/*
   File: DownSample.c
   Authors: Mike Schmidt,
            K.R. Sloan  
   Last Modified: 29 March 1991
   Purpose: Pass a WFF format image file from stdin to stdout
            down sampling by a linear factor of 2
 */
#include <stdio.h>
#include <strings.h>
#include <math.h>
#include <wff.h>

static int VERBOSE = 0;
static char *RoutineName;
static void usage()
 {
  fprintf(stderr,
          "Usage:\n\t%s [sxy][-h][-v][-x sx][-y sy]\n",
          RoutineName);
 }

static void
Pass(fdIn,fdOut, sx, sy)
 FILE *fdIn, *fdOut;
 double sx, sy;
 {
  FrameBufferType *FBin, *FBout;
  int BottomV, LeftV, TopV, RightV;
  int BottomI, LeftI, TopI, RightI;
  int widthI, widthV, heightI, heightV, bands;
  char Name[NameLength], Value[ValueLength];
  int xv,yv,xw,yw,n,b;
  unsigned short *PixelsIn, *PixelsOut,  *pIn, *pOut;
  double xwd,ywd,xwdsave;
  int passed = 0;

  FBin = (FrameBufferType *) 0;
  FBout= (FrameBufferType *) 0;
  if (FAILURE == OpenFB(&FBin))           {                        exit(-1); }
  if (FAILURE == PassImageIn(fdIn, FBin)) { (void)CloseFB(&FBin);  exit(-1); }
  if (FAILURE == OpenFB(&FBout))          { (void)CloseFB(&FBin);  exit(-1); }
 

  /*  Copy over existing NV pairs - watch for "X-PassedBy" */
  for (n=0;;n++)
   {
    GetDescriptorN(FBin, n, Name, Value);
    if (Name[0] == '\0') break;
    if (0 == strcmp(Name,"X-PassedBy"))
     {
      if ( (strlen(Value)+strlen(RoutineName)+3) > ValueLength)
       strcpy(Value,"...");
      strcat(Value,", "); strcat(Value,RoutineName);
      passed = 1;
     }
    if ( (0 != strcmp(Name,"Name")) && (0 != strcmp(Name,"Title")) )
     SetDescriptor(FBout, Name, Value);
   }
  if (FAILURE == GetBounds(FBin, &BottomI, &LeftI, &TopI, &RightI)) 
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); return; }

  /* Calculate new bounds. */


  BottomV = BottomI; LeftV = LeftI;
  widthI = RightI-LeftI+1; heightI = TopI-BottomI+1;
  widthV = widthI/sx;      heightV = heightI/sy;
  TopV   = BottomV + heightV - 1;
  RightV = LeftV   + widthV  - 1;

  ywd = (double)BottomI + 0.5 - sy;
  xwdsave = (double)LeftI + 0.5 - sx;
  
  (void)SetBounds (FBout,BottomV,LeftV,TopV,RightV);

  /*  if necessary, add "X-PassedBy" */
  if (0 == passed)
   {
    strcpy(Name,"X-PassedBy");
    strcpy(Value,RoutineName);
    SetDescriptor(FBout, Name, Value);
   }

  /* Header operations over, now we can start the output stream */
  if (FAILURE == PassImageOut(fdOut, FBout))
   { (void)CloseFB(&FBin); (void)CloseFB(&FBout); exit(-1); }

  bands = FBin->BandsPerPixel;

  PixelsIn = (unsigned short *)
              malloc( (unsigned) (widthI * bands * sizeof(unsigned short) ) );
  if ((unsigned short *)0 == PixelsIn)
   {
    fprintf( stderr, "%s: No memory for scanline buffer!\n", RoutineName );
    exit(1);
   }

  PixelsOut = (unsigned short *)
              malloc( (unsigned) (widthV * bands * sizeof(unsigned short) ) );
  if ((unsigned short *)0 == PixelsOut)
   {
    fprintf( stderr, "%s: No memory for scanline buffer!\n", RoutineName );
    exit(1);
   }

  /* Finally, pass the pixels */

  yw = BottomI - 1;
  for (yv=BottomV;yv<=TopV;yv++)
   {
    ywd += sy;
    /* step forward (if necessary) in input image to the correct scanline */
    for(; (yw < (int)ywd) && (yw < TopI);yw++) 
     if (FAILURE == NextNPixelsIn(FBin,widthI,PixelsIn))
       {
        (void)CloseFB(&FBin);
        (void)CloseFB(&FBout);
        exit(-1);
       }
    /* scan across the scanline */
    xwd = xwdsave;
    for (xv=LeftV;xv<=RightV;xv++)
     {
      xwd += sx; xw = xwd; if (xw>RightI) xw = RightI;
      pIn = &PixelsIn[xw*bands]; 
      pOut = &PixelsOut[xv*bands];
      for(b=0;b<bands;b++) *(pOut++) = *(pIn++);
     }
    /* write out the new scanline */
    if (FAILURE == NextNPixelsOut(FBout,widthV,PixelsOut))
     {
      (void)CloseFB(&FBin);
      (void)CloseFB(&FBout);
      exit(-1);
     }
    wffFlush(FBout);
   }
  (void)CloseFB(&FBin);
  (void)CloseFB(&FBout);
 }

int 
main(argc,argv)
 int argc;
 char *argv[];
 {
  int ArgsParsed = 0;
  double sx, sy;
 
  RoutineName = argv[ArgsParsed++];
  
  sx = 2.0; sy = 2.0;
  while(ArgsParsed < argc)
   {
    /* for backwards compatibility - remove SomeDay */ 
    if ('-' !=argv[ArgsParsed][0])
     {
      sx = atof(argv[ArgsParsed++]);
      sy = sx;
     }
    else switch(argv[ArgsParsed++][1])
     {
      case 'x': sx = atof(argv[ArgsParsed++]); break;
      case 'y': sy = atof(argv[ArgsParsed++]); break;
      case 'v': VERBOSE = -1; break;
      default:
      case 'h': usage(); exit(-1);
     }
   }

  Pass(stdin,stdout, sx, sy);
  exit(0);
 }

