/*
  File: devPB/DEVICE.c
  Authors: James Painter
           K.R. Sloan
  Last Modified: 10 February 1991
  Purpose: definitions for the generic graphics device used by SGP
           this version is for the adage vertigo/cubicomp paint-buffer

  See also: devTriangle.c, devCircle.c, devClip.c.
           Some of the functions promised in DEVICE.h are implemented there.
 */
#include <stdio.h>

#include <sys/types.h>

#include "vertigo.h"
#include "SGP.h"
#include "DEVICE.h"

extern devClipLine();
extern devClipRectangle();


/*  Device state  - made global for convenience ... */
int devLeft, devBottom, devRight, devTop;
int devClipL, devClipB, devClipR, devClipT;
int devCurrentColor;
FILE *devOutputStream = stdout;
double devAspectRatio = 1.0;

/* Monochrome flag */
static int mono = 0;


/*
 *  Global variables for paint buffer
 */
int	pb_id;                  /* The paint (frame) buffer's id        */
char	*pb_name;		/* The frame buffer's name		*/
char	pb_name_buffer[256];	/* Space to store the frame buffer name	*/
int     pb_xres, pb_yres;       /* Frame buffer resolution              */

/*
 *  Global variables for paint buffer
 */
int	pb_id;                  /* The paint (frame) buffer's id        */
char	*pb_name;		/* The frame buffer's name		*/
char	pb_name_buffer[256];	/* Space to store the frame buffer name	*/
int     pb_xres, pb_yres;       /* Frame buffer resolution              */

int pbInitialize()
{
    boolean	config_error();
    boolean	display_image();
    char	*getenv();

    int		status;
    char	*str;


    /*
     *  Open the frame buffer.
     */

    if ( ( pb_name = getenv( CONFIG_ENV_BUFFER ) ) == NULL )
    {
	/*
	 *  Get the default device name for the BUFFER environment
	 *  variable.  We make a copy of str because strings
	 *  returned by config_get point to static memory.
	 */

	status = config_get( CONFIG_TYPE_ENV, CONFIG_ENV_BUFFER, &str );

	if ( status == CONFIG_NO_ENTRY )
	{
	    (void) printf(
		    "missing BUFFER entry in the configuration file\n" );
	    return status;
	}
	else if ( config_error( status, CONFIG_ENV_BUFFER ) )
	{
	    /*
	     *  There was something wrong with the config file entry
	     *  for BUFFER.
	     */

	    return status;
	}
	(void) strcpy( pb_name_buffer, str );
	pb_name = pb_name_buffer;
    }


    /*
     *  Open the frame buffer.
     */

    if ( ( status = fb_open( pb_name, -1, &pb_id ) ) != FB_SUCCESS )
    {
	/*
	 *  There was a problem in opening the frame buffer.
	 */

	fb_error_save( status, pb_name );
	return status;
    }


    /*
     *  Do final initialization of the frame buffer.
     */

    config_get( CONFIG_TYPE_RESOLUTION, pb_name, &pb_xres, &pb_yres);
    fb_reset( pb_id );
    fb_display_mode( pb_id, FB_DISPLAY_RGB );
    
    return FB_SUCCESS;
}

/*
   devInit initializes the display, loading color maps if necessary
   There are two modes: color and mono.  Here, we assume a full-color
   display, and the two modes are identical

   devInit may also need to fill in the "physical" size of the display
   here, we output the "physical" limits to help your interpreter 
 */
/* the PaintBuffer seems to be offset in x, so... */
#define pb_offset (24)
int devInit(color, OutputStream)
 int color;
 FILE *OutputStream;
 {
  extern char *malloc();

  devOutputStream = OutputStream; /* but we don't use it... */

  /* Initialize the paint buffer */
  if (pbInitialize() != FB_SUCCESS) 
   {
    error_dump( E_ALL, TRUE, (pfv) NULL, stderr );
    exit(-1);
   }

  devLeft =  0;  devTop = pb_yres-1; devRight = pb_xres-1; devBottom = 0;
  devLeft +=  pb_offset; devRight += pb_offset;
  devSetClipWin(devLeft,devBottom,devRight,devTop);
  devCurrentColor = devColorCode(1.0, 1.0, 1.0);

  return( 0 ); /* success */ 
 }

/*
  devClose insures that all operations are complete and closes up
 */
int devClose()
 {
  fb_close( pb_id );
 }

/*
  devSetClipWin establishes a clipping window
  restrictions such as (L<R) must be enforced here!
 */
int devSetClipWin(L, B, R, T)
 int L, B, R, T;
 {
  if (L<R) { devClipL=L; devClipR=R; } else { devClipL=R; devClipR=L;} 
  if (B<T) { devClipB=B; devClipT=T; } else { devClipB=T; devClipT=B;} 
 
  if (devLeft < devRight)
   {
    if (devClipL < devLeft)   devClipL = devLeft;
    if (devClipR > devRight)  devClipR = devRight;
   }
  else
   {
    if (devClipL < devRight)  devClipL = devRight;
    if (devClipR > devLeft)   devClipR = devLeft;
   }

  if (devBottom < devTop)
   {
    if (devClipB < devBottom) devClipB = devBottom;
    if (devClipT > devTop)    devClipT = devTop;
   }
  else 
   {
    if (devClipB < devTop)    devClipB = devTop;
    if (devClipT > devBottom) devClipT = devBottom;
   }

  return(0); /* success */
 }

/*
 devClearScreen fills the entire screen with a particular color
 */
int devClearScreen(color)
 int color;
 {
  byte r,g,b,m;

  r = color & 0xff;
  g = (color >> 8) & 0xff;
  b = (color >> 16) & 0xff;
  m = 255;
  fb_flood( pb_id, devLeft, devBottom, devRight, devTop, r, g, b, m );
  return(0); /* success */
 }

/*
   the following routines draw primitive shapes on the physical device
   All of them clip against the device's clipping window.
   All of them paint in the device's current color
 */

/*
   Draw a line from (x1,y1) to (x2,y2) using Bresenham's algorithm.
   All coordinates are assumed to be expressed in raw device coordinates.
 */
int devLine(x1, y1, x2, y2)
 int x1, y1, x2, y2;		/* The coordinates of the endpoints */
 {
  byte r,g,b,m;

  if (devClipLine(&x1,&y1,&x2,&y2)) return(0); /* success */

  r = devCurrentColor & 0xff;
  g = (devCurrentColor >> 8) & 0xff;
  b = (devCurrentColor >> 16) & 0xff;
  m = 255;
  fb_line( pb_id, x1, y1, x2, y2, r, g, b, m );
  return(0);
 }


/*
 devRect fills a rectangle - we assume this is a primitive
 */
int devRect(L, B, R, T)
 int L, B, R, T;
 {
  int tmp;
  byte r, g, b, m;

  if (devClipRectangle(&L, &T, &R, &B)) return(0); /* success */

  if (L > R)   {  tmp = L; L = R; R = tmp; }
  if (B > T)   {  tmp = T; T = B; B = tmp; }

  r = devCurrentColor & 0xff;
  g = (devCurrentColor >> 8) & 0xff;
  b = (devCurrentColor >> 16) & 0xff;
  m = 255;
  fb_flood( pb_id, L, B, R, T, r, g, b, m );

  return(0); /* success */
 }

/*
 devDisc fills a disc - see devCircle.c
 */

/*
 devCircle draws a circle - see devCircle.c
 */

/*
 devSolidTriangle fills a triangle - see devTriangle.c
 */

/*
 devShadeTriangle shades a triangle
   with color codes specified at each vertex - see devTriangle.c
 */

/*
 devSetColor establishes the current color
 */
int devSetColor(color)
 int color;
 {
  devCurrentColor = color;
  return(0); /* success */
 }

/*
   devColorCode returns a framebuffer value - here we assume a full-color
   display, with 8-bits each of rgb, packed as:
            [00000000bbbbbbbbggggggggrrrrrrrr]
   OR, an 8-bit monochrome display, with 8-bits of i = [iiiiiiii]
   In this case, if the user program specifies a color as RGB, we convert
   to I using the CIE weights (0.30, 0.59, 0.11) 

 */ 
int devColorCode(R, G, B)
 double R,G,B; 
 {
  int devR, devG, devB;

  if (!mono) 
   {
    devR = 255.0 * R;
    devG = 255.0 * G;
    devB = 255.0 * B;
    if (devR > 255) devR = 255; else if (devR < 0) devR = 0;
    if (devG > 255) devG = 255; else if (devG < 0) devG = 0;
    if (devB > 255) devB = 255; else if (devB < 0) devB = 0;
    return (  ((devR & 0xff)      )
	    | ((devG & 0xff) <<  8)
	    | ((devB & 0xff) << 16) );
   }
  else 
   {
    devR = (.30 * R  + .59 * G + .11 * B) * 255.0;
    if (devR > 255) devR = 255; else if (devR < 0) devR = 0;
    return (devR & 0xFF);
   }
 }

/*
  devColorDecode generates r,g,b values in [0.0, 1.0] 
 */
int devColorDecode(code, R, G, B)
 int code;
 double *R, *G, *B;
 {
  int devR, devG, devB;

  devR = (code      ) & 0xff;
  *R = (double)devR / 255.0;
  if (!mono) 
   {
    devG = (code >>  8) & 0xff;
    *G = (double)devG / 255.0;
    devB = (code >> 16) & 0xff;
    *B = (double)devB / 255.0;
   }
  else 
   {
    *G = *B = *R;
   }
  return (0);
 }

/*       -30-         */
