/*

  File: SGPSpline.c
  Author: James Painter
  Last Modified: February 22, 1985
  Purpose:  Compute a natural cubic interpolatory spline through a set of
            control points.  The spline is computed using uniform knot 
            spacing.  If the first and last points are identical, the
            end conditions are modified to be appropriate for a closed
            curve. Otherwise the spline is computed so that the second
            derivitives are zero at the endpoints.

*/
#include <math.h>
#include <stdio.h>

#ifndef TRUE
#  define TRUE 1
#endif
#ifndef FALSE
#define FALSE 0
#endif


/* External Declarations */

extern char *malloc();

/* Forward declarations. */

static float ComputeCubicInterpSpline();
static int   ComputeDerivitives();
static void SolveTriDiag(), SolveTriDiagFwdElim(), SolveTriDiagBackSub();

int
SGPSpline ( NIn, InPoints, NOut, OutPoints )
 /* returns TRUE if successful. */
 int   NIn;
 float InPoints[]; /* NIn input points to be fitted.
                      Note:  For a closed curve replicate the first point at
                             the end of the array.
                    */
 int   NOut;
 float OutPoints[]; /* space for NOut output points. */
 {
  int i,j,v;
  float r, *Derivitives;

  /*
     We must have at least 3 points to avoid singular matricies.
   */
  if (NIn<3 || (NIn==3 && InPoints[0]==InPoints[2])) return FALSE;
  
  /*
      Alocate space for the derivitives.
   */
  if (! (Derivitives = (float *) malloc( NIn*sizeof(float) ) ) )
   {
    fprintf( stderr, "No memory available in SGPSpline\n" );
    exit ( 1 );
   }

  /*
    Compute the derivitives at the control points.
   */
  ComputeDerivitives( NIn, InPoints, Derivitives );

  /*
     Compute NOut evenly spaced output points.
   */
  OutPoints[0] = InPoints[0];
  for (i=1; i<NOut; i++)
   {
    v = i * (NIn-1);
    j = v / (NOut-1);
    r = (float) (v%(NOut-1)) / (float) (NOut-1);
    if (j==NIn-1)
     {
      j = NIn-2;
      r = 1.;
     }
    OutPoints[i] = ComputeCubicInterpSpline( InPoints+j, Derivitives+j, r );
   }
  return TRUE;
 }

static float
ComputeCubicInterpSpline ( InPoints, Derivitives, u )
 float InPoints[];
 float Derivitives[];
 float u;
 {
  float dx, Result;

  dx = InPoints[1] - InPoints[0];
  Result = u * ( -2.*dx + Derivitives[0] + Derivitives[1] );
  Result = u * ( Result + 3.*dx - 2.*Derivitives[0] - Derivitives[1]);
  Result = u * ( Result + Derivitives[0]);
  Result += InPoints[0];

  return Result;
 }

static int
ComputeDerivitives ( NIn, InPoints, D )
 int NIn;
 float InPoints[];
 float D[];
 {
  typedef float Array3[3];
  Array3 *A;
  int i, NA;

  /*
     Create the (almost) Tri-Diagonal set of equations to solve.
   */
  if (! (A = (Array3 *) malloc( sizeof(Array3)*NIn ) ) )
   {
    fprintf( stderr, "No memory available in SGPSpline.\n" );
    exit ( 1 );
   }
  
  for (i=1; i<NIn-1; i++)
   {
    A[i][0] = 1;
    A[i][1] = 4.;
    A[i][2] = 1.;
    D[i]    = 3.*(InPoints[i+1] - InPoints[i-1]);
   }

  if (InPoints[0] == InPoints[NIn-1])   /* Closed curve */
   {
    A[0][0] = 1.;
    A[0][1] = 4.;
    A[0][2] = 1.;
    D[0]    = 3.*( InPoints[1] - InPoints[NIn-2] );
    NA      = NIn-1;
   }
  else                                  /* "Natural" end conditions */
   {
    A[0][0] = 0.;
    A[0][1] = 2.;
    A[0][2] = 1.;
    D[0]    = 3.*( InPoints[1] - InPoints[0] );
    A[NIn-1][0] = 1.;
    A[NIn-1][1] = 2.;
    A[NIn-1][2] = 0.;
    D[NIn-1] = 3.*( InPoints[NIn-1] - InPoints[NIn-2] );
    NA = NIn;
   }
  /* Now solve it. */
  SolveTriDiag( NA, A, D );
  if (NA < NIn) D[NA] = D[0];
 }

static void
SolveTriDiag( N, A, B  )
int N;          /* Input: the number of equations. Must be at least 2 */
float A[][3];	/* On input: 
		  
                   For i=1,N-2:
		  
		   The non-zero coeffiecents of the N equations.
                   A[i][0] is the element to the left of the diagonal,
 		   A[i][1] is the diagonal element,
		   A[i][2] is the element to the right of the diagonal.
		  
                   The 0 and N-1'th rows are slightly different (hence the
		   "almost" tri-diagonal).  
		  
		   A[0][1] is the element on the diagonal of row 0;
		   A[0][2] is the element directly to the right of A[0][0]
		   A[0][0] is the element in the last column.
		  
		   A[N-1][1] is the diagonal element in row N-1.
		   A[N-1][0] is the element directly to the left of A[N-1][1]
		   A[N-1][2] is the element in the first column.
		  
		   On output:  A is Munged.
 		*/

float B[];	/* On Input:
		  
		   The column vector holding the right hand side terms of the
		   N equations.
		  
		   On Output:
		  
		   The solution vector for the N equations.
                 */
 {
  float *C;
  
  if (N >2)
   {
    if (! (C = (float *) malloc ( N*sizeof(float)) ))
     {
      fprintf( stderr, "No memory in SGPSpline.\n" );
      exit ( 1 );
     }
    C[0] = A[0][0];
    SolveTriDiagFwdElim ( N, A, B, C );
    SolveTriDiagBackSub ( N, A, B, C );
    free( C );
   }
 }

static void
SolveTriDiagFwdElim ( N, A, B, C )
/*
   Purpose: Do the forward elimination on A for SolveTriDiag.
            N, A are as above.
 */
 int N;
 float  A[][3];
 float  B[];
 float  C[];
 {
  register int i;
  
  /*  Eliminate all below diagonal terms.  */
  for (i=1; i<(N-2); i++)
   {
    A[i][0] /= A[i-1][1];
    A[i][1] -= A[i][0]*A[i-1][2];
    C[i] = -A[i][0]*C[i-1];
    B[i] -= A[i][0]*B[i-1];
    A[N-1][2] /= A[i-1][1];
    A[N-1][1] -= A[N-1][2]*C[i-1];
    B[N-1] -= A[N-1][2]*B[i-1];
    A[N-1][2] = -( A[N-1][2]*A[i-1][2] );
   }
  A[N-2][0] /= A[N-3][1];
  A[N-2][1] -= A[N-2][0]*A[N-3][2];
  A[N-2][2] -= A[N-2][0]*C[N-3];
  B[N-2]  -= A[N-2][0]*B[N-3];
  A[N-1][2] /= A[N-3][1];
  A[N-1][0] -= A[N-1][2]*A[N-3][2];
  A[N-1][1] -= A[N-1][2]*C[N-3];
  B[N-1]  -= A[N-1][2]*B[N-3];
  A[N-1][0] / = A[N-2][1];
  A[N-1][1] -= A[N-1][0]*A[N-2][2];
  B[N-1]  -= A[N-1][0]*B[N-2];
 }

static void
SolveTriDiagBackSub ( N, A, B, C )
/*
   Do the backward substitution on B.
 */
 int N; 
 float A[][3];
 float B[];
 float C[];
 {
  register int i;

  B[N-1] /= A[N-1][1];
  B[N-2] = (B[N-2] - A[N-2][2]*B[N-1]) / A[N-2][1];
  for (i=N-3; i>=0; i--)
   {
    B[i] = ( B[i] - A[i][2]*B[i+1] - C[i]*B[N-1]) / A[i][1];
   }
 }
