/*
 This file is part of GNU Taler
 (C) 2021-2024 Taler Systems S.A.

 GNU Taler is free software; you can redistribute it and/or modify it under the
 terms of the GNU General Public License as published by the Free Software
 Foundation; either version 3, or (at your option) any later version.

 GNU Taler is distributed in the hope that it will be useful, but WITHOUT ANY
 WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
 A PARTICULAR PURPOSE.  See the GNU General Public License for more details.

 You should have received a copy of the GNU General Public License along with
 GNU Taler; see the file COPYING.  If not, see <http://www.gnu.org/licenses/>
 */

/**
 *
 * @author Sebastian Javier Marchano (sebasjm)
 */

import {
  Duration,
  HttpStatusCode,
  LoginTokenRequest,
  LoginTokenScope,
  TranslatedString,
} from "@gnu-taler/taler-util";
import {
  ButtonBetterBulma,
  LocalNotificationBannerBulma,
  useChallengeHandler,
  useLocalNotificationBetter,
  useTranslationContext,
} from "@gnu-taler/web-util/browser";
import { Fragment, h, VNode } from "preact";
import { useState } from "preact/hooks";
import { SolveMFAChallenges } from "../../components/SolveMFA.js";
import { useSessionContext } from "../../context/session.js";

const TALER_SCREEN_ID = 79;

interface Props {
  showCreateAccount?: boolean;
}

export const TEMP_TEST_TOKEN = (description: TranslatedString) =>
  ({
    scope: LoginTokenScope.All,
    duration: Duration.toTalerProtocolDuration(Duration.fromMilliseconds(100)),
    description,
  }) as LoginTokenRequest;

export const FOREVER_REFRESHABLE_TOKEN = (description: TranslatedString) =>
  ({
    scope: LoginTokenScope.All_Refreshable,
    duration: Duration.toTalerProtocolDuration(Duration.getForever()),
    description,
  }) as LoginTokenRequest;

const VERSION = typeof __VERSION__ !== "undefined" ? __VERSION__ : undefined;
export function LoginPage({ showCreateAccount }: Props): VNode {
  const [password, setPassword] = useState("");

  const { state, logIn, getInstanceForUsername, config } = useSessionContext();
  const [username, setUsername] = useState(
    showCreateAccount ? "" : state.instance,
  );

  const { i18n } = useTranslationContext();

  const [notification, safeFunctionHandler] = useLocalNotificationBetter();
  const mfa = useChallengeHandler();

  const login = safeFunctionHandler(
    (usr: string, pwd: string, challengeIds: string[]) => {
      const api = getInstanceForUsername(usr);
      return api.createAccessToken(
        usr,
        pwd,
        FOREVER_REFRESHABLE_TOKEN(i18n.str`Logged in`),
        { challengeIds },
      );
    },
    !username || !password ? undefined : [username, password, []],
  );
  login.onSuccess = (success, usr) => {
    logIn(usr, success.access_token);
  };
  login.onFail = (fail) => {
    switch (fail.case) {
      case HttpStatusCode.Accepted:
        mfa.onChallengeRequired(fail.body);
        return undefined;
      case HttpStatusCode.Unauthorized:
        return i18n.str`Unauthorized`;
      case HttpStatusCode.NotFound:
        return i18n.str`Not found`;
    }
  };
  const retry = login.lambda((ids: string[]) => [
    login.args![0],
    login.args![1],
    ids,
  ]);

  if (mfa.pendingChallenge) {
    return (
      <SolveMFAChallenges
        currentChallenge={mfa.pendingChallenge}
        onCompleted={retry}
        onCancel={mfa.doCancelChallenge}
      />
    );
  }

  return (
    <Fragment>
      <LocalNotificationBannerBulma notification={notification} />
      <div class="columns is-centered" style={{ margin: "auto" }}>
        <div class="column is-two-thirds ">
          <div class="modal-card" style={{ width: "100%", margin: 0 }}>
            <header
              class="modal-card-head"
              style={{ border: "1px solid", borderBottom: 0 }}
            >
              <p class="modal-card-title">
                <i18n.Translate>Login required</i18n.Translate>
              </p>
            </header>
            <section
              class="modal-card-body"
              style={{ border: "1px solid", borderTop: 0, borderBottom: 0 }}
            >
              <div class="field is-horizontal">
                <div class="field-label is-normal">
                  <label class="label">
                    <i18n.Translate>Username</i18n.Translate>
                  </label>
                </div>
                <div class="field-body">
                  <div class="field">
                    <p class="control is-expanded">
                      <input
                        class="input"
                        type="text"
                        placeholder={i18n.str`instance name`}
                        name="username"
                        onKeyPress={(e) =>
                          e.keyCode === 13 ? login.call() : null
                        }
                        value={username}
                        onInput={(e): void =>
                          setUsername(e?.currentTarget.value)
                        }
                      />
                    </p>
                  </div>
                </div>
              </div>
              <div class="field is-horizontal">
                <div class="field-label is-normal">
                  <label class="label">
                    <i18n.Translate>Password</i18n.Translate>
                  </label>
                </div>
                <div class="field-body">
                  <div class="field">
                    <p class="control is-expanded">
                      <input
                        class="input"
                        type="password"
                        placeholder={i18n.str`current password`}
                        name="token"
                        onKeyPress={(e) =>
                          e.keyCode === 13 ? login.call() : null
                        }
                        value={password}
                        onInput={(e): void =>
                          setPassword(e?.currentTarget.value)
                        }
                      />
                    </p>
                  </div>
                </div>
              </div>
            </section>
            <footer
              class="modal-card-foot "
              style={{
                justifyContent: "space-between",
                border: "1px solid",
                borderTop: 0,
              }}
            >
              {!config.have_self_provisioning ? (
                <div />
              ) : (
                <a
                  href={
                    !username || username === "admin"
                      ? undefined
                      : `#/account/reset/${username}`
                  }
                  class="button "
                  disabled={!username || username === "admin"}
                >
                  <i18n.Translate>Forgot password</i18n.Translate>
                </a>
              )}
              <ButtonBetterBulma onClick={login} type="submit">
                <i18n.Translate>Confirm</i18n.Translate>
              </ButtonBetterBulma>
            </footer>
          </div>
          {!showCreateAccount ? undefined : (
            <div style={{ marginTop: 8 }}>
              <a href={"#/account/new"} class="has-icon">
                <span class="icon">
                  <i class="mdi mdi-account-plus" />
                </span>
                <span class="menu-item-label">
                  <i18n.Translate>Create new account</i18n.Translate>
                </span>
              </a>
            </div>
          )}
        </div>
      </div>
      <div
        style={{
          position: "absolute",
          bottom: 0,
          height: "2rem",
        }}
      >
        Version {VERSION}
      </div>
    </Fragment>
  );
}
