/* 
 *    Programmed By: Mohammed Isam Mohammed [mohammed_isam1984@yahoo.com]
 *    Copyright 2013, 2014, 2015, 2016, 2017, 2018 (c)
 * 
 *    file: main.c
 *    This file is part of Prime.
 *
 *    Prime is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    Prime is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with Prime.  If not, see <http://www.gnu.org/licenses/>.
 */    

#define MAIN    1

#include <stdio.h>
#include <string.h>
#include <sys/ioctl.h>
#include <unistd.h>
#include <poll.h>
#include <signal.h>
#include <termios.h>
#include <sys/ioctl.h>
#include <stdlib.h>
#include "defs.h"
#include "edit.h"
#include "options.h"
//#include "print.h"
#include "find.h"
#include "cutcopy.h"
#include "file.h"
#include "dirstruct.h"

#define  DIRCOLOR(x)     (dirs [(x)]->type-'a')
#define FILECOLOR(x)     (files[(x)]->type-'a')

extern void showPropertiesDialog();
char *input_str;
static sig_atomic_t end = 0;


//extern int PRINTING;
//extern void showPrintDialogBox();
//extern void freePrinters();
//extern int findInstalledPrinters();

void exit_gracefully()
{
    restoreTerminal(); 
    write_config_file();
    fcloseall();
    //freePrinters();
    showCursor();
    setScreenColors(WHITE, BGDEFAULT);
    clearScreen(); 
    fprintf(stdout, "\x1b[0m");	/* reset settings */
    exit(0);
}

/***************************************
 * sighandler(): 
 * Procedure that sets trap to the system
 * signals that are issued when user 
 * presses CTRL-C or CTRL-Z...
 * *************************************/
void sighandler(int signo)
{
    if(signo == 2) 
    {	//CTRL-C pressed
      copyMarked();
      //redraw main window but don't clear the area
      drawBox(1, 1, SCREEN_H, SCREEN_W, " Prime File Manager ", NO);
    } 
    else if(signo == SIGTSTP) 
    { //CTRL-Z pressed
      int i = msgBoxH("Are you sure you want to exit?", YES | NO, CONFIRM);
      if(i == YES) 
      { 	//exit gracefully
	exit_gracefully();
      } 
      else
      {
	refreshWindows();
      }
    } 
    else 
    {
      end = 1;
    }
}

struct termios oldtio, curtio;
struct sigaction sa;

void print_dir_highlight(int selectedDir)
{
    int pos = firstVisDir+selectedDir;
    int j;
    fprintf(stdout, "\x1b[%d;%dH%s", selectedDir+4, 3, dirHighLight);
    if(dirs[pos]->namelen >= MAX_DIR_NAME_LEN)
    {
        fprintf(stdout, "\x1b[%d;%dH%c", selectedDir+4, 3, dirs[pos]->star);
        for(j = 0; j < MAX_DIR_NAME_LEN-3; j++)
            fprintf(stdout, "%c", dirs[pos]->name[j]);
        fprintf(stdout, "..");
    }
    else
    {
        fprintf(stdout, "\x1b[%d;%dH%c%s", selectedDir+4, 3, dirs[pos]->star, dirs[pos]->name);
    }
}

void print_file_highlight(int selectedFile)
{
    int pos = firstVisFile+selectedFile;
    int j;
    fprintf(stdout, "\x1b[%d;%dH%s", selectedFile+4, (SCREEN_W/2)+1, fileHighLight);
    if(files[pos]->namelen >= MAX_DIR_NAME_LEN)
    {
        fprintf(stdout, "\x1b[%d;%dH%c", selectedFile+4, (SCREEN_W/2)+1, files[pos]->star);
        for(j = 0; j < MAX_DIR_NAME_LEN-3; j++) 
            fprintf(stdout, "%c", files[pos]->name[j]);
        fprintf(stdout, "..");
    } 
    else 
    { 
        fprintf(stdout, "\x1b[%d;%dH%c%s", selectedFile+4, (SCREEN_W/2)+1, 
	    files[pos]->star, files[pos]->name);
    }
}

void showErrorMsgBox(char *msg, char *arg)
{
    int len1 = strlen(msg);
    int len2 = strlen(arg);
    char tmp[len1+len2+1];
    strcpy(tmp, msg);
    strcat(tmp, arg);
    msgBoxH(tmp, OK, ERROR); 
}

void shiftDirsUp(int pos)
{
    if(dirs[pos]->name) free(dirs[pos]->name);
    free(dirs[pos]);
    for( ; pos < totalDirs-1; pos++)
    {
        dirs[pos] = dirs[pos+1];
    }
    dirs[pos] = NULL;
    totalDirs--;
}

void shiftFilesUp(int pos)
{
    if(files[pos]->name) free(files[pos]->name);
    free(files[pos]);
    for( ; pos < totalFiles-1; pos++)
    {
        files[pos] = files[pos+1];
    }
    files[pos] = NULL;
    totalFiles--;
}

/***************************************
 * main(): 
 * Main program loop.
 * *************************************/
int main(int argc, char **argv) 
{
    parse_args(argc, argv);
    cwd = getcwd(NULL, 0);
    if(!cwd)
    {
        fprintf(stderr, "Error: unable to detect current working directory\n");
        exit(1);
    }
    cwdlen = strlen(cwd);
    init();
    read_config_file(); 
    clearScreen();
    hideCursor();
    setScreenColorsI(COLOR_WINDOW);
    //draw main window
    drawBox(1, 1, SCREEN_H, SCREEN_W, " Prime File Manager ", YES);
    drawMenuBar(2, 2, SCREEN_W-2);
    //draw right sub-window
    drawBox(3, (int)(SCREEN_W/2), SCREEN_H-5, SCREEN_W-1, " File view ", YES);
    //draw bottom sub-window
    drawBox(SCREEN_H-4, 2, SCREEN_H-1, SCREEN_W-1, " Quick functions ", YES);
    //draw left sub-window
    drawBox(3, 2, SCREEN_H-5, (int)(SCREEN_W/2)-1, " Directory view ", YES);
    //redraw main window but don't clear the area
    drawBox(1, 1, SCREEN_H, SCREEN_W, " Prime File Manager ", NO);

    //CHECK FOR PRINTERS ON SYSTEM
    /*
    fprintf(log_file, "Checking for installed printers");
    if(findInstalledPrinters() != 0) {
    fprintf(log_file, " [ERROR]: Couldn't find printers..\n");
    } else fprintf(log_file, " [ OK ]..\n");
    */

    refreshBottomView();
    scanDir(".");

    int ch, i;
    while(!end) 
    {	//infinite program loop//
        ch = getKey();
        switch(ch)
        {
            case(DEL_KEY):
                if(GNU_DOS_LEVEL > 3) break;
do_del:
                if(numStarred == 0) 
                {
                    if(activeWindow == DIR_WIN)
                    {
                        int j = firstVisDir+selectedDir;
                        if(j < 2) break;    // ignore '.' and '..'
                        int k = msgBoxH("Are you sure you want to delete\n"
                            "this directory and all its subdirectories?", YES|NO, CONFIRM);
                        if(k == YES)
                        {
                            char *logfilename = tmpnam(NULL);
                            //msgBox(logfilename, OK, INFO);
                            FILE *logfile = fopen(logfilename, "wb");
                            deleteThisDir(dirs[j]->name, 0, logfile);
                            purgeLogFile(logfile, logfilename);
                            //free(logfilename);
                            shiftDirsUp(j);
                            if(j == totalDirs)
                            {
                                if(totalDirs <= numVisDirs) selectedDir--;
                                else firstVisDir--;
                            }
                        }
                    } 
                    else 
                    {
                        int j = firstVisFile+selectedFile;
                        if(files[j]->type == '%') break;    // ignore empty dirs
                        int k = msgBoxH("Are you sure you want to delete this file?", 
                                       YES|NO, CONFIRM);
                        if(k == YES)
                        {
                            remove(files[j]->name);
                            shiftFilesUp(j);
                            if(j == totalFiles)
                            {
                                if(totalFiles <= numVisFiles) selectedFile--;
                                else firstVisFile--;
                            }
                        }
                    }
                    refreshWindows();
                    refreshBottomView();
                }
                else 
                {
                    deleteMarked();
                    scanDir(cwd);
                }
                break;
            case(UP_KEY):
                if(GNU_DOS_LEVEL > 1) break;
do_up:
                if(activeWindow == DIR_WIN) 
                {
                    selectedDir--; 
                    if(selectedDir < 0) 
                    {
                        selectedDir = 0;
                        if(firstVisDir > 0)
                        {
                            firstVisDir--;
                            refreshDirView();
                        }
                    }
                    else 
                    {
                        setScreenColors(FILE_DIR_COLOR[DIRCOLOR(firstVisDir+selectedDir+1)],
                                        BG_COLOR[COLOR_WINDOW]);
                        print_dir_highlight(selectedDir+1);
                        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                        print_dir_highlight(selectedDir);
                    }
                    fflush(stdout);
                }
                else 
                {
                    //if(strcmp(files[0]->name, "(Empty folder)") == 0) break;
                    if(files[0]->type == '%') break;
                    selectedFile--;
                    if(selectedFile < 0) 
                    {
                        selectedFile = 0;
                        if(firstVisFile > 0)
                        {
                            firstVisFile--;
                            refreshFileView();
                        }
                    }
                    else 
                    {
                        setScreenColors(FILE_DIR_COLOR[FILECOLOR(firstVisFile+selectedFile+1)],
                                        BG_COLOR[COLOR_WINDOW]);
                        print_file_highlight(selectedFile+1);
                        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                        print_file_highlight(selectedFile);
                    }
                    fflush(stdout);
                }
                break;
            case(DOWN_KEY):
                if(GNU_DOS_LEVEL > 1) break;
do_down:
                if(activeWindow == DIR_WIN) 
                {
                    selectedDir++;
                    if(selectedDir >= totalDirs) selectedDir--;
                    if(selectedDir >= numVisDirs) 
                    {
                        selectedDir = numVisDirs-1;
                        if((firstVisDir+numVisDirs) < totalDirs) 
                        { 
                            firstVisDir++;
                            refreshDirView(); 
                        }
                    }
                    else 
                    {
                        setScreenColors(FILE_DIR_COLOR[DIRCOLOR(firstVisDir+selectedDir-1)],
                                        BG_COLOR[COLOR_WINDOW]);
                        print_dir_highlight(selectedDir-1);
                        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                        print_dir_highlight(selectedDir);
                    }
                    fflush(stdout);
                }
                else 
                {
                    //if(strcmp(files[0]->name, "(Empty folder)") == 0) break;
                    if(files[0]->type == '%') break;
                    selectedFile++;
                    if(selectedFile >= totalFiles) selectedFile--;
                    if(selectedFile >= numVisFiles)
                    {
                        selectedFile = numVisFiles-1;
                        if((firstVisFile+numVisFiles) < totalFiles)
                        {
                            firstVisFile++;
                            refreshFileView();
                        }
                    }
                    else 
                    {
                        setScreenColors(FILE_DIR_COLOR[FILECOLOR(firstVisFile+selectedFile-1)],
                                        BG_COLOR[COLOR_WINDOW]);
                        print_file_highlight(selectedFile-1);
                        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                        print_file_highlight(selectedFile);
                    }
                    fflush(stdout);
                }
                break;
            case(TAB_KEY):	//toggle active window with TAB press
                if(activeWindow == DIR_WIN) activeWindow = FILE_WIN;
                else                        activeWindow = DIR_WIN ;
                refreshFileView();
                refreshDirView();
                break;
            case(ENTER_KEY):
                if(ALT)
                {
                    if(activeWindow == FILE_WIN && files[0]->type == '%') break;
                    showPropertiesDialog();
                    scanDir(cwd);
                }
                else
                {
                    if(activeWindow == DIR_WIN) 
                    {	//navigate to the selected directory
                        scanDir(dirs[firstVisDir+selectedDir]->name);
                    }
                    else
                    {
                        if(files[0]->type != '%')
                        {
                            showPropertiesDialog();
                            scanDir(cwd);
                        }
                    }
                }
                break;
            case(SPACE_KEY):		//toggle select/unselect file or directory
                toggleSelected();
                break;
            case('s'):
            case('S'):
                if(CTRL)
                {
                    if(GNU_DOS_LEVEL <= 4) break;
                    findFile();
                    hideCursor();
                    scanDir(cwd);
                }
                else goto insert_char;
                break;
            case('f'):
            case('F'):
                if(ALT) 
                {
                    showMenu(0);
                } 
                else if(CTRL) 
                {
                    if(GNU_DOS_LEVEL > 4) break;
                    findFile();
                    //msgBox(cwd, OK, INFO);
                    cwd = getcwd(NULL, 0);
                    //msgBox(cwd, OK, INFO);
                    scanDir(cwd);
                }
                else goto insert_char;
                break;
            case('e'):
            case('E'):
                if(CTRL) 
                {
                    if(GNU_DOS_LEVEL > 2) goto do_end;
                    exportTree(YES);
                    hideCursor();
                    scanDir(cwd);
                }
                else if(ALT) 
                {
                    showMenu(1);
                } 
                else goto insert_char;
                break;
            case('h'):
            case('H'):
                if(ALT) 
                {
                    showMenu(3);
                }
                else goto insert_char;
                break;
            case('x'):
            case('X'):
                if(CTRL) 
                {
                    if(GNU_DOS_LEVEL > 4)
                    {
                        int k=(int)SCREEN_H/2;
                        int l=(int)SCREEN_W/2;
                        drawBox(k-1, l-23, k+1, l+23, NULL, YES);
                        setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
                        fprintf(stdout, "\e[%d;%dH", k, l-22);
                        fprintf(stdout, "[C-c] Quit  [C-f] Open location [C-g] Cancel");
                        int loop = 1;
                        while(loop)
                        {
                            ch = getKey();
                            if(ch == 'c' && CTRL) goto do_exit;
                            else if(ch == 'f' && CTRL)
                            {
                                file_open_location();
                                loop = 0;
                                break;
                            }
                            else if(ch == 'g' && CTRL)
                            {
                                refreshWindows();
                                loop = 0;
                                break;
                            }
                        }
                    }
                    else cutMarked();
                }
                else goto insert_char;
                break;
            case('v'):
            case('V'):
                if(CTRL)
                {
                    if(GNU_DOS_LEVEL > 3) break;
                    pasteMarked();
                } 
                else goto insert_char;
                break;
            case('p'):
            case('P'):
                if(CTRL)
                {
                    if(GNU_DOS_LEVEL > 1) goto do_up;
                    msgBoxH("Oops! This function is currently not implemented.", OK, INFO);
                    //showPrintDialogBox();
                    refreshWindows();
                }
                else goto insert_char;
                break;
            case('a'):
            case('A'):
                if(CTRL)
                {
                    if(GNU_DOS_LEVEL > 2) goto do_home;//GNU key binding
                    markAll();
                    refreshAll();
                } 
                else goto insert_char;
                break;
            case('w'):
            case('W'):
                if(CTRL) 
                {
                    if(GNU_DOS_LEVEL < 4) clearSelection();
                    else cutMarked();
                } 
                else goto insert_char;
                break;
            case('o'):
            case('O'):
                if(CTRL) 
                {
                    if(GNU_DOS_LEVEL > 4) break;
                    file_open_location();
                    refreshWindows();
                } 
                else if(ALT)
                {//open options menu
                    showMenu(2);
                }
                else goto insert_char;
                break;
            case(HOME_KEY):
                if(GNU_DOS_LEVEL > 1) break;
do_home:
                if(activeWindow == DIR_WIN) 
                {
                    firstVisDir = 0;
                    selectedDir = 0;
                    refreshDirView();
                } 
                else if(activeWindow == FILE_WIN) 
                {
                    firstVisFile = 0;
                    selectedFile = 0;
                    refreshFileView();
                }
                fflush(stdout);
                break;
            case(END_KEY):
                if(GNU_DOS_LEVEL > 1) break;
do_end:
                if(activeWindow == DIR_WIN) 
                {
                    if(totalDirs <= numVisDirs) 
                    {
                        selectedDir = totalDirs-1;
                    }
                    else 
                    {
                        firstVisDir = totalDirs-numVisDirs;
                        selectedDir = numVisDirs-1;
                    }
                    refreshDirView();
                } 
                else 
                {
                    if(totalFiles <= numVisFiles) 
                    {
                        selectedFile = totalFiles-1;
                    }
                    else 
                    {
                        firstVisFile = totalFiles-numVisFiles;
                        selectedFile = numVisFiles-1;
                    }
                    refreshFileView();
                    fflush(stdout);
                }
                break;
            case('g'):
            case('G'):
                if(CTRL) 
                {
                    if(GNU_DOS_LEVEL < 3) break;
                    goto do_exit;
                } 
                else goto insert_char;
                break;
            case('d'):
            case('D'):
                if(CTRL) 
                {
                    if(GNU_DOS_LEVEL > 3) goto do_del;
                } 
                else goto insert_char;
                break;
            case('q'):
            case('Q'):
                if(CTRL) 
                {
                    if(GNU_DOS_LEVEL > 4) break;
do_exit:
                    i = msgBoxH("Are you sure you want to exit?", YES | NO, CONFIRM);
                    if(i == YES) exit_gracefully();
                    else refreshWindows();
                }
                else goto insert_char;
                break;
            case('c'):
            case('C'):
                if(CTRL) copyMarked();
                else goto insert_char;
                break;
            case('/'):
                if(CTRL && GNU_DOS_LEVEL > 4) 
                {
                    unMarkAll(activeWindow);
                }
                break;
            case('z'):
            case('Z'):
                if(CTRL) 
                {
                    if(GNU_DOS_LEVEL > 4) break;
                    unMarkAll(activeWindow);
                } 
                else goto insert_char;
                break;
            case('n'):
            case('N'):
                if(CTRL) 
                {
                    if(GNU_DOS_LEVEL > 1) goto do_down;
                    char *res = getUserInput("Enter directory name to create:",
                                             "New Directory");
                    hideCursor();
                    if(!res) break;
                    struct stat st;
                    if(stat(res, &st) == -1) 
                    {
                        if(mkdir(res, 0775) == -1)
                        {
                            showErrorMsgBox("Error creating directory:", res);
                        }
                    }
                    else 
                    {
                        msgBoxH("Directory already exists!", OK, ERROR);
                    }
                    free(res);
                    scanDir(cwd);
                    refreshFileView();
                    refreshDirView();
                }
                else goto insert_char;
                break;
                ////////////////////////////////////////////////////
            default:
                //browse to the first entry starting with the entered character
insert_char:
                if(ch >= 33 && ch <= 126) 
                {
                    int i, x = -1;
                    if(activeWindow == DIR_WIN) 
                    {
                        //search from this point to the end
                        for(i = firstVisDir+selectedDir+1; i < totalDirs; i++)
                        {
                            if(dirs[i]->name[0] == ch || dirs[i]->name[0] == ch-32)
                            {
                                x = i;
                                break;
                            }
                        }
                        //if the previous loop didn't find anything, try again
                        //starting from the top to the current point
                        if(i >= totalDirs)
                        {
                            for(i = 0; i <= firstVisDir+selectedDir; i++)
                            {
                                if(dirs[i]->name[0] == ch || dirs[i]->name[0] == ch-32)
                                {
                                    x = i;
                                    break;
                                }
                            }
                        }
                        //check to see if we found any result
                        if(x >= 0)
                        {
                            selectedDir = x-firstVisDir;
                            if(totalDirs <= numVisDirs)
                            {
                                refreshDirView();
                                continue;
                            }
                            if(selectedDir < 0)
                            {
                                firstVisDir += selectedDir;
                                selectedDir = 0;
                            }
                            else if(selectedDir >= numVisDirs) 
                            { 
                                firstVisDir += selectedDir-numVisDirs+1;
                                selectedDir = numVisDirs-1;
                            }
                            if(totalDirs-firstVisDir < numVisDirs) 
                            { 
                                selectedDir = firstVisDir;
                                firstVisDir = totalDirs-numVisDirs; 
                                selectedDir -= firstVisDir; 
                            } 
                            refreshDirView();
                        }
                        //////////////////////////////////////////////////
                    } 
                    else if(activeWindow == FILE_WIN) 
                    {
                        //search from this point to the end
                        for(i = firstVisFile+selectedFile+1; i < totalFiles; i++)
                        {
                            if(files[i]->name[0] == ch || files[i]->name[0] == ch-32)
                            {
                                x = i;
                                break;
                            }
                        }
                        //if the previous loop didn't find anything, try again
                        //starting from the top to the current point
                        if(i >= totalFiles)
                        {
                            for(i = 0; i <= firstVisFile+selectedFile; i++)
                            {
                                if(files[i]->name[0] == ch || files[i]->name[0] == ch-32)
                                {
                                    x = i;
                                    break;
                                }
                            }
                        }
                        //check to see if we found any result
                        if(x >= 0) 
                        {
                            selectedFile = x-firstVisFile;
                            if(totalFiles <= numVisFiles)
                            {
                                refreshFileView();
                                continue;
                            }
                            if(selectedFile < 0)
                            {
                                firstVisFile += selectedFile;
                                selectedFile = 0;
                            }
                            else if(selectedFile >= numVisFiles)
                            { 
                                firstVisFile += selectedFile-numVisFiles+1;
                                selectedFile = numVisFiles-1;
                            } 
                            if(totalFiles-firstVisFile < numVisFiles) 
                            { 
                                selectedFile = firstVisFile;
                                firstVisFile = totalFiles-numVisFiles; 
                                selectedFile -= firstVisFile;
                            } 
                            refreshFileView();
                        }
                    }
                }
                break;
        }
        setScreenColorsI(COLOR_WINDOW);
    }
    
    exit_gracefully();
    return 0;
}


void toggleSelected() 
{
    setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
    int pos;
    switch(activeWindow) 
    {
        case(DIR_WIN):
            pos = firstVisDir+selectedDir;
            if((strcmp(dirs[pos]->name, ".") == 0)//ignore '.' and '..'
                || (strcmp(dirs[pos]->name, "..") == 0)) break;
            
            if(dirs[pos]->star == '*')
            {	//if selected, un-select
                dirs[pos]->star = ' ';
                numStarred--;
            } 
            else 
            {	//otherwise, select it
                if(dirs[pos]->star == '^') 
                {
                    numCut--;	//if marked for cut, remove cut
                    removeCutDir(pos);
                } 
                else if(dirs[pos]->star == '#') 
                {
                    numCopy--;	//if marked for copy, remove copy
                    removeCopyDir(pos);
                }
                dirs[pos]->star = '*';
                numStarred++;
            }
            fprintf(stdout, "\e[%d;%dH", selectedDir+4, 3);
            fprintf(stdout, "%c", dirs[pos]->star);
            refreshBottomView();
            break;
        case(FILE_WIN):
            pos = firstVisFile+selectedFile;
            if(files[pos]->type == '%') break;
            if(files[pos]->star == '*') 
            {   //if selected, unselect
                files[pos]->star = ' ';
                numStarred--;
            } 
            else 
            {	//otherwise, select it
                if(files[pos]->star == '^') 
                {
                    numCut--; //if marked for cut, remove cut
                    removeCutFile(pos);
                } 
                else if(files[pos]->star == '#') 
                {
                    numCopy--; //if marked for cut, remove cut
                    removeCopyFile(pos);
                }
                files[pos]->star = '*';
                numStarred++;
            }
            fprintf(stdout, "\e[%d;%dH", selectedFile+4, (SCREEN_W/2)+1);
            fprintf(stdout, "%c", files[pos]->star);
            refreshBottomView();
            break;
    }
}

int addDirEntry(int pos, char *name)
{
    int len = strlen(name);
    if(!dirs[pos])
    {
        dirs[pos] = allocDirStructB(len+1);
        if(!dirs[pos]) return 0;
    }
    else
    {
        free(dirs[pos]->name);
        dirs[pos]->name = (char *)malloc(len+1);
        if(!dirs[pos]->name) return 0;
    }
    strcpy(dirs[pos]->name, name);
    dirs[pos]->namelen = len;
    return 1;
}

int addFileEntry(int pos, char *name)
{
    int len = strlen(name);
    if(!files[pos])
    {
        files[pos] = allocFileStructB(len+1);
        if(!files[pos]) return 0;
    }
    else
    {
        if(files[pos]->type != '%') free(files[pos]->name);
        files[pos]->name = (char *)malloc(len+1);
        if(!files[pos]->name) return 0;
    }
    strcpy(files[pos]->name, name);
    files[pos]->namelen = len;
    return 1;
}

void scanDir(char *dir) 
{
    int dcount = 0, fcount = 0;
    int n, j;
    struct dirent **eps;
    struct stat statbuf;

    n = scandir(dir, &eps, one, alphasort);
    //msgBox(dir, OK, INFO);
    if(strcmp(dir, ".") && strcmp(dir, cwd))
    {
        j = chdir(dir);
        if(j == -1)
        {
            showErrorMsgBox("Error changing directory:", dir);
            goto finish;
        }

        if(cwd) free(cwd);
        cwd = getcwd(NULL, 0);
        cwdlen = strlen(cwd);
    }
    
    if(n >= 0) 
    {
        int cnt;
        for(cnt = 0; cnt < n; ++cnt) 
        {
            char *d = eps[cnt]->d_name;
            j = lstat(d, &statbuf);
            if(j == -1)
            {
                char tmp[strlen(d)+2];
                strcpy(tmp, ": ");
                strcat(tmp, d);
                showErrorMsgBox(strerror(errno), tmp);
                goto finish;
            }
      
            if(S_ISDIR(statbuf.st_mode)) 
            {
                if(dcount >= MAXENTRIES) break;
                if(!addDirEntry(dcount, d))
                {
                    msgBoxH("Insufficient memory.", OK, ERROR);
                    goto finish;
                }
                if(strcmp(d, "..") == 0 || strcmp(d, ".") == 0) j = 0;
                else j = checkCutOrCopyDir(dcount);
                
                switch(j)
                {
                    case 1 : dirs[dcount]->star = '^'; break;
                    case 2 : dirs[dcount]->star = '#'; break;
                    default: dirs[dcount]->star = ' '; break;
                }
                
                //check if it is a hidden dir or not
                if(dirs[dcount]->name[0] == '.') dirs[dcount]->type = 'h';
                else dirs[dcount]->type = 'd';
                dcount++;
            }
            else 
            {
                if(fcount >= MAXENTRIES) break;
                if(!addFileEntry(fcount, d))
                {
                    msgBoxH("Insufficient memory.", OK, ERROR);
                    goto finish;
                }
                j = checkCutOrCopyFile(fcount);

                switch(j)
                {
                    case 1 : files[fcount]->star = '^'; break;
                    case 2 : files[fcount]->star = '#'; break;
                    default: files[fcount]->star = ' '; break;
                }
                //check to see the file type
                if(d[0] == '.')	//is it hidden?
                    files[fcount]->type = 'h';
                else if(S_ISLNK(statbuf.st_mode))	//is it a link?
                    files[fcount]->type = 'l';
                else if((statbuf.st_mode & S_IXUSR) || (statbuf.st_mode & S_IXGRP) ||
                        (statbuf.st_mode & S_IXOTH)) //is executable?
                    files[fcount]->type = 'x';
                else
                {
                    char *ext = strrchr(d, '.');
                    if(!ext) files[fcount]->type = 'r';
                    else
                    {
                        //is it an archive?
                        if(strcasecmp(ext, ".tar") == 0 || strcasecmp(ext, ".gz") == 0 ||
                            strcasecmp(ext, ".xz"  ) == 0 || strcasecmp(ext, ".Z"  ) == 0 ||
                            strcasecmp(ext, ".rar" ) == 0 || strcasecmp(ext, ".zip") == 0 ||
                            strcasecmp(ext, ".bz2" ) == 0 || strcasecmp(ext, ".7z" ) == 0 ||
                            strcasecmp(ext, ".lzma") == 0 || strcasecmp(ext, ".lha") == 0 ||
                            strcasecmp(ext, ".jar" ) == 0)
                            files[fcount]->type = 'a';
                        //is it a picture?
                        else if(strcasecmp(ext, ".bmp") == 0 || strcasecmp(ext, ".png") == 0 ||
                            strcasecmp(ext, ".jpg") == 0 || strcasecmp(ext, ".jpeg") == 0 ||
                            strcasecmp(ext, ".pcx") == 0 || strcasecmp(ext, ".ico" ) == 0 ||
                            strcasecmp(ext, ".gif") == 0 || strcasecmp(ext, ".tiff") == 0)
                            files[fcount]->type = 'p';
                        else	//just a regular file
                            files[fcount]->type = 'r';
                    }
                }
                fcount++;
            }
            free(eps[cnt]);
        }
        free(eps);
    }
    else 
    {
        showErrorMsgBox("Failed to open directory:", strerror(errno));
        goto finish;
    }

    freeFileStructs(fcount, totalFiles);
    totalFiles = fcount;
    if(totalFiles < numVisFiles)
        files[totalFiles] = NULL;	//so the refreshFileView() will work properly
    firstVisFile = 0;
    selectedFile = 0;
    
    freeDirStructs(dcount, totalDirs);
    totalDirs = dcount;
    if(totalDirs < numVisDirs) 
        dirs[totalDirs  ] = NULL;	//so the refreshDirView() will work properly
    firstVisDir = 0;
    selectedDir = 0;
    
    numStarred = 0;
  
finish:
    refreshAll();
}



void refreshAll()
{
    setScreenColorsI(COLOR_WINDOW);
    drawBox(1, 1, SCREEN_H, SCREEN_W, " Prime File Manager ", NO);
    drawMenuBar(2, 2, SCREEN_W-2);
    refreshFileView();
    refreshDirView();
    refreshBottomView();
}

void refreshWindows()
{
    setScreenColorsI(COLOR_WINDOW);
    refreshFileView();
    refreshDirView();
}

/***************************************
 * refreshDirView(): 
 * Procedure to refresh the left window
 * showing directory tree.
 * **************************************/
void refreshDirView() 
{
    if(activeWindow == DIR_WIN) setScreenColors(GREEN, BG_COLOR[COLOR_WINDOW]);
    //draw left sub-window
    drawBox(3, 2, SCREEN_H-5, (int)(SCREEN_W/2)-1, " Directory view ", YES);
    int i, j, k;
    if(totalDirs == 0) 
    {
        fprintf(stdout, "\x1b[%d;%dH%s", 4, 3, dirHighLight);
	fprintf(stdout, "\x1b[%d;%dH No directory entries found!", 4, 3);
        return;
    }
    for(i = 0; i < numVisDirs; i++) 
    {
        k = firstVisDir+i;
        if(dirs[k] == NULL) break; 
        if(i == selectedDir && activeWindow == DIR_WIN)
            setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
        else
            setScreenColors(FILE_DIR_COLOR[DIRCOLOR(k)], BG_COLOR[COLOR_WINDOW]);
        print_dir_highlight(i);
    }

    setScreenColorsI(COLOR_WINDOW);
    //redraw main window but don't clear the area
    drawBox(1, 1, SCREEN_H, SCREEN_W, " Prime File Manager ", NO);
}


/***************************************
 * refreshFileView(): 
 * Procedure to refresh the right window
 * showing file entries.
 * **************************************/
void refreshFileView() 
{
    if(activeWindow == FILE_WIN) setScreenColors(GREEN, BG_COLOR[COLOR_WINDOW]);
    //draw right sub-window
    drawBox(3, (int)(SCREEN_W/2), SCREEN_H-5, SCREEN_W-1, " File view ", YES);
    int i, j, k;
    if(totalFiles == 0) 
    {
        addFileEntry(0, "(Empty folder)");
        files[0]->type = '%';
        files[1] = NULL;
        totalFiles = 1;
    }
    for(i = 0; i < numVisFiles; i++) 
    {
        k = firstVisDir+i;
        if(files[k] == NULL) break; 
        if(i == selectedFile && activeWindow == FILE_WIN)
            setScreenColorsI(COLOR_HIGHLIGHT_TEXT);
        else
            setScreenColors(FILE_DIR_COLOR[FILECOLOR(k)], BG_COLOR[COLOR_WINDOW]);
        print_file_highlight(i);
    }
 
    setScreenColorsI(COLOR_WINDOW);
    //redraw main window but don't clear the area
    drawBox(1, 1, SCREEN_H, SCREEN_W, " Prime File Manager ", NO);
}

/***************************************
 * refreshBottomView(): 
 * Procedure to refresh the bottom window
 * showing CWD and statistics.
 * *************************************/
void refreshBottomView() 
{
    //draw bottom sub-window
    setScreenColorsI(COLOR_WINDOW);
    drawBox(SCREEN_H-4, 2, SCREEN_H-1, SCREEN_W-1, " Quick reference ", YES);
    fprintf(stdout, "\e[%d;%dH", SCREEN_H-2, 4);
    if(cwd != NULL) 
    {	//check the current working directory var is not NULL pointer
        if(strlen(cwd) > SCREEN_W-7) 
        {	//check if cwd length is more than display width
            int i;
            fprintf(stdout, "CWD: ");
            for(i = 0; i < SCREEN_W-12; i++) 
                putchar(cwd[i]);	//show just enought chars of the string
            fprintf(stdout, "..");			//and seal it with '..'
        }
        else 
        { 				//the length is less than display width
            fprintf(stdout, "CWD: %s", cwd); 	//so spit it all out
        }
    }
    fprintf(stdout, "\e[%d;%dH", SCREEN_H-3, 4);
    if(numStarred > 0) fprintf(stdout, "Marked (%d) ", numStarred);
    if(numCut     > 0) fprintf(stdout, "Cut (%d) ", numCut);
    if(numCopy    > 0) fprintf(stdout, "Copy (%d) ", numCopy);
    //redraw main window but don't clear the area
    drawBox(1, 1, SCREEN_H, SCREEN_W, " Prime File Manager ", NO);
}

/***************************************
 * drawMenuBar(): 
 * Procedure to draw the main menu bar.
 * **************************************/
void drawMenuBar(int x, int y, int w) 
{
    printf("\x1b[0m");
    setScreenColorsI(COLOR_MENU_BAR);
    fprintf(stdout, "\x1b[%d;%dH", x, y);		//reposition the cursor
    int i,j, lastChar = y;
    for(i = 0; i < w; i++) fputc(' ', stdout);	//Draw empty menu bar
    fprintf(stdout, "\x1b[%d;%dH", x, y);		//reposition the cursor

    for(i = 0; i < totalMainMenus; i++) 
    {
        j = 0; lastChar++;
        fprintf(stdout, " ");
        while(menu[i][j] != '\0') 
        {
            if(menu[i][j] == '&') 
            {	//turn on underline feature to print the shortcut key
                //fprintf(stdout, "\x1b[4m");
                fprintf(stdout, "%c", menu[i][j+1]);
                //fprintf(stdout, "\x1b[24m");//then turn it off
            }
            else
            {
                fprintf(stdout, "%c", menu[i][j+1]);	//print normal chars (other than the
            }
            lastChar++;
            j++;					//shortcut key)
        }
        fprintf(stdout, " ");
    }
    setScreenColorsI(COLOR_WINDOW);
}
