/*
 * Electric(tm) VLSI Design System
 *
 * File: netdiff.c
 * Network tool: module for network comparison
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 *
 * This module is inspired by the work of Carl Ebeling:
 *   Ebeling, Carl, "GeminiII: A Second Generation Layout Validation Program",
 *   Proceedings of ICCAD 1988, p322-325.
 */

#include "global.h"
#include "network.h"
#include "efunction.h"
#include "egraphics.h"
#include "edialogs.h"
#include "tecschem.h"
#include "usr.h"
#include <math.h>

#define MAXITERATIONS 10
#define SYMGROUPCOMP   0
#define SYMGROUPNET    1

/* the meaning of errors returned by "net_analyzesymmetrygroups()" */
#define SIZEERRORS       1
#define EXPORTERRORS     2
#define STRUCTUREERRORS  4

GRAPHICS net_cleardesc = {LAYERH, ALLOFF, SOLIDC, SOLIDC,
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};
static GRAPHICS net_msgdesc = {LAYERH, HIGHLIT, SOLIDC, SOLIDC,
	{0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF,0xFFFF}, NOVARIABLE, 0};

#define NOSYMGROUP ((SYMGROUP *)-1)

typedef struct Isymgroup
{
	INTHUGE           hashvalue;			/* hash value of this symmetry group */
	INTBIG            grouptype;			/* SYMGROUPCOMP (components) or SYMGROUPNET (nets) */
	INTBIG            groupindex;			/* ordinal value of group */
	INTBIG            facetcount[2];		/* number of objects from facets */
	INTBIG            facettotal[2];		/* size of object list from facets */
	void            **facetlist[2];			/* list of objects from facets */
	struct Isymgroup *nextsymgroup;			/* next in list */
} SYMGROUP;

static SYMGROUP  *net_firstsymgroup = NOSYMGROUP;
static SYMGROUP  *net_symgroupfree = NOSYMGROUP;

static SYMGROUP **net_symgrouphashcomp;			/* hash table for components */
static SYMGROUP **net_symgrouphashnet;			/* hash table for nets */
static INTBIG     net_symgrouphashcompsize = 0;	/* size of component hash table */
static INTBIG     net_symgrouphashnetsize = 0;	/* size of net hash table */
static INTBIG     net_symgrouplisttotal = 0;
static INTBIG     net_symgroupnumber;
static SYMGROUP **net_symgrouplist;
static INTBIG     net_nodeCountMultiplier = 0;
static INTBIG     net_portFactorMultiplier;
static INTBIG     net_portNetFactorMultiplier;
static INTBIG     net_portHashFactorMultiplier;
static INTBIG     net_functionMultiplier;
static PCOMP     *net_pcomp1 = NOPCOMP, *net_pcomp2 = NOPCOMP;
static PNET      *net_nodelist1 = NOPNET, *net_nodelist2 = NOPNET;
static NODEPROTO *net_facet[2];
static INTBIG     net_ncc_options;				/* options to use in NCC */
static INTBIG     net_ncc_tolerance;			/* component value tolerance */
static INTHUGE    net_uniquehashvalue;
static INTBIG     net_timestamp;

/* prototypes for local routines */
static void      net_addcomptoerror(void *err, PCOMP *pc);
static void      net_addnettoerror(void *err, PNET *pn);
static void      net_addsymgrouptoerror(void *err, SYMGROUP *sg);
static INTSML    net_addtosymgroup(SYMGROUP *sg, INTBIG facetno, void *obj);
static INTBIG    net_analyzesymmetrygroups(INTSML reporterrors, INTSML checksize,
					INTSML checkexportname, INTSML ignorepwrgnd);
static INTBIG    net_assignnewhashvalues(INTBIG grouptype);
static INTSML    net_componentequalvalue(float v1, float v2);
static char     *net_describecounts(INTBIG compcount, INTBIG netcount, INTBIG buscount);
static char     *net_describesizefactor(INTHUGE sizefactor);
static INTSML    net_dogemini(PCOMP *pcomp1, PNET *nodelist1, PCOMP *pcomp2, PNET *nodelist2,
					INTSML checksize, INTSML checkexportnames, INTSML ignorepwrgnd);
static void      net_dumpcomponents(PCOMP *pcomp1, PCOMP *pcomp2, INTSML ignorepwrgnd);
static void      net_dumpnetworks(PNET *nodelist1, PNET *nodelist2, INTSML ignorepwrgnd);
static INTSML    net_findamatch(INTBIG verbose, INTSML ignorepwrgnd);
static INTHUGE   net_findcommonsizefactor(SYMGROUP *sg);
static SYMGROUP *net_findgeomsymmetrygroup(GEOM *obj);
static SYMGROUP *net_findsymmetrygroup(INTBIG grouptype, INTHUGE hashvalue);
static void      net_forceamatch(SYMGROUP *sg, INTBIG i1, INTBIG i2, INTHUGE sizefactorsplit,
					INTBIG verbose, INTSML ignorepwrgnd);
static void      net_freesymgroup(SYMGROUP *sg);
static INTSML    net_getfacets(NODEPROTO**, NODEPROTO**);
static INTSML    net_insertinhashtable(SYMGROUP *sg);
static INTSML    net_isspice(PCOMP *pc);
static INTSML    net_nccalreadydone(NODEPROTO *facet1, NODEPROTO *facet2);
static INTSML    net_ncconelevel(NODEPROTO *facet1, NODEPROTO *facet2, INTSML preanalyze);
static SYMGROUP *net_newsymgroup(INTBIG type, INTHUGE hashvalue);
static void      net_rebuildhashtable(void);
static void      net_redeemzerogroups(SYMGROUP *sgnewc, SYMGROUP *sgnewn, INTBIG verbose, INTSML ignorepwrgnd);
static void      net_removefromsymgroup(SYMGROUP *sg, INTBIG f, INTBIG index);
static INTSML    net_sameexportnames(PNET *pn1, PNET *pn2);
static void      net_showsymmetrygroups(INTBIG verbose);
static INTHUGE   net_sizefactor(PCOMP *pc);
static int       net_sortpcomp(const void *e1, const void *e2);
static int       net_sortpnet(const void *e1, const void *e2);
static int       net_sortsymgroups(const void *e1, const void *e2);
static INTHUGE   net_uniquesymmetrygrouphash(INTBIG grouptype);
static void      net_unmatchedstatus(INTBIG *unmatchednets, INTBIG *unmatchedcomps, INTBIG *symgroupcount);

/*
 * Routine to free all memory associated with this module.
 */
void net_freediffmemory(void)
{
	REGISTER SYMGROUP *sg;
	REGISTER INTBIG f;

	net_removeassociations();
	while (net_firstsymgroup != NOSYMGROUP)
	{
		sg = net_firstsymgroup;
		net_firstsymgroup = sg->nextsymgroup;
		net_freesymgroup(sg);
	}
	if (net_symgrouphashcompsize != 0) efree((char *)net_symgrouphashcomp);
	if (net_symgrouphashnetsize != 0) efree((char *)net_symgrouphashnet);
	while (net_symgroupfree != NOSYMGROUP)
	{
		sg = net_symgroupfree;
		net_symgroupfree = sg->nextsymgroup;
		for(f=0; f<2; f++)
			if (sg->facettotal[f] > 0) efree((char *)sg->facetlist[f]);
		efree((char *)sg);
	}
	if (net_symgrouplisttotal > 0) efree((char *)net_symgrouplist);
}

void net_removeassociations(void)
{
	if (net_pcomp1 != NOPCOMP)
	{
		net_freeallpcomp(net_pcomp1);
		net_pcomp1 = NOPCOMP;
	}
	if (net_pcomp2 != NOPCOMP)
	{
		net_freeallpcomp(net_pcomp2);
		net_pcomp2 = NOPCOMP;
	}
	if (net_nodelist1 != NOPNET)
	{
		net_freeallpnet(net_nodelist1);
		net_nodelist1 = NOPNET;
	}
	if (net_nodelist2 != NOPNET)
	{
		net_freeallpnet(net_nodelist2);
		net_nodelist2 = NOPNET;
	}
}

/*
 * routine to identify the equivalent object associated with the currently
 * highlighted one (comparison must have been done).  If "noise" is nonzero,
 * report errors.  Returns zero if an equate was shown.
 */
INTSML net_equate(INTSML noise)
{
	REGISTER NODEPROTO *np;
	REGISTER PORTPROTO *pp;
	REGISTER ARCINST *ai;
	REGISTER NODEINST *ni;
	REGISTER NETWORK *net, *anet;
	REGISTER PCOMP *pc;
	REGISTER PNET *pn;
	REGISTER GEOM *obj;
	REGISTER SYMGROUP *sg;
	REGISTER INTBIG i, j, f, first, fun;

	/* make sure an association has been done */
	if (net_pcomp1 == NOPCOMP || net_pcomp2 == NOPCOMP)
	{
		if (noise != 0) ttyputerr(_("First associate with '-telltool network compare'"));
		return(1);
	}

	/* get the highlighted object */
	obj = (GEOM *)asktool(us_tool, "get-object");
	if (obj == NOGEOM)
	{
		if (noise != 0) ttyputerr(_("Must select something to be equated"));
		return(1);
	}

	/* make sure this object is in one of the associated facets */
	np = geomparent(obj);
	if (np != net_facet[0] && np != net_facet[1])
	{
		if (isachildof(np, net_facet[0]) == 0 && isachildof(np, net_facet[1]) == 0)
		{
			if (noise != 0)
				ttyputerr(_("This object is not in one of the two associated facets"));
			return(1);
		}
	}

	/* highlight the associated object */
	sg = net_findgeomsymmetrygroup(obj);
	if (sg == NOSYMGROUP)
	{
		ttyputmsg(_("This object is not associated with anything else"));
		return(1);
	}
	if (sg->hashvalue == 0)
	{
		ttyputmsg(_("This object was not matched successfully"));
		return(1);
	}

	(void)asktool(us_tool, "clear");
	(void)initinfstr();
	first = 0;
	switch (sg->grouptype)
	{
		case SYMGROUPCOMP:
			for(f=0; f<2; f++)
			{
				for(i=0; i<sg->facetcount[f]; i++)
				{
					pc = (PCOMP *)sg->facetlist[f][i];
					for(j=0; j<pc->numactual; j++)
					{
						if (pc->numactual == 1) ni = (NODEINST *)pc->actuallist; else
							ni = ((NODEINST **)pc->actuallist)[j];
						if (first != 0) (void)addtoinfstr('\n');
						first++;
						(void)formatinfstr("FACET=%s FROM=0%lo;-1;0",
							describenodeproto(geomparent(ni->geom)), (INTBIG)ni->geom);
					}
				}
			}
			break;
		case SYMGROUPNET:
			for(f=0; f<2; f++)
			{
				for(i=0; i<sg->facetcount[f]; i++)
				{
					pn = (PNET *)sg->facetlist[f][i];
					net = pn->network;
					if (net == NONETWORK) continue;
					np = net->parent;
					for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
					{
						anet = ai->network;
						if (ai->proto == sch_busarc)
						{
							if (anet->signals > 1)
							{
								for(i=0; i<anet->signals; i++)
									if (anet->networklist[i] == net) break;
								if (i >= anet->signals) continue;
							} else
							{
								if (anet != net) continue;
							}
						} else
						{
							if (anet != net) continue;
						}
						if (first != 0) (void)addtoinfstr('\n');
						first++;
						(void)formatinfstr("FACET=%s FROM=0%lo;-1;0",
							describenodeproto(np), (INTBIG)ai->geom);
					}
					for(ni = np->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
					{
						if (ni->proto->primindex == 0) continue;
						fun = nodefunction(ni);
						if (fun != NPPIN && fun != NPCONTACT && fun != NPNODE && fun != NPCONNECT)
							continue;
						if (ni->firstportarcinst == NOPORTARCINST) continue;
						if (ni->firstportarcinst->conarcinst->network != net) continue;
						if (first != 0) (void)addtoinfstr('\n');
						first++;
						(void)formatinfstr("FACET=%s FROM=0%lo;-1;0",
							describenodeproto(np), (INTBIG)ni->geom);
					}
					for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
					{
						if (pp->network != net) continue;
						if (first != 0) (void)addtoinfstr('\n');
						first++;
						(void)formatinfstr("FACET=%s TEXT=0%lo;0%lo;-",
							describenodeproto(np), (INTBIG)pp->subnodeinst->geom, (INTBIG)pp);
					}
				}
			}
	}
	(void)asktool(us_tool, "show-multiple", (INTBIG)returninfstr());
	return(0);
}

/*
 * routine to compare the two networks on the screen.  If "preanalyze" is
 * nonzero, only do preanalysis and display results.
 */
INTSML net_compare(INTSML preanalyze)
{
	NODEPROTO *facet1, *facet2;
	REGISTER INTSML ret;
	REGISTER VARIABLE *var;
	REGISTER NODEPROTO *np;
	REGISTER LIBRARY *lib;

	/* make sure network tool is on */
	if ((net_tool->toolstate&TOOLON) == 0)
	{
		ttyputerr(_("Network tool must be running...turning it on for you"));
		toolturnon(net_tool, 0);
		return(1);
	}

	if (net_getfacets(&facet1, &facet2) != 0)
	{
		ttyputerr(_("Must have two windows with two different facets"));
		return(1);
	}

	/* if the top facets are already checked, stop now */
	if (net_nccalreadydone(facet1, facet2) != 0)
	{
		ttyputmsg(_("Facets are already checked"));
		return(0);
	}

	if (preanalyze != 0) ttyputmsg(_("Analyzing...")); else
	{
		ttyputmsg(_("Comparing..."));
		starttimer();
	}

	var = getvalkey((INTBIG)net_tool, VTOOL, VINTEGER, net_ncc_comptolerancekey);
	if (var == NOVARIABLE) net_ncc_tolerance = 0; else net_ncc_tolerance = var->addr;

	/* mark all facets as not-checked */
	for(lib = el_curlib; lib != NOLIBRARY; lib = lib->nextlibrary)
		for(np = lib->firstnodeproto; np != NONODEPROTO; np = np->nextnodeproto)
			np->temp1 = -1;

	ret = net_ncconelevel(facet1, facet2, preanalyze);
	return(ret);
}

/******************************** COMPARISON ********************************/

INTSML net_ncconelevel(NODEPROTO *facet1, NODEPROTO *facet2, INTSML preanalyze)
{
	INTBIG comp1, comp2, power1, power2, ground1, ground2, netcount1, netcount2,
		unmatchednets, unmatchedcomps, prevunmatchednets, prevunmatchedcomps, errors,
		symgroupcount;
	REGISTER INTBIG i, f, ocomp1, ocomp2, verbose, buscount1, buscount2;
	INTSML hierarchical, ignorepwrgnd, mergeparallel, mergeserial, recurse,
		checkexportnames, localcheckexportnames1, localcheckexportnames2,
		checksize, localchecksize1, localchecksize2, localmergeserial1, localmergeserial2,
		localmergeparallel1, localmergeparallel2, subfacetsbad, ret;
	float elapsed;
	REGISTER char *errortype;
	REGISTER UINTBIG curtime;
	REGISTER NODEINST *ni;
	REGISTER NODEPROTO *subnp1, *subnp2;
	REGISTER NETWORK *net;
	REGISTER VARIABLE *var;
	REGISTER PCOMP *pc, *opc;
	REGISTER PNET *pn;
	REGISTER SYMGROUP *sg;

	/* make sure prime multipliers are computed */
	net_initdiff();

	/* stop if already checked */
	if (facet1->temp1 == 0 && facet2->temp1 == 0) return(0);
	if (facet1->temp1 >= 0 && facet2->temp1 >= 0) return(1);
	if (net_nccalreadydone(facet1, facet2) != 0)
	{
		facet1->temp1 = facet2->temp1 = 0;
		return(0);
	}

	/* get options to use during comparison */
	var = getvalkey((INTBIG)net_tool, VTOOL, VINTEGER, net_ncc_optionskey);
	if (var == NOVARIABLE) net_ncc_options = 0; else
		net_ncc_options = var->addr;
	verbose = net_ncc_options & (NCCVERBOSETEXT | NCCVERBOSEGRAPHICS);
	recurse = 0;
	if ((net_ncc_options&NCCHIERARCHICAL) != 0) hierarchical = 1; else
	{
		hierarchical = 0;
		if ((net_ncc_options&NCCRECURSE) != 0) recurse = 1;
	}
	if ((net_ncc_options&NCCIGNOREPWRGND) != 0) ignorepwrgnd = 1; else
		ignorepwrgnd = 0;
	if ((net_ncc_options&NCCNOMERGEPARALLEL) == 0) mergeparallel = 1; else
		mergeparallel = 0;
	if ((net_ncc_options&NCCMERGESERIAL) != 0) mergeserial = 1; else
		mergeserial = 0;
	if ((net_ncc_options&NCCCHECKEXPORTNAMES) != 0) checkexportnames = 1; else
		checkexportnames = 0;
	if ((net_ncc_options&NCCCHECKSIZE) != 0) checksize = 1; else
		checksize = 0;

	/* check for facet overrides */
	localmergeparallel1 = localmergeparallel2 = mergeparallel;
	localmergeserial1 = localmergeserial2 = mergeserial;
	localcheckexportnames1 = localcheckexportnames2 = checkexportnames;
	localchecksize1 = localchecksize2 = checksize;
	var = getvalkey((INTBIG)facet1, VNODEPROTO, VINTEGER, net_ncc_optionskey);
	if (var != NOVARIABLE)
	{
		if ((var->addr&NCCNOMERGEPARALLELOVER) != 0)
		{
			if ((var->addr&NCCNOMERGEPARALLEL) == 0) localmergeparallel1 = 1; else
				localmergeparallel1 = 0;
		}
		if ((var->addr&NCCMERGESERIALOVER) != 0)
		{
			if ((var->addr&NCCMERGESERIAL) != 0) localmergeserial1 = 1; else
				localmergeserial1 = 0;
		}
		if ((var->addr&NCCCHECKEXPORTNAMESOVER) != 0)
		{
			if ((var->addr&NCCCHECKEXPORTNAMES) != 0) localcheckexportnames1 = 1; else
				localcheckexportnames1 = 0;
		}
		if ((var->addr&NCCCHECKSIZEOVER) != 0)
		{
			if ((var->addr&NCCCHECKSIZE) != 0) localchecksize1 = 1; else
				localchecksize1 = 0;
		}
	}
	var = getvalkey((INTBIG)facet2, VNODEPROTO, VINTEGER, net_ncc_optionskey);
	if (var != NOVARIABLE)
	{
		if ((var->addr&NCCNOMERGEPARALLELOVER) != 0)
		{
			if ((var->addr&NCCNOMERGEPARALLEL) == 0) localmergeparallel2 = 1; else
				localmergeparallel2 = 0;
		}
		if ((var->addr&NCCMERGESERIALOVER) != 0)
		{
			if ((var->addr&NCCMERGESERIAL) != 0) localmergeserial2 = 1; else
				localmergeserial2 = 0;
		}
		if ((var->addr&NCCCHECKEXPORTNAMESOVER) != 0)
		{
			if ((var->addr&NCCCHECKEXPORTNAMES) != 0) localcheckexportnames2 = 1; else
				localcheckexportnames2 = 0;
		}
		if ((var->addr&NCCCHECKSIZEOVER) != 0)
		{
			if ((var->addr&NCCCHECKSIZE) != 0) localchecksize2 = 1; else
				localchecksize2 = 0;
		}
	}
	if (localmergeparallel1 != 0 || localmergeparallel2 != 0) mergeparallel = 1; else
		mergeparallel = 0;
	if (localmergeserial1 != 0 || localmergeserial2 != 0) mergeserial = 1; else
		mergeserial = 0;
	if (localcheckexportnames1 != 0 || localcheckexportnames2 != 0) checkexportnames = 1; else
		checkexportnames = 0;
	if (localchecksize1 != 0 || localchecksize2 != 0) checksize = 1; else
		if (localchecksize1 == 0 && localchecksize2 == 0 && checksize != 0) checksize = 0;

	/* if recursing, look at subfacets first */
	subfacetsbad = 0;
	if (recurse != 0)
	{
		for(ni = facet1->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			subnp1 = ni->proto;
			if (subnp1->primindex != 0) continue;

			/* ignore recursive references (showing icon in contents) */
			if (subnp1->cell == facet1->cell) continue;
			if (subnp1->cellview == el_iconview)
			{
				subnp1 = anyview(subnp1, facet1->cellview);
				if (subnp1 == NONODEPROTO) continue;
			}

			/* find equivalent to this in the other view */
			subnp2 = anyview(subnp1, facet2->cellview);
			if (subnp2 == NONODEPROTO)
			{
				ttyputerr(_("Cannot find %s view of facet %s"), facet2->cellview->viewname,
					describenodeproto(subnp1));
				continue;
			}
			ret = net_ncconelevel(subnp1, subnp2, preanalyze);
			if (ret < 0)
			{
				facet1->temp1 = facet2->temp1 = 1;
				return(ret);
			}
			if (ret > 0) subfacetsbad = 1;
		}
		for(ni = facet2->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
		{
			subnp2 = ni->proto;
			if (subnp2->primindex != 0) continue;

			/* ignore recursive references (showing icon in contents) */
			if (subnp2->cell == facet2->cell) continue;
			if (subnp2->cellview == el_iconview)
			{
				subnp2 = anyview(subnp2, facet2->cellview);
				if (subnp2 == NONODEPROTO) continue;
			}

			/* find equivalent to this in the other view */
			subnp1 = anyview(subnp2, facet1->cellview);
			if (subnp1 == NONODEPROTO)
			{
				ttyputerr(_("Cannot find %s view of facet %s"), facet1->cellview->viewname,
					describenodeproto(subnp2));
				continue;
			}
			ret = net_ncconelevel(subnp1, subnp2, preanalyze);
			if (ret < 0)
			{
				facet1->temp1 = facet2->temp1 = 1;
				return(ret);
			}
			if (ret > 0) subfacetsbad = 1;
		}
	}

	/* free any previous data structures */
	net_removeassociations();

	/* build network of pseudocomponents */
	net_pcomp1 = net_makepseudo(facet1, &comp1, &netcount1, &power1, &ground1,
		&net_nodelist1, hierarchical, mergeparallel, mergeserial, 1);
	if (net_pcomp1 == NOPCOMP)
	{
		facet1->temp1 = facet2->temp1 = 1;
		return(-1);
	}
	net_pcomp2 = net_makepseudo(facet2, &comp2, &netcount2, &power2, &ground2,
		&net_nodelist2, hierarchical, mergeparallel, mergeserial, 1);
	if (net_pcomp2 == NOPCOMP)
	{
		facet1->temp1 = facet2->temp1 = 1;
		return(-1);
	}
	net_facet[0] = facet1;   net_facet[1] = facet2;

	/* separate nets into plain and busses */
	buscount1 = 0;
	for(pn = net_nodelist1; pn != NOPNET; pn = pn->nextpnet)
	{
		net = pn->network;
		if (net == NONETWORK) continue;
		if (net->signals > 1) buscount1++;
	}
	netcount1 -= buscount1;
	buscount2 = 0;
	for(pn = net_nodelist2; pn != NOPNET; pn = pn->nextpnet)
	{
		net = pn->network;
		if (net == NONETWORK) continue;
		if (net->signals > 1) buscount2++;
	}
	netcount2 -= buscount2;

	/* announce what is happening */
	ttyputmsg(_("Comparing facet %s (%s) with facet %s (%s)"),
		describenodeproto(facet1), net_describecounts(comp1, netcount1, buscount1),
		describenodeproto(facet2), net_describecounts(comp2, netcount2, buscount2));

	(void)initinfstr();
	if ((net_ncc_options&NCCHIERARCHICAL) != 0) addstringtoinfstr(_("Flattening hierarchy")); else
	{
		if ((net_ncc_options&NCCRECURSE) != 0) addstringtoinfstr(_("Checking facets recursively")); else
			addstringtoinfstr(_("Checking this facet only"));
	}
	if (ignorepwrgnd != 0) addstringtoinfstr(_("; Ignoring Power and Ground nets")); else
		addstringtoinfstr(_("; Considering Power and Ground nets"));
	ttyputmsg("- %s", returninfstr());

	(void)initinfstr();
	if (mergeparallel == 0) addstringtoinfstr(_("Parallel components not merged")); else
		 addstringtoinfstr(_("Parallel components merged"));
	if (mergeserial == 0) addstringtoinfstr(_("; Serial transistors not merged")); else
		 addstringtoinfstr(_("; Serial transistors merged"));
	ttyputmsg("- %s", returninfstr());

	if (checkexportnames != 0 && checksize != 0)
	{
		ttyputmsg(_("- Checking export names and component sizes"));
	} else if (checkexportnames == 0 && checksize == 0)
	{
		ttyputmsg(_("- Ignoring export names and component sizes"));
	} else
	{
		if (checkexportnames != 0)
		{
			ttyputmsg(_("- Checking export names; Ignoring component sizes"));
		} else
		{
			ttyputmsg(_("- Ignoring export names; Checking component sizes"));
		}
	}

	if (mergeparallel == 0 && comp1 != comp2)
	{
		/* see if merging parallel components makes them match */
		ocomp1 = comp1;   ocomp2 = comp2;
		if (comp1 < comp2)
		{
			/* try merging parallel components in facet 2 */
			(void)net_mergeparallel(&net_pcomp2, &comp2);
			if (comp1 > comp2) (void)net_mergeparallel(&net_pcomp1, &comp1);
		} else
		{
			/* try merging parallel components in facet 1 */
			(void)net_mergeparallel(&net_pcomp1, &comp1);
			if (comp2 > comp1) (void)net_mergeparallel(&net_pcomp2, &comp2);
		}
		if (ocomp1 != comp1 || ocomp2 != comp2)
		{
			ttyputmsg(_("--- Merged parallel components, now have %d in facet %s and %d in facet %s"),
				comp1, describenodeproto(facet1), comp2, describenodeproto(facet2));
		}
	}

	if (ignorepwrgnd == 0)
	{
		if (power1 != power2 || ground1 != ground2)
		{
			ttyputmsg(_("Note: facet %s has %d power and %d ground net(s),"),
				describenodeproto(facet1), power1, ground1);
			ttyputmsg(_("      while facet %s has %d power and %d ground net(s)"),
				describenodeproto(facet2), power2, ground2);
		} else if (power1 > 1 || ground1 > 1)
			ttyputmsg(_("Note: there are %d power nets and %d ground nets"), power1, ground1);
	}

	/* build list of PNODEs and wires on each net */
	if (preanalyze != 0) verbose = 0;
	net_timestamp = 0;
	if (net_buildfullpnetdata(&net_pcomp1, &net_nodelist1, verbose) != 0) return(-1);
	if (net_buildfullpnetdata(&net_pcomp2, &net_nodelist2, verbose) != 0) return(-1);

	if (preanalyze != 0)
	{
		/* dump the networks and stop */
		net_dumpcomponents(net_pcomp1, net_pcomp2, ignorepwrgnd);
		net_dumpnetworks(net_nodelist1, net_nodelist2, ignorepwrgnd);
		return(0);
	}

	/* try to find network symmetry with existing switches */
	ret = net_dogemini(net_pcomp1, net_nodelist1, net_pcomp2, net_nodelist2,
		checksize, checkexportnames, ignorepwrgnd);
	if (ret < 0) return(-1);

	/* if match failed, see if unmerged parallel components are ambiguous */
	if (ret != 0 && mergeparallel == 0)
	{
		for(sg = net_firstsymgroup; sg != NOSYMGROUP; sg = sg->nextsymgroup)
		{
			if (sg->grouptype != SYMGROUPCOMP) continue;
			if (sg->facetcount[0] > 1 && sg->facetcount[1] > 1)
			for(f=0; f<2; f++)
			{
				for(i=1; i<sg->facetcount[f]; i++)
				{
					opc = sg->facetlist[f][i-1];
					pc = sg->facetlist[f][i];
					if (net_comparewirelist(pc, opc, 0) != 0) continue;
					mergeparallel = 1;
					break;
				}
				if (mergeparallel != 0) break;
			}
			if (mergeparallel != 0) break;
		}
		if (mergeparallel != 0)
		{
			/* might work if parallel components are merged */
			ttyputmsg(_("--- No match: trying again with parallel components merged"));
			net_unmatchedstatus(&prevunmatchednets, &prevunmatchedcomps, &symgroupcount);
			(void)net_mergeparallel(&net_pcomp1, &comp1);
			(void)net_mergeparallel(&net_pcomp2, &comp2);
			ret = net_dogemini(net_pcomp1, net_nodelist1, net_pcomp2, net_nodelist2,
				checksize, checkexportnames, ignorepwrgnd);
			if (ret < 0) return(-1);
			if (ret != 0)
			{
				net_unmatchedstatus(&unmatchednets, &unmatchedcomps, &symgroupcount);
				if (unmatchednets + unmatchedcomps < prevunmatchednets + prevunmatchedcomps)
				{
					/* this improved things but didn't solve them, use it */
					ttyputmsg(_("------ Merge of parallel components improved match"));
				} else if (unmatchednets + unmatchedcomps == prevunmatchednets + prevunmatchedcomps)
					ttyputmsg(_("------ Merge of parallel components make no change")); else
						ttyputmsg(_("------ Merge of parallel components make things worse"));
			}
		}
	}

	/* free reason information */
	if (verbose != 0)
	{
		for(pn = net_nodelist1; pn != NOPNET; pn = pn->nextpnet)
		{
			efree((char *)pn->hashreason);
			pn->hashreason = 0;
		}
		for(pn = net_nodelist2; pn != NOPNET; pn = pn->nextpnet)
		{
			efree((char *)pn->hashreason);
			pn->hashreason = 0;
		}
		for(pc = net_pcomp1; pc != NOPCOMP; pc = pc->nextpcomp)
		{
			efree((char *)pc->hashreason);
			pc->hashreason = 0;
		}
		for(pc = net_pcomp2; pc != NOPCOMP; pc = pc->nextpcomp)
		{
			efree((char *)pc->hashreason);
			pc->hashreason = 0;
		}
	}

	/* see if errors were found */
	errors = net_analyzesymmetrygroups(1, checksize, checkexportnames, ignorepwrgnd);

	/* write summary of NCC */
	elapsed = endtimer();
	if (errors != 0)
	{
		switch (errors)
		{
			case SIZEERRORS:
				errortype = N_("Size");                          break;
			case EXPORTERRORS:
				errortype = N_("Export");                        break;
			case STRUCTUREERRORS:
				errortype = N_("Structural");                    break;
			case SIZEERRORS|EXPORTERRORS:
				errortype = N_("Size and Export");               break;
			case SIZEERRORS|STRUCTUREERRORS:
				errortype = N_("Size and Structural");           break;
			case EXPORTERRORS|STRUCTUREERRORS:
				errortype = N_("Export and Structural");         break;
			case SIZEERRORS|EXPORTERRORS|STRUCTUREERRORS:
				errortype = N_("Size, Export and Structural");   break;
		}
		ttyputmsg(_("******* %s differences have been found! (%s)"),
			errortype, explainduration(elapsed));
		ret = 1;
	} else
	{
		if (subfacetsbad == 0)
		{
			curtime = getcurrenttime();
			(void)setvalkey((INTBIG)facet1, VNODEPROTO, net_lastgoodncckey,
				(INTBIG)curtime, VINTEGER);
			(void)setvalkey((INTBIG)facet2, VNODEPROTO, net_lastgoodncckey,
				(INTBIG)curtime, VINTEGER);
			(void)setvalkey((INTBIG)facet1, VNODEPROTO, net_lastgoodnccfacetkey,
				(INTBIG)facet2, VNODEPROTO);
			(void)setvalkey((INTBIG)facet2, VNODEPROTO, net_lastgoodnccfacetkey,
				(INTBIG)facet1, VNODEPROTO);
		}
		ttyputmsg(_("Facets %s and %s are equivalent (%s)"), describenodeproto(net_facet[0]),
			describenodeproto(net_facet[1]), explainduration(elapsed));
		ret = subfacetsbad;
	}
	facet1->temp1 = facet2->temp1 = ret;
	return(ret);
}

/*
 * Routine to run the Gemini algorithm to match components/nets "pcomp1/pnet1" with
 * components/nets "pcomp2/pnet2".  Use "checksize" to check sizes,
 * "checkexportnames" to check port names, and "ignorepwrgnd" to ignore power and ground.
 * The value of "mergeparallel" indicates whether parallel components are merged.
 * The routine returns:
 *   -1  Hard error (memory allocation, etc.)
 *    0  Networks match
 *    1  No match, but association quiesced
 *    2  No match, and association did not quiesce
 */
INTSML net_dogemini(PCOMP *pcomp1, PNET *pnet1, PCOMP *pcomp2, PNET *pnet2,
	INTSML checksize, INTSML checkexportnames, INTSML ignorepwrgnd)
{
	REGISTER SYMGROUP *sgc, *sgn, *sgnz, *sg, *nextsg, *lastsg;
	REGISTER PCOMP *pc;
	REGISTER PNET *pn;
	REGISTER INTBIG i, j, f, changesc, changesn, verbose, redeemcount, unmatched,
		prevunmatched, prevsymgroupcount;
	INTBIG unmatchednets, unmatchedcomps, symgroupcount, errors;
	char prompt[30];

	verbose = net_ncc_options & (NCCVERBOSETEXT | NCCVERBOSEGRAPHICS);

	/* clear old symmetry group list */
	while (net_firstsymgroup != NOSYMGROUP)
	{
		sg = net_firstsymgroup;
		net_firstsymgroup = sg->nextsymgroup;
		net_freesymgroup(sg);
	}
	if (net_symgrouphashcompsize != 0) efree((char *)net_symgrouphashcomp);
	if (net_symgrouphashnetsize != 0) efree((char *)net_symgrouphashnet);
	net_uniquehashvalue = -1;
	net_symgroupnumber = 1;

	/* determine size of hash tables */
	net_symgrouphashnetsize = net_symgrouphashcompsize = 0;
	for(pc = pcomp1; pc != NOPCOMP; pc = pc->nextpcomp) net_symgrouphashcompsize++;
	for(pc = pcomp2; pc != NOPCOMP; pc = pc->nextpcomp) net_symgrouphashcompsize++;
	for(pn = pnet1; pn != NOPNET; pn = pn->nextpnet) net_symgrouphashnetsize++;
	for(pn = pnet2; pn != NOPNET; pn = pn->nextpnet) net_symgrouphashnetsize++;
	net_symgrouphashcompsize = pickprime(net_symgrouphashcompsize * 2);
	net_symgrouphashnetsize = pickprime(net_symgrouphashnetsize * 2);
	net_symgrouphashcomp = (SYMGROUP **)emalloc(net_symgrouphashcompsize * (sizeof (SYMGROUP *)),
		net_tool->cluster);
	if (net_symgrouphashcomp == 0) return(-1);
	net_symgrouphashnet = (SYMGROUP **)emalloc(net_symgrouphashnetsize * (sizeof (SYMGROUP *)),
		net_tool->cluster);
	if (net_symgrouphashnet == 0) return(-1);
	for(i=0; i<net_symgrouphashcompsize; i++) net_symgrouphashcomp[i] = NOSYMGROUP;
	for(i=0; i<net_symgrouphashnetsize; i++) net_symgrouphashnet[i] = NOSYMGROUP;

	/* reset hash explanations */
	if (verbose != 0)
	{
		for(pc = pcomp1; pc != NOPCOMP; pc = pc->nextpcomp)
			(void)reallocstring(&pc->hashreason, "initial", net_tool->cluster);
		for(pc = pcomp2; pc != NOPCOMP; pc = pc->nextpcomp)
			(void)reallocstring(&pc->hashreason, "initial", net_tool->cluster);
		for(pn = pnet1; pn != NOPNET; pn = pn->nextpnet)
			(void)reallocstring(&pn->hashreason, "initial", net_tool->cluster);
		for(pn = pnet2; pn != NOPNET; pn = pn->nextpnet)
			(void)reallocstring(&pn->hashreason, "initial", net_tool->cluster);
	}

	/* new time stamp for initial entry into symmetry groups */
	net_timestamp++;

	/* initially assign all components to the same symmetry group (ignore SPICE) */
	sgc = net_newsymgroup(SYMGROUPCOMP, 1);
	if (sgc == NOSYMGROUP) return(-1);
	for(pc = pcomp1; pc != NOPCOMP; pc = pc->nextpcomp)
	{
		pc->hashvalue = sgc->hashvalue;
		if (net_addtosymgroup(sgc, 0, (void *)pc) != 0) return(-1);
	}
	for(pc = pcomp2; pc != NOPCOMP; pc = pc->nextpcomp)
	{
		pc->hashvalue = sgc->hashvalue;
		if (net_addtosymgroup(sgc, 1, (void *)pc) != 0) return(-1);
	}

	/* initially assign all nets to the same symmetry group (with ignored pwr/gnd in zero group) */
	sgn = net_newsymgroup(SYMGROUPNET, 1);
	if (sgn == NOSYMGROUP) return(-1);

	sgnz = net_newsymgroup(SYMGROUPNET, 0);
	if (sgnz == NOSYMGROUP) return(-1);
	for(pn = pnet1; pn != NOPNET; pn = pn->nextpnet)
	{
		if (ignorepwrgnd != 0 && (pn->flags&(POWERNET|GROUNDNET)) != 0)
		{
			pn->hashvalue = sgnz->hashvalue;
			if (net_addtosymgroup(sgnz, 0, (void *)pn) != 0) return(-1);
		} else
		{
			pn->hashvalue = sgn->hashvalue;
			if (net_addtosymgroup(sgn, 0, (void *)pn) != 0) return(-1);
		}
	}
	for(pn = pnet2; pn != NOPNET; pn = pn->nextpnet)
	{
		if (ignorepwrgnd != 0 && (pn->flags&(POWERNET|GROUNDNET)) != 0)
		{
			pn->hashvalue = sgnz->hashvalue;
			if (net_addtosymgroup(sgnz, 0, (void *)pn) != 0) return(-1);
		} else
		{
			pn->hashvalue = sgn->hashvalue;
			if (net_addtosymgroup(sgn, 1, (void *)pn) != 0) return(-1);
		}
	}

	/* now iteratively refine the symmetry groups */
	net_unmatchedstatus(&unmatchednets, &unmatchedcomps, &symgroupcount);
	prevsymgroupcount = symgroupcount;
	prevunmatched = unmatchednets + unmatchedcomps;
	redeemcount = 1;
	for(i=0; i<MAXITERATIONS; i++)
	{
		if (stopping(STOPREASONNCC) != 0) break;

		/* after first pass, assign random hash values to each symmetry group */
		if (i > 0)
		{
			lastsg = NOSYMGROUP;
			for(sg = net_firstsymgroup; sg != NOSYMGROUP; sg = nextsg)
			{
				nextsg = sg->nextsymgroup;
				if (sg->facetcount[0] == 0 && sg->facetcount[1] == 0)
				{
					if (lastsg == NOSYMGROUP) net_firstsymgroup = sg->nextsymgroup; else
						lastsg->nextsymgroup = sg->nextsymgroup;
					net_freesymgroup(sg);
					continue;
				}
				if (sg->hashvalue != 0)
				{
					sg->hashvalue = (INTHUGE)rand();
					if (sg->grouptype == SYMGROUPCOMP)
					{
						for(f=0; f<2; f++)
						{
							for(j=0; j<sg->facetcount[f]; j++)
							{
								pc = (PCOMP *)sg->facetlist[f][j];
								pc->hashvalue = sg->hashvalue;
							}
						}
					} else
					{
						for(f=0; f<2; f++)
						{
							for(j=0; j<sg->facetcount[f]; j++)
							{
								pn = (PNET *)sg->facetlist[f][j];
								pn->hashvalue = sg->hashvalue;
							}
						}
					}
				}
				lastsg = sg;
			}
			net_rebuildhashtable();
		}

		/* new time stamp for entry into symmetry groups */
		net_timestamp++;

		/* assign new hash values to components */
		changesc = net_assignnewhashvalues(SYMGROUPCOMP);
		if (changesc < 0) break;

		/* show the state of the world if requested */
		if (verbose != 0)
		{
			net_showsymmetrygroups(verbose);
			sprintf(prompt, "%ld changed, %ld symmetry groups:", changesc, symgroupcount);
			if (*ttygetlinemessages(prompt) != 0) break;
		}

		/* assign new hash values to nets */
		changesn = net_assignnewhashvalues(SYMGROUPNET);
		if (changesn < 0) break;

		/* show the state of the world if requested */
		if (verbose != 0)
		{
			net_showsymmetrygroups(verbose);
			sprintf(prompt, "%ld changed, %ld symmetry groups:", changesn, symgroupcount);
			if (*ttygetlinemessages(prompt) != 0) break;
		}

		/* if things are still improving, keep on */
		net_unmatchedstatus(&unmatchednets, &unmatchedcomps, &symgroupcount);
		unmatched = unmatchednets + unmatchedcomps;
		if (unmatched < prevunmatched)
		{
			prevunmatched = unmatched;
			i--;
			prevsymgroupcount = symgroupcount;
			continue;
		}

		/* if nothing changed or about to stop the loop, look for ambiguity */
		if (changesc + changesn == 0 || symgroupcount == prevsymgroupcount || i == MAXITERATIONS-1)
		{
			/* new time stamp for entry into symmetry groups */
			net_timestamp++;

			/* see if some incremental match can be applied */
			if (net_findamatch(verbose, ignorepwrgnd) != 0)
			{
				i = 0;
				prevsymgroupcount = symgroupcount;
				continue;
			}
#if 0		/* not redeeming abandoned groups yet */
			if (redeemcount > 0)
			{
				redeemcount--;
				ttyputmsg(_("--- Redeeming abandoned groups and trying again"));
				net_redeemzerogroups(NOSYMGROUP, NOSYMGROUP, verbose, ignorepwrgnd);
			}
#endif
			break;
		}
		prevsymgroupcount = symgroupcount;
	}

	/* see if errors were found */
	errors = net_analyzesymmetrygroups(0, checksize, checkexportnames, ignorepwrgnd);
	if (errors == 0) return(0);
	if (changesc + changesn != 0) return(2);
	return(1);
}

/*
 * Routine to return a hash code for component "pc".
 */
INTHUGE net_getcomphash(PCOMP *pc, INTBIG verbose)
{
	REGISTER INTBIG function, i, portfactor;
	REGISTER INTHUGE hashvalue;
	REGISTER NODEINST *ni;
	REGISTER NETWORK *net;
	REGISTER PNET *pn;
	REGISTER PORTPROTO *pp;

	/* initialize the hash factor */
	hashvalue = 0;

	/* start the hash value with the node's function */
	if (pc->numactual == 1) ni = (NODEINST *)pc->actuallist; else
		ni = ((NODEINST **)pc->actuallist)[0];
	if (ni->proto->primindex == 0)
	{
		/* a facet instance: use the node's prototype address */
		function = (INTBIG)ni->proto->cell;
	} else
	{
		/* a primitive: use the node's function */
		function = pc->function;
	}
	hashvalue += function * net_functionMultiplier;

	if (verbose != 0)
	{
		(void)initinfstr();
		(void)formatinfstr("%d(fun)", function);
	}

	/* now add in all networks as a function of the port's network */
	for(i=0; i<pc->wirecount; i++)
	{
		pp = pc->portlist[i];
		pn = pc->netnumbers[i];
		portfactor = pc->portindices[i];
		hashvalue += (portfactor * net_portNetFactorMultiplier) *
			(pn->hashvalue * net_portHashFactorMultiplier);
		if (verbose != 0)
		{
			net = pn->network;
			if (net == NONETWORK)
			{
				(void)formatinfstr(" + %d[%s]", portfactor, pp->protoname);
			} else
			{
				(void)formatinfstr(" + %d(%s)", portfactor, describenetwork(net));
			}
			(void)formatinfstr("x%s(hash)", hugeinttoa(pn->hashvalue));
		}
	}
	if (verbose != 0)
		(void)reallocstring(&pc->hashreason, returninfstr(), net_tool->cluster);
	return(hashvalue);
}

/*
 * Routine to return a hash code for network "pn".
 */
INTHUGE net_getnethash(PNET *pn, INTBIG verbose)
{
	REGISTER INTBIG i, index, portfactor, validcomponents;
	REGISTER INTHUGE hashvalue;
	REGISTER PCOMP *pc;

	/* initialize the hash factor */
	hashvalue = 0;

	/* start with the number of components on this net */
	hashvalue += (pn->nodecount+1) * net_nodeCountMultiplier;

	if (verbose != 0)
	{
		(void)initinfstr();
		(void)formatinfstr("%d(cnt)", pn->nodecount);
	}

	/* add in information for each component */
	validcomponents = 0;
	for(i=0; i<pn->nodecount; i++)
	{
		pc = pn->nodelist[i];
		index = pn->nodewire[i];
		portfactor = pc->portindices[index];
		hashvalue += portfactor * net_portFactorMultiplier * pc->hashvalue;
		if (pc->hashvalue != 0) validcomponents = 1;
		if (verbose != 0)
			(void)formatinfstr(" + %d(port)x%s(hash)", portfactor, hugeinttoa(pc->hashvalue));
	}

	/* if no components had valid hash values, make this net zero */
	if (validcomponents == 0 && pn->nodecount != 0) hashvalue = 0;
	if (verbose != 0)
		(void)reallocstring(&pn->hashreason, returninfstr(), net_tool->cluster);
	return(hashvalue);
}

/*
 * Routine to look at the symmetry groups and return the number of hard errors and the number of
 * soft errors in "harderrors", and "softerrors".  If "reporterrors" is nonzero, these errors are
 * logged for perusal by the user.  If "checksize" is nonzero, check sizes.  If "checkexportname" is
 * nonzero, check export names.  If "ignorepwrgnd" is nonzero, ignore power and ground nets.
 */
INTBIG net_analyzesymmetrygroups(INTSML reporterrors, INTSML checksize, INTSML checkexportname,
	INTSML ignorepwrgnd)
{
	REGISTER INTBIG f, i, j, of, oi, valid, errors, first;
	REGISTER SYMGROUP *sg;
	REGISTER PCOMP *pc1, *pc2, *pc, *opc;
	REGISTER void *err;
	REGISTER char *net1name, *errmsg, *segue;
	REGISTER PNET *pn, *pn1, *pn2, *opn;
	REGISTER PORTPROTO *pp1, *pp2;
	REGISTER NODEPROTO *par1, *par2;
	REGISTER NETWORK *net;

	if (reporterrors != 0) initerrorlogging(_("NCC"), net_facet[0], net_facet[1]);
	errors = 0;
	for(sg = net_firstsymgroup; sg != NOSYMGROUP; sg = sg->nextsymgroup)
	{
		if (sg->facetcount[0] == 1 && sg->facetcount[1] == 1)
		{
			if (checksize != 0 && sg->grouptype == SYMGROUPCOMP)
			{
				/* see if sizes match */
				pc1 = (PCOMP *)sg->facetlist[0][0];
				pc2 = (PCOMP *)sg->facetlist[1][0];
				if ((pc1->flags&COMPHASWIDLEN) != 0)
				{
					if (net_componentequalvalue(pc1->width, pc2->width) == 0 ||
						net_componentequalvalue(pc1->length, pc2->length) == 0)
					{
						if (reporterrors != 0)
						{
							(void)initinfstr();
							(void)formatinfstr(_("Node sizes differ (%s/%s vs. %s/%s)"),
								frtoa(roundfloat(pc1->length)), frtoa(roundfloat(pc1->width)),
									frtoa(roundfloat(pc2->length)), frtoa(roundfloat(pc2->width)));
							err = logerror(returninfstr(), NONODEPROTO, 0);
							net_addsymgrouptoerror(err, sg);
						}
						errors |= SIZEERRORS;
					}
				} else if ((pc1->flags&COMPHASAREA) != 0)
				{
					if (net_componentequalvalue(pc1->length, pc2->length) == 0)
					{
						if (reporterrors != 0)
						{
							(void)initinfstr();
							(void)formatinfstr(_("Node sizes differ (%s vs. %s)"),
								frtoa(roundfloat(pc1->length)), frtoa(roundfloat(pc2->length)));
							err = logerror(returninfstr(), NONODEPROTO, 0);
							net_addsymgrouptoerror(err, sg);
						}
						errors |= SIZEERRORS;
					}
				}
			}
			if (sg->grouptype == SYMGROUPNET)
			{
				/* see if names match */
				pn1 = (PNET *)sg->facetlist[0][0];
				pn2 = (PNET *)sg->facetlist[1][0];

				/* ignore name match for power and ground nets */
				if ((pn1->flags&(POWERNET|GROUNDNET)) == (pn1->flags&(POWERNET|GROUNDNET)))
				{
					if ((pn1->flags&(POWERNET|GROUNDNET)) != 0) continue;
				}

				if ((pn1->flags&EXPORTEDNET) != 0)
				{
					if ((pn2->flags&EXPORTEDNET) == 0)
					{
						if (checkexportname != 0)
						{
							/* net in facet 1 is exported, but net in facet 2 isn't */
							if (reporterrors != 0)
							{
								(void)initinfstr();
								(void)formatinfstr(_("Network in facet %s is '%s' but network in facet %s is not exported"),
									describenodeproto(net_facet[0]), net_describepnet(pn1), describenodeproto(net_facet[1]));
								err = logerror(returninfstr(), NONODEPROTO, 0);
								net_addsymgrouptoerror(err, sg);
							}
							errors |= EXPORTERRORS;
						}
					} else
					{
						/* both networks exported: check names */
						if (checkexportname != 0)
						{
							if (net_sameexportnames(pn1, pn2) == 0)
							{
								if (reporterrors != 0)
								{
									(void)initinfstr();
									(void)addstringtoinfstr(net_describepnet(pn1));
									net1name = returninfstr();
									(void)initinfstr();
									par1 = pn1->network->parent;
									par2 = pn2->network->parent;
									if ((par1 == net_facet[0] && par2 == net_facet[1]) ||
										(par1 == net_facet[1] && par2 == net_facet[0]) ||
										par1->cell == par2->cell)
									{
										(void)formatinfstr(_("Export names '%s:%s' and '%s:%s' do not match"),
											describenodeproto(par1), net1name,
												describenodeproto(par2), net_describepnet(pn2));
									} else
									{
										(void)formatinfstr(_("Export names '%s:%s' and '%s:%s' are not at the same level of hierarchy"),
											describenodeproto(par1), net1name,
												describenodeproto(par2), net_describepnet(pn2));
									}
									err = logerror(returninfstr(), NONODEPROTO, 0);
									net_addsymgrouptoerror(err, sg);
								}
								errors |= EXPORTERRORS;
							}
						}

						/* check that the export characteristics match */
						if (pn2->realportcount > 0)
						{
							for(i=0; i<pn1->realportcount; i++)
							{
								if (pn1->realportcount == 1) pp1 = (PORTPROTO *)pn1->realportlist; else
									pp1 = ((PORTPROTO **)pn1->realportlist)[i];
								for(j=0; j<pn2->realportcount; j++)
								{
									if (pn2->realportcount == 1) pp2 = (PORTPROTO *)pn2->realportlist; else
										pp2 = ((PORTPROTO **)pn2->realportlist)[j];
									if ((pp1->userbits&STATEBITS) == (pp2->userbits&STATEBITS)) break;
								}
								if (j < pn2->realportcount) continue;
								if (reporterrors != 0)
								{
									if (pn2->realportcount == 1) pp2 = (PORTPROTO *)pn2->realportlist; else
										pp2 = ((PORTPROTO **)pn2->realportlist)[0];
									(void)initinfstr();
									(void)formatinfstr(_("Exports have different characteristics ('%s' is %s and '%s' is %s)"),
										pp1->protoname, describeportbits(pp1), pp2->protoname, describeportbits(pp2));
									err = logerror(returninfstr(), NONODEPROTO, 0);
									net_addsymgrouptoerror(err, sg);
								}
								errors |= EXPORTERRORS;
							}
						}
					}
				} else
				{
					if ((pn2->flags&EXPORTEDNET) != 0)
					{
						if (checkexportname != 0)
						{
							/* net in facet 2 is exported, but net in facet 1 isn't */
							if (reporterrors != 0)
							{
								(void)initinfstr();
								(void)formatinfstr(_("Network in facet %s is '%s' but network in facet %s is not exported"),
									describenodeproto(net_facet[1]), net_describepnet(pn2), describenodeproto(net_facet[0]));
								err = logerror(returninfstr(), NONODEPROTO, 0);
								net_addsymgrouptoerror(err, sg);
							}
							errors |= EXPORTERRORS;
						}
					}
				}
			}
			continue;
		}
		if (sg->facetcount[0] == 0 && sg->facetcount[1] == 0) continue;

		errmsg = 0;
		if (sg->facetcount[0] <= 0 || sg->facetcount[1] <= 0 || sg->hashvalue == 0)
		{
			if (sg->grouptype == SYMGROUPNET)
			{
				/* network group: ignore if no real associated networks */
				valid = 0;
				for(f=0; f<2; f++)
				{
					for(i=0; i<sg->facetcount[f]; i++)
					{
						pn = (PNET *)sg->facetlist[f][i];
						if (pn->network != NONETWORK) valid++;
					}
				}
				if (valid == 0) continue;

				/* network group: ignore if a bus */
				for(f=0; f<2; f++)
				{
					for(i=0; i<sg->facetcount[f]; i++)
					{
						pn = (PNET *)sg->facetlist[f][i];
						net = pn->network;
						if (net == NONETWORK) continue;
						if (net->signals > 1) break;
					}
					if (i < sg->facetcount[f]) break;
				}
				if (f < 2) continue;

				/* network group: ignore if all power and ground that is being ignored */
				if (ignorepwrgnd != 0)
				{
					valid = 0;
					for(f=0; f<2; f++)
					{
						for(i=0; i<sg->facetcount[f]; i++)
						{
							pn = (PNET *)sg->facetlist[f][i];
							if ((pn->flags&(POWERNET|GROUNDNET)) == 0)
								valid++;
						}
					}
					if (valid == 0) continue;
				}
			}
			switch (sg->grouptype)
			{
				case SYMGROUPCOMP:
					errmsg = _("Unassociated nodes");
					break;
				case SYMGROUPNET:
					errmsg = _("Unassociated networks");
					break;
			}
		} else
		{
			switch (sg->grouptype)
			{
				case SYMGROUPCOMP:
					errmsg = _("Ambiguous nodes");
					break;
				case SYMGROUPNET:
					errmsg = _("Ambiguous networks");
					break;
			}
		}

		if (reporterrors != 0 && errmsg != 0)
		{
			switch (sg->grouptype)
			{
				case SYMGROUPCOMP:
					for(f=0; f<2; f++)
					{
						for(i=0; i<sg->facetcount[f]; i++)
						{
							pc = (PCOMP *)sg->facetlist[f][i];
							if (pc->timestamp <= 0) continue;
							err = logerror(errmsg, NONODEPROTO, 0);
							net_addcomptoerror(err, pc);
							for(of=0; of<2; of++)
							{
								for(oi=0; oi<sg->facetcount[of]; oi++)
								{
									opc = (PCOMP *)sg->facetlist[of][oi];
									if (opc == pc) continue;
									if (opc->timestamp != pc->timestamp) continue;
									net_addcomptoerror(err, opc);
									opc->timestamp = -opc->timestamp;
								}
							}
							pc->timestamp = -pc->timestamp;
						}
					}
					for(f=0; f<2; f++)
					{
						for(i=0; i<sg->facetcount[f]; i++)
						{
							pc = (PCOMP *)sg->facetlist[f][i];
							pc->timestamp = -pc->timestamp;
						}
					}
					break;
				case SYMGROUPNET:
					for(f=0; f<2; f++)
					{
						for(i=0; i<sg->facetcount[f]; i++)
						{
							pn = (PNET *)sg->facetlist[f][i];
							if (pn->timestamp <= 0) continue;
							if (ignorepwrgnd != 0 && (pn->flags&(POWERNET|GROUNDNET)) != 0)
								continue;

							/* build the error message */
							(void)initinfstr();
							(void)addstringtoinfstr(errmsg);
							segue = ": ";
							for(of=0; of<2; of++)
							{
								first = 1;
								for(oi=0; oi<sg->facetcount[of]; oi++)
								{
									opn = (PNET *)sg->facetlist[of][oi];
									if (opn->timestamp != pn->timestamp) continue;
									if (ignorepwrgnd != 0 && (opn->flags&(POWERNET|GROUNDNET)) != 0)
										continue;
									net = opn->network;
									if (net == NONETWORK) continue;
									if (first != 0)
									{
										(void)addstringtoinfstr(segue);
										segue = "; ";
										(void)formatinfstr(_("from facet %s:"),
											describenodeproto(net_facet[of]));
									} else (void)addtoinfstr(',');
									first = 0;
									(void)formatinfstr(" %s", describenetwork(net));
								}
							}

							/* report the error */
							err = logerror(returninfstr(), NONODEPROTO, 0);
							net_addnettoerror(err, pn);
							for(of=0; of<2; of++)
							{
								for(oi=0; oi<sg->facetcount[of]; oi++)
								{
									opn = (PNET *)sg->facetlist[of][oi];
									if (opn == pn) continue;
									if (opn->timestamp != pn->timestamp) continue;
									if (ignorepwrgnd != 0 && (opn->flags&(POWERNET|GROUNDNET)) != 0)
										continue;
									net_addnettoerror(err, opn);
									opn->timestamp = -opn->timestamp;
								}
							}
							pn->timestamp = -pn->timestamp;
						}
					}
					for(f=0; f<2; f++)
					{
						for(i=0; i<sg->facetcount[f]; i++)
						{
							pn = (PNET *)sg->facetlist[f][i];
							pn->timestamp = -pn->timestamp;
						}
					}
					break;
			}
		}
		errors |= STRUCTUREERRORS;
	}
	if (reporterrors != 0) termerrorlogging();
	return(errors);
}

/*
 * Routine to report the number of unmatched networks and components.
 */
void net_unmatchedstatus(INTBIG *unmatchednets, INTBIG *unmatchedcomps, INTBIG *symgroupcount)
{
	REGISTER INTBIG f;
	REGISTER SYMGROUP *sg;

	*unmatchednets = *unmatchedcomps = *symgroupcount = 0;
	for(sg = net_firstsymgroup; sg != NOSYMGROUP; sg = sg->nextsymgroup)
	{
		if (sg->facetcount[0] == 0 && sg->facetcount[1] == 0) continue;
		(*symgroupcount)++;
		if (sg->facetcount[0] == 1 && sg->facetcount[1] == 1) continue;
		for(f=0; f<2; f++)
		{
			if (sg->grouptype == SYMGROUPCOMP)
			{
				*unmatchedcomps += sg->facetcount[f];
			} else
			{
				*unmatchednets += sg->facetcount[f];
			}
		}
	}
}

/*
 * Routine to look for matches in ambiguous symmetry groups.
 * Returns nonzero if any are found.
 */
INTSML net_findamatch(INTBIG verbose, INTSML ignorepwrgnd)
{
	REGISTER SYMGROUP *sg, *osg;
	REGISTER PNET *pn0, *pn1, *pn;
	REGISTER PCOMP *pc0, *pc1, *pc;
	REGISTER PORTPROTO *pp0, *pp1;
	REGISTER NETWORK *net0, *net1;
	NETWORK *nets[2];
	REGISTER char *net0name, *net1name, *node0name, *node1name;
	REGISTER INTBIG i0, i1, ip0, ip1, i, f, u, any, total;
	INTBIG is[2], unmatchednets, unmatchedcomps, symgroupcount;
	char uniquename[30];
	REGISTER NODEINST *ni0, *ni1, *ni;
	REGISTER ARCINST *ai;
	NODEINST *nis[2];
	REGISTER VARIABLE *var;
	REGISTER INTHUGE sizefactorsplit;

	/* determine the number of unmatched nets and components */
	net_unmatchedstatus(&unmatchednets, &unmatchedcomps, &symgroupcount);

	/* prepare a list of symmetry groups, sorted by size of ambiguity */
	total = 0;
	for(sg = net_firstsymgroup; sg != NOSYMGROUP; sg = sg->nextsymgroup)
	{
		if (sg->facetcount[0] == 0 && sg->facetcount[1] == 0) continue;
		total++;
	}
	if (total > net_symgrouplisttotal)
	{
		if (net_symgrouplisttotal > 0) efree((char *)net_symgrouplist);
		net_symgrouplisttotal = 0;
		net_symgrouplist = (SYMGROUP **)emalloc(total * (sizeof (SYMGROUP *)), net_tool->cluster);
		if (net_symgrouplist == 0) return(0);
		net_symgrouplisttotal = total;
	}
	total = 0;
	for(sg = net_firstsymgroup; sg != NOSYMGROUP; sg = sg->nextsymgroup)
	{
		if (sg->facetcount[0] == 0 && sg->facetcount[1] == 0) continue;
		net_symgrouplist[total++] = sg;
	}
	esort(net_symgrouplist, total, sizeof (SYMGROUP *), net_sortsymgroups);

	/* now look through the groups, starting with the smallest */
	for(i=0; i<total; i++)
	{
		sg = net_symgrouplist[i];
		if (sg->hashvalue == 0) continue;
		if (sg->facetcount[0] < 2 || sg->facetcount[1] < 2) continue;

		/* look for export names that are the same */
		if (sg->grouptype == SYMGROUPNET)
		{
			for(i0=0; i0<sg->facetcount[0]; i0++)
			{
				pn0 = (PNET *)sg->facetlist[0][i0];
				if (pn0->realportcount == 0) continue;

				for(i1=0; i1<sg->facetcount[1]; i1++)
				{
					pn1 = (PNET *)sg->facetlist[1][i1];
					if (pn1->realportcount == 0) continue;

					for(ip0=0; ip0<pn0->realportcount; ip0++)
					{
						if (pn0->realportcount == 1) pp0 = (PORTPROTO *)pn0->realportlist; else
							pp0 = ((PORTPROTO **)pn0->realportlist)[ip0];
						for(ip1=0; ip1<pn1->realportcount; ip1++)
						{
							if (pn1->realportcount == 1) pp1 = (PORTPROTO *)pn1->realportlist; else
								pp1 = ((PORTPROTO **)pn1->realportlist)[ip1];
							if (namesame(pp0->protoname, pp1->protoname) == 0)
							{
								ttyputmsg(_("--- Forcing a match based on the export name '%s' (%ld symmetry groups with %ld nets and %ld nodes unmatched)"),
									pp0->protoname, total, unmatchednets, unmatchedcomps);
								net_forceamatch(sg, i0, i1, 0, verbose, ignorepwrgnd);
								return(1);
							}
						}
					}
				}
			}
		}

		/* look for nodes that are uniquely the same size */
		if (sg->grouptype == SYMGROUPCOMP)
		{
			sizefactorsplit = net_findcommonsizefactor(sg);
			if (sizefactorsplit != 0)
			{
				ttyputmsg(_("--- Forcing a match based on size %s nodes (%ld symmetry groups with %ld nets and %ld nodes unmatched)"),
					net_describesizefactor(sizefactorsplit), total, unmatchednets, unmatchedcomps);
				net_forceamatch(sg, 0, 0, sizefactorsplit, verbose, ignorepwrgnd);
				return(1);
			}
		}

		/* look for network names that are the same */
		if (sg->grouptype == SYMGROUPNET)
		{
			for(i0=0; i0<sg->facetcount[0]; i0++)
			{
				pn0 = (PNET *)sg->facetlist[0][i0];
				net0 = pn0->network;
				if (net0 == NONETWORK || net0->namecount == 0 || net0->tempname != 0)
					continue;

				for(i1=0; i1<sg->facetcount[1]; i1++)
				{
					pn1 = (PNET *)sg->facetlist[1][i1];
					net1 = pn1->network;
					if (net1 == NONETWORK || net1->namecount == 0 || net1->tempname != 0)
						continue;

					net0name = net0->netname;
					for(ip0=0; ip0<net0->namecount; ip0++)
					{
						net1name = net1->netname;
						for(ip1=0; ip1<net1->namecount; ip1++)
						{
							if (namesame(net0name, net1name) == 0)
							{
								ttyputmsg(_("--- Forcing a match based on the network name '%s' (%ld symmetry groups with %ld nets and %ld nodes unmatched)"),
									net0name, total, unmatchednets, unmatchedcomps);
								net_forceamatch(sg, i0, i1, 0, verbose, ignorepwrgnd);
								return(1);
							}
							net1name += strlen(net1name)+1;
						}
						net0name += strlen(net0name)+1;
					}
				}
			}
		}

		/* look for commonly named nodes */
		if (sg->grouptype == SYMGROUPCOMP)
		{
			for(i0=0; i0<sg->facetcount[0]; i0++)
			{
				pc0 = (PCOMP *)sg->facetlist[0][i0];
				if (pc0->numactual != 1) continue;
				ni0 = (NODEINST *)pc0->actuallist;
				var = getvalkey((INTBIG)ni0, VNODEINST, VSTRING, el_node_name);
				if (var == NOVARIABLE) continue;
				if ((var->type&VDISPLAY) == 0) continue;
				node0name = (char *)var->addr;

				for(i1=0; i1<sg->facetcount[1]; i1++)
				{
					pc1 = (PCOMP *)sg->facetlist[1][i1];
					if (pc1->numactual != 1) continue;
					ni1 = (NODEINST *)pc1->actuallist;
					var = getvalkey((INTBIG)ni1, VNODEINST, VSTRING, el_node_name);
					if (var == NOVARIABLE) continue;
					if ((var->type&VDISPLAY) == 0) continue;
					node1name = (char *)var->addr;

					if (namesame(node0name, node1name) == 0)
					{
						ttyputmsg(_("--- Forcing a match based on the nodes named '%s' (%ld symmetry groups with %ld nets and %ld nodes unmatched)"),
							node0name, total, unmatchednets, unmatchedcomps);
						net_forceamatch(sg, i0, i1, 0, verbose, ignorepwrgnd);
						return(1);
					}
				}
			}
		}
	}

	/* random match: look again through the groups, starting with the smallest */
	for(i=0; i<total; i++)
	{
		sg = net_symgrouplist[i];
		if (sg->hashvalue == 0) continue;
		if (sg->facetcount[0] < 2 || sg->facetcount[1] < 2) continue;

		if (sg->grouptype == SYMGROUPCOMP)
		{
			for(f=0; f<2; f++)
			{
				for(is[f]=0; is[f] < sg->facetcount[f]; is[f]++)
				{
					pc = (PCOMP *)sg->facetlist[f][is[f]];
					if (pc->numactual == 1) nis[f] = (NODEINST *)pc->actuallist; else
						nis[f] = ((NODEINST **)pc->actuallist)[0];
					var = getvalkey((INTBIG)nis[f], VNODEINST, VSTRING, el_node_name);
					if (var == NOVARIABLE) break;
					if ((var->type&VDISPLAY) == 0) break;
				}
				if (is[f] >= sg->facetcount[f])
				{
					is[f] = 0;
					pc = (PCOMP *)sg->facetlist[f][is[f]];
					if (pc->numactual == 1) nis[f] = (NODEINST *)pc->actuallist; else
						nis[f] = ((NODEINST **)pc->actuallist)[0];
				}
			}

			/* find a unique name and tag the selected nodes */
			for(u=1; ; u++)
			{
				sprintf(uniquename, "NCCmatch%ld", u);
				for(f=0; f<2; f++)
				{
					for(ni = net_facet[f]->firstnodeinst; ni != NONODEINST; ni = ni->nextnodeinst)
					{
						var = getvalkey((INTBIG)ni, VNODEINST, VSTRING, el_node_name);
						if (var == NOVARIABLE) continue;
						if (namesame((char *)var->addr, uniquename) == 0) break;
					}
					if (ni != NONODEINST) break;
				}
				if (f >= 2) break;
			}
			for(f=0; f<2; f++)
			{
				startobjectchange((INTBIG)nis[f], VNODEINST);
				var = setvalkey((INTBIG)nis[f], VNODEINST, el_node_name,
					(INTBIG)uniquename, VSTRING|VDISPLAY);
				if (var != NOVARIABLE)
					defaulttextsize(3, var->textdescript);
				endobjectchange((INTBIG)nis[f], VNODEINST);
			}
			ttyputmsg(_("--- Forcing a random match of nodes '%s:%s' and '%s:%s' (%ld symmetry groups with %ld nets and %ld nodes unmatched)"),
				describenodeproto(net_facet[0]), describenodeinst(nis[0]),
					describenodeproto(net_facet[1]), describenodeinst(nis[1]),
						total, unmatchednets, unmatchedcomps);
			net_forceamatch(sg, is[0], is[1], 0, verbose, ignorepwrgnd);
			return(1);
		} else
		{
			/* look for any ambiguous networks and randomly match them */
			for(f=0; f<2; f++)
			{
				any = -1;
				for(is[f]=0; is[f] < sg->facetcount[f]; is[f]++)
				{
					pn = (PNET *)sg->facetlist[f][is[f]];
					nets[f] = pn->network;
					if (nets[f] == NONETWORK) continue;
					any = is[f];
					if (nets[f]->namecount == 0 ||
						nets[f]->tempname != 0) break;
				}
				if (is[f] >= sg->facetcount[f])
				{
					if (any < 0) nets[f] = NONETWORK; else
					{
						is[f] = any;
						pn = (PNET *)sg->facetlist[f][any];
						nets[f] = pn->network;
					}
				}
			}
			if (nets[0] != NONETWORK && nets[1] != NONETWORK)
			{
				/* find a unique name and tag the selected networks */
				for(u=1; ; u++)
				{
					sprintf(uniquename, "NCCmatch%ld", u);
					for(f=0; f<2; f++)
						if (getnetwork(uniquename, nets[f]->parent) != NONETWORK) break;
					if (f >= 2) break;
				}
				for(f=0; f<2; f++)
				{
					for(ai = net_facet[f]->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
					{
						if (ai->network != nets[f]) continue;
						startobjectchange((INTBIG)ai, VARCINST);
						var = setvalkey((INTBIG)ai, VARCINST, el_arc_name,
							(INTBIG)uniquename, VSTRING|VDISPLAY);
						if (var != NOVARIABLE)
							defaulttextsize(4, var->textdescript);
						endobjectchange((INTBIG)ai, VARCINST);

						/* pickup new net number and remember it in the data structures */
						for(osg = net_firstsymgroup; osg != NOSYMGROUP; osg = osg->nextsymgroup)
						{
							if (osg->grouptype == SYMGROUPCOMP) continue;
							for(i=0; i<osg->facetcount[f]; i++)
							{
								pn = (PNET *)osg->facetlist[f][i];
								if (pn->network == nets[f])
									pn->network = ai->network;
							}
						}
						nets[f] = ai->network;
						break;
					}
				}

				ttyputmsg(_("--- Forcing a random match of networks '%s:%s' and '%s:%s' (%ld symmetry groups with %ld nets and %ld nodes unmatched)"),
					describenodeproto(net_facet[0]), describenetwork(nets[0]),
						describenodeproto(net_facet[1]), describenetwork(nets[1]),
							total, unmatchednets, unmatchedcomps);
				net_forceamatch(sg, is[0], is[1], 0, verbose, ignorepwrgnd);
				return(1);
			}
		}
	}

	return(0);
}

int net_sortsymgroups(const void *e1, const void *e2)
{
	REGISTER SYMGROUP *sg1, *sg2;
	REGISTER INTBIG sg1size, sg2size;

	sg1 = *((SYMGROUP **)e1);
	sg2 = *((SYMGROUP **)e2);
	sg1size = sg1->facetcount[0] + sg1->facetcount[1];
	sg2size = sg2->facetcount[0] + sg2->facetcount[1];
	if (sg1->hashvalue == 0 || sg1->facetcount[0] < 2 || sg1->facetcount[1] < 2) sg1size = 0;
	if (sg2->hashvalue == 0 || sg2->facetcount[0] < 2 || sg2->facetcount[1] < 2) sg2size = 0;
	return(sg1size - sg2size);
}

int net_sortpcomp(const void *e1, const void *e2)
{
	REGISTER PCOMP *pc1, *pc2;
	REGISTER char *pt1, *pt2;

	pc1 = *((PCOMP **)e1);
	pc2 = *((PCOMP **)e2);
	if (pc2->wirecount != pc1->wirecount)
		return(pc2->wirecount - pc1->wirecount);
	pt1 = net_describepcomp(pc1);
	pt2 = net_describepcomp(pc2);
	return(namesame(pt1, pt2));
}

int net_sortpnet(const void *e1, const void *e2)
{
	REGISTER PNET *pn1, *pn2;
	REGISTER INTBIG un1, un2;

	pn1 = *((PNET **)e1);
	pn2 = *((PNET **)e2);
	if (pn2->nodecount != pn1->nodecount)
		return(pn2->nodecount - pn1->nodecount);
	un1 = un2 = 0;
	if ((pn1->flags&(POWERNET|GROUNDNET|EXPORTEDNET)) == 0 &&
		(pn1->network == NONETWORK || pn1->network->namecount == 0)) un1 = 1;
	if ((pn2->flags&(POWERNET|GROUNDNET|EXPORTEDNET)) == 0 &&
		(pn2->network == NONETWORK || pn2->network->namecount == 0)) un2 = 1;
	if (un1 == 0 && un2 == 0)
	{
		return(namesame(net_describepnet(pn1), net_describepnet(pn2)));
	}
	if (un1 != 0 && un2 != 0) return(0);
	return(un1 - un2);
}

/*
 * Routine to search symmetry group "sg" for a size factor that will distinguish part of
 * the group.  Returns zero if none can be found.
 */
INTHUGE net_findcommonsizefactor(SYMGROUP *sg)
{
	REGISTER INTHUGE sizefactora, sizefactorb, sizefactor;
	REGISTER INTBIG i0, i1, founda, foundb;
	REGISTER PCOMP *pca, *pcb, *pc;

	/* first see if facet 0 has different sizes in it */
	for(i0=1; i0<sg->facetcount[0]; i0++)
	{
		pca = (PCOMP *)sg->facetlist[0][i0-1];
		pcb = (PCOMP *)sg->facetlist[0][i0];
		if ((pca->flags&(COMPHASWIDLEN|COMPHASAREA)) == 0 ||
			(pcb->flags&(COMPHASWIDLEN|COMPHASAREA)) == 0) continue;
		sizefactora = net_sizefactor(pca);
		sizefactorb = net_sizefactor(pcb);
		if (sizefactora != sizefactorb)
		{
			/* different sizes found, see if they exist in facet 1 */
			founda = foundb = 0;
			for(i1=0; i1<sg->facetcount[1]; i1++)
			{
				pc = (PCOMP *)sg->facetlist[1][i1];
				sizefactor = net_sizefactor(pc);
				if (sizefactor == sizefactora) founda = 1;
				if (sizefactor == sizefactorb) foundb = 1;
			}
			if (founda != 0 && foundb != 0) return(sizefactor);
		}
	}

	/* next see if facet 1 has different sizes in it */
	for(i1=1; i1<sg->facetcount[1]; i1++)
	{
		pca = (PCOMP *)sg->facetlist[1][i1-1];
		pcb = (PCOMP *)sg->facetlist[1][i1];
		if ((pca->flags&(COMPHASWIDLEN|COMPHASAREA)) == 0 ||
			(pcb->flags&(COMPHASWIDLEN|COMPHASAREA)) == 0) continue;
		sizefactora = net_sizefactor(pca);
		sizefactorb = net_sizefactor(pcb);
		if (sizefactora != sizefactorb)
		{
			/* different sizes found, see if they exist in facet 0 */
			founda = foundb = 0;
			for(i0=0; i0<sg->facetcount[0]; i0++)
			{
				pc = (PCOMP *)sg->facetlist[0][i0];
				sizefactor = net_sizefactor(pc);
				if (sizefactor == sizefactora) founda = 1;
				if (sizefactor == sizefactorb) foundb = 1;
			}
			if (founda != 0 && foundb != 0) return(sizefactor);
		}
	}
	return(0);
}

/*
 * Routine to force a match between parts of symmetry group "sg".  If "sizefactorsplit" is
 * zero, then entries "i0" in facet 0 and "i1" in facet 1 are to be matched.  Otherwise,
 * those components with size factor "sizefactorsplit" are to be matched.
 */
void net_forceamatch(SYMGROUP *sg, INTBIG i0, INTBIG i1, INTHUGE sizefactorsplit, INTBIG verbose,
	INTSML ignorepwrgnd)
{
	REGISTER SYMGROUP *sgnewc, *sgnewn;
	REGISTER INTHUGE hashvalue;
	REGISTER PNET *pn, *pn0, *pn1;
	REGISTER PCOMP *pc, *pc0, *pc1;
	REGISTER INTBIG i, f;

	if (sg->grouptype == SYMGROUPCOMP)
	{
		hashvalue = net_uniquesymmetrygrouphash(SYMGROUPCOMP);
		sgnewc = net_newsymgroup(SYMGROUPCOMP, hashvalue);

		if (sizefactorsplit == 0)
		{
			/* matching two like-named nodes */
			pc0 = (PCOMP *)sg->facetlist[0][i0];
			pc1 = (PCOMP *)sg->facetlist[1][i1];
			net_removefromsymgroup(sg, 0, i0);
			net_removefromsymgroup(sg, 1, i1);
			if (net_addtosymgroup(sgnewc, 0, (void *)pc0) != 0) return;
			if (net_addtosymgroup(sgnewc, 1, (void *)pc1) != 0) return;
			pc0->hashvalue = hashvalue;
			pc1->hashvalue = hashvalue;
			if (verbose != 0)
			{
				(void)reallocstring(&pc0->hashreason, "name matched", net_tool->cluster);
				(void)reallocstring(&pc1->hashreason, "name matched", net_tool->cluster);
			}
		} else
		{
			/* matching nodes with size "sizefactor" */
			for(f=0; f<2; f++)
			{
				for(i=sg->facetcount[f]-1; i>=0; i--)
				{
					pc = (PCOMP *)sg->facetlist[f][i];
					if (net_sizefactor(pc) == sizefactorsplit)
					{
						net_removefromsymgroup(sg, f, i);
						if (net_addtosymgroup(sgnewc, f, (void *)pc) != 0) return;
						pc->hashvalue = hashvalue;
						if (verbose != 0)
							(void)reallocstring(&pc->hashreason, "size matched", net_tool->cluster);
					}
				}
			}
		}

		/* set the remaining components in this symmetry group to a nonzero hash */
		hashvalue = net_uniquesymmetrygrouphash(SYMGROUPCOMP);
		sgnewc = net_newsymgroup(SYMGROUPCOMP, hashvalue);
		for(f=0; f<2; f++)
		{
			for(i=sg->facetcount[f]-1; i>=0; i--)
			{
				pc = (PCOMP *)sg->facetlist[f][i];
				net_removefromsymgroup(sg, f, i);
				if (net_addtosymgroup(sgnewc, f, (void *)pc) != 0) return;
				pc->hashvalue = hashvalue;
				if (verbose != 0)
					(void)reallocstring(&pc->hashreason, "redeemed", net_tool->cluster);
			}
		}
		sgnewn = NOSYMGROUP;
	} else
	{
		hashvalue = net_uniquesymmetrygrouphash(SYMGROUPNET);
		sgnewn = net_newsymgroup(SYMGROUPNET, hashvalue);

		pn0 = (PNET *)sg->facetlist[0][i0];
		pn1 = (PNET *)sg->facetlist[1][i1];
		net_removefromsymgroup(sg, 0, i0);
		net_removefromsymgroup(sg, 1, i1);
		if (net_addtosymgroup(sgnewn, 0, (void *)pn0) != 0) return;
		if (net_addtosymgroup(sgnewn, 1, (void *)pn1) != 0) return;
		pn0->hashvalue = hashvalue;
		pn1->hashvalue = hashvalue;
		if (verbose != 0)
		{
			(void)reallocstring(&pn0->hashreason, "export matched", net_tool->cluster);
			(void)reallocstring(&pn1->hashreason, "export matched", net_tool->cluster);
		}

		/* set the remaining nets in this symmetry group to a nonzero hash */
		hashvalue = net_uniquesymmetrygrouphash(SYMGROUPNET);
		sgnewn = net_newsymgroup(SYMGROUPNET, hashvalue);
		for(f=0; f<2; f++)
		{
			for(i=sg->facetcount[f]-1; i>=0; i--)
			{
				pn = (PNET *)sg->facetlist[f][i];
				net_removefromsymgroup(sg, f, i);
				if (net_addtosymgroup(sgnewn, f, (void *)pn) != 0) return;
				pn->hashvalue = hashvalue;
				if (verbose != 0)
					(void)reallocstring(&pn->hashreason, "redeemed", net_tool->cluster);
			}
		}
		sgnewc = NOSYMGROUP;
	}

	net_redeemzerogroups(sgnewc, sgnewn, verbose, ignorepwrgnd);
}

void net_redeemzerogroups(SYMGROUP *sgnewc, SYMGROUP *sgnewn, INTBIG verbose, INTSML ignorepwrgnd)
{
	REGISTER SYMGROUP *sg;
	REGISTER INTHUGE hashvalue;
	REGISTER PNET *pn;
	REGISTER PCOMP *pc;
	REGISTER INTBIG i, f;

	/* redeem all zero-symmetry groups */
	sg = net_findsymmetrygroup(SYMGROUPNET, 0);
	if (sg != NOSYMGROUP)
	{
		if (sgnewn == NOSYMGROUP)
		{
			hashvalue = net_uniquesymmetrygrouphash(SYMGROUPNET);
			sgnewn = net_newsymgroup(SYMGROUPNET, hashvalue);
		}
		for(f=0; f<2; f++)
		{
			for(i=sg->facetcount[f]-1; i>=0; i--)
			{
				pn = (PNET *)sg->facetlist[f][i];
				if (ignorepwrgnd != 0 && (pn->flags&(POWERNET|GROUNDNET)) != 0) continue;
				net_removefromsymgroup(sg, f, i);
				if (net_addtosymgroup(sgnewn, f, (void *)pn) != 0) return;
				pn->hashvalue = hashvalue;
				if (verbose != 0)
					(void)reallocstring(&pn->hashreason, "redeemed", net_tool->cluster);
			}
		}
	}

	sg = net_findsymmetrygroup(SYMGROUPCOMP, 0);
	if (sg != NOSYMGROUP)
	{
		if (sgnewc == NOSYMGROUP)
		{
			hashvalue = net_uniquesymmetrygrouphash(SYMGROUPCOMP);
			sgnewc = net_newsymgroup(SYMGROUPCOMP, hashvalue);
		}
		for(f=0; f<2; f++)
		{
			for(i=sg->facetcount[f]-1; i>=0; i--)
			{
				pc = (PCOMP *)sg->facetlist[f][i];
				net_removefromsymgroup(sg, f, i);
				if (net_addtosymgroup(sgnewc, f, (void *)pc) != 0) return;
				pc->hashvalue = hashvalue;
				if (verbose != 0)
					(void)reallocstring(&pc->hashreason, "redeemed", net_tool->cluster);
			}
		}
	}
}

/*
 * Routine to return the size of component "pc".
 */
INTHUGE net_sizefactor(PCOMP *pc)
{
	REGISTER INTHUGE w, l;

	if ((pc->flags&COMPHASWIDLEN) != 0)
	{
		w = (INTHUGE)pc->width;
		l = (INTHUGE)pc->length;
		return((w<<32) | l);
	}
	return((INTHUGE)pc->length);
}

/*
 * Routine to return a string describing size factor "sizefactor".
 */
char *net_describesizefactor(INTHUGE sizefactor)
{
	static char sizedesc[50];
	REGISTER INTBIG l, w;

	w = (INTBIG)(sizefactor >> 32);
	l = (INTBIG)(sizefactor & 0xFFFFFFFF);
	if (w == 0)
	{
		strcpy(sizedesc, frtoa(l));
	} else
	{
		sprintf(sizedesc, "%sx%s", frtoa(l), frtoa(w));
	}
	return(sizedesc);
}

/*
 * Routine to return nonzero if the exported ports on "pn1" and "pn2" match.
 */
INTSML net_sameexportnames(PNET *pn1, PNET *pn2)
{
	REGISTER INTBIG i, j, c1, c2, nc1, nc2;
	REGISTER char *name1, *name2, *netname1, *netname2;
	REGISTER PORTPROTO *pp1, *pp2;

	c1 = pn1->realportcount;
	if (pn1->network == NONETWORK) nc1 = 0; else
		nc1 = pn1->network->namecount;
	c2 = pn2->realportcount;
	if (pn2->network == NONETWORK) nc2 = 0; else
		nc2 = pn2->network->namecount;

	if (nc1 > 0) netname1 = pn1->network->netname;
	for(i=0; i<nc1+c1; i++)
	{
		if (i < nc1)
		{
			name1 = netname1;
			netname1 += strlen(netname1) + 1;
		} else
		{
			if (pn1->realportcount == 1) pp1 = (PORTPROTO *)pn1->realportlist; else
				pp1 = ((PORTPROTO **)pn1->realportlist)[i-nc1];
			name1 = pp1->protoname;
		}

		if (nc2 > 0) netname2 = pn2->network->netname;
		for(j=0; j<nc2+c2; j++)
		{
			if (j < nc2)
			{
				name2 = netname2;
				netname2 += strlen(netname2) + 1;
			} else
			{
				if (pn2->realportcount == 1) pp2 = (PORTPROTO *)pn2->realportlist; else
					pp2 = ((PORTPROTO **)pn2->realportlist)[j-nc2];
				name2 = pp2->protoname;
			}
			if (namesame(name1, name2) == 0) return(1);
		}
	}
	return(0);
}

void net_addcomptoerror(void *err, PCOMP *pc)
{
	REGISTER NODEINST *ni;
	REGISTER INTBIG i;

	for(i=0; i<pc->numactual; i++)
	{
		if (pc->numactual == 1) ni = (NODEINST *)pc->actuallist; else
			ni = ((NODEINST **)pc->actuallist)[i];
		addgeomtoerror(err, ni->geom, 1, pc->hierpathcount, pc->hierpath);
	}
}

void net_addnettoerror(void *err, PNET *pn)
{
	REGISTER ARCINST *ai;
	REGISTER NETWORK *net, *anet;
	REGISTER PORTPROTO *pp;
	REGISTER NODEPROTO *np;
	REGISTER INTBIG found, i;

	found = 0;
	net = pn->network;
	if (net == NONETWORK) return;
	np = net->parent;
	for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
	{
		anet = ai->network;
		if (ai->proto == sch_busarc)
		{
			if (anet->signals > 1)
			{
				for(i=0; i<anet->signals; i++)
					if (anet->networklist[i] == net) break;
				if (i >= anet->signals) continue;
			} else
			{
				if (anet != net) continue;
			}
		} else
		{
			if (anet != net) continue;
		}
		addgeomtoerror(err, ai->geom, 1, 0, 0);
		found++;
	}
	for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
	{
		if (pp->network != net) continue;
		addexporttoerror(err, pp, 1);
		found++;
	}
	if (found == 0 && net->namecount > 0)
	{
		if (np == net_facet[0]) np = net_facet[1]; else
			np = net_facet[0];
		net = getnetwork(net->netname, np);
		if (net == NONETWORK) return;
		for(ai = np->firstarcinst; ai != NOARCINST; ai = ai->nextarcinst)
		{
			if (ai->network != net) continue;
			addgeomtoerror(err, ai->geom, 1, 0, 0);
		}
		for(pp = np->firstportproto; pp != NOPORTPROTO; pp = pp->nextportproto)
		{
			if (pp->network != net) continue;
			addexporttoerror(err, pp, 1);
		}
	}
}

/*
 * Routine to add all objects in symmetry group "sg" to the error report "err".
 */
void net_addsymgrouptoerror(void *err, SYMGROUP *sg)
{
	REGISTER INTBIG i, f;
	REGISTER PCOMP *pc;
	REGISTER PNET *pn;

	switch (sg->grouptype)
	{
		case SYMGROUPCOMP:
			for(f=0; f<2; f++)
			{
				for(i=0; i<sg->facetcount[f]; i++)
				{
					pc = (PCOMP *)sg->facetlist[f][i];
					net_addcomptoerror(err, pc);
				}
			}
			break;
		case SYMGROUPNET:
			for(f=0; f<2; f++)
			{
				for(i=0; i<sg->facetcount[f]; i++)
				{
					pn = (PNET *)sg->facetlist[f][i];
					net_addnettoerror(err, pn);
				}
			}
			break;
	}
}

/*
 * Debugging routine to show the hash codes on all symmetry groups.
 */
void net_showsymmetrygroups(INTBIG verbose)
{
	WINDOWPART *win[2];
	REGISTER WINDOWPART *w;
	REGISTER INTBIG i, f;
	UINTBIG descript[TEXTDESCRIPTSIZE];
	REGISTER SYMGROUP *sg;
	REGISTER PNET *pn;
	REGISTER PCOMP *pc;

	if ((verbose&NCCVERBOSEGRAPHICS) != 0)
	{
		/* find the windows associated with the facets */
		win[0] = win[1] = NOWINDOWPART;
		for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
		{
			for(f=0; f<2; f++)
				if (w->curnodeproto == net_facet[f]) win[f] = w;
		}
		if (win[0] == NOWINDOWPART || win[1] == NOWINDOWPART) return;

		/* clear all highlighting */
		(void)asktool(us_tool, "clear");
		for(f=0; f<2; f++)
			screendrawbox(win[f], win[f]->uselx, win[f]->usehx, win[f]->usely, win[f]->usehy,
				&net_cleardesc);

		TDCLEAR(descript);
		TDSETSIZE(descript, TXTSETPOINTS(16));
		screensettextinfo(win[0], NOTECHNOLOGY, descript);
		screensettextinfo(win[1], NOTECHNOLOGY, descript);
	}
	for(sg = net_firstsymgroup; sg != NOSYMGROUP; sg = sg->nextsymgroup)
	{
		switch (sg->grouptype)
		{
			case SYMGROUPCOMP:
				for(f=0; f<2; f++)
				{
					for(i=0; i<sg->facetcount[f]; i++)
					{
						pc = (PCOMP *)sg->facetlist[f][i];
						net_showcomphash(win[f], pc, pc->hashvalue, sg->groupindex, verbose);
					}
				}
				break;
			case SYMGROUPNET:
				for(f=0; f<2; f++)
				{
					for(i=0; i<sg->facetcount[f]; i++)
					{
						pn = (PNET *)sg->facetlist[f][i];
						net_shownethash(win[f], pn, pn->hashvalue, sg->groupindex, verbose);
					}
				}
				break;
		}
	}
}

void net_shownethash(WINDOWPART *win, PNET *pn, INTHUGE hashvalue, INTBIG hashindex, INTBIG verbose)
{
	REGISTER NETWORK *net;
	char msg[50];
	REGISTER PORTPROTO *pp;
	REGISTER ARCINST *ai;
	INTSML tsx, tsy, px, py;
	REGISTER INTBIG j;
	INTBIG xp, yp;

	net = pn->network;
	if (net == NONETWORK) return;
	if ((verbose&NCCVERBOSEGRAPHICS) != 0)
	{
		if (hashindex != 0) sprintf(msg, "%ld", hashindex); else
			strcpy(msg, hugeinttoa(hashvalue));
		screengettextsize(win, msg, &tsx, &tsy);
		for(j=0; j<net->arccount; j++)
		{
			if (net->arccount == 1) ai = (ARCINST *)net->arcaddr; else
				ai = ((ARCINST **)net->arcaddr)[j];
			xp = (ai->end[0].xpos + ai->end[1].xpos) / 2;
			yp = (ai->end[0].ypos + ai->end[1].ypos) / 2;
			xp = applyxscale(win, xp-win->screenlx) + win->uselx;
			yp = applyyscale(win, yp-win->screenly) + win->usely;
			px = (INTSML)(xp - tsx/2);   py = (INTSML)(yp - tsy/2);
			if (px < win->uselx) px = win->uselx;
			if (px+tsx > win->usehx) px = win->usehx - tsx;
			screendrawtext(win, px, py, msg, &net_msgdesc);
		}
		if (net->portcount > 0)
		{
			for(pp = win->curnodeproto->firstportproto; pp != NOPORTPROTO;
				pp = pp->nextportproto)
			{
				if (pp->network != net) continue;
				portposition(pp->subnodeinst, pp->subportproto, &xp, &yp);
				xp = applyxscale(win, xp-win->screenlx) + win->uselx;
				yp = applyyscale(win, yp-win->screenly) + win->usely;
				px = (INTSML)(xp - tsx/2);   py = (INTSML)(yp - tsy/2);
				if (px < win->uselx) px = win->uselx;
				if (px+tsx > win->usehx) px = win->usehx - tsx;
				screendrawtext(win, px, py, msg, &net_msgdesc);
			}
		}
	}
	if ((verbose&NCCVERBOSETEXT) != 0)
	{
		if (hashindex != 0)
		{
			ttyputmsg(" NET %s:%s: #%ld (%s) = %s", describenodeproto(net->parent),
				net_describepnet(pn), hashindex, hugeinttoa(hashvalue), pn->hashreason);
		} else
		{
			ttyputmsg(" NET %s:%s: %s = %s", describenodeproto(net->parent),
				net_describepnet(pn), hugeinttoa(hashvalue), pn->hashreason);
		}
	}
}

void net_showcomphash(WINDOWPART *win, PCOMP *pc, INTHUGE hashvalue, INTBIG hashindex, INTBIG verbose)
{
	REGISTER INTBIG xp, yp;
	INTSML tsx, tsy, px, py;
	REGISTER NODEINST *ni;
	char msg[50];

	if (pc->numactual == 1) ni = (NODEINST *)pc->actuallist; else
		ni = ((NODEINST **)pc->actuallist)[0];
	if ((verbose&NCCVERBOSEGRAPHICS) != 0)
	{
		xp = (ni->lowx + ni->highx) / 2;
		yp = (ni->lowy + ni->highy) / 2;
		xp = applyxscale(win, xp-win->screenlx) + win->uselx;
		yp = applyyscale(win, yp-win->screenly) + win->usely;
		if (hashindex != 0) sprintf(msg, "%ld", hashindex); else
			strcpy(msg, hugeinttoa(hashvalue));
		screengettextsize(win, msg, &tsx, &tsy);
		px = (INTSML)(xp - tsx/2);   py = (INTSML)(yp - tsy/2);
		if (px < win->uselx) px = win->uselx;
		if (px+tsx > win->usehx) px = win->usehx - tsx;
		screendrawtext(win, px, py, msg, &net_msgdesc);
	}
	if ((verbose&NCCVERBOSETEXT) != 0)
	{
		if (hashindex != 0)
		{
			ttyputmsg(" NODE %s: #%ld (%s) = %s", describenodeinst(ni), hashindex,
				hugeinttoa(hashvalue), pc->hashreason);
		} else
		{
			ttyputmsg(" NODE %s: %s = %s", describenodeinst(ni), hugeinttoa(hashvalue),
				pc->hashreason);
		}
	}
}

void net_dumpnetwork(PCOMP *pclist, PNET *pnlist)
{
	REGISTER PCOMP *pc;
	REGISTER PNET *pn;
	int i;
	REGISTER NODEINST *ni;
	char nettype[50];

	ttyputmsg("Nodes:");
	for(pc = pclist; pc != NOPCOMP; pc = pc->nextpcomp)
	{
		if (pc->numactual == 1) ni = (NODEINST *)pc->actuallist; else
			ni = ((NODEINST **)pc->actuallist)[0];
		ttyputmsg("  Node %s (fun=%d)", describenodeinst(ni), pc->function);
	}
	ttyputmsg("Nets:");
	for(pn = pnlist; pn != NOPNET; pn = pn->nextpnet)
	{
		(void)initinfstr();
		nettype[0] = 0;
		if ((pn->flags&(POWERNET|GROUNDNET)) != 0)
		{
			if ((pn->flags&POWERNET) != 0) strcat(nettype, "POWER "); else
				strcat(nettype, "GROUND ");
		}
		formatinfstr("  %sNet %s (%d nodes):", nettype, net_describepnet(pn), pn->nodecount);
		for(i=0; i<pn->nodecount; i++)
		{
			pc = pn->nodelist[i];
			if (pc->numactual == 1) ni = (NODEINST *)pc->actuallist; else
				ni = ((NODEINST **)pc->actuallist)[0];
			formatinfstr(" %s", describenodeinst(ni));
		}
		ttyputmsg("%s", returninfstr());
	}
}

char *net_describecounts(INTBIG compcount, INTBIG netcount, INTBIG buscount)
{
	(void)initinfstr();
	(void)formatinfstr(_("%ld components, %ld nets"), compcount, netcount);
	if (buscount > 0) (void)formatinfstr(_(", %ld busses"), buscount);
	return(returninfstr());
}

void net_dumpnetworks(PNET *nodelist1, PNET *nodelist2, INTSML ignorepwrgnd)
{
	REGISTER INTBIG pn1total, pn2total, ind1, ind2, pn1width, len, i,
		reportedwid, curwid;
	REGISTER char *pt;
	char line[100];
	REGISTER PNET **pn1list, **pn2list, *pn, *pn1, *pn2;

	/* count the number of networks in the first facet */
	ttyputmsg(" ");
	ttyputmsg(_("===== NETWORKS ====="));
	pn1width = strlen(describenodeproto(net_facet[0]));
	pn1total = 0;
	for(pn = nodelist1; pn != NOPNET; pn = pn->nextpnet)
	{
		/* accumulate the widest network description */
		len = strlen(net_describepnet(pn));
		if (len > pn1width) pn1width = len;
		pn1total++;
	}
	pn1width += 6;

	/* make a sorted list of the networks in the first facet */
	if (pn1total > 0)
	{
		pn1list = (PNET **)emalloc(pn1total * (sizeof (PNET *)), net_tool->cluster);
		if (pn1list == 0) return;
		pn1total = 0;
		for(pn = nodelist1; pn != NOPNET; pn = pn->nextpnet)
			pn1list[pn1total++] = pn;
		esort(pn1list, pn1total, sizeof (PNET *), net_sortpnet);
	}

	/* count the number of networks in the second facet */
	pn2total = 0;
	for(pn = nodelist2; pn != NOPNET; pn = pn->nextpnet) pn2total++;

	/* make a sorted list of the networks in the second facet */
	if (pn2total > 0)
	{
		pn2list = (PNET **)emalloc(pn2total * (sizeof (PNET *)), net_tool->cluster);
		if (pn2list == 0) return;
		pn2total = 0;
		for(pn = nodelist2; pn != NOPNET; pn = pn->nextpnet)
			pn2list[pn2total++] = pn;
		esort(pn2list, pn2total, sizeof (PNET *), net_sortpnet);
	}

	/* put out the facet names */
	(void)initinfstr();
	pt = describenodeproto(net_facet[0]);
	(void)addstringtoinfstr(pt);
	for(len = strlen(pt); len < pn1width; len++) (void)addtoinfstr(' ');
	(void)addstringtoinfstr(describenodeproto(net_facet[1]));
	ttyputmsg("%s", returninfstr());

	/* underline the facet names */
	(void)initinfstr();
	len = strlen(describenodeproto(net_facet[0]));
	for(i=0; i<len; i++) (void)addtoinfstr('-');
	for( ; len < pn1width; len++) (void)addtoinfstr(' ');
	len = strlen(describenodeproto(net_facet[1]));
	for(i=0; i<len; i++) (void)addtoinfstr('-');
	ttyputmsg("%s", returninfstr());

	ind1 = ind2 = 0;
	reportedwid = -1;
	for(;;)
	{
		if (ind1 >= pn1total) pn1 = NOPNET; else
			pn1 = pn1list[ind1];
		if (ind2 >= pn2total) pn2 = NOPNET; else
			pn2 = pn2list[ind2];
		if (pn1 == NOPNET && pn2 == NOPNET) break;
		if (pn1 != NOPNET && pn2 != NOPNET)
		{
			if (pn1->nodecount < pn2->nodecount) pn1 = NOPNET; else
				if (pn1->nodecount > pn2->nodecount) pn2 = NOPNET;
		}
		if (pn1 != NOPNET) curwid = pn1->nodecount; else
			if (pn2 != NOPNET) curwid = pn2->nodecount;
		if (curwid != reportedwid)
		{
			sprintf(line, _("%ld components:"), curwid);
			len = strlen(line);
			(void)initinfstr();
			for(i=0; i<pn1width-len+1; i++) (void)addtoinfstr(' ');
			(void)addstringtoinfstr(line);
			ttyputmsg("%s", returninfstr());
			reportedwid = curwid;
		}

		(void)initinfstr();
		if (pn1 == NOPNET) len = 0; else
		{
			if ((pn1->flags&(POWERNET|GROUNDNET|EXPORTEDNET)) == 0 &&
				(pn1->network == NONETWORK || pn1->network->namecount == 0))
			{
				/* unnamed internal network: merge with others like it */
				i = 0;
				for(;;)
				{
					i++;
					ind1++;
					if (ind1 >= pn1total) break;
					pn = pn1list[ind1];
					if (pn->nodecount != pn1->nodecount) break;
					if ((pn->flags&(POWERNET|GROUNDNET|EXPORTEDNET)) != 0 ||
						(pn->network != NONETWORK && pn->network->namecount != 0)) break;
				}
				if (i == 1) strcpy(line, _("Unnamed net")); else
					sprintf(line, _("Unnamed nets (%ld)"), i);
				len = strlen(line);
				(void)addstringtoinfstr(line);
			} else
			{
				pt = net_describepnet(pn1);
				len = strlen(pt);
				(void)addstringtoinfstr(pt);
				ind1++;
			}
		}
		for( ; len<pn1width; len++) (void)addtoinfstr(' ');
		(void)addstringtoinfstr("   ");
		if (pn2 != NOPNET)
		{
			if ((pn2->flags&(POWERNET|GROUNDNET|EXPORTEDNET)) == 0 &&
				(pn2->network == NONETWORK || pn2->network->namecount == 0))
			{
				/* unnamed internal network: merge with others like it */
				i = 0;
				for(;;)
				{
					i++;
					ind2++;
					if (ind2 >= pn2total) break;
					pn = pn2list[ind2];
					if (pn->nodecount != pn2->nodecount) break;
					if ((pn->flags&(POWERNET|GROUNDNET|EXPORTEDNET)) != 0 ||
						(pn->network != NONETWORK && pn->network->namecount != 0)) break;
				}
				if (i == 1) strcpy(line, _("Unnamed net")); else
					sprintf(line, _("Unnamed nets (%ld)"), i);
				len = strlen(line);
				(void)addstringtoinfstr(line);
			} else
			{
				(void)addstringtoinfstr(net_describepnet(pn2));
				ind2++;
			}
		}
		ttyputmsg("%s", returninfstr());
	}

	if (pn1total > 0) efree((char *)pn1list);
	if (pn2total > 0) efree((char *)pn2list);
}

void net_dumpcomponents(PCOMP *pcomp1, PCOMP *pcomp2, INTSML ignorepwrgnd)
{
	REGISTER INTBIG pc1total, pc2total, ind1, ind2, pc1width, len, i, w,
		reportedwid, curwid;
	REGISTER char *pt;
	char line[100];
	REGISTER PCOMP **pc1list, **pc2list, *pc, *pc1, *pc2;
	REGISTER PNET *pn;

	/* count the number of components in the first facet */
	ttyputmsg(" ");
	ttyputmsg(_("===== COMPONENTS ====="));
	pc1width = strlen(describenodeproto(net_facet[0]));
	pc1total = 0;
	for(pc = pcomp1; pc != NOPCOMP; pc = pc->nextpcomp)
	{
		/* accumulate the widest component description */
		len = strlen(net_describepcomp(pc));
		if (len > pc1width) pc1width = len;
		pc1total++;

		/* adjust the number of wires, removing ignored power and ground */
		w = 0;
		for(i=0; i<pc->wirecount; i++)
		{
			pn = pc->netnumbers[i];
			if (ignorepwrgnd != 0 && (pn->flags&(POWERNET|GROUNDNET)) != 0) continue;
			w++;
		}
		pc->timestamp = pc->wirecount;
		pc->wirecount = (INTSML)w;
	}
	pc1width += 6;

	/* make a sorted list of the components in the first facet */
	if (pc1total > 0)
	{
		pc1list = (PCOMP **)emalloc(pc1total * (sizeof (PCOMP *)), net_tool->cluster);
		if (pc1list == 0) return;
		pc1total = 0;
		for(pc = pcomp1; pc != NOPCOMP; pc = pc->nextpcomp)
			pc1list[pc1total++] = pc;
		esort(pc1list, pc1total, sizeof (PCOMP *), net_sortpcomp);
	}

	/* count the number of components in the second facet */
	pc2total = 0;
	for(pc = pcomp2; pc != NOPCOMP; pc = pc->nextpcomp)
	{
		pc2total++;

		/* adjust the number of wires, removing ignored power and ground */
		w = 0;
		for(i=0; i<pc->wirecount; i++)
		{
			pn = pc->netnumbers[i];
			if (ignorepwrgnd != 0 && (pn->flags&(POWERNET|GROUNDNET)) != 0) continue;
			w++;
		}
		pc->timestamp = pc->wirecount;
		pc->wirecount = (INTSML)w;
	}

	/* make a sorted list of the components in the second facet */
	if (pc2total > 0)
	{
		pc2list = (PCOMP **)emalloc(pc2total * (sizeof (PCOMP *)), net_tool->cluster);
		if (pc2list == 0) return;
		pc2total = 0;
		for(pc = pcomp2; pc != NOPCOMP; pc = pc->nextpcomp)
			pc2list[pc2total++] = pc;
		esort(pc2list, pc2total, sizeof (PCOMP *), net_sortpcomp);
	}

	/* put out the facet names */
	(void)initinfstr();
	pt = describenodeproto(net_facet[0]);
	(void)addstringtoinfstr(pt);
	for(len = strlen(pt); len < pc1width; len++) (void)addtoinfstr(' ');
	(void)addstringtoinfstr(describenodeproto(net_facet[1]));
	ttyputmsg("%s", returninfstr());

	/* underline the facet names */
	(void)initinfstr();
	len = strlen(describenodeproto(net_facet[0]));
	for(i=0; i<len; i++) (void)addtoinfstr('-');
	for( ; len < pc1width; len++) (void)addtoinfstr(' ');
	len = strlen(describenodeproto(net_facet[1]));
	for(i=0; i<len; i++) (void)addtoinfstr('-');
	ttyputmsg("%s", returninfstr());

	ind1 = ind2 = 0;
	reportedwid = -1;
	for(;;)
	{
		if (ind1 >= pc1total) pc1 = NOPCOMP; else
			pc1 = pc1list[ind1];
		if (ind2 >= pc2total) pc2 = NOPCOMP; else
			pc2 = pc2list[ind2];
		if (pc1 == NOPCOMP && pc2 == NOPCOMP) break;
		if (pc1 != NOPCOMP && pc2 != NOPCOMP)
		{
			if (pc1->wirecount < pc2->wirecount) pc1 = NOPCOMP; else
				if (pc1->wirecount > pc2->wirecount) pc2 = NOPCOMP;
		}
		if (pc1 != NOPCOMP) curwid = pc1->wirecount; else
			if (pc2 != NOPCOMP) curwid = pc2->wirecount;
		if (curwid != reportedwid)
		{
			sprintf(line, _("%ld wires:"), curwid);
			len = strlen(line);
			(void)initinfstr();
			for(i=0; i<pc1width-len+1; i++) (void)addtoinfstr(' ');
			(void)addstringtoinfstr(line);
			ttyputmsg("%s", returninfstr());
			reportedwid = curwid;
		}

		(void)initinfstr();
		if (pc1 == NOPCOMP) len = 0; else
		{
			i = 0;
			for(;;)
			{
				i++;
				ind1++;
				if (ind1 >= pc1total) break;
				if (namesame(net_describepcomp(pc1), net_describepcomp(pc1list[ind1])) != 0)
					break;
			}
			pt = net_describepcomp(pc1);
			len = strlen(pt);
			(void)addstringtoinfstr(pt);
			if (i > 1)
			{
				sprintf(line, " (%ld)", i);
				(void)addstringtoinfstr(line);
				len += strlen(line);
			}
		}
		for( ; len<pc1width; len++) (void)addtoinfstr(' ');
		if (pc2 != NOPCOMP)
		{
			i = 0;
			for(;;)
			{
				i++;
				ind2++;
				if (ind2 >= pc2total) break;
				if (namesame(net_describepcomp(pc2), net_describepcomp(pc2list[ind2])) != 0)
					break;
			}
			(void)addstringtoinfstr(net_describepcomp(pc2));
			if (i > 1)
				formatinfstr(" (%ld)", i);
		}
		ttyputmsg("%s", returninfstr());
	}

	/* restore true wire counts */
	for(pc = pcomp1; pc != NOPCOMP; pc = pc->nextpcomp)
		pc->wirecount = (INTSML)pc->timestamp;
	for(pc = pcomp2; pc != NOPCOMP; pc = pc->nextpcomp)
		pc->wirecount = (INTSML)pc->timestamp;
	if (pc1total > 0) efree((char *)pc1list);
	if (pc2total > 0) efree((char *)pc2list);
}

/*
 * Routine to assign new hash values to the components or nets (depending on the
 * value of "grouptype") in all symmetry groups.  Returns the number of changes
 * that were made (negative on error).
 */
INTBIG net_assignnewhashvalues(INTBIG grouptype)
{
	REGISTER SYMGROUP *sg, *osg;
	REGISTER INTBIG f, i, changes, verbose;
	REGISTER PCOMP *pc;
	REGISTER PNET *pn;

	verbose = net_ncc_options & (NCCVERBOSETEXT | NCCVERBOSEGRAPHICS);
	changes = 0;
	for(sg = net_firstsymgroup; sg != NOSYMGROUP; sg = sg->nextsymgroup)
	{
		if (sg->hashvalue == 0) continue;
		if (sg->grouptype != grouptype) continue;
		switch (sg->grouptype)
		{
			case SYMGROUPCOMP:
				for(f=0; f<2; f++)
				{
					for(i=0; i<sg->facetcount[f]; i++)
					{
						pc = (PCOMP *)sg->facetlist[f][i];

						/* if the group is properly matched, don't change its hash value */
						if (sg->facetcount[0] == 1 && sg->facetcount[1] == 1)
						{
							if (verbose != 0)
								(void)reallocstring(&pc->hashreason, "matched", net_tool->cluster);
							continue;
						}

						/* if the group is a singleton, set a zero hash value */
						if (sg->facetcount[0] <= 0 || sg->facetcount[1] <= 0)
						{
							if (verbose != 0)
								(void)reallocstring(&pc->hashreason, "unmatched", net_tool->cluster);
							pc->hashvalue = 0;
						} else
						{
							/* compute a new hash value for the component */
							pc->hashvalue = net_getcomphash(pc, verbose);
						}
					}
				}
				for(f=0; f<2; f++)
				{
					for(i=0; i<sg->facetcount[f]; i++)
					{
						pc = (PCOMP *)sg->facetlist[f][i];
						if (pc->hashvalue != sg->hashvalue)
						{
							/* reassign this component to a different symmetry group */
							osg = net_findsymmetrygroup(SYMGROUPCOMP, pc->hashvalue);
							if (osg == NOSYMGROUP)
							{
								osg = net_newsymgroup(SYMGROUPCOMP, pc->hashvalue);
								if (osg == NOSYMGROUP) return(-1);
							}
							net_removefromsymgroup(sg, f, i);
							i--;
							if (net_addtosymgroup(osg, f, (void *)pc) != 0) return(-1);
							changes++;
						}
					}
				}
				break;

			case SYMGROUPNET:
				for(f=0; f<2; f++)
				{
					for(i=0; i<sg->facetcount[f]; i++)
					{
						pn = (PNET *)sg->facetlist[f][i];

						/* if the group is properly matched, don't change its hash value */
						if (sg->facetcount[0] == 1 && sg->facetcount[1] == 1)
						{
							if (verbose != 0)
								(void)reallocstring(&pn->hashreason, "matched", net_tool->cluster);
							continue;
						}

						/* if the group is a singleton, set a zero hash value */
						if (sg->facetcount[0] <= 0 || sg->facetcount[1] <= 0)
						{
							pn->hashvalue = 0;
							if (verbose != 0)
								(void)reallocstring(&pn->hashreason, "unmatched", net_tool->cluster);
						} else
						{
							/* compute a new hash value for the net */
							pn->hashvalue = net_getnethash(pn, verbose);
						}
					}
				}
				for(f=0; f<2; f++)
				{
					for(i=0; i<sg->facetcount[f]; i++)
					{
						pn = (PNET *)sg->facetlist[f][i];
						if (pn->hashvalue != sg->hashvalue)
						{
							/* reassign this component to a different symmetry group */
							osg = net_findsymmetrygroup(SYMGROUPNET, pn->hashvalue);
							if (osg == NOSYMGROUP)
							{
								osg = net_newsymgroup(SYMGROUPNET, pn->hashvalue);
								if (osg == NOSYMGROUP) return(-1);
							}
							net_removefromsymgroup(sg, f, i);
							i--;
							if (net_addtosymgroup(osg, f, (void *)pn) != 0) return(-1);
							changes++;
						}
					}
				}
				break;
		}
	}
	return(changes);
}

/*
 * Routine to find a unique hash number for a new symmetry group.
 */
INTHUGE net_uniquesymmetrygrouphash(INTBIG grouptype)
{
	REGISTER SYMGROUP *sg;

	for( ; ; net_uniquehashvalue--)
	{
		sg = net_findsymmetrygroup(grouptype, net_uniquehashvalue);
		if (sg == NOSYMGROUP) break;
	}
	return(net_uniquehashvalue);
}

/*
 * Routine to find the symmetry group associated with object "obj".
 * Returns NOSYMGROUP if none is found.
 */
SYMGROUP *net_findgeomsymmetrygroup(GEOM *obj)
{
	REGISTER SYMGROUP *sg;
	REGISTER INTBIG f, i, j, fun;
	REGISTER PCOMP *pc;
	REGISTER PNET *pn;
	REGISTER NODEINST *ni, *wantni;
	REGISTER ARCINST *ai;

	if (obj->entrytype == OBJNODEINST)
	{
		/* look for a node */
		wantni = obj->entryaddr.ni;
		for(sg = net_firstsymgroup; sg != NOSYMGROUP; sg = sg->nextsymgroup)
		{
			if (sg->grouptype != SYMGROUPCOMP) continue;
			for(f=0; f<2; f++)
			{
				for(i=0; i<sg->facetcount[f]; i++)
				{
					pc = (PCOMP *)sg->facetlist[f][i];
					for(j=0; j<pc->numactual; j++)
					{
						if (pc->numactual == 1) ni = (NODEINST *)pc->actuallist; else
							ni = ((NODEINST **)pc->actuallist)[j];
						if (ni == wantni) return(sg);
					}
				}
			}
		}

		/* node not found, try network coming out of it */
		fun = nodefunction(wantni);
		if (fun == NPPIN || fun == NPCONTACT || fun == NPCONNECT)
		{
			if (wantni->firstportarcinst != NOPORTARCINST)
				obj = wantni->firstportarcinst->conarcinst->geom;
		}
		if (obj->entrytype != OBJARCINST) return(NOSYMGROUP);
	}

	/* look for an arc */
	ai = obj->entryaddr.ai;
	for(sg = net_firstsymgroup; sg != NOSYMGROUP; sg = sg->nextsymgroup)
	{
		if (sg->grouptype != SYMGROUPNET) continue;
		for(f=0; f<2; f++)
		{
			for(i=0; i<sg->facetcount[f]; i++)
			{
				pn = (PNET *)sg->facetlist[f][i];
				if (pn->network == ai->network) return(sg);
			}
		}
	}
	return(NOSYMGROUP);
}

/*
 * Routine to fill out the "nodecount/nodelist/nodewire" fields of the PNET
 * list in "pnetlist", given that it points to "pcomplist".  Returns nonzero on error.
 */
INTSML net_buildfullpnetdata(PCOMP **pcomplist, PNET **pnetlist, INTBIG verbose)
{
	REGISTER PNET *pn, *lastpn, *nextpn;
	REGISTER PCOMP *pc, *lastpc, *nextpc;
	REGISTER INTBIG i;
	REGISTER NETWORK *net;

	/* initialize all networks */
	lastpn = NOPNET;
	for(pn = *pnetlist; pn != NOPNET; pn = nextpn)
	{
		nextpn = pn->nextpnet;
		net = pn->network;

		/* remove networks that refer to busses (individual signals are compared) */
		if (net != NONETWORK && net->signals > 1)
		{
			if (lastpn == NOPNET)
			{
				*pnetlist = pn->nextpnet;
			} else
			{
				lastpn->nextpnet = pn->nextpnet;
			}
			net_freepnet(pn);
			continue;
		}
		pn->nodecount = 0;
		if (verbose != 0)
			(void)allocstring(&pn->hashreason, "initial", net_tool->cluster);
		lastpn = pn;
	}

	lastpc = NOPCOMP;
	for(pc = *pcomplist; pc != NOPCOMP; pc = nextpc)
	{
		nextpc = pc->nextpcomp;

		/* remove components that relate to SPICE simulation */
		if (net_isspice(pc) != 0)
		{
			if (lastpc == NOPCOMP)
			{
				*pcomplist = pc->nextpcomp;
			} else
			{
				lastpc->nextpcomp = pc->nextpcomp;
			}
			net_freepcomp(pc);
			continue;
		}
		if (verbose != 0)
			(void)allocstring(&pc->hashreason, "initial", net_tool->cluster);
		for(i=0; i<pc->wirecount; i++)
		{
			pn = pc->netnumbers[i];
			pn->nodecount++;
			pn->nodelist = 0;
		}
		lastpc = pc;
	}

	for(pn = *pnetlist; pn != NOPNET; pn = pn->nextpnet)
	{
		if (pn->nodecount == 0) continue;
		pn->nodelist = (PCOMP **)emalloc(pn->nodecount * (sizeof (PCOMP *)), net_tool->cluster);
		if (pn->nodelist == 0) return(1);
		pn->nodewire = (INTBIG *)emalloc(pn->nodecount * SIZEOFINTBIG, net_tool->cluster);
		if (pn->nodelist == 0) return(1);
	}
	for(pn = *pnetlist; pn != NOPNET; pn = pn->nextpnet)
		pn->nodecount = 0;
	for(pc = *pcomplist; pc != NOPCOMP; pc = pc->nextpcomp)
	{
		for(i=0; i<pc->wirecount; i++)
		{
			pn = pc->netnumbers[i];
			if (pn->nodelist == 0)
				continue;
			pn->nodelist[pn->nodecount] = pc;
			pn->nodewire[pn->nodecount] = i;
			pn->nodecount++;
		}
	}
	return(0);
}

/*
 * Routine to find the symmetry group of type "grouptype" with hash value "hashvalue".
 * Returns NOSYMGROUP if none is found.
 */
SYMGROUP *net_findsymmetrygroup(INTBIG grouptype, INTHUGE hashvalue)
{
	REGISTER SYMGROUP *sg;
	REGISTER INTBIG i, hashindex;

	if (grouptype == SYMGROUPNET)
	{
		hashindex = abs((INTBIG)(hashvalue % net_symgrouphashnetsize));
		for(i=0; i<net_symgrouphashnetsize; i++)
		{
			sg = net_symgrouphashnet[hashindex];
			if (sg == NOSYMGROUP) break;
			if (sg->hashvalue == hashvalue) return(sg);
			hashindex++;
			if (hashindex >= net_symgrouphashnetsize) hashindex = 0;
		}
	} else
	{
		hashindex = abs((INTBIG)(hashvalue % net_symgrouphashcompsize));
		for(i=0; i<net_symgrouphashcompsize; i++)
		{
			sg = net_symgrouphashcomp[hashindex];
			if (sg == NOSYMGROUP) break;
			if (sg->hashvalue == hashvalue) return(sg);
			hashindex++;
			if (hashindex >= net_symgrouphashcompsize) hashindex = 0;
		}
	}
	return(NOSYMGROUP);
}

/*
 * Routine to create a new symmetry group of type "grouptype" with hash value "hashvalue".
 * The group is linked into the global list of symmetry groups.
 * Returns NOSYMGROUP on error.
 */
SYMGROUP *net_newsymgroup(INTBIG grouptype, INTHUGE hashvalue)
{
	REGISTER SYMGROUP *sg;

	if (net_symgroupfree == NOSYMGROUP)
	{
		sg = (SYMGROUP *)emalloc(sizeof (SYMGROUP), net_tool->cluster);
		if (sg == 0) return(NOSYMGROUP);
		sg->facettotal[0] = sg->facettotal[1] = 0;
	} else
	{
		sg = net_symgroupfree;
		net_symgroupfree = sg->nextsymgroup;
	}
	sg->grouptype = grouptype;
	sg->hashvalue = hashvalue;
	sg->groupindex = net_symgroupnumber++;
	sg->facetcount[0] = sg->facetcount[1] = 0;
	sg->nextsymgroup = net_firstsymgroup;
	net_firstsymgroup = sg;

	/* put it in the hash table */
	if (net_insertinhashtable(sg) != 0)
		net_rebuildhashtable();
	return(sg);
}

void net_rebuildhashtable(void)
{
	REGISTER INTBIG i;
	REGISTER INTSML problems;
	REGISTER SYMGROUP *sg;

	for(;;)
	{
		problems = 0;
		for(i=0; i<net_symgrouphashcompsize; i++) net_symgrouphashcomp[i] = NOSYMGROUP;
		for(i=0; i<net_symgrouphashnetsize; i++) net_symgrouphashnet[i] = NOSYMGROUP;
		for(sg = net_firstsymgroup; sg != NOSYMGROUP; sg = sg->nextsymgroup)
		{
			problems = net_insertinhashtable(sg);
			if (problems != 0) break;
		}
		if (problems == 0) break;
	}
}

/*
 * Routine to insert symmetry group "sg" into a hash table.  Returns nonzero
 * if the hash table needed to be expanded (and is thus invalid now).
 */
INTSML net_insertinhashtable(SYMGROUP *sg)
{
	REGISTER INTBIG i, hashindex, newsize;
	REGISTER SYMGROUP **newhashtable;

	if (sg->grouptype == SYMGROUPNET)
	{
		hashindex = abs((INTBIG)(sg->hashvalue % net_symgrouphashnetsize));
		for(i=0; i<net_symgrouphashnetsize; i++)
		{
			if (net_symgrouphashnet[hashindex] == NOSYMGROUP)
			{
				net_symgrouphashnet[hashindex] = sg;
				break;
			}
			hashindex++;
			if (hashindex >= net_symgrouphashnetsize) hashindex = 0;
		}
		if (i >= net_symgrouphashnetsize)
		{
			newsize = pickprime(net_symgrouphashnetsize * 2);
			newhashtable = (SYMGROUP **)emalloc(newsize * (sizeof (SYMGROUP *)),
				net_tool->cluster);
			if (newhashtable == 0) return(0);
			efree((char *)net_symgrouphashnet);
			net_symgrouphashnet = newhashtable;
			net_symgrouphashnetsize = newsize;
			ttyputmsg(" -- EXPANDING SIZE OF NETWORK HASH TABLE TO %ld ENTRIES",
				newsize);
			return(1);
		}
	} else
	{
		hashindex = abs((INTBIG)(sg->hashvalue % net_symgrouphashcompsize));
		for(i=0; i<net_symgrouphashcompsize; i++)
		{
			if (net_symgrouphashcomp[hashindex] == NOSYMGROUP)
			{
				net_symgrouphashcomp[hashindex] = sg;
				break;
			}
			hashindex++;
			if (hashindex >= net_symgrouphashcompsize) hashindex = 0;
		}
		if (i >= net_symgrouphashcompsize)
		{
			newsize = pickprime(net_symgrouphashcompsize * 2);
			newhashtable = (SYMGROUP **)emalloc(newsize * (sizeof (SYMGROUP *)),
				net_tool->cluster);
			if (newhashtable == 0) return(0);
			efree((char *)net_symgrouphashcomp);
			net_symgrouphashcomp = newhashtable;
			net_symgrouphashcompsize = newsize;
			ttyputmsg(" -- EXPANDING SIZE OF COMPONENT HASH TABLE TO %ld ENTRIES",
				newsize);
			return(1);
		}
	}
	return(0);
}


/*
 * Routine to free symmetry group "sg" to the pool of unused ones.
 */
void net_freesymgroup(SYMGROUP *sg)
{
	sg->nextsymgroup = net_symgroupfree;
	net_symgroupfree = sg;
}

/*
 * Routine to add object "obj" to facet "f" (0 or 1) of symmetry group "sg".
 * Returns nonzero on error.
 */
INTSML net_addtosymgroup(SYMGROUP *sg, INTBIG f, void *obj)
{
	INTBIG newtotal, i;
	REGISTER PNET *pn;
	REGISTER PCOMP *pc;
	void **newlist;

	if (sg->facetcount[f] >= sg->facettotal[f])
	{
		newtotal = sg->facetcount[f] + 10;
		newlist = (void **)emalloc(newtotal * (sizeof (void *)), net_tool->cluster);
		if (newlist == 0) return(1);
		for(i=0; i < sg->facetcount[f]; i++)
			newlist[i] = sg->facetlist[f][i];
		if (sg->facettotal[f] > 0) efree((char *)sg->facetlist[f]);
		sg->facetlist[f] = newlist;
		sg->facettotal[f] = newtotal;
	}
	sg->facetlist[f][sg->facetcount[f]] = obj;
	sg->facetcount[f]++;
	if (sg->grouptype == SYMGROUPNET)
	{
		pn = (PNET *)obj;
		pn->timestamp = net_timestamp;
	} else
	{
		pc = (PCOMP *)obj;
		pc->timestamp = net_timestamp;
	}
	return(0);
}

/*
 * Routine to remove entry "index" from facet "f" (0 or 1) of symmetry group "sg".
 */
void net_removefromsymgroup(SYMGROUP *sg, INTBIG f, INTBIG index)
{
	REGISTER INTBIG count;

	count = sg->facetcount[f];
	sg->facetlist[f][index] = sg->facetlist[f][count-1];
	sg->facetcount[f]--;
}

/*********************** HELPER ROUTINES ***********************/

/*
 * Routine to return nonzero if the facets "facet1" and "facet2" are already NCC'd.
 */
INTSML net_nccalreadydone(NODEPROTO *facet1, NODEPROTO *facet2)
{
	REGISTER VARIABLE *var1, *var2;
	REGISTER UINTBIG lastgooddate1, lastgooddate2;

	/* see if both facets have a valid NCC date */
	var1 = getvalkey((INTBIG)facet1, VNODEPROTO, VINTEGER, net_lastgoodncckey);
	var2 = getvalkey((INTBIG)facet2, VNODEPROTO, VINTEGER, net_lastgoodncckey);
	if (var1 == NOVARIABLE || var2 == NOVARIABLE) return(0);
	lastgooddate1 = (UINTBIG)var1->addr;
	lastgooddate2 = (UINTBIG)var2->addr;
	if (facet1->revisiondate > lastgooddate1 ||facet2->revisiondate > lastgooddate2)
		return(0);

	/* make sure the facets were compared with each other */
	var1 = getvalkey((INTBIG)facet1, VNODEPROTO, VNODEPROTO, net_lastgoodnccfacetkey);
	var2 = getvalkey((INTBIG)facet2, VNODEPROTO, VNODEPROTO, net_lastgoodnccfacetkey);
	if (var1 == NOVARIABLE || var2 == NOVARIABLE) return(0);
	if ((NODEPROTO *)var1->addr != facet2 ||
		(NODEPROTO *)var2->addr != facet1) return(0);

	/* the facets are already checked */
	return(1);
}

char *net_describepcomp(PCOMP *pc)
{
	REGISTER INTBIG i;
	REGISTER NODEINST *ni;

	(void)initinfstr();
	for(i=0; i<pc->numactual; i++)
	{
		if (pc->numactual == 1) ni = (NODEINST *)pc->actuallist; else
			ni = ((NODEINST **)pc->actuallist)[i];
		if (i != 0) (void)addtoinfstr('/');
		(void)addstringtoinfstr(ntdescribenodeinst(ni));
	}
	return(returninfstr());
}

char *net_describepnet(PNET *pn)
{
	REGISTER PORTPROTO *pp;
	REGISTER INTBIG i;
	REGISTER NETWORK *net;

	net = pn->network;
	if ((pn->flags&POWERNET) != 0) return(_("POWER"));
	if ((pn->flags&GROUNDNET) != 0) return(_("GROUND"));
	if ((pn->flags&EXPORTEDNET) == 0)
	{
		(void)initinfstr();
		(void)addstringtoinfstr(_("INTERNAL"));
		if (net != NONETWORK)
			(void)formatinfstr(" %s", describenetwork(net));
		return(returninfstr());
	}
	if (pn->realportcount == 1)
	{
		pp = (PORTPROTO *)pn->realportlist;
		return(pp->protoname);
	}
	if (pn->realportcount > 1)
	{
		(void)initinfstr();
		for(i=0; i<pn->realportcount; i++)
		{
			pp = ((PORTPROTO **)pn->realportlist)[i];
			if (i > 0) (void)addtoinfstr(',');
			(void)addstringtoinfstr(pp->protoname);
		}
		return(returninfstr());
	}
	net = pn->network;
	return(describenetwork(net));
}

/*
 * Routine to see if the component values "v1" and "v2" are within the prescribed
 * tolerance (in "net_ncc_tolerance").  Returns nonzero if so.
 */
INTSML net_componentequalvalue(float v1, float v2)
{
	float tolerance, largest, diff;

	if (v1 > v2) largest = v1; else largest = v2;
	tolerance = largest * net_ncc_tolerance / 100.0f;
	diff = (float)fabs(v1 - v2);
	if (diff <= tolerance) return(1);
	return(0);
}

/*
 * Routine to return nonzero if component "pc" is a SPICE component.
 */
INTSML net_isspice(PCOMP *pc)
{
	switch (pc->function)
	{
		case NPMETER:
		case NPSOURCE:
		case NPSOURCEV:
		case NPSOURCEC:
		case NPSOURCECM:
		case NPSOURCET:
		case NPSOURCEDC:
		case NPSOURCEAC:
		case NPSOURCEN:
		case NPSOURCEX:
		case NPSOURCEB:
		case NPSOURCES:
			return(1);
	}
	return(0);
}

void net_initdiff(void)
{
	REGISTER INTBIG i;

	if (net_nodeCountMultiplier == 0)
	{
		i = 0;
		net_nodeCountMultiplier = getprime(i++);
		net_portFactorMultiplier = getprime(i++);
		net_functionMultiplier = getprime(i++);
		net_portNetFactorMultiplier = getprime(i++);
		net_portHashFactorMultiplier = getprime(i++);
	}
}

/*
 * routine to get the two facets to be compared.  These must be the only
 * two windows on the display.  Returns nonzero if facets cannot be found.
 */
INTSML net_getfacets(NODEPROTO **facet1, NODEPROTO **facet2)
{
	REGISTER WINDOWPART *w;

	/* get two windows */
	*facet1 = *facet2 = NONODEPROTO;
	for(w = el_topwindowpart; w != NOWINDOWPART; w = w->nextwindowpart)
	{
		if ((w->state&WINDOWTYPE) != DISPWINDOW &&
			(w->state&WINDOWTYPE) != DISP3DWINDOW) continue;
		if (*facet1 == NONODEPROTO) *facet1 = w->curnodeproto; else
		{
			if (*facet2 == NONODEPROTO)
			{
				if (w->curnodeproto != *facet1) *facet2 = w->curnodeproto;
			} else
			{
				if (w->curnodeproto != NONODEPROTO) return(1);
			}
		}
	}
	if (*facet2 == NONODEPROTO) return(1);
	return(0);
}
