/*
 * Electric(tm) VLSI Design System
 *
 * File: edialogs.h
 * Header file for dialog handling
 * Written by: Steven M. Rubin, Static Free Software
 *
 * Copyright (c) 2000 Static Free Software.
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 *
 * Static Free Software
 * 4119 Alpine Road
 * Portola Valley, California 94028
 * info@staticfreesoft.com
 */

/*
 *    DIALOG CONTROL
 * DiaInitDialog(dia)                   initializes dialog from table "dia"
 * item = DiaNextHit()                  obtains next "item" hit in dialog processing
 * chr = DiaGetNextCharacter(&item)     obtains next character (-1 if none, -2 if item)
 * DiaDoneDialog()                      removes and terminates dialog
 *
 *    CONTROL OF STANDARD DIALOG ITEMS
 * DiaSetText(item, msg)                puts "msg" in "item" in the dialog
 * line = DiaGetText(item)              gets text in "item" in the dialog
 * DiaSetControl(item, value)           puts "value" in "item" in the dialog
 * value = DiaGetControl(item)          gets integer in "item" in the dialog
 * valid = DiaValidEntry(item)          returns nonzero if "item" is nonblank in the dialog
 *
 *    SPECIAL DIALOG CONTROL
 * DiaDimItem(item)                     makes "item" dimmed in the dialog
 * DiaUnDimItem(item)                   makes "item" undimmed in the dialog
 * DiaNoEditControl(item)               makes "item" not editable in the dialog
 * DiaEditControl(item)                 makes "item" editable in the dialog
 * DiaOpaqueEdit(item)                  makes "item" not display text
 * DiaCharacterEdit(item)               makes "item" report keystrokes
 * DiaDefaultButton(item)               makes "item" the default button
 *
 *    CONTROL OF POPUP ITEMS
 * DiaSetPopup(item, count, names)      makes "item" a popup with "count" entries in "names"
 * DiaSetPopupEntry(item, entry)        makes popup "item" use entry "entry" as current
 * entry = DiaGetPopupEntry(item)       gets current entry in popup "item"
 *
 *    CONTROL OF SCROLLABLE ITEMS
 * DiaInitTextDialog(item, top, next, done, sortpos, flags)
 * DiaLoadTextDialog(item, top, next, done, sortpos)
 * DiaStuffLine(item, msg)              add line "msg" to item "item"
 * DiaSelectLine(item, l)               select line "l" in item "item"
 * DiaSelectLines(item, c, l)           select "c" lines in "l" in item "item"
 * which = DiaGetCurLine(item)          get current line number in scroll item "item"
 * *which = DiaGetCurLines(item)        get array of selected lines in scroll item "item"
 * msg = DiaGetScrollLine(item, l)      get line "l" in scroll item "item"
 * DiaSetScrollLine(item, l, msg)       set line "l" in scroll item "item"
 *
 *    USER-CONTROL OF DIALOG ITEMS
 * DiaItemRect(item, r)                 get rectangle for item "item"
 * DiaPercent(item, p)                  fill item "item" with "p" percent progress
 * DiaRedispRoutine(item, routine)      call "routine" to redisplay item "item"
 * DiaFillPoly(item, x, y, c, r, g, b)  draw "c" points in (x,y) with color (r,g,b)
 * DiaFrameRect(item, r)                draw frame in rectangle "r"
 * DiaDrawLine(item, fx, fy, tx, ty, m) draw line from (fx,fy) to (tx,ty) mode "m"
 * DiaDrawRect(item, rect, r, g, b)     fill rectangle "rect" with "(r,g,b)"
 * DiaGetTextInfo(msg, wid, hei)        get "wid" and "hei" of "msg"
 * DiaPutText(item, msg, x, y)          put text "msg" at (x,y)
 * DiaGetMouse(x, y)                    get current mouse position into "x/y"
 */

#if defined(__cplusplus) && !defined(ALLCPLUSPLUS)
extern "C"
{
#endif

#define OK           1
#define CANCEL       2

#define ITEMTYPE   017
#define BUTTON       0			/* standard push-button */
#define DEFBUTTON    1			/* default push-button */
#define CHECK        2			/* standard check-box */
#define RADIO        3			/* standard radio-button */
#define SCROLL       4			/* scrolling list of text */
#define SCROLLMULTI  5			/* scrolling list of text (multiple lines can be selected) */
#define MESSAGE      6			/* non-editable text */
#define EDITTEXT     7			/* editable text */
#define ICON       010			/* 32x32 icon */
#define USERDRAWN  011			/* open area for user graphics */
#define POPUP      012			/* popup menu */
#define PROGRESS   013			/* progress bar (0-100%) */
#define DIVIDELINE 014			/* dividing line */
#define INACTIVE  0200

/* the last parameter to "DiaInitTextDialog" */
#define SCSELMOUSE      1		/* mouse clicks select in scroll area */
#define SCSELKEY        2		/* key strokes select in scroll area */
#define SCDOUBLEQUIT    4		/* double-click in scroll area exits dialog */
#define SCREPORT        8		/* report selections in the scroll area */
#define SCHORIZBAR     16		/* want horizontal scroll-bar in scroll area */
#define SCSMALLFONT    32		/* want small text in scroll area */
#define SCFIXEDWIDTH   64		/* want fixed-width text in scroll area */

/* the mode to "DiaDrawLine" */
#define DLMODEON      1			/* draw the line on */
#define DLMODEOFF     2			/* draw the line off */
#define DLMODEINVERT  3			/* draw the line inverted */

typedef struct
{
	INTBIG        data;
	RECTAREA      r;
	unsigned char type;
	char         *msg;
} DIALOGITEM;

typedef struct
{
	RECTAREA    windowRect;
	char       *movable;
	INTBIG      translated;
	INTBIG      items;
	DIALOGITEM *list;
} DIALOG;

/* prototypes for dialog routines */
void    DiaCharacterEdit(INTBIG item);
void    DiaDeclareHook(char *terminputkeyword, COMCOMP *getlinecomp, void (*routine)(void));
void    DiaDefaultButton(INTBIG item);
void    DiaDimItem(INTBIG item);
void    DiaDoneDialog(void);
void    DiaDrawLine(INTBIG item, INTBIG fx, INTBIG fy, INTBIG tx, INTBIG ty, INTBIG mode);
void    DiaDrawRect(INTBIG item, RECTAREA *rect, INTBIG r, INTBIG g, INTBIG b);
void    DiaEditControl(INTBIG item);
void    DiaFillPoly(INTBIG item, INTBIG *x, INTBIG *y, INTBIG count, INTBIG r, INTBIG g, INTBIG b);
void    DiaFrameRect(INTBIG item, RECTAREA *r);
INTBIG  DiaGetControl(INTBIG item);
INTBIG  DiaGetCurLine(INTBIG item);
INTBIG *DiaGetCurLines(INTBIG item);
void    DiaGetMouse(INTSML *x, INTSML *y);
INTSML  DiaGetNextCharacter(INTBIG *item);
INTBIG  DiaGetPopupEntry(INTBIG item);
char   *DiaGetScrollLine(INTBIG item, INTBIG which);
char   *DiaGetText(INTBIG item);
void    DiaGetTextInfo(char *msg, INTBIG *wid, INTBIG *hei);
INTSML  DiaInitDialog(DIALOG *dialog);
void    DiaInitTextDialog(INTBIG item, INTSML (*toplist)(char **),
			char *(*nextinlist)(void), void (*donelist)(void),
			INTBIG sortpos, INTBIG flags);
void    DiaItemRect(INTBIG item, RECTAREA *rect);
void    DiaLoadTextDialog(INTBIG item, INTSML (*toplist)(char **), char *(*nextinlist)(void),
			void (*donelist)(void), INTBIG sortpos);
INTBIG  DiaNextHit(void);
void    DiaNoEditControl(INTBIG item);
void    DiaOpaqueEdit(INTBIG item);
void    DiaPercent(INTBIG item, INTBIG p);
void    DiaPutText(INTBIG item, char *msg, INTBIG x, INTBIG y);
void    DiaRedispRoutine(INTBIG item, void (*routine)(RECTAREA*));
void    DiaSelectLine(INTBIG item, INTBIG which);
void    DiaSelectLines(INTBIG item, INTBIG count, INTBIG *lines);
void    DiaSetControl(INTBIG item, INTBIG value);
void    DiaSetPopup(INTBIG item, INTBIG count, char **names);
void    DiaSetPopupEntry(INTBIG item, INTBIG entry);
void    DiaSetScrollLine(INTBIG item, INTBIG which, char *msg);
void    DiaSetText(INTBIG item, char *msg);
void    DiaStuffLine(INTBIG item, char *line);
void    DiaUnDimItem(INTBIG item);
INTSML  DiaValidEntry(INTBIG item);

/* a modeless dialog */
void    DiaMessageInDialog(char*, ...);

INTSML  DiaNullDlogList(char**);
char   *DiaNullDlogItem(void);
void    DiaNullDlogDone(void);
void    DiaTranslate(DIALOG *dia);

#if defined(__cplusplus) && !defined(ALLCPLUSPLUS)
}
#endif
