/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */


/*
 * Code to write pl (property list) files for tfm files and to
 * execute pltotf to generate the tfm file.
 */

#include <stdio.h>
#include <sys/wait.h>
#include "defs.h"
#include "pfd2tfm.h"
#include "char.h"
#include "extfil.h"

extern char *StrcpyAlloc();
extern char *LocateFileWithPath();
extern int Verbose;
extern char * ProgName;
extern CAFM AfmChar[1];
extern int IsFixedPitch;
extern int XHeight;
extern double HRatio;
extern double WidthDelta;
extern int MaxCharCode;
extern char *BaseFontName;
extern char *tmpdir_default;

/* The not default variables are initialized with a negative value.
 * Use defaults only, if values are negative. */
extern double DesignSize, DesignSizeDef;
extern int Slant, SlantDef;
extern int Stretch, StretchDef;
extern int Shrink, ShrinkDef;
extern int Space, SpaceDef;
extern int ExtraSpace, ExtraSpaceDef;
extern int Quad, QuadDef;
extern int XHeight, XHeightDef;
extern int FontDimenArray[1];

/* Path to locate pltotf. */
char *PlToTfPath;

/* The file name of the pltotf to exexcute, if pltotf will be executed.
   Is null, if pltotf is NOT executed. */
char *PlToTfAbsFileName;

/*
 * WritePlFile
 * ***********
 * Property list writing for a TFM file. At the end pltotf is executed
 * and all together we get a TFM file.
 *
 * pl_fn: pl file name.
 */
void
WritePlFile(pl_fn)
     char *pl_fn;
{
  int i;
  KPX_P kpxp;	
  LIG_P ligp;
  int right_index, lig_index;
  EX_FILES exPlFile; /* Write pl file there. */
  int ki; /* Kerning index in a kerning pair. */

  /* Open the property list file. */
  FExOpen(&exPlFile, EFT_WRITE, 0, pl_fn, "pl");
  if (Verbose > V_QUIET)
    printf ("WritePlFile(): writing property list to \"%s\"\n", exPlFile.ef_fn);
  
  /* Default value business */
  DesignSizeDef = 10.0;

  /*
   * What follows depends on the font: text fonts are
   * treated differently from the symbol font.
   */
  if (strcmp (BaseFontName, "Symbol") == 0) {
    SpaceDef = 0;
    SlantDef = 0;
    StretchDef = 0;
    ShrinkDef = 0;
    ExtraSpaceDef = 0;
    QuadDef = 0;
    XHeight = 0;
  } else {
    SpaceDef = AfmChar[32].c_wx;
    SlantDef = 0;
    if (IsFixedPitch) { /* Fixed pitch font */
      StretchDef = 0;
      ShrinkDef = 0;
      ExtraSpaceDef = SpaceDef; /* double spaces after the period */
      QuadDef = 0;
      XHeightDef = XHeight;
    } else {		/* Variable pitch font */
      StretchDef = AfmChar[32].c_wx/2;
      ShrinkDef =  AfmChar[32].c_wx/3;
      ExtraSpaceDef = AfmChar[32].c_wx/3;
      QuadDef = 1000;
      XHeightDef = XHeight;
    }
  }
  /* Now set all the defaults out, unless there have been values specified
   * by the user. */
  if (DesignSize < 0.0) DesignSize = DesignSizeDef;
  if (Space < 0) Space = SpaceDef;
  if (Slant < 0) Slant = SlantDef;
  if (Stretch < 0) Stretch = StretchDef;
  if (Shrink < 0)  Shrink = ShrinkDef;
  if (ExtraSpace < 0) ExtraSpace = ExtraSpaceDef;
  if (Quad < 0) Quad = QuadDef;
  if (XHeight < 0) XHeight = XHeightDef;

  /* Writing the propertly list starts here ! */
  fprintf (EX_FP(exPlFile), "(COMMENT Property list generated by %s)\n", ProgName);

  fprintf (EX_FP(exPlFile), "(COMMENT so what)\n");
  fprintf (EX_FP(exPlFile), "(DESIGNSIZE R %5.3lf)\n", DesignSize);
  fprintf (EX_FP(exPlFile), "(FONTDIMEN\n");
  fprintf (EX_FP(exPlFile), "    (SLANT        R %5.3lf)\n", Slant/1000.0);
  fprintf (EX_FP(exPlFile), "    (SPACE        R %5.3lf)\n", Space/1000.0);
  fprintf (EX_FP(exPlFile), "    (STRETCH      R %5.3lf)\n", Stretch/1000.0);
  fprintf (EX_FP(exPlFile), "    (SHRINK       R %5.3lf)\n", Shrink/1000.0);
  fprintf (EX_FP(exPlFile), "    (XHEIGHT      R %5.3lf)\n", XHeight/1000.0);
  fprintf (EX_FP(exPlFile), "    (QUAD         R %5.3lf)\n", Quad/1000.0);
  fprintf (EX_FP(exPlFile), "    (EXTRASPACE   R %5.3lf)\n", ExtraSpace/1000.0);
  for (i=8; i<23; i++) {
    if (FontDimenArray[i] >= 0)
      fprintf (EX_FP(exPlFile), "    (PARAMETER D %d R %5.3lf)\n",
	       i, FontDimenArray[i]/1000.0);
  }
  fprintf (EX_FP(exPlFile), ")\n");

  /* Ligature and kerning table written here. */
  fprintf (EX_FP(exPlFile), "(LIGTABLE\n");
  for (i=0; i<=MaxCharCode; i++) {
    if (!AfmChar[i].c_used)
      continue;
    kpxp = AfmChar[i].c_kern;
    ligp = AfmChar[i].c_lig;
    if ((ligp==NULL) && (kpxp==NULL))
      continue; /* neither ligature, nor kerning */
    
    if (Verbose > V_SOME)
      fprintf (stderr, "WritePlFile(): Kern and lig for \"%s\" ['%o]\n",
	       AfmChar[i].c_string, i);
    
    fprintf (EX_FP(exPlFile), "    (LABEL D %d)\n", i);

    /* Ligatures ? */
    while (ligp != NULL) {
      if (CharNameToIndex (ligp->l_cleft) != i)
	Fatal ("WritePlFile(): inconsistency in ligature stuff.");
      right_index  = CharNameToIndex (ligp->l_cright);
      lig_index =    CharNameToIndex (ligp->l_lig_name);
      fprintf (EX_FP(exPlFile), "    (LIG D %d D %d)\n", right_index, lig_index);
      ligp = ligp->l_next;
    }

    /* Kerning pairs next. */
    while (kpxp != NULL) {
      if ((ki = CharNameToIndexNoError(kpxp->k_cright)) != -1)
	fprintf (EX_FP(exPlFile), "    (KRN D %d R %6.5lf)\n",
		 ki, (double)kpxp->k_dist/1000.0);
      else {
	if (Verbose > V_SOME)
	  fprintf (stderr, "WritePlFile(): NO kerning \"%s\" & \"%s\" (2nd char: eliminated)\n",
		   kpxp->k_cleft, kpxp->k_cright);
      }
      kpxp = kpxp->k_next;
    }
    
    /* Done with this character */
    fprintf (EX_FP(exPlFile), "    (STOP)\n");
  }
  fprintf (EX_FP(exPlFile), "    )\n");
  
  /* Write character sizes here: they are relative to the design size */
  for (i=0; i<=MaxCharCode; i++) {
    if (!AfmChar[i].c_used)
      continue;
    fprintf (EX_FP(exPlFile), "(CHARACTER D %d\n", i);
    fprintf (EX_FP(exPlFile), "    (CHARWD R %6.5lf)\n",
	     (double)AfmChar[i].c_wx/1000.0 + WidthDelta/1000.0);
    fprintf (EX_FP(exPlFile), "    (CHARHT R %6.5lf)\n",
	     HRatio*AfmChar[i].c_ury/1000.0);
    fprintf (EX_FP(exPlFile), "    (CHARDP R %6.5lf)\n",
	     -HRatio*AfmChar[i].c_lly/1000.0);
    fprintf (EX_FP(exPlFile), "    )\n");
  }
  if (Verbose > V_QUIET)
    fprintf (stderr, "WritePlFile(): Ligature and kerning table done\n");
  
  /* Done: close pl file. */
  FExClose (&exPlFile);
}

/*
 * ForkOffPlToTf
 * *************
 * Fork off pltotf, after the properly list file was generated.
 *
 * pl_fn: file name, propertly list file.
 * tfm_fn: file name, tfm file.
 */
void
ForkOffPlToTf(pl_fn, tfm_fn)
     char *pl_fn;
     char *tfm_fn;
{
  int childpid; /* Child process id, that is the id of pltotf. */
  int wait_pid; /* Process id returned by wait call. */
  union	wait wait_status;
  char buffer[256]; /* Temporary buffer. */

  if (Verbose > V_QUIET)
    fprintf (stderr, "ForkOffPlToTf(): execute pltotf now (following output by pltotf)\n");

  if ((childpid=fork()) == 0)	{
    /* Child: modify ProgName so it appears to be a different program. */
    sprintf (buffer, "%s-CHILD", ProgName);
    ProgName = StrcpyAlloc(buffer);
    InitProgName(ProgName, tmpdir_default);

    if (execl (PlToTfAbsFileName, "pltotf", pl_fn, tfm_fn, NULL) == -1) {
      perror ("pltotf");
      Fatal ("ForkOffPlToTf() [child]: execl of pltotf failed.");
    }
    exit(0);
  }

  if (childpid == -1)
    Fatal ("ForkOffPlToTf(): fork() of pltotf failed.");

  /* Parent: wait on completion of child. */
  wait_status.w_status = 0;
  if ((wait_pid = wait(&wait_status)) == -1)
    Fatal ("ForkOffPlToTf(): parent, wait() returns -1 (error: no child).");
  if (wait_pid != childpid)
    Fatal3 ("ForkOffPlToTf(): wait() returns wrong process id of %d (child: %d).",
	    wait_pid, childpid);
  /* Now evaluate the result of the wait operation. The value of w_status is
     to be interpreted as follows:
     low byte: termination status of child.
     high_byte: exit code of child. */
  fflush (stdout);
  if (Verbose > V_SOME) {
    fprintf (stderr, "ForkOffPlToTf(): signal byte:      %x (hex)\n",
	     wait_status.w_status & 0xff);
    fprintf (stderr, "ForkOffPlToTf(): exit code pltotf: %x (hex)\n",
	     (wait_status.w_status>>8) & 0xff);
  }
  /* Generates an error here. */
  if (wait_status.w_status != 0) {
    fprintf (stderr, "ForkOffPlToTf(): signal byte:      %x (hex)\n",
	     wait_status.w_status & 0xff);
    fprintf (stderr, "ForkOffPlToTf(): exit code pltotf: %x (hex)\n",
	     (wait_status.w_status>>8) & 0xff);
    Fatal ("ForkOffPlToTf(): fork error.");
  }

  if (Verbose > V_QUIET)
    fprintf (stderr, "ForkOffPlToTf(): pltotf done.\n");
}
