/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

#include <stdio.h>

extern char *malloc();

/* This malloc works in two different modes. If MallocMode == 0 then
   it's just like malloc. If MallocMode == 1 then memory is allocated
   from a fixed size pool. There is no dynamic reallocation.
   This fixed size pool can be flushed. */
int MallocMode = 0;
int MallocAllocated = 0;

char * MallocMode1Pointer; /* Point always to the beginning of the malloc mode 1
			      buffer area. */
int MallocMode1Index; /* Next free byte in that buffer */
int MallocMode1Size; /* Maximum size */
int MallocMode1UsedMaximum = 0; /* Maximum used in mode 1 */

/* If the flag MALLOC_NASTY is defined then after an allocation the area
   allocated is filled with bytes 1F. This means the area is intentionally
   not initialiazed. This should cause programs to fail which rely on
   zeroed memory areas which they should not. */

#define MALLOC_NASTY


/*
 * Malloc
 * ******
 * Like malloc but does include testing whether allocation
 * was ok or not. Generates an error and aborts, if it failed.
 *
 * size:  how much to allocates, in bytes. Normally a construct
 *        like "K*sizeof(XX)".
 * RET: a pointer to the allocated memory.
 */
char*
Malloc (size)
     int size;
{
  char* ret;
  int i;

  /* Branch according to the mode of the memory allocation business. */
  switch (MallocMode) {
    case 0:
      if ((ret=malloc((unsigned) size)) != NULL) {
	MallocAllocated += size;
#ifdef MALLOC_NASTY
	for (i=0; i<size; i++)
	  *(ret+i) = 0x1f;
#endif	
	return (ret);
      }
      Fatal3 ("Malloc(): malloc() failed, allocated successfully so far %d bytes,\
 requested now: %d.", MallocAllocated, size);

    case 1:
      if (MallocMode1Index + size + 3 >= MallocMode1Size)
	Fatal ("Malloc(): Mode 1: ran out of memory.");
      ret = MallocMode1Pointer + MallocMode1Index;
      MallocMode1Index += ((size+3)/4) * 4; /* Start at word boundaries */
      MallocMode1Index += size;
#ifdef MALLOC_NASTY
      for (i=0; i<size; i++)
	*(ret+i) = 0x1f;
#endif	
      return (ret);

    default:
      Fatal2 ("Malloc(): illegal mode %d", MallocMode);
    }
  return (NULL); /* lint */
}

/*
 * InitMallocMode1
 * ***************
 * Initialize memory mode allocation 1. In that mode a fixed size
 * buffer is used for allocation. It is reused when ResetMallocMode1()
 * is called. There is no switch to this mode. You have to change
 * MallocMode for that purpose.
 * size: number of bytes of this buffer.
 */
void
InitMallocMode1(size)
{
  if (MallocMode != 0)
    Fatal ("InitMallocMode1(): MallocMode must be zero.");
  MallocMode1Pointer = Malloc(size);
  MallocMode1Index = 0;
  MallocMode1Size = size;
}

/*
 * ResetMallocMode1
 * ****************
 * Reset memory allocation mode 1.
 */
void
ResetMallocMode1()
{
  /* Compute maximum memory usage under mode 1 */
  if (MallocMode1UsedMaximum < MallocMode1Index)
    MallocMode1UsedMaximum =   MallocMode1Index;

  /* Reset to restart at the beginning of the buffer */
  MallocMode1Index = 0;
}
