/* Copyright 1988 Stephan v. Bechtolsheim */

/* This file is part of the TeXPS Software Package.

The TeXPS Software Package is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY.  No author or distributor
accepts responsibility to anyone for the consequences of using it
or for whether it serves any particular purpose or works at all,
unless he says so in writing.  Refer to the TeXPS Software Package
General Public License for full details.

Everyone is granted permission to copy, modify and redistribute
the TeXPS Software Package, but only under the conditions described in the
TeXPS Software Package General Public License.   A copy of this license is
supposed to have been given to you along with TeXPS Software Package so you
can know your rights and responsibilities.  It should be in a
file named CopyrightLong.  Among other things, the copyright notice
and this notice must be preserved on all copies.  */

/* Some simple directory manipulation functions. */

#include <stdio.h>
#include <sys/param.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>

#if SYS_V == 1
#include <string.h>
#else
#include <strings.h>
#endif

/* External declarations. */
#if SYS_V == 1
extern char *getcwd();
#define index  strchr
#define rindex strrchr
#else
extern char *getwd();
#endif

extern char *StrcpyAlloc();
extern int FileBusinessVerbose;

/* Forward declarations. */
char * GetWd();

/*
 * ChDir
 * *****
 * Change directory to the specified directory.
 * Generate a fatal error if you can't.
 *
 * dir_name: directory to change to.
 */
void
ChDir(dir_name)
     char *dir_name;
{
  char tmp[MAXPATHLEN];
  char *p;

  if (Strlen(dir_name) == 0)
    Fatal ("ChDir(): directory name is empty.");
  if (chdir(dir_name)) {
    p = GetWd();
    Fatal3 ("ChDir(): cannot chdir to \"%s\". (directory \"%s\")",
	    dir_name, p);
  }
  if (FileBusinessVerbose)
    fprintf (stderr, "ChDir(): path name after chdir: \"%s\"\n", GetWd());
}

/*
 * GetWd
 * *****
 * Return a pointer to a string containing the current
 * directory. Generate a fatal error if this is not possible.
 *
 * RET: pointer to string of current working directory.
 */
char *
GetWd()
{
  char tmp[MAXPATHLEN];
  char *p;

#if SYS_V == 1
  p = getcwd(tmp, sizeof(tmp));
#else
  p = getwd(tmp);
#endif

  /* Everything ok, return. */
  if (p != NULL)
    return(StrcpyAlloc(p));

  /* An error occurred: getwd() or getcwd() did not work. */
  Fatal2 ("GetWd(): cannot getwd(), message: \"%s\".", tmp);
}

/* The absolute path name of the home directory of the user. */
char *HomeDirAbsolutePathName = NULL;

/*
 * GetWdCshSyntax
 * ***************
 * Return a pointer to a string containing the current
 * directory. In this case "~" is used at the beginning
 * in case the file name is specified with respect to the home directory
 * of the user who executes the program.
 */
char *
GetWdCshSyntax()
{
  char tmp[MAXPATHLEN];
  char *p;

  if (Strlen(HomeDirAbsolutePathName) == 0) {
    /* The absolute path name is not yet specified! */
  }

  p = GetWd();

  if (strncmp(p, HomeDirAbsolutePathName, strlen(HomeDirAbsolutePathName)) == 0) {
    ReplaceASubString(p, strlen(HomeDirAbsolutePathName), "~");
  }

  return(StrcpyAlloc(p));
}

/*
 * RmDir
 * ******
 * Remove a directory and generate a fatal error if it failed.
 * The directory must be empty.
 *
 * dirn: file name of file to RmDir.
 */
void
RmDir(dirn)
    char *dirn;
{  
  if (Strlen(dirn) == 0)
    Fatal ("RmDir(): empty directory name.");
  if (FileBusinessVerbose)
    fprintf (stderr, "RmDir(): directory \"%s\"\n", dirn);
  if (rmdir(dirn) == -1)
    Fatal2 ("RmDir(): RmDir of \"%s\" failed.", dirn);
}


/* The following two routines are for the manipulation of
   directory and file names when a directory is traversed
   in a recursive fashion. */

/*
 * DirAppendFileName
 * *****************
 * Append a file name (which can be another directory name) to
 * a directory name.
 *
 * dirn: directory name (can be empty).
 * fn: file name.
 * RET: the added on file name.
 */
char *
DirAppendFileName (dirn, fn)
     char *dirn;
     char *fn;
{
  char tmp[512];
  if (Strlen(fn) == 0)
    Fatal ("DirAppendFileName(): fn is empty.");

  /* "." as directory name is treated the same way as if no directory name
     were provided. */
  if (Strcmp(dirn, ".") == 0)
    dirn = NULL;

  if (Strlen(dirn) == 0)
    return(StrcpyAlloc(fn));
  else {
    sprintf (tmp, "%s/%s", dirn, fn);
    return(StrcpyAlloc(tmp));
  }
}

/*
 * DirDropLastFileName
 * *******************
 * Given a file name such as "xx/yy/zz" drop the last
 * component and return it (that is it would return xx/yy in
 * this case).
 *
 * fn: file name.
 * RET: file name minus the last component in it. If the file
 *      name fn did not have a / in it, NULL is returned.
 */
char *
DirDropLastFileName (fn)
     char *fn;
{
  char *slash_ptr;
  char tmp[512];

  if (Strlen(fn) == 0)
    Fatal ("DirDropLastFileName(): fn == 0");

  /* Make a copy of the provided file name. */
  strcpy(tmp, fn);

  /* Return NULL if there is no '/' in the file name. */
  if ((slash_ptr = rindex(tmp, '/')) == NULL)
    return (NULL);

  /* Cut off everything after the slash and return it. */
  *slash_ptr = '\0';
  return (StrcpyAlloc(tmp));
}

/*
 * KeepTrackOfDirectories
 * **********************
 * This procedure, when called, updates a global variable which
 * is used to keep track of current directories.
 *
 * cdir: current directory.
 * chdir: directory to which a change occurs. It can be either of
 *        the following:
 *       .: nothing happens.
 *       ..: the last path is removed from cdir.
 *       xx: the directory xx is appended to cdir.
 * RET: the new value for cdir (cdir itself is not updated).
 */
char *
KeepTrackOfDirectories(cdir, chdir)
     char *cdir;
     char *chdir;
{
  char *slash_ptr;
  char tmp[MAXPATHLEN];

  if (Strlen(cdir) == 0)
    Fatal ("KeepTrackOfDirectories(): cdir is empty.");
  if (Strlen(chdir) == 0)
    Fatal ("KeepTrackOfDirectories(): chdir is empty.");

  if (Strcmp(chdir, ".") == 0) /* no change. */
    return (StrcpyAlloc(cdir));

  if (Strcmp(chdir, "..") == 0) { /* Parent directory: strike everything after the last /. */
    strcpy(tmp, cdir);
    if ((slash_ptr = rindex(tmp, '/')) == NULL)
      Fatal2 ("KeepTrackOfDirectories(): chdir = \"..\", cdir = \"%s\" (no slash)", cdir);
    *slash_ptr = '\0';
    return (StrcpyAlloc(tmp));
  }

  /* Append. */
  if ((slash_ptr = rindex(chdir, '/')) != NULL)
    Fatal2 ("KeepTrackOfDirectories(): chdir = \"%s\" (contains slash)", chdir);
  sprintf (tmp, "%s/%s", cdir, chdir);
  return (StrcpyAlloc(tmp));
}

/*
 * NoDotDotInFileName
 * ******************
 * This function generates a fatal error, if a file name
 * contains '..', that is tries to access it's parent
 * directory.
 *
 * fn: file name
 * extra: extra error message printed, may be NULL.
 */
void
NoDotDotInFileName(fn, extra)
     char *fn;
     char *extra;
{
  if (Strlen(fn) == 0)
    Fatal ("NoDotDotInFileName(): file name empty.");
  if (! IsSubString(fn, ".."))
    return;

  if (Strlen(extra) == 0)
    Fatal ("NoDotDotInFileName(): filename contains \"..\"");
  else
    Fatal2 ("NoDotDotInFileName(): filename \"%s\" contains \"..\"", fn);
}

/*
 * RemoveAndMkDir
 * **************
 * Assume a directorn name dn is given. This dn must either
 * point to an non-existing file, in which case a directory
 * of that name is created or to an existing directory, in which
 * case this directory is removed and a new one is created.
 */
void
RemoveAndMkDir(dn)
     char * dn;
{
  struct stat stat_buf;
  char buffer[256];

  if (Strlen(dn) == 0)
    Fatal ("RemoveAndMkDir(): directory name is empty.");

  if (access(dn, F_OK) == 0) {

    /* Something exists. */
    if (stat(dn, &stat_buf) != 0)
      Fatal2 ("RemoveAndMkDir(): cannot stat \"%s\"", dn);

    if (! stat_buf.st_mode & S_IFDIR)
      Fatal2 ("RemoveAndMkDir(): \"%s\" exists, not a directory.", dn);

    sprintf (buffer, "/bin/rm -rf %s", dn);
    SafeSystemCall(buffer);
    /* RmDir(dn); */
  }

  /* Here nothing should be left over. */
  if (access(dn, F_OK) == 0)
    Fatal2 ("RemoveAndMkDir(): second access of \"%s\"", dn);

  if (mkdir(dn, 0755))
    Fatal2 ("RemoveAndMkDir(): mkdir of \"%s\" failed", dn);
}
