/* SIMPLE - Simple Is a Macro Processing Language Element */
/* Copyright (c) 1998 David A. Madore (david.madore@ens.fr) */

/*
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version
 * 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty
 * of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See
 * the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

/*
 * This file contains the routines to handle token lists.
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "simple.h"
#include "token.h"

#define ALLUNIT 128

void
make_empty_list(token_list *tl)
     /* ``Create'' a new (hence empty) token list. */
{
  tl->l=NULL;
  tl->size=0;
  tl->allsize=0;
}

char
is_empty(token_list *tl)
     /* Test to see whether the token list is empty. */
{
  return (tl->size==0);
}

void
append_token(token_list *tl,token tk)
     /* Append token tk at the end of list tl. */
{
  if (tl->size>=tl->allsize) {
    if ((tl->l=REALLOC(tl->l,(tl->allsize+=ALLUNIT)*sizeof(token)))==NULL)
      fatal(0,"Out of memory!");
  }
  tl->l[tl->size++]=tk;
}

void
ensure_length(token_list *tl,int len)
     /* Force the allocated size of tl to be at least len. */
{
  if (tl->allsize>=len) return;
  if ((tl->l=REALLOC(tl->l,len*sizeof(token)))==NULL)
    fatal(0,"Out of memory!");
  tl->allsize=len;
}

token
first_token(token_list *tl)
     /* Return the first token of the list, which must be non empty. */
{
  if (tl->size==0)
    fatal(1,"first_token() called on empty list!");
  return tl->l[0];
}

void
remove_first(token_list *tl)
     /* Remove the first token of the list, which must be non empty. */
{
#if !defined(HAVE_MEMMOVE)
  int i;
#endif
  if (tl->size==0)
    fatal(1,"remove_first() called on empty list!");
  tl->size--;
#if defined(HAVE_MEMMOVE)
  memmove(tl->l,tl->l+1,tl->size*sizeof(token));
#else
  for (i=0;i<tl->size;i++)
    tl->l[i]=tl->l[i+1];
#endif
}

void
remove_last(token_list *tl)
     /* Remove the last token of the list, which must be non empty. */
{
  if (tl->size==0)
    fatal(1,"remove_first() called on empty list!");
  tl->size--;
}

void
free_list(token_list *tl)
     /* Release memory occupied by list and make it empty. */
{
  FREE(tl->l);
  tl->l=NULL;
  tl->size=0;
  tl->allsize=0;
}

void
copy_list(token_list *td,token_list *ts)
     /* Copy list ts to list td. */
{
  if (((td->l=REALLOC(td->l,ts->allsize*sizeof(token)))==NULL)&&(ts->allsize>0))
    fatal(0,"Out of memory!");
  td->allsize=ts->allsize;
  td->size=ts->size;
  if (td->allsize==0) td->l=NULL; /* libc should take care of that */
  if (td->size>0) {
    memcpy(td->l,ts->l,td->size*sizeof(token));
  }
}

void
prepend_list(token_list *td,token_list *ts)
     /* Insert ts at the beginning of td. */
{
  int total;
#if !defined(HAVE_MEMMOVE)
  int i;
#endif
  if (ts->size==0) return;
  total=((td->size+ts->size)/ALLUNIT+1)*ALLUNIT;
  if ((td->l=REALLOC(td->l,total*sizeof(token)))==NULL)
    fatal(0,"Out of memory!");
  td->allsize=total;
#if defined(HAVE_MEMMOVE)
  memmove(td->l+ts->size,td->l,td->size*sizeof(token));
#else
  for (i=td->size-1;i>=0;i--)
    td->l[i+ts->size]=td->l[i];
#endif
  memcpy(td->l,ts->l,ts->size*sizeof(token));
  td->size+=ts->size;
}

void
append_list(token_list *td,token_list *ts)
     /* Insert ts at the end of td. */
{
  int total;
  if (ts->size==0) return;
  total=((td->size+ts->size)/ALLUNIT+1)*ALLUNIT;
  if ((td->l=REALLOC(td->l,total*sizeof(token)))==NULL)
    fatal(0,"Out of memory!");
  td->allsize=total;
  memcpy(td->l+td->size,ts->l,ts->size*sizeof(token));
  td->size+=ts->size;
}

char
compare_list(token_list *t1,token_list *t2)
     /* Returns 1 if the two token lists are identical, 0 otherwise. */
{
  int i;
  if (t1->size!=t2->size) return 0;
  for (i=0;i<t1->size;i++)
    if (t1->l[i]!=t2->l[i]) return 0;
  return 1;
}

unsigned char *
token_list_to_string(token_list *tl)
     /* Convert the token list to a dynamically allocated string,
      * producing an error if the list contains a special token. */
{
  unsigned char *s;
  int i;
  if ((s=malloc(tl->size+1))==NULL)
    fatal(0,"Out of memory!");
  for (i=0;i<tl->size;i++) {
    if ((tl->l[i]>=FIRST_SPECIAL)||(tl->l[i]<=0))
      fatal(0,"Expected a string (with no special token).");
    s[i]=tl->l[i];
  }
  s[i]=0; /* Ooops! Had forgotten that... */
  return s;
}
