/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains routines that control access to a device.
 *	It supports the requirements of both the DVIOUT program and
 *	the print symbiont.
 *
 *	These routines support buffered output I/O and non-buffered
 *	input I/O. In all cases, I/O termination is handled by an AST.
 */

#include descrip
#include jpidef
#include dvidef
#include devdef
#include ttdef
#ifndef TT$C_BAUD_38400
#define TT$C_BAUD_38400 (TT$C_BAUD_19200+1)
#endif
#include tt2def
#include dcdef
#include psldef
#include msgdef
#include syidef
#include "ast.h"
#include "vms.h"
#include "iostatus.h"
#include "vmsio.h"

#define MAX_DEV_BUF_SIZE  1580	/* 1584 is VMS default maximum
				   (SYSGEN option MAXBUF) */

struct DCB *Connect_Device (Name, Buffer_Size, N_Buffers, Status)
char *Name;
unsigned short Buffer_Size, N_Buffers;
unsigned long Status[];
{
	auto   struct DCB *Dcb_Ptr;
	auto   unsigned long Sys_Status;
	auto   unsigned int Index;
	static struct dsc$descriptor Dev_Name_Desc;
	static unsigned long MaxBuf = 0, DVI_Status[2], Buffer[3], Device_Spl;
#define SS_IO_Status ((struct Set_Sense_IO_Status_Block *) &DVI_Status[0])
	static unsigned short Device_Name_Length;
	static struct Item_List DVI_Spl_List[] = {
                { 4, DVI$_SPL + DVI$M_SECONDARY, &Device_Spl, 0 },
		{ 0, 0, 0, 0 }
	}, DVI_List[] = {
		{ 4, DVI$_DEVCHAR, 0, 0 },
		{ 4, DVI$_DEVCHAR2, 0, 0 },
		{ 4, DVI$_DEVDEPEND, 0, 0 },
		{ 4, DVI$_DEVDEPEND2, 0, 0 },
		{ 4, DVI$_DEVCLASS, 0, 0 },
		{ 4, DVI$_DEVTYPE, 0, 0 },
		{ 4, DVI$_DEVBUFSIZ, 0, 0 },
		{ 0, DVI$_DEVNAM, 0, &Device_Name_Length },
		{ 0, 0, 0, 0 }
	}, SYI_List[] = {
		{ 4, SYI$_MAXBUF, &MaxBuf, 0 },
		{ 0, 0, 0, 0 }
	};
	static struct {
		unsigned char Baud_Code;
		unsigned long Baud_Rate;
	} Baud_Table[] = {
		{ TT$C_BAUD_50, 50 },      { TT$C_BAUD_75, 75 },
		{ TT$C_BAUD_110, 110 },    { TT$C_BAUD_134, 134 },
		{ TT$C_BAUD_150, 150 },    { TT$C_BAUD_300, 300 },
		{ TT$C_BAUD_600, 600 },    { TT$C_BAUD_1200, 1200 },
		{ TT$C_BAUD_1800, 1800 },  { TT$C_BAUD_2000, 2000 },
		{ TT$C_BAUD_2400, 2400 },  { TT$C_BAUD_3600, 3600 },
		{ TT$C_BAUD_4800, 4800 },  { TT$C_BAUD_7200, 7200 },
		{ TT$C_BAUD_9600, 9600 },  { TT$C_BAUD_19200, 19200 },
		{ TT$C_BAUD_38400, 38400 }
	};
	extern struct DCB *Dcb_Alloc();
	extern unsigned long Enable_Mailbox_Interrupt(), Sys$GetDvIW();
	extern unsigned long Lib_Asn_Wth_Mbx(), Lib$Get_EF(), Sys$DAssgn();
	extern unsigned long Sys$GetSyIW(), Sys$QIOW();
	extern int Check_System_Status();
	globalvalue IO$_SENSEMODE;
/*
 *	Get system's maximum terminal i/o buffer size:
 */
	if (MaxBuf == 0) {
		Sys_Status = Sys$GetSyIW (0, 0, 0, SYI_List, DVI_Status, 0, 0);
		if ((Sys_Status & 0x01) == 0 || (DVI_Status[0] & 0x01) == 0)
			MaxBuf = MAX_DEV_BUF_SIZE;
	}
/*
 *	Allocate a control block:
 */
	Dcb_Ptr = Dcb_Alloc ((Buffer_Size == 0 || Buffer_Size > MaxBuf) ?
			      MaxBuf : Buffer_Size, N_Buffers);
/*
 *	Assign a channel and a mailbox to the device:
 */
	Make_VMS_Descriptor (Name, &Dev_Name_Desc);
	Sys_Status = Lib_Asn_Wth_Mbx (&Dev_Name_Desc, 0, 0, &Dcb_Ptr->dcb$w_chan,
				      &Dcb_Ptr->dcb$w_mbxchan);
	Status[0] = Sys_Status;
	if (Check_System_Status (Sys_Status) == 0) {
		if (Dcb_Ptr->dcb$w_mbxchan != 0)
			Sys$DAssgn (Dcb_Ptr->dcb$w_mbxchan);
		Free_Dcb (Dcb_Ptr);
		return (0);
	}
/*
 *	Obtain the device characteristics:
 */
	Device_Spl = 0;
	Sys_Status = Sys$GetDvIW (0, Dcb_Ptr->dcb$w_chan, 0, DVI_Spl_List, DVI_Status,
				  0, 0, 0);
	for (Index = 0; Index < 8; Index++)
		DVI_List[Index].Item_Code = (DVI_List[Index].Item_Code & ~1) + Device_Spl;
	for (Index = 0; Index < 7; Index++)
		DVI_List[Index].Buffer_Addr = &Dcb_Ptr->dcb$l_devchar[Index];
	DVI_List[7].Buffer_Addr = Dcb_Ptr->dcb$b_devname;
	DVI_List[7].Buffer_Len = sizeof (Dcb_Ptr->dcb$b_devname) - 1;
	Sys_Status = Sys$GetDvIW (0, Dcb_Ptr->dcb$w_chan, 0, DVI_List, DVI_Status,
				  0, 0, 0);
	Dcb_Ptr->dcb$b_devname[Device_Name_Length] = '\0';
	Dcb_Ptr->dcb$l_devchar[7] = 0;
	if (Check_System_Status (Sys_Status) != 0 && Check_System_Status (DVI_Status[0]) != 0 &&
	    (Dcb_Ptr->dcb$l_devchar[0] & DEV$M_TRM) != 0 &&
	    Check_System_Status (Sys_Status = Sys$QIOW (0, Dcb_Ptr->dcb$w_chan, IO$_SENSEMODE, SS_IO_Status,
							0, 0, Buffer, sizeof (Buffer), 0, 0, 0, 0)) != 0 &&
	    Check_System_Status (SS_IO_Status->Status) != 0) {
		for (Index = 0; Index < sizeof (Baud_Table) / sizeof (*Baud_Table); Index++)
		if (Baud_Table[Index].Baud_Code == SS_IO_Status->Transmit_Speed) {
			Dcb_Ptr->dcb$l_devchar[7] = Baud_Table[Index].Baud_Rate;
			break;
		}
	}
/*
 *	Obtain an event flag for device reads and set up the read terminator
 *	mask descriptor:
 */
	if (Check_System_Status (Lib$Get_EF (&Dcb_Ptr->dcb$l_evf)) == 0)
		Dcb_Ptr->dcb$l_evf = 0;
	if (Check_System_Status (Lib$Get_EF (&Dcb_Ptr->dcb$l_mbxevf)) == 0)
		Dcb_Ptr->dcb$l_mbxevf = 0;
	Dcb_Ptr->dcb$l_maskdesc.dsc$w_length = sizeof (Dcb_Ptr->dcb$l_tmask);
	Dcb_Ptr->dcb$l_maskdesc.dsc$a_pointer = &Dcb_Ptr->dcb$l_tmask[0];
/*
 *	Enable the Mailbox interrupts on the device:
 */
	Check_System_Status (Enable_Mailbox_Interrupt (Dcb_Ptr));
	return (Dcb_Ptr);
}
#undef SS_IO_Status

/*
 *	Routine Disconnect_Device terminates activity on the device.
 *	Note that all device I/O must be completed prior to disconnection.
 */

Disconnect_Device (Dcb_Ptr)
struct DCB *Dcb_Ptr;
{
	auto   unsigned long Sys_Status;
	extern unsigned long Lib_DAssgn(), Sys$DAssgn();
	extern int Check_System_Status(), strcmp();
/*
 *	Deassign the device and mailbox channels, then evaporate
 *	the control block:
 */
	Check_System_Status (Sys$DAssgn (Dcb_Ptr->dcb$w_mbxchan));
	Check_System_Status (Lib_DAssgn (&Dcb_Ptr->dcb$w_chan, &1));
	Free_Dcb (Dcb_Ptr);
}

Free_Dcb (Dcb_Ptr)
struct DCB *Dcb_Ptr;
{
	auto   struct BCB *Bcb_Ptr;
	extern unsigned long Lib$Free_EF();
	extern int Check_System_Status();
/*
 *	Evaporate the device buffers and their control blocks, then the
 *	main control block; free all non-zero event flags:
 */
	if (Dcb_Ptr->dcb$l_mbxevf != 0)
		Check_System_Status (Lib$Free_EF (&Dcb_Ptr->dcb$l_mbxevf));
	if (Dcb_Ptr->dcb$l_evf != 0)
		Check_System_Status (Lib$Free_EF (&Dcb_Ptr->dcb$l_evf));
	while ((Bcb_Ptr = Dcb_Ptr->dcb$l_bcb) != 0) {
		if (Bcb_Ptr->bcb$l_evf != 0)
			Check_System_Status (Lib$Free_EF (&Bcb_Ptr->bcb$l_evf));
		Dcb_Ptr->dcb$l_bcb = Bcb_Ptr->bcb$l_nxt;
		Mem_Free (Bcb_Ptr);
	}
	Mem_Free (Dcb_Ptr);
}

/*
 *	Routine Set_Device_Characteristics sets the characteristics for
 *	a terminal:
 */

unsigned long Set_Device_Characteristics (Attr_Str, Dcb_Ptr)
char *Attr_Str;
struct DCB *Dcb_Ptr;
{
	struct Loc_Table {
		char *Characteristic_Name;
		unsigned char Data_Type;
#define BOOLEAN    0
#define REV_BOOL   1
#define LOW_NIBBLE 2
#define BYTE       4
#define WORD       5
#define CLASS      6
#define TYPE       7
#define BAUD       8
		unsigned char Offset;
		unsigned short Mask;
	};
	struct Match_Table {
		char *Name;
		unsigned long Value;
	};
	auto   struct Loc_Table *Loc_Ptr;
	auto   struct Match_Table *Match_Ptr;
	auto   unsigned long Sys_Status, Attr_Value;
	auto   int Index;
	auto   char *In_Ptr, *Out_Ptr;
	static struct Set_Sense_IO_Status_Block IO_Status;
	static unsigned char Characteristics_Buffer[24];
#define P3 *(unsigned long *) &Characteristics_Buffer[12]
#define P4 *(unsigned long *) &Characteristics_Buffer[16]
#define P5 *(unsigned long *) &Characteristics_Buffer[20]
	static char Name_Buffer[20], Value_Buffer[20];
	static struct Loc_Table Characteristics_Table[] = {
		{ "altdispar",   BOOLEAN, 20, TT$V_ALTDISPAR },
		{ "altframe",    BOOLEAN, 20, TT$V_ALTFRAME },
		{ "altrpar",     BOOLEAN, 20, TT$V_ALTRPAR },
		{ "altypeahd",   BOOLEAN, 8, TT2$V_ALTYPEAHD },
		{ "ansicrt",     BOOLEAN, 8, TT2$V_ANSICRT },
		{ "app_keypad",  BOOLEAN, 8, TT2$V_APP_KEYPAD },
		{ "autobaud",    BOOLEAN, 8, TT2$V_AUTOBAUD },
		{ "avo",         BOOLEAN, 8, TT2$V_AVO },
		{ "block",       BOOLEAN, 8, TT2$V_BLOCK },
/*		{ "break",       BOOLEAN, 20, TT$V_BREAK },	*/
		{ "brdcst",      REV_BOOL, 4, TT$V_NOBRDCST },
		{ "brdcstmbx",   BOOLEAN, 8, TT2$V_BRDCSTMBX },
		{ "class",       CLASS, 0, 0 },
		{ "crfill",      BYTE, 16, 0 },
/*		{ "dcl_ctrlc",   BOOLEAN, 8, TT2$V_DCL_CTRLC },
		{ "dcl_mailbx",  BOOLEAN, 8, TT2$V_DCL_MAILBX },
		{ "dcl_outbnd",  BOOLEAN, 8, TT2$V_DCL_OUTBND },	*/
		{ "deccrt",      BOOLEAN, 8, TT2$V_DECCRT },
		{ "deccrt2",     BOOLEAN, 8, TT2$V_DECCRT2 },
		{ "dialup",      BOOLEAN, 8, TT2$V_DIALUP },
		{ "disconnect",  BOOLEAN, 8, TT2$V_DISCONNECT },
		{ "disparerr",   BOOLEAN, 20, TT$V_DISPARERR },
		{ "dma",         BOOLEAN, 8, TT2$V_DMA },
		{ "drcs",        BOOLEAN, 8, TT2$V_DRCS },
		{ "edit",        BOOLEAN, 8, TT2$V_EDIT },
		{ "echo",        REV_BOOL, 4, TT$V_NOECHO },
		{ "eightbit",    BOOLEAN, 4, TT$V_EIGHTBIT },
		{ "escape",      BOOLEAN, 4, TT$V_ESCAPE },
		{ "fallback",    BOOLEAN, 8, TT2$V_FALLBACK },
		{ "formfeed",    BOOLEAN, 4, TT$V_MECHFORM },
		{ "frame_size",  LOW_NIBBLE, 20, 0 },
		{ "halfdup",     BOOLEAN, 4, TT$V_HALFDUP },
		{ "hangup",      BOOLEAN, 8, TT2$V_HANGUP },
		{ "holdscreen",  BOOLEAN, 4, TT$V_HOLDSCREEN },
		{ "hostsync",    BOOLEAN, 4, TT$V_HOSTSYNC },
		{ "insert",      BOOLEAN, 8, TT2$V_INSERT },
		{ "lffill",      BYTE, 17, 0 },
		{ "localecho",   BOOLEAN, 8, TT2$V_LOCALECHO },
		{ "lower",       BOOLEAN, 4, TT$V_LOWER },
		{ "mbxdsabl",    BOOLEAN, 4, TT$V_MBXDSABL },
		{ "modem",       BOOLEAN, 4, TT$V_MODEM },
		{ "modhangup",   BOOLEAN, 8, TT2$V_MODHANGUP },
		{ "odd",         BOOLEAN, 20, TT$V_ODD },
		{ "oper",        BOOLEAN, 4, TT$V_OPER },
		{ "page_length", BYTE, 7, 0 },
		{ "page_width",  WORD, 2, 0 },
		{ "parity",      BOOLEAN, 20, TT$V_PARITY },
		{ "pasthru",     BOOLEAN, 8, TT2$V_PASTHRU },
		{ "printer",     BOOLEAN, 8, TT2$V_PRINTER },
		{ "readsync",    BOOLEAN, 4, TT$V_READSYNC },
		{ "receive",     BAUD, 13, 0 },
		{ "regis",       BOOLEAN, 8, TT2$V_REGIS },
		{ "scope",       BOOLEAN, 4, TT$V_SCOPE },
		{ "script",      BOOLEAN, 4, TT$V_SCRIPT },
		{ "secure",      BOOLEAN, 8, TT2$V_SECURE },
		{ "setspeed",    BOOLEAN, 8, TT2$V_SETSPEED },
		{ "sixel",       BOOLEAN, 8, TT2$V_SIXEL },
		{ "syspwd",      BOOLEAN, 8, TT2$V_SYSPWD },
		{ "tab",         BOOLEAN, 4, TT$V_MECHTAB },
		{ "transmit",    BAUD, 12, 0 },
		{ "ttsync",      BOOLEAN, 4, TT$V_TTSYNC },
		{ "twostop",     BOOLEAN, 20, TT$V_TWOSTOP },
		{ "type",        TYPE, 1, 0 },
		{ "typeahd",     REV_BOOL, 4, TT$V_NOTYPEAHD },
		{ "wrap",        BOOLEAN, 4, TT$V_WRAP },
		{ "xon",         BOOLEAN, 8, TT2$V_XON },
		{ 0 }
	};
	static struct Match_Table Class_Table[] = {
		{ "disk", DC$_DISK },
		{ "tape", DC$_TAPE },
		{ "scom", DC$_SCOM },
		{ "card", DC$_CARD },
		{ "term", DC$_TERM },
		{ "lp",   DC$_LP },
		{ "realtime", DC$_REALTIME },
		{ "mailbox", DC$_MAILBOX },
		{ "workstation", DC$_WORKSTATION },
		{ "bus", DC$_BUS },
		{ "misc", DC$_MISC },
		{ 0 }
	};
	static struct Match_Table Type_Table[] = {
		{ "unknown", TT$_UNKNOWN },
		{ "ft1", TT$_FT1 },
		{ "ft2", TT$_FT2 },
		{ "ft3", TT$_FT3 },
		{ "ft4", TT$_FT4 },
		{ "ft5", TT$_FT5 },
		{ "ft6", TT$_FT6 },
		{ "ft7", TT$_FT7 },
		{ "ft8", TT$_FT8 },
		{ "lax", TT$_LAX },
		{ "la12", TT$_LA12 },
		{ "la24", TT$_LA24 },
		{ "la36", TT$_LA36 },
		{ "la34", TT$_LA34 },
		{ "la38", TT$_LA38 },
		{ "la80", TT$_LA80 },
		{ "la84", TT$_LA84 },
		{ "la100", TT$_LA100 },
		{ "la120", TT$_LA120 },
		{ "la210", TT$_LA210 },
		{ "lqp02", TT$_LQP02 },
		{ "ln01k", TT$_LN01K },
		{ "ln03", TT$_LN03 },
		{ "vt05", TT$_VT05 },
		{ "vt5x", TT$_VT5X },
		{ "vt52", TT$_VT52 },
		{ "vt55", TT$_VT55 },
		{ "vt80", TT$_VT80 },
		{ "vk100", TT$_VK100 },
		{ "vt100", TT$_VT100 },
		{ "vt101", TT$_VT101 },
		{ "vt102", TT$_VT102 },
		{ "vt105", TT$_VT105 },
		{ "vt125", TT$_VT125 },
		{ "vt131", TT$_VT131 },
		{ "vt132", TT$_VT132 },
		{ "vt173", TT$_VT173 },
		{ "vt200_series", TT$_VT200_SERIES },
		{ "pro_series", TT$_PRO_SERIES },
		{ "tq_bts", TT$_TQ_BTS },
		{ "tek401x", TT$_TEK401X },
		{ 0 }
	};
	static struct Match_Table Baud_Table[] = {
		{ "50", TT$C_BAUD_50 },
		{ "75", TT$C_BAUD_75 },
		{ "110", TT$C_BAUD_110 },
		{ "134", TT$C_BAUD_134 },
		{ "150", TT$C_BAUD_150 },
		{ "300", TT$C_BAUD_300 },
		{ "600", TT$C_BAUD_600 },
		{ "1200", TT$C_BAUD_1200 },
		{ "1800", TT$C_BAUD_1800 },
		{ "2000", TT$C_BAUD_2000 },
		{ "2400", TT$C_BAUD_2400 },
		{ "3600", TT$C_BAUD_3600 },
		{ "4800", TT$C_BAUD_4800 },
		{ "7200", TT$C_BAUD_7200 },
		{ "9600", TT$C_BAUD_9600 },
		{ "19200", TT$C_BAUD_19200 },
		{ "38400", TT$C_BAUD_38400 },
		{ 0 }
	};
	extern unsigned long Sys$QIOW();
	extern long atol();
	extern int strcmp();
	globalvalue IO$_SENSEMODE, IO$_SETMODE;
/*
 *	Get the current device settings, initialize:
 */
	Sys_Status = Sys$QIOW (0, Dcb_Ptr->dcb$w_chan, IO$_SENSEMODE, &IO_Status, 0, 0,
			       Characteristics_Buffer, 12, 0, 0, 0, 0);
	if ((Sys_Status & 0x01) == 0 || ((Sys_Status = IO_Status.Status) & 0x01) == 0)
		return (Sys_Status);
	P3 = 0;
	Characteristics_Buffer[12] = IO_Status.Transmit_Speed;
	Characteristics_Buffer[13] = IO_Status.Receive_Speed;
	P4 = 0;
	Characteristics_Buffer[16] = IO_Status.CR_Fill_Count;
	Characteristics_Buffer[17] = IO_Status.LF_Fill_Count;
	P5 = (unsigned long) IO_Status.Parity_Flags << 4;
/*
 *	Parse the name string for each specification. Specifications
 *	are separated by a comma; specification values follow the
 *	specification name by an equals sign, e.g.:
 *	"crfill=3,hostsync=on,brdcst=off,receive=1200,transmit=1200".
 *
 *	Note that to change the parity setting, "altrpar=on" must be
 *	specified.
 */
	for (In_Ptr = Attr_Str; *In_Ptr != '\0'; ) {
		for (Out_Ptr = &Name_Buffer[0]; *In_Ptr != '\0' && *In_Ptr != '=' && *In_Ptr != ','; In_Ptr++)
		if (Out_Ptr < &Name_Buffer[19] && *In_Ptr != ' ')
			*Out_Ptr++ = *In_Ptr;
		*Out_Ptr = '\0';
		if (*In_Ptr == '=')
			In_Ptr++;
		for (Out_Ptr = &Value_Buffer[0]; *In_Ptr != '\0' && *In_Ptr != ','; In_Ptr++)
		if (Out_Ptr < &Value_Buffer[19] && *In_Ptr != ' ')
			*Out_Ptr++ = *In_Ptr;
		*Out_Ptr = '\0';
		if (*In_Ptr == ',')
			In_Ptr++;
/*
 *	Locate the name string in the table:
 */
		for (Loc_Ptr = &Characteristics_Table[0]; Loc_Ptr->Characteristic_Name != 0; Loc_Ptr++)
		if (strcmp (Loc_Ptr->Characteristic_Name, Name_Buffer) == 0) {
			Index = Loc_Ptr->Offset;
			switch (Loc_Ptr->Data_Type) {

			case BOOLEAN:
				if (strcmp (Value_Buffer, "on") == 0)
					*(unsigned long *) &Characteristics_Buffer[Index] |= (1 << Loc_Ptr->Mask);
				else if (strcmp (Value_Buffer, "off") == 0)
					*(unsigned long *) &Characteristics_Buffer[Index] &= ~(1 << Loc_Ptr->Mask);
				break;

			case REV_BOOL:
				if (strcmp (Value_Buffer, "off") == 0)
					*(unsigned long *) &Characteristics_Buffer[Index] |= (1 << Loc_Ptr->Mask);
				else if (strcmp (Value_Buffer, "on") == 0)
					*(unsigned long *) &Characteristics_Buffer[Index] &= ~(1 << Loc_Ptr->Mask);
				break;

			case LOW_NIBBLE:
				Characteristics_Buffer[Index] = (Characteristics_Buffer[Index] & 0xF0) |
								(atol (Value_Buffer) & 0x0F);
				break;

			case BYTE:
				Characteristics_Buffer[Index] = atol (Value_Buffer) & 0xFF;
				break;

			case WORD:
				*(unsigned short *) &Characteristics_Buffer[Index] = atol (Value_Buffer) & 0xFFFF;
				break;

			case CLASS:
				Match_Ptr = &Class_Table[0];
				goto Search_Match;

			case TYPE:
				Match_Ptr = &Type_Table[0];
				goto Search_Match;

			case BAUD:
				Match_Ptr = &Baud_Table[0];
Search_Match:			for (; Match_Ptr->Name != 0; Match_Ptr++)
				if (strcmp (Match_Ptr->Name, Value_Buffer) == 0) {
					Characteristics_Buffer[Index] = Match_Ptr->Value;
					break;
				}
			}
			break;
		}
	}
/*
 *	Check if values were specified for CRFILL or LFFILL, set those
 *	bits if so:
 */
	if (Characteristics_Buffer[16] != 0)
		*(unsigned long *) &Characteristics_Buffer[4] |= TT$M_CRFILL;
	if (Characteristics_Buffer[17] != 0)
		*(unsigned long *) &Characteristics_Buffer[4] |= TT$M_LFFILL;
/*
 *	Set the characteristics. Note that the various parameters obtained
 *	by the $GETDVI system service when the device was connected may no
 *	longer reflect the changes made here. They are left in their original
 *	state so that the characteristics can be restored when the device is
 *	disconnected, if the caller so chooses.
 */
	Sys_Status = Sys$QIOW (0, Dcb_Ptr->dcb$w_chan, IO$_SETMODE, &IO_Status, 0, 0,
			       Characteristics_Buffer, 12, P3, P4, P5, 0);
	if ((Sys_Status & 0x01) != 0)
		Sys_Status = IO_Status.Status;
	return (Sys_Status);
}
#undef P3
#undef P4
#undef P5

unsigned long Flush_Device_IO (Dcb_Ptr)
struct DCB *Dcb_Ptr;
{
	extern unsigned long Initiate_Write();

	return (Initiate_Write (Dcb_Ptr->dcb$l_abf));
}

Wait_for_Device_IO (Dcb_Ptr)
struct DCB *Dcb_Ptr;
{
	auto   struct BCB *Bcb_Ptr;
	extern unsigned long Sys$Hiber();

loop:	for (Bcb_Ptr = Dcb_Ptr->dcb$l_bcb; Bcb_Ptr != 0; Bcb_Ptr = Bcb_Ptr->bcb$l_nxt)
	if ((Bcb_Ptr->bcb$w_flags & BCB$M_ACTIVE) != 0) {
		Sys$Hiber ();
		goto loop;
	}
}

/*
 *	Routine Write_Device outputs a block to the device:
 */

unsigned long Write_Device (Block_Ptr, Count, Dcb_Ptr)
char *Block_Ptr;
unsigned short Count;
struct DCB *Dcb_Ptr;
{
	auto   struct BCB *Bcb_Ptr;
	auto   unsigned long Sys_Status;
	auto   unsigned short Cnt;
	auto   char *Ptr;
	extern struct BCB *Get_Buffer();
	extern unsigned long Initiate_Write();
	globalvalue SS$_NORMAL;

	Sys_Status = SS$_NORMAL;
	Ptr = Block_Ptr;
	Bcb_Ptr = Dcb_Ptr->dcb$l_abf;
	Cnt = Count;
	while (Cnt > 0)
	if (Bcb_Ptr->bcb$w_hsz == 0) {
		Sys_Status = Initiate_Write (Bcb_Ptr);
		if ((Sys_Status & 0x01) == 0)
			break;
		Dcb_Ptr->dcb$l_abf = Bcb_Ptr = Get_Buffer (Dcb_Ptr);
	} else {
		*Bcb_Ptr->bcb$l_bfp++ = *Ptr++;
		Bcb_Ptr->bcb$w_hsz--;
		Cnt--;
	}
	return (Sys_Status);
}

unsigned long Initiate_Write (Bcb_Ptr)
struct BCB *Bcb_Ptr;
{
	auto   struct DCB *Dcb_Ptr;
	auto   unsigned long Sys_Status;
	auto   unsigned short Count;
	extern unsigned long Sys$QIO(), Output_AST_Handler();
	extern int Check_System_Status();
	globalvalue SS$_NORMAL, IO$_WRITELBLK, IO$M_NOFORMAT;

	Dcb_Ptr = Bcb_Ptr->bcb$l_dcb;
	if ((Count = Bcb_Ptr->bcb$w_bsz - Bcb_Ptr->bcb$w_hsz) == 0) {
		Bcb_Ptr->bcb$l_ios.Status = Sys_Status = SS$_NORMAL;
		Bcb_Ptr->bcb$l_ios.Byte_Count = 0;
		EnQueue_AST_Locked (AST_DEVICE_WRITE_IO, Dcb_Ptr, Bcb_Ptr, PRI_DEVICE_WRITE_IO);
	} else {
		Bcb_Ptr->bcb$w_flags |= BCB$M_ACTIVE;
		Sys_Status = Sys$QIO (Bcb_Ptr->bcb$l_evf, Dcb_Ptr->dcb$w_chan,
				      IO$_WRITELBLK | IO$M_NOFORMAT, &Bcb_Ptr->bcb$l_ios,
				      &Output_AST_Handler, Bcb_Ptr, Bcb_Ptr->bcb$l_buf,
				      Count, 0, 0, 0, 0);
		if (Check_System_Status (Sys_Status) == 0) {
			Bcb_Ptr->bcb$w_flags &= ~BCB$M_ACTIVE;
			Bcb_Ptr->bcb$l_bfp = Bcb_Ptr->bcb$l_buf;
			Bcb_Ptr->bcb$w_hsz = Bcb_Ptr->bcb$w_bsz;
		}
	}
	return (Sys_Status);
}

unsigned long Write_Device_Immediate (Buffer, Buffer_Size, Dcb_Ptr)
struct DCB *Dcb_Ptr;
{
	auto   unsigned long Sys_Status;
	static struct Write_IO_Status_Block IO_Status;
	extern unsigned long Sys$QIOW();
	globalvalue IO$_WRITELBLK, IO$M_NOFORMAT;

	Sys_Status = Sys$QIOW (0, Dcb_Ptr->dcb$w_chan,
			       IO$_WRITELBLK | IO$M_NOFORMAT, &IO_Status,
			       0, 0, Buffer, Buffer_Size, 0, 0, 0, 0);
	if ((Sys_Status & 0x01) != 0)
		Sys_Status = IO_Status.Status;
	return (Sys_Status);
}

unsigned long Read_Device (Buffer, Buffer_Size, Terminators, TimeOut, Dcb_Ptr)
char *Buffer;
unsigned char *Terminators;
unsigned short Buffer_Size;
unsigned long TimeOut;
struct DCB *Dcb_Ptr;
{
	auto   unsigned long IO_Function, Sys_Status;
	extern unsigned long Sys$QIO(), Input_AST_Handler();
	globalvalue IO$_READLBLK, IO$M_NOECHO, IO$M_NOFILTR, IO$M_TIMED;
/*
 *	Set up the I/O function code:
 */
	IO_Function = IO$_READLBLK | IO$M_NOECHO | IO$M_NOFILTR;
	if (TimeOut > 0)
		IO_Function |= IO$M_TIMED;
/*
 *	Generate the terminator mask:
 */
	Generate_Terminator_Mask (Terminators, Dcb_Ptr->dcb$l_tmask);
/*
 *	Perform the read. The AST routine will complete it:
 */
	Sys_Status = Sys$QIO (Dcb_Ptr->dcb$l_evf, Dcb_Ptr->dcb$w_chan, IO_Function,
			      &Dcb_Ptr->dcb$l_ios, &Input_AST_Handler, Dcb_Ptr, Buffer,
			      Buffer_Size, TimeOut, &Dcb_Ptr->dcb$l_maskdesc, 0, 0);
	return (Sys_Status);
}

/*
 *	Routine Read_Device_Now performs a read from the specified device,
 *	but returns immediately, either with a completed read, or an
 *	imcomplete one. It returns SS$_TIMEOUT if the buffer is incomplete
 *	and no terminator seen, SS$_NORMAL if a terminator was seen or if
 *	the requested number of characters were read and no terminators
 *	were specified; otherwise it returns the Sys$QIOW status value,
 *	unless the error is SS$_DATAOVERUN, which means that the typeahead
 *	buffer has gotten full prior to the read request. This is
 *	unfortunate, but there is no reasonable recovery from this error,
 *	so SS$_NORMAL is returned in this case.
 */

unsigned long Read_Device_Now (Buffer, Buffer_Size, Terminators, Dcb_Ptr)
char *Buffer;
unsigned char *Terminators;
unsigned short Buffer_Size;
struct DCB *Dcb_Ptr;
{
	auto   struct Read_IO_Status_Block *Stat_Ptr;
	auto   unsigned long Sys_Status;
	extern unsigned long Sys$QIOW();
	globalvalue IO$_READLBLK, IO$M_NOFILTR, IO$M_NOECHO, IO$M_TIMED;
	globalvalue SS$_NORMAL, SS$_TIMEOUT, SS$_DATAOVERUN;
/*
 *	Generate the terminator mask:
 */
	Generate_Terminator_Mask (Terminators, Dcb_Ptr->dcb$l_tmask);
/*
 *	Perform the read; check return status:
 */
	Stat_Ptr = &Dcb_Ptr->dcb$l_ios;
	if (Buffer_Size == 0) {
		Stat_Ptr->Status = SS$_NORMAL;
		Stat_Ptr->Byte_Count = 0;
		Stat_Ptr->Terminator = 0;
		Stat_Ptr->Terminator_Size = 0;
		Sys_Status = SS$_NORMAL;
	} else {
		Sys_Status = Sys$QIOW (Dcb_Ptr->dcb$l_evf, Dcb_Ptr->dcb$w_chan,
				       IO$_READLBLK | IO$M_NOECHO | IO$M_NOFILTR | IO$M_TIMED,
				       Stat_Ptr, 0, 0, Buffer, Buffer_Size, 0,
				       &Dcb_Ptr->dcb$l_maskdesc, 0, 0);
	}
	if ((Sys_Status & 0x01) != 0) {
		Sys_Status = Stat_Ptr->Status;
		if (Sys_Status == SS$_DATAOVERUN ||
		    (Sys_Status == SS$_TIMEOUT && *Terminators == '\0' && Stat_Ptr->Byte_Count == Buffer_Size))
			Sys_Status = SS$_NORMAL;
	}
	return (Sys_Status);
}

Generate_Terminator_Mask (Terminators, Mask)
unsigned char *Terminators;
unsigned long Mask[8];
{
	auto   int Index;
	auto   unsigned char *Ptr;

	for (Index = 0; Index < 8; Index++)
		Mask[Index] = 0;
	for (Ptr = Terminators; *Ptr != '\0'; Ptr++)
		Mask[*Ptr >> 5] |= (1 << (*Ptr & 0x1F));
}

/*
 *	Routine Cancel_Device_IO cancels all outstanding I/O requests
 *	to the device:
 */

unsigned long Cancel_Device_IO (Dcb_Ptr)
struct DCB *Dcb_Ptr;
{
	extern unsigned long Sys$Cancel();
/*
 *	Issue a cancel I/O request. The AST will complete the I/O
 *	normally, to reset the buffer pointers. The active buffer
 *	will remain unchanged.
 */
	return (Sys$Cancel (Dcb_Ptr->dcb$w_chan));
}

/*
 *	These next routines provide information from the various control
 *	blocks to prevent outside routines from having to access the
 *	individual control block items:
 *
 *	Routine Is_a_Terminal returns whether or not the specified
 *	device is a terminal:
 */

int Is_a_Terminal (Dcb_Ptr)
struct DCB *Dcb_Ptr;
{
	return (((Dcb_Ptr->dcb$l_devchar[0] & DEV$M_TRM) == 0) ? 0 : 1);
}

/*
 *	On_a_Modem determines if the device is connected through a modem:
 */

int On_a_Modem (Dcb_Ptr)
struct DCB *Dcb_Ptr;
{
	return (((Dcb_Ptr->dcb$l_devchar[2] & TT$M_MODEM) == 0) ? 0 : 1);
}

char *Device_Name (Dcb_Ptr)
struct DCB *Dcb_Ptr;
{
	return (Dcb_Ptr->dcb$b_devname);
}

unsigned long Device_Speed (Dcb_Ptr)
struct DCB *Dcb_Ptr;
{
	return (Dcb_Ptr->dcb$l_devchar[7]);
}

struct DCB *Device_Control (Bcb_Ptr)
struct BCB *Bcb_Ptr;
{
	return (Bcb_Ptr->bcb$l_dcb);
}

Set_Device_Write_Status (Bcb_Ptr, Status)
struct BCB *Bcb_Ptr;
struct Write_IO_Status_Block *Status;
{
	*Status = Bcb_Ptr->bcb$l_ios;
}

Set_Device_Read_Status (Dcb_Ptr, Status)
struct DCB *Dcb_Ptr;
struct Read_IO_Status_Block *Status;
{
	*Status = Dcb_Ptr->dcb$l_ios;
}

/*
 *	This next routine returns the amount of buffer left for the
 *	current output operation. It is used to detect when the buffer
 *	is almost full in order to circumvent the automatic multiple
 *	buffering of output data, in case a process cannot be held up
 *	waiting for a new buffer to become available:
 */

unsigned short Device_Buffer_Size_Left (Dcb_Ptr)
struct DCB *Dcb_Ptr;
{
	return (Dcb_Ptr->dcb$l_abf->bcb$w_hsz);
}

struct DCB *Dcb_Alloc (Buf_Size, N_Bufs)
unsigned short Buf_Size, N_Bufs;
{
	auto   struct DCB *Dcb_Ptr;
	auto   struct BCB *Bcb_Ptr;
	auto   unsigned short Buf_Count;
	extern char *Mem_Alloc();
	extern struct BCB *Bcb_Alloc();

	Dcb_Ptr = (struct DCB *) Mem_Alloc (DCB$C_BLN);
	Table_Init (Dcb_Ptr, DCB$C_BID, DCB$C_BLN);
	Dcb_Ptr->dcb$w_nbf = N_Bufs;
	Bcb_Ptr = 0;
	for (Buf_Count = N_Bufs; Buf_Count > 0; Buf_Count--)
		Bcb_Ptr = Bcb_Alloc (Buf_Size, Bcb_Ptr, Dcb_Ptr);
	Dcb_Ptr->dcb$l_abf = Dcb_Ptr->dcb$l_bcb = Bcb_Ptr;
	return (Dcb_Ptr);
}

struct BCB *Bcb_Alloc (Buf_Size, Head_Ptr, Dcb_Ptr)
unsigned short Buf_Size;
struct BCB *Head_Ptr;
struct DCB *Dcb_Ptr;
{
	auto   struct BCB *Bcb_Ptr;
	auto   char *Ptr;
	extern char *Mem_Alloc();
	extern unsigned long Lib$Get_EF();

	Bcb_Ptr = (struct BCB *) Mem_Alloc (BCB$C_BLN + Buf_Size);
	Table_Init (Bcb_Ptr, BCB$C_BID, BCB$C_BLN);
	Ptr = (char *) Bcb_Ptr;
	if ((Lib$Get_EF (&Bcb_Ptr->bcb$l_evf) & 0x01) == 0)
		Bcb_Ptr->bcb$l_evf = 0;
	Bcb_Ptr->bcb$l_nxt = Head_Ptr;
	Bcb_Ptr->bcb$l_dcb = Dcb_Ptr;
	Bcb_Ptr->bcb$l_bfp = Bcb_Ptr->bcb$l_buf = &Ptr[BCB$C_BLN];
	Bcb_Ptr->bcb$w_hsz = Bcb_Ptr->bcb$w_bsz = Buf_Size;
	return (Bcb_Ptr);
}

/*
 *	Routine Get_Buffer finds a buffer not in use. If necessary,
 *	it waits for the AST routine to signal an I/O completion if
 *	all the buffers are tied up.
 */

struct BCB *Get_Buffer (Dcb_Ptr)
struct DCB *Dcb_Ptr;
{
	auto   struct BCB *Bcb_Ptr;
	extern unsigned long Sys$Hiber();

loop:	for (Bcb_Ptr = Dcb_Ptr->dcb$l_bcb; Bcb_Ptr != 0; Bcb_Ptr = Bcb_Ptr->bcb$l_nxt)
	if ((Bcb_Ptr->bcb$w_flags & BCB$M_ACTIVE) == 0)
		return (Bcb_Ptr);
	Sys$Hiber ();
	goto loop;
}

/*
 *	Routine Get_Device_Stats returns the number of reads and writes
 *	from/to the device:
 */

Get_Device_Stats (Dcb_Ptr, Stats)
struct DCB *Dcb_Ptr;
unsigned long Stats[2];
{
	Stats[0] = Dcb_Ptr->dcb$l_nreads;
	Stats[1] = Dcb_Ptr->dcb$l_nwrites;
}

/*
 *	This AST routine is responsible for I/O completion for device write
 *	I/O. It marks the I/O completed, checks the I/O status, and wakes
 *	up the main process. This is also a convenient place to reset the
 *	buffer pointer.
 */

unsigned long Output_AST_Handler (Bcb_Ptr, R0, R1, PC, PSL)
struct BCB *Bcb_Ptr;
unsigned long R0, R1, PC, PSL;
{
	extern unsigned long Sys$Wake();
	globalvalue SS$_NORMAL, SS$_CANCEL, SS$_ABORT;

	if (Bcb_Ptr->bcb$l_ios.Status != SS$_CANCEL && Bcb_Ptr->bcb$l_ios.Status != SS$_ABORT)
		Bcb_Ptr->bcb$l_dcb->dcb$l_nwrites++;
	Bcb_Ptr->bcb$l_bfp = Bcb_Ptr->bcb$l_buf;
	Bcb_Ptr->bcb$w_hsz = Bcb_Ptr->bcb$w_bsz;
	Bcb_Ptr->bcb$w_flags &= ~BCB$M_ACTIVE;
	EnQueue_AST (AST_DEVICE_WRITE_IO, Bcb_Ptr->bcb$l_dcb, Bcb_Ptr, PRI_DEVICE_WRITE_IO);
	Sys$Wake (0, 0);
	return (SS$_NORMAL);
}

/*
 *	This AST routine is responsible for I/O completion for device read
 *	I/O. It marks the I/O completed and wakes up the main process.
 */

unsigned long Input_AST_Handler (Dcb_Ptr, R0, R1, PC, PSL)
struct DCB *Dcb_Ptr;
unsigned long R0, R1, PC, PSL;
{
	extern unsigned long Sys$Wake();
	globalvalue SS$_NORMAL, SS$_CANCEL, SS$_ABORT;

	if (Dcb_Ptr->dcb$l_ios.Status != SS$_CANCEL && Dcb_Ptr->dcb$l_ios.Status != SS$_ABORT)
		Dcb_Ptr->dcb$l_nreads++;
	EnQueue_AST (AST_DEVICE_READ_IO, Dcb_Ptr, 0, PRI_DEVICE_READ_IO);
	Sys$Wake (0, 0);
	return (SS$_NORMAL);
}

/*
 *	Routine Enable_Mailbox_Interrupt sets a write attention AST
 *	for the terminal mailbox:
 */

unsigned long Enable_Mailbox_Interrupt (Dcb_Ptr)
struct DCB *Dcb_Ptr;
{
	auto   unsigned long Sys_Status;
	extern unsigned long Sys$QIOW(), Mailbox_AST_Handler();
	globalvalue IO$_SETMODE, IO$M_WRTATTN;

	Sys_Status = Sys$QIOW (Dcb_Ptr->dcb$l_mbxevf, Dcb_Ptr->dcb$w_mbxchan,
			       IO$_SETMODE | IO$M_WRTATTN, &Dcb_Ptr->dcb$l_mbxios, 0, 0,
			       &Mailbox_AST_Handler, Dcb_Ptr, PSL$C_USER, 0, 0, 0);
	if ((Sys_Status & 0x01) != 0)
		Sys_Status = Dcb_Ptr->dcb$l_mbxios.Status;
	return (Sys_Status);
}

/*
 *	This AST Handler processes messages in the terminal mailbox
 *	and queues an appropriate AST.
 */

unsigned long Mailbox_AST_Handler (Dcb_Ptr, R0, R1, PC, PSL)
struct DCB *Dcb_Ptr;
unsigned long R0, R1, PC, PSL;
{
	auto   unsigned long Sys_Status;
	static struct {
		unsigned short msg$w_type;
		unsigned short msg$w_unit;
		unsigned char  msg$b_namlng;
		char msg$t_controller[15];
		unsigned short msg$w_brdcstlng;
		char msg$t_brdcstmsg[80];
	} Mailbox_Buffer;
	extern unsigned long Sys$QIOW(), Sys$Wake(), Enable_Mailbox_Interrupt();
	globalvalue IO$_READLBLK, IO$M_NOW, SS$_ENDOFFILE, SS$_NORMAL;
/*
 *	Read in the message:
 */
	Sys_Status = Sys$QIOW (Dcb_Ptr->dcb$l_mbxevf, Dcb_Ptr->dcb$w_mbxchan,
			       IO$_READLBLK | IO$M_NOW, &Dcb_Ptr->dcb$l_mbxios, 0, 0,
			       &Mailbox_Buffer, sizeof (Mailbox_Buffer), 0, 0, 0, 0);
	if ((Sys_Status & 0x01) != 0)
		Sys_Status = Dcb_Ptr->dcb$l_mbxios.Status;
/*
 *	Determine the message type and queue the appropriate AST;
 *	wake up the main process:
 */
	if ((Sys_Status & 0x01) != 0) switch (Mailbox_Buffer.msg$w_type) {

	case MSG$_TRMUNSOLIC:
		EnQueue_AST (AST_DEVICE_INPUT, Dcb_Ptr, 0, PRI_DEVICE_INPUT);
		Sys$Wake (0, 0);
		break;

	case MSG$_TRMHANGUP:
		EnQueue_AST (AST_CARRIER_LOSS, Dcb_Ptr, 0, PRI_CARRIER_LOSS);
		Sys$Wake (0, 0);
		break;

	case MSG$_TRMBRDCST:
		EnQueue_AST (AST_BROADCAST, Dcb_Ptr, 0, PRI_BROADCAST);
		Sys$Wake (0, 0);
	}
/*
 *	Re-enable the interrupt and return:
 */
	Sys_Status = Enable_Mailbox_Interrupt (Dcb_Ptr);
	return (SS$_NORMAL);
}
