/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains routines that process run options.
 */

#include "debug.h"
#include "types.h"
#include "devtab.h"
#include "options.h"
#include "tracedef.h"

#include "arith.p"
#include "strng.p"

#define Times10(a) (((a << 2) + a) << 1)

/*
 *	Routine Process_Run_Options retrieves information from the
 *	command line used to invoke Dviout.
 */

int Process_Run_Options (Option_Ptr, Scope)
struct Run_Options *Option_Ptr;
int Scope;
{
	auto   struct Run_Options *O_Ptr;
	auto   struct Pages_Spec *PD_Ptr;
	auto   int Count, Index;
	static struct {
		char *Mag_Name;
		char *Mag_Value;
	} Mag_Table[] = {
		{ "MAG0", "1000" }, { "MAGH", "1095" }, { "MAG1", "1200" },
		{ "MAG2", "1440" }, { "MAG3", "1728" }, { "MAG4", "2074" },
		{ "MAG5", "2488" }, { "MAG6", "2986" }
	};
	extern int Load_PXL(), Load_PK(), Load_GF();
	static struct {
		char *Source_Name;
		int (*Func)();
	} Source_Table[] = {
		{ "PK", &Load_PK },
		{ "PXL", &Load_PXL },
		{ "GF", &Load_GF }
	};
	static int Numeric_Values[2];
	static char *Value_String[80];
	extern struct Device_Table Device_Table[];
	extern int Check_Run_Option(), Get_Run_Option_Value(), strcmp();
	extern unsigned int Parse_Unsigned_Int();
	extern long Parse_Dimension();
	extern char *Mem_Alloc();

	O_Ptr = Option_Ptr;
/*
 *	Set default values:
 */
	O_Ptr->Device_Ptr = &Device_Table[0];
	O_Ptr->Pixel_Load_Func = &Load_PK;
	O_Ptr->Magnification = 1000;
	O_Ptr->Hoffset = 4736286;	/* 1 inch */
	O_Ptr->Voffset = 4736286;
	O_Ptr->Output[0] = '\0';
	O_Ptr->Page_Mark[0] = '\0';
	O_Ptr->Mark_Font_Name[0] = '\0';
	O_Ptr->Mark_Font_Scale = 1000;
	O_Ptr->Form_Size[0].Numerator = 85;
	O_Ptr->Form_Size[0].Denominator = 10;
	O_Ptr->Form_Size[1].Numerator = 11;
	O_Ptr->Form_Size[1].Denominator = 1;
	O_Ptr->Pages_Head = 0;
	O_Ptr->Number_of_Copies = 1;
	O_Ptr->Memory_Size = 0;
	O_Ptr->Flags = 0;
	O_Ptr->Neg_Flags = 0;
	O_Ptr->Trace = 0;
	O_Ptr->Neg_Trace = 0;
/*
 *	Obtain /OUTPUT option:
 */
	if (Check_Run_Option ("OUTPUT", "", Scope) != 0) {
		Get_Run_Option_Value ("OUTPUT", O_Ptr->Output, sizeof (O_Ptr->Output));
		O_Ptr->Flags |= OUTPUT;
	}
/*
 *	Obtain /FORM option:
 */
	if (Check_Run_Option ("FORM", "", Scope) != 0) {
		Get_Run_Option_Value ("FORM", Value_String, sizeof (Value_String));
		Parse_Form_Type (Value_String, O_Ptr->Form_Size);
		O_Ptr->Flags |= FORM;
	}
/*
 *	Obtain /MAGNIFICATION option:
 */
	if (Check_Run_Option ("MAGNIFICATION", "", Scope) != 0) {
		Get_Run_Option_Value ("MAGNIFICATION", Value_String, sizeof (Value_String));
		for (Index = 0; Index < arraysize (Mag_Table); Index++)
		if (strcmp (Mag_Table[Index].Mag_Name, Value_String) == 0) {
			stringcpy_m (Value_String, Mag_Table[Index].Mag_Value, sizeof (Value_String));
			break;
		}
		O_Ptr->Magnification = Parse_Unsigned_Int (Value_String);
		O_Ptr->Flags |= MAGNIFICATION;
	}
/*
 *	Obtain /HOFFSET option:
 */
	if (Check_Run_Option ("HOFFSET", "", Scope) != 0) {
		Get_Run_Option_Value ("HOFFSET", Value_String, sizeof (Value_String));
		O_Ptr->Hoffset = Parse_Dimension (Value_String, 4736286);
		O_Ptr->Flags |= HOFFSET;
	}
/*
 *	Obtain /VOFFSET option:
 */
	if (Check_Run_Option ("VOFFSET", "", Scope) != 0) {
		Get_Run_Option_Value ("VOFFSET", Value_String, sizeof (Value_String));
		O_Ptr->Voffset = Parse_Dimension (Value_String, 4736286);
		O_Ptr->Flags |= VOFFSET;
	}
/*
 *	Obtain /COPIES option:
 */
	if (Check_Run_Option ("COPIES", "", Scope) != 0) {
		Get_Run_Option_Value ("COPIES", Value_String, sizeof (Value_String));
		O_Ptr->Number_of_Copies = Parse_Unsigned_Int (Value_String);
		O_Ptr->Flags |= COPIES;
	}
/*
 *	Obtain /MEMORY option:
 */
	if (Check_Run_Option ("MEMORY", "", Scope) != 0) {
		Get_Run_Option_Value ("MEMORY", Value_String, sizeof (Value_String));
		O_Ptr->Memory_Size = Parse_Unsigned_Int (Value_String);
		O_Ptr->Flags |= MEMORY;
	}
/*
 *	Obtain /PAGES option:
 */
	if (Check_Run_Option ("PAGES", "", Scope) != 0) {
		PD_Ptr = (struct Pages_Spec *) &O_Ptr->Pages_Head;
		while (Get_Run_Option_Value ("PAGES", Value_String,
					     sizeof (Value_String)) != 0) {
			PD_Ptr = PD_Ptr->Link = (struct Pages_Spec *)
				Mem_Alloc (sizeof (struct Pages_Spec));
			PD_Ptr->Link = 0;
			Parse_Page_Specification (Value_String, PD_Ptr);
		}
		O_Ptr->Flags |= PAGES;
	}
/*
 *	Obtain /DEVICE option:
 */
	if (Check_Run_Option ("DEVICE", "", Scope) != 0) {
		for (Index = 0; Device_Table[Index].Device_Name != 0; Index++)
		if (Check_Run_Option ("DEVICE", Device_Table[Index].Device_Name, Scope) != 0) {
			O_Ptr->Device_Ptr = &Device_Table[Index];
			break;
		}
		O_Ptr->Flags |= DEVICE;
	}
/*
 *	Obtain PIXEL_SOURCE; set function address:
 */
	if (Check_Run_Option ("PIXEL_SOURCE", "", Scope) != 0) {
		for (Index = 0; Index < arraysize (Source_Table); Index++)
		if (Check_Run_Option ("PIXEL_SOURCE", Source_Table[Index].Source_Name, Scope) != 0) {
			O_Ptr->Pixel_Load_Func = Source_Table[Index].Func;
			break;
		}
		O_Ptr->Flags |= PIXEL_SOURCE;
	}
/*
 *	Obtain /MARK, /MKFONT, /MKSCALE parameters:
 */
	if (Check_Run_Option ("MARK", "", Scope) != 0) {
		Get_Run_Option_Value ("MARK", O_Ptr->Page_Mark, sizeof (O_Ptr->Page_Mark));
		O_Ptr->Flags |= MARK;
	}
	if (Check_Run_Option ("MKFONT", "", Scope) != 0) {
		Get_Run_Option_Value ("MKFONT", O_Ptr->Mark_Font_Name, sizeof (O_Ptr->Mark_Font_Name));
		O_Ptr->Flags |= MKFONT;
	}
	if (Check_Run_Option ("MKSCALE", "", Scope) != 0) {
		Get_Run_Option_Value ("MKSCALE", Value_String, sizeof (Value_String));
		for (Index = 0; Index < arraysize (Mag_Table); Index++)
		if (strcmp (Mag_Table[Index].Mag_Name, Value_String) == 0) {
			stringcpy_m (Value_String, Mag_Table[Index].Mag_Value, sizeof (Value_String));
			break;
		}
		O_Ptr->Mark_Font_Scale = Parse_Unsigned_Int (Value_String);
		O_Ptr->Flags |= MKSCALE;
	}
/*
 *	Obtain flags /[NO]INTERACTIVE, [NO]USETFM, [NO]LANDSCAPE,
 *	[NO] ODD_PAGES, [NO]EVEN_PAGES, [NO]TWO_SIDED, [NO]CONCATENATE,
 *	[NO]REVERSE, [NO]EXCLUDE, [NO]DELETE, [NO]SORTED:
 */
	Set_Option_Mask ("INTERACTIVE", "", INTERACTIVE, &O_Ptr->Flags, Scope);
	Set_Option_Mask ("USETFM", "", USETFM, &O_Ptr->Flags, Scope);
	Set_Option_Mask ("LANDSCAPE", "", LANDSCAPE, &O_Ptr->Flags, Scope);
	Set_Option_Mask ("ODD_PAGES", "", ODD_PAGES, &O_Ptr->Flags, Scope);
	Set_Option_Mask ("EVEN_PAGES", "", EVEN_PAGES, &O_Ptr->Flags, Scope);
	Set_Option_Mask ("TWO_SIDED", "", TWO_SIDED, &O_Ptr->Flags, Scope);
	Set_Option_Mask ("CONCATENATE", "", CONCATENATE, &O_Ptr->Flags, Scope);
	Set_Option_Mask ("REVERSE", "", REVERSE, &O_Ptr->Flags, Scope);
	Set_Option_Mask ("EXCLUDE", "", EXCLUDE, &O_Ptr->Flags, Scope);
	Set_Option_Mask ("DELETE", "", DELETE, &O_Ptr->Flags, Scope);
	Set_Option_Mask ("SORTED", "", SORTED, &O_Ptr->Flags, Scope);
/*
 *	Obtain /TRACE options. Note that specifying /TRACE=(ALL,NOxxxx)
 *	will allow selective exclusion of /trace options:
 */
	if (Check_Run_Option ("TRACE", "", Scope) != 0) {
		Set_Option_Mask ("TRACE", "ALL", TRACE_ALL, &O_Ptr->Trace, Scope);
		Set_Option_Mask ("TRACE", "PIXEL", TRACE_PIXEL, &O_Ptr->Trace, Scope);
		Set_Option_Mask ("TRACE", "FONT", TRACE_FONT, &O_Ptr->Trace, Scope);
		Set_Option_Mask ("TRACE", "CHAR", TRACE_CHAR, &O_Ptr->Trace, Scope);
		Set_Option_Mask ("TRACE", "RASTER", TRACE_RASTER, &O_Ptr->Trace, Scope);
		Set_Option_Mask ("TRACE", "DVI", TRACE_DVI, &O_Ptr->Trace, Scope);
		Set_Option_Mask ("TRACE", "DRIVER", TRACE_DRIVER, &O_Ptr->Trace, Scope);
		Set_Option_Mask ("TRACE", "PASTE", TRACE_PASTE, &O_Ptr->Trace, Scope);
		Set_Option_Mask ("TRACE", "SPECIAL", TRACE_SPECIAL, &O_Ptr->Trace, Scope);
		if (O_Ptr->Trace == 0 && O_Ptr->Neg_Trace == 0)	/* just /trace */
			O_Ptr->Trace |= TRACE_ALL;
	}
}

Set_Option_Mask (Option, Value, Mask, Flags, Scope)
char *Option, *Value;
unsigned long Mask;
unsigned long Flags[2];
int Scope;
{
	auto   int Indicator;
	extern int Check_Run_Option();

	if ((Indicator = Check_Run_Option (Option, Value, Scope)) > 0)
		Flags[0] |= Mask;
	else if (Indicator < 0)
		Flags[1] |= Mask;
}

/*
 *	Routine Apply_Options applies any run options that have
 *	been specified in global context to any local options that
 *	weren't specified.
 */

Apply_Options (Global_Ptr, Local_Ptr)
struct Run_Options *Global_Ptr, *Local_Ptr;
{
	auto   struct Run_Options *G_Ptr, *L_Ptr;

	G_Ptr = Global_Ptr;
	L_Ptr = Local_Ptr;
	if ((L_Ptr->Flags & MAGNIFICATION) == 0)
		L_Ptr->Magnification = G_Ptr->Magnification;
	if ((L_Ptr->Flags & MEMORY) == 0)
		L_Ptr->Memory_Size = G_Ptr->Memory_Size;
	if ((L_Ptr->Flags & HOFFSET) == 0)
		L_Ptr->Hoffset = G_Ptr->Hoffset;
	if ((L_Ptr->Flags & VOFFSET) == 0)
		L_Ptr->Voffset = G_Ptr->Voffset;
	if ((L_Ptr->Flags & OUTPUT) == 0)
		stringcpy_m (L_Ptr->Output, G_Ptr->Output, sizeof (L_Ptr->Output));
	if ((L_Ptr->Flags & FORM) == 0) {
		L_Ptr->Form_Size[0] = G_Ptr->Form_Size[0];
		L_Ptr->Form_Size[1] = G_Ptr->Form_Size[1];
	}
	if ((L_Ptr->Flags & PAGES) == 0)
		L_Ptr->Pages_Head = G_Ptr->Pages_Head;
	if ((L_Ptr->Flags & DEVICE) == 0)
		L_Ptr->Device_Ptr = G_Ptr->Device_Ptr;
	if ((L_Ptr->Flags & PIXEL_SOURCE) == 0)
		L_Ptr->Pixel_Load_Func = G_Ptr->Pixel_Load_Func;
	if ((L_Ptr->Flags & COPIES) == 0)
		L_Ptr->Number_of_Copies = G_Ptr->Number_of_Copies;
	if ((L_Ptr->Flags & MARK) == 0)
		stringcpy_m (L_Ptr->Page_Mark, G_Ptr->Page_Mark, sizeof (L_Ptr->Page_Mark));
	if ((L_Ptr->Flags & MKFONT) == 0)
		stringcpy_m (L_Ptr->Mark_Font_Name, G_Ptr->Mark_Font_Name, sizeof (L_Ptr->Mark_Font_Name));
	if ((L_Ptr->Flags & MKSCALE) == 0)
		L_Ptr->Mark_Font_Scale = G_Ptr->Mark_Font_Scale;
/*
 *	Combine the global and local flag words into a single
 *	word. The rules that apply here are:
 *
 *	1) If a flag has not been specified locally, but has
 *	   been specified globally, use the global specification.
 *	2) If a flag has been specified locally, it over-rides
 *	   any global setting of the flag.
 *	3) If a flag has a default, specifying /NO..., either
 *	   locally or globally, over-rides the default. Default
 *	   values affect only the 'on' flag value.
 */
	L_Ptr->Flags |= (G_Ptr->Flags | DEFAULT_FLAG) & ~G_Ptr->Neg_Flags;
	L_Ptr->Flags &= ~L_Ptr->Neg_Flags;
	L_Ptr->Trace |= G_Ptr->Trace & ~G_Ptr->Neg_Trace;
	L_Ptr->Trace &= ~L_Ptr->Neg_Trace;
}

/*
 *	Routines Parse_Page_Specification and Parse_Page_Number
 *	parse a string containing the character representation of
 *	page numbers (possibly separated by colons) into an signed
 *	output array. Roman numerals are assigned negative values.
 */

Parse_Page_Specification (Page_String, Page_Ptr)
char *Page_String;
struct Pages_Spec *Page_Ptr;
{
	auto   char *Ptr;
	auto   long Temp;
	auto   unsigned int Index, Count;
	static int Type[10];
	static char String[80];
	extern int Compare_Page_Number();
	msgcode DVIOUT_SEARCHINRANGE;

	stringcpy_m (String, Page_String, sizeof (String));
	for (Ptr = String; *Ptr != '\0' && *Ptr != ':'; Ptr++)
		;
	if (*Ptr == '\0') {	/* Search-style page specification */
		Parse_Page_Number (String, Page_Ptr->Low_Page, Type);
		for (Index = 0; Index < 10; Index++)
			Page_Ptr->High_Page[Index] = (Type[Index] == 1) ? 0 : 1;
		Page_Ptr->Range = 0;
	} else {		/* Range-style page specification */
		*Ptr++ = '\0';
		Parse_Page_Number (String, Page_Ptr->Low_Page, Type);
		Count = 0;
		for (Index = 0; Index < 10; Index++)
		if (Type[Index] == 2)
			Count++;
		Parse_Page_Number (Ptr, Page_Ptr->High_Page, Type);
		for (Index = 0; Index < 10; Index++)
		if (Type[Index] == 2)
			Count++;
		if (Count > 0)
			Message (DVIOUT_SEARCHINRANGE, Page_String, 1);
		Page_Ptr->Range = 1;
		if (Compare_Page_Number (Page_Ptr->Low_Page, Page_Ptr->High_Page) > 0) {
			for (Index = 0; Index < 10; Index++) {
				Temp = Page_Ptr->Low_Page[Index];
				Page_Ptr->Low_Page[Index] = Page_Ptr->High_Page[Index];
				Page_Ptr->High_Page[Index] = Temp;
			}
		}
	}
}

Parse_Page_Number (String, Out_Value, Type)
char *String;
long Out_Value[10];
int Type[10];
{
	auto   char *Ptr0, *Ptr;
	auto   long Value;
	auto   unsigned int Index;
	extern unsigned int Convert_Roman_to_Arabic(), Parse_Unsigned_Int();
	msgcode DVIOUT_EXTRACHAR;
/*
 *	Parse the individual components of the page number. Each can
 *	be a non-negative integer, a roman numeral, or an asterisk:
 */
	Ptr = String;
	for (Index = 0; Index < 10 && *(Ptr0 = Ptr) != '\0'; Index++) {
		for (Ptr = Ptr0; *Ptr != '.' && *Ptr != '\0'; Ptr++)
			;
		if (*Ptr == '.')
			*Ptr++ = '\0';
		Type[Index] = 1;
		if (*Ptr0 == '*') {
			Type[Index] = 2;
			Value = 0;
			if (*++Ptr0 != '\0')
				Message (DVIOUT_EXTRACHAR, Ptr0, 1);
		} else if ((Value = Convert_Roman_to_Arabic (Ptr0)) != 0)
			Value = -Value;
		else
			Value = Parse_Unsigned_Int (Ptr0);
		Out_Value[Index] = Value;
	}
	if (Index == 10 && *Ptr != '\0')
		Message (DVIOUT_EXTRACHAR, Ptr, 1);
	for (; Index < 10; Index++) {
		Out_Value[Index] = 0;
		Type[Index] = 0;
	}
}

/*
 *	Routine Parse_Unsigned_Int converts an ascii string
 *	containing a number into a binary unsigned integer value.
 */

unsigned int Parse_Unsigned_Int (String)
char *String;
{
	auto   char *Ptr, c;
	auto   unsigned int Value;
	msgcode DVIOUT_INVALNUMERIC;

	Value = 0;
	for (Ptr = String; (c = *Ptr) >= '0' && c <= '9'; Ptr++)
		Value = Times10 (Value) + (c - '0');
	if (c != '\0')
		Message (DVIOUT_INVALNUMERIC, String, 1);
	return (Value);
}

/*
 *	Routine Convert_Roman_to_Arabic converts a string containing a
 *	roman numeral into a conventional Arabic number. If an error
 *	is encountered, zero is returned instead.
 *
 *	The little trick used here is that every prefix of a valid
 *	roman numeral string is itself a valid roman numeral, if
 *	one considers 'atoms' of the string as indivisible, atoms
 *	being the numbers 1z-9z for each of the four magnitudes (z)
 *	of: i (1), x (10), c (100) and m (1000).
 *
 *	Even so, this is not a particularly efficient algorithm.
 */

unsigned int Convert_Roman_to_Arabic (String)
char *String;
{
	auto   char *Ptr;
	auto   unsigned int Tok_Value, Value;
	auto   int Char_Type, Next_Type, Temp, Count;
	static unsigned int R_Values[7] = { 1, 5, 10, 50, 100, 500, 1000 };
	extern int Roman_Letter_Type();

	Ptr = String;
	if (*Ptr == '\0' || (Char_Type = Roman_Letter_Type (*Ptr)) < 0)
		return (0);
	Tok_Value = Value = R_Values[Char_Type];
	if ((Char_Type & 0x01) == 0) {	/* Ten's value (i,x,c,m) */
		Count = 0;
		while (*++Ptr != '\0') {
			if ((Next_Type = Roman_Letter_Type(*Ptr)) == Char_Type) {
				Tok_Value += Value;
				if (++Count > 2) {
					Tok_Value = 0;
					break;
				}
			} else if (Count == 0 && (Temp = Next_Type - Char_Type) == 1 || Temp == 2) {
				Ptr++;
				Tok_Value = R_Values[Next_Type] - Value;
				break;
			} else break;
		}
	} else {	/* Five's value (v,l,d) */
		Value = R_Values[--Char_Type];
		Count = 0;
		while (*++Ptr != '\0' && Roman_Letter_Type(*Ptr) == Char_Type) {
			Tok_Value += Value;
			if (++Count > 3) {
				Tok_Value = 0;
				break;
			}
		}
	}
	if (Tok_Value == 0 || *Ptr == '\0')
		;
	else if ((Value = Convert_Roman_to_Arabic (Ptr)) == 0 || Value >= R_Values[Char_Type])
		Tok_Value = 0;
	else
		Tok_Value += Value;
	return (Tok_Value);
}

int Roman_Letter_Type (c)
char c;
{
	auto   int Char_Type;
	auto   unsigned char Ascii_Type;
	static int Type[22] = {
		4, 5, -1, -1, -1, -1, 0, -1, -1, 3, 6, -1, -1, -1, -1, -1, -1, -1, -1, 1, -1, 2
	};
	extern unsigned char XOrd[];
	extern char tolower();

	if ((Ascii_Type = XOrd[tolower(c)]) < 99 || Ascii_Type > 120)
		Char_Type = -1;
	else
		Char_Type = Type[Ascii_Type-99];
	return (Char_Type);
}

/*
 *	Routine Parse_Dimension parses a string containing a dimension
 *	into the equivalent number of scaled points.
 */

long Parse_Dimension (String, Default)
char *String;
long Default;
{
	auto   unsigned int Index;
	auto   char *Ptr;
	static struct Ratio Value;
	static struct {
		char *Dim_Name;
		struct Ratio Dim_Fraction;
	} Dim_Table[] = { /* Denominators should be less than (2^31-1)/1000000 */
		{ "PT", { 65536, 1 } },
		{ "POINT", { 65536, 1 } },
		{ "PC", { 12 * 65536, 1 } },
		{ "PICA", { 12 * 65536, 1 } },
		{ "IN", { 7227 * 65536, 100 } },
		{ "INCH", { 7227 * 65536, 100 } },
		{ "BP", { 7227 * 8192, 9 * 100 } },
		{ "BIGPOINT", { 7227 * 8192, 9 * 100 } },
		{ "CM", { 7227 * 65536, 254 } },
		{ "CENTIMETER", { 7227 * 65536, 254 } },
		{ "MM", { 7227 * 32768, 127 * 10 } },
		{ "MILLIMETER", { 7227 * 32768, 127 * 10 } },
		{ "DD", { 1238 * 65536, 1157 } },
		{ "DIDOT", { 1238 * 65536, 1157 } },
		{ "CC", { 12 * 1238 * 65536, 1157 } },
		{ "CICERO", { 12 * 1238 * 65536, 1157 } },
		{ "SP", { 1, 1 } },
		{ "SCALEDPOINT", { 1, 1 } },
		{ 0 }
	};
	extern char *Parse_Number(), *Skip_Blanks();
	extern int strcmp();
	msgcode DVIOUT_INVALDIMEN;
/*
 *	Obtain the numeric part of the dimension:
 */
	Ptr = Parse_Number (String, &Value);
/*
 *	Skip trailing blank. What is left must match an entry in the
 *	dimension table; if not, assume a raw value in scaled points:
 */
	Ptr = Skip_Blanks (Ptr);
	for (Index = 0; Dim_Table[Index].Dim_Name != 0; Index++)
	if (strcmp (Ptr, Dim_Table[Index].Dim_Name) == 0)
		return (XN_Div_D_R_M (Value.Numerator, Dim_Table[Index].Dim_Fraction.Numerator,
				      Dim_Table[Index].Dim_Fraction.Denominator * Value.Denominator));
	if (*Ptr != '\0') {
		Message (DVIOUT_INVALDIMEN, String, 1);
		return (Default);
	}
	return ((long) (Value.Numerator + (Value.Denominator >> 1)) / Value.Denominator);
}

char *Parse_Number (Num_String, Num_Value)
char *Num_String;
struct Ratio *Num_Value;
{
	auto   long Value;
	auto   unsigned long Scale;
	auto   char *Ptr, SeenPt, Sign, c;
	extern char *Skip_Blanks();

	Sign = SeenPt = 0;
	Value = 0;
	Scale = 1;
/*
 *	Skip leading blanks; check if a minus sign is present:
 */
	Ptr = Skip_Blanks (Num_String);
	if (*Ptr == '-') {
		Sign++;
		Ptr++;
	}
/*
 *	Parse the numeric part:
 */
	for (; (c = *Ptr) != '\0'; Ptr++)
	if (c == '.' && SeenPt == 0)
		SeenPt++;
	else if (c >= '0' && c <= '9') {
		if (Scale < 1000000) {	/* Ignore ridiculous accuracy */
			Value = Times10 (Value) + (c - '0');
			if (SeenPt != 0)
				Scale = Times10 (Scale);
		}
	} else break;
	if (Sign != 0 && Value != 0)
		Value = -Value;
	Num_Value->Numerator = (unsigned long) Value;
	Num_Value->Denominator = Scale;
	return (Ptr);
}

/*
 *	Routine Parse_Form_Type translate the form type into physical
 *	dimensions, in inches.
 */

Parse_Form_Type (Form_Name, Form_Dimension)
char *Form_Name;
struct Ratio Form_Dimension[2];
{
	auto   char *Ptr;
	auto   int Index;
	static struct Ratio Form_Width, Form_Length;
	static struct {
		char *Name;
		unsigned int Min_Chars;
		struct Ratio Width;
		struct Ratio Length;
	} Form_Table[] = {
		{ "LETTER",  3, {   85,  10 }, {   11,   1 } },
		{ "LEGAL",   3, {   85,  10 }, {   14,   1 } },
		{ "LEDGER",  3, {   11,   1 }, {   17,   1 } },
		{ "GLETTER", 4, {    8,   1 }, {   10,   1 } },
		{ "GLEGAL",  4, {    8,   1 }, {   13,   1 } },
		{ "FOLIO",   1, {   83,  10 }, {   13,   1 } },
		{ "A3",      2, { 2970, 254 }, {  165,  10 } },
		{ "A4",      2, { 2100, 254 }, { 2970, 254 } },
		{ "A5",      2, {   58,  10 }, { 2100, 254 } },
		{ "A6",      2, {   41,  10 }, {   67,  10 } },
		{ "A",       1, {   85,  10 }, {   11,   1 } },
		{ "B4",      2, {  101,  10 }, {  143,  10 } },
		{ "B5",      2, {   72,  10 }, {  101,  10 } },
		{ "B6",      2, {   51,  10 }, {   72,  10 } },
		{ "B",       1, {   11,   1 }, {   17,   1 } },
		{ "C",       1, {   17,   1 }, {   22,   1 } },
		{ "D",       1, {   24,   1 }, {   36,   1 } },
		{ "E",       1, {   36,   1 }, {   48,   1 } }
	};
	extern char *Parse_Number(), *Skip_Blanks();
	msgcode DVIOUT_INVALFORM;
/*
 *	First, search for common form names:
 */
	for (Index = 0; Index < arraysize (Form_Table); Index++)
	if (Compare_Keyword_M (Form_Name, Form_Table[Index].Name, Form_Table[Index].Min_Chars, 4, 0) == 1) {
		Form_Dimension[0] = Form_Table[Index].Width;
		Form_Dimension[1] = Form_Table[Index].Length;
		return;
	}
/*
 *	Otherwise, check for the syntax nnXnn, specifying the paper
 *	size in inches:
 */
	Ptr = Skip_Blanks (Parse_Number (Form_Name, &Form_Width));
	if (*Ptr != 'X' || (long) Form_Width.Numerator <= 0)
		Message (DVIOUT_INVALFORM, Form_Name, 1);
	else {
		Ptr = Skip_Blanks (Parse_Number (Skip_Blanks (&Ptr[1]), &Form_Length));
		if (*Ptr != '\0' || (long) Form_Length.Numerator <= 0)
			Message (DVIOUT_INVALFORM, Form_Name, 1);
		else {
			Form_Dimension[0] = Form_Width;
			Form_Dimension[1] = Form_Length;
		}
	}
}

char *Skip_Blanks (String)
char *String;
{
	auto   char *Ptr;

	for (Ptr = String; *Ptr == ' ' || *Ptr == '\t'; Ptr++)
		;
	return (Ptr);
}

#ifdef DEBUG
Show_Options (Option_Ptr, Header)
struct Run_Options *Option_Ptr;
char *Header;
{
	auto   struct Run_Options *O_Ptr;
	auto   struct Pages_Spec *PS_Ptr;
	auto   int Index;
	static char Page_Str[32];

	O_Ptr = Option_Ptr;
	printf ("%s options:\n", Header);
	printf ("  Device is \"%s\"\n", O_Ptr->Device_Ptr->Device_Name);
	printf ("  Output specification is \"%s\"\n", O_Ptr->Output);
	printf ("  Form size is (%u / %u) by (%u / %u)\n", O_Ptr->Form_Size[0].Numerator,
		O_Ptr->Form_Size[0].Denominator, O_Ptr->Form_Size[1].Numerator,
		O_Ptr->Form_Size[1].Denominator);
	printf ("  Magnification is %d\n", O_Ptr->Magnification);
	printf ("  Horizontal offset is %d\n", O_Ptr->Hoffset);
	printf ("  Vertical offset is %d\n", O_Ptr->Voffset);
	printf ("  Amount of Memory is %u\n", O_Ptr->Memory_Size);
	printf ("  Number of copies is %d\n", O_Ptr->Number_of_Copies);
	printf ("  Page Mark is \"%s\"\n", O_Ptr->Page_Mark);
	printf ("  Mark Font is %s scaled %u\n", O_Ptr->Mark_Font_Name, O_Ptr->Mark_Font_Scale);
	printf ("  Pages specified:\n");
	for (PS_Ptr = O_Ptr->Pages_Head; PS_Ptr != 0; PS_Ptr = PS_Ptr->Link) {
		if (PS_Ptr->Range == 0) {
			printf ("    ");
			for (Index = 0; Index < 10; Index++) {
				if (Index > 0)
					printf (".");
				if (PS_Ptr->High_Page[Index] != 0)
					printf ("*");
				else
					printf ("%ld", PS_Ptr->Low_Page[Index]);
			}
		} else {
			Format_Page_Number (PS_Ptr->Low_Page, Page_Str, sizeof (Page_Str));
			printf ("    %s", Page_Str);
			Format_Page_Number (PS_Ptr->High_Page, Page_Str, sizeof (Page_Str));
			printf (" to %s", Page_Str);
		}
		printf ("\n");
	}
}
#endif
