/***************************************************************************
** Although considerable effort has been expended to make this software   **
** correct and reliable, no warranty is implied; the author disclaims any **
** obligation or liability for damages, including but not limited to      **
** special, indirect, or consequential damages arising out of or in       **
** connection with the use or performance of this software.               **
***************************************************************************/

/*
 *	This file contains the MacPaint image-paste functionality.
 *
 *	The transformation capabilities of this function is limited
 *	to a 1-to-1 aspect ratio, with integer scaling. The resulting
 *	image is placed in the output area with the top left corners
 *	aligned, and using the largest scaling factor that will fit
 *	the image into the output area, with a minimum scaling factor
 *	of 1 (1.92 inches by 2.4 inches on a 300 dpi output device).
 *	If the scaling factor is 1 and the image will not fit in the
 *	specified area, the image is truncated at the bottom and right
 *	of the output area.
 *
 *	The bitmap is output as transparent, that is, anything
 *	underneath it shows through.
 *
 *	The MacPaint file format is described in Macintosh Technical
 *	notes #86 and #171.
 *
 *	"MacPaint" is a trademark of Apple Computer, Inc.
 */

#include "types.h"
#include "devtab.h"
#include "tracedef.h"

#include "raster.p"
#include "strng.p"
#include "fileio.p"
#include "fio.p"

struct {
	unsigned long Version;
	unsigned char Pattern[38][8];
	unsigned char Filler[204];
} MPP_Header;

pointer *MPP_File;

/*
 *	The parameters to MP_Paste specify the output device descriptor,
 *	the location of the upper left corner of the output box, the
 *	width, depth and orientation of the output box, the relative
 *	picture number and the file name containing the graphical input
 *	data. All dimensions are in 1/1024 pixels.
 *
 *	The 'orientation' option allows the picture to be pasted into
 *	the box in any of four orientations, corresponding to rotations
 *	of multiples of 90 degrees. Additionally, if X_Scale is negative,
 *	it indicates a mirror-image of the picture is to be produced.
 *
 *	A problem may exist here if the output device has extremely high
 *	resolution. The width and/or the height of the bitmap image may
 *	exceed 65535, the limit that can be stored in an unsigned short
 *	value.
 */

int MP_Paste (Dev_Ptr, File, X0, Y0, X_Scale, Y_Scale, Orientation, Picture,
	      Optv, Optc)
struct Device_Table *Dev_Ptr;
char *File;
long X0, Y0;
long X_Scale, Y_Scale;
unsigned int Orientation, Picture;
char *Optv[];
int Optc;
{
	auto   unsigned char *Image_Buffer, *Row_Buffer, *Ptr;
	auto   long Scale_X;
	auto   int Row, Scale, Width, Height, Count, Size;
	auto   int Left, Right, Top, Bottom;
	auto   unsigned short N_Rows, N_Columns;
	static unsigned long Context;
	static int Left_Bound, Right_Bound;
	static unsigned char Row_of_Data[72];
	msgcode DVIOUT_INOPENFAIL, DVIOUT_BADPAINTFILE;
	extern int MPP_Get_Row();
	extern char *Mem_Alloc();
/*
 *	Open the graphics file for input:
 */
	if (trace_paste)
		printf ("MacPaint file name: %s\n", File);
	if ((MPP_File = Open_Gen_File_M (File, ".MCP", "r", 0)) == 0) {
		Message (DVIOUT_INOPENFAIL, File, 1);
		return (0);
	}
/*
 *	Read in the file header, containing the unused patterns and
 *	version number:
 */
	MPP_Header.Version = (((((Read_Gen_Character_M (MPP_File) << 8) +
				 Read_Gen_Character_M (MPP_File)) << 8) +
			       Read_Gen_Character_M (MPP_File)) << 8) +
			     Read_Gen_Character_M (MPP_File);
	if (trace_paste)
		printf ("  File version is %ld\n", MPP_Header.Version);
	Ptr = &MPP_Header.Pattern[0][0];
	for (Count = 8*38; Count > 0; Count--)
		*Ptr++ = Read_Gen_Character_M (MPP_File);
	Ptr = &MPP_Header.Filler[0];
	for (Count = 204; Count > 0; Count--)
		*Ptr++ = Read_Gen_Character_M (MPP_File);
	if (Gen_At_EOF_M (MPP_File) != 0) {
		Close_Gen_File_M (MPP_File);
		Message (DVIOUT_BADPAINTFILE, File, 1);
		return (0);
	}
/*
 *	Determine the number of rows and columns in the output image,
 *	and the required transformation parameters:
 */
	Scale_X = (X_Scale > 0) ? X_Scale : -X_Scale;
	Width = (Scale_X + 10) >> 10;	/* +10 is integer truncation fix */
	Height = (Y_Scale + 10) >> 10;
	if (trace_paste)
		printf ("  Non-scaled width and height in pixels: %u by %u\n", Width, Height);
	if ((Orientation & 0x01) == 0) {	/* UP or DOWN */
		if (Width <= 576) {
			Scale = 1;
			N_Columns = Width;
		} else {
			Scale = Width / 576;
			N_Columns = 576;
		}
		if ((N_Rows = Height / Scale) > 720)
			N_Rows = 720;
		Width = Scale * N_Columns;
		Height = Scale * N_Rows;
	} else {				/* LEFT or RIGHT */
		if (Height <= 576) {
			Scale = 1;
			N_Columns = Height;
		} else {
			Scale = Height / 576;
			N_Columns = 576;
		}
		if ((N_Rows = Width / Scale) > 720)
			N_Rows = 720;
		Width = Scale * N_Rows;
		Height = Scale * N_Columns;
	}
	if (trace_paste)
		printf ("  Scale: %u  Columns: %u  Rows: %u\n", Scale, N_Columns, N_Rows);
/*
 *	Allocate a buffer large enough to hold one scaled row and another
 *	to hold the entire bitmap:
 */
	Row_Buffer = (unsigned char *) Mem_Alloc ((Scale * N_Columns + 7) >> 3);
	Image_Buffer = (unsigned char *) Mem_Alloc (((Width + 7) >> 3) * Height);
/*
 *	Do some initial set-up:
 */
	switch (Orientation) {
	case 0: Size = (Width + 7) >> 3; Ptr = Image_Buffer; break;
	case 1: Size = (X_Scale > 0) ? 0 : Width; break;
	case 2: Size = (Width + 7) >> 3; Ptr = &Image_Buffer[Size * Height]; break;
	case 3: Size = (X_Scale > 0) ? Width : 0;
	}
	Top = 720; Left = 576; Bottom = Right = -1;
/*
 *	Read in each row of the image and place it into the unpacked
 *	image buffer:
 */
	Context = 0;
	for (Row = 0; Row < N_Rows; Row++) {
		if (MPP_Get_Row (Row_of_Data, MPP_File, &Left_Bound, &Right_Bound, &Context) == 0) {
			Message (DVIOUT_BADPAINTFILE, File, 1);
			break;
		}
/*
 *	Scale and (possibly) reflect the row of data; compute
 *	bitmap bounds;
 */
		if (Left_Bound >= N_Columns)	/* All zero row */
			Clear_Memory_M (Row_Buffer, (Scale * N_Columns + 7) >> 3);
		else {
			Magnify_Raster_Row_M (N_Columns, Scale, Row_of_Data, Row_Buffer);
			if (Right_Bound >= N_Columns)
				Right_Bound = N_Columns - 1;
			if ((Orientation == 0 && X_Scale < 0) ||
			     Orientation == 1 ||
			    (Orientation == 2 && X_Scale > 0))
				Reflect_Raster_M (N_Columns * Scale, 1, Row_Buffer);
			if (Top == 720)
				Top = Row;
			Bottom = Row;
			if (Left_Bound < Left)
				Left = Left_Bound;
			if (Right_Bound > Right)
				Right = Right_Bound;
		}
/*
 *	Now place the scaled row into the output buffer according to the
 *	orientation of the image:
 */
		switch (Orientation) {

		case 0:	/* Up */

			for (Count = Scale; Count > 0; Count--) {
				Move_Memory_M (Row_Buffer, Ptr, Size);
				Ptr = &Ptr[Size];
			}
			break;

		case 1:	/* Left */

			for (Count = Scale; Count > 0; Count--) {
				if (X_Scale > 0)
					Write_Raster_Column_M (Width, Height, Size++, Row_Buffer, Image_Buffer);
				else
					Write_Raster_Column_M (Width, Height, --Size, Row_Buffer, Image_Buffer);
			}
			break;

		case 2:	/* Down */

			for (Count = Scale; Count > 0; Count--) {
				Ptr = &Ptr[-Size];
				Move_Memory_M (Row_Buffer, Ptr, Size);
			}
			break;

		case 3:	/* Right */

			for (Count = Scale; Count > 0; Count--) {
				if (X_Scale > 0)
					Write_Raster_Column_M (Width, Height, --Size, Row_Buffer, Image_Buffer);
				else
					Write_Raster_Column_M (Width, Height, Size++, Row_Buffer, Image_Buffer);
			}
		}
	}
/*
 *	Output the bitmap to the device. Offsets are required
 *	for orientations other than UP because the bitmap does not
 *	necessarily fit exactly into the output area:
 */
	if (Top != 720) switch (Orientation) {
	case 0: (*Dev_Ptr->Typeset_Pixel) (X0, Y0 + ((Top * Scale) << 10),
					   Width, (Bottom + 1 - Top) * Scale,
					   &Image_Buffer[((Width + 7) >> 3) * Top * Scale]); break;
	case 1: (*Dev_Ptr->Typeset_Pixel) (X0, Y0 + (Y_Scale - (Height << 10)) +
						    (((N_Columns - (Right + 1)) * Scale) << 10),
					   Width, (Right + 1 - Left) * Scale,
					   &Image_Buffer[((Width + 7) >> 3) * (N_Columns - (Right + 1)) * Scale]); break;
	case 2: (*Dev_Ptr->Typeset_Pixel) (X0 + (Scale_X - (Width << 10)),
					   Y0 + (Y_Scale - (Height << 10)) + (((N_Rows - (Bottom + 1)) * Scale) << 10),
					   Width, (Bottom + 1 - Top) * Scale,
					   &Image_Buffer[((Width + 7) >> 3) * (N_Rows - (Bottom + 1)) * Scale]); break;
	case 3: (*Dev_Ptr->Typeset_Pixel) (X0 + (Scale_X - (Width << 10)), Y0 + ((Left * Scale) << 10),
					   Width, (Right + 1 - Left) * Scale,
					   &Image_Buffer[((Width + 7) >> 3) * Left * Scale]);
	}
/*
 *	Finish up:
 */
	Close_Gen_File_M (MPP_File);
	Mem_Free (Image_Buffer);
	Mem_Free (Row_Buffer);
}

/*
 *	Read in a row of the image. The bytes of the MacPaint file
 *	are arranged as follows: Each segment of each row is preceded
 *	by a count byte. If this byte is positive, it is followed by
 *	that many data bytes plus one; if it is negative, the byte
 *	following the count byte is to be repeated a number of times
 *	equal to the two's complement of the count byte plus one.
 *	Additionally, data segments do not cross rows; there must
 *	be exactly 72 bytes in each row (for version 2+ files, at least;
 *	others are unknown).
 */

int MPP_Get_Row (Row_Buffer, File, Left, Right, Context)
unsigned char Row_Buffer[72];
pointer File;
int *Left, *Right;
struct { unsigned char Cnt; unsigned char Chr; unsigned char Typ; } *Context;
{
	auto   int Index, Jndex, Error, Pos;
	auto   unsigned char Count, Value;

	Index = 0;
	Error = 0;
	do {
		if ((Count = Context->Cnt) == 0) {
			Count = Read_Gen_Character_M (File);
			if (Gen_At_EOF_M (File) != 0)
				Error++;
			else if ((Count & 0x80) == 0) {
				Count++;
				Context->Typ = 0;
			} else {
				Count = ~Count + 2;	/* That's (-Count + 1) */
				Context->Typ = 1;
				Context->Chr = Read_Gen_Character_M (File);
				if (Gen_At_EOF_M (File) != 0)
					Error++;
			}
		}
		if (Error != 0)
			;
		else if (MPP_Header.Version >= 2 && Index+Count > 72)
			Error++;
		else if (Context->Typ == 0) {
			for (; Count > 0 && Index < 72; Count--) {
				Row_Buffer[Index++] = Read_Gen_Character_M (File);
				if (Gen_At_EOF_M (File) != 0) {
					Error++;
					break;
				}
			}
			Context->Cnt = Count;
		} else {
			for (; Count > 0 && Index < 72; Count--)
				Row_Buffer[Index++] = Context->Chr;
			Context->Cnt = Count;
		}
	} while (Index < 72 && Error == 0);
	if (Error != 0)
		return (0);
/*
 *	Determine left-most and right-most non-zero columns:
 */
	Pos = 0;
	for (Index = 0; Index < 72; Index++)
	if ((Value = Row_Buffer[Index]) != 0) {
		while ((Value & 0x80) == 0) {
			Pos++;
			Value <<= 1;
		}
		break;
	}
	if ((*Left = (Index << 3) + Pos) != 576) {	/* Not all zero */
		for (Jndex = 71; (Value = Row_Buffer[Jndex]) == 0; Jndex--)
			;
		Pos = 7;
		while ((Value & 0x01) == 0) {
			Pos--;
			Value >>= 1;
		}
		*Right = (Jndex << 3) + Pos;
	}
	return (1);
}
