/*
 * java.net.PlainSocketImpl.c
 *
 * Copyright (c) 1996 T. J. Wilkinson & Associates, London, UK.
 *
 * See the file "lib-license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@tjwassoc.demon.co.uk>, 1996.
 **/

/*** CHANGELOG ***
 *
 * 12.1.1998   Olli-Pekka Auvinen   Rewrote socketCreate, socketConnect,
 *                                  socketBind, socketListen, socketAccept
 *                                  and socketClose.
 *                                  
 * 13.1.1998   Olli-Pekka Auvinen   Modified setSocketOptions and
 *                                  getSocketOptions.
 *                                  Removed #include <netinet/in.h>.
 *
 * 15.1.1998   Olli-Pekka Auvinen   Rewrote socketAvailable.
 *
 * 16.1.1998   Olli-Pekka Auvinen   Added seek to start of ctl file.
 *
 * 10.2.1998   Olli-Pekka Auvinen   Added the dokumatic documentation. 
 *
 */

#include <u.h>
#include <libc.h>

#include "config.h"
#include "config-std.h"
#include "config-io.h"
#include "config-mem.h"
#include <native.h>
#include "../../APIcore/java.io.stubs/FileDescriptor.h"
#include "../../APIcore/java.lang.stubs/Integer.h"
#include "../java.net.stubs/SocketImpl.h"
#include "../java.net.stubs/InetAddress.h"
#include "../java.net.stubs/PlainSocketImpl.h"
#include "../java.net.stubs/SocketOptions.h"
#include "nets.h"
#include "kthread.h"

/**
  @title java_net_PlainSocketImpl
  @desc Instances of this class represent plain stream (TCP) sockets.
  @funcidx
  @end
  */

/**
  @function    java_net_PlainSocketImpl_socketCreate
  @description Create a new stream socket.
  @parameter   Reference to the current (this) object.
  @parameter   Ignored parameter, uses 1.0 API semantics for PlainSocketImpl.
  @end
  */

void java_net_PlainSocketImpl_socketCreate(struct Hjava_net_PlainSocketImpl* this, jbool stream)
{
  int ctlFileFD;

  /* Creating a socket in Plan9 is implemented with a tcp device which is
     controlled though its ctl file. Socket's file descriptor is emulated
     with this ctl file's file descriptor. The Java version 1.0 option of
     creating a datagram socket is not supported anymore. Use datagram
     socket instead. */
  
  if (stream == 0) {
    SignalError(NULL, "java.net.SocketException", "Deprecated Socket mode,"
		"use DatagramSocket class instead");	 
  }
  ctlFileFD = open("/net/tcp/clone", ORDWR);
  
  if (ctlFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Ctl file opening failed");
  }
  
  unhand(unhand(this)->fd)->fd = ctlFileFD;
}

/**
  @function    java_net_PlainSocketImpl_connect
  @description Connect the socket to someone.
  @parameter   Reference to the current (this) object.
  @parameter   InetAddress object of the party to be connected to.
  @parameter   The port number of the party to be connected to.
  @end
  */

void java_net_PlainSocketImpl_socketConnect(struct Hjava_net_PlainSocketImpl* this, struct Hjava_net_InetAddress* daddr, jint dport)
{
  char connString[MAX_PATH_LEN], connNumber[MAX_CNUM_LEN];
  char *portString;
  int ctlFileFD, localFileFD, ipAddress, ip1, ip2, ip3, ip4, r;
  
  /* Construct the connect call string and write it to the ctl file of the
     tcp device. */
  
  ipAddress = unhand(daddr)->address;
  ip1 = (ipAddress & 0xff000000) >> 24;
  ip2 = (ipAddress & 0x00ff0000) >> 16;
  ip3 = (ipAddress & 0x0000ff00) >> 8;
  ip4 = (ipAddress & 0x000000ff);
  sprint(connString, "connect %d.%d.%d.%d!%d", ip1, ip2, ip3, ip4, dport);

  ctlFileFD = unhand(unhand(this)->fd)->fd;  
  if (ctlFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Invalid ctl file fd");
  }
  
  r = strlen(connString);
  if (write(ctlFileFD, connString, r) < r) {
    SignalError(NULL, "java.net.SocketException", "Connecting failed");
  }
  
  /* Open the local file from the protocol directory and parse the local
     port number from there. */
  
  if (seek(ctlFileFD, 0, 0)) {
    SignalError(NULL, "java.net.SocketException", "Ctl file seeking failed");
  }
  r = read(ctlFileFD, connNumber, MAX_CNUM_LEN); 
  if (r <= 0) {
    SignalError(NULL, "java.net.SocketException", "Ctl file reading failed");
  }
  connNumber[r] = '\0';
  
  sprint(connString, "/net/tcp/%d/local", atoi(connNumber));
  
  localFileFD = open(connString, OREAD);
  if (localFileFD <0) {
    SignalError(NULL, "java.net.SocketException", "Local file opening failed");
  }

  r = read(localFileFD, connString, MAX_PATH_LEN);
  if (r <= 0) {
    SignalError(NULL, "java.net.SocketException", "Local file reading failed");
  }
  connString[r] = '\0';
  
  portString = strstr(connString, "!") + 1;
  
  /* Set the member variables according to the new state of the tcp device. */
  
  unhand(this)->address = daddr;
  unhand(this)->port = dport;
  unhand(this)->localport = atoi(portString);
  
  if (close(localFileFD) < 0) {
    SignalError(NULL, "java.net.SocketException", "Local file closing failed");
  }
}

/**
  @function    java_net_PlainSocketImpl_socketBind
  @description Bind a port to the socket (address is ignored in Plan9).
  @parameter   Reference to the current (this) object.
  @parameter   The port number to bind to.
  @parameter   The local address to bind to.
  @end
  */

void java_net_PlainSocketImpl_socketBind(struct Hjava_net_PlainSocketImpl* this, struct Hjava_net_InetAddress* laddr, jint lport)
{
  char bindString[MAX_PATH_LEN], connNumber[MAX_CNUM_LEN];
  char *portString, *ip1, *ip2, *ip3, *ip4;
  int ctlFileFD, localFileFD, r, daddr;
  
  /* Construct the bind call string and write it to the ctl file of the tcp
     device. The Plan 9 BSD emulation bind is incompatible with the semantics
     of Java stream socket semantics. The binding is accomplished by simply
     announcing the "bound" port number, not by binding the device to that
     port. Emulated bind would cause that no further action would be
     possible in server type of sockets. */
  
  sprint(bindString, "announce %d", lport);
  
  ctlFileFD = unhand(unhand(this)->fd)->fd;
  if (ctlFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Invalid ctl file fd");
  }
  
  r = strlen(bindString);
  if (write(ctlFileFD, bindString, r) < r) {
    SignalError(NULL, "java.net.SocketException", "Socket binding failed");
  }
  
  if (seek(ctlFileFD, 0, 0)) {
    SignalError(NULL, "java.net.SocketException", "Ctl file seeking failed");
  }
  r = read(ctlFileFD, connNumber, MAX_CNUM_LEN);
  if (r <= 0) {
    SignalError(NULL, "java.net.SocketException", "Ctl file reading failed");
  }
  connNumber[r] = '\0';
  
  sprint(bindString, "/net/tcp/%d/local", atoi(connNumber));

  localFileFD = open(bindString, OREAD);
  if (localFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Local file opening failed");
  }
  
  r = read(localFileFD, bindString, MAX_PATH_LEN);
  if (r <= 0) {
    SignalError(NULL, "java.net.SocketException", "Local file reading failed");
  }
  bindString[r] = '\0';

  portString = strstr(bindString, "!");

  *portString = '\0';
  portString++;

  ip1 = bindString;
  ip2 = strstr(ip1, ".");
  *ip2 = '\0';
  ip2++;
  ip3 = strstr(ip2, ".");
  *ip3 = '\0';
  ip3++;
  ip4 = strstr(ip3, ".");
  *ip4 = '\0';
  ip4++;

  daddr = (atoi(ip1) << 24) + (atoi(ip2) << 16) + (atoi(ip3) << 8) + atoi(ip4);
  USED(daddr);

  /* Set the member variables according to the new state of the tcp device. */

  unhand(this)->localport = atoi(portString);
  unhand(this)->address = laddr;
  
  if (close(localFileFD) < 0) {
    SignalError(NULL, "java.net.SocketException", "Local file closing failed");
  }
}

/**
  @function    java_net_PlainSocketImpl_socketListen
  @description Turns this socket into a listener.
  @parameter   Reference to the current (this) object.
  @parameter   The backlog queue length.
  @end
  */

void java_net_PlainSocketImpl_socketListen(struct Hjava_net_PlainSocketImpl* this, jint count)
{
  char backlogString[15];
  int ctlFileFD, r;
  
  ctlFileFD = unhand(unhand(this)->fd)->fd;
  if (ctlFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Invalid ctl file fd");
  }
  
  /* Construct the backlog announcement string used to initialize listening
     of the socket and then write it to the ctl file of the tcp device. */
  
  sprint(backlogString, "backlog %d", count);
  
  r = strlen(backlogString);
  if (write(ctlFileFD, backlogString, r) < r) {
    SignalError(NULL, "java.net.SocketException", "Backlog announcing failed");
  }
}


/**
  @function    java_net_PlainSocketImpl_socketAccept
  @description Accept a connection.
  @parameter   Reference to the current (this) object.
  @parameter   The SocketImpl object of the new accepted connection.
  @end
  */

void java_net_PlainSocketImpl_socketAccept(struct Hjava_net_PlainSocketImpl* this, struct Hjava_net_SocketImpl* sock)
{
  char listenString[MAX_PATH_LEN], bindString[MAX_PATH_LEN],
    connNumber[MAX_CNUM_LEN];
  char *portString, *ip1, *ip2, *ip3, *ip4;
  int ctlFileFD, newCtlFileFD, remoteFileFD, localFileFD, daddr, r;

  ctlFileFD = unhand(unhand(this)->fd)->fd;
  if (ctlFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Invalid ctl file fd");
  }
  
  if (seek(ctlFileFD, 0, 0)) {
    SignalError(NULL, "java.net.SocketException", "Ctl file seeking failed");
  }
  r = read(ctlFileFD, connNumber, MAX_CNUM_LEN);
  if (r <= 0) {
    SignalError(NULL, "java.net.SocketException", "Ctl file reading failed");
  }
  connNumber[r] = '\0';
    
  sprint(bindString, "/net/tcp/%d/local", atoi(connNumber));
  
  localFileFD = open(bindString, OREAD);
  if (localFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Local file opening failed");
  }
  
  r = read(localFileFD, bindString, MAX_PATH_LEN);
  if (r <= 0) {
    SignalError(NULL, "java.net.SocketException", "Local file reading failed");
  }
  bindString[r] = '\0';
  
  if (close(localFileFD) < 0) {
    SignalError(NULL, "java.net.SocketException", "Local file closing failed");
  }
  
  portString = strstr(bindString, "!") + 1;
  
  if (atoi(portString) == 0)
    {
      /* Construct the announce string used to allow accepting connections
	 default port and write it to the ctl file of the tcp device. */
      
      sprint(listenString, "announce *");
      
      r = strlen(listenString);
      if (write(ctlFileFD, listenString, r) < r) {
	SignalError(NULL, "java.net.SocketException",
		    "Accepting port announcing failed");
      }
    }
  
  /* Construct the listen file name with protocol directory path and then
     open the listen file which is Plan9's way of waiting connections to be
     accepted. Open will block until a new connection is accepted. */
  
  sprint(listenString, "/net/tcp/%d/listen", atoi(connNumber));
  
  newCtlFileFD = open(listenString, ORDWR);
  if (newCtlFileFD < 0) {
    SignalError(NULL, "java.net.SocketException",
		"Listen file opening failed");
  }
  unhand(unhand(sock)->fd)->fd = newCtlFileFD;
  
  /* Open the remote file from the protocol directory and parse the name
     and port of the peer protocol entity from there. */
  
  r = read(newCtlFileFD, connNumber, MAX_CNUM_LEN);
  if (r <= 0) {
    SignalError(NULL, "java.net.SocketException",
		"Listen file reading failed");
  }
  connNumber[r] = '\0';
  
  sprint(listenString, "/net/tcp/%d/remote", atoi(connNumber));
  
  remoteFileFD = open(listenString, OREAD);
  if (remoteFileFD <0) {
    SignalError(NULL, "java.net.SocketException",
		"Remote file opening failed");
  }
  
  r = read(remoteFileFD, listenString, MAX_PATH_LEN);
  if (r <= 0) {
    SignalError(NULL, "java.net.SocketException",
		"Remote file reading failed");
  }
  listenString[r] = '\0';
  
  portString = strstr(listenString, "!");
  *portString = '\0';
  portString++;
  
  ip1 = listenString;  
  ip2 = strstr(ip1, ".");
  *ip2 = '\0';
  ip2++;  

  ip3 = strstr(ip2, ".");
  *ip3 = '\0';
  ip3++;  

  ip4 = strstr(ip3, ".");
  *ip4 = '\0';
  ip4++;
  daddr = (atoi(ip1) << 24) + (atoi(ip2) << 16) + (atoi(ip3) << 8) + atoi(ip4);
  
  /* Set the member variables according to the new state of the tcp device. */
  
  unhand(unhand(sock)->address)->address = daddr;
  unhand(sock)->port = atoi(portString);
  
  if (close(remoteFileFD) < 0) {
    SignalError(NULL, "java.net.SocketException",
		"Remote file closing failed");
  }
}


/**
  @function    java_net_PlainSocketImpl_socketAvailable
  @description Returns how many bytes can be read without blocking.
  @parameter   Reference to the current (this) object.
  @rvalue      Integer number of non-blocking readable bytes.
  @end
  */

jint java_net_PlainSocketImpl_socketAvailable(struct Hjava_net_PlainSocketImpl* this)
{
  char availableString[MAX_PATH_LEN], connNumber[MAX_CNUM_LEN],
    statBuffer[DIRLEN];
  int ctlFileFD, dataFileFD, r;
  
  ctlFileFD = unhand(unhand(this)->fd)->fd;
  if (ctlFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Invalid ctl file fd");
  }

  /* Construct the data file path and open it for the following fstat call. */

  if (seek(ctlFileFD, 0, 0)) {
    SignalError(NULL, "java.net.SocketException", "Ctl file seeking failed");
  }
  r = read(ctlFileFD, connNumber, MAX_CNUM_LEN);
  if (r <= 0) {
    SignalError(NULL, "java.net.SocketException", "Ctl file reading failed");
  }
  connNumber[r] = '\0';
  
  sprint(availableString, "/net/tcp/%d/data", atoi(connNumber));
  
  dataFileFD = open(availableString, ORDWR);
  if (dataFileFD < 0) {
    SignalError(NULL, "java.net.SocketException", "Data file opening failed");
  }
  
  /* The call to fstat returns the number of bytes that can be read without
     blocking if the argument file descriptor is a network connection. The
     statBuffer including various statistics about the file is not used for
     anything but it must be reserved to allow the system call to succeed. */
  
  r = fstat(dataFileFD, statBuffer);
  
  if (close(dataFileFD) < 0) {
    SignalError(NULL, "java.net.SocketException", "Data file closing failed");
  }
  
  return (r);
}


/**
  @function    java_net_PlainSocketImpl_socketClose
  @description Closes this socket.
  @parameter   Reference to the current (this) object.
  @end
  */

void java_net_PlainSocketImpl_socketClose(struct Hjava_net_PlainSocketImpl* this)
{
  int ctlFileFD, r;
  
  /* Simply close the ctl file and set the file descriptor instance variable
     to -1. */

  ctlFileFD = unhand(unhand(this)->fd)->fd;
  
  if (ctlFileFD != -1) {
    
    r = close(ctlFileFD);
    unhand(unhand(this)->fd)->fd = -1;
    
    if (r < 0) {
      SignalError(NULL, "java.net.SocketException", "Ctl file closing failed");
    }
  }
}

void java_net_PlainSocketImpl_initProto(struct Hjava_net_PlainSocketImpl* this)
{
  USED(this);
  /* ??? **/
}

void java_net_PlainSocketImpl_socketSetOption(struct Hjava_net_PlainSocketImpl* this, jint v1, jbool v2, struct Hjava_lang_Object* v3)
{
  USED(this);
  USED(v2);
  USED(v3);

  switch(v1) {
    
  case java_net_SocketOptions_SO_REUSEADDR:
  case java_net_SocketOptions_TCP_NODELAY:
    
    /* Option TCP_NODELAY handled here "the Tim Wilkinson way" */
    break;
    
  case java_net_SocketOptions_IP_MULTICAST_IF:
  case java_net_SocketOptions_SO_BINDADDR: /* JAVA takes care **/
  case java_net_SocketOptions_SO_TIMEOUT: /* JAVA takes care **/
    
  default:
    
    SignalError(NULL, "java.net.SocketException",
		"Unimplemented socket option");    
  }
}

jint java_net_PlainSocketImpl_socketGetOption(struct Hjava_net_PlainSocketImpl* this, jint val)
{
  int r;
  USED(this);
  SET(r);

  switch(val) {
    
  case java_net_SocketOptions_SO_BINDADDR:
    
    r = INADDR_ANY;
    break;
    
  case java_net_SocketOptions_SO_REUSEADDR:
  case java_net_SocketOptions_IP_MULTICAST_IF:  
  case java_net_SocketOptions_SO_TIMEOUT: /* JAVA takes care **/
    
  default:

    SignalError(NULL, "java.net.SocketException",
		"Unimplemented socket option");    
  }
  return (r);
}
