/*
 * java.io.RandomAccessFile.c
 *
 * Copyright (c) 1996 T. J. Wilkinson & Associates, London, UK.
 *
 * See the file "lib-license.terms" for information on usage and redistribution
 * of this file, and for a DISCLAIMER OF ALL WARRANTIES.
 *
 * Written by Tim Wilkinson <tim@tjwassoc.demon.co.uk>, 1996.
 */

/*** CHANGELOG ***
 *
 * 22.1.1998   Arttu Kuukankorpi          removed some includes
 *
 *  1.2.1998   Arttu Kuukankorpi          included u.h and libc.h
 *                                        rewrote RandomAccessFile_Open
 *                                        rewrote RandomAccessFile_Length
 *                                        changed call to seek to call to lseek in
 *                                         RandomAccessFile_Seek and
 *                                         RandomAccessFile_GetFilePointer
 *
 * 15.2.1998   Arttu Kuukankorpi          added dokumatic-documentation
 *
 *  3.3.1998   Arttu Kuukankorpi          fixed a bug in RandomAccessFile_Length
 *
 * 16.3.1998   Arttu Kuukankorpi          changed include-directories
 */

#include <u.h>
#include <libc.h>

#include "config.h"
#include "config-std.h"
#include "config-io.h"
#include "config-mem.h"
#include "defs.h"
#include "files.h"
#include "../java.io.stubs/RandomAccessFile.h"
#include "../java.io.stubs/FileDescriptor.h"
#include "kthread.h"

/**
  @title java_io_RandomAccessFile
  @desc Instances of this class support both reading and writing to a random access file.
  @funcidx
  @end
  */

/**
  @function java_io_RandomAccessFile_open
  @description Open a file for random access.
  @parameter Reference to the current (this) object.
  @parameter Name of the file to be opened.
  @parameter Open for reading or writing?
  @end
  */

void
java_io_RandomAccessFile_open(struct Hjava_io_RandomAccessFile* this, struct Hjava_lang_String* name, jbool rw)
{
	int fd;
	char str[MAXPATHLEN];

	javaString2CString(name, str, sizeof(str));

	if( rw ){
		if( access(str, 0) < 0 ){
			if( create(str, ORDWR, 0777) < 0 ){
			    SignalError(NULL, "java.io.IOException", SYS_ERROR);
			}
		}
		fd = open(str, ORDWR);
	}else
		fd = open(str, OREAD);

	unhand(unhand(this)->fd)->fd = fd;
	if (fd < 0) {
		SignalError(NULL, "java.io.IOException", SYS_ERROR);
	}
}

/**
  @function java_io_RandomAccessFile_length
  @description Return length of file.
  @parameter Reference to the current (this) object.
  @end
  */

jlong
java_io_RandomAccessFile_length(struct Hjava_io_RandomAccessFile* this)
{
    /*
	struct stat buf;
	int r;

	r = fstat(unhand(unhand(this)->fd)->fd, &buf);
	if (r < 0) {
		SignalError(NULL, "java.io.IOException", SYS_ERROR);
	}
	return (off_t2jlong(buf.st_size));
    */
        int cur;
	jlong cur2;

	/* get current file position */
	cur2 = java_io_RandomAccessFile_getFilePointer(this);

	/* get file size, this changes the file pointer to the end of file */
	if( (cur = seek(unhand(unhand(this)->fd)->fd, 0L, 2)) < 0 ){
	    SignalError(NULL, "java.io.IOException", SYS_ERROR);
	}

	/* restore file pointer */
	java_io_RandomAccessFile_seek(this, cur2);
	return (jlong)cur;
}

/**
  @function java_io_RandomAccessFile_seek
  @description Seek into file.
  @parameter Reference to the current (this) object.
  @parameter Position.
  @end
  */

void
java_io_RandomAccessFile_seek(struct Hjava_io_RandomAccessFile* this, jlong pos)
{
	int r;

	/*
	r = lseek(unhand(unhand(this)->fd)->fd, jlong2off_t(pos), SEEK_SET);
	*/
        r = seek(unhand(unhand(this)->fd)->fd, (long)pos, 0);
	if (r < 0) {
		SignalError(NULL, "java.io.IOException", SYS_ERROR);
	}
}

/**
  @function java_io_RandomAccessFile_readBytes
  @description Read in bytes from file.
  @parameter Reference to the current (this) object.
  @parameter Reference to an array of bytes where to store read data to.
  @parameter Offset.
  @parameter Length
  @rvalue Number of bytes read, -1 in case of error.
  @end
  */

jint
java_io_RandomAccessFile_readBytes(struct Hjava_io_RandomAccessFile* this, HArrayOfByte* bytes, jint off, jint len)
{
	jint ret;

	/* Adjust length */
	len = (len < obj_length(bytes) - off ? len : obj_length(bytes) - off);

	ret = read(unhand(unhand(this)->fd)->fd, &unhand(bytes)->body[off], len);
	if (ret < 0) {
		SignalError(NULL, "java.io.IOException", SYS_ERROR);
	}
	return (ret > 0 ? ret : -1);
}

/**
  @function java_io_RandomAccessFile_read
  @description Read a byte from file.
  @parameter Reference to the current (this) object.
  @rvalue Byte read, -1 in case of error.
  @end
  */

jint
java_io_RandomAccessFile_read(struct Hjava_io_RandomAccessFile* this)
{
	jint ret;
	/* unsigned */ char byte;

	ret = read(unhand(unhand(this)->fd)->fd, &byte, 1);
	if (ret < 0) {
		SignalError(NULL, "java.io.IOException", SYS_ERROR);
	}

	return (ret > 0 ? byte : -1);
}

/**
  @function java_io_RandomAccessFile_write
  @description Write a byte to file.
  @parameter Reference to the current (this) object.
  @rvalue Byte to be written.
  @end
  */

void
java_io_RandomAccessFile_write(struct Hjava_io_RandomAccessFile* this, jint data)
{
	jint ret;
	/* unsigned */ char byte;

	byte = data;

	ret = write(unhand(unhand(this)->fd)->fd, &byte, 1);
	if (ret < 0) {
		SignalError(NULL, "java.io.IOException", SYS_ERROR);
	}
}

/**
  @function java_io_RandomAccessFile_writeBytes
  @description Write a number of bytes to file.
  @parameter Reference to the current (this) object.
  @parameter Reference to the array of bytes to be written.
  @parameter Offset.
  @parameter Length.
  @end
  */

void
java_io_RandomAccessFile_writeBytes(struct Hjava_io_RandomAccessFile* this, HArrayOfByte* bytes, jint off, jint len)
{
	jint ret;

	ret = write(unhand(unhand(this)->fd)->fd, &unhand(bytes)->body[off], len);
	if (ret < 0) {
		SignalError(NULL, "java.io.IOException", SYS_ERROR);
	}
}

/**
  @function java_io_RandomAccessFile_getFilePointer
  @description Get current file position.
  @parameter Reference to the current (this) object.
  @rvalue Position.
  @end
  */

jlong
java_io_RandomAccessFile_getFilePointer(struct Hjava_io_RandomAccessFile* this)
{
    /*
	off_t r;

	r = lseek(unhand(unhand(this)->fd)->fd, 0, SEEK_CUR);
	if (r < 0) {
		SignalError(NULL, "java.io.IOException", SYS_ERROR);
	}
	return (off_t2jlong(r));
    */
	int cur;

	if( (cur = seek(unhand(unhand(this)->fd)->fd, 0L, 1)) < 0 ){
		SignalError(NULL, "java.io.IOException", SYS_ERROR);
	}
	
	return (jlong)cur;
}

/**
  @function java_io_RandomAccessFile_close
  @description Close file.
  @parameter Reference to the current (this) object.
  @end
  */

void
java_io_RandomAccessFile_close(struct Hjava_io_RandomAccessFile* this)
{
	int r;

	if (unhand(unhand(this)->fd)->fd >= 0) {
		r = close(unhand(unhand(this)->fd)->fd);
		unhand(unhand(this)->fd)->fd = -1;
		if (r < 0) {
			SignalError(NULL, "java.io.IOException", SYS_ERROR);
		}
	}
}
