/* 
 * TiffDocView.m
 *
 * This file provides the code for the TiffDocView, which is the DocView of
 * the ScrollView in the application's main window.    The TiffDocView must
 * communicate with the image it displays, and be able to redraw itself when
 * the image is scrolled.
 *
 * You may freely copy, distribute, and reuse the code in this example.
 * NeXT disclaims any warranty of any kind, expressed or  implied, as to its
 * fitness for any particular use.
 */

#import "TiffDocView.h"
#import  <appkit/NXImage.h>

@implementation TiffDocView
    
- drawSelf:(const NXRect *)rects :(int)rectCount
{
	
    /* 
     * "rects" is passed in to drawSelf::.    It's a pointer to an array of
     * NXRects that describe the area of the TiffDocView that needs to be
     * drawn.  In this case, we can use the very same NXRects to describe the
     * area of the image that must be composited in to the TiffDocView,
     * because:
     *
     * 1.) the NXImage and the TiffDocView are in the same coordinate system,
     * and  
     *
     * 2.) we want to render the rect at the same coordinates that it occupies
     * in the source image. 
     *
     * Since we're only expecting one rect to be modified by a scrolling
     * operation, "rects" actually is a pointer to the one we want.
     * 
     * The composite method requires pointers for its arguments, so we leave
     * "rects" as a pointer and take the address of its origin coordinates.
     */
     [docImage composite:NX_SOVER fromRect:rects toPoint: &rects->origin];
     return self;
}
	
	

/* Read in image data from a file */

- readImageFile:(const char *)imageFile
{
    NXSize			imSize;

    /* Remove old NXImage if there is one */
    if (docImage != nil)
    {
       	[docImage free];
    }
    
    /* Allocate, initialize, and load a new NXImage */
    docImage = [[NXImage alloc] initFromFile: imageFile];    
    
    /* Find out how big the image is */
    [docImage getSize: &imSize];
    
    /* ...and size the TiffDocView to fit. */
    [self  sizeTo: imSize.width : imSize.height];
    
    /* Invoke drawSelf:, as defined above */
    [self display];
    
    return self;
}

@end
