/*
 * This is a simple example of an EasyBTX client without using the AppKit.
 *
 * How to compile:
 *
 * cc -Wall SimpleClient.m -o SimpleClient -lNeXT_s -lsys_s
 *
 * Examples of usage:
 *
 * launch EasyBTX.app and perform an autologin procedure:
 * ./SimpleClient open "account-number" "user" "password"
 *
 * request a page and send some strings:
 * ./SimpleClient "*page-number#" "string1" "string2" ...
 *
 *
 * Written by: Max Boehm, 02/20/95
 *
 * You may freely copy, distribute and reuse the code in this example.  
 * Don't even talk to me about warranties.
 *
 */

#import <appkit/appkit.h>
#import <remote/NXProxy.h>
#import <machkit/NXPort.h>
#import "EasyBTXProtocol.h"


id server;			/* EasyBTX server (NXProxy*) */

char **send_strings;		/* string list */


@interface SimpleClient:Object <btxClientMethods,NXSenderIsInvalid>
{
}
@end


@implementation SimpleClient

- senderIsInvalid:sender
/* This message is sent if connection is broken (server died). */
{
    [sender free];		/* -> run is terminated */
    server=nil;
    return self;
}

- (oneway void) userInputAtRow:(in int)row col:(in int)col
/* This method is called every time when the videotex system expects user
 * input. row and col is the current cursor position.
 */
{
    char *str;
    
    printf("userInputAtRow:%d col:%d\n",row,col);

    /* read ASCII data of row 1 of videotex screen */
    [server readString:&str row:1 col:1 length:40];
    printf("readString:%s\n",str);
    free(str);
    
    if (*send_strings!=NULL) {			/* send next string */
	printf("sendString:%s\n",*send_strings);
	[server sendString:*send_strings++];
    }
    else					/* no more strings -> exit */
	exit(0);
}

@end



void main (int argc, char **argv)
{
    id client;
    
    /* connect to EasyBTXServer */
    
    server=[NXConnection connectToName:"EasyBTXServer"];
    if (!server) {			/* try again */
	NXPortFromName("EasyBTX",NULL);	/* launch EasyBTX.app */
	usleep(1000000);
	server=[NXConnection connectToName:"EasyBTXServer"];
    }
    
    if (!server) {
	fprintf(stderr,"Can't connect to EasyBTX server.\n");
	fprintf(stderr,"Launch EasyBTX or put it in /LocalApps, ~/Apps,...\n");
	exit(-2);
    }
    
    /* setup receiver of senderIsInvalid: and userInputAtRow:col: messages */
    
    client=[SimpleClient new];		/* create the client object */
    
    /* if connection is broken, send senderIsInvalid: message */
    [NXPort worryAboutPortInvalidation];
    [[server connectionForProxy] registerForInvalidationNotification:client];

    [server setProtocolForProxy:@protocol(btxServerMethods)];
    [server setNotify:client];		/* receiver of userInputAtRow:col: */

    /* handle command line arguments */

    if (argc<2) {
	fprintf(stderr,"usage: %s [open] send-string ...\n",argv[0]);
	exit(-1);
    }
    
    if (strcmp(argv[1],"open")==0) {	/* connect to videotex system */
	const char *msg[8] = { "ok", "IO-error", "no/wrong modem echo",
	"unexpected modem answer", "timeout", "device in use",
	"can't open device","connection already established" };
	int r;

	[[server connectionForProxy] setInTimeout:80000];
	r=[server openBTX];		/* dial out */

	if (r!=0)
	    fprintf(stderr,"can't connect to videotex system (%s).\n",msg[-r]);
    }
    else {				/* send first string immediately */
	printf("sendString:%s\n",argv[1]);
	[server sendString:argv[1]];
    }

    send_strings=argv+2;		/* string list (terminated by NULL) */
    
    /* run the connection in the current thread. This handles the
     * userInputAtRow:col: messages.
     */
    [[server connectionForProxy] run];
    
    fprintf(stderr,"run terminated (connection died)!\n");
    exit(-3);
}

