// Format: 80 columns, tabs = 4 spaces
#import <appkit/appkit.h>

/******************************************************************************
	StringList - by Jeff Martin

	StringList (a subclass of Storage) is a convenient way to deal with lists of character strings. It contains methods for adding strings, inserting strings, sorting strings and searching for strings.
		
	StringList can also add strings from a delimited string (such as the ones passed to an app by the workspace). The delimiters can be supplied or assumed to be white space.
	
	Finally, StringList implements a browser delegate method so that it can easily display itself if set to be a browser delegate.

	To use this object inside of InterfaceBuilder, simply drag its icon from the palettes window into the suitcases window. Then drag in a browser and set its delegate to be the stringlist. Type in some entries into StringList's inspector and test interface. The browser fills up!

Written by: Jeff Martin (jmartin@next.com)
You may freely copy, distribute and reuse the code in this example.  
Don't even talk to me about warranties.
******************************************************************************/
	
@interface StringList : Storage
{
	BOOL	isSorted;			// Whether or not the list is currently sorted
}

// Convenience adding methods (defaulting to ifAbsent=NO, noCopy=NO, 
//		sorted=NO and at=count)
- addString:(const char *)string;
- addStringIfAbsent:(const char *)string;
- addStringNoCopy:(const char *)string;
- addStringIfAbsentNoCopy:(const char *)string;

- addString:(const char *)string at:(int)at;
- addStringIfAbsent:(const char *)string at:(int)at;
- addStringNoCopy:(const char *)string at:(int)at;
- addStringIfAbsentNoCopy:(const char *)string at:(int)at;

- addStringSorted:(const char *)string;
- addStringIfAbsentSorted:(const char *)string;
- addStringNoCopySorted:(const char *)string;
- addStringIfAbsentNoCopySorted:(const char *)string;

// The main adding method (all other adds call this)
- addString:(const char *)string ifAbsent:(BOOL)ifAbsent noCopy:(BOOL)noCopy sorted:(BOOL)sorted at:(int)at;

// Adding Multiple strings
- addStrings:(const char *const*)strings;
- addStrings:(const char *const*)strings ifAbsent:(BOOL)ifAbsent noCopy:(BOOL)noCopy sorted:(BOOL)sorted at:(int)at;

// Adding another StringList
- addStringList:stringListObject;
- addStringList:stringListObject ifAbsent:(BOOL)ifAbsent noCopy:(BOOL)noCopy sorted:(BOOL)sorted at:(int)at;

// Adding individual strings from delimited (tab or otherwise) string
- addDelimitedStrings:(const char *)string delimiters:(const char *)dels;
- addDelimitedStrings:(const char *)string delimiters:(const char *)dels ifAbsent:(BOOL)ifAbsent sorted:(BOOL)sorted at:(int)at;

// Finding strings in list
- (const char *const*)strings;
- (const char *)stringAt:(int)at;

// Finding the index of a given string and/or whether it is in list
- (BOOL)stringExists:(const char *)string;
- (unsigned)indexOfString:(const char *)string;
- (unsigned)indexOfString:(const char *)string exists:(BOOL *)exists;

// Removing strings from list
- removeString:(const char *)string;
- removeStrings:(const char *const*)strings;
- (char *)removeStringAt:(int)at;

// Sorting list
- (BOOL)isSorted;
- sortStrings:sender;

// Archiving
- write:(NXTypedStream *)stream;
- read:(NXTypedStream *)stream;

// Cleanup
- freeStrings;
- free;

// Bonus NXBrowser support
- (int)browser:sender fillMatrix:matrix inColumn:(int)column;

@end