/* server and client protocols of EasyBTX V1.1 */

@protocol btxClientMethods

- (oneway void) userInputAtRow:(in int)row col:(in int)col;
/* This method is sent to the client whenever user input is expected by the
 * videotex system. The current cursor position is row (1..24) and col (1..40).
 * The receiver of this method has to be set by the setNotify: server method.
 */

@end

@protocol btxServerMethods

- (int) openBTX;
/* Dial out and connect to the videotex system. Autologin is not done.
 * Return:  0 = ok, connection established
 *         -1 = IO-Error on serial device
 *         -2 = no / wrong echo of AT modem command.
 *         -3 = unexpected modem answer (other than "CONNECT" or "OK")
 *         -4 = timeout
 *         -5 = device in use (lockfile already exists in /usr/spool/uucp/LCK)
 *         -6 = can't open serial device
 *         -7 = connection already established.
 */

- (void) closeBTX;
/* Close the videotex session and hang up the modem. */

- (void) setNotify:(id <btxClientMethods>)obj;
/* Set the receiver of the userInputAtRow:col: message, which is sent whenever
 * user input is expected by the videotex system. obj=nil disables the
 * notification.
 */

- (void) sendString:(const char*)str;
/* Send the string *str to the videotex system. The string may contain control
 * characters as described below. If the cursor is the last row, '*' is sent
 * as INI and '#' is sent as TER. Control characters are:
 * INI      = Command introducer ('*'). '*' is displayed in the last row.
 * TER      = Command terminator ('#') or termination of an data field.
 *            If the cursor is located in an data field, the rest of the field
 *            is overwritten by spaces and the cursor moves to the first
 *            character of the next field. TER at the first position of a
 *            data field does not cancel any entries.
 * APH      = The cursor is moved to the first position of the current data
 *            field. If the cursor is already at this position, it is moved
 *            to the first position of the preceding data field.
 * APB, APF = Cursor is moved one position to the left or to the right.
 * APU, APD = Cursor is moved one position up or down within a data field.
 * APR      = Cursor is moved to the beginning of the next row of the same
 *            data field. If no next row exists, it is moved the the beginning
 *            of the next data field.
 * DCT      = Send the page to the videotex system.
 */

- (void) readString:(out char**)str row:(int)row col:(int)col length:(int)length;
/* Read length characters of the current page starting at position row (1..24)
 * and col (1..40). The videotex characters are converted to NeXT ASCII
 * whenever possible. Graphical characters (DRCS) and characters with no
 * ASCII equivalent are read as space (' ').
 * Memory is allocated for the string and a pointer to the buffer is returned
 * in *str. The string is terminated by '\0'. The string has to be freed by
 * the receiver of this message.
 * Multiple lines may be read, when col+length-1>40.
 */

- (void) getCursorRow:(out int*)row col:(out int*)col;
/* Return the current cursor position in *row (1..24) and *col (1..40). */

- (int)  saveCEPT:(const char*)path;
/* Write the CEPT code of the videotex session to a file.
 * Return:  0 = ok
 *         -1 = can't open (creat) file
 *         -2 = can't write file
 */

- (void) openWindow;
/* Open the EasyBTX window. The user may enter commands. */

- (void) closeWindow;
/* Close the EasyBTX window. The user can't enter commands. */

@end


/* CEPT control characters codes */

#define INI 0x13	/* ini       (*)     */
#define TER 0x1C	/* ter       (#)     */

#define APH 0x1E	/* home      (*021#) */
#define APU 0x0B	/* up        (*022#) */
#define APB 0x08	/* back      (*024#) */
#define APF 0x09	/* forward   (*026#) */
#define APR 0x0D	/* return    (*027#) */
#define APD 0x0A	/* down      (*028#) */

#define DCT 0x1A	/* send page (*029#) */

