.\"
.\" TclX.man
.\"
.\" Extended Tcl binary file search command.
.\"----------------------------------------------------------------------------
.\" Copyright 1992 Karl Lehenbauer and Mark Diekhans.
.\"
.\" Permission to use, copy, modify, and distribute this software and its
.\" documentation for any purpose and without fee is hereby granted, provided
.\" that the above copyright notice appear in all copies.  Karl Lehenbauer and
.\" Mark Diekhans make no representations about the suitability of this
.\" software for any purpose.  It is provided "as is" without express or
.\" implied warranty.
.\"----------------------------------------------------------------------------
.\" $Id: TclX.man,v 2.3 1992/11/17 06:26:44 markd Exp $
.\"----------------------------------------------------------------------------
.\"
.TH "TclX" TCL "" "Tcl"
.ad b
.SH NAME
TclX - Extended Tcl: Extended command set for Tcl
'
.SH "INTRODUCTION"
.PP
This man page contains the documentation for all of the extensions that are
added to Tcl 6.2 by Extended Tcl (TclX 6.2b).
These extensions provide extend Tcl's
capabilities by adding new commands to it, without changing the syntax of
standard Tcl.  Extended Tcl is a superset of standard Tcl and is built
alongside
the standard Tcl sources.  Extended Tcl has three basic functional areas:
A set of new commands, a Tcl shell (i.e. a Unix shell-style command line
and interactive environment),  and a user-extensible library of useful
Tcl procedures, any of which can be automatically loaded on the first
attempt to execute it.
.PP
The command descriptions are separated into several sections:
.RS 5
.TP
o General Commands
.TP
o Unix Access Commands
.TP
o File I/O Commands
.TP
o File Scanning Commands
.TP
o Math Commands
.TP
o List Maninipulation Commands
.TP
o Keyed Lists
.TP
o String and Character Manipulation Commands
.TP
o XPG/3 Message Catalog Commands
.TP
o Tcl Shell and Development Environment
.TP
o Help Facility
.TP
o Tcl Shell Initialization Sequence
.TP
o Tcl Initialization File
.TP
o Tcl Shell Variables
.TP
o Tcl Shell Procedures
.TP
o Tcl Command Autoloading
.TP
o Tcl Package Libraries
.TP
o Tcl Package Library Procedures
.TP
o Tcl Shell Internal Structure
.RE
'
.SH "GENERAL COMMANDS"
.PP
A set of general, useful Tcl commands, includes a command to begin
an interactive session with Tcl, a facility for tracing execution,
and a looping command.
'
'@help: control/commandloop
'@brief: Create an interactive command loop.
.TP
.B commandloop [\fIprompt\fR] [\fIprompt2\fR]
.br
Create an interactive command loop for the current TCL interpreter.  This
command receives commands from stdin and executes them.  It is
useful TCL scripts that do not normally converse interactively with
a user through a Tcl command interpreter, but which sometimes want
to enter this mode.
.sp
\fIPrompt\fR is a Tcl command string that is evaluated to determine
the text of the prompt string.
\fIPrompt2\fR is a command string that is evaluated to determine
the ``downlevel prompt'', which is
the prompt which is issued for continuation input.
When the command
terminates, the variables for the prompt hooks will be set to their old value.
If these arguments are not specified, the prompt hooks use their current
value.  Prompt hooks are TCL code that return as their result the prompt to
output.  The result of the last command executed in the command string (which
may be a \fBreturn\fR) will be used as the prompt.
'@endhelp
'@help: debug/cmdtrace
'@brief: Trace Tcl execution.
.TP
.B cmdtrace \fIlevel\fR|\fBon\fR [\fBnoeval\fR] [\fBnotruncate\fR] [\fBflush\fR] [\fIprocs\fR] [\fIfilehandle\fR]
.br
Print a trace statement for all commands executed at depth of \fIlevel\fR or
below (1 is the top level).  If \fBon\fR is specified, all commands at any
level are traced.  The following options are available:
.RS 5
.TP
.B noeval
Causes arguments to be printed unevaluated.  If \fBnoeval\fR
is specified, the arguments are printed before
evaluation.  Otherwise, they are printed afterwards.
.sp
 If the command line is longer than 60 characters, it is truncated
to 60 and a "..." is postpended to indicate that there was more output
than was displayed.
If an evaluated 
argument contains a space, the entire argument will be enclosed inside of
braces (`{}') to allow the reader to visually separate the arguments from
each other.
.TP
.B notruncate
Disables the truncation of commands and evaluated arguments.
.TP
.B flush
Causes the output buffer to be flushed after each line is printed.  This is
useful when tracing C code that cause an application to abort, making it easy
to narrow the problem down to the command that caused the abort.
.TP
.B procs
Enables the tracing of procedure calls only.  Commands that aren't
procedure calls (i.e. calls to commands that are written in C, C++
or some object-compatible language) are not traced if the \fBprocs\fR
option is specified.  This option is particularly useful for greatly
reducing the output of \fBcmdtrace\fR while debugging.
.TP
.B filehandle
If specified, then the trace output will be written to the file rather than
stdout.
.RE
.TP
.B cmdtrace off
Turn off all tracing.
.TP
.B cmdtrace depth
.br
Returns the current maximum trace level, or zero if trace is disabled.
'@endhelp
'@help: files/echo
'@brief: Echo one or more strings to stdout, followed by a newline.
.TP
.B echo [\fIstr1..\fR]
.br
Writes zero or more strings to standard output, followed by a newline.
'@endhelp
'@help: status/infox
'@brief: Return information about Extended Tcl, or the current application.
.TP
.B infox \fIoption\fR
.br
Return information about Extended Tcl, or the current application.  
The following \fBinfox\fR command options are available:
.RS 5
.TP
.B version
Return the version number of Extended Tcl.  The version number for
Extended Tcl is generated by combining the base version of the standard Tcl
code with a letter indicating the version of Extended Tcl being used.  This is
the documentation for version \fBtcl6.2b\fR.
.TP
.B patchlevel
Return the patchlevel for Extended Tcl.
.TP
.B appname
Return the symbolic application name of the current application linked with
the Extended Tcl library.  The C variable \fBtclAppName\fR must be set by the
application to return an application specific value for this variable.
.TP
.B applongname
Return a natural language name for the current application. The C variable
\fBtclLongAppName\fR must be set by the application to return an application
specific value for this variable.
.TP
.B appversion
Return the version number for the current application.  The C variable
\fBtclAppVersion\fR must be set by the application to return an 
application-specific value for this variable.
.RE
'@endhelp
'@help: control/loop
'@brief: Higher-performance for-style loop.
.TP
.B loop \fIvar first limit [increment] body\fR
.br
\fBLoop\fR is a looping command, similar in behavior to the Tcl \fBfor\fR
statement, except that the \fBloop\fR statement achieves substantially higher
performance and is easier to code when the beginning and ending values of a
loop are known, and the loop variable is to be incremented by a known, fixed
amount every time through the loop.
.sp
 The \fIvar\fR argument is the name of a Tcl variable that will contain the
loop index.  The loop index is set to the value specified by
\fIfirst\fR.  The Tcl interpreter is invoked upon \fIbody\fR zero or more
times, where \fIvar\fR is incremented by \fIincrement\fR every time through
the loop, or by one if \fIincrement\fR is not specified.  \fIIncrement\fR can
be negative in which case the loop will count downwards.
.sp
When \fIvar\fR reaches \fIlimit\fR, the loop terminates without a subsequent
execution of \fIbody\fR.  For instance, if the original \fBloop\fR parameters
would cause \fBloop\fR to terminate, say \fIfirst\fR was one, \fIlimit\fR was
zero and \fIincrement\fR was not specified or was non-negative, \fIbody\fR is
not executed at all and \fBloop\fR returns.
.sp
If a \fBcontinue\fR command is invoked within \fIbody\fR then
any remaining commands in the current execution of \fIbody\fR are skipped,
as in the \fBfor\fR command.
If a \fBbreak\fR command is invoked
within \fIbody\fR
then the \fBloop\fR command will
return immediately.
\fBLoop\fR returns an empty string.
'@endhelp
'
'@help: debug/profile
'@brief: Collect Tcl script performance profile data.
.TP
.B profile [-commands] on
.B profile off \fIarrayVar\fR
.br
This command is used to collect a performance profile of a Tcl script.  It
collects data at the Tcl procedure level. The number of calls to a procedure,
and the amount of real and CPU time is collected. Time is also collected for
the global context.  The procedure data is collected by bucketing it based on
the procedure call stack, this allows determination of how much time is spent
in a particular procedure in each of it's calling contexts.
.sp
The \fBon\fR option enables profile data collection. If the \fB-commands\fR
option is specifed, data on all commands within a procedure is collected
as well a procedures.  Multiple occurrences of a command within a procedure
are not distinguished, but this data may still be useful for analysis.
.sp
The \fBoff\fR option turns off profiling and moves the data collected to the
array \fIarrayVar\fR.  The array is address by a list containing the procedure
call stack.  Element zero is the top of the stack, the procedure that the
data is for.  The data in each entry is a list consisting of the procedure
call count and the real time and CPU time in milliseconds spent in the
procedure (and all procedures it called). The list is in the form {\fIcount
real cpu\fR}.
A Tcl procedure \fBprofrep\fR
is supplied for reducing the data and producing a report (see Tcl Package
Library Procedures section).
'@endhelp
'
.SH "UNIX ACCESS COMMANDS
.PP
These commands provide access to many basic Unix facilities, including process
handling, date and time processing, signal handling,
linking and unlinking files,
setting file, process, and user attributes, and the executing
commands via the shell.
'@help: files/chgrp
'@brief: Change file group.
.TP
.B chgrp \fIgroup\fR \fIfilelist\fR
.br
Set the
group id of each file in the list \fIfilelist\fR to \fIgroup\fR, which can
be either a
group name or a numeric group id.
'@endhelp
'
'@help: files/chmod
'@brief: Set file permissions.
.TP
.B chmod \fImode\fR \fIfilelist\fR
.br
Set permissions of each of the files in the list \fIfilelist\fR to \fImode\fR,
where \fImode\fR is an absolute numeric mode or symbolic permissions as in the
UNIX \fBchmod(1)\fR command.  To specify a mode as octal, it should be
prefixed with a "0" (e.g. 0622).
'@endhelp
'
'@help: files/chown
'@brief: Change file owner and/or group.
.TP
.B chown \fIowner\fR|{\fIowner group\fR} \fIfilelist\fR
.br
Set owner of each file in the list \fIfilelist\fR to \fIowner\fR, which can
be a user
name or numeric user id.  If the first parameter is a list, then the
owner is set to the first element of the list and the group is set to the
second element.  \fIGroup\fR can be a group name or numeric group id.
If \fIgroup\fR is {}, then the file group will be set to the login
group of the specified user.
'@endhelp
'
'@help: time/convertclock
'@brief: Parse and convert a date and time string to integer clock value.
.TP
.B convertclock \fIdateString\fR [\fBGMT\fR|{}] [\fIbaseClock\fR]
.br
Convert \fIdateString\fR to an integer clock value (see \fBgetclock\fR).
This command can parse and convert virtually any standard date and/or time
string, which can include standard time zone mnemonics.  If only a time is
specified, the current date is assumed.  If the string does not contain a
time zone mnemonic, the local time zone is assumed, unless the \fBGMT\fR 
argument is specified, in which case the clock value is calculated assuming
that the specified time is relative to Greenwich Mean Time.
If \fIbaseClock\fR is specified,
it is taken as the current clock value.  This is useful for determining the
time on a specific day.  Some examples are:
.sp
.nf
.ft CW
    convertclock "14 Feb 92"
    convertclock "Feb 14, 1992 12:20 PM PST"
    convertclock "12:20 PM, Feb 14, 1992"
.ft R
.fi
'@endhelp
'
'@help: processes/execl
'@brief: Perform a process exec, executing a file.
.TP
.B execl \fIprog\fR [\fIarglist\fR]
.br
Do an execl, replacing the current program (either Extended Tcl or an
application with Extended Tcl embedded into it) with \fIprog\fR and
passing the arguments in the list \fIarglist\fR.
'@endhelp
'
'@help: time/fmtclock
'@brief: Convert an integer time value to human-readable format.
.TP
.B fmtclock \fIclockval\fR [\fIformat\fR] [\fBGMT|{}\fR]
.br
Converts a Unix integer time value, typically returned by
\fBgetclock\fR, \fBconvertclock\fR, or the \fBatime\fR, \fBmtime\fR,
or \fBctime\fR options of the \fBfile\fR command, to human-readable
form.  The \fIformat\fR argument is a string that describes how the
date and time are to be formatted.
Field descriptors consist of a ``%'' followed by a field
descriptor character.  All other characters are copied into the result.
Valid field descriptors are:
.sp
.nf
.ft CW
    %% - Insert a %.
    %a - Abbreviated weekday name.
    %A - Full weekday name
    %b - Abbreviated month name.
    %B - Full month name.
    %d - Day of month (01 - 31).
    %D - Date as %m/%d/%y.
    %e - Day of month (1-31), no leading zeros.
    %h - Abbreviated month name.
    %H - Hour (00 - 23).
    %I - Hour (00 - 12).
    %j - Day number of year (001 - 366).
    %m - Month number (01 - 12).
    %M - Minute (00 - 59).
    %n - Insert a new line.
    %p - AM or PM.
    %r - Time as %I:%M:%S %p.
    %R - Time as %H:%M.
    %S - Seconds (00 - 59).
    %t - Insert a tab.
    %T - Time as %H:%M:%S.
    %U - Week number of year (01 - 52), Sunday is the first 
         day of the week.
    %w - Weekday number (Sunday = 0).
    %W - Week number of year (01 - 52), Monday is the first 
         day of the week.
    %x - Local specific date format.
    %X - Local specific time format.
    %y - Year within century (00 - 99).
    %Y - Year as ccyy (e.g. 1990)
    %Z - Time zone name.
.fi
.ft R
.sp
If format is not specified, "%a %b %d %H:%M:%S %Z %Y" is used.  If \fBGMT\fR
is specified, the time will be formated as Greenwich Mean Time. If the
argument is not specified or is empty, then the local timezone will be used as
defined by the TIMEZONE environment variable.
'@endhelp
'@help: time/getclock
'@brief: Return current date and time as an integer value.
.TP
.B getclock
.br
Return the current date and time as a system-dependent integer value.  The
unit of the value is seconds, allowing it to be used for relative time
calculations.
'@endhelp
'@help: status/id
'@brief: Access, set or convert process, user and group information.
.TP
.B id options
.br
This command provides a means of getting, setting and converting user, group
and process ids.  The \fBid\fR command has the following options:
'
.RS 5
.TP
.B id user \fR[\fIname\fR]
.TP
.B id userid \fR[\fIuid\fR]
Set the real and effective user ID to \fIname\fR or \fIuid\fR, if the
name (or uid) is valid and permissions allow it.  If the name (or uid)
is not specified, the current name (or uid) is returned.
.TP
.B id convert userid \fIuid\fR
.TP
.B id convert user \fIname\fR
Convert a user ID number to a user name, or vice versa.
.TP
.B id group \fR[\fIname\fR]
.TP
.B id groupid \fR[\fIgid\fR]
Set the real and effective group ID to \fIname\fR or \fIgid\fR, if the
name (or gid) is valid and permissions allow it.  If the group name
(or gid) is not specified, the current group name (or gid) is returned.
.TP
.B id convert groupid \fIgid\fR
.TP
.B id convert group \fIname\fR
Convert a group ID number to a group name, or vice versa.
.TP
.B id effective user
.TP
.B id effective userid
Return the effective user name, or effective user ID number, respectively.
.TP
.B id effective group
.TP
.B id effective groupid
Return the effective group name, or effective group ID number, respectively.
.TP
.B id process
.br
Return the process ID of the current process.
.TP
.B id process parent
.br
Return the process ID of the parent of the current process.
.TP
.B id process group
.br
Return the process group ID of the current process.
.TP
.B id process group set
.br
Set the process group ID of the current process to its process ID.
.RE
'@endhelp
'
'@help: processes/kill
'@brief: Send a signal to the specified process.
.TP
.B kill [\fIsignal\fR] \fIprocesslist\fR
.br
Send a signal to the each process in the list \fIprocesslist\fR, if permitted.
\fISignal\fR, if present, is the signal number or the symbolic name of the
signal, see the signal system call manual page.  The leading ``SIG''
is optional
when the signal is specified by its symbolic name.
The default for \fIsigno\fR is 15, SIGTERM.
'@endhelp
'@help: files/link
'@brief: Create a link to a file.
.TP
.B link [\fB-sym\fR] \fIsrcpath destpath\fR
.br
Create a directory entry, \fIdestpath\fR, linking it to the existing file,
\fIsrcpath\fR.  If \fB-sym\fR is specified, a symbolic link, rather than
a hard link, is created.  (The \fB-sym\fR option is only available
on systems that support symbolic links.)
'@endhelp
'@help: files/mkdir
'@brief: Create a new directory
.TP
.B mkdir [-path] \fIdirList\fR
.br
Create each of the directories in the list \fIdirList\fR.  The mode on the
new directories is 777, modified by the umask.  If \fB-path\fR is specified,
then any non-existent parent directories in the specified path(s) are also
created.
'@endhelp
'@help: processes/system
'@brief: Execute command via `system' call.
.TP
.B system \fIcommand\fR
.br
Executes \fIcommand\fR via the \fIsystem\fR(3) call.  Differs from \fBexec\fR
in that \fBsystem\fR doesn't return the executed command's standard output
as the
result string, and
\fBsystem\fR goes through the default shell to provide wildcard
expansion, redirection, etc,
as is normal from an \fBsh\fR command line.  The exit code of
the command is returned.
'@endhelp
'@help: time/times
'@brief: Get process and child execution times.
.TP
.B times
.br
Return a list containing the process and child execution times in the form:
.br
    \fIutime stime cutime cstime\fR
.br
Also see the \fItimes\fR(2) system call manual page.
The values are in milliseconds.
'@endhelp
'@help: status/umask
'@brief: Get or set the file-creation mode mask.
.TP
.B umask [\fIoctalmask\fR]
.br
Sets file-creation mode mask to the octal value of \fIoctalmask\fR.
If \fIoctalmask\fR is omitted, the current mask is returned.
'@endhelp
'@help: files/unlink
'@brief: Delete (unlink) files.
.TP
.B unlink [\fB-nocomplain\fR] \fIfilelist\fR
.br
Delete (unlink) the files whose names are in the list \fIfilelist\fR.
If \fB-nocomplain\fR is specified, then errors will be ignored.
'@endhelp
'@help: processes/wait
'@brief: Wait for a child process to terminate.
.TP
.B wait \fIpid\fR
.br
Waits for the process identified by process id \fIpid\fR to terminate,
either due to an untrapped signal or
call to \fIexit\fR system call.  \fBWait\fR returns a list containing
three elements:  The first element is the process id of the process
that terminated.
If the process exited normally, the second element is `EXIT',
and the third contains the numeric exit code.
If the process terminated due to a signal,
the second element is `SIG', and the third contains
the signal name.  If the process is
currently stopped (on systems that support SIGSTP), the second element
is `STOP', followed by the signal name.
'@endhelp
'
.SH "FILE I/O COMMANDS"
.PP      
These commands extend the stdio-style file I/O capabilities present in Tcl 6.2
and above.
These extensions include searching ASCII-sorted data files, copying files,
duplicating file descriptors, control of file access options, retrieving open
file status, and creating pipes with the \fBpipe\fR system call.  An interface
to the \fBselect\fR system call is available on Unix systems that support
it.
.PP
It should be noted that Tcl file I/O is implemented on top of the stdio 
library.  By default, the file is buffered.  When communicating to a process
through a pipe, a \fBflush\fR command should be issued to force the data
out.  Alternatively, the \fBfcntl\fR command may be used to set the buffering
mode of a file to line-buffered or unbuffered.
'@help: files/bsearch
.TP
.B bsearch \fIfilehandle key\fR [\fIretvar\fR] [\fIcompare_proc\fR]
.br
Search an opened file containing lines of text sorted into ascending
order for a match.  \fIFilehandle\fR is a Tcl filehandle as returned
by the \fBopen\fR command.  \fIKey\fR contains the string to match.
If \fIretvar\fR is specified, then the line from the
file is returned in \fIretvar\fR, and the command returns \fB1\fR if \fIkey\fR
was found, and \fB0\fR if it wasn't.  If \fIretvar\fR is not specified
or is a null name, then the command returns the line that was found, or an
empty string if \fIkey\fR wasn't found.
.sp
By default, the key is matched against the first white-space separated field
in each line.  The field is treated as an ASCII string.  If \fIcompare_proc\fR
is specified, then it
defines the name of a Tcl procedure to evaluate against each
line read from the sorted file during the execution of the
\fBbsearch\fR command.  \fICompare_proc\fR takes two arguments, the key
and a line extracted from the file.  The compare routine should return a
number less than zero if the key is less than the line, zero if the key
matches the line, or greater than zero if the key is greater than the line.
The file must be sorted in ascending order according to the same criteria
\fIcompare_proc\fR uses to compare the key with the line, or errouenous
results will occur.
'@endhelp
'@help: files/copyfile
'@brief: Copy the remainder of one open file into another.
.TP
.B copyfile \fIhandle1 handle2\fR
.br
Copies the rest of the file specified by \fIhandle1\fR, starting
from its current position,
to the file specified by \fIhandle2\fR, starting from its current
position.
'@endhelp
'@help: files/dup
'@brief: Duplicate an open filehandle.
.TP
.B dup \fIfilehandle\fR [\fIstdhandle\fR]
.br
Duplicate an open file.  A file handle is created that addresses the
same file as \fIfilehandle\fR.
.sp
A special case is allowed for duping files to stdin, stdout or stderr.
If \fIstdhandle\fR is specified, then it must contain either
\fBstdin\fR, \fBstdout\fR, or \fBstderr\fR.  In this form, the
file corresponding to \fIstdhandle\fR is
closed, and the dup is performed from \fIfilehandle\fR with the
result going to \fIstdhandle\fR.
.sp
The procedure shown below will create a child process and
set its standard input and output filehandles to a pair
of pipe filehandles we pass as arguments.
Finally the program does an \fBexecl\fR of a specified command,
with the program's stdin and stdout coming from and going to
our pair of pipes.
.sp
.nf
.ft CW
    proc ChildProcess {cmd inPipe outPipe} {
        if {[set childPid [fork]] == 0} {
            close stdin
            dup $inPipe stdin
            close $inPipe

            close stdout
            dup $outPipe stdout
            close $outPipe

            execl $cmd
            # will never make it here...
        }
        return $childPid
    }
.ft R
.fi
'@endhelp
'@help: files/fcntl
'@brief: Get or set file access options for an open file.
.TP
.B fcntl \fIhandle\fR \fIattribute [value\fR] 
.br
This command either sets or clears a file option or returns its current
value.  If \fIvalue\fR are not specified, then the current value of
\fBattribute\fR is returned. The following attributes may be specified:
.IP
\fBRDONLY\fR - The file is opened for reading only. (Get only)
.sp
\fBWRONLY\fR - The file is opened for writing only.  (Get only)
.sp
\fBRDWR\fR - The file is opened for reading and writing.  (Get only)
.sp
\fBREAD\fR - If the file is readable. (Get only).
.sp
\fBWRITE\fR - If the file is writable. (Get only).
.sp
\fBAPPEND\fR - The file is opened for append-only writes.  All writes will
be forced to the end of the file.
.sp
\fBNONBLOCK\fR - The file is to be accessed with non-blocking I/O.  See the
\fBread\fR system call for a description of how it affects the behavior of
file reads.
.sp
\fBCLOEXEC\fR - Close the file on an process exec.  If the \fBexecl\fR
command or some other mechanism causes the process to do an
exec, the file will
be closed if this option is set.
.sp
\fBNOBUF\fR - The file is not buffered. If set, then there no stdio buffering
for the file.
.sp
\fBLINEBUF\fR - Output the file will be line buffered. The buffer will
be flushed when a newline is written, when the buffer is full,
or when input is requested.
.sp
The \fBAPPEND\fR, \fBNONBLOCK\fR, and \fBCLOEXEC\fR attributes may be set or
cleared by specifying the attribute name and a value \fB1\fR to set the
attribute and \fB0\fR to clear it.
.sp
The \fBNOBUF\fR and \fBLINEBUF\fR attributes may only be set (a value of
\fB1\fR) and only one of the options may be selected.
Once set, it may not be
changed.  These options should be set before any I/O operations have been done
on the file or data may be lost.
'@endhelp
'
'@help: files/flock
'@brief: Lock all or part of a file.
.TP
.B flock \fIoptions handle\fR [\fIstart\fR] [\fIlength\fR] [\fIorigin\fR]
.br
This command places a lock on all or part of the file specified by
\fIhandle\fR.  The lock is either advisory or mandatory, depending on the mode
bits of the file.  The lock is placed beginning at relative byte offset
\fIstart\fR for \fIlength\fR bytes.  If \fIstart\fR or \fIlength\fR is omitted
or empty, zero is assumed.  If \fIlength\fR is zero, then the lock always
extents to end of file, even if the file grows.  If \fIorigin\fR is
"\fBstart\fR", then the offset is relative to the beginning of the file. If it
is "\fBcurrent\fR", it is relative to the current access position in the file.
If it is "\fBend\fR", then it is relative to the end-of-file (a negative is
before the EOF, positive is after).  If \fIorigin\fR is omitted, \fBstart\fR
is assumed.
.sp
The following \fIoptions\fR are recognized:
.IP
\fB-read\fR - Place a read lock on the file.  Multiple processes may be
accessing the file with read-locks.
.IP
\fB-write\fR - Place a write lock on the file.  Only one process may be
accessing a file if there is a write lock.
.IP
\fB-nowait\fR - If specified, then the process will not block if the lock
can not be obtained.  With this option, the command returns 1 if the lock
is obtained and 0 if it is not.
.sp
See your system's \fBfcntl\fR system call documentation for full details of
the behavior of file locking.  If locking is being done on ranges of a
file, it is best to use unbuffered file access (see the \fBfcntl\fR command).
'@endhelp
'@help: files/funlock
'@brief: Remove a lock from part or all of a file.
.TP
.B funlock \fIhandle\fR [\fIstart\fR] [\fIlength\fR] [\fIorigin\fR]
.br
Remove a locked from a file that was previously placed with the \fIflock\fR
command.  The arguments are the same as for the \fIflock\fR command, see
that command for more details.
'@endhelp
'
'@help: files/fstat
'@brief: Obtain status information about an open file.
.TP
.B fstat \fIhandle\fR [\fIitem\fR]|[\fBstat \fIarrayvar\fR]
.br
Obtain status information about an open file.
.sp
The following keys are used to identify data items:
.br
.IP
o\ \fBatime\fR - The time of last access.
.IP
o\ \fBctime\fR - The time of last file status change
.IP
o\ \fBdev\fR - The device containing a directory for the file.  This value
uniquely identifies the file system that contains the file.
.IP
o\ \fBgid\fR - The group ID of the file's group.
.IP
o\ \fBino\fR - The inode number.  This field uniquely identifies the file in a
given file system.
.IP
o\ \fBmode\fR - The mode of the file (see the \fBmknod\fR system call).
.IP
o\ \fBmtime\fR - Time when the data in the file was last modified.
.IP
o\ \fBnlink\fR - The number of links to the file.
.IP
o\ \fBsize\fR - The file size in bytes.
.IP
o\ \fBtty\fR - If the file is associated with a terminal, then 1 otherwise 0.
.IP
o\ \fBtype\fR - The type of the file in symbolic form, which is one of the
following values: \fBfile\fR, \fBdirectory\fR, \fBcharacterSpecial\fR,
\fBblockSpecial\fR, \fBfifo\fR, \fBlink\fR, or \fBsocket\fR.
.IP
o\ \fBuid\fR - The user ID of the file's owner.
.sp
If one of these keys is specified as \fIitem\fR, then that data item is
returned
.sp
If \fBstat \fIarrayvar\fR is specified, then the information is returned in
the array \fIarrrayvar\fR.  Each of the above keys indexes an element of the
array containing the data.
.sp
If only \fIhandle\fR is specified, the command returns the data as a keyed
list.
'@endhelp
'
'@help: files/lgets
'@brief: Read a list for a file, handles embedded newlines.
.TP
.B lgets \fIfileId\fR [\fIvarName\fR]
.br
Reads the next Tcl list from the file given by \fIfileId\fR and discards
the terminating newline character.  This command differs from the \fBgets\fR
command, in that it reads Tcl lists rather than lines.  If the list
contains a newline, then that newline will be returned as part of the result.
Only a newline not quoted as part of the list indicates the end of the list.
There is no corresponding command for outputing lists, as \fBputs\fR will
do this correctly.
If \fIvarName\fR is specified, then the line is placed in the variable
by that name and the return value is a count of the number of characters
read (not including the newline).
If the end of the file is reached before reading
any characters then \-1 is returned and \fIvarName\fR is set to an
empty string.
If \fIvarName\fR is not specified then the return value will be
the line (minus the newline character) or an empty string if
the end of the file is reached before reading any characters.
An empty string will also be returned if a line contains no characters
except the newline, so \fBeof\fR may have to be used to determine
what really happened.
'@endhelp
'
.SH "FILE SCANNING COMMANDS"
.PP
These commands provide a facility to scan files, matching lines of the file
against regular expressions and executing Tcl code on a match.  With this
facility you can use Tcl to do the sort of file processing
that is traditionally done with \fIawk\fR.  And since Tcl's approach is more
declarative, some of the scripts that can be rather
difficult to write in awk are simple to code in Tcl.
.PP
File scanning in Tcl centers around the concept of a \fIscan context\fR.
A scan context contains one or more match statements, which associate
regular expressions to scan for with Tcl code to be executed when the
expressions are matched.
'
'@help: filescan/scancontext
'@brief: Manage file scan contexts.
.TP
.B scancontext [\fIoption\fR]
.br
This command manages file scan contexts.  A scan context is a collection of
regular expressions and commands to execute when that regular expression
matches a line of the file.  A context may also have a single default match,
to be applied against lines that do not match any of the regular expressions.
Multiple scan contexts may be defined and they may be reused on multiple files.
A scan context is identified by a context handle.  The \fBscancontext\fR
command takes
the following forms:
.TP
.B scancontext create
Create a new scan context.  The \fBscanmatch\fR command is used to define
patterns in the context.  A contexthandle is returned, which the Tcl
programmer uses to refer to the newly created scan context in calls
to the Tcl file scanning commands.
'
.TP
.B scancontext delete \fIcontexthandle\fR
.br
Delete the scan context identified by \fIcontexthandle\fR, and free all
of the
match statements and compiled regular expressions associated with the
specified context.
'@endhelp
'@help: filescan/scanfile
'@brief: Scan a file, executing match code when their patterns are matched.
.TP
.B scanfile \fIcontexthandle\fR \fIfilehandle\fR
.br
Scan the file specified by \fIfilehandle\fR, starting at the
current file position.  Check all patterns in the scan context specified by
\fIcontexthandle\fR against
it, executing the match commands corresponding to patterns matched.
'@endhelp
'@help: filescan/scanmatch
'@brief: Specify tcl code to execute when scanfile pattern is matched.
.TP
.B scanmatch [\fB-nocase\fR] \fIcontexthandle\fR [\fIregexp\fR] \fIcommands\fR
.br
Specify Tcl \fIcommands\fR, to be evaluated when \fIregexp\fR is matched by a
\fBscanfile\fR command.  The match is added to the scan context specified by
\fIcontexthandle\fR.  Any number of match statements may be
specified for a give context.  \fIRegexp\fR is a regular expression (see the
\fBregexp\fR command).  If \fB-nocase\fR is specified as the first argument,
the pattern is matched regardless of
alphabetic case.
.sp
If \fIregexp\fR is not specified, then a default match is
specified for the scan context.  The default match will be executed when a
line of the file does not match any of the regular expressions
in the current scancontext.
.sp
The array \fBmatchInfo\fR is available to the Tcl code that is executed
when an expression matches (or defaults).  It contans information about
the file being scanned and where within it the expression was matched.
.sp
\fBmatchInfo\fR is local to the top
level of the match command unless declared global at that level
by the Tcl \fBglobal\fR command.  If it is to
be used as a global, it \fImust\fR be declared global before \fBscanfile\fR is
called (since \fBscanfile\fR sets the \fBmatchInfo\fR before the match code is
executed, a subsequent \fBglobal\fR will override the local variable).
The following array entries are available:
.RS 5
.TP
.B matchInfo(line)
Contains the text of the line of the file that was matched.
.TP
.B matchInfo(offset)
The byte offset into the file of the first character of
the line that was matched.
.TP
.B matchInfo(linenum)
The line number of the line that was matched. This is relative to the first
line scanned, which is usually, but not necessarily, the first line of the
file.  The first line is line number one.
.TP
.B matchInfo(handle)
The file handle of the file currently being scanned.
.RE
.PP
All \fBscanmatch\fR patterns that match a line will be processed in the order
in which their
specifications were added to the scan context.  The remainder of the
\fBscanmatch\fR pattern-command pairs may be skipped for a file line if a
\fBcontinue\fR is executed by the Tcl code of a preceding, matched
pattern.
.sp
If a \fBreturn\fR is
executed in the body of the match command, the \fBscanfile\fR command
currently in
progress returns, with the value passed to \fBreturn\fR as its
return value.
'@endhelp
'
.SH "MATH COMMANDS"
.PP
These commands make many additional math functions available in Tcl, including
minimum, maximum, trig functions, exponent, logarithm, square root,
and more. An integer random number generator is provided as well.
.PP
The results of floating point calculations will remain floating point numbers
(containing a decimal point) and will be accurate to the precision of a
\fIdouble\fR.
.PP
All of these commands (except random), take floating point expressions, thus
the command can be used without the \fBexpr\fR command.  This greatly eases
the construction of expressions.  For example:
.sp
.RS 5
set x [sin {1.0 - [cos 3.4]}]
.RE
'
'@help: math/acos
'@brief: Return the arccosine of a number.
.TP
.B acos \fIexpr\fR
.br
Returns the arccosine of \fIexpr\fR.  \fIexpr\fR is in radians.
'@endhelp
'@help: math/asin
'@brief: Return the arcsin of a number.
.TP
.B asin \fIexpr\fR
.br
Returns the arcsin of \fIexpr\fR.  \fIexpr\fR is in radians.
'@endhelp
'@help: math/atan
'@brief: Return the arctangent of a number..
.TP
.B atan \fIexpr\fR
.br
Returns the arctangent of \fIexpr\fR.  \fIexpr\fR is in radians.
'@endhelp
'@help: math/ceil
'@brief: Return the smallest integer not less than a floating point number.
.TP
.B ceil \fIexpr\fR
.br
Returns the smallest integer not less than \fIexpr\fR (floating point).
'@endhelp
'@help: math/cos
'@brief: Return the cosine of a number.
.TP
.B cos \fIexpr\fR
.br
Returns the cosine of \fIexpr\fR.  \fIexpr\fR is in radians.
'@endhelp
'@help: math/cosh
'@brief: Return the hyperbolic cosine of a number.
.TP
.B cosh \fIexpr\fR
.br
Returns the hyperbolic cosine of \fIexpr\fR.
'@endhelp
'@help: math/exp
'@brief: Return e to the power of a number.
.TP
.B exp \fIexpr\fR
.br
Returns \fIe\fR to the power of \fIexpr\fR.
'@endhelp
'@help: math/fabs
'@brief: Return the absolute value of the floating point number.
.TP
.B fabs \fIexpr\fR
.br
Returns the absolute value of \fIexpr\fR (floating point).
'@endhelp
'@help: math/floor
'@brief: Return the largest integer not greater than a floating point number.
.TP
.B floor \fIexpr\fR
.br
Returns the largest integer not greater than \fIexpr\fR (floating point).
'@endhelp
'@help: math/fmod
'@brief: Perform a floating point modulus operation.
.TP
.B fmod \fInum1\fR \fInum2\fR
.br
Returns \fInum1\fR modulo \fInum2\fR.
'@endhelp
'@help: math/max
'@brief: Return the argument that has the highest numeric value.
.TP
.B max \fInum1 num2\fR [..\fInumN\fR]
.br
Returns the argument that has the highest numeric value. The arguments,
\fInumN\fR may be any interger or floating point values.
'@endhelp
'@help: math/min
'@brief: Return the argument that has the lowest numeric value.
.TP
.B min \fInum1 num2\fR [..\fInumN\fR]
.br
Returns the argument that has the lowest numeric value. The arguments,
\fInumN\fR may be any interger or floating point values.
'@endhelp
'@help: math/log
'@brief: Return the natural logarithm of a number.
.TP
.B log \fIexpr\fR
.br
Returns the natural logarithm of \fIexpr\fR.
'@endhelp
'@help: math/log10
'@brief: Return the logarithm base 10 of a number.
.TP
.B log10 \fIexpr\fR
.br
Returns the logarithm base 10 of \fIexpr\fR.
'@endhelp
'@help: math/pow
'@brief: Return a number to the power of another number.
.TP
.B pow \fInum1\fR \fInum2\fR
.br
Returns \fInum1\fR to the power of \fInum2\fR.
'@endhelp
'@help: math/random
'@brief: Return a pseudorandom integer or set the seed.
.TP
.B random \fBlimit\fR | \fBseed\fR [\fIseedval\fR]
.br
Generate a pseudorandom integer number greater than or equal to zero and
less than \fIlimit\fR.  If \fBseed\fR is specified, then the command
resets the random number generator to a starting point derived from 
the \fBseedval\fR. This allows one to reproduce 
pseudorandom number sequences
for testing purposes.
If \fIseedval\fR is omitted, then the seed is set to a value based on current
system state and the current time, providing a reasonably interesting and
ever-changing seed.
'@endhelp
'@help: math/sin
'@brief: Return the sin of a number.
.TP
.B sin \fIexpr\fR
.br
Returns the sin of \fIexpr\fR.  \fIexpr\fR is in radians.
'@endhelp
'@help: math/tan
'@brief: Return the tangent of a number.
.TP
.B tan \fIexpr\fR
.br
Returns the tangent of \fIexpr\fR.  \fIexpr\fR is in radians.
'@endhelp
'@help: math/sinh
'@brief: Return the hyperbolic sin of a number.
.TP
.B sinh \fIexpr\fR
.br
Returns the hyperbolic sin of \fIexpr\fR.
'@endhelp
'@help: math/sqrt
'@brief: Return the square root of a number.
.TP
.B sqrt \fIexpr\fR
.br
Returns the square root of \fIexpr\fR.
'@endhelp
'@help: math/tanh
'@brief: Return the hyperbolic tangent of a number.
.TP
.B tanh \fIexpr\fR
.br
Returns the hyperbolic tangent of \fIexpr\fR.
'@endhelp
'
.SH "LIST MANINIPULATION COMMANDS"
.PP
Extended Tcl provides two additional list manipulation commands.
'@help: lists/lempty
'@brief: Determine if a list is empty. 
.TP
.B lempty \fIlist\fR
.br
Determine if the specified list is empty.
If empty, 1 is returned, otherwise, 0 is returned.  This command is an
alternative to comparing a list to an empty string.
'@endhelp
'
'@help: lists/lvarcat
'@brief: Concatenate the contents lists or strings into a variable
.TP
\fBlvarcat\fI var string [\fR\fIstring...\fR]
.br
This command treats each \fIstring\fR argument as a list and concatenates them
to the end of the contents of \fIvar\fR, forming a a single list.  The list is
stored back into \fIvar\fR and also returned as the result.  if \fIvar\fR does
not exist, it is created.
'@endhelp
'
'@help: lists/lvarpop
'@brief: Pop or replace the specified element from a list.
.TP
.B lvarpop \fIvar\fR [\fIindex\fR \fR[\fIstring\fR]]
.br
The \fBlvarpop\fR command pops (deletes) the element indexed by
\fIindex\fR from the list contained in the variable \fIvar\fR.
If \fIindex\fR is omitted, then 0 is assumed.
If \fIstring\fR, is specified, then the deleted element is replaced by 
\fIstring\fR. The replaced or deleted element is returned.
Thus ``lvarpop argv 0'' returns the first element of argv, setting
argv to contain the remainder of the string.
'@endhelp
'
'@help: lists/lvarpush
'@brief: Push or insert the an element into a list.
.TP
.B lvarpush \fIvar string\fR [\fIindex\fR]
.br
The \fBlvarpush\fR command pushes (inserts) \fIstring\fR as an element in the
list contained in the variable \fIvar\fR.  The element is inserted before
position \fIindex\fR in the list. If \fIindex\fR is omitted, then 0 is
assumed.  If \fIvar\fR does not exists, it is created.
'@endhelp
'
'@help: intro/keyedlists
'@brief: Introduction to keyed lists
.SH "KEYED LISTS"
.PP
Extended Tcl defines a special type of list referred to as \fIkeyed lists\fR.
These lists provided a structured data type built upon standard Tcl lists.
This provides a functionality similar to \fIstruct\fRs in the C 
programming language.
.sp
A keyed list is a list in which each element contains a key and value pair.
These element pairs are stored as lists themselves, where the key is the first
element of the list, and the value is the second.  The key-value pairs are
refered to as \fIfields\fR.
This is an example of a keyed list:
.IP
    {{NAME {Frank Zappa}} {JOB {musician and composer}}}
'
.PP
If the variable \fBperson\fR contained the above list, then
\fBkeylget person NAME\fR would return \fB{Frank Zappa}\fR.
Executing the command:
.IP
\fBkeylset person ID 106\fR
.PP
would make \fBperson\fR contain 
.IP
    {{ID 106} {NAME {Frank Zappa}} {JOB {musician and composer}}
.PP
Fields may contain subfields; `.' is the seperator character.  Subfields
are actually fields where the value is another keyed list.  Thus 
the following list has the top level fields \fIID\fR and \fINAME\fR, and
subfields \fINAME.FIRST\fR and  \fINAME.LAST\fR:
.IP
    {ID 106} {NAME {{FIRST Frank} {LAST Zappa}}}
.PP
There is no limit to the recursive depth of subfields, allowing one
to build complex data structures.
.PP
Keyed lists are constructed and accessed via a number of commands.
All keyed list management commands take the name of the variable containing
the keyed list
as an argument (i.e. passed by reference), rather than passing the
list directly.
'@endhelp
'
'@help: keyedlists/keyldel
'@brief: Delete a field of a keyed list.
.TP
.B keyldel \fIlistvar\fR \fIkey\fR
.br
Delete the field specified by \fIkey\fR from the keyed list in the
variable \fIlistvar\fR.  This removes both the key and the value from
the keyed list.
'@endhelp
'
'@help: keyedlists/keylget
'@brief: Get the value of a field of a keyed list.
.TP
.B keylget \fIlistvar\fR [\fIkey\fR [\fIretvar\fR | {}]]
.br
Return the value associated with \fIkey\fR from the keyed list in the
variable \fIlistvar\fR.  If \fIretvar\fR is not specified, then the value will
be returned as the result of the command.  
In this case, if \fIkey\fR is not found in the
list, an error will result.
.sp
If \fIretvar\fR is specified and \fIkey\fR is in
the list, then the value is returned in the variable \fIretvar\fR and the
command returns \fB1\fR if the key was present within the list.
If \fIkey\fR isn't in the list, the command will return \fB0\fR,
and \fIretvar\fR will be left unchanged.  If \fB{}\fR is specified for
\fIretvar\fR, the value is not returned, allowing the Tcl programmer
to determine if a key is present in a keyed list without setting a
variable as a side-effect.
.sp
If \fIkey\fR is omitted, then a list of all the keys in
the keyed list is returned.
'@endhelp
'
'@help: keyedlists/keylkeys
'@brief: Get the keys in a keyed list.
.TP
.B keylkeys \fIlistvar\fR [\fIkey\fR]
.br
Return the a list of the keyes in the keyed list in the
variable \fIlistvar\fR.  If \fIkeys\fR is specified, then it is the
name of a key field  who's subfield keys are to be retrieve.
'@endhelp
'
'@help: keyedlists/keylset
'@brief: Set the value of a field of a keyed list.
.TP
\fBkeylset\fR \fIlistvar\fR \fIkey\fR \fIvalue\fR [\fIkey2\fR \fIvalue2\fR ...]
.br
Set the value associated with \fIkey\fR, in the keyed list contained in the
variable \fIlistvar\fR, to \fIvalue\fR.
If \fRlistvar\fR does not exists, it is created.  If \fIkey\fR
is not currently in the list, it will be added.  If it already exists, 
\fIvalue\fR replaces the existing value.  Multiple keywords and values may
be specified, if desired.
'@endhelp
'
.SH "STRING AND CHARACTER MANIPULATION COMMANDS"
.PP
The commands provide additional functionality to classify characters, convert
characters between character and numeric values, index into a string,
determine the length of a string, extract a range of character from a string,
replicate a string a number of times, and transliterate a string (similar to
the Unix \fItr\fR program).
'
'@help: strings/cindex
'@brief: Return indexed character from string.
.TP
.B cindex \fIstring indexExpr\fR
.br
Returns the character indexed by the expression \fIindexExpr\fR (zero based)
from \fIstring\fR. This command is a shortcut for:
.sp
.nf
.ft CW
    \fBstring index\fR \fIstring\fR [\fBexpr\fI indexExpr]\fR
.ft R
.fi
'@endhelp
'@help: strings/clength
'@brief: Return length of specified string.
.TP
.B clength \fIstring\fR
.br
Returns the length of \fIstring\fR in characters.  
This command is a shortcut for:
.sp
.nf
.ft CW
    \fBstring length\fR \fIstring\fR
.ft R
.fi
'@endhelp
'@help: strings/crange
'@brief: Return range of characters from string.
.TP
.B crange \fIstring firstExpr lastExpr\fR
.br
Returns a range of characters from \fIstring\fR starting at the character
indexed by the expression \fIfirstExpr\fR (zero-based) until the character
indexed by the expression \fIlastExpr\fR.  The special keyword \fBend\fR may
be specified for \fIlast\fR to indicate that the remainder of the string is to
be extracted.  This command is a shortcut for:
.sp
.nf
.ft CW
    \fBstring range\fR \fIstring\fR [\fBexpr\fI firstExpr\fR] [\fBexpr\fI lastExpr]\fR
.ft R
.fi
'@endhelp
'@help: strings/csubstr
'@brief: Return a substring from within a string.
.TP
.B csubstr \fIstring firstExpr lengthExpr\fR
.br
Returns a range of characters from \fIstring\fR starting at the character
indexed by the expression \fIfirstExpr\fR (zero-based) for \fIlengthExpr\fR
characters.  The special keyword \fBend\fR may be specified for
\fIlengthExpr\fR to indicate that the remainder of the string is to be
extracted.  This command is a short cut for:
.sp
.nf
.ft CW
    \fBstring range\fR \fIstring\fR [\fBexpr\fI firstExpr\fR] [\fBexpr\fR \fIlength\fB-1\fR]
.ft R
.fi
'@endhelp
'@help: strings/ctype
'@brief: Determine if a string has various characteristics.
.TP
.B ctype \fIclass string\fR 
.br
\fBctype\fR
determines whether all characters in \fIstring\fR are of the specified
\fIclass\fR.  It returns \fB1\fR if they are all of \fIclass\fR,
and \fB0\fR if
they are not, or if the string is empty.  This command also provides another
method (besides \fBformat\fR and \fBscan\fR) of converting between an ASCII
character and its numeric value.  The following \fBctype\fR commands are
available:
.RS 5
.TP
.B ctype alnum \fIstring\fR
Tests that all characters are alphabetic or numeric characters as defined by
the character set.
.TP
.B ctype alpha \fIstring\fR
Tests that all characters are alphabetic characters as defined by the
character set.
.TP
.B ctype ascii \fIstring\fR
Tests that all characters are an ASCII character (a non-negative number less
than 0200).
.TP
.B ctype char \fInumber\fR
Converts the numeric value, \fIstring\fR, to an ASCII character.  Number must
be in the range 0 through 255.
.TP
.B ctype cntrl \fIstring\fR
Tests that all characters are ``control characters'' as defined by the
character set.
.TP
.B ctype digit \fIstring\fR
Tests that all characters are valid decimal digits, i.e. 0 through 9.
.TP
.B ctype graph \fIstring\fR
Tests that all characters within are any character for which \fIctype print\fR
is true, except for space characters.
.TP
.B ctype lower \fIstring\fR
Tests that all characters are lowercase letters as defined by the character
set.
.TP
.B ctype ord \fIcharacter\fR
Convert a character into its decimal numeric value.  The string must be one
character long.
.TP
.B ctype space \fIstring\fR
Tests that all characters are either a space, horizontal-tab, carriage
return, newline, vertical-tab, or form-feed.
.TP
.B ctype print \fIstring\fR
Tests that all characters are a space or any character for which \fIctype
alnum\fR or \fIctype punct\fR is true or other ``printing character'' as
defined by the character set.
.TP
.B ctype punct \fIstring\fR
Tests that all characters are made up of any of the characters
other than the ones for which
\fBalnum\fR, \fBcntrl\fR, or \fBspace\fR is true.
.TP
.B ctype upper \fIstring\fR
Tests that all characters are uppercase letters as defined by the character
set.
.TP
.B ctype xdigit \fIstring\fR
Tests that all characters are valid hexadecimal digits, that is \fI0\fR
through \fI9\fR, a through \fIf\fR or \fIA\fR through \fIF\fR.
.RE
'@endhelp
'@help: strings/replicate
'@brief: Replicate string a number of times.
.TP
.B replicate \fIstring countExpr\fR
.br
Returns \fIstring\fR, replicated the number of times indicated by the
expression \fIcountExpr\fR.
'@endhelp
'
'@help: strings/translit
'@brief: Translate characters in a string according to patterns.
.TP
.B translit \fIinrange outrange string\fR
.br
Translate characters in \fIstring\fR, changing characters
occuring
in \fIinrange\fR
to the corresponding character in \fIoutrange\fR. \fIInrange\fR and 
\fIoutrange\fR may be list of characters or a range in the form `A-M'.
For example:
.nf
.ft CW
        translit a-z A-Z foobar
.ft R
.if
returns "FOOBAR".
'@endhelp
.SH "TCL SHELL AND DEVELOPMENT ENVIRONMENT"
.sp
'@help: tclshell/intro
'@brief: Introduction to the tcl shell.
.B tcl [-qn] [[-f] \fIscript\fB]|[-c \fIcommand\fB] [\fIargs\fR]
.PP
\fBTcl\fR starts the interactive Tcl command interpreter.  The Tcl shell
provides an environment for writing, debugging and executing Tcl scripts.  The
functionality of the Tcl shell can be easily obtained by any application
that includes Tcl.
.PP
The \fBtcl\fR command, issued without any arguments,
invokes an interactive Tcl shell, allowing the user to interact
directly with Tcl, executing any Tcl commands at will and viewing
their results.
.PP
If \fIscript\fR is specified, then the script is executed noninteractively
with any additional arguments, \fIargs\fR, being supplied in the
global Tcl
variable `\fBargv\fR'.  If 
\fIcommand\fR is supplied, then this command (or semicolon-separated
series of commands) is executed, with `\fBargv\fR' containing
any \fIargs\fR.
.PP
The Tcl shell is intended as an environment for Tcl program development and
execution.  While it is not a full-featured interactive shell, it provides a
comfortable environment for the interactive development of Tcl code.  Note
that the package library code described here overrides the \fBunknown\fR
command provided as part of the standard Berkeley Tcl library facility,
although Tcl source libraries coded to that standard can be loaded and used by
Extended Tcl.
.PP
The following command line flags are recognized by the Tcl shell command
line parser:
.TP
.B \-q
Quick initialization flag.  If this flag is set the only initialization step
performed is to locate the Tcl default file and save its name in the Tcl
variable TCLDEFAULT.  The file is not evaluated nor is the TclInit.tcl file.
This provides for a fast startup, but does not make any of the standard
Tcl procedures and facilities available.
.TP
.B \-n
No procedure call stack dump.  The procedure call stack will not be displayed
when an error occurs, only the error message.  Useful in the #! line of
already debugged scripts.
.TP
.B \-f
Takes the next argument as a script for Tcl to source, rather than entering
interactive
mode.  The \fB-f\fR flag is optional.  Normally the first argument that does
not start with a `-' is taken as the script to execute unless the `-c' option
is specified.  Any following arguments are passed to the script
via \fBargv\fR,
thus any other Tcl shell command-line flags must precede this option.
.TP
.B \-c
Take the next argument as a Tcl command to execute.  It may contain series of 
commands to execute, separated by `;'.  Any following arguments are passed in
\fBargv\fR, thus, as with \fB-f\fR, any other Tcl shell flags must precede
this option.
.TP
.B \-\-
Mark the end of the arguments to the Tcl shell. All arguments following this
are passed in the Tcl variable \fBargv\fR.  This is useful to pass arguments
without attempting to execute a Tcl script.
.PP
The result string returned by a command executed from the Tcl shell command
line is
normally echoed back to the user.  If an error occurs, then the string result
is displayed, along with the error message.  The error message will be
preceded by the string ``Error:''.
.PP
The \fBset\fR command is a special case.  If the command is called to set
a variable (i.e. with two arguments), then the result will not be echoed.
If only one argument, the name of a variable, is supplied to \fBset\fR, then
the result will be echoed.
.PP
If an unknown Tcl command is entered from the command line, then the Unix
command path, specified in the environment variable \fBPATH\fR,
will be searched for a command of the same name.
If the command is found, it will be executed with any arguments remaining
on the Tcl command line being passed as arguments to the command.
This feature is provided to enhance the interactive environment for developing
Tcl scripts.
.PP
Automatic execution of programs in this manner is only supported from the
command line, not in script files
or in procedures, to reduce confusion and mistakes while programming
in Tcl.  Scripts should use the Tcl \fBexec\fR or \fBsystem\fR 
commands to run Unix commands.
'@endhelp
'
.SH "TCL SHELL INITIALIZATION SEQUENCE"
'@help: tclshell/initialize
'@brief: Tcl shell initialization sequence.
.PP
The standard Tcl shell initialization consists of the following steps:
.IP
Search for a default file.  The default file is a Tcl script that is executed
to set important Tcl variables that govern startup, such as \fBTCLPATH\fR.
The current Extended Tcl version, represented by \fI$ver\fR,
is included
in the default file name to allow multiple Tcl versions to exists on a
system.  Note that the Extended Tcl version is the standard Tcl version number
with an alphabetic character added to indicated the version of Extended Tcl.
(The \fBinfox version\fR command will return this version number.)
.PP
The Tcl default file is searched for using the following algorithm:
.sp
.RS 10
- An environment variable, \fBTCLDEFAULT\fR, is checked for.  If present,
it is used as the name of the
default file.
.sp
- A global default file, which
is usually either \fB/etc/default/tcl\fI$ver\fR or
\fB/usr/local/lib/tcldefault\fI$ver\fR.  (Tcl may be compiled to use a
different directory if desired).
.RE
.IP
First, \fBTcl\fR executes the default file.  This file normally sets at 
least two Tcl variables:
\fBTCLPATH\fR, which contains a list of directories that contain tcl source
files and libraries, and \fBTCLINIT\fR, the
full pathname of the Tcl source file that performs Tcl initialization.
Other site-specific variables may also be set in
this file.
.IP
Next, \fBtcl\fR executes the initialization file specified by the Tcl 
variable \fBTCLINIT\fR.
This is normally the \fBTclInit.Tcl\fR file distributed with Tcl.  This
file defines the Tcl environment, including standard Extended
Tcl procedures and variables.
'@endhelp
'
.SH "TCL INITIALIZATION FILE"
'@help: tclshell/tclinit
'@brief: Tcl shell initialization file.
.PP
The Tcl initialization file, normally \fITclInit.tcl\fR in the main Tcl script
directory, initializes the Tcl shell.  It defines various Tcl procedures
that are required to implement loading of Tcl source from libraries via the
\fIpackage library\fR and \fIautoload\fR facilities.
.PP
If the Tcl is invoked interactively, it will source a file
named \fI.tclrc\fR in the
user's home directory, if it exists.  Tcl is viewed primarly as a programming
language, not an interactive shell, so the \fI.tclrc\fR is intended for use for
loading development utilities, not to support applications,
which should not have
to rely
on the user's environment in such a manner.
'@endhelp
'
.SH "TCL SHELL VARIABLES"
'@help: tclshell/variables
'@brief: Tcl shell variables.
.PP
The following variables are set and/or used by the Tcl shell.
.TP
.B argv
A list containing the arguments passed in from the command line, excluding
arguments used by the Tcl shell.  The first element is the first passed
argument, not the program name.
.TP
.B interactiveSession
Set to \fB1\fR if Tcl shell is invoked interactively, or \fB0\fR if the
Tcl shell is directly executing a script.  Normally checked by scripts so
that they can function as a standalone application if specified on the
command line, but merely load in and not execute if loaded during an
interactive invocation of Tcl.
.TP
.B noAutoExec
If this variable exists and has a value of \fB1\fR, then the Tcl shell will not
attempt to exec an non-existent command as a shell command.
.TP
.B programName
The name that the Tcl shell is executing as.  All directory
components are removed from the program name.
.TP
.B TCLDEFAULT
Name of the default file that was used to locate all other files used by the
Tcl shell.
.TP
.B TCLENV
Array that contains information used internally by various Tcl procedures that
are part of the Tcl shell.  Entries that you might want to modify are defined
here.  Don't change any other entries.
.TP
.B TCLENV(topLevelPromptHook)
Contains code to run to generate the prompt used when interactively prompting
for commands.  The code in this hook will be evaluated and the result will be
used for the prompt.
.TP
.B TCLENV(downLevelPromptHook)
Contains code to run to generate the prompt used when interactively prompting
for continuation of an incomplete command.  The code in this hook
will be evaluated and the result will be used for the prompt
.TP
.B TCLINIT
The name of the file that initializes, and does startup processing of,
Tcl.
.TP
.B TCLPATH
Path to search to locate Tcl scripts.  Used by the \fBload\fR and 
\fBdemand_load\fR commands.  The path is a Tcl list (i.e.
zero or more space-separated directory names).
'@endhelp
'
.SH "TCL SHELL PROCEDURES"
.PP
The follow additional procedures are defined by the Tcl shell and may be 
generally useful:
.TP
'@help: libraries/load
'@brief: Search the TCLPATH for a file to source.
.B load \fIfile\fR
Source a file, as with the \fBsource\fR command, except search \fBTCLPATH\fR
for the file.
'@endhelp
.TP
'@help: libraries/utilprocs
'@brief: Search a path list for a file.
.B searchpath \fIpath file\fR
Search all directories in the specified path, which is a Tcl list, for the
specified file.  Returns the full path name of the file, or an empty string
if the requested file could not be found.
'@endhelp
'
.TP

