/* Copyright (c) 1999 Thorsten Kukuk
   Author: Thorsten Kukuk <kukuk@suse.de>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <sys/time.h>
#include "nis_db.h"
#include "db_private.h"
#include "xdb.h"

static db_status
__add_entry (table_t *tobj, const entry_obj *entry)
{
  xdb_logaction action;
  db_status retval;
  entry_t *eobj;
  XDR xdrs;

  retval = __open_log (tobj);

  action = LOG_ADD;

  xdrstdio_create (&xdrs, tobj->fp_log, XDR_ENCODE);
  if (!xdr_xdb_logaction (&xdrs, &action))
    {
      xdr_destroy (&xdrs);
      fclose (tobj->fp_log);
      tobj->fp_log = NULL;
      return DB_INTERNAL_ERROR;
    }

  if (!__nisdb_xdr_entry_obj (&xdrs, entry))
    {
      xdr_destroy (&xdrs);
      fclose (tobj->fp_log);
      tobj->fp_log = NULL;
      return DB_INTERNAL_ERROR;
    }
  xdr_destroy (&xdrs);

  eobj = malloc (sizeof (entry_t));
  if (eobj == NULL)
    return DB_MEMORY_LIMIT;

  eobj->object = __clone_entry (entry);
  if (eobj->object == NULL)
    return DB_MEMORY_LIMIT;
  retval = calc_entry_hash (tobj, eobj);
  if (retval != DB_SUCCESS)
    {
      xdr_free ((xdrproc_t) __nisdb_xdr_entry_obj, (caddr_t) eobj->object);
      free (eobj->object);
      free (eobj);
      return retval;
    }
  eobj->next = tobj->entry;
  tobj->entry = eobj;

  return DB_SUCCESS;
}

db_result *
db_add_entry (const char *table_name, const u_int numattrs,
	      const nis_attr *attrs, const entry_obj *entry)
{
  struct timeval t_start, t_end;
  table_t *tobj;
  db_result *res, *res2;

  gettimeofday (&t_start, NULL);

  if ((res = calloc (1, sizeof (db_result))) == NULL)
    return NULL;

  if (table_name == NULL || strlen (table_name) == 0)
    {
      res->status = DB_BADTABLE;
      goto bailout;
    }

  /* The Sun implementation requires that at least one attribute
     be specified even if we don't want to use any. */
  if (numattrs == 0 || attrs == NULL)
    {
      res->status = DB_BADQUERY;
      goto bailout;
    }

  /* If the user specified an attributes, he wants to do a
     replacement. Find the entry to be replaced (can't replace
     more than one). */
  res2 = db_list_entries (table_name, numattrs, attrs);
  if (res2 == NULL)
    {
      res->status = DB_MEMORY_LIMIT;
      goto bailout;
    }

  if (res2->status != DB_SUCCESS)
    {
      /* It's Ok if we doesn't found an entry. But if the search
         ends in an error, bail out with this error code. */
      if (res2->status != DB_NOTFOUND)
	{
	  res->status = res2->status;
	  db_free_result (res2);
	  goto bailout;
	}
      db_free_result (res2);
    }
  else
    {
      if (res2->objects.objects_len > 1)
	{
	  res->status = DB_NOTUNIQUE;
	  db_free_result (res2);
	  goto bailout;
	}

      db_free_result (res2);
      res2 = db_remove_entry (table_name, numattrs, attrs);
      if (res2 == NULL)
	{
	  res->status = DB_MEMORY_LIMIT;
	  goto bailout;
	}
      if (res2->status != DB_SUCCESS)
	{
	  res->status = DB_INTERNAL_ERROR;
	  db_free_result (res2);
	  goto bailout;
	}
      db_free_result (res2);
    }


  /* Get the table entry to add the new entry */
  tobj = __get_table (table_name);
  if (tobj == NULL)
    {
      res->status = DB_BADTABLE;
      goto bailout;
    }

  res->status = __add_entry (tobj, entry);
  if (res->status != DB_SUCCESS)
    goto bailout;

bailout:
  gettimeofday (&t_end, NULL);
  res->ticks +=
    (t_end.tv_usec - t_start.tv_usec) +
    ((t_end.tv_sec - t_start.tv_sec) * 1000000);

  return res;
}
