/****************************************************************************** 
 *
 * File:        smfrecord.c
 * Version:     $Id: smfrecord.c,v 1.20 1995/07/28 15:34:08 burgaard Exp $
 *              $Version: 0.5$
 *
 * Purpose:     Standard MIDI file recorder.
 *
 * Project:     MIDI/Sequencer library.
 *              Roland MPU-401 Device driver.
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 * Copyrights:  Copyright (c) 1995 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************
 *
 * Description: This is a ``little'' example application that utilizes my Roland
 *              MPU-401 Device Driver.
 *
 *              This example demonstrates recording.
 *
 *              It should give a pretty clear picture of what you can do with
 *              /dev/mpu401data if you ignore all the MIDI file parsing.
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <signal.h>
#include <time.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include "mpu401.h"
#include "mpuioctl.h"

#include "midiconst.h"
#include "midiqueue.h"
#include "midiprint.h"
#include "midifile.h"

/*** GLOBAL VARIABLES *********************************************************/

static char nooverdub   = 0;
static word bpm         = 100; char dobpm = 0;
static word numerator   = 4;   char dotimesig = 0;
static word denominator = 4;
static char metronome   = 0; /* 0 = off, 1 = GS metro, 2 = GM metro */
static char gmsysex     = 0;
static char gssysex     = 0;

static int  mpudev    = 0;
static int  filec     = 0;

static char isrecord   = 0;

static midi_file        midifile;
static midi_file_timing timing;
static midi_file_data   queues;

static midi_file        play_midifile;
static midi_file_timing play_timing;
static midi_file_data   play_queues;

/*
 * # of events to transfer to driver per demand.
 *
 * If you think replay gets sloppy sometimes, especially if a lot
 * of notes is being played at once, try make this amount bigger.
 * Recommended range is 50 to 200 events. There's no point in making
 * it bigger than 200 events (except for momentarily vasting kernel-
 * memory! ;-)
 */
#define PACK_SIZE 50

byte gmon[6] = {0xf0,0x7e,0x7f,0x09,0x01,0xf7};
byte gsreset[11] = {0xf0,0x41,0x10,0x42,0x12,0x40,0x00,0x7f,0x00,0x41,0xf7};

/*** ARGUMENT PARSING *********************************************************/

void usage(int exitcode)
{
  printf("Standard MIDI File Recorder version 0.5\n");
  printf("Copyright (c) 1995 Kim Burgaard\n");
  printf("\n");
  printf("Usage: smfrecord [options] filename\n");
  printf("\n");
  printf("  -t --title <title>    Set song title.\n");
  printf("  -c --copy <copyright> Set copyright notice.\n");
  printf("\n");
  printf("  -s --source <file>    Overdub record with file as source.\n");
  printf("  -n --nooverdub        Save recorded input seperatly (no overdub).\n");
  printf("\n");
  printf("  -b --bpm <tempo>      Set tempo in beats per minute\n");
  printf("  -T --timesig <n/d>    Set time signature to n/d, where d is 1, 2, 4, 8 or 16\n");
  printf("  -m --metronome        Turn on metronome (GS)\n");
  printf("  -M --gmmetronome      Turn on metronome (GM)\n");
  printf("\n");
  printf("  -g --gmsysex          Add General MIDI On SysEx message\n");
  printf("  -G --gssysex          Add GS reset SysEx message\n");
  printf("\n");
  printf("  -h --help             This help text.\n");
  exit(exitcode);
}

inline char isoption(char *s)
{
  return  (s && (strlen(s) > 1) && (s[0] == '-'));
}

inline char islongoption(char *s)
{
  return  (s && (strlen(s) > 2) && (s[0] == '-') && (s[1] == '-'));
}

int parsearg(int argc, char *argv[])
{
  int i = 1;
  int j = 0;
  int k = 0;

  if (argc < 2)
    {
      fprintf(stderr, "Try `smfrecord --help' for more information\n");
      exit(-1);
    };
  while (i < argc)
    {
      if ( islongoption(argv[i]) )
	{
	  if ( !strcmp("--nooeverdub", argv[i]) ) nooverdub = 1;
	  else if ( !strcmp("--title", argv[i]) )
	    {
	      if (midifile.title != NULL)
		{
		  fprintf(stderr, "Title already set to: %s\n", midifile.title);
		  usage(1);
		}
	      else if (++i >= argc)
		{
		  fprintf(stderr, "Title expected after --title\n");
		  usage(1);
		};
	      midifile.title = argv[i];
	    }
	  else if ( !strcmp("--copy", argv[i]) )
	    {
	      if (midifile.copyright != NULL)
		{
		  fprintf(stderr, "Copyright notice already set to: %s\n", midifile.copyright);
		  usage(1);
		}
	      else if (++i >= argc)
		{
		  fprintf(stderr, "Copyright notice expected after --copy\n");
		  usage(1);
		};
	      midifile.copyright = argv[i];
	    }
	  else if ( !strcmp("--source", argv[i]) )
	    {
	      if (play_midifile.name != NULL)
		{
		  fprintf(stderr, "Source file already given: %s\n", play_midifile.name);
		  usage(1);
		}
	      else if (++i >= argc)
		{
		  fprintf(stderr, "Source file name expected after --source\n");
		  usage(1);
		};
	      play_midifile.name = argv[i];
	    }
	  else if ( !strcmp("--bpm", argv[i]) )
	    {
	      if (++i >= argc)
		{
		  fprintf(stderr, "Tempo argument expected after --bpm\n");
		  usage(1);
		};
	      if (sscanf(argv[i], "%u", &bpm) != 1)
		{
		  fprintf(stderr, "Invalid integer argument to --bpm: %s\n", argv[i]);
		  usage(1);
		};
	      if ((bpm < 8) || (bpm > 240))
		{
		  fprintf(stderr, "Argument to --bpm out of range: %s\n", argv[i]);
		  fprintf(stderr, "Must be between 8 and 240!\n");
		  usage(1);
		};
	      dobpm = 1;
	    }
	  else if ( !strcmp("--timesig", argv[i]) )
	    {
	      if (++i >= argc)
		{
		  fprintf(stderr, "Time signature argument expected after --timesig\n");
		  usage(1);
		};
	      if (sscanf(argv[i], "%u/%u", &numerator, &denominator) != 2)
		{
		  fprintf(stderr, "Invalid integer argument to --timesig: %s\n", argv[i]);
		  usage(1);
		};
	      if ((numerator < 1) || (numerator > 127))
		{
		  fprintf(stderr, "Numerator argument to --timesig out of range: %s\n", argv[i]);
		  fprintf(stderr, "Must be between 1 and 127!\n");
		  usage(1);
		}
	      else if (numerator > 16)
		{
		  fprintf(stderr, "Warning: Do you really want a numerator on %d?\n", numerator);
		};
	      switch (denominator)
		{
		case 1:
		case 2:
		case 4:
		case 8:
		case 16:
		  break;
		default:
		  fprintf(stderr, "Denominator argument to --timesig out of range: %s\n", argv[i]);
		  fprintf(stderr, "Must be 1, 2, 4, 8 or 16!\n");
		  usage(1);
		  break;
		};
	      dotimesig = 1;
	    }
	  else if ( !strcmp("--metronome", argv[i]) )
	    {
	      if (metronome != 2) metronome = 1;
	      else fprintf(stderr, "Warning: Ambigeous option --metronome and --gmmetronome\n");
	    }
	  else if ( !strcmp("--gmmetronome", argv[i]) )
	    {
	      if (metronome != 1) metronome = 2;
	      else fprintf(stderr, "Warning: Ambigeous option --gmmetronome and --metronome\n");
	    }
	  else if ( !strcmp("--gmsysex", argv[i]) ) gmsysex = 1;
	  else if ( !strcmp("--gssysex", argv[i]) ) gssysex = 1;
	  else if ( !strcmp("--help", argv[i]) ) usage(0);
	  else
	    {
	      fprintf(stderr, "Unknown option `%s':\n\n", argv[i]);
	      fprintf(stderr, "Try `smfrecord --help' for more information\n");
	      exit(-1);
	    };
	}
      else if ( isoption(argv[i]) )
	{
	  k = strlen(argv[i]);
	  j = 1;
	  while (j < k)
	    {
	      switch (argv[i][j])
		{
		case 't':
		  if (midifile.title != NULL)
		    {
		      fprintf(stderr, "Title already set to: %s\n", midifile.title);
		      usage(1);
		    }
		  else if (++i >= argc)
		    {
		      fprintf(stderr, "Title expected after -t\n");
		      usage(1);
		    };
		  j = k;
		  midifile.title = argv[i];
		  break;
		case 'c':
		  if (midifile.copyright != NULL)
		    {
		      fprintf(stderr, "Copyright notice already set to: %s\n", midifile.copyright);
		      usage(1);
		    }
		  else if (++i >= argc)
		    {
		      fprintf(stderr, "Copyright notice expected after -c\n");
		      usage(1);
		    };
		  j = k;
		  midifile.copyright = argv[i];
		  break;
		case 's':
		  if (play_midifile.name != NULL)
		    {
		      fprintf(stderr, "Source file already given: %s\n", play_midifile.name);
		      usage(1);
		    }
		  else if (++i >= argc)
		    {
		      fprintf(stderr, "Source file name expected after -s\n");
		      usage(1);
		    };
		  j = k;
		  play_midifile.name = argv[i];
		  break;
		case 'n':
		  nooverdub = 1;
		  break;
		case 'b':
		  if (++i >= argc)
		    {
		      fprintf(stderr, "Tempo argument expected after -b\n");
		      usage(1);
		    };
		  j = strlen(argv[i-1]);
		  if (sscanf(argv[i], "%u", &bpm) != 1)
		    {
		      fprintf(stderr, "Invalid integer argument to --bpm: %s\n", argv[i]);
		      usage(1);
		    };
		  if ((bpm < 8) || (bpm > 240))
		    {
		      fprintf(stderr, "Argument to --bpm out of range: %s\n", argv[i]);
		      fprintf(stderr, "Must be between 8 and 240!\n");
		      usage(1);
		    };
		  dobpm = 1;
		  break;
		case 'T':
		  if (++i >= argc)
		    {
		      fprintf(stderr, "Time signature argument expected after -T\n");
		      usage(1);
		    };
		  j = k;
		  if (sscanf(argv[i], "%u/%u", &numerator, &denominator) != 2)
		    {
		      fprintf(stderr, "Invalid integer argument to -T: %s\n", argv[i]);
		      usage(1);
		    };
		  if ((numerator < 1) || (numerator > 127))
		    {
		      fprintf(stderr, "Numerator argument to -T out of range: %s\n", argv[i]);
		      fprintf(stderr, "Must be between 1 and 127!\n");
		      usage(1);
		    }
		  else if (numerator > 16)
		    {
		      fprintf(stderr,"Warning: Do you really want a numerator on %d?\n",numerator);
		    };
		  switch (denominator)
		    {
		    case 1:
		    case 2:
		    case 4:
		    case 8:
		    case 16:
		      break;
		    default:
		      fprintf(stderr, "Denominator argument to -T out of range: %s\n", argv[i]);
		      fprintf(stderr, "Must be 1, 2, 4, 8 or 16!\n");
		      usage(1);
		      break;
		    };
		  dotimesig = 1;
		  break;
		case 'm':
		  if (metronome != 2) metronome = 1;
		  else fprintf(stderr,"Warning: Ambigeous option --metronome and --gmmetronome\n");
		  break;
		case 'M':
		  if (metronome != 1) metronome = 2;
		  else fprintf(stderr,"Warning: Ambigeous option --gmmetronome and --metronome\n");
		  break;
		case 'g':
		  gmsysex = 1;
		  break;
		case 'G':
		  gssysex = 1;
		  break;
		case 'h':
		  usage(0);
		  break;
		default:
		  fprintf(stderr, "Unknown option `-%c'\n\n", argv[i][j]);
		  fprintf(stderr, "Try `smfrecord --help' for more information\n");
		  exit(-1);
		};
	      j++;
	    };
	}
      else
	{
	  filec++;
	  midifile.name = argv[i];
	};
      i++;
    };

  if (filec > 1)
    {
      fprintf(stderr, "You may only specify one desination file\n\n");
      fprintf(stderr, "Try `smfrecord --help' for more information\n");
      exit(-1);
    }
  else if (filec < 1)
    {
      fprintf(stderr, "A desination file is required\n\n");
      fprintf(stderr, "Try `smfrecord --help' for more information\n");
      exit(-1);
    };

  return 0;
}

/*** MEMORY HANDLING **********************************************************/

char * my_malloc(long size)
{
  char * tmp = NULL;
  if (!size) return NULL;
  if ( !(tmp = malloc(size)) )
    {
      fprintf(stderr, "Out of memory!\n");
      exit(-3);
    };

  return tmp;
}

/*** MIDI FILE HANDLING *******************************************************/

void put_gmon_sysex(midi_queue * queue, long time)
{
  byte * data = (byte *)my_malloc(6);
  memcpy(data, &gmon[0], 6);
  midi_queue_put_event(queue, MIDI_TYP_SYSEX, time, 6, data);
};

void put_gsreset_sysex(midi_queue * queue, long time)
{
  byte * data = (byte *)my_malloc(11);
  memcpy(data, &gsreset[0], 11);
  midi_queue_put_event(queue, MIDI_TYP_SYSEX, time, 11, data);
};

void put_tempo(midi_queue * queue, long time, byte bpm)
{
  long j = 0;
  byte * data = (byte *)my_malloc(4);
  j = (60.0 * 1000000.0)/(float)(bpm);
  data = (byte *)my_malloc(4);
  data[0] = MID_META_SET_TEMPO;
  data[1] = (j >> 16) & 0xff;
  data[2] = (j >> 8) & 0xff;
  data[3] = j & 0xff;
  midi_queue_put_event(queue, MIDI_TYP_META, time, 4, data);
};

void put_time_sig(midi_queue * queue, long time, byte numerator, byte denominator)
{
  byte j = 0;
  byte k = 0;
  byte * data = (byte *)my_malloc(5);
  for (j = 0, k = 1; k < denominator; j++) k *= 2;
  data = (byte *)my_malloc(5);
  data[0] = MID_META_TIME_SIG;
  data[1] = numerator;
  data[2] = j;
  data[3] = 24;
  data[4] = 8;
  midi_queue_put_event(queue, MIDI_TYP_META, time, 5, data);
};

int load(void)
{
  midi_queue queue;
  midi_event * event;
  long time;
  byte thisbpm = 0;
  byte thisnum = 0;
  byte thisden = 0;
  int ch, i, j;

  if ( !(play_midifile.file = fopen(play_midifile.name, "rb")) )
    {
      fprintf(stderr, "Could not open `%s': ", play_midifile.name);
      perror("");
      return -EINVAL;
    };

  play_midifile.title = play_midifile.copyright = NULL;
  
  play_midifile.format = 0;
  play_midifile.tracks = 0;
  play_timing.division = 96;
  play_timing.newdivision = 120;
  play_timing.time = 0.0;
  
  midi_queue_flush(&play_queues.voice);
  midi_queue_flush(&play_queues.sysex);
  midi_queue_flush(&play_queues.meta);

  play_queues.dovoice = play_queues.dosysex = play_queues.dometa = 1;
  
  if ( (i = midi_load(&play_midifile, &play_timing, &play_queues, 0)) &&
       ferror(play_midifile.file))
    {
      fprintf(stderr, "Error while reading `%s': ", play_midifile.name);
      perror("");
      i = ferror(play_midifile.file);
    };
  fclose(play_midifile.file);

  /* remove any tempo and time sig settings in 1st measure! */
  midi_queue_reset(&queue);
  while ( (event = midi_queue_get(&play_queues.meta)) )
    {
      switch (event->data[0])
	{
	case MID_META_SET_TEMPO:
	  j  = (event->data[1]&0xff) << 16;
	  j |= (event->data[2]&0xff) << 8;
	  j |= (event->data[3]&0xff);
	  thisbpm = ( 60.0 * 1000000.0 / (float)j );
	  midi_event_free(event);
	  break;
	case MID_META_TIME_SIG:
	  thisnum = event->data[1];
	  for (j = 0, thisden = 1; j < event->data[2]; j++) thisden *= 2;
	  midi_event_free(event);
	  break;
	default:
	  midi_queue_put(&queue, event);
	};
      if (event->time > (4*play_timing.division)) break;
    };
  while ( (event = midi_queue_get(&queue)) ) midi_queue_put(&play_queues.meta, event);

  /* if the source file already contains GM ON and GS Reset sysex messages 
   * then ignore those options */
  time = 1;
  event = play_queues.sysex.head;
  while (event && (event->time < 4*play_timing.division))
    {
      if (event->size == 6 && memcmp(event->data, &gmon[0], 6) == 0) gmsysex = 0;
      else if (event->size == 11 && memcmp(event->data, &gsreset[0], 11) == 0) gssysex = 0;
      event = event->next;
    };

  /* check if an option supplied tempo collides with source file tempo */
  if (thisbpm && !dobpm)
    {
      dobpm = thisbpm != 100;
      bpm = thisbpm;
    };
  if (dobpm && (thisbpm != bpm))
    {
      fprintf(stderr,"\nWarning: Source file defines tempo in first measure:\n");
      fprintf(stderr, "   Source file tempo = %3d, option supplied tempo = %3d\n", thisbpm, bpm);
      do
	{
	  fprintf(stderr, "\nOverride source file tempo with option tempo? (y/n) ");
	  ch = toupper(getchar());
	  while (ch == '\n' || ch == '\r') ch = toupper(getchar());
	}
      while ((ch != 'Y') && (ch != 'N'));
      if (ch == 'Y') bpm = thisbpm;
    };

  /* check if an option supplied time sig. collides with a source file time sig. */
  if (thisnum && thisden && !dotimesig)
    {
      dotimesig = (thisnum != 4) && (thisden != 4);
      numerator = thisnum;
      denominator = thisden;
    };
  if (dotimesig && ((thisnum != numerator) || (thisden != denominator)))
    {
      fprintf(stderr,
	      "\nWarning: Source file defines time signature in first measure\n");
      fprintf(stderr, "   Source file = %d/%d, option supplied = %d/%d\n",
	      thisnum, thisden, numerator, denominator);
      do
	{
	  fprintf(stderr,"Override source file time sig with option time sig? (y/n) ");
	  ch = toupper(getchar());
	  while (ch == '\n' || ch == '\r') ch = toupper(getchar());
	}
      while ((ch != 'Y') && (ch != 'N'));
      if (ch == 'Y')
	{
	  numerator = thisnum;
	  denominator = thisden;
	};
    };
  return i;
}

static char wheel[4] = {'-','\\','|','/'};

int mpuaction(void)
{
  mpu_metrosetup_t metrosetup;
  mpu_demand_t demand;
  midi_event * pevent = NULL;
  midi_event event;
  int s_t = 0;
  int c_t = 0;
  int c_s = 0;
  int c_m = 0;
  int c_h = 0;
  int i = 0;
  long ec = 0;
  long bc = 0;
  long j = 0;
  byte k = 0;

  printf("Standard MIDI File Recorder version 0.5\n");
  printf("Copyright (c) 1995 Kim Burgaard\n\n");
 
  midifile.format = 0;
  midifile.tracks = 1;

  if (play_midifile.name)
    {
      timing = play_timing;
    }
  else
    {
      timing.division = 120;
      timing.newdivision = 120; /* save in recorded timebase */
      timing.time = 0.0;
    };

  midi_queue_flush(&queues.voice);
  midi_queue_flush(&queues.sysex);
  midi_queue_flush(&queues.meta);

  queues.dovoice = queues.dosysex = queues.dometa = 1;
  isrecord = 1;

  /* store sysex, time signature and tempo */
  put_time_sig(&play_queues.meta, 0, 4, 4); /* 1st measure ALWAYS in time sig. 4/4 */
  put_tempo(&play_queues.meta, 0, 100); /* 1st measure ALWAYS in tempo 100 bpm */
  if (gmsysex) put_gmon_sysex(&play_queues.sysex, 1);
  if (gssysex) put_gsreset_sysex(&play_queues.sysex, 4);
  if (dobpm && (bpm != 100)) put_tempo(&play_queues.meta, timing.division-1, bpm);
  if (dotimesig) put_time_sig(&play_queues.meta, timing.division, numerator, denominator);
  
  if (play_midifile.name)
    {
      printf("%s ", (nooverdub) ? "Replaying file while recording:" : "Overdubbing with:");
      printf("%s\n\n", play_midifile.name);
    };

  /* usually there is none of few sysex events */
  while ( (pevent = midi_queue_get(&play_queues.sysex)) )
    {
      if (pevent->type == MIDI_TYP_SYSEX) ioctl(mpudev, MPUIOC_PUT_EVENT, pevent);
      if (nooverdub) midi_event_free(pevent);
      else midi_queue_put(&queues.sysex, pevent);
    };
  /* preload 2 * PACK_SIZE meta events */
  for (i = 0; i < 2*PACK_SIZE && play_queues.meta.count > 0; i++)
    {
      if ( !(pevent = midi_queue_get(&play_queues.meta)) ) break;
      ioctl(mpudev, MPUIOC_PUT_EVENT, pevent);
      if (nooverdub) midi_event_free(pevent);
      else midi_queue_put(&queues.meta, pevent);
    };
  /* preload 2 * PACK_SIZE voice events */
  for (i = 0; i < 2*PACK_SIZE && play_queues.voice.count > 0; i++)
    {
      if ( !(pevent = midi_queue_get(&play_queues.voice)) ) break;
      ioctl(mpudev, MPUIOC_PUT_EVENT, pevent);
      if (nooverdub) midi_event_free(pevent);
      else midi_queue_put(&queues.voice, pevent);
    };

  if (metronome == 2)
    {
      metrosetup.numerator = 4;
      metrosetup.denominator = 4;
      metrosetup.channel = 9; /* channel 10 */
      metrosetup.click_key = 37;
      metrosetup.click_velo = 73;
      metrosetup.acc_click_key = 37;
      metrosetup.acc_click_velo = 100;
      metrosetup.meas_click_key = 37;
      metrosetup.meas_click_velo = 127;
      ioctl(mpudev, MPUIOC_SET_METRONOME, &metrosetup);
    }
  else
    {
      k = 4;
      ioctl(mpudev, MPUIOC_SET_NUMERATOR, &k);
      ioctl(mpudev, MPUIOC_SET_DENOMINATOR, &k);
    };
  if (metronome) ioctl(mpudev, MPUIOC_METRONOME_ON);

  j = 0;
  k = 100;
  ioctl(mpudev, MPUIOC_SET_TIMEBASE, &timing.division);
  ioctl(mpudev, MPUIOC_SET_TEMPO, &k);

  printf("Press <ctrl>+<c> to stop\n");
  fflush(NULL);

  ioctl(mpudev, MPUIOC_START_OVERDUB, &j);

  ec = bc = 0;
  s_t = time(NULL);

  while (isrecord)
    {
      demand.voice = (play_queues.voice.count>PACK_SIZE) ? PACK_SIZE : play_queues.voice.count;
      demand.control = (play_queues.meta.count>PACK_SIZE) ? PACK_SIZE : play_queues.meta.count;
      demand.record = PACK_SIZE;
      demand.sync = 1;

      if (ioctl(mpudev, MPUIOC_BLOCK_DEMAND, &demand)) return -EINVAL;

      if (demand.message & MPU_BLK_SYNC)
	{
	  bc++;
	  if (ioctl(mpudev, MPUIOC_GET_TEMPO, &bpm)) return -EINVAL;
	  if (ioctl(mpudev, MPUIOC_GET_NUMERATOR, &numerator)) return -EINVAL;
	  if (ioctl(mpudev, MPUIOC_GET_DENOMINATOR, &denominator)) return -EINVAL;
	};

      c_t = time(NULL) - s_t;
      c_s = (c_t % 60);
      c_m = (c_t/60) % 60;
      c_h = (c_t/(60*60)) % 60;
      printf("\r[%02d:%02d:%02d] BPM = %3d, Time signature = %2d/%-2d, events recorded = %-6ld",
	     c_h, c_m, c_s, bpm, numerator, denominator, ec);
      printf(" %c\r", wheel[bc%4]);
      fflush(NULL);

      if (demand.message & MPU_BLK_VOICE)
	{
	  for (i = 0; i < PACK_SIZE && play_queues.voice.count > 0; i++)
	    {
	      if ( !(pevent = midi_queue_get(&play_queues.voice)) ) break;
	      ioctl(mpudev, MPUIOC_PUT_EVENT, pevent);
	      if (nooverdub) midi_event_free(pevent);
	      else midi_queue_put(&queues.voice, pevent);
	    };
	};
      if (demand.message & MPU_BLK_CONTROL)
	{
	  for (i = 0; i < PACK_SIZE && play_queues.meta.count > 0; i++)
	    {
	      if ( !(pevent = midi_queue_get(&play_queues.meta)) ) break;
	      ioctl(mpudev, MPUIOC_PUT_EVENT, pevent);
	      if (nooverdub) midi_event_free(pevent);
	      else midi_queue_put(&queues.meta, pevent);
	    };
	};
      if (demand.message & MPU_BLK_RECORD)
	{
	  for (i = 0; i < PACK_SIZE; i++)
	    {
	      if (ioctl(mpudev, MPUIOC_GET_EVENT_SIZE, &event.size))
		{
		  fprintf(stderr, "\nsmfrecord: Couldn't get event size from driver\n");
		  isrecord = 0;
		  break;
		}
	      else if (event.size < 1) break;
	      event.data = my_malloc(event.size);
	      if (ioctl(mpudev, MPUIOC_GET_EVENT, &event) || !event.size)
		{
		  free(event.data);
		  break;
		};
	      /* midi_printevent(&event, MIDI_PRINT_ALL); */
	      switch (event.type & 0xff)
		{
		case MIDI_TYP_VOICE:
		  ec++;
		  midi_queue_put_event(&queues.voice,event.type,event.time,event.size,event.data);
		  break;
		case MIDI_TYP_META:
		  ec++;
		  midi_queue_put_event(&queues.meta,event.type,event.time,event.size,event.data);
		  break;
		case MIDI_TYP_SYSEX:
		  ec++;
		  midi_queue_put_event(&queues.sysex,event.type,event.time,event.size,event.data);
		  break;
		default:
		  free(event.data);
		  fprintf(stderr, "\nWarning: Received unknown event type (0x%x) from driver\n",
			  event.type&0xff);
		  break;
		};
	    };
	};
    };

  isrecord = 0;
  ioctl(mpudev, MPUIOC_STOP_OVERDUB, 0);
  if (metronome) ioctl(mpudev, MPUIOC_METRONOME_OFF);

  /* get any pending events from record buffer */
  while (!ioctl(mpudev, MPUIOC_GET_EVENT_SIZE, &event.size) && event.size > 0)
    {
      event.data = my_malloc(event.size);
      if (ioctl(mpudev, MPUIOC_GET_EVENT, &event) || !event.size)
	{
	  free(event.data);
	  break;
	};
      /* midi_printevent(&event, MIDI_PRINT_ALL); */
      switch (event.type & 0xff)
	{
	case MIDI_TYP_VOICE:
	  ec++;
	  midi_queue_put_event(&queues.voice,event.type,event.time,event.size,event.data);
	  break;
	case MIDI_TYP_META:
	  ec++;
	  midi_queue_put_event(&queues.meta,event.type,event.time,event.size,event.data);
	  break;
	case MIDI_TYP_SYSEX:
	  ec++;
	  midi_queue_put_event(&queues.sysex,event.type,event.time,event.size,event.data);
	  break;
	default:
	  free(event.data);
	  fprintf(stderr,
		  "\nWarning: Received unknown event type (0x%x) from driver\n",
		  event.type&0xff);
	  break;
	};
    };

  c_t = time(NULL) - s_t;
  c_s = c_t % 60;
  c_m = (c_t/60) % 60;
  c_h = (c_t/(60*60)) % 60;
  printf("\r[%02d:%02d:%02d] BPM = %3d, Time signature = %2d/%-2d, events recorded = %-6ld",
	 c_h, c_m, c_s, bpm, numerator, denominator, ec);
  printf(" %c", wheel[bc%4]);
  fflush(NULL);

  if (event.size) /* oops, we stopped because of error in ioctl call */
    {
      fprintf(stderr, "\nsmfrecord: Couldn't get event size from driver\n");
    };

  printf("\n\n");

  ioctl(mpudev, MPUIOC_RESET, 0);
  ioctl(mpudev, MPUIOC_GM_ON, 0);
  ioctl(mpudev, MPUIOC_GS_RESET, 0);

  while(play_queues.voice.count > 0)
    {
      if ( !(pevent = midi_queue_get(&play_queues.voice)) ) break;
      if (nooverdub) midi_event_free(pevent);
      else midi_queue_put(&queues.voice, pevent);
    };
  while(play_queues.meta.count > 0)
    {
      if ( !(pevent = midi_queue_get(&queues.meta)) ) break;
      if (nooverdub) midi_event_free(pevent);
      else midi_queue_put(&queues.meta, pevent);
    };

  printf("Saving: %s", midifile.name);
  if (midifile.title && midifile.copyright) printf(" [ %s, %s ]", midifile.title, midifile.copyright);
  else if (midifile.copyright) printf(" [ %s ]", midifile.copyright);
  else if (midifile.title) printf("[ %s ]", midifile.title);
  printf("...");

  if ( !(midifile.file = fopen(midifile.name, "wb")) )
    {
      perror("smfrecord");
      return -EINVAL;
    };

  if ((i = midi_save(&midifile, &timing, &queues, 0))) perror("smfrecord");
  fclose(midifile.file);
  printf(" done\n");
  return i;
}

int record(void)
{
  int i = 0;

  if ( (mpudev = open("/dev/mpu401data", O_RDWR)) == -1 )
    {
      perror("Could not open `/dev/mpu401'");
      return -EINVAL;
    };

  if ((play_midifile.name) && (i = load())) return i;
  return mpuaction();
}

void save(void)
{
  isrecord = 0;
};

void closedown(void)
{
  printf("\nsmfplay: killed...\n");
  if (mpudev != -1)
    {
      ioctl(mpudev, MPUIOC_RESET, 0);
      ioctl(mpudev, MPUIOC_GM_ON, 0);
      ioctl(mpudev, MPUIOC_GS_RESET, 0);
      close(mpudev);
    };
  exit(1);
}

/*** MAIN *********************************************************************/

int main(int argc, char *argv[])
{
  midifile.name = midifile.title = midifile.copyright = NULL;
  play_midifile.name = play_midifile.title = play_midifile.copyright = NULL;
  if ( parsearg(argc, argv) ) return 1;

  mpudev = -1;

  /* Prevent hanging sounds if terminated by fx. Ctrl-C */
  signal(SIGINT,  (void(*)()) save);
  signal(SIGQUIT, (void(*)()) closedown);
  signal(SIGTERM, (void(*)()) closedown);
  signal(SIGABRT, (void(*)()) closedown);

  midi_queue_reset(&queues.voice);
  midi_queue_reset(&queues.sysex);
  midi_queue_reset(&queues.meta);

  midi_queue_reset(&play_queues.voice);
  midi_queue_reset(&play_queues.sysex);
  midi_queue_reset(&play_queues.meta);

  return record();
}

/*** END OF FILE **************************************************************/
