/****************************************************************************** 
 *
 * File:        smfedit.c
 * Version:     $Id: smfedit.c,v 1.5 1995/07/21 17:24:47 burgaard Exp $
 *              $Version: 0.3$
 *
 * Purpose:     Standard MIDI File Editor.
 *
 * Project:     MIDI/Sequencer library.
 *              Roland MPU-401 Device driver.
 * Authors:     Kim Burgaard, <burgaard@daimi.aau.dk>
 * Copyrights:  Copyright (c) 1994, 1995 Kim Burgaard.
 *
 *      This package is free software; you can redistribute it and/or modify it
 *      under the terms of the GNU General Public License as published by the
 *      Free Software Foundation; either version 2, or (at your option) any
 *      later version.
 *
 *      This package is distributed in the hope that it will be useful, but
 *      WITHOUT ANY WARRANTY; without even the implied warranty of
 *      MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
 *      Public License for more details.
 *
 *      You should have received a copy of the GNU General Public License along
 *      with this program; see the file COPYING. If not, write to the Free
 *      Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 ******************************************************************************/

/*** INCLUDES & DEFINES *******************************************************/

#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>

#include "miditypes.h"
#include "midiqueue.h"
#include "midiprint.h"
#include "midifile.h"

/*** GLOBAL VARIABLES *********************************************************/

static char *source   = NULL;
static char *dest     = NULL;
static char *title    = NULL;
static char *copyright= NULL;
static char *text     = NULL;

static int deltext    = 0;
static int delmetau   = 0;
static int delmeta    = 0;
static int delsysex   = 0;
static int deltrack   = -1;

static int savformat  = 0;

static midi_file        midifile;
static midi_file_timing timing;
static midi_file_data   queues;

/*** MEMORY HANDLING **********************************************************/

static char *my_malloc(long size)
{
  static char *tmp = NULL;

  if ( !(tmp = malloc(size)) )
    {
      fprintf(stderr, "Out of memory!\n");
      exit(-3);
    };

  return tmp;
}

/*** EVENT HANDLING ***********************************************************/

struct event_kill
{
  struct event_kill *next;
  enum MIDI_TYPE type;
  long number;
  midi_event *event;
};
typedef struct event_kill event_kill;

static event_kill *first = NULL;
static event_kill *last = NULL;
static event_kill *current = NULL;
static long eventkills = 0;

/*** ARGUMENT PARSING *********************************************************/

void usage(int exitcode)
{
  printf("Standard MIDI File Player Editor 0.3\n");
  printf("Copyright (c) 1994, 1995 Kim Burgaard\n");
  printf("\n");
  printf("Usage: smfedit [options] source\n");
  printf("\n");
  printf("   -output <filename> Destination file\n");
  printf("\n");
  printf("   -title <text>   Change/add song title\n");
  printf("   -copy <text>    Change/add copyright notice\n");
  printf("   -text <text>    Change/add song description\n");
  printf("\n");
  printf("   -delete t n     Delete the n'th event of type t\n");
  printf("   -delrange t s e Delete from the s'th to the e'th event of type t\n");
  printf("\n");
  printf("                   t is on of V = voice, M = meta and S = sysex\n");
  printf("\n");
  printf("   -deltext        Delete most META text events\n");
  printf("   -delmetamost    Delete all non-critic META events\n");
  printf("                   Makes the file compliant with almost any MIDI sequencer\n");
  printf("   -delmeta        Delete all META events\n");
  printf("   -delsysex       Delete all SYSEX events\n");
  printf("\n");
  printf("   -deltrack n     Delete all events in track n\n");
  printf("\n");
  printf("   -smf            Equivalent to -delmetamost\n");
  printf("   -1              Save in format 1. Default is to save in format 0\n");
  printf("\n");
  printf("   -help           This help text.\n");
  exit(exitcode);
}

inline char isoption(char *s)
{
  return !( !s || !strlen(s) || s[0] != '-' );
}

static char buffer[6];

int parsearg(int argc, char *argv[])
{
  event_kill tmp;
  long fe, te;
  int i = 1;

  if (argc < 2) usage(1);
  while (i < argc)
    {
      if ( isoption(argv[i]) )
	{
	  if ( !strcmp("-output", argv[i]) )
	    {
	      if (dest)
		{
		  fprintf(stderr, "Multiple destination files not allowed!\n");
		  exit(-1);
		}
	      else
		{
		  i++;
		  if (!(i < argc))
		    {
		      fprintf(stderr, "Destination file name expected\n");
		      exit(-1);
		    }
		  else
		    dest = argv[i];
		};
	    }
	  else if ( !strcmp("-title", argv[i]) )
	    {
	      if (title)
		{
		  fprintf(stderr, "Multiple titles not allowed!\n");
		  exit(-1);
		}
	      else
		{
		  i++;
		  if (!(i < argc))
		    {
		      fprintf(stderr, "Title text expected\n");
		      exit(-1);
		    }
		  else
		    title = argv[i];
		};
	    }
	  else if ( !strcmp("-copy", argv[i]) )
	    {
	      if (copyright)
		{
		  fprintf(stderr, "Multiple copyright notices not allowed!\n");
		  exit(-1);
		}
	      else
		{
		  i++;
		  if (!(i < argc))
		    {
		      fprintf(stderr, "Copyright notice expected\n");
		      exit(-1);
		    }
		  else
		    copyright = argv[i];
		};
	    }
	  else if ( !strcmp("-text", argv[i]) )
	    {
	      if (text)
		{
		  fprintf(stderr, "Multiple text events not allowed!\n");
		  exit(-1);
		}
	      else
		{
		  i++;
		  if (!(i < argc))
		    {
		      fprintf(stderr, "Text expected\n");
		      exit(-1);
		    }
		  else
		    text = argv[i];
		};
	    }
	  else if ( !strcmp("-delete", argv[i]) )
	    {
	      i++;
	      if (!(i+1 < argc))
		{
		  fprintf(stderr, "Argument expected\n");
		  exit(-1);
		}
	      else
		{
		  if (sscanf(argv[i], "%[V]", &buffer[0]))
		    tmp.type = MIDI_TYP_VOICE;
		  else if (sscanf(argv[i], "%[M]", &buffer[0]))
		    tmp.type = MIDI_TYP_META;
		  else if (sscanf(argv[i], "%[S]", &buffer[0]))
		    tmp.type = MIDI_TYP_SYSEX;
		  else
		    {
		      fprintf(stderr, "Invalid argument to -delete: %s\n", argv[i]);
		      exit(-1);
		    };

		  i++;
		  if (!sscanf(argv[i], "%ld", &tmp.number) || tmp.number < 0)
		    {
		      fprintf(stderr, "Invalid argument to -delete: %s\n", argv[i]);
		      exit(-1);
		    };

		  current = (event_kill *)my_malloc(sizeof(event_kill));
		  current->event = NULL;
		  current->next = NULL;
		  eventkills++;
		  if (last)
		    {
		      last->next = current;
		      last = current;
		    }
		  else
		    first = last = current;

		  current->type = tmp.type;
		  current->number = tmp.number;
		};
	    }
	  else if ( !strcmp("-delrange", argv[i]) )
	    {
	      i++;
	      if (!(i+2 < argc))
		{
		  fprintf(stderr, "Argument expected after -delrange\n");
		  exit(-1);
		}
	      else
		{
		  if (sscanf(argv[i], "%[V]", &buffer[0]))
		    tmp.type = MIDI_TYP_VOICE;
		  else if (sscanf(argv[i], "%[M]", &buffer[0]))
		    tmp.type = MIDI_TYP_META;
		  else if (sscanf(argv[i], "%[S]", &buffer[0]))
		    tmp.type = MIDI_TYP_SYSEX;
		  else
		    {
		      fprintf(stderr, "Invalid argument to -delrange: %s\n", argv[i]);
		      exit(-1);
		    };

		  i++;
		  if (!sscanf(argv[i], "%ld", &fe) || fe < 0)
		    {
		      fprintf(stderr, "Invalid argument to -delrange %s\n", argv[i]);
		      exit(-1);
		    };

		  i++;
		  if (!sscanf(argv[i], "%ld", &te) || te < fe)
		    {
		      fprintf(stderr, "Invalid argument to -delrange %s\n", argv[i]);
		      exit(-1);
		    };

		  for (tmp.number = fe; tmp.number < te; tmp.number++)
		    {
		      current = (event_kill *)my_malloc(sizeof(event_kill));
		      current->event = NULL;
		      current->next = NULL;
		      eventkills++;
		      if (last)
			{
			  last->next = current;
			  last = current;
			}
		      else
			first = last = current;
		      
		      current->type = tmp.type;
		      current->number = tmp.number;
		    };
		};
	    }
	  else if ( !strcmp("-deltext", argv[i]) )
	    {
	      deltext = 1;
	    }
	  else if ( !strcmp("-delmetamost", argv[i]) ||  !strcmp("-smf", argv[i]) )
	    {
	      delmetau = 1;
	    }
	  else if ( !strcmp("-delmeta", argv[i]) )
	    {
	      delmeta = 1;
	    }
	  else if ( !strcmp("-delsysex", argv[i]) )
	    {
	      delsysex = 1;
	    }
	  else if ( !strcmp("-deltrack", argv[i]) )
	    {
	      if (deltrack != -1)
		{
		  fprintf(stderr, "You can only specify one track to delete.\n");
		  exit(-1);
		}
	      else
		{
		  i++;
		  if (!(i < argc))
		    {
		      fprintf(stderr, "Integer argument expected to -deltrack\n");
		      exit(-1);
		    }
		  else if (!sscanf(argv[i], "%d", &deltrack) || deltrack < 1 || deltrack > 16)
		    {
		      fprintf(stderr, "Invalid argument to -delrange: %s\n", argv[i]);
		      exit(-1);
		    };
		};
	    }
	  else if ( !strcmp("-1", argv[i]) )
	    {
	      savformat = 1;
	    }
	  else if ( !strcmp("-help", argv[i]) )
	    usage(0);
	  else
	    {
	      fprintf(stderr, "Unknown option `%s' :\n\n", argv[i]);
	      fprintf(stderr, "Try `smfplay -help' for more information\n");
	      exit(-1);
	    };
	}
      else if (source)
	{
	  fprintf(stderr, "Multiple source files is not allowed!\n");
	  exit(-1);
	}
      else
	source = argv[i]; 

      i++;
    };

  if (!source)
    { 
      fprintf(stderr, "Try `smfedit -help' for more information\n");
      exit(-1);
    };

  if (!dest) dest = source;

  return 0;
}

/*** MIDI FILE HANDLING *******************************************************/

static midi_event *event = NULL;
static midi_event *tempe = NULL;
static event_kill *ekt = NULL;
static event_kill *ek = NULL;

void eventkill(void)
{
  static midi_event *ve, *me, *se;
  static long i, j, dc, vc, mc, sc;

  vc = mc = sc = 0;
  ve = queues.voice.head;
  me = queues.meta.head;
  se = queues.sysex.head;

  ek = first;
  while (ek)
    {
      switch (ek->type)
	{
	case MIDI_TYP_VOICE:
	  if (ek->number >= queues.voice.count)
	    {
	      fprintf(stderr, "VOICE delete %ld out of range\n", ek->number);
	      ek->event = NULL;
	    }
	  else if (ek->number < vc)
	    {
	      j = vc;
	      for (i = 0; i < j - ek->number && ve; i++) 
		{
		  ve = ve->prev;
		  vc--;
		};
	      ek->event = ve;
	    }
	  else
	    {
	      j = vc;
	      for (i = 0; i < ek->number - j; i++)
		{
		  ve = ve->next;
		  vc++;
		};
	      ek->event = ve;
	    };
	  break;
	case MIDI_TYP_SYSEX:
	  if (ek->number >= queues.sysex.count)
	    {
	      fprintf(stderr, "SYSEX delete %ld out of range\n", ek->number);
	      ek->event = NULL;
	    }
	  else if (ek->number < sc)
	    {
	      j = sc;
	      for (i = 0; i < j - ek->number && se; i++) 
		{
		  se = se->prev;
		  sc--;
		};
	      ek->event = se;
	    }
	  else
	    {
	      j = sc;
	      for (i = 0; i < ek->number - j; i++)
		{
		  se = se->next;
		  sc++;
		};
	      ek->event = se;
	    };
	  break;
	case MIDI_TYP_META:
	  if (ek->number >= queues.meta.count)
	    {
	      fprintf(stderr, "META delete %ld out of range\n", ek->number);
	      ek->event = NULL;
	    }
	  else if (ek->number < mc)
	    {
	      j = mc;
	      for (i = 0; i < j - ek->number && me; i++) 
		{
		  me = me->prev;
		  mc--;
		};
	      ek->event = me;
	    }
	  else
	    {
	      j = mc;
	      for (i = 0; i < ek->number - j; i++)
		{
		  me = me->next;
		  mc++;
		};
	      ek->event = me;
	    };
	  break;
	};
      ek = ek->next;
    };

  ek = first;

  dc = 0;

  while (ek)
    {
      switch (ek->type)
	{
	case MIDI_TYP_VOICE:
	  if (ek->event)
	    {
	      midi_queue_remove_event(&queues.voice, ek->event);
	      printf("VOICE delete %ld\n", ek->number);
	      dc++;
	    };
	  break;
	case MIDI_TYP_SYSEX:
	  if (ek->event)
	    {
	      midi_queue_remove_event(&queues.sysex, ek->event);
	      printf("SYSEX delete %ld\n", ek->number);
	      dc++;
	    };
	  break;
	case MIDI_TYP_META:
	  if (ek->event)
	    {
	      midi_queue_remove_event(&queues.meta, ek->event);
	      printf("META delete %ld\n", ek->number);
	      dc++;
	    };
	  break;
	};

      ekt = ek;
      ek = ek->next;
      free(ekt);
    };
  if (dc) printf("Successfully deleted %ld events out of %ld requested\n", dc, eventkills);
  else printf("No event(s) deleted\n");
}

void deletetext(void)
{
  long dc = 0;
  
  event = queues.meta.head;
  while (event)
    {
      tempe = event->next;
      switch (event->data[0]&0xff)
	{
	case MID_META_SEQ_NAME:
	case MID_META_INST_NAME:
	case MID_META_LYRIC:
	case MID_META_MARKER:
	case MID_META_CUE_POINT:
	  midi_queue_remove_event(&queues.meta, event);
	  dc++;
	};
      event = tempe;
    };
  if (dc) printf("Successfully deleted %ld META text events\n", dc);
  else printf("No META text event(s) in file to delete\n");
}

void deletemeta(void)
{
  long dc = 0;
  
  event = queues.meta.head;
  while (event)
    {
      tempe = event->next;
      
      switch (event->data[0]&0xff)
	{
	case MID_META_COPYRIGHT:
	case MID_META_SEQ_NAME:
	case MID_META_EOT:
	case MID_META_SET_TEMPO:
	case MID_META_SMPTE_OFFSET:
	case MID_META_TIME_SIG:
	case MID_META_KEY_SIG:
	  break;
	default:
	  midi_queue_remove_event(&queues.meta, event);
	  dc++;
	};
      event = tempe;
    };
  if (dc) printf("Successfully deleted %ld META events\n", dc);
  else printf("No META event(s) in file to delete\n");
}

void textreplace(void)
{
  char *data = NULL;
  int size = 0;
  
  if (text)
    {
      event = queues.meta.head;
      while (event && event->type != MIDI_TYP_META &&
	     (event->data[0]&0xff) != MID_META_TEXT)
	{
	  event = event->next;
	};
      if (event && event->type == MIDI_TYP_META &&
	  (event->data[0]&0xff) == MID_META_TEXT)
	{
	  midi_queue_remove_event(&queues.meta, event);
	};
      
      size = strlen(text);
      data = my_malloc(size + 1);
      memcpy(&data[1], text, size);
      data[0] = MID_META_TEXT;
      midi_queue_put_least_event(&queues.meta, MIDI_TYP_META, 0, size + 1, data);
    };
  
  if (title)
    {
      event = queues.meta.head;
      while (event && event->type != MIDI_TYP_META &&
	     (event->data[0]&0xff) != MID_META_SEQ_NAME)
	{
	  event = event->next;
	};
      if (event && event->type == MIDI_TYP_META &&
	  (event->data[0]&0xff) == MID_META_SEQ_NAME)
	{
	  midi_queue_remove_event(&queues.meta, event);
	};
      
      size = strlen(title);
      data = my_malloc(size + 1);
      memcpy(&data[1], title, size);
      data[0] = MID_META_SEQ_NAME;
      midi_queue_put_least_event(&queues.meta, MIDI_TYP_META, 0, size + 1, data);
    };
  
  if (copyright)
    {
      event = queues.meta.head;
      while (event && event->type != MIDI_TYP_META &&
	     (event->data[0]&0xff) != MID_META_COPYRIGHT)
	{
	  event = event->next;
	};
      if (event && event->type == MIDI_TYP_META &&
	  (event->data[0]&0xff) == MID_META_COPYRIGHT)
	{
	  midi_queue_remove_event(&queues.meta, event);
	};
      
      size = strlen(copyright);
      data = my_malloc(size + 1);
      memcpy(&data[1], copyright, size);
      data[0] = MID_META_COPYRIGHT;
      midi_queue_put_least_event(&queues.meta, MIDI_TYP_META, 0, size + 1, data);
    };
};

void trackdelete(void)
{
  midi_event * event = queues.voice.head;
  midi_event * tmp = NULL;
  long k = 0;

  while (event)
    {
      tmp = NULL;
      if ((event->data[0]&0x0f) == (deltrack-1)) tmp = event;
      event = event->next;
      if (tmp)
	{
	  midi_queue_remove_event(&queues.voice, tmp);
	  k++;
	};
    };
  if (k) printf("Successfully deleted %ld events from track %d\n", k, deltrack);
  else printf("No event(s) in track %d to delete\n", deltrack);
};

int edit(void)
{
  char *tn = NULL;
  int i = 0;


  if ( (tn = strrchr(source, '/')) ) tn++;
  else tn = source;

  midifile.name = tn;

  if ( !(midifile.file = fopen(source, "rb")) )
    {
      fprintf(stderr, "Could not open `%s': ", tn);
      perror("");
      return -EINVAL;
    };

  midifile.title = midifile.copyright = NULL;
      
  midifile.format = 0;
  midifile.tracks = 0;
  timing.division = 96;
  timing.newdivision = 0;
  timing.time = 0.0;
      
  midi_queue_flush(&queues.voice);
  midi_queue_flush(&queues.sysex);
  midi_queue_flush(&queues.meta);
  
  queues.dovoice = 1;
  queues.dometa = !delmeta;
  queues.dosysex = !delsysex;

  if ( (i = midi_load(&midifile, &timing, &queues, 0)) == 0 ) 
    {
      fclose(midifile.file);

      if (midifile.title && midifile.copyright)
	{
	  printf("[ %s, %s ]\n", midifile.title, midifile.copyright);
	  free(midifile.title);
	  free(midifile.copyright);
	}
      else if (midifile.copyright)
	{
	  printf("[ %s, %s ]\n", midifile.name, midifile.copyright);
	  free(midifile.copyright);
	}
      else if (midifile.title)
	{
	  printf("[ %s ]\n", midifile.title);
	  free(midifile.title);
	};

      midifile.title = midifile.copyright = NULL;

      if (deltrack > -1) trackdelete();
      if (eventkills) eventkill();
      if (deltext) deletetext();
      if (delmetau) deletemeta();
      if (title || copyright || text) textreplace();
      
      tn = dest;
      if ( (tn = strrchr(dest, '/')) ) ++tn; /* strip leading path */
      midifile.name = tn;
      midifile.title = midifile.copyright = NULL;
      
      midifile.format = savformat;
      timing.newdivision = 0;
  
      queues.dovoice = 1;
      queues.dometa = !delmeta;
      queues.dosysex = !delsysex;
      
      if ( !(midifile.file = fopen(dest, "w+b")) )
	{
	  fprintf(stderr, "Could not open `%s': ", source);
	  perror("");
	  return -EINVAL;
	};
      
      if ((i = midi_save(&midifile, &timing, &queues, 1)) != 0 &&
	  (i = ferror(midifile.file)))
	{
	  fprintf(stderr, "Error while writing `%s': ", midifile.name);
	  perror("");
	};

      fclose(midifile.file);
    }
  else if ( (i = ferror(midifile.file)) )
    {
      fprintf(stderr, "Error while reading `%s': ", midifile.name);
      perror("");
    };
  
  return i;
}

/*** MAIN *********************************************************************/

int main(int argc, char *argv[])
{
  if ( parsearg(argc, argv) ) return 1;
  return edit();
}

/*** END OF FILE **************************************************************/
