-----------------------------------------------------------------------------
--                                                                         --
--                GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                         --
--                       R T S _ R e n d e z v o u s                       --
--                                                                         --
--                                 S p e c                                 --
--                                                                         --
--                            $Revision: 1.2 $                            --
--                                                                         --
--          Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                         --
-- GNARL is free software; you can redistribute it and/or modify it  under --
-- terms  of  the  GNU  Library General Public License as published by the --
-- Free Software Foundation; either version 2, or  (at  your  option)  any --
-- later  version.   GNARL is distributed in the hope that it will be use- --
-- ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
-- MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
-- eral Library Public License for more details.  You should have received --
-- a  copy of the GNU Library General Public License along with GNARL; see --
-- file COPYING. If not, write to the Free Software Foundation,  675  Mass --
-- Ave, Cambridge, MA 02139, USA.                                          --
--                                                                         --
-----------------------------------------------------------------------------

with System.Compiler_Exceptions;
with System.Task_IDs; use System.Task_IDs;
with System.Task_Entries;
with System.Task_Entry_Queue;
with System.Task_Protected_Objects;

package System.Task_Rendezvous is

   Null_Task_Entry : constant := Task_Entries.Null_Entry;
   Max_Task_Entry : constant Integer := Task_Entries.Max_Entry;
   type Task_Entry_Index is new Task_Entries.Entry_Index
         range Task_Entries.Null_Entry .. Task_Entries.Max_Entry;

   procedure Task_Entry_Call (
         Acceptor : Task_ID;
         E : Task_Entry_Index;
         Parameter : System.Address;
         Mode : Task_Protected_Objects.Call_Modes;
         Rendezvous_Successful : out Boolean);

   --  general entry call

   procedure Call_Simple (
         Acceptor : Task_ID;
         E : Task_Entry_Index;
         Parameter : System.Address);

   --  simple entry call

   procedure Cancel_Task_Entry_Call (Cancelled : out Boolean);

   --  cancel pending task entry call

   type Select_Modes is (
         Simple_Mode,
         Else_Mode,
         Terminate_Mode);

   Max_Select : constant Integer := Integer'Last;            --  RTS-defined

   subtype Select_Index is Integer range 0 .. Max_Select;
   --  This is a subtype so that operations on it will be visible to
   --  the code generated by GNAT.

   No_Rendezvous : constant Select_Index := 0;
   subtype Positive_Select_Index is
         Select_Index range 1 .. Select_Index'Last;
   type Accept_Alternative is record --  should be packed
      Null_Body : Boolean;
      S : Task_Entry_Index;
   end record;
   type Accept_List is
         array (Positive_Select_Index range <>) of Accept_Alternative;

   type Accept_List_Access is access constant Accept_List;

   procedure Selective_Wait (

         Open_Accepts : Accept_List_Access;

         Select_Mode : Select_Modes;
         Parameter : out System.Address;
         Index : out Select_Index);

   --  selective wait

   procedure Accept_Call (
         E : Task_Entry_Index;
         Parameter : out System.Address);

   --  accept an entry call

   procedure Accept_Trivial (E : Task_Entry_Index);

   --  accept an entry call that has no parameters and no body

   function Task_Count (E : Task_Entry_Index) return Natural;

   --  return number of tasks waiting on the entry E (of current task)

   function Callable (T : Task_ID) return Boolean;

   --  return T'CALLABLE

   procedure Complete_Rendezvous;

   --  called by acceptor to wake up caller

   procedure Exceptional_Complete_Rendezvous (
         Ex : Compiler_Exceptions.Exception_ID);

   --  called by acceptor to mark the end of the current rendezvous and
   --  propagate an exception to the caller.

   --  RTS Internal Declarations---Not part of the GNARLI.

   procedure Close_Entries (Target : Task_ID);
   --  Not part of the CARTS interface.
   --  close entries, purge entry queues
   --  called by Task_Stages.Complete

   procedure Complete_on_Sync_Point (T : Task_ID);
   --  Not part of the CARTS interface.
   --  If a task is suspended on an accept, select, or
   --  entry call (but not yet *in* rendezvous) then complete the task.

   procedure Reset_Priority (
         Acceptor_Prev_Priority : Task_Entry_Queue.Rendezvous_Priority;
         Acceptor : Task_ID);

   pragma Inline (Reset_Priority);
   --  Not part of the CARTS interface.
   --  reset priority to original value

end System.Task_Rendezvous;
