-----------------------------------------------------------------------------
--                                                                         --
--                GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                         --
--                       R T S _ R e n d e z v o u s                       --
--                                                                         --
--                                 B o d y                                 --
--                                                                         --
--                            $Revision: 1.2 $                            --
--                                                                         --
--          Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                         --
-- GNARL is free software; you can redistribute it and/or modify it  under --
-- terms  of  the  GNU  Library General Public License as published by the --
-- Free Software Foundation; either version 2, or (at  your  option)  any --
-- later  version.   GNARL is distributed in the hope that it will be use- --
-- ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
-- MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
-- eral Library Public License for more details.  You should have received --
-- a  copy of the GNU Library General Public License along with GNARL; see --
-- file COPYING. If not, write to the Free Software Foundation,  675  Mass --
-- Ave, Cambridge, MA 02139, USA.                                          --
--                                                                         --
-----------------------------------------------------------------------------

with Unchecked_Conversion;

with System.Task_Primitives; use System.Task_Primitives;
with System.Task_Abortion; use System.Task_Abortion;
with System.Task_Protected_Objects; use System.Task_Protected_Objects;
with System.Error_Reporting; use System.Error_Reporting;
with System.Machine; use System.Machine;
with System.Task_Entries; use System.Task_Entries;
with System.Task_Entry_Queue; use System.Task_Entry_Queue;
with System.Task_ATCB;
with System.Task_Stages; use System.Task_Stages;
with System.Compiler_Exceptions;
with System.Task_Memory;

package body System.Task_Rendezvous is

   function "<" (L, R : Task_ATCB.Task_Stage)
         return Boolean renames Task_ATCB."<";
   function ">=" (L, R : Task_ATCB.Task_Stage)
         return Boolean renames Task_ATCB.">=";
   function "=" (L, R : Task_ATCB.Accepting_State)
         return Boolean renames Task_ATCB."=";
   function "=" (L, R : Compiler_Exceptions.Exception_ID)
         return Boolean renames Compiler_Exceptions."=";

   type Select_Treatment is (
         Accept_Alternative_Selected,
         Else_Selected,
         Terminate_Selected,
         Accept_Alternative_Open,
         No_Alternative_Open);

   Default_Treatment : constant array (Select_Modes) of Select_Treatment :=
         (Simple_Mode         => No_Alternative_Open,
         Else_Mode           => Else_Selected,
         Terminate_Mode      => Terminate_Selected);

   procedure Boost_Priority (
         Call : Entry_Call_Link;
         Acceptor : Task_ATCB.ATCB_Ptr) is
      Caller : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Call.Self);
   begin
      if Get_Priority (Caller.LL_TCB'access) >
            Get_Priority (Acceptor.LL_TCB'access) then
         Call.Acceptor_Prev_Priority := Acceptor.Current_Priority;
         Acceptor.Current_Priority := Caller.Current_Priority;
         Set_Priority (Acceptor.LL_TCB'access,
               Acceptor.Current_Priority);
      end if;
   end Boost_Priority;
   pragma Inline (Boost_Priority);

   procedure Reset_Priority (
      Acceptor_Prev_Priority : Rendezvous_Priority;
      Acceptor : Task_ID) is
      Acceptor_ATCB : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Acceptor);
   begin
      if Acceptor_Prev_Priority /= Priority_Not_Boosted then
         Acceptor_ATCB.Current_Priority := Acceptor_Prev_Priority;
         Set_Priority (Acceptor_ATCB.LL_TCB'access,
               Acceptor_ATCB.Current_Priority);
      end if;
   end Reset_Priority;
   pragma Inline (Reset_Priority);

   --  Test if a rendezvous can be made right away;
   --  return True if the rendezvous has occurred (and finished).
   --  Problem: Try not to call this when the acceptor is not accepting.
   procedure Test_Call (
         Entry_Call : in out Entry_Call_Link;
         Rendezvous_Completed : out Boolean) is
      Temp_Entry : Entry_Index;
      TAS_Result : Boolean;
      Acceptor_ID : Task_ID;
      Acceptor : Task_ATCB.ATCB_Ptr;
      Caller : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Entry_Call.Self);
   begin
      Acceptor := Task_ATCB.ID_To_ATCB (Entry_Call.Called_Task);
      if Acceptor.Accepting = Task_ATCB.Trivial_Accept then

         Temp_Entry := Entry_Index (Acceptor.Open_Accepts (1).S);

         if Entry_Call.E = Temp_Entry then              --  do rendezvous
            Acceptor.Accepting := Task_ATCB.Not_Accepting;
            Entry_Call.Acceptor_Prev_Call := Acceptor.Call;
            Acceptor.Call := Entry_Call;
            Entry_Call.Done := True;
            Rendezvous_Completed := True;
            Cond_Signal (Acceptor.Cond);                  --  Inefficient
         else --  wait for acceptor
            Rendezvous_Completed := False;
         end if;

      elsif Acceptor.Accepting = Task_ATCB.Not_Accepting then
         if Callable (Task_ATCB.ATCB_To_ID (Acceptor)) then
         --  wait for acceptor
            Rendezvous_Completed := False;
         else
            if Entry_Call.Mode /= Asynchronous_Call then
               Caller.ATC_Nesting_Level := Caller.ATC_Nesting_Level - 1;
            end if;
            Unlock (Acceptor.L);
            Undefer_Abortion;
            raise Tasking_Error;
         end if;

      else                                  --  try to do immediate rendezvous

         for I in Acceptor.Open_Accepts'range loop
            Temp_Entry := Entry_Index (Acceptor.Open_Accepts (I).S);

               --  WARNING: The following code will be incorrectly indented
               --  in the Verdix version, due to the conditional inclusion
               --  of a declare block for Verdix.

            if Entry_Call.E = Temp_Entry then --  do rendezvous
               Test_And_Set (Entry_Call.Call_Claimed'Address, TAS_Result);
               if not TAS_Result then --  this task has been aborted
                  Unlock (Acceptor.L);
                  Write_Lock (Caller.L);
                  Caller.Suspended_Abortably := True;
                  while Caller.Pending_ATC_Level >= Caller.ATC_Nesting_Level
                        loop
                     Cond_Wait (Caller.Rend_Cond, Caller.L);
                  end loop;
                  Caller.Suspended_Abortably := False;
                  Unlock (Caller.L);
                  Write_Lock (Acceptor.L);

               end if;
               Entry_Call.Acceptor_Prev_Call := Acceptor.Call;
               Acceptor.Call := Entry_Call;
               Acceptor.Chosen_Index := I;
               Acceptor.Accepting := Task_ATCB.Not_Accepting;
               Boost_Priority (Entry_Call, Acceptor);
               Cond_Signal (Acceptor.Cond);

               --  This needs to be protected by the caller's mutex, not the
               --  acceptor's.  Otherwise, there is a risk of loosing a
               --  signal.  This is dumb code, and probably could be fixed to
               --  some extent by getting rid of Test_Call.
               Unlock (Acceptor.L);
               Write_Lock (Caller.L);
               Caller.Suspended_Abortably := True;
               while not Entry_Call.Done loop
                  Cond_Wait (Caller.Rend_Cond, Caller.L);
               end loop;
               Caller.Suspended_Abortably := False;
               Unlock (Caller.L);
               Write_Lock (Acceptor.L);
               Rendezvous_Completed := True;
               return;
               --  rendezvous is over

            end if;
         end loop;

         Rendezvous_Completed := False;
      end if;
   end Test_Call;
   pragma Inline (Test_Call);

   procedure Vulnerable_Cancel_Task_Entry_Call (
         Call : Entry_Call_Link;
         Cancel_Was_Successful : out Boolean) is
      TAS_Result : Boolean;
      Caller : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Call.Self);
      Acceptor : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Call.Called_Task);
   begin
      Cancel_Was_Successful := False;
      Test_And_Set (Call.Call_Claimed'Address, TAS_Result);
      if TAS_Result then

         if not Call.Done then
         --  We should be able to check this flag at this point; we have
         --  claimed the call, so no one will be able to service this call, so
         --  no one else should be able to change the Call.Done flag.

            Write_Lock (Acceptor.L);
            Dequeue (Acceptor.Entry_Queues (Task_Entry_Index (Call.E)), Call);
            Unlock (Acceptor.L);
            Cancel_Was_Successful := True;
            --  Note: this will indicate failure to cancel if the acceptor has
            --  canceled the call due to completion.  Of course, we are going
            --  to raise an exception in that case, so I think that this is
            --  OK; the flag retuned to the application code should never be
            --  used.
         end if;
      else
         Write_Lock (Caller.L);
         while not Call.Done loop
            Cond_Wait (Caller.Rend_Cond, Caller.L);
         end loop;
         Unlock (Caller.L);
      end if;
      Caller.ATC_Nesting_Level := Caller.ATC_Nesting_Level - 1;

      Write_Lock (Caller.L);
      if Caller.Pending_ATC_Level = Caller.ATC_Nesting_Level then
         Caller.Pending_ATC_Level := ATC_Level_Infinity;
         Caller.Aborting := False;
      end if;
      Unlock (Caller.L);
      --  If we have reached the desired ATC nesting level, reset the
      --  requested level to effective infinity, to allow further calls.

      Caller.Exception_To_Raise := Call.Exception_To_Raise;

   end Vulnerable_Cancel_Task_Entry_Call;

   --  simple entry call
   procedure Call_Simple (Acceptor : Task_ID;
         E : Task_Entry_Index;
         Parameter : System.Address) is
      Caller : constant Task_ATCB.ATCB_Ptr :=
            Task_ATCB.ID_To_ATCB (Task_Stages.Self);
      Acceptor_ATCB : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Acceptor);
      Rendezvous_Completed : Boolean;
      Level : ATC_Level;
      Entry_Call : Entry_Call_Link;
      Cancel_Was_Successful : Boolean;
   begin
      Defer_Abortion;
      Write_Lock (Acceptor_ATCB.L);
      Caller.ATC_Nesting_Level := Caller.ATC_Nesting_Level + 1;
      Level := Caller.ATC_Nesting_Level;

      Entry_Call := Caller.Entry_Calls (Level)'access;

      Entry_Call.Next := null;
      Entry_Call.Call_Claimed := False;
      Entry_Call.Mode := C_M_To_EQ_Call_Modes (Simple_Call);
      Entry_Call.Abortable := True;
      Entry_Call.Done := False;
      Entry_Call.E := Entry_Index (E);
      Entry_Call.Prio := Caller.Current_Priority;
      Entry_Call.Parameter := Parameter;
      Entry_Call.Called_Task := Acceptor;
      Entry_Call.Exception_To_Raise := Compiler_Exceptions.Null_Exception;

      Test_Call (Entry_Call, Rendezvous_Completed);
      if not Rendezvous_Completed then
         Enqueue (Acceptor_ATCB.Entry_Queues (E), Entry_Call);
         Unlock (Acceptor_ATCB.L);
         Write_Lock (Caller.L);
         Caller.Suspended_Abortably := True;
         while not Entry_Call.Done and then
               (Caller.Pending_ATC_Level >= Caller.ATC_Nesting_Level or else
               Entry_Call.Call_Claimed) loop
            Cond_Wait (Caller.Rend_Cond, Caller.L);
         end loop;
         Caller.Suspended_Abortably := False;
         Unlock (Caller.L);
      else
         Unlock (Acceptor_ATCB.L);
      end if;

      --  WARNING:
      --  Is this right?  In the original runtime, abortion would decrement
      --  the nesting level as part of completing the task.  Now I'm not
      --  so sure; there is a distinction made between
      --  asynchronous calls and the rest that I think I may come to regret;
      --  the asynchronous calls are always cleaned up by
      --  Cancel_Task_Entry_Call, but the others get cleaned up by
      --  Task_Entry_Call or its equivalent.
      --  Complete no longer does a decrement, so who does if this task is
      --  aborted at this point?  I think that the decrement should take
      --  place before undeferring abortion, and that this should include
      --  taking the call off any queue it might be on.
      --  Problem: What if it is claimed in the meantime by an acceptor?  The
      --  test for Call_Claimed in the wait loop is really vulnerable to race
      --  conditions on this point.  We can't get out of the loop until
      --  Call_Claimed is false, but there is nothing to keep it from
      --  staying false.  By the time we get here, rendezvous could be in
      --  progress.  The only solution is to claim the call here in order
      --  to cancel it.  However, what do we do if we loose?  Wait again?
      --  I think so.  I also think that that works: wait until done or
      --  aborted; if aborted, attempt to cancel the call; if that fails, wait
      --  until the call (now well and truly started) completes, without
      --  benefit of Suspended_Abortably.
      --  Problem: The acceptor might also claim the call on completion, to
      --  cancel it.  In that case, it has already awakened us, and won't do it
      --  again.
      --  I think this is OK.  Close_Entries already pretends that the
      --  call has been completed, and has already set the exception at that
      --  point.

      Vulnerable_Cancel_Task_Entry_Call (
            Entry_Call,
            Cancel_Was_Successful);
      Undefer_Abortion;
      Assert (
            Caller.Pending_ATC_Level >= Caller.ATC_Nesting_Level,
            "Continuing after aborting self!");
      Check_Exception;
   end Call_Simple;

   procedure Cancel_Task_Entry_Call (Cancelled : out Boolean) is
      Caller : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Task_Stages.Self);
      Call : Entry_Call_Link;
      Acceptor : Task_ATCB.ATCB_Ptr;
   begin
      Assert (Caller.ATC_Nesting_Level > ATC_Level_Base'FIRST,
            "Attempt to cancel nonexistant task entry call.");
      Call := Caller.Entry_Calls (Caller.ATC_Nesting_Level)'access;
      Assert (Call.Mode = Asynchronous_Call,
            "Attempt to perform ATC on a non-asynchronous task entry call");
      Assert (Call.Called_PO = EQ_Null_PO,
            "Attempt to use Cancel_Task_Entry_Call on protected entry call.");
      Acceptor := Task_ATCB.ID_To_ATCB (Call.Called_Task);
      Defer_Abortion;
      Vulnerable_Cancel_Task_Entry_Call (Call, Cancelled);
      Undefer_Abortion;
      Check_Exception;
   end Cancel_Task_Entry_Call;

   procedure Requeue_Task_Entry (
         Acceptor : Task_ID;
         E : Task_Entry_Index;
         With_Abort : Boolean) is
      Old_Acceptor : Task_ATCB.ATCB_Ptr :=
            Task_ATCB.ID_To_ATCB (Task_Stages.Self);
      Entry_Call : Entry_Call_Link;
      Acceptor_ATCB : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Acceptor);
   begin
      Write_Lock (Old_Acceptor.L);
      Entry_Call := Old_Acceptor.Call;
      Old_Acceptor.Call := null;
      Unlock (Old_Acceptor.L);

      Entry_Call.Abortable := With_Abort;
      Entry_Call.E := Entry_Index (E);
      if With_Abort then
         Entry_Call.Call_Claimed := False;
      end if;

      Write_Lock (Acceptor_ATCB.L);
      Enqueue (Acceptor_ATCB.Entry_Queues (E), Entry_Call);
      Unlock (Acceptor_ATCB.L);
   end Requeue_Task_Entry;

   procedure Task_Entry_Call (Acceptor : Task_ID;
         E : Task_Entry_Index;
         Parameter : System.Address;
         Mode : Call_Modes;
         Rendezvous_Successful : out Boolean) is
      Caller : constant Task_ATCB.ATCB_Ptr :=
            Task_ATCB.ID_To_ATCB (Task_Stages.Self);
      Acceptor_ATCB : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Acceptor);
      Rendezvous_Completed : Boolean;
      Entry_Call : Entry_Call_Link;
      Cancel_Was_Successful : Boolean;
   begin
      if Mode = Simple_Call then
         Call_Simple (Acceptor, E, Parameter);
         Rendezvous_Successful := True;
         return;
      elsif Mode = Conditional_Call then
         Defer_Abortion;
         Caller.ATC_Nesting_Level := Caller.ATC_Nesting_Level + 1;

         Entry_Call := Caller.Entry_Calls (Caller.ATC_Nesting_Level)'access;

         Entry_Call.Next := null;
         Entry_Call.Call_Claimed := False;
         Entry_Call.Mode := C_M_To_EQ_Call_Modes (Mode);
         Entry_Call.Abortable := True;
         Entry_Call.Done := False;
         Entry_Call.E := Entry_Index (E);
         Entry_Call.Prio := Caller.Current_Priority;
         Entry_Call.Parameter := Parameter;
         Entry_Call.Called_Task := Acceptor;
         Entry_Call.Exception_To_Raise := Compiler_Exceptions.Null_Exception;
         Write_Lock (Acceptor_ATCB.L);
         Test_Call (Entry_Call, Rendezvous_Completed);
         Unlock (Acceptor_ATCB.L);
         if not Rendezvous_Completed then

            --  This is real junk code.  If the acceptor wasn't ready, the
            --  call shouldn't be queued at all.  Since it is, the acceptor can
            --  accept it while before we get here to check for failed
            --  rendezvous.  The code below breaks the race, but it would
            --  be better if it never arose.  Think about dumping Test_Call.
            Vulnerable_Cancel_Task_Entry_Call (
                  Entry_Call,
                  Cancel_Was_Successful);
         end if;
         Undefer_Abortion;
         Assert (
               Caller.Pending_ATC_Level >= Caller.ATC_Nesting_Level,
               "Continuing after aborting self!");
         Check_Exception;
         Rendezvous_Successful := Entry_Call.Done;
         return;
      else         --  Mode = Asynchronous_Call
         Defer_Abortion;
         Caller.ATC_Nesting_Level := Caller.ATC_Nesting_Level + 1;

         Entry_Call := Caller.Entry_Calls (Caller.ATC_Nesting_Level)'access;

         Entry_Call.Next := null;
         Entry_Call.Call_Claimed := False;
         Entry_Call.Mode := C_M_To_EQ_Call_Modes (Mode);
         Entry_Call.Abortable := True;
         Entry_Call.Done := False;
         Entry_Call.E := Entry_Index (E);
         Entry_Call.Prio := Caller.Current_Priority;
         Entry_Call.Parameter := Parameter;
         Entry_Call.Called_Task := Acceptor;
         Entry_Call.Called_PO := EQ_Null_PO;
         Entry_Call.Exception_To_Raise := Compiler_Exceptions.Null_Exception;

         Write_Lock (Acceptor_ATCB.L);
         Test_Call (Entry_Call, Rendezvous_Completed);
         if not Rendezvous_Completed then
            Enqueue (Acceptor_ATCB.Entry_Queues (E), Entry_Call);
         end if;
         Unlock (Acceptor_ATCB.L);
         Undefer_Abortion;
         Rendezvous_Successful := Entry_Call.Done;
         --  Amazingly, this seems to be all the work that is needed.
         --  Asynchronous calls are set up so that they are always explicitly
         --  canceled in in the compiled code.  It might be worth considering
         --  unifying the various calls, and explitely cancelling all of them.
         --  This is not very efficiant, unfortunately.  Perhaps this call
         --  should unify them, with other calls for optimization?  Then who
         --  would want to use this call?
      end if;
   end Task_Entry_Call;

   --  accept an entry call
   procedure Accept_Call (
         E : Task_Entry_Index;
         Parameter : out System.Address) is
      Acceptor : constant Task_ATCB.ATCB_Ptr :=
            Task_ATCB.ID_To_ATCB (Task_Stages.Self);
      Caller : Task_ATCB.ATCB_Ptr := null;
      TAS_Result : Boolean;

      Open_Accepts : aliased Accept_List (1 .. 1);

      Entry_Call : Entry_Call_Link;
   begin
      Defer_Abortion;
      Write_Lock (Acceptor.L);

      --  If someone is completing this task, it must be because they plan
      --  to abort it.  This task should not try to access its pending entry
      --  calls or queues in this case, as they are being emptied.  Wait for
      --  abortion to kill us.
      if Acceptor.Stage >= Task_ATCB.Completing then
         while Acceptor.Pending_ATC_Level >= Acceptor.ATC_Nesting_Level loop
            Cond_Wait (Acceptor.Cond, Acceptor.L);
         end loop;
         Unlock (Acceptor.L);
         Undefer_Abortion;
         Assert (False, "Continuing execution after being aborted.");
      end if;

      loop
         Dequeue_Head (Acceptor.Entry_Queues (E), Entry_Call);
         if Entry_Call /= null then
            Test_And_Set (Entry_Call.Call_Claimed'Address, TAS_Result);
            exit when TAS_Result;
            --  TAS_Result = False only when the caller is already aborted
            --  or timed out;
            --  in that case, we go on to the next caller on the queue
         else
            exit;
         end if;
      end loop;
      if Entry_Call /= null then
         Caller := Task_ATCB.ID_To_ATCB (Entry_Call.Self);
         Boost_Priority (Entry_Call, Acceptor);
         Entry_Call.Acceptor_Prev_Call := Acceptor.Call;
         Acceptor.Call := Entry_Call;
         Parameter := Entry_Call.Parameter;
      else
      --  wait for a caller
         Open_Accepts (1).Null_Body := false;
         Open_Accepts (1).S := E;
         Acceptor.Open_Accepts :=
               Open_Accepts'access;

         Acceptor.Accepting := Task_ATCB.Simple_Accept;
         --  wait for normal call
         Acceptor.Suspended_Abortably := True;
         while Acceptor.Accepting /= Task_ATCB.Not_Accepting and then
               Acceptor.Pending_ATC_Level >= Acceptor.ATC_Nesting_Level loop
            Cond_Wait (Acceptor.Cond, Acceptor.L);
         end loop;
         Acceptor.Suspended_Abortably := False;

         if Acceptor.Pending_ATC_Level >= Acceptor.ATC_Nesting_Level then
            Caller := Task_ATCB.ID_To_ATCB (Acceptor.Call.Self);
            Parameter :=
                  Caller.Entry_Calls (Caller.ATC_Nesting_Level).Parameter;
         end if;
         --  If this task has been aborted, skip the parameter load (Caller
         --  will not be reliable) and fall through to Undefer_Abortion which
         --  will allow the task to be killed.

      end if;

      --  At this point, the call has been claimed, either by the acceptor
      --  or by the caller on behalf of the acceptor.

      --  Acceptor.Call should already be updated by the Caller
      Unlock (Acceptor.L);
      Undefer_Abortion;
      --  start rendezvous
   end Accept_Call;

   --  accept an entry call that has no parameters and no body
   --  so, there is no exception to be propagated to the caller?
   procedure Accept_Trivial (E : Task_Entry_Index) is
      Acceptor : constant Task_ATCB.ATCB_Ptr :=
            Task_ATCB.ID_To_ATCB (Task_Stages.Self);
      Caller : Task_ATCB.ATCB_Ptr := null;
      TAS_Result : Boolean;

      Open_Accepts : aliased Accept_List (1 .. 1);

      Entry_Call : Entry_Call_Link;
   begin
      Defer_Abortion;
      Write_Lock (Acceptor.L);

      --  If someone is completing this task, it must be because they plan
      --  to abort it.  This task should not try to access its pending entry
      --  calls or queues in this case, as they are being emptied.  Wait for
      --  abortion to kill us.
      if Acceptor.Stage >= Task_ATCB.Completing then
         while Acceptor.Pending_ATC_Level >= Acceptor.ATC_Nesting_Level loop
            Cond_Wait (Acceptor.Cond, Acceptor.L);
         end loop;
         Unlock (Acceptor.L);
         Undefer_Abortion;
         Assert (False, "Continuing execution after being aborted.");
      end if;

      loop
         Dequeue_Head (Acceptor.Entry_Queues (E), Entry_Call);
         if Entry_Call = null then --  need to wait for call
            Open_Accepts (1).Null_Body := false;
            Open_Accepts (1).S := E;
            Acceptor.Open_Accepts :=
                  Open_Accepts'access;

            Acceptor.Accepting := Task_ATCB.Trivial_Accept;
            --  wait for normal entry call
            Acceptor.Suspended_Abortably := True;
            while Acceptor.Accepting /= Task_ATCB.Not_Accepting and then
                  Acceptor.Pending_ATC_Level >= Acceptor.ATC_Nesting_Level loop
               Cond_Wait (Acceptor.Cond, Acceptor.L);
            end loop;
            Acceptor.Suspended_Abortably := False;
            Entry_Call := Acceptor.Call;
            Acceptor.Call := Entry_Call.Acceptor_Prev_Call;
            Caller := Task_ATCB.ID_To_ATCB (Entry_Call.Self);
            if EQ_C_M_To_Call_Modes (Entry_Call.Mode) = Asynchronous_Call then
               Abort_To_Level (Task_ATCB.ATCB_To_ID (Caller),
                     Entry_Call.Level);
            end if;
            Unlock (Acceptor.L);
            if Acceptor.Pending_ATC_Level < Acceptor.ATC_Nesting_Level then
               Undefer_Abortion;
               Assert (False, "Continuing after being aborted!");
            end if;
            exit;
         end if;
         Test_And_Set (Entry_Call.Call_Claimed'address, TAS_Result);
         if TAS_Result then      --  caller is waiting; there is no accept body
            Caller := Task_ATCB.ID_To_ATCB (Entry_Call.Self);
            Unlock (Acceptor.L);
            Write_Lock (Caller.L);
            Entry_Call.Done := True;
            --  Done with mutex locked to make sure that signal is not lost.
            Unlock (Caller.L);
            Entry_Call.Call_Claimed := False;
            if EQ_C_M_To_Call_Modes (Entry_Call.Mode) = Asynchronous_Call then
               Abort_To_Level (Task_ATCB.ATCB_To_ID (Caller),
                     Entry_Call.Level);
            else
               Cond_Signal (Caller.Rend_Cond);
            end if;
            exit;
         end if;
         --  TAS_Result = False only when the caller is already aborted
         --  or has timed out;
         --  in that case, we go on to the next caller on the queue
      end loop;
      Undefer_Abortion;
   end Accept_Trivial;

   --  called by acceptor to wake up caller and (optionally) propagate
   --  exception
   procedure Universal_Complete_Rendezvous (
         Ex : Compiler_Exceptions.Exception_ID) is
      Acceptor : constant Task_ATCB.ATCB_Ptr :=
            Task_ATCB.ID_To_ATCB (Task_Stages.Self);
      Caller : Task_ATCB.ATCB_Ptr;
      Call : Entry_Call_Link;
      Prev_Priority : Rendezvous_Priority;
   begin
      Defer_Abortion;
      Call := Acceptor.Call;
      Acceptor.Call := Call.Acceptor_Prev_Call;
      Prev_Priority := Call.Acceptor_Prev_Priority;
      Call.Exception_To_Raise := Ex;
      Caller := Task_ATCB.ID_To_ATCB (Call.Self);
      Call.Call_Claimed := False;
      Write_Lock (Caller.L);
      Call.Done := True;
      Unlock (Caller.L);
      if EQ_C_M_To_Call_Modes (Call.Mode) = Asynchronous_Call then
         Abort_To_Level (Task_ATCB.ATCB_To_ID (Caller), Call.Level);
      else
         Cond_Signal (Caller.Rend_Cond);
      end if;
      Reset_Priority (Prev_Priority, Task_ATCB.ATCB_To_ID (Acceptor));

      Acceptor.Exception_To_Raise := Ex;
      --  Save the exception for Complete_Rendezvous.
      Undefer_Abortion;
   end Universal_Complete_Rendezvous;
   pragma Inline (Universal_Complete_Rendezvous);

   --  called by acceptor to mark the end of the current rendezvous.
   procedure Complete_Rendezvous is
   begin
      Universal_Complete_Rendezvous (Compiler_Exceptions.Null_Exception);
   end Complete_Rendezvous;

   --  called by acceptor to mark the end of the current rendezvous and
   --  store an exception for later propagation by Complete_Rendezvous.
   --  called by acceptor to wake up caller and propagate exception
   procedure Exceptional_Complete_Rendezvous (
         Ex : Compiler_Exceptions.Exception_ID) is
   begin
      Universal_Complete_Rendezvous (Ex);
   end Exceptional_Complete_Rendezvous;

   --  test if there is a call waiting on any entry,
   --  and whether any selects are open;
   --  set Acceptor.Chosen_Index to selected alternative,
   --  if an accept alternative can be selected.
   function Test_Selective_Wait
         (Acceptor : Task_ATCB.ATCB_Ptr;

         Open_Accepts : Accept_List_Access;

         Select_Mode : Select_Modes) return Select_Treatment is
      Temp_Entry : Task_Entry_Index;
      Caller : Task_ATCB.ATCB_Ptr := null;
      TAS_Result : Boolean;
      Treatment : Select_Treatment;
      Entry_Call : Entry_Call_Link;
   begin
      Treatment := Default_Treatment (Select_Mode);
      Acceptor.Chosen_Index := No_Rendezvous;
      for I in Open_Accepts'range loop
         Temp_Entry := Open_Accepts (I).S;
         if Temp_Entry /= Null_Task_Entry then --  guard is open
            loop
               Dequeue_Head (Acceptor.Entry_Queues (Temp_Entry), Entry_Call);
               if Entry_Call /= null then
                  Test_And_Set (Entry_Call.Call_Claimed'Address, TAS_Result);
                  exit when TAS_Result;
                  --  TAS_Result = False only when the call is already canceled
                  --  in that case, we go on to the next call on the queue
               else exit;
               end if;
            end loop;
            if Entry_Call /= null then
               Caller := Task_ATCB.ID_To_ATCB (Entry_Call.Self);
               Boost_Priority (Entry_Call, Acceptor);
               Acceptor.Chosen_Index := I;
               Entry_Call.Acceptor_Prev_Call := Acceptor.Call;
               Acceptor.Call := Entry_Call;
               Treatment := Accept_Alternative_Selected;
               exit;
            elsif Treatment = No_Alternative_Open then
               Treatment := Accept_Alternative_Open;
            end if;                        --  do rendezvous
         end if;                          --  open alternative
      end loop;

      return Treatment;
      --  to get around the problem with optimizer when returning
      --  from 2 level-depthed if statement inside for loop.
      --  Otherwise this function will always return False.!@#?

   end Test_Selective_Wait;
   pragma Inline (Test_Selective_Wait);

   --  selective wait
   procedure Selective_Wait (

         Open_Accepts : Accept_List_Access;

         Select_Mode : Select_Modes;
         Parameter : out System.Address;
         Index : out Select_Index)is
      Acceptor : constant Task_ATCB.ATCB_Ptr :=
            Task_ATCB.ID_To_ATCB (Task_Stages.Self);
      Treatment : Select_Treatment;

      I_Result : Integer;
   begin
      Defer_Abortion;
      Write_Lock (Acceptor.L);

      --  If someone is completing this task, it must be because they plan
      --  to abort it.  This task should not try to access its pending entry
      --  calls or queues in this case, as they are being emptied.  Wait for
      --  abortion to kill us.
      if Acceptor.Stage >= Task_ATCB.Completing then
         while Acceptor.Pending_ATC_Level >= Acceptor.ATC_Nesting_Level loop
            Cond_Wait (Acceptor.Cond, Acceptor.L);
         end loop;
         Undefer_Abortion;
         Assert (False, "Continuing execution after being aborted.");
      end if;

      Treatment := Test_Selective_Wait (Acceptor, Open_Accepts, Select_Mode);

      case Treatment is

      when Accept_Alternative_Selected =>
      --  ready to rendezvous already
         Parameter := Acceptor.Call.Parameter;

      when Accept_Alternative_Open =>
      --  wait for caller.

         Acceptor.Open_Accepts := Open_Accepts;

         Acceptor.Accepting := Task_ATCB.Select_Wait;
         Acceptor.Suspended_Abortably := True;
         while Acceptor.Accepting /= Task_ATCB.Not_Accepting and then
               Acceptor.Pending_ATC_Level >= Acceptor.ATC_Nesting_Level loop
            Cond_Wait (Acceptor.Cond, Acceptor.L);
         end loop;
         Acceptor.Suspended_Abortably := False;
         if Acceptor.Pending_ATC_Level >= Acceptor.ATC_Nesting_Level then
            Parameter := Acceptor.Call.Parameter;
         end if;
         --  Acceptor.Call should already be updated by the Caller if
         --  not aborted.

      when Else_Selected =>
         Acceptor.Accepting := Task_ATCB.Not_Accepting;

      when Terminate_Selected =>
      --  terminate alternative is open

         Acceptor.Open_Accepts := Open_Accepts;

         Acceptor.Accepting := Task_ATCB.Select_Wait;
         --  We need to check if a signal is pending on an open interrupt
         --  entry. Otherwise this task would become passive (since terminate
         --  alternative is open) and, if none of the siblings are active
         --  anymore, the task could not wake up anymore, even though a
         --  signal might be pending on an open interrupt entry.

         Unlock (Acceptor.L);
         Terminate_Alternative;
         --  wait for normal entry call or termination
         --  consider letting Terminate_Alternative assume mutex L
         --  is already locked, and return with it locked, so
         --  this code could be simplified?
         --  no return here if Acceptor completes, otherwise
         --  Acceptor.Call should already be updated by the Caller
         Index := Acceptor.Chosen_Index;
         Parameter := Acceptor.Call.Parameter;
         Undefer_Abortion;
         return;

      when No_Alternative_Open =>
      --  Acceptor.Chosen_Index := No_Rendezvous; => Program_Error
         null;
      end case;
      --  caller has been chosen
      --  Acceptor.Call should already be updated by the Caller
      --  Acceptor.Chosen_Index should either be updated by the Caller
      --    or by Test_Selective_Wait
      Index := Acceptor.Chosen_Index;
      Unlock (Acceptor.L);
      Undefer_Abortion;
      --  start rendezvous
   end Selective_Wait;

   --  return number of tasks waiting on the entry E (of current task)
   function Task_Count (E : Task_Entry_Index) return Natural is
      T : constant Task_ATCB.ATCB_Ptr
            := Task_ATCB.ID_To_ATCB (Task_Stages.Self);
      Return_Count : Natural;
   begin
      Write_Lock (T.L);

      Return_Count :=
            System.Task_Entry_Queue.Count_Waiting (T.Entry_Queues (E));

      Unlock (T.L);
      return Return_Count;
   end Task_Count;

   --  return T'CALLABLE
   function Callable (T : Task_ID) return Boolean is
   begin
      return Task_ATCB.ID_To_ATCB (T).Stage < Task_ATCB.Complete and then
            Task_ATCB.ID_To_ATCB (T).Pending_ATC_Level > ATC_Level_Base'FIRST;
   end Callable;

   --  *LATER* get rid of T.Callable component
   --  and use Stage >= Complete or Abnormal?

   --  close entries, purge entry queues
   --  called by Task_Stages.Complete
   --  T.Stage must be Completing before this is called.
   procedure Close_Entries (Target : Task_ID) is
      T : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Target);
      Temp_Call : Entry_Call_Link;
      Null_Call : Entry_Call_Link := null;
      Temp_Caller : Task_ATCB.ATCB_Ptr;
      TAS_Result : Boolean;
   begin

      --  purging pending callers that are in the middle of rendezvous

      Temp_Call := T.Call;
      while Temp_Call /= null loop
         Temp_Call.Exception_To_Raise := Compiler_Exceptions.Tasking_Error_ID;

         Temp_Caller := Task_ATCB.ID_To_ATCB (Temp_Call.Self);

         --  Problem: Once this lock is unlocked, the target gan go on to
         --  accept other calls, which will be missed by loop.  The question
         --  is, is there something else that will prevent this?
         --  If the target is in an abortion deferred region at this point,
         --  I don't know what it would be.

         --  By the time we get here, we do know that the target is complete
         --  and not callable.  Callable is unprotected, but Stage is protected
         --  by T.L.  If all forms of accept made sure under the protection of
         --  T.L that they were not complete before accepting a call, then it
         --  should be safe to unlock this here.

         --  Problem: what about multiple aborters?  If two tasks are in this
         --  routine at once, then there could contention if this mutex is
         --  unlocked.  We need some other form of claim mechanism to prevent
         --  this.  I think that the mechanism outlined in the implementation
         --  sketch, where an aborter waits for a previous aborter to finish
         --  its work, might solve this.

         --  What if T itself is at exactly this point and gets aborted?  In
         --  that case, I think that the aborter has to wait for T to finish
         --  completing itself.  This was previously done by contending for the
         --  mutex; it might now have to be done with some kind of flag, or
         --  maybe another stage.  Perhaps we are setting Stage=Complete too
         --  soon, and abortion should wait on that.  That would require
         --  some other flag to claim the right to complete, however.  This
         --  flag could probably be protected by T.L; there should not be
         --  any need for a TAS or global mutex.  Perhaps the Aborting flag
         --  could do this, though right now all it means is that an abortion
         --  exception has been sent.  We really need a separate Completing
         --  flag (ugh).  On the bright side, this might mean that completion
         --  can be treated as once-and-once-only, and need not be reentrant.

         --  Problem: What does an acceptor do when it finds that it is being
         --  completed?  I guess it should wait until completion is finished,
         --  just like a second aborter.  Otherwise, it might continue on
         --  with a rendezvous that it never really accepted.

         Write_Lock (Temp_Caller.L);
         Temp_Call.Done := True;
         Unlock (Temp_Caller.L);
         --  The caller can break out of its loop at this point, and never
         --  notice the abortion.
--       Temp_Call.Call_Claimed:= False;
--  Wrong, I think.  This should look like a completed call to everyone.
         Abort_To_Level (Task_ATCB.ATCB_To_ID (Temp_Caller),
               Temp_Call.Level - 1);
         --  I think this might be wrong; Abortion takes precedence over
         --  exceptions in the call block.
         --  Not true; the last call to be canceled won't raise Abortion again;
         --  it raises the chosen exception instead.  This is true of leaf
         --  (suspending) calls as well; they decrement the nesting level
         --  before undeferring abortion, which will prevent further abortion
         --  (providing that abortion is not to an outer level).

         Temp_Call := Temp_Call.Acceptor_Prev_Call;
      end loop;

      --  purging entry queues

      for I in 1 .. T.Entry_Num loop
         Dequeue_Head (T.Entry_Queues (I), Temp_Call);
         if Temp_Call /= Null_Call then
            loop
               Test_And_Set (Temp_Call.Call_Claimed'Address, TAS_Result);
               if TAS_Result then
                  Temp_Caller := Task_ATCB.ID_To_ATCB (Temp_Call.Self);
                  Temp_Call.Exception_To_Raise :=
                        Compiler_Exceptions.Tasking_Error_ID;
                  Temp_Call.Done := True;
                  Abort_To_Level (Task_ATCB.ATCB_To_ID (Temp_Caller),
                        Temp_Call.Level - 1);
               --  else
                  --  Someone else claimed this call.  It must be to cancel it,
                  --  since the acceptor can't have accepted it at this point.
                  --  So far as we are concerned, this call is not on the
                  --  queue, and we don't have to raise tasking error in the
                  --  caller.
               end if;
               Dequeue_Head (T.Entry_Queues (I), Temp_Call);
               exit when Temp_Call = Null_Call;
            end loop;
         end if;
      end loop;

   end Close_Entries;

   --  If a task is suspended on an accept, select, or
   --  entry call (but not yet *in* rendezvous) then complete the task.
   procedure Complete_on_Sync_Point (T : Task_IDs.Task_ID) is
      Target : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (T);
      Call : Task_Entry_Queue.Entry_Call_Link;
      TAS_Result : Boolean;
   begin
      Write_Lock (Target.L);
      if Target.Suspended_Abortably then

         if Target.Accepting /= Task_ATCB.Not_Accepting then
            Unlock (Target.L);
            Task_Stages.Complete (T);
         else  --  Hopefully suspended on an entry call by elimination.
            if Target.ATC_Nesting_Level > ATC_Level_Base'First then
               Call :=
                     Target.Entry_Calls (Target.ATC_Nesting_Level)'access;
               Test_And_Set (Call.Call_Claimed'Address, TAS_Result);
               if TAS_Result then
                  Unlock (Target.L);
                  Task_Stages.Complete (T);
                  Call.Call_Claimed := False;
                  --  To allow abortion to claim it.
               else
                  Unlock (Target.L);
               end if;
            end if;
         end if;
      else
         Unlock (Target.L);
      end if;
   end Complete_on_Sync_Point;

end System.Task_Rendezvous;
