------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                        R T S _ P r i m i t i v e s                       --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.2 $                            --
--                                                                          --
--           Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                          --
--  GNARL is free software; you can redistribute it and/or modify it  under --
--  terms  of  the  GNU  Library General Public License as published by the --
--  Free Software Foundation; either version 2, or (at  your  option)  any --
--  later  version.   GNARL is distributed in the hope that it will be use- --
--  ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
--  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
--  eral Library Public License for more details.  You should have received --
--  a  copy of the GNU Library General Public License along with GNARL; see --
--  file COPYING. If not, write to the Free Software Foundation,  675  Mass --
--  Ave, Cambridge, MA 02139, USA.                                          --
--                                                                          --
------------------------------------------------------------------------------

with Unchecked_Deallocation;

with System.Machine_Specifics;
with System.POSIX_Error; use System.POSIX_Error;
with System.POSIX_RTE; use System.POSIX_RTE;
with System.Pthreads; use System.Pthreads;
with System.Task_Clock;

with Unchecked_Conversion;

package body System.Task_Primitives is

   Abort_Signal : constant Signal_Number := SIGUSR1;

   function "=" (L, R : System.Address) return Boolean renames System."=";

   ATCB_Key : pthread_key_t;

   Abort_Handler : Abort_Handler_Pointer;

   LL_Signals : sigset_t;
   Task_Signal_Mask : sigset_t;

   type Signal_Set is array (Machine_Specifics.Interrupt_ID) of Boolean;
   Reserved_Signals : Signal_Set;

   Assertions_Checked : constant Boolean := True;

   procedure Put_Character (C : Integer);
   pragma Interface (C, Put_Character);
   pragma Interface_Name (Put_Character, "putchar");

   procedure Write_Character (C : Character) is
   begin
      Put_Character (Character'Pos (C));
   end Write_Character;

   procedure Write_EOL is
   begin
      Write_Character (ascii.lf);
   end Write_EOL;

   procedure Write_String (S : String) is
   begin
      for I in S'range loop
         Write_Character (S (I));
      end loop;
   end Write_String;

   procedure Prog_Exit (Status : Integer);
   pragma Interface (C, Prog_Exit);
   pragma Interface_Name (Prog_Exit, "exit");
   --  for UNIX version only

   procedure LL_Assert (B : Boolean; M : String) is
   begin
      if not B then
         Write_String ("Failed assertion: ");
         Write_String (M);
         Write_String (".");
         Write_EOL;
         Prog_Exit (1);
      end if;
   end LL_Assert;

   procedure Assert (B : Boolean; M : String) is
   begin
      if Assertions_Checked then
         LL_Assert (B, M);
      end if;
   end Assert;
   pragma Inline (Assert);

   procedure Initialize_LL_Tasks (T : TCB_Ptr) is
      Old_Set : sigset_t;

      Mask : sigset_t;
      Result : Return_Code;
      function Pointer_to_Address is new
            Unchecked_Conversion (TCB_Ptr, System.Address);
   begin

   --  WARNING : SIGALRM should not be in the following mask.  SIGALRM should
   --          be a normal user signal under 1, and should be enabled
   --          by the client.  However, the current RTS built on 1
   --          uses nanosleep () and pthread_cond_wait (), which fail if all
   --          threads have SIGALRM masked.
      LL_Signals := Sigmask (Abort_Signal) + Sigmask (SIGALRM) +
            Sigmask (SIGILL) + Sigmask (SIGABRT) + Sigmask (SIGFPE) +
            Sigmask (SIGSEGV) + Sigmask (SIGPIPE);
      Task_Signal_Mask := All_Signals - LL_Signals;
      pthread_init;
      Reserved_Signals := (others => False);
      Reserved_Signals (Machine_Specifics.Interrupt_ID (SIGILL)) := True;
      Reserved_Signals (Machine_Specifics.Interrupt_ID (SIGABRT)) := True;
      Reserved_Signals (Machine_Specifics.Interrupt_ID (SIGFPE)) := True;
      Reserved_Signals (Machine_Specifics.Interrupt_ID (SIGSEGV)) := True;
      Reserved_Signals (Machine_Specifics.Interrupt_ID (SIGPIPE)) := True;
      Reserved_Signals (Machine_Specifics.Interrupt_ID (Abort_Signal)) := True;

      pthread_key_create (ATCB_Key, System.Null_Address, Result);

      if Result = Failure then
         raise Storage_Error;               --  Insufficiant resources.
      end if;

      sigprocmask (SIG_SETMASK, Task_Signal_Mask, Old_Set, Result);
      Assert (Result /= Failure, "GNULLI failure---sigprocmask");

      T.LL_Entry_Point := null;

      T.Thread := pthread_self;
      pthread_setspecific (ATCB_Key, Pointer_to_Address (T), Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_setspecific");
   end Initialize_LL_Tasks;

   function Self return TCB_Ptr is
      Temp : System.Address;
      Result : Return_Code;
      function Address_to_Pointer is new
            Unchecked_Conversion (System.Address, TCB_Ptr);
   begin
      pthread_getspecific (ATCB_Key, Temp, Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_getspecific");
      return Address_to_Pointer (Temp);
   end Self;

   procedure Initialize_Lock (Prio : System.Priority;
         L : in out Lock) is
         Attributes : pthread_mutexattr_t;
         Result : Return_Code;
   begin
      pthread_mutexattr_init (Attributes, Result);
      if Result = Failure then
         raise STORAGE_ERROR;  --  should be ENOMEM
      end if;
      pthread_mutexattr_setprotocol (Attributes, PRIO_PROTECT, Result);
      Assert (Result /= Failure,
            "GNULLI failure---pthread_mutexattr_setprotocol");
      pthread_mutexattr_setprio_ceiling (Attributes, Prio, Result);
      Assert (Result /= Failure,
            "GNULLI failure---pthread_mutexattr_setprio_ceiling");
      pthread_mutex_init (pthread_mutex_t (L), Attributes, Result);
      if Result = Failure then
         raise STORAGE_ERROR;  --  should be ENOMEM
      end if;
   end Initialize_Lock;

   procedure Finalize_Lock (L : in out Lock) is
      Result : Return_Code;
   begin
      pthread_mutex_destroy (pthread_mutex_t (L), Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_mutex_destroy");
   end Finalize_Lock;

   procedure Write_Lock (L : in out Lock) is
      Result : Return_Code;
   begin
      pthread_mutex_lock (pthread_mutex_t (L), Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_mutex_lock");
   end Write_Lock;

   procedure Read_Lock (L : in out Lock) is
   begin
      Write_Lock (L);
   end Read_Lock;

   procedure Unlock (L : in out Lock) is
      Result : Return_Code;
   begin
      pthread_mutex_unlock (pthread_mutex_t (L), Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_mutex_unlock");
   end Unlock;

   procedure Initialize_Cond (Cond : in out Condition_Variable) is
      Attributes : pthread_condattr_t;
      Result : Return_Code;
   begin
      pthread_condattr_init (Attributes, Result);
      if Result = Failure then
         raise STORAGE_ERROR;  --  should be ENOMEM
      end if;
      pthread_cond_init (pthread_cond_t (Cond), Attributes, Result);
      if Result = Failure then
         raise STORAGE_ERROR;  --  should be ENOMEM
      end if;
      pthread_condattr_destroy (Attributes, Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_condattr_destroy");
   end Initialize_Cond;

   procedure Finalize_Cond (Cond : in out Condition_Variable) is
      Result : Return_Code;
   begin
      pthread_cond_destroy (pthread_cond_t (Cond), Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_cond_destroy");
   end Finalize_Cond;

   procedure Cond_Wait (Cond : in out Condition_Variable;
         L : in out Lock) is
      Result : Return_Code;
   begin
      pthread_cond_wait (pthread_cond_t (Cond), pthread_mutex_t (L), Result);

      Assert (Result /= Failure or else errno = EINTR,
            "GNULLI failure---pthread_cond_wait");
      --  EINTR is not considered a failure.  We have been assured that
      --  Pthreads will soon guarantee that a thread will wake up from
      --  a condition variable wait after it handles a signal.  EINTR will
      --  probably go away at that point.

   end Cond_Wait;

   procedure Cond_Timed_Wait (Cond : in out Condition_Variable;
         L : in out Lock; Abs_Time : Task_Clock.Stimespec;
         Timed_Out : out Boolean) is
      Result : Return_Code;
      I_Result : Integer;
      function Stimespec_to_timespec is new
            Unchecked_Conversion (Task_Clock.Stimespec, timespec);
   begin
      pthread_cond_timedwait (pthread_cond_t (Cond),
            pthread_mutex_t (L), Stimespec_to_timespec (Abs_Time), Result);
      Timed_Out := Result = Failure and then errno = EAGAIN;
      Assert (Result /= Failure or else errno = EAGAIN,
            "GNULLI failure---pthread_cond_timedwait");
   end Cond_Timed_Wait;

   procedure Cond_Signal (Cond : in out Condition_Variable) is
      Result : Return_Code;
   begin
      pthread_cond_signal (pthread_cond_t (Cond), Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_cond_signal");
   end Cond_Signal;

   procedure Cond_Broadcast (Cond : in out Condition_Variable) is
      Result : Return_Code;
   begin
      pthread_cond_broadcast (pthread_cond_t (Cond), Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_cond_signal");
   end Cond_Broadcast;

   procedure Set_Priority (T : TCB_Ptr; Prio : System.Priority) is
      Attributes : pthread_attr_t;
      Result : Return_Code;
   begin
      pthread_attr_init (Attributes, Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_attr_init");
      pthread_getschedattr (T.Thread, Attributes, Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_getschedattr");
      pthread_attr_setprio (Attributes, Priority_Type (Prio), Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_setprio");
      pthread_setschedattr (T.Thread, Attributes, Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_setschedattr");
   end Set_Priority;

   procedure Set_Own_Priority (Prio : System.Priority) is
      Attributes : pthread_attr_t;
      Result : Return_Code;
   begin
      Set_Priority (Self, Prio);
   end Set_Own_Priority;

   function Get_Priority (T : TCB_Ptr) return System.Priority is
      Attributes : pthread_attr_t;
      Prio : Priority_Type;
      Result : Return_Code;
   begin
      pthread_attr_init (Attributes, Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_attr_init");
      pthread_getschedattr (T.Thread, Attributes, Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_getschedattr");
      pthread_attr_getprio (Attributes, Prio, Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_getprio");
      return System.Priority (Prio);
   end Get_Priority;

   function Get_Own_Priority return System.Priority is
   begin
      return Get_Priority (Self);
   end Get_Own_Priority;

   procedure LL_Wrapper (T : TCB_Ptr) is
      Result : Return_Code;
      function Pointer_to_Address is new
            Unchecked_Conversion (TCB_Ptr, System.Address);
   begin
      pthread_setspecific (ATCB_Key, Pointer_to_Address (T), Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_setspecific");
      T.LL_Entry_Point (T.LL_Arg);
      --  the entry point call may not return!
   end LL_Wrapper;

   procedure Create_LL_Task (
         Priority : System.Priority;
         Stack_Size :  Machine_Specifics.Task_Storage_Size;
         LL_Entry_Point : Machine_Specifics.Init_State;
         Arg : System.Address;
         T : TCB_Ptr) is
      Attributes : pthread_attr_t;
      Result : Return_Code;
      function Pointer_to_Address is new
            Unchecked_Conversion (TCB_Ptr, System.Address);
      Old_Set : sigset_t;
   begin
      T.LL_Entry_Point := LL_Entry_Point;
      T.LL_Arg := Arg;
      T.Stack_Size := Stack_Size;
      pthread_attr_init (Attributes, Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_attr_init");
      pthread_attr_setdetachstate (Attributes, 1, Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_setdetachstate");
      pthread_attr_setstacksize (Attributes, size_t (Stack_Size),
            Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_setstacksize");
      pthread_attr_setprio (Attributes, Priority_Type (Priority),
            Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_attr_setprio");

      --  It is not safe for the task to be created to accept signals until
      --  it has bound its TCB pointer to the thread using
      --  pthread_setspecific ().
      --  The handler wrappers use the TCB pointers to restore the stack limit.
      sigprocmask (SIG_BLOCK, LL_Signals, Old_Set, Result);
      Assert (Result /= Failure, "GNULLI failure---sigprocmask");

      pthread_create (T.Thread,
            Attributes,
            LL_Wrapper'Address,
            Pointer_to_Address (T),
            Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_create");

      sigprocmask (SIG_UNBLOCK, LL_Signals, Old_Set, Result);
      Assert (Result /= Failure, "GNULLI failure---sigprocmask");

   end Create_LL_Task;

   procedure Exit_LL_Task is
   begin

      pthread_exit (System.Null_Address);

   end Exit_LL_Task;

   procedure Deallocate_TCB (T : in out TCB_Ptr) is

      procedure Old_TCB is new
            Unchecked_Deallocation (Task_Control_Block, TCB_Ptr);

   begin

      Old_TCB (T);

      T := null;
   end Deallocate_TCB;

   procedure Abort_Task (T : TCB_Ptr) is
      Result : Return_Code;
   begin
      pthread_kill (T.Thread, Abort_Signal, Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_kill");
   end Abort_Task;

   procedure Test_Abort is
   begin
      null;
   end Test_Abort;

   function Get_Stack_Limit return System.Address is
   begin
      return Self.Stack_Limit;
   end Get_Stack_Limit;
   pragma Inline (Get_Stack_Limit);

   procedure Abort_Wrapper (signo : Integer;
         info : siginfo_ptr;
         context : System.Address) is

--  These are temporary things

   function aaa is new Unchecked_Conversion (
         System.Address,
         Machine_Specifics.Pre_Call_State);

   begin
      Abort_Handler (aaa (context));
--  begin
--    Abort_Handler (Machine_Specifics.Pre_Call_State (context));

   end Abort_Wrapper;

   --  Note that this currently takes System.Address.  The 1 specifies
   --  access procedure (Context : Pre_Call_State) for the handler type.
   --  This may be a mistake of the interface in commiting to this 9X type.
   --  The correct way to do it may be to make this a type in
   --  Machine_Specifics,
   --  which can then be created with a constructor funciton in one place.
   --  However, Ada 83 compilers are always going to have to take the address
   --  of the procedure, if only to pass it to a constructor function.

   procedure Install_Abort_Handler (
         Handler : Abort_Handler_Pointer) is

      act, old_act : struct_sigaction;
      Result : Return_Code;
   begin
      Abort_Handler := Handler;
      act.sa_handler := Abort_Wrapper'Address;
      act.sa_mask := 0;
      act.sa_flags := 0;
      sigaction (Abort_Signal, act, old_act, Result);
      Assert (Result /= Failure, "GNULLI failure---sigaction");
   end Install_Abort_Handler;

   procedure Install_Error_Handler (
         Handler : System.Address) is
      act, old_act : struct_sigaction;
      Result : Return_Code;
   begin
      act.sa_handler := Handler;
      act.sa_mask :=
            Sigmask (SIGILL) +
            Sigmask (SIGABRT) +
            Sigmask (SIGFPE) +
            Sigmask (SIGSEGV) +
            Sigmask (SIGPIPE);
      act.sa_flags := 0;
      sigaction (SIGILL, act, old_act, Result);
      Assert (Result /= Failure, "GNULLI failure---sigaction");
      sigaction (SIGABRT, act, old_act, Result);
      Assert (Result /= Failure, "GNULLI failure---sigaction");
      sigaction (SIGFPE, act, old_act, Result);
      Assert (Result /= Failure, "GNULLI failure---sigaction");
      sigaction (SIGSEGV, act, old_act, Result);
      Assert (Result /= Failure, "GNULLI failure---sigaction");
      sigaction (SIGPIPE, act, old_act, Result);
      Assert (Result /= Failure, "GNULLI failure---sigaction");
   end Install_Error_Handler;

   procedure Signal_Task (T : TCB_Ptr;
         I : Machine_Specifics.Interrupt_ID) is
         Result : Return_Code;
   begin
      pthread_kill (T.Thread, Signal_Number (I), Result);
      Assert (Result /= Failure, "GNULLI failure---pthread_kill");
   end Signal_Task;

   procedure Wait_for_Signal (I : Machine_Specifics.Interrupt_ID) is
      Temp_Signal : Signal_Number;
      Result : Return_Code;
   begin
      sigwait (Sigmask (Signal_Number (I)), Temp_Signal, Result);
      Assert (Result /= Failure, "GNULLI failure---sigwait");
   end Wait_For_Signal;

   function Reserved_Signal (
         I : Machine_Specifics.Interrupt_ID) return Boolean is
   begin
      return Reserved_Signals (I);
   end Reserved_Signal;

end System.Task_Primitives;
