------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                        R T S _ E n t r y _ Q u e u e                     --
--                                                                          --
--                                  S p e c                                 --
--                                                                          --
--                             $Revision: 1.2 $                             --
--                                                                          --
--           Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                          --
--  GNARL is free software; you can redistribute it and/or modify it  under --
--  terms  of  the  GNU  Library General Public License as published by the --
--  Free Software Foundation; either version 2, or (at  your  option)  any  --
--  later  version.   GNARL is distributed in the hope that it will be use- --
--  ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
--  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
--  eral Library Public License for more details.  You should have received --
--  a  copy of the GNU Library General Public License along with GNARL; see --
--  file COPYING. If not, write to the Free Software Foundation,  675  Mass --
--  Ave, Cambridge, MA 02139, USA.                                          --
--                                                                          --
------------------------------------------------------------------------------

with System.Task_IDs; use System.Task_IDs;
with System.Task_Entries; use System.Task_Entries;
with System.Task_Abortion; use System.Task_Abortion;
with System.Compiler_Exceptions;

package System.Task_Entry_Queue is

   type EQ_Protection_Access is new System.Address;
   EQ_Null_PO : constant EQ_Protection_Access :=
         EQ_Protection_Access (System.Null_Address);

   type EQ_Call_Modes is (Simple_Call, Conditional_Call, Asynchronous_Call);

   Priority_Not_Boosted : constant Integer := System.Priority'First - 1;
   subtype Rendezvous_Priority is Integer
         range Priority_Not_Boosted .. System.Priority'LAST;

   type Entry_Call_Record;
   type Entry_Call_Link is access Entry_Call_Record;

   type Server_Kind is (Task_Server, PO_Server);
   type Server_Record (Kind : Server_Kind := Task_Server) is record
      case Kind is

      when Task_Server =>
         Called_Task : Task_ID;
         Acceptor_Prev_Call : Entry_Call_Link;

         Acceptor_Prev_Priority : Rendezvous_Priority;
         --  For a task servicing a task entry call,
         --  information about the most recent prior call being serviced.
         --   Not used for protected entry calls;
         --  this function should be performed by GNULLI ceiling locking.

      when PO_Server =>
         Called_PO : EQ_Protection_Access;

      end case;
   end record;

   type Entry_Call_Record is record

      Next : Entry_Call_Link;

      Call_Claimed : Boolean;
      --  This flag is True if the call has been queued
      --  and subsequently claimed
      --  for service or cancellation.
      --  Protection : Test_And_Set/gloabal update or some similar mechanism
      --  (e.g. global mutex).
      --  Caution : on machines were we use Test_And_Set, we may not want this
      --  field packed.  For example, the SPARC atomic ldsub instruction
      --  effects a whole byte.

      Self : Task_ID;
      Level : ATC_Level;
      --  One of Self and Level are redundent in this implementation, since
      --  each Entry_Call_Record is at Self.Entry_Calls (Level).  Since we must
      --  have access to the entry call record to be reading this, we could
      --  get Self from Level, or Level from Self.  However, this requires
      --  non-portable address arithmetic.

      Mode : EQ_Call_Modes;
      Abortable : Boolean;

      Done : Boolean;
      --  Protection : Self.L.

      E : Entry_Index;

      Prio : System.Any_Priority;

      --  The above fields are those that there may be some hope of packing.
      --  They are gathered together to allow for compilers that lay records
      --  out contigously, to allow for such packing.

      Parameter : System.Address;

      Exception_To_Raise : Compiler_Exceptions.Exception_ID;
      --  The exception to raise once this call has been completed without
      --  being aborted.

      --  Server : Server_Record;
      --  Warning : VERDIX COMPILER BUG.
      --  The remaining componants in this record
      --  used to be contained in a variant Server_Record field.  One variant
      --  contained fields for task entry calls and the other for protected
      --  entry calls.  This entry call record is in turn a componant of an
      --  array which is used as a componant of the ATCB record.  This
      --  was too much for the Verdix compiler, which generated some very
      --  peculiar (and incorrect) code for accessing entry call records in
      --  the ATCB.  In any case, little or no space is saved, because space
      --  had to be allocated for all of the task fields plus the discriminant,
      --  whereas this way space has to be allocated for all of the task
      --  fields plus the called protected object field.

      Called_Task : Task_ID;
      --  For task entry calls only.

      Acceptor_Prev_Call : Entry_Call_Link;
      --  For task entry calls only.

      Acceptor_Prev_Priority : Rendezvous_Priority;
      --  For task entry calls only.
      --  The priority of the most recent prior call being serviced.
      --  For protected entry calls, this function should be performed by
      --  GNULLI ceiling locking.

      Called_PO : EQ_Protection_Access;
      --  For protected entry calls only.

   end record;

   type Entry_Queue is record
      Head : Entry_Call_Link;
      Tail : Entry_Call_Link;
   end record;

   procedure Enqueue (E : in out Entry_Queue; Call : Entry_Call_Link);
   --  enqueue Call at the end of entry_queue E

   procedure Dequeue (E : in out Entry_Queue; Call : Entry_Call_Link);
   --  dequeue Call from entry_queue E

   function Head (E : in Entry_Queue) return Entry_Call_Link;
   --  return the head of entry_queue E

   procedure Dequeue_Head (
         E : in out Entry_Queue;
         Call : out Entry_Call_Link);
   --  remove and return the head of entry_queue E

   function Onqueue (Call : Entry_Call_Link) return Boolean;
   --  return True if Call is on any entry_queue at all

   function Count_Waiting (E : in Entry_Queue) return Natural;
   --  return number of calls on the waiting queue of E

end System.Task_Entry_Queue;
