------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                             R T S _ C l o c k                            --
--                                                                          --
--                                  S p e c                                 --
--                                                                          --
--                             $Revision: 1.2 $                            --
--                                                                          --
--           Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                          --
--  GNARL is free software; you can redistribute it and/or modify it  under --
--  terms  of  the  GNU  Library General Public License as published by the --
--  Free Software Foundation; either version 2, or (at  your  option)  any  --
--  later  version.   GNARL is distributed in the hope that it will be use- --
--  ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
--  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
--  eral Library Public License for more details.  You should have received --
--  a  copy of the GNU Library General Public License along with GNARL; see --
--  file COPYING. If not, write to the Free Software Foundation,  675  Mass --
--  Ave, Cambridge, MA 02139, USA.                                          --
--                                                                          --
------------------------------------------------------------------------------

--  This package interfaces with the Ada RTS and defined the low-level
--  timer operations, based on POSIX.

with System.POSIX_RTE; use System.POSIX_RTE;

with Unchecked_Conversion;

package System.Task_Clock is

   type Stimespec_sec is new integer range integer'first + 1 .. integer'last;
   type Stimespec_nsec is new integer range 0 .. 10#1#E9 - 1;

   type Stime_t is new Integer range integer'first + 1 .. integer'last;
   type Fractional_Second is new POSIX_RTE.Fractional_Second;

   type Stimespec is record
      tv_sec : Stime_t;
      tv_nsec : Fractional_Second;
   end record;

   function Stimespec_to_timespec is new
         Unchecked_Conversion (Stimespec, POSIX_RTE.timespec);
   function timespec_to_Stimespec is new
         Unchecked_Conversion (POSIX_RTE.timespec, Stimespec);

   Stimespec_First : constant Stimespec :=
         Stimespec' (Stime_t'FIRST, Fractional_Second'FIRST);
   Stimespec_Last : constant Stimespec :=
         Stimespec' (Stime_t'LAST, Fractional_Second'FIRST);
   --  limit Stimespec_Last for symmetricity of Stimespec
   Stimespec_Zero : constant Stimespec :=
         Stimespec' (0, Fractional_Second'First);
   Stimespec_Unit : constant Stimespec :=
         Stimespec' (0, Fractional_Second'First + 1);
   Stimespec_Sec_Unit : constant Stimespec :=
         Stimespec' (Stime_t (1), Fractional_Second (0));
   Stimespec_Ticks : constant Stimespec :=
         Stimespec' (0, 1000000);
   --  Sun os 4.1 has clock resoultion of 10ms.

   --  This is dependent on the POSIX.4 implementation (POSIX_RTE).
   --  This type extends the unsigned POSIX.4 timespec type to a signed
   --  timespec type. The sign may only occur in the sec field. There
   --  should be no loss of accuracy since POSIX.4 requires the presence
   --  of the sign bit to report error conditions (-1).

   function Stimespec_Seconds (TV : Stimespec) return Stimespec_sec;

   function Stimespec_NSeconds (TV : Stimespec) return Stimespec_nsec;

   function Time_Of (S : Stimespec_sec; NS : Stimespec_nsec) return Stimespec;

   function Clock return Stimespec;

   --  These functions convert between duration and Stimespec.
   function To_Duration (TV : Stimespec) return Duration;

   function To_Stimespec (Time : Duration) return Stimespec;

   function LL_Less (LTV, RTV : Stimespec) return Boolean;

   function LL_U_Minus (TV : Stimespec) return Stimespec;

   function LL_Plus (LTV, RTV : Stimespec) return Stimespec;

   function LL_Minus (LTV, RTV : Stimespec) return Stimespec;

   function LL_Multiply (TV : Stimespec; N : integer) return Stimespec;

   --  According to ADA9x spec, we are following the rule of
   --  integer division in
   --  the following division operations. The error in division is therefore,
   --  when +/+ or -/- : 0 <= error < |divisor|
   --  when +/- or -/+ : -|divisor|< error <= 0
   function LL_Int_Divide (TV : Stimespec; N : integer) return Stimespec;

   function LL_Divide (LTV, RTV : Stimespec) return Stimespec;

   function "<" (LTV, RTV : Stimespec) return Boolean renames LL_Less;
   function "-" (TV : Stimespec) return Stimespec renames LL_U_Minus;
   function "+" (LTV, RTV : Stimespec) return Stimespec renames LL_Plus;
   function "-" (LTV, RTV : Stimespec) return Stimespec renames LL_Minus;
   function "*" (TV : Stimespec;
         N : integer) return Stimespec renames LL_Multiply;
   function "/" (TV : Stimespec; N : integer) return Stimespec
            renames LL_Int_Divide;
   function "/" (LTV, RTV : Stimespec) return Stimespec renames LL_Divide;

end System.Task_Clock;
