------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                             R T S _ C l o c k                            --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.2 $                            --
--                                                                          --
--           Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                          --
--  GNARL is free software; you can redistribute it and/or modify it  under --
--  terms  of  the  GNU  Library General Public License as published by the --
--  Free Software Foundation; either version 2, or (at  your  option)  any --
--  later  version.   GNARL is distributed in the hope that it will be use- --
--  ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
--  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
--  eral Library Public License for more details.  You should have received --
--  a  copy of the GNU Library General Public License along with GNARL; see --
--  file COPYING. If not, write to the Free Software Foundation,  675  Mass --
--  Ave, Cambridge, MA 02139, USA.                                          --
--                                                                          --
------------------------------------------------------------------------------

--  This package interfaces with the Ada RTS and defined the low-level
--  timer operations, based on POSIX.

with System.Std; use System.Std;

package body System.Task_Clock is

   --  Following definitions are used for multiple percision arithmetic
   --  operations in base 10000. Only positive numbers can be manipulated.

   Base : constant := 10000;
   Max_Digits : constant := 14;
   --  Total of 14 base 10000 digits used
   subtype Digit is integer range 0 .. Base - 1;
   type Digit_Range is range 0 .. Max_Digits - 1;
   type Number is array (Digit_Range) of Digit;

   --  Things  needed to represent Base 10000 numbers in base 10 numbers.
   Base_10_Digits : constant := 4;
   --  Number of base 10 number digits to represent Base 10000 Number
   Log_Of_Base : constant := 13;
   --  log base 2 of Base. closest smaller integer
   subtype Base_Shift_Range is integer range 0 .. Max_Digits * Base_10_Digits;
   --  Range of shifting Base 10000 numbers

   --  Some of the useful constants
   Number_Zero : constant Number :=
         (0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0);
   Unit_Number : constant Number :=
         (0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1);

   subtype Non_Negative is integer range 0 .. integer'last;
   Non_Negative_Bound : constant Digit_Range := 3;
   --  assuming integers are represented using 32bits,
   --  they can be represented using 3 base 10000 digits.

   --  These functions convert between duration and Stimespec.

   --  1) Nanoseconds may be too large to be converted into duration type.
   --  But if we convert into Float first, we may introduce rounding errors.
   --  We assume that duration is defined with a delta 0.001 to represent
   --  steps in milliseconds.
   --  2) Rounding of Nanosecond is assumed below the third decimal digits.
   --  For the rounding of exactly halfway value, rounding up is inforced.
   function To_Duration (TV : Stimespec) return Duration is
      Temp_TV : Stimespec;
   begin
      if TV.tv_sec >= 0 then
         return Duration (TV.tv_sec) + (Duration (TV.tv_nsec / 10#1#E5) +
                  Duration (5)) / 10#1#E4;
      else
         Temp_TV := -TV;
         return -Duration (Temp_TV.tv_sec) -
               (Duration (Temp_TV.tv_nsec / 10#1#E5) +
               Duration (5)) / 10#1#E4;
      end if;
      --  According to the Ada language specification, rounding the value
      --  around the half way between two consecutive integers can go
      --  either way.
      --  Here, however we are making sure that the middle value
      --  to be round up.
   end To_Duration;

   function To_Stimespec (Time : Duration) return Stimespec is
      sec :   Stime_t;
      nsec :  Fractional_Second;
   begin
      if Time <= 0.0 then
         sec := Stime_t (Time + 0.5);  --  to truncate the Time
      else
         sec := Stime_t (Time - 0.5);  --  to truncate the Time
      end if;
      --  According to the Ada language specification, rounding the value
      --  around the half way between two consecutive integers can go either
      --  way.
      --  Here, however we are making sure that the middle value to be round
      --  up.
      if (Time - duration (sec)) >= 1.0 then
         sec := sec + 1;
      elsif (Time - duration (sec)) <= -1.0 then
         sec := sec - 1;
      end if;
      --  Now sec has the integer part of Time

      if Time < 0.0 and then Time - Duration (sec) /= 0.0 then
         nsec := Fractional_Second ((1.0 + (Time - Duration (sec))) *
               10#1#E3) * 10#1#E6;
         sec := sec - 1;
      else
         nsec := Fractional_Second ((Time - Duration (sec)) * 10#1#E3)
               * 10#1#E6;
      end if;
      return Stimespec' (sec, nsec);
   end To_Stimespec;

   function Negative (T : Stimespec) return boolean is
   begin
      return (T.tv_sec < 0);
   end Negative;

   function Max (X, Y : Digit_Range) return Digit_Range is
   begin
      if X <= Y then
         return Y;
      end if;
      return X;
   end Max;

   --  returns the index of the  most significant digit
   function Most_Signif_Digit (N : Number) return Digit_Range is
   begin
      for i in Digit_Range loop
         if N (Digit_Range'last - i) /= 0 then
            return Digit_Range'last - i;
         end if;
      end loop;
      return 0;
      --  equals to Number_Zero;
   end Most_signif_Digit;

   --  Shift left for D number of base 10 digits.
   --  Filling number is 0. No overflow.
   function Shift_Left (N : Number; D : Base_Shift_Range) return Number is
      Result :        Number;
      Base_Bound :    Digit_Range := Digit_Range (D / Base_10_Digits);
      Base_10_Bound : Base_Shift_Range := D mod Base_10_Digits;
      Carry :         integer := 0;
      Temp_Int :      integer;
   begin
      if N = Number_Zero then
         return Number_Zero;
      end if;
      if D = 0 then
         return N;
      end if;
      for i in Base_Bound .. Digit_Range'last loop
      --  shift by base 10000 digit
         Result (i) := N (i - Base_Bound);
      end loop;
      if Base_10_Bound /= 0 then
         for i in Base_Bound .. Digit_Range'last loop
         --  shift by base 10 digit
            Temp_Int := Result (i) * (10 ** Base_10_Bound) + Carry;
            Carry := Temp_Int / Base;
            Result (i) := Temp_Int mod Base;
         end loop;
      end if;
      for i in 0 .. Base_Bound - 1 loop
         Result (i) := 0;
      end loop;
      return Result;
   end Shift_Left;

   --  Opposite of shift left. Filling number is 0. No overflow
   function Shift_Right (N : Number; D : Base_Shift_Range) return Number is
      Result :        Number;
      Base_Bound :    Digit_Range := Digit_Range (D / Base_10_Digits);
      Base_10_Bound : Base_Shift_Range := D mod Base_10_Digits;
      Carry :         integer := 0;
      Temp_Int :      integer;
   begin
      if N = Number_Zero then
         return Number_Zero;
      end if;
      if D = 0 then
         return N;
      end if;
      for i in 0 .. Digit_Range'last - Base_Bound loop
      --  shift by base 10000 digit
         Result (i) := N (i + Base_Bound);
      end loop;
      if Base_10_Bound /= 0 then
         for i in 0 .. Digit_Range'last - Base_Bound loop
         --  shift by base 10 digit
            Temp_Int :=
                  Result (Digit_Range'last - Base_Bound - i) + Carry * Base;
            Carry := Temp_Int mod 10 ** integer (Base_10_Bound);
            Result (Digit_Range'last - Base_Bound - i) :=
                     Temp_Int / 10 ** integer (Base_10_Bound);
         end loop;
      end if;
      for i in Digit_Range'last - Base_Bound + 1 .. Digit_Range'last loop
         Result (i) := 0;
      end loop;
      return Result;
   end Shift_Right;

   function "<" (LN, RN : Number) return boolean is
      L : Digit_Range := Most_Signif_Digit (LN);
      R : Digit_Range := Most_Signif_Digit (RN);
   begin
      if L < R then
         return true;
      elsif L > R then
         return false;
      end if;

      --  when Digits of LN = RN
      for i in 0 .. L loop
         if LN (L - i) < RN (L - i) then
            return true;
         elsif LN (L - i) > RN (L - i) then
            return false;
         end if;
      end loop;
      --  two mumbers are equal
      return false;
   end "<";

   function "+" (LN, RN : Number) return Number is
      Carry : Digit := 0;
      Temp_Digit : integer;
      Result :  Number := Number_Zero;
      L :          Digit_Range := Most_Signif_Digit (LN);
      R :          Digit_Range := Most_Signif_Digit (RN);
   begin
      for i in 0 .. Max (L, R) loop
         Temp_Digit := LN (i) + RN (i) + Carry;
         Result (i) := Temp_Digit mod Base;
         Carry := Temp_Digit / Base;
      end loop;
      if Carry > 0 then
         Result (Max (L, R) + 1) := Carry;
      end if;
      return Result;
   end "+";

   function "-" (LN, RN : Number) return Number is
      Borrower : Digit := 0;
      Temp_Digit : integer;
      Result : Number := Number_Zero;
      L :          Digit_Range := Most_Signif_Digit (LN);
      R :          Digit_Range := Most_Signif_Digit (RN);
   begin
      if LN < RN then
         raise numeric_error;
         --  negative number not allowed
      end if;
      for i in 0 .. L loop
         Temp_Digit := LN (i) - RN (i) - Borrower;
         if Temp_Digit < 0 then
            Result (i) := Base + Temp_Digit;
            Borrower := 1;
         else
            Result (i) := Temp_Digit;
            Borrower := 0;
         end if;
      end loop;
      return Result;
   end "-";

   --  Maximum of (Max_Digits / 2) Digits are assumed for mutiplication
   function "*" (LN, RN : Number) return Number is
      Carry :      Digit;
      Temp_Digit : integer;
      Result : Number := Number_Zero;
      L :          Digit_Range := Most_Signif_Digit (LN);
      R :          Digit_Range := Most_Signif_Digit (RN);
   begin
      for i in 0 .. R loop
         if RN (i) /= 0 then
            Carry := 0;
            for j in 0 .. L loop
               Temp_Digit := RN (i) * LN (j) + Result (j + i) + Carry;
               Carry := Temp_Digit / Base;
               Result (j + i) := Temp_Digit mod Base;
            end loop;
            Result (L + i + 1) := Carry;
         end if;
      end loop;
      return Result;
   end "*";

   --  Division function returns the quotient only.
   --  Algorithm taken from Charles J. Mifsud's multiple percision
   --  division algorithm. (Communications of the ACM, Vol.13, No.11,
   --  Nov. 1970)
   --  Adjustment steps are not implemented. Instead we check the
   --  correct quotient using iterations.
   function "/" (LN, RN : Number) return Number is
      Dividend :         Number := LN;
      Divisor :          Number := RN;
      Quotient :         Number := Number_Zero;
      Remainder :        Number := Number_Zero;
      Partial_Dividend : Number := Number_Zero;
      Temp_Number :      Number := Number_Zero;
      L :                Digit_Range := Most_Signif_Digit (LN);
      R :                Digit_Range := Most_Signif_Digit (RN);
      Temp_Int :         integer;
   begin
      if Divisor = Number_Zero then
         raise Numeric_Error;
      end if;
      --  step 0
      if Dividend < Divisor then
         return Number_Zero;
      end if;
      --  step 5
      for i in 0 .. R loop
         Partial_Dividend (i) := Dividend (L - R + i);
      end loop;
      Partial_Dividend (R + 1) := 0;
      --  step 6
      for j in 0 .. L - R loop
         Temp_Int := integer (Partial_Dividend (R + 1) * Base +
                  Partial_Dividend (R)) / integer (Divisor (R));
         if Temp_Int = Base then
         --  special case when the division result is equal to Base.
         --  Temp_Int can be equal to Base since we are dealing
         --  with only first and second digits of the numbers. For example,
         --  we can have 10000 / 1 situation, because the remaining digits
         --  for Partial_Dividend and Divisor determine the Remainder and
         --  Remainder in turn determines the Partial_Dividend.
         --  100000100  / 100001000 is an example for this kind of situation.
            Temp_Number (0) := Temp_Int - 1;
         else
            Temp_Number (0) := Temp_Int;
         end if;
         --  step 7
         while Partial_Dividend < (Divisor * Temp_Number) loop
         --  step 8
            Temp_Number (0) := Temp_Number (0) - 1;
         end loop;
         Quotient (L - R - j) := Temp_Number (0);
         Remainder := Partial_Dividend - (Divisor * Temp_Number);
         --  step 9
         exit when j > L - R - 1;
         Partial_Dividend := Shift_Left (Remainder, 4);
         Partial_Dividend (0) := Dividend (L - R - j - 1);
      end loop;
      return Quotient;
   end "/";

   function Non_Negative_To_Number (P : Non_Negative) return Number is
      Temp_N : Number := Number_Zero;
      Temp_P : Non_Negative := P;
   begin
      for i in 0 .. Non_Negative_Bound - 1 loop
         Temp_N (i) := Temp_P mod Base;
         Temp_P := Temp_P / Base;
      end loop;
      return Temp_N;
   end Non_Negative_To_Number;

   --  will raise exception when
   --  N >  Non_Negative_To_Number (Non_Negative'last)
   function Number_To_Non_Negative (N : Number) return Non_Negative is
      Result : Non_Negative := 0;
   begin
      for i in 0 .. Non_Negative_Bound - 1 loop
         Result := Result + N (i) * (Base ** integer (i));
      end loop;
      return Result;
   end Number_To_Non_Negative;

   --  Followings are the conversion functions between Stimespecs and
   --  multiple pecision numbers.

   --  fuction to build mutiple percision number out of Stimespec.
   --  Decimal point will  be ignored. Positive Stimespec is assumed
   function Stimespec_To_Base_Number (TV : Stimespec) return Number is
      Temp1 :     Number := Non_Negative_To_Number (Non_Negative (TV.tv_sec));
      Temp2 :     Number := Non_Negative_To_Number (Non_Negative (TV.tv_nsec));
   begin

      --  Adjust Temp1 and Temp2 to form a Base Number with no decimal point
      return Temp2 + Shift_Left (Temp1, 9);
   end Stimespec_To_Base_Number;

   --  function to build Stimespec from the number represented in
   --  mutilple percision. Decimal point position has to be specified
   function Base_Number_To_Stimespec (
         N : Number;
         D : Base_Shift_Range)
         return Stimespec is
      Temp_D : Base_Shift_Range := D;
      Temp_N : Number := N;
      Result : Stimespec;
   begin
      --  adjust the sparation point so that the conversion to
      --  non negative value does not raise overflow
      if D > 9 then
         Temp_D := 9;
         Temp_N := Shift_Right (Temp_N, D - Temp_D);
      end if;
      Result.tv_nsec := Fractional_Second (Number_To_Non_Negative (Temp_N -
            Shift_Left (Shift_Right (Temp_N, Temp_D), Temp_D)));
      Result.tv_sec := Stime_t (Number_To_Non_Negative (
            Shift_Right (Temp_N, Temp_D)));
      if Stimespec_Last < Result then
      --  forcing the boundary check
         raise constraint_error;
      end if;
      return Result;
   end Base_Number_To_Stimespec;

   function Abs_Stimespec (TV : Stimespec) return Stimespec is
   begin
      if Negative (TV) then
         return -TV;
      end if;
      return TV;
   end Abs_Stimespec;

   function LL_Less (LTV, RTV : Stimespec) return Boolean is
   begin
      return LTV.tv_sec < RTV.tv_sec or else
            (LTV.tv_sec = RTV.tv_sec and then LTV.tv_nsec < RTV.tv_nsec);
   end LL_Less;

   --  unary minus
   function LL_U_Minus (TV : Stimespec) return Stimespec is
      Result : Stimespec;
   begin
      if TV.tv_nsec = Fractional_Second'First then
         return (-TV.tv_sec, TV.tv_nsec);
      end if;
      Result := Stimespec' (-TV.tv_sec - 1,
            (Fractional_Second'LAST - TV.tv_nsec + 1));
      if Stimespec_Last < Result then
      --  forcing the boundary check
         raise constraint_error;
      end if;
      return Result;
   end LL_U_Minus;

   --  exception when LTV + RTV > Stimespec'LAST or LTV + RTV < Stimespec'FIRST
   --  assumes that Nanoseconds'LAST >= 2 * Fractional_Second'LAST
   function LL_Plus (LTV, RTV : Stimespec) return Stimespec is
      Out_Sec : Stime_t;
      Carry : Stime_t := 0;
      Out_Nsec : Nanoseconds;
      Result : Stimespec;
   begin
      Out_Nsec := Nanoseconds (LTV.tv_nsec) + Nanoseconds (RTV.tv_nsec);
      if Out_Nsec > Nanoseconds (Fractional_Second'LAST) then
         Out_Nsec := Out_Nsec - (Nanoseconds (Fractional_Second'LAST) + 1);
         Carry := 1;
      end if;

      --  To avoid unnecessary underflow or overflow.
      if LTV.tv_sec < 0 and then RTV.tv_sec < 0 then
         Out_Sec := LTV.tv_sec + Carry;
         Out_Sec := Out_Sec + RTV.tv_sec;
      else
         Out_Sec := LTV.tv_sec + RTV.tv_sec;
         Out_Sec := Out_Sec + Carry;
      end if;
      Result := Stimespec' (Out_Sec, Fractional_Second (Out_Nsec));
      if Stimespec_Last < Result then
      --  forcing the boundary check
         raise constraint_error;
      end if;
      return Result;
   end LL_Plus;

   function LL_Minus (LTV, RTV : Stimespec) return Stimespec is
   begin
      return (LTV + (-RTV));
   end LL_Minus;

   function LL_Multiply (TV : Stimespec; N : integer) return Stimespec is
      Negative_Result : boolean := Negative (TV) xor N < 0;
      Result : Stimespec;
      Temp_Num : Number;
   begin
      if N = integer'first then
      --  condition check for integer'First. There is no abs value for this.
         Temp_Num := Non_Negative_To_Number (abs (N + 1)) +
               Non_Negative_To_Number (1);
         --  Temp_Num has absolute value of N
         Result :=
               Base_Number_To_Stimespec (
               Stimespec_To_Base_Number (Abs_Stimespec (TV)) * Temp_Num, 9);
      else
         Result :=
               Base_Number_To_Stimespec (
               Stimespec_To_Base_Number (Abs_Stimespec (TV)) *
               Non_Negative_To_Number (abs (N)), 9);
      end if;
      if Stimespec_Last < Result then
      --  forcing the boundary check
         raise constraint_error;
      end if;
      if Negative_Result then
         return -Result;
      end if;
      return Result;
   end LL_Multiply;

   --  integer division of Stimespec
   function LL_Int_Divide (TV : Stimespec; N : integer) return Stimespec is
      Negative_Result : boolean := Negative (TV) xor N < 0;
      Result : Stimespec;
   Temp_Num : Number;
   begin
      if N = integer'first then
      --  condition check for integer'First. There is no abs value for this.
         Temp_Num := Non_Negative_To_Number (
               abs (N + 1)) + Non_Negative_To_Number (1);
         --  Temp_Num has absolute value of N
         Temp_Num := Stimespec_To_Base_Number (Abs_Stimespec (TV)) / Temp_Num;
         Result := Base_Number_To_Stimespec (Temp_Num, 9);
      else
         Result := Base_Number_To_Stimespec (
               Stimespec_To_Base_Number (Abs_Stimespec (TV)) /
               Non_Negative_To_Number (abs (N)), 9);
      end if;
      if Stimespec_Last < Result then
      --  forcing the boundary check
         raise constraint_error;
      end if;
      if Negative_Result then
         return -Result;
      end if;
      return Result;
   end LL_Int_Divide;

   function LL_Divide (LTV, RTV : Stimespec) return Stimespec is
      Negative_Result : boolean := Negative (LTV) xor Negative (RTV);
      Temp_Num : Number;
      Result : Stimespec;
   begin
      Temp_Num := Stimespec_To_Base_Number (Abs_Stimespec (LTV));
      Temp_Num :=
               Shift_Left (Temp_Num, 9) /
                     Stimespec_To_Base_Number (Abs_Stimespec (RTV));
      --  Base Number division returns quotient only. Therefore, in order to
      --  get the nanosecond part of the division shifting is necessary
      Result := Base_Number_To_Stimespec (Temp_Num, 9);
      if Stimespec_Last < Result then
      --  forcing the boundary check
         raise constraint_error;
      end if;
      if Negative_Result then
         return -Result;
      end if;
      return Result;
   end LL_Divide;

   function Stimespec_Seconds (TV : Stimespec) return Stimespec_sec is
   begin
      return Stimespec_sec (TV.tv_sec);
   end Stimespec_Seconds;

   function Stimespec_NSeconds (TV : Stimespec) return Stimespec_nsec is
   begin
      return Stimespec_nsec (TV.tv_nsec);
   end Stimespec_NSeconds;

   function Time_Of (S : Stimespec_sec;
         NS : Stimespec_nsec) return Stimespec is
   begin
      return Stimespec' (Stime_t (S), Fractional_Second (NS));
   end Time_Of;

   function Clock return Stimespec is
      Now : timespec;
      Result : Integer;
   begin
      clock_gettime (CLOCK_REALTIME, Now, Result);
      return timespec_to_Stimespec (Now);
   end Clock;

end System.Task_Clock;
