------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                             P t h r e a d s                              --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.2 $                            --
--                                                                          --
--           Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                          --
--  GNARL is free software; you can redistribute it and/or modify it  under --
--  terms  of  the  GNU  Library General Public License as published by the --
--  Free Software Foundation; either version 2, or (at  your  option)  any --
--  later  version.   GNARL is distributed in the hope that it will be use- --
--  ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
--  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
--  eral Library Public License for more details.  You should have received --
--  a  copy of the GNU Library General Public License along with GNARL; see --
--  file COPYING. If not, write to the Free Software Foundation,  675  Mass --
--  Ave, Cambridge, MA 02139, USA.                                          --
--                                                                          --
------------------------------------------------------------------------------

--  This package interfaces with Pthreads.  It is not a complete interface; it
--  only includes what is needed to implement the Ada runtime.

with Unchecked_Conversion;

package body System.Pthreads is

   -----------------------------------------------------------------------
   --  These unchecked conversion functions are used to convert a variable
   --  to an access value referencing that variable.  The expression
   --  Address_to_Pointer(X'Address) evaluates to an access value referencing
   --  X; if X is of type T, this expression returns a value of type
   --  access T.  This is necessary to allow structures to be passed to
   --  C functions, since some compiler interfaces to C only allows scalers,
   --  access values, and values of type System.Address as actual parameters.
   -----------------------------------------------------------------------
   function Address_to_Pointer is new
         Unchecked_Conversion (System.Address, POSIX_RTE.sigset_t_ptr);
   type pthread_t_ptr is access pthread_t;
   function Address_to_Pointer is new Unchecked_Conversion (System.Address,
         pthread_t_ptr);
   type pthread_attr_t_ptr is access pthread_attr_t;
   function Address_to_Pointer is new Unchecked_Conversion (System.Address,
         pthread_attr_t_ptr);
   type pthread_mutexattr_t_ptr is access pthread_mutexattr_t;
   function Address_to_Pointer is new Unchecked_Conversion (System.Address,
         pthread_mutexattr_t_ptr);
   type pthread_mutex_t_ptr is access pthread_mutex_t;
   function Address_to_Pointer is new Unchecked_Conversion (System.Address,
         pthread_mutex_t_ptr);
   type pthread_condattr_t_ptr is access pthread_condattr_t;
   function Address_to_Pointer is new Unchecked_Conversion (System.Address,
         pthread_condattr_t_ptr);
   type pthread_cond_t_ptr is access pthread_cond_t;
   function Address_to_Pointer is new Unchecked_Conversion (System.Address,
         pthread_cond_t_ptr);
   type pthread_key_t_ptr is access pthread_key_t;
   function Address_to_Pointer is new Unchecked_Conversion (System.Address,
         pthread_key_t_ptr);
   type Address_Pointer is access System.Address;
   function Address_to_Pointer is new Unchecked_Conversion (System.Address,
         Address_Pointer);
   type timespec_ptr is access POSIX_RTE.timespec;
   function Address_to_Pointer is new Unchecked_Conversion (System.Address,
         timespec_ptr);
   type integer_ptr is access Integer;
   function Address_to_Pointer is new Unchecked_Conversion (System.Address,
         integer_ptr);

   function pthread_attr_init_base (attr : pthread_attr_t_ptr)
         return Return_Code;
   pragma Interface (C, pthread_attr_init_base);
   pragma Interface_Name (pthread_attr_init_base,
         "pthread_attr_init");

   function pthread_attr_setstacksize_base (attr : pthread_attr_t_ptr;
        stacksize : size_t) return Return_Code;
   pragma Interface (C, pthread_attr_setstacksize_base);
   pragma Interface_Name (pthread_attr_setstacksize_base,
         "pthread_attr_setstacksize");

   function pthread_attr_setdetachstate_base (attr : pthread_attr_t_ptr;
         detachstate : integer_ptr) return Return_Code;
   pragma Interface (C, pthread_attr_setdetachstate_base);
   pragma Interface_Name (pthread_attr_setdetachstate_base,
         "pthread_attr_setdetachstate");

   function pthread_create_base (thread : pthread_t_ptr;
         attr : pthread_attr_t_ptr;
         start_routine : System.Address; arg : System.Address)
         return Return_Code;
   pragma Interface (C, pthread_create_base);
   pragma Interface_Name (pthread_create_base, "pthread_create");

   procedure pthread_init_base;
   pragma Interface (C, pthread_init_base);
   pragma Interface_Name (pthread_init_base, "pthread_init");

   function pthread_detach_base (thread : pthread_t_ptr) return Return_Code;
   pragma Interface (C, pthread_detach_base);
   pragma Interface_Name (pthread_detach_base, "pthread_detach");

   function pthread_mutexattr_init_base (attr : pthread_mutexattr_t_ptr)
         return Return_Code;
   pragma Interface (C, pthread_mutexattr_init_base);
   pragma Interface_Name (pthread_mutexattr_init_base,
         "pthread_mutexattr_init");

   function pthread_mutexattr_setprotocol_base (
         attributes : pthread_mutexattr_t_ptr;
         protocol : pthread_protocol_t) return Return_Code;
   pragma Interface (C, pthread_mutexattr_setprotocol_base);
   pragma Interface_Name (pthread_mutexattr_setprotocol_base,
         "pthread_mutexattr_setprotocol");

   function pthread_mutexattr_setprio_ceiling_base (
         attributes : pthread_mutexattr_t_ptr;
         prio_ceiling : Integer) return Return_Code;
   pragma Interface (C, pthread_mutexattr_setprio_ceiling_base);
   pragma Interface_Name (pthread_mutexattr_setprio_ceiling_base,
         "pthread_mutexattr_setprio_ceiling");

   function pthread_mutex_init_base (mutex : pthread_mutex_t_ptr;
         attr : pthread_mutexattr_t_ptr) return Return_Code;
   pragma Interface (C, pthread_mutex_init_base);
   pragma Interface_Name (pthread_mutex_init_base, "pthread_mutex_init");

   function pthread_mutex_destroy_base (mutex : pthread_mutex_t_ptr)
         return Return_Code;
   pragma Interface (C, pthread_mutex_destroy_base);
   pragma Interface_Name (pthread_mutex_destroy_base,
         "pthread_mutex_destroy");

   function pthread_mutex_trylock_base (mutex : pthread_mutex_t_ptr)
         return Return_Code;
   pragma Interface (C, pthread_mutex_trylock_base);
   pragma Interface_Name (pthread_mutex_trylock_base,
         "pthread_mutex_trylock");

   function pthread_mutex_lock_base (mutex : pthread_mutex_t_ptr)
         return Return_Code;
   pragma Interface (C, pthread_mutex_lock_base);
   pragma Interface_Name (pthread_mutex_lock_base, "pthread_mutex_lock");

   function pthread_mutex_unlock_base (mutex : pthread_mutex_t_ptr)
         return Return_Code;
   pragma Interface (C, pthread_mutex_unlock_base);
   pragma Interface_Name (pthread_mutex_unlock_base, "pthread_mutex_unlock");

   function pthread_cond_init_base (cond : pthread_cond_t_ptr;
         attr : pthread_condattr_t_ptr)
         return Return_Code;
   pragma Interface (C, pthread_cond_init_base);
   pragma Interface_Name (pthread_cond_init_base, "pthread_cond_init");

   function pthread_cond_wait_base (cond : pthread_cond_t_ptr;
         mutex : pthread_mutex_t_ptr)
         return Return_Code;
   pragma Interface (C, pthread_cond_wait_base);
   pragma Interface_Name (pthread_cond_wait_base, "pthread_cond_wait");

   function pthread_cond_timedwait_base (cond : pthread_cond_t_ptr;
         mutex : pthread_mutex_t_ptr;
         abstime : timespec_ptr)
         return Return_Code;
   pragma Interface (C, pthread_cond_timedwait_base);
   pragma Interface_Name (pthread_cond_timedwait_base,
         "pthread_cond_timedwait");

   function pthread_cond_signal_base (cond : pthread_cond_t_ptr)
         return Return_Code;
   pragma Interface (C, pthread_cond_signal_base);
   pragma Interface_Name (pthread_cond_signal_base, "pthread_cond_signal");

   function pthread_cond_broadcast_base (cond : pthread_cond_t_ptr)
         return Return_Code;
   pragma Interface (C, pthread_cond_broadcast_base);
   pragma Interface_Name (pthread_cond_broadcast_base,
         "pthread_cond_broadcast");

   function pthread_cond_destroy_base (cond : pthread_condattr_t_ptr)
         return Return_Code;
   pragma Interface (C, pthread_cond_destroy_base);
   pragma Interface_Name (pthread_cond_destroy_base, "pthread_cond_destroy");

   function pthread_condattr_init_base (cond : pthread_condattr_t_ptr)
         return Return_Code;
   pragma Interface (C, pthread_condattr_init_base);
   pragma Interface_Name (pthread_condattr_init_base,
         "pthread_condattr_init");

   function pthread_condattr_destroy_base (cond : pthread_condattr_t_ptr)
         return Return_Code;
   pragma Interface (C, pthread_condattr_destroy_base);
   pragma Interface_Name (pthread_condattr_destroy_base,
         "pthread_condattr_destroy");

   function pthread_setspecific_base (key : pthread_key_t;
         value : System.Address)
         return Return_Code;
   pragma Interface (C, pthread_setspecific_base);
   pragma Interface_Name (pthread_setspecific_base, "pthread_setspecific");

   function pthread_getspecific_base (key : pthread_key_t;
         value : Address_Pointer)
         return Return_Code;
   pragma Interface (C, pthread_getspecific_base);
   pragma Interface_Name (pthread_getspecific_base, "pthread_getspecific");

   function pthread_key_create_base (key : pthread_key_t_ptr;
         destructor : System.Address)
         return Return_Code;
   pragma Interface (C, pthread_key_create_base);
   pragma Interface_Name (pthread_key_create_base, "pthread_key_create");

   function pthread_attr_setprio_base (attr : pthread_attr_t_ptr;
         priority : Priority_Type) return Return_Code;
   pragma Interface (C, pthread_attr_setprio_base);
   pragma Interface_Name (pthread_attr_setprio_base,
         "pthread_attr_setprio");

   function pthread_attr_getprio_base (attr : pthread_attr_t_ptr)
         return Return_Code;
   pragma Interface (C, pthread_attr_getprio_base);
   pragma Interface_Name (pthread_attr_getprio_base,
         "pthread_attr_getprio");

   function pthread_setprio_base (thread : pthread_t;
               priority : Priority_Type) return Return_Code;
   pragma Interface (C, pthread_setprio_base);
   pragma Interface_Name (pthread_setprio_base,
         "pthread_setprio");

   function pthread_setschedattr_base (thread : pthread_t;
         attr : pthread_attr_t_ptr) return Return_Code;
   pragma Interface (C, pthread_setschedattr_base);
   pragma Interface_Name (pthread_setschedattr_base,
         "pthread_setschedattr");

   function pthread_getschedattr_base (thread : pthread_t;
         attr : pthread_attr_t_ptr) return Return_Code;
   pragma Interface (C, pthread_getschedattr_base);
   pragma Interface_Name (pthread_getschedattr_base,
         "pthread_getschedattr");

   function pthread_self_base return System.Address;
   pragma Interface (C, pthread_self_base);
   pragma Interface_Name (pthread_self_base, "pthread_self");

   function sigwait_base (set : POSIX_RTE.sigset_t_ptr) return Return_Code;
   pragma Interface (C, sigwait_base);
   pragma Interface_Name (sigwait_base, "sigwait");

   function pthread_kill_base (thread : pthread_t;
               sig : Signal_Number) return Return_Code;
   pragma Interface (C, pthread_kill_base);
   pragma Interface_Name (pthread_kill_base, "pthread_kill");

   ----------------------------------------------------------------------
   --  These subprograms are part of the interface to Pthreads exported by
   --  this package.
   ----------------------------------------------------------------------
   procedure pthread_attr_init (attributes : out pthread_attr_t;
         result : out Return_Code) is
   begin
      result :=
            pthread_attr_init_base (Address_to_Pointer (attributes'Address));
   end pthread_attr_init;

   procedure pthread_attr_setstacksize (attr : in out pthread_attr_t;
         stacksize : size_t; result : out Return_Code) is
   begin
      result :=
            pthread_attr_setstacksize_base (Address_to_Pointer (attr'Address),
            stacksize);
   end pthread_attr_setstacksize;

   procedure pthread_attr_setdetachstate (attr : in out pthread_attr_t;
         detachstate : Integer; result : out Return_Code) is
   begin
      Result := pthread_attr_setdetachstate_base (
            Address_to_Pointer (attr'Address),
            Address_to_Pointer (detachstate'Address));
   end pthread_attr_setdetachstate;

   procedure pthread_create (thread : out pthread_t;
         attributes : pthread_attr_t;
         start_routine : System.Address; arg : System.Address;
         result : out Return_Code) is
   begin
      result := pthread_create_base (Address_to_Pointer (thread'Address),
            Address_to_Pointer (attributes'Address), start_routine, arg);
   end pthread_create;

   --  This procedure provides a hook into Pthreads initialization that allows
   --  the addition of initializations speicific to the Ada interface to
   --  Pthreads.
   procedure pthread_init is
   begin
      pthread_init_base;
   end pthread_init;

   procedure pthread_detach (thread : in out pthread_t;
         result : out Return_Code) is
   begin
      result := pthread_detach_base (Address_to_Pointer (thread'Address));
   end pthread_detach;

   procedure pthread_mutexattr_init (attributes : out pthread_mutexattr_t;
         result : out Return_Code) is
   begin
      result :=
            pthread_mutexattr_init_base (Address_to_Pointer
                  (attributes'Address));
   end pthread_mutexattr_init;

   procedure pthread_mutexattr_setprotocol (
         attributes : in out pthread_mutexattr_t;
         protocol : pthread_protocol_t;
         result : out Return_Code) is
   begin
      result := pthread_mutexattr_setprotocol_base (
            Address_to_Pointer (attributes'Address), protocol);
   end pthread_mutexattr_setprotocol;

   procedure pthread_mutexattr_setprio_ceiling (
         attributes : in out pthread_mutexattr_t;
         prio_ceiling : Integer;
         result : out Return_Code) is
   begin
      result := pthread_mutexattr_setprio_ceiling_base (
            Address_to_Pointer (attributes'Address), prio_ceiling);
   end pthread_mutexattr_setprio_ceiling;

   procedure pthread_mutex_init (mutex : out pthread_mutex_t;
         attributes : pthread_mutexattr_t; result : out Return_Code) is
   begin
      result := pthread_mutex_init_base (Address_to_Pointer (mutex'Address),
            Address_to_Pointer (attributes'Address));
   end pthread_mutex_init;

   procedure pthread_mutex_destroy (mutex : in out pthread_mutex_t;
         result : out Return_Code) is
   begin
      result :=
            pthread_mutex_destroy_base (Address_to_Pointer (mutex'Address));
   end pthread_mutex_destroy;

   procedure pthread_mutex_trylock (mutex : in out pthread_mutex_t;
         result : out Return_Code) is
   begin
      result :=
            pthread_mutex_trylock_base (Address_to_Pointer (mutex'Address));
   end pthread_mutex_trylock;

   procedure pthread_mutex_lock (mutex : in out pthread_mutex_t;
         result : out Return_Code) is
   begin
      result := pthread_mutex_lock_base (Address_to_Pointer (mutex'Address));
   end pthread_mutex_lock;

   procedure pthread_mutex_unlock (mutex : in out pthread_mutex_t;
         result : out Return_Code) is
   begin
      result := pthread_mutex_unlock_base (Address_to_Pointer (mutex'Address));
   end pthread_mutex_unlock;

   procedure pthread_cond_init (condition : out pthread_cond_t;
         attributes : pthread_condattr_t;
         result : out Return_Code) is
   begin
      result := pthread_cond_init_base (Address_to_Pointer (condition'Address),
            Address_to_Pointer (attributes'Address));
   end pthread_cond_init;

   procedure pthread_cond_wait (condition : in out pthread_cond_t;
         mutex : in out pthread_mutex_t;
         result : out Return_Code) is
   begin
      result := pthread_cond_wait_base (Address_to_Pointer (condition'Address),
            Address_to_Pointer (mutex'Address));
   end pthread_cond_wait;

   procedure pthread_cond_timedwait (condition : in out pthread_cond_t;
         mutex : in out pthread_mutex_t;
         absolute_time : POSIX_RTE.timespec;
         result : out Return_Code) is
   begin
      result := pthread_cond_timedwait_base (
            Address_to_Pointer (condition'Address),
            Address_to_Pointer (mutex'Address),
            Address_to_Pointer (absolute_time'Address));
   end pthread_cond_timedwait;

   procedure pthread_cond_signal (condition : in out pthread_cond_t;
         result : out Return_Code) is
   begin
      result :=
            pthread_cond_signal_base (Address_to_Pointer (condition'Address));
   end pthread_cond_signal;

   procedure pthread_cond_broadcast (condition : in out pthread_cond_t;
         result : out Return_Code) is
   begin
      result := pthread_cond_broadcast_base (
            Address_to_Pointer (condition'Address));
   end pthread_cond_broadcast;

   procedure pthread_cond_destroy (condition : in out pthread_cond_t;
         result : out Return_Code) is
   begin
      result := pthread_cond_destroy_base (
            Address_to_Pointer (condition'Address));
   end pthread_cond_destroy;

   procedure pthread_condattr_init (attributes : out pthread_condattr_t;
         result : out Return_Code) is
   begin
      result := pthread_condattr_init_base (
            Address_to_Pointer (attributes'Address));
   end pthread_condattr_init;

   procedure pthread_condattr_destroy (attributes : in out pthread_condattr_t;
         result : out Return_Code) is
   begin
      result := pthread_condattr_destroy_base (
            Address_to_Pointer (attributes'Address));
   end pthread_condattr_destroy;

   --  suppress all checks to prevent stack check on entering routine
   procedure pthread_setspecific (key : pthread_key_t;
         value : System.Address;
         result : out Return_Code) is

   begin
      result := pthread_setspecific_base (key, value);
   end pthread_setspecific;

   procedure pthread_getspecific (key : pthread_key_t;
         value : out System.Address;
         result : out Return_Code) is
   begin
      result := pthread_getspecific_base (key,
            Address_to_Pointer (value'Address));
   end pthread_getspecific;

   procedure pthread_key_create (key : out pthread_key_t;
         destructor : System.Address;
         result : out Return_Code) is
   begin
      result := pthread_key_create_base (Address_to_Pointer (key'Address),
            destructor);
   end pthread_key_create;

   procedure pthread_attr_setprio (attr : in out pthread_attr_t;
         priority : Priority_Type; result : out Return_Code) is
   begin
      result := pthread_attr_setprio_base (Address_to_Pointer (attr'Address),
            priority);
   end pthread_attr_setprio;

   procedure pthread_attr_getprio (attr : pthread_attr_t;
         priority : out Priority_Type; result : out Return_Code) is
      Temp_Result : Return_Code;
   begin
      Temp_Result :=
            pthread_attr_getprio_base (Address_to_Pointer (attr'Address));
      if Temp_Result /= Failure then
         priority := Priority_Type (Temp_Result);
         result := 0;
      else priority := Priority_Type'FIRST;
      --  send out lowest priority, is it ok?
         result := Failure;
      end if;
   end pthread_attr_getprio;

   procedure pthread_setprio (thread : pthread_t;
         priority : Priority_Type; result : out Return_Code) is
   begin
      result := pthread_setprio_base (thread, priority);
   end pthread_setprio;

   --  procedure pthread_getprio (thread : pthread_t;
   --     priority : out Priority_Type; result : out Return_Code) is
   --  Temp_Result : Return_Code;
   --  begin
   --     Temp_Result := pthread_getprio_base (thread);
   --     if Temp_Result /= Failure
   --     then priority := Priority_Type (Temp_Result);
   --        result := 0;
   --     else priority := Priority_Type'FIRST;
   --     --  send out lowest priority, is ok?
   --        result := Failure;
   --     end if;
   --
   --  end pthread_getprio;

   procedure pthread_setschedattr (thread : pthread_t;
         attributes : pthread_attr_t; result : out Return_Code) is
   begin
      result := pthread_setschedattr_base (thread,
            Address_to_Pointer (attributes'Address));
   end pthread_setschedattr;

   procedure pthread_getschedattr (thread : pthread_t;
      attributes : out pthread_attr_t; result : out Return_Code) is
      Temp_Result : Return_Code;
   begin
      result := pthread_getschedattr_base (thread,
            Address_to_Pointer (attributes'Address));
   end pthread_getschedattr;

   function pthread_self return pthread_t is
   begin
      return pthread_t (pthread_self_base);
   end pthread_self;

   procedure sigwait (set : sigset_t;
         sig : out Signal_Number;
         result : out Return_Code) is
         Temp_Result : Return_Code;
   begin
      Temp_Result := sigwait_base (Address_to_Pointer (set'Address));
      if Temp_Result /= Failure then
         sig := Signal_Number (Temp_Result);
      else
         sig := 0;
      end if;
      result := Temp_Result;
   end sigwait;

   procedure pthread_kill (thread : pthread_t; sig : Signal_Number;
         result : out Return_Code) is
   begin
      result := pthread_kill_base (thread, sig);
   end pthread_kill;

end System.Pthreads;
