------------------------------------------------------------------------------
--                                                                          --
--                 GNU ADA RUNTIME LIBRARY (GNARL) COMPONENTS               --
--                                                                          --
--                    C o m p i l e r _ E x c e p t i o n s                 --
--                                                                          --
--                                  B o d y                                 --
--                                                                          --
--                             $Revision: 1.2 $                            --
--                                                                          --
--           Copyright (c) 1991,1992,1993, FSU, All Rights Reserved         --
--                                                                          --
--  GNARL is free software; you can redistribute it and/or modify it  under --
--  terms  of  the  GNU  Library General Public License as published by the --
--  Free Software Foundation; either version 2, or  (at  your  option)  any --
--  later  version.   GNARL is distributed in the hope that it will be use- --
--  ful, but but WITHOUT ANY WARRANTY; without even the implied warranty of --
--  MERCHANTABILITY  or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Gen- --
--  eral Library Public License for more details.  You should have received --
--  a  copy of the GNU Library General Public License along with GNARL; see --
--  file COPYING. If not, write to the Free Software Foundation,  675  Mass --
--  Ave, Cambridge, MA 02139, USA.                                          --
--                                                                          --
------------------------------------------------------------------------------

with System.Storage_Elements;

with System.Task_ATCB;
with System.Task_Stages;
with System.POSIX_RTE; use System.POSIX_RTE;
with System.Error_Reporting; use System.Error_Reporting;
with System.Machine_Specifics;

package body System.Compiler_Exceptions is

   SPARC_MAXREGWINDOW : constant := 31;

   --  for SparcStation

   type sc_spbuf_t is array (1 .. SPARC_MAXREGWINDOW) of System.Address;
   type sc_wbuf_t is array (1 .. SPARC_MAXREGWINDOW, 1 .. 16) of Integer;
   type sigcontext is record
      sc_onstack : Integer;  --  sigstack state to restore
      sc_mask : sigset_t;  --  signal mask to restore
      sc_sp : System.Address;  --  sp to restore
      sc_pc : System.Address;  --  pc to restore
      sc_npc : System.Address;  --  next pc to restore
      sc_psr : Integer;  --  psr to restore
      sc_g1 : Integer;  --  register that must be restored
      sc_o0 : Integer;
      sc_wbcnt : Integer;  --  number of outstanding windows
      sc_spbuf : sc_spbuf_t;  --  sp's for each wbuf, actually in C is char *
      sc_wbuf : sc_wbuf_t;  --  window save buf
   end record;

   type sigcontext_ptr is access sigcontext;
   for sigcontext_ptr'Size use 32;

--  The above operations will be available as predefined operations on
--  the modula Address type in GNARL, since this package is a child
--  of system.

   FPE_INTOVF_TRAP : constant Integer := 16#1#;    --  integer overflow
   FPE_STARTSIG_TRAP : constant Integer := 16#2#;  --  process using fp
   FPE_INTDIV_TRAP : constant Integer := 16#14#;   --  integer divide by zero
   FPE_FLTINEX_TRAP : constant Integer
                            := 16#c4#;   --  [floating inexact result]
   FPE_FLTDIV_TRAP : constant Integer
                            := 16#c8#;   --  [floating divide by zero]
   FPE_FLTUND_TRAP : constant Integer
                            := 16#cc#;   --  [floating underflow]
   FPE_FLTOPERR_TRAP : constant Integer
                            := 16#d0#;   --  [floating operand error]
   FPE_FLTOVF_TRAP : constant Integer
                            := 16#d4#;   --  [floating overflow]

   ILL_CHECK_TRAP : constant Integer :=  --  SIGILL generated by
         16#80# + 16#5#;                 --    trap 5 instruction.

   function Pre_Call_To_Context is new
         Unchecked_Conversion (
         Machine_Specifics.Pre_Call_State,
         sigcontext_ptr);

   procedure Raise_Exception (E : Exception_ID) is
   begin
      if E /= Null_Exception then

         Unimplemented_Operation;

      end if;
   end Raise_Exception;

   procedure Return_From_Handler (T : in out Task_ATCB.ATCB_Ptr) is
   --  The UNIX sigtramp () routine returns here rather than to the code that
   --  caused the error.  This routine does not actually take an arguement;
   --  this is just a trick to get a local variable.

   begin

      Unimplemented_Operation;

   end Return_from_Handler;

   procedure Notify_Exception (
         Which : Machine_Specifics.Machine_Exceptions;
         Info :  Machine_Specifics.Error_Information;
         Modified_Registers : Machine_Specifics.Pre_Call_State) is
      T : Task_ATCB.ATCB_Ptr := Task_ATCB.ID_To_ATCB (Task_Stages.Self);
      context : sigcontext_ptr :=
            Pre_Call_To_Context (Modified_Registers);
      sig : Signal_Number := Signal_Number (Which);
   begin

      --  Check for stack overflow.
      --  WARNING---Dubious code alert.
      --  This code is here  (I think) for the case where synchronous signals
      --  are handled on a separate stack.  Otherwise, the stack check should
      --  raise an exception before we get here.  I am not sure this is a good
      --  idea; since we shouldn't assume that the exception will propagate
      --  back to the user code, and might not want it to if it could.
      --  Ted Giering                     July 2, 1993
      --  Assert (Address_To_Integer (T.Stack_Limit) <
      --  Address_To_Integer (context.sc_sp),
      --  "Stack overflow in error handler");

      --  The following code updates Exception_Address and Exception_To_Raise
      --  in the ATCB.  I don't think this requires mutual exclusion; no other
      --  task should use Exception_Address, and Exception_To_Raise is only
      --  used to propagate an exception from an accept statement to its
      --  caller.
      --  In this case, the caller will have been waiting in the runtime code,
      --  which should also not get any synchronous errors.

      T.Exception_Address := context.sc_pc;

      case sig is

      when SIGFPE =>

         case Info.si_code is

            when FPE_INTDIV_TRAP | FPE_FLTINEX_TRAP |
                 FPE_FLTDIV_TRAP | FPE_FLTUND_TRAP  |
                 FPE_FLTOVF_TRAP =>
               T.Exception_To_Raise := Numeric_Error_ID;

            when FPE_FLTOPERR_TRAP =>
               T.Exception_To_Raise := Constraint_Error_ID;

            when FPE_INTOVF_TRAP =>
               T.Exception_To_Raise := Constraint_Error_ID;

         --  The SunOS signal delivery mechanism appears to provide a return
         --  address in context.sc_pc for this case, rather than the address
         --  of the instruction that caused the trap.
         --  The Verdix exception mechanism, on the other
         --  hand, appears to want the exact address of the instruction.  This
         --  caused a handler to be bypassed when an exception was raised in a
         --  return statement.
         --  Subtracting 4 is a kludge that I hope will
         --  cure this problem.
         --  According to the SunOS source code, this is also done for some
         --  cases of FPE_INTDIV_TRAP, but there does not appear to be any way
         --  for the Verdix compiler to generate this, so we should be safe.
               T.Exception_Address := T.Exception_Address - 4;

            when others =>

               Assert (false, "Unexpected SIGFPE signal");

         end case;

      when SIGILL =>
         case Info.si_code is

            when ILL_CHECK_TRAP =>
               T.Exception_To_Raise := Constraint_Error_ID;

            when others =>

               Assert (false, "Unexpected SIGILL signal");

         end case;
      when SIGSEGV =>
      --  If the address that caused the error was in the first page, this
      --  was caused by accessing a null pointer.

         if context.sc_o0 >= 0 and context.sc_o0 < 16#2000# then
            T.Exception_To_Raise := Constraint_Error_ID;

         else
            T.Exception_To_Raise := Storage_Error_ID;
         end if;

      when others =>

         Assert (false, "Unexpected signal");

      end case;

      context.sc_pc := Return_From_Handler'Address;
      context.sc_npc := Return_From_Handler'Address + 4;

   end Notify_Exception;

   function Current_Exception return Exception_ID is
   begin
      Unimplemented_Operation;
      return Null_Exception;
   end Current_Exception;

   function Image (E : Exception_ID) return Exception_ID_String is
   begin
      Unimplemented_Operation;
      return "Not Implemented*";
   end Image;

end System.Compiler_Exceptions;
