------------------------------------------------------------------------------
--                                                                          --
--                         GNAT RUNTIME COMPONENTS                          --
--                                                                          --
--                A D A . S T R I N G S . U N B O U N D E D                 --
--                                                                          --
--                                 S p e c                                  --
--                                                                          --
--                            $Revision: 1.3 $                              --
--                                                                          --
--           Copyright (c) 1992,1993,1994 NYU, All Rights Reserved          --
--                                                                          --
-- GNAT is free software;  you can  redistribute it  and/or modify it under --
-- terms of the  GNU General Public License as published  by the Free Soft- --
-- ware  Foundation;  either version 2,  or (at your option) any later ver- --
-- sion.  GNAT is distributed in the hope that it will be useful, but WITH- --
-- OUT ANY WARRANTY;  without even the  implied warranty of MERCHANTABILITY --
-- or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License --
-- for  more details.  You should have  received  a copy of the GNU General --
-- Public License  distributed with GNAT;  see file COPYING.  If not, write --
-- to the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA. --
--                                                                          --
------------------------------------------------------------------------------


with Ada.Strings.Maps;

package Ada.Strings.Unbounded is

   Blank : constant Character := ' ';

   type Unbounded_String is private;

   Null_Unbounded_String : constant Unbounded_String;

   type String_Access is access all String;

   ----------------------------
   -- String Length Function --
   ----------------------------

   function Length (Source : in Unbounded_String) return Natural;

   -----------------------------------------------------
   -- Conversion, Catenation, and Selection Functions --
   -----------------------------------------------------

   function To_Unbounded_String (Source : in String)
      return Unbounded_String;

   function To_String (Source : in Unbounded_String) return String;

   function "&" (Left, Right : in Unbounded_String)
      return Unbounded_String;

   function "&" (Left : in Unbounded_String; Right : String)
      return Unbounded_String;

   function "&" (Left : in String; Right : Unbounded_String)
      return Unbounded_String;

   function "&" (Left : in Unbounded_String; Right : Character)
      return Unbounded_String;

   function "&" (Left : in Character; Right : Unbounded_String)
      return Unbounded_String;

   function Element (Source : in Unbounded_String;
                      Index  : in Positive)
      return Character;

   procedure Replace_Element
     (Source : in out Unbounded_String;
      Index  : in Positive;
      By     : in Character);

   function Slice (Source : in Unbounded_String;
                   Low    : in Positive;
                   High   : in Natural)
      return String;

   function "="  (Left, Right : in Unbounded_String) return Boolean;

   function "<"  (Left, Right : in Unbounded_String) return Boolean;
   function "<=" (Left, Right : in Unbounded_String) return Boolean;
   function ">"  (Left, Right : in Unbounded_String) return Boolean;
   function ">=" (Left, Right : in Unbounded_String) return Boolean;

   ------------------------
   -- Search Subprograms --
   ------------------------

   function Index (Source   : in Unbounded_String;
                   Pattern  : in String;
                   Going    : in Direction := Forward;
                   Mapping  : in Maps.Character_Mapping
                                := Maps.Identity)
      return Natural;

   function Index (Source : in Unbounded_String;
                   Set    : in Maps.Character_Set;
                   Test   : in Membership := Inside;
                   Going  : in Direction  := Forward) return Natural;


   function Index_Non_Blank (Source : in Unbounded_String;
                             Going  : in Direction := Forward)
      return Natural;

   function Count (Source   : in Unbounded_String;
                   Pattern  : in String;
                   Mapping  : in Maps.Character_Mapping
                                := Maps.Identity)
      return Natural;

   function Count (Source   : in Unbounded_String;
                   Set      : in Maps.Character_Set)
      return Natural;


   procedure Find_Token (Source : in Unbounded_String;
                         Set    : in Maps.Character_Set;
                         Test   : in Membership;
                         First  : out Positive;
                         Last   : out Natural);

   ------------------------------------
   -- String Translation Subprograms --
   ------------------------------------

   function Translate (Source : in Unbounded_String;
                        Mapping  : in Maps.Character_Mapping)
      return Unbounded_String;

   procedure Translate (Source : in out Unbounded_String;
                         Mapping  : in Maps.Character_Mapping);

   ---------------------------------------
   -- String Transformation Subprograms --
   ---------------------------------------

   function Replace_Slice
      (Source   : in Unbounded_String;
       Low      : in Positive;
       High     : in Natural;
       By       : in String)
      return Unbounded_String;


   function Insert (Source   : in Unbounded_String;
                    Before   : in Positive;
                    New_Item : in String)
      return Unbounded_String;


   function Overwrite (Source    : in Unbounded_String;
                       Position  : in Positive;
                       New_Item  : in String)
      return Unbounded_String;


   function Delete (Source  : in Unbounded_String;
                    From    : in Positive;
                    Through : in Natural)
      return Unbounded_String;

   function Trim
      (Source : in Unbounded_String)
      return Unbounded_String;

   function Trim
      (Source : in Unbounded_String;
       Left   : in Maps.Character_Set;
       Right  : in Maps.Character_Set)
      return Unbounded_String;

   function Head (Source : in Unbounded_String;
                  Count  : in Natural;
                  Pad    : in Character := Blank)
      return Unbounded_String;

   function Tail (Source : in Unbounded_String;
                  Count  : in Natural;
                  Pad    : in Character := Blank)
      return Unbounded_String;

   function "*" (Left  : in Natural;
                 Right : in Character)
      return Unbounded_String;

   function "*" (Left  : in Natural;
                 Right : in String)
      return Unbounded_String;

   function "*" (Left  : in Natural;
                 Right : in Unbounded_String)
      return Unbounded_String;

private

   type Unbounded_String is record
      Reference : String_Access := new String'("");
   end record;

   Null_Unbounded_String : constant Unbounded_String :=
                               (Reference => new String'(""));

end Ada.Strings.Unbounded;
