/*
**                      Softwarepraktikum iMaze
**                              1993/94
**                Joerg Czeranski    Hans-Ulrich Kiel
**
** Datei: rechne3d.c
**
** Kommentar:
**  Berechnet aus den Labyrinth-Koordinaten der Spieler die aktuelle
**  3D-Sicht und legt die zu zeichnenden Grafik-Objekte in den
**  Datenstrukturen ab (in Bildpunkten)
*/


#include <stdio.h>
#include <memory.h>

#include "global.h"
#include "speicher.h"
#include "labyrinth.h"
#include "farben.h"
#include "spieler.h"
#include "grafik.h"
#include "rechne.h"

static char sccsid[] = "@(#)rechne3d.c	1.24 5/26/94";


/* Struct fuer temporaere Speicherung von Spielern und Schuessen */
struct objekt
{
	struct position pos; /* wie in struct spieler/struct schuss */
	s_char blick;        /* -1 fuer Schuesse, sonst wie in struct spieler */
	u_char farbe;        /* wie in struct spieler/struct schuss */
	int feldx, feldy;    /* Block im groben Raster,
	                        in dem das Objekt gemalt wird */
	u_char radius;       /* KUGELRAD oder SCHUSSRAD */
	int abstandq;        /* Quadrat des Abstandes von 'ich' */
};


/*
** projx
**  berechnet x-Koordinate der Zentralprojektion
**
** Parameter:
**  x: Abstand von 'ich' nach rechts
**  y: Abstand von 'ich' in Blickrichtung
**
** Rueckgabewert:
**  x-Koordinate in Bildpunkten (0 ... FENSTER_BREITE)
*/
static int projx(x, y)
int x, y;
{
	return x * (FENSTER_BREITE / 2 + 1) / y + FENSTER_BREITE / 2;
}


/*
** projy
**  berechnet y-Koordinate der Zentralprojektion
**
** Parameter:
**  y: Abstand von 'ich' in Blickrichtung
**  z: Abstand von 'ich' nach oben geteilt durch WANDH/2
**
** Rueckgabewert:
**  y-Koordinate in Bildpunkten (0 ... FENSTER_HOEHE)
*/
static int projy(y, z)
int y, z;
{
	return -z * (WANDH / 2) * (FENSTER_HOEHE / 2 + 1) / y +
		FENSTER_HOEHE / 2;
}


/*
** drehx
**  berechnet x-Koordinate des um den Winkel w gedrehten Punktes (x,y)
**
** Parameter:
**  x: alte x-Koordinate
**  y: alte y-Koordinate
**  w: Drehwinkel in WINKANZ Schritten pro Vollkreis
**
** Rueckgabewert:
**  neue x-Koordinate
*/
static int drehx(x, y, w)
int x, y;
u_char w;
{
	return (x * costab[(int)w * (TRIGANZ / WINKANZ)] +
		y * sintab[(int)w * (TRIGANZ / WINKANZ)]) / TRIGFAKTOR;
}


/*
** drehy
**  berechnet y-Koordinate des um den Winkel w gedrehten Punktes (x,y)
**
** Parameter:
**  x: alte x-Koordinate
**  y: alte y-Koordinate
**  w: Drehwinkel in WINKANZ Schritten pro Vollkreis
**
** Rueckgabewert:
**  neue y-Koordinate
*/
static int drehy(x, y, w)
int x, y;
u_char w;
{
	return (y * costab[(int)w * (TRIGANZ / WINKANZ)] -
		x * sintab[(int)w * (TRIGANZ / WINKANZ)]) / TRIGFAKTOR;
}


/*
** drehblick
**  berechnet die relative Blickrichtung einer Kugel
**
** Parameter:
**  kugelx: relative x-Koordinate der Kugel
**  kugely: relative y-Koordinate der Kugel
**  kugelblick: absolute Blickrichtung der Kugel
**              oder -1, falls Kugel kein Gesicht hat
**  ichblick: absolute Blickrichtung des Spielers
**
** Rueckgabewert:
**  relative Blickrichtung der Kugel oder -1
*/
static int drehblick(kugelx, kugely, kugelblick, ichblick)
int kugelx, kugely, kugelblick, ichblick;
{
	int x;         /* Abstand der Kugel von 'ich' nach rechts */
	int y;         /* Abstand der Kugel von 'ich' in Blickrichtung */
	int korrektur; /* Korrekturwinkel fuer Quadranten */

	/* Position der Kugel drehen */
	x = drehx(kugelx, kugely, ichblick);
	y = drehy(kugelx, kugely, ichblick);

	/* hat Kugel kein Gesicht oder steht er auf demselben Punkt? */
	if (kugelblick < 0 || (x == 0 && y == 0))
		return -1;

	/* Position um TRIGANZ/4 drehen, bis y > abs(x) */
	korrektur = 0;
	if (y < x)
	{
		korrektur = TRIGANZ / 2;
		x = -x;
		y = -y;
	}
	if (y < -x)
	{
		int tmp;

		korrektur += 3 * TRIGANZ / 4;
		tmp = x;
		x = -y;
		y = tmp;
	}

	/* Blickrichtung berechnen */
	return (atantab[x * (TRIGANZ / 2) / y + TRIGANZ / 2] + korrektur +
		(kugelblick - ichblick) * (TRIGANZ / WINKANZ) + TRIGANZ) % TRIGANZ;
}


/*
** ausschnitt_eintragen
**  einen weiteren sichtbaren Ausschnitt einer Kugel in die
**  Clipping-Liste eintragen
**
** Parameter:
**  sichtanz: Zeiger auf die Anzahl der sichtbaren Ausschnitte
**  sichtbar: Zeiger auf die Liste der sichtbaren Ausschnitte
**  von: x-Koordinate des linken Randes des Ausschnitts in Bildpunkten
**  bis: x-Koordinate des rechten Randes des Ausschnitts in Bildpunkten
**
** Seiteneffekte:
**  *sichtanz wird erhoeht, *sichtbar um ein Element verlaengert
*/
static void ausschnitt_eintragen(sichtanz, sichtbar, von, bis)
int *sichtanz, von, bis;
struct ausschnitt **sichtbar;
{
	liste_verlaengern(sichtanz, (void **)sichtbar,
		sizeof(struct ausschnitt));

	(*sichtbar)[*sichtanz - 1].x = von;
	(*sichtbar)[*sichtanz - 1].breite = bis - von;
}


/*
** rechnekugel
**  eine weitere Kugel projezieren, Clipping-Liste berechnen und,
**  sofern sichtbar, an die Kugel-Liste anhaengen
**
** Parameter:
**  anzwaende: Anzahl der sichtbaren Waende
**  waende: Liste der sichtbaren Waende (fuer Clipping)
**  anzkugeln: Zeiger auf die Anzahl der sichtbaren Kugeln
**  kugeln: Zeiger auf die Liste der sichtbaren Kugeln
**  x: Abstand der Kugel von 'ich' nach rechts
**  y: Abstand der Kugel von 'ich' in Blickrichtung
**  blick: Blickrichtung der Kugel relativ zur Blickrichtung von 'ich'
**  farbe: Farbe der Kugel
**  radius: Radius der Kugel im feinen Raster
**
** Seiteneffekte:
**  *anzkugeln wird erhoeht, *kugeln um ein Element verlaengert
*/
static void rechnekugel(anzwaende, waende, anzkugeln,
	kugeln, x, y, blick, farbe, radius)
struct flaeche *waende;
struct kugel **kugeln;
int anzwaende, *anzkugeln, x, y;
int blick;
u_char farbe, radius;
{
	struct ausschnitt *sichtbar; /* Clipping-Liste */
	int sichtanz;                /* Laenge der Clipping-Liste */
	int mittelpunktx;            /* x-Koordinate des Kugelmittelpunkt in
									Bildpunkten */
	int radiusx;                 /* Kugelradius in x-Richtung
									in Bildpunkten */

	/* falls Kugel hinter 'ich', nicht zeichnen */
	if (y <= 0)
		return;

	/* falls Kugel nicht im 90-Grad-Blickfeld, nicht zeichnen
	   sqrt(2) und 557 / 408 sind beide 1.41421... */
	if (x - y > (int)radius * 557 / 408 || x + y < -(int)radius * 557 / 408)
		return;

	mittelpunktx = projx(x, y);
	radiusx = projx(radius, y) - projx(0, y);

	liste_initialisieren(&sichtanz, (void **)&sichtbar);

	{
		int links; /* linker Rand des aktuellen sichtbaren Auschnitts */
		int i;     /* Index fuer Waende */

		/* ganz links anfangen */
		links = 0;

		/* lineare Suche, koennte man durch
		   vorherige binaere Suche optimieren */
		for (i = 0; i < anzwaende &&
			links <= min(mittelpunktx + radiusx, FENSTER_BREITE - 1); i++)
		{
			int rechts; /* rechter Rand des sichtbaren Auschnitts */

			rechts = min(waende[i].ecke[0].x, FENSTER_BREITE);

			/* Ausschnitte, die ganz links der Kugel liegen,
			   nicht eintragen */
			if (waende[i].ecke[0].x > links &&
				rechts >= mittelpunktx - radiusx)
				ausschnitt_eintragen(&sichtanz, &sichtbar, links, rechts);

			/* ab neuem 'links' weitersuchen */
			links = max(waende[i].ecke[2].x, links);
		}

		/* letzten Ausschnitt eintragen */
		if (FENSTER_BREITE > links && links <= mittelpunktx + radiusx)
			ausschnitt_eintragen(&sichtanz, &sichtbar,
				links, FENSTER_BREITE);
	}

	/* falls kein sichtbarer Bereich, Kugel nicht eintragen */
	if (sichtanz == 0)
	{
		liste_freigeben(&sichtanz, (void **)&sichtbar);
		return;
	}

	liste_verlaengern(anzkugeln, (void **)kugeln, sizeof(struct kugel));

	{
		struct kugel *kugel; /* Zeiger auf die neue Kugel in *kugeln */

		kugel = &(*kugeln)[*anzkugeln - 1];

		if (farbe == TRANSPARENT || y <= (int)radius)
			/* transparente Kugeln haben keinen Schatten: */
			kugel->schatteny = kugel->schattenry = 0;
		else
		{
			int schatten_oben, schatten_unten;

			/* Schatten projezieren
			   (Pfusch, in Wirklichkeit keine Ellipse) */
			schatten_oben = projy(y + radius, -1);
			schatten_unten = projy(y - radius, -1);
			kugel->schattenry = (schatten_unten - schatten_oben) / 2;
			kugel->schatteny = schatten_unten - kugel->schattenry;
		}

		kugel->mittelpunkt.x = mittelpunktx;
		kugel->mittelpunkt.y = projy(y, 0);
		kugel->radiusx = radiusx;
		/* Breiten-/Hoehen-Verhaeltnis des Fensters muss 3/2 sein,
		   damit die Kugeln rund werden */
		kugel->radiusy = radiusx * 3 / 2;
		kugel->blick = blick;
		kugel->farbe = farbe;
		kugel->sichtbar = sichtbar;
		kugel->sichtanz = sichtanz;
	}
}


/*
** rechnewand
**  eine weitere Wand projezieren und, sofern sichtbar,
**  in die sortierte Wand-Liste eintragen
**
** Parameter:
**  anzwaende: Zeiger auf die Anzahl der sichtbaren Waende
**  waende: Zeiger auf die Liste der sichtbaren Waende
**  x1: Abstand des einen Wandendes von 'ich' nach rechts
**  y1: Abstand des einen Wandendes von 'ich' in Blickrichtung
**  x2: Abstand des anderen Wandendes von 'ich' nach rechts
**  y2: Abstand des anderen Wandendes von 'ich' in Blickrichtung
**  farbe: Farbe der Wand
**  tuer: Flag, ob Wand eine Tuer ist
**
** Seiteneffekte:
**  *anzwaende wird erhoeht, *waende um ein Element verlaengert
*/
static void rechnewand(anzwaende, waende, x1, y1, x2, y2, farbe, tuer)
struct flaeche **waende;
int *anzwaende, x1, y1, x2, y2;
u_char farbe, tuer;
{
	struct flaeche wand; /* gesamte Wand, projeziert */

	if (farbe == TRANSPARENT)
		return;

	/* falls Wand rechts des 90-Grad-Blickfeldes, nicht zeichnen */
	if (x1 >= y1 && x2 >= y2)
		return;

	/* falls Wand rechte Grenze des Blickfeldes schneidet, abschneiden */
	if (x1 >= y1 || x2 >= y2)
	{
		int m, e, d;

		e = (x1 * y2 - x2 * y1);
		d = (x1 - x2 + y2 - y1);
		m = (e + d / 2) / d;

		/* erstes oder zweites Wandende veraendern? */
		if (x1 >= y1)
			x1 = y1 = m;
		else
			x2 = y2 = m;
	}

	/* falls Wand links des 90-Grad-Blickfeldes, nicht zeichnen */
	if (x1 <= -y1 && x2 <= -y2)
		return;

	/* falls Wand linke Grenze des Blickfeldes schneidet, abschneiden */
	if (x1 <= -y1 || x2 <= -y2)
	{
		int m, e, d;

		e = (x1 * y2 - x2 * y1);
		d = (x1 - x2 + y1 - y2);
		m = (e + d / 2) / d;

		/* erstes oder zweites Wandende veraendern? */
		if (x1 <= -y1)
			x1 = -m, y1 = m;
		else
			x2 = -m, y2 = m;
	}

	/* falls Wand hinter oder auf Bildebene, korrigieren */
	if (y1 <= 0)
		y1 = 1;
	if (y2 <= 0)
		y2 = 1;

	/* projezieren */
	wand.ecke[0].x = wand.ecke[1].x = projx(x1, y1);
	wand.ecke[0].y = projy(y1, 1);
	wand.ecke[1].y = projy(y1, -1);
	wand.ecke[2].x = wand.ecke[3].x = projx(x2, y2);
	wand.ecke[2].y = projy(y2, -1);
	wand.ecke[3].y = projy(y2, 1);
	wand.farbe = farbe;
	wand.tuer = tuer;

	/* ecke[0] und ecke[1] links, ecke[2] und ecke[3] rechts,
	   sonst tauschen */
	if (wand.ecke[0].x > wand.ecke[2].x)
	{
		int tmp;

		tmp = wand.ecke[0].x;
		wand.ecke[0].x = wand.ecke[1].x = wand.ecke[2].x;
		wand.ecke[2].x = wand.ecke[3].x = tmp;
		tmp = wand.ecke[0].y;
		wand.ecke[0].y = wand.ecke[3].y;
		wand.ecke[3].y = tmp;
		tmp = wand.ecke[1].y;
		wand.ecke[1].y = wand.ecke[2].y;
		wand.ecke[2].y = tmp;
	}

	/* alle sichtbaren Ausschnitte suchen, bei spaeteren Durchlaeufen
	   werden die frueheren Ausschnitte durch sich selbst verdeckt */
	for (;;)
	{
		int i;                     /* Index der Wand, vor der
								      eingefuegt werden muss */
		struct flaeche ausschnitt; /* ein sichtbarer Ausschnitt der Wand */

		/* lineare Suche, koennte durch binaere Suche optimiert werden,
		   gesucht wird die erste Wand, die weiter rechts
		   anfaengt als 'wand' */
		for (i = 0; i < *anzwaende && (*waende)[i].ecke[0].x <=
			wand.ecke[0].x; i++);

		{
			int links, rechts; /* die Grenzen des sichtbaren Ausschnitts
								  in Bildpunkten */

			if (i == 0)
				/* falls links von allen Waenden */
				links = wand.ecke[0].x;
			else
			{
				/* erste Luecke in den naeher liegenden Waenden suchen,
				   durch die man 'wand' sehen oder rechts an 'wand'
				   vorbeisehen kann */
				while (i < *anzwaende &&
					(*waende)[i - 1].ecke[2].x == (*waende)[i].ecke[0].x)
					i++;

				/* Luecke ist rechts von 'wand', kein weiterer
				   sichtbarer Ausschnitt */
				if ((*waende)[i - 1].ecke[2].x >= wand.ecke[2].x)
					/* hier verlassen, wer findet schon diese Zeile? */
					return;

				/* Anfang der Luecke */
				links = max(wand.ecke[0].x, (*waende)[i - 1].ecke[2].x);
			}

			if (i >= *anzwaende)
				/* falls rechts von allen Waenden */
				rechts = wand.ecke[2].x;
			else
				rechts = min(wand.ecke[2].x, (*waende)[i].ecke[0].x);

			/* y-Koordinaten und Farbe kopieren */
			memcpy(&ausschnitt, &wand, sizeof(struct flaeche));

			/* x-Koordinaten */
			ausschnitt.ecke[0].x = ausschnitt.ecke[1].x = links;
			ausschnitt.ecke[2].x = ausschnitt.ecke[3].x = rechts;

			/* falls 'wand' nicht nur ein Strich, y-Koordinaten anpassen */
			if (wand.ecke[0].x != wand.ecke[2].x)
			{
				/* linke y-Koordinaten anpassen */
				if (links > wand.ecke[0].x)
				{
					ausschnitt.ecke[0].y =
						(links - wand.ecke[0].x) *
						(wand.ecke[3].y - wand.ecke[0].y) /
						(wand.ecke[2].x - wand.ecke[0].x) + wand.ecke[0].y;
					ausschnitt.ecke[1].y =
						(links - wand.ecke[0].x) *
						(wand.ecke[2].y - wand.ecke[1].y) /
						(wand.ecke[2].x - wand.ecke[0].x) + wand.ecke[1].y;
				}

				/* rechte y-Koordinaten anpassen */
				if (rechts < wand.ecke[2].x)
				{
					ausschnitt.ecke[2].y =
						(rechts - wand.ecke[0].x) *
						(wand.ecke[2].y - wand.ecke[1].y) /
						(wand.ecke[2].x - wand.ecke[0].x) + wand.ecke[1].y;
					ausschnitt.ecke[3].y =
						(rechts - wand.ecke[0].x) *
						(wand.ecke[3].y - wand.ecke[0].y) /
						(wand.ecke[2].x - wand.ecke[0].x) + wand.ecke[0].y;
				}
			}
		}

		liste_verlaengern(anzwaende, (void **)waende,
			sizeof(struct flaeche));

		{
			int j;

			/* weiter rechts liegende Waende in *waende verschieben */
			for (j = *anzwaende - 1; j > i; j--)
				memcpy(&(*waende)[j], &(*waende)[j - 1],
					sizeof(struct flaeche));
		}

		/* Ausschnitt in *waende kopieren */
		memcpy(&(*waende)[i], &ausschnitt, sizeof(struct flaeche));
	}
}


/*
** objekt_cmp
**  vergleicht zwei Kugelobjekte nach feldx/y,
**  welches bei Blick in Nordrichtung zuerst einsortiert werden muss
**  (wird fuer qsort verwendet)
**
** Parameter:
**  objekt1: Zeiger auf ein Objekt
**  objekt2: Zeiger auf anderes Objekt
**
** Rueckgabewert:
**  < 0, falls Objekt 1 vor Objekt 2
**  = 0, falls egal
**  > 0, falls Objekt 1 nach Objekt 2
*/
static int objekt_cmp(objekt1, objekt2)
struct objekt *objekt1, *objekt2;
{
	int d1, d2;

	if (objekt1->feldy > 0 || objekt2->feldy > 0)
		return objekt2->feldy - objekt1->feldy;

	d1 = max(abs(objekt1->feldx), abs(objekt1->feldy));
	d2 = max(abs(objekt2->feldx), abs(objekt2->feldy));

	if (d1 != d2)
		return d1 - d2;

	if (objekt1->feldx == objekt2->feldx)
		if (objekt1->feldy == objekt2->feldy)
			return objekt1->abstandq - objekt2->abstandq;
		else
			return objekt2->feldy - objekt1->feldy;

	if (objekt1->feldy > -d1 && objekt2->feldy > -d2)
		return objekt1->feldx - objekt2->feldx;

	if (objekt1->feldy > -d1 || objekt2->feldy > -d2)
		return objekt2->feldy - objekt1->feldy;

	if (objekt1->feldx == objekt2->feldx)
		return objekt1->abstandq - objekt2->abstandq;

	if (objekt1->feldx > 0)
		return objekt1->feldx - objekt2->feldx;

	if (objekt2->feldx > 0)
		return -1;

	return objekt2->feldx - objekt1->feldx;
}


/*
** einnorden
**  dreht eine Position um 90 Grad * himricht nach rechts
**
** Parameter:
**  neu: Zeiger auf die neue Position
**  alt: Zeiger auf die alte Position
**  himricht: Winkel um den gedreht werden soll
**            (grobe Blickrichtung von 'ich')
**  feldbreite: Breite des Spielfeldes
**  feldlaenge: Laenge des Spielfeldes
**
** Seiteneffekte:
**  *neu wird auf die neue Position gesetzt
*/
static void einnorden(neu, alt, himricht, feldbreite, feldlaenge)
struct position *neu, *alt;
u_char himricht;
int feldbreite, feldlaenge;
{
	int xalt, yalt; /* alte Koordinaten im feinen Raster*/
	int xneu, yneu; /* neue Koordinaten im feinen Raster*/

	xalt = XPOS(*alt);
	yalt = YPOS(*alt);

	switch (himricht)
	{
		case 0: /* NORD */
			xneu = xalt;
			yneu = yalt;
			break;

		case 1: /* WEST */
			xneu = RASPT * feldlaenge - yalt;
			yneu = xalt;
			break;

		case 2: /* SUED */
			xneu = RASPT * feldbreite - xalt;
			yneu = RASPT * feldlaenge - yalt;
			break;

		case 3: /* OST */
			xneu = yalt;
			yneu = RASPT * feldbreite - xalt;
			break;
	}

	SETX(*neu, xneu);
	SETY(*neu, yneu);
}


/*
** tueren_offen_waagerecht
**  markiert ein West-/Ost-Tuerenpaar (oder Wandpaar) als offen
**
** Parameter:
**  anzoffen: Zeiger auf die Anzahl der offenen Tueren
**  offen: Zeiger auf die Liste der offenen Tueren
**  feld: Spielfeld in aktuelle Nordrichtung gedreht
**  feldbreite: Breite des Spielfeldes in Bloecken
**  feldlaenge: Laenge des Spielfeldes in Bloecken
**  x: x-Koordinate der zu markierenden West-Tuer
**  y: y-Koordinate der zu markierenden Tueren
**
** Seiteneffekte:
**  *anzoffen wird erhoeht, *offen um ein Element verlaengert,
**  in feld werden die tueroffen-Flags gesetzt
*/
static void tueren_offen_waagerecht(anzoffen, offen, feld,
	feldbreite, feldlaenge, x, y)
int *anzoffen, feldbreite, feldlaenge, x, y;
u_char (**offen)[3];
block **feld;
{
	/* falls Koordinaten ausserhalb des Spielfeldes oder alle Tueren/Waende
	   bereits TRANSPARENT, nichts tun */
	if (x < 0 || x > feldbreite || y < 0 || y >= feldlaenge ||
		(x == 0 || feld[x - 1][y][OST].farbe == TRANSPARENT) &&
		(x == feldbreite || feld[x][y][WEST].farbe == TRANSPARENT))
		return;

	if (x > 0)
	{
		/* Osttuer (linke Wand) in Liste eintragen */
		liste_verlaengern(anzoffen, (void **)offen, sizeof(u_char [3]));
		(*offen)[*anzoffen - 1][0] = x - 1;
		(*offen)[*anzoffen - 1][1] = y;
		(*offen)[*anzoffen - 1][2] = OST;

		/* Tuer im Spielfeld markieren */
		feld[x - 1][y][OST].tueroffen = 1;
	}

	if (x < feldbreite)
	{
		/* Westtuer (rechte Wand) in Liste eintragen */
		liste_verlaengern(anzoffen, (void **)offen, sizeof(u_char [3]));
		(*offen)[*anzoffen - 1][0] = x;
		(*offen)[*anzoffen - 1][1] = y;
		(*offen)[*anzoffen - 1][2] = WEST;

		/* Tuer im Spielfeld markieren */
		feld[x][y][WEST].tueroffen = 1;
	}
}


/*
** tueren_offen_senkrecht
**  markiert ein Nord-/Sued-Tuerenpaar (oder Wandpaar) als offen
**
** Parameter:
**  anzoffen: Zeiger auf die Anzahl der offenen Tueren
**  offen: Zeiger auf die Liste der offenen Tueren
**  feld: Spielfeld in aktuelle Nordrichtung gedreht
**  feldbreite: Breite des Spielfeldes in Bloecken
**  feldlaenge: Laenge des Spielfeldes in Bloecken
**  x: x-Koordinate der zu markierenden Tueren
**  y: y-Koordinate der zu markierenden Nord-Tuer
**
** Seiteneffekte:
**  *anzoffen wird erhoeht, *offen um ein Element verlaengert,
**  in feld werden die tueroffen-Flags gesetzt
*/
static void tueren_offen_senkrecht(anzoffen, offen, feld,
	feldbreite, feldlaenge, x, y)
int *anzoffen, feldbreite, feldlaenge, x, y;
u_char (**offen)[3];
block **feld;
{
	/* falls Koordinaten ausserhalb des Spielfeldes oder alle Tueren/Waende
	   bereits TRANSPARENT, nichts tun */
	if (x < 0 || x >= feldbreite || y < 0 || y > feldlaenge ||
		(y == 0 || feld[x][y - 1][SUED].farbe == TRANSPARENT) &&
		(y == feldlaenge || feld[x][y][NORD].farbe == TRANSPARENT))
		return;

	if (y > 0)
	{
		/* Suedtuer (obere Wand) in Liste eintragen */
		liste_verlaengern(anzoffen, (void **)offen, sizeof(u_char [3]));
		(*offen)[*anzoffen - 1][0] = x;
		(*offen)[*anzoffen - 1][1] = y - 1;
		(*offen)[*anzoffen - 1][2] = SUED;

		/* Tuer im Spielfeld markieren */
		feld[x][y - 1][SUED].tueroffen = 1;
	}

	if (y < feldlaenge)
	{
		/* Nordtuer (untere Wand) in Liste eintragen */
		liste_verlaengern(anzoffen, (void **)offen, sizeof(u_char [3]));
		(*offen)[*anzoffen - 1][0] = x;
		(*offen)[*anzoffen - 1][1] = y;
		(*offen)[*anzoffen - 1][2] = NORD;

		/* Tuer im Spielfeld markieren */
		feld[x][y][NORD].tueroffen = 1;
	}
}


/*
** tueren_offen_markieren
**  markiert alle Tueren/Waende, die ein Spieler durchdringt als offen
**
** Parameter:
**  anzoffen: Zeiger auf die Anzahl der offenen Tueren
**  offen: Zeiger auf die Liste der offenen Tueren
**  feld: Spielfeld in aktuelle Nordrichtung gedreht
**  feldbreite: Breite des Spielfeldes in Bloecken
**  feldlaenge: Laenge des Spielfeldes in Bloecken
**  pos: Position des Spielers
**  radius: Radius des Spielers
**
** Seiteneffekte:
**  *anzoffen, *offen werden veraendert
**  in feld werden die tueroffen-Flags gesetzt
*/
static void tueren_offen_markieren(anzoffen, offen, feld,
	feldbreite, feldlaenge, pos, radius)
int *anzoffen, feldbreite, feldlaenge, radius;
u_char (**offen)[3];
struct position *pos;
block **feld;
{
	/* ragt Spieler links raus? */
	if ((int)pos->xfein < radius)
		tueren_offen_waagerecht(anzoffen, offen, feld,
			feldbreite, feldlaenge, pos->xgrob, pos->ygrob);

	/* ragt Spieler oben raus? */
	if ((int)pos->yfein < radius)
		tueren_offen_senkrecht(anzoffen, offen, feld,
			feldbreite, feldlaenge, pos->xgrob, pos->ygrob);

	/* ragt Spieler rechts raus? */
	if ((int)pos->xfein > RASPT - radius)
		tueren_offen_waagerecht(anzoffen, offen, feld,
			feldbreite, feldlaenge, pos->xgrob + 1, pos->ygrob);

	/* ragt Spieler unten raus? */
	if ((int)pos->yfein > RASPT - radius)
		tueren_offen_senkrecht(anzoffen, offen, feld,
			feldbreite, feldlaenge, pos->xgrob, pos->ygrob + 1);

	/* ragt Spieler links oben raus? */
	if (sqr((int)pos->xfein) + sqr((int)pos->yfein) < sqr(radius))
	{
		tueren_offen_waagerecht(anzoffen, offen, feld,
			feldbreite, feldlaenge, pos->xgrob, pos->ygrob - 1);
		tueren_offen_senkrecht(anzoffen, offen, feld,
			feldbreite, feldlaenge, pos->xgrob - 1, pos->ygrob);
	}

	/* ragt Spieler links unten raus? */
	if (sqr((int)pos->xfein) + sqr(RASPT - (int)pos->yfein) < sqr(radius))
	{
		tueren_offen_waagerecht(anzoffen, offen, feld,
			feldbreite, feldlaenge, pos->xgrob, pos->ygrob + 1);
		tueren_offen_senkrecht(anzoffen, offen, feld,
			feldbreite, feldlaenge, pos->xgrob - 1, pos->ygrob + 1);
	}

	/* ragt Spieler rechts unten raus? */
	if (sqr(RASPT - (int)pos->xfein) + sqr(RASPT - (int)pos->yfein) <
		sqr(radius))
	{
		tueren_offen_waagerecht(anzoffen, offen, feld,
			feldbreite, feldlaenge, pos->xgrob + 1, pos->ygrob + 1);
		tueren_offen_senkrecht(anzoffen, offen, feld,
			feldbreite, feldlaenge, pos->xgrob + 1, pos->ygrob + 1);
	}

	/* ragt Spieler rechts oben raus? */
	if (sqr(RASPT - (int)pos->xfein) + sqr((int)pos->yfein) < sqr(radius))
	{
		tueren_offen_waagerecht(anzoffen, offen, feld,
			feldbreite, feldlaenge, pos->xgrob + 1, pos->ygrob - 1);
		tueren_offen_senkrecht(anzoffen, offen, feld,
			feldbreite, feldlaenge, pos->xgrob + 1, pos->ygrob);
	}
}


/*
** rechne_nordwand
**  fuehrt fuer eine Nordwand eine Koordinatentransformation relativ zu
**  Position und Blickrichtung des Spielers durch und stellt uebergibt
**  sie, sofern sie nicht offen ist, an rechnewand zur weiteren Bearbeitung
**
** Parameter:
**  anzwaende: Zeiger auf die Anzahl der sichtbaren Waende
**  waende: Zeiger auf die Liste der sichtbaren Waende
**  feld: Spielfeld in aktuelle Nordrichtung gedreht
**  x: gedrehte x-Koordinate des Blocks, zu der die Wand gehoert
**  y: gedrehte y-Koordinate des Blocks, zu der die Wand gehoert
**  ichpos: gedrehte Position des Spielers
**  ichblick: gedrehte Blickrichtung des Spielers
**
** Seiteneffekte:
**  *anzwaende wird erhoeht, *waende verlaengert
*/
static void rechne_nordwand(anzwaende, waende, feld, x, y,
	ichpos, ichblick)
int *anzwaende, x, y, ichblick;
struct flaeche **waende;
block **feld;
struct position *ichpos;
{
	if (!feld[x][y][NORD].tueroffen)
		rechnewand(anzwaende, waende,
			drehx((x * RASPT - XPOS(*ichpos)),
				(YPOS(*ichpos) - y * RASPT), ichblick),
			drehy((x * RASPT - XPOS(*ichpos)),
				(YPOS(*ichpos) - y * RASPT), ichblick),
			drehx(((x + 1) * RASPT - XPOS(*ichpos)),
				(YPOS(*ichpos) - y * RASPT), ichblick),
			drehy(((x + 1) * RASPT - XPOS(*ichpos)),
				(YPOS(*ichpos) - y * RASPT), ichblick),
			feld[x][y][NORD].farbe,
			feld[x][y][NORD].tuer);
}


/*
** rechne_westwand
**  fuehrt fuer eine Westwand eine Koordinatentransformation relativ zu
**  Position und Blickrichtung des Spielers durch und stellt uebergibt
**  sie, sofern sie nicht offen ist, an rechnewand zur weiteren Bearbeitung
**
** Parameter:
**  anzwaende: Zeiger auf die Anzahl der sichtbaren Waende
**  waende: Zeiger auf die Liste der sichtbaren Waende
**  feld: Spielfeld in aktuelle Nordrichtung gedreht
**  x: gedrehte x-Koordinate des Blocks, zu der die Wand gehoert
**  y: gedrehte y-Koordinate des Blocks, zu der die Wand gehoert
**  ichpos: gedrehte Position des Spielers
**  ichblick: gedrehte Blickrichtung des Spielers
**
** Seiteneffekte:
**  *anzwaende wird erhoeht, *waende verlaengert
*/
static void rechne_westwand(anzwaende, waende, feld, x, y,
	ichpos, ichblick)
int *anzwaende, x, y, ichblick;
struct flaeche **waende;
block **feld;
struct position *ichpos;
{
	if (!feld[x][y][WEST].tueroffen)
		rechnewand(anzwaende, waende,
			drehx((x * RASPT - XPOS(*ichpos)),
				(YPOS(*ichpos) - y * RASPT), ichblick),
			drehy((x * RASPT - XPOS(*ichpos)),
				(YPOS(*ichpos) - y * RASPT), ichblick),
			drehx((x * RASPT - XPOS(*ichpos)),
				(YPOS(*ichpos) - (y + 1) * RASPT), ichblick),
			drehy((x * RASPT - XPOS(*ichpos)),
				(YPOS(*ichpos) - (y + 1) * RASPT), ichblick),
			feld[x][y][WEST].farbe,
			feld[x][y][WEST].tuer);
}


/*
** rechne_ostwand
**  fuehrt fuer eine Ostwand eine Koordinatentransformation relativ zu
**  Position und Blickrichtung des Spielers durch und stellt uebergibt
**  sie, sofern sie nicht offen ist, an rechnewand zur weiteren Bearbeitung
**
** Parameter:
**  anzwaende: Zeiger auf die Anzahl der sichtbaren Waende
**  waende: Zeiger auf die Liste der sichtbaren Waende
**  feld: Spielfeld in aktuelle Nordrichtung gedreht
**  x: gedrehte x-Koordinate des Blocks, zu der die Wand gehoert
**  y: gedrehte y-Koordinate des Blocks, zu der die Wand gehoert
**  ichpos: gedrehte Position des Spielers
**  ichblick: gedrehte Blickrichtung des Spielers
**
** Seiteneffekte:
**  *anzwaende wird erhoeht, *waende verlaengert
*/
static void rechne_ostwand(anzwaende, waende, feld, x, y,
	ichpos, ichblick)
int *anzwaende, x, y, ichblick;
struct flaeche **waende;
block **feld;
struct position *ichpos;
{
	if (!feld[x][y][OST].tueroffen)
		rechnewand(anzwaende, waende,
			drehx(((x + 1) * RASPT - XPOS(*ichpos)),
				(YPOS(*ichpos) - y * RASPT), ichblick),
			drehy(((x + 1) * RASPT - XPOS(*ichpos)),
				(YPOS(*ichpos) - y * RASPT), ichblick),
			drehx(((x + 1) * RASPT - XPOS(*ichpos)),
				(YPOS(*ichpos) - (y + 1) * RASPT), ichblick),
			drehy(((x + 1) * RASPT - XPOS(*ichpos)),
				(YPOS(*ichpos) - (y + 1) * RASPT), ichblick),
			feld[x][y][OST].farbe,
			feld[x][y][OST].tuer);
}


/*
** objekt_block_finden
**  stellt den Block im Labyrinth fest, an der ein Objekt eingeordnet werden
**  muss, damit es beim Zeichnen korrekt verdeckt wird
**
** Parameter:
**  feldx: Zeiger auf die x-Koordinate des Blocks
**  feldy: Zeiger auf die y-Koordinate des Blocks
**  radius: Radius des Objekts
**  pos: Zeiger auf die Position des Objekts
**  ichpos: Zeiger auf die Position des Spielers
**
** Seiteneffekte:
**  *feldx und *feldy werden gesetzt
*/
static void objekt_block_finden(feldx, feldy, radius, pos, ichpos)
int *feldx, *feldy;
u_char radius;
struct position *pos, *ichpos;
{
	/* y-Koordinate des Blocks, in dem das Objekt gemalt wird,
	   ist in fast allen Faellen die des Blocks, in den der
	   vordere Kugelrand (YPOS + radius) faellt, faellt der vordere
	   Kugelrand auf den Blockrand, wird das Objekt trotzdem im hinteren
	   Block gemalt, deshalb radius - 1 */
	*feldy = (YPOS(*pos) + (int)radius - 1) / RASPT;

	/* Objekt links */
	if (XPOS(*pos) <= (int)ichpos->xgrob * RASPT + RASPT / 2)

		/* falls das Objekt nicht nach vorne aus einem Block ragt,
		   muss das Objekt in dem Block gemalt werden, in den der
		   rechte Kugelrand (XPOS + radius) faellt, es sei denn, er
		   faellt auf den Blockrand */
		if (*feldy == pos->ygrob)
			*feldx = (XPOS(*pos) + (int)radius - 1) / RASPT;

		/* falls das Objekt nach schraeg rechts vorne aus einem Block
		   ragt, muss das Objekt in dem Block gemalt werden, in den
		   es ragt */
		else if (sqr(RASPT - (int)pos->xfein) + sqr(RASPT - (int)pos->yfein) <
			sqr((int)radius))
			*feldx = pos->xgrob + 1;

		/* falls das Objekt nur nach vorne und eventuell nach rechts
		   aus einem Block ragt, muss es im vorderen Block gemalt
		   werden */
		else
		{
			*feldx = pos->xgrob;

			/* falls das Objekt nicht nach rechts, sondern nur nach
			   vorne aus einem Block ragt und sich der Mittelpunkt
			   rechts vom Blockmittelpunkt befindet, darf das Objekt
			   erst einen Block weiter hinten gemalt werden, da
			   es von einer zusaetzlichen Wand verdeckt sein kann */
			if ((int)pos->xfein <= RASPT - (int)radius &&
				(int)pos->yfein - (int)pos->xfein < RASPT / 2)
				(*feldy)--;
		}

	/* sonst analog :-) */
	else

		/* falls das Objekt nicht nach vorne aus einem Block ragt,
		   muss das Objekt in dem Block gemalt werden, in den der
		   links Kugelrand (XPOS - radius) faellt, es sei denn, er
		   faellt auf den Blockrand */
		if (*feldy == pos->ygrob)
			*feldx = (XPOS(*pos) - (int)radius + 1) / RASPT;

		/* falls das Objekt nach schraeg links vorne aus einem Block
		   ragt, muss das Objekt in dem Block gemalt werden, in den
		   es ragt */
		else if (sqr((int)pos->xfein) + sqr((int)(RASPT - pos->yfein)) <
			sqr((int)radius))
			*feldx = pos->xgrob - 1;

		/* falls das Objekt nur nach vorne und eventuell nach links
		   aus einem Block ragt, muss es im vorderen Block gemalt
		   werden */
		else
		{
			*feldx = pos->xgrob;

			/* falls das Objekt nicht nach links, sondern nur nach
			   vorne aus einem Block ragt und sich der Mittelpunkt
			   links vom Blockmittelpunkt befindet, darf das Objekt
			   erst einen Block weiter hinten gemalt werden, da
			   es von einer zusaetzlichen Wand verdeckt sein kann */
			if (pos->xfein >= radius &&
				(int)pos->yfein + (int)pos->xfein < 3 * RASPT / 2)
				(*feldy)--;
		}
}


/*
** objekte_sortieren
**  sortiert eine Liste von Objekten in die Reihenfolge, in der sie spaeter
**  gezeichnet werden muessen
**
** Parameter:
**  anzobjekte: Anzahl der Objekte
**  objekte: Liste von Positionen, etc. der Objekte
**  ichpos: Zeiger auf die Position des Spielers
**  feldbreite: Breite des Spielfeldes
**  feldlaenge: Laenge des Spielfeldes
**
** Seiteneffekte:
**  *objekte wird umsortiert
*/
static void objekte_sortieren(anzobjekte, objekte, ichpos,
	feldbreite, feldlaenge)
int anzobjekte, feldbreite, feldlaenge;
struct objekt *objekte;
struct position *ichpos;
{
	int i; /* Index fuer Objekte */

	/* feststellen, in welchem Block die Objekte gemalt werden muessen */
	for (i = 0; i < anzobjekte; i++)
	{
		int dx, dy;

		dx = (int)objekte[i].pos.xgrob - (int)ichpos->xgrob;
		dy = (int)objekte[i].pos.ygrob - (int)ichpos->ygrob;

		/* rechts? */
		if (dx > -dy)
		{
			int feldx, feldy;
			struct position tmppos, tmpichpos;

			einnorden(&tmppos, &objekte[i].pos, OST,
				feldbreite, feldlaenge);
			einnorden(&tmpichpos, ichpos, OST, feldbreite, feldlaenge);

			objekt_block_finden(&feldx, &feldy,
				objekte[i].radius, &tmppos, &tmpichpos);

			objekte[i].feldx = feldbreite - 1 - feldy;
			objekte[i].feldy = feldx;
		}
		/* links? */
		else if (-dx > -dy)
		{
			int feldx, feldy;
			struct position tmppos, tmpichpos;

			einnorden(&tmppos, &objekte[i].pos, WEST,
				feldbreite, feldlaenge);
			einnorden(&tmpichpos, ichpos, WEST, feldbreite, feldlaenge);

			objekt_block_finden(&feldx, &feldy,
				objekte[i].radius, &tmppos, &tmpichpos);

			objekte[i].feldx = feldy;
			objekte[i].feldy = feldlaenge - 1 - feldx;
		}
		/* sonst mitte */
		else
			objekt_block_finden(&objekte[i].feldx, &objekte[i].feldy,
				objekte[i].radius, &objekte[i].pos, ichpos);

		/* ab hier ist feldx/y relativ zur groben Position von 'ich' */
		objekte[i].feldx -= ichpos->xgrob;
		objekte[i].feldy -= ichpos->ygrob;

		/* Quadrat des Abstands zwischen Objekt und 'ich' berechnen */
		objekte[i].abstandq = sqr(XPOS(objekte[i].pos) - XPOS(*ichpos)) +
			sqr(YPOS(objekte[i].pos) - YPOS(*ichpos));
	}

	/* Objekte sortieren */
	if (anzobjekte)
		qsort(objekte, anzobjekte, sizeof(struct objekt), objekt_cmp);

	/* ab hier ist feldx/y wieder absolut */
	for (i = 0; i < anzobjekte; i++)
	{
		objekte[i].feldx += ichpos->xgrob;
		objekte[i].feldy += ichpos->ygrob;
	}
}


/* bis hier lokaler Teil                       */
/***********************************************/
/* ab hier globaler Teil                       */


/*
** rechne3d
**  berechnet alle sichtbaren Waende und Kugeln
**
** Parameter:
**  objektdaten: Zeiger auf Liste/Anzahl der sichtbaren Waende und Kugeln
**  ich: Position/Blickrichtung von 'ich'
**  anzgegner: Anzahl der anderen Spieler
**  gegner: Liste von Positionen/Blickrichtungen/Farben der anderen Spieler
**  anzschuesse: Anzahl der Schuesse
**  schuesse: Liste von Positionen/Farben der Schuesse
**
** Seiteneffekte:
**  die Komponenten von *objektdaten werden gesetzt
*/
void rechne3d(objektdaten, ich, anzgegner, gegner, anzschuesse, schuesse)
struct objektdaten *objektdaten;
int anzgegner, anzschuesse;
struct spieler *ich, *gegner;
struct schuss *schuesse;
{
	int i;                         /* Index fuer Objekte */
	u_char himricht;               /* grobe Blickrichtung von 'ich' */
	u_char nordbreite, nordlaenge; /* Breite/Lange des
									  gedrehten Labyrinths */
	int anzobjekte;                /* Anzahl der Objekte, die
									  beruecksichtigt werden */
	struct objekt *objekte;        /* Spieler/Schuesse, die
									  beruecksichtigt werden */
	u_char ichblick;               /* gedrehte Blickrichtung von 'ich' */
	struct position ichpos;        /* gedrehte Position von 'ich' */
	int *anzwaende;                /* Zeiger auf Anzahl der Waende */
	int *anzkugeln;                /* Zeiger auf Anzahl der Kugeln */
	struct flaeche **waende;       /* Zeiger auf Liste der Waende */
	struct kugel **kugeln;         /* Zeiger auf Liste der Kugeln */
	block **feld;                  /* gedrehtes Spielfeld */
	int anzoffen;                  /* Anzahl der offenen Tueren */
	u_char (*offen)[3];            /* offene Tueren: x/y/Himmelsrichtung */

	anzwaende = &objektdaten->anzwaende;
	waende = &objektdaten->waende;
	anzkugeln = &objektdaten->anzkugeln;
	kugeln = &objektdaten->kugeln;

	/* blickrichtung grob NORD = 0, WEST = 1, SUED = 2, OST = 3 */
	himricht = (ich->blick * (TRIGANZ / WINKANZ) + TRIGANZ / 8) %
		TRIGANZ / (TRIGANZ / 4);

	feld = feld_himricht[himricht];

	/* bei Blickrichtung WEST und OST Breite/Laenge vertauschen */
	if (himricht % 2)
	{
		nordbreite = feldlaenge;
		nordlaenge = feldbreite;
	}
	else
	{
		nordbreite = feldbreite;
		nordlaenge = feldlaenge;
	}

	/* Position/Blickrichtung von 'ich' drehen */
	einnorden(&ichpos, &ich->pos, himricht, feldbreite, feldlaenge);
	ichblick = ((int)ich->blick - WINKANZ / 4 * (int)himricht +
		WINKANZ) % WINKANZ;

	liste_initialisieren(&anzobjekte, (void **)&objekte);
	liste_initialisieren(&anzoffen, (void **)&offen);

	/* Tueren, durch die der Spieler selbst laeuft, auch
	   als offen markieren */
	tueren_offen_markieren(&anzoffen, &offen, feld,
		nordbreite, nordlaenge, &ichpos, KUGELRAD);

	/* Objekte ausserhalb des Blickfeldes
	   koennten hier wegoptimiert werden */

	for (i = 0; i < anzgegner; i++)
	{
		struct position tmppos; /* temporaer fuer die neue Position */

		/* Position des Spielers drehen */
		einnorden(&tmppos, &gegner[i].pos, himricht,
			feldbreite, feldlaenge);

		/* Spieler nur beruecksichtigen, falls innerhalb der Sichtweite
		   uns nicht hinter 'ich' */
		if (tmppos.ygrob <= ichpos.ygrob &&
			(int)tmppos.ygrob >= (int)ichpos.ygrob - SICHTW - 1)
		{
			liste_verlaengern(&anzobjekte, (void **)&objekte,
				sizeof(struct objekt));

			memcpy(&objekte[anzobjekte - 1].pos, &tmppos,
				sizeof(struct position));
			objekte[anzobjekte - 1].blick =
				((int)gegner[i].blick - WINKANZ / 4 * (int)himricht +
				WINKANZ) % WINKANZ;
			objekte[anzobjekte - 1].farbe = gegner[i].farbe;
			objekte[anzobjekte - 1].radius = KUGELRAD;
		}

		tueren_offen_markieren(&anzoffen, &offen, feld,
			nordbreite, nordlaenge, &tmppos, KUGELRAD);
	}

	for (i = 0; i < anzschuesse; i++)
	{
		struct position tmppos; /* temporaer fuer die neue Position */

		/* Position des Schusses drehen */
		einnorden(&tmppos, &schuesse[i].pos, himricht,
			feldbreite, feldlaenge);

		/* Schuss nur beruecksichtigen, falls innerhalb der Sichtweite
		   uns nicht hinter 'ich' */
		if (tmppos.ygrob <= ichpos.ygrob &&
			(int)tmppos.ygrob >= (int)ichpos.ygrob - SICHTW - 1)
		{
			liste_verlaengern(&anzobjekte, (void **)&objekte,
				sizeof(struct objekt));

			memcpy(&objekte[anzobjekte - 1].pos, &tmppos,
				sizeof(struct position));
			objekte[anzobjekte - 1].blick = -1;
			objekte[anzobjekte - 1].farbe = schuesse[i].farbe;
			objekte[anzobjekte - 1].radius = SCHUSSRAD;
		}
	}

	objekte_sortieren(anzobjekte, objekte, &ichpos, nordbreite, nordlaenge);

	/* alle Objekte hinter 'ich' ueberspringen */
	for (i = 0; i < anzobjekte && objekte[i].feldy > (int)ichpos.ygrob; i++);
	{
		int y; /* y-Koordinate des aktuellen Blocks */
		int d;

		for (d = 0; d <= SICHTW; d++)
		{
			int x; /* x-Koordinate des aktuellen Blocks */

			if ((x = (int)ichpos.xgrob - d) >= 0)
				for (y = ichpos.ygrob; y >= max((int)ichpos.ygrob - d + 1,
					0); y--)
				{
					/* solange noch Objekte im aktuellen Block
					   beruecksichtigt werden muessen, die Objekte
					   projezieren und eintragen */
					while (i < anzobjekte && x == objekte[i].feldx &&
						y == objekte[i].feldy)
					{
						rechnekugel(*anzwaende, *waende, anzkugeln, kugeln,
							drehx((XPOS(objekte[i].pos) - XPOS(ichpos)),
								(YPOS(ichpos) - YPOS(objekte[i].pos)),
								ichblick),
							drehy((XPOS(objekte[i].pos) - XPOS(ichpos)),
								(YPOS(ichpos) - YPOS(objekte[i].pos)),
								ichblick),
							drehblick(XPOS(objekte[i].pos) - XPOS(ichpos),
								YPOS(ichpos) - YPOS(objekte[i].pos),
								objekte[i].blick, ichblick),
							objekte[i].farbe, objekte[i].radius);
						i++;
					}

					/* Westwand projezieren und eintragen */
					rechne_westwand(anzwaende, waende, feld, x, y,
						&ichpos, ichblick);

					/* Nordwand projezieren und eintragen */
					rechne_nordwand(anzwaende, waende, feld, x, y,
						&ichpos, ichblick);
				}

			if ((x = (int)ichpos.xgrob + d) < (int)nordbreite)
				for (y = ichpos.ygrob; y >= max((int)ichpos.ygrob - d + 1,
					0); y--)
				{
					/* solange noch Objekte im aktuellen Block
					   beruecksichtigt werden muessen, die Objekte
					   projezieren und eintragen */
					while (i < anzobjekte && x == objekte[i].feldx &&
						y == objekte[i].feldy)
					{
						rechnekugel(*anzwaende, *waende, anzkugeln, kugeln,
							drehx((XPOS(objekte[i].pos) - XPOS(ichpos)),
								(YPOS(ichpos) - YPOS(objekte[i].pos)),
								ichblick),
							drehy((XPOS(objekte[i].pos) - XPOS(ichpos)),
								(YPOS(ichpos) - YPOS(objekte[i].pos)),
								ichblick),
							drehblick(XPOS(objekte[i].pos) - XPOS(ichpos),
								YPOS(ichpos) - YPOS(objekte[i].pos),
								objekte[i].blick, ichblick),
							objekte[i].farbe, objekte[i].radius);
						i++;
					}

					/* Ostwand projezieren und eintragen */
					rechne_ostwand(anzwaende, waende, feld, x, y,
						&ichpos, ichblick);

					/* Nordwand projezieren und eintragen */
					rechne_nordwand(anzwaende, waende, feld, x, y,
						&ichpos, ichblick);
				}

			if ((y = ichpos.ygrob - d) >= 0)
			{
				/* dann von der Mitte nach links malen */
				for (x = ichpos.xgrob;
					x >= max((int)ichpos.xgrob - d, 0); x--)
				{
					/* solange noch Objekte im aktuellen Block
					   beruecksichtigt werden muessen, die Objekte
					   projezieren und eintragen */
					while (i < anzobjekte && x == objekte[i].feldx &&
						y == objekte[i].feldy)
					{
						rechnekugel(*anzwaende, *waende, anzkugeln, kugeln,
							drehx((XPOS(objekte[i].pos) - XPOS(ichpos)),
								(YPOS(ichpos) - YPOS(objekte[i].pos)),
								ichblick),
							drehy((XPOS(objekte[i].pos) - XPOS(ichpos)),
								(YPOS(ichpos) - YPOS(objekte[i].pos)),
								ichblick),
							drehblick(XPOS(objekte[i].pos) - XPOS(ichpos),
								YPOS(ichpos) - YPOS(objekte[i].pos),
								objekte[i].blick, ichblick),
							objekte[i].farbe, objekte[i].radius);
						i++;
					}

					/* Westwand projezieren und eintragen */
					rechne_westwand(anzwaende, waende, feld, x, y,
						&ichpos, ichblick);

					/* Nordwand projezieren und eintragen */
					rechne_nordwand(anzwaende, waende, feld, x, y,
						&ichpos, ichblick);
				}

				/* zuerst von der Mitte nach rechts malen */
				for (x = ichpos.xgrob;
					x <= min((int)ichpos.xgrob + d, (int)nordbreite - 1);
					x++)
				{
					/* solange noch Objekte im aktuellen Block
					   beruecksichtigt werden muessen, die Objekte
					   projezieren und eintragen */
					while (i < anzobjekte && x == objekte[i].feldx &&
						y == objekte[i].feldy)
					{
						rechnekugel(*anzwaende, *waende, anzkugeln, kugeln,
							drehx((XPOS(objekte[i].pos) - XPOS(ichpos)),
								(YPOS(ichpos) - YPOS(objekte[i].pos)),
								ichblick),
							drehy((XPOS(objekte[i].pos) - XPOS(ichpos)),
								(YPOS(ichpos) - YPOS(objekte[i].pos)),
								ichblick),
							drehblick(XPOS(objekte[i].pos) - XPOS(ichpos),
								YPOS(ichpos) - YPOS(objekte[i].pos),
								objekte[i].blick, ichblick),
							objekte[i].farbe, objekte[i].radius);
						i++;
					}

					/* Ostwand projezieren und eintragen */
					rechne_ostwand(anzwaende, waende, feld, x, y,
						&ichpos, ichblick);

					/* Nordwand projezieren und eintragen */
					rechne_nordwand(anzwaende, waende, feld, x, y,
						&ichpos, ichblick);
				}
			}
		}
	}

	for (i = 0; i < anzoffen; i++)
		feld[offen[i][0]][offen[i][1]][offen[i][2]].tueroffen = 0;
	liste_freigeben(&anzoffen, (void **)&offen);

	liste_freigeben(&anzobjekte, (void **)&objekte);
}
