/***********************************************************
*  "Das mit dem Affen..." -- A one-night hack              *
*----------------------------------------------------------*
*  1995 Artsoft Development                               *
*        Holger Schemel                                    *
*        33659 Bielefeld-Senne                             *
*        Telefon: (0521) 493245                            *
*        eMail: aeglos@valinor.owl.de                      *
*               aeglos@uni-paderborn.de                    *
*               q99492@pbhrzx.uni-paderborn.de             *
***********************************************************/

#define XK_MISCELLANY
#define XK_LATIN1

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Intrinsic.h>
#include <X11/keysymdef.h>

#include <stdio.h>
#include <stdlib.h>
#include <strings.h>
#include <errno.h>
#include <signal.h>

#include "affe.xbm"
#define ICON_WIDTH	affe_width
#define ICON_HEIGHT	affe_height
#define ICON_BITS	affe_bits

#define TRUE		1
#define FALSE		0

#define PRESSED		TRUE
#define UNPRESSED	FALSE
#define RELEASED	FALSE

#define MIN(a,b) 	(((a)<(b))?(a):(b))
#define MAX(a,b) 	(((a)>(b))?(a):(b))
#define ABS(a)		(((a)<0)?(-a):(a))

#define TILESIZE_X      40
#define TILESIZE_Y      40
#define TILES_X         4
#define TILES_Y         5
#define WIN_XPOS	0
#define WIN_YPOS	0
#define BORDER_X	(TILESIZE_X / 2)
#define BORDER_Y	(TILESIZE_Y / 2)
#define WIN_XSIZE       ((TILES_X * TILESIZE_X) + (2 * BORDER_X))
#define WIN_YSIZE       ((TILES_Y * TILESIZE_Y) + (2 * BORDER_Y))
#define AFFENFUESSE	13

#define BACK		0
#define FRONT		1
#define LIGHT		2
#define DARK		3
#define EMPTY		4
#define INVERT		5
#define MAX_COLORS	6

#ifndef SAVED_GAME_NAME
#define SAVED_GAME_NAME	"/tmp/affenspiel.saved"
#endif

int playfield[TILES_X][TILES_Y];
int startover[TILES_X][TILES_Y] =
{
  1,  1,  0,  5,  5,
  2,  2,  4,  6,  9,
  2,  2,  4,  7, 10,
  3,  3,  0,  8,  8
};

int solution[TILES_X][TILES_Y] =
{
  0,  0,  0,  0,  0,
  0,  0,  0,  2,  2,
  0,  0,  0,  2,  2,
  0,  0,  0,  0,  0
};

Display        *display;
int		screen;
Window  	window;
Pixmap          pixmap, affe_pixmap;
Drawable        drawto;
GC      	gc, color_gc[MAX_COLORS];
Colormap        cmap;

int     	width, height;
unsigned long	pen_fg, pen_bg;
unsigned long	color[MAX_COLORS];

char 		*progname;
char		*name_of_saved_game;
FILE		*saved_game=NULL;
int		game_solved=FALSE;

void OpenAll(int, char **);
void InitDisplay();
void InitWindow(int, char **);
void CloseAll();
void EventLoop();
void HandleExposeEvent(XExposeEvent *);
void HandleButtonEvent(XButtonEvent *, int);
void HandleKeyEvent(XKeyEvent *);
void ClearWindow(void);
void DisplayDrawBuffer(void);
void DrawBorder(int, int, int, int, int);
void DrawTile(int, int, int);
void DrawTiles();
void TryToMoveTile(int, int, int, int);
void CheckIfSolved();
void SillyEffect();
void NewGame();
void LoadGame();
void SaveGame();

int main(int argc, char *argv[])
{
  progname = argv[0];
  name_of_saved_game = (argc>1 ? argv[1] : SAVED_GAME_NAME);
  OpenAll(argc, argv);

  EventLoop();

  CloseAll();
  exit(0);
}

void OpenAll(int argc, char *argv[])
{
  InitDisplay();
  InitWindow(argc, argv);
  NewGame();
  DrawTiles();

  signal(SIGINT, CloseAll);
  signal(SIGTERM, CloseAll);

  XMapWindow(display, window);
  XFlush(display);
}

void InitDisplay()
{
  int i;
  XColor screen_def, exact_def;
  char *color_name[MAX_COLORS] = 
  {
    "gray90",
    "black",
    "white",
    "gray60",
    "gray80",
    "gray90"
  };
  int mono_color[MAX_COLORS] = { 0, 1, 1, 1, 1, 0 };

  if (!(display=XOpenDisplay(NULL)))
  {
    fprintf(stderr,"%s: cannot connect to X server %s\n", 
	    progname, XDisplayName(NULL));
    exit(-1);
  }
  
  screen = DefaultScreen(display);
  cmap = DefaultColormap(display, screen);

  pen_bg = WhitePixel(display,screen);
  pen_fg = BlackPixel(display,screen);

  for(i=0;i<MAX_COLORS;i++)
  {
    if (!XAllocNamedColor(display,cmap,color_name[i],&screen_def,&exact_def) ||
	screen_def.red   - exact_def.red   > 0x0800 ||
	screen_def.green - exact_def.green > 0x0800 ||
	screen_def.blue  - exact_def.blue  > 0x0800)
      color[i] = (mono_color[i] ? pen_fg : pen_bg);
    else
      color[i] = screen_def.pixel;
  }
}

void InitWindow(int argc, char *argv[])
{
  unsigned int border_width = 4;
  Pixmap icon_pixmap;
  XSizeHints size_hints;
  XWMHints wm_hints;
  XClassHint class_hints;
  XTextProperty windowName, iconName;
  XGCValues gc_values;
  unsigned long gc_valuemask;
  char *window_name = "Das mit dem Affen...";
  char *icon_name = "ffle";
  long window_event_mask;
  int i;

  width = WIN_XSIZE;
  height = WIN_YSIZE;

  window = XCreateSimpleWindow(display, RootWindow(display, screen),
			       WIN_XPOS, WIN_YPOS, width, height, border_width,
			       pen_fg, pen_bg);

  affe_pixmap = XCreateBitmapFromData(display, window, ICON_BITS, 
                                      ICON_WIDTH, ICON_HEIGHT);
  icon_pixmap=affe_pixmap;

  size_hints.flags = PSize | PMinSize | PMaxSize;
  size_hints.width  = size_hints.min_width  = size_hints.max_width  = width;
  size_hints.height = size_hints.min_height = size_hints.max_height = height;

  if (!XStringListToTextProperty(&window_name, 1, &windowName))
  {
    fprintf(stderr, "%s: structure allocation for windowName failed.\n",
	    progname);
    exit(-1);
  }

  if (!XStringListToTextProperty(&icon_name, 1, &iconName))
  {
    fprintf(stderr, "%s: structure allocation for iconName failed.\n",
	    progname);
    exit(-1);
  }

  wm_hints.initial_state = NormalState;
  wm_hints.input = True;
  wm_hints.icon_pixmap = icon_pixmap;
  wm_hints.flags = StateHint | IconPixmapHint | InputHint;

  class_hints.res_name = progname;
  class_hints.res_class = "Das mit dem Affen";

  XSetWMProperties(display, window, &windowName, &iconName, 
		   argv, argc, &size_hints, &wm_hints, 
		   &class_hints);

  /* Select event types wanted */
  window_event_mask = ExposureMask | ButtonPressMask | ButtonReleaseMask |
                      KeyPressMask | StructureNotifyMask;
  XSelectInput(display, window, window_event_mask);

  /* create GC for normal drawing */
  gc_values.graphics_exposures = False;
  gc_values.foreground = pen_fg;
  gc_values.background = pen_bg;
  gc_valuemask = GCGraphicsExposures | GCForeground | GCBackground;
  gc = XCreateGC(display, window, gc_valuemask, &gc_values);

  /* create GCs for drawing in all (five) colors */
  for(i=0;i<MAX_COLORS;i++)
  {
    gc_values.graphics_exposures = False;
    gc_values.foreground = color[i];
    gc_values.background = color[BACK];
    gc_values.line_width = 1;
    gc_values.line_style = LineSolid;
    gc_values.cap_style = CapProjecting;
    gc_values.join_style = JoinBevel;
    gc_values.function = (i==INVERT ? GXinvert : GXcopy);
    gc_valuemask = GCGraphicsExposures | GCForeground | GCBackground |
                   GCLineWidth | GCLineStyle | GCCapStyle | GCJoinStyle |
		   GCFunction;
    color_gc[i] = XCreateGC(display, window, gc_valuemask, &gc_values);
  }

  gc = color_gc[FRONT];

  pixmap = XCreatePixmap(display, window,
			 WIN_XSIZE,WIN_YSIZE,
			 XDefaultDepth(display,screen));
  drawto = pixmap;
}

void CloseAll(void)
{
  XFreeGC(display, gc);
  XCloseDisplay(display);
  exit(0);
}

void EventLoop(void)
{
  XEvent event;
  int pending;

  for(;;)
  {
    pending=XPending(display);

    if (pending)	/* got an event */
    {
      for(; pending>0; pending--)
      {
	XNextEvent(display, &event);
	switch(event.type)
	{
	  case Expose:
            HandleExposeEvent((XExposeEvent *) &event);
	    break;
	  case ButtonPress:
            HandleButtonEvent((XButtonEvent *) &event, PRESSED);
	    break;
	  case ButtonRelease:
            HandleButtonEvent((XButtonEvent *) &event, RELEASED);
	    break;
	  case KeyPress:
	  case KeyRelease:
            HandleKeyEvent((XKeyEvent *) &event);
	    break;
	  default:
	    break;
	}
      }
    }
    usleep(100000);
  }
}

void HandleExposeEvent(XExposeEvent *event)
{
  int x=event->x, y=event->y;
  int w=event->width, h=event->height;

  XCopyArea(display,pixmap,window,gc,x,y,w,h,x,y);
}

void HandleButtonEvent(XButtonEvent *event, int button_state)
{
  int x=event->x, y=event->y, button=event->button;
  int bx=(x-BORDER_X+TILESIZE_X)/TILESIZE_X-1;
  int by=(y-BORDER_Y+TILESIZE_Y)/TILESIZE_Y-1;
  static int old_bx=-1, old_by=-1;

  if (button_state == PRESSED)
  {	
    old_bx=bx;
    old_by=by;

    if (bx>=0 && bx<TILES_X && by>=0 && by<TILES_Y && playfield[bx][by])
    {
      DrawTile(bx,by,PRESSED);
      DisplayDrawBuffer();
    }
  }
  else
  {
    if (old_bx>=0 && old_bx<TILES_X && old_by>=0 && old_by<TILES_Y)
    {
      DrawTile(old_bx,old_by,UNPRESSED);
      DisplayDrawBuffer();

      if (playfield[old_bx][old_by]==playfield[bx][by])
	TryToMoveTile(bx,by,
		      x-(BORDER_X+bx*TILESIZE_X+TILESIZE_X/2),
		      y-(BORDER_Y+by*TILESIZE_Y+TILESIZE_Y/2));
    }
  }
}

void HandleKeyEvent(XKeyEvent *event)
{
  KeySym key = XLookupKeysym(event,event->state);

  switch(key)
  {
    case XK_L:
    case XK_l:
      if (name_of_saved_game)
      {
	LoadGame();
	DrawTiles();
	DisplayDrawBuffer();
      }
      break;
    case XK_N:
    case XK_n:
      NewGame();
      DrawTiles();
      DisplayDrawBuffer();
      break;
    case XK_S:
    case XK_s:
      if (name_of_saved_game)
      {
	SaveGame();
	DrawTiles();
	DisplayDrawBuffer();
      }
      break;
    case XK_Q:
    case XK_q:
      CloseAll();
      exit(0);
      break;
    default:
      break;
  }
}

void ClearWindow()
{
  XFillRectangle(display,drawto,color_gc[BACK],0,0,WIN_XSIZE,WIN_YSIZE);
  XFlush(display);
}

void DisplayDrawBuffer()
{
  XCopyArea(display,pixmap,window,gc,0,0,WIN_XSIZE,WIN_YSIZE,0,0);
  XFlush(display);
}

void DrawBorder(int x1, int y1, int x2, int y2, int pressed)
{
  int upperleft  = (pressed ? DARK : LIGHT);
  int lowerright = (pressed ? LIGHT : DARK);

  XDrawLine(display,drawto,color_gc[upperleft],x1,y1,x2,y1);
  XDrawLine(display,drawto,color_gc[upperleft],x1+1,y1+1,x2-1,y1+1);
  XDrawLine(display,drawto,color_gc[upperleft],x1,y1,x1,y2);
  XDrawLine(display,drawto,color_gc[upperleft],x1+1,y1+1,x1+1,y2-1);
  XDrawLine(display,drawto,color_gc[lowerright],x1+1,y2,x2,y2);
  XDrawLine(display,drawto,color_gc[lowerright],x1+2,y2-1,x2-1,y2-1);
  XDrawLine(display,drawto,color_gc[lowerright],x2,y1+1,x2,y2);
  XDrawLine(display,drawto,color_gc[lowerright],x2-1,y1+2,x2-1,y2-1);
}

void DrawTile(int x, int y, int pressed)
{
  int x1=BORDER_X+x*TILESIZE_X;
  int y1=BORDER_Y+y*TILESIZE_Y;
  int x2=BORDER_X+(x+1)*TILESIZE_X-1;
  int y2=BORDER_Y+(y+1)*TILESIZE_Y-1;

  if (x>0 && playfield[x-1][y]==playfield[x][y])
    x1-=TILESIZE_X;
  if (x<TILES_X-1 && playfield[x+1][y]==playfield[x][y])
    x2+=TILESIZE_X;
  if (y>0 && playfield[x][y-1]==playfield[x][y])
    y1-=TILESIZE_Y;
  if (y<TILES_Y-1 && playfield[x][y+1]==playfield[x][y])
    y2+=TILESIZE_Y;

  if (!playfield[x][y])
  {
    XFillRectangle(display,drawto,color_gc[EMPTY],x1,y1,x2-x1+1,y2-y1+1);
    return;
  }

  if (pressed)
    DrawBorder(x1,y1,x2,y2,PRESSED);
  else
    DrawBorder(x1,y1,x2,y2,UNPRESSED);

  if (x2-x1+1==2*TILESIZE_X && y2-y1+1==2*TILESIZE_Y) /* das mit dem Affen */
    XCopyPlane(display,affe_pixmap,drawto,gc,
	       0,0,ICON_WIDTH,ICON_HEIGHT-AFFENFUESSE,
	       x1+2,y1+2+(2*TILESIZE_Y-(ICON_HEIGHT-AFFENFUESSE)-4),0x1);
}

void DrawTiles()
{
  int x,y;

  ClearWindow();
  DrawBorder(BORDER_X-2,BORDER_Y-2,
	     WIN_XSIZE-BORDER_X+1,WIN_YSIZE-BORDER_Y+1,PRESSED);

  for(y=0;y<TILES_Y;y++)
    for(x=0;x<TILES_X;x++)
      DrawTile(x,y,UNPRESSED);

  /* die Fuesse des Affen */
  XCopyPlane(display,affe_pixmap,drawto,gc,
	     0,ICON_HEIGHT-AFFENFUESSE,ICON_WIDTH,AFFENFUESSE,
	     WIN_XSIZE/2-ICON_WIDTH/2,WIN_YSIZE-BORDER_Y+2,0x1);
}

void TryToMoveTile(int bx, int by, int movehint_x, int movehint_y)
{
  int x,y;
  int bbx[2],bby[2];
  int move_x=(movehint_x<0 ? -1 : 1);
  int move_y=(movehint_y<0 ? -1 : 1);
  int we_can_move=TRUE;
  int value=playfield[bx][by];

  bbx[0]=bbx[1]=bx;
  bby[0]=bby[1]=by;

  if (ABS(movehint_x) > ABS(movehint_y))
    move_y=0;
  else
    move_x=0;

  if (bx>0 && playfield[bx-1][by]==playfield[bx][by])
  {
    bbx[0]=bx-1;
    move_x=1;
    move_y=0;
  }
  if (bx<TILES_X-1 && playfield[bx+1][by]==playfield[bx][by])
  {
    bbx[1]=bx+1;
    move_x=-1;
    move_y=0;
  }
  if (by>0 && playfield[bx][by-1]==playfield[bx][by])
  {
    bby[0]=by-1;
    move_x=0;
    move_y=1;
  }
  if (by<TILES_Y-1 && playfield[bx][by+1]==playfield[bx][by])
  {
    bby[1]=by+1;
    move_x=0;
    move_y=-1;
  }

  /* test if we can move */
  for(x=0;x<2;x++)
  {
    for(y=0;y<2;y++)
    {
      int new_x=bbx[x]+move_x;
      int new_y=bby[y]+move_y;

      if (new_x<0 || new_x >=TILES_X || new_y<0 || new_y>=TILES_Y ||
	  (playfield[new_x][new_y]!=playfield[bx][by] &&
	   playfield[new_x][new_y]!=0))
	we_can_move=FALSE;
    }
  }
  
  /* no success: try the opposite direction... */
  if (!we_can_move)
  {
    move_x=-move_x;
    move_y=-move_y;
    we_can_move=TRUE;

    for(x=0;x<2;x++)
    {
      for(y=0;y<2;y++)
      {
	int new_x=bbx[x]+move_x;
	int new_y=bby[y]+move_y;
	
	if (new_x<0 || new_x >=TILES_X || new_y<0 || new_y>=TILES_Y ||
	    (playfield[new_x][new_y]!=playfield[bx][by] &&
	     playfield[new_x][new_y]!=0))
	  we_can_move=FALSE;
      }
    }
  }

  /* don't give up too fast... */
  if (!we_can_move)
  {
    int i;
    int try_move_x[4] = {  0, 0, -1, 1 };
    int try_move_y[4] = { -1, 1,  0, 0 };

    for(i=0;i<4;i++)
    {
      we_can_move=TRUE;

      for(x=0;x<2;x++)
      {
	for(y=0;y<2;y++)
	{
	  int new_x=bbx[x]+try_move_x[i];
	  int new_y=bby[y]+try_move_y[i];
	  
	  if (new_x<0 || new_x >=TILES_X || new_y<0 || new_y>=TILES_Y ||
	      (playfield[new_x][new_y]!=playfield[bx][by] &&
	       playfield[new_x][new_y]!=0))
	    we_can_move=FALSE;
	}
      }
      
      if (we_can_move)
      {
	move_x=try_move_x[i];
	move_y=try_move_y[i];
	break;
      }
    }

    if (!we_can_move)
      return;
  }

  /* delete at old position */
  for(x=0;x<2;x++)
    for(y=0;y<2;y++)
      playfield[bbx[x]][bby[y]]=0;

  /* set at new position */
  for(x=0;x<2;x++)
  {
    for(y=0;y<2;y++)
    {
      int new_x=bbx[x]+move_x;
      int new_y=bby[y]+move_y;

      playfield[new_x][new_y]=value;
    }
  }

  DrawTiles();
  DisplayDrawBuffer();

  CheckIfSolved();
}

void CheckIfSolved()
{
  int x,y;
  int affe_in_einem_stueck=TRUE;

  for(x=0;x<TILES_X;x++)
    for(y=0;y<TILES_Y;y++)
      if (solution[x][y] && playfield[x][y]!=solution[x][y])
	affe_in_einem_stueck=FALSE;

  if (affe_in_einem_stueck)
  {
    if (!game_solved)
    {
      game_solved=TRUE;
      SillyEffect();
    }
  }
  else
  {
    if (game_solved)
      game_solved=FALSE;
  }
}

void SillyEffect()
{
  int i;

  for(i=0;i<20;i++)
  {
    XFillRectangle(display,drawto,color_gc[INVERT],0,0,WIN_XSIZE,WIN_YSIZE);
    DisplayDrawBuffer();
    usleep(20000);
  }
}

void NewGame()
{
  int x,y;

  for(y=0;y<TILES_Y;y++)
    for(x=0;x<TILES_X;x++)
      playfield[x][y]=startover[x][y];
}

void LoadGame()
{
  int x,y;

  if (!(saved_game=fopen(name_of_saved_game,"r")))
  {
    fprintf(stderr,"cannot read '%s'.\n",name_of_saved_game);
    return;
  }

  for(y=0;y<TILES_Y;y++)
  {
    for(x=0;x<TILES_X;x++)
    {
      fscanf(saved_game,"%d ",&playfield[x][y]);
    }
    fscanf(saved_game,"\n");
  }

  fclose(saved_game);
}

void SaveGame()
{
  int x,y;

  if (!(saved_game=fopen(name_of_saved_game,"w")))
  {
    fprintf(stderr,"cannot write '%s'.\n",name_of_saved_game);
    return;
  }

  for(y=0;y<TILES_Y;y++)
  {
    for(x=0;x<TILES_X;x++)
    {
      fprintf(saved_game,"%d ",playfield[x][y]);
    }
    fprintf(saved_game,"\n");
  }

  fclose(saved_game);
}
