#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/Xaw/Label.h>
#include <X11/Xaw/Command.h>
#ifndef NOSTRIP
#include <X11/Xaw/StripChart.h>
#endif /* NOSTRIP */
#include <X11/Xmu/CharSet.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <mymalloc.h>
#include <except.h>
#include <myxlib.h>
#include <YShell.h>

#include "connect.h"
#include "events.h"
#include "gointer.h"
#include "players.h"
#include "tell.h"
#include "utils.h"
#include "xgospel.h"

#define PRTPTR    "#%p"        /* Nice outputformat for a pointer          */
#define NULLNAME  "(null)"     /* How to print the null pointer            */

#define BEGINESCAPE       "\\{"
#define ENDESCAPE         "} "
#define XGOSPEL           "xgospel"

#define XGOSPELESCAPE BEGINESCAPE XGOSPEL ENDESCAPE

#define CREATE    1
#define DELETE    2
#define PlayerWidget(player) (((player)->Widget != 0) +                 \
                              ((player)->WidgetPlan & CREATE != 0) -    \
                              ((player)->WidgetPlan & DELETE != 0))
#define PlayerRefresh()                                                     \
do {                                                                        \
    if (!WidgetProc && !WantUpdate)                                         \
        if (PlayerUpdate) WantUpdate = 1;                                   \
        else WidgetProc = XtAppAddWorkProc(AppContext(players),             \
                                           WorkShowPlayers, NULL);          \
} while (0)
#define AppContext(w)   XtWidgetToApplicationContext(w)

struct _Player {
    struct _Player *Next, *Previous;
    char           *Name;
    char           *Strength;
    int             AutoRated;
    char           *State;
    char           *Idle;
    NameList       *Results;
    int             Pos, OnServer, XgospelUser;
    State           Found;
    Tell           *Tell;
    Game           *Game;
    int             WidgetPlan;
    Widget          Widget;
};

#define CHANGEINT(ptr, field, value, extra)     \
do {                                            \
    if ((ptr)->field != (value)) {              \
        (ptr)->field = (value);                 \
        extra;                                  \
    }                                           \
} while(0)

#define CHANGESTRING(ptr, field, value, extra)  \
do {                                            \
    if ((ptr)->field && (!value || strcmp((ptr)->field, (value))) || \
        !(ptr)->field && value) {               \
        char *sillystr;                         \
                                                \
        sillystr = mystrdup(value);             \
        myfree((ptr)->field);                   \
        (ptr)->field = sillystr;                \
        extra;                                  \
    }                                           \
} while(0)

Widget  PlayersButton;
int     nrplayers, maxplayers;
Player *Me;

extern int nrgames;

static void DoPlayer(Widget w, XEvent *evnt, String *str, Cardinal *n);
static XtActionsRec actionTable[] = {
    { "doplayer",    DoPlayer   },
};

static Player PlayerBase      = { &PlayerBase,  &PlayerBase };
static Player DummyPlayerBase = { &DummyPlayerBase,  &DummyPlayerBase };

static const char   StrengthOrder[] = "p dkR?";
static Widget       playerinfo, players, playerstats;
static XtWorkProcId WidgetProc;
static MyContext    PlayerContext;
static XrmQuark     WidgetToPlayer;
static int          PlayerPos, WantUpdate, WantWho;
static long         PlayerUpdate;

static int     PlayerInsert(Widget w);
static Player *MakePlayer(const char *Name);
static void    ChangeStrength(Player *player);
static void    ShowPlayer(Player *player);
static void    DeletePlayer(Player *player);

static const char *NameStrength(const char *Name, const char *Strength,
                                char AutoRated)
{
    static char Text[80];

    sprintf(Text, "%s[%3s]%c", Name, Strength, AutoRated);
    return Text;
}

const char *PlayerString(const Player *player)
{
    char AutoRated;

    switch(player->AutoRated) {
      case 0:  AutoRated = ' '; break;
      case 1:  AutoRated = '*'; break;
      default: AutoRated = '?'; break;
    }

    return NameStrength(player->Name, player->Strength, AutoRated);
}

const char *PlayerNameToString(const char *Name)
{
    return PlayerString(NameToPlayer(Name));
}

char *PlayerTemplateDescription(const Player *player, const char *Template)
{
    const char *AutoRated, *autorated;

    switch(player->AutoRated) {
      case 0:  AutoRated =      autorated = " "; break;
      case 1:  AutoRated =      autorated = "*"; break;
      default: AutoRated = "?"; autorated = ""; break;
    }

    return StringToFilename(Template,
                            (int) 'N', player->Name,
                            (int) 'n', player->Strength,
                            (int) 'A', AutoRated,
                            (int) 'a', autorated,
                            0);
}

static char *PlayerTitle(const char *Pattern, XtPointer Closure)
{
    return PlayerTemplateDescription((const Player *) Closure, Pattern);
}

void SetPlayerTitles(Widget w, const Player *player)
{
    SetWidgetTitles(w, PlayerTitle, (XtPointer) player);
}

/*****************************************************************************/

#ifndef NOSTRIP
static void SetNrPlayers(Widget w, XtPointer client_data, XtPointer value)
{
    *(double *) value = ((double) nrplayers) / appdata.PlayersScale;
}
#endif /* NOSTRIP */

void InitPlayers(Widget Toplevel)
{
    Widget PlayerRoot;
    Widget PlayerCollect;
    Widget PlayerStrip;

    Me             = NULL;
    nrplayers      = maxplayers = 0;
    WidgetProc     = 0;
    PlayerUpdate   = 0;
    WantUpdate     = 0;
    WantWho        = 0;

    WidgetToPlayer = XrmPermStringToQuark("WidgetToPlayer");
    PlayerContext  = YShellContext(Toplevel);
    
    XtAppAddActions(AppContext(Toplevel), actionTable, XtNumber(actionTable));

    PlayerRoot = MyVaCreateManagedWidget("players", Toplevel, NULL);
                                      
    players       = XtNameToWidget(PlayerRoot, "*playerset");
    XtVaSetValues(players, XtNinsertPosition, (XtArgVal) PlayerInsert, NULL);
    PlayerCollect = XtNameToWidget(PlayerRoot, "*collect");
    playerstats   = XtNameToWidget(PlayerRoot, "*stats");
    PlayerStrip   = XtNameToWidget(PlayerRoot, "*strip");
    if (PlayerStrip)
        XtAddCallback(PlayerStrip, XtNgetValue, SetNrPlayers, NULL);
    playerinfo    = XtNameToWidget(PlayerRoot, "*info");

    XtAddCallback(PlayersButton, XtNcallback,        CallToggleUpDown,
                  (XtPointer) PlayerRoot);
    XtAddCallback(PlayerRoot,    XtNpopupCallback,   CallToggleOn,
                  (XtPointer) PlayersButton);
    XtAddCallback(PlayerRoot,    XtNpopdownCallback, CallToggleOff,
                  (XtPointer) PlayersButton);

    XtRealizeWidget(PlayerRoot);
    XtInstallAllAccelerators(PlayerCollect,    PlayerCollect);
    DeleteProtocol(PlayerRoot);
    CallToggleUpDown(PlayersButton, (XtPointer) PlayerRoot, NULL);
}

/*****************************************************************************/

void CheckPlayerStrength(Player *player, const char *Strength)
{
    char   *str;
    int     Length, AutoRated;

    Length    = strlen(Strength)-1;
    AutoRated = Strength[Length] == '*';
    if (!AutoRated) Length++;
    str = mystrndup(Strength, Length);
    WITH_UNWIND {
        CHANGEINT   (player, AutoRated, AutoRated, player->Found = CHANGED);
        CHANGESTRING(player, Strength, str,
                     ChangeStrength(player); player->Found = CHANGED);
    } ON_UNWIND {
        myfree(str);
    } END_UNWIND;
    if (player->Found != UNCHANGED) ShowPlayer(player);
}

static void NewPlayer(Player *player)
{
    if (playerinfo)
        AddText(playerinfo, "%16s connected\n", PlayerString(player));
    if (player->Tell) TellMessage(player->Tell, "player connected");
    nrplayers++;
    Logon(player);
}

Player *PlayerConnect(const char *Name, const char *Strength)
{
    Player *player;
    int     OldOn;

    player = PlayerFromName(Name);
    OldOn  = player->OnServer;
    player->OnServer = 1;
    player->Found    = CHANGED;
    CheckPlayerStrength(player, Strength);
    if (!OldOn) NewPlayer(player);
    return player;
}

void PlayerDisconnect(const char *Name)
{
    Player *player;

    player = NameToPlayer(Name);
    if (player->OnServer) DeletePlayer(player);
}

/*****************************************************************************/

Player *FindPlayerByNameAndStrength(const char *Name,
                                    const char *Strength)
{
    Player *player;

    if (DebugFun) {
        printf("FindPlayerByNameAndStrength(%s, %s)\n", Name, Strength);
        fflush(stdout);
    }

    player = PlayerFromName(Name);
    CheckPlayerStrength(player, Strength);
    return player;
}

Player *PlayerFromName(const char *Name)
{
    Player *player;

    for (player = PlayerBase.Next; 
         player != &PlayerBase;
         player = player->Next)
        if (0 == strcmp(Name, player->Name)) return player;
    return MakePlayer(Name);
}

Player *NameOnServer(const char *Name)
{
    Player *player;

    for (player = PlayerBase.Next; 
         player != &PlayerBase;
         player = player->Next)
        if (0 == strcmp(Name, player->Name)) return player;
    return NULL;
}

Player *NameToPlayer(const char *Name)
{
    Player *player;

    player = NameOnServer(Name);
    if (!player) {
        if (!TersePlay(NULL) &&
            /* Remove guest clutch as soon as server gets fixed --Ton */
            strncmp(Name, "guest", 5)) {
            if (Entered) Warning("Player %s not found, "
                                 "player database corrupted ?\n", Name);
            AutoCommand(NULL, "who");
        }
        player = MakePlayer(Name);
    }
    return player;
}

void PlayerInGame(Player *player, Game *game)
{
    if (player) player->Game = game;
}

void PlayerOutGame(Player *player, Game *game)
{
    if (player && player->Game == game) player->Game = NULL;
}

Player *FindPlayer(const char *Name,  const char *Strength,
                   const char *state, const char *Idle)
{
    Player   *player;
    int       OldOn;

    if (DebugFun) {
        printf("FindPlayer(%s, %s, %s)\n", Name, Strength, state);
        fflush(stdout);
    }

    player = PlayerFromName(Name);
    OldOn = player->OnServer;
    player->OnServer = 1;
    if (PlayerWidget(player)) player->Found = UNCHANGED;
    else                      player->Found = CHANGED;
    CHANGESTRING(player, Idle, Idle, /* player->Found = CHANGED */;);
    CHANGESTRING(player, State, state, player->Found = CHANGED);
    CheckPlayerStrength(player, Strength);
    if (!OldOn) NewPlayer(player);
    return player;
}

static Player *RealMakePlayer(const char *Name)
{
    Player *player;

    if (DebugFun) {
        printf("MakePlayer(%s)\n", Name);
        fflush(stdout);
    }

    player = mynew(Player); 
    player->Name = player->Strength = player->State = player->Idle = NULL;
    WITH_HANDLING {
        player->Name        = mystrdup(Name);
        player->Strength    = mystrdup("???");
        player->State       = mystrdup("?????  ???");
        player->Idle        = mystrdup(UNKNOWN);
        player->AutoRated   = 2;
        player->Results     = NULL;
        player->Tell        = 0;
        player->WidgetPlan  = 0;
        player->Widget      = 0;
        player->Found       = NEW;
        player->Pos         = 0;
        player->OnServer    = 0;
        player->XgospelUser = NONXGOSPELUSER;
        player->Game        = NULL;
    } ON_EXCEPTION {
        if (player->Idle)     myfree(player->Idle);
        if (player->State)    myfree(player->State);
        if (player->Strength) myfree(player->Strength);
        if (player->Name)     myfree(player->Name);
        myfree(player);
        ReRaise();
    } END_HANDLING;
    return player;
}

static Player *MakePlayer(const char *Name)
{
    Player *player;

    player = RealMakePlayer(Name);
    player->Previous       =  PlayerBase.Previous;
    player->Next           = &PlayerBase;
    player->Next->Previous =  player->Previous->Next = player;
    return player;
}

Player *MakeDummyPlayer(void)
{
    Player *player;

    player = RealMakePlayer("??????");
    player->Previous       =  DummyPlayerBase.Previous;
    player->Next           = &DummyPlayerBase;
    player->Next->Previous =  player->Previous->Next = player;
    return player;
}

void RenameDummyPlayer(Player *player, const char *Name)
{
    CHANGESTRING(player, Name, Name,/* Here we should change game entries */;);
}

void FreeDummyPlayer(Player *player)
{
    player->Previous->Next = player->Next;
    player->Next->Previous = player->Previous;

    myfree(player->Name);
    myfree(player->Strength);
    myfree(player->State);
    myfree(player->Idle);
    myfree(player);
}

void AddResults(const char *Name, NameList *Results)
{
    Player     *player;
    NameList   *Line, *Next;
    const char *Ptr;
    char        Name1[20], Name2[20];

    for (Line = Results->Next; Line != Results; Line = Next) {
        Next = Line->Next;
        Ptr = Line->Name;
        *Name1 = *Name2 = 0;
        sscanf(Ptr, "%19s", Name1);
        Name1[sizeof(Name1)-1] = 0;
        Ptr = strchr(Ptr, ':');
        if (Ptr) {
            sscanf(Ptr+1, "%19s", Name2);
            Name2[sizeof(Name1)-1] = 0;
            if (!*Name1 || !*Name2)
                Warning("Invalid results line deleted: %s\n", Line->Name);
            else if (!strcmp(Name, Name1) || !strcmp(Name, Name2)) continue;
        } else Warning("Invalid results line deleted: %s\n", Line->Name);
        Line->Previous->Next = Next;
        Next->Previous = Line->Previous;
        myfree(Line->Name);
        myfree(Line);
    }
    player = PlayerFromName(Name);
    if (player->Results) FreeNameList(player->Results);
    player->Results = Results;
}

/*****************************************************************************/

int StrengthCompare(const Player *player1, const Player *player2)
{
    const char *Strength1, *Strength2, *ptr1, *ptr2;
    int   temp;

    Strength1 = player1->Strength;
    Strength2 = player2->Strength;

    temp = Strength1[strlen(Strength1)-1];
    ptr1 = strchr(StrengthOrder, temp);
    if (!ptr1) ptr1 = StrengthOrder+1;

    temp = Strength2[strlen(Strength2)-1];
    ptr2 = strchr(StrengthOrder, temp);
    if (!ptr2) ptr2 = StrengthOrder+1;

    if (ptr1 < ptr2) return -1;
    if (ptr1 > ptr2) return  1;
    if (*ptr1 == 'p' || *ptr1 == 'd') {
        temp = atoi(Strength1) - atoi(Strength2);
        if (temp < 0) return  1;
        if (temp > 0) return -1;
    } else if (*ptr1 == 'k') {
        temp = atoi(Strength1) - atoi(Strength2);
        if (temp < 0) return -1;
        if (temp > 0) return  1;
    }
    return 0;
}

int PlayerCompare(const Player *player1, const Player *player2)
{
    const char *Strength1, *Strength2, *ptr1, *ptr2;
    int   temp;

    if (appdata.SortPlayersByStrength != False) {
        Strength1 = player1->Strength;
        Strength2 = player2->Strength;

        temp = Strength1[strlen(Strength1)-1];
        ptr1 = strchr(StrengthOrder, temp);
        if (!ptr1) ptr1 = StrengthOrder+1;

        temp = Strength2[strlen(Strength2)-1];
        ptr2 = strchr(StrengthOrder, temp);
        if (!ptr2) ptr2 = StrengthOrder+1;

        if (ptr1 < ptr2) return -1;
        if (ptr1 > ptr2) return  1;
        if (*ptr1 == 'p' || *ptr1 == 'd') {
            temp = atoi(Strength1) - atoi(Strength2);
            if (temp < 0) return  1;
            if (temp > 0) return -1;
        } else if (*ptr1 == 'k') {
            temp = atoi(Strength1) - atoi(Strength2);
            if (temp < 0) return -1;
            if (temp > 0) return  1;
        }
    }
    return XmuCompareISOLatin1(player1->Name, player2->Name);
}

/* to be used in qsorts */
int PlayersCompare(const void *player1, const void *player2)
{
    return PlayerCompare(*(const Player **) player1,
                         *(const Player **) player2);
}

void GetStats(Widget w, XtPointer clientdata, XtPointer calldata)
{
/* Idle time now available in stats */
/*    SendCommand(NULL, NULL, "who"); */
    SendCommand(NULL, NULL, "results %s", (char *) clientdata);
    SendCommand(NULL, NULL, "stats %s", (char *) clientdata);
}

static void DumpPlayer(const Player *player)
{
    Output ("-------Player Dump------\n");
    Outputf("Player %s\n", PlayerString(player));
    Outputf("Me = " PRTPTR ", Next = " PRTPTR ", Previous = " PRTPTR
            ", Results = " PRTPTR "\n",
            player, player->Next, player->Previous, player->Results);
    Outputf("Name = %s, Strength = %s, AutoRated = %d, Idle = %s\n",
            player->Name, player->Strength, player->AutoRated, player->Idle);
    Outputf("Pos = %d, OnServer = %d, XgospelUser = %d, Found = %d\n",
            player->Pos, player->OnServer, player->XgospelUser, player->Found);
    Outputf("Game = %s, Tell = " PRTPTR ", Widget = %08lx, WidgetPlan = %d\n",
            player->Game ? GameLongDescription(player->Game) : NULLNAME,
            player->Tell, (long) player->Widget, player->WidgetPlan);
}

void DumpPlayers(const char *args)
{
    const Player *player;

    for (player = PlayerBase.Next; player != &PlayerBase;
         player = player->Next) DumpPlayer(player);
}

static void SelectPlayer(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Player *player;
    String  text;

    player = (Player *) clientdata;
    text   = (String)   calldata;

    if      (strcmp(text, PLAYERKEY1) == 0) FindTell(player->Name);
    else if (strcmp(text, PLAYERKEY3) == 0)
        GetStats(w, (XtPointer) player->Name, NULL);
    else if (strcmp(text, DUMPPLAYER) == 0) DumpPlayer(player);
    else Warning("Unknown argument to DoPlayer\n");
}

void UnassumePlayers(Connection conn)
{
    Player *player;

    for (player = PlayerBase.Next; player != &PlayerBase;
         player = player->Next) if (player->XgospelUser == XGOSPELUSER)
             player->XgospelUser = EXXGOSPELUSER;
}

void TellXgospelUsers(Connection conn, const char *message)
{
    Player *player;

    for (player = PlayerBase.Next; player != &PlayerBase;
         player = player->Next)
        if (player->XgospelUser == XGOSPELUSER && player != Me)
            UserSendCommand(conn, NULL, "tell %s %s", player->Name, message);
}

void NewXgospelUser(Connection conn, const Player *user)
{
    Player *player;

    if (user == Me) return;
    for (player = PlayerBase.Next; player != &PlayerBase;
         player = player->Next)
        if (player->XgospelUser == XGOSPELUSER && player != user)
            UserSendCommand(conn, NULL, "tell %s " XGOSPELESCAPE
                        "USER %s VERSION ????", user->Name, player->Name);
}

void ShowPlayerStats(void)
{
    char Text[80];

    if (nrplayers > maxplayers) maxplayers = nrplayers;
    sprintf(Text, "%d player%s (max %d), %d game%s\n", 
            nrplayers, nrplayers == 1 ? "" : "s", maxplayers, 
            nrgames, nrgames == 1 ? "" : "s");
    XtVaSetValues(playerstats, XtNlabel, (XtArgVal) Text, NULL);
}

static void SetPlayerEntry(int *i, Arg *args, const Player *player)
{
    static char Text[80];

    sprintf(Text, "%16s %s", PlayerString(player), player->State);
    XtSetArg(args[*i], XtNlabel, Text); (*i)++;
}

/* Result is usable until the next call */
const char *GetPlayerType(const Player *player, const char *Prefix)
{
    static char Name[80];
    char   ch, *ptr;

    ptr = player->Strength;
    while (isdigit(*ptr)) ptr++;
    sprintf(Name, "%.*s%.*s",
            (int) sizeof(Name)/2, Prefix, (int) sizeof(Name)/2-1, ptr);
    for (ptr = Name; (ch = *ptr) != 0; ptr++)
        if (!isalnum(ch) && ch != '-') *ptr = '_';
    return Name;
}

static int WidgetPlayerCompare(const void *player1, const void *player2)
{
    return PlayerCompare((Player *)*(Widget *)player1,
                         (Player *)*(Widget *)player2);
}

static Boolean WorkShowPlayers(XtPointer Closure)
{
    int        i, j, ToCreate, ToDelete;
    XtPointer  Data;
    Player    *player;
    Widget    *Create, *Cre, *Delete, *Del, *Children;
    Cardinal   NrChildren;
    Arg        arg[5];

    ToCreate = ToDelete = 0;
    for (player = PlayerBase.Next; player != &PlayerBase;
         player = player->Next) switch(player->WidgetPlan) {
           case CREATE:
             ToCreate++;
             break;
           case DELETE:
             ToDelete++;
             break;
           case CREATE | DELETE:
             ToDelete++;
             ToCreate++;
             break;
         }

    XtVaGetValues(players,
                  XtNchildren,    (XtArgVal) &Children,
                  XtNnumChildren, (XtArgVal) &NrChildren,
                  NULL);
    Create = mynews(Widget, ToCreate);
    WITH_UNWIND {
        Delete = mynews(Widget, ToDelete);
        WITH_UNWIND {
            Cre = Create;
            Del = Delete;
            for (player = PlayerBase.Next; player != &PlayerBase;
                 player = player->Next) switch(player->WidgetPlan) {
                   case CREATE:
                     *Cre++ = (Widget) player;
                     break;
                   case DELETE:
                     *Del++ = (Widget) player;
                     break;
                   case CREATE | DELETE:
                     *Cre++ = (Widget) player;
                     *Del++ = (Widget) player;
                     break;
            }
            if (ToCreate) {
                qsort(Create, ToCreate, sizeof(Widget), WidgetPlayerCompare);
                j = 0;
                for (i=0; i<NrChildren; i++) {
                    if (MyFindContext(PlayerContext, Children[i],
                                      WidgetToPlayer, &Data))
                        Raise1(AssertException,
                               "Could not find player for widget");
                    player = (Player *) Data;
                    while (PlayerCompare(player, (Player *) Create[j]) > 0) {
                        ((Player *)Create[j])->Pos = i+j;
                        if (++j >= ToCreate) goto done;
                    }
                }
                for (;j<ToCreate; j++)
                    ((Player *) Create[j])->Pos = NrChildren+j;
            }
          done:
            for (j=0; j<ToDelete; j++) {
                player = (Player *) Delete[j];
                MyDeleteContext(PlayerContext, player->Widget, WidgetToPlayer);
                Delete[j] = player->Widget;
                player->Widget = 0;
                player->WidgetPlan = 0;
            }
            for (j=0; j<ToCreate; j++) {
                player = (Player *) Create[j];
                player->WidgetPlan = 0;
                i = 0;
                SetPlayerEntry(&i, arg, player);
                PlayerPos = player->Pos;
                player->Widget = Create[j] =
                    XtCreateWidget((String)
                                   GetPlayerType(player, "playerEntry"),
                                   commandWidgetClass, players, arg, i);
                XtAddCallback(player->Widget, XtNcallback, SelectPlayer, 
                              (XtPointer) player);
                MySaveContext(PlayerContext, player->Widget, WidgetToPlayer,
                              (XtPointer) player);
            }
            SetManagementChildren(Create, ToCreate, Delete, ToDelete);
            for (j=0; j<ToDelete; j++) {
                XtDestroyWidget(Delete[j]);
            }
        } ON_UNWIND {
            myfree(Delete);
        } END_UNWIND;
    } ON_UNWIND {
        myfree(Create);
    } END_UNWIND;
    ShowPlayerStats();
    WidgetProc = 0;
    PlayerUpdate = appdata.PlayerUpdateTimeout;
    return True;
}

void PlayersResort(void)
{
    Player *player;
    int     changes;

    changes = 0;
    for (player = PlayerBase.Next; player != &PlayerBase;
         player = player->Next)
        if (player->Widget && player->WidgetPlan == 0) {
            player->WidgetPlan = DELETE | CREATE;
            changes = 1;
        }
    if (changes) PlayerRefresh();
}

static void ChangeStrength(Player *player)
{
    Player *pl;
    Arg     arg[5];
    int     i;

    if (DebugFun) {
        printf("ChangeStrength(%s, %s, %s)\n", player->Name,
               player->Strength, player->State);
        fflush(stdout);
    }

    if (player->Widget) {
        XtDestroyWidget(player->Widget);
        MyDeleteContext(PlayerContext, player->Widget, WidgetToPlayer);
        player->Widget = 0;
        PlayerPos = 0;
        i = 0;
        SetPlayerEntry(&i, arg, player);
        for (pl = PlayerBase.Next; pl != &PlayerBase; pl = pl->Next)
            if (pl->Widget) {
                if (pl->Pos > player->Pos) pl->Pos--;
                if (PlayerCompare(player, pl) > 0) PlayerPos++;
                else pl->Pos++;
            }
        player->Pos = PlayerPos;
        player->Widget =
            XtCreateManagedWidget((String)
                                  GetPlayerType(player, "playerEntry"),
                                  commandWidgetClass, players, arg, i);
        XtAddCallback(player->Widget, XtNcallback, SelectPlayer, 
                      (XtPointer) player);
        MySaveContext(PlayerContext, player->Widget, WidgetToPlayer,
                      (XtPointer) player);
    }
    if (player->Tell) SetTellDescription(player->Tell);
    if (player->Game) ChangeGameDescription(player->Game);
}

static void ShowPlayer(Player *player)
{
    if (DebugFun) {
        printf("ShowPlayer(%s, %s, %s)\n", player->Name,
               player->Strength, player->State);
        fflush(stdout);
    }
    if (!player->OnServer) return;

    if (player->Widget) player->WidgetPlan = CREATE | DELETE;
    else                player->WidgetPlan = CREATE;
    PlayerRefresh();
    player->Found = UNCHANGED;
}

static void DeletePlayer(Player *player)
{
    if (DebugFun) {
        if (player) printf("DeletePlayer(%s, %s, %s)\n", player->Name,
                           player->Strength, player->State);
        else printf("DeletePlayer(NULL)\n");
        fflush(stdout);
    }
    if (player) {
        if (player->Widget) {
            player->WidgetPlan = DELETE;
            PlayerRefresh();
        } else player->WidgetPlan = 0;
        if (playerinfo) AddText(playerinfo, "%16s disconnected\n",
                                PlayerString(player));
        if (player->Tell) TellMessage(player->Tell, "player disconnected");
        Logoff(player);
        nrplayers--;
        player->OnServer = 0;
    }
}

static int PlayerInsert(Widget w)
{
    return PlayerPos;
}

static void DoPlayer(Widget w, XEvent *evnt, String *str, Cardinal *n)
{
    if (*n) XtCallCallbacks(w, XtNcallback, (XtPointer) str[0]);
    else    XtCallCallbacks(w, XtNcallback, (XtPointer) "none");
}

/*****************************************************************************/

void AssertPlayersDeleted(void)
{
    Player *player;

    for (player = PlayerBase.Next; 
         player != &PlayerBase;
         player = player->Next)
        player->Found = DELETED;
}

void TestPlayersDeleted(void)
{
    Player *player, *next;
                    
    for (player = PlayerBase.Next; 
         player != &PlayerBase;
         player = next) {
        next = player->Next;
        switch(player->Found) {
          case NEW:
          case CHANGED:
            ShowPlayer(player);
            break;
          case DELETED:
            if (player->OnServer) DeletePlayer(player);
            break;
          case UNCHANGED: /* Can also mean: already (re)displayed */
            break;
          default:
            Raise1(AssertException, "Impossible WhoState");
            break;
        }
    }
}

void PlayerStatusLine(int Players, int MaxPlayers, int Games)
{
/*  nrplayers  = Players; */
    if (MaxPlayers > 0 && MaxPlayers > maxplayers) {
        maxplayers = MaxPlayers;
        ShowPlayerStats();
    }
    if (nrgames != Games) AutoCommand(NULL, "games");
}

const char *PlayerToStrength(const Player *player)
{
    return player ? player->Strength : NULL;
}

const char *PlayerToName(const Player *player)
{
    return player ? player->Name : NULL;
}

const char *PlayerToAutoRated(const Player *player)
{
    switch(player->AutoRated) {
      case 0:  return " ";
      case 1:  return "*";
    }
    return "?";
}

const char *PlayerToIdle(const Player *player)
{
    return player ? player->Idle : NULL;
}

const NameList *PlayerToResults(const Player *player)
{
    return player ? player->Results : NULL;
}

int OnServerP(const Player *player)
{
    return player->OnServer;
}

Tell **PlayerTellAddress(Player *player)
{
    return &player->Tell;
}

int *PlayerXgospelUser(Player *player)
{
    return &player->XgospelUser;
}

void PlayersTime(unsigned long diff)
{
    if (PlayerUpdate) {
        PlayerUpdate -= diff;
        if (PlayerUpdate <= 0) {
            PlayerUpdate = 0;
            if (WantUpdate) {
                WantUpdate = 0;
                PlayerRefresh();
            }
        }
    }
}
