#include <X11/StringDefs.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/Xaw/AsciiText.h>
#include <X11/Xaw/Toggle.h>
#include <X11/Xmu/CharSet.h>

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#if !STDC_HEADERS && HAVE_MEMORY_H
# include <memory.h>
#endif /* not STDC_HEADERS and HAVE_MEMORY_H */

#include <except.h>
#include <myxlib.h>
#include <mymalloc.h>

#include "broadcast.h"
#include "connect.h"
#include "players.h"
#include "utils.h"
#include "xgospel.h"

#ifdef    HAVE_NO_MEMCHR_PROTO
extern void *memchr(const void *s, int c, size_t n);
#endif /* HAVE_NO_MEMCHR_PROTO */

#define MINCHLENGTH     2

Widget BroadcastButton, YellButton;

static void Broadcast(), Yell(), ChangeChannel(), ChangeChannelTitle();
static XtActionsRec actionTable[] = {
    { "broadcast",              Broadcast  },
    { "yell",                   Yell },
    { "changechannel",          ChangeChannel },
    { "changechanneltitle",     ChangeChannelTitle },
};

struct ChannelData {
    ChannelData *Next, *Previous;
    char        *Name, *Title, *Moderator, *State;
    NameList    *Names;
};

static ChannelData ChannelBase = { &ChannelBase, &ChannelBase };
static int    MyChannel, NoChannelLength;
static char  *NoChannel;
static char  *BroadcastFileName, *YellFileName;
static Widget broadcastinfo, broadcastinput, broadcastbeep, broadcastraise;
static Widget broadcastoverwrite;
static Widget BroadcastErrorBeep, BroadcastErrorRaise;
static Widget yellinfo, yellinput, YellBeep, YellRaise, yellchannel;
static Widget yelloverwrite, YellErrorBeep, YellErrorRaise, YellTitle;
static Widget YellModerator, YellState;

static void ChangeBroadcastFilename(Widget w,
                                    XtPointer clientdata, XtPointer calldata)
{
    char **Filename;
    Widget Root;

    Filename = (char **) clientdata;
    Root = ChangeFilename(toplevel, "broadcastFilename",
                          "Enter name of broadcast file",
                          Filename, "filename", *Filename, NULL);
    MyDependsOn(Root, w);
}

static void SaveBroadcast(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Boolean                   Overwrite;
    static char const * const ErrorType = "Broadcast save error";

    if (broadcastinfo) {
        if (broadcastoverwrite) XtVaGetValues(broadcastoverwrite, XtNstate,
                                              (XtArgVal) &Overwrite, NULL);
        else Overwrite = False;

        SaveWrite(BroadcastFileName, Overwrite,
                  BroadcastErrorBeep, BroadcastErrorRaise, ErrorType,
                  SaveTextFun, (XtPointer) broadcastinfo);
    } else {
        IfBell(BroadcastErrorBeep);
        IfRaise(BroadcastErrorRaise, BroadcastErrorRaise);
        PopMessage(ErrorType, "Broadcasts were ignored so there is "
                   "nothing to be saved");
    }
}

void InitBroadcast(Widget Toplevel)
{
    Widget BroadcastRoot, BroadcastQuit, BroadcastSave, BroadcastFile;
    Widget BroadcastCollect;

    XtAppAddActions(XtWidgetToApplicationContext(Toplevel),
                    actionTable, XtNumber(actionTable));
    BroadcastRoot = MyVaCreateManagedWidget("broadcasts", Toplevel, NULL);

    BroadcastFileName = StringToFilename(appdata.BroadcastFilename,
                                         (int) 'T', "Broadcasts",
                                         (int) 't', "_", 0);
    XtAddCallback(BroadcastRoot, XtNdestroyCallback,
                  CallFree, (XtPointer) BroadcastFileName);
    BroadcastCollect    = XtNameToWidget(BroadcastRoot, "*collect");
    BroadcastQuit       = XtNameToWidget(BroadcastRoot, "*quit");
    broadcastbeep       = XtNameToWidget(BroadcastRoot, "*beep");
    broadcastraise      = XtNameToWidget(BroadcastRoot, "*raise");
    broadcastinfo       = XtNameToWidget(BroadcastRoot, "*info");
    broadcastinput      = XtNameToWidget(BroadcastRoot, "*input");
    BroadcastFile       = XtNameToWidget(BroadcastRoot, "*file");
    BroadcastSave       = XtNameToWidget(BroadcastRoot, "*save");
    broadcastoverwrite  = XtNameToWidget(BroadcastRoot, "*overwrite");
    BroadcastErrorBeep  = XtNameToWidget(BroadcastRoot, "*errorBeep");
    BroadcastErrorRaise = XtNameToWidget(BroadcastRoot, "*errorRaise");

    if (BroadcastSave) XtAddCallback(BroadcastSave, XtNcallback, SaveBroadcast,
                                     NULL);
    if (BroadcastFile)
        XtAddCallback(BroadcastFile, XtNcallback, ChangeBroadcastFilename,
                      &BroadcastFileName);
    XtAddCallback(BroadcastButton,   XtNcallback, CallToggleUpDown,
                  (XtPointer) BroadcastRoot);
    XtAddCallback(BroadcastRoot,    XtNpopupCallback, CallToggleOn,
                  (XtPointer) BroadcastButton);
    XtAddCallback(BroadcastRoot,    XtNpopdownCallback, CallToggleOff,
                  (XtPointer) BroadcastButton);
    XtAddCallback(BroadcastQuit, XtNcallback, CallPopdown,
                  (XtPointer) BroadcastRoot);
    XtSetKeyboardFocus(BroadcastCollect, broadcastinput);

    XtRealizeWidget(BroadcastRoot);
    XtInstallAllAccelerators(BroadcastCollect, BroadcastCollect);
    DeleteProtocol(BroadcastRoot);
    CallToggleUpDown(BroadcastButton, (XtPointer) BroadcastRoot, NULL);
}

static void RealSendBroadcast(Widget w, const char *Name,
                              const char *Text, int length)
{
    int         i;
    const char *Ptr;

    for (i=length, Ptr = Text; i>0; i--, Ptr++) if (!isspace(*Ptr)) break;
    if (i) {
        UserSendCommand(NULL, NULL, "shout %.*s", length, Text);
        if (w) BatchAddText(w, "%16s: %.*s\n", Name, length, Text);
        else        Outputf("%16s: %.*s\n", Name, length, Text);
    } else if (w) BatchAddText(w, ".....       Empty broadcast not sent\n");
    else                Output(".....       Empty broadcast not sent\n");
}

static void Broadcast(Widget w, XEvent *event, String *string, Cardinal *n)
{
    String      Buffer;
    const char *From, *To, *End, *Name;

    XtVaGetValues(w, XtNstring, &Buffer, NULL);
    Name = PlayerString(Me);
    End = strchr(Buffer, 0);
    for (From = Buffer;
         (To = memchr((char *)From, '\n', End-From)) != NULL;
         From = To+1)
        RealSendBroadcast(broadcastinfo, Name, From, To-From);
    RealSendBroadcast(broadcastinfo, Name, From, End-From);
    XtVaSetValues(w, XtNstring, "", NULL);
}

void ShowBroadcast(const char *Who, const char *What)
{
    BatchAddText(broadcastinfo, "%16s: %s\n", PlayerNameToString(Who), What);
    IfBell(broadcastbeep);
    IfRaise(broadcastraise, broadcastraise);
}

static void ChangeYellFilename(Widget w,
                                    XtPointer clientdata, XtPointer calldata)
{
    char **Filename;
    Widget Root;

    Filename = (char **) clientdata;
    Root = ChangeFilename(toplevel, "yellFilename",
                          "Enter name of channel file",
                          Filename, "filename", *Filename, NULL);
    MyDependsOn(Root, w);
}

static void SaveYell(Widget w, XtPointer clientdata, XtPointer calldata)
{
    Boolean                   Overwrite;
    static char const * const ErrorType = "Yell save error";

    if (yellinfo) {
        if (yelloverwrite) XtVaGetValues(yelloverwrite, XtNstate,
                                         (XtArgVal) &Overwrite, NULL);
        else Overwrite = False;

        SaveWrite(YellFileName, Overwrite,
                  YellErrorBeep, YellErrorRaise, ErrorType,
                  SaveTextFun, (XtPointer) yellinfo);
    } else {
        IfBell(YellErrorBeep);
        IfRaise(YellErrorRaise, YellErrorRaise);
        PopMessage(ErrorType, "channels were ignored so there is "
                   "nothing to be saved");
    }
}

void InitYell(Widget Toplevel)
{
    Widget YellRoot, YellQuit;
    Widget YellCollect, YellFile, YellSave, YellChannels;
    String Name;

    MyChannel = -1;
    NoChannel = NULL;
    XtAppAddActions(XtWidgetToApplicationContext(Toplevel),
                    actionTable, XtNumber(actionTable));
    YellRoot = MyVaCreateManagedWidget("yells", Toplevel, NULL);
    YellFileName = StringToFilename(appdata.YellFilename,
                                    (int) 'T', "Channels",
                                    (int) 't', "_", 0);
    XtAddCallback(YellRoot, XtNdestroyCallback,
                  CallFree, (XtPointer) YellFileName);

    YellCollect    = XtNameToWidget(YellRoot, "*collect");
    YellQuit       = XtNameToWidget(YellRoot, "*quit");
    YellBeep       = XtNameToWidget(YellRoot, "*beep");
    YellRaise      = XtNameToWidget(YellRoot, "*raise");
    yellinfo       = XtNameToWidget(YellRoot, "*info");
    yellinput      = XtNameToWidget(YellRoot, "*input");
    YellFile       = XtNameToWidget(YellRoot, "*file");
    YellSave       = XtNameToWidget(YellRoot, "*save");
    YellChannels   = XtNameToWidget(YellRoot, "*channels");
    yelloverwrite  = XtNameToWidget(YellRoot, "*overwrite");
    YellErrorBeep  = XtNameToWidget(YellRoot, "*errorBeep");
    YellErrorRaise = XtNameToWidget(YellRoot, "*errorRaise");
    YellTitle      = XtNameToWidget(YellRoot, "*title");
    YellModerator  = XtNameToWidget(YellRoot, "*moderator");
    YellState      = XtNameToWidget(YellRoot, "*state");

    if (YellSave) XtAddCallback(YellSave, XtNcallback, SaveYell, NULL);
    if (YellFile) XtAddCallback(YellFile, XtNcallback,
                                ChangeYellFilename, &YellFileName);
    
    if (YellChannels)
        XtAddCallback(YellChannels, XtNcallback, CallSendCommand, "channels");

    yellchannel = XtNameToWidget(YellRoot, "*channel");
    if (yellchannel) {
        XtVaGetValues(yellchannel, XtNstring, &Name, NULL);
        NoChannelLength = strlen(Name);
        if (NoChannelLength < MINCHLENGTH) {
            int Temp;

            NoChannel = mynews(char, MINCHLENGTH+1);
            Temp = MINCHLENGTH-NoChannelLength;
            memset(NoChannel, ' ', Temp);
            memcpy(NoChannel+Temp, Name, NoChannelLength+1);
            NoChannelLength = MINCHLENGTH;
            XtVaSetValues(yellchannel, XtNstring, NoChannel, NULL);
        } else NoChannel = mystrndup(Name, NoChannelLength);
    }

    XtAddCallback(YellButton,  XtNcallback, CallToggleUpDown,
                  (XtPointer) YellRoot);
    XtAddCallback(YellRoot,    XtNpopupCallback, CallToggleOn,
                  (XtPointer) YellButton);
    XtAddCallback(YellRoot,    XtNpopdownCallback, CallToggleOff,
                  (XtPointer) YellButton);
    XtAddCallback(YellQuit, XtNcallback, CallPopdown,
                  (XtPointer) YellRoot);

/*  if (YellTitle)     XtUnmanageChild(YellTitle); */
    if (YellModerator) XtUnmanageChild(YellModerator);
    if (YellState)     XtUnmanageChild(YellState);

    XtSetKeyboardFocus(YellCollect, yellinput);

    XtRealizeWidget(YellRoot);
    if (YellCollect) XtInstallAllAccelerators(YellCollect, YellCollect);
    DeleteProtocol(YellRoot);
    if (YellButton) CallToggleUpDown(YellButton, (XtPointer) YellRoot, NULL);
    else XtPopup(YellRoot, XtGrabNone);
}

void CleanYell(void)
{
    myfree(NoChannel);
}

static void RealSendYell(Widget w, int channel, const char *Name,
                         const char *Text, int length)
{
    int         i;
    const char *Ptr;

    for (i=length, Ptr = Text; i>0; i--, Ptr++) if (!isspace(*Ptr)) break;
    if (i) {
        UserSendCommand(NULL, NULL, "yell %d %.*s", channel, length, Text);
        if (w) BatchAddText(w, "%16s: %.*s\n", Name, length, Text);
        else Outputf("%16s: %.*s\n", Name, length, Text);
    } else if (w) BatchAddText(w, ".....       Empty message not sent\n");
    else Output(".....       Empty message not sent\n");
}

static void Yell(Widget w, XEvent *event, String *string, Cardinal *n)
{
    String      Buffer;
    const char *From, *To, *End, *Name;

    XtVaGetValues(w, XtNstring, &Buffer, NULL);
    Name = PlayerString(Me);
    End = strchr(Buffer, 0);
    for (From = Buffer;
         (To = memchr((char *)From, '\n', End-From)) != NULL;
         From = To+1)
        RealSendYell(yellinfo, MyChannel, Name, From, To-From);
    RealSendYell(yellinfo, MyChannel, Name, From, End-From);
    XtVaSetValues(w, XtNstring, "", NULL);
}

static ChannelData *FindChannel(int num)
{
    ChannelData *Data;

    for (Data = ChannelBase.Next; Data != &ChannelBase; Data = Data->Next)
        if (num == atoi(Data->Name)) return Data;
    return NULL;
}

/* Maybe combine with ChannelName ? -Ton */
static void ShowChannelInfo(void)
{
    ChannelData *Data;
    String       Current;

    Data = FindChannel(MyChannel);
    if (Data) {
        if (YellTitle)
            if (Data->Title) {
                XtVaSetValues(YellTitle, XtNstring, Data->Title, NULL);
/*                XtManageChild(YellTitle); */
            }
/*          else XtUnmanageChild(YellTitle); */
            else XtVaSetValues(YellTitle, XtNstring, "", NULL);
        if (YellModerator)
            if (Data->Moderator) {
                XtVaGetValues(YellModerator, XtNstring, &Current, NULL);
                if (strcmp(Current, Data->Moderator)) {
                    XtVaSetValues(YellModerator, XtNstring, "", NULL);
                    AddText(YellModerator, Data->Moderator);
                }
                XtManageChild(YellModerator);
            } else XtUnmanageChild(YellModerator);
        if (YellState)
            if (Data->State) {
                XtVaGetValues(YellState, XtNstring, &Current, NULL);
                if (strcmp(Current, Data->State)) {
                    XtVaSetValues(YellState, XtNstring, "", NULL);
                    AddText(YellState, Data->State);
                }
                XtManageChild(YellState);
            } else XtUnmanageChild(YellState);
    } else {
/*      XtUnmanageChild(YellTitle); */
        XtVaSetValues(YellTitle, XtNstring, "", NULL);
        XtUnmanageChild(YellModerator);
        XtUnmanageChild(YellState);
    }
}

static void ChannelName(void)
{
    char   Buffer[80];
    String Buf;

    if (yellchannel) {
        switch(MyChannel) {
          case -1:
            Buf = NoChannel;
            break;
          default:
            sprintf(Buffer, "%*d", NoChannelLength, MyChannel);
            Buf = Buffer;
            break;
        }
        XtVaSetValues(yellchannel, XtNstring, Buf, NULL);
    }
}

static void ChangeChannel(Widget w, XEvent *event, String *string, Cardinal *n)
{
    String Buffer, Buf;
    int    Chan;   

    if (yellchannel) {
        XtVaGetValues(yellchannel, XtNstring, &Buffer, NULL);
        while (isspace(*Buffer)) Buffer++;
        if (XmuCompareISOLatin1(NoChannel, Buffer) == 0) Chan = -1;
        else {
            Chan = strtol(Buffer, &Buf, 0);
            if (Buf == Buffer) Chan = MyChannel;
            else {
                while (isspace(*Buf)) Buf++;
                if (*Buf) Chan = MyChannel;
            }
        }
        if (MyChannel != Chan) {
            SendCommand(NULL, NULL, "; %d", Chan);
            /* Next line is needed as long as
               changing to -1 gives no answer -Ton */
            if (Chan == -1) SendCommand(NULL, (XtPointer) 1, "channels");
        }
        ChannelName();
    }
}

static void ChangeChannelTitle(Widget w, XEvent *event,
                               String *string, Cardinal *n)
{
    String       Buffer;
    ChannelData *Data;

    if (YellTitle) {
        XtVaGetValues(YellTitle, XtNstring, &Buffer, NULL);
        if (MyChannel >= 0 && Buffer && Buffer[0]) {
            Data = FindChannel(MyChannel);
            if (!Data || !Data->Title || strcmp(Data->Title, Buffer)) {
                SendCommand(NULL, NULL, "channel %d title %s",
                            MyChannel, Buffer);
                SendCommand(NULL, (XtPointer) 1, "channels");
            } else if (!Data) SendCommand(NULL, (XtPointer) 1, "channels");
        }
        ShowChannelInfo();
    }
}

static void InconsistentChannel(int channel)
{
    if (yellinfo) BatchAddText(yellinfo, "Somehow you seem to have moved from"
                               " channel %d to %d\n", MyChannel, channel);
    MyChannel = channel;
    ChannelName();
    ShowChannelInfo();
}

void WrongChannel(const char *Message)
{
    if (yellinfo) BatchAddText(yellinfo, "Channel must be %s\n", Message);
}

void ChannelDisallowed(void)
{
    if (yellinfo) BatchAddText(yellinfo, "Access is not allowed\n");
}

void JoinChannel(int channel)
{
    if (yellinfo && channel != MyChannel)
        if (channel == -1)
             BatchAddText(yellinfo, "You left all channels\n", MyChannel); 
        else BatchAddText(yellinfo, "You moved to channel %d\n", channel);
    MyChannel = channel;
    ChannelName();
    ShowChannelInfo();
    MyChannel = channel;
    ChannelName();
    ShowChannelInfo();
}

void RejoinChannel(void)
{
    if (MyChannel >= 0) {
        FirstCommand(NULL, "; %d", MyChannel);
        MyChannel = -1;
        ChannelName();
        ShowChannelInfo();
    }
}

void ChannelJoin(int channel, const char *Who)
{
    if (channel != MyChannel) InconsistentChannel(channel);
    if (yellinfo) BatchAddText(yellinfo, "%16s has joined channel %d\n",
                               PlayerNameToString(Who), channel);
}

void ChannelLeave(int channel, const char *Who)
{
    if (channel != MyChannel) InconsistentChannel(channel);
    if (yellinfo) BatchAddText(yellinfo, "%16s has left channel %d\n",
                               PlayerNameToString(Who), channel);
}

void ShowYell(int channel, const char *Who, const char *What)
{
    if (channel != MyChannel) InconsistentChannel(channel);

    if (yellinfo)
        BatchAddText(yellinfo, "%16s: %s\n", PlayerNameToString(Who), What);
    IfBell(YellBeep);
    IfRaise(YellRaise, YellRaise);
}

typedef struct {
    int            channel;
    const Player  *player;
} ChannelPlayer;

static void CleanChannelData(void)
{
    ChannelData *Here, *Next;

    for (Here = ChannelBase.Next; Here != &ChannelBase; Here = Next) {
        Next = Here->Next;
        myfree(Here->Name);
        myfree(Here->Moderator);
        myfree(Here->Title);
        myfree(Here->State);
        FreeNameList(Here->Names);
        myfree(Here);
    }
    Here->Previous = Here->Next = Here;
}

ChannelData *OpenChannelData(void)
{
    CleanChannelData();
    ChannelBase.Name = NULL; /* Here we will remember our own channel */
    return &ChannelBase;
}

void AddChannelData(ChannelData *Data, char *Name, char *Moderator,
                    char *Title, char *state, NameList *Names)
{
    ChannelData *NewData;

    NewData = mynew(ChannelData);
    NewData->Name = NewData->Moderator =
        NewData->Title = NewData->State = NULL;
    WITH_HANDLING {
        NewData->Name      = mystrdup(Name);
        NewData->Moderator = mystrdup(Moderator);
        NewData->Title     = mystrdup(Title);
        NewData->State     = mystrdup(state);
        NewData->Names     = Names;
    } ON_EXCEPTION {
        if (NewData->Name)      free(NewData->Name);
        if (NewData->Moderator) free(NewData->Moderator);
        if (NewData->Title)     free(NewData->Title);
        if (NewData->State)     free(NewData->State);
        ReRaise();
    } END_HANDLING;

    NewData->Next = Data;
    NewData->Previous = Data->Previous;
    NewData->Previous->Next = NewData->Next->Previous = NewData;
}

void         CloseChannelData(ChannelData *Data)
{
    ShowChannelInfo();
}

void         ChannelList(const ChannelData *Data)
{
    ChannelData   *Here;
    int            Special, NrCols, col, NrPeople, i;
    NameList      *Now, *From;
    const char    *Text, *WasChannel;
    const Player **People, **Ptr;

    WasChannel = ArgsCommand(NULL, "channels");
    if (WasChannel && CommandClosure(NULL)) {
        Text = PlayerToName(Me);
        for (Here = Data->Next; Here != Data; Here = Here->Next) {
            From = Here->Names;
            for (Now = From->Next; Now != From; Now = Now->Next)
                if (strcmp(Text, Now->Name) == 0) {
                    ((ChannelData *)Data)->Name = Here->Name;
                    JoinChannel(atoi(Here->Name));
                    return;
                }
        }
        JoinChannel(-1);
        return;
    }

    Special = yellinfo && !(UserCommandP(NULL) && WasChannel);
    if (Special) NrCols = 3;
    else         NrCols = 4;

    for (Here = Data->Next; Here != Data; Here = Here->Next) {
        if (Special)
            BatchAddText(yellinfo, "%2s %10s -- %s -- %s", Here->Name,
                         Here->Moderator ? Here->Moderator : "",
                         Here->Title, Here->State);
        else Outputf("%2s %10s -- %s -- %s", Here->Name,
                     Here->Moderator ? Here->Moderator : "",
                     Here->Title, Here->State);
        From = Here->Names;
        col = 0;
        NrPeople = 0;
        for (Now = From->Next; Now != From; Now = Now->Next) NrPeople++;
        People = mynews(const Player *, NrPeople);
        WITH_UNWIND {
            Ptr = People;
            for (Now = From->Next; Now != From; Now = Now->Next) {
                *Ptr = PlayerFromName(Now->Name);
                if (Me == *Ptr++) ((ChannelData *)Data)->Name = Here->Name;
            }
            qsort((void *) People, NrPeople, sizeof(*People), PlayersCompare);
            for (Ptr = People, i = NrPeople; i > 0; Ptr++, i--) {
                Text = PlayerString(*Ptr);
                if (Special) BatchAddText(yellinfo, "%s%16s",
                                          col ? "  " : "\n  ", Text);
                else Outputf("%s%16s", col ? "  " : "\n  ", Text);
                if (++col == NrCols) col = 0;
            }
        } ON_UNWIND {
            myfree(People);
            if (Special) BatchAddText(yellinfo, "\n");
            else Output("\n");
        } END_UNWIND;
    }
    if (Data->Name) JoinChannel(atoi(Data->Name));
    else            JoinChannel(-1);
}
