/*
 * test_plotmtv.c - program to test the mtv read and plotting routines
 */

#include <stdio.h>
#include <math.h>
#include <strings.h>
#include "PXplot.h"
#include "CNplot.h"
#include "patchlevel.h"

/* 
 * Definitions
 */
#define PSFILE     "dataplot.ps"

/* 
 * Globals
 */
static char   progname[CN_MAXCHAR];
static char   printer [CN_MAXCHAR];
static char   printCmd[CN_MAXCHAR];
static char   psfile  [CN_MAXCHAR];
static double scale;
static int    landscape;
static int    postscript;
static int    printplot;
static int    noxplot;
static int    verbose;
static int    debug;
static int    color_postscript;
static int    nodate;
static int    plot3d;
static int    nopixmap;

static CNplotsetptr plot_listhead;   /* Head to list of plots    */
static CNplotsetptr plot_listtail;   /* Tail to list of plots    */
static int          plotID;          /* Plot ID                  */

static CNdatasetptr data_listhead;   /* Head to list of datasets */
static CNdatasetptr data_listtail;   /* Tail to list of datasets */
static int          dataID;          /* Data ID                  */

static char *plot_foreground=NULL;
static char *plot_background=NULL;

/*
 * Procedure Declarations
 */
static void         usage();
static void         initialize_data();
static void         print_init_message();
static void         read_mtvdata();
static CNplotsetptr create_plotset();
static CNplotsetptr create_single_plotset();
static void         draw_view();

/* 
 * Main program 
 */
main(argc,argv)
int argc;
char **argv;
{
   CNdatasetptr dptr;
   CNplotsetptr pptr;
   char         datafile[CN_MAXCHAR];
   int          i, FILEFOUND = CN_FALSE;
   int          plot_all = CN_FALSE;
   int          plottype;
   int          err     = CN_FALSE;

   /* Initialize data first */
   initialize_data(argv[0]);

   /* Parse the line first */
   for (i=1; i<argc; i++) {
      if (strncmp(argv[i],"-verbose",2) == 0) {   /* verbose     */
         verbose = CN_TRUE;
         (void) fprintf(stdout,"   Commands: verbose         = ON\n");
         continue;
         }

      if (strncmp(argv[i],"-debug",2) == 0) {     /* debug       */
         verbose = CN_TRUE;
         debug   = CN_TRUE;
         (void) fprintf(stdout,"   Commands: debug           = ON\n");
         continue;
         }

      if (strcmp(argv[i],"-quick") == 0) {        /* program identifier */
         continue;
         }

      if ( (strcmp(argv[i],"-nops") == 0) ||
           (strcmp(argv[i],"-ps"  ) == 0) ) {     /* postscript */
         (void) fprintf(stdout,
                "***Sorry, the \"%s\" option has been disabled\n",argv[i]);
         continue;
         }

      if ( (strcmp(argv[i],"-print") == 0) ||
           (strcmp(argv[i],"-prt")   == 0) ) {    /* print */
         printplot = CN_TRUE;
         (void) fprintf(stdout,"   Commands: printplot       = ON\n");
         continue;
         }

      if (strncmp(argv[i],"-noxplot",5) == 0) {    /* noxplot */
         noxplot = CN_TRUE;
         (void) fprintf(stdout,"   Commands: noxplot          = ON\n");
         continue;
         }

      if (strncmp(argv[i],"-landscape",2) == 0) { /* landscape */
         landscape = CN_TRUE;
         (void) fprintf(stdout,"   Commands: landscape       = ON\n");
         continue;
         }

      if (strncmp(argv[i],"-scale",2) == 0) {       /* scale  */
         if (++i >= argc) { usage(progname); exit(-1); }
         scale = atof(argv[i]);
         if (scale < 0.1 || scale > 1.0) {
            scale = 1.0;
            (void) fprintf(stdout,"   Warning : Invalid scale -");
            (void) fprintf(stdout," scale reset to %g\n",scale);
         }
         (void) fprintf(stdout,"   Commands: scale           = %g\n",scale);
         continue;
         }

      if ( (strcmp(argv[i],"-colorps") == 0) || 
           (strcmp(argv[i],"-pscolor") == 0) ) {    /* color ps */
         color_postscript = CN_TRUE;
         (void) fprintf(stdout,"   Commands: color postscript= ON\n");
         continue;
         }

      if (strcmp(argv[i],"-nodate") == 0) {       /* no postscript date */
         nodate = CN_TRUE;
         (void) fprintf(stdout,"   Commands: nodate          = ON\n");
         continue;
         }

      if ((strncmp(argv[i],"-3d",3) == 0) ||   
          (strncmp(argv[i],"-3D",3) == 0)) {      /* 3D plot  */
         plot3d           = CN_TRUE;
         (void) fprintf(stdout,"   Commands: plot3D          = ON\n");
         continue;
         }

      if (strncmp(argv[i],"-nopixmap",5) == 0) {   /* don't use pixmap */
         nopixmap = CN_TRUE;
         (void) fprintf(stdout,"   Commands: nopixmap        = ON\n");
         continue;
         }

      if (strncmp(argv[i],"-plotall",4) == 0) {    /* Plot all */
         plot_all         = CN_TRUE;
         (void) fprintf(stdout,"   Commands: plotall         = ON\n");
         continue;
         }

      if ((strncmp(argv[i],"-pfg",3) == 0) ||        /* foreground color */
          (strcmp (argv[i],"-fg") == 0)) {
         if (++i >= argc) { usage(progname); exit(-1); }
         plot_foreground  = argv[i];
         (void) fprintf(stdout,"   Commands: foreground color= %s\n",
                        plot_foreground);
         continue;
         }

      if ((strncmp(argv[i],"-pbg",3) == 0) ||        /* background color */
          (strcmp (argv[i],"-bg") == 0)) {
         if (++i >= argc) { usage(progname); exit(-1); }
         plot_background  = argv[i];
         (void) fprintf(stdout,"   Commands: background color= %s\n",
                        plot_background);
         continue;
         }

      if (strncmp(argv[i],"-P",2) == 0) {         /* Printer */
         (void) strcpy(printer,argv[i]+2);
         (void) fprintf(stdout, "   Commands: Printer         = %s\n",printer);
         continue;
         }

      if (strcmp(argv[i],"-printcmd") == 0) {    /* print command */
         if (++i >= argc) { usage(progname); exit(-1); }
         (void) strcpy(printCmd, argv[i]);
         (void) fprintf(stdout,"   Commands: Print command   = \"%s\"\n",
                          printCmd);
         continue;
         }

      if (strcmp(argv[i],"-o") == 0) {           /* postscript file */
         if (++i >= argc) { usage(progname); exit(-1); }
         (void) strcpy(psfile, argv[i]);
         (void) fprintf(stdout,"   Commands: Postscript file = \"%s\"\n",
                          psfile);
         continue;
         }

      if (argv[i][0] == '-') err = CN_TRUE;

      if (argv[i][0] != '-' && argv[i][0] != '=') {
         FILEFOUND = CN_TRUE;
         (void) strcpy(datafile,argv[i]);
         read_mtvdata(datafile);
         (void) fprintf(stdout,
                "   Commands: Data File       = %s\n",datafile);
      }
   }

   /* Serious syntax error */
   if (!FILEFOUND) {
      usage(progname);
      (void) fprintf(stderr,"%s: Datafile has not been specified!\n",progname);
      exit(-1);
   }

   /* Incorrect specifications (non-critical syntax error) */
   if (err) usage(progname);

   /* Go thru the datasets */
   if (data_listhead == NULL) {
      (void) fprintf(stderr,
            "%s: Unable to find valid data in the file \"%s\"!\n",
            progname,datafile);
      exit(-1);
   }

   if (plot_all) {
      /* Create a single plotset containing all the datasets */
      plottype = plot3d ? CN_PLOT3D : CN_PLOT2D;
      if ((pptr = create_single_plotset(plottype, verbose)) != NULL) {
         draw_view(pptr);
         CNdelete_plotset(&plot_listhead, &plot_listtail, pptr);
      }
   } else {
      /* Loop thru the datasets and plot each */
      for (dptr=data_listhead; dptr!=NULL; dptr=dptr->next) {
         plottype = plot3d ? CN_PLOT3D : CN_PLOT2D;
         if ((pptr = create_plotset(dptr, plottype, verbose)) != NULL) {
            draw_view(pptr);
            CNdelete_plotset(&plot_listhead, &plot_listtail, pptr);
         }
      }
   }
   /* Quit */
   (void) fprintf(stdout,"Congratulations!!!  You have been TERMINATED...\n");
   exit(0);
}


/*
 * Print out valid command line options
 */
static void usage(progname)
char *progname;
{
   (void) fprintf(stderr,"%s [-Pprinter] \
[-3d] \
[-colorps] \
[-debug] \
[-l] \
[-nodate] \
[-nopixmap] \
[-noxplot] \
[-pfg foreground color] \
[-pbg background_color] \
[-plotall] \
[-print] \
[-printcmd command] \
[-scale scale] \
[-v] \
datafile1 datafile2...\n", progname);
}

/*
 * Initialize data 
 */
static void initialize_data(pgname)
char *pgname;
{
   static void print_init_message();

   char   *getenv();
   char   *ptr;

#ifndef PRINTER_NAME
   static char *printer_name = "hpps";
# else
   static char *printer_name = PRINTER_NAME;
#endif

#ifndef PRINTER_CMD
   static char *printer_cmd  = "lpr -h";
#else
   static char *printer_cmd  = PRINTER_CMD;
#endif

   /* Program name */
   (void) strcpy(progname,pgname);

   /* Print the initial message */
   print_init_message();

   /* printer type */
   if ((ptr = getenv("PRINTER")) == NULL) {
      (void) strcpy(printer,printer_name);
   } else {
      (void) strcpy(printer,ptr);
   }
   (void) fprintf(stdout,"   Default Printer           = %s\n",printer);

   /* printer command */
   if ((ptr = getenv("MTV_PRINTER_CMD")) == NULL) {
      (void) strcpy(printCmd,printer_cmd);
   } else {
      (void) strcpy(printCmd,ptr);
   }
   (void) fprintf(stdout,"   Default Print Command     = %s\n",printCmd);

   /* Global variables */
   verbose           = CN_FALSE;
   debug             = CN_FALSE;

   /* Plot variables */
   postscript        = CN_TRUE;
   printplot         = CN_FALSE;
   noxplot           = CN_FALSE;
   landscape         = CN_FALSE;
   scale             = 1.00;
   plot3d            = CN_FALSE;
   nopixmap          = CN_FALSE;
   nodate            = CN_FALSE;

   /* Postscript file */
   (void) strcpy(psfile,PSFILE);

   /* check the POSTSCRIPT environment variable */
   postscript = CN_TRUE;
   if ((ptr = getenv("MTV_POSTSCRIPT")) != NULL) {
      CNstring_to_lower(ptr);
      if ((strcmp(ptr,"off")==0) || (strcmp(ptr,"false")==0)) {
         postscript = CN_FALSE;
         (void) fprintf(stdout,"   Postscript                = OFF\n");
      }
   }

   /* check the PSCOLOR    environment variable */
   color_postscript = CN_FALSE;
   if ((ptr = getenv("MTV_PSCOLOR")) != NULL) {
      CNstring_to_lower(ptr);
      if ((strcmp(ptr,"on")==0) || (strcmp(ptr,"true")==0)) {
         color_postscript = CN_TRUE;
         (void) fprintf(stdout,"   Color Postscript          = ON\n");
      }
   }

   /* Linked lists */
   plotID            = 1;
   plot_listhead     = NULL;
   plot_listtail     = NULL;
   dataID            = 1;
   data_listhead     = NULL;
   data_listtail     = NULL;
}

/*
 * Print the inital prompt
 */
static void print_init_message()
{
   (void) fprintf(stdout,"\nWelcome to %s V%d.%d.%d %s\n",
           progname,VERSION,PATCHLEVEL,SUBPATCHLEVEL,DATE);
}

/*
 * Read data in "mtvplot" format
 */
static void read_mtvdata(filename)
char        *filename;              /* Data file to read             */
{
   CNdslistptr dshead=NULL, dstail=NULL, ds;
   CNdatasetptr dptr=NULL;

   /*
    * Read the plotmtv data from a file and store in a data-structure
    */
   CNread_plotmtv(filename,(FILE *)NULL,CN_FILE,
                  (CNdatasetptr) NULL, (CNdatasetptr) NULL,
                   &dshead,&dstail,&dataID,verbose);

   /* Copy the data to the local list */
   for (ds=dshead; ds!=NULL; ds=ds->next)
      if (ds->Dptr) CNstore_dataset(&data_listhead, &data_listtail, 
                                    ds->Dptr, 0);

#ifdef INTEL_ONLY
   /* Fill out quantity datasets */
   for (dptr=data_listhead; dptr!=NULL; dptr=dptr->next)
      if (dptr->datatype == CN_PIF_CHILD)
         (void) CNfill_pif_quantity_dataset(dptr,verbose);
#endif

   /* Delete the dslist */ 
   CNdelete_dslist_list(&dshead, &dstail);
}


/*
 * Create a plotset
 * Take a dataset and encapsulate it in a plotset
 */
static CNplotsetptr create_plotset(dptr, plottype, debug)
CNdatasetptr dptr;
int          plottype;
int          debug;
{
   CNplotsetptr WP;
   CNdslistptr  dshead=NULL, dstail=NULL, DS;
   int          contour=CN_FALSE, eqscal=CN_FALSE;

   if (debug) (void) fprintf(stdout,"Creating a new plotset...\n");

   /* Error checking */
   if (dptr == NULL) {
      (void) fprintf(stderr,"create_plotset() : Error! NULL dataset!\n");
      return(NULL);
   }

   /* Put the dataset in a list */
   (void) CNinsert_dslist(&dshead, &dstail, dptr);

   /* Create the plotset */
   WP = CNinsert_plotset(&plot_listhead, &plot_listtail, CN_NONAME, plotID++);

   /*
    * Set the options/properties of the plotset
    */

   /* plot-type */
   WP->plottype = plottype;

   /* Insert the dataset into the plotset */
   WP->datahead = dshead;
   WP->datatail = dstail;

   /* Check the list for contour type datasets */
   for (DS=dshead; DS!=NULL && !contour; DS=DS->next)
      if (DS->Dptr->datatype == CN_CONTOUR) contour = CN_TRUE;

   /* Check the list for equalscale datasets */
   for (DS=dshead; DS!=NULL && !eqscal; DS=DS->next)
      if (DS->Dptr->plot_pr.equalscale) eqscal = CN_TRUE;

   /* Set the plot shape */
   if (contour || eqscal) {
      WP->plot_pr.equalscale = CN_TRUE;
      WP->plot_pr.fitpage    = CN_FALSE;
      WP->plot_pr.flag = WP->plot_pr.flag | CNequalscale;
      WP->plot_pr.flag = WP->plot_pr.flag | CNfitpage;
   }

   /* Apply the properties based on the first dataset */
   CNset_plotset_property(&(WP->plot_pr),&(dshead->Dptr->plot_pr));

   /* Apply the view properties based on the first dataset */
   CNset_view_property(WP->view_pr,dshead->Dptr->view_pr);

   /* Set the viewport of the plotset */
   CNreset_plotset_viewport(WP);

   /* Return the plotset */
   return(WP);
}

/*
 * Create a plotset
 * Take the dataset list and encapsulate it in a plotset
 */
static CNplotsetptr create_single_plotset(plottype, debug)
int          plottype;
int          debug;
{
   CNplotsetptr WP;
   CNdatasetptr dptr;
   CNdslistptr  dshead=NULL, dstail=NULL, DS;
   int          contour=CN_FALSE, eqscal=CN_FALSE;

   if (debug) (void) fprintf(stdout,"Creating a new plotset...\n");

   /* Error checking */
   if (data_listhead == NULL) {
      (void) fprintf(stderr,"create_plotset() : Error! NULL dataset!\n");
      return(NULL);
   }

   /* Put all the datasets in a list */
   for (dptr=data_listhead; dptr!=NULL; dptr=dptr->next)
   (void) CNinsert_dslist(&dshead, &dstail, dptr);

   /* Create the plotset */
   WP = CNinsert_plotset(&plot_listhead, &plot_listtail, CN_NONAME, plotID++);

   /*
    * Set the options/properties of the plotset
    */

   /* plot-type */
   WP->plottype = plottype;

   /* Insert the dataset into the plotset */
   WP->datahead = dshead;
   WP->datatail = dstail;

   /* Check the list for contour type datasets */
   for (DS=dshead; DS!=NULL && !contour; DS=DS->next)
      if (DS->Dptr->datatype == CN_CONTOUR) contour = CN_TRUE;

   /* Check the list for equalscale datasets */
   for (DS=dshead; DS!=NULL && !eqscal; DS=DS->next)
      if (DS->Dptr->plot_pr.equalscale) eqscal = CN_TRUE;

   /* Set the plot shape */
   if (contour || eqscal) {
      WP->plot_pr.equalscale = CN_TRUE;
      WP->plot_pr.fitpage    = CN_FALSE;
      WP->plot_pr.flag = WP->plot_pr.flag | CNequalscale;
      WP->plot_pr.flag = WP->plot_pr.flag | CNfitpage;
   }

   /* Don't modify overlapping datasets */
   (void) CNparse_plotset_property(&(WP->plot_pr),"overlay","off",0);

   /* Apply the properties based on the first dataset */
   CNset_plotset_property(&(WP->plot_pr),&(dshead->Dptr->plot_pr));

   /* Apply the view properties based on the first dataset */
   CNset_view_property(WP->view_pr,dshead->Dptr->view_pr);

   /* Set the viewport of the plotset */
   CNreset_plotset_viewport(WP);

   /* Return the plotset */
   return(WP);
}

/*
 * Plot the data 
 */
static void draw_view(pptr)
CNplotsetptr pptr;
{
   int PXcreateXWindow();
   int fileprt=CN_FALSE;
   int prtdate=!nodate;
   int c;
   char error_message[CN_MAXCHAR];
   int  status;

   /* Check the pptr */
   if (pptr == NULL) {
      (void) fprintf(stderr,"Error! NULL data in draw_view()!\n");
      return;
   }

   if (!noxplot) {
      /*
       * X11 Plot 
       */
      status = PXcreateXWindow(pptr,progname,"",
                             plot_foreground,plot_background,nopixmap,
                             psfile, printCmd, printer,
                             scale, landscape, prtdate, color_postscript,
                             debug);
      if (status == 0) noxplot = CN_TRUE;
   }

   if (noxplot) {
      /*
       * Postscript Plot
       */
      fileprt = printplot;
      if (!printplot) {
         /* prompt for a print out */
         (void) fprintf(stdout,"   Type return (or n/q) to ");
         (void) fprintf(stdout,"send the plot to the %s printer :",printer);
         if ( (c=getc(stdin)) == '\n' || c == 'y' || c == 'Y' ) {
            fileprt = CN_TRUE;
         } else if (c == 'q')
            exit(1);
         if (c != '\n') {
            while ( (c=getc(stdin)) != '\n')
               /*EMPTY*/
               ;
         }
      } 

      /* print directly */
      PXplotps(psfile, printCmd, printer, 
               scale, landscape, prtdate, PX_EPSI, color_postscript,
               1, fileprt, pptr, error_message, verbose);
   }
}

