use strict;
use warnings;

use Test::More tests => 2304;
use Crypt::Stream::Salsa20;

# http://www.ecrypt.eu.org/stream/svn/viewcvs.cgi/ecrypt/trunk/submissions/salsa20/reduced/12-rounds/verified.test-vectors?rev=161&view=markup
# http://www.ecrypt.eu.org/stream/svn/viewcvs.cgi/ecrypt/trunk/submissions/salsa20/reduced/8-rounds/verified.test-vectors?rev=161&view=markup
# http://www.ecrypt.eu.org/stream/svn/viewcvs.cgi/ecrypt/trunk/submissions/salsa20/full/verified.test-vectors?rev=161&view=markup

my $pt  = "\x00" x 140000;
my $t;
my $last;
my $rounds;

while (my $l = <DATA>) {
  chomp($l);
  if ($l =~ /^(Set.+|End of test vectors)$/) {
    if (defined $t->{key} && defined $t->{IV}) {
      my $key = pack("H*", $t->{key});
      my $iv  = pack("H*", $t->{IV});
      my $keylen = length $key;
      my $ivlen  = length $iv;
      die "undefined rounds" unless $rounds;
      my $enc = Crypt::Stream::Salsa20->new($key, $iv,0, $rounds)->crypt($pt);
      for my $s (sort keys %$t) {
        if ($s =~ /stream\[([0-9]+)..([0-9]+)\]/) {
          my $data = substr($enc, $1, $2 - $1 + 1);
          is(unpack("H*", $data), lc($t->{$s}), "$t->{comment} | $rounds=rounds keylen=$keylen ivlen=$ivlen | $s");
        }
      }
    }
    $t = { comment => $l };
    $last = undef;
  }
  elsif ($l eq 'Primitive Name: Salsa20/8') {
    $rounds = 8;
  }
  elsif ($l eq 'Primitive Name: Salsa20/12') {
    $rounds = 12;
  }
  elsif ($l eq 'Primitive Name: Salsa20') {
    $rounds = 20;
  }
  elsif ($l =~ /^\s+([0-9A-F]+)$/i) {
    die "trouble with empty 'last'" if !defined $last;
    $t->{$last} .= $1;
  }
  elsif ($l =~ /^\s+(\S+)\s*=\s*([0-9A-F]+)$/) {
    $last = $1;
    die "trouble with last=$last" if defined $t->{$last};
    $t->{$last} = $2;
  }
}

__DATA__
********************************************************************************
*                          ECRYPT Stream Cipher Project                        *
********************************************************************************

Primitive Name: Salsa20/8
=========================
Profile: SW & HW
Key size: 128 bits
IV size: 64 bits

Test vectors -- set 1
=====================

(stream is generated by encrypting 512 zero bytes)

Set 1, vector#  0:
                         key = 80000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = A9C9F888AB552A2D1BBFF9F36BEBEB33
                               7A8B4B107C75B63BAE26CB9A235BBA9D
                               784F38BEFC3ADF4CD3E266687EA7B9F0
                               9BA650AE81EAC6063AE31FF12218DDC5
            stream[192..255] = BB5B6BB2CC8B8A0222DCCC1753ED4AEB
                               23377ACCBD5D4C0B69A8A03BB115EF71
                               871BC10559080ACA7C68F0DEF32A80DD
                               BAF497259BB76A3853A7183B51CC4B9F
            stream[256..319] = 4436CDC0BE39559F5E5A6B79FBDB2CAE
                               4782910F27FFC2391E05CFC78D601AD8
                               CD7D87B074169361D997D1BED9729C0D
                               EB23418E0646B7997C06AA84E7640CE3
            stream[448..511] = BEE85903BEA506B05FC04795836FAAAC
                               7F93F785D473EB762576D96B4A65FFE4
                               63B34AAE696777FC6351B67C3753B89B
                               A6B197BD655D1D9CA86E067F4D770220
                  xor-digest = 58499752D174577BFD3E4B5762576E0A
                               7D9C5314F320C5E25FCCC23540F52D48
                               4E8DB98AC2744A1B788A571422BAE8F7
                               E57859A4C44374797F0988D911A0DC73

Set 1, vector#  9:
                         key = 00400000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = EEB20BFB12025D2EE2BF33356644DCEF
                               467D377176FA74B3C110377A40CFF1BF
                               37EBD52A51750FB04B80C50AFD082354
                               9230B006F5994EBAAA521C7788F5E31C
            stream[192..255] = 6F397C82EC0D708CBE01F7FFAC0109EE
                               E7D2C564046CE22B8F74DF12A111CBED
                               9697A492C9147BFBB26613D8FFC29762
                               DA009207E2038F7BCE7FBB53BF1D6128
            stream[256..319] = 278CD2F0E90E66BCEF73D0FEB66FB5AF
                               F2F2083C1B6C462E1F1E6D864F6A7473
                               C0988F721AD673C23C4E70DDF67505AC
                               017B84DFF1983BD1ED81F8D64C8D9347
            stream[448..511] = EB55A9195DEE506F1C56E99DF24AE40C
                               F7F942B577BA241692AC85EACEE58B38
                               CE2F05F0E2C492D7FFAA07EF6CA36916
                               34BA12B68476C95F583F2723C498A6E7
                  xor-digest = A497B3521388286F6F6BC714F1F5F4E0
                               1BEE3FA18543B04FA454157C378B1993
                               616916B0601114F60516AAAE15B6A149
                               3CD0196BFDD26999162C3A29E91FF987

Set 1, vector# 18:
                         key = 00002000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 714DA982330B4B52E88CD0AC151E77AB
                               72EECEA2023139DA39FCCC3ABC12F83F
                               455733EDC22808318F10499EA0FCEEB4
                               0F61EF121C39F62D92CA62DA885BDF21
            stream[192..255] = 2EE0006A6B8F39086D981B9F332330C6
                               25531E002B8A28E7AEDA658D4A59558E
                               788CC2A24B073FA4E523F4BDA4EFF218
                               6AE54BADEA96283266035DCAB57CC935
            stream[256..319] = BD63C4E505BB28A14D2C25DA1A233905
                               578560105F7DF219653B8B3FB0436933
                               3B84259A1F2E866BC3F90EDC7192B03B
                               D83EEEB33CB9FD63D65BE8F5A8B93905
            stream[448..511] = 9DDBAAFAF7E490A32059C79D76854C17
                               0F3AEFAB39D3B16F964C9A36AB07DBA7
                               A66AF2D1C73DE7732DB4E0D51ABB71B6
                               56EF23A130C4C1DD901523019FC75ACD
                  xor-digest = 6742A21F8190E3CA5A8EBC1F454D4807
                               64646E8919D4E243241440CDB8F01576
                               67E79A4C5349DD1F9A8ACE3D642E59C8
                               101D5ED1D80F2C20DA3BE4FE57C86D3A

Set 1, vector# 27:
                         key = 00000010000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 78516F569C48CE781EEADD7861648833
                               3ED99FFC70DD8CA8D386486DAE2121A2
                               37FE4DBDFC1338BD603470533708A59C
                               526E3A0466A439BB10790101DA3A342F
            stream[192..255] = EF3CADD23A2ABFCB02835CD29777872A
                               CB36694FCC8CCAA1EFE4CABFD2446429
                               2E52D5EE2E71A372616D2E07A8054A8E
                               F6D37BEB5209B0D0858A5A97B0937F0F
            stream[256..319] = A65405017E6B9369E806DAB5B85E8051
                               17439EFE4E24E90D51A32D7B47B65AE0
                               8E9E216492CCF7AB090B0E03E15A6D85
                               E426D78568DC1BA39D4F710E39A4ED07
            stream[448..511] = 330885086270E10A64CF373C424A917D
                               DA6F4C0EA1A6FFBFF5ED1425301ABF3C
                               276B95CB3335F25AD506DC8D7E7CD9AE
                               DD67008E73CA62BDC5968F9C9C0C034E
                  xor-digest = B7D13D3900A87E835BF1715B0A6EBD1F
                               876F9678557EBBA267935B0CB79B71CF
                               6FEAF0EC2CFEDBB498B7D7AC475E6021
                               828965E6FF764517C88CB4536624BD16

Set 1, vector# 36:
                         key = 00000000080000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 452532EAF196AAD7D60FA945697135F4
                               35AAF74CFCC81675CC6782C184CF06B0
                               3F50DB531902395EA7EF9B35A8824CD0
                               7C6BA842EF47E50782A699B668A624AC
            stream[192..255] = 3B2F7C7E2A62EE9BD3B903A4F2B9E2F9
                               BF4E569CAB5CD01C9E42F30BA811720E
                               7961D33F1F9415CA7396F99A84CBF8FC
                               B21C458EEC06BAC69C4E53BDC111D975
            stream[256..319] = 59725D03330E7DBC611B8CACC980397D
                               0E65CDA80C69D2EB407F97F553B45FAB
                               98DF881797EECB167ABDDB89EA32F0C8
                               5D4F6E46B620BAC7F11AC63D32E995D8
            stream[448..511] = D66BC4FEC98FEA101E0513877A7C4BFE
                               D05D536F07F0F807057FDA44904EFD7A
                               8909FC37BF187D5FAEEE178594AE6969
                               685A8752174C1B6CD0F5B55B3727B308
                  xor-digest = 1ABCE128F08B654F6E7C7035393588B1
                               D4DB5DDAE571678A002F100C203D0E03
                               5FBCC21FDEDA27835E757F85AD8B8B43
                               07EF4D37F4CCE5A6BC6396AB442FEE63

Set 1, vector# 45:
                         key = 00000000000400000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 2FC78557A987F16837973BEA8DCD7D10
                               0F633BB08DD179A833FC540459837B23
                               24D58C89080DC685A06280BE0A0F6018
                               DB231C0F3CF2E70C4F6FD7659516D634
            stream[192..255] = 4E4526B3D9A1BD18C1A5C71F299465DF
                               9651F8EA8F0FA68084392AAA6FA5D8C3
                               69568941C36A7FD27ECEFBB15FF0B4AF
                               C71B158445B8B10C89AE2CFA4B34A486
            stream[256..319] = C1AA4A8C888AE011D95485D48B94A1F9
                               F21E3E6FEA9C7B52477644F69E2D9F4D
                               7EE1F8BF0A7E1E846DB6A3715D6A3FAD
                               5C5AA5C8D9AEACA85486B9A946403F3D
            stream[448..511] = 9F3198790884423AAE689D697014A25B
                               7007276148D33CDAD09500E4A954A357
                               FA82D9B5407D66295D528945107F12E0
                               332819DFE1CEB91FE5029590418629A0
                  xor-digest = 9871E7AA45717A2B0C54D4CE847F382A
                               D64F04E93D10DC70D6ADAD64733BDD5C
                               C1E54E3994E8402741487E4302D4A51C
                               492C5B0E2DCE13E013A15315C755F405

Set 1, vector# 54:
                         key = 00000000000002000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 82D07E74D2C06674F745FBF3FB0555F4
                               9C337A062310BAB83A8510CE2F4A8A9E
                               D460F96CCD9C91C37433BAC0A0E0040E
                               66D8ABE66A4FAAF7E28D38976F6A17D9
            stream[192..255] = C23B0C1FE9410B7D03B8B3395FDCE2A0
                               CD98B7FEDE9701D1E163115E14E7B888
                               62A9DF85606808DDB1C78AD618A99235
                               70370315E803509B207891646E907A8E
            stream[256..319] = 0C53C196F0BF02B99E07C38D669C9997
                               703D37F38932F49DD4BBCA62394C9C82
                               B2DE44E076B3614D7D8A6F37A2D6129F
                               FD9998BE11FE5A362693DD892188B58A
            stream[448..511] = 9ED70CCE2359423A06341EA80A0A19F4
                               CCE223A6260E1361AFB48269CBAE6CFF
                               E55F69AE3F7CA1C5DBAD2FB4C40A30FF
                               5346150168BE038671C8E591A41201BD
                  xor-digest = 3B6316D341EC17A767B7582A2EE3A1CF
                               A730C47CB02E92E8FDA06C7210619FEC
                               4003C940A7F08F72183D2B3BE0F04BC2
                               DEB4743B5E0E8F2B5911883AC76BB8A2

Set 1, vector# 63:
                         key = 00000000000000010000000000000000
                          IV = 0000000000000000
               stream[0..63] = 157890C4B0087062E201648A7E96F342
                               265863A22636A38B3BA67CD21B4CEAC7
                               7EF495EE6572650B1013BA5F62647240
                               CC73360159EF7E7201A29844ABF2C13E
            stream[192..255] = C82BC4AF671087758641B264EEACD0FB
                               56E67D2774BC48E9714B284CFAD09C8C
                               2E4CE5CF2965FEE7772173B0E386A0BD
                               C5E1D7112A7E8F9DDAA3073011C237C0
            stream[256..319] = 2AC3EC2897DCDB2FC2A25FEAD30D24FE
                               FDEE2D08FE45BCFE07799CF33EC3A051
                               099F12316EC0FB948A6640D013782859
                               A43BB5270B252EE04AC0112E2C9A29C9
            stream[448..511] = E03777A666E20A55DE01AFDD7DCCE004
                               CB77128738191A679A8A206E34F1210F
                               12D5AAC3FB98CDA2D4E817F35509975A
                               A0F215AC8F06FECEC0A31C3CA72ACC42
                  xor-digest = 8A61D3F972A404EDEFEC35F6AEEF6ACD
                               E36A3F0F1EC671D0FCF22D041A45687A
                               55591C70CD120BF1A08F91EA17F0C47C
                               DFBBB5683F2AB4756CB030BED6D55047

Set 1, vector# 72:
                         key = 00000000000000000080000000000000
                          IV = 0000000000000000
               stream[0..63] = E4522CCF0A00DE07D16CD3214FF41A0E
                               49145F226913A741E86673F4732ECA30
                               FC7392785E2E77F8ED01001BB9CB63EB
                               6447DFB8AF742A046F46B1D65872F5E9
            stream[192..255] = 5A93ED214863CF1B4B67010C1998D2D6
                               337286B51508019C9DC8393BF5BF1E83
                               74C862094BD44C1ADBD4DEC4328AE947
                               49B496E754ABCF153ED604FF0BA38229
            stream[256..319] = 965BBE8F68A976288316F729147E5519
                               7720CC7A2D6C90CC65AABAC772C3FE99
                               68B6DB151929CA66BC92B155B1EFA110
                               90AFF664DBBFB8E92D0361511CEDB20D
            stream[448..511] = 96C3CED39C72409AB659409B4364797A
                               9C3117F6BE80D5050481129E465358CB
                               81AA03DF85CB979AA5AB9D9F75F07E88
                               9B3269B849C07C1F7F2ECD3E5433F2DA
                  xor-digest = F951CC778BC0E1217736CD2AD309ED90
                               A15555CEA11E13BF71AC0AC55A585001
                               F8701AC1C0208863DE3614113C1C3ED8
                               A2C7E7C84B9B3A702DC2264A8D9A807F

Set 1, vector# 81:
                         key = 00000000000000000000400000000000
                          IV = 0000000000000000
               stream[0..63] = 05FF9B90485DE76693FE86F109D6F9AF
                               E445C851DC1DA1ACD43F8F4724A0DF74
                               5BB5413455D490229B139E378D28098D
                               63280A28F6344384A45D80151671BA03
            stream[192..255] = 0517E140CD1157EDEEA5236AB8EBC532
                               C95185527DD506DBE60FFC6A96B84CF0
                               2865AD5FE3E9FAEC599B4E0B4858A9B7
                               40EDDA87B8646EC77D0A89D46FA8C8C4
            stream[256..319] = 7745361F8E28CB05D3188CB4CF114A94
                               D03A66EBCE7ED5A49738B352679509D2
                               0BF9502FF9DA729528D4BFE4FBD10576
                               697B8F90585EEBB19625EA94E09B1967
            stream[448..511] = 68CFC4F550FCFE8644F95F7618ED59BC
                               D3C7C215C961B891284C784CCCA29432
                               040CFB4016D36E992C762F100A0EBEAA
                               4E0254FAC3129FDBAB85165AE13286C3
                  xor-digest = DFD7DEFC02C4A76B86EBC3512F29BC60
                               64F3DD92D002094065F9A02DB3585845
                               D21708A27326953E67C650F6AEDE0A8D
                               2E5CAAC90E41CE94AEF7EC6D5EE9455D

Set 1, vector# 90:
                         key = 00000000000000000000002000000000
                          IV = 0000000000000000
               stream[0..63] = 4E4C778B3C0DFE1D34FF9A616ADBB205
                               AA2855AB4415998F3DADD3E111DC322A
                               E00BFFCBC300DDED661BFC82D63706ED
                               B769559BBAA023003B590CA5537C99CB
            stream[192..255] = 3EBEC9B4771FE7D30205E323ED0B084A
                               2BB1C1559E68812611AED24AB20F1B43
                               CA5E39D517B98DAE88CF03F122D4BA85
                               A7DC86A19D426E1330060B1A8248F5E9
            stream[256..319] = 8B0547A837188E323861661F659F67C4
                               81DDB19DEDB33B4E886321175CA51A3E
                               267C888029BDD7454D20906129369F7D
                               302C8692867FAB7B27C9315E99472D03
            stream[448..511] = 2B03F0DE389D8749C3B23AFE8822D0EF
                               EC2E4F02483B062A6FFEB8EF495DCA15
                               E8F46D5CB53AB7AE8974872253DCEB71
                               18CA37BCD610E831445BF68FF475FD2B
                  xor-digest = CE69F2238067D38F5CB4122CF79CE481
                               A864C8177601A4DA56E602EDEDBB6639
                               B6B9185B91A58AE4D49DBEFC43C2D5D3
                               6C94D47819543E41060BE8BF5461646C

Set 1, vector# 99:
                         key = 00000000000000000000000010000000
                          IV = 0000000000000000
               stream[0..63] = 0346C54A065AC9779940C3F1674258CA
                               603E7B78C4022D4984C2794F6C72FF2D
                               E06D1CD8DB1C8B616FD050D1A5E54D93
                               56D452F817987D56906B0C6DFF97D395
            stream[192..255] = E54D565152A5B776BC921B05170AE1E9
                               C7E7B33EDE88B74286FA0686649D1A49
                               3FEEFD6227D63A6E3B8C3B3977D426BB
                               33EFD59CBF0B393807218AE891BD29AA
            stream[256..319] = 37DAACE15A677EA6E7E2ACC3A897B4D3
                               EC8B0C32F7D18A2245FC337C21F5DF9C
                               1C85DA16040468B813D765D63DB1BCDE
                               57C6B41A4A3FD5523B504ADE2D2C5099
            stream[448..511] = D05E56A10EBCF446E3E07BDC88B26B0D
                               85614AE78F2DA6ACAA1F7C6375CF47FD
                               584FCEE4CDD646F495830E0AD49DA4B1
                               96CCC074B67E40C4E15CF96B50D227F9
                  xor-digest = 125BCE67E0A4D1BC9F3E99F989C41019
                               41E75976D30FB7F492F33F79CE1E281B
                               994C0B4AE17331E102D2E850C65CDEE1
                               621C53F0DE4710933C8D9933EB97C3DF

Set 1, vector#108:
                         key = 00000000000000000000000000080000
                          IV = 0000000000000000
               stream[0..63] = FEC1757CB5722AAB452B1E7E8A7382B4
                               9B72AB16FB1C7C50E0960943B417B976
                               5E8620899715BF97FC41D865D8C9F38E
                               FE79D39A56007706D96649C88561860A
            stream[192..255] = BC9007C6DEC316C5DB279B18DA5C07E8
                               70155DC032B8EED16DB6FEAD5784534B
                               08A09882C8E9F7791DBC1A7512D27B25
                               D42F80BE61FB94EE643082DF4FE167F3
            stream[256..319] = B4D9BDEE5B73165AABFCE52C5FC338A0
                               C0BF57A985E3333377D463B17EF5A02C
                               4475508FB697A7F0130AC88E8A1D4A84
                               281AD13ECC7191C598366686DE55DC66
            stream[448..511] = E2F99407F98FCA990319231920A7A9FE
                               0E780E12F4822D600D1401DF5E8B6E1D
                               77FC82FEAC53A52C5CD14D875C4F6326
                               BA8F8EF706A4A9CA0873F9F06D69DEA0
                  xor-digest = D69030D6BC9386A6EFDD404F82BF938E
                               061C27BDA5EDA09C674B8E18FF3EA236
                               418A76D9C4038A198B8E238E6C9C7111
                               EE98337C7EE70192D30FFA9FF7A1E672

Set 1, vector#117:
                         key = 00000000000000000000000000000400
                          IV = 0000000000000000
               stream[0..63] = 2DC1D28AD3537E7AECE5A7BD757B0E56
                               E75EE869BBAFAA0B148E24FAA89D6209
                               11C01825869209B0905D225B6A71358E
                               FD1F33011ECEB81F918293E77C6CBC69
            stream[192..255] = FD3C8BB4472FCD8E5D3DD97B1FBF9712
                               63C97DDAFB105BAABA24FB9B409B963F
                               DB7790F26E19DA0B41B3A4023BC3FA16
                               6EE4BF97BD5FB119671B34F9004F58D9
            stream[256..319] = 545A1D0EBDCCC12F4E9F802EBE828C1E
                               C8DC448DB801C6C48F1B4FABC6D09643
                               FD76A68B52DD5AF8D4734924A6B3F0B9
                               AD7BDE848477CE9580D101BD37DCA52B
            stream[448..511] = 1E1D1E8DF03F3F48DD41B147FD6A8A63
                               ABCA633CD576B5F4767ABEA5CD721CAA
                               DFEE6FE44D107C29256A6D65284B883A
                               1C0CDF3B5E4780FD385DF19EEF1C412F
                  xor-digest = ADBEE1D8C71C5F4F5916FF689121C396
                               600702AABB23986678FA0EB4A8650306
                               E1CD3A4135B1DC429C63504A6A20E89D
                               D969F703CAA56EE4B9C0E507CA098DA5

Set 1, vector#126:
                         key = 00000000000000000000000000000002
                          IV = 0000000000000000
               stream[0..63] = 06C80B8CEC60F0C2E73EB6ED5DCB1B9C
                               39B210F1AB76FEDF1A6B7AE370DA0F20
                               0CEBCAD6EF6E57AC80E4375C035FA44D
                               3AE4DC2C2507757DAF37B14F36643489
            stream[192..255] = 218AE79578CA357BB0E74152C1B63258
                               38E1F8DBA5B43515571905B51EAD76AC
                               2BDEC397CC153E07EF7A956799157129
                               9FA3CC973C64E10C0A3E97B9CC6CDE4F
            stream[256..319] = BBA90DD4C0BDA240EDE42069A94DFE3F
                               C36EB12C62C993D836957BADE0DCA961
                               58486165F546894385166E1A5EBD7CA4
                               BBDBC03DE417E811FA6621EA64A9080B
            stream[448..511] = 7974CB949C44C149393BC9F10BDFB541
                               D4260C71BC52C90D52F9AC71A6F8BC97
                               11026831159AF43986EF161DB3762471
                               91C86F6DA07BAC1F496A9D1D73F7DF25
                  xor-digest = D5BBE5ACC61FB80E3001E4E0B2C1DDC5
                               7008D8510B70E8DEF16D58773EECDFA2
                               01027B3961174E3390FEBC2B15581F46
                               ABEAD4C0E7F63851EB6891EC86C20CB6

Test vectors -- set 2
=====================

Set 2, vector#  0:
                         key = 00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = A39D5AF8FE112FAFBD6EAE7D4E925708
                               BA05F9A3C2E24BA025A55C9AFD468F03
                               77692205C44AE9DF37B88E30CB5526A6
                               2D8699A37CC4457793F59AC1592A5998
            stream[192..255] = 1ED0FFAB9657CC6DBE4D83A5B7A4238D
                               30D7118FFD2D651E2B11822AAE986C35
                               FB2469A7FCD62F9B80F242B75C803D0E
                               47247FBE9E8E5A0EC7354707D837A54A
            stream[256..319] = F7863220BBFD89CBB0F04C030102A90E
                               9F33C525A85BACF3C3211EEA56F27492
                               3EDB8660AE792F750697BF4004C1BA3B
                               C5A8B1F47CCF9974EDA1308024D5AB30
            stream[448..511] = 41154479F67F5A8E0FEDF10E5D1E2576
                               FF4B4692C5B2ACA5C7851E7F02D935A1
                               37BF2F7104ADB87EE864AA01546E108A
                               8D2CD6FCB69589FB2E38210C1256DC7B
                  xor-digest = CEE35F4B1A524595ED606972E4E8EF4F
                               2F4086C0A9A6C4EA7A511100D892B4E8
                               3DD840342D363C265FAAF90CF140C158
                               C5BBA92B975A4AE1A46A4174B6CFBD3D

Set 2, vector#  9:
                         key = 09090909090909090909090909090909
                          IV = 0000000000000000
               stream[0..63] = BB5DA95AB6911162EAB6D25F4C6A08B0
                               C28FE8DB7F8D0F153D0340FC7E3E8E94
                               2ABAB31A218BDE0C899F72F1338A4C0C
                               8194F8958740F4BAC3EAA43B5313494B
            stream[192..255] = EF970695F118C274652E9C4DBCF1BE79
                               D5872DBD4EBE88BC02930DC20649B98C
                               05310BC16CD4678394CFF0F7EF10449B
                               971497EFDE0CAFAE4FFACE8E4893EA0F
            stream[256..319] = F9C403AD799743777B6A263E31E312E1
                               7955728FCECCBDD0416BDE77600A6B45
                               03CB7C6F3FC1624057F2C04CFB23C709
                               F8F8BFE43699AB0503FD4B45B2484C37
            stream[448..511] = 75BFAC8776A7B48494D779093A796204
                               17727B72184850E6B4C4EC03C7A1DCD5
                               0D60888C32E36478B49270A678A09571
                               3C580DE6CB791A7AE9FD76DF7A1FDF45
                  xor-digest = 8E7052ADADA2BBD0DF15FB28EDE8381D
                               FB7CB35D04A0662A8CD3E06CAB944496
                               CBB0C69A13DC97E6D97C3934211D30FF
                               19585C92C9A27265D4459221C6CED840

Set 2, vector# 18:
                         key = 12121212121212121212121212121212
                          IV = 0000000000000000
               stream[0..63] = C63D8BEC3A6F864E72486A71D45D7C2C
                               C2CA3595793A236044F18AC52EEF8A7F
                               9957C96C06BF11ABD673EC07F22924AA
                               F242804E87BC7488340B036DAB4E31DD
            stream[192..255] = 563460549F78966F8270BD0308A757D9
                               6FB48D03A37F7B4303ED12C4855A600E
                               01664515817095F04C8A341DFFFE252D
                               0C82FFDB70B51EDD30407364134D5054
            stream[256..319] = C6249019D499D3D663AE2E1F0A211EF3
                               57537D2BAC64A918E19D636014DDF27A
                               E8021EC6FB4EC8E4BE91ED37C8672310
                               D133E1986113E0A7C862B7B18DF2BCEC
            stream[448..511] = D3FC7786DD5CD0029726B72A33AE5363
                               8026492F63A0BB934BCE10BB15BAEB99
                               1D06F78DC59BB00FEE2A69C37BDD594F
                               57F4CFD67A5AD988EAE5C185D49CE89F
                  xor-digest = 61D337375308A033C556265C7B09C55E
                               9EE682FD7E2A61F92DD1926654BBCA90
                               1FF95549B35C22E149035674653F3A68
                               84883560E37F3F99CAD268BC1AE7A144

Set 2, vector# 27:
                         key = 1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B
                          IV = 0000000000000000
               stream[0..63] = D374D11AF756CEB281472D57CD02DB43
                               E2AE871E1DAD4ADD8F8CA1AD9AA12ABD
                               37F9369D0D8BE2D8748D30CEBA091667
                               8BE57241B456AE2719256FDE657D2ED2
            stream[192..255] = 4E75CF9C17BFD6AD5389864F22A2BCE6
                               1CF09E0FDE30828022758078A624824B
                               0D5ED4B67E3CB0CE2343A078AB3618A8
                               5AC854862BAC0A097A588B3D4DDE1C88
            stream[256..319] = 58F12B730DB83F43B9E052F39BBC1348
                               2BFAA6606700D30CB2361A667111C674
                               F0DDBD8033EBE4C4541BCB6C98757FDB
                               7D36E9DE27FCEB7AA8F2D878E2B69E28
            stream[448..511] = 1C6217BFC62A7E4AF4B003C16F305680
                               12F72C66369591A7C6EF6C520DF6285E
                               BF17FF18E19ECD33D8260D7A60DDED46
                               08720A3917390A79485A600631085165
                  xor-digest = A8E06B465D709568A8800C803338BB67
                               3E901ED8785491BDE31FF991F7EA3536
                               D5B57AAE2DEFA64CDB897A8C03BA5F2C
                               91404DE23728C38A2D5273C71560F133

Set 2, vector# 36:
                         key = 24242424242424242424242424242424
                          IV = 0000000000000000
               stream[0..63] = 88FB4992E84E90EA3DF9AD945224DC15
                               E0DB77BC715E2DC155862B3EFCFF9067
                               EFE51ACDCEDD5A9D982126963B254671
                               A599BC89C8747C8E54E826D7C20A959A
            stream[192..255] = F6BADC2C89AD8496EB5FAB4767EDE96F
                               4DCB9FF3BA6760AFE2E51ED0AEE46014
                               076AF626A23E861A0558BEF12AC50A10
                               B20BC66D5A8CF35CF14D42EE20F794FB
            stream[256..319] = DA90E5DFA6C82228571367DAE0F0EA74
                               BAFFF758B750A6A4AB27C1D2702CBF01
                               BC5A4C773E8C9B374930F8BDB956CA7E
                               0BD7F7BBBB1949022C359BB07E33075F
            stream[448..511] = 831539EC87739BD1FCBC42FDFCDB7CA4
                               38C306D2F4F791A50639843E43BFDA1A
                               65E852A1047F129941657D94B97A2B37
                               B4A725DF800CE4FE282C8D3CDC32CDDE
                  xor-digest = 85E747BEC4D893F24D6B204D171AF40C
                               5A42E7CDBEA6C9F30F9BB3F392EEC49A
                               65B74A67D76177B25DB2CC809490B131
                               E7C14348EF88D97E700D377670D82511

Set 2, vector# 45:
                         key = 2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D
                          IV = 0000000000000000
               stream[0..63] = 830A4E757568CA23C1D348A5AAE0B907
                               C2944998C859D78E3CDD52F5FC688060
                               66CA8FDB40AFC236B0296AED171DC68C
                               BFA377D434837A7AD1D9D5DD9971834E
            stream[192..255] = 14F62A57BBC81B437DA25D8370C8B260
                               147BC758F4A492F6657C9572B4696909
                               1E09618D36D551353829EF952BE7DB6F
                               401CC74F59E80B0BABF1A4385CF4C53F
            stream[256..319] = 0F247823939030FE792CCE36CEE0439B
                               17191B5C0513B5B09E90217D5F3AD654
                               D6F69C6B0212AB0B681DB7B16C7F3171
                               7B2B897FD605834288D9EED181376E12
            stream[448..511] = 6174EFDCE81F3A1844477D14EDFB18F0
                               36DFF353C9669E88799E8DC08C71377A
                               4E499BC561BDEEA32F01B7C91980C50E
                               323D6483B73ABD6252637408F5FA30C4
                  xor-digest = BF961EAA26648407AA78146F136CBF30
                               545BCED900B3A4E5636E26AC10C04C4F
                               04CF27A65D01078F1BEA30C2F9756947
                               402830CFD923432B378E9607C0EC4EB1

Set 2, vector# 54:
                         key = 36363636363636363636363636363636
                          IV = 0000000000000000
               stream[0..63] = 23A9EBAC1F774F95B3D611D6E23496B4
                               01D2B70D598539BFA1C0ACCAF8D93D9D
                               AE41675D63C9069E2E07ABDBD1386F06
                               515FF17F0E60E136059CF9E49845CB7B
            stream[192..255] = 97FCDE1AE674DC70E6B5C410698F09E8
                               3B735A8A76A6E08DF50496F2DB8FD63C
                               A97FD4FEB7C6EE08CC5968AD8F2D816F
                               B1E52C1211D61EA916846206A54503B9
            stream[256..319] = D6949609665B062DEEFDEA52DC769AAA
                               F4B86A7643A10F1D4BEAE0E4DB39D733
                               24632215963BFB23CA2D3BAB3E84CC13
                               83F2F43DE60D2F8AD78867A6F236C364
            stream[448..511] = E1D18B9D4BE9B49A14A202325BDD933D
                               15905B210F4C9A89F3092824C3440E49
                               F484C00B08958765B75BF654E25DEE47
                               B552411BCC75E2D1EEF34C72B0E0BD31
                  xor-digest = C958BFA65C96B7EEDCCA22BC9AD63FD9
                               22A94D3F6EE59E285C0C736AFE83769B
                               5DA2E16EA79009B6729EF005EEE38B20
                               3F33FFA70EC07E0E5465CF292C942886

Set 2, vector# 63:
                         key = 3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F
                          IV = 0000000000000000
               stream[0..63] = 46C516660185D516FC2ACF6A0C759C03
                               A3BB36BE93F544FDC9DA3B8DF4EFBC17
                               F9C1533061065B557A511B8C2BA33436
                               524AD2E7214A6004C37DCC61DF6EBADE
            stream[192..255] = B93519A1F798A4D1C44B5AA6509AC8A1
                               26F7B632BC55127865AC343AF70D5659
                               8FC2C418047F934C22660D4D1D03F3CC
                               84FB07231576CA4AA6EE598094B5E133
            stream[256..319] = B28549E92A6447859D829F25AC5AA670
                               08D0463DDAE8F1F99D18F391584D3A36
                               22845AA770176F41660C69A8E8D9D4DB
                               CD5EA82F1C696AD30B6CC2ED880FC8FB
            stream[448..511] = A37AA3AB4501857BB177C6578F74A399
                               7F0F9B624A29986C954F51E7223DB7FE
                               A55976940EFA83D479083F87C416893C
                               1B344D55497525C86DBB4FAFBD0C050A
                  xor-digest = D56F29DD8D7579987A8002A0B5264E3A
                               A187C61ABC77F6F5D7B7BB19B7C3E601
                               992AAAE92F736CB67F49F705859582B6
                               987815955F15A163E012841D80B57151

Set 2, vector# 72:
                         key = 48484848484848484848484848484848
                          IV = 0000000000000000
               stream[0..63] = F79DF6AAA57B1F45D1125990AC02D96B
                               F956AF2ED8C54332178F6EECF4598E74
                               AA1F536EE22E784B45BFB083A60E3828
                               7A360C06BE1EA762546AB0AD6899FE20
            stream[192..255] = 8D6D8DC5D1133FC3915D8460456A2094
                               AB68B9569D42A400185551141BE60851
                               7628970C1B46137FAEFFBDE6BCDD7916
                               AE114475C55378C26B2C00F5D0A365DD
            stream[256..319] = 3BAF985ED8A33AD9FA7A03BB0508A228
                               88F9DAD037D5B0EE2A89AF15217FBEFE
                               284D8EF5C166E4FED8157B61CEA4316E
                               CEBA1852A058AA4E93ECAFC37D67EEB8
            stream[448..511] = EDBE4C8FF3AF3278765A9340A927C5AF
                               DF9784D7A11E8C7B7985B972B84096BF
                               4449BFD0A01443F59644B217C4F07BBE
                               1394996F1A628042D82B78BC03817244
                  xor-digest = 7761F4B5E1B2D6FEFB0299FFD8B7AE64
                               4B0820E32E577325FE4E5BF5C129ED1C
                               E95FA763E48F68BE3D6104904D56236C
                               0097210AED56853638339518385E1D1C

Set 2, vector# 81:
                         key = 51515151515151515151515151515151
                          IV = 0000000000000000
               stream[0..63] = 7C436ED4AA18C3FD3442737D2425FD08
                               479604F203C057A080DCD7DA6777D0CD
                               E1DC0DF50A18F210C7C34F0A2F8FBA63
                               A132E6AF9F143550489E277CB11B899B
            stream[192..255] = 2FA20A460467D3213675805154EA262E
                               69BD8AB67D3CFC2054D18AACAE846D60
                               34DA1F11EA767C54EFE75672F3AD1835
                               81AC23D9A3B6170939528E31ECD451C4
            stream[256..319] = E553D095C339720C0149CFC37371CCF6
                               B2CFE5DF88117E0918D0C36A33662CEF
                               E986F8992BA20C53237A16552A31609D
                               68BB27E9FF6851814924805103FD95B8
            stream[448..511] = 85021EF68D4DCDA72CF66C2AE7C9B51D
                               4EC54DDBF38EB4CAD25AF75BA26F03E5
                               7E7CC9E56ECAD33A31B2255F0EC91149
                               EF5D4B163F593595B70E1DF1D4F1C2BE
                  xor-digest = 848894EF165D6833E75C2EAC631E9384
                               2504D1C834C111B02EAB2CA558D2C54C
                               AF282C2E60D49B7DDDF0479C0D91A89C
                               CAEC9C46DFB203E46027D3A6635F0131

Set 2, vector# 90:
                         key = 5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A
                          IV = 0000000000000000
               stream[0..63] = 3EFD3CB48D101B96C476FC474B90C34D
                               28C70FAE0B02560E5DBBD78CF6242580
                               7FD93F1095A491CD610B93B3E1D5CB25
                               025B46FA8EF9293AA10A04A005BC859F
            stream[192..255] = 734A67A9F3F709ADA26DB9A2AB67074B
                               08B000B93A346DB1A1A7E85BE6F08762
                               D12728B3A2ACFE92DC5A55C11D314217
                               95EC96B2F864A99CFC8DC19A3968D557
            stream[256..319] = E38257962CF6421F02408FD830A207B3
                               EDCB4C2813777F1852D2D67914C81FD6
                               94CCEF4FA18BF869E1B713F8D32F75FF
                               6F3EFDAEC72DAE4BE95A6216E8B5EFE2
            stream[448..511] = 6579A6020CE99121C55931495022AE37
                               3E804C544F66DD8AE19E301EA6A59D35
                               E4AA6C276D603C81970DB6FC9873145D
                               5CE1EF50E0FCBD83BDD1B695639B6619
                  xor-digest = E52056F99E72713F62269BE85D7D1C83
                               5BD387B4B70B80354071C3831F150843
                               AAF05D4B68F203EEDD2D13AFD8FD70C6
                               ACB076B237C9EEE406D98B958EAFCA6C

Set 2, vector# 99:
                         key = 63636363636363636363636363636363
                          IV = 0000000000000000
               stream[0..63] = E389A5F5FA91DFA9EF33BC5D59A32A1E
                               5785CD3332AB4181920B6424DA428BC7
                               FC741625CC0E71681450C6874A25CAB0
                               CD82F50021058F1F9AF0A79FE1D523E9
            stream[192..255] = 61228955734A2229D4FBF81FD7CCBEA3
                               67B444ADED9F9BBFC58AF64344C0D3DD
                               7A95E09AA03981CC153C30BFA32F8911
                               0D8A5DFE7B6A72056B4F3F794C703CEF
            stream[256..319] = 7B2C5FE9B147DE30DBB11497045BF9C5
                               73F1CDEA6B6D6BC5A7D6A7D5AFCD71CB
                               58B053881D0758E07BD6C28E3AFEE6D2
                               C30229F7B7B1F825C12DC83F22F2B764
            stream[448..511] = 0A9FA3443A110696F7E09E5A8517D5D9
                               9CA97CB84DC79F6B244005B601746DF8
                               C891FAE30A18EB4FB6BBFFBDC0B2C946
                               E58A6379ED8011F587B5E699BB110687
                  xor-digest = 0915C7A8096764B4971FA1F0E16396EA
                               4DDEDC089AECC5427E8E608C035F508D
                               B9F88438DD187DC96425CFCF05B42B10
                               CADD8CE70A3B3727BC5DA947767CFF53

Set 2, vector#108:
                         key = 6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C
                          IV = 0000000000000000
               stream[0..63] = 6B0801E18D411FB380E9CD019D64C1CE
                               BAF4321F9698D780DDC901F4C9A5B054
                               3E3B68D13D541132278134635FF88BDE
                               F53A2FECE2907CF204C72F2D44793F5C
            stream[192..255] = A3F68C6843F746642C25861C08532D40
                               230F1B89F2AACFEBF37CF4AD0755885A
                               D264089A6FD7888EFF21BDE59260252E
                               BEBD5048B0F71754A1FD2A84347D9983
            stream[256..319] = 7B7424E3EF501CCD176F52CAC9F76A6E
                               163D24898C0B1B392386E510AB1306D0
                               506BDD75250CC3AFBC1107D085B786F5
                               E15C443C8ABF24239C809957EA5E2472
            stream[448..511] = 2FD527B2EE6F47B4B309C98B9D89F90D
                               ECB62003000DB9E368005C040D2AAD0D
                               96B4C4C28A4D73FD4B77C9B5F5AE306A
                               58480DB0C23FB8019966361301BEF871
                  xor-digest = 0047F785832EC6C4D53F74F71385F6AD
                               4AF2EC30C47DECE026227C33DC2F6046
                               C2079AF6FCE40A3E1AEC09555B11B638
                               4ECFE0C1FFD06C64B70E3166EA5B033C

Set 2, vector#117:
                         key = 75757575757575757575757575757575
                          IV = 0000000000000000
               stream[0..63] = 8E84EF7CD82BD7D9D0AC60618E810CB4
                               B27A487571C37DC9A6687C88909C5184
                               25E344CE10F7932F8712239E3E9B3C98
                               F0C577E9717E067F249ECE881B8A4B44
            stream[192..255] = 3197B6DEBB0F6BBBC64D73C70F92CD8C
                               AD9F9126774FC8D5D7860713B34B692A
                               CFADF08CA40B6F0F17886B707BC0E19C
                               75ECB27C3DE001FA89C8326079243988
            stream[256..319] = 6CB8307FAE14BB9BFD35DC3113FAB89C
                               79B83B3FBBFA858D3A66BFE2EA0667F7
                               69D81556E64B993BB264899F9A137EFE
                               68C23C8D0159579EB91338EDD7641DA4
            stream[448..511] = FC163278A81CC1D68B0DC63AB8653EF5
                               E7B4AE6D83FFFE7FCE25A1A2499F4A99
                               58F174379D12B07D93689BDF4EBEF5A9
                               FC1B0197A8196608996D00931C3F74C3
                  xor-digest = D67F17889A2627C46E669184A70764D7
                               4CDB38D233E86CA7B68BE30BAD7A417C
                               8A8069965949DC949A91BE1E3F9261B9
                               DE7F3ECD400A1DDB183AEF4FF5521B3C

Set 2, vector#126:
                         key = 7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
                          IV = 0000000000000000
               stream[0..63] = 5636194A7942D413E1DC4C00472ED448
                               52AAC699E6EC70CCBE90DD52CE22A2C5
                               C181574D6ED0C812BD11322C37E6707E
                               51C2DDB180D169EED688DD4FB0CDFFC2
            stream[192..255] = B2A519FFF3609F005F6DCD3C358EC11C
                               C7C7B9DBF5673CC9F8DEB2C8692A8C3F
                               B692CF6996C72D44E4AB31986C63F196
                               3E2805518AA25D173A9726B96D5FF68E
            stream[256..319] = 4D4B1930AACC609028188AB283764A51
                               AA5FA78372680EBFDCB37D231B991996
                               E1F7F587161B3CB07BC53E2C32A13404
                               7BAD04FC4247DC91092C2C6E158C0001
            stream[448..511] = 44F0EBF50F57A17B05FB1C00E1E9C2D1
                               74A46D274E3F7771CF3BE60A6353DF35
                               95027FBE56429251328D14DFEAAAF341
                               73FBFCD64A5B0CEBF809B8B5A810A4A5
                  xor-digest = 2420CBCFFFD7FF619CF90C47E8254CAE
                               35C9A5829D3D12555F63F74D7EB7AD8C
                               38ED07EF32A99A37C7D491CD75057310
                               42C9AB76EC402AE80068EC46EB09F5BD

Set 2, vector#135:
                         key = 87878787878787878787878787878787
                          IV = 0000000000000000
               stream[0..63] = A07C7453D74FFB728C811461E65CFD4A
                               C4DD1E51204850E300289CFF77AD5AC7
                               5FE981783AD7142FC5A1F824FCE4603E
                               7304D87DAE839A8709E904A154EA4C5E
            stream[192..255] = FC0B10CAD79C24DF62B5B0C0D72CD392
                               BE1BBAB711338658223900B3A91CE3CA
                               E1CD0695B48FA70E378F516333299401
                               7CCE453F2C6BB15741AB59EAE40D8740
            stream[256..319] = CB265FD47653D4C1D1C6127AC9944F48
                               395254BCC98C4098B531F9DD3F8E090F
                               E32F08305A4E3BE52A6C7248514E748F
                               50045A8B5DF32F40772CBE6B7CA37C51
            stream[448..511] = 3F713D14BF3D6DCCAC799054628DC343
                               7EA1ED3E584E7D66A5AF77E32D304332
                               3471DEDE870EDD6C80EDB13044D3C7EE
                               CC5222BEDD632312A502CC4E09657D1B
                  xor-digest = 781C6F09E76920437B776A40BDC6A98A
                               83580F69C33153FB7EC84648442DE65E
                               85044B20713E86518943E7C00CE3B555
                               2141185478569F1BEB433425E88C5E16

Set 2, vector#144:
                         key = 90909090909090909090909090909090
                          IV = 0000000000000000
               stream[0..63] = 1B99E94FF97C2BF5C9855990FC723112
                               6848AEAE4358108DF55613A029711E22
                               5DE0F71B8AF909CA8B30D29528F094F5
                               BA422655E865CF0415E2873D6EA3C052
            stream[192..255] = C853430564969D769643242F6528A333
                               26242D6C15C00177F5D323FCEE231A06
                               DC11D438F628783E051F7298FF9CEE16
                               27A8914C10E4CFBDC821D224C768251E
            stream[256..319] = 312B62F94B4C2F07188476E094EE188E
                               9024CCFB48387F0CC8DE8C94546A915F
                               06A51F858F2D350394AD28E14AEFB487
                               8BECF9DD0B41D5E31B72E5FDE4B95A21
            stream[448..511] = A6E94F9AC8750EDB5FB501B0BF383CF5
                               0F71501FA3AD6B2E615C96C7DE1F9B8D
                               90BD511AB1CA84CB4E9E8E932B87F8D2
                               FA85F13185BC3A5A76EEB45D5A8662A8
                  xor-digest = F58C4EA9798FF90D61638922CB3FA58D
                               BBC40E5D7E818A68729ADB95359BE1AB
                               E6A109447DAFA455653C2A9BAC4790DE
                               95677D3FCB5DDEBC8B6B222C71232ACF

Set 2, vector#153:
                         key = 99999999999999999999999999999999
                          IV = 0000000000000000
               stream[0..63] = B00BADCC296EF974EF399D65BD221739
                               22C6BAAB85CFAD316017C90BCF9BAF0C
                               68931C22752B9CD4E93B383DF17126CA
                               D13C0484E6F86966F231C07E80D55F17
            stream[192..255] = DD2CD88C628CEA5D553905DA1881990D
                               D6E5203F1B6DAA7A1CA924F46E04E30B
                               077C4E956412D013DF3EB8EFE0EA0BCF
                               B45E7D56F8FF5259011227F2108E18BA
            stream[256..319] = 8839583F86643B07ADCACF096A72F114
                               EB47EFB6EA007CD8013589E0AB80C820
                               C1A05836BC87F1125BB0C7814F493FEC
                               669DC4A165CE9B7293DC435BFA7CCFB3
            stream[448..511] = 9DEF9B71E4171DDB457227EC967EBAB4
                               F73862FE73118D198F6C69B70F8C3360
                               33515E0676F5FCF227EB5C2E956F2064
                               A4FEACBB2A14F3F1105EDBD4EAFADB44
                  xor-digest = 57B91698D02B8368F2948688F281498D
                               399488EB653AABA93F0B6755189042D7
                               6E9A139B43535A36A227C8ACC85ACBC5
                               969A228E2AED7CD89C053268B494146F

Set 2, vector#162:
                         key = A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2
                          IV = 0000000000000000
               stream[0..63] = 589DC88A91119E50DD431DF324E48370
                               BB47CD18B47EE7141D005061CE405B6C
                               1F3D584943D5E16AEE589D20760E3B6B
                               F79162F246636FBC514ACFE0EDD562AE
            stream[192..255] = 08C7D6A5B251FA2D3015C656886B2BFB
                               7203DE8FB6588D77B2282D8FB3C79A27
                               598686E2477D058A6EB652DF04DA688C
                               B06252A6C1219E03AFA3DAC8745037E7
            stream[256..319] = 39F28C8DAF72EE9207D24132A5822CEC
                               2BA1F924864F1888ACE29F09BBB95EC1
                               7A72F78656441362FDB9FBB1645B7E25
                               FCC120A1279D50F588AA35FC40B5FC76
            stream[448..511] = 0B4069B2EA0C49F0924A8B9A0E70D825
                               AD779E2F09C5CC4781C5DB7D3774B037
                               7F9FC0061062018B69AA740D4F3DA9DA
                               A03C9D84973F31A666B867B2C3603416
                  xor-digest = 29E34FA4885AD39640BE9177D8ABFB84
                               FA9C357F2237D8010B713D3312FE922A
                               CBEB6E5248C592EC2E3CDA0F41F3E27B
                               A8BA6C7A587EF204D350CD691ED9FFC7

Set 2, vector#171:
                         key = ABABABABABABABABABABABABABABABAB
                          IV = 0000000000000000
               stream[0..63] = 0F2784597DF16C84CF34D2576F5D4C04
                               73649281DEB117EA380AC1141E5C915B
                               4132A0763EC70AB96F0E144CCD6CDFF2
                               54713EC93D2577916728734127AFC3B2
            stream[192..255] = E1707B1C569E2011E5160248B0ABC86B
                               DFCAB8CF49BDD01463034DB9E869FE47
                               F1F0064AFCFE97E19F910FD303635794
                               9BB5E0B3D6F4804B538309DD73F2CA44
            stream[256..319] = F5A1B6FAE8A52E5DC413D13C96DAA15E
                               D4DD714A8894AEA5007839C66C4E45C7
                               71E49317594C3A939B81A3BF9642C0A1
                               53AF0EFA32465818F3B1F7E966505EBA
            stream[448..511] = 05966F8B413793626B2F0BAD2B33A53F
                               C924A775C883295EC9576A93AF8AD59D
                               6CA5F747BB8988A52D84F563492148DC
                               A86C538EF719BE0600ADAAB154A2D897
                  xor-digest = E1C96EBFFCCF71083FBCFB7C8433ADF9
                               2178ECD4226A57880FE55E65CF94ABD3
                               B3C54D8841E40AD4B9B7DC0F2E520106
                               622E3A0CC99E56174376FD3DD4165519

Set 2, vector#180:
                         key = B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4
                          IV = 0000000000000000
               stream[0..63] = B4F82DFEE6108192539DA8653FE99FE1
                               770899983C846489677CA0C5C08697F6
                               A6629EF3EA50B3D87CB2A442425A44D5
                               B48B52513CF5A24C89E0CD1C17E3A2C7
            stream[192..255] = 5891DBC20064DEE60378EB7AB35DCE9A
                               C494577C54B686A17CE1622AAF527659
                               6221A24E35545760DD29AC979053BF7D
                               733F5F9CE86F67EBC588F08A9261E513
            stream[256..319] = 37E32F34DDDE6D717C8F05BEFC175BEC
                               55B7D0A1ABB5A6A7C5DB07C4BD15CEEF
                               FA4B484A181CB04A6D4DE20A169C2970
                               E912F79FED019C51142DE283CC750FBA
            stream[448..511] = AE521271F96BA0BD6BC6E18DD2416919
                               66841DDF106280FBB1EC4C38D551911F
                               6293C49F1397F0BB277D4DC3109AA636
                               5F07C05839F7483D4E9B0289B746F490
                  xor-digest = F94E8A31779FA7BF7B9C4107F0AC9D0F
                               3EA9798DC4D96971EC3344837D0D1B74
                               92F7E0B1FC9A81BE559D13782AA2A75F
                               DF1AEDE66D8C2DAE4EAC91092F5FFFF9

Set 2, vector#189:
                         key = BDBDBDBDBDBDBDBDBDBDBDBDBDBDBDBD
                          IV = 0000000000000000
               stream[0..63] = 412250869F182E744D3AE61AEC1A0871
                               10CEEE9D09AF9808A01BFF85750E796F
                               2928C4641510FD593B9D066433774EEE
                               CCF06FA00B0BE03A1384B4DF8DCE705E
            stream[192..255] = 203AF3950D1D8BA96F360A2057193849
                               9CC4FAEF4706A4F080E0D9A9BC1D24E4
                               F0D5F21868E6D2548CE4D736544C252D
                               D0A63F1031631091F1A06ADAD01FBA69
            stream[256..319] = 0AD812997275D2A81A1E16498C6D7BE5
                               F3B6B1403D33D93A75F00C7A74FD63C5
                               00D9B7236BD2DDB86B1C1650ADB2922D
                               8AE1DBAFD23CC2442909E4F7AEF57F20
            stream[448..511] = C2FD8DF92A5641DBDFBFAE49B4234B46
                               BEC7B2C5961BC7A4451632721F16D095
                               718FDB7C5D6B79513384FCBFE3953767
                               169D6362EAAC1693467E661126798F8F
                  xor-digest = 065073AA3CB7A0CB26CF0A1707E403A8
                               781C4D73B679D44E6BBF678087A463BA
                               14D8BFBA831A54B3660B7FDC60199B30
                               3FC04AC175F3DBDB8817475EE6901DF2

Set 2, vector#198:
                         key = C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6
                          IV = 0000000000000000
               stream[0..63] = BF078B482CA987C6FE0DB35B57C57438
                               CE8EFA08D03C33CD57AD4C4011AA4613
                               849849C85639B07D011C042B479430E8
                               6FBFE86A564FD24DB1D452D011BB2431
            stream[192..255] = BC93B4BD800FA5AF7A8C7FE140A3427B
                               017BF881DE34CF2CF0790371A67D55A6
                               46E4C5B2A6E2D261094FA38FE43D4F58
                               356E0A8E1DF95C6E9EE997BAF93D8FBE
            stream[256..319] = 35A186FEC358F0F71622E2036736E9E4
                               95A1DE01B2262A041A0DE4C6E2458FA7
                               74968E12C9336105E01666A495980445
                               486C96C2A7442C07507DF7BF2DDA69B2
            stream[448..511] = 8AE093CE4F53DA6719F30D1C9B46F77D
                               3706A217C042D3D10E04BE0F1196367E
                               536C2BB4D1EA803B627CD5FFD61E1EFE
                               BCD2D9FDAD73B9BFBAF60F2E49EBC51A
                  xor-digest = 4BE42C2034BBDC4C79739A65893628D9
                               71D293605517905761B967F43C6106AD
                               42B778F92A250AE71EAD233855AB47B8
                               6FE1B1D38727C5C7BB9EE23BB3A9B8B0

Set 2, vector#207:
                         key = CFCFCFCFCFCFCFCFCFCFCFCFCFCFCFCF
                          IV = 0000000000000000
               stream[0..63] = AA2E1D58F1C2D1EA6E199EE52D0463EC
                               E9968AF14D65E60B598D812B5B3005C4
                               489CA1440C4F054B3D860C3F02975D5E
                               384620173F35DBCC2BE16CA51F6D51D4
            stream[192..255] = 1B3EF59FE7787FF542A3642D494BFD2E
                               4CF23B0E9414726B8E9A33D7248023BE
                               5F1BB7CC4DCDAFF26D032B2BD4BDA70A
                               7DF83DEFD97D55A7CFB5C87D0403BB36
            stream[256..319] = B2AB1C1703379A771D14F51EBDE3E9D5
                               FA7F64282B022198DF5AFEB47EE6161C
                               6AC3E475271E255AEB8DE3B76B9F72C2
                               6FBD8D9A07E22567CBA64E3D2B358868
            stream[448..511] = 2F82A11DC74591559B41F405D92061B4
                               6A2C3C3E7861D3DB223DF33F0883278A
                               7F2501C8210D4ED83F12908DD2A24DB2
                               974FF5FB017D18C6231906A4349A0254
                  xor-digest = FB802899DCF605A73152A329475C6498
                               6A73BDC0ABD79F343256375A8C99B3BA
                               1420A670C9224D935ED571F14D76B581
                               FE7359EC6A81F9DE87621A929BBDB80F

Set 2, vector#216:
                         key = D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8
                          IV = 0000000000000000
               stream[0..63] = 8FFFA4908E599C9F54F71CC3CAD466E2
                               7172C0B28A89CDF401807621F2CEB860
                               DBE97E4506CBAC5A553823F237AF44FB
                               172E28482EBBE258CD5D900257DBF600
            stream[192..255] = EC0A8A7E2F056AB09CE7247A43D6AC99
                               848194A71B1C8BEBE3F8344DEE2E90A1
                               3DBF9919BC1CBF1F9EC1574E8894EEAC
                               A165F6596BA764170F84B7B2C198B041
            stream[256..319] = B97BAF3540F110EBA6C0B83242F8B92A
                               F916A2BD90C746D44E92C569EE46FBF3
                               8134D3CD7FED347D59BDE6799C4A22FB
                               E772F8257228B635A780435564173652
            stream[448..511] = 9B9F0345D3CDC28880A8AC32475CDDAC
                               B1FE8042FB27BF550D65F60D04FB1BE9
                               84409377EE3F287D7D86DF8AA577E68E
                               7CDB45B14809FBD8414E5C68671AD797
                  xor-digest = B3A6802F51E6830D24A09CDEEC6CA56F
                               BF6740F8A258E7932E7911887492B13B
                               1B8589F0010299F2C5B84BC1FAE55A67
                               C8FE002C8E96B5E92C6E7F7F9B00303E

Set 2, vector#225:
                         key = E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1
                          IV = 0000000000000000
               stream[0..63] = D691618BCBB4597F2F002A282DB08F18
                               8260243C7B71C458252C00FDEF11C5CE
                               F88FF0590F74BB4F8010ADFBA887F296
                               94AEEFA49E1DF385E5F6C909FD64B2A2
            stream[192..255] = 51FC236B38D321BDC28E882CE0C1677F
                               6C8FA911149224208C39996846EFE47B
                               3B3116202C6442252D772EDA5E6FA2D2
                               444EE4E26DB885C30B3D742BA575078E
            stream[256..319] = 45B0E8B6A8608E595B7389FBFD3A278A
                               AB8A69AF3534E053C26C46F3876C6DDC
                               041DF569182BF4C1734E0D76EB703F08
                               165C922B7CE38E992BE35BDB6650CDAD
            stream[448..511] = 9F0F51945B3B64DF58DC01E7FB3FA1BE
                               19506AAA0E8E192601E1F318A5A7BE0D
                               FE90766376B66BB26BF2D341491E5AEA
                               B16319B6327FC4BF2F552635F2484D8E
                  xor-digest = E1D71EFF4F6AC143217154C55CCA88F6
                               3BBB0F92469AE5F9F2F6E7EA62828B40
                               52B099AAE0DFBADB956D18096CDC0BA1
                               6FB164BD344E295DDE07002FD747CAF9

Set 2, vector#234:
                         key = EAEAEAEAEAEAEAEAEAEAEAEAEAEAEAEA
                          IV = 0000000000000000
               stream[0..63] = 422F7069C9A50B477617644969866641
                               9865D441451E2B298A6CB45ED810EC70
                               60128C304AB69E1925B0032D6CD33F0F
                               8761CD0F6255CCB85E5EC6599EADC85C
            stream[192..255] = E231F4AB8F76ACBF29A17B7BF89B89E9
                               8C6666D93897F4B91E37D2EBF192E89A
                               19C118EE64A06026EC2B30F8F5FA875F
                               15748FDC0A6943A9E4E76081AE7578D1
            stream[256..319] = BABA7317873F97E79DDE6ADC5AE3D054
                               1194B4C4F3CCAB2738D8766BB638908A
                               31BCC0002BE58B3D2E1A574278681236
                               E146A095CF376FEA265BA434F5BB0404
            stream[448..511] = 02409059238B1864F85A37B9132C1C43
                               456D471F26C2B345CD70DFB64C84E888
                               06B0382D54A97328EAF7F159723FB4E2
                               EBF773BEA260ACBDA9FD2A276E0BA1DE
                  xor-digest = A04E4083B5BAEFD3497DE23B9A9381F2
                               5E2CA36D8D6E9A9B2AC7DD93CE5AE0F4
                               7C633C9FBCCA66FA43B059E5499C55B3
                               B3E36FCA0AA45DD575763236CB3DA253

Set 2, vector#243:
                         key = F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3
                          IV = 0000000000000000
               stream[0..63] = 91F87B5DBA560E3FBFC5CD43CA50DA55
                               0AA2648AAB28BC62EB0AD4414F8692EB
                               977D93381284D0F26820DAAEB8B5C44F
                               2C874C6921528998267C822553C5F745
            stream[192..255] = A48DEC79C1ADF6D6BB99947621F9E046
                               5DA6FFBED93C66E5AB56F3CC09D3F631
                               CBC8BF78A03EED31DC6430330E9C0CE9
                               B0F3DF41FD94B1FC81D44F82AAAF9CA5
            stream[256..319] = F31878CCF2591DE5684249F2FD2CAB6F
                               3E7E94CE507CE1401B0F5085CF27C7AD
                               3E776468FD098D60389A08C960BA604B
                               1CCEE0D69D1803F254D2E1D9171CB66F
            stream[448..511] = 50ADC820ECC329E14B66BDD159DEA2CC
                               F6A4B80353F7B783A5FE972DA013AA9A
                               5DB1D92C250D46F3A4F2FFD1248516DE
                               7F36DD29443B9EFC5FDD39C0521E73DE
                  xor-digest = 9C80115F3D6C42E0057EF4962D048866
                               99D6795A8ACFE2C71A47B70D8E7626D1
                               CEA8C9E5875C0E69357B20348FBA1F46
                               6EC41669F8709DF5C72AFEFCE62D6170

Set 2, vector#252:
                         key = FCFCFCFCFCFCFCFCFCFCFCFCFCFCFCFC
                          IV = 0000000000000000
               stream[0..63] = 91D2128ED1DDC3800D570CB4ADAC8CA9
                               760BA6223C7EF19A81F62CDB3F8D7901
                               4175D8B57840FD0D0A9EBB8CA6CA997E
                               6AB212C4C1CE9528131DF8C5C3CBA992
            stream[192..255] = DF4916D02189492FD92477865F768DEE
                               24B95846BE302D516498A197AA834C95
                               8F48705F772C76503386E14D41461E8D
                               106BEC629C8AB86987E89BE9E58B3B35
            stream[256..319] = DAEBC4D6BAFD05ABEF3E80F10977300C
                               22B5794337329F78CC406920BD61A908
                               E17BDF09CAF99C06B4471EA7A5CA3067
                               08FD08BEB4FE6CFC5589DB0EA5B753F7
            stream[448..511] = BA0795ECE69FBDDAA293D933B7FBF964
                               8C4F72DEE388777E906C2824CCB8C347
                               E605F6132C47AB2397FB6B63F66C3BFF
                               84459633D646F5672854EF0DAC8DD10E
                  xor-digest = 8884D9ED4BE75614F04D92ED6B19BB8B
                               54589B95EC4409F9EF07623E64A164DF
                               848EFC9E77AFBF157993AF25E7281D2F
                               6D69C23D520BA71E3BD3884DF8E1D6A9

Test vectors -- set 3
=====================

Set 3, vector#  0:
                         key = 000102030405060708090A0B0C0D0E0F
                          IV = 0000000000000000
               stream[0..63] = B944F975FA4F57828CD395EC502244C7
                               EC8CC5C03B0855362712093A0F3E4D7F
                               AD253A30AE26C6A4D5533581684E4C92
                               99F95B4A04D0E0FB045EECAD98B9BED9
            stream[192..255] = 8605AC4B01C25BAA23358420468B9EC9
                               38AACC8B1C48E9714A10D45D1B5FB040
                               88FAC8F978092A7F00C8997E44B957AD
                               A886DF3355CAE292865A56BA9C40D47A
            stream[256..319] = D67A9CB65DCCAF2CBED3EE49B8F741B7
                               BC905ED2582434AB5841C30F907DD4C8
                               BB64E8CF349AC9E014C730B7AC47D0D8
                               A35BD764C9062147986BDF185C8F6ED1
            stream[448..511] = CCC41E287637F5133B57B63ABCFAB9F6
                               EF08E7F64C35A3B1178BD24AEFC88D38
                               7A533337E31BD8DAA9CDA684D4DE3A73
                               EE090CEF165750EE875BB3DD0B4FA7D7
                  xor-digest = CA58C4673A943A086CAD0FC68C44D797
                               1E6901DC71311D7C6D3E806CE3F4D7E7
                               708D15FBB7EFEFEBE8E8816E7F32BEA3
                               474ED4A6AA4C24BD85D4284DCB46D95D

Set 3, vector#  9:
                         key = 090A0B0C0D0E0F101112131415161718
                          IV = 0000000000000000
               stream[0..63] = 5C98076C5B9B23A9330FBDB0E4FB13B1
                               D065A47F1269E4B5E8C220B6F149AD20
                               7A087542D7C36A438B293C7A9E2F3F4B
                               3690D8450C76302E15988C3E087416A7
            stream[192..255] = AB081A893DC93F74D5B2E9DE1052AADA
                               21886DDEDBB54A3B634D88D48429D6AB
                               A376B062E2C67926F75CB34790FF5582
                               0449943A0A2BDFB5E7F1627E65FC55A9
            stream[256..319] = ECCDA197A1ED2F786822F9AFE8762953
                               6EE878E01E33B4A1F75A7C85451B0B44
                               88D72F523E823EF564E14A232AA9B6DF
                               4047978520A1189C3E939C98A6B04789
            stream[448..511] = BDAF7803BD065FC8CF4A8D362D32345C
                               2287F7286AD6474109090AE120AA084B
                               C8267300DD1D6EF55603BA19BFD6342D
                               B0CB33D49DCBF34E98E0CA68A1CAE5AA
                  xor-digest = 5DA4029EE351EE7F695DFB1DF4329BED
                               6713D7A9DB3EABB04A42F1BE6EB2F0F2
                               8236C2B0717F9925CF01C1D2972C0CAC
                               A2E79B955C8BD38C36EA1F7C7A1C5C70

Set 3, vector# 18:
                         key = 12131415161718191A1B1C1D1E1F2021
                          IV = 0000000000000000
               stream[0..63] = F6E50642F21A69DCF979A0453BC857E6
                               972343AE67F31458200AC2AACC6157BE
                               CE3B8615786FBE7F3527B82675EBFB1E
                               1D2ED84C8B8A54E3FA949328EBB6213F
            stream[192..255] = 55BCC142834D5779DC122CD86A71C918
                               595F9299054073846FBA12A91BDCB435
                               B67F076E771ECD7F1E5E1F4F5F517D1E
                               F985AE4D86DD8CE1F7929D8872EF77B2
            stream[256..319] = 1417C2E51E22C43E5650C350B17BD8E0
                               804306C6870133AD90A7B0B5A0E6B203
                               5A3280F400EE5B23D6FE31FDAA0B84C4
                               DC316C6E3C7B0AD0B9D49881CA62610C
            stream[448..511] = 2FD3470B0B72AA966E985534EA923DE0
                               D674BDE7A069B8BA7DCB457196BC258E
                               8D88358DBCCBF5288ACA6754275CD0A2
                               9D13F14B9BAF140F51F99009A14AD8BB
                  xor-digest = 78EC5CA63A428B24EB9B8B23C87980D1
                               796D4AF8E2B08D179E260418715E92A5
                               AB313A37078057B96170E97922B0BF13
                               D44A173B114D1C111489E81469093379

Set 3, vector# 27:
                         key = 1B1C1D1E1F202122232425262728292A
                          IV = 0000000000000000
               stream[0..63] = 7C29443DDF39B33D8FF9A097EA8925D1
                               3D9096915697657E2918E1A4E9BCAC64
                               FFA13F9CC2AA13247D36DFF11D3EFA10
                               4DA78914F7BF83F7BB6FC6592FAE47B0
            stream[192..255] = 2E207793F18C28BAD7E80AD3C5E01B71
                               75FEDD0B880F21322EEEFCB4FD6B3760
                               E99C57A93771A451C75184D93D95AF28
                               3C5DA66315AA81D5BF6486AE2ED08611
            stream[256..319] = 02692FC646AD96EA9443FC2D3F4D1857
                               DBAB8C3F1ACDCBF98AF443626CE42FEB
                               B1019CF9E8104B1EBE9A2A08C11CC238
                               BF8197F7CD8603BFEB8BBC1320B9D0C7
            stream[448..511] = 19A98541A6AFFFDAFFD061FE5DB77652
                               64CC6026D76A86910D3155ED8D60B38C
                               99F2DB1E03C9EBF448C5B9A7DC9E33FC
                               766DB3530C90F58BBAA424D924F8BBE2
                  xor-digest = 000B02661425A8F6D6F251C943DC92C3
                               7942D14DC404A2E6998AFE8FF254019F
                               06EB15004965F32EDF58FC96F3AE5AB0
                               D3965835B8BA8588B9A97B0DFF8D913E

Set 3, vector# 36:
                         key = 2425262728292A2B2C2D2E2F30313233
                          IV = 0000000000000000
               stream[0..63] = BA7EEE789E15F3333605F76743991843
                               E7562AC1CCA5C3D9EB99BB0DFF6AE19B
                               6CC17B13D4A0E254A7247C6FE5C76975
                               3935BFF84677896FCE01E90F4AADEF40
            stream[192..255] = 4BCF8E1B5C7BEBC373AA7B38C4B656D4
                               2BCC756A87556B5D0727A6AF871A3DF6
                               737F2E167AB4A42F2D9040E67B99C9FF
                               3B0E904BF81C9E53C341B91DDD4E3D22
            stream[256..319] = CC8DA753458731AE170F456B045DBBDB
                               0F062FEE904F004D3AE286957303B003
                               FD92B87D4CB28102D4C4FBE37BB41D2E
                               A9E328E966C37BC9A9AA2F445A9F426C
            stream[448..511] = AFE8E4308AEB0A5FB4991BF31F875F36
                               3B2F8DC34844CB1A1463CBD9CF54E09C
                               371B3ADDA8B0124D2784EC3E8C90B645
                               48E43BDE3EB284A5A9B0F2276151AD11
                  xor-digest = A9C52671735B2BE995F705A02D51E53E
                               55BB1277209784BA227946B31012419A
                               6B34470DF6ED63B5A89F38C3B1C9F5A8
                               24826B57A8A025659EABBA9846FA116E

Set 3, vector# 45:
                         key = 2D2E2F303132333435363738393A3B3C
                          IV = 0000000000000000
               stream[0..63] = DC69F2D64F6AD02C5ADA0BBD1923F9A7
                               1832BE0B6288583F761A7E6E57E5395C
                               529D482A065E2628CBED1D149D1A85A3
                               F61361180D5F8A35BBC82CFCC7106049
            stream[192..255] = CF733BFAE9287E8A10BB77F7EF0B2A75
                               47A0DA18331ACBE842D77ADE611E0B54
                               A7EAD5EC2E002211E44EF6232A9449EC
                               CA2F21893160F4F133D139D84DF73FDA
            stream[256..319] = 684BEDDEA8D9F5F508D3BB727A136A9E
                               CE5AF4B5DFCEC1016680B4419F5FC1F5
                               AD738F55E548FA735DBC5A4EC75FE566
                               4BD5425D29506237EDA944CE4A9B5BBE
            stream[448..511] = D2B016A4711E95A2AA2CBFF865F50765
                               EDB3A49BFC1C09C6B409421453A980F3
                               E0F1AE196C7B8EA38E6DDB110C46FB3B
                               F583688AB54A7A5FAE7EF583FE5B3BE4
                  xor-digest = 2FA9EC703471755C068C73ECA433AE33
                               1A694ABF8FABCE0A707B2837DD1FB869
                               25115A4D820C590558BCDFF7E85DB069
                               43D3000AD4DA71A9A221FAD93EE3A6EA

Set 3, vector# 54:
                         key = 363738393A3B3C3D3E3F404142434445
                          IV = 0000000000000000
               stream[0..63] = A3B532C5BCB55591B44388EE91AD3CE8
                               8AC256302851D8FAAD2290E333751962
                               187D4A822B1C6FDC8AB0F0457373EE1E
                               903CFBEDB8D70DE45E14FF8DD00A54E8
            stream[192..255] = 4B8A5E9C75F6E8A616466AE69BADB056
                               1749B37E68A4C8A3503EB6D0F2A9E37B
                               CEE771A7313F88515B860E012D2F28E0
                               295D744B853417AC10D98ECB95E6DD21
            stream[256..319] = 873A9C172E6C01B7814342BBEA87CEB4
                               0C7B0ED9A253715330D82CDD1873A1A6
                               5F7A7359662A16E6DD60974F62FE490C
                               F331A110E5A66AE7F5A53E3F6A912079
            stream[448..511] = 3153B05CD0678B062B33F52F4D60D49B
                               F5DFE40E1C5C605B6D2BFEE77D10F14C
                               D98B00D73418DF4EA60E7CB7FF4068F4
                               36D0B14C06C3962128467E40A2DFDD49
                  xor-digest = CAE300665E16A7D8509603600A3D76B4
                               3258CAA2B9B98000AF6E00F44EAB7E70
                               2E2FD01AECC892D277C6BF354AAE2F33
                               16C029A7F17034F5C67F92F1A4CEC9B9

Set 3, vector# 63:
                         key = 3F404142434445464748494A4B4C4D4E
                          IV = 0000000000000000
               stream[0..63] = EDAAB24D0022ED40F04442E864B9DBD0
                               DF7CC54BA8E458CC1E068CA3C9D58B88
                               5A7664B5D3E94EC88655548E8A921A8F
                               31CB53419812C0E9E17A58B2448E7C41
            stream[192..255] = A1173C4990A76700F12FD02816198EDE
                               C3BDEB176C50611C547C04B1B7552AF2
                               93747C2AE373BB588713ED28A0689AB7
                               7AEAF19422776E1EBA329BE428B74D04
            stream[256..319] = 93DBE7FDFB017559B2CEABE954A1F301
                               1201A34DD6B6598F7A0FBEB404F926D3
                               B87D7A7D724E28575BC9447B4608F60E
                               EE758DD59585B772BC93DF0D7B73C2AE
            stream[448..511] = 34C77AA7D38E7DC3B1A4A01336A4E25B
                               D8599BA3A1B027CD45EF89B77D06EFFD
                               63AF9E262DE68B3AC916DA054281BEEF
                               AD9728A57F285F540C03A7E0C0903111
                  xor-digest = A808E567B1B53B30F3824CA83F308C10
                               3FA5FA2E9CE6002DFA098CCA7F226442
                               41E57D6E7770299DAF947AC714516575
                               B78B3AE29620FF074E5923D4C76CB01A

Set 3, vector# 72:
                         key = 48494A4B4C4D4E4F5051525354555657
                          IV = 0000000000000000
               stream[0..63] = 33BEFFE76180DB2077EC33F3770BBCDE
                               D4322DECA27140564470133955D41ABB
                               4CF0E944DA722B308F4EE954382477AF
                               D803852D1A818E7051C2371EB420D0FB
            stream[192..255] = 6C7804B4274940CB37B751A235F5B115
                               4EE1A63612727609EFEAF8E5D4C91E71
                               5B609953E0CDDFE24C01B2F4F9588CB7
                               52B1AC58D8E5B251FF9199798513FBB2
            stream[256..319] = D806FC774052A3B2EB237D363075755C
                               596576BE5C43F3D27AA016374858985E
                               D949312D49F3DD6BBBADFD6D55CF4CEA
                               A6BD59A11DB075CF7ABB01622FDB4FF2
            stream[448..511] = 755710AEDDFB7AFA4D03904764EB5D61
                               770CF5BE0D89902AB662638844303F98
                               6758FEAFDE9AA4ADE1A3F0D790B91D46
                               88C9E40D915F25B27FAC224BBAAAC96D
                  xor-digest = 85CF09841916C9CAEA81CFE7E4814548
                               FED359FA48850DE2DA310187E55E88C7
                               270B6FB478B4CF078198EE6B45AA8ABC
                               33679F5F3B49928C30DB686C52A7F9F1

Set 3, vector# 81:
                         key = 5152535455565758595A5B5C5D5E5F60
                          IV = 0000000000000000
               stream[0..63] = 2B3A8E25B066E8FF776E25A4C1B22E79
                               536C631CC89A283A67E2BD7E8F75044D
                               7F09E0E04392C2B4052CC01E1483816B
                               E6F19C0000D245B2D049017712443ABF
            stream[192..255] = 75C34F189278541F5A08C2E058E05AB3
                               89D3DE534F5BD181F8072050C7333532
                               3435A980CF08C360C154636C990DC79D
                               C00EA4703D7E35F25A31B3DCEDCEBE7B
            stream[256..319] = AA5614230F170D786F5523783046069A
                               D1A1FC030A34472A8277CC0CBB799D0D
                               8654ACC5865D7138497416F522D61A0C
                               5AE7E3D1FF8E2CDE5E15A0F5E7598F97
            stream[448..511] = F93AFF781427F4AE31AE5EE7685BC42B
                               96A0476F9A060EB9CB2696B9D7BB7BD2
                               1587291960527D6ACBDA69C4D51387DA
                               9F13A4C95D33C60CDD26B325DD753C17
                  xor-digest = 43DDD201CF4DBA25014B2AC59E54A9BA
                               85FBFD6DF3303E1819A10A26E9E7EC7C
                               C22C0AF034E6A618C23B57AA41651BEF
                               0CB022E1B77071E406C84260E0FF3FB8

Set 3, vector# 90:
                         key = 5A5B5C5D5E5F60616263646566676869
                          IV = 0000000000000000
               stream[0..63] = E85A04576D4A0FA50503CF29F23B7D7D
                               BEA8D6238A8554DD1B51DEC856473FA7
                               D15927461688CE09BC5928152D29CC9D
                               8754E7269162B822A184C1B8F3562E9B
            stream[192..255] = 1EC6811C3CE10CAD5FC97A60AA055FEF
                               48EA84A8417F70D8A3A1011C6C388EAC
                               86F950DA86216B4B5AEED2F39B479E69
                               49C6ABEBBD31A4C3EAA58C4D995A26A7
            stream[256..319] = E7CC00EE4477BCC25474075D94686A60
                               8EA949109F5EB77D8C3172A122691202
                               206CC16B28914B79CC44775698D6566D
                               2626768A8C83B295B8DB36C2C5B25D0B
            stream[448..511] = 19457396FC73F2AFFCE8F35B08623DD1
                               B60169F985951396CC18D07A4BC4F07C
                               404F05A7823A49B8E627CDF1F9A21923
                               F3CB8472BDA02B056CA975903CC3ED96
                  xor-digest = B9C824CA69EF003EA7DE921E439DF8DD
                               77AD3C3D8B38E55D1D4E2C41AE887E65
                               9B6396CB0D63F4D56A56CD339EAB537F
                               90BF4999EDF0F782204FF93C1DFF8041

Set 3, vector# 99:
                         key = 636465666768696A6B6C6D6E6F707172
                          IV = 0000000000000000
               stream[0..63] = C7FDC1B00C56E5A205CB493495BA8506
                               DBE95D7326D9D79C0C39D2FDAC9094ED
                               134DF87CA8D95D793FBDCBF8024EB3FE
                               43722409521D20E0F86AD630369D6BE3
            stream[192..255] = 826860F82D28237B1D5B435AA3308C7D
                               EFDFC4A4DFFE7A4879F38502796AF8CD
                               D2D67B38B409617B205C790BAF27C502
                               D2EC7F6811709508BDC4BF86974E2C25
            stream[256..319] = 09D0C20C211C7263C9CF16EDD1D4F03B
                               59F6657B09AE8BE0517117199D68DCB1
                               4C2903A7B06B574E80BADD37AC63EFEC
                               F7C00D7C3C67D47B86AF6DCFCF39AF7F
            stream[448..511] = 03CEF238533B10318CB35B069F727D1F
                               617D526E5E655459C9EFADF9137F46D4
                               FADD462FFA976EA0E1814B69FDD188A2
                               55123FE689F957CB0EB6ED3E7587F9B5
                  xor-digest = 3E6631C919065D3D4887F8A060085DB3
                               B20D84F79C78AED24AEF6A5E3E3D725F
                               110C4A3EC079B39E9B538CD92FEAF8C9
                               09E072591B0A0B95143D89AFD15DA2CF

Set 3, vector#108:
                         key = 6C6D6E6F707172737475767778797A7B
                          IV = 0000000000000000
               stream[0..63] = 97A18C197254BF1D7232C984B0FF0306
                               D5A741A7A19E169733E4076AD823FB6F
                               783B02EB3A40DCBDF3BF1F28E2D316CA
                               99FB85A8D97A64E1963365897367B1DC
            stream[192..255] = 63BC208E85DADA87879A67177668405D
                               158757517E1D8A9544D80DEA8B207A84
                               56451B5B00CE33438C3440875D4BCED7
                               FB39E1CD2DA93D52AEB5B0FDA81FC2DA
            stream[256..319] = 7BA6CC540AD5DC786D4742020F9D3DDD
                               C9EC3D91619A749DCD93F22DED7AAEE8
                               E05E933CC350684B6DA632CDE2210BCC
                               EEDE8C5D872CC0A0C4EFC780461ECC5E
            stream[448..511] = D68E284793EC5D9F74409E2A550E8C5E
                               49E0ECE77B836DB25A16E94CE96A4E6C
                               76F24266A796896A99E2C25A3A19C2D7
                               4D90D17467EC7C40BDB2B1B987BCCC5B
                  xor-digest = 815CF2F9220C9DFC610D9DBC8A8397E5
                               F87D8D9D11D02039ABE18E23C393F237
                               061F73B97F4A69B01F9562610C1F5528
                               0D7E648F8E066064CA6E7BB91B23051C

Set 3, vector#117:
                         key = 75767778797A7B7C7D7E7F8081828384
                          IV = 0000000000000000
               stream[0..63] = 2352F5E4C303C8D0940E75B63CB1A1DC
                               561B74409FDC14651B0FCD85027C750C
                               1515E452223422CACFD68F12A0760B40
                               E857CF75E3F4F38899C92A96AF257A18
            stream[192..255] = D9D4B02FB09F9581A5E3FB9F1D81D416
                               6ED1D30DB8D94C5D29407A9906A4F43C
                               945ADB4AD0863727AEF0272AC084CCD8
                               4CDB36D2B5A16537ECBA4A9FDD189E41
            stream[256..319] = 7A6FB5B7F99422F6441F26ED4A8269B8
                               410505B7A705E23FAB1DC4326DF4F680
                               A056F60DAD69999DBE0A4EB3A976E0CA
                               32F3B5DDF6719C8FBE0CA7EE6EBC2C0A
            stream[448..511] = 1DC9779A05576BC41C42C4D3207D0E3B
                               24386EFBB4637019DE4BD1DAF78974BB
                               E54B679E63F5D03A17C12936444A8BC3
                               6BDE14FCCE62EABDE5A11ACDDE42EA32
                  xor-digest = 420ED300C4B0ECBF2B700EADA9532D36
                               72D2B6582B7BB8F1B92F492F600B3000
                               5352AA1452A412A6C8EEA2A23EC208AB
                               74B0D24CD1B49BE9D3B6645F528F50E1

Set 3, vector#126:
                         key = 7E7F808182838485868788898A8B8C8D
                          IV = 0000000000000000
               stream[0..63] = 8FE443C77F2BC040BFDD0A2C725216D2
                               C05159764BC60DE59AB28D394DCF8B8C
                               C16832C43C9573D21B48B0493B0FC85B
                               63CE9540475CA5ECE01C0CE7316E153B
            stream[192..255] = 5CF9021BDD35450D639784A42E3FA7A8
                               506B4C82AF1B7DDB2708534F49B7DC81
                               098A571C1D37EF6162525C7E1B111B19
                               33FBA1B814C2122DD455B40D360ECCD9
            stream[256..319] = 47C629B2A4781155B466EE9BE4A5291D
                               9053AC1C76CFD7B2D1EDC717E33A0495
                               C3178A9A95FB835439D6557EDAA750BD
                               E52DA333A9D800688B59B5A16A5078EF
            stream[448..511] = 2B765CD2E584631098F5FB130F6486DC
                               98BF0000CB5A9045538A1447A5B2FE9F
                               F8AF23700577DBFF637DBE0530508128
                               0F8B92A7D3D56DC3085E740D0A6BDB65
                  xor-digest = 806ECE0BB175321343FCE5350BA035ED
                               7991EC9FE3EAC640A0A683ED0B7EAAD9
                               64FDFDC0CB9B1589F39442A18EB52D77
                               8DC88C484EFFD719606D92E4B9DD31AA

Set 3, vector#135:
                         key = 8788898A8B8C8D8E8F90919293949596
                          IV = 0000000000000000
               stream[0..63] = E599B28FD0BA76F00094AF342FDBEA56
                               D1FF02C6126A95539932129D63EF8632
                               2FFB8541D2792FD0AF4E59A071855DB8
                               F43E24FA45306BA3063003A8FB4F1F51
            stream[192..255] = 2C7670CA9A5F47A6C7A109964A45ADAA
                               4C301E3FD28BB265C5DF1114B68E482E
                               2DEB4D713A9B9B4CA0D728037D5FD7D1
                               45BA27098948393AFA4051EB40B433C0
            stream[256..319] = 9B8CA929B01F8E730EA8FBD997A8C725
                               0440EA528335F78482E63E19726A54CC
                               F7DDB88614AF0AF274E892396D5D77A6
                               2C914754F9B5678B5BA6D88E6C244B54
            stream[448..511] = 3B8151ABA00E58B20EEAF7CF3965BEE1
                               CD51D35E182C6DA1493DAFACC3E54CDB
                               C749ED581B9B77410437E8CEAEFDE999
                               B1D79AC88206E56C93192528A78B59DA
                  xor-digest = AF5D35333E216B7D64EF5D29D5AAAB02
                               BFD146D91580408F616AF7514B5658CE
                               16CBFD820108F31B97E2F3110CF98D4F
                               B29820DB9F2D6F159B32994D4A0A49E2

Set 3, vector#144:
                         key = 909192939495969798999A9B9C9D9E9F
                          IV = 0000000000000000
               stream[0..63] = 6ADFF9CDFF88A3E5E0874703C86DCC19
                               6C090972272630DE3B5C79810783450E
                               B679CAFB089DB31DA4BB3E2BB6B1B514
                               6C8047F84091481DB0051711BB3A4C10
            stream[192..255] = 762E3B6A8219440027CC0A2DE2C02FCD
                               CB06E9E7DBE22DDFAD59D4E091E3F036
                               E4EC32029923BACF8CFD5A78A6D719C8
                               D26C49F728CFB3A32006C2DE62D36770
            stream[256..319] = 46FAFC71119EE0051D5BE4233FFEB553
                               875FE446A7EFFAD95AF9BE476FA561BA
                               01C67EFA3F3E5C4CB00BC4F9A100EDA1
                               E440EB4289709D409B501E6B1CCB70DF
            stream[448..511] = 9E230DADB6721A1842AA9ADBCD5F49B1
                               5203F423504EF26F379DF52AFFB6A76F
                               655A1E40C20C22440CDC96FB1DD3E2E6
                               1496F9A890487D0B304D7F373F1DEB9E
                  xor-digest = 9110486A23538EEBD87C686A601BDC89
                               B6C16D55BC61C3AFF142FC67F0902A98
                               8D7025C38F9F0945F309ECADA3C2FA7D
                               1EE4E189B63AC5756C4B8B28C9913F96

Set 3, vector#153:
                         key = 999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8
                          IV = 0000000000000000
               stream[0..63] = 7381223CFDE233803F7484C801A38B24
                               1AE1A485BA644857EECA5422FA1E35C9
                               4FE3840EDC7F890FCC50F3D29F4A01F2
                               8F7AF08AFA954F4DA6B743630BE1E41B
            stream[192..255] = AA95F34B2873F5767E0467112DBDF16F
                               03762BBA19EF7F059804B62F47126201
                               0FADD0C259359439FEE338D8F3AA3E4D
                               24C0C2B1492FAB73A927CD6FF17634A5
            stream[256..319] = 1A70441AA9B2878C3E41607E6EFA0597
                               774057CC14D5822BB1265FBDFF3B906B
                               66F920FA3141F6FD81CEA11D50925464
                               66C3FF1190ADD1DDAE0F46B69FE7B94A
            stream[448..511] = 1AD6757713D7A54F8EA743134FF7A457
                               671F6371A9483DAA9C50CECDF752A0BA
                               B8D1B5B28EA4E624E5792BAD49A40CA8
                               C3DA218956D2D047E856AF6E636419BD
                  xor-digest = 0A1780D9C3FCAA533A7F08B116B34BE0
                               112F23DC6F9DB9FD6D4C6BDB07533C57
                               2946865D54F05505067B256CDFC001B0
                               ED085B4E7E070A868E7E859031A28252

Set 3, vector#162:
                         key = A2A3A4A5A6A7A8A9AAABACADAEAFB0B1
                          IV = 0000000000000000
               stream[0..63] = F6F7130D50A6E0DEC78F265468ABD6F1
                               C63D745CF1D142F1A39A96C4699F6859
                               46C9F18B9CEDF2F7C73B0493C10BC80E
                               7DC8EAF2F04988663D9A77C78DF0B72C
            stream[192..255] = 7B956074C6B217A50AC0605B24FC6C52
                               1B7C909E9F1CE3124C0C4A7D2F6CCA35
                               EEE79997109C7DCFF250A353FDC32834
                               55AA68C5EB7C30741613C3CCAD65B640
            stream[256..319] = 916D59366AFD20B58056B55F83B52755
                               404FBD696050C1520EDB3AE455314D68
                               9F094698B7252315859AE6DC3B7146A4
                               5249C337A1881A4147004B8E62ED8F25
            stream[448..511] = B120226D4936515678C011E198151AB6
                               BF2A2047623CB706EF9A4512B3524E76
                               78BE6F5A834008AD14BA2770B8693DA3
                               5401988E11816B7BE2A7F9243E290F2C
                  xor-digest = 93AF172B224A2038A613CF2D79BD0D7D
                               7E401E03CD689862FDD61283A4BE0AD4
                               E42F968D036A8DDA54D9D86D481CC738
                               9B4FF57050D6C680DCBD63684F3DF756

Set 3, vector#171:
                         key = ABACADAEAFB0B1B2B3B4B5B6B7B8B9BA
                          IV = 0000000000000000
               stream[0..63] = 613C8EFB0D205DDA40169D61287EC948
                               18DDA41DCC49BEE0DB3054C9CE8E4E89
                               7292454A94BFB6EBB2FB0584E5DB8870
                               F6FF57B964D9847DD7EF44EDF4715585
            stream[192..255] = 2978F08A57DC4A243F59F387654FF981
                               E3F07E2D711AE6E7D384120D05DB4DFD
                               0A452F64B3DA7B23DC2099968B6CBC04
                               F7332AADDB765AEA590A6889DC49D51E
            stream[256..319] = 72200777E6080069A41D6B756557751C
                               0829E698CC1F09F571D7DDF00546EEE8
                               073CB57E5A755DCED2741378B4DAED5B
                               76BDB9F50D807C1DA96BD129FE116AA1
            stream[448..511] = 69A08B376AD8850F7C75774BC038B75E
                               4B163242B0BD8122401022F9DF56CFF5
                               C3D58EC228AEE220ED764E485A7287E3
                               B5135F6A139218C8B1F51ED0481E5829
                  xor-digest = E184B83201BC0FC54DEB236FC1B3CFC7
                               B0E94F55042EBB704D7866CADA3A08E4
                               E65B26B76E4DE62FD923BB2E22CBB14E
                               6DDAA9F5D0833A399F27A9F1B980153C

Set 3, vector#180:
                         key = B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3
                          IV = 0000000000000000
               stream[0..63] = 7BFFD6D7DCB9FF327697ADC2D12848D1
                               7BA15705CA2970C9933C32B64E0BAA59
                               EAC9B06E3535B0FC7FFAE74A5111D6F8
                               99CEFE939F9AFF49207855A2F10B175B
            stream[192..255] = 48D39C1434C75BBE42B8CAFAAE6C687F
                               B7428D650D0C9E00724053090A26081C
                               9AD32440631C2EBF511FBCB1EC3F0312
                               F35AC981EBE01F0843F9496EF66C7794
            stream[256..319] = DD5309EC9C2BE3C3735E05E0839ACD6D
                               2CA2C01FB569A1A0D982D19450686EBF
                               C6D426A5A6D7A51381196F13333D7A52
                               6B91EDA2A52FFCCEEB06E7BB60C14C49
            stream[448..511] = A22AFF21C9C7E653BB8801E8F39E81CE
                               3FF2989BD49B038D226ED1A579F38E73
                               2EDF7BD34C7EC27BCC0A286DBEF6A678
                               9DA85A830344EEEACFD69551023881B5
                  xor-digest = D6FBFAADFD01DE36505348F103EC3213
                               3ECCEFD135B9B9583BDDDD0AEE996FE3
                               1625609FFA9F495429395E5B4190F1F5
                               5FD473D23DD06965EF2059CD460D8975

Set 3, vector#189:
                         key = BDBEBFC0C1C2C3C4C5C6C7C8C9CACBCC
                          IV = 0000000000000000
               stream[0..63] = 1A9DF5590D8771C3AEC7E6CB20694711
                               275AFBE88E8985FF397F7E907F8EF989
                               C25D2C5B64781E4A1691D0274ACFAD57
                               77BC56989181F2908510FD4235AFD934
            stream[192..255] = 72352AFEC181287F5399E7C0996E1080
                               248574341A1695FE3A6EC3DE036EAE4B
                               2F7BDD624648AEFD05F6326B12CA2E18
                               A6800B8025ECBB879F53A3FA42C154FC
            stream[256..319] = EC71A134059D4518D9DB485668893EB1
                               061DFD0ECE314422F43D7AEA70714974
                               2F89854CAA39CFDCEA015944BC530B28
                               29B60DA10C47F00098157A79D934D9B1
            stream[448..511] = 036B7C6A61E8C4E223F6A7243A4819ED
                               310F3FD445791044F17F4E222ACDD710
                               F8A8B6579FDA58BEED97F7F0654FB910
                               B9E634D4EE5B8C06A4608FC5BCCADA9B
                  xor-digest = 4C202E08A01A28FE33C2EC0FABFF6C19
                               5EE25F3407D248DF91BDB9636F083BA6
                               AD5090170AD1263B87505A08B667C015
                               A41A9BBFBC99A8AC0FC1A4A7B9F8F150

Set 3, vector#198:
                         key = C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5
                          IV = 0000000000000000
               stream[0..63] = 9FA10DDDAAC350A1F78BD5673A4A6FB0
                               0521EFC93AF3574FE71B7EC78A99E131
                               320010E4ED662E4563AB09B6B21B6523
                               EBBC240F7FFEFD9BAA62A917984E10C2
            stream[192..255] = F18A19233A4C37312E479C9E1565F764
                               49AD73DDF7E5E281C6351F0FA1A65F57
                               9845C6C855051228645F0369881EA998
                               278DC5FD7780AED780557ECD34B1D074
            stream[256..319] = 7D3F4613D5AD1877AE002E2B12E71434
                               F4DC04D0618AB601D514730094FDCCE2
                               5485640A7CEF20460C5D56F2608A086C
                               D98E4DE795371F4F2992FF290E787B00
            stream[448..511] = 282CC627EA1EEC5F8A40F48B56DD6EC3
                               5B163FC8FAB2437757DBF62D7766C601
                               727BA2C632DACE61C02C5135EA9F8676
                               93D018B81A22D3AD527A403DE0914CBF
                  xor-digest = ECB149C6EAD3648025A47AD7F0F4081F
                               4811CC09197FABFBC7E5B5667105F94F
                               79CF9C0E4C29236FEE28B213181113DE
                               5F998A9AE6140EA6B6666E0077D12B5D

Set 3, vector#207:
                         key = CFD0D1D2D3D4D5D6D7D8D9DADBDCDDDE
                          IV = 0000000000000000
               stream[0..63] = D1F08D8C8E95707FA2E9184F556EC73E
                               4ECA16F39D27B14081D0F7E7D67C05D1
                               A94BF246B2B2D3F3467BA596A5ED65C8
                               869BD4DADF8ED011B8C33A33FAFBF5A6
            stream[192..255] = 937FBBB1C29C3AFA03FD4AD05F43EE3A
                               0F3CB44AA0F404778B5750C1248D3566
                               FB86A72CB1B74EBBB13852A1A0A9D21C
                               81AA7DB5D4FE46B2A9FF89A066A46E9D
            stream[256..319] = 4E4856115CF07DA91BD54ED41B390807
                               30731DABDD3E26194E063867DD9C7144
                               44C932AB01DF56FF78EA5545962AE3A5
                               FE075AD67BC7854B82764B522583C038
            stream[448..511] = 96E0E817BA67F3C72A7DDE649BD7653C
                               D2B719C8DDD2E84CCE9A2B6E61DE55BF
                               A9907DD9A1F1434E78D685C61DCD7992
                               91D61D393CF883CD6713499B38E2EBD7
                  xor-digest = A3A626CC9A396E2615A6893CDE75EDB0
                               083E2D617388F47D1DDD2BA94A6F19F3
                               3840018C11309EB666C60488FAD0EF90
                               51500F52CA9B2B2D46F2CC4DE586428B

Set 3, vector#216:
                         key = D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7
                          IV = 0000000000000000
               stream[0..63] = BFE4AFB5BBEE1F71F254DA01F751627A
                               9B752EFDEC2BC9AD6385A2E0FC8C0DB9
                               82036A87FE2A55100B8E741F6FAF3FDF
                               5776BCE0AC121C1C3722D47F4DE93315
            stream[192..255] = 79FBB1A72383DCDB92BB1E8E7E4DCA94
                               A60DC4944F5BC29EA15A802D9DE63000
                               D433E935FAAFE80C4AF4DABBE2D852D5
                               5119631389BEC8E4D9FF1C45D3BA6ACF
            stream[256..319] = E9FFD92D8423E1CD9D22A776C7EFF47B
                               4FD2DB15DD08DD8474B97F4662760B68
                               37E03EAA16DCB7D857B2BCA7DF11A210
                               1BD18852167C929256442D2875CD6888
            stream[448..511] = AC1CC9EE33AD5E544A8070EDB73FA55E
                               7335997AAFFE2D06DBD891A13D202BC7
                               C0BC2C9120BD580DC508E010E9F55B6D
                               5EC84189BCCEE8495F5FE9189DE12A0B
                  xor-digest = 1F092F2D0087085EB577733C30CA70F2
                               D75E78D19C0915646A8AC3DD0959FF64
                               B2528FAABEE57AF5AA562CA9DEAA12F5
                               58BEEEACCB4AD4E1C7C969D69B54B6A3

Set 3, vector#225:
                         key = E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0
                          IV = 0000000000000000
               stream[0..63] = F216623931049297AF5957EE91576935
                               8B4B1719A2AEB5BABF9E5D154B326B05
                               8FCCD2455FED38875F6FCFA2EA5AF3C1
                               930ED03BF536E1B4B1692A27B2373372
            stream[192..255] = 3AAC0DF2509AE42B77495CB79A3D841A
                               305648225D3A30671235FBA6F957EA1B
                               CF4BD302AF56C6B7473040A8E3EA393F
                               5FCD8FAA1A8AEFFD297F8ECDD320DB57
            stream[256..319] = C049E4E06A2A62F92CA4CB7C548F82A7
                               253C8BEBA6DE150E27F867E3EE9FF138
                               041CFF16343E41076C9FA651B8A9FD24
                               78AC9AE3544A32605E543628BD9F7635
            stream[448..511] = 72E338F197C96C3E2EF8959AE3C91658
                               4AD39506CF846236C5CA71FD0731C945
                               A2626360DDE50EF086ED0A297F041E4D
                               C8C7B532D97B56E7E5643106AB0D43E1
                  xor-digest = 15D8FE0ED72672D71CCD70AF00641CDD
                               5493A6C997F05D47E258E875D5399582
                               A02D9B09ABC7F7A7BB4C1BFFE3A2EB4F
                               80A2871820A009880F821F5AD82D55DF

Set 3, vector#234:
                         key = EAEBECEDEEEFF0F1F2F3F4F5F6F7F8F9
                          IV = 0000000000000000
               stream[0..63] = A3C7DA822802F2C5D821F4E6D4147591
                               37410418EEA3E8A46E8C9A99FB1F102A
                               2F6F04AC7BF10114BFF750E5D1A9F366
                               D117AF9CE88191FA04442B3D58159FB5
            stream[192..255] = 507280D7AAEB20216CE74D0A3BD67853
                               EACEE2687FD705EE33F930199DD80578
                               8C699E1334CBB81D8B390BAFB1A3F64C
                               6719101BBFFB835E738CD20D590D0E86
            stream[256..319] = 78A9D106A485BC68EB5CCA176A8ECDDC
                               386394B59DB79BA6E849BED84F7C8DF9
                               C489FAE426E574749A18585C1A0B3B54
                               C9027FBEAB8D43C2A049711FDE0544DE
            stream[448..511] = E2F09B30F3274DF4DCE9C6B66FEF9F42
                               7CA8F18E59E16CED86BD108726AD65A0
                               25675D6C397B075A8F2DA9FC6F197CEA
                               EAED1E13B4FEBE0C76309318F9D681AF
                  xor-digest = 30C7F2FCF068E213FE9E9D604171331C
                               8C7A161F9625E655FDC10A6A755865BE
                               62B0E68B34C9BF8D5E67F998A3D05733
                               E7D47D14DB2ACBF243013B8378C3BE5B

Set 3, vector#243:
                         key = F3F4F5F6F7F8F9FAFBFCFDFEFF000102
                          IV = 0000000000000000
               stream[0..63] = 9A80437B4C5DA35067C0E86CDAFA0931
                               36B1034568C5EABB72C7C47C12A7678E
                               30138CC730E23F7A5B688D6368B2C4AA
                               9AC0563124E5E16E53742AE78FAED8D4
            stream[192..255] = D7B7B44300F54D5E21B5A19EA3239993
                               A3D0C6BC3CEF82216AEDF4B756F99C22
                               AE85E6D625C6C884242C09FBF815B176
                               A417A359E764EE0237E6C78E62B5393C
            stream[256..319] = 10A116FA29F470265B03E0A83A9D76FD
                               B541174C2D921CB776C78B190B76B0B9
                               2C218A291BC201BFFAA7A309BCBCA016
                               B6C47B99EAB7E9D3FFAA2DC413BE179A
            stream[448..511] = E926BBBCAD67A57CAD0DF98AB2755318
                               87A97C868E93D8AC2D2ED3ED0165C41C
                               A5A5F00DB54132E667B4A5BC803FBBC7
                               073A19A210D7B27DC744F87165D6C2E3
                  xor-digest = 07CA4706A8B983104B42CC6D3A889AC4
                               22DA578FFBA6CAFDE3C43142F6A3D61D
                               262B3E2AF75D02F7726EBA179F07FFAD
                               4E2C1D1D7E3F14A2F32EAD1246E6A850

Set 3, vector#252:
                         key = FCFDFEFF000102030405060708090A0B
                          IV = 0000000000000000
               stream[0..63] = 62F253E12CECC98AAF3C22E89CD4EB55
                               623CB74D4244B36FA82E6341DCAEEC20
                               C5DEF4639F447DA6EFB5A9F3CF264B38
                               2509823644C013E2B0204636FE26ABBA
            stream[192..255] = 3B9B855A964E0C6778CBB6F5AF2EB74E
                               F359A782E8C12A76AE03EC82C8ACEC88
                               8D573E236368AC4C118FAD826184B610
                               5420700039734ECFD3C71554DD238A5B
            stream[256..319] = 997FA19FC4E1D3D0D79691AEFDF9F308
                               9628B6000FA4F4AD24C38C498A28EEED
                               1F31467E6926FADEDCC3B3381EFFD18A
                               F7B40DBD47C54568C2ED6F107868B52A
            stream[448..511] = 1ACD6317330621AA3D901CC2D70E6ABF
                               2F48913C4AAC536C02815263903DB1CF
                               5F1BB2475B30D1E1F58FEAFD30F510D1
                               47D5419AD3B9427B5899CFE6985602A1
                  xor-digest = 51C7EF9556DEA1FDE69C71C6A64221FC
                               33C63690514F356F3AE8F132A979AEF8
                               95E237D811EEDBDAC852D6AC47A6CE45
                               0E0A7A7E363559C3BA457882A920E9E5

Test vectors -- set 4
=====================

Set 4, vector#  0:
                         key = 0053A6F94C9FF24598EB3E91E4378ADD
                          IV = 0000000000000000
               stream[0..63] = A1F5956D10E53E41180CFFAA3B8B7D8E
                               DA33CF3E197F52DF22CBA222AFCE1D55
                               3C53DDC4F0873BEE714E34B4FBBE466F
                               A507C94236FCA32DA00E95E9C2BA44ED
        stream[65472..65535] = 147B13171C4DDB42263F68F536EC524E
                               CE906DD84264B152B7ABEC487B40193A
                               4221BA8FCF81E98092DD091FDB198C2C
                               00C9866D2A341D8347C34B354EA1346C
        stream[65536..65599] = 9F48A9D58E55639741E205E70191EC12
                               FB4A72FB54C721611B3FE1E90A93F62F
                               5D5A9850A694B40C65DFAF11E6BA1B72
                               3CDC00E0B549889BCD7F68B8996AF850
      stream[131008..131071] = D7A0D1281015F8E9D19E6D5A4650D0F7
                               9D1E5D18F4183032771E323A78A46BDF
                               307C31BB8D7A7944BD4E2356E91759C0
                               C1149E07E7C7209808BB45BAAEC6FEBF
                  xor-digest = 285DE6030515BCEB9CDFE1D833D41C0B
                               439CB7EFAD46D951FDDE2643E5D09DBF
                               BE4ABA9D88DF2AF67E75988BA1C61CBA
                               BD5168EA8092EB2C0FDC56D57FD7BE83

Set 4, vector#  1:
                         key = 0558ABFE51A4F74A9DF04396E93C8FE2
                          IV = 0000000000000000
               stream[0..63] = 091FB6E98D0C86961E4551BA1AE4BFE4
                               D4A6CF75F1FDEA0AF4A24E2F00A18874
                               09D66E2169FB0D0655B8B05C642FC01A
                               2A2BC4982E4A4BED6ABEA4E66D7632EE
        stream[65472..65535] = 58B1AA6F98D68D82DA62E206FD0A95C3
                               870F3BB84267E126C872C5E26FA6700C
                               F05BDEC1485ED1527C4CC6AE33455C4F
                               FBF157CAFFD6C8F51AEA41FFE7D5B17B
        stream[65536..65599] = C98302DC16F953112900DC0F33DCE74A
                               0DE74E0DADA02BDA83903D36697994E4
                               B3008F7E1F4CBCAAE84269561F2B42AF
                               2AFA02264F74F74962C85836915AEDD1
      stream[131008..131071] = D78DC1D6B9C74E1F26F973DD46F12711
                               F08FE7E8F170803435D237913038F49B
                               6DA74457987C0B0603C8B3ED5A4E06D3
                               CB480BF7453B1DD19AD9E7A73645E733
                  xor-digest = 9606653705505DEA9B4465CC1A229769
                               495A3BD5E42E788C49C393C0DB675805
                               1E203077638ED2B812C90DD33B0387F2
                               33A3EAB068D9FD6E24DD6AE405DF30FC

Set 4, vector#  2:
                         key = 0A5DB00356A9FC4FA2F5489BEE4194E7
                          IV = 0000000000000000
               stream[0..63] = E66B5DEB6404BD60AC89512A06C2A1B8
                               19C18EF46DC52271254F8AB2AF1A5DB4
                               89C5E0488222C0D780DB6DD2F04A9980
                               CE24E0B98E6B8A3824C7F1DF8AA982BF
        stream[65472..65535] = EEE4CD03A91363326DED2FDA955AD34C
                               F420608DCDB756B74EA4E17DAD69A41E
                               5F8CDAEFF82EB1AB9EB5884633938D9B
                               3BCFB0471BC005A242D48FAC14854DB0
        stream[65536..65599] = 8A07A1252147C0C748E222077D1FCD9E
                               868A4A2D0316E0081BFD241349CD56BF
                               47D1F05DCDC93EF380958604F6B3B925
                               91EE893B990DF1432ABDE08CB87C6F9B
      stream[131008..131071] = D316553A63887BCABE8F766A737BC41E
                               3C0D7C91D7D753AC407F84C8E4A0DC44
                               F782A1826E92D7E5C782C4DAA10700F3
                               42B6E6E36A86B5BC3C9EC65C6836D97F
                  xor-digest = 140D55CFB34D2C0D5B49C48DE6213B27
                               C1EA25846F98CA58025524FE32A0968C
                               A01E12C83285011FD7FC0C84FD0E7680
                               13808BCFA1B363CC9A1D3B085B94D04E

Set 4, vector#  3:
                         key = 0F62B5085BAE0154A7FA4DA0F34699EC
                          IV = 0000000000000000
               stream[0..63] = 87DEAEFDF287A53E17E1F229C6ABA15B
                               45BE2CF9A43E03F097E2B1AE4983274C
                               AA95A13B2F12F776D09A0338E824B806
                               409E2BF4B76431FDBDBC2E60ABDCFA08
        stream[65472..65535] = 15A9748C457B794334BDDAD73C0C6F3B
                               55EBD8CB6AF0B3FBEBEE27AE6F01E7B6
                               B2D265DBE8A917C98C92A13120D06267
                               1FF252ED3D21BC5980DFBF246796BAF9
        stream[65536..65599] = 65718EC42E807C67EC3902B6562B577A
                               0FA7B870673DF5FA05F14588F5138118
                               CEDA98B62ED8F18A3481CE99CEEDEF76
                               11B9656AF3988E187251A693A1E563F7
      stream[131008..131071] = 4A416B9F083234191D157569E551BB33
                               48664FC2A2C6873A5F6A437CDFFD0734
                               19CECDB26CFDC628837C9C5B23581E42
                               E1611BDE2961F3DC46908EAB6C8EC5B8
                  xor-digest = 3B105651F6FA69B88D06E86FF7DC3FC0
                               8A248974FBF732F4708D1FF893297CFB
                               2BE3DB8EC19169916DA207F6E070A79B
                               9266280B2585E4F44548712C3DB23EF1

Test vectors -- set 5
=====================

Set 5, vector#  0:
                         key = 00000000000000000000000000000000
                          IV = 8000000000000000
               stream[0..63] = 472251678E82F653A05AB5E7240365AC
                               3141ECDAA952DA2B5DDFE52DC6F7E8C4
                               703E48C7EC5A7A48A51E4235602E1D2C
                               214C9AD181BAD0038E22650FF6F5C8F6
            stream[192..255] = F629EF5987FC4EFF6DEEFB0E06BA7EDF
                               E5C58B95DF1DF27D0CF8DBFA5CB56B78
                               988194F3B8089A60712001B3CECC9D81
                               DE66AC3E92AE685E52A2F3108398A6D2
            stream[256..319] = 78AE6BF0074315ECA839492DC62707CC
                               850CA3F5B73DF4CC0267FFC50EC21D20
                               43AEE7580DA658763CFD2CD2CA488312
                               97A3D90268965C50C7065C6823B3EC64
            stream[448..511] = 2F827A849ECD9F12A1B8FC60421EC3E0
                               FF869E096E1AD5594F31E781687C8AFA
                               BB6224517D04D3AC715D4459EB79037B
                               B574C6FE170D47CBFA8D5AA1D27808C0
                  xor-digest = 7288E2143E8E378A542DEF5AAE2E8F26
                               2AC9C78B9EB627E107AC01B922B3F584
                               DFC9727AB8FF5C9D6D987E858AD590D4
                               F8FB9B6EE08AEAC7372FC2E463601CF5

Set 5, vector#  9:
                         key = 00000000000000000000000000000000
                          IV = 0040000000000000
               stream[0..63] = DCA9F0C7328C0D00896F3A8AF0AE9039
                               C68CD8E0B4F8641032C8C46531E7945E
                               D9C76756F9B208DB476E6CAFBE32ED69
                               F2671E0FF9EDF1B1D00AC5541E58B0D1
            stream[192..255] = F5DD29AFD81492A0B8707F2345518234
                               95294AF432CAE43434FC0F501871AF25
                               4E99B1DB409C4BD20BA052D97A83BB16
                               13F82477EE3F7113711D289EBA51DCCA
            stream[256..319] = 3849353AB57F6700876D9260D34A1B62
                               DEF88327A99A63DE2FE0C8BD43DE6685
                               46BE2599C8570B43D833BD5B9236FC96
                               AE5C19E6F7D15B56B6925D930D414513
            stream[448..511] = 69796448153827076DD0DEEB7A137E1C
                               4B6C2DF4775754B5BE065749CB5F3DEC
                               18B12DBAEA3A6CA9176651657FDF76CB
                               B5A844F2DA5EAF43551E9A2504BD880B
                  xor-digest = 8978CC47B5FB5561C76F9AB31017778B
                               286B6933C0C1E4CAA8F624A1F60384EF
                               64393592B97AECBE412FBF7D195CEEEE
                               F6863641AF2F4E9E473D2992ACA3EA54

Set 5, vector# 18:
                         key = 00000000000000000000000000000000
                          IV = 0000200000000000
               stream[0..63] = 8EB7774E5EC6869056565767B351FC16
                               A7CBCE69FF44C44341CBEBFAA36AB1B6
                               990038F7AD25D286D2C0D70A70F4D0BF
                               760D0D825363EB397F5A796CE0F6679D
            stream[192..255] = F3C02A63C144C954FB8A5D1BDF1D1491
                               133C5040F2777D805DF6CA59C8827BA5
                               39F65D047CCF71A03735F821CF278EFD
                               05D9577569E270E871923E39742810A6
            stream[256..319] = 39DA7BDD213BA84DC008D02A2BF4A554
                               57D2D17540CED48E8A8F99A8FBD3FEA9
                               EC92029BAA162C913C985DDFDE5E8EAF
                               350D5D34F790D8A2F7EDFA2A4E5F27AE
            stream[448..511] = 9DE0054021939CE424C13FB9E97F16B3
                               FCC8F2E61CF92E36F0C5B50CA483BDFE
                               F243A35B559A57036A7DE9E071D04A34
                               BD28DDD2309BD698568BB4C762DF87DA
                  xor-digest = 7DB6B434FD2C6951B09D5B1241CF90A1
                               F07E69607739B4EBACD60ABC1BF2EBA1
                               CDB606EDC66388178A99040AC5D48CAC
                               E6938686E5B60427EA5D4777C4503273

Set 5, vector# 27:
                         key = 00000000000000000000000000000000
                          IV = 0000001000000000
               stream[0..63] = EE94037B1AA04C6947469FFC96B5BE2F
                               3407679BFCBAB69B0A3B63D0D1D37CF2
                               9AA443B236CE7E8505B60C6020A98936
                               26600A22213633DAF818D2CD8AAAC3A1
            stream[192..255] = 1E02522FD7F3D9DBCCCE100A21A84DD6
                               E38A990986EC2CD3C12932F56AE76246
                               913C03C3318A4BE75259C8AA6A4B3106
                               25DC56ED95A366DF8F6FDD4900F4248F
            stream[256..319] = C134B74F0277195D1015C78DF776CE08
                               EB3DF3416BD24121BFA7BAB106539EE4
                               C519115064618CBCAF22AC21998949C2
                               81CAEE8B5567B508CBE9EE42E2FCD8DE
            stream[448..511] = 7A827B43AEB818E05C26923880A2C516
                               7B996B7CC567410DEE2F0341C2041AC7
                               21FE6750991E83B0C84F40FC460788B6
                               8F0080E763804A36BEA97E40E7BA24F4
                  xor-digest = F6B1CADAC448B58CDE69FB03AA10B701
                               DD33CF9CB6259C6A1B214E707FB692F0
                               D951B185FB59C1EDAD1E5EB4AECA9B08
                               B7A267D5F9878D180D198853B712FC0A

Set 5, vector# 36:
                         key = 00000000000000000000000000000000
                          IV = 0000000008000000
               stream[0..63] = 8C03FC8A3D1AAC8015012626A2AC1D68
                               3B757CEF6BE6DA25D2B63115FFD02932
                               04996C7DF715A857631336AF68536D5D
                               CF1AA416B7238D139A6F2E94279ECA6B
            stream[192..255] = 050D99876B2C2ED7D8ABBC2C9DDA5DDA
                               90F69363E7A5283FF1349F16B1E1D311
                               88644C64378D701BEC2259C2C84E66D9
                               67A81E4154C801DF145B80AEB9FC2865
            stream[256..319] = A9E480A21C7A14CD922CA3424F962CD7
                               95DC2E5AACD57101CD9B9FA3B0735F22
                               0AF0D0E89A47B95084B3E50731606307
                               9511AFC432893EC4C01D59D8DF48B283
            stream[448..511] = 16211A7DAF005F4278357135DFFBD5AB
                               A12EB5F997646A4DF275F0EC56E1E425
                               FA5145C9431E00B94B2D163856AAFE3A
                               058CBBB368EABAF2BA38B99451C7497F
                  xor-digest = 07394AB07CB7A366ACFCD4D7F9DD02AD
                               DB8497544D4DFF12E5279EA54C6DE419
                               A45DDC65DF3A8311E604D24EAF2DA30F
                               8C9B482E87BCD65D6C67CCFF0DFB1319

Set 5, vector# 45:
                         key = 00000000000000000000000000000000
                          IV = 0000000000040000
               stream[0..63] = 830A8F698485CA3DC7717448933D139E
                               C3E47C392C028D60BA4A0830F39993AA
                               D8B6312FB2B878D2316FB1EEDA48DA16
                               F5A931BA8EBDFEBAE8062F16448673C0
            stream[192..255] = E2C6B92FBAD227F290A7A9D1531A8D3C
                               F3A917C82EB11D06906A4346AFF319F5
                               2775D6C7E875735CB27F2AEB47602447
                               6B082B5456451E59F32D7D686D7089F2
            stream[256..319] = ED0FA8F6AA4FAFE305B50B3BFB15D343
                               3CF3C2FBDC26EA22FD338E3092CF241A
                               CA7811A6B43926C44B3793B688249699
                               D14BD7B96B673D1523AAC4319EA20BD7
            stream[448..511] = 16AE0F67EE8D38383C99346E6CF4B2C6
                               0D9BCE5D51079DABE12ECC3DF35FCC87
                               DF262C0C29682B4781B6EF6C78FED514
                               16315F6FE4D13E35161F258F265D844C
                  xor-digest = DF2AD8769752D0C96BC203E2BFE0CE51
                               A06960745BD6BB27E9225F3BC03C363E
                               534220F2878D06049BBE7D962F65FB68
                               39114CEBD2F1FC8BD1DF4456AF73E159

Set 5, vector# 54:
                         key = 00000000000000000000000000000000
                          IV = 0000000000000200
               stream[0..63] = 21BB7990348C2D479D705887028D098A
                               665EC277854EC889A9F74FCFE6AAA0C7
                               4AE0925DDF53B60AEFAE5E4E0264AB90
                               B7B9A3BF997705E84C5287D7FD999168
            stream[192..255] = 7D5C6AC259F16A873EC6D7AFE8405326
                               350AABFFF35FB606751883A82FD52129
                               A6206A2362B26F00315514F280C9463F
                               025D120635CDF53D774A42D6D25DD505
            stream[256..319] = 9583BB2B1E762C0AB62117E6C3DC71D2
                               C9AFA9C6A52448FC32B1D0AA0032B7EF
                               296AFC632D840815E183B0C4C4223BCA
                               42EB1E7C7B59F61A657A3ABFF5FA167D
            stream[448..511] = DD760E2C49D6044263A94F16299588C9
                               2D231044196A3716E6D4EC5F40DC3CDB
                               4974F425C1150F414D7EE7C23F0BAD1D
                               3F8FADA282157BB259E02CC69F46A8A8
                  xor-digest = 30FCC419AE938C3DC100014EF4C8C485
                               515444DD710F6C3F31C4A5D8D541F3D5
                               909A8FEBDBF9D66A8B3707311D96FC77
                               82549AC53B629F4CFF86C84EAD1D2A8B

Set 5, vector# 63:
                         key = 00000000000000000000000000000000
                          IV = 0000000000000001
               stream[0..63] = 12767FCC2C2272AFE2885051D9067C2F
                               6A90198FCC2791127AA2C4C60F0EB034
                               15FAD1B68FDFFCD45856B4D41C2BA14E
                               A62E7CA54DF9C4ED578C977089561EC1
            stream[192..255] = 1FAA9DC03921FB02E2585AD83F0C7FE5
                               73E7D0EEF5DD8506F75832C3954E7446
                               86AA581696BE2EB08BE60DB1FD55CD85
                               503988D4A6D02DB35F99808FCFE2C0AA
            stream[256..319] = C1E0ED9DD57E773DAAF0F64D59E89055
                               17ED01A17BD285583FB4637C64AF0237
                               E1569ACC212CA0B7C71BA61711C2D18C
                               8C160F8F977FBECB551F4EDE059D2FBD
            stream[448..511] = 22E09B96DF7AA5217DD9420365AD72A6
                               187BD671C128F654B3574E184CC126EF
                               E76CF3792FB90C4A150F9E38C378E0B7
                               11674B61A75D85D412967B61A1580269
                  xor-digest = F94EF3ADDAAED466EDF2BC980BCE9A71
                               F00F03188ECDD7989714F47B1413ECDA
                               BA150B8DD479514BFFF379B52BB2A330
                               AB56A0778A70A8B6CB651C6FC14178E3

Test vectors -- set 6
=====================

Set 6, vector#  0:
                         key = 0053A6F94C9FF24598EB3E91E4378ADD
                          IV = 0D74DB42A91077DE
               stream[0..63] = 75FCAE3A3961BDC7D2513662C24ADECE
                               995545599FF129006E7A6EE57B7F33A2
                               6D1B27C51EA15E8F956693472DC23132
                               FCD90FB0E352D26AF4DCE5427193CA26
        stream[65472..65535] = EA75A566C431A10CED804CCD45172AD1
                               EC4930E9869372B8EDDF303098A8910C
                               EE123BF849C51A33554BA1445E6B6268
                               4921F36B77EADC9681A2BB9DDFEC2FC8
        stream[65536..65599] = 227CBD5D7AAC2DACA9D3A1D86E8F7628
                               ACF6787019B4FBD77EF63478C19A51B4
                               9F30EDE4FFD8623DD321A35D615FECF4
                               8D97AE7B33FBF5C0DE5E6B4CA286002F
      stream[131008..131071] = 11D4302F3C7A3E406AF5AF012787B688
                               2FA8339F65CB1D2C5FA794A0FEECB9A2
                               3F3702D754F3C3D66DF6F528F5E7BB71
                               EF182B4231B142A1845191D38F0FC578
                  xor-digest = F749D03DA53D00A655D2838FC0ABF859
                               8CAD7E8CA2B07D4F9B52388EA4261B66
                               51E180480B9564819C068F43594C59F3
                               211C498A56EC1257A5D90BB8E03FB45F

Set 6, vector#  1:
                         key = 0558ABFE51A4F74A9DF04396E93C8FE2
                          IV = 167DE44BB21980E7
               stream[0..63] = 1BA441EA65D9A3E7AC17D2BF53A1B987
                               CEA9E0B112AA62D55B62E7BDF207C841
                               B73FBCFF77B8E6ECA376023AA19C4DE1
                               2DE0E74E8F428C309680377ECACF0F26
        stream[65472..65535] = 9C28D7AB26676CB3A971CCCF8F6C82A2
                               5E6C7BB937871B20A93BC36961E821D5
                               7A9A3A29D7E6AA92EA07095E100D0F51
                               256E43A055D6558B349452F6D346F422
        stream[65536..65599] = 8E0829DB5BCAE53C76B340AA85F3EBC7
                               CCB867957681D0E225DD9C2430383279
                               1B4DBB74CF22F8CCC228C11C0C137DA5
                               5D463DC22B2F12F871991F345C5928AB
      stream[131008..131071] = E13A1C5FF0B146050B890EEC6A914616
                               54FFE14C4204DAFA62A24818DB3C8A1C
                               74DD045DE609E371920CCD8576A4E941
                               1CF0ACC364E599835526B962F9842158
                  xor-digest = 3582DC0AF7F3AC561463091B3D2854A2
                               A607E241637302BF7C6EDCD49D5746C1
                               8661C1D25A51FAA8B7BE8563E7426FB2
                               B4E0EDC240182161EF961E9F98436C8C

Set 6, vector#  2:
                         key = 0A5DB00356A9FC4FA2F5489BEE4194E7
                          IV = 1F86ED54BB2289F0
               stream[0..63] = 2E5DD6E06C30836003EBE45FF08E156F
                               497F1493C8F5D244E82D64101D24ADBD
                               C30071E5A3345F7023F42A5BCE77E202
                               40F81554ACE2A277CEB68DF0DFE38FF7
        stream[65472..65535] = C4859A0C0760B39C79FC57E5196C3395
                               B656E2DCC0A1298921BF41ED63A27DFC
                               0953F9104B757415C128664163DE7F88
                               F2CEA02C079E12C8AE3E25B0E5CECF4A
        stream[65536..65599] = 32E910AC7BA4236E870D888883CED15B
                               FD2523B36A2E994C1B06C17E53BDF9EC
                               E0DBDFDFE869E4CDAD83B60CB431D727
                               0770820FFA9DB6E7534B08E6A55421A3
      stream[131008..131071] = 9756C4798B22B7716E69D119764D8249
                               2B96B0FAD9460C83CB178812C4A78A4F
                               846E404CE0189ABAF2D015E9389F2A07
                               674B1065EC9BD45EF6ADCAE5579BE0ED
                  xor-digest = 7217FD4D1913595DB6A837A594A2627B
                               F08C7764863A446261AE5A4EF4F4FADC
                               3D2C040E3D08C3CD4B702B7A977C8700
                               4ADC316A1F8958B612661D2B0A591A10

Set 6, vector#  3:
                         key = 0F62B5085BAE0154A7FA4DA0F34699EC
                          IV = 288FF65DC42B92F9
               stream[0..63] = 32224DA57726457257BC771AD362A4E7
                               868B71C660AF74A622CEC4D2179B8C16
                               6CF4D9942BC8FAF2B7E4ABC537FE47C7
                               1CB7C861448B15DDBE0ED54688CADF01
        stream[65472..65535] = DA0C4633EF6446BB09F2F55FD959B20E
                               940B5C9372EF631ACB603CD1A20950DD
                               DE5EFBFCDF3EA6C6941C1ADD22072458
                               CC611177FF6E9BE2EDA335C2D846E8E1
        stream[65536..65599] = 67BCE4E9D7768610F8E157524369F727
                               B85E6A649F8A90BB2AB57CBEBED587DF
                               497E119A7244C4BF186E13B7A4E8306E
                               8BB6E21F5B7AC1902DC144FE86738821
      stream[131008..131071] = 64F6FD593AEB95BA4EE8E36C850B9BEC
                               3241F8FE72FCEF1FD7B0F4C245C74651
                               D9C7C22F61C9AACE0BF757128075C5A6
                               020BBD5F20819A8D5E6B474B3546D60D
                  xor-digest = 4794F7ED490F7F0A9923036BA1E1E50F
                               52E811B63B3935D851D92E86E4BDD638
                               9028D17DC34D8ADBA58E9EEF54FC53F0
                               B7925B8BDA29BA73F46C52AE52A47798



End of test vectors

Primitive Name: Salsa20/8
=========================
Profile: SW & HW
Key size: 256 bits
IV size: 64 bits

Test vectors -- set 1
=====================

(stream is generated by encrypting 512 zero bytes)

Set 1, vector#  0:
                         key = 80000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = B1F599E9B0D96DF436AE31F5EF589565
                               B92D245DB5A1D4C7A78E5E8D0146F8A4
                               9D326C1A3BF50C052C9C8F114DC74972
                               C4469591E31C9ED11927AA9871F38583
            stream[192..255] = 0C427AD0D68D752517649EC6D311FE7C
                               D71DBFF6E6217B91A83D45F33F5E5224
                               BDF86D09A132884B152105842B5BDCA8
                               6A0B1ADC568F5C04B4D021A2AD3E0E26
            stream[256..319] = F6E86C2E6F768A167E484A4116A63322
                               FCE370FE40623A13856550E6A3452478
                               D8257E4B7322D3B98CD8505E21C54A31
                               D78F9444366C72AC9B31FB747F7F2592
            stream[448..511] = 53BF865C66A344CFCD19177476A05ACA
                               5851CC45224B196ABF3206D899E7FE3B
                               13B3F028FA849B5564561A9181EA69E5
                               12BC34DA29180CDF6811E40A9A06A8D1
                  xor-digest = 49D1E5B6AF120ACBC4DA7CDAF3A5DF90
                               0FFC60D671AF58AEEF9F13EB2C182755
                               C7C105AA3986BBF3F9BE4D70A190A0E5
                               C220D40C1A367508F79730E900478C62

Set 1, vector#  9:
                         key = 00400000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 4CB569029A0127DD65804C8012699F64
                               D65CD08465CD01913AA15609B2D35DF2
                               E6BA9D5FE91CB92BD90B8DF2F0EA0FBF
                               6AAC7BD2A15113DC760177392B262BD5
            stream[192..255] = 7C3919176346880B76B9C35B0A795D1B
                               290511E7F12A5F245675B0A14B0F538C
                               9DA8427AE3854E9ED15C4C2E2FA2C2E6
                               D473827114B77E4E71FCE5BCB65B9A26
            stream[256..319] = B25050372A8BDB17E770128CB4F9A17C
                               719DC8011FB46D3DED31E460F83E5062
                               D259CA69D9B6D5705897415C84A1AD9D
                               4982146386500027CF407937BED8D26E
            stream[448..511] = 9C82A7E1DD2465AEC5217EE872D3DF71
                               9DA787F72CCB18063BFB2BF53CAF01AA
                               A3B11ECD69883A86C0780B06A508FC2C
                               4651F4C31C39B4070D8F6D2CD4AC2670
                  xor-digest = 07D09FC6FA2B01E6DC525BD9245FD45C
                               540F2AE6FBEACE56280B35AAD326BC1B
                               E6FD402B0B6AB726B677A9023098D44C
                               9994DA8CF944B6BDF05DF1EBEE51EB16

Set 1, vector# 18:
                         key = 00002000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = C0A04790E17A7A886926ABFE0E3D5E58
                               63E56A6FBC83C0D77FA3F25A9E686427
                               519E31D1002611C8FCE04D66A28BDD15
                               61EC451BE47829EC2FA1CDF9BFA67EE8
            stream[192..255] = 695121BF3742871F4434A18215C6AC86
                               9A7942DE2F6F690A7E90DF81F514E0F9
                               F604B8BF7F0CA50EDB683F5B993D718F
                               85A2E7B2291A2F25B5BF56074F8E5DE3
            stream[256..319] = 1F8E83DBC6D2BCB52081EF727D4322A8
                               426064E91F77AF3D4F4EF2D89CA23B8B
                               BF9F33A77F877E03A04A3AAA6668C4F1
                               6F986B9AD75D2F44C4C5F0B6F54B5CCD
            stream[448..511] = 141DFAF364F212DB39E6474D83B5FFAE
                               A16EBF06F9B8FF3285FF49183891F210
                               2EB3FD2EC15A8995B265D43A0C4A0688
                               B4FD16777F6FA59F8BD43E3DF0E10533
                  xor-digest = 9D54797BCE381D777FD5FD3F7A31557F
                               E049E858A3A438F939612D121DB40D9C
                               08A48300967D606D6249468B40717541
                               71A67AA1CF8BF70393D2506E625EE4DA

Set 1, vector# 27:
                         key = 00000010000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 07ADBA33B91F2442F4FF5CA8BD0F42C9
                               638D1BF3FD502E32EABAC76D42ED1FB7
                               3DDA60E286C47702A7C1C25C5FAEDCFF
                               E82D3CF45B0D0067529AAEB322EFAF0B
            stream[192..255] = 0EAD07205E46F5296FFE1501EF519C12
                               00E14862FDF94B158648EEB670A23834
                               597C45CD4ED741210CB2706CC92FC772
                               C6A4CE9F943516F983D3465D4B110109
            stream[256..319] = 279402FA512C11465E6B284DFD807DA6
                               4BF1C57D9FB56F22B699864C53055E69
                               EA0C92BAB3579A7303A6BE57183F6FA7
                               4D37A3289B17DD44CFCBE114B2F17141
            stream[448..511] = FBF61845C9876405CD3F870145AE8481
                               5152F3096E5C6CADCB0E5315994635FC
                               8BF7464C9C485ADD489CFCE9A9726137
                               604508267FFC4B51C30B9B8CE9209AEC
                  xor-digest = DC5375238A230A60CCCF708D023831C3
                               42D8EECEE0CFBCBDFC367925F6C20D3B
                               ED686B77D2EAEC2B228A45C7E4097BF3
                               F6E337104105819703C83E09F29AD293

Set 1, vector# 36:
                         key = 00000000080000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 9B861B2BABF0E5471083DCD672635228
                               1A9E4FE7930B84D2F54D44F177A4A816
                               BD5612A5E62F04BA4525A09774DC1D52
                               B623C19063E6C47E10A227313D7CCEBA
            stream[192..255] = DA9DAADFB3C80EE4BA989F19244E1455
                               69DF581AEE6EB2465788FB90069B090C
                               B86974CFAE8854521E40F55F894BFD89
                               A602A3ECABAE65D688268DD14A7A0272
            stream[256..319] = B62EE6B6B96CF39221BA017B69485B4B
                               8B906BD9748359676862CB1A573A1DAD
                               21FD852886D1917E22BABF8C8D1166BF
                               C3930378CCACEF5E147F451806A0E998
            stream[448..511] = 7F42DAF4A82E6CDFB55E0A39605DA988
                               5DD6305D933F22EEFDDAEFE5AAB21F8C
                               E3E236D48597E511EE7FEBDF2118D476
                               18A1BCC340744F9B99763E946215B337
                  xor-digest = 8D6758522C2FF11EC704DAE50E403C36
                               A6388F4E1D0EEA6EEC0E9D2AB81C9785
                               65BD094BD406143F870F5A3F6D427250
                               29FA2CAAA287C2B47632CE1DCB675E65

Set 1, vector# 45:
                         key = 00000000000400000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = EFD63D5F2EA5E92AEB9497CEB1D9890F
                               4CF4BA101F6B7B6F249C8F374245722A
                               D3FCE9F513C43AC16354BD586FB8EF28
                               916D8201879CAFE95BFDF7CA50EC4860
            stream[192..255] = 59BCA4481F0DFAD0EA84AD04A6D11F75
                               780FE012D2CB4D98DC6E4CAC594B56E6
                               E35E932DF1FC517E317F6509522D5DEA
                               8E87EFE299443F6D66BDFCA965F12302
            stream[256..319] = AC241F1FA1588689C330FE1EC23EE831
                               377C2AC1109E45FAA234967E1A2F5BBA
                               FBBDAF34CA55FB87BCF1028C92A463BE
                               630AEFAF54B55C0D1F6BED248AA47192
            stream[448..511] = DCD6FB74A6272A2805B204A50B05C8F5
                               C2A1C9737617F0ECD8ADDA477334F659
                               804BF91ACC9F8D2CB17920497B1EB869
                               82196F1D747D412D2BF64B68F4EBACF6
                  xor-digest = 90F59875E07D0A48AEEF182C2C3EA28C
                               3A74EE0545F2166166BA830D6370162D
                               604FC2D004BA140A34C4654450FE5F2B
                               E5E8CF2A783F3374343DFBCD9E63E185

Set 1, vector# 54:
                         key = 00000000000002000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = CA042B270792B9B52C590A84E62E1B50
                               CAD3779C8E289C56DD2D51D948AEA365
                               B0F3E7DF129FFFDE8E5040DBD24A3873
                               5F527D0C644D149AC5D24107B1CD8B7B
            stream[192..255] = BFFADF9484E10627ACBAAB8F40485CC4
                               530670AA9BD9212E7C9A60DDFDA5E907
                               BAD8C68871208D34286AFFB3AC196151
                               AFD2D923DC42B44205C95E3BD477252D
            stream[256..319] = 5C990BBD0CBE3C1EF941574CB0782104
                               C2E9B5A907DB7D2B63F5B193EED2423A
                               524C8DEF1C0D7BD6ECA189B9D52D0AB7
                               8C4CDE063444F2F220318C251D8DAD7E
            stream[448..511] = A503586BB9140732CE9221EE39C22F06
                               CC4974F9908C68A4C60CAE1D5AEF395D
                               9659D5DDFE29559BB95EE03F799DF676
                               B74DCEF03E249A140446D258FF806F61
                  xor-digest = 77FAD92BEE362C9B37FF7B0FDFBF768F
                               35DB5698B9E61EC3AE3548AAA900F6A2
                               9CC73F1FB05B84E6D6B645B133AEF05F
                               B6A7CBE64DFFAA3C67FAF919EFA1C95F

Set 1, vector# 63:
                         key = 00000000000000010000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 447536116FCEFB706304EFB5959CA278
                               6F8F99F4EC41475403B1B296DEB39A6A
                               7B727213948804C896651BE8B109B311
                               52F419E77778E8B78A3C6F0CB6F3EA12
            stream[192..255] = CA38A7080D9B997A113AE6FD9AB64876
                               842F9652D68C897AB1152D65299A79F3
                               C6BC389A6E88719C10F5EFC6E16BC251
                               8B6132A97396439D1A6ED4C172E7D9E5
            stream[256..319] = A8E8682144B118BF37DE2767809DD270
                               698B95A2AB8D60B938244970575DC4CB
                               A4405C50521C14A014A1937CBB6C6050
                               BC94F395272C509C68D38F384700909B
            stream[448..511] = C9DBCBFF3A5A1E7694BBF1B095A1DFED
                               7AAA951462FDA270A9C594990F187DCF
                               77BAA00E35F6D92522B2546ECAE0C549
                               9FEB6C398ED2CFFDB42D79E6D27BCC01
                  xor-digest = 69DBA63E3F931CA3C389C163B436A393
                               2E130C1F3AD10B2EC40C976922E50F34
                               D2F7FD5E6A7F8431F793986B6249B00D
                               2411FB03476D230A514D13C6160A408B

Set 1, vector# 72:
                         key = 00000000000000000080000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 519BEC70D73F65AED973F1CA27BB09C1
                               15E1CBF35B983DDEE060A8874B958ECE
                               DD9A2BD8F34E449A920D4356FF5D704F
                               96159D51F77C72D960AE20603CBE8DEB
            stream[192..255] = D934B37B0122898DE47979A6320A2339
                               C50270DA713941BB36BB8673596E3A94
                               7F419E36D9BCBCB217766085E2BFAD15
                               5B95CFEA99827B4375D7EB872A4A450D
            stream[256..319] = BE6D58BF305C672AB4B88A17C34D1735
                               5D900889F71871853D4F96E516606007
                               6734F63D91F63C78FBE33A548E7E3F98
                               F50193F48C5DC8AD84CB73EA7190D4CB
            stream[448..511] = 7591FDC95089C12DEAE0D09AC184DAAD
                               C82196221E44F6B64708F8569C884AB4
                               6CF600E10171C27D6B79D37865BD258F
                               AF812ADCB91212154393DBA8CB929AFD
                  xor-digest = A7762C9455180049C6C210FED3F21A5D
                               062C2EAA3D5079F0F855FF7784BB2FB2
                               440293CB863EDC530A2EA4E0AC7A8571
                               013704A1F0C21C8EF3E7B2B37D3DB18E

Set 1, vector# 81:
                         key = 00000000000000000000400000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 8700CE466B97E3A2D800527A2C1FF8FB
                               47C9FAAC61DFF5933FD4D3FABD7B3D60
                               0330BC60048906CC5985BF4DBE596D3E
                               CAFE0D419A93495953F46F269734064B
            stream[192..255] = 238AC7EC0C6A93320F2424D50219718D
                               FAAE74CE5D00F8EF75818B3F67FC5ADA
                               D6B1DFD0D2819CBB7646636618815E50
                               4DBEB131C9C5769656A49D9147F7092A
            stream[256..319] = 14F5ED9BFC53D4659BE0B2A7156FF011
                               88781ED21AD718047F09ECE0E49D95F6
                               853743B3CD3288B8C4AD2A0E80BA446C
                               7C8EF3534B4AD897E824F960F00E4D2E
            stream[448..511] = 2021937E8B5E89C749C0577133C99265
                               ACE44D66576158FB23D1E25D91483CB6
                               EC6FBB5841E434D1B5A825611283C524
                               28B5953D04340C5568F0823BE8AEC8DA
                  xor-digest = 93F5F5B07297F5A4BACA9D7C012C960D
                               400585244F0B7A9E6A6E69B39932CD3B
                               505F00765C49CC1142B9245976430110
                               44F9E02B6CD9400D9A50F9DABEAED5AE

Set 1, vector# 90:
                         key = 00000000000000000000002000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 72202D20E067C42A49F5645B8D1560EB
                               A4630180DCD525856CE9B6676C8E0F5C
                               C2AE48A28593A7699976C28BC5B14B62
                               077097FC95CC07A9DD4F6E61F8979DB1
            stream[192..255] = 4D629A3E50D7B7D65FF1B1C88A2C7C04
                               A33D6174303C341DADD45A31E513172F
                               D0CE19EDE7D20B0865F7995999C4AE61
                               B8C5362F767D9C3E720F52647A9870D7
            stream[256..319] = 318CAD0E25D89C16D7DBBD747E7BAF35
                               DABB3174BCFA045EBCFECFE98AB958EA
                               B860BCFD96276CA3FC019C01ACB0B8D2
                               E913E018A6E3BFB7B72E9792028770D1
            stream[448..511] = AA7BF09450349CBF872B926E56CAC975
                               2F283DA6BFB57408E30AF518510C74A8
                               B2C7CDDE5BE0D73130F65D18961AF873
                               602C3A4F96D69F2C9D1121BFBCFC42F9
                  xor-digest = B2332974E876A8B779A1794D1CB9A1ED
                               D4D82261C9B8992E6109A398E7C6D469
                               24D46C6532668C21B5AAEA2C2FAEC623
                               DF752CD4D7D23B3537CDC9F67A18F213

Set 1, vector# 99:
                         key = 00000000000000000000000010000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = BF617D33B6D76276B90B009F2E1629C1
                               8FB4644C13D8B016D6825D3B3C243933
                               36467E8FED98F3470628A3365C1238A6
                               2557E564A3E7E63D4F38D0178BB008BB
            stream[192..255] = 121A21F86100F080ED24CEFDC40C7BCC
                               73F972DDBE185C7AAECC222B14E85ABA
                               A68987F463157AA3C6EE7007EC9053D9
                               37D63204B57F477DD38F12ADA39305E8
            stream[256..319] = EB5C3917709D8259E0BC46B2D5D8D91B
                               215E6B9CFB767C2BD20CB78B1E317024
                               E548123E2BF286F9D97B4CFE20EC9A54
                               CBC86E4ADE6D132B0CA51B31F527D83A
            stream[448..511] = 011EEC5019295F0A57C0080125EE678F
                               25753D98C89B84DD2A43693EF523F637
                               A58BC4D3F87A8C7B5BD534678DC9A927
                               1AF4931184C848BD45ACD781BFA34F13
                  xor-digest = BEF2FC04056939BAD7EB5903EA31FC15
                               B1C84C84B830872BF32FBF9C8E766B57
                               A672620743ED2B281E3422A416FF940E
                               DF36283008D368A2C75911C12A6E5392

Set 1, vector#108:
                         key = 00000000000000000000000000080000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 96442969A2C1DCAE04BF53D9CE9548FC
                               DCBFCEA8E359F44D38BBD6E3CFDAE006
                               F3B82C55610AEB8EDCD97418FB80E332
                               17ADB7BAA671183CB18207BEC9815BBF
            stream[192..255] = 3BE157F78E46B64DCBE7160DA4EE7469
                               5D62D99256CCC55B4A52C15A00C6EE04
                               254DBA97CB076B2659DB7A27010F588D
                               FD2E4FF758A152F49C3832DB15453AFD
            stream[256..319] = 7C1089E9717EA02791F17783A908C24C
                               45C7954073FC601F9C76413F00F34A14
                               85C9D30A735E47ABE0EB411E188CB58D
                               A11C21377210C0206E990D3EFEC086FB
            stream[448..511] = FBDCAE406023E5A63192616A230A1CE8
                               161BCD7D8F53A036A2122F1C5B43824A
                               0DE3F44233F6B53081752D38580CDB36
                               8CF2B9C40F47864C92A969A8D005834B
                  xor-digest = FC5952C6971DA1DD0F3E5AF2BBF0B59B
                               4DBF4FB8E8670B3AE8F8E44C76A11ADD
                               0C1B6C69BFF35D92CC5E193732087997
                               88B68F21BF932117DB89D07EFAD9735D

Set 1, vector#117:
                         key = 00000000000000000000000000000400
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 10A84B30D0AEF7DC7A03915BB4EAAD37
                               7F43F8A1D5FBA4C8C86A0E2003AC7E53
                               47DA3690CF9F67B614343E8329DB664E
                               F5214C17A8C80C81342F838D53323C92
            stream[192..255] = 8CB02D8A4470E72D38A8BF4658D7DEA4
                               60830816D331E1BE0560BF877824D2F1
                               9F215CD189469AAE71C5E59DF8EAD3BB
                               19FCC868F8BCE27A004E1FD8EEAD1FAA
            stream[256..319] = 328965EA1972678761A7FAC57CE2B616
                               38D091E805D3D80FF8C89A60276881A6
                               0FD5275FABB3B4BCAEBE3F4F1FCC5830
                               8F6C7052DE6199FD9E68309E2428AD4D
            stream[448..511] = 9E6A09D9474DEC4BD4855DD9D10301B7
                               1A337D046D71710C378D43099BC920B4
                               FE8DF45FA221B5DA69C7BFB159A90DC1
                               0E32CB9B16D0275F4FFE9FFA58269885
                  xor-digest = 1EE2BA9F4B06813F5A3D07060CA33CF6
                               627114562C94191A7B5BFAF360A620AE
                               DA2A096A57D324CFDE21739F40636B7B
                               4491A674E6EAFCD5D68F88FB4B509844

Set 1, vector#126:
                         key = 00000000000000000000000000000002
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 4F213EF1F637777C8415D708A4528C50
                               B9BCA3266A8A4E833D0917AE389E10EB
                               375FD20F0B276F171B967E93DDAA9ED8
                               38408EFB6F3C863C409AED7A71EB3083
            stream[192..255] = 7F99377159EEFF8214A76A2066B6995C
                               56D28AD7D8AFD65575ED8DA9F26EBF7E
                               32D6348010DCE043378B633CBA4103F2
                               36E8C44705D318DC64BD68B30316A128
            stream[256..319] = 17B7242DE05C29A81C99DEE4FFBEFBF7
                               3CC6263AB735AB75955055FEF4298023
                               560FA8C1A075E435E0B7054E57EB8830
                               E634EC4CDEFDB5136725259F8B3E538E
            stream[448..511] = 6AF6813DA738CEF56D933B449AFB8286
                               EF986CCDC94387E283034061E7397747
                               85D28C9B12C48876CDE0031D8666CF51
                               D8CD2DA3D665BF2BE82421D9DA3D91A1
                  xor-digest = 4FA0B2B334897E5C7FC06AED04EA97C9
                               EC594C95BCB4336B09FC5A29212185FF
                               AFA959C5C72FE0772927EE3735448472
                               958C2E3609B0FCC14E10C5884FCD1B6E

Set 1, vector#135:
                         key = 00000000000000000000000000000000
                               01000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 93B9B64F1CF33E32AB793B6C011D9D55
                               F30ED3468ED2804AAD47DBF37DADCF7E
                               835EE714DF2EB7F735EE0B6F7977A4C0
                               C12EFADE37DDB779302387A4C2BBDE99
            stream[192..255] = 7EB6BB8434D3A21045DEBF33367706A5
                               37D99FFB920EAD1142C26A3827E5A629
                               16F664867C8B0D75BB504A522E26C91B
                               30C5279650D28B62222E7221E6BCF5C0
            stream[256..319] = 0FA62A7BD369C9711E46B430B72A4697
                               5CCAA00F5C10368F73FA235217415572
                               8AF6FE6F88C2DCD84B8E61E52A526EA4
                               791AC2BB17E5E3C24803B70DC85A9134
            stream[448..511] = D40442FB3A73590BB338263E75636B4C
                               C35FC8E5BC69B5158AFABF263EC75CCB
                               F46DA40285C90F100EFF8C2D61996B1A
                               A6EE155C9DA4E268EB9B4CF6DECCDF96
                  xor-digest = 1258509D8CEE47D464B1AB7FBD755F0E
                               8FA37399C669E0B1E58B51602E6BD55B
                               6E23225865BBF5CEE948E494390B504E
                               7BE2D89E920B7DCA88B821FE32112F0A

Set 1, vector#144:
                         key = 00000000000000000000000000000000
                               00008000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 2ABD4F4AC5206509C5D881C4EBEBE70E
                               9D449E81D026C1C91A6AA1E23605246D
                               2A6DEE518CBE2B7F106FCD104C176614
                               3DB4412A56F112AFE7B466C2EA345803
            stream[192..255] = 4850BBEEF99E7140942709515032BB62
                               7C77C95C8F2B78397F94DE35B7471B09
                               21208B475369CF843A54EB51BEE4E696
                               CEE39F24DF13DCF8B4C6C0198C2B5063
            stream[256..319] = 5D2C579DBD5623C5A8DC55C6F56922D5
                               70066703851217A856E512E5D0DF9862
                               C635D71D1BAD417FD0404B55741B4B66
                               A3DD966EAEDC7CB832E9178F2852C32F
            stream[448..511] = CDA231E1C20FA371404A0E4416FFEF34
                               80F84C27541761EFA64753341762E22B
                               837D492A671BCBF8F653DAC91EE9DA7E
                               48097BF80B49CC1B035CC9F2D7965114
                  xor-digest = 60635D5975273E169A2CD1DB9FCE2309
                               0139649729C0C4468E604865B6DFF800
                               38576336E80459F69A97A01CCA1CBC0F
                               321A27875A1C5BF43B43005B2181F660

Set 1, vector#153:
                         key = 00000000000000000000000000000000
                               00000040000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 1721330A66CD0691D553CB18DE9E5766
                               7014598D8146EE13F7347B687020C12F
                               936CD399F7B7CA653939E86A085AA5F2
                               8D2B52A05EA55E192B9EBA9E8775316C
            stream[192..255] = 14561C341893FC8428F529260B27AD62
                               4686DA00F3FAEC86B3FCA66693213BEA
                               13055A0C7C6CFED044791D07BE39A29B
                               3F267A120D26CAD0C546AEA3EC77FB7F
            stream[256..319] = 64C6553F6B6B591E2B69E73FBA76EE5D
                               5AD7532BC4F2C79CCB4BD71ADAF73D9E
                               91BF0CA33D74A364216E5C6A466F57B9
                               62B6BE4EDA87E6C2A34404032E99BE5F
            stream[448..511] = 8BD6CED81C74859FAAEA46487D5DC621
                               99BE28D2D6F986985A55B2581DB2174B
                               4EA1A1F6E34AF69EE060EA06692349FD
                               4F75595E1D1F041996BD12B818342E7F
                  xor-digest = 1D144973B630BE518A8FDB2C311C86DD
                               D01B0B4BECFBF3B0A750530C942B684B
                               62EE9CFBBE3CD34EDD162B31CAC86605
                               F15997704378056A3C7B4012DBEE1CDF

Set 1, vector#162:
                         key = 00000000000000000000000000000000
                               00000000200000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = AE0B0C8E1E83B245F8F7D9E1EE8AC0EB
                               551CB5F02F89C233BF6F9D255FED4EA3
                               0D050B09B4F93BBCFC13D92F0AD2B92A
                               08BD102B75C7B460DBD0EBA78309E4A6
            stream[192..255] = 5BDF8FF9D06CAC965535476FEBC33EB1
                               C02601FB7794A0ABF883DDC0440CE7C9
                               25147614FE7EC19C133975FCD5037FDD
                               4F8674A878A8C5DD9118CA458EE8A23F
            stream[256..319] = 735F2EB52956E2E90A36FB7CB6173108
                               78F204A9541A57005EE230C3B48DFE46
                               85CD0A2A0A2524BDF0D9750B845FCA06
                               14FD99A42CF5C4FDACE838BCEC5409DB
            stream[448..511] = CCA2EAD7C6F773AB74FFB4306C0BF663
                               6FF4DDD03295A0523DAB96F79904C644
                               D67A4B8A194D469D2686A84FC34144CE
                               2BDF552222B62C6BE25EA5F946C41C6C
                  xor-digest = 33E7B12281A2E4F36ABF7F232A9AB873
                               F92D08AF10033D77DD01F44CA788A259
                               124610BEBAFC0A843253BB788694BBC2
                               F5361E43EA6BF71101E7BCFC0D15157E

Set 1, vector#171:
                         key = 00000000000000000000000000000000
                               00000000001000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 1E99CD4AAD296CB1CA8212A4379316F9
                               685C46F41FD2A16D41391C3EDC5F29F4
                               6E852AF7E148560ABE4A041C17607D0B
                               A07011D870495AF05C2B8170C9D7EC00
            stream[192..255] = 1B7FD751EC259E504E2ABF79BCFAA73C
                               48EBC50E2FA79389A40D2B1A8BDC095A
                               0B0DB6637601843E65017A3162663D5C
                               B8BB2EB6E8E4554C822B0C2C71F4D703
            stream[256..319] = 49415396E1B5DE631AB0EF5D1678B7E3
                               61745B5BE630B703B8742EF81828E8B0
                               E3659EC3CB96CE85687815DC156BD97F
                               47470A16070502664736F9591DBD4092
            stream[448..511] = 801592A56923CCC15DCB304003AD6365
                               2D0FA02745A27951D250CAD369AEF453
                               CAD4357CE78D37B2E7FD0FFCEE0129DA
                               EF81E6B7F745AD973F639543C9286FEC
                  xor-digest = 4B010997F497D09C30FD7121C8CD905C
                               0CE3F6CC89FF2670C60615F33CED3370
                               A30CCE37A595AB986912BA6E3AD1C2E7
                               84E45D0D9EB801204CCFED471EA83B1F

Set 1, vector#180:
                         key = 00000000000000000000000000000000
                               00000000000008000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 9A2BEDD8BD0B375C0A75ADD271762671
                               EA5185A04B157DF446731E48234AF756
                               4A4CF77DDE456C01051C6B7E4B75E9F8
                               5F7B3835DAF3089161B34D1EC5A9387B
            stream[192..255] = 949ED265D2A7C47E09B0C2C68B51E867
                               B0E2BCED4C6F1BDF34F18BB3542A1A42
                               97CE27D8E5573B90F6F578187D734E17
                               66D670C9F3581CC0AF5540FEE65CCFBF
            stream[256..319] = BE33D393CA7B090BB75C2A4311FA2F51
                               FEB72A898FFAF9511E1930A7594D195B
                               AD346C394B8934FA06FC5FCF67C46DED
                               9AA390F66390DD96B5804C8AF2F53042
            stream[448..511] = E3BB844575F02F946779A1CF49CA86B9
                               F8523EF427A75754C225D632A1463B94
                               08B92BF20CF653FD4DAEA3495092D152
                               8B78525E6D274A9E0542B469D724630F
                  xor-digest = F3E1F312E101F545E58C55CECB47E7FB
                               F83775B7E1D5893CDEE5330E0B9BE4B2
                               0D7A6B792208ABB400C42A566EBEE0C9
                               CB87EBA01B5278925167EAF90EFEADDB

Set 1, vector#189:
                         key = 00000000000000000000000000000000
                               00000000000000040000000000000000
                          IV = 0000000000000000
               stream[0..63] = 0CC2D49A59818C0B74EFD0A555B0552A
                               6F030F836F963E2CB521593C0BB6A8BB
                               C2EEF63A443FE51264F4F7D9AE304E35
                               0CB90D7823DB58C59E4852ACDFA1CE34
            stream[192..255] = 73EFFEC58DF9641C6FFFA948AA1091D6
                               AE3D95DBB04FA525A7AD22461274B332
                               ABBBF6C8C2618D65D68191EBF27AFAE2
                               1D186243B97843E9CBE6642D4D75AD58
            stream[256..319] = 0201BA02840063D51C7941E0611F86CB
                               954D4226EE4D339309A28F8A84FAF0E1
                               52B7436CAD5A50045618F877F04F7595
                               D11E466A2CA6C9063B66FED078036FC3
            stream[448..511] = 5E695FF23AE6809FE3B01B33507065B2
                               7C249ACD3D4B7CC0111D245CC4C03357
                               9EE05A25BC853BE98095E4FB35DA6326
                               0D1BAC735D5174A54E0F82E1F21FC60C
                  xor-digest = 92309EA0E6AD6726DB6206F0F2B4E4D5
                               F2EE891AE518DF98793BF15D6586682B
                               5B1AB611287F1FE908B36D6F83E30A49
                               5735CF123BC47DD8319B28CC47F7B7D2

Set 1, vector#198:
                         key = 00000000000000000000000000000000
                               00000000000000000200000000000000
                          IV = 0000000000000000
               stream[0..63] = 59423E6E9B1A21CC918175AA7EE9402A
                               E3D0F851CCA4BB133065351258F86B46
                               B86D5B99B2A04FF3A6D875802ADC6E0A
                               7AA25D82D372AE47D781D938DE3C264D
            stream[192..255] = CFFBA066A54C8539BEE98D250FFA7050
                               2CE46F4980F7C153DB394E3716DE007D
                               F3BE6450A5DD7C07F63942FFFECB4984
                               739C5425E888EA0D1EEA6764FBE71E97
            stream[256..319] = AAE3D2C3C8AF44EC77DB8D85B6C8B786
                               DA367E2C8B80226B41835A22315421CB
                               724DA2B410052A31383D573F357712DB
                               7C090FC6B72B4FB6DF97C9BDC546F4C3
            stream[448..511] = 01CAD4D634885EA427463E3497A24CC7
                               8693ED36C969ACF63129F8E258DD33CF
                               24926D30B152AF4BED63704688B65890
                               C5CD1C11ADAC9B2BC7261BD8EEC6AF27
                  xor-digest = D3ABF9A42F5D38E479F6244AFE033C49
                               DB612CD455ECE14910E224D6BF82A85C
                               AA5FBCDBA15973C2DCA40DDF31C6F934
                               9615C2109DE780884B1C5E739BE07FC8

Set 1, vector#207:
                         key = 00000000000000000000000000000000
                               00000000000000000001000000000000
                          IV = 0000000000000000
               stream[0..63] = 24340A3925933EEC1B024399305480A5
                               B2BC0413253A55E3E207E1FDADB6C02D
                               663BC849006E66404959D0CF201199CC
                               08D3A2F5B43A4E46F811ACBB6C25C5DC
            stream[192..255] = 159EF67E0C3A6859D0B668EAADD38F66
                               1AE3E58F4ADE4C7B4B6DD69E74AF32C9
                               187FDD6D23127A7B52F8BC9A316F72BD
                               6908EB96E8A45CE55A0D9E437D340301
            stream[256..319] = 87FB9628930F05C6B63A6D35E079793E
                               4DBFC0A6715CBFE372983414D0BE145E
                               47FEC39F509385DBB1974C11D93A1696
                               50B0A3D9D88E242F03C7C3538835A8EE
            stream[448..511] = 81C30946B8AD366786BABEB3FFAF279F
                               8946AC7AC82D9130F0AEFC59196C56DB
                               3FEF19797F59560B8D03DFC67033FD90
                               E2268F33A911E632A491523678F6E96D
                  xor-digest = 554D7D2449638F619200887D59806D48
                               9A8A2556E546551C24250CEF82D7165D
                               34B04636E6A0CF53E2810C57911FD6AB
                               BA443BF607FFB04AE0499077B0C54828

Set 1, vector#216:
                         key = 00000000000000000000000000000000
                               00000000000000000000008000000000
                          IV = 0000000000000000
               stream[0..63] = E655934F2C201912ED5B7A1BDFA083C0
                               C695761B65ED14382DFA01578BC9E70B
                               0BC88B968361B6260135FE30F73A1228
                               BC0A8A381F3A9A39E07CF5696F9E02AE
            stream[192..255] = 984BBE27CF2DF2DC23B016D65FD8D539
                               18B0E87E018E3E77F750C3812993DD74
                               C7F5FFBFFC1C625C7886771019EBCE05
                               9EB679E205620E31517BFA2D8A7446F3
            stream[256..319] = B687D52DADF614FDE06DA36C8708297C
                               F3D9ADC18E700BFEAFB610ED48C44A2C
                               CF1A0F20AB470F4D6DE458C4A5E54321
                               6E8E08E5AB1978F268B756FF010167E4
            stream[448..511] = 12843C080BCC1BBD9DCE071D3C524732
                               57FFB02F8551C4352C849E1B0E0BFC84
                               382A089F3DFAF0BF0FA17AB280FE0498
                               253B6E2ABBCA5007D13F45E58120E006
                  xor-digest = D1B01B8ACA42B7D69EEE9CEA972BEDDA
                               517850DD23DF04F9011E9950D5680DF0
                               E681457A45CA4DDDB6383A7C90C01F60
                               F7388A468E375AEA65B68F17CE7D250F

Set 1, vector#225:
                         key = 00000000000000000000000000000000
                               00000000000000000000000040000000
                          IV = 0000000000000000
               stream[0..63] = CC6643F48ACDCF92546CD95FD763BA29
                               CAE0BB9A49AD8AC80F4806DF387659F0
                               2ECAF6A762BBBCA98DAC2AB63D443C8E
                               00C7546D1D9A6913B7789AF22344D58A
            stream[192..255] = ABBF25FFF6D469B4FF969AA334E3F45D
                               4274B51085CF929A525B6ABC7BCE23E5
                               1F4636B8044201679C1E8BD360BCC4F9
                               DDB1B153396A2FB1C99433ED173F6848
            stream[256..319] = 976C95C7BA4C47BD12829C74EFD8DE54
                               0A8080673DEAFB8146D3EF2B0D4D5822
                               9DB1AAC242456E82EC9075633D652D6D
                               13C52E7515D621A783485AEE50F1D584
            stream[448..511] = D8D7AA10029FFFABF0E4A37C97B79B7D
                               FE288042C59EC3FE2DFC3A5C85AB1222
                               DE83DEC02DF258D40CEA916285C8DB15
                               E1583AB9A723A034A08EAC54F86D1B37
                  xor-digest = F0E89C4AFBB4A73A394DD888B4A4047B
                               A5FF6B4377241463D97859E2B36EB3B5
                               96FC9E712D73A66B9314581631480D9E
                               FB78381498F173D8DDCBBFA3ACEBB770

Set 1, vector#234:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000200000
                          IV = 0000000000000000
               stream[0..63] = E0A3179484B3C28C11EDEB480C5745F8
                               2A5A2297E4308958A844667AABEC9B0C
                               5918972CFB3F2DE2A368AE8C7FADE377
                               B6244A498FA1C9057323F88E28215E67
            stream[192..255] = D8F3A3D0314529DB70DB8C82ED6B3B50
                               B331A4FEFFF324924B3DECF79A895D55
                               D2DAAE6745E981CE92769B331839A9F4
                               D3689A381010B2378629F8CA1EECF61D
            stream[256..319] = 26E0D285B77620FA653B95E2B9FC7114
                               0D54DAB219462FC8156460E865EBF278
                               1C22CC62717AAB0308AD43E4648DC84F
                               5D5EFCB5D731151F819B94589AAE8145
            stream[448..511] = DAEABEAF8F9BDCAAEF83849DFF5E80D0
                               8DF0C7C28B650E27CAA493393A520936
                               CD5D0F677EF1605023A6ED133F9A45B0
                               B7CAC749FA615C49C275D8AF655F1602
                  xor-digest = 242F93ADAA7E4735371F70B7FDE011E8
                               040ECD9DFFAF11233CCDE9F0DB50D982
                               000F109C6DDE7A53267AF7CD9B3D70AB
                               3EF657F5BCC2CA951AFD54FBFA246985

Set 1, vector#243:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000001000
                          IV = 0000000000000000
               stream[0..63] = C7455FC561090BD124FF876E0098AD68
                               FDEF6722E6759807D15CCC905EBFD3DD
                               58853FAAC0CA4144DBC0E97BA86A637F
                               F2BB40909024B144C287BDC4A5E62206
            stream[192..255] = DD7B556C974A18D612F0CF2FFA1F826B
                               ED8DF2BD4D61B713CB945CD1E3EDF862
                               873A9BB084CD0EC125242BDC5B7EFB09
                               7E7CF1C634899A30DE57394F65BEF9E1
            stream[256..319] = E5E338F34041779F0DD8AAB73AA5341C
                               0A8E311B99131D633585CCB63D82C6EC
                               F4CD1C6D13CD4586642C06FE872840BF
                               8320DA5C839B43815B868EEA224C8FCA
            stream[448..511] = 7EBD49D1552E0EC54D4DCFF4143D7429
                               467F49E46061409FFFCA0DBB2F014FD5
                               58DA77E1F049A47ED1F2346212B13C8D
                               CE762AF4F33C280BCADF1FBEDB287052
                  xor-digest = D0D6D92C686188AB848A53AB6149E323
                               42808FFCA03C93B1EC8ABB32F53E72CF
                               F319754A1FF358946A9C604063EE8D71
                               AE216644224922184ED670C191A0EB9D

Set 1, vector#252:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000008
                          IV = 0000000000000000
               stream[0..63] = 0AB8FD02B07D9FE49B557B1C474C812B
                               6B46A16D2BB8000710F67DC2613C51F9
                               B29E6F4EB5C92DF84EF2C7E8662F4008
                               EEF15E133AF72E0ACE417975FE15F9FF
            stream[192..255] = DCDA45C9E560DD7A13375CA8F7E36F54
                               971DFC4CEBEB1455A5239668EF583479
                               9C013D6E39B2676ECA90E8EAD76209ED
                               F6F14ECD554BF3B75CDD50404CEB0F24
            stream[256..319] = 04A9AB73A738B9EC7E4AE6593E37FBE0
                               41597FB99C0136EF8F2DD79C4D77D142
                               DC5FE3A93413D581A94E6072F842EF3C
                               D939E0EC4482C6017AC4C2178BA30D0A
            stream[448..511] = EE9222E8F3FF28C620E43DDACC8BB933
                               EBC78C402B856F20FF55EB53A4E4B242
                               9D628BAD7A6820AF0B54CC491B93A106
                               3C81C4BBF22B6DE292D0DCF8E8B16A83
                  xor-digest = A6A3FE92F4DDF512D460355CD12171FF
                               CDD9D24A09033BC92BCC2C3B001409BD
                               E0995A71C357E35127DA25603E55A210
                               F1733B5AEB61651F137E2477D072D11D

Test vectors -- set 2
=====================

Set 2, vector#  0:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 9F591DA5F99C235445EA91866EAD681B
                               977C4FFA036D770FBCA79D41FB014178
                               CF8ECF3164E5E77D7495DC0195081EDB
                               2F45C8A1B17D2BEC8DF3EF9FB7618075
            stream[192..255] = 8A234BC08DFF8CB862DE088AA0A5ADFF
                               0033BCCCD9535B83E30406F7B54BB69D
                               38F7E5B3A620095D36026D3FA6A71D7D
                               04AEFBA7D86D2EFC3DCB260C64F0FFA4
            stream[256..319] = 4F10E45BE6F5EF5E04DD2605B15A6172
                               A2F547A9AD7CD3ABD01E1640215ADB7E
                               FF80C93858C21A56E17DC361B118F56B
                               19A56908577B88A8AD0636AE624DFDB5
            stream[448..511] = 7EE3A5F3236BB2098539CC540A937AE1
                               75F7A86338E144FA1C6AD519EA07A123
                               8A4CEB122A6ED79A7F1318D1087E6922
                               51D73B2FF0778F3E0B31908918C6813D
                  xor-digest = 3CF25B5B65171A19AC9474F21528D564
                               15DC4B2B9075FB73AF0FDB3B9BE00857
                               CD6DC3A43920952696505EBB7E5E93F8
                               CFDA13175D18BD3B98CBDFBE6F4492D5

Set 2, vector#  9:
                         key = 09090909090909090909090909090909
                               09090909090909090909090909090909
                          IV = 0000000000000000
               stream[0..63] = 17DE94BEBD3E434305ABDBFF24B73C0F
                               AE33E04F3088F3D3842D089661CC4074
                               359C2B01B24D0A7645002D6DBE58549C
                               FA24E7BBB2D0A01AA97B0088FCFEC19A
            stream[192..255] = 9A50A2554917EDD01E43F06566C32E6E
                               C6EA97801F3AEBDAF1B9CDA9424E4293
                               E6235F79FA535AA9C88028CC295100A6
                               34624E3A1BC7474BAA7F3FEDCDD51DEF
            stream[256..319] = 3889FC4E6BDF1E48ED490FBD024DD70D
                               82796A854ECFE1F7CDCCD5A63C5C7B5C
                               3E5BC8B55D9B71E9ECC29DDA8E5E0F62
                               5AB1F50C7352420E2398B39C9868B84C
            stream[448..511] = 04E726136229B276ECD72BF75768FAC7
                               B01259867F02E39F6FA34C158AC0B3E7
                               4649DA7276220D7C637675BB1E2FC9BA
                               0B44B39E7B795F61CDFB5F2F9F78E733
                  xor-digest = B100A147765682CF0AB8021743A37242
                               69CC2E15DE974BFB7C5EF6B011738CE7
                               36D5B62D35F96E05D25FF4CBA6AAFECA
                               E157A2A602D2AD3D2044319275C97702

Set 2, vector# 18:
                         key = 12121212121212121212121212121212
                               12121212121212121212121212121212
                          IV = 0000000000000000
               stream[0..63] = 3A77AB8543D5506C22E4067E3E3C8E66
                               CFBADEC128BE90404DDC75721ECBE143
                               7F9956BEF51CA1F9F8E98541DE989AA8
                               2A36E0C8CF93BB6C641CFABC0A8A0B81
            stream[192..255] = 79C1DBC3073AA02090A21B9A15503E2D
                               AF0E2A36D36B77AAA49B7ED023988CD2
                               B8AD12328F0AA0753D813435A9D81DB2
                               63DF8AFEBAE01BC7C9C2BCF66B69022E
            stream[256..319] = 192DD4570FF21FC0224F3ED99662C4C4
                               B4FF6BEC22E48B33E0462E8F6887A4F0
                               E660F2FC69AF17DA12566EEE65536FA3
                               D22F204CEF6443E1E1814D1E6ECF5BA2
            stream[448..511] = 25614AA36A8EDDE2637F5A4A22207A1B
                               FF5E46F00FC4DDCA81D21C0847758748
                               C2FA10D8BEFED7DC301C4C6AD9E04A78
                               FA18A2B5A5A0072922E709F2ABE1997F
                  xor-digest = A33DF028C8654C793C1ECF308B9F2521
                               B3402FD0DBCDEFE02D8B1A4CC5D9005B
                               F572EB615F6CF0E029451BC80A9F2205
                               4B1EDB8576641293FE06B3847B1512E2

Set 2, vector# 27:
                         key = 1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B
                               1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B
                          IV = 0000000000000000
               stream[0..63] = 749BD4074F30227614C75239E42400F0
                               49C90E6CE14DC662ED466B95A26ED52C
                               906C96EC0F2CACD38481B88D061596CE
                               E803A957051C967F149BBC4C26821229
            stream[192..255] = 680B2C35EAF80BC5721740E361021D1C
                               4461058DE6890092C50F3216FCEBB875
                               F04DE1D0180662B06CD1F1B12B3FEF9D
                               F91162782B02E172194CD65014521389
            stream[256..319] = 57DE5A528C1D79A7D0AB1945B19B070D
                               C29887E44670E10261A31B2FD31F2AC7
                               84F59CC6924DA848D61F68E38D68C12B
                               DC6F7CEE7E5AE4E9095C47C8A3A9BD12
            stream[448..511] = 4A19EF588AA5893ABDFC4E5E22A7059A
                               B7868C175F26007EAB3C2A20E6BEDA4A
                               C5A0F4D3FE0B507B434664AD1C2A12A1
                               108266CD12C093847F53C0130C1CFFF0
                  xor-digest = 11A1441DA542081F4ABE927C7CEB1C8F
                               22A23698C5C86CC0D3993BFA1D6D9E4E
                               3EDF2DC88D8E0CEF9259E8225AC6D709
                               82C3E5FB081266AD96D5AAFCBAA53492

Set 2, vector# 36:
                         key = 24242424242424242424242424242424
                               24242424242424242424242424242424
                          IV = 0000000000000000
               stream[0..63] = 03C710F55D85129824858240AF22D6C6
                               002AB3396E1055AEAC09194DE408149B
                               0BAE22139CDFC3F8300E89284CCD0A6D
                               ADD4B8F6A949D95C33F71A8C56822A90
            stream[192..255] = A9D4337EDC62B31CB8EAF0C33B2D2B5D
                               52067F4F38B58E46257BF8356389F8AC
                               9BD1971A32669DFA27D096FADDDD2799
                               4867213ECEDEA75FAB9BC8DC76A52E0F
            stream[256..319] = 8C274A6B84153218AD5240AF34C3E170
                               B14FA74FE4D3BB91821B79C59ABC2BFE
                               EF9FB90BB15B1944F89A09532CFD968F
                               6760D42F4ECF5131BC0B2F3E6FE7FAC3
            stream[448..511] = F644E47F0A49EE249DDDF85A4B263FA3
                               19BDAFB542DE545160D1F72E84FAAB9D
                               C65B4CF3B316D2803AC7345EF1743DD9
                               290288DDBA99A8874BCC8B62F67D1272
                  xor-digest = 60DACD72279B83B9E4AEE20A6B454727
                               D3C1D983DB22534F3D09C4D1199616EB
                               4FB0CC6DE139240640FED8D0FE4FCC5D
                               359DED42948C62F41F28957E58FD2D22

Set 2, vector# 45:
                         key = 2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D
                               2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D
                          IV = 0000000000000000
               stream[0..63] = F1D4B91380013CC5804EBACB87038932
                               5A9152C213847A39A2246FCC73895A8F
                               B0DFA4C4CD27A383229A87B56E294E6D
                               66798616559B647E6A68D16F03D49842
            stream[192..255] = 0631D81FBD9FEBC7D2135210A3581087
                               8DC2C48B9349BD9B0538BBBAD1E3797E
                               74672741BD8CBA3BD43F1199D384BDD8
                               DE52D483DFA4E2AAB76D832FA2B65C0A
            stream[256..319] = 2C288C7B676ADEB31CCF97F39C6DACA7
                               960FFBE3792B3605582AEAE5B8BC8B5E
                               E09D6F02A7DD367D5B64F63FF38E95FD
                               35397D43C7A7A73C43C6DE7D79A1008B
            stream[448..511] = 30A20E916AB3899DC4CE0BD7E998DFF1
                               EBB94C5476E446F020EDE56E4FE8E968
                               99C3341DC8A727216E66A75DC48A7181
                               D2AE06C2C6ECD4B3CA91E32148E674D8
                  xor-digest = 9D16803B217853203D3AB370B93C4166
                               B1FFE023A0E0C00648CBFE26246DA050
                               00E48209EA2062BB7E9AF910BE8630F4
                               E2590159E9B72A5A8A1C91D191530CCD

Set 2, vector# 54:
                         key = 36363636363636363636363636363636
                               36363636363636363636363636363636
                          IV = 0000000000000000
               stream[0..63] = BA07F8D5A310A4A555E707B4A723BC6C
                               02AEC0E59D347E9EEB4621393B8598E2
                               20DB155A92407BADEB4BB2326A9D4AAF
                               145722174608DC70C6792BADBBCCE501
            stream[192..255] = CCCE849D66069FBF00E17B6A8469451D
                               1116726CB9E5B39F95D3471823EAD799
                               FD1D24CA929F7BA11936040CDC711C3F
                               2114600912D259591DD05D3266E51C24
            stream[256..319] = 5A024E63D55074677ABF8AF215B46564
                               F3CC3FB9ED2C1DC3CF6BF30C48101A66
                               F2C48088E8C5623610B9ECDF868BFF12
                               C4E854B37BEB8EE91E210CB2E8A64173
            stream[448..511] = 9FC9EBAAE6EF32724DD6B8CDCD0DF8C7
                               736B58E4BC8D261B6DBD5DCDDBE06598
                               D32D27827D9D8123341C20C416F97A2F
                               4400D2B2D1999C1EB1043B071F475131
                  xor-digest = 2E40BECC3B6473F1906C708399B419E6
                               4027405F7B9CE4A2AD7B6E5FC7A00D37
                               D8F36FA1D5FEF4A303C9198B9065CA18
                               E6A27A10076DFE42C49E8C43001E9BF0

Set 2, vector# 63:
                         key = 3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F
                               3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F
                          IV = 0000000000000000
               stream[0..63] = 1DC029760CC691E3BAFE5B88F011C5F1
                               F2C8B1EF1C45490EE291B8E727C1CCB1
                               4523AE2AF236DB983033A1B9A25129CD
                               E9BED6D7E725A2BBEEED9B2EE1D89940
            stream[192..255] = 2C67C33F4FA3F58C5D167ED97E122D10
                               3C185B9E23F2732EDF4494393A8EEA6D
                               8E24051BF1B55B2B8F850E1219B9F326
                               69EC9FAF428DE59AE3B8052F76FDBB4D
            stream[256..319] = 72BBFE5054B35B8BE30EE01B6D3E52E8
                               AE3FAC16CFD90BD6028647D6394786E8
                               659149263D36407C665BB2106DED7229
                               7F4864C7547D3CDC3B468F6972AA3740
            stream[448..511] = D3909C48153EB8346E19D19A86E44FE6
                               26AC398933530CBB912CE2E71C822AFA
                               121928712758D4C8E5C48425BC5A17F0
                               1C1FC5367FB9D30184062A987ECA5CDF
                  xor-digest = EC6FE6AAAA305C82ED442B954AF5BDDB
                               E04C9266303B55D8499E230BE9AB047F
                               EEEA542F827E6C3037EAF12365F58996
                               202A9A9F6A6E452778D840061AC1F405

Set 2, vector# 72:
                         key = 48484848484848484848484848484848
                               48484848484848484848484848484848
                          IV = 0000000000000000
               stream[0..63] = 5F43041F50CFBBEF9FB21D646D0043F9
                               BD79708A964E144E5DCE24641BF7D442
                               D25D3722E57621D8C9AEF76D2B3C118E
                               E43370B409A327C8118C27202492AD44
            stream[192..255] = 1C9E3BDBC49C0EFC9E039DC17AE15C71
                               43017B87CB040584AF8809559CF7BE9A
                               EE17130C5DCDBAD67EED43C23F3E7BBD
                               29122B9651E50EA023655D72FFF8B50F
            stream[256..319] = 6472C876499000BEF8DDD9B0EC47DCC9
                               60175DE66B5549CAD0CF8D90021858DE
                               4C8C4CB43878136B50D633BA51D01438
                               541F8E41A2EF10AB6AEB127188C354CD
            stream[448..511] = F040B4B84713DC71DECDF4413BBB42EE
                               345DF8190D176B8E7971FAFD07FF99BE
                               E87B52C3F3DBD9A72A32A79988E8265E
                               E13462E339544AD049A1872EDA01BA12
                  xor-digest = CC18BFB48DC60665115F89881A41A71D
                               D6590BEF99A3C0540E60A0D12EBE7D80
                               2BF163D1490FB3C94C9EAA13C07E0CB1
                               A6E4595B46787D6B29E7EEA5883BF374

Set 2, vector# 81:
                         key = 51515151515151515151515151515151
                               51515151515151515151515151515151
                          IV = 0000000000000000
               stream[0..63] = 5400274066761C7D3780A2CAAFF102BD
                               D49966227AB356514A091095769E094B
                               4CCE40E8A8803973714B59C5B67DFCE3
                               979AB4D9A1EABDB99563009E3E00A1B7
            stream[192..255] = EEF851A7869149D57E09829099C053FA
                               043DB0477D016E8AA2FC8F5B36A40B50
                               204922BC4AA5EC0C6601B9C5E1974935
                               60F9EA748D24A504AD509EFB2E4EDC13
            stream[256..319] = 26DE4A71C1C2C2BF6BF65CAFC0631F3C
                               B316CE5289F47BA0691FE120A5E41C71
                               9B670E34D27AF88822E03338D8B2561C
                               15AF68388C02391C594513EF47373C1C
            stream[448..511] = B5AA8433A24598EB913A63E5059DE564
                               5D06DF00BB0FD3BCCBC3B386925984F3
                               6AE59430772283B667949CB08BADE486
                               246547D949426C9F7A976544DE6F339A
                  xor-digest = 02E19E46000972398C48B45FC8CD9FF7
                               FAD3932595A2701785BCA4EC4B3D5594
                               6F8675785B9325E79EA61D06A39E79B3
                               23AED963D56715C35A25315A3FB1077A

Set 2, vector# 90:
                         key = 5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A
                               5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A
                          IV = 0000000000000000
               stream[0..63] = 5D1A4EF0C43016A7A577374132E72026
                               8F16D1F21A996D628F4EC8FFAC83CC89
                               DB32C80655FE849C0933C50D3A838386
                               7641F09C13FDD1D1176E563A44F2BF93
            stream[192..255] = AAADEF53BD349367DF7B3C4AA28A7D91
                               183BFF64E469608569FF4F74196DC820
                               50E828DB10F75737E187348899A379D1
                               A9A3700B081F2E97B016F631D3BD9D3C
            stream[256..319] = 5A82B87E84E0B509130470463ED669C0
                               156AD12B86A015DA49F19F36E3FBFDCE
                               F5EA39F47504FEA91C771A8B7767A0F5
                               7F62D7541ACFB22CF3FB9C684BC9412E
            stream[448..511] = 23B63749AABE5DEEBDCF4FC8F79E626E
                               AB3C69C094A2592325A0D8FBCC3DBA6F
                               607152C6D257442517559697BC663D93
                               71D4D12116A053E5722DFF9C1BF22CF4
                  xor-digest = C70E2BBF4BD8F1850FC96712772D4907
                               C534EAE119C1C057516A0C7CEFF84DDF
                               11C7D25D38E634080F924911E248F0F4
                               74BC5B0A3F1FC145F85C34F28C34E1A2

Set 2, vector# 99:
                         key = 63636363636363636363636363636363
                               63636363636363636363636363636363
                          IV = 0000000000000000
               stream[0..63] = 9288C9999C48482ACB6E465D8DCF37CA
                               04EFA423F94EC3D076582EE42C8CE15A
                               CB2008D1F8CB95828336B08F63460521
                               38F636E9B304F8D53A4893DF546059BA
            stream[192..255] = 03BD0C2381CE78EC3C9163F1B354AEB9
                               A7FD8D0E1C3AD6037C338FC5E6A25C8C
                               F25166FBC06912A3006F4B23ACED866A
                               94E0EFD30FB7207920954698CD8F1CDA
            stream[256..319] = 83A8F0110C8C210C6CB79A05FD17A25D
                               86BB96A7AC28ED772C3E5BEF61A9CF13
                               828D6357F34A3DF0281B60623516F26F
                               1F1AD3F4459552202A33AA512AC9BC79
            stream[448..511] = 76CE75A0AA88E936DAE288C472365448
                               5ACCA2B4EAF164BECBD3FEE0B1F99332
                               BC032238228E2FA02D0446D9DA2E1FAD
                               88EFAC423E14F80F83DF3F7D3797A1AC
                  xor-digest = 94B3F82549D6B2CABE12F715C6278821
                               2BA28BC2B239897254035133E55B3DE3
                               C7D4820513964140777EFABE5778BF59
                               113A782BAEC51B7808CA1E41AC060B32

Set 2, vector#108:
                         key = 6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C
                               6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C
                          IV = 0000000000000000
               stream[0..63] = 377A5562EC30B2C41F3C6974A70429F4
                               13C23803F942442D1C478D175FA2F65B
                               06C7A5F7010202B5EF530102E76B81CA
                               595E078F02CD3DDBBD1AE9F3AD2FCC74
            stream[192..255] = 5DA38B8C5EBF1A2F1774E62A59D1B707
                               53A11B5B176D007D5CF30A8D32A8BEA4
                               02F295EB6FA7E86E709E03B0C5960AA8
                               EA929444047238E8D53F92FF5C96F461
            stream[256..319] = 2D3E36441075B1BBF3B5B9BBF9493114
                               AA83D82DF9485B5D1DC3C5792C6D55AA
                               4D89ED0FC408C59DF194126AA1DD84BA
                               69699B253AA94C3F883B08875ED2DB65
            stream[448..511] = 3657669287141F08CE1F962AC3AB785A
                               7AD5D8FE1F1D3B4200013664593D6672
                               9A2B8405824714A0CEA3B356BAE7FBF0
                               828F3F5F532ECB631FC508E183145C4C
                  xor-digest = 6C28460CC179D623FA0A50746FA35D29
                               D6724F0A37BBEDBC8A172C6FC2802CE9
                               7A1A9C3A94C71152FE47F9D624E42504
                               B9A20E73DAD0343DF707B2DF4131454D

Set 2, vector#117:
                         key = 75757575757575757575757575757575
                               75757575757575757575757575757575
                          IV = 0000000000000000
               stream[0..63] = 12916AC2BDAE6CC9A6D88A632CCCEDF3
                               0DE0974DD19B1D5C2A38491E2290B27A
                               7A316B25C4CFE17196F5F8799DA773CF
                               6F0A3F0FB483182D1E3DE98AEE5E1A3E
            stream[192..255] = 75FF4EE150B2B9A3149F00318A2D6594
                               7C766E2337BD1CE734A59D6BA33C6600
                               3C9D8D68D0313635FB37D9A9EA6E8681
                               32A949BECDE19E9419E5C73D4DADE022
            stream[256..319] = 7CCF7A5AE3A3EE4AC6A63AA689856D09
                               54BFA12D835AFF25F3F710EE428C9410
                               E33730403A8C239B69A4107B976C8F9A
                               6833C8DAC74BAEC1715C62CAA4D4229D
            stream[448..511] = 6381F00FDAE9D95F575CDC528FBCD334
                               440E561D61B439FEEBC056391ADC13CA
                               C87E4879BE98844843D38100601D4251
                               C46A2828438418993F060CE403E24941
                  xor-digest = 586E01385FAE512BE1412005ACCBE586
                               2E87F1DB4B2A876078E8710A022CB44F
                               CCBFE009D8589EAFA869ED673984C148
                               55A99873CC5256A34E27B0B2A7F971AB

Set 2, vector#126:
                         key = 7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
                               7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
                          IV = 0000000000000000
               stream[0..63] = 8F751521CC45165D650F8476D1B4F94F
                               0D47452278B6B02C56D688F10213B5F8
                               FF30856088625B3B07C10A0B7601D0DC
                               7C8B5ADFD671B14CDF735214B8548B77
            stream[192..255] = 044FC97F0737864C87784BB38D7188F0
                               ED04B83F3206363DA7CE0B717FE23917
                               4FB0ECCEBA1CD188A76C6744EE5CDD06
                               502A61802F94CC6AE10166E14DFD1D53
            stream[256..319] = 2A7A186C9A3DC780355B6CBEBB61A82C
                               EE07E79169B5A1F5C11D9739FBD51F55
                               C11ACE99F77BB307FDDCBD6111EFBA17
                               65AAC500116A57EF06C838475C765D5A
            stream[448..511] = A0C4715850742D600AC43A1CB5F02167
                               BE0322464F4BE66EE5F1C5AD86F126BA
                               D5CDB4B61537A7D979FA774F8DDAE361
                               F11EFA63B76A54385EF0CC623E318E0D
                  xor-digest = 83071CF21B3F2A3473AF42A6C6F3841F
                               BA16EE97BA74A0BB0E3D5CED2D45FF90
                               3777597D8E0A28CDDF6A92CF812BDA60
                               8E1EBC189D792F5EFFBCC810CA89ED89

Set 2, vector#135:
                         key = 87878787878787878787878787878787
                               87878787878787878787878787878787
                          IV = 0000000000000000
               stream[0..63] = C4000E8A5231BF724A8DADAC00032676
                               F092A855766AFE62D3A15501098917D0
                               336B73D927854B5C4DF74C199E6F05F6
                               0527C3FD21C5294366362C0CABAF535D
            stream[192..255] = F24F025C83DF8DB30DE4E88E1929E43C
                               86C4B337130E791800321B85A8551130
                               744552BDCF17CD64CDF31C44BADB5855
                               89AF4A40C33DF783EDC8E2AD922C1E5E
            stream[256..319] = 509D9D1E15FCA283BF51962535AFCDEA
                               971512F5DF6870EA78AD708FA3CCA90C
                               56FD2FDB3C8C069D1D38863D64FF21C3
                               DC4894742001447982E26374604D3780
            stream[448..511] = B94B58B1B1571C78C24463E35D9B0ABE
                               05D3C017DEE329D7EF472728904380A8
                               E93F9FEC8929E3C09EAF341C03D17EDB
                               60F4DF6207E5491C04F58AB3EAACA579
                  xor-digest = 7243A35FEEA2CC5102710649ABEDD9CF
                               9D424558C8AFE243F3BAABA1B72FEDE8
                               79F7C58BF4903C4279126540553CB5B2
                               7155665084B2486B657E1E11C82211F1

Set 2, vector#144:
                         key = 90909090909090909090909090909090
                               90909090909090909090909090909090
                          IV = 0000000000000000
               stream[0..63] = B57A0A1F4A7931491F31760261E865F9
                               3A5DBB49FEA69627E2CF1E596F50B4B2
                               D566159DD9132B8B3C4FC382C10315C3
                               131A28711F1A131A604BDBBE73BBDDD8
            stream[192..255] = B16B458C1E5F24494EFCAC727BF82918
                               0DF881CACA9AAF93923E744985C3CE30
                               96CC008E0CF5CF0BB538CF76DEE0C411
                               EA3F1D9673DA4A464481DCD7A8E67388
            stream[256..319] = 12FFF7CF7FC1ABF5E53EDC0254111862
                               3D132FEF1DBC125E6BEEDFDC4E1E1424
                               2918880E634B58DAABED2D8D62626A74
                               8F6B8E1BCA7293E5FD080485267EA794
            stream[448..511] = 562D9280A4E9B3D7B69AF52785D0042E
                               AC42C3678FBAEE910ECD5957DC2EF8C2
                               51B0C9F1D33389F28173A549E0208715
                               4F11B4365A38127E878EE5ECBFBC7F2C
                  xor-digest = 2C2D2C4D9D209778A7570F3A448035F6
                               19DC11DB7127E2BC08E7C8A50936BB33
                               11A92783DF213584A504245F5F7C2CCB
                               7027FDFB102924A39A4AA73A7B14F1D3

Set 2, vector#153:
                         key = 99999999999999999999999999999999
                               99999999999999999999999999999999
                          IV = 0000000000000000
               stream[0..63] = 216EDF3DB5C8BB5FF80B10362F7F249F
                               27628192E6A3861F287B2A6D8C4D71E6
                               069E691FC49AF0E768E336B9F907435C
                               697CA739A8C3C985B82DFC17825DC979
            stream[192..255] = 0D550F517C3C37F71C42AF7F18C062B2
                               0FD374B266471581DDA673C1883668BA
                               F227CE4276EB1E1273C72BBD8D2A2B1C
                               49C6966A14105598D1248AAEBCC776A3
            stream[256..319] = 188A5307177F93100E3D4125D410CA62
                               7E35CA8238BAC94E4EEE246CBE7C32B0
                               AEA05D2340DFF7BC01BB527E44568E81
                               4407D1FCAD63648487B3E306FD066895
            stream[448..511] = 5AD72DC4F62F8A9655FEC7C2B6E09123
                               AA8F26F45DC4F1883D1443C92F089E7E
                               05517D82469961770759BA95B4010338
                               3F917EEFD979D1B561A6D6CFC8A55210
                  xor-digest = 477449B7532089DEA8A1D5901374B05F
                               A2A36835E747BE5FDB110302BC1ECAD1
                               69F1A2DFFCD90E2626D5F4797225EA2C
                               CC408F75AE6E0D6B3CCF5B1ECA6C206D

Set 2, vector#162:
                         key = A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2
                               A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2
                          IV = 0000000000000000
               stream[0..63] = 9898E66BFD4CAEE704CDD338B49EBDCE
                               6EC8A7C0067718255A72701498641096
                               75404A8D4E11F5E56CD01F2586DFEF25
                               0577B3F0317B823309CEAF96CC43E633
            stream[192..255] = 64C06887F8C12828266DCE2ED67E713B
                               799B5F3D6FFC0D2152023D96FB6FE83E
                               E810FE5A0417B7BD2EBD4AB48AE32E1D
                               1955FFC869615A3878726BF06DB50F55
            stream[256..319] = 3FDC5C7865257708CDF067F6B6664D98
                               A288B834CE51237B7D46DB04CBFB1D79
                               C9719ABE0327AEA6B2480DE711191125
                               E89F23A97776025B6430F681D1888CF5
            stream[448..511] = 2CDEC4F2932777F35CF260D981033C74
                               60DCB770B5E53D952E4AB7DCBD394603
                               6A1AF3309CA1DFE9D5589138273ED42E
                               CAA2B1F0AFA99D9E08E0B3873CE92BE0
                  xor-digest = 544F9FE9711B6C792AEF0E7C79CB84D9
                               24E6D34515E70C6AF57E6D35C3E67B2C
                               F1AEB0C665D29222E7A2D0FE4F355C80
                               893CAC3A026E7A42FA867ECABE2D1E86

Set 2, vector#171:
                         key = ABABABABABABABABABABABABABABABAB
                               ABABABABABABABABABABABABABABABAB
                          IV = 0000000000000000
               stream[0..63] = 45FCBCEB09F6EE9A70E4FBFA49409B41
                               1D979DE84CC551E9D17971AD78DED211
                               4B2C1EB7038BA4F45214B68DC79D7EC6
                               B9A2A18D24A14DB7C4B0823931C37B1B
            stream[192..255] = B1689FB8B13F7EACF79847D75C741AFA
                               0215F919C3190A2737AB2577779B74CF
                               5ADD9B70FF6DED4BB6E537978112F72C
                               F6F6818E88C91A3EF8C0B21F90759075
            stream[256..319] = 0A3FFBB5FD1EB9BD0101DE9C4BB3CC88
                               4FF3D410592922901FC31132661EB33B
                               359E266418EA88A74016942DB7C584B2
                               DDA18312C583182E6CB239D415090FF2
            stream[448..511] = EE48E5057246A92F66494DF967A3635B
                               0D5E3A113D25CCBE8688D5D9768155E6
                               EE4700694B5C756F175DE60A4F11F879
                               EE6939A97716D6C9BC77ABF711EA3747
                  xor-digest = FE6CC121CF62DE4811E2869A12A18B78
                               710CF53BE4837D6B824A9E4B971DB487
                               B3160C372069043A2F995EEFBCC17B9A
                               DB57D26FBBA388516FB723301839BD89

Set 2, vector#180:
                         key = B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4
                               B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4
                          IV = 0000000000000000
               stream[0..63] = 34A45DCF85C3F2BD4687B0EBEC2DF46E
                               AF0139F34D6A6D34C701F13C6672EC09
                               1286D36FF46853CF2F702D94A1FA520B
                               CB067DAE0CC559EAA02284CF57117862
            stream[192..255] = F4DBD0BD871A3D7EB607FCBDE41F83B9
                               179FB16C82303E25E498CCFD16CDB309
                               DC87BEF14DA71302DC6FF98E095DEEAB
                               BFBAD7B7EA584A723B05765227A670FE
            stream[256..319] = E51C19F34E220915FCA9BB0A01BCBF3E
                               29E78AB68B0C6078004A310D8471972A
                               AA5B852057E81D1D1E1DA66C51B9A676
                               C08BE2B93C88F02D1378139FD5721E12
            stream[448..511] = 3BF0DEE1CF4ECD7E7FD0EBDA2695A05A
                               F6D70C7414BFD80A70666F68AEC24E1A
                               671D30BA1F49CE124A14F467EB7D2D25
                               FE62976E7203E38E9D5D4ACD7AF5BF1A
                  xor-digest = 090B394EF92FD645DA7684666973CE24
                               64E3E4CAA16BF209C9506EDD82C07ED1
                               E5CEB0AC34188596FB98F70BCE08D676
                               EE474CA0DCBB22406C1824F8E7362F38

Set 2, vector#189:
                         key = BDBDBDBDBDBDBDBDBDBDBDBDBDBDBDBD
                               BDBDBDBDBDBDBDBDBDBDBDBDBDBDBDBD
                          IV = 0000000000000000
               stream[0..63] = 51F7A8666C52459AF653BCF329EFE90B
                               DFCB1034C0346C7C36395A80A362D97C
                               631DE15C3C0FEF7FBC914414356934E0
                               82E6A298EFA73C8CAF2359D548F31BB5
            stream[192..255] = D94ABA1D1FC7B27FA77CA274EDA19CE2
                               DE72F4F592201B47D7E50F372C0E91FC
                               8DE3F21AD1BFFD35D88A60DEAB25221A
                               04DB7E32FD335598675D5ADE6AD782CF
            stream[256..319] = E0324B0A8EBCCC5A9CB98411D2F7C22C
                               706F7307486ECCCF8E829680385A5400
                               F162CE0F110E22A512E9FD5F00134102
                               071B89D665071A7310D4C543D5C3A433
            stream[448..511] = B1E03A75FA5E9BD0356E2FA725791419
                               6A42DFBBC799AD1A1EDEE3005121E567
                               9770808402A3412D6D111DDF1C64AA7F
                               8B48E4F0560D083AC85A30E2006B1DA0
                  xor-digest = 7C1F1D90844CE18FDBF0EE1A20A3CBD4
                               79E4FFB1573FAE389127C07D891EBCDE
                               068E90C7BE2F4F3521A5AE45FC3DBBB4
                               4F5D1A4FF3980430E72414CA2B34BCF5

Set 2, vector#198:
                         key = C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6
                               C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6
                          IV = 0000000000000000
               stream[0..63] = F6DC2B1D33E63C84C1BE154A2FBE379C
                               E57F03A9533159873B13A957E53D8D62
                               E4CE4107C393D45BBDE62625606B545B
                               C8F5E01B95078D1EACFE42FA8B37BF26
            stream[192..255] = 93A4563829F20C9CF8DE184B7D3C8BE9
                               9ACF7DCB6E8DCAB8835CB31494820268
                               CDA2731D696F92645D38DFC4FC2A7E96
                               A1C420DEFF69AB3675DA7F73770484D6
            stream[256..319] = D585F862D60E0FF7A441ACFC2E30206B
                               1521120930DFAA995BBAFF7C3574321B
                               E70FB567F7DA9ACC9373F0C49E1D8C02
                               AFADB738D1D948B1880CD9F6508CEFCD
            stream[448..511] = CB41E685D84F049AD4FB848E337084A7
                               6320C93538D4CEF3B6A8FBAD5A495494
                               D57AD41DD93D52E429D1A19658AC6DC2
                               DD68CA8A9D6FDF2E1717939FAC9DFCE6
                  xor-digest = 15539669BC4F561EB5C75387FFE852A3
                               C1A60CDC3A8441654E6493814AEFDDBD
                               0D734B621F722058F05EA046001FC621
                               D9F636BDC7EAF785A8515B0302B1121A

Set 2, vector#207:
                         key = CFCFCFCFCFCFCFCFCFCFCFCFCFCFCFCF
                               CFCFCFCFCFCFCFCFCFCFCFCFCFCFCFCF
                          IV = 0000000000000000
               stream[0..63] = 374D70803C0FB5C994C8B882102BF2E6
                               00E569B32B590DFBFC393F9BF070644A
                               406C1C07121E0D3DCA8987DF3A5A82D1
                               6744FB514560A81D5E6578B34FC59A5F
            stream[192..255] = 66CAFEE22C82D9FA3116C1CEDD116835
                               A5345DB4D4E08DE09A27BCAB02158064
                               4F23DFFEDFD006219183C6F19A26D21C
                               BD4B0E5BB974E829B89969F1EA6C03AA
            stream[256..319] = 2C0FDB57FC1320362AAE984F1358F233
                               8D01BF7C7F3A6598DFEB7EC2F29FB8C9
                               3961D5A82A5B92ACAF2667A97AFA4C8F
                               3489D641A9BAF3303F8B8AA863F8C799
            stream[448..511] = 30D95005B32D0CE1868811E29544E396
                               D9C1431A73C2ED60103D0B1EA3F7D0AF
                               3F4058ED8E62AB3C3CB7DE98BF667168
                               8279673A8CC9DDAA88BDD3718ED973B3
                  xor-digest = A5B05D4596DCF905617CD17590856F2D
                               5FCC02237C4C6315C38DCBF72CACB353
                               7A0EC4E5C5A92838C1F6491B68EB148C
                               A22DEEEBBEFD4F29DA895FB1B5C2FE66

Set 2, vector#216:
                         key = D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8
                               D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8
                          IV = 0000000000000000
               stream[0..63] = BAF637DA83F1B4257EB31DE10C5DAC0E
                               7A233C0CD51191E90F952C9ED9274582
                               3EA858F8BE3074BD895B1E2D7B0B3959
                               7E10F503D7C9D6C0466561764E3945A1
            stream[192..255] = 0A0D75E35523978C426EE16E0968C28E
                               4850E47CB13D74606A93E35CEFCE2672
                               37321C80CA35F7BE15E1E96929683E4B
                               E46629B5682E236FC675B0212877955B
            stream[256..319] = 4BB0F289AEEF00513FBE876FCD4C4238
                               19F56E153479FAC2806ED3A5145FFE50
                               854F8DECD6BD6CBC1CF659809E4C7099
                               2B3ED7AFC337731FA51016D9B95CB975
            stream[448..511] = E9E4B6CFCC740654BF8116B1807370F4
                               BE1DA1C67AC1919E1248F948FE187BC0
                               20D52134CE17F83174D2D562FAF531D5
                               A2AF78F28F17E8DAC613538064DD95AE
                  xor-digest = 140B5CD872127DF8F1905EDE1AE1E92B
                               83883FC7C437F419803A9480B73193B8
                               7B8B784CE3017A9A528F38FEC92B4AB4
                               E4B168BE69BCB40049A5AC2802021F74

Set 2, vector#225:
                         key = E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1
                               E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1
                          IV = 0000000000000000
               stream[0..63] = 40FDD02D51823166306BCDF8B77607F1
                               D4EB2B3876E1FC73417742F24BE45BEA
                               80BE07DEDC173023A9BD6BCEA49D6F05
                               28230A637233A06029A4A7A63C1B7817
            stream[192..255] = 0F1D7BC5EE19D5E4577C9A6A0E6DE608
                               9DF193A9F916BA818C867450DA4094EA
                               9379256FD8BC9FCA31E04CA5CE3479D1
                               EF430C4ABE94042063964C6EB3B0AB04
            stream[256..319] = 6993D14C4511351CEF3292EC39BD33F0
                               D112C1C72E2B040696CA11EFE4513523
                               B6741542A1E536F5AD94905A336EC236
                               17E79FB72EC8F9FB2E973F3F847D4CFD
            stream[448..511] = D86B9B0674DB00B67EE9E762F0E61165
                               34E9CD6F5115946A081EA6C5875F2182
                               6B31F109887499144D5179995C8F7533
                               A8653A87E7CE832AD7D5B5984F8FD781
                  xor-digest = 69472F97EB5509E8F10C71DC976A2A13
                               C75CDB24EBACC412D18D7A865244FEEA
                               EB6691B9E925FE87E7B30748150E03FA
                               333C0CDF442932070B03229D04D902F8

Set 2, vector#234:
                         key = EAEAEAEAEAEAEAEAEAEAEAEAEAEAEAEA
                               EAEAEAEAEAEAEAEAEAEAEAEAEAEAEAEA
                          IV = 0000000000000000
               stream[0..63] = 0B95F3973089E4FA452F0EFAE825CE1C
                               92F37F7633E73F324DDEBB156D551E0B
                               E70B2B1028FC5DB8CC25CC08EB6B7A00
                               768D2D5477933D3DCEBC79F54D2C5E8C
            stream[192..255] = 591BF91FC97F535EC390A8FBBAC2175B
                               5D4D8F305C6D76EBBA5A6F572D797D1D
                               BAF5E6FEF47C118762B818ADB7BB3D4A
                               F97BF42359477303F1171C068E2A78C0
            stream[256..319] = 94B812EBE731EB385B202D15BCFA10A7
                               D576CF9CD522543C4B0F3187730AC758
                               729059CDE6E4B4AD913D91B03934F21B
                               A6633E9EA3EEFB164EFDD9F17DF0F7DD
            stream[448..511] = 833376932091A4DBDEBCEAB7B93795A5
                               41D10C9D699E5A4FA54B33D1E018742C
                               33094EDBE2A76B3BB429136DDC20791E
                               0354579EA3095C6D7A0DE29B7639DA63
                  xor-digest = 14FEB3FCF65145B6ED7F8D17D48E4DC2
                               8E18A44BC6F45C7972ED1C6D2C061788
                               A77CB36D9BBD6F8D4350035BE35BDD22
                               F6F41AC561460FB54DA14AEC4910321A

Set 2, vector#243:
                         key = F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3
                               F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3
                          IV = 0000000000000000
               stream[0..63] = D5AAAEB6B360B05E60728CB7715F064D
                               688272DE4D247EC825EB5B86B9EC637F
                               872E9C609D7F072EE4D6B2A0706073CF
                               7BC8EBD669798A9A0A4C57F3BCD4AD96
            stream[192..255] = BAE49771D050446C339A54859DF3C0E3
                               1E979A221895658F4CF25264EC15EB0D
                               3636C835ACE612CB1356612139179AD2
                               D0A2F509E7F024A7B3E36767E4BF5FCB
            stream[256..319] = 5FF6401D11EA0DF1C741E8A49F2474F7
                               ED21B1D06C11AF82A10E8050A2DE0C7D
                               092B07D58F5C87A64F10952B0DE31E08
                               CA7DBACB5F7F566E8996923F5638F25E
            stream[448..511] = 86C59730ACF2CDD8EDCD7282C5BAAB3E
                               DAE59D9E4676979FBEE5833EA788B269
                               F60389D08B5514CE579F947F16021B45
                               014D7E4A533E00D085AB4D4113B15C18
                  xor-digest = 214C32075CD0535E7F57E38E7EED8B75
                               56F9BF0729601DCBFC36DBB1D3C90C39
                               4DCA2BC3D0EE166903419C2746ED5DE0
                               6E980DA3316CC0322F36FE64B8C82522

Set 2, vector#252:
                         key = FCFCFCFCFCFCFCFCFCFCFCFCFCFCFCFC
                               FCFCFCFCFCFCFCFCFCFCFCFCFCFCFCFC
                          IV = 0000000000000000
               stream[0..63] = 69DE3E4C2FF081EB7F60C6E3D6A80212
                               5CA029E2BF4017AFB33A96EE4674F2AD
                               0587532926ED63D219416A985040B7D8
                               2F843893DBF7D0EB3EE1C3850359D259
            stream[192..255] = 8E7CD9BE41820B9A8BFE5ACA633E9437
                               C14753BC9EA615C8F7C2CD01115CF4C5
                               84D8E77321FE7691F062D0FA6CA629EE
                               C023EAB96AEBE93B1E5659BCBFDA3744
            stream[256..319] = 0C3E756A47DFFA8F79EA84E76E7D3156
                               B7E3B426E6EC45C3889B6B80BBDB5F81
                               1799693CFC5275F9592BC02723798A76
                               00941D12C49B7D5958FCE09E1008064E
            stream[448..511] = 63F215BAC5A814F2365FAF2C64AAB050
                               36AEC3A73CB80D1CA167500373F3663A
                               C400FA946F12D7C89ECC0D1D996D251E
                               439326BB36807E215B89414A2AF99E2C
                  xor-digest = 8886830183F33BCE411CAC92338B725A
                               F64BD7953ECDB6C2E347B1ADEC3CE20A
                               C0FEFB9D816A4CC99DE15F94391AD370
                               E36B0AB409D92B704764536EC65CCA30

Test vectors -- set 3
=====================

Set 3, vector#  0:
                         key = 000102030405060708090A0B0C0D0E0F
                               101112131415161718191A1B1C1D1E1F
                          IV = 0000000000000000
               stream[0..63] = BA3FB682EB86D3672981D4D7C39BA93A
                               66B78D1D867A8EF6BD9523C8CA39CAE4
                               7A0CDF6118AFBAF8CD60D5F162696721
                               6AA433DF22A7BE576C80DF29197CCD0F
            stream[192..255] = FAF50E0FF8D5F26443351B25E0D04501
                               B0EC7C23E538A0557D981D4AD91A8ECD
                               65484D1948DC7153FB39DFAC5389B290
                               A916C1D42119E84D5AF96F4605D4E318
            stream[256..319] = 96D23B04984AADA3D4B12AAF5B04EADE
                               440D1831B36EE8E824CED1C4E2FFD41A
                               FBCF4A7996F5BCD52124E3D5B3F9847F
                               71A3933E4F10C60EE01CCB5A727CDD84
            stream[448..511] = 375E3480C20FC8319B173D9642758759
                               10C513D36229E15C42C92EFE604AB6AE
                               4C8464123E6B9B6B3070897D029C287D
                               FBFDEC2B8BFE1E78F30CB97F5AED613A
                  xor-digest = 3B0723526F1C51B0C943375C030087BE
                               D6FD8162BF3EE3589EEDB84CAB870032
                               3CABEAC277B1A19B12E3F8B5C641D8E2
                               D1D3CA68AE6F356BF2BE577BAB4A40BC

Set 3, vector#  9:
                         key = 090A0B0C0D0E0F101112131415161718
                               191A1B1C1D1E1F202122232425262728
                          IV = 0000000000000000
               stream[0..63] = 5AA7FC2554941FBAA27C70A5F83359A0
                               250D31284A12582924D86139BF3296C3
                               9D22E5E9C6275D4B10A38F2CC80FE0F9
                               01807090D71E339485B6439A80354A5A
            stream[192..255] = 75E1DF44FAEFE21824D9BA7BC4B10490
                               09F90A2AFF10829DA8D21B7FA9B02E53
                               A8F1844E7994916DFC034D18E146770F
                               A6BB7A36D8B86316C24EE40111B0CC39
            stream[256..319] = 4946FAE6887BE478DFDAF6A697C535EF
                               FABCE3DC64FCC6D10CA5D376B97641AB
                               E30D9907512FE05BA0108290DF12CC77
                               3508B672147A05EE76F1A4CEE5E0DDDF
            stream[448..511] = 433D262DA3259850DF2D17229533D2B6
                               6F2E7CD72DCDC1AB8CA9875E411A91BD
                               8D14AD0C44F55A0437F25803B8101D96
                               F0C1CFBD9365CF28F48FA3BEF9527FD8
                  xor-digest = 1B2BEAC501C91E98D9EB00F3B86DCEE7
                               1E94EDBA2ECD08F4B389E492C34588DE
                               DF0181949073ED76FF0C6C278A56E24F
                               5D7353DB202DB031CDCB7BE9B59538BE

Set 3, vector# 18:
                         key = 12131415161718191A1B1C1D1E1F2021
                               22232425262728292A2B2C2D2E2F3031
                          IV = 0000000000000000
               stream[0..63] = 23A022FDDFCD7BE7C9322AD0CE317505
                               64C05D78BE9379540948C9077F1D505B
                               930400D7605B4758546F8EA8F9DA961D
                               B84B6372EFA2DC34B1EED24880F50459
            stream[192..255] = 72BC67A34C4B6BDC13A57FCD0599AD90
                               462CA2E205279376FF5AB4B27D55B860
                               46BE2144347C307BFDEDC06FCE28DC02
                               66D90BFC7C5CCE133CBD01503D981670
            stream[256..319] = 9C7DBC85FB4BBBE086CBA25DC1663115
                               53B3F81BEC202E8D39CE1C0E17E6A92C
                               0F27F7CD32483AE934F2CA19B1EA6F9D
                               E617D6FA8665F54DB30C91363274EC38
            stream[448..511] = 2FA63569E0147FA5F738CC8B2E526CDC
                               0A2A1D3AAC0C737E1F52E4F12F469049
                               2D106E142C75FF23BB4180220F7B239D
                               AE1E6E66C1CF1FEF634038FBAF080CEB
                  xor-digest = 3578A4C3D5D95EDA7184D3A45611E931
                               78EA4FFC9D9B22E2BFBA694B92A80709
                               27695EB5A0F420530AA4192DF99DBC39
                               D5491646222EF528747F1624DDA80A38

Set 3, vector# 27:
                         key = 1B1C1D1E1F202122232425262728292A
                               2B2C2D2E2F303132333435363738393A
                          IV = 0000000000000000
               stream[0..63] = D3F71E40C924EB10356C72909F560DF3
                               58E1871B786865FD648C0F7698F24DD3
                               40603E96DB692EB383A99E8361A16C1E
                               12B550A653E48D5E61BE57413218C527
            stream[192..255] = 3F98CC06A7EF083AF032386327CD2253
                               A66FB323D5BE68E90343F33804BF7FF5
                               40F2FB5A786A37367ED9715365D2B4DE
                               EFA87CDD5E9554FB78C2BACA053D99B6
            stream[256..319] = 1534A1FF522DD2B3EF21C46E5943CD75
                               09AFA99B3B0972EC1EFE3BBF938A003A
                               61D2A82ECAB87F6EFF8B9253FF16EC6B
                               1BBDCCB286420694361CBA573635D136
            stream[448..511] = EA5FD6B0357F67367F30773EDB2E8839
                               0FE19BFA13FD03794936BBF1CD5DDB8B
                               22E85019FBAD90E35ED21F27B04B2422
                               B9A4F4C78946A1EF3E8FEA0B18FA3790
                  xor-digest = 37164AF19204384ED577F9561584948B
                               9928274C3432C76D08D13EAA90DA09BE
                               C6785BA86F611187C6C099DBF38328DE
                               7A2FC168191150CF43B5F6D59CBA1ADC

Set 3, vector# 36:
                         key = 2425262728292A2B2C2D2E2F30313233
                               3435363738393A3B3C3D3E3F40414243
                          IV = 0000000000000000
               stream[0..63] = 5E6CB90E447C519B72F4BD206E788644
                               4328E6D0A8194660C0597F09BF55913E
                               44D616B62ED7D58DACB73E7372C20435
                               DC4FFD07698C96661CB0AC779A09A755
            stream[192..255] = B5FC78E8AFECCBD64AE32E09E626D58A
                               0724C1B65FB0DA4B3B27A51EBABB381B
                               AD3CAC814CF0C15AEA0BB7E9B02DF672
                               3FBFEF9D56D39DCF3F212F7E74EE9C56
            stream[256..319] = D81FD8C385D13C64005346AA9426943B
                               5CEEBA07DC644CEAF98A2135986EA2CD
                               F6C95E7CACDBB819A8DB3EC52DCDF68F
                               6958982B3CCFAA94A9ACE5631D3443F5
            stream[448..511] = F1A605A14802E9087AD3AB5B3BC711D0
                               FB1A3AAB4BD7B1B55EB41B956A9EA8A6
                               7F080F7B11993D6A2136854B99646C11
                               19D458FD72A3CDC4E6C92D34E2AE04D5
                  xor-digest = 4364C17C638B54542D8AA1FF920D6245
                               131E7EF82AE604405CC02B0775E40B79
                               531669963B9A1387C5CE31396DD860C3
                               9D511C062BE3C14EA17A8E185C42FA98

Set 3, vector# 45:
                         key = 2D2E2F303132333435363738393A3B3C
                               3D3E3F404142434445464748494A4B4C
                          IV = 0000000000000000
               stream[0..63] = 754297ED844669861E90BD6EE111AC4F
                               0C7FF32CAA43C9FA5E6B0D99B09161FD
                               90C413E0EDC23EA034BA7AF6588B4E5A
                               CE15E4A68691380508DED0765C853C46
            stream[192..255] = 14C904192B5D9C4D435E090F56BFEE26
                               E8E7F7F018A6634D599C22C41B6F7CF9
                               D8465E1AC223567A1FA86B0675B7B3C1
                               40A385AAF76F04481E80EB0689D1B57B
            stream[256..319] = 64969C0E4E92FEA2E6ABCD5ADAE9A440
                               11957C6524766D30F2E7E8ADB3546C86
                               B6D2CA6E40FDDFF18A4D3B9E68CF9BB2
                               640C18449009628FD1A8F8AAB53E1219
            stream[448..511] = 741D18626C35D5B53E21CD224CB755A9
                               1699F57BA64268C3B37297BB0C676E5D
                               3FC29AEBA6FA5B5C0437E47A819FA7A0
                               5662EE8F0190CEFB8D68112B7E4358FC
                  xor-digest = D444F5ADBC6FC03088397080E6EA457A
                               4804B555C0F62F2E190C7884BFCD5F62
                               2BB8B53CD7AFA2FB4544DC6A9EEA9036
                               B094606C54F8360C54774EF7124FCF1D

Set 3, vector# 54:
                         key = 363738393A3B3C3D3E3F404142434445
                               464748494A4B4C4D4E4F505152535455
                          IV = 0000000000000000
               stream[0..63] = EFE25573E0FF9AFA4B658740D4D8795B
                               98A7EAAE5D9A43699DD3D6CB7702681F
                               613CD37CFAB324869CD3CC4F9A2BC0A7
                               2F04F650E5819D7D28BDFE09929D0AE3
            stream[192..255] = FBCBF6545D64863C8EED507944CFC8C6
                               7398EA1A4CB2D909C95BFA9357083BD4
                               1A2F425E7DF544F4B2CDF26D669201D6
                               CDA546E5661EC8F65C27CD3DA9FEFFFE
            stream[256..319] = 145AB666C6840F3C79C4876D1040EDA4
                               6D51D532728F31E2F20BCDAFD869AC00
                               B3A134C49DD40B846F49C35494D5F288
                               509393F522881D32B524A51C1C6E1FD5
            stream[448..511] = 3567346A27E8F0F35481801B3D9A934D
                               4E67688C496CFFEE816CE772029979A4
                               CABE0CE905E23584717F43E4F04F3133
                               89EA50C2660ED2ED57C6823407204396
                  xor-digest = B57486F1AF7CF540AE0681D7DAF16E73
                               445EDCF224CE4DD6AA8CD201B1829EE7
                               413512ABB97AC537C6C16B5C411C383E
                               72AD8A3F3450F74F60E16580C225CB29

Set 3, vector# 63:
                         key = 3F404142434445464748494A4B4C4D4E
                               4F505152535455565758595A5B5C5D5E
                          IV = 0000000000000000
               stream[0..63] = 6F4EB5FF1D58A8AE8ADFE295D6281296
                               36C186F144C393A63DB62E04B8321960
                               200238C6D0A75EAFFEC4A8D8FFD73CB1
                               CD204793152B4C095E5056C20961B805
            stream[192..255] = BC57E64A9DBEF123AA4B766226D68A3F
                               FE1D16DBB6400649D9FF54043916E5A4
                               8EA6D011173A1C8B353E73D7117E1A54
                               FECA401E5B9C37364D4C46B93E3361A7
            stream[256..319] = 6192F9AC8AC31BDA04D5FFB22B94050D
                               EA2C41F33F47E34BD132EBE0DEBCAE12
                               52D711F09283EEF6410328FBC9675817
                               BB909993BB19440ED2B19B06F85D9CC2
            stream[448..511] = 293C1E8254286EACCD7300365E6E7DBD
                               6EA70A68546E13FEE54ADF54254AAC5A
                               352F2A8FF9BC609F497335C19D97CA8D
                               6F84C6FE96EA3C6AA82619A1F11B4E92
                  xor-digest = 2A5A737D2EEF7B4A3050BD4D96FA8D23
                               FC94BB5526724EFC6306CF0102D45EC6
                               13D5B0DE4C5E10499DB6010C35D10790
                               D1B7F26C1CA8872818312A17035B005D

Set 3, vector# 72:
                         key = 48494A4B4C4D4E4F5051525354555657
                               58595A5B5C5D5E5F6061626364656667
                          IV = 0000000000000000
               stream[0..63] = A6CC8B404F2AF4FDAF7D6A8795F7CC5D
                               9BB2741B787B51538BF9BDA816486BC0
                               D334348DB81317951E1DF416988A0942
                               7ECC9214B02B8C4AA816A6014758FE06
            stream[192..255] = D4FA0D1AB59C779C4BA1B64391356884
                               06DA8137C88A3642295C6285E9535438
                               D86DAC9C5B88628CD04D99CCEA1EC82E
                               F58720BF5AEBC02A453A5601A9CFEE46
            stream[256..319] = 1768DEDD9085B9EA5D36B507855B7413
                               63500AEEDFEAE88D6996E45DC8A21BDF
                               2553BFD191E2CF697FC4D8D6AC7A3AE8
                               AF1B140EBABB5F1B8D1524E087147291
            stream[448..511] = B53ACC37176F2DD5E48250F1334E40B0
                               282FBF02B12953AE229A9175CF973DED
                               0B8A637E2E55EE26E7E2E8E962AA4BC0
                               98B0B1A82348DEEC1BDE9D370859937F
                  xor-digest = 9EA2C38E03E01567DD7B969946638AFD
                               A0EEAB5E0DAE16E6FD33196B923D57FD
                               22964B080FCFC07E64EC5464EFA9EB21
                               4ADCFE4792CEAAEABC73DB10E4178FD1

Set 3, vector# 81:
                         key = 5152535455565758595A5B5C5D5E5F60
                               6162636465666768696A6B6C6D6E6F70
                          IV = 0000000000000000
               stream[0..63] = EC6D0E3259CD756BAD85ED4EADF9851C
                               2E1ABFB3FA5D3B71CD094F90E325D960
                               49ED9E7F20B583A489FC1084B2B6699E
                               68FCA80FCAEE2A7A4946FFED7B316FDD
            stream[192..255] = 0C645D3281EE8E879E0BAD2351FA2623
                               9688A13B9711B425F7B03CE3E2806B49
                               25C0A528969C8FDADFCE1CB05C73CCB9
                               56504222DA9D5BB1395113B8BFE1B9E9
            stream[256..319] = F55E779CADB76CCDC754AC5685F3EC14
                               DEC0B31E72ABA48F72052BBE7B3A77FB
                               E7D915C545E1DDCC6E59DF9C080CADB4
                               59A3BEF25B3E7698D675D5D0E68C52B1
            stream[448..511] = 43F152BDBDB22D16E35F4E668CB0F76E
                               846DDEBDE9A38C407CF2CD4DC2A218A2
                               FFD0C2DFD191B9CAF757B9B4E2EF51D1
                               8F989ED878EF38385C67A9181EDD92CD
                  xor-digest = F70553BE38C7578DF6C8AF8F093BCF86
                               E1059769931E6C1B63B9C0E840E24D17
                               6DCE6578AF4FB9B23D034D0754218F2C
                               32FE312FCA7E08C00961FC6D56B02DA7

Set 3, vector# 90:
                         key = 5A5B5C5D5E5F60616263646566676869
                               6A6B6C6D6E6F70717273747576777879
                          IV = 0000000000000000
               stream[0..63] = 39FFB32108B18B6EE3A973F74EC2DB90
                               761B3B7F3EA76D9222C604A3EB0CE840
                               6BA03A2D72FEB41F37D6D4A996EF3918
                               D4BB2ABD32464DA343FF7E9EF91F078A
            stream[192..255] = CBD3382826E40B48ED3F1584D06E5331
                               4F30311D1CF63583CEB32E9969000D9C
                               28C70D83D5B6DA6042A090D2B23E9321
                               553F297EE3CC00661883B0F97EBEB9C1
            stream[256..319] = EB43CF6FFF18536DB8607CF15EE3BE04
                               6B01725EBBFCE04611B3E801DFA3E054
                               89335C4FB4043EAD3DF44D17DDE10064
                               E0FCBB727C3B223918A847C190AB35B6
            stream[448..511] = D19D8D8D87164F3057118BB61A215A36
                               15FE0E1346DDF57E068C7DD3990C86B9
                               9AD2FA0A1882C702AE368A7EA451408E
                               5E697ADD8E5163025C6F7E204EEBA47A
                  xor-digest = 0DEE89775F959C0BCD8DB47353239C9F
                               584AA8A94C2D8F70B0313F1BA8343BBE
                               4F676C1A0A01360EC40D1642F24D8ECC
                               56DEBEA05AABD12B7BE0B3DC81A0A19D

Set 3, vector# 99:
                         key = 636465666768696A6B6C6D6E6F707172
                               737475767778797A7B7C7D7E7F808182
                          IV = 0000000000000000
               stream[0..63] = F6D9AA962C57060D96630B813DE68E9A
                               054F9B2699D5AE322572E8FEEE823BD9
                               791B51D31AA87AD890983CC8AFE51D46
                               885E9BB0E2BB94996878DD9BA0806B0A
            stream[192..255] = 3C798B433F6B01D2984FEF29FE1C590F
                               B9BD8CEF54B4DFA2EE4534A121346489
                               03CC0289A55F6C67932B6D13A92A1C69
                               76FE803087178377C79951DADBC005E7
            stream[256..319] = DF973FB508E631B3D8F89DD3D3AE41EA
                               54F99CE2EB38591801BD0B2E6FE24BDA
                               A819A8E3D35FCA7396D4F6701E9A047B
                               E9C78984C83C3EB1AF36301FA7DE36C0
            stream[448..511] = B77933A3F5E9242D3C74B7A66E1CC47B
                               E7B7C9622BC2EC9151EEB578EF3046DF
                               9A09E5945C16C0670C3A27687EC192B0
                               1C41CBCD723A7090AE02B70978A8AC7D
                  xor-digest = 0FB64FBD95C135DC44C79B3EC27C7F04
                               B7CC5F42A9E2A05ED9E355CFDBF010B2
                               0F51581C026264301147D5B9229BAD6D
                               6B2D52E3ABC081E172CA302F71435641

Set 3, vector#108:
                         key = 6C6D6E6F707172737475767778797A7B
                               7C7D7E7F808182838485868788898A8B
                          IV = 0000000000000000
               stream[0..63] = BFC99C4BE36C5F0E8768CA215FB89AF7
                               35CB307D0944E7C5DA2723E6FEAC88A3
                               CBD3DD611B10D4A106269F10C277D1A7
                               BB9E2165CAE4807A3E9CF6F5EEB52E5F
            stream[192..255] = 6348C228D90563328BBA1470EA46F1D2
                               84513C15271DFF9647121D7454A1DE04
                               DF79E4A13044504538C31602E73D434B
                               E6A78B1B3072D6CCF79AE0B59048CD84
            stream[256..319] = B954F2BC8E354EDFD9848AF57B14489E
                               84B7A295F16E09F13D6A589C3631E7D7
                               4B1D05020C92EC9DB97BB95C64195B12
                               168DE2AC4418EEA1843B1523A0717DD5
            stream[448..511] = 15913C068D8C34D68BD388C19B86F4E7
                               204FAA3D276AF6A7CFC0D7B9CF5D18CF
                               0AFE0518B7B59491B2A827BE34760D76
                               765CB932E9AAC50164A615F99132A429
                  xor-digest = B0EF9F244B31864DC9548F47C4645B50
                               0B9F0D5ED29CCCE0775E7C2AE176FF51
                               98CC837F47422BCD403E501A9B810A31
                               093D708B49F9CAD7762E014B2B7BD47D

Set 3, vector#117:
                         key = 75767778797A7B7C7D7E7F8081828384
                               85868788898A8B8C8D8E8F9091929394
                          IV = 0000000000000000
               stream[0..63] = 7753B7864359B6457C83680F1ABCE6BC
                               8C84701CC7D301598E4EC3E4BD001366
                               4D4DC9AF32BB00F03B59AFD9D8AC0204
                               7D5A1A0A2F108FD635CE53D166289C02
            stream[192..255] = B45E4F98939CED60307F5609C2CEFC40
                               4F75ADED40BD987013707BE6A9098F19
                               0C17D87A194809A486B73E5E15A9C88C
                               8AD20789927F0E82A9A29F4F779E7963
            stream[256..319] = 9CEDD8EDE5F436B48C781E0D32688B31
                               BF7C7DA3AAD10B28FB4CFE0A0B98122B
                               6BB8E22D30F73F22BF7979EBD624B446
                               3FD3928201D9223463277D91B4FEEB49
            stream[448..511] = DFCD892955FDE7910B4089935F2D9D77
                               FA67EA7CC2D2DE9C97F0EE82D59A4D55
                               4A4774069DE82B23C9AF0438EFC9719A
                               FBEE89279ACFAFCF39BB5150D255E17C
                  xor-digest = 5A42BDB360C7A463723DDED9DC7C8EC6
                               4BDBABED6F3B746909A06D68DCA990E5
                               5EF73AF7FB4A30323FA385A0A97C67F7
                               65187F0691B0BF3FBCEFFB512E7314F6

Set 3, vector#126:
                         key = 7E7F808182838485868788898A8B8C8D
                               8E8F909192939495969798999A9B9C9D
                          IV = 0000000000000000
               stream[0..63] = 9C0FF1BFB93AC046B8F7DEC7C801034B
                               8213D302AC9F86BECFA8CDFB1D1E2B01
                               4F06C231BB336F84C42D2C0698FD342E
                               AB969B2F02D597BFB62B41B9D93F39AF
            stream[192..255] = 247D7F30B7A189D4A0540A8DF815141D
                               F80675FE4A0DBF36186BA4B66C46207C
                               CBAA36E7F9B60EC4021231DACAFC8750
                               5A84144B71DDD9D0813EC732E4FEE41A
            stream[256..319] = 093F61AE766178EC810384DFFC674F72
                               40AC5F77E09A826150D1E6BB2C328D68
                               A42D8FD61ACE57736FE1014FC44D8764
                               CFE0FCFFF7EE249721C8F414E0F2AC66
            stream[448..511] = 878A8D74547F783BA5E10F3E182F2BAF
                               AD55020C48EA97E22CB4B186C23A3986
                               609054BCB9EA79DE268107F8BD7E233C
                               E33DDCF537B1455A7D88F63783904A3D
                  xor-digest = 9D25476D67E9C3BC97CD33BF999F3A67
                               D2CEA66F51BAC2D09FABE1A2FC9F003F
                               C2B7EF56E59CE1DBC93ACD044D499A17
                               E254E78BAFE0F8C090C432E22427E15A

Set 3, vector#135:
                         key = 8788898A8B8C8D8E8F90919293949596
                               9798999A9B9C9D9E9FA0A1A2A3A4A5A6
                          IV = 0000000000000000
               stream[0..63] = 065FBC2107DA4A11A20D0957EDE31242
                               F4C3C647B703E124320D339821DD8E68
                               3156106E22A15A5C520382BCA5A617D2
                               5B2F625597140D0F77FDCEB50F92E920
            stream[192..255] = E888535B56D880565B4C0A9FFF8C300B
                               6ED37DC38B83D788C3145770F0B55A6C
                               ABF25AA6B1AE5B2BDDF31D0AD7EC57C4
                               C52F05E70B460851E4E83A6D3E02B909
            stream[256..319] = F1F33465565F1204F114B137BD033480
                               8E3CE20D604BF09CA61B3EC091ADBE02
                               97FE2DA9E04EBABB0BB85AE81F5591A6
                               4B395B59BE6B61B95D2516FC2368623B
            stream[448..511] = 955054108EBCA3851D09B651335F8981
                               61D3CFFE90AE3B1AC84720E55D7F32F9
                               A701490DF7AE35EEDFC4F4BE5389891F
                               85CA78A60E12AB6ACBF5B2ADAAB5F5ED
                  xor-digest = AFF011AABB1864C51A90291DE2801133
                               A8704102F43D64D29BD1A635AC909A00
                               7AA39A2CEFD15F780418B9A8B3AA41C1
                               B1F6F81B39CD174BBEE946876FC3F8CE

Set 3, vector#144:
                         key = 909192939495969798999A9B9C9D9E9F
                               A0A1A2A3A4A5A6A7A8A9AAABACADAEAF
                          IV = 0000000000000000
               stream[0..63] = 2EC2E3873A2D19A18D62B741832192FC
                               5D223A598F0764221625B65361B79CB7
                               73ECC0A8658F7C97935C4CC4B1C98F02
                               7D7F39718D75CD424D078B493FDED1B4
            stream[192..255] = D3A490309322EA17BEC6347C7B8C3A2D
                               1CAA22C5F13819BBC698A4E65071AF2C
                               B6E97E9CC2CF7598AD71FFD06F82D34F
                               02C14E2E6E32DD06344FEC91B07E0C30
            stream[256..319] = 3061A7CFED3B3C182E1E124DF5BE0EFF
                               6907B2FEE90515894E96680AE494494E
                               FDF3A2ED6741961AA243D6C1FA2C4087
                               BA7D5867DB9336BC8732E4A2D43C4F7D
            stream[448..511] = A8606B968458099BA15972F1C2614ECD
                               6A1965F29703CD6922867ED554637155
                               E6887DD5F1162EB1C3AC6040A6C00BD7
                               8A2E7564827A99C1B68783A2C4E98EE4
                  xor-digest = 3247C2543E64FD12CD04881BA0EC9451
                               7CC993CAFF15C564F138BB2BE0B2F416
                               F889566B77A75FC061AF9BD4876EF5C5
                               5940F9892EEFF7486C379C9C06AD82A4

Set 3, vector#153:
                         key = 999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8
                               A9AAABACADAEAFB0B1B2B3B4B5B6B7B8
                          IV = 0000000000000000
               stream[0..63] = E96318181A269F54CCE5ADE990DC2213
                               59C05F5FF387A4665EF4219DE4E82AD3
                               7B7164538D49A63D7C577DDB9535E7D0
                               B9DE2AB33ECAF9EB2B4E0251215FDBE8
            stream[192..255] = 9894C7E3008E4CB7C90DD524753B3BFD
                               7FDEB01759D9D7CE922B8C573BA79450
                               2D22B59D19A8120CF955106B197A328D
                               847299B96CE7EAB66E5404C9C8FB54C7
            stream[256..319] = B08042A43B7D7ECE459D78DAAA240B05
                               58AA2129613671D12B71E38B1ED65ED1
                               7C94FA972A5F53D342603357D2EB649E
                               80420BBDA39100EE3E26D7C63097666A
            stream[448..511] = 201F890EE62BA9E56679F78BECA40980
                               E791B94D2BFC53932E3CDAF55D010DDE
                               663A21598F3A266EB25E2AE5B3F6BD42
                               2FA252F889CB919B9526BD8D989C4ADB
                  xor-digest = 3C3B4A734021F6D34B02F9FFF0ECC97C
                               8904FBE220C2247158134454FE6AF10A
                               904708ECAC36B898437066C34DB1DFCC
                               35ACFB104ED8F45E658698ABC1BA36B0

Set 3, vector#162:
                         key = A2A3A4A5A6A7A8A9AAABACADAEAFB0B1
                               B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1
                          IV = 0000000000000000
               stream[0..63] = 8AB2AAB730AD2BE35D67AEDAFAB6AF1A
                               1B37B60321B0FBC8E5DAC75E9BA3801F
                               213112344E6B48C65C7EBFC3E3A1FFCF
                               7F4363EC94475EA00291EE3E8E1DF2BE
            stream[192..255] = 1A6EE4C1747D7C8396AF9E926919EAF1
                               683CBBDE91489136BFB1AF7B8DCD6DDF
                               DEE853AC653DA04B2753EF30B719054C
                               39DCEAF37D896E95A43B5A1CE7F0ADA5
            stream[256..319] = 4DD8A53CD6F54E7CE7B2DE8C4A2F785B
                               DB29C5416207D0B9F889B881C646F677
                               A164B968B0D2034C6B16E2FAD02AA972
                               5D2D7BD40023DE32DC85D10A5B1155EE
            stream[448..511] = 396A2EAA2A6DA0021B00811185589D38
                               E7CD9BF7C0595B92C9AE95C6AD8E23E6
                               C5D15D8668A459744C965B48B6C05EE1
                               47C396450FF200530DAAFD860846B241
                  xor-digest = 280F50953916BB7F753EA341D1828BD1
                               7BC69E329541E03364D6E98DE0CFB46E
                               33CBB179B80B43393B2224CB813F04FE
                               5241D3820DD908DC0F573CF25B87DC4C

Set 3, vector#171:
                         key = ABACADAEAFB0B1B2B3B4B5B6B7B8B9BA
                               BBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CA
                          IV = 0000000000000000
               stream[0..63] = 9186DE40D5CC5E7C2A1D8B01BC735F55
                               BC203F79D306D77EDCBEAC719C47885D
                               DD8E4C4E6298456733113ECCAFB0CC83
                               E67D395BDA726F39259C6F088473A324
            stream[192..255] = BE271DDDEF0B8B1CE3E0C98FFCE3225A
                               DE6EDBC37D79D07AFCD5FDF704CB344D
                               BFCF9DEA3599F9A138F03F973E368957
                               8C818655906ED30942B43815911A96E4
            stream[256..319] = 8269370C46ED550734134178939E07CA
                               FEC25AEFF8AB3EB39535FEA2B4BEB982
                               9201BC39ACC51165F03260FCFC523129
                               52CCEF82F417AD2B7B8C3C81A6F51888
            stream[448..511] = F5260FFCF2CAF56A0CDE99D17888C1B0
                               F73522C8DDC7BEA70B9CF2C596554556
                               9B90F5D14E92ECAD8FC4EC839B9B24AD
                               3103813089250F446D9F583A9B092E2C
                  xor-digest = 29A4DB1174ABA47C7AF4FA87F5D64BB2
                               C2958E324781E6E01FB9F683287A937F
                               27C1EB6C277B72C1D664A2B2ADCDC5F8
                               0108FA158A9169D2D4822C027C09D42B

Set 3, vector#180:
                         key = B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3
                               C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3
                          IV = 0000000000000000
               stream[0..63] = 0807452F2F205199755AFCF75169468F
                               2101D8EF99C56654F43A09C2DDECE891
                               C311F5E7238EA5A133B8EA6EC02B29A2
                               8AD2F7BFF999E3DCB64A8B2E71C89C23
            stream[192..255] = 8546E72891619012DFF9FC77EE48BAD2
                               35BA5502902C15C032A4F7C40CE8AD50
                               300893E656015207DE10C9F9F8661B0F
                               B3EC6C8D57DF4298BE0592A937B008A6
            stream[256..319] = 2F756804EB674EA89E6EF40D6B175DE5
                               79C92528EC48E1EE9486DC6BC79B612A
                               AF1A1188136DCAEB637E89DD3D7F928B
                               5F86B698C4243CC7FC23EE82671A0863
            stream[448..511] = 6940F708B98BD18645945DC24C1AD37A
                               A763A4DBE7E57980EF3796423270C6D4
                               6E61D99F7C38F5CDE07539562835660D
                               51BB2F440FE1C3733D3D27C1D9878BAC
                  xor-digest = 5A790F2C4225F3D126CD36F3CF1BB7FB
                               9CC72C536862B41FA4E54B5540BCE09C
                               B3621BDC4FC0DC7F1B77DC8DEE62ACA8
                               E972D238FF39222EA430712EAC38C2EF

Set 3, vector#189:
                         key = BDBEBFC0C1C2C3C4C5C6C7C8C9CACBCC
                               CDCECFD0D1D2D3D4D5D6D7D8D9DADBDC
                          IV = 0000000000000000
               stream[0..63] = 01C630447E90A0389E4D291195ABF89E
                               212178C06289808F9B43B6BEFCA39DC1
                               076AC748AA754187EED31AAE0F6649F9
                               BEFB22AA292B6261AD32AA4A319607BC
            stream[192..255] = 735992D7402D2BCE2192FD24DBE31536
                               6A4C85BE7328CF3A53FE1D8E63A8E06B
                               DF33E9B2404F4C8A3D74EFB906431733
                               71D671C69B4962EF02854E1AFE7EDB80
            stream[256..319] = 32879D22D3B1B494025AFF30B5B16555
                               EDD942DE616BE35C102031EC58425D8C
                               00DF47A83CE569433C3D03652675003D
                               6C466B39EB15827D79F0BFCCADBF266A
            stream[448..511] = B036F057DC30312015C88F176721838A
                               4AADA494D604F5147679C1DEF285802B
                               929296FF3E9049A48039F41C03A40121
                               EB2FA555E57F18E0782F52147255CA14
                  xor-digest = 411CA17EF9B8AD1E74F1F9943620D4F0
                               F289817035E85CE86F1D879855DC2A1D
                               EF5CBAA9FF0A3F501F58FE9E42FD89FC
                               15800938AEAE2E8BA5758C6701C9CD2A

Set 3, vector#198:
                         key = C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5
                               D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5
                          IV = 0000000000000000
               stream[0..63] = 62F05BBE952327AC4933CEE3850BA77B
                               C49F6E0FEFD55CA9B5F5164849E3187E
                               C1652E371BFA9582586B593EFB410ED4
                               BEEFA9B50CA53F93290B2F3D7889A2E7
            stream[192..255] = 5B1CCEFACE75306DB9D1E9A45B7555FF
                               1527126733B48F6AD70CCB64EE0E0690
                               64B2F884AD47A028A6701D36B1086D64
                               E57587E933260389078BA7B9355C195E
            stream[256..319] = 882B2D05D7CC6098C4B1556B3DD45C50
                               A9A403F34AD670764923F55EBED3D8AE
                               2C6B1FD7F5904AA17461C2ECA6BFC2A9
                               C7F077629A8D5AB0B9807B1688C22F30
            stream[448..511] = EF623D1243ECE17CB99322597D59214B
                               4C50E572B5B30FF60C6AB14A95987CB5
                               BD18F0CC8494D619F6D950A2C72CB9BB
                               F2AE14B0E499F65F8401F29DF5ADD75B
                  xor-digest = 3253187DAF7A4BF8C2646C6F921B60A2
                               B6A79F9EFD121EC5769F278B76BC984B
                               3FD6AF376B8C0CC90ABCE386E6C8097C
                               DEA03930FB2D84AEEF7B0C627FB00BB5

Set 3, vector#207:
                         key = CFD0D1D2D3D4D5D6D7D8D9DADBDCDDDE
                               DFE0E1E2E3E4E5E6E7E8E9EAEBECEDEE
                          IV = 0000000000000000
               stream[0..63] = 8904DC48B8479DCF6FBD212DDE6ED694
                               F5E57A7AC714EB4B4D1667E3E2726B87
                               2C8DB0DFDF48AE1C703953DCD8D80D87
                               98ADB5D736FF9E8D319597FD625CEA26
            stream[192..255] = C24A62103EF1129B09FDDD36E896C990
                               E46C0E03C239C43A1E21E764DFD8008A
                               B32070164CC2544B77353C0C0B7BF636
                               B4925BC0B9FE08D9869CCCD81838747F
            stream[256..319] = FD0753C8B8A8B1B89E851C59E37E5712
                               31A9D11946B985A1BF1A0F36CFAF3569
                               5A5071D1843C3232982B77E73B710066
                               BEF3DD1FE125752C70946FE618D15A0F
            stream[448..511] = 0A3A64FD42FA8E51BBACFC20442AA3E8
                               552C14C5F9273C2E287347FC6E9EC7AA
                               C2663998B187C0E3EECD56BE4A9E5066
                               8B3328194EAF1CCC179FFDF7725F05B3
                  xor-digest = 83E573D4DC583AAE78A8D59A6848E229
                               23859B733276026BDA517F1254959EF8
                               69A95A9BAB40EAE62CE06E6B5E1F6985
                               D7F3C2A92D7D67932C8CB4CE169E3A7D

Set 3, vector#216:
                         key = D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7
                               E8E9EAEBECEDEEEFF0F1F2F3F4F5F6F7
                          IV = 0000000000000000
               stream[0..63] = 277AA5A661FB5E7EA2319A63A3926DCB
                               54C56197D38BBC450681ECED2C0D418D
                               AF064E850791B6D88E43C8FAB2E89CC8
                               141C3A1B454F70110273B2E6C6DF4D4A
            stream[192..255] = BBB17BDE03F95053111321A5E7E34C6A
                               37E291DAFF18B74C8D47758384A21162
                               4B69682436535C053F79AF13F45477F2
                               DF505B227C48190D6005B3B7B8649FF5
            stream[256..319] = B61AF6C3C8AE55A78DC691651C85C6B7
                               810B5F2817E6998419EBA095202B7E5D
                               38793DFA03DE604A8565D8AA72EAD3EE
                               77A4C20F536F8BCE659F0580C60D0D11
            stream[448..511] = 64F7BC91F92A09BD45276F0D912847F6
                               4381EEA9B9AB0C4B2E4C0FCC4783B370
                               4077E35A083839793D29D8600AE838E1
                               60B76B81BFE9E0DB0405CF0D035E125B
                  xor-digest = 4767CE51AA5FE743C98BE4671D81EBAF
                               B535000B7E478286B93DB4B2AFE056FE
                               1D4D9026C1FF44F50DEDAD4D51030D9F
                               B32C996D4AB69D0027FCAF739FDB3DBD

Set 3, vector#225:
                         key = E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0
                               F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF00
                          IV = 0000000000000000
               stream[0..63] = F041D614A8916534F5E80EDCB78A47F7
                               07080264357A0F814712C11687820C7A
                               29C669962868EB0BD810D3CCA09F81EB
                               004DE78E4E6CD29EF44B8594A823E345
            stream[192..255] = 7CF42A19A22B7C3AAE0F5E7BF4C2F808
                               D561CC2020D44F05302F892DA8EE292B
                               F44ADF860000029B57A17AF1513E0ECA
                               687B0107F4CB4E2D631AC5B8747272A3
            stream[256..319] = E7618C318AB60F74791494FBF8ED1A09
                               15D553E6AD82A4F149E76D6505A2CAFE
                               A50DC7E13170EACE5A72B2DE38A0CA95
                               53CDF70BF68D96FAB39FA22980709EA5
            stream[448..511] = BC6840F2F5E7F92E4F955134B6503A1E
                               75898A917221A3BC5D80BAA500AB92EB
                               A2D3502D65B18D0D63B9884ED4C96F7A
                               1ADBBF66ED93C517939870578835F7DB
                  xor-digest = ECAAAE574C624941B2723F0DC1CE5393
                               B99122129CF10BA734B1E858A23B1E73
                               7D8C6D29A8D82C8E71A25ECEEF252D1E
                               CF8E4DF286CC2E04C5EA8931EA0740A6

Set 3, vector#234:
                         key = EAEBECEDEEEFF0F1F2F3F4F5F6F7F8F9
                               FAFBFCFDFEFF00010203040506070809
                          IV = 0000000000000000
               stream[0..63] = 13BD6B62B3C42484F1548FFE20D522A0
                               843B2370735BD08FCE18541E1964FDE5
                               3079514D7B919F3C07F5225DE58A2EF8
                               470E1D7C03347298434AC74442CF9F95
            stream[192..255] = 74C077AC0BBB0860CD1A75C2B8E6DAE5
                               5EF0BF6BA4CCB77AFE91D0A906AA2088
                               543B9BAE37C5ADB0AF2896813F272139
                               2EEFDA7507A4CCABAD1FED44C9F70714
            stream[256..319] = 06C1AEF89DF7CB47CA2377FC82EA37D1
                               E0CF511E2DD24C73BCDA112771E37CAF
                               B6B1E850BE5DB85D3AF1F54B059FFE0E
                               86C4079A87A8EF1AA38E8322B6E9E670
            stream[448..511] = 4A62251F516A841469A921B58A7BC891
                               AE1B92CE68B111B99CE35BF72FF1C15F
                               F4C1BBAD93D07E08A565E9047160DEC5
                               4BC605A8701ADC52FE2F1E3EC45CBADD
                  xor-digest = E8E44185DA30F7CE11C4BFEFFAD6C3D2
                               30D5D79F3DAC30566CD1B3E82F8A8ABC
                               B41D7694F186F5032292D89FD71A3CAA
                               299354AA5C922504708553212BDC5003

Set 3, vector#243:
                         key = F3F4F5F6F7F8F9FAFBFCFDFEFF000102
                               030405060708090A0B0C0D0E0F101112
                          IV = 0000000000000000
               stream[0..63] = CFABF67FA1DF699A709808DE3E2786C2
                               9C69B21CB6E6654CC602FBD08237566B
                               E7CA0EA13F7C9F269CECF8074922A8E0
                               02E0319E2DD58ED35BDBCB82C15A415B
            stream[192..255] = 0D8DFCD4458356A7F9F0849D110B1A26
                               357E064B0BDB7893356F86C87B326FDC
                               E4C9C1DABA8A3DF925BC7355B7A0B8FA
                               6DAEAB125879A5188DE35E8D907960FA
            stream[256..319] = 31C8238CD692FB30572F36882464FFAA
                               D1D876BC0A3A92FCE8B4A0B5A011E0DF
                               31AACCADFDA0489292E36C43F121C713
                               958E515317FC9BC8154775ABA0E9CEB0
            stream[448..511] = B121C8A34A296E5051BE2B7AD3C5573D
                               E7A09D1CE84195363E5999CA33CA17FE
                               801D3938621A804D6B85CB38DDD3D235
                               53CDBB6ACD75E261D45348AA1CA9E0CC
                  xor-digest = C3B97443E9051ECAD7918AFBB80B1F2D
                               80B9A7185AE4CC008E67ACE7507730A0
                               3A543573BB415B222B6D014D76E3DE66
                               44CFF644DCCD9F72480D9AF5832F590E

Set 3, vector#252:
                         key = FCFDFEFF000102030405060708090A0B
                               0C0D0E0F101112131415161718191A1B
                          IV = 0000000000000000
               stream[0..63] = C02874278891C94ED1C3186D276AFC76
                               D4704B38BBDF53C057B5711BD7514F81
                               6C2037DDEA1E87647633FC6250BAAD03
                               F6442A185AC2417834D054AFE5D1B518
            stream[192..255] = 67175439808C73E544314A8357C9D1D4
                               941C977D280B7ED7B55D85517E215E3C
                               F84E1F7D89BDE2F7B02B57365CF0B8B7
                               D2F5C18CDA3A1ED4083082C870B80A80
            stream[256..319] = 221FBAC59A3C7B68AF9C61FB0570721D
                               65AB60C261D471F6953F08B63B660CCB
                               95F3348FF7B94BA5D8269062208383C4
                               23D9714C88952C2B72653569C4ABE00D
            stream[448..511] = 45A0E29E968D8C239989C01A19D569F6
                               8EAE85E11BB667CBFD245D9E86B19CB7
                               7F2B1C25A191C676AEC95702DCE50CFA
                               41D733C8734B1337DAAC68934038BB26
                  xor-digest = DC1F86C26FAB7918B18CD75E9C8D3A14
                               DA6D2CB90A9AC3A78A420B7D067291CF
                               06B522D78F9038CBF24B7E8324BBEAA0
                               3EB826D4F1E25BC9510DA2248F26268A

Test vectors -- set 4
=====================

Set 4, vector#  0:
                         key = 0053A6F94C9FF24598EB3E91E4378ADD
                               3083D6297CCF2275C81B6EC11467BA0D
                          IV = 0000000000000000
               stream[0..63] = BC4BB7DC20EFA2DF9E7DB7B8C6734A6A
                               EF8B0936F0E2579638162411BCFA4536
                               A816DAF73978C6DF8FA1B6F84568D609
                               616ED37ED0A8B05FE85CB064057F1E5B
        stream[65472..65535] = 45CCA5EEB6363023FADCD9C021698A8F
                               24AAB4275E8EE6337E800892B749F9A3
                               E1C2878CA21F2FE905E108EEEA30580E
                               E04B35D2755C9994010D156D39D01341
        stream[65536..65599] = FF7F12A00733D2FB1906C1479E82EB06
                               20C3D695EA3EB5E146C49EF2B47BF618
                               523288D6AAD86345D3E799BDF2E8DE9B
                               A315BBD2195D3EEE4AAE83A125EFEE5A
      stream[131008..131071] = 63EA72323994AF3B58C8A0AAB71B4FEC
                               31661F67E7B6339ECA60E8BC7D6D9BD7
                               24A47E3C459BF803250206D89E8BEA9F
                               CD4FDB7F08B27A8420468043D0845D8A
                  xor-digest = BD8E5D99907AA3CEBA1A56D21839152C
                               4759DE9BFD7829CF53139307476B0BCB
                               3C1166EF29424C72AB7EB81AAF1E259A
                               241CFD18BF4E5CB97AE38F47C3AB02BF

Set 4, vector#  1:
                         key = 0558ABFE51A4F74A9DF04396E93C8FE2
                               3588DB2E81D4277ACD2073C6196CBF12
                          IV = 0000000000000000
               stream[0..63] = A3C365600F43EFB9F64482A5E9B6AF05
                               29DD849A3A8D3E4ABB3087B020EBAC62
                               C91CC168782672CA092856E687E2DE53
                               B1EE3180714D014EDBD0A7DEF739E33E
        stream[65472..65535] = 2357CCD0C1E83CFD73AFE89ACBC91160
                               2E7765AC08A27C9ED48191B2A7F20441
                               EACD599FD7EBB693CB00A03337834703
                               0FDD4C7FE3AB29192BADE53612EC5B6A
        stream[65536..65599] = 68BC3654B7A0EE6B749A274214F9683D
                               DC3B5A5617412A9A0EE28E348988EDBC
                               754790ADA72831C7BA1B2888CCA4D8B3
                               A043488E03A9A7374E0E825E017A1699
      stream[131008..131071] = 273727736F4F8F55EEC8F48551584C69
                               1665F7CAB18E0F3F72DCE262AC4811CA
                               E6E68B5E5567B609E6E544D9051BA339
                               4462B63F30120125A42F0011B2E95F0C
                  xor-digest = 6045F536D3DA9851E352B149A23CB356
                               6D662C83EA61D2DDA2345CD39BD6A348
                               B3A56816F095BE86C46CA9ABDC1B3372
                               B28EE0480B918DA7C86F5A0CDCBF185A

Set 4, vector#  2:
                         key = 0A5DB00356A9FC4FA2F5489BEE4194E7
                               3A8DE03386D92C7FD22578CB1E71C417
                          IV = 0000000000000000
               stream[0..63] = 1ABE0185937D528F94A35C84393EF1DA
                               2FD2646B547EAB7176407EE62B91DF9A
                               09D45A42C7FB65A33525294217564A2E
                               F1FA5F3D25C84E5545D28D1C0F2BF9A3
        stream[65472..65535] = 59B3ACFF89A5DC317B5855914C5858EA
                               5A7A67DB6CC484E2FE7A7F9ECCBE636B
                               C538A519BCA67C35A3AC1704EB52282A
                               B1C3E12A1979DC1D008D52E86B230423
        stream[65536..65599] = 2EC1785C1D3BE5FF74F87AAA4D4123C8
                               EC0D60D932B34AC5C0E4530F9A8538DC
                               980B0BD7DA19BCF185606DE356137862
                               23C0BAD6FC58C683EFBA9929006ED557
      stream[131008..131071] = C0FC679B12FCFE53F881A5A10C61ACD6
                               9F138A54DE03D3F65CCCB217CF279B7E
                               0306762406111174AF7A8298D4717854
                               D76E4E42915A7EC55AB639CF9F83CAA2
                  xor-digest = 2B9375FBCB3C247C8CB40703040A8EAF
                               3D376E61FFE14D42ECC2664D9DD9D812
                               4441FF8E7A48C7DCEE86D1BF0C3C4666
                               A7A2ECB23FEAEE77F5269B34B5577AA9

Set 4, vector#  3:
                         key = 0F62B5085BAE0154A7FA4DA0F34699EC
                               3F92E5388BDE3184D72A7DD02376C91C
                          IV = 0000000000000000
               stream[0..63] = 6802649BAA9512CB488D48B5B4D00FB5
                               3B7F37FF0A0FDA6DD8CDAAB63699153E
                               885AF17BAB77C3058E6C6679FE66A76A
                               D2531980B488F90326A4A949AD4C4178
        stream[65472..65535] = 254771F3D5E8EE31B66FF56B5BA1B7CA
                               63DD3CC5174D7D78DC0AE9A7DCE6FCB6
                               BD9D1A2BE92E507243FD1683A11180DD
                               1043D3CE6D0B14991A8F042C19804006
        stream[65536..65599] = 8F1F57EF9243BEDCB44A6B348D50E803
                               C594E5D824AFE7306A66A5A99C1E79F4
                               FC2B4713F5761278779D886A20FBBF3B
                               B521C03B1D1123AA7987AD07B522BD53
      stream[131008..131071] = BC35FF22267FD9AE240B5611B55D8759
                               2A442C9D150A08144EB100B3F520A14D
                               C66854C7A31675AC96264EF8ABCF37D2
                               4F16AA8F53FB8D5E6B5A696F292D0EBD
                  xor-digest = 9251B6497928E1F4EBA24CFB7F035007
                               915BFB543B1ED802524363A1F468BF02
                               31884D14097354F84D94C501EA757624
                               D5DCA7DB1BB96FDD1748EA070CA980BE

Test vectors -- set 5
=====================

Set 5, vector#  0:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 8000000000000000
               stream[0..63] = 675E88B4EBB7C9E13B48B8391BB4DD6A
                               112C7F688F623FCC3025212158267D59
                               078C39226DFA40A6EFE5B8476C66D9E2
                               01EA921C0BBAC2A06E478578EAB2E141
            stream[192..255] = 196EB7F2CFEDA01ECD73FD1D5CDEA854
                               0E87AF023B664C6F8392226AB2B2D4B4
                               F1A7D6123DCF3F1235563F2DD664115E
                               AAAB98529C8714486327ECCCC7E1573A
            stream[256..319] = 83D9E5E28E20C888E27AB822BF50AF19
                               6E1EB89A8A078DF20F0BE97C0EF8B441
                               0D51802C66B245F6314A4AF1713F372E
                               3237C1B6FD04753A310EF21405E07E51
            stream[448..511] = 2EC336F64E4587BB8B578C323C025554
                               5A152F7340E83C64F1422D5D77B3B1BB
                               E1552AB5A2085B88962C1BEEEF0799EF
                               84CBD94C433A8C9E0D6D7E6A0B0B6C04
                  xor-digest = BEEDAE03356A16E4BAA3D9329B6DC83B
                               2C2287A33E2A4352C2CBC634CFD45622
                               547F167B14323B454448EFBAFC235D29
                               6D27C4CB18BA43A6371AB4770F1B5B28

Set 5, vector#  9:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0040000000000000
               stream[0..63] = EC23882036E96BC32349C38CF9AF39E6
                               BE54597D5751E2F4E8462F94AB50CF4A
                               E84C593E79BBB1C516E5D94C1C20DFE7
                               852874FD8A27F16B0C8214AB746B8378
            stream[192..255] = DE98F68F118D69EB9A6D9C1FFBE26484
                               404D91C8910B7152CA5936965537059C
                               81CCA83D524CBFB1463DE36999DEC074
                               A3B6DF4B8B1F0145057DDFCC86D9479E
            stream[256..319] = 7AD87D960F9E32AC40F24FACB907FA60
                               75D4111F80BD92C4B9EEE0A4AA047C43
                               6EC7FCEBA114EE2BF06465EC828ECDF9
                               9B8FA81B2BA8EA4B975361BCF45DC5F1
            stream[448..511] = 998059FB9F1CCB625594C54E4A7CB857
                               1B09032A1C259A1BCB0A6E10E691DC61
                               3E59A238EEB77FBECDF86F89F5F9DA13
                               08F4554E6567F37850D0BF3B571924E9
                  xor-digest = 2C18B90499A32D427405FF25D31F1656
                               B832FB1620ED584D6BC39E5751C20123
                               48CB162921C94D9B47C60DB648C9FE3F
                               095C1759DE242828B22FB37904A16A92

Set 5, vector# 18:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000200000000000
               stream[0..63] = ADEA4C486447ED9B1753752EF188EA2C
                               E2ABB0F1E4138DB089A77BA4B9792FE3
                               05C9A14A652C51434968FE71DFFFF1B2
                               73AF1E54A10FDC6BE6EB4DEA63306365
            stream[192..255] = CC7E7C21F3FD39DE0A06C47D45DA9C33
                               2E052EDF0043E01D7A833B96A1DFA06C
                               D673F10464037C89A5552DFE7CFF2E72
                               52BF2A16F5C9B28076435BB9F01CC7F0
            stream[256..319] = 1BFD92B9C409330C708CDE770A92B496
                               A5EC43FBD829FD13ECBF45604731A123
                               22D9B4C76932917402098CA0AC423696
                               EA683B7A21EAD3350DE2C0BC29E94E5F
            stream[448..511] = CD29867142874CC1D5B0D74D9A955995
                               EECA7960F20E10E5692394435B6E1DC0
                               6B89C357181C59473A151CD46A9B9C35
                               ACF1C6EEAAC80BFED5856EE8823AA8A1
                  xor-digest = CDBF897BA3D3925CEEF1782891F9EB80
                               23FBE3E16680DED6FFF9934F3723FDCF
                               6361CB0E61120F59A3C31F1EF1093D15
                               9C1804F91A1B3544DB33D689720AEE7D

Set 5, vector# 27:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000001000000000
               stream[0..63] = 7F08734392894C728C2F04D5AFAC53B1
                               3EE75C2E64C10CE7288048F169A1ED42
                               46918C4D0DFE1D4598D493508CA21204
                               ABDF934E992B2519F5D5141AA4BFE070
            stream[192..255] = 71EFEE65DD4967AF04D6094338F0427F
                               78770CBD5777437DB3FAAAAFAA981568
                               4F6B024D69D18EB46D4F30B9F3A63C5B
                               064490C9FE1EDB8ACD28B392199FF604
            stream[256..319] = 31B7212C81883D0BB9B43788773E7E5C
                               0D7561272141720D5E7AD183E57CB7F0
                               7A338FF849B5D99C72239675954F4480
                               8F22C5A3F012208ACE5220A75B38ABFC
            stream[448..511] = 5D86F22CB32C99D3F8EE982275D61686
                               1BDEE942065FCC60097FCFE00DE69325
                               65EF1C9FD2CDDD4AAFF1EDED35A5D6B5
                               8010AC4375DCFCC0AE6F8AC90D4CB279
                  xor-digest = 7E849F63079253C3780AA208EEA5D9C9
                               D0104EF0FD7C9CF82258F509DD4F6E71
                               B9BC6C2CEB4E41F84BE73D32CC3CEC2B
                               58845C1023C785D55514C2296CD9A709

Set 5, vector# 36:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000008000000
               stream[0..63] = B361CF9742D9B51A3F58D77ECEE24BF7
                               4F2B610C3C9C20FF0788718203FAD591
                               A294E9965CE79D9D3A6FC13F98E4B36F
                               7D14B81CB2B56650BFEF65BC4A687D25
            stream[192..255] = 567F472CBBE17F7021015BA0E0F71B52
                               EB60AB21C5FAF6C7386002C985985A3A
                               78351F79FF50B5835DEE97C83BF44780
                               7AB8A55D408FFAA4DB88E3CAE53400B0
            stream[256..319] = 06B90181E4952C14E5CE7467A82F709A
                               E6AD384962826E7DE3F05AC3DA3DE72A
                               3495624F546334836D522365DBA4C43D
                               2827F0E332BB98DAD8937CF66045B5E4
            stream[448..511] = 6DD88B105261CAE39E9D1F1853A49CCC
                               182D6AF3E4AB31410C50AEE77ADA5B05
                               22D13B584B0F165227D4C5413F893ED1
                               742D6AD92CEDB766A90817D4DD8651CB
                  xor-digest = BDF763D8211E2D8F81088B812FA70E07
                               AA6D90C6CB6D5967ED412AF23900ED53
                               B120C2AA91A737FB1FBE5E17D464A415
                               A56D473460465F412C9AF48180C59EB1

Set 5, vector# 45:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000040000
               stream[0..63] = 45C7AC47C39DE1C9FF0C2107F52853DF
                               B62E17430420B44E94C0C8A43BAC06FC
                               A5D17C06252E8F2C061ADD684F403AB7
                               2F66B0077EF40927CD69CF88C3006935
            stream[192..255] = 7FA0790E8E895CA378F1B5899A7384D7
                               E084C27065A0CF69DF06790EEEE50529
                               E0F6709A48AD3A38D61D151ACE6A0B8C
                               D4382AF60BDB6538C50A437E7C0EFDFB
            stream[256..319] = B9EBD0F081387A117E8E8AA77A06671D
                               4B5E559E47715E9B0C36248183832DFD
                               22B394E359B6435F3DB6ABAE9C0ADB66
                               61292A4901B5CF6A8728D02335F26B5E
            stream[448..511] = 184B25F5BD8C5DEF413EE9FD4E2E19DD
                               251EEB5BEF34AC76EC629525C1A87AC4
                               50B323464D51C59DE539755761E87432
                               B87F4CA9D91A0E8A7F39B42903E8B671
                  xor-digest = 10A5A277C10212398789DCD19F20E7B8
                               C0E8F62CF13160C43A0E8DAD97454E39
                               A9E98C1F26B3631BF7FE58036E98636C
                               38FBBE9860478DCE634D4941178BC89C

Set 5, vector# 54:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000200
               stream[0..63] = 85500DAB66ADAC980365E36105EE9BAD
                               5F352E48E50BE5C36BAD0CB1BDF04791
                               81047B35775C89C9DDAB499C1F659DE4
                               4139E56506E1EE60634F46E5B63C0541
            stream[192..255] = 6540C5CA47613812535E87974ACD34CD
                               885B3E6C00D9132B50238DD543F177D4
                               5EBBB58D9F54358B3E1FD085389ED5B1
                               B2E0A9561F58A541936E2CB82A9BB731
            stream[256..319] = DCED1A5684A1A52790FAC7E510E508B6
                               D6E09C79487E8DF95794727EE7DB4CA3
                               076F1438396C3A0BB525FB24EA115293
                               4432A769F437B22DB57B44D53DFCDE7F
            stream[448..511] = 6CA881015FEF1D6C0900125A952C5CCB
                               49A0E1E2A33B1F2FA8E60B2EC7FBCB20
                               9B03F73B773CFF4ABB741C05C923D0A8
                               C31EA6B6B11E5A53F1D1DA3CB7C50675
                  xor-digest = 19C172C1FA6B0CB99D605E926467F77F
                               18C70C32ED8C71226B668CF98A9C1E77
                               58DC03FAE59DBD92B4AE5F5CA05DF636
                               90069F4EF02BB8DF8E208595D8907978

Set 5, vector# 63:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000001
               stream[0..63] = A2D286F3455C44D4F1B2FF7693076CD7
                               BAC089D1957B515BF255E69D1D84F836
                               2C06FBA6D05F59422578A0345CFE42FC
                               9645E3BB300EEB81025A1C60B4FF8DD9
            stream[192..255] = 9AACACDF178BBF9D9D28185FF1241AA4
                               CC0DB3E6C19A4D77DE559A5CBFC4CAF4
                               0E720FC9DAC5A833E87A9C63F0195DBC
                               DF6CFC8F41B34D0A8583C134F7D5A1D5
            stream[256..319] = 5A7D71E055D16E925D9C65657B02943B
                               5B4A69ADB9FA3918C65608D8457F40C7
                               C6D46480101382F75DFB9DAB1BC70E82
                               B5C49D51A989921EE2896FB78C1DAB3F
            stream[448..511] = 8BBE64307669E8FD83EAE1B8EBDDE7ED
                               8380D256BF5A7D8C7B104056D84EED32
                               E41BA579882E67C28E17AA5D5BF90583
                               D907D423A3017B0FFCB68F4D67172466
                  xor-digest = F4453F7147346281294A38B3E83B3F9D
                               E7F7370FB9765CACC6410A7E92919C7D
                               D270C421F932B744014EEB0952807AD7
                               B9D4E53A77B24C252B97BA722AD14004

Test vectors -- set 6
=====================

Set 6, vector#  0:
                         key = 0053A6F94C9FF24598EB3E91E4378ADD
                               3083D6297CCF2275C81B6EC11467BA0D
                          IV = 0D74DB42A91077DE
               stream[0..63] = 420609C7CDDA902D6FA7CB264AB0C89A
                               030DB2E4BF18D179BF7A3114F2266E2D
                               5519F123A079233B5FF82481E508828D
                               AB2620E5521056FAF60BE1489A716971
        stream[65472..65535] = FB60D119C4443C7978FD8D68900D5478
                               D5ED042DC632FB5AE025896B8E61F20C
                               D169FB57A72DB4BCAEBC9671B146B99B
                               ADB2EAFF56E83CECC94A913A852B6711
        stream[65536..65599] = 854FE0BE60EA49F4D389D9A63E6BCD74
                               263211722D6D4B3865466D806F61E719
                               F4C76909436897E64C3C34B91E8D0220
                               776DEA862B0C692DAD4EFF58E0CC08E4
      stream[131008..131071] = BC4B88C2D58367F7A523D6DD118CFA6C
                               7EFA657B115E9B4D9E54C10BFC2DA2BD
                               D71F2001EA6A22FC2DC5BBE682C2A072
                               DAAA8520E7B74CD59B75305DFE816F9B
                  xor-digest = 70174FB4365E24E02D83F99287BA8E5D
                               DF1E4B9463880C702C084E5624EFA8E3
                               17D384DF0EA64E5A0D038237E0D5857B
                               A140D0FD20946D0DBEA04DD83F12BE2D

Set 6, vector#  1:
                         key = 0558ABFE51A4F74A9DF04396E93C8FE2
                               3588DB2E81D4277ACD2073C6196CBF12
                          IV = 167DE44BB21980E7
               stream[0..63] = 2178746B16C0E5A374796B103897EFC3
                               ED577627927509B689ABD22549EFDDA1
                               066D7838E2A27446C7C9A76D3F6F6D4D
                               15D1068F48ABF9E90527DD3C9B1F26F2
        stream[65472..65535] = 05E36B4C91E994FC8B472882AB771283
                               39B5B9D592A2CBDC6EA7F13B3B8A0741
                               FE0D4D8C6C6F0A97DCA3682A42CD473F
                               DE17834C56336EDC093BDAA32F048449
        stream[65536..65599] = 8C76D08E44F7AAFC9385FE492D9061E8
                               BEEF34CB156A1C486A4EE09141BCBF64
                               A5BACA7BC26CDBAA7E3D0444C38E90D0
                               8DEC4D9C1F5674F369ABAA675DE9D6EC
      stream[131008..131071] = 803F9EC6663E5C22CD2C83AE8042C1A1
                               EABA7FF342897B9B1EEA3B59672DEE7E
                               6BADE60C31E1F39F134735BA9CD0FC0E
                               183287819EBAB974438CD44288797B9D
                  xor-digest = B7FB27DD93BAD8298AB78590E1285047
                               F457923FF11FBB9CBD3AE57CDD5FA3B8
                               F0A3CD8D9F443EB12AC2FC75789B4ECD
                               C1C5D5F46136401DA5922E35B98B2CA2

Set 6, vector#  2:
                         key = 0A5DB00356A9FC4FA2F5489BEE4194E7
                               3A8DE03386D92C7FD22578CB1E71C417
                          IV = 1F86ED54BB2289F0
               stream[0..63] = 3A87FB3AA2A4C29EC035203034518F38
                               88715C19A388D01F698483FC88E38716
                               BC432FD340ACD9C75838776E42DD04D7
                               3FBC567A61FF3FD176DBF9C714DAB434
        stream[65472..65535] = 5513BF93B6342E402993FC2B4EED9F7A
                               8225D65B1969BA6F3965001DE4D06119
                               40E142CE86EC31DA6B70C80BBCA6526C
                               CBE6C55D26374DAC8AB5E4D439AB109D
        stream[65536..65599] = 0EA0A9E444818A8AD6E1929067445797
                               96B65E391D42EEF424E61B4E811F9909
                               6452AECF24508D2A71A1E248C9B0C1CF
                               21B09409434B1E916C0D9EA430B8248F
      stream[131008..131071] = 5CF71DC4DB6CEADB293AF79754E3EB80
                               C3A30DBDFA712031EBF050079301F490
                               38C278F5CE6C6EB684192505382A935E
                               1BCF27C4A41C97624A531ECA4B2514D4
                  xor-digest = 7E014E5D31459EF92EC398F92AD75FE5
                               28AEE41CDB206DF39D453A5A37D43604
                               BF1F54121FBF222C21579A00B7E0F776
                               DB481F31A445592F918CE5665C662A5D

Set 6, vector#  3:
                         key = 0F62B5085BAE0154A7FA4DA0F34699EC
                               3F92E5388BDE3184D72A7DD02376C91C
                          IV = 288FF65DC42B92F9
               stream[0..63] = 36CEB42E23CE2FED61D1A4E5A6E0A600
                               DCCA12CE4F1316C175C0BDE0825D9097
                               2F574A7A25665FE6C3B91A70F1B83795
                               330F5CFA8922C8F9B0589BEADE0B1432
        stream[65472..65535] = 09632B7D639BEBA755BEFDF5351049D3
                               E39ED53E3F61CBF84BF4976B61569A18
                               968D7DD5998BDC61728D84EC9008EB04
                               F958A75B74E6E2B9AAFB2D62736101A8
        stream[65536..65599] = 67A7D6E68367F72A50451DD6FB73DE9C
                               4A9731758AA4E8CD9B3F0D5F3BB18FA4
                               71836828AE901667990431DAF2F25B90
                               146C4E92CC8B16F335B49B8DD7B06EFA
      stream[131008..131071] = 70ABFF26A28D19CE894EE592E8CFBE7A
                               CD7200EF317C6D7847C31CD3630A87FA
                               AE0BB6FC52649DDA8ED0C5F0EDF553C1
                               9B991C4086BFA4D78F5108399CA910E5
                  xor-digest = 3024FF4FF97C1C1FC9BEA85B46BB2441
                               B13EB173BBC874698DC796C12A6AB654
                               00E233A5F9629E42E22AD8D830B8CD7A
                               3B537D103872C0F2280551B9486070E0



End of test vectors

********************************************************************************
*                          ECRYPT Stream Cipher Project                        *
********************************************************************************

Primitive Name: Salsa20/12
==========================
Profile: SW & HW
Key size: 128 bits
IV size: 64 bits

Test vectors -- set 1
=====================

(stream is generated by encrypting 512 zero bytes)

Set 1, vector#  0:
                         key = 80000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = FC207DBFC76C5E1774961E7A5AAD0906
                               9B2225AC1CE0FE7A0CE77003E7E5BDF8
                               B31AF821000813E6C56B8C1771D6EE70
                               39B2FBD0A68E8AD70A3944B677937897
            stream[192..255] = 4B62A4881FA1AF9560586510D5527ED4
                               8A51ECAFA4DECEEBBDDC10E9918D44AB
                               26B10C0A31ED242F146C72940C6E9C37
                               53F641DA84E9F68B4F9E76B6C48CA5AC
            stream[256..319] = F52383D9DEFB20810325F7AEC9EADE34
                               D9D883FEE37E05F74BF40875B2D0BE79
                               ED8886E5BFF556CEA8D1D9E86B1F68A9
                               64598C34F177F8163E271B8D2FEB5996
            stream[448..511] = A52ED8C37014B10EC0AA8E05B5CEEE12
                               3A1017557FB3B15C53E6C5EA8300BF74
                               264A73B5315DC821AD2CAB0F3BB2F152
                               BDAEA3AEE97BA04B8E72A7B40DCC6BA4
                  xor-digest = 21B4FE2B96EF241D540C8ACEB15649E8
                               1F14F842A86DE5F79EDB11480F3FE6AD
                               817F8E559ABECDF734FE171EAEA9D27C
                               BB371955BFF422550AE57649D44B569E

Set 1, vector#  9:
                         key = 00400000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 6C11A3F95FEC7F48D9C16F93CC901EEC
                               8D347BEA4C64B63F3E1CD88DF4F03A59
                               5ACC0500EFC616DCFEBA3E839F0F72C5
                               A54A0801B90C864EEAA7F48CF37DC365
            stream[192..255] = DEE7D7D48794FC33F7920A21E5C4451E
                               109C86E19CB9B78F6B1648E8389E415A
                               20F8CCD89D2ED0A5AEA12794ECD04DFA
                               CBC7675C82EEA65F35C27431F96838FB
            stream[256..319] = 0D0234066686CE699BC8634B4B1983BA
                               04A153B7447D13DC1371F2F5198FF556
                               182451F0ED4227818172E59F443406F9
                               9D52736DA7C52235A2DC91ED11793CBB
            stream[448..511] = 8E1F1453E17B186C968751E8D7EF7EDB
                               6C7E8B2EE2C40071E6AB2DEF81D28358
                               0657DBF6DC5E651D7B953661169707ED
                               5337E7317EC78D6F4C2A4C95E354F455
                  xor-digest = CC44B389976BBBF545E4A5C52710E57B
                               7114F17EE9226D300683EE53B6626AD4
                               D04F134160150185176687E3791A4523
                               B8FBF1E15C60BB7F558AA20AD24E7EDD

Set 1, vector# 18:
                         key = 00002000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = E27E394CC6B72EB535FD92D1BDF9F5D6
                               24671D5BFC9EF233F6B51F12BF338AE1
                               72DC8B7F4CE899BD5FF85B0546F022DE
                               B91FEA1ABAC32EE1F7B671E7D6DBF9D6
            stream[192..255] = 0D404B2411A81D85FBDB104B5FB2C42C
                               84FCDB61147DF1D12CDF8B3D74A3B570
                               EFA84498D94841C548628CCD66D9B9AB
                               22ADCDAB6E829A697A9E4FD1AC9F788A
            stream[256..319] = E0239CB5125FCF463A96ADA0EFF1DDAF
                               7ADC1964F28AB98FDC304538145190DC
                               4A6810413A370EF20776DBF005FC4244
                               C946A1251A4258B01BD28655D91AE182
            stream[448..511] = 3CF060F8989604C82CB269A963A2BA57
                               D66AB146ECE690A34170944406836A6C
                               77414BB4FAA5E6EC5E2C31C292645F69
                               C481FB1D7E33E7E09CAB6064CA8B568D
                  xor-digest = BF30A8729FFDECA0433C3B00047FF233
                               E5F849498660DBEAB7A6F86BFC98057E
                               2BF86DC83A887AB7990DE1090F8D30D9
                               4192CC769E002D66509972AEB0F28962

Set 1, vector# 27:
                         key = 00000010000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 7099AE9D9CB1AB463F1386999890EEDE
                               6B02D8D6679BAB095546A93A3B3718CF
                               105A2BFC94472D8AFDE35DB1A033063B
                               F723639C2031DCEB76641B40D467F440
            stream[192..255] = 4CD159A07A3306134984017876A68392
                               D38062F88DC1E28CC98669372971F28F
                               73E6138C74E6E2D116A8AB5754746D8D
                               80CACC2584D95D6AE2AFBB99A17F0438
            stream[256..319] = 0BF7FCE46946038217A999FB9F43677D
                               20537393021ED90AA6C869A349E0D54D
                               6DECE3E15570C6774CB9AB337DDE57BF
                               73BBF71F0771BD5AB172BFA1D6EA6035
            stream[448..511] = A128783FA0BC1706701518E9B176E912
                               639627536C0CAA99B230183343BF4C46
                               D5A74C7867931C307731FF8028C9A1DC
                               B4B8487DB28326E8A1E9840C02AB0453
                  xor-digest = 5424FFA3ED6240E71F60DF954EC9B729
                               265735DBE7C86B724B6F06328AE71F81
                               926A1146261373381D666422875F429A
                               3C32829F9BB2B66094422307F5AE1C2C

Set 1, vector# 36:
                         key = 00000000080000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 1AD36B099C145A144D041FD42EB2630D
                               70D402934C712C49E57B93F9744ACB5D
                               D9FA38F54B8F23595F7B5639C68056F3
                               3D607077C6FA583AA6039E943C7EB610
            stream[192..255] = D751D6B282EC6D64E056D74AC3B5F7A6
                               72D28952CFE4A3A440A4F1DAACCB7871
                               35D647689C95B92A803BC7BCDF1E87D3
                               E7A8DFDBA625286FCF60CF7470C6BBA8
            stream[256..319] = A5E4245548214E6DC5CB767720BD6FDE
                               44A34AF58979E57C5CAB2258B7990B53
                               DA217C33BA5DBC63F4398BC08CB3F77F
                               9056E88A3370DDB4EB7E49494E2BF291
            stream[448..511] = AE45D93DDF3DF28D6D6D359D63DC5EE4
                               25D5A2CAE27ED3EF4328D162E5162F44
                               4982345D965C1C559361839FE34BDBBC
                               8D9FA0A607AF26A8744BFFAA9F06590C
                  xor-digest = 63F3E823ABDB4EBD45FE02D0538ACCE4
                               38A8EB93C6F1859CA0BFDE75F617AD5E
                               782BE769CB8A62D8F4500D4394B7A0AA
                               B710A83D93125D2242B166A6B68462E4

Set 1, vector# 45:
                         key = 00000000000400000000000000000000
                          IV = 0000000000000000
               stream[0..63] = FECD00A21964C769B7C758C4CDC71D1F
                               5AE789EEC3F86D51CE6607C5F7BFA94A
                               2A33B291B806B660A37014728D413066
                               15E76B71C814637335ED143EDDA88B47
            stream[192..255] = A4F77595FCC27BDCC93369BCB56B5BC1
                               836231722ADE1487F7CFEA34DAD05838
                               BFAFB3BD4FF23C06E3E247559C50E22C
                               3E1CAEAE31159BFAF8F18268A3EF1414
            stream[256..319] = A006010DA2B96B14559BA35BA4D2BFE6
                               3F43C6237038C2ADF8B7A05719774846
                               A48311D1571309DB1AAA89F89F18CAFE
                               F060D8A2DE29ACBFA51F965A0D93A3BC
            stream[448..511] = EB6DF337C0616D64ED606711BB0369B7
                               36C3B8102E3DF0FD58CC9C58833BD88F
                               093077030D0DCF4BFD92FFA738B10F81
                               491F4811DC4650B1A562CABF7611CD41
                  xor-digest = E743A5C3502FFEE64131ED9754D58FD5
                               2E53D4E2B993159CCD2F017CC363A6C8
                               A3AC5FC7332E6849DF9DC5007B32015A
                               F8FFBE21631A532D1535615F195941A2

Set 1, vector# 54:
                         key = 00000000000002000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 7776759D4299A92C5B0A584555539ABA
                               EDF25E153EB4BAEDCAC7CD08611E65CF
                               DC705A7D6221ECC8B18B9605332FEDA9
                               FA04BB8770083F2C26DE4C185816C67C
            stream[192..255] = 47D040516AB62987A882B29D78D7ECE6
                               45F303025F81D46DE11E3BC8DE1B8108
                               96B7FC69A0BF7E215C72BF40C29C31B3
                               5C0DFCC8B919BE2CDDE46F0D56F485AB
            stream[256..319] = FC87ECAAD4B508BAB006560D15A56EE2
                               4822CEE1DB3CD0C7850000A4A2C83040
                               E96D3BC41131D5ED701A8D8C2DFA2E02
                               92E174DF205FA6CAB7312A1AACB29B86
            stream[448..511] = 43A19249F5110965AFECFEC772798E1A
                               8C0995B4E14E378C41751B3D55A16D1A
                               15E15A193D48EA3E6FDB774050367637
                               5E5C84263C3D95270C3E57D1C2B75398
                  xor-digest = 8460CC004BABF78FA08E113B0FD920CF
                               3FA66E62E530B46AC102CCF2E12AA0F3
                               075D46968DBD41235B9EDE36E6E2010A
                               663B16C2994BF48DBC7B31F24A4CBB43

Set 1, vector# 63:
                         key = 00000000000000010000000000000000
                          IV = 0000000000000000
               stream[0..63] = B4355B1DABB58F928C557EB6F36B0E1D
                               000865108C2949A3DD0F6C24350F73B7
                               1799E221F65C21F2F97DFD8DF4176FA0
                               B9484B4E015601AC32CB2168B1670953
            stream[192..255] = 78E083E4A3A16F3528179B02198D91FF
                               74911312C68A8BAB5E6909FB8322484F
                               D16346A9D5F422524D9E6EBFC688203E
                               973C717D336B469BE5277E963AF9847F
            stream[256..319] = 74F483674D84B36D05FDD103B0436180
                               14A27E4C91209E669FBA4E2505A2746F
                               223C332104164DBBC618F9136111E80A
                               F0B157D9800EDA4DEDF6B0809919290C
            stream[448..511] = 1080DB7D1C29F1FFE980A3914381D9D6
                               B01205DC497A98CDF24D9B7BF1E02B45
                               CCF2729B673D309ED8F2C23709F54DF3
                               588E1BBC2BB4A73495DDDC3800B436F4
                  xor-digest = 3232850FEA37ED5AE9D42C9781A1DAA9
                               766E77B5B07C8AA7772F735F5F462714
                               DD8394AB20413D174B77892C7FE578D3
                               861887A66A774A47A1686661BF9CA45B

Set 1, vector# 72:
                         key = 00000000000000000080000000000000
                          IV = 0000000000000000
               stream[0..63] = 5D23D5CC42E76F960CFBB202FDDB6569
                               2686F0325E0590E8B55DED9DFFB821E7
                               9AC149C6138D9D30A18B09F5BA06EB91
                               B7051E00B7EEFD9F2217E0E5607DF031
            stream[192..255] = 2E31DAED8F8BB7EC2134F72E992776D9
                               315E9DACCDA1FB3C2FF60F4AAA0B1B41
                               6846E66CA20472E955C647241E30F6C5
                               D6769CB8A52D33776D25EADCCE63B0A9
            stream[256..319] = BAEBFCAE4A0B77CD9DAB4ACE819FCE20
                               37CD8F36043957605D245BE91A60C324
                               CA6B75075D327717F23967E6B34D23E6
                               2FF8A608D12EA6049887AA221FD55080
            stream[448..511] = E9327BDB9BF1A454562357E832C0B3D6
                               B88005CA2DB661DAB2081A76FDC3EB9B
                               C1220A0A583B602CF6A929FE88397334
                               C4F262D2843A6DB46DCE4F2E9778D5D2
                  xor-digest = 7E2E09B8D1EEA9163A1BAA3F16836000
                               E6929E16BDA46362C0ECCA675CF7D396
                               94C05CF4E450C1FA0A9F61D8D29CE610
                               AF7BA90D44B49E378CC266F5F265855C

Set 1, vector# 81:
                         key = 00000000000000000000400000000000
                          IV = 0000000000000000
               stream[0..63] = 1F8DF67D1B37A46589BF09E697E5BFDC
                               4B3DAA4EDF853DFFE6B40FE5C22A9654
                               335AFD1F4E69CEE9908F975D05C46872
                               E2FDD596B5322927456CB729AE4640CD
            stream[192..255] = 733DEA06FEBE13519DDF1EDDCFD11ACF
                               3BF9026C1DAB35BD5071D7FEA4CF5817
                               0151BF93F872293980BE610C78629A15
                               C58A4AA2CD19501241E5B122AA36412F
            stream[256..319] = 179B914891E56D4819DF4390B7E4BB61
                               04FDEA72D27380054A46042C86257FC0
                               22CF20238085F6B1DD25BF2F476A895A
                               3F0DEA186F2955CA10F98E4DC8C1438A
            stream[448..511] = 1573E4E628DF9B38C4DD8051C377D6C9
                               EADD740DC70690721802F93D3408BF94
                               5F91567144DC1A1972902BFA40BCC5B9
                               4976AA7EFC5D1E62AA37151C2DCEF158
                  xor-digest = 2D8A5522DF5C30F4CC57B01F8909D2F1
                               049C7F8AA62F6F25FF02962D0F994707
                               F509E48AE3F79A40D79F57DEE548E6C8
                               0560A0BF5DB82FAAD606DE5817DB3AEA

Set 1, vector# 90:
                         key = 00000000000000000000002000000000
                          IV = 0000000000000000
               stream[0..63] = 6D1B868ABA77A90F9C6F9EAAE097A0AD
                               C07CA39DD34C5DE507C3460118EE1CF4
                               7E6C9FBFD8C99D0D13F9D99C8A398453
                               3D32F7709B04BE6099DEC1AEDCB2EB09
            stream[192..255] = 2A5EE11DCAEA07D4F2786334F9B9E7C8
                               618C930CF6938B6DE60DC0BD7D2CAB6C
                               385A80731C86D7C0679114B41D9ED711
                               6BE0C4219B0C2E948C7DA08F225A7DDE
            stream[256..319] = 46FDD22A56AF991776BCBC1C7D31CFB1
                               66236A67D69907DC7364013D079F1115
                               5FBD9615D93483A3EBBE6FDADD4A7D37
                               9BFEF3505BD825BA78F829F370C5E204
            stream[448..511] = C43AC2769A7584C15AC38C11420C1F6F
                               C58DD5FDFE6AFC40D1080A9D39DB8BF7
                               1FE0B71D418BEA72B358A09733307495
                               6FABD024FC3173DC63625CDEA8D39F98
                  xor-digest = B0BAED050A475E6668B52FA435EFB53F
                               878EA3779B64233DC755DD7352DB725B
                               1334552C6D637D02EB66171C6E23E65B
                               C7761073315059CD7C0ECC96E89D184C

Set 1, vector# 99:
                         key = 00000000000000000000000010000000
                          IV = 0000000000000000
               stream[0..63] = 02FEE3556AB7BE3918225C7E2C9DD564
                               4C652C559E125506CE5C95422A863B65
                               EE12CAA0C8ADCEE4AC1C79C4CA88FBB9
                               DCB04D3AE378A4FB677ECE629429A19D
            stream[192..255] = AF07B7F4D4F6384D26780550882FE358
                               CB6E7EF1A47106FE58142C632A64326B
                               5746E9025A5E2C9E0EC581AE5EBE0401
                               587DA227D857848B5BBB3162C7752221
            stream[256..319] = 76D77B034955D535612587FE0C0E08C8
                               86C77456E460CAD6BE9306C6C314C291
                               09BC5A9CC871651D2F501DE771E972B4
                               9C713E8C83BAE968904F66927CBB17D7
            stream[448..511] = 8C96C93A67952AE3DD9F3B4BCB43ACD6
                               3C58F2D411823CB034B1945298F48230
                               C28742F6D0F84501307D32B2E7E59BA3
                               87C80800B52F07D903079C37785235B2
                  xor-digest = 2D54E5EB884F7BE5997BCDFD6779BA7B
                               84183AB0C400E330F25EB94AEDB96D39
                               1BEC73CD6FEEC8FC04B40B39C5F6581D
                               4A20634FC49E5C047D0922AFDB067482

Set 1, vector#108:
                         key = 00000000000000000000000000080000
                          IV = 0000000000000000
               stream[0..63] = E9C8B0B8026ED49553BC4056B6EDB496
                               153FBF2473B014882908F6C304C3B6C6
                               A876F362D88AED4706DC6E55E2D32C63
                               9F055B75A27D73A52CF5D31CA01205DB
            stream[192..255] = 0B4E2AEA3375E11B13FE15A4D9DCFB88
                               F2B5082DC2BF6AD45819B85B7751F388
                               68F5C1420A9AEF893DBD7FD2E4E1ACB9
                               126E03C0D3C34DF32F2A9E90045CB5A0
            stream[256..319] = 0253CC8A858FD5E6083539619D870D2F
                               5D2EED3959EE1CD6FC313540B9978324
                               9989680E440289E0B522A15DC12919B7
                               DD4245F5C96DBB8FC63A62627397D7AB
            stream[448..511] = CF7C0864F79C2600B76E20ECCD1274E1
                               303953B3F71DDB8B5F06578A71E074AA
                               5C9A3BA8781676059F3C2183CB2D7406
                               564DDBBD414D825B93F2494CE67D1D79
                  xor-digest = 731EAC3F6951E217045D40068C525102
                               B54AA5066B5FFA76698F7708731F7780
                               93C053FAF16C35A0601700854EE3DB13
                               9329C0DE373C31BA0F43943C3A45D3D2

Set 1, vector#117:
                         key = 00000000000000000000000000000400
                          IV = 0000000000000000
               stream[0..63] = E50126F5DEF07EC86CBD598C6D73EAC7
                               48BC668A409889DB23E4981F2602EFF8
                               D131E427690647A978B4EDE5200A857C
                               69B8BA545ED510A5630EDEAFC01D032B
            stream[192..255] = 0400F9A8A08D870E68AEA32A53C8F26E
                               3AEC66BD06B9A1A2D2D546ED510F3299
                               BE1FEDD82BDADA6C01EDC115ADD2AA88
                               280E98B6BCBDC69A9F23D36BFE5552D2
            stream[256..319] = 6345F2227B5756CF1F78503998B0270C
                               DF983AAF91041470F36C60040DE9FAD8
                               05AE2155D53A9FC78CFACC7F2EB0BCB2
                               2840E67AAFB677939E05E59A4C32A559
            stream[448..511] = 1ABEC75411C0F9BEF100B62E2F0DA94B
                               49477B80AE0A0A00AFAE39C48E0DDE25
                               DC38BB8D31DD48B421FC3AAA32BFF6A9
                               914F6C86210D2A69392F131837CDD0B9
                  xor-digest = 7F6E7CC68CDC24342DA46EA89F73655C
                               7B340CBA9B3163F2C0ECE9B3F1FE5B92
                               C5B899EB9EF06885B9C386BB09D67439
                               7BFF03EF055DCEEF0E04272D05D64462

Set 1, vector#126:
                         key = 00000000000000000000000000000002
                          IV = 0000000000000000
               stream[0..63] = 8AD3F25EB62A13EBCA454419818DC668
                               DD1E67ED28144FB5D869D88065B4EA6B
                               1ECB51AC8166E5DA2760FC09398DBD4B
                               6AB526A41618F6D3376BF8495A8B28C3
            stream[192..255] = 89F80B154220C573D31A58B7E5ACB9E3
                               9D0DB61A8A99364D1706D12F32212084
                               6A8A1B2A28AB75EC74C65A278666361C
                               96D428D95C7B385B08EA53C6CD136378
            stream[256..319] = 5DDD72ECD0CBBAD3370CF2BBFE5F3486
                               2DF34F6B53344BE7CAC5A64B92E916A0
                               9FEBA6BB9EBA6E8DEAB0BD72B0EE08A3
                               832C1C1134F8B77138171D488C9B9562
            stream[448..511] = B15C38CF4356E5E993C77B0877A4E2E1
                               DDCE158DD44A0FE98AF874477FB75823
                               DBD6FA48B0B178FC8948D4A819A85EF3
                               5DF3C32D96D5B1AAC90B10331695E529
                  xor-digest = BC73346C9CBF137AB25ECC242182AE96
                               111327FE917C2857DDA56422E336D147
                               48DCBAE5144CEAE8BEF1DA044650D094
                               D0F3ED83E88A54D115CFE57DA314B086

Test vectors -- set 2
=====================

Set 2, vector#  0:
                         key = 00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = CFDC41F0C6E0A38ABF71D5C7E8EC7B5D
                               5EB266F8F0A55401426D5D636A68C1AD
                               F02CED40350A1EDCBCFCC342AB7C4D68
                               E08D9CE2D03FFFFDF4C7F5B95FDF95A7
            stream[192..255] = 125FCFAFCFEFD21E7200AD213FE0A8E5
                               F629ACF0596FFAABEAB52C6547681581
                               FA49E7483C20EC31D16594BA20C74355
                               626EFC6CAC741FA180D28FD90FF6F610
            stream[256..319] = F4C9678918B0FE1DB92675517AC3BFBF
                               F360E034BFB6700675C80F4D3FE46DB9
                               09A9D7C40BB74698B74FDC3D3ADAC1F4
                               DD4D215EF36D87E2F046D6ABEF43E2AE
            stream[448..511] = 68E4DE43893B3F0ED0799EFDE4EE3662
                               B928B1381F9054F04C85A0CDEF4C50B4
                               C0B23134E3F5C00319468DB8ECB686E5
                               606F0F43261F458ABC03253FD8BA579F
                  xor-digest = 3E0BC0CBC0BFC458D7874B8DB954B7D2
                               3A009D4A511C549C8A2F850B880148D5
                               09B4FEBF89CB72B121A5BA29AEF26A38
                               FCED66F609C256C0D2D51DBAB63BE529

Set 2, vector#  9:
                         key = 09090909090909090909090909090909
                          IV = 0000000000000000
               stream[0..63] = 78E11FC333DEDE8876EE681EDB8373B2
                               08ED4460F39A1CF5B5DECC2D89F942AA
                               2D7748D56BBB0284128E12B4B2888E25
                               E12100C14B8590A9D17F87F42EBDA7F1
            stream[192..255] = E1A648215B15C4A0712F51DF26A0828B
                               F4C8559F509508401437BE136061223A
                               C04EF0033A32EA04F81F4731FAAE8B39
                               AF59DA5DE48E78FD410DE743F908DDAC
            stream[256..319] = 21481C65E5B9922516A30871EA9E00D7
                               43E862755C6C611DE736A07B65BEDD96
                               5A453327A2D71A6347B992DEF0E27C9B
                               8F98F0E5E02037282722602B79F1F970
            stream[448..511] = DC98915A2905BBC0B03C49961B2472DB
                               F4F2BD1879C340229A715B15B93711CA
                               503711A1CCE36A645055107C7EEA06B3
                               02E87EFDBD839038B159C24FA913BA58
                  xor-digest = 4C01A7AA0E65B9587C66E6DBF89862C7
                               6C34B4668EA699BB3CE010BB74F67080
                               1A4CAD15528B9B74B2CE3D8DAF2B167C
                               F36AC635D70FB5C4E4BBFE211780284F

Set 2, vector# 18:
                         key = 12121212121212121212121212121212
                          IV = 0000000000000000
               stream[0..63] = BBA326041B9CFEC1A32290A1030A30C2
                               DD5544534E8774E9D868E96C747417B6
                               DF2E18AF0905E85C6A0B3062FA3B2532
                               F4A3D27A71FCF77403C2D40A7F44922E
            stream[192..255] = C55D288330D41E2349402F110CA8ADC8
                               734481DD5893A738A3F2DECA42A387CB
                               394F009B7E50B76A489FF1389FC9254D
                               4B64F14FFDE2412A85346D6A4F3701A3
            stream[256..319] = 17D9537A8A1E1D2E4FE9F9209C4B122C
                               151069BBA738BB678F5CC51E270D9B39
                               A053B794FAF9A8DFF1B1B2660481CEC5
                               010750E8379A000C50BDFD13BACEED75
            stream[448..511] = B4B0A6EFDD9E35EA6CD719B1D81BDF1A
                               523AFD162B48B6F8D939C83747AD1BDB
                               223040FD2547CDFF4FDA76D227F61A16
                               E2258CBC6D33FABACED488256C4DF4E1
                  xor-digest = D95DF94005DF0ECAD93B60039688E21C
                               972CA6201334BCBA69B86FE0508A7AC5
                               B817615F15C55B0493C9FD337FF95501
                               FDF14A8658584B08A0A976D96A696F03

Set 2, vector# 27:
                         key = 1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B
                          IV = 0000000000000000
               stream[0..63] = A67474611DF551FF9F24E125E6C27562
                               82B01CD9AB06CCBAA4D0FB7B2ECEDB59
                               4146F1510358522E00A69705FEF2926F
                               0444E91A5779BED68827046E808A12AF
            stream[192..255] = B4188E97925DA2B95680E28FB1B68455
                               96E487401BF361CDB05874E0BFF6F9E8
                               15D174EB06EFFBCF8E3E6C9FB5B06E9A
                               594A9D54AA2E375618405872DB4949B9
            stream[256..319] = 16D4E31B6AE838770B257A563F5A0233
                               BB6EBCCA2671C68D43205AA09D888EE6
                               7DC3FD52E94D2D631FA8F55B13042519
                               D5734FC59D5CEE55769B90D9DD14204A
            stream[448..511] = 1ACB44EBE39B2D6F2C7EFD92D7DB9BC4
                               DE18B3BC57DD3737FD737449FA14BEDE
                               278185A6EF7B5F92D8827EC804F63647
                               CE71BB252A212C9B6EA4705F50D69D67
                  xor-digest = 9C2EAD3D8559E5735135A8EE945911E1
                               7824F544118F8E1C69B6EB94C3ED9879
                               93094AC359CE20A832442E7672669BC0
                               522B97B1C0992F5F0B14F5E9B84D4E7F

Set 2, vector# 36:
                         key = 24242424242424242424242424242424
                          IV = 0000000000000000
               stream[0..63] = 07485149A5680F497E2B0FCDEB37890D
                               696D6D3ADD14610A8D0063C96C0BE7C3
                               32179FB7F0E7A8ACA9EF1C3CEC347FF7
                               C3EAC020B6FCE95C7CDD205880432C85
            stream[192..255] = BD33D0A581FD95E057D4BF8B82683F34
                               5A232D288E6E3ED0088E391E1BC24FBC
                               8ABE1841F0EAD8A32FC68870D7892158
                               C80E675A49240117905691481E713520
            stream[256..319] = 70675ECBF6E95EC74A7999D62CEA9FF8
                               A079B448CF276289E3A720B849014201
                               B06A0C83AF4FE5535D43C9FF3EBCFF9D
                               BFA932EAC3C6C42B3D3287781B2955F8
            stream[448..511] = 75A8D363EE0FE056C7632CC121CFCDAA
                               085D30E061566AB70892608393A90EF6
                               79B3BDEF79914EF744D41F50F1EEDAB0
                               83E23CD6CEFAEBBF650DE1943F2EB89B
                  xor-digest = 9D4DAC3967121D02AF3F4F658199F3D4
                               821D40DEEBA10F6CC1972D2BE20FC741
                               C2A893BC156A25E8B7595C13BE35B472
                               E19466FD8B4E054A4C3E6C012BDC7165

Set 2, vector# 45:
                         key = 2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D
                          IV = 0000000000000000
               stream[0..63] = F5F56F11F0E654DC661D77353889C1E1
                               F64D45C7745C165C4B5E3994A21FB1E9
                               52C2D4491A9F4076E6578FFE21D60588
                               410E37E09D1595E955F06C6C550752F7
            stream[192..255] = BAF3991C3D88A6A1264FDBF18D35F055
                               76D93B4BB1067E90036081B45E5B604A
                               35EDFF80D5222362B65AC7346B93B9A1
                               E923BE69B784618B457F2ECDFFC1A24C
            stream[256..319] = 62522C45283A43947B4E179A87465BFB
                               FEEACE0BA736E175B6F0AE30E95AB265
                               8289EBB6C93B441AAEDE3044BDF6271C
                               A51BE90C0B366EC9F0808C52D2433191
            stream[448..511] = F006B16753B092F44D18CABA9D9EF113
                               BB4E0F1A552907A9205EC9A2CE2DE3C5
                               02D44D70B9984A1BBDE7A3084D7E8B51
                               1662C21BC2E1B93AEFE5F3A8B2E4CFEF
                  xor-digest = 6DDBD964F04BEA00294B4B81CA366120
                               4F1EAFFAA1CBA953D61CF2B8F87BA419
                               13492CE33929CDC7D746E114EEC875E4
                               C9CE2B5D9A1B68D421C50C9109EE974D

Set 2, vector# 54:
                         key = 36363636363636363636363636363636
                          IV = 0000000000000000
               stream[0..63] = 8CD04921BDCA3F23F8FAD91B887A7D87
                               1A0AC2830B84A310BA35215F5BDC41E5
                               C8222B777638607108C8EC423B91A268
                               DCC0FEF10BD67426C423035B09469760
            stream[192..255] = 2E4425343560FD982EE40636EB4BD657
                               84172BB7469D8D529906D8062E8F6955
                               3FCC22C04E756520DE02C80EA808406E
                               EF00625BE1A8A6B9DC55D02EF7534478
            stream[256..319] = 95CA583B48204E4DC734781280F82AEA
                               88978EC1EF561EA6B327F025258C1853
                               459ECB320FA0E44EB635599CF50D229C
                               1508164EED9D683D9B7A25EB65FABDED
            stream[448..511] = F9796C4F5AA7C0E16C1F4E14AE990A00
                               F9E0B4E681A562059EE61ECE757B7097
                               72BB50F5E90F8FD489E92048439F1597
                               2F432157EEBF095B0C377CD78151D73E
                  xor-digest = 154249E978966D38CE78D87A4DDA23DE
                               F6F51645F74E72962753F5C51646989B
                               0C291C5E6E5304F12A7C2DA590863A3C
                               52394653C7DC9C34629D5284BCF7CACD

Set 2, vector# 63:
                         key = 3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F
                          IV = 0000000000000000
               stream[0..63] = EF17465AE4BCECABC45429A601825C9F
                               5AAEBE3CFFC8CB91AD0E439ACFC042F2
                               7979FB2D806628E3A149EA114297E410
                               1F944CBF47D5AA51763130864EFEE9A4
            stream[192..255] = 2F4B8F726D54A2B638B55CBCFAF96E49
                               39752375B569995B0C0A09C03C6A9D12
                               5641D4957C383EE72EB4A250161ACC85
                               B8AF5BD72940F782FE8DE3BAF82FBA0C
            stream[256..319] = 55150E658F9DB8790B58ACA1A486CA99
                               D8E3D44D8359A9A4919F3EC159E84369
                               CF0C09276457EE9E908068671DA65279
                               FB6B8998FE5EF599FA23C20DBAA19A0E
            stream[448..511] = 24217553E5014979ED1A9D6DC757B87F
                               7A5F47E9BD84A8C9E2147B71C7C31164
                               DEB57BBCE06CFF64F7D087C649071EDC
                               C8B9FF28368C232F8929A153ECCCE773
                  xor-digest = 88E7A828484DB7C9FB22B22466E46990
                               0EB5A324B1194941477077C0F3058D5F
                               40B08B3982A2C217DFCE23CB0B01EB59
                               8C6D6E94D8AAA6E70F40BA0E59B2AF5B

Set 2, vector# 72:
                         key = 48484848484848484848484848484848
                          IV = 0000000000000000
               stream[0..63] = 07AF4CE0E37D80814FDB58F4A4CB725D
                               51F509A2D1BC7ABF0BFFDE47230F3E40
                               5BB6CE0FB4BDA1998B4178219BA43E34
                               DC5BED1ECE0271EB8542429C007762C5
            stream[192..255] = 7319CE489DD0E275D66CFF965E582E9B
                               A43A08990B12AA403658177FD4857247
                               644C166C87DDCA701C75CAAE020C7E1B
                               98D4A6F02970CB133014149A055AF6FD
            stream[256..319] = 7E7FC4F3EA3D98FDB2FA4D421BF5A402
                               9CF4AD859A621F655684CEFB5C951001
                               DDE136362147D4A4029C43DE793BA72D
                               B73CD0B4EC87C3FD97D8B3BC86C2ED09
            stream[448..511] = 0F53F6308F5DD651B0B8799EE0819585
                               91869FFEEA3C8751E61968B478AEAF94
                               C3A82AB036EF78F63A9910B7F78FEF10
                               4FFF8C30B1AB9FFC3D23A511F4296AAF
                  xor-digest = 2EC84A79CE10C0BF63229351B7BB1548
                               7E040F5CEC2880BBA84FA3C0DE507858
                               15FC12DB6C2033CED1C40F6BC09A5472
                               B1E63064DDB1CDD12142B8BB15B83127

Set 2, vector# 81:
                         key = 51515151515151515151515151515151
                          IV = 0000000000000000
               stream[0..63] = E9B6F6953BBFC93BA8527F3C90F4A36C
                               162AF8F9EADCB6662A984BA69559A5FE
                               136D88EB6259ED699A25EBF8F7E908F3
                               B09563AE35295D26538EACA4D40343E6
            stream[192..255] = FB5EB4AF07DB5014900AA0D4EDD6BAED
                               D5E9447A2D66A657B21AD8AB6EAAA06C
                               F01F8DF775ABAD4B73FF45FF57CA2823
                               15B2EEE8B59E98DB67488CED22AA7374
            stream[256..319] = 86C9BD502AE78CDC27C4D5A72C1637A1
                               DDEF6CF46607C3259659F05083D21D08
                               5BD9C33273B31C15D412490373C51C9B
                               F71768963DC5AA7886124AC28D83FBC6
            stream[448..511] = 90472D1D60C576E7B53B6F6F4FDA9AE3
                               DF9FA89C2C01843E717A6AC7CE85CDE4
                               2E7531370F601CAFBB00660706DF8A87
                               946F73B5693F703B7889EDEB292356A8
                  xor-digest = BBB2BA188E8A42A843388DAB06CEE4B5
                               1FF3D58931BAE3A853FA4D22CA3ECF3E
                               456369E532CEA76F9803A8598D18FC58
                               D4AC2CBDCB50029A597F535A9C6DB76F

Set 2, vector# 90:
                         key = 5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A
                          IV = 0000000000000000
               stream[0..63] = 090A2830022A66BC98C337E73263E83B
                               97F09EF9F7D801393AB5C4B16F3E1154
                               ED58C98E1A12DE95805231A1828DD036
                               18FEC59D4A73C2E71A8C272BADB9E432
            stream[192..255] = 6757AA638AAB9314CFBC0A6631672C7E
                               F3B8502B0777E573C812BCB0B38F83D5
                               E96C7843F87B7DD4E87868387A6504DD
                               21FE428973B49A20BECDB7BBC9EB7782
            stream[256..319] = 3E25B9B889414A14CCBEBB467C074EBB
                               A25E57D9EABDC284E079CC4BED1955CE
                               CD3B7F5DD241EADFA000BF2076293CD9
                               D14100E1FB749B68CFE3F705459253F2
            stream[448..511] = 31D93F583064053A0D73B8784030D409
                               7BDC2DC54EA9844A14AE13485FA561F4
                               E2133740D25528C2E5F6960A835EE1AA
                               C17A963C65B5CECECC63506F4914FE5B
                  xor-digest = F8F89B47AFAAE2B64C2DC0E7CF5591B4
                               341EFEFB8B12F4968A5E43612D56A503
                               D304A7E0DC244135668D9D06EFF5D49C
                               7E884B93CAD1B24D0E76E0A9DA31D7EC

Set 2, vector# 99:
                         key = 63636363636363636363636363636363
                          IV = 0000000000000000
               stream[0..63] = 56CB640AFD649199033BC5F5213F5CF5
                               D029C3FF9C108BEA0F9812C4ED7CD367
                               738728A7773AB2E79F252CCEBB66626A
                               98DB1C5EF6DC0A03A5A51EB26452E01D
            stream[192..255] = EB58F1A715ECD2743457D11876C6A05D
                               01330C012B483F4A6DE472A9420315B5
                               613DF16B06750C362ED46FB33131429E
                               5BEF0F9E92D43FA46B91E611B04F273F
            stream[256..319] = 986F4FB7887A3DA93CA5DF4E26FC5B2A
                               9828A6CA43852C008B479FB080254F65
                               9953244C0DE15446A6A3D74363FFDF18
                               F932DCEA517BA8678A9753F79A54B25B
            stream[448..511] = D4A5B242B38CBDF3E39D99F02182EC99
                               5CCE5D4716A12095413F24D6C4AE1CE9
                               99F40A71EFAB7DFB1130368036E6C009
                               1FF73C2DB7AF67B6E2D8A82B07A78DA0
                  xor-digest = AB7CDB3F14EF7EE7D4D98ECA28C19705
                               502B87B91890383918EA770F7FB232D8
                               8CAFA481F580279079E4E6376DC40E02
                               244755D532352D91FE09E0CA49F998EE

Set 2, vector#108:
                         key = 6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C
                          IV = 0000000000000000
               stream[0..63] = 22325FFBA854334AC36E293F3FFA4790
                               90757449B65D37A140E8727D2937564D
                               DC9FFE9A122D69D2E364632B27F1A136
                               AC905B207B2FB4E396A065327DDA7284
            stream[192..255] = 513B6B80AF1693CC9947EA2C1A20948F
                               45EDE955BF99B71C64EB21639528A983
                               D01B0692188C2A716E3994839EE44CCA
                               6ABFF9ADAB28C074FA17DBEE92E91AA8
            stream[256..319] = 00E0E88C9632CAE45AF8266907F507D4
                               830C08942BCEB75E92D83E1A9CA3C490
                               102FBCA07A87FB83211A222A1719F688
                               A8DC901F2D054A87932764BCD4678AE6
            stream[448..511] = AB4DB8C9E2A238D11037B0E0E046D890
                               37A2E6CF41DBFD5EB18BDDA0BF60BC54
                               3E4032BF1E9539868DEF2F0E835760B5
                               DED276D09C816BADED16E4D3E25035BF
                  xor-digest = 8E31A459C7345502419C785ADFACDB28
                               CBD5E7D67311ADB5A419299A3E643C66
                               7CB6BC826CE470268A387BB99BD495CF
                               4E9214C8513A0FA261AD9AEC2CF0269C

Set 2, vector#117:
                         key = 75757575757575757575757575757575
                          IV = 0000000000000000
               stream[0..63] = 5A140B690718155C243AB78BB1C4FE46
                               2E48CC460D52A407620BE5688D61982E
                               BA3AD5C6132F9D8A5D2DB8E8A7CD8477
                               CB59FDC4BAD5BC02B7EAA617E978682E
            stream[192..255] = 1A4EBAE065228FF74F64975E7AC14B88
                               D0CA9E8FF9F32293D43B657D189E709F
                               A814A1C43C2949BF7985F54173912629
                               0D150C136072A48AD488C1BBFBA09400
            stream[256..319] = 871E8ED16860413E5361175321EB5FC2
                               A7122526E79661C462C5C6CAA0B3562B
                               C1C23D398158B588F417F83B1A2829A4
                               73A6F401AC07ABEEE41CCFE923639580
            stream[448..511] = D0C9289AA1F09D5DC47955D848FE23E4
                               F173C07E894A6533D408F7267BF570D1
                               CCB9EB35C0A72C92B8361A580308ECBB
                               166A4D6E29DE9C207090F5C06BFE3054
                  xor-digest = 2CC07920652888ED82437792FB545529
                               B9AF5A5ADC3251AD1145766A89CD443D
                               C5BFD34F67A0AC444616A22D6A88F510
                               F33B380960FCE450604628CD07C9B53D

Set 2, vector#126:
                         key = 7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
                          IV = 0000000000000000
               stream[0..63] = 92355DCB30659BA4CE97C0000A318B34
                               BFC122972C5900D042988ED842B14EE6
                               63F9A951B176D9F857356470C67A2186
                               BE30157E0B0E9C76CBA5411031912C5F
            stream[192..255] = 0014E93A6C01562523146116F7AE909E
                               F18C2C216E27ECDC50E283818FC6E83F
                               391F0B5A2603D40786E51797BC1EC921
                               1BB56EB9A289DE0050150B6BB0ADB4B1
            stream[256..319] = E93675C0783FEC49445EBA356562EDB5
                               3D0075FEA42682FFBFC33BBE399D2DCF
                               1E863A0C481D63F84F51A2ED70EE5BFC
                               0A5988770B90B48A322C087F1B53284A
            stream[448..511] = F50C246D24C26C1226A9BC4C37AB3497
                               A103F9739EDC1B79FDCD572E46051297
                               D822B79829B8A6D9A533E1A3A917AC0D
                               BAFF4AA6910DEF689C5697601C1127BF
                  xor-digest = 73BDCA61DF56027E6C4B6A066B4D0D2D
                               D898DF673AF48DBEB74EDC2DA78F1243
                               9E5637D4B7219AB6ABF25121422E615E
                               DF9C29033E9BD06E7E57F01D4A4004EA

Set 2, vector#135:
                         key = 87878787878787878787878787878787
                          IV = 0000000000000000
               stream[0..63] = 1836D71DA5F984BDCD28C0DCEF2B2DD8
                               5C162E6D7568E0279B5A3FA073776A29
                               CC74D4EA54E12283CF4D218CB36CBE23
                               B5BACE976859E9F6B20A19358A2B9020
            stream[192..255] = 56F9FF4761A66D40C13AC9C09C3F5369
                               06AFE4A6741EBF75639C90D8D173D98D
                               2C1F020266F12051D834E1F9BE880877
                               9A0F81ECBC83BC299572F4D7102E0393
            stream[256..319] = 626438E5C504DA221FD56B2D6F224C1D
                               F3B13EF20995114BCC47C19E0E919AD7
                               BF2F267831CF73F7FD0BA8E9BE495563
                               68A4BAE0E08910BBFDD4C227FEF1BBFA
            stream[448..511] = 57D2B968B4D8575D9789EA408261DAB7
                               825D8128BDAEF7882C0829CEA94053C4
                               61708FE529E81D59EBBB3E6DCE7B4F21
                               79168E2BA8C6CBC28088C9A3A5A88810
                  xor-digest = D18F00D6039DF1F9563A22BFD123B4F8
                               430C3DF7BDB3E21BF19F743E727C00B2
                               A2DBF47A72456EF16C9EC210B8EE3030
                               BA62FD8A6C5BB0CAFED5C88BE291CAFA

Set 2, vector#144:
                         key = 90909090909090909090909090909090
                          IV = 0000000000000000
               stream[0..63] = 7BDA07E149DE0ED3528770E9BC05B97E
                               CC31477057E4ECECA2DBAA3D770989FB
                               9AA552202DB95E06EFEC88225EA51A91
                               945192C98058E5C3071B9E04AA93E765
            stream[192..255] = E70DAF641EDA601F418FB511FBD30D8E
                               4E39E8C11925137655ACF13C5657C462
                               2B30C28A5D209D0FA21EAAEB1E509158
                               223E7110CB5FE16587830914C87EC990
            stream[256..319] = 50AF6EC0E3E832B3B845D88002A83FA0
                               4AB42F9354D8B932E7AF68FAE37AFB7D
                               25D390334F04AFA43A02EA933A2EF083
                               EFC36025B6EEDD198502EB161E54CA32
            stream[448..511] = 0EE92778B862BF6BD646364B9B670442
                               FF454BF6E513D667300AC32D05C14EF6
                               4CC0E7EAF43D2972EFA4E0F0DD4299AC
                               73FE564FF212B8D38A5AF8635E693F2B
                  xor-digest = BCD8F69336D939197FD1D40F83E768B5
                               E0E6B124F8F69BCCAA9AC34BBAF88A74
                               048E782B9B3C59BCDC0CD987045BEEE0
                               062A9982AB32ACFE71EC8D07634D75DA

Set 2, vector#153:
                         key = 99999999999999999999999999999999
                          IV = 0000000000000000
               stream[0..63] = 3E5395C73A8771815996D38445725E37
                               8BA4FA0F217950549ED0C70A8934ADCF
                               28363587BE499CEF032F9D5072B125F2
                               B279219A3A6836BBE51D4ACEBB9CA50A
            stream[192..255] = 1AFC20CFD4C85E04F4ECF65BF2CB2B7C
                               1DE6F6BC923CB47B242CB38DA6BBC431
                               7E2F1AC7E7D71379AF42E871A732BCC0
                               C02C7197724F665A8E08179A9367E661
            stream[256..319] = 4D6A999B2B52769F382E1C37AC4EE7F2
                               A630C192F5D4C2E377DC4B35634EE220
                               AEE8203C53C7D35E4AD7EDA3687C6592
                               AE24B5B31AAD317F99311F1D17B21639
            stream[448..511] = 4FF3D17D6D8E3C253568E85C5B9DADBC
                               F3B2541F0F3B4C837F019F9A6087ADDF
                               13C4EAB99831089A81B00EBDBC25916A
                               E56C5E63413304AB18A3B0BCB422AB44
                  xor-digest = 46DF2DCC246F54C20AB9C20F6BCFE53C
                               FF0B7EDE2ED3988543AB88FA5C169D91
                               8727A95C300BA603565AB96E4AF8748D
                               5B8BB2EA6D8C327898CD94CA74AB013E

Set 2, vector#162:
                         key = A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2
                          IV = 0000000000000000
               stream[0..63] = 698FC3DCF4490045D3858D66159A29A9
                               48C8B7C5A86D5F81919411E24033DDEA
                               B6DA0E028046DA1B56F5521EDD9F3B30
                               93C44DE2A84BA29E14F003E8AE721E15
            stream[192..255] = 123B07C4B259B26B3CE84194A5CB68C2
                               81CF2F18BF9FBCC4BFD60AB8B285B6FE
                               90BC756978306B90EF8B3EAE8E1963F4
                               DC74543FE6D24EAD29701202CA04C578
            stream[256..319] = C70CCE37139ECF6B0C6256E12DA250BA
                               FA1A58BE59A423D51A5575FA3BAE800D
                               27AFEF402D995141D78532090CE2E855
                               ACFED3F73E6A3FC0A8569097CA44826A
            stream[448..511] = 714976777161E25428A71AE897355E3E
                               F8693BA4522AA532EB8557DE01052107
                               1849D965E9ABEA2690A0A3EA9BF5E07E
                               6BAE6A0C620DE22929F8F4AD1C65BE4D
                  xor-digest = 5184DDFF6D12941A54ECB94C990530CD
                               F447DB1563A07854D1C3F5754CF979F4
                               AB482F0BA75D46354B2237EC56981F1C
                               30CDFF39A0E85DC4CA4656129AA1EFA7

Set 2, vector#171:
                         key = ABABABABABABABABABABABABABABABAB
                          IV = 0000000000000000
               stream[0..63] = CED3F5C0F21359C84503F03B20EB7415
                               8032DF39BDEBF273182F60CD40D076E9
                               1D185E07094643CAB4D2968AF1DD0D75
                               5F14A49E2624B99E59C6FE8D9D301075
            stream[192..255] = C8AF7AA4D4AA225D9B50063045EF29BB
                               5972451CDF44E110C05DF5ACC4CB102C
                               6629311A8DFA96B7FF7CD0B9D97C0F1C
                               64691DF1FFB56F3F0F77E743FA921854
            stream[256..319] = 6729BF84E64DC1B4E4F67A567BF0712F
                               203573A4AA52A9B8BBBC8EBC221D2E2E
                               01F9C8C77527A095040C1B02B7FBDB3E
                               8641A63BB159BF37DCBFC99D147D8F22
            stream[448..511] = 1E5F1AF1D8E0CE41B72370BD5592D405
                               77B5F8BF547B8C4153D8777BFB471B74
                               2920736F9940773B93644F09101CEE92
                               6857833E0AF52948AE726A90E166B6EC
                  xor-digest = 55CA85DCC67AE7102FC6483000DA0189
                               93BE79A09AF73A48AAD518BC2FA6F95E
                               AE0C5C28D8F8B58853294F2FD6FFE797
                               C63320FF330352D1DAE289749BC0E7EA

Set 2, vector#180:
                         key = B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4
                          IV = 0000000000000000
               stream[0..63] = A733DD08E72D49905440CDA035153793
                               84CB32168AF827AA9FDAE868B0612141
                               8C26AFD518BFC63F82980B34844BB1BA
                               30E9D143D03BD0BCE0A3D7E2D0D4BC20
            stream[192..255] = 990548311CBBF2CE0BF25BDC82ACE9BA
                               E6D04A7515ED451708EC34152B793B33
                               F1A3CDB646DD858A03FD9E952D2B69B8
                               346DAD426DDBBAC1D39FA5BBE067116F
            stream[256..319] = 8CC25A51F176D8D88760D665CE035CB9
                               3FE67495A345187A5C28A7209380D8C5
                               75566CC37BEAA30FB21F32775EE83E26
                               1B6FA2A2BF7BBED19C9AEE7D648C6AC9
            stream[448..511] = AFFDA4749ACFEC636FB3B527EF291881
                               A505B423AA10F6EBFBB4E81284C151BA
                               7E4737D11E26DAC0CAAEA6EA5386503D
                               8EB475BB2FF9D5B76F2210AA0404ACC3
                  xor-digest = BB360A15B6B5F3324BA690E9B0658A82
                               E64B8B16F90ACD37C3727128E67DBDEA
                               BDF2DF0BFD90445766826FF2AADFB89B
                               5D6162E2C098219ED9E3302C3E9FAF5F

Set 2, vector#189:
                         key = BDBDBDBDBDBDBDBDBDBDBDBDBDBDBDBD
                          IV = 0000000000000000
               stream[0..63] = 6DAE32A5EE068D0F26BBEC67C4CCAC6D
                               B82107D3FCE239BECFE1E0B31EBE55C1
                               8076847832A79981EB9F9DA2D45053BD
                               E8DAC204F5987EE59D35207A9C0F4A01
            stream[192..255] = 2A71B1E8BA23B8E9ED529E2445F9D377
                               0EE967C4DA449BDDE88214D6A115A12E
                               17DA6C2C87471391DF46476BA797CA0D
                               4E72F207C5849C36B137FA016F6C27E2
            stream[256..319] = 8498927FF1D3F7B51832A2ACD6406E51
                               3F91006A65055778184B0F79D14B9537
                               8B1B39F96364B120346F55E6C52EED26
                               BE12F69870751907422DBDAFE228D476
            stream[448..511] = DDAE18BF1161775715361BF759BB6DD3
                               8D9FC537EF66E6591C129A28E73630EC
                               94F08CD67A94DE53A0518CE315C635A9
                               130D49A199EB2E685826F9BE763E226E
                  xor-digest = DD01E7648076CBB9ABB66CE9622EF58D
                               E0E4AA898803DA9A3B87DC621314EC5F
                               A564748BCA218C2A799115F23D7D889D
                               8E883B4D2FACACF089197E6C437162A9

Set 2, vector#198:
                         key = C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6
                          IV = 0000000000000000
               stream[0..63] = 90523A4272FAFD7D63A349EF9BDDA08E
                               0B1B5B8493C051EC417F7508AC57FF0C
                               39027D265C87EA3E1E2F0F691A5D90B7
                               60AFA074390D23A379D867F18DA1431D
            stream[192..255] = F03A5024C1A61D4D35180DBFF29A9246
                               6DA994292A426335DFF23881AB290BCF
                               7DAAC704DD8086BC8A9E9C835BFB4F29
                               917B6CFFB52A6CC4BCA15C706DC9FAC2
            stream[256..319] = 65053C82497CD7126554895C526020F1
                               2CB2A03F0206D1A246F5B3C23DD64FB6
                               0A527F1CE688AC11FB41326728EEFAA0
                               DB82F621F3D244C5A4B70AA627FE4E5C
            stream[448..511] = 0F063672A95738C77169F5B9B3E0BE21
                               F1209CECA99C899B557F6156837A3687
                               D832968CE46266A8AC60266F6EC1F572
                               B47F0669E1C84BEF15DE2A851F0E42BB
                  xor-digest = A16DBFD8B2ED4A5B592F49D411B9FB3C
                               E4DDF597B7A11640B7397285565E7B6F
                               105E647216FF93D3F5C7B2A6EFAD03F7
                               13B57F779FE4D5860C381BEAEBE2E3B1

Set 2, vector#207:
                         key = CFCFCFCFCFCFCFCFCFCFCFCFCFCFCFCF
                          IV = 0000000000000000
               stream[0..63] = C6422EA2BB2E44F2F5BC28FF3A59B6E3
                               F9C7971D24A3E1A59E88E68633665C07
                               BCE6F10C099C48F6D6520F173D94C86D
                               81258DE671A2AC773B20020B15BAD7E8
            stream[192..255] = 214646D229BA58D95DDAC142F6CDB99A
                               834A189CCC8996DB8AFA25FF668B42CE
                               2DE6B64D3523C7F674C05345896552DB
                               283459FA7EA86C0F267122E53636D3D2
            stream[256..319] = 4D0D760CE1736F4A19143BD862CB1A1B
                               E521422F5BB73FCF1BC223DD970F79B6
                               9F118FB78D2AF8B8D3BF0649CFCA97D2
                               94390BBC50CFCD057F6F48221499DE02
            stream[448..511] = 24AE0B32E2915E5733292D002DEB9254
                               46B0C9825E1F05DFC774FA3F5BFA798D
                               0DF3A079BA658C82B16EB9D6146DB6A0
                               69D3F5225D67E538E9CB586633AB3466
                  xor-digest = 12C4822CBDD4FC6E3B254710E61305F6
                               3D0439CE25E562EADA27C673BC159667
                               D23CC93D63D12C4CCAE1BC8A67FAEB0B
                               33A5C9562CD6B379838EC76B35205335

Set 2, vector#216:
                         key = D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8
                          IV = 0000000000000000
               stream[0..63] = 0FAD07F9ACF7CE73F195ACBB9036F170
                               57D2A214AA34FABC55034DF2D80032A3
                               36538F0A0DEFEAB1E5EE9F01483222B9
                               235547A47AE3E4D55C99B6E1AF4D9828
            stream[192..255] = 1E794F647101D673B5495569540C6CDF
                               426AFD912228D4DC98F8F4B34085F1E6
                               251571ED570EC849DE31872183E51780
                               99B94E5B51A0E2A62D41773001E444C7
            stream[256..319] = 0456B4207545FFFE19693DCF92DE4FA2
                               2FFD79E6D60B2ACE85396391890CF3C1
                               078D4B1C4108AD821E136EC5E656208A
                               8BEE5039B1A0E351A8B0C73CF040099A
            stream[448..511] = 357B312597FD4B1A9C597CB78FFC9F41
                               3C57F20DBC1656FA949DCFC95B57CCDE
                               9720DB8A5A19E238D3274D4D2CACBD2C
                               4F181F94C5BFCCF023F28829822D7FDD
                  xor-digest = A5AB00EB947A992530AD77BC736E51EE
                               7949A3F645D08616F65A433FD42D485B
                               AF9A30DAC2822686768D4227B93B0D68
                               A1C684A1761FCA32474B7BC87F36B664

Set 2, vector#225:
                         key = E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1
                          IV = 0000000000000000
               stream[0..63] = B47B5F9794E772CB733FD145F0E65883
                               1B2ADDA3394AC9B8A37B7368BA7E94D7
                               A5C9DB4BE302836814B3E01C9B49B7CF
                               3A120E68F8A9A2BD307EE3CDC1264FBF
            stream[192..255] = B15453A0BE82D96BD0FB63683290BE26
                               2DFF1C96C9672CF1494EE943D2040256
                               5378883F84D41278F08C0B1605EAA42B
                               9C370521C95B2677BDCF7FE47E88F8CD
            stream[256..319] = 1F8E2C661E15BCC6A7C9E8EF4137C182
                               41529236699F5A03C6AD731F0B820136
                               9D3431BC8CFEA7670907072E44FD663A
                               5B1A65F11D83041E1352243FCDC856DF
            stream[448..511] = 4FDE385E29E397A276B68758FCF15428
                               80C62027E6EF206039DB0CC3831F148B
                               31328FD9F31E24EF1AF9E365E6DA1A50
                               A691E6184C3B2358174B7FB1686A2816
                  xor-digest = F861FB77202C5D0A6E4DE015C3188837
                               24F1FCEB54802972C6DF03EF33B9F20A
                               16DB369016A81B179F970E372BED0823
                               B5AD8D0B66479D324035C9508D98D6A6

Set 2, vector#234:
                         key = EAEAEAEAEAEAEAEAEAEAEAEAEAEAEAEA
                          IV = 0000000000000000
               stream[0..63] = 14D023DCA3A16C64C523346E71F08822
                               A8D29492E2AA88A6033724345A534CD1
                               8A5555494F474CA47B56E65A7349B1B3
                               236A08830CFCA250ACFDE016DC6FDCAB
            stream[192..255] = 935E39469AC41E7F9C9F38890B7FE56B
                               D471E5E88823B04F4D65209E956996FC
                               E7D89EEFE41D5B178EBC3BCB94D24C53
                               8D71B5C25CC3548F89AD3DE84C723C51
            stream[256..319] = 2F05505EBFF80CA353E519BCD9AAB89B
                               A461FF960DD26F3EF59C5C800F34FEEE
                               059F438379E5C0878DC304D37484DF67
                               AFC5170D8B4375A023754071B6FB9ACE
            stream[448..511] = BCCFE516A16CF8530E7E7D891E925C3A
                               D7D00B3059830BF13B7F9BFEFD321957
                               30EEB2D6D14814987A4C5FDC6F130E73
                               26F32BFEA36EB32E2843A5D9FA70877C
                  xor-digest = F30A8774895983AE8E6657A9DF55C153
                               9B3F70860006E69435F05ABB1E574453
                               4A4C0DFA798CE6C08F1EDD84CCEC99B2
                               33CF9BF06885ACBBDB65811D0401F072

Set 2, vector#243:
                         key = F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3
                          IV = 0000000000000000
               stream[0..63] = 1B0765B22A19DEA0F745011D1FEFE28B
                               CFA0F16AB591AD3F96B9D6C03B5B1ADD
                               C0CAC604BA1F87A010D329017C496A63
                               6A5BF061CA194676FD49DD4FD710730A
            stream[192..255] = 3D94352577027AC320D58A080CF0F9EE
                               2C8FA013816C35E0D7D661178B41E12A
                               978848552F3BD9DF31442B162D427523
                               7273E82EB844732A3163E8C4864877D2
            stream[256..319] = A2402233F166780E6F587CD2CB290BCC
                               8A18AA4FCB1D198B9F72411805D3A697
                               5C970266A197AFC3C13DE9F336677CA2
                               6E6C1CC5BE9B6547ED8EB40A8C87B708
            stream[448..511] = E92FC3C9419C4C5999D587491E723B85
                               01AC168981D955CEAF54FD38004C98B4
                               0F285278B394BA063A3241419FDAA5A0
                               CA3407824B02C99379642A748F12F7E2
                  xor-digest = 61F6D54699921E9A888A09B432F00F40
                               93A094E513BB9E3B7F9FA07753DB5E35
                               72C2CAB598A9A279C096E2E02A4B0E0A
                               09E03459B44C8B8CB6C7A4990D280227

Set 2, vector#252:
                         key = FCFCFCFCFCFCFCFCFCFCFCFCFCFCFCFC
                          IV = 0000000000000000
               stream[0..63] = 87BEE5606EC66CD5D591403A87238C06
                               869FDFFB82C77EC24B1154158815D585
                               0E2B5E1DF9DA83E71878A4ED5ECEB69A
                               46B902BE797DE757C11520D66A20C6D5
            stream[192..255] = 651D82DC936F745C26D99B83B2EED833
                               98C04F1DCB01545AC3A76CEAF515EF52
                               54BE684B3F53C44EE0DCD2AB051598E8
                               7FC2D7AED4FA01698A7CD8CB40FD4D03
            stream[256..319] = 5DC0691D570AE03B66F2A095273C9ED7
                               8202913DE705FED10ABBED45D49E816B
                               65869F24822F45D1D2BF039EE1A73761
                               B1B976C09CF0DF6156DDFD182C073943
            stream[448..511] = 879FCD39718AEE3AAB8F33FA3955ED06
                               0EE6184CDE7D3F77C1A963A810D1853A
                               0311AD641DAD428836387430E56B206F
                               2CDEE26378E926A721FD1BBEBE038D74
                  xor-digest = 6D93E964A2EBDAE83FA0D55678E0CCBE
                               65A4E74A57F54077D7096DA145B3F51E
                               9CD19F43E9FF8BFC3F096F601428A3B4
                               2677C1EA75FDE9055EBBE24DFB432156

Test vectors -- set 3
=====================

Set 3, vector#  0:
                         key = 000102030405060708090A0B0C0D0E0F
                          IV = 0000000000000000
               stream[0..63] = 5397BEEEB4A0CB7FDDE18F47E8ED66FE
                               65024BDC6F00571F26D4C8881436C921
                               E783F2740A538109B3A44D845D0181EA
                               D196104E9F8CC5D83E7DB2B28A233229
            stream[192..255] = 140B7C4D9338A08B75920F37AAEFC037
                               D1253A6D1BC2FCFF4AACF91AFE4A3118
                               FDCB25456CC2DA821EF79961A6597E53
                               0F5B10EA0487AA3265089FDB5E65A0C2
            stream[256..319] = E904754906189D8C083EF0E0D817F704
                               D1BB2354AFD58CE6AC883B3E12CB4AA9
                               C8D33844C91AF0F8C917282B74A234A8
                               C56BD2232423005E82F85A3A582A4CA1
            stream[448..511] = 790F3108ACD858933A80633B9E63CCEE
                               DBA2526AE050462EA70CB6385754E1BA
                               AAB5B878E78DCF78CF68DE42FD1F154F
                               93E0149D1AFC2AC92035955774781A40
                  xor-digest = D3F9A3F16223EF02826A59A2B999B043
                               477975C6C104B6CE11193AEC747AD34B
                               074EF0BDC8B90BC445E0B6D8D9151610
                               C49F98914CA49AAB06756BDCEE491A34

Set 3, vector#  9:
                         key = 090A0B0C0D0E0F101112131415161718
                          IV = 0000000000000000
               stream[0..63] = C11DF1A7CB68D2B35F7D909BDD911E90
                               7C23630F3B8BA5F17D2F0368A9CD1CDD
                               EEE0FC94F2E690BCA17B09A83E365BC5
                               65E90B12095EF276821CCAB37129B96D
            stream[192..255] = 092E78A7FE843643420B6432358A9964
                               BCC6BA1D836D89B3032EA30C8277365A
                               7C747D6B6F3EA2A2612EF474455E4712
                               0CCB78BC778B7047D3330C2741F348C8
            stream[256..319] = 2DF648A909C583A4C39387724F1D9CDB
                               DC30EA3062F277D9FABCE03F0BC1DA8A
                               3A64A46FAA23CBC06011682D19A2CAAD
                               FFCD62C1B3776CE3816FE00B567799E5
            stream[448..511] = 97CDDE0221419B30BD5C7EADE13A1534
                               000527C3A6F1D70E7C62C6B64C22812C
                               A51F86061B78E9D4465143065A52A1A7
                               29797709F59E7B50611DC0634BBC9506
                  xor-digest = 566C9BADB71180D43DA1497DB178ED64
                               55C7C2E46CC5B5DE1A13A9EC66383828
                               6A46A0A22516644A4A4FD3BCC382E42C
                               9FE46CC0A98BFA6E644C2024042A5CBC

Set 3, vector# 18:
                         key = 12131415161718191A1B1C1D1E1F2021
                          IV = 0000000000000000
               stream[0..63] = B6F01140EFD1027CF0B7515A9456D712
                               720480A171442566DB201EE1D90B4415
                               8514744D2F5EC48997DF03B33BD97547
                               DD47A312015B43C8CA6C90276E4A26A5
            stream[192..255] = 778E7146F8F68B9AF2360BAFCA60EA84
                               176770ADBB25388DA33D0D6A4C17260F
                               6E15FC1A522E5BA30306F4DB3F428D49
                               DF915E647E88858B633AFE123977407B
            stream[256..319] = 6B79DA132D3A033D6F226E6048303CF3
                               896AABA478A7987B64561BBF1432DF9F
                               06AB8B6ECA20C0AA201BFEC4714D625A
                               2868ED43B0992782E5B2C39FD8938A33
            stream[448..511] = 6B314347AA1F639881B15E7DA053FE75
                               4C755B80EBC9401BB7A2B101D2A4189D
                               DFD899A972133898B14476D67AC77BA6
                               D4FE69048F4563614FD3DDD1E0929576
                  xor-digest = 38A0C125DC33DAD39CA7BEFAC6D971E6
                               CB50A85F955D3C2319A72E12CA890104
                               48B608BEC933660850417FE2AC9660E8
                               4FEB3E22274C3FC2A32F41DAB10549FA

Set 3, vector# 27:
                         key = 1B1C1D1E1F202122232425262728292A
                          IV = 0000000000000000
               stream[0..63] = 308F29D5D939240F372A454D60C80462
                               F3C89424CFE19D4B107A77ACC3257A6C
                               68813AE7608C271C17963C09E6621A39
                               0DC565BEF20366D3F50632D3FAB0D1A1
            stream[192..255] = 1CB8C092196E37E6A6D6172E98767A02
                               3138E298C5D9D65638EA8F2232B1799C
                               2F073A4FA02332710D986257335E4FAF
                               0373E849AC4623534F0967A1737C5000
            stream[256..319] = A5A1E78C6EB0E61248C99743395C3820
                               583C5E1F508D2E5468B9042CEA9B4C58
                               946D23262A8C2A3D3F2B8416C011A365
                               1D6C99A7D70D28EFDC10CEFF480DDAFA
            stream[448..511] = ABF6709D31D5621120E04EAABE265ADC
                               FE7DB703E7B3E987B06B2D5EDD8DC1A9
                               82BE507A2FAE1538977C8C32871AEDEA
                               D7D36CDC4E256B4844515853764EB309
                  xor-digest = 53A5EF9F08DFA2BDD4436A14B7FFC7BB
                               6BA7A3A3F856977D386308C45B3DC34F
                               C91D4933A68CF8C33C81AEC28A73AC04
                               65777FC47470D80F3365135365B331D1

Set 3, vector# 36:
                         key = 2425262728292A2B2C2D2E2F30313233
                          IV = 0000000000000000
               stream[0..63] = 232C14CC6763202B97A6E67264247D56
                               DD90F0F6D2B12E8D7D1F3BABB9F10336
                               1E5FEE044E17D74DBC5F1CE17CE39F87
                               BE123779FB8893E71FD42B34764E1237
            stream[192..255] = 54B674843A1D99A877C1BF411373DAFA
                               1B4279E7E89B11412A41102519C20D2A
                               42915170C520533B8556860B05147CA0
                               B0CF956F1AC08C298292EA0A63FD7AE1
            stream[256..319] = 1207D2853AFE2F448EBC5563418C48FA
                               0F0A7C5A611296F21AFD48162680B48A
                               57C4F28D473FFF287F5405494752E9C7
                               8AFAE014173CABF6169D1974D1C7E2B9
            stream[448..511] = C0BD83D8A3E45887BA0122AA67B1BE49
                               6501F88586D85A0BFF6C056A6A6C433D
                               57A4F394D892B3024E6F73EE352390E7
                               FC5853AB30678086663E688D333A8FCA
                  xor-digest = 62E04B744F3B02B92932E79FA3809F42
                               7A24BEE058FEBCAA578D18790870DD5A
                               4C07D986688ABF151299E8D51BDF11AB
                               69D4A748CCAD9890B6B87D074E44DA75

Set 3, vector# 45:
                         key = 2D2E2F303132333435363738393A3B3C
                          IV = 0000000000000000
               stream[0..63] = 290619378BDE835D7C800D57CB80BE02
                               A70AA94B84C17D4A061DB1F805BEA33B
                               8165D188E77DB20CBE63FBDB6E6FE171
                               21F62DBA1D73ED949B51540FDF646FE2
            stream[192..255] = 9AC1524873D1F8D7F85555C652AA5691
                               9CB899F5C4DD371DD84B9DA11A1685AF
                               4C72E0C8571B207525E20ABE1DEE785A
                               C2F4E13765C79BDDB97ABCBD06568B98
            stream[256..319] = 6F84D4B6420C56909A865F02689F7B80
                               8B7CA9711EAA119A3F46AC7986177520
                               B436688948772562088F91BFE00305A8
                               40AB8387E19AB91C32B574D527DB1153
            stream[448..511] = AE37DAFC5B67547F7890262388E09A64
                               AC447CBBBBD4119C4EC3FE91FAB87982
                               448CAF21278491A723F0221BA9B2F5B2
                               608F72DD61B957144FA959982A9971F4
                  xor-digest = 535F20F8B086B82676BCCA049448194F
                               C9C840F5BED66E4C2AF62958D3912B25
                               29AD2256F05489F1AAE77601D61D087E
                               2105BBEA8C8F848037E2D4ED852D346D

Set 3, vector# 54:
                         key = 363738393A3B3C3D3E3F404142434445
                          IV = 0000000000000000
               stream[0..63] = 17A31A8BA79842C29205BCBE29FFC6CF
                               C873AFEAFE6D628C42D0D84C232312D4
                               8DD1EA98A6B60776EE0F2B7FA10E5076
                               040FB7068D5B7FF0F81D01F65139B96F
            stream[192..255] = C318C9A635DCC0FBF269B678B58DF2D4
                               EB280E7F2C07F3FFC1F1CB31C8B854C7
                               421EDFE2988A913C41B3DFE5AF320A7D
                               83F72C9DB58B9D7E8784E791EFED03C8
            stream[256..319] = AE2D8DE43CA7B6861F4935746B4714A0
                               A4C0033B6BD9CF4214BC6C0CD7BCF0E1
                               330272EB7D5D69B830E842FA67E79D1A
                               C8324B9D2F4AAE75BA447E0545F959A4
            stream[448..511] = 80460099FC4B9BAA5511E33898C58990
                               18C36B43593A48EFC2FFA3076CFAEAD0
                               F69E0AC1EC25200E6F3EC4ED1292F7F2
                               5EEB2F7121EF36939409D3EC2F1827A2
                  xor-digest = F0FA847C971FA7D0972210B44E026CEE
                               5AC1604E8646434BD52794272F1A40BF
                               73AFBC66C2D9E4A7E1B604E4015B6AA7
                               45D32792663EFF3528FD0A09A5F3A230

Set 3, vector# 63:
                         key = 3F404142434445464748494A4B4C4D4E
                          IV = 0000000000000000
               stream[0..63] = FEFBA0A5E5E771EFC9E75023AB125620
                               DE09A651FF9FFB2021DC311E834E9C92
                               2B3AD31E507D68CE031E1292F615CFFB
                               24E5363BE64C1D1C466AD91AE63BC683
            stream[192..255] = 7F65C6B6C0D9C0DDF828B260C182CA3F
                               367EDB245BD743FA63ED5FC4558D454B
                               CF34ACFDEDDAB2BDA491D8EE0DCE29FF
                               66887328AB8D0137575304F1A6707A79
            stream[256..319] = 2B1C6F22231243C268BFFA586531B8EA
                               C5F2D9AFA595FFBA49B9E1438913CA4F
                               B7A549449CC7E3DFA75EEB7842B79B58
                               93742A45D1B241499527522F0C849614
            stream[448..511] = 38E8E30A67336A042D38ECBDFFF07619
                               B001BBC717898853DF2D0731E2ED81C5
                               1EEEB4BDD3E43CBDF55A46E83B40EF15
                               CDD880A4B23BD8DECC2FC62D6FA56C0E
                  xor-digest = E1304D6D4948403A6D5E1CFB6399CF2D
                               2902AFCFE525B3765C6FB649695528DF
                               4C89C5756998C979E36FD2416A23260B
                               72CA2CDC57FFD2CD28ADD8BABE916A64

Set 3, vector# 72:
                         key = 48494A4B4C4D4E4F5051525354555657
                          IV = 0000000000000000
               stream[0..63] = FED4CEA15B954A9031F3F509D47C6898
                               CB0ABA8DFBF076F8AE2503A69C7ADB9F
                               24A7E3746FE6A2AAF3A617547A34EEA0
                               530A335CB2C930F33C84792371A3E314
            stream[192..255] = AC3F11785FF7CE0A61AF70982CE78C1D
                               627B98F373B96E296F2ED4F0C630725B
                               11258DFEBBE6A2FA44FC26AB5E7306EC
                               A80BA1815408923B61CDD46A9EE26063
            stream[256..319] = F282CAABB7D6C62EA6634E18FD0BA907
                               59F3137449BFD38FE0C936462C570D6D
                               C44E9BEDD0CD683DC9ED4E1ABFF46BBD
                               416995DDF557DE2B682052EB6748F1C9
            stream[448..511] = BD376656B4E7ADC49A72D65A1300AC49
                               66629CE0988AD268DD426EC31E84111E
                               B821E54A1F44CD8DBADD589A5C040E0A
                               91F7D0CC77E4B5DC87816F70BF46C7C4
                  xor-digest = F25436D5D0DE65BF9875EBBCED1B49B4
                               6B0A46216439D4B72FA3098B687BEAEE
                               40B4603CA5DB6DCC7DDBF2754C45050F
                               C54BA735B0715A2482267AABE7F07D15

Set 3, vector# 81:
                         key = 5152535455565758595A5B5C5D5E5F60
                          IV = 0000000000000000
               stream[0..63] = E757F6A32456DA42D1603BDC738011FD
                               087EC1036E0CDE20AC27229CD7ED69AC
                               36BB8FC65606E0C586DEB164999DDF1D
                               D99AA63893A2EA1F10050C47814EF88A
            stream[192..255] = CCF4DD2BB13307DA437BDD6E5AC52862
                               74EAB0319659E2B834939888DDB7F96A
                               147DF3C7AFC9087EA84E4521A635AF9B
                               ECF687846AE7D1DF44B1135E8DBEC166
            stream[256..319] = A027EAB3CE9C95232A204C816C995BE2
                               3AD5A733801D44CA5CEE33E6AF6E0912
                               C1EEB407C66DB2B3CE7147EA6C6388D9
                               DD5634B1AFB88F79CFB65817C65C1808
            stream[448..511] = C1B87A7F4CAC945649355EFEA3D3A4AA
                               2F9BEF5E1C84E6A3C9E4DF79BABF6730
                               04CF21F770EF973C77A6DF00847F6DCC
                               45AF6FA37921FB9CE37DC9F7324C459C
                  xor-digest = 1E550FF29096A82C80FA7320B05E9E73
                               7D094BA4333AFA55F41397E54402D8E4
                               D3613DB34B274BD11650451E5C207E9B
                               529F168ED713A7E3C3025FEFD088AFC0

Set 3, vector# 90:
                         key = 5A5B5C5D5E5F60616263646566676869
                          IV = 0000000000000000
               stream[0..63] = E417A2579AEA928836BC8AE3F0E6E7D0
                               38415B944193D447B372E1819D58E7F0
                               D9EC9AE6089E1D03F45720BD597F2A68
                               3F8E99B546C5AD9393413CAF8F278685
            stream[192..255] = 8CAE88DCC954D8BB2C82674D753CEF1F
                               8AD03FDCC8D08F5B5A5097E4656B1036
                               C334BFF216D450412956C676AF107469
                               B117112D07126DDF791E8E1DD876095E
            stream[256..319] = 2D16FB13087EADB63706D0350B54E793
                               E8BEBDEEC696F90DB557E61BF62EA2C4
                               92861701EBE0C061518FA2E6BEC521B0
                               5F9B99AC1CEDEDD763D1916366CD1D1D
            stream[448..511] = 0C09FA64D9D4F9AB15134AD7ACBCE10E
                               560E6348533A289BB5D8A6C10A5C7227
                               FCA99BC371AB9E415EEE1218D6819C28
                               7FD23DEAC3CFF6BBE78B8257DDFC7123
                  xor-digest = FE6B276F3F1C7B17B1C6BE848900F73E
                               1D91E58C89B3164B03FA05C0CFCED7E3
                               C0CC3203028DEE79375DC7CF7E46135A
                               C80E24AA5CBAAFF149957C0CB18729E0

Set 3, vector# 99:
                         key = 636465666768696A6B6C6D6E6F707172
                          IV = 0000000000000000
               stream[0..63] = FD451AA175871C17DCEE4886BBF3255E
                               00399765843F72003D42051189A30BAB
                               D94B5B42C1CEA0569279D43642C18DC6
                               E70201F31BC458A812965D3AA581CFCC
            stream[192..255] = 1EF9271B17651BDC9EBCAB10F5EC0C6F
                               B6D7A822D4AFADE2999F6421185134C6
                               EF6CF59B921CBB31E8C82ABF612AE43C
                               8BD246AF08085695E63C76369CA7C890
            stream[256..319] = A4736A0AE5CBF75D46DCA2E7E09EBE97
                               8CB15F75FA0384AC8F3E5D40F38B834C
                               9B38BF8F6CFE69CEEDF74B5C3D73694E
                               AF990AA0A6059D4F4448846E8AAEE075
            stream[448..511] = F636924F3A1875C4BA3C1D24D1C07774
                               7B23C5D41B4F0FA61F518772C6009032
                               9B8CE5B92CD25D2D6C8C27FCA670790B
                               F8B63ED9AF9741A69C38D9A5B4B45FAA
                  xor-digest = 9F5B93B948046E8CBB41F542796D78D9
                               CE87C9AFBEC1C09C690DC05722C635F4
                               EFA114D505A222BB767974DAF2D0CF50
                               BC9DF203AE9E478829F5096C7F4C29E5

Set 3, vector#108:
                         key = 6C6D6E6F707172737475767778797A7B
                          IV = 0000000000000000
               stream[0..63] = BB760E2B64BFC58620EC7F8FB3D3F42F
                               FB40CAAB7B77A42CA7D6960843B62C7F
                               5B7D17F10E0F34395ABF73B1C8118B60
                               6EDDA501B8254B8E3FE1EFE920425A08
            stream[192..255] = 8CEC2BB2B33FDA155B311E68722AE0BB
                               E19C1ABAF094F3B36A557D155E72A0C3
                               DCDA130971D1783296271A24F020EE80
                               96675F8A58304BF6BE00C5E5F4115F5D
            stream[256..319] = 9C7586AE59D9A040C6065017BF71A64B
                               B37FFD61FCB9155FD89D0E611A2BFDF8
                               777CB70D9C57994F45B36523B778E60C
                               87DFA1BA49D77F4654FB4B25E43C8447
            stream[448..511] = 3B600C1F11244D0BCE66E8D6F5448CD4
                               57965C93F6CE5DE7E8C9A9EB1446B782
                               20741679EDE1472ADF7202A6BCA58902
                               F1A6D39D7426E1014FD690CA5DAD555C
                  xor-digest = 5FC61D1A8D50DBF883B2A64DAB2A26E2
                               E00D5B1FB84C24ED4424F9D9631BBF2E
                               4719D35C9B6C1EE12705CD741AFA2A62
                               C850DB40F650FE4999AAC513400DD9C2

Set 3, vector#117:
                         key = 75767778797A7B7C7D7E7F8081828384
                          IV = 0000000000000000
               stream[0..63] = FBA447771E8832FE47A4E061CD66B547
                               45D65C4D883B4CCD1E3A9DB626B87FED
                               DF516B5B11C7ABBC0D8EC0FF36C7A271
                               6B20554BC80120E2A9BD6BEC22769920
            stream[192..255] = 30722CFCBD9C8EF9EAE6FCC09807DA6B
                               3E185657F854A2082B9ACE87CCF43067
                               FC73B62C1852F2A2C4343FBA574A1BFD
                               B7CB39296CECC9536BF57DC2DD74DC19
            stream[256..319] = 0330C8D06A9946EA98198B3FB863D55C
                               0BE0779D615E393E593F5F89952F289C
                               0CF721A68953265DB2D67E7F0BE05211
                               EB6686137F78BCFE0CD33AD065D56BA8
            stream[448..511] = BC89D5CDB46BE3726DC93FE14F10E58A
                               F9D08072FD99D3A8A3B41BAEAB2E4490
                               E0B3740624167A5DD676962A5212BA21
                               B45E2400AD4105BD9FE0C5512E9759B1
                  xor-digest = 5E8DF18A1EECA70FE1EFCE61550EAB0C
                               E35E8AF51A9EF05B80C2163567E2497D
                               DB7C93757B2A3FF3FC544EB04E45E5B0
                               C14909CBA266F568EB1EF5832D8C8C64

Set 3, vector#126:
                         key = 7E7F808182838485868788898A8B8C8D
                          IV = 0000000000000000
               stream[0..63] = CEAFF7FF7E9AF9B652ADAA7E8E3E0398
                               0DC178E5C7A5F1B376CD9600BEBB8E21
                               15A031198F084B00EBEEC2A093B1536B
                               51D3B096E9FF2F941B390085C9D0D2FD
            stream[192..255] = 47DCD0A6839807EDDC4018BE0BBD479E
                               26F5A3589C1690FC3FD77DD608ACF600
                               77B401FF00324B9A3823A46DDB16878A
                               819380DA48D55110453B4FD6362D78D0
            stream[256..319] = 93A7430A5EF8029499E0DEF2D25AF72A
                               207BDF8499072764C27A1842B560493C
                               853AF518A01ED85BA41359F91F21D22B
                               CDB6DAD533D2F2992BA4CCDC606A31B4
            stream[448..511] = E46BACA9A4E7416D00F62248DC367A42
                               6FB1044B8A6E2830A53A7F4099764353
                               05AE1C858B7C69CA4F09E4399DF02218
                               E8AAA5A6E9B739D3A8A8DF5D03C8BF60
                  xor-digest = 6E5531771D96D80CCA6B84091A42BBB9
                               4608AEE22A24B0411636A271EAE0F6A4
                               C2E4AFD1EEAAD9F09F704C2958269EA2
                               F7A27EA4AB995CA036D5017E1FCA6F54

Set 3, vector#135:
                         key = 8788898A8B8C8D8E8F90919293949596
                          IV = 0000000000000000
               stream[0..63] = 268449C44DFA8DF4281BB51BF9C4C303
                               DC0238241067DE3D7195B803FAA8562C
                               FD702D38402C24983ADB6834EB7B438A
                               1BDE40DD0A21BA1FC0FEF901FDDBBFFD
            stream[192..255] = 28E5EFCFF0190A3DE03F9933ACBAE306
                               02F0176E5FB515783974AC57F09FA521
                               68AB986848529DCC92FA305E9BD7DAE9
                               B9BD8CD923E2DA99F355942D79A87F5B
            stream[256..319] = 5F0465BA4760A1243701060FE5100E04
                               28EF0BFCB43C00395DBDA6D625E4A705
                               8842EC9E469F8D43BAE331B42A09B168
                               A54A9A356FF5044DDF233060A994FB7D
            stream[448..511] = B85CF2F2890D46A2B0F4E2E7FEBC51FD
                               D2C8848E47E9C26A15394ABFC3F27153
                               E6E303C407D65015B4A63A1D5009B37B
                               FD8962C8051D525D4A2ECF9560C0E9D6
                  xor-digest = D514DEB2D31F585A0718E8E3FF0B6021
                               414ED8ED8E6CEE56442408B60C966A78
                               0A0CC86EF6DC4CDC0F378D0DC2EC1FB4
                               D7B58CDB58C57D416C4BA886FBC2614B

Set 3, vector#144:
                         key = 909192939495969798999A9B9C9D9E9F
                          IV = 0000000000000000
               stream[0..63] = 7ADEFA66AEF35F468311136B2FA2A7F9
                               9E2E4D5B89A69BD3BE86705C55047A9A
                               0C94E03D4156BB062D317B7D7EA038BE
                               6AA2BF3F1A1727A35CE7F87892F0B909
            stream[192..255] = F473CA9BCC5E43832A8E97E506625392
                               595B03D1DE9CC004C129E0D31A11EA7F
                               74141EC9CCA8016364D5CA0B42A24A28
                               D0C2A504E7AC17C8BCD8072CEED339BF
            stream[256..319] = 9EC647DDAA52EC4F0FBCAE9763465C56
                               8033D53A96023E67B8D572FD1CB3A79C
                               F1F0B1C9F7270B1A7837ABA3403E49EF
                               F1BCC76ACBC7F010302B7C7D3EFF49D3
            stream[448..511] = 7F8A1652AE49A46389F854062BB63626
                               694E66EBC8F03695999B8B97C4E76EE2
                               68D518E4DF73AA6101079519E5969828
                               41E3B5FD119EAA2ED80A8668586307F7
                  xor-digest = 09847670910A615423CBCDE9A25DFAFA
                               40C02B1D12BCA5F7E9F7E63710714BF9
                               3C5DF0F48F62EC03E8851742B02AE212
                               8A4E9E3FDC713F5E4D33704B32057F40

Set 3, vector#153:
                         key = 999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8
                          IV = 0000000000000000
               stream[0..63] = AF55D49EF49CFDB407804BA971F63076
                               D9E2AEC8AFDAAA41D1C0408815DC0271
                               CEC76E1AA000BA7674CF2AA32E22948D
                               8815E24C41217035D095E12B86DECA56
            stream[192..255] = 2C8156890A4FD8047CA50A1CED951B0D
                               5A96A7954BB3C4F3428D645676F4AC20
                               2E4C040FF745DF0BEC3D17FDC8B2ED63
                               B4F80769B39B51F5017EE8E6A1EF235A
            stream[256..319] = 1784D88B20B7123B353CBFB779187E43
                               7F00E23E6B54473C92FB014292076341
                               6ACA7E4B5BD6C7FF76F363C3DFE1428E
                               EFCF954BA93BEAFF346426473CF9533B
            stream[448..511] = EE6726314B2D89BC084BCA41D32C2F16
                               95B0462C9B3693A5028F5CEFD4DC66EA
                               5D2715D2FB560474B5CC1C80FAAA24EF
                               5A366A3AA894C3552021C9DDDD2AA7B7
                  xor-digest = 6D4FAD71650127118BD37FAF6D39F47E
                               6C67FA165B32DA20A60E664387561BDC
                               E81CA0ED4A7F08811570D6FA51B1702B
                               EA83EB6BA9AA77CDDED6F86F5A9382F1

Set 3, vector#162:
                         key = A2A3A4A5A6A7A8A9AAABACADAEAFB0B1
                          IV = 0000000000000000
               stream[0..63] = 7C8BB3A2FF8FEF19285EE7ECF43855D5
                               59318E0BCE4A730C5804FF45DDE07E26
                               BB7004169FB82A8C298B55B12A44C180
                               7DF46B13252547F1BEC0C71CB3715925
            stream[192..255] = 2B56543BC1FFD42650023DA52E1F6D1B
                               E176FE3D7EBA7BA6202934D82A6980BD
                               B4F3C0866B9619E9374AF7FE6D75552E
                               2EFED0D5D340AE0700D79AB9CD62110E
            stream[256..319] = FD4B5865981DEF8DA85FC271EA01050E
                               1D5D93044B49953C7B4E10F69EAE6628
                               F5D242C0B18BCBFCB6638F98A5AEFDFE
                               50EFB14B1CEC4B9800A53FA79DAE7A07
            stream[448..511] = 6705AEBFBBC289D1348809339A337AA9
                               BB09F967F370F7B6F91B31340983F30C
                               7BE18FCAF8A98AD2068811086304E015
                               27631D4575A757E23603A1754CC0C05C
                  xor-digest = 3CFDE7C771319F1F8235A42910F2F272
                               A32A21FC232AA6FB19BEED7150C42115
                               11A28CA72A91130DF09D5647554C5B05
                               7E4158CA0E5EE6E620381A1886AED3AF

Set 3, vector#171:
                         key = ABACADAEAFB0B1B2B3B4B5B6B7B8B9BA
                          IV = 0000000000000000
               stream[0..63] = 0007438F1FF92182808D6C9086DAD491
                               4AD4D9A254B76C9E6DED0BBCD38A9005
                               369C5E22D9526AE58E77D1DD795CEB58
                               5AB8A3B23F3CBDAC2B5250D5AF4A4ABA
            stream[192..255] = BD0E5A42DF166C28A30133F947633BBE
                               EF6E349FB4C4CAA7735D293F4FD70E1E
                               25C7E02D84899DF7FD783676FD8E4E5F
                               8D0373CD688D9552D52A3520AF8AB4F4
            stream[256..319] = C664C363DA0A849C12E4540B1F48BB38
                               7F7470AE8D5EFF416BD56FF0F6B5B1FE
                               699E6957D7EDD362050A984E76214A36
                               B2C233AB4E1BE0C7D144A9CA8FC82C43
            stream[448..511] = B570F16AFE52CAD7829DCA4396611F2E
                               FC3D6BDE216BB164C58CE6E633E36130
                               25B2877036D43FA5A8F8FB9BD183F055
                               408C9C8BE11BF02A673C931E16EA2210
                  xor-digest = 2A17AD481BCD0B1B9E556628DD1262FA
                               641087E211857C0A4D2236DB806AD2D2
                               BE77469DB5C157C39AE8CAFC5EB11373
                               DE00DF97F944552C361E3B3EF7EEA196

Set 3, vector#180:
                         key = B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3
                          IV = 0000000000000000
               stream[0..63] = 708206F874440E673B7FB6A50F40BC4F
                               A0E5CF42665997448B3BFFB9E985F808
                               7A030A89D42944DC3982F991426A1C03
                               860D9973D02B4BB55BE333852C456AAC
            stream[192..255] = 5FE29259ACF760E97850A41F6A189F4E
                               3F7DED15364B350F0CCECC4195DFB44C
                               3CEC850471C5BA43AE9AD3C5E4E189F0
                               3AACCCDCD9B356BA78826601736DE8B5
            stream[256..319] = 2E5B18B5581B17A0DDC722214FB8242C
                               3FEED6A3E70292E66BD1AB5E6A1187DF
                               0F37F82F4CAABAD5FA07BD75C5217E65
                               A299F6592CA89439BB4C07167CDBA9C8
            stream[448..511] = F1B23A106031762A429A507CB8CD64B7
                               B8E1C83945C0C90EF76205C7A4AAA559
                               FD53770152BDEA97756DCDAF2A70B946
                               C5C2CD1303A47F878B8FBEA602F01F77
                  xor-digest = F0FDD0D60AF3A5BB7A8D9335EE4BF6AC
                               A21D197753E23C69F5B30D8557730B25
                               C49F117EBDB802BD4F54342AE8266304
                               9370254337F93A8FFB7774082A72E27F

Set 3, vector#189:
                         key = BDBEBFC0C1C2C3C4C5C6C7C8C9CACBCC
                          IV = 0000000000000000
               stream[0..63] = 8C299B9FE894439B2EC792B2F1EAAD12
                               E9F36EB166009238751CE1C0B7EDFA4D
                               415E51710DC0571BCCC7A1D4CAF8539C
                               0A0AA3772F606E0BA1A9619C0072F6D0
            stream[192..255] = 13F41122028FE0E3FDF69182A14BD84F
                               9CD8F0BD1889F7A279B6C3FC086262D1
                               BD3EFF7EB624B9CDE80E01AE0FBA7926
                               3BB96046B5781F86B1F74C770C9D1B95
            stream[256..319] = 9BF09E9F13251656327A965532A86076
                               19E1D1D53BAE1073B97F4F031CC04E3E
                               28A477EBAC545A3AF323523D0E576387
                               54D888FFB7354BBEFF8012843B37E2B0
            stream[448..511] = 1C11D6314A15362CA9E192A85590A2E2
                               CAAE7981C4D97012CE0FA647113BE9A5
                               0F085F327CC0F4B85189A156DE2F47EB
                               E306B2BC8E10635A769C78A6F01A39EA
                  xor-digest = E6B25AA23ECA3C7E605C6B481A7910F5
                               779757CAC74536EA42034E53581BF184
                               8A5A8CE0625457F76ECCDDB58033480A
                               DA5B5CE83CE32B0E87F7A942DE7FD4F9

Set 3, vector#198:
                         key = C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5
                          IV = 0000000000000000
               stream[0..63] = 6284621BFC63CAA22BE6048DE4143CB4
                               1D2196177C15063C53B3E2996D0CDFE2
                               20859E4888EFDAB46AFD3173531B70C2
                               C8DE433AF0FE22CF139E5D620FC40B1B
            stream[192..255] = BB85177A47E808FA2688FAF5405DD862
                               2337120C4D7D7A4946181D4D2575749F
                               B48B3592BE5E273718AE7BEF239CDCF2
                               EF961CB12CC7F046F855693E1CD2BFDA
            stream[256..319] = B0040C9A95D85399A61E034E71C0DFE5
                               9E358159FFF7C47D67DB866B84377830
                               30C9FADE324B21223366869F14044EA3
                               2B356E32AD60EC37C053AC5724D00219
            stream[448..511] = 15F147A8A3964C7C3D799F73F9594575
                               95C6CAEBF8FD50FCA0E92172C6F4FA82
                               7ECD666EB528EAB224B1EE1393A8D633
                               FA3F70AE0C17A934DDE048D5418052CD
                  xor-digest = 68221AEE731E380B79D1B3B173C77A4F
                               DD4E6B4178BCB73D8045D4B3EF0712D2
                               CD00058A71FCD0DC9536489F7BBF767F
                               6E78A8000BD5BC505BD1B65ADC0C1015

Set 3, vector#207:
                         key = CFD0D1D2D3D4D5D6D7D8D9DADBDCDDDE
                          IV = 0000000000000000
               stream[0..63] = C369B102AC455FCFDA25313FF1F6CEA0
                               177D195DF2F41BFD359FB0F362D060C5
                               6350FE870F674D5133CBF0F63AE1C2BF
                               A612408A0854FE859ABE703FF2688E69
            stream[192..255] = 6DD87040F10D4FD13701929FB4A5C0C4
                               EC76F16C597D797DF8F3FC7E55945F7C
                               38EC20A928BB1FAFB44334C7149BD5C0
                               0615CFD669EA306D7C7924488DEA4FD9
            stream[256..319] = 51ACAB1262C6AB00125DA521996F75CA
                               291BDEA32FE13A15EB5D5CB9EF85D7EE
                               E8C99FC23401E3BFACF9E396CE9DD53C
                               EB1333212EF859BBCE956C785C21AEA2
            stream[448..511] = 115D5394C6B246A4105FAD22439D9FD6
                               C7113BEDDEC9BCB4179C55FC15AF583A
                               C709698BCAC504FD8049D16375F8EB28
                               2D02A18B67785024615E46A34DA17152
                  xor-digest = 65C60BE6EB8FD3DC1AC7DD9B72504FEF
                               2A405791788501AEEDD12CB9A1BB43AA
                               C2F8922082390F34DDA885A88C71FEAF
                               A5DF105DE56E83848C888216F7AAC0AB

Set 3, vector#216:
                         key = D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7
                          IV = 0000000000000000
               stream[0..63] = 06483C3CDB88620B6CF0FEB0EEF4A9D3
                               475405D73363F9233345D017605536E0
                               E768E446AE16D28A8832B3BADC78F681
                               FB3FB8BDDD5BDADCAB512AD345FF5B84
            stream[192..255] = 8C4DFAEC5ECA598CEE9FA61CA1A5CB51
                               6B9504D5576109A89F413E426FEA07CB
                               02AC70D8B4747FBC24D32E32020C822D
                               D6A89B7EA54F27104778C8B7ED08025B
            stream[256..319] = 59484F76FE8CA11B9C2DE0136C9D29D1
                               DBF36CF9DD7D2F1C0D9C30CC51BB2020
                               8B8002A4424422330EB9A6B1E21320DC
                               A96E593F029C1B66D18E2DFF83817A8E
            stream[448..511] = 4A65AADE2FF6C8AD061ADBABE7856F75
                               1E342505174509362FB9121074DFA525
                               7EF8091214A1620AD23A7F3C2FB5D672
                               8AF27FA4CC0FE7C4B9645DD2E073AB82
                  xor-digest = FB7510E690653B72CE801125D97BB1F0
                               05E1B6B3A2EC937BF49DCDF894B5D73A
                               C0D3ABF2748CD8F0659FA6366356D69F
                               7AB5F795EA3AA95928EC909359056827

Set 3, vector#225:
                         key = E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0
                          IV = 0000000000000000
               stream[0..63] = 67B48A3BEF62737FC10CB6689E8BC1D9
                               41F0A93D46F475BCB63EAA68CE070C78
                               980026FDDBEBD79F5FDDA7C877058451
                               DBD497E18006D4FF15CDFC09BBA2EE27
            stream[192..255] = D92977D84D234D416D308B73E9E4C6A9
                               00F674D23191AB78739AA996C5BC1883
                               7991E006F814F4C9F5C2DC0473193819
                               A82ADB66EFAB431CB835C11E7A3FEE33
            stream[256..319] = D9AB982EC450D428031B9930C92ED65D
                               64483A3E7E172BF74E09AA3FC0EEB195
                               35B2903BFF7A4D6E2ED45C3EA4F82222
                               B4F2321B72362F811C5A56F828E7EF06
            stream[448..511] = 923A9A794AB86E2FBF9E92F4B2CB1D40
                               238181AF9F6A1B84C3D4F4EDD50A0A38
                               04EDA438F65F00D1468E4597118BD48B
                               FFABC3E58B57DA964D659C9AB0497A45
                  xor-digest = 5C37B660EFFEFF6DA251C85C144E1F9E
                               5810CAEDF8D677703C6529733D4A8054
                               E156358FEF2CD2EED6AA5A4D03A16571
                               DC46269521B79E252FB6B9E09CBC6815

Set 3, vector#234:
                         key = EAEBECEDEEEFF0F1F2F3F4F5F6F7F8F9
                          IV = 0000000000000000
               stream[0..63] = F06EF8CE77E8C41D62FE94A8E0731883
                               BA178230EDE0F58CBD806C6064B7A75E
                               222CD1A42BAA861F9C1F8829A55C5E90
                               E8B899C163260B6A8208EEA68850A41D
            stream[192..255] = B5CC125B4ED0EC30DC28CEC004FDF4DD
                               F4E22616739328E65AD5369F291DF3F1
                               D6E266E92F025C8B37D3B8975B97640B
                               58A3650A19C7B0FAAE4A0ACE5A5857EA
            stream[256..319] = 56FD201B527616699547E0B05CA3189C
                               6A725A0E10F8A2F49790A51406ED5E16
                               F651874DE023DAA11462F93BE45C04F7
                               4A2E2B1B974D49DE95B75C4A516F484B
            stream[448..511] = C0121496D93BDAE5954B054D9E3A0BF3
                               D44FEA7192706F1F243354C4FA4CCA8E
                               AF58187DA4B0A0E7684950FDAE15DD90
                               E239BDA38D09402019361E66F193CFA1
                  xor-digest = 553B309CB49013F6F1705B1C08A3399C
                               CE8DBC390580A57449EFE8494FDD8732
                               72ACB6DF1240395434B417BEE04B801A
                               92B4E958502F751B38A6DA1CDD2C0360

Set 3, vector#243:
                         key = F3F4F5F6F7F8F9FAFBFCFDFEFF000102
                          IV = 0000000000000000
               stream[0..63] = 9D3EC48369BA558E4DDC12BD341B661A
                               F6DB6660E17748C9046C3983BA9C1183
                               F91D0F3CACAF911A5E7523F257EE9785
                               3351FC2657352BC936A53B2C5E8C2C66
            stream[192..255] = EA384BD0F7DE413AE69DBB1F4F613BFF
                               29304700CE7EE8CAD5E762A3423867DB
                               AF3FA9824CA3C96784854F53D22B224E
                               26E8F246C02CD7488A930EB6E523DBEF
            stream[256..319] = 4471E3B8C67D4FA42D0D6BE905DB7D45
                               C2CAB1FCE87EE8BC24C8E45045C60268
                               80F07DB69104EB728D30725EA04B6582
                               2701660595458F3EC645C50B29A1E984
            stream[448..511] = A51766E5540CDA5078C2FC84EE4F93D3
                               3F6786E52216BDAE578BF9C41AFDBD75
                               1FC9C03E55A0C992590A24FAD0E6F3DA
                               CCB441F5673D5DC6D173B4FE19E7E1FA
                  xor-digest = B947BFBC7B03189D204602D252C20AA0
                               9A08542334C38241E3C4BC6D85CA1B3A
                               67E8FD274515E3205B53DD752EF2FE1C
                               C545E40272A5DE0E5498EA4F1F04F3FD

Set 3, vector#252:
                         key = FCFDFEFF000102030405060708090A0B
                          IV = 0000000000000000
               stream[0..63] = 83DEF7DF15EA8893F9FC22533773ACC2
                               5B1DD61ED4DD029F270D0F58FD49A622
                               490D814996416E83930CFE9B259E5941
                               1433ECA932FC575567F44928B9D6E112
            stream[192..255] = 3BF7CC3CA842E1BD6FBDFE768EE5A8DD
                               F6750C06DF275AB5DE771FE794AAB9C6
                               CA9D8D3F29F352B51CFE1108696CD44C
                               BA273A849ED031D9795754B9AB83AD3F
            stream[256..319] = 18AF34CFB1CECF1250A7D2D3F34FBD1A
                               D0EDC388B447AB65A10920888CF0A576
                               A8E213A8846F6AB5BF2986AEEA1B063A
                               3EA9150B5E75B7EB42836DF2445DB1BA
            stream[448..511] = DD3169A11FDB78BDBC353B9069589C55
                               8EF6367DCACCDE933961911FC1C74160
                               84855C8064C94F2DC852E3A82B1206DE
                               D72B14FC853F95B896560264F652A370
                  xor-digest = 6B7986D5AA8435658DF421B948D612C0
                               909ABF812F6FC11F4B14C0637DAABCCC
                               9FBD704CE6C883D16C4B5E9CF85AE809
                               1D4D699EF975BFE16B3882F3A1BF7007

Test vectors -- set 4
=====================

Set 4, vector#  0:
                         key = 0053A6F94C9FF24598EB3E91E4378ADD
                          IV = 0000000000000000
               stream[0..63] = CBCE4343A827C5B222F6A140085EB98C
                               A15A67927D6748FC49CEEE08640620C1
                               DB83BAACE7CACEC889C055D5B7B98B43
                               1C82E5E5D672AF797F1D4193AC53852D
        stream[65472..65535] = 0C1129F4FE09DC6BD32B25875581837E
                               64FB76CBDD2317F4143A34A58209315D
                               C12AA9339EF76A279F6A529BAA836018
                               492D356CF9DBE8FBF135E77B4291E702
        stream[65536..65599] = F585568D176E59918D12B6B71925F81A
                               3F76992D796B931DC526FB461ECF88C4
                               953D38670A4D000458C478F93F9CB6F4
                               313952410469D8EAED8A70EC48B7AC10
      stream[131008..131071] = 1514D5C32665795D09E990B35096415D
                               9A3DF0F61967E73FA07EA4B7CE303FEB
                               3E2C3FCC8C03D1E23293DCA0BF4DCEA6
                               439F64EDE144F13FA5AD552942D53D53
                  xor-digest = 8DF99C400C6D338F61822764553CF642
                               3BAD2A2126AC2F40D8933F7EC748C1A6
                               59BC40DD5F02E41DEDBBD1D5FF1529D5
                               E3B3E13459E65B71896BF5D89877A62B

Set 4, vector#  1:
                         key = 0558ABFE51A4F74A9DF04396E93C8FE2
                          IV = 0000000000000000
               stream[0..63] = 419A8767890D24103D47B8D0DE832200
                               D826E062C404F460E53B14379812A33E
                               2AD32D056AD1290B7F20BD83B3F78D7D
                               2F14D50145B462B197B781AF7A3DA977
        stream[65472..65535] = 2E66D8C770522E3F8033E9D978C87A6D
                               35F6D0EE60972098B3BF979ED7F3C96F
                               B9DB178035275B3D92DB05DC0A3E8C0B
                               9495B77758172590BFAEA7BA3892CE8E
        stream[65536..65599] = A009410F8C39393E3D4B06D47D8CEFBB
                               4462463C17E10A7CFAB32E4A7475B065
                               A521C6F63103C22E247F8A8334EBE5D6
                               F8B12ADCFE2190D38FD2ED92DAD940D1
      stream[131008..131071] = 564179140BCBBADA9EE3082C95043D22
                               B1D3766164756A2084243CDA0F21D36C
                               A7C2C2528479D455648CB88345FA7E1B
                               E3CF27EC28DE07B8D3E1B3570FC118B1
                  xor-digest = EC22B54FDE927C17907FAF996DE68C9D
                               56145967C06E2F771B8DD949D3BB4A67
                               9F1CC88C2D7928BCC5E47C9C1FED39F7
                               940804F455590BF51A960A405FD6AA0A

Set 4, vector#  2:
                         key = 0A5DB00356A9FC4FA2F5489BEE4194E7
                          IV = 0000000000000000
               stream[0..63] = 7CD369FFBD79D5C20325665E9B1E4FDA
                               C0B6CDA6594516B48B9FD6FA6132DD0A
                               ADF349D4361BB04247E73AEC4AC44DE9
                               7A83D5D20A53727D96A3554E7F5FDC01
        stream[65472..65535] = AF9B464B3F7D06F6E1798091FF352EF2
                               2E2D68967598200C43B095BD7FEB510D
                               139B8E91B998C727A645ECD617ABF524
                               01AA11DEFAA089CF283BD1A9D55E00E9
        stream[65536..65599] = E11F434C0E44BB64EAA8667FB84DDA50
                               88CA4B42E80EF80EA8E65A161B3C3C08
                               480678F4C377C1AE9DF1A9A8B8F7D9FB
                               D4528E222E1E5C08CC8D7C5C85E3373E
      stream[131008..131071] = 00C53CF9FB77EED33D0823F50F643B35
                               183AA9455A1A5F08DE30BB9C62F488BD
                               257477C9A08412EA768D09F3CB5B5794
                               86E8D82DC3F298AA52C54FA640B41A60
                  xor-digest = 5B703418BA6BE980986E1486763868E8
                               57CDE274AE11E86D38EFC59B47FB6FC9
                               3E74FFEA7BF1AF01678DEFBA06DE929C
                               55FE6CF7876B9C3104169464B34D6AB6

Set 4, vector#  3:
                         key = 0F62B5085BAE0154A7FA4DA0F34699EC
                          IV = 0000000000000000
               stream[0..63] = 351A653254E7F9D7C2EBB4DF3B6058C3
                               33B6A9D6AB0F0307C53B0127F08AD353
                               7C9BB64AAC8F8996F64D034FA6629D98
                               BFC7117D46E664CC129E6BDF80EF5226
        stream[65472..65535] = 301786260DCD50DEFB2629D5BA20FC03
                               429B0F519685D24944C1C9657EE53C3D
                               AC05F3C1B23C04F79A8C65FE649D79D3
                               4D66D3A9E8F50E5AA17768A318BBD5BC
        stream[65536..65599] = 7BA32D064DD260AA3C9A22C26462ED8F
                               23523F08A3CFE867CB5ECA0AD6380062
                               9857EDA7A8DEEE22CC5C8D4C9CF61E07
                               9CEE75F73F4AB2DE8B97BB3D70BD7FE0
      stream[131008..131071] = 3F210F2E77340A70DAA93BD3CD15E4C2
                               E8774623B94953D92B3C4AF39286F513
                               79052364DAA94D69FA3804E83010DDD5
                               17BC3E8E57BF2FA8AC6131EBE5628DA8
                  xor-digest = A19082EBCCEF8B398453DAFC9345C209
                               0D96AB8A2085DAD67DB21BA9E03A331A
                               7E1E3BC39A911E28E09329B3B0B5918C
                               3C1D6FBF292E0B7F280E4A92F21278DE

Test vectors -- set 5
=====================

Set 5, vector#  0:
                         key = 00000000000000000000000000000000
                          IV = 8000000000000000
               stream[0..63] = 0828399A6FEF20DA26620D2DF296A666
                               7011AEA49BD5499E538A0F720618CD5C
                               0C2D8B3F836413B73C797046F619E3F4
                               4E828FF834E60082F74BF661E115A89B
            stream[192..255] = FEDCA28213479CECAD2E73EAC5B2E3F0
                               F28279130D065D8375919CF8516139C4
                               185EC1FCC46B83FE1D97EB1631A61CA5
                               D359435EDE101422C8111C73F874D472
            stream[256..319] = 4BDD65B2652AAD22821AA6526999BC0A
                               DF4A0F4B7BCAAADBA9C2DCB96D7D8783
                               5FFD884ED00664DA9E5C31CF01C36903
                               B4EEF0D6EF5CF8D9A618BB1BC269146D
            stream[448..511] = A69DD628514FD484510E4311DE80DB69
                               216E9EE6ABCB9951ADF271AD7437C46C
                               F26064E829F3A6F766C7F2B605D9FE04
                               177E45CBBFD87F8040ED4E8990BFEBBF
                  xor-digest = D7A6A0F641AE7E16AD674CD8849B53C0
                               786A5CE177684C84CDE53382192A7A45
                               4210667BE29BAEDF8D24C0D604B4A590
                               7CAAD8742B2AD51F5B2963E85BF8D86A

Set 5, vector#  9:
                         key = 00000000000000000000000000000000
                          IV = 0040000000000000
               stream[0..63] = FD2DA0E40468588B1543EE85E09C4FCD
                               79508B7C218E036BB7F2B318A307CCE8
                               4AC24265835CA28B71E47AC8438A1B9F
                               0E27DDF55B026E89A03F4D85D761B8CC
            stream[192..255] = 0D71956B4507F2B4AB8BF13E0D119314
                               6BD49407EB382010B536F324C608526A
                               7F847BA2B566C114064DED7D3EB3C1A6
                               D7E7E01DE6AFC97281D5578D2A090AF6
            stream[256..319] = A30714284C89032C322259BC65897F9A
                               B19C5DFF3F08C7E283CC4F84D13EFFA3
                               4EC4395ABB409D8DCBE46FBEFD6AE284
                               1CDB61FF4A567781EE8FF7C497E33E87
            stream[448..511] = C81287935240C50F85F91878FF6105BB
                               3B2AE809F8ED6A81806AE10448A1110F
                               63F4D39351246746538CDCBA21E58903
                               DF2E79EBBB25C3C6D11FEEBE9EB64CEC
                  xor-digest = DF18788794641697952A30EE2872858F
                               9183BE0CA958E8C92D07D12E16E61A70
                               7C177FD3615DAC73A6BC8D4EED4F37B9
                               3B6FA6401673A18F00ACB3B5FDB26D14

Set 5, vector# 18:
                         key = 00000000000000000000000000000000
                          IV = 0000200000000000
               stream[0..63] = 84E40F5CF15FC05B7DCFA4F2FEFB4728
                               13691C2304CF5869F0FBD4B6AC1EC4E7
                               9C586206C183DCD3F46E57DF23C0D6EB
                               F5A576EBFBFD7293191D1D0ED7FEFE79
            stream[192..255] = 7333C34FE964FE554AF81B0BB46322CA
                               C89E8792F894941C9AF353ABD94AFAF8
                               D7A15AC1DFC525312888BCB732B7E1AB
                               6370D696DDC99567770CC012A48E01B8
            stream[256..319] = 19C776193BCF11896B65656D7E83B08B
                               64EFF3E047D53509887000A75F1F6FDD
                               9535105C4D257A8BD733596FDA75454F
                               D4A9D328DE452ED1056E61CF6E737C72
            stream[448..511] = 5DD14EFA6D628F675CD800F93E455394
                               C6EB5DA5D59858E72D2513CF1307E150
                               40EB0357033080C9F3ED32EFED7EA162
                               C308B0A8A87E222F095940D9E59CC65B
                  xor-digest = 1D4D00D61A6D7AF13B0F992EC8C084BC
                               0AA3D51ECEA8D16C3F61FBB00F6EAC29
                               AEA93FF7278D5E4F54FF4AA37FB26403
                               844DD750DCF9F545213F1D18B7ADE56A

Set 5, vector# 27:
                         key = 00000000000000000000000000000000
                          IV = 0000001000000000
               stream[0..63] = CE11522BFB447432E4050B040AA99DF8
                               99D6BBFC09385CB1DA154A12B999168E
                               F911951E77CF1F13FA172BF785159E54
                               9A2811E582225A9C47B4B795BC9C989A
            stream[192..255] = EED57DCF8FE2667A72E1C805808A2425
                               70034E1C984FD9A7A4AE4CC8272363DA
                               00C25244BF0C981AFDE88D023D16F054
                               930221E64DA40EE29BA0EB9CBC4379E8
            stream[256..319] = FCFAF63C16B1FDF2F35EED64C66F7C30
                               44E3D10988A1557433276F62198ADD1D
                               B4915239D582EB5D26A1C8B0E779D402
                               359AE48EF37F42B2791CCB148BAA271B
            stream[448..511] = 8926F249478DB9511A0B458F00F1C3CA
                               4920BAA6398AB226C54625E8ECC5C279
                               6F3CD50530BD48FCCE1F1C12319DFE9C
                               7985ED9039EA39BA3DBB805EE24CFBEA
                  xor-digest = BEFC1100F1D51C9C0969FE2C6D61CC7B
                               F005B6D5A6AA1599700BEB5C41388DFF
                               FFC45ED41DB2D64809E6559FB27DAA84
                               B20BF61DE2F1E587B4E2AC374563B522

Set 5, vector# 36:
                         key = 00000000000000000000000000000000
                          IV = 0000000008000000
               stream[0..63] = E250C895D66BEC675D8133DA375FB62B
                               93F0E827BC3F2FAA71FC128ACA7BAEFA
                               678E14A9B3E96557B27C420690104A36
                               246FAC96F4F0F1F01737CF73FB9F5835
            stream[192..255] = ECD7C3259F7254138CF155607665F337
                               8314BD1517311898E83F803D25FFF71B
                               30DB78068FE4441E42353D891E44C64D
                               D0506639612E3CD09F02C7B70D97B6F1
            stream[256..319] = 25851A96F3CC05B22C6896C142801673
                               C1656A2BB16B782C4056A05BAB6B4CF8
                               0027896E3523D9A2E614C0EF99C2D13A
                               1A4588F6D706235FF458E0E58A2B37A7
            stream[448..511] = 7F81A5653BEF2841C95FB32E82F812F6
                               A303795A484AB757B289B99849FC001C
                               A93E82ADF8DD1E70212AD4E337E84293
                               E402B9A226900E6E897F349020485321
                  xor-digest = 862A4A72A2B584743B18950C0352A7C2
                               641B46F8D825A7D896B07F45408A194B
                               98FFF5CF26E59CBBB18E585B306A5376
                               93481B83384987F859B2D439BF103C3C

Set 5, vector# 45:
                         key = 00000000000000000000000000000000
                          IV = 0000000000040000
               stream[0..63] = 5CFB6A5E9CEBE0A9ACC60822EB99E528
                               4CC9800F02965CC246DD9FCF30AFDCF6
                               74D5CD9B0C82F56E996D76311343DC3B
                               26B8A5A84683A814B30B64130DFC1312
            stream[192..255] = B53D6B7B2B406EE0995C482EAC7F3FC2
                               F53B8030DB140CCC349B2D838D593578
                               5EF8213934FCF97025B6E5C176BADFF6
                               7D7E32F1628211F0B367A09D360AF3E5
            stream[256..319] = 00C88495EC56E8E60377014E6CEA1F01
                               D4A242E2CEA59C926A60CB5E7B324F92
                               086121F67F6A0D054BB981A31FFD1B78
                               238271FC12A105D9C8E374E310F49FC4
            stream[448..511] = 26CD7F8B35DE4E3B93C6D4FC1CF094E6
                               B1D4FD414CE3AC1BA83693D45508A9DE
                               9E0382E3D59E746A69406C909503F1A4
                               1B5C2F09B60556FDCB7DA979304058B9
                  xor-digest = 87B7C4F5A489E29309CCF0F73EF24777
                               0A90021EF4EAF1D1946D7516E08A88D0
                               A34C630D3B9ED48B0ED21A5A13E7A89E
                               3A79E1295315B934A20B33EA0271682C

Set 5, vector# 54:
                         key = 00000000000000000000000000000000
                          IV = 0000000000000200
               stream[0..63] = 838C9C791A294AD47E1590635B78F6EC
                               C85B041EDD3E57B86C85AF3A2B6E7863
                               324DADEF9067141973FC08AD6B1244DF
                               A714B9A199258321240AF38889C2C0FD
            stream[192..255] = 6D43D906013113B0DA875025307E00AD
                               9CDB202313443A98F99650C51AECF089
                               F500C9CECEF33B2CB9D4F3FE9738F1B8
                               DE3BF7666BE7FABF3F4AB2D1D39E9D19
            stream[256..319] = 36D293409D8F671ABAFF0BD24790C13B
                               DD865D2F198836138F60ABE8AF3B110B
                               2C31DED1AD4C2CA0D72AD79C4A004ED0
                               6D230C23EA76C546EBAD6D9F0ED10BD8
            stream[448..511] = 727C90BDA82077EB3F6BFAF61EDB0C6D
                               9E3A161CA91A567BB46E8FC863D5B639
                               1A89A80865054F24EEC232BE3AADE61D
                               4D756E02902AAB6E93E133F966ACEDFA
                  xor-digest = E1043BB3428B80C1518636DB551D36BD
                               97BF077364E765560B7E21441E440865
                               8741A82729433FDCCEC8E3912770600B
                               B164D2F5AE4318189243E0F14CE5370A

Set 5, vector# 63:
                         key = 00000000000000000000000000000000
                          IV = 0000000000000001
               stream[0..63] = 92AD3F36D69DBCEE09DD51BF185FE689
                               E4D074B3182CE55DC5FB2A8E1EF3535B
                               5F24B53A86ACCC883F1E2E32775FCFD8
                               0B8C9B770C3E68972944BD37B58CC46A
            stream[192..255] = A03F61F6E9763F882689C818D2B38FCF
                               B9EBCC65DC02033743B60759B929EE8D
                               6E7778225EF67B41FB2CB94D987F25C6
                               A68AAEC278FCFFEA338F879219DEC6A5
            stream[256..319] = 98A0B25B987D40BE455E44DA2AE884F7
                               A44A385B08599D4120DB4CF2A1119861
                               581E096148EE9610211283F35C46FBA8
                               7F26CA77511C8AD8B8A2F8216692D3A4
            stream[448..511] = 96CBADFCE460FCF939648B50F2E77974
                               5734BB6F9DA81CA6D5B437AB064BA60E
                               0A764853926869D865648C7D6330E8E1
                               025E08ADC3C94F8BCE91262B1AE61D35
                  xor-digest = 5DC773662339265807BDE53BFA583DC8
                               DD97A326B95CF8208E325908C3E3CCB2
                               9D4B2BE5F248E1D50CDFD18F2E2A652F
                               96D61BB0AECEAF6578B46E560969031F

Test vectors -- set 6
=====================

Set 6, vector#  0:
                         key = 0053A6F94C9FF24598EB3E91E4378ADD
                          IV = 0D74DB42A91077DE
               stream[0..63] = AD9E60E6D2A264B89DFF9FB129C43BE7
                               AF76941B496AA3D2CD43489DB59AB424
                               491A7E48421DA3AAAFBD841E86AEADD7
                               62A08B2198FFC403D1023C90C1D5C45C
        stream[65472..65535] = FCFCEDDB8BB103AE3D0F838F16D38790
                               3345EC7EF5BFF71767116F8B12AB648B
                               8CA707BFE466D340C9047C4777FDAB3D
                               87BDAE93ACF7CE284FBA25B3426B479E
        stream[65536..65599] = 7B134CC1E9DDE0CE5B3177106DE6BDB9
                               7793A531FE5A8A1B01B5FD10649E2D71
                               09795C572456A2C3E18B0E1BF938766F
                               9944B31A178BECBD9F2191C6DD608A2A
      stream[131008..131071] = 727730F9D6F5B2C9F14849B7E7E03291
                               E83BFB478A50F8E67D0FC5C4722011BB
                               B75B76D60604734BB89F7FB2146C29B4
                               2F0949F29FA37B8E1B8E2F99E8429F9A
                  xor-digest = D5FB304D3680C469C47B3196F3B71C5D
                               EE1926EF2B9EC70AAD5EA27B314A9899
                               8F062928FC9EA1080F825D70FB99A616
                               4CA3982C4E0A7163E7B69E68DF9E2EA5

Set 6, vector#  1:
                         key = 0558ABFE51A4F74A9DF04396E93C8FE2
                          IV = 167DE44BB21980E7
               stream[0..63] = 04C8326DFDC89A24265C30C0D04B66DA
                               F4878363F561B727B6C02701BD5C1660
                               2AD9F50CB7FF0CB1886FDEF38839CA38
                               F20C87C8918A798AB99BBD3CD166FB82
        stream[65472..65535] = 99DB955CD5F1F7EFEA6AAEB7E11CB087
                               22617BA13D345B625FB56F9F5F2FAD2C
                               92EBF354107B37FA83A2761C8C156D8B
                               645A7988487DFDA074B301AC43F4D8AC
        stream[65536..65599] = D413787359E60809D3BFA094EAB05655
                               26C3B9BB416F44E60705ECE97A8CAE5D
                               81BA3CB308688E88A1F4507986EBBA88
                               11EEB3F37D9F3BB2CB863A3F44695125
      stream[131008..131071] = A857DD0BA71938837D808CE2E20D168D
                               494DD4BE691F03174974D524F8C4A465
                               253EDD5EFCE042A10B27F507D9E1EB09
                               65DDE53C8E36C09855FA98A1BFF605BE
                  xor-digest = 28F9B2682ACCB532009D0AE9FB90AE5C
                               0EAE5A875DFA1833AC6980A69A77B317
                               A86F3F673E9784A4A9C95809F531B7C7
                               49C4F5ABE1652996B4BC3FA5DF5A8429

Set 6, vector#  2:
                         key = 0A5DB00356A9FC4FA2F5489BEE4194E7
                          IV = 1F86ED54BB2289F0
               stream[0..63] = 70C2BCB290CCF3FEAEC1B40C1E76A484
                               1183DB1EC369406AA2413E68BEE065F3
                               C8FC0A70F27BE05EC3F7752369AC0A81
                               C6634867FE27B216FFD615ADF774A123
        stream[65472..65535] = CF6A14D9229A19E0D06B1F5F8F3F70A3
                               420AFAD9EB87D8D90246D57525C4A9EE
                               A98CDCE2F4571F8996188DDB5F8C683C
                               1529B214D85E49A792EC524DC49B42C1
        stream[65536..65599] = 51377F55EE34CA14F28DF83AA281F835
                               47D5CDF2645D491DDDB5D94B0336AA61
                               B4DE25FBECFDC76EE511675EE2E078EA
                               8E13AB68F5144B9E0B3A7DE1B819110B
      stream[131008..131071] = AFD6F1C39D1779585FE4DA29B48E7C84
                               C08C029D319742A039AFDF17E86B149F
                               970073741D0F4183ADE1C6FBA439A8F9
                               C5B1C3ABE04927DEA0CE6EA80BE174EB
                  xor-digest = AF57FD3AF27C4C9805D693F890366A17
                               9C07FDC915CE74084F6FACE7DCB87B43
                               0B9E30D04D609EC22EC71D37A59968A6
                               BCCD6054EB548CCF492379B625FCB68F

Set 6, vector#  3:
                         key = 0F62B5085BAE0154A7FA4DA0F34699EC
                          IV = 288FF65DC42B92F9
               stream[0..63] = D24D58B33AFCCA127AE53329E41661D3
                               F7B04DD3CE4B4B450E7BE4E95D433D60
                               FCE717B8FB74378409339779619E6451
                               811ACE2C9C7A5A307A79658574A3A357
        stream[65472..65535] = 38A7A26D4D095F59E5B1765B2C6D5256
                               C43EBF4C3DB6C34E8194B8316A275A14
                               CB03B08107FA0ACD018F7337ED75BE0A
                               081019DD0FA3EEE16B90E07755FC3C7F
        stream[65536..65599] = C6DFD83431D95832A70479E7DE00F978
                               CE963BE6CFC466FD10BED45948A802D5
                               2D4F0065F76AC633275D0404AFE331E0
                               A20FA4910132EEEBD0BEE8133EA79305
      stream[131008..131071] = 492AD6E33898DE41A6CA6763B64C8EAB
                               D15D12C71609A19618F29F4924A138CE
                               7E19AE0701140F7592FA56F05B290F86
                               A6E29E1F779C261ACB14C8D262157844
                  xor-digest = 6CCEB4175D6626620E1FE517CDA7D686
                               86AD1B678BAAB2C073112A520109806D
                               D10F458F1623BB0F4E8EA3762F283D27
                               FC613A9A8C298577E37B62B35C246920



End of test vectors

Primitive Name: Salsa20/12
==========================
Profile: SW & HW
Key size: 256 bits
IV size: 64 bits

Test vectors -- set 1
=====================

(stream is generated by encrypting 512 zero bytes)

Set 1, vector#  0:
                         key = 80000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = AFE411ED1C4E07E4D0CDE3B33E31EC19
                               0FA4CC796A58BAFB848EAD8D07D02CD2
                               D4B6F9F30CB0B57007E3733895CC8D10
                               60107975ACAEEB689B6CF614AB64A3D6
            stream[192..255] = 8966E93E875E8065AC6F3A1A3E2146F8
                               3D5EA93CA987FF9F13ED6ADE169665AE
                               3527FCA5613AF081C0E773DA6E7C74C5
                               642ECAC53FEBF15A699AC2C8255CC100
            stream[256..319] = C89DB39DD8872492ABF8109462B3639B
                               B18C64ED500B70D2836B6194D11A77AC
                               8C14DD8E1DF0B3924DDA24563E2719E2
                               635C61F63B9AE60D56D5F3512851B4B1
            stream[448..511] = 87A5191EC2E3C9049FA524CD8673E067
                               7C77ADCF8AB5328FD828C4ACB3ECCCA5
                               49ADEDA04872518ECDF874ADCB2420C7
                               BD1CCFE561B074080224FA7176F0CB5F
                  xor-digest = 61148960B846452EB2D8CD69A80ACE1F
                               F373967FFE5BA516F6755B67669B08B8
                               C036C18DA119A1579C0D1505C4CC1072
                               9F11B058AFD80D8F266DF4B1F777DF4D

Set 1, vector#  9:
                         key = 00400000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = D35370FC409EE2729711067694626557
                               E18A0C423B5EDE308CD51EF08D129737
                               44BDA1F9E10F33C7D2B0E4CC31DB1D7C
                               2A475D1A9C3ABFF48BBB7D7E901653E2
            stream[192..255] = 6087B4A538A3C730DC0AB44FA29083DA
                               E1E25529E5055AA991A91E8A91B58F9F
                               BF3502250F9C1C4CE26A0A68609AE1C2
                               7CB5099141D78ED20A0605ED5E52D102
            stream[256..319] = B8EA18F0FE7B2AC9E62564D8B5AAC4FB
                               EECC2F34AF327C6858EA61326CE5F7CD
                               29A3A493FB90F74BE8C33AAD6B3C6D77
                               F872F6A22A90662499EA89F06C467178
            stream[448..511] = 4DAFDF774BA1BF2EB56E00C8EC4D0BD2
                               E25332E76739CDDF9CD2DA5AC6A4E0E2
                               C1D08E36094C617E0A3039EA3763F8A2
                               C4C0D6D2EE0DCF62E66B74138025AF7C
                  xor-digest = 2F5DBB70C486E011FA247C063695CD44
                               A401E81F35A25CEC96C12E6349F21F81
                               6CE0265EDB03B7F464E59F03C0B6633E
                               9760855743ECA943F7E4260B52BE688C

Set 1, vector# 18:
                         key = 00002000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 2817BAA279C2E38743A8971A554E1DCA
                               A76833AFA424C3D5C4A2FE1CBECFCA3B
                               617F46AF068A542D96D7064312311CD3
                               B8685BBFA8129FDFAACD4F1320ADA997
            stream[192..255] = 74581CE56B91F4D0754DE4E24CBAAE31
                               45E6B1A2837A24C2BFB7018A2C0E6EE4
                               6EC36E29734187CC3377A04BF0F6A60B
                               6096EF5F82FE2003D7ED1632EE0FFD39
            stream[256..319] = A3BA6FEAE953CAD1D555A7515B696D45
                               598CDA3B642709FFDD8EECEC3CAB7035
                               3330E6272ACB0363544F969B1821B072
                               9BD48F724700CF59EC4AF5B0F614B0ED
            stream[448..511] = BFC43D8A25395A7336A0699A1C0269A2
                               8B073E6E16CE8F894FD15FF4F346F2CB
                               DC92F5E06C34AF7C18C51EC888FFB2AD
                               FAA438BE01B58B0D15EA9422255B809E
                  xor-digest = B473B218C02E949A076FF5680305C2A7
                               EB4224C244BF901F06EE3530B59F1A2D
                               DAEBB7AF7C23CD7C244D4791339CCA09
                               9BD301FCEC9290981A86D58535E55632

Set 1, vector# 27:
                         key = 00000010000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 0B5B3D1E653759C67FDA0A4EA40F030F
                               791B0CF5DB733DAE33F83AB056A772C0
                               D310DBBF5372909A625B7BAB4D1FC47D
                               6BC5D8427FB69A74E5CD8969D150DD84
            stream[192..255] = ED412E2D0767B19937EC5E40589F22C3
                               2B664EC630072125F79F21D0FB6C8ACA
                               B9A7878A3A0BD1373EEDBE6D2423CE8C
                               2C919F2E984F55BA9A3F8C2448FD0696
            stream[256..319] = 1BF05992C88A50A7BE08A101ACFF7CBF
                               3858AA19CB63DF58B650BCE68B55DB89
                               B513211B66FFACB56280ABDB42EE0962
                               8DB84754656AD56348F206CE5C435065
            stream[448..511] = 453D632932E54389AF9F0BF2B9181D18
                               F4EAF0724D407AC1C01EB4A03A1F6390
                               F0AF4F7C7E919BF920FC0BFB4411EC80
                               AC21462ECE37751F4B2D8B08CBBE983F
                  xor-digest = BB28BD314F2EA4E87EED413D526148FD
                               D2E42DCBED256663F87846E420282C4E
                               690FFBE14F3E9E530E15F802923DD2B4
                               57015F004764DF3F08F8CFD005904546

Set 1, vector# 36:
                         key = 00000000080000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = AACE59C86B2E5AB9DCA66FBEBC3E7E0C
                               866F3605BB4C920E926C311CF1A18FBF
                               74DFE08FEF7457B87245143BFC181DD9
                               4630570E710052C0E7A0D14886581F6D
            stream[192..255] = C589F2FA7249642EE4AD73436871FD54
                               6E3993139FDE18F5BCF310D754D66E23
                               339A9E7A33A112D3418B5987AD1C0E6D
                               3A4A785F78B0B596FC6AC5B5490CF0AD
            stream[256..319] = D88B511C4CFFC17137E017926B753B3D
                               FE5367FA3F3D38144F7AEF0792A641C1
                               698DA70D222F999A78F7AF69C14EF312
                               BE8EA870AD873E6685B7561382482749
            stream[448..511] = 87E8902715929753474278A7B53EA327
                               EA9E497C26AD48BB763A18B2C399213A
                               BD1204240B997A3A8711E395AFA6D121
                               B49402EA3E992A2698A318CC02C7F981
                  xor-digest = AAC5EB7C11DD6FF3120A804C81A06D3C
                               BFF22A0BA76725B3426BE11C5E9FC60B
                               C55035E7BF3F2732F86967A44EB2773E
                               D1FBB0CD8E82CF75305284FD46E3531E

Set 1, vector# 45:
                         key = 00000000000400000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = B0C25602B985D242368955757B27FB07
                               6EDAB23F4985A1BC4B30BCA0FE767871
                               5E68FE3FBA4DB140E555E515D0D62CF1
                               8B26A04387A87635FCC6C9A4BCF488BE
            stream[192..255] = B009F3F8EAF75703D49CA78AC924B613
                               AEC45473713CE31AFBC1FCD263927666
                               DBA064BD0A26E4D57146DA8821BB0CAA
                               5A768F9E126034D6FB9B181A5AE623AA
            stream[256..319] = D3EA08879D62D9C2C640010D4971C2B5
                               B476A9E38EAEAEF693B421BEE680D7AC
                               4EC743A9F5B3E85ECFDEC81749126A55
                               3D963CB21A77BCF954796DA3B3E5080A
            stream[448..511] = 10C6C500636A6DC4FA2101D66DB53A54
                               B6810DB73DCE979DDAC331F44F4AC10E
                               CFA79AB105F35C8A7E0F91560253EFCC
                               7AAD27787D5CF8B1475B6CA9DCE80097
                  xor-digest = 77CFED6347E72A650399430CCDE3ACD0
                               7CE45B7D5C9ECC725AA76F33913E5A56
                               ECBEF3DAB8DAC3E3FDD98D587A1A6E6A
                               71D01EC892FBA7B0B2E82A1C4FC447B0

Set 1, vector# 54:
                         key = 00000000000002000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 6A4115CBC62CF756FEEDF37FC45E0628
                               9A3B36D1CFCFEE9D586CF5FA72E6614F
                               17D572B02D55F3781CF8C28E0F0FF7FB
                               A44F67F65302184DA7EA1F50DD887710
            stream[192..255] = AD086017E3E0253DC9D0B7A4A50959E4
                               F9000D8FEC2683438159EBFC53F45E9C
                               85FA2B7CFFEEA333C031E9C44A4B37D3
                               475A10B7B07CEFC897037255DE34E115
            stream[256..319] = 256A5BDA46A30D87B505723A6C742951
                               2A7F600E12527A95A9F4BEA80A76A792
                               BBA9F59FD5BAC9DD7EABF75D04C4C432
                               2E35D9B5B54ADC5A6B917EB167A15F0B
            stream[448..511] = 37BA4A3BDBC0A3959E5E6A80F877969D
                               FC11733DAB4D1AECA5D019E0F6F3D180
                               1EE134F691E830BFC1B48D7AA2F4314A
                               BD17452CF884BD4B6C808FBEBE44A5BB
                  xor-digest = 676B9CC931C13642FDD6D27E2DB32991
                               BAB0F142C825AA1706F53DC42CB28F20
                               BF03B191813E1F6992A9859F9FCCAE0A
                               5BD89773ED2E24D8E35A5152DAFA9EED

Set 1, vector# 63:
                         key = 00000000000000010000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = BF891676CCCECD390A2E711D12235A58
                               61DEA9AA913CA081FC98B7EFCBED1580
                               93E75AA27B458AE2AE2D4515E9ACA19D
                               B0C2A195CB7C726234B4788677B16D32
            stream[192..255] = 854D77AB1E25999EF5409F10CE870B8C
                               B86939A212BE570A719FB7E7EC1FF329
                               42B29A2A299FAC22B8FB01F76C8F48A7
                               7A5D43DBCB49D4A0C637A4FC17438130
            stream[256..319] = C103EE4DFB2019D3F80E4DEBF98CF341
                               DD177B9877ADB0A11E80CCA9E22B582A
                               BB84CDDFE4579C7D32C6DD52C2BEB963
                               E4DF92AF314A9F1D2C5819792875263E
            stream[448..511] = 250B26FACCA0C4656A6F16E1A9F7A426
                               BCD0B574DD0DE37D5D2143587CBB037F
                               2CD00C3260E0BBA15E1D26C441AA0DBF
                               D0149971A933EF1623654D5436762D24
                  xor-digest = 7B2F208B0F1624CF61BED60E98BF5A0F
                               26E57806A6894CA043635F9A85D5AA11
                               0BE9CC2ADE70BA9CD47673C89FE15389
                               231EBC70AF24D11E8AA3560429EDBFC1

Set 1, vector# 72:
                         key = 00000000000000000080000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 9D1812D4D1CDBF39F0B265E8871B0690
                               062292678733DE2D7BF114B4FCACC319
                               8CE01D56EE3C3E6F943DADA2588C4F65
                               DF6D541F06E2947509F1E009DB9D0F10
            stream[192..255] = 7B884B3F3BB9F2936B66919C5993339D
                               D3E5C619C7B8F815BB131DA59A4FC7E5
                               741B337EF188C18FE1268A23F7BF83E0
                               6357E5AD512042F0FBFE2236C0EFFF31
            stream[256..319] = 59A4A95742B1FDEB81D22557B6C22B3D
                               DD3B47CB85FB15AFAE657EE5A61FC8BE
                               F3D5F082C43AAF86A76FFF9DB4D1F40A
                               F666D03EB4A516A9399C0E1D493C3A97
            stream[448..511] = 6767AD6AD7846F6DC81A65D1C4DD3A2E
                               F6709B91536504B5FB16F0620E9AB76D
                               9D528BFF107BD2BD18D06147984822B2
                               EEC70DBCAC7229817EAB2B17F6768E62
                  xor-digest = E6AA8C7631B393AF94920234D16F1C3B
                               A8EF81756FFBB64167B799C66E789D24
                               063CA052EC322A62BDF931D2B0EE6C63
                               F3144492EB24B94087E1379A45A01197

Set 1, vector# 81:
                         key = 00000000000000000000400000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 6389093BB9AB75E59C2789D97C5B37EA
                               24DD0651D7496D614D30002FCFD48030
                               3FF3D642C6B44E594D800318B5CDAEEC
                               7F2126A3B03B41FEF10B13A85E742F7C
            stream[192..255] = 8DED781C470FDCC61D18C5A728D790B0
                               0E82B0A73ADB83DCB407B910A362ABA8
                               6B6133B4D7609A261D2D7DBAC525808B
                               23BE34184AE83F19549CACFF0CEC99B2
            stream[256..319] = 70A92B03DE3F73980B0AC3DA43432909
                               D9281837B28634F4440D47BE640B0CB9
                               8302A6C68B43BEFA72100BBEC1570E6B
                               380C9F064659AF621C57BAE4ECCDAC01
            stream[448..511] = 7C535EC78A02E334A514ACDE70E5D7B0
                               53F30CA42CBC6507077F68A6D367F5A7
                               E7CBDE7BF2B200FE34EC35220C792F3C
                               116F58A4A3CF39E9555179C1569F36C5
                  xor-digest = D6956DEF953F4D1D8DCB131371977605
                               FA0C8A4C7EDF2590F0E7A1AC16FA388C
                               37A969B9183B250780976191B37F4404
                               6929110B9D1513B2CF4E2D155D172B09

Set 1, vector# 90:
                         key = 00000000000000000000002000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = D215C7511591B798595393396587ADFA
                               4F9560F29E8C51F3B78BD66E5AEAF6E6
                               FD8AA0C1F118550B0922D468E89BBEB4
                               22A0BB03D3C182713033C7E8F3584DFE
            stream[192..255] = 42A967B3958812A03DFAF28A1D3F14BD
                               C33AF3A80324A714BFC89D1F203FACE4
                               75BD46C774CF71A83D73778D314020B2
                               07CA5D6D4F519CF1342213AEBE0B1DB1
            stream[256..319] = EAAAE1629F4D0B2ABE666FE1183E9A99
                               E3BDF6AF7A7DD791850AC732CA2BBF26
                               95552C58E76A2DA0CB5716476D1DF5D5
                               F0C1A594B3A96B93ECC7D9A8049E9C7A
            stream[448..511] = 1745E471E99050ACDD50CD00C9C2391E
                               87EB23A8D5DD637F99DB10F236930415
                               7E0D203655B22CFE1C630A0FABDBB030
                               B8BD796B8FF244B1A6C31351C5FC9888
                  xor-digest = 38973BBFC3107DE4236AD0C4B3BD6E7D
                               4AA6182203C4908828636A848410E435
                               44E82C12A4E1C8025D702BB767EFE648
                               91C894CBD647BB03EF65D7BA4BC459F5

Set 1, vector# 99:
                         key = 00000000000000000000000010000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 7E11EE1FF504341A20D8D7BEBADD2251
                               E1C53177DCFAA98AD9A7A6F47F8DD769
                               B130223B7B461AF569B3D34F6A84A9F3
                               0F0B9DB1CB9EE64F29087810BAB13AFD
            stream[192..255] = 1B5AD6AD85063A7600814FFC0FA14FEF
                               ECAC68D40BFFFEA3009001D488301240
                               16CD145E7DCD887B3A547A95FFD299DB
                               7487DB06CF964FA588EB8EBF629EA778
            stream[256..319] = A8284D274B3737613835669E2A531213
                               93656BAE438D66EA41C19DAF455ABE4A
                               9908BED31D09CE5DDF3BB28E0CD87C49
                               8DCD82E0E1A7AE2827AF9B84E5CFD659
            stream[448..511] = 1539E0BDABB5524A6E75BA30D92595A2
                               9E17A16D9EA9B198D2C0BDC7084B7E82
                               2043F27575F03E5956253BD2DB7DA51B
                               3BBBB8091F8CBB2518B8026A4F863563
                  xor-digest = 01F0F87A7F91D9FBEE2A4912A010CB03
                               35E05A12CF8562D0D94FD0661CC7E79B
                               FBEE4F6F5F31732EC899C8A4F0ACF506
                               270AF836E7A82BBFDF2DC201A1772C35

Set 1, vector#108:
                         key = 00000000000000000000000000080000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 334483A8D13DCC20B7586229B4431804
                               1CFE2A919ECF0BD14094E65ED57E1B41
                               84C61F35E7B6A27B3BBFE9C5773B4D1B
                               C2B4CB1553D65CBF237DD0C80629033D
            stream[192..255] = 496FF41AAC9104377A6353F6C0C7DC0A
                               DB9EB70E55578537FE01333A0376DC36
                               C3BDBF74D912E39491B5994C38722DE0
                               DEEAF4C56F11C9223F9BA8615CAA1DBA
            stream[256..319] = EE7CDB35ED5D2297D7920352A62E4F78
                               E932EFE0280C176C99B7E458D8A689A7
                               976197522DB47EADAA45B45702E2A8B8
                               D4E86A6E242ACFA1A2DC8F116E417A7D
            stream[448..511] = F6E280593F4EEF22E989795B675118AA
                               C56155B55347207A88F1284588E75344
                               C428FF85EB49504CD261D9FFE3629015
                               21B4FC67925E48002795A9E01F358896
                  xor-digest = E1CC8FB100B628DC16A5EC988A0F3F4C
                               9E8E4C654DFC79A16ECA7DE901C66E0A
                               9F02C4B9E1364AB1D51AAB100BAEACD3
                               2CF76B8A56AF37A84AC367543670900A

Set 1, vector#117:
                         key = 00000000000000000000000000000400
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = F14213AAE4B9009E3E186711D38D2E5F
                               7F9E1582C3119FAF35E2BD769C27AF98
                               F91B1F3BE438B9C4D9E34A660C9503AF
                               1028E46991791DD5BDA0F72402E918B8
            stream[192..255] = 261FBE670CF7AC3717B109D0DCDD11BB
                               9D459DCD28170BD213B1504ECA80F682
                               0A5C00DE042F716D446321554185A067
                               F518BBF40D494EE66299FC4ACCC35C9D
            stream[256..319] = 19C6D1D61A4E25915E84155F6600FFC5
                               D9A1D3D642B867A3A2C7220F31AAEEB5
                               BC70D2B4FA31C8F33CEDDF4ACFBCC6C4
                               62BB6A20CC887332A0663A70AF3172E1
            stream[448..511] = 92A5DE3470C31234BAE98E5B7400353E
                               4D38429A3908BABA871466E2F11FFF26
                               9A0FFB444A10E492C9F12E79D2B702FE
                               DBF8BD08B0CFCD25BCD001559E475DDD
                  xor-digest = 4AC00A08ACF888E710C9A077CE363905
                               55C62CAD164AFA1653A6BD93DCC25689
                               42807CC8547FDC5B7BB923C9C9CB7525
                               F291ED7AB926A14A30CAD96DD8AF2638

Set 1, vector#126:
                         key = 00000000000000000000000000000002
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 17EDFEFE97F853B1F939DA6457560C27
                               235019D1D732D361F97A7CD2E820F8C9
                               6E4027CCE09699E9C2A3C3B9BB286718
                               2F2A032E33DA2FAE39783D4F51A830C9
            stream[192..255] = 5BC8F17A5D6EBB9A6C33DCD757891988
                               81E9D874426DF815B70189ED3BF71C4A
                               2F1D767BA5214ADAB15A40A41CEDD65F
                               B2F6DB56ADD96AB53662DEBE1A0E9D3F
            stream[256..319] = B267B2046D33190085B6C5D775C76C49
                               1976BCF86A5319DA1D47F7F92B1A3D3F
                               AEB4AD3B7ED8DA3796E27ABF9929DFE2
                               5B3CD790F5A8225554CF740A9DED9615
            stream[448..511] = D58E8620DB3317075B3C868E0403A1D3
                               6D98304037F5AA089775430118D91841
                               BEC652CD06F4EF138D2F17CEDE63D3A8
                               714DB3251830AA1C9A93E0431CF23812
                  xor-digest = 9DDFD866E937DE4EBB0CE08FC162794B
                               5FD602CA6C0F96A32E3838F5A1554C07
                               CBCB15F21000CA7E56227FA54C940B02
                               CF3F7DA0A8B366555B2712776D5859F2

Set 1, vector#135:
                         key = 00000000000000000000000000000000
                               01000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 351C49EE8DA408D2CD177722640F1259
                               E1D4A1A55A955E08F228669C6F5D70F9
                               A70DF66B1E6125034E7D95D9CD8178BE
                               A06C1F3261B213E6A13605F44A50347C
            stream[192..255] = 546F38423FB0531043A311A8D97F93F7
                               8BA2825D1FD4AF93AB9A84A12212FECF
                               BE4EF5DEBF1848BB92EC2BABE2222FB7
                               E5010E81AB1976FD4AD5CC40A804B906
            stream[256..319] = 4B6B16E92BDEDF277B5FAE0721A97066
                               CB08A4C1FDE7CB6BA864A86C11311407
                               2B83BDFF5104E7BA30E48798FA749323
                               65C4A2BB63BE9CC1DAEA27E41E492659
            stream[448..511] = BBAD9D3BF9B2AB18F53CC61B5B2BC669
                               B5D099A28AF0A0D6A767A9DF069BD93B
                               AC1C1573DC64653E4750C6B5F119607F
                               3D87F4B9608CEB7B697655442198F581
                  xor-digest = 9EC62446507887A4BAEF4AA0881C350F
                               2AD1E176D807057D6894E82BC0BC7DC6
                               E56090A3B932214181FC8628153E9AAB
                               171CF1E134C03593FBFE861AAD59D695

Set 1, vector#144:
                         key = 00000000000000000000000000000000
                               00008000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = A978B3B7AD8626992E684D011963B296
                               1C11D347FAABAE59012DF088B2158B95
                               347F9D5727D46511F797BE502AAA59FB
                               1433AE57A76DA01B9EFE2B2F18D3E1C3
            stream[192..255] = CABDE46B1A00AFFFB7960C98E75D7636
                               894E24F3DC28DB1CD7F19015AEDB83D0
                               6E9B0C3C71F875ECA8006A9E6297B909
                               E8C2AF9D9EA50B2EE4A1DB6BE26E422F
            stream[256..319] = 671B2F0ED8042E321D16FED1F4F49B6F
                               0F1BFA7A63C935A850BFE915661E4228
                               23005AB4DDF97123A5FF7E3ADF3C6CC1
                               0D35D01E004B3D0F84727E702C35DC01
            stream[448..511] = 4118ABF80B94D331328D18713BD0E175
                               1F3FC85716097AC87E5ED38C95805389
                               43BEB16E1F1DEB61E4D4E805177D29A6
                               BFDD0FE173880375CACBB75F81C8CEEA
                  xor-digest = 7FE0C63F27E1B19CA76C791B55577730
                               3056B20E05426261D7F3A45A2755FADE
                               DB9C82B7A16484200FDECFFC48DCCC56
                               2B03E3C4F0EC6049EA0FBDC10185C023

Set 1, vector#153:
                         key = 00000000000000000000000000000000
                               00000040000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 77B1F4E8565511F209D9F4E70857AC6E
                               EAE6FD4295B77111078C52428F13C4F1
                               EA3994D84A3776F022D1440D311BE28F
                               18B13F86EDBDDA1038EDC3430D99FDB6
            stream[192..255] = 752C994D76E55E4BC3F4F99FE4849D7C
                               FD43C2ED45ED7FD22A820F88FEF67D84
                               B61318A343034D99889E1A6D23363BC0
                               75052E87EAAC0FEA0110FEEA75336B6D
            stream[256..319] = 0533B286A8606720FAAB245089F872F7
                               902FD62FF24D6365DED10CC8B79352A9
                               D427D43F94E272E6601B77766BA92A6A
                               B81B0DC43F4318F2FF3FFC4BEB5F6E40
            stream[448..511] = FECF0372097707D5201D44B990ED9BDD
                               37BBC59BEEFB42A4B9C50BB34E86307E
                               08DAE67D4828397C3AEB5C61A5354738
                               E925C28A7C3586E9EDEB9F3EF2E683A5
                  xor-digest = 26BF1531988579B573AB3AE977DDCBFC
                               F0AF50970CB5B0F65CDC5CC2B6AC426E
                               E1B3CAABFDACC976DEA70C78485B00B1
                               811F4DBE4D669D3BC285C547B56B9856

Set 1, vector#162:
                         key = 00000000000000000000000000000000
                               00000000200000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 3FD280CFA9C36C58EB56F51FA6080D65
                               6A311646F7F67AE841E4B53A3034B4FD
                               FE2B02103AE3E21832980973FEE5A2B4
                               4C5106D024892EBE4173F592AE773B5A
            stream[192..255] = BB8FEB595B5F8B4AF5F5B1816ADEE0A2
                               C7811F3D6506E1F23F7A4998DB4F5957
                               BA4C4392E11040A2B09691B297DEA4B2
                               8645743BCAD424992518CE276D76F072
            stream[256..319] = D1BCDE2A3AD99412D78415A8A8A0B6C9
                               AC77B55EDC515274EFDCBCA243D905DE
                               A06DFBB0E4A6D54C32F8802E57C3570C
                               82CBBBD38684027F952239A05704B0AB
            stream[448..511] = C9AAEC8B510A7DBC4AECE67C058187F7
                               B6A0101D2BE4D3168F51CEE50FAD4A08
                               E79A3100AB7AC6144D24B3FB0EBE7327
                               25FEF2899DC2F93D1BDE45BC6EAFA01D
                  xor-digest = 1F5F5C678021795E41085C6B4C8B89E8
                               EA5C4235EC260ACB8298E7B41B4214C8
                               0D9548F729C8A3BEA148640D36469804
                               063F77575346E5C5DAF592E6C8F6EDF2

Set 1, vector#171:
                         key = 00000000000000000000000000000000
                               00000000001000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 9A9B00959B1DF85E0FDFB6D11F8041B7
                               0275DECB0DA973A3FE1350AFEBF53638
                               1B03F5EDD52C5BDBA41006EE6A2445C5
                               9202537B26B3C449563A68337751C5AB
            stream[192..255] = 472D13AECC0A21B08B5AC0BC6F78AE66
                               0A8BB9A8A0342299368AE190EA45771F
                               EA778449ECF64B5444BE97AB67B7C6C5
                               740AFD88003146EAF8DF37EA74C09270
            stream[256..319] = E5A0D8170DFEA041CFB8098C77727642
                               64C56953A3F8E38361021687AE91AF63
                               A3D68A9F13CE0616FD28328EA4A5C83F
                               7326C1F0D6E3B9117411AF8F883E75A4
            stream[448..511] = F964DDBD3742CE89648D21EA49B14348
                               5CC815C641A787E67D7489E5AE0545F0
                               A8C8534E195E47F9C0D224AD54BD72C0
                               6B677D0F7C9CF3554F09B481DFF6EDE7
                  xor-digest = 76D7257AB0DD8BCA4D3509CD097ADBA2
                               663437D30B9D9E442E4B0E8D88430FB8
                               4D462E77BD84D9F02255AACB45F31C99
                               023FAFFFCEAB7EAEB6A2875CC0C1F406

Set 1, vector#180:
                         key = 00000000000000000000000000000000
                               00000000000008000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 95658A5DC01ADE96440F57FC3131C507
                               E5BC4EFCA919FC4BC0A56DB364592260
                               8C19F042594B5C874757DFB8B0C0EF68
                               D08F22ED5E078F9E7903B269E897313B
            stream[192..255] = E120EFC63885CB2B93966C18875611D1
                               5ACBA57F51A61F723611747D8E3B0BC5
                               12213A722C12AE9FBC27943BAFF41A0C
                               89ED713C2F7EA5F37166BDABB3F9B9D3
            stream[256..319] = 449CB554BB4E7C22D0BF1E1126DEC463
                               9699459CA6308C1DBD9DEAA9EC1C3986
                               128F85787D6D69B67468B85DF5CD09CE
                               079B2C6653DAC10D3E2B45587C217488
            stream[448..511] = E193D5EE9B76BA0937190C5217A0A0CF
                               23D56DF7C3CBB238BDEAC22582F7F190
                               95B54B0EF331217BD9FA14805B68CDC8
                               DCAB90D45F5CA4D0F99B21977D4B04C3
                  xor-digest = E1E5321447B92364109C47B538D9FA18
                               0C827EE76B94C162395D25A3A684B3CD
                               C685330FC8F3850AC996EAD26BDB7C01
                               AC3F7EDDA7E45F5084EA26D8B1219185

Set 1, vector#189:
                         key = 00000000000000000000000000000000
                               00000000000000040000000000000000
                          IV = 0000000000000000
               stream[0..63] = F115DEA8B1647092400004A0577D54FF
                               8D098F83F67F425546C3F6FD92F70FBF
                               8ECEEF26037840B9E70E09A2FDE0AF4C
                               4C1826FACEDDC94CCEF1FD12AB84F0A2
            stream[192..255] = BFF169D7BB9B109E819AA8F338CCB626
                               E6C9BD82E4B9C283EA8041200B72D576
                               D8C0557BB31631F34377AB443C158B87
                               6797FFA1A434035BA37C3199A6EFC086
            stream[256..319] = 3423344BE7ED86D3A3FE5E84639BB7F3
                               C9B597CD4B335179AB6B7DA36A1736EF
                               1E2C2F266AA4DC2D82CA5B245115BEEA
                               0F0D0F0B9E8B49570446CE25755D6363
            stream[448..511] = 9620F9FFE31ADB4D73704DFFFFC44387
                               32C0974FA3461FFA8EA448F371BF90B9
                               5A790DA7A9FED8A75DF2AE65582523CA
                               A4C1C0639F85678F9C0E7FDF947C15B2
                  xor-digest = CFF2F6EEADA4C95BD7AF9DD28A6DD4DC
                               5CB3D1AEEA820E56D11A7CCF64E0C1B1
                               EC654B9C27E9F7B76F8F3E12D76E78E9
                               42B6531B05B2E359AB63C5E863515DC5

Set 1, vector#198:
                         key = 00000000000000000000000000000000
                               00000000000000000200000000000000
                          IV = 0000000000000000
               stream[0..63] = A8AAEE8B9C37DBDBAB9363CB1FD3BA04
                               941BE75883DFA5EF99E0BDC26F6A6FA5
                               312BD677F61E31DD249E3B3EFE8ED5FF
                               41C3FFAE7AEDCA129A800A3DF0E675DB
            stream[192..255] = DBB517248B189B1568F645D94AF87F81
                               1BA90D83FBFBC2DCFF33507424F8C5B5
                               C0C59E4D92B640872FC47CF92D613549
                               8153ECC945BB9E97079F06C7671F289B
            stream[256..319] = DF6D89322BF30452DBC5F11D16EB9F9B
                               DDC8B29A3B126AD076A32E14CBC6B71F
                               AB1F916D6D0FA3021A489D114614FA87
                               CFD26BCBA656AD2722AA45192AAA0888
            stream[448..511] = 0E636E94D6B530AFA6BE7483D3A9147D
                               4B0CB4E49BCF09A5F77804E27D0F5715
                               B0064ADE088EB906978B25A76E1C8583
                               80E5BFF7C35A3DA9E4B904312AD4A9E6
                  xor-digest = D74F629CE53414F34009F1DD2A6A7002
                               58B4E96AE3E6E7F5ABBD1BC77A4DE3C8
                               C715B1E536AC3486CCCD7A3153A1A917
                               6A70B0602012371730647BFA6F732C2D

Set 1, vector#207:
                         key = 00000000000000000000000000000000
                               00000000000000000001000000000000
                          IV = 0000000000000000
               stream[0..63] = AEDB2F5EF24F0EEF9E2B4C628176B6BC
                               4C24583434A114B66FA10E8BC272B4AC
                               38032EA4E99E8AE6FFD12C0B86BBFE83
                               C92BD33C8486B84B37B9CD978CE3BD38
            stream[192..255] = F29797C98D102CAABBDA2E032068FEC9
                               04F06D9A533F95BEA95CB2E77932A4EF
                               C18B2C8FB073D52CD6CF40891EDD997E
                               4C2A69EF5F35639BC2E228F23BF1D17C
            stream[256..319] = 4303E0EAEAC68EA81D0C642755C404F8
                               7ECE033ACF4B9138AF510D6C19AA7048
                               5C297BB1B934063184FC999F04A946FD
                               77657813CF4A6209FCC977A6EDC8DC81
            stream[448..511] = 960D9D1031F75897BC92449D8FC84809
                               108B084FC2069FC69A1DD5087E203221
                               D04A640A6D4E2D855DB79844C330B402
                               627A33E2CC415B822D49BA1FDF7B2663
                  xor-digest = 143732BEE18644909A29035D8AD8B127
                               E90BA86920EF5A84A53576889868D4A4
                               1A40FC53AFEBD8F92FA34B4EFCFF2B7A
                               6BB2390D0CD1D1F05E33C60B44449331

Set 1, vector#216:
                         key = 00000000000000000000000000000000
                               00000000000000000000008000000000
                          IV = 0000000000000000
               stream[0..63] = 84B3B0E73D6D7AC3A5D9766A369AF105
                               8803934EFC4E3BE6D43CA981A8D93EE0
                               D0C9FED2F483535FCC5D8BE6523C756C
                               443E4B0D9262AA7FA1EF5C516809374D
            stream[192..255] = 21722B0F64F730D9F3459413C475593D
                               90DE1304BE7A80692A8AE5B20431B749
                               FC2573B06C7ACC7173461CAB4B2869C3
                               12B45B2D6DFBBECACFA6A6EE67EB0EF1
            stream[256..319] = 3D7561C8F63E41C115C629BEE3C9CBEB
                               17B5B9856CE2AA52A1215ADEA831E50B
                               224472DD4781E398399B7A0D5C35D5A6
                               749DEA84255F7423D86905275CCF4251
            stream[448..511] = E10BDFE94C2E2F4CD60DE35B9F39A924
                               CF56A70765C00241A8AE8CB25F527CE1
                               2C4DBB4FE0AFD828C22489D7F4BC0434
                               AB3503217882E11D289A4FCC06FF549E
                  xor-digest = CCF2F481440C02E0326157D942869C70
                               942F90AA283EEF7CFA4A63B8EF8AFBD4
                               5D24EB83FC6885AD0213EE54CE208C52
                               C284F78F1A11E7C4BB899052857DEBEE

Set 1, vector#225:
                         key = 00000000000000000000000000000000
                               00000000000000000000000040000000
                          IV = 0000000000000000
               stream[0..63] = 9304017033ECE78F3789AAE6633F6275
                               EBECE0C1090C951E0F74F1976A6A47E2
                               72C6183A8B18384D3E4997F000C7DC38
                               AE0AC0AF03AE0C97964BCAA4B37EE6E4
            stream[192..255] = 03CFB50D0C1540CB800B8FB2EDA6E433
                               87131BDF1B53980A76BA82E23BBE1E02
                               D4B41598C5090DB93AA3B3428E8252BB
                               F62F44D0FB2559FB0F80EBA91F4AECA5
            stream[256..319] = B28BF06ECD6D020AF76278AA056D89C6
                               093E64C0F9CB3DD8947E4C383497CC39
                               CB17D7EAB3DC79EE50838A0F69678CC6
                               82C5E508777FCF5BE10419A8DA01DE9D
            stream[448..511] = C041CCE57D8C431409902CA93C7ACFDB
                               52C5DB935CEFFA978EF1C620D010E854
                               52251E5F5D2BE25FB983F664336C7CAC
                               ED4E9B5890CCBAFDA213F5C1D5765601
                  xor-digest = 1ABC41147D20DB5A3CEF47C67BBF2362
                               1FD6BDFE1242AE82427B97FC6003B5B1
                               141C9DC8348F0EC68A7BA72B626FCF2F
                               71E93B1540E816FED2CA637CA45492FA

Set 1, vector#234:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000200000
                          IV = 0000000000000000
               stream[0..63] = C8703AC9568B26608DB924A742B57C96
                               CD56F92F0406C24950EA4643AEB90270
                               19B389A4E430961AC5D77189187E7982
                               C1131B1DA50A0D93B128925F69AACB45
            stream[192..255] = 5FC26EB16EB4AF14A389CC9D61EBC278
                               3614B7DC6597C2C3AE81DFA3F1FE539B
                               F98EC9EF2F8155DE7ECAED71C0502776
                               B6B2DF312C1DC37F28F4D96EC5026CF9
            stream[256..319] = D023769606D9342B4D3F2E286E8B147E
                               9CC022B86F8FDE6E735CF651783DDD8C
                               F477058656B60CBA44041A048417772E
                               2F3FC737B880E2A565BBF8465729F1E2
            stream[448..511] = 96028706B0B88E2871ECFC1FAF4907D5
                               8D46469CD13AC6390190F7BE6550A45D
                               6C178421C99395901F36DE791C045B09
                               E28CA4891B54F93BB4EB9397866818DA
                  xor-digest = E439F6436C8A01AC4D3F93FFB1592FA3
                               FA59696024C0D7E46D1DC35D3E3D2BE9
                               B1E538C7E7423CB71FB5950521ABBC83
                               DC052000A6EE087CE171F8DDEDD42567

Set 1, vector#243:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000001000
                          IV = 0000000000000000
               stream[0..63] = 14D46414B0FCD93B5D38DA16353997C9
                               A17AC2D871F443775400446ACCA2A6B4
                               B4161EE20086C0177159D382F1C1FD82
                               64629B8AA17030BCC8DF5A57B2F3320F
            stream[192..255] = 6996AB9DC7D174CE8AEC33BF239858E2
                               4764CBFBE6ADCDB53DFE638EF04C80BF
                               856D6ACD0DD860C884CC5BE2C1AA0294
                               FADA6A6DB1019F651909DF6AE7D0FFD4
            stream[256..319] = 9E4FB6192A93CDF609BA07F14EA4CF1E
                               0966B1022BCF84AB9B19413408B67A8F
                               10AC8ACE77C5B23CF340271CDF120747
                               1E66449A7E4618A458FEB6C18BE00D60
            stream[448..511] = 167D1151BDB937E40C7C43651CB0F977
                               04F506B26326EEC3827F13C3884D181C
                               768A09037CE283171F16BA31FB96D7A8
                               2C7769FE78B0E9C97599DDA06CE6D1AC
                  xor-digest = A043B753B458D58BDAA1B675A132932A
                               98953F0D7DAE2C798BBB96A9A0C23E27
                               EA1DC2CC2013FF31262523752C954A4F
                               4CBFC22A7FC3D570E01BEA392510A503

Set 1, vector#252:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000008
                          IV = 0000000000000000
               stream[0..63] = 7F198D70B4B1D1634C1820DE2004B3BF
                               C5307C2498EAC95F0005CDE5EC9E1B6D
                               A514829EB2315B6B43DE24263422B601
                               9FCDDF446AB4869209948AF5D85EBC6C
            stream[192..255] = A82DE85BD15E433D9E30ECFA1037A459
                               7ABB72CC02F23E54E851FB088DFDA84B
                               F454E568C36011DF2B1AAA60B0A9704B
                               D5A5B7AD0E9AE9196914805C97FAAA9F
            stream[256..319] = 36A531D53D846DE8AC03A126337D2AAF
                               0E4078BA7B13B3CF75522A0BB1D82D98
                               672C7EBFD0F45EDBF6091C39F59324D7
                               AC1793EA30A264510F5EDBC6A656498D
            stream[448..511] = 7FD9052862D77E7577C663E457C4379E
                               D61735F39CBBB3B832204F00D9E7B617
                               378DC8C45E177931B4ADEB2DC1AC6830
                               B376B019B547C5C75671BF4183E5B22F
                  xor-digest = 958691D27F52C5C8D9B33123AA8DD942
                               10BA24B37689B3AC795EAF71CFBF329B
                               CF37CB4B90B6DBF0D434BB24D114A881
                               52DE2E2C8E3A24C1C8544E7A81323A23

Test vectors -- set 2
=====================

Set 2, vector#  0:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = BD78A2F8118A563C761DB4F2FBE055DA
                               97F90988D27594D9C5DFD13A3EFEAA3F
                               68F0D2564850ADF5017433968E4B3405
                               AC49A39532124FCD6F47E415C7028A83
            stream[192..255] = B26D54A90A4AEF7DE8B88B79455D3B04
                               D0004055E07F9C05DB26D41B62FB8BFD
                               0BAD2CCDC915BAEF1953A6BAB7BD123D
                               07B5D2589418AA12668DDB6D0BC09DC2
            stream[256..319] = 47C62DB21CAFADE41016CF4A5AA4DDC4
                               E8A6D57DED1387F24A52D5FF462D8185
                               2B137782AA194514D0DA7E1DC256D4A8
                               5EF48002B1007A61D402738B6E43D125
            stream[448..511] = BF6D97632ABCB97B3C36A9B75238EE66
                               5A9DB8E36559131F6D4AC794C2DC3AD2
                               33224485A3E9A7970A75518307DA2B12
                               9B508AD24B585BDEBBC862A081590058
                  xor-digest = 705061CAA93B6FDAD318167490426A28
                               5F290366788B9DD29524DA4F634735F4
                               3A207EADFD68AE4E79AC1FF9848AE84E
                               A0E6E6ACBE6AC760D6E57D451FF22499

Set 2, vector#  9:
                         key = 09090909090909090909090909090909
                               09090909090909090909090909090909
                          IV = 0000000000000000
               stream[0..63] = 502FD12A40DDA86F250C55F568945A27
                               BE9C24BF82368D4D7EA486ACE021871F
                               F3F9C9312F40C58FD0B416E6A2F5C66E
                               B74C6860942E39DC148A606BEABCA7EB
            stream[192..255] = 3A4B7E72E1E9013EA527DD37F6A51314
                               FE28795C8B6E6A96BC0D0D8F3D5F20CF
                               5F35123A8CFDDCCDDFD5577349BFAD70
                               A65A18A22C5016F446E4C49F26D9FB4B
            stream[256..319] = EBECDF20D74758F17B66ED18EAF26E92
                               E2B689416B3DB2A6842387F08062409D
                               6E90DFA56CDAEB88A7024A8A349799A2
                               BE3CB7E85C03BA23EA6BC9ADE64F9493
            stream[448..511] = 711D518891711EEF4CB033FE55B7F1D8
                               49DDA255F786BCBD337C90FC4B80A0C6
                               21643A6035B501DD7499658DD398F097
                               9F6B289F19875201F36D3DF51DF4B632
                  xor-digest = E15D07DB9BC223533EA9BD687FF62357
                               BBFBF568DF5287E32D0BDD2D1471BE57
                               561E938601578E9FFF91619C7C86DA91
                               722E7F1EEC48B6FF8F57B9F7ABF020F7

Set 2, vector# 18:
                         key = 12121212121212121212121212121212
                               12121212121212121212121212121212
                          IV = 0000000000000000
               stream[0..63] = C4BB00CBC2F1124644BDC37DAEBE800B
                               4271593267C0CA277C47C7B19E10D5DF
                               4DB5B6A7D2B6EDD40F1BE0FD820E71E7
                               2AFFAF5F3F7E6B406F3BC5E21895556C
            stream[192..255] = 3D481F6F1284523C87D4F464103510DE
                               2A9E47EF11D03AFCD850BF0FC377FECD
                               FA4057C44C184EB3D8F6CC12AD9A6B57
                               974142BEC7BF6E36436A7661A145F351
            stream[256..319] = A2F86C1089C3CB2632D845D8E373AAB9
                               AA98ECDB22B64F6AF63AA21439E15A56
                               30275D9AF8CD410F05077532C70ED4F9
                               C323983C39897D0993CED092C85F25A3
            stream[448..511] = 84334AE3D87C00B82CB91D453B365AFF
                               6FEF3D128B15C57B208DA43A4DE3D504
                               633A282724E5DD71519E157B33515D96
                               BF0762248933F1B1C7FA16E024FD8DB4
                  xor-digest = C510020D8CED8452F08E369A1E6BE347
                               4C4095E89059EF437FE342E0C02F17A0
                               FA087D33C3356DCA99E9E13DAD3C8919
                               92745749F426AEA36396D8AB23AF6FCE

Set 2, vector# 27:
                         key = 1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B
                               1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B
                          IV = 0000000000000000
               stream[0..63] = A1C341AE3AE9CBF35A2FFDA900DC78FB
                               519539C0F72386FD1B1ECECB73AB7C24
                               FCC7D96273C4893994F27758F9FEA16F
                               E474DC77F7E3A08810B5BF62909E5804
            stream[192..255] = B8CFCE5902A2A2F7FFA470E3D7532B0D
                               7200201BBDA9CDD5B3CAE361F05C1C6C
                               F96FD25D7073079F64FC7BB18A2592D8
                               ABE1929259CBBC2AE89BEB766FB6B465
            stream[256..319] = EF4487C6CF18093E846CB1327E38115E
                               E5087E60DE0852FB5264BF68110859A6
                               BC33F409258FB2C188092B9436E8F057
                               440663DAB906C76AEE616964CDB3DC7F
            stream[448..511] = CB6A19F6596B5753AD573F026E4F8743
                               9AE38CF26ACC2DF5634697F504FE48E4
                               6E956D41A9979EA0DAB9AFFEE397CB04
                               311B05370041B159EA4EBF12B58245A5
                  xor-digest = B0B017CD43CBBA4407B585CEBB3BADA8
                               1BE65B2A04653E3C3912FB6AD5AB0383
                               408503B86EE0802F8699C9E02E77D9FA
                               DF710E11A6B7523A7649851EFE06DEB8

Set 2, vector# 36:
                         key = 24242424242424242424242424242424
                               24242424242424242424242424242424
                          IV = 0000000000000000
               stream[0..63] = BF93B87312B2856584B1CD4036A61DA9
                               56476639F76AFC1B96FE8A5A8ABEA075
                               BB16048F0CF9917D115B442C3233DC2C
                               429D0E19B12FCD50433645EDA7FF03C3
            stream[192..255] = 2B28D6A80F35CB0C16AB7EC6375B69DC
                               7C6D66815F4298F930520D374CF6DAE2
                               11341B82490E004814C3D3FDC92E075E
                               97F987215E087EA4970938BA4FCDCF93
            stream[256..319] = D165570413A606D818F606F7BEE5B924
                               B229AA26F691B691861D8F06701BAC41
                               19B7FBEFBED06C4CEA2910B2D4DD58D1
                               CA5C3F232C23525235C22A62B6266E8E
            stream[448..511] = 095CFF8CF39798CDC4029FBF3DB62BD3
                               B8C0E3BCE1E7C91EA8B5D3322F1A4292
                               2ABDC1203335174C5D0BFD3769D9A579
                               CB118FE140469F474FD64F04B6334A6A
                  xor-digest = A71C71DCFC3ED3CAF7A3E061C7954EEA
                               CB6B6AB503C7E8E34186768A35925C52
                               40E12AA9631CE5C696FC9462DDECA3EA
                               7C30932182C1F3DAC105BC1534B4E805

Set 2, vector# 45:
                         key = 2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D
                               2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D
                          IV = 0000000000000000
               stream[0..63] = 1120733B9370D243F6F1B52BF775E95D
                               9AB2B7EA4B092D65DE2908DCA58DFADA
                               9ECA767AA96700C61B37A5940C1C2E9F
                               7E1A09179DE268EA974D269884293986
            stream[192..255] = 5C5BBA191E2FC38197B186A2C5A17694
                               B0FFEFA6A51EDC5FD5CA8D21232E9389
                               48465D4097F489F5047D6227E3BD80A3
                               FDF3EDBB7D73E8576ED1E85FC5F2C97E
            stream[256..319] = 462D51A6931892C1D9226396E6667F4B
                               86BC6803034357174F547E617F4C9A70
                               B1BD76149EB15E378205ADF4FC97683B
                               C833A7924497FD3427FC15CD92D1B082
            stream[448..511] = 8ED91B4DC9473FF9F709B0B1936D54A5
                               21130ED889BAE71483A95477E0741EB8
                               C80C04BA68585B2B9142747261B6515B
                               63E986BC80C743A3A7E67AC068D552BD
                  xor-digest = AA1D2E8DEBD092EAF725D24E58916CB7
                               AE51A821D9C02E7986D21E376B89AC21
                               C311039D3AA6AB46FE530BFEDBBC894B
                               FECFD746C32714115A7586A2E920E17F

Set 2, vector# 54:
                         key = 36363636363636363636363636363636
                               36363636363636363636363636363636
                          IV = 0000000000000000
               stream[0..63] = C601B6BC6027BA68B64393F248DE8B3E
                               3E089ECBF716EB676AD04292FAA784F9
                               BD6AD8E3A77180CA09CC46F027F437A0
                               F0C36F36B565D1D96E16E98BF772DD12
            stream[192..255] = 8137B013574D6A3B8340F3CCCBA65172
                               78BB1CB29ABE8B52205C23F75C575058
                               CDA9DC91D2B1F6078FCCD023B8F23308
                               531DDD41AA479B418D8BD2DC8D63FC06
            stream[256..319] = E7F7E1DF52B5F991B88615C70DA7F216
                               E25D4B3648654FFF43F365360A7EF29B
                               34BDC3AC1130E6785498FAC553B51638
                               8B68A801DE16025975DAE8D53863CE60
            stream[448..511] = C46BD8B37ADBEF8E11F078A5DF4C8C42
                               E8DF021D8A94D1B41137F89353DCAC05
                               CFFAEBA9E7D1618F86059C9E6C825A9F
                               B58FD32AE0C4C473C6704EB39BA8677F
                  xor-digest = 2B3DDED292619EC67AD252009C564CEA
                               FD99A959C064C3CB5ECD0FB43EB0D2F0
                               3941CF1739E10B84863D55E10E8697E6
                               D39B5557CF61A41476BB73546B8985E4

Set 2, vector# 63:
                         key = 3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F
                               3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F
                          IV = 0000000000000000
               stream[0..63] = A342F1F443E77317CC3BE8A16D1152E9
                               684D89A49D647B9C641D66D475E84BEC
                               37B190EBCBD9CFD2CDF1875ECA1E207E
                               FC63F2611C9C1D2EC709802DFF4B092E
            stream[192..255] = 496C4D445B61EEBEA7F3178A16DF39FE
                               FB683B65D725605508156A747DBCEC76
                               8CEC9EE154F498B83B669B5CDED64972
                               D561AD5B380CC935C66B95CCD6931B41
            stream[256..319] = 5D52BB16C1AB53C6B39136B5ECD0FF0E
                               D37CFF1CD7B795B8C6DC2FC6D11E8D7D
                               053EEFCA3F4EAA49B68684FE584125AB
                               0D26B970A06E3B0934F37AEB59A1D274
            stream[448..511] = 589012527D6CFE88F487F503657F35F2
                               7CB688ED762381B10A96184E381963E2
                               5F4172B28F6BC975364D416706966D20
                               8E1EB12F461086D43C6F8566CD00F1DE
                  xor-digest = 20B395380C965521DBBE6E58A9628D7A
                               BF86355DCB8343C8AE843AE661A23F33
                               55B38DE2F02878B08DC9EF9ADCB25D70
                               8D3F247AFC165C7DE17ABCB3F8AD6E5C

Set 2, vector# 72:
                         key = 48484848484848484848484848484848
                               48484848484848484848484848484848
                          IV = 0000000000000000
               stream[0..63] = AAB04C19C70C17749B9BE1E07255A5EB
                               05EFF0E71D832D3A6BA0329318B3E283
                               53472B0D558886792BC5836A7E265825
                               8F3A4F3787C52A9526177D37F8F08C4F
            stream[192..255] = E1EBE11AFD3B7627C46A77A197920F1C
                               3D2C5DDBBF65A0438DA3F83EEE372323
                               5BEA04DF7E08504BA7FF34A940C1BDE3
                               89062B9F02E7E74E5175751B7A57F548
            stream[256..319] = EBB7317413CAA37B4ECF5D70C4F5DB5C
                               1EC54659CCAD09D9D6FB33ACEFE9E029
                               7619CA8CCE41E32407FB10A8D02BDC06
                               872DF91778963DA73CC80E1CF28B978E
            stream[448..511] = 6B4C2C58B1AF907204935261B54130BA
                               2FC71423A25C214CEB1E11FFEA3205B9
                               C6D906CEF4A49756DE67AC5B90C6D55D
                               5498F1A3482DA5BC0AAD468B9F6DF49E
                  xor-digest = 435BF985729B54177902F6FD552A4893
                               C8BEF9F0B020A1A5BD8E7332EBE39B70
                               238DE493722977518AF8422ADBFD51E4
                               5723F85A3DBD65B874878A9F7324528D

Set 2, vector# 81:
                         key = 51515151515151515151515151515151
                               51515151515151515151515151515151
                          IV = 0000000000000000
               stream[0..63] = BEB253BF80EC236DFE52CF8B9812312B
                               03A5E6E836144A756D8318612B0CE00B
                               44DB4E59572091BF52967977975C71B4
                               18D2215C754A7A8FB7C5BE7E778625F9
            stream[192..255] = EDB3F9282B8E29C0D67A746A6DE88AEC
                               121BF597651323E83999CA305B5D66BC
                               05BEFE85FDAACACBA5EA97AA8918A535
                               D67B545CE3FCC25F63CCA10768584A26
            stream[256..319] = 7DE9BBCC97E07450381B9173E2ED5093
                               9DB570582EAB5785C8413341E00648EB
                               42FC7032C5F05D52D6D303B1E6E22A85
                               D3EA6B5875D38BA436D36C6B13C8FCE9
            stream[448..511] = 339F6AA77F7FD1654FAB0C7CBF847FD7
                               D3D6C3D0B545CBBBE9DE9139F7727155
                               9B46FEE27430B0D0EDC8B2CFE872DAA8
                               0DCF920FE5C09FBB0C05296AF336C538
                  xor-digest = 1A8EA71728D1C75E6BB431CC770139A7
                               9A9A48031B344564A0A0B572A3BD74C3
                               4DA48355C0C0737362E51423DE283546
                               716FA15C0FD39C4E47EC5C4B989FCA5D

Set 2, vector# 90:
                         key = 5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A
                               5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A
                          IV = 0000000000000000
               stream[0..63] = 60CA11B797C8548C519E04286EF8E5D6
                               4B5099A29FC4D48059A1239C490450EB
                               D212D0A79A19BE683D6D5A4BD5DE162D
                               68D73D34440D329567DCFE2E7181CF66
            stream[192..255] = E8E23059AAE2F3B5C36192513102401E
                               F32E7E60B1E3E9C0B2005724BD7CAB32
                               2D52C1C158EEA04E6341B386FBCA6EE6
                               AB22ABCF4F9567D55B18658839B0F415
            stream[256..319] = A68BE851B4594EF4EC2384823ACDFA42
                               E4178F6B249E8187BC2FDC8AA57CECE7
                               B42260DE02C7976F4A8382072DE8DD71
                               77109610B96295084632AE5C221DFC3E
            stream[448..511] = FFE73FEC0D0A6BC1017F9BB7D63706E2
                               8C2395971FBE6DF39C4EF21AC50C2B0D
                               B438681BC0863EBDD535270197B6B477
                               7BA13610569D4FE5D7AB12445AC41D17
                  xor-digest = A2FEA7C6FCC7408D325B4CD49A2F5AEB
                               D7ECC0E665633B054352D58B548A4D5F
                               8804E32149D13FF34726CE20883294E5
                               CF84231A165004AD1B1FC7EDD476D710

Set 2, vector# 99:
                         key = 63636363636363636363636363636363
                               63636363636363636363636363636363
                          IV = 0000000000000000
               stream[0..63] = 086EE1F7FE6A3702AAD117D83367C282
                               715082C01B4ACC70CE0D68D89793DB57
                               D284CA73C919109EA81C1D0DAD01067E
                               2A570F2882B1A8BFEBE06A4E621F87A0
            stream[192..255] = 78C21A6DC2A3674D09C81C72BB40B843
                               72A50C4BE38EE7B1ECB614F6334EA452
                               A668B2BF5C235942ED4BDA1478645218
                               9FCABE8EE3885F705357559B2855EB53
            stream[256..319] = 3D31E2343A710E05FFD8D5182580A8CA
                               D6CC95D3A784D3EF114E6DB8AF71F2DD
                               49B565EF1406FC96EFC7D4973418EB92
                               11D35CA399DC7CA8A0F80DC036B2437B
            stream[448..511] = FDF571E6806116B684FD7D274BA37A6B
                               9B28A339C045A410B467C61E5DB890DC
                               F3A2D0987321D563BF61A3D2077E5724
                               CCC9AE68068F3EBC36D85E2F767726B8
                  xor-digest = 86C275579ACBD9629C6C55C5AA26AA36
                               32319A00F6103820C47DF1E87449C933
                               4E0A3F6653976DC8821D34111174E935
                               D4E0D22F8C11844DF6481FFA665F1084

Set 2, vector#108:
                         key = 6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C
                               6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C
                          IV = 0000000000000000
               stream[0..63] = 86CFD326356856ED1972371B615030EC
                               40B08E23AA0C2864642702D5B43D339E
                               10CD594AF1F8F9E1334285B9AA850D7D
                               AFE359BDBE843E5C585B8A52D78F60EC
            stream[192..255] = CE2C4B0D6F8A2D8552F50FBF6E34132E
                               6743A649D6DA5A80C390CE95DCFD2DD6
                               C6DDF8C1A9D5DB12651BE63213020B1B
                               B8CD7543162B688B07C2191FCDC4EB72
            stream[256..319] = 9AD4DB6D6DC9CC08FA9E28B6152A58AE
                               A7F5C943ACE7460FED25DC22757FFF9E
                               3FB10AF1F0C151557A188CB5600494AF
                               F615DCC780849C3201B7FDC85C594912
            stream[448..511] = F29A436C80B185F949FE69CC4926E6AF
                               E74D179BA7BA6875C767210E6209F196
                               66667005D2D32D446F7E0C2DE6518665
                               5201523FC952CC92EEEE5FB46087F189
                  xor-digest = 02823CD1801107255F8C4E822E76A042
                               3B783E0A7E5D33F5A15561DDECB99AEE
                               91B41CB507FB010ED2D4582794F4B2C4
                               2B8D26CCDE7BE6204DEFF44BCDD43EF6

Set 2, vector#117:
                         key = 75757575757575757575757575757575
                               75757575757575757575757575757575
                          IV = 0000000000000000
               stream[0..63] = 213DD3DCE3BD83E16D92224AD4CBBC93
                               B1E6B037AD26F7F80CE93A1BCEA7AA22
                               F414237641A275697FDDFA83803DA526
                               A044BC04F298F82BD97135CA821742B9
            stream[192..255] = BA2B544DF3F9FAA3E85C1A62313C3A84
                               B0B72E1DBE6AA06CE5B3BE62E121C3B1
                               D7ACE4118BF04CB816BD05B1732AEBEF
                               18C48826A3A9BEE2DC19245667363EC9
            stream[256..319] = 4D7362FCFDA991F642D05B332DC008FF
                               F754EF715C7BDF0CCC22E1A3B53B38FA
                               D26F17D685D724A015289E400DC27EF4
                               A471CA4F41EEEE5EA39679C48BDA06D6
            stream[448..511] = 684C3961C25A4D137C3AE50E9C21736F
                               E7DF4EBEF1D734EEC3F25537FF84E2BF
                               235B9BF74A5E572897972D94F6C8197F
                               C6DE4D9F6B6D5F42D89C2E6E808A1D2B
                  xor-digest = 985A5757D3D1AEBDA6BB9D46ABDF8606
                               EE80DB153B7AA794099877E24A270A38
                               E5C7E6E713443259E85FD768636BB957
                               8DE0A9D9FBB9AFDC6137B72CB9E3DC49

Set 2, vector#126:
                         key = 7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
                               7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
                          IV = 0000000000000000
               stream[0..63] = 0551BAB03BB4BE93992ADFF8C21A46F1
                               4A800C1F13FFB271A81CFFC7F1D2C4C6
                               77EE5A9A2FFDA1D9C8C34712ADD11F5F
                               065F9E3309036AAC05A8CDC11A69BF7A
            stream[192..255] = A48E21CE027ADFF9376F8DF73F083B58
                               22889D88AB09CB60E16B9CB7B31A5668
                               D78C44EF5BA268D39608761323103E5F
                               CF57E473C0AE71891B4C569DCDD6D15C
            stream[256..319] = 84293E84C7918880D268A75B59F418AB
                               19E2E65D84C71B243BD361F567EE4E75
                               8B4D92E23E106D0C681DBDF21F6A1F79
                               128B49A0C44CE7C2A8E362DB98C4001D
            stream[448..511] = F01AC73489301A414B97226F50B66E93
                               9B9C938A917EAAF0BA257AC7BCA7DA65
                               7C61EBA38BEBCABE03EB57EF85DE9F76
                               092B42810811CC5C8B2D38C157830618
                  xor-digest = A03E63F97E770277278437C78FD6260B
                               EC6C48700E7ED9AC7DBFB9B29BEA55C1
                               B188C8DF2238D4E2A723402CECB2082C
                               146B51050E0E44321CC82A59DEBAFEC0

Set 2, vector#135:
                         key = 87878787878787878787878787878787
                               87878787878787878787878787878787
                          IV = 0000000000000000
               stream[0..63] = CEDBEDBBEA889C2F1297E66143D49331
                               6B17D6D49903C3526D491B50594669AA
                               CD5448D143130676316344D12F370738
                               6932EEDB15A4FF0C4AA884C50ED7597B
            stream[192..255] = 73E08145584DED40393467D34F53CAFB
                               C1D49E187320601F673E9C83A6673027
                               3C16B05031C6AF55478280985C7B3B74
                               600FDD37F1786C7C11384150DB055DA5
            stream[256..319] = B8ED2CD03780285CAE3929D32D803292
                               5E23C9FBBC5793F9051C4471B14FDBAC
                               967ACF4FF1AE5FA2EFC82E074397BD3F
                               F050503BC9C9DA0FDEC16A2B976F09B4
            stream[448..511] = E36AADF2E9EDB4F27575F95BBC56B7C7
                               EE375DF2FB260F394D6B7AE2E0996747
                               AB0CE3169072041471BC13DE9EE2EFE5
                               8D9D939EF3DF5F41A3B30109B2DC9CA3
                  xor-digest = DA3B913915B231C14D7592448B78E54C
                               71B96C1D4657AEAA9D35BF0A260E1B81
                               F5848E18863585DD997D3B0B356B182E
                               42FB631BF12DCFD04D49CA848E0C199F

Set 2, vector#144:
                         key = 90909090909090909090909090909090
                               90909090909090909090909090909090
                          IV = 0000000000000000
               stream[0..63] = 1EAF9987509E6FFC3CFC0AC8C285EE2E
                               270A123FAA1B22ADD2E42F46D6A268A9
                               BE109873E2B0F2503AB7DD6EDAFCCC86
                               7BB723956716A6DB5DB82EE62697F1A8
            stream[192..255] = 48B47209D27B090CFD828733860FDAC7
                               4A06FE3FBB0FF0DF60C0E8E433E0D726
                               7A95110A2525DFA636F7FDDC5B5095A6
                               7710EFA1FC913F4FA35BE2FB4D694390
            stream[256..319] = 570F478690B4E918F8A813B790A4CD0F
                               05D3AB1FCC9A7659CEF9D3128620576C
                               8414F818ECE1D79724D2AFB599061BC6
                               E281D1F9D145B7A2B222B17DEC46A08C
            stream[448..511] = 5C28A972E602CADF119A184C4237F043
                               401944E09C82080D9BD6E519DEB6E0AC
                               2DAE03A1A7AC16CAB95A7C1DB03E3521
                               E816942939DD1F9822A5E0C4FDA1A263
                  xor-digest = 9CE61928679DEC4956169C8547C4232E
                               53A49690D65E4A151F29FA45640F5932
                               F371F9785C9139A3A21116912FA3714F
                               15D598192F5E003F6AB5DE5591EA5F78

Set 2, vector#153:
                         key = 99999999999999999999999999999999
                               99999999999999999999999999999999
                          IV = 0000000000000000
               stream[0..63] = 838BF86C9324F9EC7961B3BF5AFE68BA
                               F55583FCB27864ED3329F0613F20F04C
                               9FD625F1C6C7F0F5B1E9A1FEA3A9E18C
                               AAA45993E014FD813C968112CA642E7B
            stream[192..255] = 2403B2952FB9366AE4CAA38B5F28DE39
                               60D968107A0DB23C5160013DC08765AD
                               7137781658B368F5CE80FAD4FBCC2368
                               85230940A4A09EDB80B6AD2C7739D690
            stream[256..319] = 76FDE2E157CCB2996694E32018F17F89
                               F985175FEE5F81AC1D761582CC805FC6
                               2429F85A410BC66CFD1184ADD932C52A
                               F354120D7B39918BC11CAD25FA902E4A
            stream[448..511] = 0628DB9A15D3294F2A4ABAA982C4CB08
                               9434446F61E6DEF2B552F42A3A76633B
                               030E39C0F981FA2016F00C72ED2074CC
                               F09BC1D2D5F316139798C648A064EA77
                  xor-digest = A4471FFB0477B66D98D8FF1A8D0B190D
                               BF08CAF284812D1C58DD199A7925F650
                               FD68C5A9120AD708C6BAB4696874D88C
                               D49BE92A34049E8BBD0E119A9AE22ACD

Set 2, vector#162:
                         key = A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2
                               A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2
                          IV = 0000000000000000
               stream[0..63] = 95E4B88A9404C242E7738754680F3EDB
                               59DDBD3CB681E74643149DE0E8CB92C5
                               F7759AA64709CB4690A677C1B00D5484
                               C753D0840E77F3136A1BF8B548E6F992
            stream[192..255] = A82BDAE1A0587DD1BD42529A234EA560
                               67AE5B0646ED5D2F0D91E91B05AAB2F2
                               590E5E6FEBE556FAEE2EEBD18B146962
                               54ACB8D5FA530E1286D9BF25263B2B0E
            stream[256..319] = 39B6F0D96EEE6BF0F535156CFF2DEBAF
                               76E402D33A17B90E0A6E2594455E6350
                               08DC040016912FE5957F81F1C5957F20
                               62E297E437B524BEA66B18DB71DEB8A6
            stream[448..511] = 90575BC9ACE284748A5D5C88FBB67B14
                               1F0B2F997A286E499814B4450355E765
                               DD8A12A758C482C77C318FF0224E8101
                               A3A8AE151BE6FE749CFC540164BD57C9
                  xor-digest = C0ED332BEC2029E9DE93B8B58109894F
                               855180EB9F2345F344FA185DB23D81F8
                               3305D48C1A38D209555D535AB2AE6EC3
                               B1EF3F40369D9DBA40C64CC08AC1F1BE

Set 2, vector#171:
                         key = ABABABABABABABABABABABABABABABAB
                               ABABABABABABABABABABABABABABABAB
                          IV = 0000000000000000
               stream[0..63] = D776DBB5A293624CAC4E80E0EA62CA46
                               EB16BA6F2416DFABAC5A788D20D7F99B
                               D192AD8DE4B8088F3FD610436645BB18
                               F0977084CFF58EEFA9114B90737EAACC
            stream[192..255] = 5B2CE8996EE58CD54F518470CC59B352
                               FE36D1479D23B997AE4D5A3D775C9BE2
                               8944E1F227F73955A36CA488986A40E8
                               56D810044DFC60801CE0CEB1C7BDD74B
            stream[256..319] = 8FA5126A205EDCE05992F46C67DA7BC7
                               4D7CE666F82698A0D01CD6F3C295E223
                               6ACB7ADC9DD16D1DA2D518EFA347AEE6
                               CA40A41B915A2DBBE8DE7CBC77585679
            stream[448..511] = 53E4AC4B44A8FBA54F6360E9C9474E06
                               DA67777F25D8A898E2E00F30DE66B145
                               63112FD92B8DEB2511A58951A2C5F4C3
                               86F0CFB5CC8EAEFC40F08DE5E4E35B57
                  xor-digest = CEA5BB6799866C5D9ADF354C9C18D70C
                               D2CF126E90347393A7BCAC9526075F4D
                               5F2CC464056006E4291F1ACA4C506D21
                               F5E0C6DF1EC4E5EE5612E48649E92EDE

Set 2, vector#180:
                         key = B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4
                               B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4
                          IV = 0000000000000000
               stream[0..63] = 66CC59EA2C9BD7A54DB3A6108CAE9CE5
                               18A6CCEDE8DE2494DD719BD85CBF2DBF
                               8EA69971C985BDF5EE86183DF6C2ADF4
                               76968AC97BEF6F857248F2D2FE733CD1
            stream[192..255] = C62A000ECCCEB3196FAAA2C952428C9C
                               86F886FE40F8D7BE4C036FF5D6179A8A
                               15EF8916DD3E5FC94538D7125CED434F
                               6AB07EC4364E853F38532FB438803D05
            stream[256..319] = 999D12E53E5DDFA5F1F1F9AE02A79D8F
                               CBE83316BEBF3A693CF5BC69C6CDF910
                               40A214335C287674E156C1DA21685433
                               3203C2A15CE8EE1F559A0FB904475B51
            stream[448..511] = F6C77629FF659B41A71A61CE2E00DBA4
                               AF665389228FB92BB162231DE4F11CBE
                               8FE89B1ABCC0FA090F900BEDC4CA7061
                               72E6D49AA6A75DD429945A7BE10DC6BA
                  xor-digest = B189D7A68E5A011EE4D7A74B13793C30
                               78DF4B22897763BD3576501ABF4EB945
                               740F3398D1DA826B44A7E3223340070C
                               33F75157F1185734E51C7A8A1D626331

Set 2, vector#189:
                         key = BDBDBDBDBDBDBDBDBDBDBDBDBDBDBDBD
                               BDBDBDBDBDBDBDBDBDBDBDBDBDBDBDBD
                          IV = 0000000000000000
               stream[0..63] = 2A0A0D51E38B3D4C958262699626F403
                               62B3DD64855AD052F5FF35634A5ADD29
                               FE07711A312D244E0E788CB17AB9FA0F
                               80F2011AB76C101989D1256FCF92ED27
            stream[192..255] = 9015DCB12AD8CDD7AE00B74DEE9756BC
                               FE1E96DB82DE2103046C8D32BA4AADFB
                               F4DCABD8DB61D3C2D24E6778DDA3D0D9
                               69DCAC2D9031DFFD817E32CDF93E7BF2
            stream[256..319] = 88FCD9683309F597CC1FEE3B597BD871
                               0ED01AA0205F0673B9B89571BD4268B9
                               BC339CDB7C150DCF8A86E5F907798639
                               9A1C2D93A1A7130CE8EB3ACC153C2C64
            stream[448..511] = C471C3A2B9CCC7750DE5BE6FD0079A7B
                               7B820E220B2F7E221300883158B4ABFD
                               173275BC4CD3E268383EBA8B8FB9DBF4
                               501AD0A8EAF044154D23E31D46849E09
                  xor-digest = D31495B1005A0AB28051F883E9C1D021
                               E654646F328A2570F453D3A31CE1FF38
                               41FA0D3A70C0256042F26696B2518A83
                               F6364F8C21E11BBB3A453786E4E63954

Set 2, vector#198:
                         key = C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6
                               C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6
                          IV = 0000000000000000
               stream[0..63] = 400E1EE3FEEC8039F95C584ABB0F0109
                               1FBD677D35A588D00DDC1CB01CC4B6F2
                               DBB96B093383CDA0C0C97E94C024834E
                               FED938CE6FED291E4529409C4B617443
            stream[192..255] = 4D6F8C12AA9A58EEF4928BDB441B7021
                               305ABE8FD224CCDB6769269AF74D5588
                               54EAC5E317B72023A6174AA2271435DF
                               ADB19B30D5C0D25CD5CFC45047F613FE
            stream[256..319] = 17376A07C62F9902B5D0A094A31A25E7
                               453E593E9A00B455D3749F1E7FD1453C
                               3FF1CF4DAD5C4D55EC22A074BA61B29E
                               E8B6873A6EE03EDF7A8D8A735D48F088
            stream[448..511] = C043C07842007AED090DBBBAD900EC4D
                               62F9485DF314970027F0A831C632A47B
                               5821E1DBE0B2F87A1C4695712D9954FB
                               8431FC54FC6B6BCF31CC6CFE6226FA67
                  xor-digest = F1DF86E9DDB7D9319A43E3FE1053FFE5
                               0AB03172ADFCDED6D647E714B93725DB
                               0F72B11BABE129230F183199412CEB1F
                               9E26A1F55E23264AA5371C7B8D9DD760

Set 2, vector#207:
                         key = CFCFCFCFCFCFCFCFCFCFCFCFCFCFCFCF
                               CFCFCFCFCFCFCFCFCFCFCFCFCFCFCFCF
                          IV = 0000000000000000
               stream[0..63] = 3A884901CE2FE7E2C92B83FCC7851711
                               DC1A248C93D3D0CF51E60E723E3B286D
                               7278705D7170915035A3A5633AB266D1
                               7800D09BC1D88BD30AF579F7A7741350
            stream[192..255] = 469BF47A16925A4A66A5FAACCE90DD17
                               63B025F46840645545CA8E631432CB63
                               5F6B40FE4D4E5FB11E8B0C6763B0809B
                               1D33F3465AB3307EFCF9A07D63692182
            stream[256..319] = D2E3E70A50C52FAF3177DA9B37774E9C
                               CB0993025FFF6D512D09470D30F6CE5B
                               9FD74CF69866894579C13873630BB756
                               8934B6C5734F1C248D844CA5F8FEB697
            stream[448..511] = F99A709234B57B1F480A636E418388FA
                               5157F5E735C95CCF6F87DE0DE719685C
                               CA02458660E824211653B6A513E00E14
                               53499F9C4A316FCD1B3C3B0AABC4767A
                  xor-digest = 1E5D0C73621DCE21D371F75459529E8F
                               C06678530B2979DF405BCBB0B6902FCE
                               4791404564791B9BD02EBDB2D7B177A7
                               672759322184041ADE6EE019047E805A

Set 2, vector#216:
                         key = D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8
                               D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8
                          IV = 0000000000000000
               stream[0..63] = 6C79462F32EC653185AFDEF23A228CC5
                               E0CD4516BBB3514544362D07AD9F5B64
                               9A7601234BD0C72C57825A6A3DBBCB41
                               ED82FC1C98D5A9AD7AFC5AC84653BC11
            stream[192..255] = 6198891C9AF821BF222323674D8629DA
                               64BE13792E65704653ECA9B24322847F
                               5BBAA386465D5CA6C52C7863B83C45EA
                               C026B3D2EC279F2EB997EF6C09AD99A7
            stream[256..319] = 4895D171F47480D853E19921C35CA02C
                               F4C6D389F09B7CF449DB6EA59237ACE8
                               AEF1A4A3AB2C07B065E60CF17B696BB3
                               4CF9423039D80B9214A7B8F770E5EAE2
            stream[448..511] = 228AE54F43635BD6CD905FC882E33330
                               557B0A31DC71A6B7C9809A5E7C91C4E8
                               45B4A78D40D3833FD03E2527D3246427
                               73B43E290F797857C510114B60719AE7
                  xor-digest = 7A3F26F2DF71AF6726F7EA34B30B4019
                               A7CA0339D8914620FEC91753267D8226
                               EEE5CC55C45F34D1FC60A63920A3288D
                               960FE1B82105079241AA8E3691C97107

Set 2, vector#225:
                         key = E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1
                               E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1
                          IV = 0000000000000000
               stream[0..63] = 3B353E3E965216E8A472F98B8EF40551
                               5DB8F69ABEA40EDAAD45218F42E5D179
                               107D4F1B687B816948E6EC56B4CA7187
                               9AEBF0E038E66EDB45891F689DF25D76
            stream[192..255] = 190BE689A52CD2AAE9766D37E3208390
                               D6AE3F26D46CD3E806F3EDDBCBE23BE9
                               5E352A5789C6FFDE9ED9AD3D16FCEC08
                               0CDBFAF768FC7A78AB2DFE1E95828AA4
            stream[256..319] = DF75A7F6B728E94DFF932DFD47CA2725
                               71A38607D7006E99DB8A5A184397BDEB
                               6B9E7233444FDEBBE0FB26504B7AE2D3
                               9CE38C7EF60358A8B87440EC831F4CA3
            stream[448..511] = C22D227A340E0CA579AB4D5B7373C306
                               8BC88D5941079D465E1AAAB9182003B9
                               2248E697F2945DC3B048482417538156
                               82E48992D6D91C4EF63802D8EB97A9EA
                  xor-digest = 66242C4343D005DAC38B70886C518AF2
                               CCAE53B67EC60251C9782D0476642AC4
                               53BD63EB7CF634E9CA2505C9FC430054
                               DE068DB0045D190013468FED7C7FAACB

Set 2, vector#234:
                         key = EAEAEAEAEAEAEAEAEAEAEAEAEAEAEAEA
                               EAEAEAEAEAEAEAEAEAEAEAEAEAEAEAEA
                          IV = 0000000000000000
               stream[0..63] = 028F2FFF9F4A343F56ACD18E417BF1F4
                               06E5372F8675DA604B87F04658BCAACD
                               2CA8BAC22310F418A9808D1BF51602EB
                               B40B17AF425FA62145AE9F07AEBCBE64
            stream[192..255] = 37B404CE286F761D03A5DFDF13998194
                               B6E8B52E9C086A0735DC8D8CBA92E40E
                               024EC906C0E07914BE6C94DF48EA6DAE
                               8270E04D029F2464C4C2FE6E77B11FB7
            stream[256..319] = 6B168F77D7944C84EB60AD8786A2CBF9
                               E013A440ED8BA0A349E90414B9B5FB3B
                               7A69AD6E1B71079FBC7EBEF3B048AC20
                               717538534FC69019EF153071EA3000DF
            stream[448..511] = B8BCDAC9713AFBB489E7065A5A1FAE83
                               C72FD27E4F61A2EB9A9DE3D3EBB83727
                               B33AA5A1B858B12DC056693305B35092
                               7D9F0F939C2EF9F5B968F840CCC2165F
                  xor-digest = AD64EE8B437E6631ACE86B34B0D2EDA5
                               F7EAE0EE690DBCFAD119C51DAAC3CDBB
                               B41D6B0C34FAA0935C497FE8AEE2FCAB
                               8A7BA6ABD43674EC54F6252C80882C87

Set 2, vector#243:
                         key = F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3
                               F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3
                          IV = 0000000000000000
               stream[0..63] = 3EF82100AEB325413DDC66C1F0F3BB8B
                               32CB76FC2BC6F554F227BE4F558BB529
                               D16DE5C00742C9343B1C7697EDF58AED
                               F3DC8FF3D0F7AB4FCE824861A1325014
            stream[192..255] = 4938860A4056C19547C2F3D154BBC628
                               BD04DACAF493C12835F2C9CFFD6E6363
                               8FB0C3DB9D128E9E89ACA87C1B8E846D
                               63D16FD6113FA5701C793FA22F3DABEF
            stream[256..319] = 9FE59D216EB945C2DFEC191C537A36F5
                               FE9D314B81599347AC7C44BC66A4DFFE
                               CA30A8B5CFA643BDEA4F2328898D2133
                               8D5FB9656DBF839FA6CDCB2239970C22
            stream[448..511] = 738092AC1C945AA515EC7015EC1E0069
                               E019E493AE21C3E591AF6B073EF291A2
                               494FA42AFDBE140586BCBC07E9097294
                               36D7A16E2EE7C9FEB2BB2C8296C5F719
                  xor-digest = EF7E13981E3584A23EC45EF5D87A9987
                               C0C1793CA69B4B1D45F5659441F94368
                               7F5B8BF8E927B8D6F9C457647769FBA5
                               EF96F5FF096310B6A43F691815EF4071

Set 2, vector#252:
                         key = FCFCFCFCFCFCFCFCFCFCFCFCFCFCFCFC
                               FCFCFCFCFCFCFCFCFCFCFCFCFCFCFCFC
                          IV = 0000000000000000
               stream[0..63] = 576C594F6C5D1A1B37DDCBB458E4015B
                               F0825704553740C0DDAA61FA45CB0402
                               673E22FED29FAF73E4DBC9B5182FF904
                               44D0FDAFB6C7857B81E21193526CA955
            stream[192..255] = 953F56F89601AD6EC1B73E218406E917
                               AD5EF96C6996069DDD51AC5E68FC4502
                               805698AA2E6536FF630CE0E9BAB6B769
                               2E5E0C13747A849444CFFCD53E901B03
            stream[256..319] = D1A5E8AAC58378BA64A931CB7D871FD0
                               D472E121E6518449D75E2C43D5A4A04B
                               D7FB9FAFE7EEBB6E6F3BCD9BEC01A77A
                               E41701FC3D91F43F899E5B54286630EB
            stream[448..511] = BA6DC9518AAF9E21E5AD3B5A5385C1CB
                               93F201825B73FA907DF9BA5FCF018910
                               329721FA9F255024566D16C3DC90E448
                               BB9198C46B013E123D7FFFA3E0B8F495
                  xor-digest = 1BFB2203D3EA5A1E5CE6EE224F6DA465
                               81850FD8C686341B2DB7FC30A4C99B61
                               C20B8EC02C21EDE6E18FC59AB1086E49
                               7D6B6F5E69C45B65C10387AD7787BD98

Test vectors -- set 3
=====================

Set 3, vector#  0:
                         key = 000102030405060708090A0B0C0D0E0F
                               101112131415161718191A1B1C1D1E1F
                          IV = 0000000000000000
               stream[0..63] = 83B8FAFC9D0877ACE5B32BE82F2789C6
                               20E76D38E6E102FD5FBB7030C983F7CE
                               0DE2CFFA74753DC7E267D26B6ADEDC24
                               EBCD0F85C2F2384D7A46F7E0A3F12801
            stream[192..255] = 8CD28DE6B67C68923F9D43EE80B79A8A
                               17CD699EF88219544714020835004093
                               D31EB2569A49DAF0FEC5CEC7E5C0D109
                               B81A37E1F283228D526B74B1121E4882
            stream[256..319] = 5380893F6A6101E0A326034DBEE3C4FB
                               A5CAFC6935396A83FBD9F6D8BB82F005
                               ECF507F185E34D21E27CDF50BC7E860A
                               DC843EC1288B16DBE28189AB59BD2FD8
            stream[448..511] = BA8FAA38C8E337089F7BFCDC9F86E49E
                               AE5540C9CD3A9F69B739156FD68A2A53
                               002E0646DE04655BAEF5CFAA1CD2B103
                               130EE442087A9A67E0FEBB2024802FC5
                  xor-digest = 0C1BD98719AAB70E1D01C8DEA9A44BC3
                               0D8529BA05F413A75A95003B4C4FF43A
                               856A8C39DDE290074B2C95D0C8AFDC92
                               5CF768BA4A8076BACAB76A9FC8921D2B

Set 3, vector#  9:
                         key = 090A0B0C0D0E0F101112131415161718
                               191A1B1C1D1E1F202122232425262728
                          IV = 0000000000000000
               stream[0..63] = EE057BD65B2AE68A93C63B6990F05B12
                               B18BB6DF74ECE11AF4549821D6206F5E
                               D0EB0FF4294A4637BA096F393BC0BCB3
                               482E126C6BB9483878D4644954C8EB80
            stream[192..255] = FDE4CC34D720F6C87C5664E246D4D638
                               D59596DD9A53BB4701F5CC84473A2C54
                               F312B8BF1E56BD3A9236524283088532
                               D50E437F8653FF70D376162CD046CCF9
            stream[256..319] = 0D6666E291DA4072A2AD160B5DB1BB32
                               0BD1FD43B058F85DD7A65567BEDB9B40
                               EAF9191A965E59D36BC859C2A99B79DE
                               71076B3EAAC7CD3949246D27CB729EAA
            stream[448..511] = 3370DDF88E879DBCA5EC693C2666468F
                               7DB8C70FEA8BADB27977059D2D3F9968
                               75900742E7B867116F74BA3EF389E5EC
                               F9031A95E1E9DBFE54DCCAB7319CF42C
                  xor-digest = D92FB594A2D9DCA9A741D46E93843C14
                               1F896DAAE538AF8B390E2448F0A7AB90
                               F9F98FBDC90161BE28F002C538932335
                               3B9BAF717D551F1272E0AC32EFBE3156

Set 3, vector# 18:
                         key = 12131415161718191A1B1C1D1E1F2021
                               22232425262728292A2B2C2D2E2F3031
                          IV = 0000000000000000
               stream[0..63] = EBD0586F334692B88D4E5CD9CE8B0E2F
                               BA04F466344845B61C83C280D1D07699
                               A1A8086762C51410B64BCAC7A21E9F33
                               34634E492D5E5F0E61DCBFF2A1FE5F42
            stream[192..255] = 155ADA79705DC5962CCE1A726F2B4607
                               A82B724254BC834F7EDDDE385350BBED
                               E5DA4E31CC928190F6AC3AE3FBEF43FA
                               46403CB408BEE59058DEB6BA705738E4
            stream[256..319] = FD6587B08C97AC268CD53DBC4AFB61C3
                               DEC3AEC5AE9C0BF3200645704D88FA07
                               65C44D046410B6CE17562586E78E2388
                               0ABD53B287A745B92B807D929F08B044
            stream[448..511] = 96C6BFD84F37FE32BE534E1E7F0CDC04
                               4B5E52E6F882048AFBE4FEEE3865EB2F
                               9C15671B7AA32F8F11DA799B32B3403F
                               8D6A2649A9BAD8845D0F3D906C7FA9AD
                  xor-digest = E50EB70DD12B95502CC8A976F930F59C
                               19DD52DC3C1A62E4A1A4BA343740C146
                               078C77A33F4A4501D6F80AABC5984C47
                               9FD19C560DF3DA0EC5E36C5CAF888229

Set 3, vector# 27:
                         key = 1B1C1D1E1F202122232425262728292A
                               2B2C2D2E2F303132333435363738393A
                          IV = 0000000000000000
               stream[0..63] = 514715C4CB7F1960A1E64645A5CE592E
                               70609C87FA3918753B82DD519711CAC9
                               A2B3C52EBF5C3CD1AA4488026D133BA9
                               A46931427722DEB5C38E45D0A64FB543
            stream[192..255] = 2C378DE00F67EC95C7AE7C44BE0BC183
                               9B3A6BD1FED4D4F0EC1C95BAB47953A8
                               BEF51B0DA372CE06D892F1B130F65EFD
                               2E41A5CD11A972709C6FE46FBEEFF7EA
            stream[256..319] = 9E7A50E19E3FCD649BE2FAE1561C4596
                               E7FB3CAD3C68B8C3E778AF650360C244
                               1788E381E9E91E98F19207C309BCB295
                               F7FA67F5333B965AC7A909F0903B0FA9
            stream[448..511] = 4D3B613F5C0AF52800B3F20D2868A32B
                               F1F43FB2CDF878C1A4F08878F5E5ECCF
                               197A880F654FF75712409B58A4EE9081
                               C2AEE121485BF0736FF048171F00C7E3
                  xor-digest = 82BD892940506AC1FF885F8B67570CA9
                               0219401950BCECDEC40A08C5D115CE1C
                               EFCD0CAF2458940AE4D3EA5BB5E863E3
                               9C1E6B508D0704960CF579770A44F012

Set 3, vector# 36:
                         key = 2425262728292A2B2C2D2E2F30313233
                               3435363738393A3B3C3D3E3F40414243
                          IV = 0000000000000000
               stream[0..63] = D8EE9A7DE99A531FD0313A295960E7C2
                               903BF71C2E35B503979A04654FF0C474
                               B97FF731859AFFD50233DF7F79D0D677
                               C8A060D8035F24846C6988F8CC913150
            stream[192..255] = BD83BBC63837503B0D00D3887804D981
                               25FF300C52972BFBBE2B791CEC7700FC
                               8E3D16A0244856E61EEF1B28F1B451DF
                               FE3EBA4CE8F7485875B729C074402C29
            stream[256..319] = AA470544D499CD6445757996DE604279
                               DF407FEBD3DF80FCF5A95A41209635B1
                               923F7B397D6DB60242599F665020363C
                               8F55DE21A12C0DAF16CC329680B261CA
            stream[448..511] = 0780E9DC2B6AB4C580E048BC6BFE8287
                               9A402FD7E1E3FFF0BDD87E04ADED2CF6
                               75433652EB92D3EC693BABD5788477F8
                               E496DA9D9E56A531DDB10D39FBAE953D
                  xor-digest = 468B0B5D26EEF40C460B8B27F99E296B
                               D5B1AEBF4BB067120318F90C4B7D695F
                               B46B432ED4300CFF3ED224A38AB1B886
                               D061ACF1DED454E7A7C82FF37CD3EACA

Set 3, vector# 45:
                         key = 2D2E2F303132333435363738393A3B3C
                               3D3E3F404142434445464748494A4B4C
                          IV = 0000000000000000
               stream[0..63] = 4C02E827DD736F3796F789602F437D2C
                               A857849F69BA55F9785D75F3B7F8B28C
                               98E42A9BCC7BF598A7C24BB575B3BE3D
                               B2F192DDC3DD287531187536B8B7BE3C
            stream[192..255] = 46D07278B5FC215DD83E2DFCAC7B72B9
                               27AF97AE3AE5713CA75E7B17E51C34EF
                               8CA4BC628D7D1A786DA83856DC73FAD8
                               DC87F3E279D248E8FEFB15EE23793C89
            stream[256..319] = A4C851C1238F37ABA543F792284682E8
                               B7675C8A5E77D582E20E17A2F2D0A257
                               A489F6157FEBA80F7CF77BF4D167D0F5
                               857FD88FAB6AE129324F9DDA0EF6FA6E
            stream[448..511] = 8B236D38816556729804E4CAEB3FBE5E
                               1FDD0A8E894CBCD45CC628626BC36918
                               D2B995F32D76832C099CFC5F8B48B6AA
                               B971FBF598DC5456CFBC0C5C1C1D8096
                  xor-digest = EB36CC013C52A9B8F1B4C9F6FCD74833
                               6FF43C7C61FED948B0E81D6BC391436C
                               0C67A68734596C6A670134B141E13A35
                               AC02A781635C6B48B65F1F613F22AE9F

Set 3, vector# 54:
                         key = 363738393A3B3C3D3E3F404142434445
                               464748494A4B4C4D4E4F505152535455
                          IV = 0000000000000000
               stream[0..63] = 39B089939B2A5F6C6B14ECC43D6685E5
                               CF46968FD736EF7212AC957E33D16BEB
                               19368EFF049046C385B9261FE840B70F
                               2FFF653A44B722FA3A4D137F3975B8A5
            stream[192..255] = FD9ADF7990758BA30294EE4757BF166D
                               FBD15BC966FB7BA2D0139C95FC8D8180
                               4FC0BA88732D99D002002074C820EDB5
                               78AA07D5CF463D75DFC0EF4E727A27BE
            stream[256..319] = D5C110F13C35C5B4A4518DE641AE74DA
                               DB29E7D737E770DF52588DD51A36CC97
                               C6E68756932BAC8DCA84BB706CCE7E12
                               A0488F02806740830C0EEA760416CB7F
            stream[448..511] = 78DA32974D8BDE553FD204AC4833044A
                               A6D05511D9B78829D6815193BFD98788
                               7D26EB145AA7429AE3DD0AC34B83860A
                               BD746E6861921380B617CEBE08885AD3
                  xor-digest = 8A1241E4B69825395BB9D5AB3F32D758
                               81A6D778CBEE4F6F515B64355C622360
                               1A86245DD75B0940BBDE6C28247A3B5E
                               BC9F5BC95096D030D76FBB989B306948

Set 3, vector# 63:
                         key = 3F404142434445464748494A4B4C4D4E
                               4F505152535455565758595A5B5C5D5E
                          IV = 0000000000000000
               stream[0..63] = 4C025870D8B17661C922FE2F8E431C9B
                               73FD6E280DF37B32EA6729069770D4A0
                               D285FEE2D1213D79FBDFE861B3877FF7
                               81E259F88EB76980DE9BEEC0D3BD28C4
            stream[192..255] = F18DEA2E04E3F82F13008F9E9974388F
                               24036378D927E5BEBA35DA6A9AA5A8A3
                               5F7068BF8989BE6547A0343C48322630
                               637C0208228FC9F01FF182191414065B
            stream[256..319] = 7011EB6F07792AAADE37BEB8135C1F22
                               C94495B11271F53DD090102AC84239E0
                               1027F25A7346B33521C866CC3D1DE42D
                               4035CC9252992C7A98878CF929E63BF7
            stream[448..511] = B67AA826A02C7FBC328D8B1EB72E9114
                               CBFD2879C98B5E2231D9E96D30F84C95
                               D5F44F9E4F7B7759733785C65D31735D
                               F597A30692EACBD35DA21F3A71821AC7
                  xor-digest = 3F27B46CBF54BC94C3C556CEF9371972
                               B73452AC9831D7D9C2C6031DCE1B5D08
                               DE4E50C6334516228E1A890AB3E8FE70
                               D72D058E63D202854AE5F2B1A2A09FF0

Set 3, vector# 72:
                         key = 48494A4B4C4D4E4F5051525354555657
                               58595A5B5C5D5E5F6061626364656667
                          IV = 0000000000000000
               stream[0..63] = 9E1B005B1072B05213869162B5E9940E
                               C8847C4A2D196E421C2BF7ACEA349EB7
                               4AFCA3DE0A62416A5B7EA34D90E83EF0
                               608B45F727326C1DB59ED8C7A82EF46D
            stream[192..255] = 247D730C7FDE4B1CA99F39D29F535DE1
                               3ADBA3493B705CC5E279BBB8B190F325
                               2F21E742D8057CB3B4715CC696755540
                               AA090950A422AE1C6C9087A6AC3C0314
            stream[256..319] = D6919E05F350D80BF9927EF17004A684
                               02FD3A990388478AFC98760FDCF0DDA6
                               0797C78224B0C7899721999C8806D6FC
                               2C2CE8D428D273FE5FE8D6AD0F0CEE46
            stream[448..511] = 37B8AE6780C719C5F89E9B13147E915B
                               4027B0419F52CC68D287391EB3954ED5
                               E7BD1F1B653F146D8D0E6A13E6B8253C
                               09FE17E11D5A99F719CD0072CEA40E80
                  xor-digest = DD32748517CA537D50CE908F5934461B
                               B2BAD80FFD6CA8673B4E72A5F0DBDB08
                               03DA7BC2F61AB452D570DF1A589783E7
                               3F4216C3244D460147749053F4091E3F

Set 3, vector# 81:
                         key = 5152535455565758595A5B5C5D5E5F60
                               6162636465666768696A6B6C6D6E6F70
                          IV = 0000000000000000
               stream[0..63] = F4A7F5AD819C8C5091C4AA8DCF2B990F
                               D75FDC9A5781F4876C31B0996251B3FC
                               BAB9E49DDD3174E521DEF3648BF7CD2F
                               E8CF29AD7A5CBDD38A67F2DB0FB25F8C
            stream[192..255] = ED03729B234A7324BCAB8B12A6534449
                               7BC68E52D222248D2FA6BB2D057A597A
                               1B1C61526E707DE6080E790B4BC8D6E0
                               0A674B050F4D1F80821D9EF93A48D790
            stream[256..319] = 78AF0C754358D14CA3236738C3F6081F
                               015BCBCC2A91F1EC31777BEDD87727FA
                               38A8E04BC5207507BB7D5DEACA995B40
                               BCCC5AE61F13BDD048DAAB703165D173
            stream[448..511] = 94DEDE89A1EFA1C2EBF4197C50E6FAAC
                               128E1CE69B4DD634851B4A542309C4A4
                               BEC042388575B9F1FFD2BE7BF7C54230
                               3FAEF32100CECF47443900529C22E577
                  xor-digest = CF3FFFC81C902D50374E2F5A61944421
                               8FB1A8B57A1509D7B3BFBC95AFAFBA85
                               79E9C73DA1177A4B1679F09297F7C544
                               56B0B59A20F3B9984F822091D03CE790

Set 3, vector# 90:
                         key = 5A5B5C5D5E5F60616263646566676869
                               6A6B6C6D6E6F70717273747576777879
                          IV = 0000000000000000
               stream[0..63] = 2C3179280B545CEA99FC6F30C0E23FE1
                               C4EB42EF4E5482BB832EB22C1C905692
                               07B1E1C2B9F0902159875470647F352C
                               C0C8614E084451A4365AE5DD103D76C9
            stream[192..255] = 64DC29E6EDBB55DB8E91CC286B4B901C
                               98DE95307B8ADFF92336A30C5BE03E4F
                               7533845EB44570427991461EDA67CE9A
                               4B9ABB9A8FE989FDFABD5DCE7E6B7B4A
            stream[256..319] = 8909CFCB450F16639F054474B628D9AF
                               E1FDB4CBF7468DD35CB78A148958578E
                               87F85D1B374629E6C6DFA1C2C650A884
                               954A421E95586F8E451847265D868C47
            stream[448..511] = E5B942A3F2D4F4195CDBAD9030E603C4
                               53D86BF65B2A1A63C64EEF0FC5628C96
                               F978F11C39CF588785DF38C11227999C
                               787DD42A794502F6C8863DD2FA0DBC9C
                  xor-digest = 017E8528FE9F64A444AE6D69373598F7
                               0FD32398F3544EF98262B5DEC4014922
                               4E5FDC7690A243608750F8A4FCCE22AF
                               8A1F5C8197DC7DC7E952C816FD6EB584

Set 3, vector# 99:
                         key = 636465666768696A6B6C6D6E6F707172
                               737475767778797A7B7C7D7E7F808182
                          IV = 0000000000000000
               stream[0..63] = FCC7F5C877E6A71303FB8861E1C54073
                               059FA8FB2AD9707A70A80EEC402245DF
                               645A83570A59C9E0A8B7EEFA1B4C3E11
                               A57DF5B6FFF412B5F71A2323D07527A0
            stream[192..255] = B176A3DD2C02C3F858A0F5C0A247578C
                               9985A2B6401CC4BAF54FB3DD2FCDF07A
                               987AA528366D17A013D67FD3C705E157
                               3259EB342CB4741D6BF59066D70E5B6C
            stream[256..319] = A569B7591D79555C3DB66A552DD38D4F
                               617A680972E8CEC4DEEB23CFEEE112A1
                               4CEBDBCEE70993C1BBE7CACD334F1D8C
                               756F15E3D36EE82A8D9824959573CEDF
            stream[448..511] = D2064A8B35ECE7717BDD68AFE71108CB
                               4E7729FF3469818752A4A39616369BED
                               53E5273E21D453B61EBE8D519B0023D6
                               15699D42F5135A7BE7A0222C02EA50CB
                  xor-digest = 27ACC332A47E720F902945AA735026EB
                               9DB1C07CADA36F9320A3875E27C0519F
                               6E542673B6437FD44D58FF70C1FFDD77
                               1232AFA84AAA3A0132CEBF47E9ED67D9

Set 3, vector#108:
                         key = 6C6D6E6F707172737475767778797A7B
                               7C7D7E7F808182838485868788898A8B
                          IV = 0000000000000000
               stream[0..63] = F417C17C2094CAF58DDD74601A53C17D
                               5E50B247A730553E5157C5CA4486E5C7
                               4A4ED2F006CB89197DE1886A86AA7E24
                               56F995B1E044F7569FCC2C18BC65B006
            stream[192..255] = C717A0CB4BA11F522FC1D408878AF2EE
                               7CD420430A4B768AC523B0E8A2182473
                               861055EBB25D223E4C7DB61469939025
                               745C4943ABB8BA68C1D199CED88D455C
            stream[256..319] = 81794132A1B7B56B67D8BD0432405671
                               160FB5216146C20ED5C0CB8BD63C08E3
                               5C045397C7B7403AFE0D7C05B5840A06
                               1C67F4E6D0F0C3F7AB492D125B8B1101
            stream[448..511] = BE36DFCD4760BED0A1547D5A22FC335B
                               E6AE9AF6AD45C1E3136014251201C688
                               C0EB6C80BC55B58554D3F795581C603B
                               583B6453DA87BA4EC5D9A70BF05A31DD
                  xor-digest = E70D37AB9A241AE19C2098E5B2F9EB32
                               F2C5910EDAAC4C404D72F2B37249EC02
                               06552F2F46AC890DC3D92061E248179B
                               2FAE062C44FF451A573F25FE7C64828A

Set 3, vector#117:
                         key = 75767778797A7B7C7D7E7F8081828384
                               85868788898A8B8C8D8E8F9091929394
                          IV = 0000000000000000
               stream[0..63] = 0867C15DED499FA15E38A7EE81BED09C
                               85C44E90D325C4081309279C79C494F8
                               CE5EAC8DCC62C9ED5743C43E8DA21F5F
                               4B3ED70D8C59EAE886B2428E5C0458B3
            stream[192..255] = CFA60BEA15AC7EA54C2D251B99C13F2B
                               6CFBA54F5C57E1E1681CA3CF9F94E225
                               5C3F9AF9CF3181184FD5A8F6FDC728D6
                               E43817977B649D1CC4742FCBFAEBFD1E
            stream[256..319] = 7B0DD2AB4EDA5252E8E28D1B20929286
                               45F148ACF54C4F8239C6209E96014F85
                               64151D9338124301DB25AF2369F5FCA7
                               475EB55CCC0386CE29FA615E11BCB45D
            stream[448..511] = 115417B139716618C4E7865B44739636
                               4E62EC23201866EFF48D99DC4F2F98E8
                               4B98581D955A577D84B75E446CD41FC9
                               A06DF6FAD36CE86CC43D0F24030A8D12
                  xor-digest = 1E6F6CF9B80191E1A0A0EC4328174271
                               6226A180B32C2A4D87CEA1FF808A4E7B
                               BF2A85DA581C31235C2DE3C86542678F
                               02A1198FAFADC50A1EAC40E95BE6ADD3

Set 3, vector#126:
                         key = 7E7F808182838485868788898A8B8C8D
                               8E8F909192939495969798999A9B9C9D
                          IV = 0000000000000000
               stream[0..63] = 327B9777AE40E57CA0C9FB19615F9902
                               E4FF711EF44375234CE7616A66B66BB4
                               B6837A86DB7A6E4D4AAC2B996644706A
                               EAF0713ECFEF9E0D123829B5396898C2
            stream[192..255] = B8B1C65FA18B4009FCD2F9DC6A4C5CD9
                               050F812F8D98B5760E7844677DD72DB9
                               B7EEFE64AE1E363CEF56CF80409E992B
                               C550118D605A90800CBE666288374584
            stream[256..319] = 6C99751BC6ED9DB85B292810979B4016
                               3BE35BEE7A3D662F906CF0393CDABC52
                               31861520581756B95CBA55A5E779B38B
                               C75FE5DD568C77EE397CE56DC09826A6
            stream[448..511] = FB113E907F6EB8FAD09BEC825207BE7F
                               8078AA354C6978540F922BF185A93FCA
                               C3DB0ACF50CA48C7E237D67AD2FE9BB0
                               1A5BFBCACC9FEFE7C7C56506602B39FA
                  xor-digest = D0E390289A19FB5D067C4BD19D4204EE
                               0AEFA3C13172AED8766357802B46C883
                               E584EEA330F74E10D94466A238093DEA
                               C13C98F139B96443E0FD0EC778ACD673

Set 3, vector#135:
                         key = 8788898A8B8C8D8E8F90919293949596
                               9798999A9B9C9D9E9FA0A1A2A3A4A5A6
                          IV = 0000000000000000
               stream[0..63] = CB09EF82D2180BDCFC098C9B8F6C1F4D
                               2BCEF48FB4EAFDB57C35B09CCEB62349
                               C09006CC68A53BC1BE4257383091C981
                               2BE8B5FFBFA0FABA69E45732A8EBE011
            stream[192..255] = 8871EE16888B8C4CA6228AC60223EE31
                               323219056E38FDCF0E56320F11D06E1F
                               48F3588A9776E9AA884837B9D648E9C1
                               71C1B5AB947E7520473ADB18F3F417F8
            stream[256..319] = 7DCFFC61967BCE7E7B04446CDDA8D6BA
                               C2F34418C63460F8AE83A56DAD434396
                               6DE060CF13FACD50220B1F19F2C0FFDE
                               08660ED68A1903BD2EB879465F8ED0DB
            stream[448..511] = E7140E9E6E9844DD22BA00981BF850F4
                               BA1055D2615EC80E76201B7BE0C20EF9
                               3209330A43E6F1F9F306D667AF2E3533
                               AD59623987E9B0522250FF86EFCF6207
                  xor-digest = 4F572881E72B2D303CF8DD072A8F01C0
                               3666FBCC8DBB69C14C6448112D42A65D
                               58AC5C398C41258237D185473253FEF3
                               B4D60E1938EB15F48DAFE5AC79ABF26D

Set 3, vector#144:
                         key = 909192939495969798999A9B9C9D9E9F
                               A0A1A2A3A4A5A6A7A8A9AAABACADAEAF
                          IV = 0000000000000000
               stream[0..63] = AF4A88B6D7BA5444F0804F6F2CE87EB6
                               3EE7E623001307852D51F79A12E6BD6E
                               3107D7F76F48A6EC7BAF8FC53BBAA86B
                               5F4F9E7B04E29527232FE800259C525C
            stream[192..255] = 94848BE2392AFB3684A2ADA5FD4ECCE6
                               1DF6501700541A1703631FC8BFF13093
                               67F6EC80784A5ACFE444DFFBFECE61FB
                               9DFF5A5409DDC1548FA8D679DB6A9757
            stream[256..319] = B0A6606F5BFC5C905D00E25F2B7CD2AD
                               F28775D2AA12DF98F3DA93D2A3B78423
                               9ADFB4E390FC6CD781D0337A518C0AE2
                               BB854FF28EFA0936AB6DDCE49D78E6F7
            stream[448..511] = 658C2DF3F0BC5EEB68F3D14250414BDF
                               692BFC1D567C6C51AF7831E57F3B427A
                               F0606199418BB6117C80C99EF8F28FAF
                               4129FBFD71A5D76E597E55964434F0E2
                  xor-digest = 6F54D491451FBB37E5D1506437C032E0
                               8677EFCA3371028C868F33089EEF0F28
                               B741E89056C7275852132E5FCA348066
                               7119F0E79AE4CC526026AED9DAB42030

Set 3, vector#153:
                         key = 999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8
                               A9AAABACADAEAFB0B1B2B3B4B5B6B7B8
                          IV = 0000000000000000
               stream[0..63] = 95733E2F1D8E8059A48968A151F73E39
                               6184A263525B81BD6096B43A92E50C21
                               ED323850FE31DF4BCBB8F911D8E7C54D
                               CEDAEDD354F8BF8B3B5BE7C2FBF84231
            stream[192..255] = 0E889D735D17BE1DCA0BF6E82E51E2CA
                               B2A9E80F7EE3BD8DBEBC82DF4A3933B1
                               98248EC7CF944788F22CA93512F25A33
                               D45635B09E09742B381B5A4FD15717AB
            stream[256..319] = CF290909510205343FB2F9DA9A5A8716
                               7941F72E206AD184F8FE3FF30400CB53
                               80E28DA8E4BC18D79A0CBB120F5FD880
                               514E3272C876B13C0E7AB8DF2B594FCD
            stream[448..511] = AA509B41C5443412ECAC27226A63A655
                               27C3FB7363A1559824B16EEB9171EC0D
                               7C7E577592701B6D30621B2F3FE06DF7
                               4E9DC64EB0B4E95CA62FBA6758C393F2
                  xor-digest = 48624F9FA8E8067FDFA3BA33B7859978
                               01C1C7FCB58A192E7DA4551BD19D8C3C
                               1A4782A5479715E9CBF8A2CC08F19B11
                               AAC8F58DD9F4CFCF1CDEFDC755E2B221

Set 3, vector#162:
                         key = A2A3A4A5A6A7A8A9AAABACADAEAFB0B1
                               B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1
                          IV = 0000000000000000
               stream[0..63] = A27A589019A08049661CF58BC71C9AD2
                               B98DCF1683F0BA96E6AD59E4DA61DE24
                               205B68B7926E7F8A8F2C549C2E650544
                               03CBD383BE73C75796B04EB476170501
            stream[192..255] = 281F8ABD5D7B64E9CFCE3D26DB934BC2
                               7E7E39C357A32D1B860897549219315E
                               190599E5087AFF3249076100B192450A
                               5B454FA8E18D6D012F46E52A50AE4A74
            stream[256..319] = 7647AB37D6350AA67973353A1F6AFDC3
                               540B7EB247FE3F02336108470B589EDD
                               0C8BB044025B8212E75F7A86ECC19206
                               48C92AE445EF0E5F98CACC3A7F57EF62
            stream[448..511] = 98DC00FD1858D6E503916B9DA7249502
                               C4B201EC03DD86EB77C0A3FF4290175D
                               021B8297278BE49B797E685AD628EC94
                               523F8E5B5AEFBD7952586DC97BF1BAEA
                  xor-digest = 99DB035B11F05E0E8906FF868D08DC65
                               186AFD6A7A6E685062BADC746B9E4C10
                               6BBBB19ABBBF89B66AA3ABECCEDA2129
                               862975A3A935C5D2604CA083FD98C90D

Set 3, vector#171:
                         key = ABACADAEAFB0B1B2B3B4B5B6B7B8B9BA
                               BBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CA
                          IV = 0000000000000000
               stream[0..63] = D3320C4FC079F31269225D8CC7137059
                               6B3A921E7EA42F52D0A891C250625421
                               02358F00B9FB35C579488414B74EABAB
                               631AF7F3E2E73288C5A893030DE610D4
            stream[192..255] = D1EB9B01EB5B05939519E0F9416B8508
                               F4E2C6A75038E16BA347E90E4AD76148
                               CB0EE152776D876FF8640C98829E3DA7
                               DC45765707FB94089D91B8471C9DC087
            stream[256..319] = 69F42A91588B78EAAFF0E48E280FF630
                               50A1CDBB5B2F5A2FF285BC39EC606063
                               67287A0035D1855DFF8A6C2DD9F35088
                               A8F2DA4DC28CE4520568577BC966EDEC
            stream[448..511] = 2DE18A3614529AEF470D0DD62653FACA
                               B69452151125D7859AE6966CAA20A72E
                               6C3446A78BBE769387F74FF41BDD1BA2
                               9494036F2D1112BC3EAD251E421873EB
                  xor-digest = 09958596ABF053D368FD158A8BE9DF0B
                               66142B4DB31B9CC9C2E5081B237BDF1B
                               09C5723DACAC0558D18BFB46694BA3E7
                               9DACF93CCB774586FB5BD8751988ABE0

Set 3, vector#180:
                         key = B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3
                               C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3
                          IV = 0000000000000000
               stream[0..63] = 01DA5AFB731512CA67F33A9CB49ECD96
                               52E9E29E5FB1C2C9FB64318D964BDCB1
                               5105AB5ECECB3CAFDB97730F4748F19D
                               9B4C823752FE47770317939E99DE8036
            stream[192..255] = 3E096A7C107536127451F1754B26C236
                               5D258056510E15CFA379956D9C7802AF
                               C099050654D4C30D403A61E2F8F92807
                               8B57DE1A928F65138FC97B64C0271406
            stream[256..319] = 10D7B5BBB3574860532962464C27EC17
                               35DF90FBE3ABB38E6E265C3C5CF90FB9
                               0EF97238AC7B3565EEF84D32F4A99ED7
                               2127B83A0103A893C635980E0B1F8FE5
            stream[448..511] = 0ECC0EAC7F18E0CF8B08155430C19322
                               113B6A9665738F783BE47BA6DDFC74AD
                               6D8CA697E7492F6E775F3D652EC103BF
                               CC00CB99F00481B89082E748C31C86A2
                  xor-digest = 73941DA6A3CBF15F8AB88BA53884A105
                               7D9A3092AC274294100474D89BB032C5
                               26D57C3A42956B45C4517C6ACFDE293C
                               39361A3C7B0755EF78ABE5018A9CDD8A

Set 3, vector#189:
                         key = BDBEBFC0C1C2C3C4C5C6C7C8C9CACBCC
                               CDCECFD0D1D2D3D4D5D6D7D8D9DADBDC
                          IV = 0000000000000000
               stream[0..63] = CE4844C64AF769329745E0283CFA4062
                               F85743714AA1A8921EACD6D9C83AB1EF
                               25CE0B40DACC9A009C3F5B35CCF0BDD7
                               69EDF975A8B6AFBB2A21A0BF8A8D01AE
            stream[192..255] = 7BB1B235885507C8195B59DCE4F87F98
                               44A55E37BB4DA46D61DD21C68424AE37
                               360E4F1FA1AEED7B4324A746171D3265
                               D987A6FD23C866CDDFBC670BC0D47631
            stream[256..319] = 083B1FD1E84B72DA7EEB0F7963D6C44A
                               FD12BD9C442E14A56A9B4C0EF3598153
                               E719C21DEABA2B40D6CF30BFC2A321D2
                               E53D6FD09154D4FE99DDEC2E0F10D47F
            stream[448..511] = E5589173547F4707536BA8AFA3286FBC
                               C6B2FEC43E6966E3030E560D8CD9B20E
                               138D38BD0AEFB9B7A79336BB8837FD51
                               7A8E98D529F1AB2974C4009A19485332
                  xor-digest = 5263179028C8DC5BC5E1AF65AA51B07D
                               4425B164A11EDE46C4CDA4715B269386
                               CBA943C5BBA3083C8485B472B68C1B73
                               4421457A9E5718D7115D94C627D09920

Set 3, vector#198:
                         key = C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5
                               D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5
                          IV = 0000000000000000
               stream[0..63] = BF3F28F4838DE2A352EEE91E447E9440
                               840D5D34DF25A03FB6EF26EF8E640B4E
                               533AE200E028BD67A586935DFB203D0A
                               7D8DEDC9AD57292A67E8EC8098BE460A
            stream[192..255] = 1DA6FB724EE6C2DAF2AED86ABBDCD1DD
                               52A935EA39297AAFAF0E9D05BFADF1D6
                               30DACB531C48305CA617506F496BB269
                               8F58892AB489639582007A44E94EC531
            stream[256..319] = 9EC574D70037EE7455013FCE3A3E252A
                               05FF49AB501DD10FAE97793AC7C54031
                               FBA89060D673C88FD2661F7A04470D06
                               7536FFFED49EA5F285D62AED5C8A7A2A
            stream[448..511] = 25FD505F53027E3736F2A84290EAE81B
                               8E8212AF289CC6EFAC8388FB66CF5F76
                               15904A3A8B5701590A0FC47B02DFF16C
                               CAD62CEA5416BD8AFA3515D589D048D4
                  xor-digest = 00A9BE6FD2C4BB3D44631BB15673E4B6
                               467E1F3BD78D4D2D3C266A2428420F36
                               11A7757F2715C70D18863E82458F3A18
                               3049C6DF7C9120A7AB706BCC8A6DE966

Set 3, vector#207:
                         key = CFD0D1D2D3D4D5D6D7D8D9DADBDCDDDE
                               DFE0E1E2E3E4E5E6E7E8E9EAEBECEDEE
                          IV = 0000000000000000
               stream[0..63] = 5D68C95D62D06896065752A27290F788
                               FEED275ECB6447A00310DDF78DBE77CF
                               83AB043340AA7DE7D512191B0B38B3DD
                               25271D07C3B3D3182FE2A88260AA724F
            stream[192..255] = 611C102D9E6A3DC5E230BE3533F6830F
                               3FDF4517EB9F414DB5C9A8364BEF5590
                               5317B7A8ADE564984DCED895ACA19C28
                               B38751E3BD5927F0FA95640E1DA044FC
            stream[256..319] = 65617CA477EE1522A5F5B26D2CC7E91A
                               93B744DA97BA28A14EAD5EF15DB7A32F
                               1D17A9C64D1BC471242CFF40EFE9E33A
                               7C4B7FD01810314C49621EA1700E7690
            stream[448..511] = E127D03FC9A331ACAD2B0F51532C3CDC
                               F094801747A23CEE9C6769578CB18FB3
                               8EF4A328D50ACEC17C4B33FB48DD8863
                               F4395B9C0ED139F312F0128974C29FF5
                  xor-digest = 4C2395D5F8A3B71781E513B2CD19E8AE
                               B031D176C3284BC10FC41BFCCCECFA1D
                               426F125BCF3EF6E8F12CB39A4278A412
                               1B2384F31EF8CA3A8E643545F5AA9529

Set 3, vector#216:
                         key = D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7
                               E8E9EAEBECEDEEEFF0F1F2F3F4F5F6F7
                          IV = 0000000000000000
               stream[0..63] = BB1874177DAC85AB55895D51F9FA009F
                               2486B9CD65EE0D1F8B15AD3D3EE17585
                               ED8772D6E05F766D6E7200A53DAEC7D1
                               68A048FC952235145CF223214B016316
            stream[192..255] = 82D9EFB1885B9B5F244F9CC6BA84A953
                               1C641EAE5FF9923122BF738C4EB7A44E
                               7270FCA689A385F05A02693B3EFF8170
                               9AA9A2546F5865851DB9741FA4C73E51
            stream[256..319] = 84FE0056F5F896A91395E7477BDD1101
                               9B5ADAB25AAF626D86EA4E2341F3A0E6
                               7CCBB10283C3078587C1AB41A4E0026F
                               E4EDEDC19C30893C3596B1323694A0CF
            stream[448..511] = AFD139C94A2EB414057012F68B6E98A6
                               04605A5DECF6A502A570AC530C2AFACE
                               C987E470855125D1B2CCDDF782E90AB7
                               D2D21F202447E138886746ABBFA87591
                  xor-digest = 73834176FBE16E8A046CEE2056BE0556
                               41E1EB10AF31C0CAAB708B5A0B4BD158
                               D158145B351AD1D8911376A98EB157D7
                               6F4CB7D471BD466DA3CE8F665EB0261B

Set 3, vector#225:
                         key = E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0
                               F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF00
                          IV = 0000000000000000
               stream[0..63] = B3613AF18B5CDCB42F7C1D4E3A06E42D
                               F8D2EB01CC7A176F821AB7603146F557
                               8B8A5FA1B8CD5C4E83252E35DF206097
                               B3BDAC7F39C15C5E8AF9A604406FE41E
            stream[192..255] = 946BC7E1C2EF60D94C08CC8CE421BAF2
                               8A15EEA34D3CB17FEE82693B1CB988F3
                               14A78A462DF5524B67A328E2867E82C7
                               344D9F412716DDF7BAC3002CF6219CAA
            stream[256..319] = 2AF96C50F3108FDA22021898E5644E0F
                               1D4C7DAEF3A6A87A3E5B84C10D418EA2
                               E387A44459A5F3DAFBA9CF0FBD74C6EA
                               46F1F515F49937C54C4FC4C80620598C
            stream[448..511] = BFFECD78A142429D7A6336419FDCC484
                               951A0C4A89F896547834D60138AFA0D1
                               357457931F71C87BAB84161683F705B6
                               84EF5A8F8506D8C717562A0C9F99F555
                  xor-digest = 250AC7E5D470732151CF8038D31A137B
                               0D5867985EBE43FBEB172F56D9B56833
                               3592B8A161019C8CE1467EA082973E0A
                               0465B7C940419997F00B6AE05D138C19

Set 3, vector#234:
                         key = EAEBECEDEEEFF0F1F2F3F4F5F6F7F8F9
                               FAFBFCFDFEFF00010203040506070809
                          IV = 0000000000000000
               stream[0..63] = 0F715A05810CFDB16519EED13BBD80F4
                               3A026D941D01554A8A881FA47D4206FF
                               972C46F955AA6AD19E8E9ACD21D132C3
                               8ABBD6AEFD705E3FFC0E7662E1F35576
            stream[192..255] = B54A17D617EB0D6F49024484693F5274
                               BBD6AD0B3C92973DEF35B9467DE8560F
                               D8540FBD152169A385556AC617198167
                               5FA748BC35E0D1228BAE6F29753FDA65
            stream[256..319] = 0BE6A98F6BA8DCCE357C62148A59CA54
                               1FFFAA5A3A8AE7A22B1EE8B7B17205A8
                               986BBBA25F4046680F4188131B0FED5F
                               C09849A28E605F4E32E024E8BCC18D73
            stream[448..511] = 1C07D9B563C2CC4996988514BCDC47E2
                               F9211B5CE8860BD7B2CA336AC345A960
                               F3BD198C08DDFE5C215719A628CA7EA4
                               FD060DD654229F3D44A29B05188F450F
                  xor-digest = EDBEC56A789EF2CFAA351B2DD011A02F
                               F2147CD06CCBEA7E9DB53CD00AF70DCE
                               4AAF3B75CC6A97487CD2837D6E81DA20
                               7568DCE27B027BED29CDD32F296FA317

Set 3, vector#243:
                         key = F3F4F5F6F7F8F9FAFBFCFDFEFF000102
                               030405060708090A0B0C0D0E0F101112
                          IV = 0000000000000000
               stream[0..63] = A909E3526CD204AE2FE6CC8C2013A733
                               5A76FF4B9DACC7427632440C2B949151
                               50CBDB96BB6208F0AA5982DD6F7D8141
                               7800E433390F640A4F7C975446E5DD45
            stream[192..255] = B96665B5E1A97BC7C31F04202903A19E
                               7D593DF8849ECD8F0E0BEDE52530C463
                               D8CECCEB18C53543412962DCE00FBD8D
                               8BE5464749AEBC92CC0D0CFD353BF8A8
            stream[256..319] = ABA2978EBE03D0D59BA09E7539F25E82
                               AB26376E0CCAC02BCBAF54D9B9B34322
                               6AC9F14268EE9591EC2DEE2E78DF107B
                               A735488688E3B0608D6D100031EA8C0D
            stream[448..511] = BBD78EAD8DD6D88FFD2CFE4D2BC86DBC
                               2DBAAC634F6A7C4D1F4451C33AFBBFA4
                               995EB25998291D975E21C9D5D42434C2
                               45EE6FA3EDEE0BFEA31E39EBFB644520
                  xor-digest = 8EE841D7F30138B1E3E353FF8CA8F038
                               040508DBF641C4401A8E19B18AF663C8
                               82EA0B5AE66035865504A716F18ADE89
                               EB5B5A6012A0056D650E412F1933CC21

Set 3, vector#252:
                         key = FCFDFEFF000102030405060708090A0B
                               0C0D0E0F101112131415161718191A1B
                          IV = 0000000000000000
               stream[0..63] = 177D9C9719AABD1CF9E0CC0C20367D5C
                               1795AAFE99AB1E4FA4509E088024F502
                               8D2DF7DF44E6A3FDAC343354C2FF78B4
                               51DCB2C28482E637E4AE4352BBE4034A
            stream[192..255] = C75A98394355CFF1E940286FC4AD7800
                               40C1E7E5CE1885465A98BDBAB1DD4611
                               4D532B1FFF605E64A85D1B65CBFC7F26
                               E21D06D6511B4399801E2EACD0CE8F9D
            stream[256..319] = F8F9A9EB1A6663FF782C4449ED4E9BBF
                               27E8D70EEEA2B12514F0DA47029D3DDE
                               340B479CCA5CF51C9885833FD0343AD4
                               3F275A6635CD0022B69C05D312B86762
            stream[448..511] = 310B1C1055CE26E44D0AABD93C40C26F
                               CE6505A4CBAB6A21FE4BFDFC982E4A5E
                               11F1ED02BAC8E8B33781EC1C564C9789
                               9896D1D9E739C2A7D23F0999E4090866
                  xor-digest = 6E7E74B60DD78B4B36FC745FFC4C9CC6
                               EFD771077BAEEDE737C51A030188D700
                               498190B6A0C8700BB94B3A4E6EFAE776
                               3C7EB049C5A84DE8E99A0BE4CB90B524

Test vectors -- set 4
=====================

Set 4, vector#  0:
                         key = 0053A6F94C9FF24598EB3E91E4378ADD
                               3083D6297CCF2275C81B6EC11467BA0D
                          IV = 0000000000000000
               stream[0..63] = 1F1465B6F9EF7D3D2B3409220F2D07C2
                               98680C9E2B598E8E8D77A1E04B37CE32
                               12F5ECA98D70E4C701495695127611CF
                               A5D2BF93707B33A66FB78401E77A5CA1
        stream[65472..65535] = 95388EAB63F86B786E8F66ABEF56B220
                               4861087F93458CDC41439832616CCC91
                               A1A2746AD8F1C77DB3A5734BE3C41951
                               C9816A2282EEDC3E7D517AD4DC9193BC
        stream[65536..65599] = 72ED7F020F1F2B83753B25912ACC1E61
                               E0EE3546E78A5CA40DA02482F31BDA1E
                               C3EAB316DAA9B1E1ED1770FC46ED3E0A
                               6E0530F0C0CBFC04EDADDC40879DA1AD
      stream[131008..131071] = 4A59395170EFD1B06C53D494AE6CA1C0
                               A8CA1E43F7FF2EB1F3AD6681D2A625AB
                               A05D28A1CAF0814198BC0F61D5D8A375
                               57CA253AFC2F45A326CBE60B4E9D9DFD
                  xor-digest = 5F7DE14883814992002FAA1466A63335
                               F0B673217DD236BBF6EDA3BB80B0C32D
                               AFD116A65F8B97AED60F1D8E7E7033EC
                               39B2CFDE48ACC0643BC4106EBFE13957

Set 4, vector#  1:
                         key = 0558ABFE51A4F74A9DF04396E93C8FE2
                               3588DB2E81D4277ACD2073C6196CBF12
                          IV = 0000000000000000
               stream[0..63] = 04E01132C752BB8CC398FF4B2E39BC01
                               9ACB60879A7104DF4941FFE4DC12789A
                               3D3027DE343A5E473A78D4A57F191E2B
                               9300262A74ED2B57EAD32615B98D59AD
        stream[65472..65535] = 53348E80938BD3FF9576957989240C42
                               F14359EAF50E46D232F2E3D429C8724E
                               A71302A288CF7A0197420087CA7B974E
                               6AA766899F95748C3E45E649B9B99238
        stream[65536..65599] = 3562F5EA35E66D5C89C865FC074E73D4
                               8D16D73AEFD538E78986E49A22E1753E
                               6FC290EC0ED6DC29BE88319756D584A3
                               FEA8A9F83AE078AAD17E55C0C28BF143
      stream[131008..131071] = FC09CC3BFBCC21CF3102BFB43DEEF449
                               94B50E7B005C3D4F70304BC1949B8940
                               16376091484BCCF747199F9F700A7277
                               201E9C46F6A8679DCA4386990DF997B9
                  xor-digest = 1AB900B643FA352027A51320FCDA3428
                               1EDEAAABFE7CBC326932566312A85150
                               6BC9A380CD2AD7B340E6F51D1719BC44
                               F5ED3951A04491FFCE0BB0E9CF80FFD4

Set 4, vector#  2:
                         key = 0A5DB00356A9FC4FA2F5489BEE4194E7
                               3A8DE03386D92C7FD22578CB1E71C417
                          IV = 0000000000000000
               stream[0..63] = 378AF306330D8023A3284FD664E84DAE
                               CA69FD5970A47A4596572C6B64102D04
                               9F0E5A37E0396F61D83963C68404BB83
                               DFBDAB1E52AAA14CC665775AEFFE7AD4
        stream[65472..65535] = 22361925B3A9F33250F22974852411B4
                               57D090592C99E736F67E1B72509DF404
                               5E9B5C9D88087900CC9EC76537212892
                               332414F646AB8F17D4E22D8908B7822C
        stream[65536..65599] = DEF1DA8721D35E1B0A4E03E127338CE5
                               AE733FA60C76E4B73350624431BFD562
                               0E284C6666D0D4E81C179CD741FC0893
                               C797B80C0644D3AABACDB617083EADB1
      stream[131008..131071] = 8EAC42F2CCD8C99F6C4427D5DAFDBFCD
                               094C83004181D5C76E7A064FA68AA65F
                               83B41ABAEFEDB962802EC933A18F289E
                               B128C5A1B9B57061E09B3505CAD9FF6C
                  xor-digest = 91A1BC3C1CD27F1579B55E9198135E36
                               18649894E5A3877EB54E351D9870A3AC
                               B8FFBE945FADD27E6AB8111666B992EE
                               55BE43D41F44A13659B9B6C9307506B1

Set 4, vector#  3:
                         key = 0F62B5085BAE0154A7FA4DA0F34699EC
                               3F92E5388BDE3184D72A7DD02376C91C
                          IV = 0000000000000000
               stream[0..63] = FC6AA05E99E4A91A2447F41FA0683DA8
                               82E1A6687CFFF5C7039DE6FA7FD37DED
                               25DA5E60FA01A277F8B905FD71C1702C
                               5A376AEF82B99CF88A5624F7C7F2F633
        stream[65472..65535] = A3E9D58192EBE39CB80A3F5C1F32A798
                               C81721B6F7290EDBEDBBC897C32E502F
                               F4F2A4304CB33642E079E1548BAF4E03
                               7EB64D2E4CD69BB145451302B349DD56
        stream[65536..65599] = 174B70F75C49589AAA5CDC484A6C25D7
                               170A0AA32E68AF7CBEB80A885663B48E
                               BD377912F50998A34A993088A67D25C5
                               70C4E2CA6A1999445A10C8AC53DAC991
      stream[131008..131071] = B1D27DDFF03A06B18F2CC3217E184E98
                               493ECE07B9FA3A176FFD1436BD7A1C87
                               5AB56D30B4A10F961A9E120F95312334
                               B6AB55A3F3EFBBF7AE2AC53AF4E7D6A7
                  xor-digest = 59BC4CEE8169ED61E598D257EEC7AD5E
                               D5B44F794BB71E416FAE8C40D17F110B
                               A7338A498D8A4E16235791CDCB7C561B
                               2E0290362CC9E555E787B5508A24FE7B

Test vectors -- set 5
=====================

Set 5, vector#  0:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 8000000000000000
               stream[0..63] = 172C5192CB6E645BC527B2539A8D7412
                               67C191CC59B5EBF40BE5276E4EECA9B4
                               B31F3B434B2AAF28ECF61C58E927F7D0
                               A021D87BF287C9C6D0754A5795C4C6B4
            stream[192..255] = AE2C5824C80C782B9243A0B6B4C67B65
                               EC43D2812C6ACFA26B74BB515798FA4F
                               639B4C644853FA9D1F8AB4EE052DB3BC
                               97334FE7A564284926FE40AC9BFA970C
            stream[256..319] = 14AE8E3920C80738C3981B839798876F
                               5991F1BD0CD291FFA0405E8B8DBE9C2E
                               B9DBC808CB3BBBA11E8BFB7F6ED08AE3
                               D21358F5A8162F1A165C4CEF7A1AC7C2
            stream[448..511] = 40E6619A9E4CBEEF3D99472CC523EA3C
                               06E71DE9B54CED37BFA32772C99659D6
                               759AD3890D1FFF43CD560306BA39967F
                               1B6618F7B021144E20A9DAC7F45B9492
                  xor-digest = 4B5CA8587A2ABD6943CB7C856AF802F5
                               176051E66CD843E7272E260B46CA76C9
                               E7AFE2D681B65741C61E9AA654FC5A31
                               3A28BD50B1DD0843A72FD47C231B6D7E

Set 5, vector#  9:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0040000000000000
               stream[0..63] = 8F63797D01A6C6B2D77C9413F5D31BCA
                               4C7CC8BB21F94D45C81760EF9D91EE72
                               50C4E3C6A15C600F6EAB0AA236D97B9F
                               C144AC7E12CAA6CD7D5838D56F98657B
            stream[192..255] = 0B1DE2F3711FB2A77CA531B35469CCCF
                               11EAF706F0F6BB483EA5901E256D726B
                               3117AC59BA673A36776E0F396859BCBD
                               50EE61FC1432071AD4DF281164EB8545
            stream[256..319] = E069B0A27639F94E54906FA4138163B3
                               F62560E0D93CC92D314276A3C1D707B4
                               5A4847DEAB9ED6E55F383DDC0E89B452
                               8EA98AAC4010DE942C1567E1E8ED7C19
            stream[448..511] = E55385BFD08171C5E13A7767CF3FD62B
                               ABDAEFACE62E54E18E053B60FD63EC95
                               69E85DCE892DF38A1BBB832B3E431A82
                               A7C57603726ECFD79A2770C402B918CE
                  xor-digest = E0075B464F69E8A0750783BF2DD8554D
                               0A03AE993D4A1744270F398F234AD159
                               A7C9E892F88EA53064DBF4A869AF3347
                               533CA1F31AFC9533C5100425DA6A043D

Set 5, vector# 18:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000200000000000
               stream[0..63] = FB7F2990D5CDE4AB446B5EADEDF8D8B4
                               8F729EA00C2262B16CB679472C99383E
                               1994B8A08C8815548F739420F1C061DD
                               BD322248E3A923627D4EEA41F91EBAB7
            stream[192..255] = 9EE5558CF5E0948E8E3CBC3E9703E698
                               526ADD94CAB6D1849BDBD249F369BD6C
                               A2B650FCC023DFD7098F4351106E67F5
                               3A05639435A36BAC5FFD5DB6869B6158
            stream[256..319] = 583905D6D767FA9997FEF521D15858AB
                               DAC052C08CB0D9BB80F58AF0929EB8CA
                               459AA99282439BEB1E9D8AF7E224CD1B
                               18C2B53DACDA84007FC92A9675B6B61B
            stream[448..511] = F6566C867579E835967F8797907B7158
                               C5B9AFAB9112C93D6BF6C1A0C4E1AA60
                               3A1BC71DDAC476E9A5F647C3104AEECB
                               5F838EACD6DCB0BC33E4D2A28BA25A87
                  xor-digest = 2AD3B6F8335AEE45373C225C1E8A396A
                               43E5BF7F3103527044EDF4247C77A43E
                               484C55738C4ED787427DBF118616E080
                               740D5EE2714C2833C9AC4D0133520615

Set 5, vector# 27:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000001000000000
               stream[0..63] = B2FC0377EEA35AA3AAF01ADD5EFCFF1C
                               FE60265276CFCF133227C29DB57A5BA0
                               1BD2BF9E89D8886DED21CB0D5596E64A
                               4C4E56A476B8FD40659543706880A564
            stream[192..255] = 92C68371198642E1B604E35E0F761D32
                               B314F371FC992FA301908C597A0816C6
                               F8E905D7CFAB1FBB781BB2A8D0273394
                               C250C141DDF563BE8667C12407114458
            stream[256..319] = A3B3F2B537B5D14D9C5D3C6908768D29
                               D0AB912161D07CF8405F1F817BC03C6C
                               08A7783CB19F2A482AF90878C3C3718F
                               1CA490BC6C822C9C421987A6520F93B5
            stream[448..511] = C13E14A58F98700AE84D829903A73FFA
                               12A518FF70DD9A646C142955626EC924
                               583F037E9A068FC61F08855DA46AEBE8
                               51439EEC2664E24317ADACEEA448A0F1
                  xor-digest = E163E45F61BE8A2BCE80012F85E8B88D
                               5E030B7F780644D4D9317CD3AC7C9016
                               795623569400FF041BB6FF2791F0963D
                               B3FC44ECB6256EE33CE18E15365A38AA

Set 5, vector# 36:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000008000000
               stream[0..63] = 174C5FB5D06A1697FBFDE5C4944C66B0
                               E693E31544C371C4672833EDF54E376A
                               ED0954477C953D3A99B7C8DE28786A5F
                               35B3E24777CCC482CE76E33EC37DD27D
            stream[192..255] = 9C306A235BE1BABC7B4CF43AA82AD588
                               EBA9A732D6319F046BC866B2FC4A286A
                               E0FAB642168B77A7469831955A70286F
                               E96B36CDF21F17EB2CC0A6FBC03EA70B
            stream[256..319] = FBC4B1347B71CF7FB8D5561C123FFF59
                               730A2EA26AECC2C3CB69CFFE019C48F3
                               85FF6A2496EF1BC1418F0CD51A23084E
                               EC6456414C6280CE0ED1DAD55C2777F2
            stream[448..511] = 5FBC2A0F72C313E1E7B1DEB8181DEAAD
                               1FD832FCA05D8B41F86C309B65A21502
                               5F5C349BE749969CB4AB25D69D0CA3D0
                               6ACC72613E68AE20F8E868F9D0D03D5F
                  xor-digest = CD90497EB1D63C244CF6F1CE9075594D
                               02C12ACE88A444EFB40B275317036A85
                               73686B955B5FEE1436042298E7567D15
                               8A2E521D5BFFDFB4ACB78F04C67F56C9

Set 5, vector# 45:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000040000
               stream[0..63] = 5E0DFECBFC8A0F838F37D16B2931EF1D
                               0522930B3C84F7BD153A51BC11AF6286
                               EED716F8FD6260F9BC378C247BB93DDB
                               CEB2673BA987FBF376593C7462D6DF5A
            stream[192..255] = C1B1EFF210DF58DB48E9FB1B9AB4B011
                               2078BD28DED5A30E43FFE76AC1DB0690
                               8ADA2EDC57E0DCCA377E1E878C5FF3BC
                               1EDC55131626D8988B37EC96F04DDB35
            stream[256..319] = 45B760A94E1755A79E656FDED62D20AB
                               FEEABF0CC92807DF40EF2C6C2DB8E271
                               F559B38344F45F6D34F7C001E5CFBE45
                               98E3ABA99D754DFA0E7721E88423529B
            stream[448..511] = 9C5391341CFB7C971FD65341A9AF2089
                               7E9C037B5971121E34F036FB835E7AC8
                               C41191677E942692024787869299C47F
                               2807BBC6696BBB38E252CEC5100C06D1
                  xor-digest = FE62CF9D9F0E93BB8DA29344A6C40E85
                               253077FA0B541A9396E48AFEA38EDA9F
                               9F24B1AF19F434E83CBC6F300E1B3B8A
                               64DC64CD2C2C15FBCD062D473EB36F45

Set 5, vector# 54:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000200
               stream[0..63] = CB662B5518DD67BE68AAE547DB30E0D3
                               B515117767C06B4C30B12FCC0C9A41D3
                               3BE1BB5752A13AEA54EC1D25765B59CE
                               B60C3D94BC6101EFD927954B79F1B60A
            stream[192..255] = EA2E3C6B117E519A761FF82AED339A00
                               C37C6BA0DA42ED10DD4EE65E7B1793C2
                               E541B33B674698083CACD9BA3084D699
                               8E22B093D516CE830495C05F1FDC7390
            stream[256..319] = A42CD8A6C27B3DE8BB05A6456729A3FC
                               83E8AC7FCAD766081D24F1567C88A267
                               1B58B78BC14481B5C9FCFAF911707ED8
                               66D87877E38DFD1C260E83E814C76CD5
            stream[448..511] = 2E00E43720FC16C9689ADFA41D75FFBE
                               38B044E81963A303848DE66CAEDDF240
                               DCF164BA7A2FBCC015A24155C7F93745
                               BE13B53B986599CA0F108684517FD76A
                  xor-digest = E3ED010D96914D7D0F66B0647B5D316A
                               F7A623B7E9CCA3E753421694BC36389F
                               5244D1FBE8CD5A65E32695F5BE91198A
                               4D88394AD9A312EB99D1B6ADFB5FCC45

Set 5, vector# 63:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000001
               stream[0..63] = 51C469E3B4F9F1384612C4E52D03D714
                               CC7169177A464839AB9E4ADE1863E40B
                               69141F4E6D3ACEDCA5664D55526690E8
                               CB6F06093AAD5B8FF617EB0E42140E6A
            stream[192..255] = 723DA5F68B235084FFC5845C5D74D728
                               35D0F449A67E4A988786DAFF629C8B72
                               88340941529F46DDB8F4DA2627494C27
                               670DC1367841EDFBAAC45ED28AD5EDCE
            stream[256..319] = AD71BB3D609BC1BC133389E4A1DF7429
                               1BFC75A1D9096CDC24C882B4097B864D
                               3572AA3777D20F8696FC1213F8BABFE8
                               BF622A77E5C6D669368D57EC68E26275
            stream[448..511] = 60BA081C35F53B712C52AF8CE62A2B4D
                               AB89914C8B8B1A8F21F8DD3FB9F5BAED
                               82EAA11F24A289B5E7F6C9279BE5CB5B
                               1C9BD8CEC87C953B512461A0C8CFDD13
                  xor-digest = 06E1A7EA3D460C742549C4F12DE2DA39
                               5D10F6FB08E5A63FBDA7E8D5F3B16899
                               49E9CE174080E8E59DD7BA63D36D69E3
                               7BD295EE2D4B515584859D75A9E87582

Test vectors -- set 6
=====================

Set 6, vector#  0:
                         key = 0053A6F94C9FF24598EB3E91E4378ADD
                               3083D6297CCF2275C81B6EC11467BA0D
                          IV = 0D74DB42A91077DE
               stream[0..63] = 52E20CF8775AE882F200C2999FE4BA31
                               A7A18F1D5C9716191D123175E147BD4E
                               8CA6ED166CE0FC8E65A5CA608420FC65
                               44C9700A0F2138E8C1A286FB8C1FBFA0
        stream[65472..65535] = 8FBC9FE8691BD4F082B47F5405EDFBC1
                               6F4D5A12DDCB2D754E8A9998D0B21955
                               7DFE2984F4A1D2DDA76B9596928CCE05
                               56F50066CD599E44EF5C14B226683AEF
        stream[65536..65599] = BCBD01DD28961CC7AD3047386CBCC67C
                               108D6AF11167E40D7AE1B2FC4518A867
                               EFE402651D1D8851C4FD2330C597B36A
                               46D5689E00FC96FECF9CE3E2211D44BE
      stream[131008..131071] = 9166F31CD85B5BB18FC614E54E4AD67F
                               B8658E3BF9FB19B7A82F0FE7DC902DF5
                               63C6AC4F446748C4BC3E1405E124820D
                               C40941998F44A810E722787FCD47784C
                  xor-digest = 7E288D6B1C8C521CB4A535CC446E092D
                               B2EFFA99321A5C6E0C52DD15F383DB4C
                               4EB394C3BBA4ABA7892E96DF4A02B324
                               ABBB78AD72963BBFBF395A941D6E96FD

Set 6, vector#  1:
                         key = 0558ABFE51A4F74A9DF04396E93C8FE2
                               3588DB2E81D4277ACD2073C6196CBF12
                          IV = 167DE44BB21980E7
               stream[0..63] = C07560B3E776B471C5E2931426CAF1ED
                               3AE4B867087682CA9DFDC2BAE89350BD
                               84821CAEFF85AAC49D7435A7D9889352
                               F5279E36123F41728A14EF269FCB944B
        stream[65472..65535] = EED1BB58F90C89E05CC68B2DB6055849
                               B3D2B187B7F02F9A24CE342AF0FC47A3
                               74BD7590FBF4FD9EE59B1A381EBFD229
                               AD2A2901B3FB610812900B9230E622E9
        stream[65536..65599] = 70F0493A1B6253CC5ED3450A31CF377D
                               834BAD2072302927CCD830104BD305FF
                               59D29417B232884EC959194D6047C3DD
                               6656C47E320064EB0144F7341BC3D697
      stream[131008..131071] = D2CCF7C1AF2AB466E627DB440840969A
                               BDAB68D886AE6A38A13FEE1750CA97B5
                               D3315B84084728862FBCC7D4A97C75C8
                               655FF9D6BBC26188636F3EDFE15C7D30
                  xor-digest = 986A4F031423FB9FB0F03FF3EB0C992F
                               6A97B8E5545C6E441C35CAA01D5CBE1C
                               5854B8C2D327ACACC1B70603D7917B5D
                               A7F824B0CFE21CA857B438B940BD67EF

Set 6, vector#  2:
                         key = 0A5DB00356A9FC4FA2F5489BEE4194E7
                               3A8DE03386D92C7FD22578CB1E71C417
                          IV = 1F86ED54BB2289F0
               stream[0..63] = 512252910190D154D14D0B9232B88431
                               8CCB43819BD5421932C03A13F07B4010
                               83D789725AA9DA0B41CB6224945EDCB0
                               FB6FD7C2342235C970F64E101C256864
        stream[65472..65535] = 97967455840A4AE5C1CACE491519138A
                               A35E5F02407D4A1FE5086D35F3551EF4
                               77D9289D1723797C1A49EC26629AFADC
                               56A038A38C75881B6217FD7467255909
        stream[65536..65599] = 1BF82E3D5C54DAABCF8415F8A2A1A22E
                               868806334FF3113604741C1DF2B9840F
                               87DEEFB00723A8A1B24A4DA17ECDAD00
                               01F979DDAE2DF0C5E1E532C48F8E0D34
      stream[131008..131071] = 06D84F6A71348420329FCD0C41759AD1
                               8F9957A38F22893BA558C5051197285C
                               6BE2FD6C96A5C662AFD31178E70F960A
                               AB3F479623A444B68191E4C528469388
                  xor-digest = 1E26089DB2177310078B99B0AD543FD4
                               80F3704035BF62F70E6BAE530FAA011A
                               18FA40A6BE989938EBEC6DD26BA501E4
                               05F09A9D77E98D292A299EABF7A5684B

Set 6, vector#  3:
                         key = 0F62B5085BAE0154A7FA4DA0F34699EC
                               3F92E5388BDE3184D72A7DD02376C91C
                          IV = 288FF65DC42B92F9
               stream[0..63] = 99DB33AD11CE0CCB3BFDBF8D0C181604
                               52D014CDE989B4C411A559FF7C20A169
                               E6DC9909D816BECEDC4063CE07CEA828
                               F44BF9B6C9A0A0B200E1B52AF41859C5
        stream[65472..65535] = 2FF20264EEAF47AB7D57C36224535451
                               735AC836D32DD28AE63645CE952F7FDB
                               E6689C695977B1C76E60DD5B27ACA476
                               D2620FDC9313E8489BA56A70C9F4C3A8
        stream[65536..65599] = EB30CDA727C0F8B7E45D5EF30DB7CBE0
                               21F2291E5F56938D56F687B737C3B427
                               545C56A6D3A0BF2B2F47B48493FAE45E
                               D50C2E9BBE49FD92D67C7649055F06FD
      stream[131008..131071] = 0EBF6CC3CBCBE74E6EE807471B492A67
                               39A52F571131A250BCDFA076A26590D7
                               EDE6751C0326A02CB11C58773552804F
                               D8686715355C5A5CC591963A75E994B4
                  xor-digest = 08900BE1E024034E5884974485A84E21
                               FC771F4E2F283292AF5CA780CEBE4745
                               18F9DCC5B45C10F3446B5AC5D55CE1D9
                               D75AA2BAA93144FC634F200C6023EE29



End of test vectors

********************************************************************************
*                          ECRYPT Stream Cipher Project                        *
********************************************************************************

Primitive Name: Salsa20
=======================
Profile: SW & HW
Key size: 128 bits
IV size: 64 bits

Test vectors -- set 1
=====================

(stream is generated by encrypting 512 zero bytes)

Set 1, vector#  0:
                         key = 80000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 4DFA5E481DA23EA09A31022050859936
                               DA52FCEE218005164F267CB65F5CFD7F
                               2B4F97E0FF16924A52DF269515110A07
                               F9E460BC65EF95DA58F740B7D1DBB0AA
            stream[192..255] = DA9C1581F429E0A00F7D67E23B730676
                               783B262E8EB43A25F55FB90B3E753AEF
                               8C6713EC66C51881111593CCB3E8CB8F
                               8DE124080501EEEB389C4BCB6977CF95
            stream[256..319] = 7D5789631EB4554400E1E025935DFA7B
                               3E9039D61BDC58A8697D36815BF1985C
                               EFDF7AE112E5BB81E37ECF0616CE7147
                               FC08A93A367E08631F23C03B00A8DA2F
            stream[448..511] = B375703739DACED4DD4059FD71C3C47F
                               C2F9939670FAD4A46066ADCC6A564578
                               3308B90FFB72BE04A6B147CBE38CC0C3
                               B9267C296A92A7C69873F9F263BE9703
                  xor-digest = F7A274D268316790A67EC058F45C0F2A
                               067A99FCDE6236C0CEF8E056349FE54C
                               5F13AC74D2539570FD34FEAB06C57205
                               3949B59585742181A5A760223AFA22D4

Set 1, vector#  9:
                         key = 00400000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 0471076057830FB99202291177FBFE5D
                               38C888944DF8917CAB82788B91B53D1C
                               FB06D07A304B18BB763F888A61BB6B75
                               5CD58BEC9C4CFB7569CB91862E79C459
            stream[192..255] = D1D7E97556426E6CFC21312AE3811425
                               9E5A6FB10DACBD88E4354B0472556935
                               2B6DA5ACAFACD5E266F9575C2ED8E6F2
                               EFE4B4D36114C3A623DD49F4794F865B
            stream[256..319] = AF06FAA82C73291231E1BD916A773DE1
                               52FD2126C40A10C3A6EB40F22834B8CC
                               68BD5C6DBD7FC1EC8F34165C517C0B63
                               9DB0C60506D3606906B8463AA0D0EC2F
            stream[448..511] = AB3216F1216379EFD5EC589510B8FD35
                               014D0AA0B613040BAE63ECAB90A9AF79
                               661F8DA2F853A5204B0F8E72E9D9EB4D
                               BA5A4690E73A4D25F61EE7295215140C
                  xor-digest = B76A7991D5EE58FC51B9035E077E1315
                               D81F131FA1F26CF22005C6C4F2412243
                               C401A850AFEFAADC5B052435B51177C7
                               0AE68CB9DF9B44681C2D8B7049D89333

Set 1, vector# 18:
                         key = 00002000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = BACFE4145E6D4182EA4A0F59D4076C7E
                               83FFD17E7540E5B7DE70EEDDF9552006
                               B291B214A43E127EED1DA1540F33716D
                               83C3AD7D711CD03251B78B2568F2C844
            stream[192..255] = 56824347D03D9084ECCF358A0AE410B9
                               4F74AE7FAD9F73D2351E0A44DF127434
                               3ADE372BDA2971189623FD1EAA4B723D
                               76F5B9741A3DDC7E5B3E8ED4928EF421
            stream[256..319] = 999F4E0F54C62F9211D4B1F1B79B227A
                               FB3116C9CF9ADB9715DE856A8EB31084
                               71AB40DFBF47B71389EF64C20E1FFDCF
                               018790BCE8E9FDC46527FE1545D3A6EA
            stream[448..511] = 76F1B87E93EB9FEFEC3AED69210FE4AB
                               2ED577DECE01A75FD364CD1CD7DE1027
                               5A002DDBC494EE8350E8EEC1D8D6925E
                               FD6FE7EA7F610512F1F0A83C8949AEB1
                  xor-digest = B9D233247408CD459A027430A23E6FCF
                               3E9A3BAF0D0FC59E623F04D9C107D402
                               880620C64A111318ECE60C22737BECA4
                               21F7D3D004E7191ECE2C7075289B31BF

Set 1, vector# 27:
                         key = 00000010000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 24F4E317B675336E68A8E2A3A04CA967
                               AB96512ACBA2F832015E9BE03F08830F
                               CF32E93D14FFBD2C901E982831ED8062
                               21D7DC8C32BBC8E056F21BF9BDDC8020
            stream[192..255] = E223DE7299E51C94623F8EAD3A6DB045
                               4091EE2B54A498F98690D7D84DB7EFD5
                               A2A8202435CAC1FB34C842AEECF643C6
                               3054C424FAC5A632502CD3146278498A
            stream[256..319] = 5A111014076A6D52E94C364BD7311B64
                               411DE27872FC8641D92C9D811F2B5185
                               94935F959D064A9BE806FAD06517819D
                               2321B248E1F37E108E3412CE93FA8970
            stream[448..511] = 8A9AB11BD5360D8C7F34887982B3F658
                               6C34C1D6CB49100EA5D09A24C6B835D5
                               77C1A1C776902D785CB5516D74E87480
                               79878FDFDDF0126B1867E762546E4D72
                  xor-digest = 0423874278AE11EF0A29B3E6E1A5BA41
                               E43671636615E3F1F6215750E5A1749A
                               CDFE0CEB74A11AC4862527C5849110C9
                               A7A6F01E419372824BCAB90550340E81

Set 1, vector# 36:
                         key = 00000000080000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 9907DB5E2156427AD15B167BEB0AD445
                               452478AFEE3CF71AE1ED8EAF43E001A1
                               C8199AF9CFD88D2B782AA2F39845A26A
                               7AC54E5BE15DB7BDFBF873E16BC05A1D
            stream[192..255] = EBA0DCC03E9EB60AE1EE5EFE3647BE45
                               6E66733AA5D6353447981184A05F0F0C
                               B0AD1CB630C35DC253DE3FEBD10684CA
                               DBA8B4B85E02B757DED0FEB1C31D71A3
            stream[256..319] = BD24858A3DB0D9E552345A3C3ECC4C69
                               BBAE4901016A944C0D7ECCAAB9027738
                               975EEA6A4240D94DA183A74D649B789E
                               24A0849E26DC367BDE4539ADCCF0CAD8
            stream[448..511] = EE20675194FA404F54BAB7103F6821C1
                               37EE2347560DC31D338B01026AB6E571
                               65467215315F06360D85F3C5FE7A359E
                               80CBFE735F75AA065BC18EFB2829457D
                  xor-digest = 19B8E721CD10577375FC6D0E6DC39B05
                               4E371860CE2AA310906EA7BAB28D737F
                               2357B42E7DC1C48D597EA58B87602CE5
                               C37EEDED2E0F4819938878AE7C50E151

Set 1, vector# 45:
                         key = 00000000000400000000000000000000
                          IV = 0000000000000000
               stream[0..63] = A59CE982636F2C8C912B1E8105E2577D
                               9C86861E61FA3BFF757D74CB9EDE6027
                               D7D6DE775643FAF5F2C04971BDCB56E6
                               BE8144366235AC5E01C1EDF8512AF78B
            stream[192..255] = DF8F13F1059E54DEF681CD554439BAB7
                               24CDE604BE5B77D85D2829B3EB137F4F
                               2466BEADF4D5D54BA4DC36F1254BEC4F
                               B2B367A59EA6DDAC005354949D573E68
            stream[256..319] = B3F542ECBAD4ACA0A95B31D281B930E8
                               021993DF5012E48A333316E712C4E19B
                               58231AAE7C90C91C9CC135B12B490BE4
                               2CF9C9A2727621CA81B2C3A081716F76
            stream[448..511] = F64A6449F2F13030BE554DB00D24CD50
                               A89F80CCFE97435EBF0C49EB08747BF7
                               B2C89BE612629F231C1B3398D8B4CC3F
                               35DBECD1CF1CFDFDECD481B72A51276A
                  xor-digest = 4134A74A52EA89BF22E05A467E37E082
                               15537896BE4D2BBDF29EA52A2303E64B
                               D954A18928543C82B68A21E4B830A775
                               CBA9D1176EBF8DB92938DF6E59117B74

Set 1, vector# 54:
                         key = 00000000000002000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 7A8131B777F7FBFD33A06E396FF32D7D
                               8C3CEEE9573F405F98BD6083FE57BAB6
                               FC87D5F34522D2440F649741D9F87849
                               BC8751EF432DEE5DCC6A88B34B6A1EA9
            stream[192..255] = 6573F813310565DB22219984E0919445
                               9E5BB8613237F012EBB8249666582ACA
                               751ED59380199117DDB29A5298F95FF0
                               65D271AB66CF6BC6CDE0EA5FC4D304EB
            stream[256..319] = 0E65CB6944AFBD84F5B5D00F307402B8
                               399BF02852ED2826EA9AA4A55FB56DF2
                               A6B83F7F228947DFAB2E0B10EAAA09D7
                               5A34F165ECB4D06CE6AB4796ABA3206A
            stream[448..511] = 11F69B4D034B1D7213B9560FAE89FF2A
                               53D9D0C9EAFCAA7F27E9D119DEEEA299
                               AC8EC0EA0529846DAF90CF1D9BFBE406
                               043FE03F1713F249084BDD32FD98CD72
                  xor-digest = E9CFBD15B5F4AD02903851F46728F2DD
                               5910273E7360F1571EF1442199143B6C
                               28E5368A2E00E08ADAE73AF3489E0D6F
                               0D8032984ADD139B6BF508A5EEE4434B

Set 1, vector# 63:
                         key = 00000000000000010000000000000000
                          IV = 0000000000000000
               stream[0..63] = FE4DF972E982735FFAEC4D66F929403F
                               7246FB5B2794118493DF068CD310DEB6
                               3EEEF12344E221A2D163CC666F5685B5
                               02F4883142FA867B0BA46BF17D011984
            stream[192..255] = 4694F79AB2F3877BD590BA09B413F1BD
                               F394C4D8F2C20F551AA5A07207433204
                               C2BC3A3BA014886A08F4EC5E4D91CDD0
                               1D7A039C5B815754198B2DBCE68D25EA
            stream[256..319] = D1340204FB4544EFD5DAF28EDCC6FF03
                               B39FBEE708CAEF6ABD3E2E3AB5738B32
                               04EF38CACCC40B9FBD1E6F0206A2B564
                               E2F9EA05E10B6DD061F6AB94374681C0
            stream[448..511] = BB802FB53E11AFDC3104044D70448079
                               41FDAEF1042E0D35972D80CE77B4D560
                               083EB4113CDBC4AC56014D7FF94291DC
                               9387CEF74A0E165042BC12373C6E020C
                  xor-digest = FF021AEC5DC82F40BBF44CEA85287BCF
                               D70F16F557F07B1BF970407051F71C41
                               5B703A67CAF8E81CB22D9F09E0CBD247
                               5E9859355A48FDA9F48E38E2748BE41B

Set 1, vector# 72:
                         key = 00000000000000000080000000000000
                          IV = 0000000000000000
               stream[0..63] = 8F8121BDD7B286465F03D64CA45A4A15
                               4BDF44560419A40E0B482CED194C4B32
                               4F2E9295C452B73B292BA7F55A692DEE
                               A5129A49167BA7AABBEED26E39B25E7A
            stream[192..255] = 7E4388EDBBA6EC5882E9CBF01CFA6786
                               0F10F0A5109FCA7E865C3814EB007CC8
                               9585C2653BDCE30F667CF95A2AA425D3
                               5A531F558180EF3E32A9543AE50E8FD6
            stream[256..319] = 527FF72879B1B809C027DFB7B39D02B3
                               04D648CD8D70F4E0465615B334ED9E2D
                               59703745467F1168A8033BA861841DC0
                               0E7E1AB5E96469F6DA01B8973D0D414A
            stream[448..511] = 82653E0949A5D8E32C4D0A81BBF96F6A
                               7249D4D1E0DCDCC72B90565D9AF4D0AC
                               461C1EAC85E254DD5E567A009EEB3897
                               9A2FD1E4F32FAD15D177D766932190E1
                  xor-digest = B2F239692CE50EECABD7A846AC333885
                               43CFC1061F33420B6F205809F3965D89
                               9C56C02D208DD3E9A1F0D5BBED8F5DAC
                               B164FD005DF907002302F40ADB6665CC

Set 1, vector# 81:
                         key = 00000000000000000000400000000000
                          IV = 0000000000000000
               stream[0..63] = 52FA8BD042682CD5AA21188EBF3B9E4A
                               EE3BE38AE052C5B37730E52C6CEE33C9
                               1B492F95A67F2F6C15425A8623C0C2AE
                               7275FFD0FCF13A0A293A784289BEACB4
            stream[192..255] = 5F43C508BA6F728D032841618F96B103
                               19B094027E7719C28A8A8637D4B0C4D2
                               25D602EA23B40D1541A3F8487F25B14A
                               8CBD8D2001AC28EADFDC0325BA2C140E
            stream[256..319] = 5C802C813FF09CAF632CA8832479F891
                               FB1016F2F44EFA81B3C872E37468B818
                               3EB32D8BD8917A858AEF47524FCC05D3
                               688C551FC8A42C8D9F0509018706E40E
            stream[448..511] = 4CDD40DC6E9C0E4F84810ABE712003F6
                               4B23C6D0C88E61D1F303C3BBD89B58AA
                               098B44B5CD82EDCFC618D324A41317AC
                               6FED20C9A0C54A9ED1F4DA3BF2EC3C66
                  xor-digest = B72D2FEE4BFBC0F65005EE2797B0608A
                               7A6D9CD1114B67C0ADEC7B4B6D793182
                               880777B0279E3DF27CBA820714629A96
                               034E4C71F5356254A0116CF3E9F7EF5C

Set 1, vector# 90:
                         key = 00000000000000000000002000000000
                          IV = 0000000000000000
               stream[0..63] = 6262315C736E88717E9627EECF4F6B55
                               BD10D5960A9961D572EFC7CBDB9A1F01
                               1733D3E17E4735BEFA16FE6B148F8661
                               4C1E37065A48ACF287FFE65C9DC44A58
            stream[192..255] = B43439584FB2FAF3B2937838D8000AC4
                               CD4BC4E582212A7741A0192F71C1F11B
                               58D7F779CA0E6E4B8BD58E00B50C3C53
                               DAF843467064A2DBE2FAD6FF6F40ECD8
            stream[256..319] = EE51EE875F6F1B8AF0334F509DF5692B
                               9B43CC63A586C2380AF3AE490DCD6CFF
                               7907BC3724AE3BBEAD79D436E6DADDB2
                               2141B3BA46C9BEC0E01B9D4F7657B387
            stream[448..511] = E5A4FE4A2FCA9A9ED779A9574283DC21
                               C85216D54486D9B182300D0593B1E2B0
                               10814F7066AEB955C057609CE9AF0D63
                               F057E17B19F57FFB7287EB2067C43B8D
                  xor-digest = 8866D8F9E6F423A7DF10C77625014AA5
                               82C06CD861A88F40FB9CD1EBF0911188
                               4344BEEA5A724E6FD8DB98BF4E6B9BEA
                               5318FA62813D1B49A2D529FC00CB5777

Set 1, vector# 99:
                         key = 00000000000000000000000010000000
                          IV = 0000000000000000
               stream[0..63] = 82FD629BD82C3BE22910951E2E41F8FE
                               187E2BD198F6113AFF44B9B0689AA520
                               C8CCE4E8D3FBA69EDE748BCF18397214
                               F98D7ACF4424866A8670E98EBAB715A3
            stream[192..255] = 342D80E30E2FE7A00B02FC62F7090CDD
                               ECBDFD283D42A00423113196A87BEFD8
                               B9E8AAF61C93F73CC6CBE9CC5AEC182F
                               3948B7857F96B017F3477A2EEC3AEB3B
            stream[256..319] = 8233712B6D3CCB572474BE200D67E540
                               3FC62128D74CE5F790202C696BFFB7EE
                               3CAD255324F87291273A7719278FA313
                               1ABA12342692A2C0C58D27BA3725761B
            stream[448..511] = 782600E7357AC69EA158C725B3E1E940
                               51A0CB63D0D1B4B3DF5F5037E3E1DE45
                               850578E9D513B90B8E5882D4DCA9F42B
                               E32621F4DCC1C77B38F1B0AC1227C196
                  xor-digest = F8AE82F9B77EF090AE0C72A5EAE21405
                               68BEF0B354BCDF4BD39732CD86C63A82
                               AFD27F58C459272B3E8A4B9B558D856F
                               8475CF3A1AD99074822A836CFE520DC5

Set 1, vector#108:
                         key = 00000000000000000000000000080000
                          IV = 0000000000000000
               stream[0..63] = D244F87EB315A7EEF02CA314B440777E
                               C6C44660020B43189693500F3279FA01
                               7257BE0AB087B81F85FD55AAC5845189
                               C66E259B5412C4BDFD0EBE805FC70C8A
            stream[192..255] = 5A2D8D3E431FB40E60856F05C7976206
                               42B35DAB0255764D986740699040702F
                               6CDE058458E842CB6E1843EBD336D374
                               23833EC01DFFF9086FEECAB8A165D29F
            stream[256..319] = 443CEF4570C83517ED55C2F57058BB70
                               294CC8D7342597E2CD850F6C02E355CA
                               EB43C0A41F4BB74FFE9F6B0D25799140
                               D03792D667601AD7954D21BD7C174C43
            stream[448..511] = 959C8B16A0ADEC58B544BE33CCF03277
                               E48C7916E333F549CDE16E2B4B6DCE2D
                               8D76C50718C0E77BFBEB3A3CB3CA14BF
                               40F65EBFAE1A5001EAB36E531414E87F
                  xor-digest = 4DC82B00DC54141CC890348496115C68
                               1DB10ABE8454FBD10B49EF951CD20C6F
                               7FE8AAA10906E57CF05EE838F76C8B7A
                               3F9E6BD6D21C49F1590C913026C71A3E

Set 1, vector#117:
                         key = 00000000000000000000000000000400
                          IV = 0000000000000000
               stream[0..63] = 44A74D35E73A7E7C37B009AE712783AC
                               86ACE0C02CB175656AF79023D91C909E
                               D2CB2F5C94BF8593DDC5E054D7EB726E
                               0E867572AF954F88E05A4DAFD00CCF0A
            stream[192..255] = FEC113A0255391D48A37CDF607AE1226
                               86305DDAD4CF1294598F2336AB6A5A02
                               9D927393454C2E014868137688C0417A
                               2D31D0FE9540D7246FE2F84D6052DE40
            stream[256..319] = 79C2F7431D69E54C0474D8160113F364
                               8156A8963817C34AC9A9AD222543666E
                               7EAF03AF4EE03271C3ECED262E7B4C66
                               B0F618BAF3395423274DD1F73E2675E3
            stream[448..511] = 75C1295C871B1100F27DAF19E5D5BF8D
                               880B9A54CEFDF1561B4351A32898F3C2
                               6A04AB1149C24FBFA2AC963388E64C43
                               65D716BCE8330BC03FA178DBE5C1E6B0
                  xor-digest = 65D58F845F973928ADF5803799901856
                               A08952CF215154C52A5FF2DAD71E8B70
                               3DE107E5531491666353F323E790EB02
                               1B5EF66C13F43401F4F6A27F08CE11D5

Set 1, vector#126:
                         key = 00000000000000000000000000000002
                          IV = 0000000000000000
               stream[0..63] = E23A3638C836B1ACF7E27296E1F5A241
                               3C4CC351EFEF65E3672E7C2FCD1FA105
                               2D2C26778DB774B8FBA29ABED72D058E
                               E35EBA376BA5BC3D84F8E44ABD5DC2CC
            stream[192..255] = 2A8BEB3C372A6570F54EB429FA7F562D
                               6EF14DF725861EDCE8132620EAA00D8B
                               1DFEF653B64E9C328930904A0EEB0132
                               B277BB3D9888431E1F28CDB0238DE685
            stream[256..319] = CCBEB5CA57104B95BF7BA5B12C8B8553
                               4CE9548F628CF53EF02C337D788BCE71
                               D2D3D9C355E7D5EB75C56D079CB7D99D
                               6AF0C8A86024B3AF5C2FC8A028413D93
            stream[448..511] = D00A5FDCE01A334C37E75634A8037B49
                               BEC06ACBD2243320E2CA41FB5619E6D8
                               75AB2007310D4149379C91EF4E199805
                               BE261E5C744F0DF21737E01243B7116F
                  xor-digest = 2D72232A4485E0D2EEDC061939602077
                               4C100C5424FF742B2868E3A68E67E165
                               4C4711C54A34DA937359A26B8386AD20
                               39EB2021DCFBB6A11603AF56225DE098

Test vectors -- set 2
=====================

Set 2, vector#  0:
                         key = 00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 6513ADAECFEB124C1CBE6BDAEF690B4F
                               FB00B0FCACE33CE806792BB414801998
                               34BFB1CFDD095802C6E95E251002989A
                               C22AE588D32AE79320D9BD7732E00338
            stream[192..255] = 75E9D0493CA05D2820408719AFC75120
                               692040118F76B8328AC279530D846670
                               65E735C52ADD4BCFE07C9D93C0091790
                               2B187D46A25924767F91A6B29C961859
            stream[256..319] = 0E47D68F845B3D31E8B47F3BEA660E2E
                               CA484C82F5E3AE00484D87410A1772D0
                               FA3B88F8024C170B21E50E0989E94A26
                               69C91973B3AE5781D305D8122791DA4C
            stream[448..511] = CCBA51D3DB400E7EB780C0CCBD3D2B5B
                               B9AAD82A75A1F746824EE5B9DAF7B794
                               7A4B808DF48CE94830F6C9146860611D
                               A649E735ED5ED6E3E3DFF7C218879D63
                  xor-digest = 6D3937FFA13637648E477623277644AD
                               AD3854E6B2B3E4D68155356F68B30490
                               842B2AEA2E32239BE84E613C6CE1B9BD
                               026094962CB1A6757AF5A13DDAF8252C

Set 2, vector#  9:
                         key = 09090909090909090909090909090909
                          IV = 0000000000000000
               stream[0..63] = 169060CCB42BEA7BEE4D8012A02F3635
                               EB7BCA12859FA159CD559094B3507DB8
                               01735D1A1300102A9C9415546829CBD2
                               021BA217B39B81D89C55B13D0C603359
            stream[192..255] = 23EF24BB24195B9FD574823CD8A40C29
                               D86BD35C191E2038779FF696C712B6D8
                               2E7014DBE1AC5D527AF076C088C4A8D4
                               4317958189F6EF54933A7E0816B5B916
            stream[256..319] = D8F12ED8AFE9422B85E5CC9B8ADEC9D6
                               CFABE8DBC1082BCCC02F5A7266AA074C
                               A284E583A35837798CC0E69D4CE93765
                               3B8CDD65CE414B89138615CCB165AD19
            stream[448..511] = F70A0FF4ECD155E0F033604693A51E23
                               63880E2ECF98699E7174AF7C2C6B0FC6
                               59AE329599A3949272A37B9B2183A091
                               0922A3F325AE124DCBDD735364055CEB
                  xor-digest = 30209DD68D46E5A30034EF6DCE74FE1A
                               B6C772AB22CD3D6C354A9C4607EF3F82
                               900423D29FB65E07FFA3AEAD94E940D6
                               E52E305A10D60936D34BD03B3F342AB1

Set 2, vector# 18:
                         key = 12121212121212121212121212121212
                          IV = 0000000000000000
               stream[0..63] = 05835754A1333770BBA8262F8A84D0FD
                               70ABF58CDB83A54172B0C07B6CCA5641
                               060E3097D2B19F82E918CB697D0F347D
                               C7DAE05C14355D09B61B47298FE89AEB
            stream[192..255] = 5525C22F425949A5E51A4EAFA18F62C6
                               E01A27EF78D79B073AEBEC436EC8183B
                               C683CD3205CF80B795181DAFF3DC9848
                               6644C6310F09D865A7A75EE6D5105F92
            stream[256..319] = 2EE7A4F9C576EADE7EE325334212196C
                               B7A61D6FA693238E6E2C8B53B900FF1A
                               133A6E53F58AC89D6A695594CE03F775
                               8DF9ABE981F23373B3680C7A4AD82680
            stream[448..511] = CB7A0595F3A1B755E9070E8D3BACCF95
                               74F881E4B9D91558E19317C4C254988F
                               42184584E5538C63D964F8EF61D86B09
                               D983998979BA3F44BAF527128D3E5393
                  xor-digest = AD29013FD0A222EEBE65126380A26477
                               BD86751B3B0A2B4922602E63E6ECDA52
                               3BA789633BEE6CFF64436A8644CCD7E8
                               F81B062187A9595A8D2507ED774FA5CD

Set 2, vector# 27:
                         key = 1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B
                          IV = 0000000000000000
               stream[0..63] = 72A8D26F2DF3B6713C2A053B3354DBA6
                               C10743C7A8F19261CF0E7957905748DD
                               D6D3333E2CBC6611B68C458D5CDBA2A2
                               30AC5AB03D59E71FE9C993E7B8E7E09F
            stream[192..255] = 7B6132DC5E2990B0049A5F7F357C9D99
                               7733948018AE1D4F9DB999F4605FD78C
                               B548D75AC4657D93A20AA451B8F35E0A
                               3CD08880CCED7D4A508BA7FB49737C17
            stream[256..319] = EF7A7448D019C76ED0B9C18B5B2867CF
                               9AD84B789FB037E6B107B0A4615737B5
                               C1C113F91462CDA0BCB9ADDC09E8EA6B
                               99E4835FED25F5CC423EEFF56D851838
            stream[448..511] = 6B75BDD0EC8D581CB7567426F0B92C9B
                               B5057A89C3F604583DB700A46D6B8DE4
                               1AF315AE99BB5C1B52C76272D1E262F9
                               FC7022CE70B435C27AE443284F5F84C1
                  xor-digest = 484F9FCB516547DD89AF46991B18F1DE
                               C4C6CBC7D52735E00FC3201B4650151C
                               3D4FB9C119442B368B28E3C68ED83F10
                               D9DA2FDED7DEB8F04827FA91CCDBF65B

Set 2, vector# 36:
                         key = 24242424242424242424242424242424
                          IV = 0000000000000000
               stream[0..63] = 76240D13C7E59CBD4183D162834A5D36
                               37CD09EE4F5AFE9C28CFA9466A4089F6
                               5C80C224A87F956459B173D720274D09
                               C573FCD128498D810460FDA1BB50F934
            stream[192..255] = 71AF115217F3B7F77A05B56E32AD0889
                               BFA470B6DDC256D852C63B45688D7BC8
                               DC610D347A2600D7769C67B28D1FA25F
                               1AACFB8F9BB68BFE17357335D8FAC993
            stream[256..319] = 6573CC1ADC0DE744F6694E5FBB59E5BF
                               5939CE5D13793E2F683C7F2C7DD9A460
                               575746688A0F17D419FE3E5F88654559
                               7B6705E1390542B4F953D568025F5BB3
            stream[448..511] = 809179FAD4AD9B5C355A09E99C8BE931
                               4B9DF269F162C1317206EB3580CAE58A
                               B93A408C23739EF9538730FE687C8DAC
                               1CE95290BA4ACBC886153E63A613857B
                  xor-digest = D1781DCE3EFB8B13740F016264051354
                               F323C81A13D42CE75E67180849AC49FF
                               A7EA95720696F86848A1A4B8506A95E3
                               A61371DDE7F21167CC147173BFC4D78F

Set 2, vector# 45:
                         key = 2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D
                          IV = 0000000000000000
               stream[0..63] = 3117FD618A1E7821EA08CDED410C8A67
                               BDD8F7BE3FCA9649BD3E297FD83A80AD
                               814C8904C9D7A2DC0DCAA641CFFF502D
                               78AFF1832D34C263C1938C1ADF01238F
            stream[192..255] = 1E8CB540F19EC7AFCB366A25F74C0004
                               B682E06129030617527BECD16E3E3E00
                               27D818F035EDCDF56D8D4752AEF28BDB
                               FA0D3B008235173475F5FA105B91BEED
            stream[256..319] = 637C3B4566BBEBBE703E4BF1C978CCD2
                               77AE3B8768DB97DF01983CDF3529B3EC
                               6B1137CA6F231047C13EA38649D0058E
                               BE5EF7B7BBA140F22338E382F1D6AB3F
            stream[448..511] = D407259B6355C343D64A5130DA55C057
                               E4AF722B70AC8A074262233677A457AF
                               EAA34E7FD6F15959A4C781C4C978F7B3
                               BC571BF66674F015A1EA5DB262E25BDC
                  xor-digest = 1F64F78101768FF5067B9A918444EF70
                               3FF06561E23B31C61BD43BCF86CFAD24
                               9942F73DC8F40AE49B14874B08F2A527
                               A53DF496F37D067F1168268D4A134740

Set 2, vector# 54:
                         key = 36363636363636363636363636363636
                          IV = 0000000000000000
               stream[0..63] = 7FED83B9283449AD8EBFC935F5F36407
                               5C9008ADE8626D350770E2DBD058F053
                               F7E5300B088B1341EC54C2BEE72A520C
                               35C673E79CC4ED0A6D8F4C15FBDD090B
            stream[192..255] = D780206A2537106610D1C95BF7E9121B
                               EDE1F0B8DFBE83CBC49C2C653DD187F7
                               D84A2F4607BF99A96B3B84FB792340D4
                               E67202FB74EC24F38955F345F21CF3DB
            stream[256..319] = 6CA21C5DC289674C13CFD4FCBDEA8356
                               0A90F53BB54F16DBF274F5CC56D7857C
                               D3E3B06C81C70C828DC30DADEBD92F38
                               BB8C24136F37797A647584BCEE68DF91
            stream[448..511] = 471936CE9C84E131C4C5792B769654B8
                               9644BFAFB1149130E580FD805A325B62
                               8CDE5FAE0F5C7CFFEF0D931F8F517A92
                               9E892D3789B74217A81BAEFE441E47ED
                  xor-digest = 0073DA29855E96EA5C414B9BD2E1C0F4
                               987D3F1EB1CA73C4AA10180B99A43774
                               4857EB36586593B81088AADE5D89BBC6
                               8FBD8B0D268080746D6BE38DBC9396CD

Set 2, vector# 63:
                         key = 3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F
                          IV = 0000000000000000
               stream[0..63] = C224F33B124D6692263733DFD5BF5271
                               7D1FB45EC1CEDCA6BF92BA44C1EADA85
                               F7B031BCC581A890FD275085C7AD1C3D
                               652BCA5F4D7597DECDB2232318EABC32
            stream[192..255] = 090325F54C0350AD446C19ABDCAEFF52
                               EC57F5A13FB55FEDE4606CEC44EC658B
                               BB13163481D2C84BF9409313F6470A0D
                               A9803936094CC29A8DE7613CBFA77DD5
            stream[256..319] = 1F66F5B70B9D12BC7092C1846498A2A0
                               730AA8FA8DD97A757BBB878320CE6633
                               E5BCC3A5090F3F75BE6E72DD1E8D95B0
                               DE7DBFDD764E484E1FB854B68A7111C6
            stream[448..511] = F8AE560041414BE888C7B5EB3082CC7C
                               4DFBBA5FD103F522FBD95A7166B91DE6
                               C78FB47413576EC83F0EDE6338C9EDDB
                               81757B58C45CBD3A3E29E491DB1F04E2
                  xor-digest = 542B2672401C5D1225CC704365753E33
                               D0827A863C4897FFCE1B724CD10B2A0E
                               8A4E4CDAB7357424FC6DC78440037240
                               B8FD5299907A946CE77DAFA5322AB73D

Set 2, vector# 72:
                         key = 48484848484848484848484848484848
                          IV = 0000000000000000
               stream[0..63] = 11BF31E22D7458C189092A1DE3A4905B
                               A2FA36858907E3511FB63FDFF2C5C2A1
                               5B651B2C2F1A3A43A718642152806967
                               2B6BB0AEC10452F1DAA9FC73FF5A396A
            stream[192..255] = D1E1619E4BD327D2A124FC52BC15B194
                               0B05394ECE5926E1E1ADE7D3FC8C6E91
                               E43889F6F9C1FD5C094F6CA25025AE4C
                               CC4FDC1824936373DBEE16D62B81112D
            stream[256..319] = F900E9B0665F84C939D5FE4946FA7B41
                               E34F06058522A2DB49E210E3E5385E58
                               97C24F6350C6CCA578285325CC16F558
                               6DC662FFBEA41BAC68996BAAB9F32D1F
            stream[448..511] = 40587ECAD15841F1BD1D236A61051574
                               A974E15292F777ABDED64D2B761892BE
                               F3DD69E479DE0D02CC73AF76E81E8A77
                               F3CEE74180CB5685ACD4F0039DFFC3B0
                  xor-digest = C3E5CC5C7CEA1B3885EB9CEF2D1FAF18
                               E7DE1CFD7237F2D6D344F3DF7168A88E
                               C88C1314CB6F5A3EAE1BC468B4FAD75E
                               8A42BE8607705C9A7950302461AD9B3F

Set 2, vector# 81:
                         key = 51515151515151515151515151515151
                          IV = 0000000000000000
               stream[0..63] = EBC464423EADEF13E845C595A9795A58
                               5064F478A1C8582F07A4BA68E81329CB
                               26A13C2EA0EFE9094B0A749FDB1CC6F9
                               C2D293F0B395E14EB63075A39A2EDB4C
            stream[192..255] = F4BBBBCE9C5869DE6BAF5FD4AE835DBE
                               5B7F1752B2972086F3383E9D180C2FE5
                               5618846B10EB68AC0EB0865E0B167C6D
                               3A843B29336BC1100A4AB7E8A3369959
            stream[256..319] = 3CEB39E3D740771BD49002EA8CD99851
                               8A8C70772679ECAF2030583AED43F77F
                               565FECDBEF333265A2E1CC42CB606980
                               AEF3B24C436A12C85CBDC5EBD97A9177
            stream[448..511] = EF651A98A98C4C2B61EA8E7A673F5D4F
                               D832D1F9FD19EE4537B6FEC7D11C6B2F
                               3EF5D764EEAD396A7A2E32662647BFC0
                               7F02A557BA6EF046C8DE3781D74332B0
                  xor-digest = 88A96FF895BF2A827FC26DB2BB75DC69
                               8E8E1B7E231997AB2942E981EF1633EA
                               061F6B323B99519828FB41A6F5CCC79C
                               57F6DDDD34DEAB38514A54C4886626E5

Set 2, vector# 90:
                         key = 5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A
                          IV = 0000000000000000
               stream[0..63] = F40253BAA835152E1582646FD5BD3FED
                               638EB3498C80BFB941644A7750BBA565
                               3130CC97A937A2B27AFBB3E679BC42BE
                               87F83723DC6F0D61DCE9DE8608AC62AA
            stream[192..255] = A5A1CD35A230ED57ADB8FE16CD2D2EA6
                               055C32D3E621A0FD6EB6717AA916D478
                               57CD987C16E6112EDE60CCB0F7014642
                               2788017A6812202362691FDA257E5856
            stream[256..319] = 81F0D04A929DB4676F6A3E6C15049779
                               C4EC9A12ACF80168D7E9AA1D6FA9C13E
                               F2956CEE750A89103B48F22C06439C5C
                               E9129996455FAE2D7775A1D8D39B00CE
            stream[448..511] = 3F6D60A0951F0747B94E4DDE3CA4ED4C
                               96694B7534CD9ED97B96FAAD3CF00D4A
                               EF12919D410CD9777CD5F2F3F2BF160E
                               BBA3561CC24345D9A09978C3253F6DCB
                  xor-digest = 554F89BF1AD5602655B800DB9B3CCFFA
                               1B267D57654DCF3FDDA81A59DF68B022
                               555E63DE51E7A83668E7F1AE09EEB5B8
                               748DEF8580B304199C4D117CF9A94E78

Set 2, vector# 99:
                         key = 63636363636363636363636363636363
                          IV = 0000000000000000
               stream[0..63] = ED5FF13649F7D8EDFC783EFDF2F843B3
                               68776B19390AF110BEF12EAC8EC58A2E
                               8CDAB6EC9049FBDA23A615C536C3A313
                               799E21668C248EC864D5D5D99DED80B3
            stream[192..255] = 845ACE9B870CF9D77597201988552DE5
                               3FD40D2C8AC51ABE1335F6A2D0035DF8
                               B10CACAD851E000BAC6EA8831B2FBCFE
                               B7C94787E41CC541BAC3D9D26DB4F19D
            stream[256..319] = 981580764B81A4E12CA1F36634B59136
                               5E4BDB6C12DE13F2F337E72E018029C5
                               A0BECDA7B6723DD609D81A314CE39619
                               0E82848893E5A44478B08340F90A73F3
            stream[448..511] = 4CD3B072D5720E6C64C9476552D1CFF4
                               D4EF68DCBD11E8D516F0C248F9250B57
                               1990DD3AFC0AE8452896CCCC0BD0EFDF
                               17B616691AB3DF9AF6A42EDCA54BF9CD
                  xor-digest = 52D590BB5E396FCC2E00D9C51B3C0BF0
                               73E123C7EE69B528B0F0F87B57DC6907
                               F4B57FD5F5B10D602B1F723E9FDD5510
                               AEC60CD0DD50ED4B60FA355859638C2C

Set 2, vector#108:
                         key = 6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C
                          IV = 0000000000000000
               stream[0..63] = 78ED06021C5C7867D176DA2A96C4BBAA
                               494F451F21875446393E9688205ED63D
                               EA8ADEB1A2381201F576C5A541BC8887
                               4078608CA8F2C2A6BDCDC1081DD254CC
            stream[192..255] = C1747F85DB1E4FB3E29621015314E3CB
                               261808FA6485E59057B60BE82851CFC9
                               48966763AF97CB9869567B763C745457
                               5022249DFE729BD5DEF41E6DBCC68128
            stream[256..319] = 1EE4C7F63AF666D8EDB2564268ECD127
                               B4D015CB59487FEAF87D0941D42D0F8A
                               24BD353D4EF765FCCF07A3C3ACF71B90
                               E03E8AEA9C3F467FE2DD36CEC00E5271
            stream[448..511] = 7AFF4F3A284CC39E5EAF07BA6341F065
                               671147CA0F073CEF2B992A7E21690C82
                               71639ED678D6A675EBDAD48336584213
                               15A2BA74754467CCCE128CCC62668D0D
                  xor-digest = FB3FE601D4E58B0766F02FA15C332391
                               3CD745E905AD74EA5DABA77BC25D282D
                               D66D98204E101F06D60BA446A21331AF
                               6DDEB70679DEF46B886EB8A75C916380

Set 2, vector#117:
                         key = 75757575757575757575757575757575
                          IV = 0000000000000000
               stream[0..63] = D935C93A8EBB90DB53A27BF9B41B3345
                               23E1DFDE3BFFC09EA97EFB9376D38C7D
                               6DC67AAB21EA3A5C07B6503F986F7E8D
                               9E11B3150BF0D38F36C284ADB31FACF8
            stream[192..255] = DA88C48115010D3CD5DC0640DED2E652
                               0399AAFED73E573CBAF552C6FE06B1B3
                               F3ADE3ADC19DA311B675A6D83FD48E38
                               46825BD36EB88001AE1BD69439A0141C
            stream[256..319] = 14EA210224DAF4FC5D647C78B6BFEF7D
                               724DC56DCDF832B496DEAD31DD948DB1
                               944E17AB2966973FD7CCB1BC9EC0335F
                               35326D5834EE3B08833358C4C28F70DE
            stream[448..511] = D5346E161C083E00E247414F44E0E737
                               5B435F426B58D482A37694331D7C5DC9
                               7D8953E6A852625282973ECCFD012D66
                               4C0AFA5D481A59D7688FDB54C55CD04F
                  xor-digest = BB5EAC1AB84C70857245294309C023C4
                               B1A4199D16877BC847BCBB1B0A8D1B54
                               4289D6C8BF27212AAFFD42021669BB24
                               77A4F815FA01B3F7E88299240155265B

Set 2, vector#126:
                         key = 7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
                          IV = 0000000000000000
               stream[0..63] = 45A43A587C45607441CE3AE200467977
                               88879C5B77FDB90B76F7D2DF27EE8D94
                               28A5B5AF35E2AAE242E6577BEC92DA09
                               29A6AFB3CB8F8496375C98085460AB95
            stream[192..255] = 14AE0BA973AE19E6FD674413C276AB9D
                               99AA0048822AFB6F0B68A2741FB5CE2F
                               64F3D862106EF2BDE19B39209F75B92B
                               DBE9015D63FDFD7B9E8A776291F4E831
            stream[256..319] = C26FA1812FFC32EFF2954592A0E1E5B1
                               26D5A2196624156E3DFD0481205B24D5
                               613B0A75AF3CBC8BBE5911BE93125BD3
                               D3C50C92910DBA05D80666632E5DF9EF
            stream[448..511] = AD0DABE5AF74AB4F62B4699E0D667BBF
                               01B4DCF0A45514554CAC4DFDE453EFF1
                               E51BE5B74B37512C40E3608FB0E65A3F
                               D4EAFA27A3BB0D6E1300C594CB0D1254
                  xor-digest = 0F1A4B0994EE03B6C381FE4BB8E33C0E
                               E47C395BB59922C5537EEBFD12549422
                               0F743B93D867085E027E56623F795056
                               08179A39FF52D4C00A45A5FB8F618C49

Set 2, vector#135:
                         key = 87878787878787878787878787878787
                          IV = 0000000000000000
               stream[0..63] = 09E15E82DFA9D821B8F68789978D0940
                               48892C624167BA88AD767CAEFDE80E25
                               F57467156B8054C8E88F3478A2897A20
                               344C4B05665E7438AD1836BE86A07B83
            stream[192..255] = 2D752E53C3FCA8D3CC4E760595D588A6
                               B321F910B8F96459DBD42C6635063246
                               60A527C66A53B406709262B0E42F11CB
                               0AD2450A1FB2F48EA85C1B39D4408DB9
            stream[256..319] = 1EC94A21BD2C0408D3E15104FA25D15D
                               6E3E0D3F8070D84184D35B6302BF62AE
                               A282E3640820CC09E1528B684B740018
                               0598D6960EC92E4EC4C9E533E1BA06F1
            stream[448..511] = D0AC302C5CC256351E24CFFD11F0BD8A
                               0BE1277EDDCB3EE4D530E051712A710D
                               F4513FD6438B7A355CCF4FEDA9A60F2A
                               C375508F998C642E6C51724FE9462F7F
                  xor-digest = B7F32B6FADB48BB8DA231BDBDC469723
                               2BAE5F8F8345F9F14A991FF851CC3C64
                               1DF4913A5C550FC898F95AC299ED8915
                               5A434DC4B1E37D82EA137BB763F68BC7

Set 2, vector#144:
                         key = 90909090909090909090909090909090
                          IV = 0000000000000000
               stream[0..63] = EA869D49E7C75E07B551C24EBE351B4E
                               7FD9CB26413E55A8A977B766650F81EF
                               CA06E30107F76DC97EA9147FFA7CA66A
                               FD4D4DA538CDA1C27E8D948CC406FB89
            stream[192..255] = 436A8EC10421116CD03BF95A4DAAE630
                               1BB8C724B3D481099C70B26109971CCE
                               ACBCE35C8EE98BBB0CD553B5C4181125
                               00262C7EA10FAAC8BA9A30A04222D8E2
            stream[256..319] = 47487A34DE325E79838475B1757D5D29
                               3C931F9E57579FCA5E04A40E4A0A38CF
                               D1614F9CEF75F024FFF5D972BD671DC9
                               FB2A80F64E8A2D82C3BAA5DDFD1E6821
            stream[448..511] = 3FDCAD4E7B069391FAB74C836D58DE23
                               95B27FFAE47D633912AE97E7E3E60264
                               CA0DC540D33122320311C5CFC9E26D63
                               2753AC45B6A8E81AC816F5CA3BBDB1D6
                  xor-digest = E30E770C75C94EE022BEA6B95241E5D7
                               163D7C55AAF20FE7150768CEE6E11037
                               42902FA4F928CDCF31335944DCDEBADD
                               E36FE089D2EB93677E9DF75234E1B3C8

Set 2, vector#153:
                         key = 99999999999999999999999999999999
                          IV = 0000000000000000
               stream[0..63] = 7B3AA4599561C9059739C7D18D342CF2
                               E73B3B9E1E85D38EDB41AEFADD81BF24
                               1580885078CA10D338598D18B3E4B693
                               155D12D362D533494BA48142AB068F68
            stream[192..255] = D27864FC30D5FD278A9FB83FADADFD2F
                               E72CE78A2563C031791D55FF31CF5946
                               4BE7422C81968A70E040164603DC0B0A
                               EEE93AC497CC0B770779CE6058BE80CF
            stream[256..319] = 4C5A87029660B65782FD616F48CFD600
                               6DFB158682DC80E085E52163BE2947E2
                               70A0FD74DC8DC2F5920E59F28E225280
                               FAC96BA78B8007E3D0DF6EF7BF835993
            stream[448..511] = F5A2ECD04452358970E4F8914FC08E82
                               926ECFF33D9FC0977F10241E7A50E528
                               996A7FB71F79FC30BF881AF6BA19016D
                               DC077ED22C58DC57E2BDBDA1020B30B2
                  xor-digest = 8C9995B52F4AC9CA25E5C956850FFE90
                               D396530617298D89659C2F863995FB06
                               0B65ADFED6AA977EDBB4FC2F6774335E
                               9DEBC61E05E92718A340F79368E74273

Set 2, vector#162:
                         key = A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2
                          IV = 0000000000000000
               stream[0..63] = 9776A232A31A22E2F10D203A2A1B60B9
                               D28D64D6D0BF32C8CCA1BBF6B57B1482
                               BCC9FCF7BBE0F8B61C4BF64C540474BC
                               F1F9C1C808CCBE6693668632A4E8653B
            stream[192..255] = 5C746D64A3195079079028D74CE029A8
                               7F72B30B34B6C7459998847C42F2E44D
                               843CF196229EED471B6BBDBA63BE3B52
                               9B8AF4B5846EB0AB008261E161707B76
            stream[256..319] = F780FE5204AC188A680F41068A9F5018
                               2D9154D6D5F1886034C270A8C3AF61DF
                               945381B7ADCA546E153DBF0E6EA2DDDA
                               4EDA3E7F7CF4E2043C5E20AF659282B4
            stream[448..511] = 71D24CD8B4A70554906A32A5EFDFA8B8
                               34C324E6F35240257A0A27485103616D
                               D41C8F4108D1FC76AB72AF166100AB17
                               212492A72099ACF6F9EB53AC50BD8B8B
                  xor-digest = B2217FF55077D373B735C1A7D8B784F5
                               187AF2F028FE906F85B938277CAC918C
                               E87BEA508AFF86B9071F2B7E4F88A3B1
                               F3323151C9DF441FE6F266CF8F01A0B9

Set 2, vector#171:
                         key = ABABABABABABABABABABABABABABABAB
                          IV = 0000000000000000
               stream[0..63] = 62DF49A919AF1367D2AAF1EB608DE1FD
                               F8B93C2026389CEBE93FA389C6F28458
                               48EBBE70B3A3C8E79061D78E9ED24ED9
                               AA7BB6C1D726AA060AEFC4FFE70F0169
            stream[192..255] = E7A4DF0D61453F612FB558D1FAE198AA
                               B1979F91E1792C99423E0C5733459365
                               70915B60210F1F9CA8845120E6372659
                               B02A179A4D679E8EDDDDF8843ABAB7A4
            stream[256..319] = C9501A02DD6AFB536BD2045917B016B8
                               3C5150A7232E945A53B4A61F90C5D0FB
                               6E6AC45182CBF428772049B32C825D1C
                               33290DBEEC9EF3FE69F5EF4FAC95E9B1
            stream[448..511] = B8D487CDD057282A0DDF21CE3F421E2A
                               C9696CD36416FA900D12A20199FE0018
                               86C904AB629194AECCC28E59A54A1357
                               47B7537D4E017B66538E5B1E83F88367
                  xor-digest = 4EB0E761F6BD6A738DC295C0B1B737FC
                               FDB2A68FF50EB198D699CC71141EC6EB
                               54434D40B592A65F2F5C50B6027D4F52
                               9307969E1D74028FF4BD6A44CEAA121C

Set 2, vector#180:
                         key = B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4
                          IV = 0000000000000000
               stream[0..63] = 6F703F3FF0A49665AC70CD9675902EE7
                               8C60FF8BEB931011FC89B0F28D6E176A
                               9AD4D494693187CB5DB08FF727477AE6
                               4B2EF7383E76F19731B9E23186212720
            stream[192..255] = AD26886ABF6AD6E0CA4E305E468DA1B3
                               69F0ADD3E14364C8A95BD78C5F2762B7
                               2915264A022AD11B3C6D312B5F6526E0
                               183D581B57973AFB824945BFB78CEB8F
            stream[256..319] = FE29F08A5C157B87C600CE4458F274C9
                               86451983FE5AE561DF56139FF33755D7
                               1100286068A32559B169D8C2161E215D
                               BC32FAEA11B652284795C144CF3E693E
            stream[448..511] = 7974578366C3E999028FA8318D82AAAA
                               8ED3FD4DFB111CBF0F529C251BA91DC6
                               ACFA9795C90C954CEA287D23AD979028
                               E974393B4C3ABA251BCB6CECCD09210E
                  xor-digest = 88BE85838404EA4F0FFDD192C43E3B93
                               329C4A4919234D116E4393EA26110022
                               BED2B427EC719178E6F1A9B9B08BEF5B
                               F2FE4A9CC869CB6BD2D989F750EDA78F

Set 2, vector#189:
                         key = BDBDBDBDBDBDBDBDBDBDBDBDBDBDBDBD
                          IV = 0000000000000000
               stream[0..63] = 61900F2EF2BEA2F59971C82CDFB52F27
                               9D81B444833FF02DD0178A53A8BFB9E1
                               FF3B8D7EC799A7FBB60EADE8B1630C12
                               1059AA3E756702FEF9EEE7F233AFC79F
            stream[192..255] = D27E0784038D1B13833ACD396413FF10
                               D35F3C5C04A710FC58313EEBC1113B2C
                               FA20CBD1AEA4433C6650F16E7C3B6830
                               2E5F6B58D8E4F26D91F19FE981DEF939
            stream[256..319] = B658FB693E80CE50E3F64B910B660BEB
                               142B4C4B61466424A9884D22EB80B8B4
                               0C26BEA869118ED068DCC83F9E4C68F1
                               7A3597D0FE0E36700D01B4252EE0010E
            stream[448..511] = 9FC658A20D3107A34680CC75EB3F76D6
                               A2150490E9F6A3428C9AD57F2A252385
                               C956B01C31C978E219BE351A534DB23B
                               99908DACC6726196742D0B7E1D88472C
                  xor-digest = DA74A6EC8D54723B1797751F786CB1B5
                               17995EBF297A034AF744EEF86833CC5B
                               A3DCBDB4D3FAB47F5BA37463CEC80F45
                               DAE1A48FBB80148A39CA789BAE09D39F

Set 2, vector#198:
                         key = C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6
                          IV = 0000000000000000
               stream[0..63] = 42D1C40F11588014006445E81C8219C4
                               370E55E06731E09514956834B2047EE2
                               8A9DAECC7EB25F34A311CC8EA28EDCD2
                               4A539160A0D8FDAA1A26E9F0CDFE0BE3
            stream[192..255] = 976201744266DEABBA3BFE206295F40E
                               8D9D169475C11659ADA3F6F25F11CEF8
                               CD6B851B1F72CD3E7D6F0ABAF8FB929D
                               DB7CF0C7B128B4E4C2C977297B2C5FC9
            stream[256..319] = D3601C4CD44BBEEFD5DAD1BDFF12C190
                               A5F0B0CE95C019972863F4309CE566DE
                               62BECB0C5F43360A9A09EB5BAB87CF13
                               E7AB42D71D5E1229AF88667D95E8C96F
            stream[448..511] = 69EAA4BAAAA795BCF3B96E79C931A1F2
                               D2DD16A242714358B106F38C1234A5BB
                               D269E68A03539EFAFA79455ADBE1B984
                               E9766B0720947E1365FDF076F73639CD
                  xor-digest = 54E422EB1EB2DBDB338798E0D352A87A
                               D5F5A28BC5F77E1B42913E6500723A93
                               6D4019D703DC93A1DF7C65AB74F1FC1A
                               4D38C519A8338B73A435FC7491DFC769

Set 2, vector#207:
                         key = CFCFCFCFCFCFCFCFCFCFCFCFCFCFCFCF
                          IV = 0000000000000000
               stream[0..63] = 9C09F353BF5ED33EDEF88D73985A14DB
                               C1390F08236461F08FDCAF9A7699FD7C
                               4C602BE458B3437CEB1464F451ED021A
                               0E1C906BA59C73A8BA745979AF213E35
            stream[192..255] = 437E3C1DE32B0DB2F0A57E41A7282670
                               AC223D9FD958D111A8B45A70A1F863E2
                               989A97386758D44060F6BFFF5434C908
                               88B4BB4EDAE6528AAADC7B81B8C7BEA3
            stream[256..319] = 94007100350C946B6D12B7C6A2FD1215
                               682C867257C12C74E343B79E3DE79A78
                               2D74663347D8E633D8BE9D288A2A64A8
                               55C71B4496587ADECCB4F30706BB4BD9
            stream[448..511] = 585D0C2DB901F4004846ADBAA754BCA8
                               2B66A94C9AF06C914E3751243B87581A
                               FAE281312A492DBEE8D6BB64DD748F44
                               5EF88F82AB44CBA33D767678914BDE77
                  xor-digest = BB97F09B9FCEC06B6124310BBDD1E9CE
                               8D3793F62FF1337F520DE2A90FE2592A
                               F2636DFA20466FDAA9329443ACC0E9A5
                               0492621AF5790CAE5642E6F7D9AF400D

Set 2, vector#216:
                         key = D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8
                          IV = 0000000000000000
               stream[0..63] = 4965F30797EE95156A0C141D2ACA5232
                               04DD7C0F89C6B3F5A2AC1C59B8CF0DA4
                               01B3906A6A3C94DA1F1E0046BD895052
                               CB9E95F667407B4EE9E579D7A2C91861
            stream[192..255] = 8EDF23D6C8B062593C6F32360BF271B7
                               ACEC1A4F7B66BF964DFB6C0BD93217BB
                               C5FACC720B286E93D3E9B31FA8C4C762
                               DF1F8A3836A8FD8ACBA384B8093E0817
            stream[256..319] = 44FA82E9E469170BA6E5E8833117DAE9
                               E65401105C5F9FEA0AF682E53A627B4A
                               4A621B63F7CE5265D3DFADFBFD4A2B6C
                               2B40D2249EB0385D959F9FE73B37D67D
            stream[448..511] = 828BA57593BC4C2ACB0E8E4B8266C1CC
                               095CE9A761FB68FC57D7A2FCFF768EFB
                               39629D3378549FEE08CCF48A4A4DC2DD
                               17E72A1454B7FA82E2ACF90B4B8370A7
                  xor-digest = 8A365EE7E7BC9198EC88A39F5047431D
                               1632CBB0D1E812957595E7A0763DFA46
                               953070863838812A9504F7A376078FEA
                               9444B27E15FC043AE2D375D37DB1C6C3

Set 2, vector#225:
                         key = E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1
                          IV = 0000000000000000
               stream[0..63] = 5C7BA38DF4789D45C75FCC71EC9E5751
                               B3A60AD62367952C6A87C0657D6DB3E7
                               1053AC73E75FF4B66177B3325B1BBE69
                               AEE30AD5867D68B660603FE4F0BF8AA6
            stream[192..255] = B9C7460E3B6C313BA17F7AE115FC6A8A
                               499943C70BE40B8EF9842C8A934061E1
                               E9CB9B4ED3503165C528CA6E0CF2622B
                               B1F16D24657BDAEDB9BA8F9E193B65EB
            stream[256..319] = 406CD92883E991057DFD80BC8201067F
                               35700264A4DFC28CF23EE32573DCB420
                               91FEF27548613999E5C5463E840FE957
                               60CF80CC5A05A74DE49E7724273C9EA6
            stream[448..511] = F13D615B49786D74B6591BA6887A7669
                               136F34B69D31412D4A9CB90234DAFCC4
                               1551743113701EF6191A577C7DB72E2C
                               B723C738317848F7CC917E1510F02791
                  xor-digest = B31C13C287692760C2710CC4812A4CD3
                               535248839E0B5220185BE58BBCE6A70D
                               629E0749D40D9E79F698FFAFF7B9C530
                               06419AAAD9AC1FAC2286F66DEC96AEB3

Set 2, vector#234:
                         key = EAEAEAEAEAEAEAEAEAEAEAEAEAEAEAEA
                          IV = 0000000000000000
               stream[0..63] = 5B06F5B01529B8C57B73A410A61DD757
                               FE5810970AA0CBFAD3404F17E7C7B645
                               9DD7F615913A0EF2DCC91AFC57FA660D
                               6C7352B537C65CD090F1DE51C1036AB5
            stream[192..255] = 0F613F9E9F03199DF0D0A5C5BE253CDF
                               138903876DE7F7B0F40B2F840F322F27
                               0C0618D05ABB1F013D8744B231555A8E
                               CB14A9E9C9AF39EDA91D36700F1C25B3
            stream[256..319] = 4D9FAB87C56867A687A03BF3EDCC224A
                               C54D04450AB6F78A642715AF62CF5192
                               15E2CDF5338E45554B852B6FB552BCAF
                               5C599BDF9FA679962F038976CDA2DEFA
            stream[448..511] = E0F80A9BF168EB523FD9D48F19CA96A1
                               8F89C1CF11A3ED6EC8AEAB99082DE99B
                               E46DE2FB23BE4A305F185CF3A8EA377C
                               CA1EF46FD3192D03DCAE13B79960FEF4
                  xor-digest = AB020EA09B2573D7106EAA1D177F2E4A
                               1F8E2237AD1481F9923DDF973A79CFC2
                               1A0B8CDDD22D3D78C488D0CC9BE8FAA8
                               C74F0F2CFE619B7D7EA5B2E697E23372

Set 2, vector#243:
                         key = F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3
                          IV = 0000000000000000
               stream[0..63] = E7BC9C13F83F51E8855E83B81AF1FFB9
                               676300ABAB85986B0B44441DDEFAB83B
                               8569C4732D8D991696BD7B6694C6CB20
                               872A2D4542192BE81AA7FF8C1634FC61
            stream[192..255] = 0B429A2957CBD422E94012B49C443CBC
                               2E13EFDE3B867C6018BABFDE9ED3B803
                               6A913C770D77C60DCD91F23E03B3A576
                               66847B1CACFCBCFF57D9F2A2BAD6131D
            stream[256..319] = EA2CBD32269BB804DD2D641452DC09F9
                               64CB2BCD714180E94609C1209A8C26D1
                               256067F1B86AA4F886BB3602CF96B4DD
                               7039F0326CD24D7C2D69DE22D9E24624
            stream[448..511] = CA0DD398EA7E543F1F680BF83E2B773B
                               BB5B0A931DEADDEC0884F7B823FC686E
                               71D7E4C033C65B03B292426CE4E1A7A8
                               A9D037303E6D1F0F45FDFB0FFE322F93
                  xor-digest = 0D67BC1CFE545A6AE2F51A7FB2F32FC6
                               2E08707F9CBF2E08245E4594E9DB2A7E
                               CBB6AB7190831C3D7D8F9D606231668E
                               447C4EA29D69B4344952A97A77CC71CB

Set 2, vector#252:
                         key = FCFCFCFCFCFCFCFCFCFCFCFCFCFCFCFC
                          IV = 0000000000000000
               stream[0..63] = C93DA97CB6851BCB95ABFAF547C20DF8
                               A54836178971F748CF6D49AEF3C9CE8C
                               E7D284571D871EFD51B6A897AF698CD8
                               F2B050B6EB21A1A58A9FC77200B1A032
            stream[192..255] = 5B4144FD0C46CEE4348B598EEF76D16B
                               1A71CBF85F4D9926402133846136C59F
                               BE577B8B7EB8D6A67A48358573C06876
                               6AC76A308A14154E2FA9BD9DCA8842E6
            stream[256..319] = 3BF67A79DF6FE3C32DA7A53CD0D37237
                               16A99BF7D168A25C93C29DF2945D9BCB
                               F78B669195411BD86D3F890A462734AB
                               10F488E9952334D7242E51AC6D886D60
            stream[448..511] = 65629AA9654930681578EEC971A48D83
                               90FBF82469A385B8BCF28B2C1E9F13CE
                               FC06F54335B4D5DE011F3DCE2B94D38F
                               1A04871E273FCD2A8FA32C0E08710E69
                  xor-digest = E308FAEC064EC30CA1BEA7C2A02E95F4
                               ABCBF7D7762557BE9872726F9020162F
                               9B4EA11F621426EED6297C947BB3FAC2
                               69A8D0F38672EFBD72FDCCBEB8475221

Test vectors -- set 3
=====================

Set 3, vector#  0:
                         key = 000102030405060708090A0B0C0D0E0F
                          IV = 0000000000000000
               stream[0..63] = 2DD5C3F7BA2B20F76802410C68868889
                               5AD8C1BD4EA6C9B140FB9B90E21049BF
                               583F527970EBC1A4C4C5AF117A5940D9
                               2B98895B1902F02BF6E9BEF8D6B4CCBE
            stream[192..255] = AB56CC2C5BFFEF174BBE28C48A17039E
                               CB795F4C2541E2F4AE5C69CA7FC2DED4
                               D39B2C7B936ACD5C2ECD4719FD6A3188
                               323A14490281CBE8DAC48E4664FF3D3B
            stream[256..319] = 9A18E827C33633E932FC431D697F0775
                               B4C5B0AD26D1ACD5A643E3A01A065821
                               42A43F48E5D3D9A91858887310D39969
                               D65E7DB788AFE27D03CD985641967357
            stream[448..511] = 752357191E8041ABB8B5761FAF9CB9D7
                               3072E10B4A3ED8C6ADA2B05CBBAC298F
                               2ED6448360F63A51E073DE02338DBAF2
                               A8384157329BC31A1036BBB4CBFEE660
                  xor-digest = F3BCF4D6381742839C5627050D4B227F
                               EB1ECCC527BF605C4CB9D6FB0618F419
                               B51846707550BBEEE381E44A50A406D0
                               20C8433D08B19C98EFC867ED9897EDBB

Set 3, vector#  9:
                         key = 090A0B0C0D0E0F101112131415161718
                          IV = 0000000000000000
               stream[0..63] = 0F8DB5661F92FB1E7C760741430E15BB
                               36CD93850A901F88C40AB5D03C3C5FCE
                               71E8F16E239795862BEC37F63490335B
                               B13CD83F86225C8257AB682341C2D357
            stream[192..255] = 002734084DF7F9D6613508E587A4DD42
                               1D317B45A6918B48E007F53BEB3685A9
                               235E5F2A7FACC41461B1C22DC55BF82B
                               54468C8523508167AAF83ABBFC39C67B
            stream[256..319] = 3C9F43ED10724681186AC02ACFEC1A3A
                               090E6C9AC1D1BC92A5DBF407664EBCF4
                               563676257518554C90656AC1D4F167B8
                               B0D3839EB8C9E9B127665DCE0B1FD78C
            stream[448..511] = 46B7C56E7ED713AAB757B24056AF58C6
                               AD3C86270CFEAE4AADB35F0DB2D96932
                               1A38388D00ED9C2AD3A3F6D8BE0DE7F7
                               ADA068F67525A0996DE5E4DF490DF700
                  xor-digest = FDAEDE318DDD9EE44670318D51E812A2
                               F9B6EAEB18B9EBDC0FB76D95CD0AE8C9
                               5792F6EA71332404798505D947B89B04
                               1D56FAD3B0D92BEC06428EC5A841EB82

Set 3, vector# 18:
                         key = 12131415161718191A1B1C1D1E1F2021
                          IV = 0000000000000000
               stream[0..63] = 4B135E9A5C9D54E6E019B5A2B48B9E6E
                               17F6E6667B9D43BC3F892AD6ED64C584
                               4FE52F75BD67F5C01523EE026A385108
                               3FBA5AC0B6080CE3E6A2F5A65808B0AC
            stream[192..255] = E45A7A605BCFBBE77E781BBE78C270C5
                               AC7DAD21F015E90517672F1553724DDA
                               12692D23EC7E0B420A93D249C4383566
                               22D45809034A1A92B3DE34AEB4421168
            stream[256..319] = 14DEA7F82A4D3C1796C3911ABC2EFE9D
                               C9EB79C42F72691F8CB8C353ECBCC0DC
                               6159EC13DFC08442F99F0F68355D704E
                               5649D8B34836B5D2C46F8999CD570B17
            stream[448..511] = CA6A357766527EA439B56C970E2E089C
                               30C94E62CB07D7FE1B1403540C2DA9A6
                               362732811EF811C9D04ED4880DC0038D
                               5FDCE22BDE2668CD75107D7926EC98B5
                  xor-digest = DE518E6B67BAEC2A516CCAB0475341C4
                               BCC652ABE49ECCAA64E87248441A8F72
                               7BE173CACEBF8895B07DE8DDD28F1EE8
                               AA739855F1E6DB70765AB1B55BC3B1ED

Set 3, vector# 27:
                         key = 1B1C1D1E1F202122232425262728292A
                          IV = 0000000000000000
               stream[0..63] = E04A423EF2E928DCA81E10541980CDE5
                               C8054CC3CF437025B629C13677D41167
                               21123EE13F889A991C03A2E5ADC0B12B
                               9BBC63CB60A23543445919AF49EBC829
            stream[192..255] = F6E1D7DBD22E05430EBFBEA15E751C83
                               76B4743681DE6AC3E257A3C3C1F9EC6A
                               63D0A04BF3A07F64E6B167A49CD3FDAA
                               B89A05E438B1847E0DC6E9108A8D4C71
            stream[256..319] = FC2B2A1A96CF2C73A8901D334462ED56
                               D57ABD985E4F2210D7366456D2D1CDF3
                               F99DFDB271348D00C7E3F51E6738218D
                               9CD0DDEFF12341F295E762C50A50D228
            stream[448..511] = 1F324485CC29D2EAEC7B31AE7664E8D2
                               C97517A378A9B8184F50801524867D37
                               6652416A0CA96EE64DDF26138DB5C58A
                               3B22EF9037E74A9685162EE3DB174A0E
                  xor-digest = 697048C59621DBC7D47B6BE93A5060C4
                               B2DFBDB1E7E444F1FC292C06C12974D1
                               26EA9C8FD09C63945E4D9107CD0A1AC5
                               7161CA8C7CFEF55CB60E52666C705EC6

Set 3, vector# 36:
                         key = 2425262728292A2B2C2D2E2F30313233
                          IV = 0000000000000000
               stream[0..63] = 361A977EEB47543EC9400647C0C16978
                               4C852F268B34C5B163BCA81CFC5E746F
                               10CDB464A4B1365F3F44364331568DB2
                               C4707BF81AA0E0B3AB585B9CE6621E64
            stream[192..255] = E0F8B9826B20AEEC540EABA9D12AB8EB
                               636C979B38DE75B87102C9B441876C39
                               C2A5FD54E3B7AB28BE342E377A328895
                               6C1A2645B6B76E8B1E21F871699F627E
            stream[256..319] = 850464EEED2251D2B5E2FE6AE2C11663
                               E63A02E30F59186172D625CFF2A646FA
                               CB85DC275C7CA2AF1B61B95F22A5554F
                               BAD63C0DCC4B5B333A29D270B6366AEF
            stream[448..511] = 4387292615C564C860AE78460BBEC30D
                               ECDFBCD60AD2430280E3927353CEBC21
                               DF53F7FD16858EF7542946442A26A1C3
                               DA4CEFF5C4B781AD6210388B7905D2C7
                  xor-digest = 2FADEF81A5C4051CAC55E16C68CC6EEF
                               CEE2D4966BAE782E3D885CAA2271EFBB
                               E33F9313FD00632DC73441823713A487
                               94C21E812E30A1DD4B2AE858A27E7C88

Set 3, vector# 45:
                         key = 2D2E2F303132333435363738393A3B3C
                          IV = 0000000000000000
               stream[0..63] = 9F25D8BD7FBC7102A61CB590CC69D1C7
                               2B31425F11A685B80EAC771178030AF0
                               52802311ED605FF07E81AD7AAC79B6A8
                               1B24113DB5B4F927E6481E3F2D750AB2
            stream[192..255] = DAEF37444CB2B068124E074BAD188195
                               3D61D5BA3BFBF37B21BC47935D74820E
                               9187086CEF67EB86C88DDD62C48B9089
                               A9381750DC55EA4736232AE3EDB9BFFE
            stream[256..319] = B6C621F00A573B60571990A95A4FEC4A
                               C2CA889C70D662BB4FF54C8FAAE0B7C4
                               5B8EC5414AE0F080B68E2943ABF76EA2
                               ABB83F9F93EF94CB3CFE9A4CEED337CD
            stream[448..511] = 6F17EAE9346878BB98C97F6C81DD2E41
                               5FDEB54305FE2DF74AFC65627C376359
                               FB2E7841FF75744A715DF952851C1CBC
                               DD241BADF37B3618E0097B3A084E1B54
                  xor-digest = 8D1890B66A56552BE334B3472344F53D
                               D2782D4ABB4514D0F5B761436C997402
                               02A4B1244A1A7F485EFDB52C0065263F
                               EE5A7D7DFC2BB754304CE9B2724119EB

Set 3, vector# 54:
                         key = 363738393A3B3C3D3E3F404142434445
                          IV = 0000000000000000
               stream[0..63] = 3466360F26B76484D0C4FD63965E5561
                               8BDBFDB2213D8CA5A72F2FE6E0A13548
                               D06E87C8A6EEA392FE52D3F5E0F6559D
                               331828E96A07D99C6C0A42EFC24BA96D
            stream[192..255] = AB7184066D8E0AB537BB24D777088BC4
                               41E00481834B5DD5F6297D6F221532BC
                               56F638A8C84D42F322767D3D1E11A3C6
                               5085A8CA239A4FDD1CDF2AC72C1E354F
            stream[256..319] = 55F29F112B07544EDA3EBB5892DBB91E
                               46F8CBC905D0681D8E7109DF816ABFB8
                               AE6A0F9833CDF34A29F25D67A60D3633
                               8A10346FEBE72CCF238D8670C9F2B59C
            stream[448..511] = 0657453B7806D9EA777FFFBE05028C76
                               DCFF718BC9B6402A3CAEC3BCCB7231E6
                               D3DDB00D5A9637E1E714F47221FFCC11
                               B1425D9653F7D777292B146556A89787
                  xor-digest = C2A8D317E3B1CB884A2C3B07F11FD388
                               33282A9FBD1F6AF5C33CBE1E18D99B64
                               99A241EA83A56605BC6B99259FBAAED4
                               BDDA788B08CAAA93D2E00C6B5392ECF0

Set 3, vector# 63:
                         key = 3F404142434445464748494A4B4C4D4E
                          IV = 0000000000000000
               stream[0..63] = 40AD59C99464D95702727406E4C82C85
                               7FA48911319A3FCC231DC91C990E19D4
                               D9D5972B6A6F21BD12C118365ECAABC8
                               9F9C3B63FFF77D8EA3C55B2322B57D0E
            stream[192..255] = DBF23042C787DDF6FFCE32A792E39DF9
                               E0332B0A2A2F2A5F96A14F51FAAB7C27
                               14E07C3ADCA32D0DE5F8968870C7F0E8
                               1FE263352C1283965F8C210FC25DE713
            stream[256..319] = 455E3D1F5F44697DA562CC6BF77B9309
                               9C4AFAB9F7F300B44AD9783A9622BD54
                               3EFDB027D8E71236B52BEE57DD2FB3EE
                               1F5B9022AB96A59AE7DF50E6933B3209
            stream[448..511] = F11D47D8C57BBF862E0D6238BC0BF6A5
                               2500A62BB037B3A33E87525259B8E547
                               35F664FCEDF11BA2C0F3AEB9C944BCE7
                               7FFD26D604674DF8905A73CB7E230A4F
                  xor-digest = F021DE2B24C80A48DE6F7F807F1EF2F8
                               13D72A77E7BFC12515F9F5755CEFF64C
                               B5829CA780627A7920F3963E28005677
                               B85A56017A6F5A403DA49F8F8B71581D

Set 3, vector# 72:
                         key = 48494A4B4C4D4E4F5051525354555657
                          IV = 0000000000000000
               stream[0..63] = D8B1A4CB2A5A8DE1F798254A41F61DD4
                               FB1226A1B4C62FD70E87B6ED7D57902A
                               69642E7E21A71C6DC6D5430DCE89F16F
                               CCC9AAD48743974473753A6FF7663FD9
            stream[192..255] = D4BA9BC857F74A28CACC734844849C3E
                               DCB9FB952023C97E80F5BFA445178CAB
                               92B4D9AA8A6D4E79B81993B831C73765
                               10E74E30E7E68AD3188F8817DA8243F2
            stream[256..319] = B7039E6F6C4D5D7F750ED014E6501188
                               17994F0D3C31B071CC16932A412E627D
                               2486CCB9E43FCA79039D3E0F63577406
                               F5B6420F5587CF9DAC40118AA6F170A8
            stream[448..511] = 1ABA14E7E9E6BA4821774CBC2B63F410
                               381E4D661F82BAB1B182005B6D42900D
                               C658C6224F959E05095BC8081920C8AD
                               11148D4F8BD746B3F0059E15C47B9414
                  xor-digest = AD0620EB4E71605CDEA447A02E638F0C
                               2A0096EA666010761DB03CFC85629680
                               44D213B15EC69E1E5811EEBE7C96B616
                               6BE36E42B16F9F4BE0CC71B456C1FCA1

Set 3, vector# 81:
                         key = 5152535455565758595A5B5C5D5E5F60
                          IV = 0000000000000000
               stream[0..63] = 235E55E2759C6781BBB947133EDD4D91
                               C9746E7E4B2E5EF833A92BE6086C57C6
                               729655D4C4253EC17ACF359012E80175
                               7E7A6EB0F713DEC40491266604B83311
            stream[192..255] = 247BEAAC4A785EF1A55B469A1AEE8530
                               27B2D37C74B8DA58A8B92F1360968513
                               C0296585E6745E727C34FFCE80F5C72F
                               850B999721E3BF1B6C3A019DBEE464C1
            stream[256..319] = E7DDB25678BF6EECA2DA2390C9F333EB
                               61CD899DD823E7C19474643A4DA31335
                               2556E44A9C0006C8D54B1FD0313D574A
                               08B86138394BA1194E140A62A96D7F01
            stream[448..511] = DB417F9C1D9FD49FC96DB5E981F0C3F8
                               484E3BDC559473963D12D982FEA287A3
                               9A36D69DDBBCF1CA2C9FB7F4B2B37F3D
                               A755838A67C48822F4C1E82E65A07151
                  xor-digest = 119D1DDC7C95982B6B035FD4A4D8C5C9
                               FD2518FFBC69C3C6A7F600174A391614
                               6287F19BDDDAB385D2C6A39C593935F2
                               88B2F3E8895B9519EC71BA453319CC1F

Set 3, vector# 90:
                         key = 5A5B5C5D5E5F60616263646566676869
                          IV = 0000000000000000
               stream[0..63] = F27A0A59FA3D1274D934EACCFA0038AF
                               C3B866D2BFA4A8BA81D698DBCA5B65D5
                               2F3A1AC9855BEEEB3B41C510F7489E35
                               AB22CB4444816208C282C461FF16A7BC
            stream[192..255] = 522594154A2E4843083ABCA886102DA8
                               14500C5AADAAB0C8FB40381B1D750F9D
                               A9A1831D8000B30BD1EFA854DC903D63
                               D53CD80A10D642E332DFFC9523792150
            stream[256..319] = 5D092D8E8DDA6C878A3CFBC1EC8DD13F
                               2A1B073916097AEC4C3E56A229D8E282
                               DDB656DAD60DBC7DF44DF124B19920FC
                               C27FCADB1782F1B73E0A78C161270700
            stream[448..511] = 8F75BF72995AD23E9ADFEA351F26E42B
                               E2BE8D67FB810ABCBD5FAE552DC10D1E
                               281D94D5239A4EA311784D7AC7A764FA
                               88C7FD7789E803D11E65DD6AC0F9E563
                  xor-digest = 55AC113CC018689601F39AA80FA4FA26
                               EE655D40F315C6B694FFAE74A09D382B
                               62A4E7C60F75167361871A82561FFAC4
                               53BFED061D6B01672008308C92D241FF

Set 3, vector# 99:
                         key = 636465666768696A6B6C6D6E6F707172
                          IV = 0000000000000000
               stream[0..63] = 654037B9120AEB60BD08CC07FFEC5985
                               C914DAD04CD1277312B4264582A4D85A
                               4CB7B6CC0EB8AD16475AD8AE99888BC3
                               FDE6A5B744851C5FC77EAB50CFAD021D
            stream[192..255] = E52D332CD0DE31F44CDCAB6C71BD38C9
                               4417870829D3E2CFDAC40137D066EA48
                               2786F146137491B8B9BC05675C4F88A8
                               B58686E18D63BE71B6FEFEF8E46D0273
            stream[256..319] = 28959548CE505007768B1AA6867D2C00
                               9F969675D6E6D54496F0CC1DC8DD1AFB
                               A739E8565323749EAA7B03387922C50B
                               982CB8BC7D602B9B19C05CD2B87324F9
            stream[448..511] = D420AEC936801FEE65E7D6542B37C919
                               0E7DB10A5934D3617066BEA8CC80B8EA
                               AAFC82F2860FA760776418B4FF148DFD
                               58F21D322909E7BF0EC19010A168FAF7
                  xor-digest = 5BAFB9BEA29B3658A5BBF649E09455B7
                               0FB262AB938B65FE71652A0662FF0FB5
                               14C35AF438A72A6122AC1AA8591477AE
                               AEB78214C63E41255E87230481D1A793

Set 3, vector#108:
                         key = 6C6D6E6F707172737475767778797A7B
                          IV = 0000000000000000
               stream[0..63] = 0DB7EA55A79C045818C29E99D8A4B664
                               33E4C77DF532D71BA720BD5D82629F12
                               76EF0BF93E636A6F71F91B947DFA7CAA
                               A1B0512AA531603197B86ABA2B0829D1
            stream[192..255] = A62EAFD63CED0D5CE9763609697E78A7
                               59A797868B94869EC54B44887D907F01
                               542028DEDDF420496DE84B5DA9C6A401
                               2C3D39DF6D46CE90DD45AF10FA0F8AAF
            stream[256..319] = 7C2AD3F01023BC8E49C5B36AFE7E67DC
                               A26CCD504C222BD6AF467D4C6B07B792
                               61E9714FDD1E35C31DA4B44DB8D4FC05
                               69F885F880E63B5ABB6BA0BFEE2CE80C
            stream[448..511] = 066D3C8D46F45891430A85852FF53744
                               8EBDD6CE8A799CCF7EAF88425FBD60D3
                               2A1741B39CC3C73371C2C9A36544D3C3
                               B0F02D2596ACC61C60A6671F112F185E
                  xor-digest = 6EE5BF7E194B03A7DDC92FC74A398FF8
                               22471FEF6DD399426F7372E445E1EE36
                               5ED7164CD09120A79CCF03D0A2A309DC
                               5932441B64DDC6FDC9E183DA9F825106

Set 3, vector#117:
                         key = 75767778797A7B7C7D7E7F8081828384
                          IV = 0000000000000000
               stream[0..63] = 3FE4BD60364BAB4F323DB8097EC189E2
                               A43ACD0F5FFA5D65D8BDB0D79588AA9D
                               86669E143FD5915C31F7283F1180FCAB
                               CDCB64B680F2B63BFBA2AF3FC9836307
            stream[192..255] = F1788B6CA473D314F6310675FC716252
                               8285A538B4C1BE58D45C97349C8A3605
                               7774A4F0E057311EEA0D41DFDF131D47
                               32E2EAACA1AB09233F8124668881E580
            stream[256..319] = FEF434B35F024801A77400B31BD0E735
                               22BEC7D10D8BF8743F991322C660B4FD
                               2CEE5A9FDE0D614DE8919487CBD5C6D1
                               3FEB55C254F96094378C72D8316A8936
            stream[448..511] = 338FD71531C8D07732FD7F9145BBC368
                               932E3F3E4C72D2200A4F780AF7B2C3AA
                               91C1ED44DBEAA9A2F1B3C64DCE8DCD27
                               B307A4104D5C755693D848BEA2C2D23B
                  xor-digest = 7ABF3C4E6E8CCAC05AA336DF2156E195
                               7DFDAD45995FF6268B9708DAED9C2097
                               F8F0F2A0EE5FBF4A7B511ED2E8E56179
                               93E915E9BAABA30D758A9691E9D8578A

Set 3, vector#126:
                         key = 7E7F808182838485868788898A8B8C8D
                          IV = 0000000000000000
               stream[0..63] = 062187DAA84742580D76E1D55EE4DE2E
                               3B0C454F383CFDDE567A008E4E8DAA3C
                               E645D5BEDA64A23F0522D8C15E6DA0AD
                               88421577A78F2A4466BD0BFA243DA160
            stream[192..255] = 4CC379C5CF66AA9FB0850E50ED8CC58B
                               72E8441361904449DAABF04D3C464DE4
                               D56B22210B4336113DAA1A19E1E15339
                               F047DA5A55379C0E1FE448A20BC10266
            stream[256..319] = BD2C0F58DBD757240AEB55E06D5526FE
                               7088123CE2F7386699C3E2780F5C3F86
                               374B7CB9505299D639B89D7C717BA8A2
                               AEED0C529F22F8C5006913D1BE647275
            stream[448..511] = 54D61231409D85E46023ED5EFF8FDC1F
                               7A83CACDDB82DD8D1FA7CDEA0E088A61
                               D02BCE7FA7EC3B73B66953DA467BE4B9
                               12EBE2A46B56A8BF0D925A919B7B22E3
                  xor-digest = 9F569A8133067D1D4651BAE70DB3FE20
                               1649A1DA469C7D7C0B0DF16968285BF4
                               ED0F36ED1CF9F213B2EC4BFF83D455FF
                               C8B19E82DAE61408141F221C255DDFAB

Set 3, vector#135:
                         key = 8788898A8B8C8D8E8F90919293949596
                          IV = 0000000000000000
               stream[0..63] = 1A74C21E0C929282548AD36F5D6AD360
                               E3A9100933D871388F34DAFB286471AE
                               D6ACC48B470476DC5C2BB593F59DC17E
                               F772F56922391BF23A0B2E80D65FA193
            stream[192..255] = B9C8DAC399EF111DE678A9BD8EC24F34
                               0F6F785B19984328B13F78072666955A
                               B837C4E51AC95C36ECBEFFC07D9B37F2
                               EE9981E8CF49FD5BA0EADDE2CA37CC8D
            stream[256..319] = 3B0283B5A95280B58CEC0A8D65328A7A
                               8F3655A4B39ECBE88C6322E93011E13C
                               FF0A370844851F4C5605504E8266B301
                               DD9B915CA8DCD72E169AEA2033296D7F
            stream[448..511] = 4F9CA1676901DDC313D4EE17B815F6B5
                               AC11AF03BF02517FB3B10E9302FCBF67
                               C284B5C7612BBE7249365BCAC07FD4C2
                               C7AE78F3FDA1880B2DAA20E4EC70F93B
                  xor-digest = 9B9EA936FD4385D3516304BEFC44BC6D
                               5B60C97925B52CE269F2843496DEBD33
                               5A07ADA2EC87BA27E306CFFB884935D7
                               74EE317C7307740B884095278D1DB0C2

Set 3, vector#144:
                         key = 909192939495969798999A9B9C9D9E9F
                          IV = 0000000000000000
               stream[0..63] = 0281FB6B767A90231AB6A19EB1E4FB76
                               A041063FE23AC835797DFA178CC2D7C2
                               8DFAD591D2EAF26A985332F8DC74537D
                               F7E0A5F26946BCF7D70B6C3D9DD859D2
            stream[192..255] = 088ED6D7AB26EEC97518EBF387B0644F
                               D22266E578F141A7218F94AE2EE5885A
                               67A9FA304F6880A781EE05C1251A7EAD
                               4C3025D833B59739C68D3D7F3A844148
            stream[256..319] = 6B48D13EC0EB1CD0CDAC5D5E09DC7BE4
                               AE02BE4283DDC7FA68E802A31508E6EA
                               7197E5AC10805FDEB6824AEEF8178BAA
                               45D7E419CF9237155D379B38F994EF98
            stream[448..511] = 7E71823935822D048B67103FF56A709A
                               25517DCE5CFBB807B496EEF79EFFBCD1
                               0D23BAD02758814F593B2CD4AC062699
                               AEC02B25A7E0D1BAE598AFDBE4333FE7
                  xor-digest = 0D4802AF0B0F92FFF2F80FE65FE5D1FB
                               DFEF122231028FE36CC164D1D39185A1
                               869AD43D08C6E1C9F8A9113CE2CEF0A0
                               22629C6FAC1C27E6DDF2A46C52293681

Set 3, vector#153:
                         key = 999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8
                          IV = 0000000000000000
               stream[0..63] = D4ACE9BF4A76822D685E93E7F77F2A79
                               46A76E3BF0910854C960331A41835D40
                               902BC1CF3F8A30D4C8391087EC3A03D8
                               81E4734A5B830EFD55DA84159879D97F
            stream[192..255] = 5BD8BB7ED009652150E62CF6A17503BA
                               E55A9F4ECD45B5E2C60DB74E9AE6C8BF
                               44C71000912442E24ED2816243A7794D
                               5B1203A246E40BE02F285294399388B1
            stream[256..319] = 55433BDEA349E8849D7DF899193F029A
                               9F09405D7AFE842CB2C79F0E55C88913
                               B0776825D8D036A69DDDCD6AFCA6588F
                               69F0946A86D32C3585F3813B8CCB56AF
            stream[448..511] = 0B67F00FA0BB7D1ED5E4B46A68794864
                               5239422656F77EF2AFEA34FFF98DA7A8
                               90970F09137AF0FABD754C296DD3C6F2
                               7539BC3AE78FFA6CDCCC75E944660BB4
                  xor-digest = 9D6D8BAB5F6EDB5450EA2D5751741351
                               199ED720B0572410FD698C99F2E0DB92
                               C0E62E68AEE0CC6CDB6EA8898BFD29E8
                               E106470DE4E5C66F94FE0258A2D24CA3

Set 3, vector#162:
                         key = A2A3A4A5A6A7A8A9AAABACADAEAFB0B1
                          IV = 0000000000000000
               stream[0..63] = 92A067C3724F662120C25FAF4B9EC419
                               C392D98E5CB8C5EE5842C1D5C704DE87
                               8C8C68C55BA83D63C5DEEC24CFF7230D
                               3F6FBF6E49520C20CFE422798C676A47
            stream[192..255] = 133C9A30B917C583D84FB0AAC2C63B5F
                               6758AC8C2951196E9460ADBE3417D914
                               90F0A195DC5682F984069506CA75DC1D
                               79A7AE1DCDF9E0219D4E6A005BA72EDD
            stream[256..319] = 091D38749503B63238B1E3260855B76C
                               5CFE9D012265FB7F58EB8CAA76B45645
                               9C54F051274DDAE06BEC6D7EB8B9FF59
                               5302D9D68F2AF1057581D5EE97CCEEDD
            stream[448..511] = 3FCCB960792B7136768BBA4C3D69C597
                               88F04602C10848A7BCBED112F860998D
                               9E9A788998D1DC760F7ECF40597446D8
                               F39CD4D4013F472BB125DE6A43E9799D
                  xor-digest = 12464226235C1DDDAFA37DF12F3A0444
                               42C0EEE521DBB7B3239C86ADB61AD6A0
                               A418D3804252DC3658A3AE82473023A8
                               D190E1EDB1DAFA3CF566573511CF8F19

Set 3, vector#171:
                         key = ABACADAEAFB0B1B2B3B4B5B6B7B8B9BA
                          IV = 0000000000000000
               stream[0..63] = AC3DE1B9F6DF6D6117B671A639BF0761
                               24A0A6D293B107554E9D662A8BFC3F34
                               17C59437C981A0FDF9853EDF5B9C38FE
                               74072C8B78FE5EBA6B8B970FE0CE8F1F
            stream[192..255] = 23112BD4E7F978D15F8B16F6EDB130D7
                               2F377233C463D710F302B9D7844C8A47
                               FB2DFDD60235572859B7AF100149C87F
                               6ED6CE2344CDF917D3E94700B05E2EEF
            stream[256..319] = E8DDFE8916B97519B6FCC881AEDDB42F
                               39EC77F64CAB75210B15FBE104B02FC8
                               02A775C681E79086D0802A49CE6212F1
                               77BF925D10425F7AD199AB06BD4D9802
            stream[448..511] = F9D681342E65348868500712C2CA8481
                               D08B7176A751EF880014391A54680992
                               6597B10E85761664558F34DA486D3D44
                               54829C2D337BBA3483E62F2D72A0A521
                  xor-digest = 75BEFA10DACA457FFE4753A13543F996
                               4CF17E6941318C931575A0865B1C86C1
                               2EE5E031EFD125A3D56C4B7846C19484
                               507CC551C5CB558533E288BA0D2C14F1

Set 3, vector#180:
                         key = B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3
                          IV = 0000000000000000
               stream[0..63] = 21BD228837BFB3ACB2DFC2B6556002B6
                               A0D63A8A0637533947615E61FE567471
                               B26506B3D3B23F3FDB90DFAC6515961D
                               0F07FD3D9E25B5F31B07E29657E000BF
            stream[192..255] = 2CF15E4DC1192CA86AA3B3F64841D8C5
                               CD7067696674B6D6AB36533284DA3ABF
                               D96DD87830AE8FA723457BE53CB3404B
                               7A0DCBB4AF48A40FC946C5DEB7BD3A59
            stream[256..319] = E3B15D2A87F61C2CE8F37DCEB896B5CA
                               28D1DA6A3A71704309C0175BB6116911
                               9D5CBE34FC8F052961FF15F2C8F06CD6
                               F8E889694E2C69E918DD29C33F125D31
            stream[448..511] = CCD1C951D6339694972E902166A13033
                               A1B0C07313DC5927FE9FB3910625332C
                               4F0C96A8896E3FC26EFF2AF9484D28B8
                               CB36FF4883634B40C2891FA53B6620B1
                  xor-digest = 1E6FA2DF675C21D1AA9819BA05D3C96D
                               3463D6F0758286BBB41A63F8748B94C8
                               B652C60C5D4655E8436F2379CA7088B4
                               9625667F386BC5A2F25FD0BFB0088FAA

Set 3, vector#189:
                         key = BDBEBFC0C1C2C3C4C5C6C7C8C9CACBCC
                          IV = 0000000000000000
               stream[0..63] = 7943AD4AA5F62E08E1AE450E84CFF27D
                               E3B204A2BCA315B981906D5A13F68AB0
                               34D3396EA8A41001AF49834368805B37
                               D5380FB14821E3F7F4B44231784306F3
            stream[192..255] = 415F5381C9A58A29045E77A1E91E6726
                               DFCEBC71E4F52B36DBD7432D158F2ADB
                               31CF5F52D8456952C09B45A16B289B7A
                               32687716B8EDFF0B1E5D0FC16DCCFA88
            stream[256..319] = CE317CB853E2AFA22392D4B8AE345A91
                               0807F8DE3A14A820CDA771C2F2F3629A
                               65A1CC7A54DDEC182E29B4DACEA5FBFA
                               4FAC8F54338C7B854CD58ABA74A2ACFF
            stream[448..511] = 5804F61C5C07EC3C2D37DF746E4C96A1
                               AD5E004C2585F3F401CB3AF62CB975F8
                               64375BE3A7117079810418B07DABCCEE
                               61B6EC98EA4F28B0D88941CB6BE2B9D2
                  xor-digest = 9DBDBD0C3B340F294B1EB42CAD3111F0
                               A5CF6A0B6206976022C6A2D6303A235B
                               717542C25397879A27480D67AC5A245D
                               0C58334CD801764A948060CA6F99E2D6

Set 3, vector#198:
                         key = C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5
                          IV = 0000000000000000
               stream[0..63] = A4FB9A02500A1F86145956E16D04975E
                               2A1F9D2283D8AD55C17A9BD6E0C8B561
                               6658132B8928F908FEC7C6D08DBFBC55
                               73449F28AA0EF2884E3A7637233E45CD
            stream[192..255] = 74D169573560C14692BBE2498FDA0ED7
                               866A11EE4F26BB5B2E9E2559F089B35E
                               C9972634C5A969DD16EB4341782C6C29
                               FBBF4D11ECB4133D1F9CA576963973EB
            stream[256..319] = D28966E675759B82EDE324ABA1121B82
                               EAB964AB3E10F0FE9DF3FCC04AFC8386
                               3A43FD6B7FC0AD592C93B80BE99207CB
                               A8A55DDEA56DD811AAD3560B9A26DE82
            stream[448..511] = E362A817CCD304126E214D7A0C8E9EB9
                               3B33EB15DE324DDDFB5C870EA22279C7
                               8E28EFF95974C2B935FC9F1BF531D372
                               EF7244D2CC620CEBDE5D8096AD7926B3
                  xor-digest = 3DD73F824FD1D9CB55B7E37C9C8A55C7
                               EBB0866564AEA680BBBD431554D89E81
                               FF280B563D5991438CEA5C183C607ADC
                               23CC72CDE3A4D2CEB27B81ED8E5C9215

Set 3, vector#207:
                         key = CFD0D1D2D3D4D5D6D7D8D9DADBDCDDDE
                          IV = 0000000000000000
               stream[0..63] = FF879F406EAF43FABC6BE563ADA47C27
                               872647F244C7FAE428E4130F17B47138
                               0E1E1CD06C50309760FDEE0BC91C31D0
                               CA797E07B173C6202D2916EEBA9B6D1C
            stream[192..255] = 61E724B288AECF393483371C1BE653F3
                               7BBA313D220173A43459F0BCE195E45C
                               49B3B5FB1B0539DE43B5B4F2960D8E6E
                               5BC81DAF07E9EFBB760881441FA8823B
            stream[256..319] = F77AC22945ECD60EBCAF4BA19A59B078
                               B3C3BC36D1DDA6B9969B458C2019D68E
                               FD04D75DDC6041BBCD69747651D2DA7F
                               BED721081F8147367585CABB1C50CF0C
            stream[448..511] = 7475DCD3545B810445AFCA0C0AFA93A9
                               11EA99991A5D639AB32DDF69AA21C45A
                               53DCB998FDAE5F9A82EC8501123EAE3D
                               99351C43311F8430DB3D230E12DA77D2
                  xor-digest = A61CDBCF6F79213D2A789543B0EA3D8A
                               22BA4FB8118C1D40AE56EC8238861566
                               20CED8AA76FFE917C1E52060F91EE73B
                               C75E913D072C50B3D939E04F69493553

Set 3, vector#216:
                         key = D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7
                          IV = 0000000000000000
               stream[0..63] = 2B4C4185E2FDFAE75DABFF32632FB5D9
                               823359F15E2D17FF74FAC844E5016A4A
                               64C2C47498A15029FBEB6E4893381E65
                               6D2A2B9712524827B151C6E67D990388
            stream[192..255] = D870A94C4856EF818C5D93B2187F09C7
                               32E4491103B8A49B14CDC118F1607E2D
                               8443740F20220DF076B981D90436E9C3
                               09282C1CEAAE6375002AD1CA9CCF720C
            stream[256..319] = 5091AE53E13948DAE57F6B0BE95B8F46
                               A1F53553767B98F9799A0F0AC468AEB3
                               40C20E23FA1A8CAE7387CEA127A7A0F3
                               635667BF028DE15179093B706306B99C
            stream[448..511] = 02323B1FA2C863D3B4A89CFC143013A6
                               EEA8265BBD1B8FE243DEA2F4B19A5726
                               593564E7E7021FD042F58077A5821C2F
                               415BC38D6DD2BE29A5400E4B1D65B2A2
                  xor-digest = 9B29085D13B4992B077E3A878A5918B5
                               92C98C8A83956EC20EFE673A24C48C91
                               5D8DB1A4A66F62F1A3E7D6ADF6DC8845
                               DD7A6D43F9DBF6C1EA21639060469AD6

Set 3, vector#225:
                         key = E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0
                          IV = 0000000000000000
               stream[0..63] = 9A5509AB6D2AB05C7DBA61B0CC9DD844
                               B352A293E7D96B5C0066ACDB548DB857
                               0459E989B83AF10A2C48E9C00E02671F
                               436B39C174494787D1ECEB3417C3A533
            stream[192..255] = 8A913EBA25B4D5B485E67F97E83E10E0
                               B858780D482A6840C88E7981F59DC51F
                               2A86109E9CD526FCFA5DBF30D4AB5753
                               51027E5A1C923A00007260CE7948C53D
            stream[256..319] = 0A901AB3EBC2B0E4CBC154821FB7A0E7
                               2682EC9876144C4DC9E05098B6EFCCCB
                               90E2F03837553C579CDD0A647D6A6963
                               50000CA57628B1E48E96242226A92ECC
            stream[448..511] = 9CDB39B79A464F2CCA3637F04EBAEA35
                               7A229FC6A9BA5B83171A0A8945B6F117
                               56EBC9F4201D0BA09C39F97767213046
                               32AA6A68ADE5B90268AEE335E13B1D39
                  xor-digest = 695757EDF4992CE9E1C088D62CAB18A3
                               8F56EE71F1F4866E88D1A02E07CB89B9
                               133F0B02A23BA39622E84E19DACDF323
                               97F29E50151F78524B717093131A10B1

Set 3, vector#234:
                         key = EAEBECEDEEEFF0F1F2F3F4F5F6F7F8F9
                          IV = 0000000000000000
               stream[0..63] = 37EDAFA4F5EDC64EBF5F74E543493A53
                               93353DE345A70467A9EC9F61EEFE0ED4
                               532914B3EA6C2D889DA9E22D45A7DD32
                               1EA5F1F6978A7B2E2A15D705DE700CE4
            stream[192..255] = C415739777C22430DAB2037F6287E516
                               B1CE142111456D8919E8CD19C2B2D30D
                               8A1B662C26137F20F87C2802A2F3E66D
                               8CEB9D3C1B4368195856249A379BD880
            stream[256..319] = 0381733EC9B2073F9E4E995447118411
                               2D99B23FA4A87B4025C6AF955E93E0D5
                               7DD37011E1624175F970BDA7D625224B
                               AB0F021E6453DBA894A5074C447D24BC
            stream[448..511] = F9D45C7E0E7A26F2E7E2C07F68AF1191
                               CC699964C01654522924A98D6790A946
                               A04CD9586455D5A537CBA4D10B3C2718
                               745C24875156483FE662B11E0634EAEA
                  xor-digest = E0FE8129B73BCADA14FB385E6D3DB22D
                               84C9755D63E93141202576FB5B2D3647
                               D47B2F6378BC8567E4416976443FAE76
                               3C2B5FA46F2670C301A5B22802513D2D

Set 3, vector#243:
                         key = F3F4F5F6F7F8F9FAFBFCFDFEFF000102
                          IV = 0000000000000000
               stream[0..63] = B935A7B6D798932D879795A182E7C194
                               BECEFF32522C2F3FFF55A5C6D32A91D2
                               BA9F144DB280ABA7BA8A7921AFA3BD82
                               CA742DDBEAF8AF72299936E9C2FEA59E
            stream[192..255] = 6F32248B6EF4CDAE06864B6477893440
                               F0E0217421D7081D1F0DA197B5263674
                               0E9BDD59068BEDE48BF52C43446C12CD
                               4F10ED22BFDDFA915FA0FB1A73F9139C
            stream[256..319] = BF01A4ED868EF9080DF80689E589897C
                               021DCA18073F9291E1D158DC26266556
                               728DD130629D3760F541439147F4C1CA
                               279FB98040E9FCE50998E42D6259DE1F
            stream[448..511] = 0F2B116CD687C91FBA1EDEAD586411E9
                               66D9EA1076863EC3FDFC254DD5C93ED6
                               AE1B01982F63A8EB13D839B2510AD02C
                               DE24210D97A7FA9623CAC00F4C5A1107
                  xor-digest = C6970385CA89CDFCACA9E90DA2A2FE99
                               58EF83B9BF04DBE7A3B3437503688831
                               05FF6665D9F91D4DBBBCAF31B555ED3D
                               D07C3AC824281730BF834693C596AD54

Set 3, vector#252:
                         key = FCFDFEFF000102030405060708090A0B
                          IV = 0000000000000000
               stream[0..63] = 09D36BFFDDCD3ADC8EB0ABEEB3794CE1
                               FFBDED9CFC315D21A53C221B27722FE3
                               F10E20D47DDCFD3CCDE9C1BAAF01F551
                               1D3F14F88BF741A7F6578C3BC9024B2B
            stream[192..255] = 552502A1B2D0F29806DE512F3314FC8E
                               19518E35D9DB1EBC9034EA46E5815AB9
                               DF0F403E997E676BF47C0116D5E9B817
                               26B99D65AA4315F1E5906F6E39B1297E
            stream[256..319] = 6BF351A501E8D1B4BAF4BFD04726DC4F
                               50200463DCC13FF3BE93E6C4D4304CE0
                               9E6A1CEA41BFB93D6DBAD713298F79CF
                               F6F5BB81F456E33A3396D02F2E33BDC5
            stream[448..511] = 715F8FFB2BC25CD89E46B706EF871207
                               EFE736AA3CB961B06E7B439E8E4F76E2
                               944AF7BD49EEC47B4A2FD716D191E858
                               59C74FD0B4A505ACE9F80EEB39403A1F
                  xor-digest = D51B519D78CDBC8DF5CB1CEA5EBBA6E4
                               6530535D84CBF1696EBF238D3F7AA4A1
                               D2F1EF5FF092DB57943E28501C64CFF0
                               4619197ED4A3D82EEEB2B2E9648D7494

Test vectors -- set 4
=====================

Set 4, vector#  0:
                         key = 0053A6F94C9FF24598EB3E91E4378ADD
                          IV = 0000000000000000
               stream[0..63] = BE4EF3D2FAC6C4C3D822CE67436A407C
                               C237981D31A65190B51053D13A19C89F
                               C90ACB45C8684058733EDD259869C58E
                               EF760862BEFBBCA0F6E675FD1FA25C27
        stream[65472..65535] = F5666B7BD1F4BC8134E0E45CDB69876D
                               1D0ADAE6E3C17BFBFE4BCE02461169C5
                               4B787C6EF602AF92BEBBD66321E0CAF0
                               44E1ADA8CCB9F9FACFC4C1031948352E
        stream[65536..65599] = 292EEB202F1E3A353D9DC6188C5DB434
                               14C9EF3F479DF988125EC39B30C014A8
                               09683084FBCDD5271165B1B1BF54DAB4
                               40577D864CD186867876F7FDA5C79653
      stream[131008..131071] = C012E8E03878A6E7D236FEC001A9F895
                               B4F58B2AF2F3D237A944D93273F5F3B5
                               45B1220A6A2C732FC85E7632921F2D36
                               6B3290C7B0A73FB61D49BC7616FC02B8
                  xor-digest = 196D1A0977F0585B23367497D449E11D
                               E328ECD944BC133F786348C9591B35B7
                               189CDDD934757ED8F18FBC984DA377A8
                               07147F1A6A9A8759FD2A062FD76D275E

Set 4, vector#  1:
                         key = 0558ABFE51A4F74A9DF04396E93C8FE2
                          IV = 0000000000000000
               stream[0..63] = BA1A48247B8C44AAF12F5645D65FF7F4
                               E4D7C404EE0CBB691355FAEB82D03B99
                               AD0FDFC20A1E593973E5B8F0264F7FB0
                               538292A4C8FE8218A1DA3EB7B71EEA64
        stream[65472..65535] = 03A24E89D69D5E1DA98B0367CF626F33
                               D558B1208AB120B6B1778BFF640F56DA
                               715FE1B681D8CC0F305D6645B439BA81
                               D3C446A428B31BB18E9DA1E2A900B0FD
        stream[65536..65599] = 6A28ADD4F926759CEBB0AFC5D5DA5243
                               1F2E7ECBBD1E9DEAF368137E35F1AFBD
                               65852214FA06310C3175FCF364810F62
                               7E3703E9AC5458A8B681EB03CEECD872
      stream[131008..131071] = E8D8AB5E245B9A83A77B30F19E3706F0
                               37272E42F9C6CD7E8156C923535EF119
                               B633E896E97C404C6D87565EEA08EB7F
                               F6319FF3E631B6CDD18C53EE92CCEEA0
                  xor-digest = 2BD4F834BC7B3C128E291B2BCE7DA0A5
                               BA1A17E2785093B7F32B7D605AE63276
                               F8256998EC1E0B5A7FD2D66EE9B0B705
                               E49435EDF8BACE1BE770738A403B8F14

Set 4, vector#  2:
                         key = 0A5DB00356A9FC4FA2F5489BEE4194E7
                          IV = 0000000000000000
               stream[0..63] = 8313F4A86F697AAC985182862E4FC623
                               3511C46B6DAEEDB94B63461111CB4768
                               72F1BC3B4E8EE80A4ADE7D1A8CD49C17
                               1D3A550D3F39B7775734225579B8B60A
        stream[65472..65535] = 6AFA6F539C0F3B0B9DEB0235E7EB2E14
                               B111615D4FBC5BF7FFE75E160DEDA3D9
                               932125469AEC00539ECE8FCF8067CB0F
                               B542C2064267BEA7D9AD6365314D5C2C
        stream[65536..65599] = 296F2B5D22F5C96DA78304F5800E0C87
                               C56BC1BACD7A85D35CFECE17427393E1
                               611975CC040D27DF6A5FABC89ADDE328
                               AE8E9CB4F64CFA0CB38FE525E39BDFE4
      stream[131008..131071] = 86C8139FD7CED7B5432E16911469C7A5
                               6BDD8567E8A8993BA9FA1394348C2283
                               F2DF5F56E207D52A1DA070ABF7B516CF
                               2A03C6CD42D6EA2C217EC02DF8DDCA9C
                  xor-digest = DEEBF1FCF222519E26EC6556EA449080
                               92923B357CB88D1A1C1B03341F5C6A98
                               4C70E9DB735377615C0476D46DA9897B
                               48127A0D224241E79FE8CF51B005EF93

Set 4, vector#  3:
                         key = 0F62B5085BAE0154A7FA4DA0F34699EC
                          IV = 0000000000000000
               stream[0..63] = 62765613D127804ECD0F82D208D70156
                               3B1685EEF67945DAE2900307CDB14EA6
                               2474A439D8BAE8005493455471E7BCB9
                               DB75F0596F3FB47E65B94DC909FDE140
        stream[65472..65535] = 00A0D5B2CE7B95E142D21B57B187C29C
                               19B101CD063196D9B32A3075FB5D54A2
                               0D3CE57CBEC6CA684CB0E5306D5E21E5
                               657F35B8FB419A0251EA5CD94113E23B
        stream[65536..65599] = AAC2D29404A015047DEFB4F11460958D
                               A989141026FE9325F15954363FC78898
                               D4A20F6870F4D2B124590973F6956096
                               940E2324F7C63384A85BACF53F7755E3
      stream[131008..131071] = 0A543607FE352336ACFEDFE6B74359E0
                               B26B19FD45A8938C6C0A6DB68A137749
                               5B65211558D0CB9ECA9DA2C0E50702B6
                               88B2DEC53AAA2FBF11BD149F4F445696
                  xor-digest = D124AA942DC1D54D5B9B4BC6804F9990
                               543EAF31FF441F0CD16B961C817EA4A7
                               6AF71F678BBB482052B2BA767B4F9265
                               B65C3D839D182D093B560AEB09184C0C

Test vectors -- set 5
=====================

Set 5, vector#  0:
                         key = 00000000000000000000000000000000
                          IV = 8000000000000000
               stream[0..63] = B66C1E4446DD9557E578E223B0B76801
                               7B23B267BB0234AE4626BF443F219776
                               436FB19FD0E8866FCD0DE9A9538F4A09
                               CA9AC0732E30BCF98E4F13E4B9E201D9
            stream[192..255] = 462920041C5543954D6230C531042B99
                               9A289542FEB3C129C5286E1A4B4CF118
                               7447959785434BEF0D05C6EC8950E469
                               BBA6647571DDD049C72D81AC8B75D027
            stream[256..319] = DD84E3F631ADDC4450B9813729BD8E7C
                               C8909A1E023EE539F12646CFEC03239A
                               68F3008F171CDAE514D20BCD584DFD44
                               CBF25C05D028E51870729E4087AA025B
            stream[448..511] = 5AC8474899B9E28211CC7137BD0DF290
                               D3E926EB32D8F9C92D0FB1DE4DBE452D
                               E3800E554B348E8A3D1B9C59B9C77B09
                               0B8E3A0BDAC520E97650195846198E9D
                  xor-digest = 104639D9F65C879F7DFF8A82A94C130C
                               D6C727B3BC8127943ACDF0AB7AD6D28B
                               F2ADF50D81F50C53D0FDFE15803854C7
                               D67F6C9B4752275696E370A467A4C1F8

Set 5, vector#  9:
                         key = 00000000000000000000000000000000
                          IV = 0040000000000000
               stream[0..63] = 1A643637B9A9D868F66237163E2C7D97
                               6CEDC2ED0E18C98916614C6C0D435B44
                               8105B355AE1937A3F718733CE1526231
                               6FA3243A27C9E93D29745C1B4DE6C17B
            stream[192..255] = CDDB6BD210D7E92FBFDD18B22A03D66C
                               C695A93F34FB033DC14605536EEEA06F
                               FC4F1E4BACFCD6EB9DA65E36C46B26A9
                               3F60EAA9EC43307E2EA5C7A68558C01A
            stream[256..319] = 5FC02B90B39F3E90B8AEC15776F2A94F
                               D8C26B140F798C93E1759957F99C613B
                               8B4177A7B877D80A9B9C76C2B84E21A6
                               DF803F0DB651E1D0C88FB3743A79938F
            stream[448..511] = B4BC18F7279AC64BB6140A586F45AC96
                               E549C0CA497F59B875C614DE605A8BFF
                               63AB3F1E00DAEAE7A5CC7A7796E9BACC
                               DD469E9100EABCD6E69301EA59C4B76A
                  xor-digest = 4EF8F9A7D50D7ABEC1A104565E9E20BF
                               35FACFDD5600B0360E3ECBDE626CC693
                               4A52173415C05BA5EE681D649CB60D18
                               6970CF18BC028AF829054903FDEB37BA

Set 5, vector# 18:
                         key = 00000000000000000000000000000000
                          IV = 0000200000000000
               stream[0..63] = 94B7B07E184BC24A0904290B2601FC3A
                               C70BEAD7B1FC3294360ED4EF16813453
                               0B4D1F3F28A3C3B248B2E914A8DCBD53
                               26A240C9BB361A8A93D023725BDCD4E3
            stream[192..255] = 27C7A2C4EAA1E2E8798CA71EA50B7E5A
                               CD9FC82263D11781EFC16142CFD21A63
                               4DB2B860B54A9979AFA187CE0667D176
                               23FC91EC1E5E6C31A8089628AC76F9F0
            stream[256..319] = C2CD243516E5919D6C5C478469260813
                               ABE8E6F54BE8E11D48FEC043CDADA19B
                               EFE9CB0C22A9BB30B98E4CFCF1A55EF1
                               263B209CE15FEAEF8237CFAF7E5286D6
            stream[448..511] = 84489BD680FB11E5CAA0F5535ABA86DC
                               FF30AC031CEFED9897F2528035977726
                               70E1E164FA06A28DD9BAF625B576166A
                               4C4BF4CADD003D5DF2B0E6D9142DD8B3
                  xor-digest = 783AD910F37369EFB54DD9A00D54CDB7
                               2EEAF2693C121B13344025E08DF874AC
                               4BBC08B8FA916B423B0F4667A6D1BAEC
                               3016B999FF9FAB317161422E4FF925AB

Set 5, vector# 27:
                         key = 00000000000000000000000000000000
                          IV = 0000001000000000
               stream[0..63] = 2E6C8BE7DD335292EE9152641B0E4EFB
                               43D27434E4BE70EAC4CAFAE5C38B2E5B
                               06E70B9966F4EDD9B4C4589E18E61F05
                               B78E7849B6496F33E2FCA3FC8360824C
            stream[192..255] = 1006D6A04165A951C7EE31EEB0F6C32B
                               D0B089683C001942886FCEF9E700D15A
                               DB117652735C546D30177DC14FA68708
                               D591C3254C05B84BF0DCBC3105F06A6F
            stream[256..319] = 2196ADA05BED2BD097A43E4C5BE6C940
                               4A353689939DCB9C4F82278BDB0EB505
                               F70FFD9921B46645EDDFCF47405FD3E6
                               7CAE732B367A0B0F2B57A503161FA5DE
            stream[448..511] = 4A3504DAC25F59489C769090D822E89E
                               1338AC73F22DB2614B43D640525EF996
                               9D6B7E3900ADCBE056AB818E0FF708E3
                               B0A8E63531F252C384DD3DE7318EA866
                  xor-digest = 33533F81725EA5444E0642A07A334AE5
                               AC3DD16214F6FE196A60A4343AFA5026
                               E1602E84D3E672EEDB9FB5BB6F44C023
                               66C28BD8E3CF673BB34F438CF82561E2

Set 5, vector# 36:
                         key = 00000000000000000000000000000000
                          IV = 0000000008000000
               stream[0..63] = 1D3FD8BAF2A13BCD2A49B50F8DFB0522
                               8E366B4FD2ECD6973DFF116289D7E0AF
                               55EFB875345204B5FCE27A1C6DF79531
                               B3175647526BF5C028C454BADEFBECD6
            stream[192..255] = F639D0D23CC5817501517216ADA14241
                               D08495F17CDEAFB883CE619A3255EC3F
                               EAADFA224CF354C425A74D3DDAAA0C86
                               E44016238C142B36944EF53A1EC7DF92
            stream[256..319] = 9CAE4D4639696A188E08BC1B01774608
                               5D18418F82DC90742BB6D172414ACC13
                               A4721B018B2CC002CB6E6FFE4A4E252C
                               C4BF5DE975684C8805036F4C76660DC8
            stream[448..511] = CB2A2CB3136F5CC71FD95A4A242B15E5
                               1C8E3BAE52FEC9C1B591B86DFDDC2442
                               353DF500B2B9868A6C609655FC1A3E03
                               347608D12D3923457EEEB34960F4DB31
                  xor-digest = D623CA4753D2197E68B87B1ACBD84CC9
                               A056EC02F83D7E399CE2C4ACCF7934A5
                               A0CAE68FC0EB88098AA39DA88881C7B2
                               4C137195F32DA5CA86631CB84A6BC3B2

Set 5, vector# 45:
                         key = 00000000000000000000000000000000
                          IV = 0000000000040000
               stream[0..63] = 2DCAD75F5621A673A471FDE8728FACF6
                               D3146C10A0903DE12FBDCE134CC0F11B
                               2D2ABBDBADFA19303E264011A1B9EFEC
                               AB4DFBC37E3D0F090D6B069505525D3A
            stream[192..255] = 02C401ACF6D160CC1D80E11CB4F3038A
                               4C5B61C995CD94E15D7F95A0A18C49D5
                               DA265F6D88D68A39B55DB3505039D13E
                               AB9DEBD408CE7A79C375FD3FEBEF86C8
            stream[256..319] = 83D92AF769F5BF1FA894613D3DF447EB
                               D461CFFC0CA3A9843E8441EC91DEBC67
                               BE9162EABC5607A6D3FCAD4426EF4F9F
                               3B42CEC8C287C194B2211DEA4549D5D5
            stream[448..511] = D3F86930112EAFC7AA430444693BAE77
                               3F014D0798CAF3652A3432460F326DA8
                               8E82BE1E08C220B5FCBCE238B982E37D
                               1E60DCBF1747D437D42DB21ADF5EECF2
                  xor-digest = 0BF26BADEFCB5BB32C43410920FF5E0F
                               2720E8BB1C94DD5D04F0853F298C3ABA
                               8FF670AF163C5D24BCAF13AD0A04196A
                               2B89E82CF88846C77C77A097E234010F

Set 5, vector# 54:
                         key = 00000000000000000000000000000000
                          IV = 0000000000000200
               stream[0..63] = D8E137C510CDBB1C788677F44F3D3F2E
                               4C19FCEB51E7C2ECBDB175E933F44625
                               C7B0168E446CCCA900B9DB12D53E89E1
                               B917A69BDB888935B3B795D743D0D0E6
            stream[192..255] = E168F81B5BFB769F3380690D423E251E
                               0F4BEEBE0B02F19AFFADBD94212B8063
                               D77A665FD53F8F1A1CC682599C74F415
                               3642EC7DADA034403A90E1E5DA40C896
            stream[256..319] = 574774CFB8452E82777371616E0AC224
                               E29939E725B99EA8CFB4A9BF459A70D6
                               AB1991E85E06905ACCDA8D1911F82835
                               9C4FD7614A55C1E30171934412D46B3E
            stream[448..511] = 21FE9B1F82E865CC305F04FA2C69EA97
                               6D90A41590A3BD242337D87D28E3041D
                               3D0F74CA24A74453CB679FDFFEE45AA6
                               3B2DDE513D3F9E28E86346D9A4114CD7
                  xor-digest = 3E25D50331D9840FBD4F8B0FD10A9D64
                               6A5E8E0ADE57CCDECF346B2973631740
                               382139165B0E0E78A53E4B6CAABE6517
                               BF02B7B2905F9A64A60F412CA78E6929

Set 5, vector# 63:
                         key = 00000000000000000000000000000000
                          IV = 0000000000000001
               stream[0..63] = 42DCF10EA1BCBA82C88DDCDF905C9C78
                               42A78AE57117F09CE51517C0C70063CF
                               1F6BC955EF8806300972BD5FC715B0ED
                               38A111610A81EBA855BB5CD1AEA0D74E
            stream[192..255] = 261E70245994E208CDF3E868A19E26D3
                               B74DBFCB6416DE95E202228F18E56622
                               521759F43A9A71EB5F8F705932B0448B
                               42987CEC39A4DF03E62D2C24501B4BDE
            stream[256..319] = 9E433A4BF223AA0126807E8041179CC4
                               760516D3537109F72124E3534A24EA7D
                               B225C60063190FD57FF8595D60B2A8B4
                               AE37384BB4FCD5B65234EE4FB0A1EBEA
            stream[448..511] = 3F9803DD763449758F008D77C8940F8A
                               FB755833ED080A10513D800BA3A83B1C
                               028A53AED0A65177C58B116E574745D0
                               F28506A9DACD6F8A3D81613E00B12FDB
                  xor-digest = C0CA35A30730FCE3A6B08FD9707EBD1C
                               8154F54266696A99430BCA8B9F94FDD1
                               A78CCB43CB67C58EFF3B171A38597F12
                               AA6A424088C062B97613691B7D12CDE6

Test vectors -- set 6
=====================

Set 6, vector#  0:
                         key = 0053A6F94C9FF24598EB3E91E4378ADD
                          IV = 0D74DB42A91077DE
               stream[0..63] = 05E1E7BEB697D999656BF37C1B978806
                               735D0B903A6007BD329927EFBE1B0E2A
                               8137C1AE291493AA83A821755BEE0B06
                               CD14855A67E46703EBF8F3114B584CBA
        stream[65472..65535] = 1A70A37B1C9CA11CD3BF988D3EE4612D
                               15F1A08D683FCCC6558ECF2089388B8E
                               555E7619BF82EE71348F4F8D0D2AE464
                               339D66BFC3A003BF229C0FC0AB6AE1C6
        stream[65536..65599] = 4ED220425F7DDB0C843232FB03A7B1C7
                               616A50076FB056D3580DB13D2C295973
                               D289CC335C8BC75DD87F121E85BB9981
                               66C2EF415F3F7A297E9E1BEE767F84E2
      stream[131008..131071] = E121F8377E5146BFAE5AEC9F422F474F
                               D3E9C685D32744A76D8B307A682FCA1B
                               6BF790B5B51073E114732D3786B985FD
                               4F45162488FEEB04C8F26E27E0F6B5CD
                  xor-digest = 620BB4C2ED20F4152F0F86053D3F5595
                               8E1FBA48F5D86B25C8F31559F3158072
                               6E7ED8525D0B9EA5264BF97750713476
                               1EF65FE195274AFBF000938C03BA59A7

Set 6, vector#  1:
                         key = 0558ABFE51A4F74A9DF04396E93C8FE2
                          IV = 167DE44BB21980E7
               stream[0..63] = EF5236C33EEEC2E337296AB237F99F56
                               A48639744788E128BC05275D4873B9F0
                               FAFDA8FAF24F0A61C2903373F3DE3E45
                               9928CD6F2172EA6CDBE7B0FBF45D3DAD
        stream[65472..65535] = 29412152F2750DC2F951EC969B4E9587
                               DCD2A23DAADCBC20677DDFE89096C883
                               E65721FC8F7BFC2D0D1FD6143D8504CB
                               7340E06FE324CE3445081D3B7B72F3B3
        stream[65536..65599] = 49BFE800381794D264028A2E32D318E7
                               F6FD9B377ED3A12274CE21D40CCEF04D
                               55791AF99849989C21D00E7D4E7B9FF4
                               D46AABC44AED676B5C69CF32BE386205
      stream[131008..131071] = C3E16260DD666D8D8FBF1529D0E8151A
                               931663D75FA0046132E4AD78D8BE7F8D
                               7F41AAEFDE58BA80B962B8B68762CDF3
                               E4B06E05D73D22CC33F1E1592D5116F4
                  xor-digest = 10879B33D24115E4774C71711B563B67
                               CCD891E3825EDB58E182EC92648AE51C
                               DDC29A6A776C0AB3182DDDA1E180D55D
                               FAB024A3121BE45ECA59FF1A3715434C

Set 6, vector#  2:
                         key = 0A5DB00356A9FC4FA2F5489BEE4194E7
                          IV = 1F86ED54BB2289F0
               stream[0..63] = 8B354C8F8384D5591EA0FF23E7960472
                               B494D04B2F787FC87B6569CB9021562F
                               F5B1287A4D89FB316B69971E9B861A10
                               9CF9204572E3DE7EAB4991F4C7975427
        stream[65472..65535] = B8B26382B081B45E135DF7F8C468ACEA
                               56EB33EC38F292E3246F5A90233DDDC1
                               CD977E0996641C3FA4BB42E7438EE04D
                               8C275C57A69EEA872A440FC6EE39DB21
        stream[65536..65599] = C0BA18C9F84D6A2E10D2CCCC041D736A
                               943592BB626D2832A9A6CCC1005DDB9E
                               A1694370FF15BD486B77629BB363C3B1
                               21811BCCFB18537502712A63061157D8
      stream[131008..131071] = 870355A6A03D4BC9038EA0CB2F4B8006
                               B42D70914FBFF76A80D2567BE8404B03
                               C1124BCE2FD863CE7438A5680D23C5E1
                               F8ED3C8A6DB656BFF7B060B8A8966E09
                  xor-digest = 888FA87DB4EC690A180EF022AF6615F0
                               677DB73B6A9E0CFACEBBB5B2A8816B2A
                               D0338A812E03F4DFB26AF9D66160348C
                               B9EE72B63B2866E8281A2DB793A3A68E

Set 6, vector#  3:
                         key = 0F62B5085BAE0154A7FA4DA0F34699EC
                          IV = 288FF65DC42B92F9
               stream[0..63] = 71DAEE5142D0728B41B6597933EBF467
                               E43279E30978677078941602629CBF68
                               B73D6BD2C95F118D2B3E6EC955DABB6D
                               C61C4143BC9A9B32B99DBE6866166DC0
        stream[65472..65535] = 906258725DDD0323D8E3098CBDAD6B7F
                               941682A4745E4A42B3DC6EDEE565E6D9
                               C65630610CDB14B5F110425F5A6DBF18
                               70856183FA5B91FC177DFA721C5D6BF0
        stream[65536..65599] = 09033D9EBB07648F92858913E220FC52
                               8A10125919C891CCF8051153229B958B
                               A9236CADF56A0F328707F7E9D5F76CCB
                               CAF5E46A7BB9675655A426ED377D660E
      stream[131008..131071] = F9876CA5B5136805445520CDA425508A
                               E0E36DE975DE381F80E77D951D885801
                               CEB354E4F45A2ED5F51DD61CE0994227
                               7F493452E0768B2624FACA4D9E0F7BE4
                  xor-digest = 0F4039E538DAB20139A4FEDCF07C00C4
                               5D81FD259D0C64A29799A6EE2FF2FA8B
                               480A8A3CC7C7027A6CE0A197C4432295
                               5E4D4B00C94BF5B751E61B891F3FD906



End of test vectors

Primitive Name: Salsa20
=======================
Profile: SW & HW
Key size: 256 bits
IV size: 64 bits

Test vectors -- set 1
=====================

(stream is generated by encrypting 512 zero bytes)

Set 1, vector#  0:
                         key = 80000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = E3BE8FDD8BECA2E3EA8EF9475B29A6E7
                               003951E1097A5C38D23B7A5FAD9F6844
                               B22C97559E2723C7CBBD3FE4FC8D9A07
                               44652A83E72A9C461876AF4D7EF1A117
            stream[192..255] = 57BE81F47B17D9AE7C4FF15429A73E10
                               ACF250ED3A90A93C711308A74C6216A9
                               ED84CD126DA7F28E8ABF8BB63517E1CA
                               98E712F4FB2E1A6AED9FDC73291FAA17
            stream[256..319] = 958211C4BA2EBD5838C635EDB81F513A
                               91A294E194F1C039AEEC657DCE40AA7E
                               7C0AF57CACEFA40C9F14B71A4B3456A6
                               3E162EC7D8D10B8FFB1810D71001B618
            stream[448..511] = 696AFCFD0CDDCC83C7E77F11A649D79A
                               CDC3354E9635FF137E929933A0BD6F53
                               77EFA105A3A4266B7C0D089D08F1E855
                               CC32B15B93784A36E56A76CC64BC8477
                  xor-digest = 50EC2485637DB19C6E795E9C73938280
                               6F6DB320FE3D0444D56707D7B456457F
                               3DB3E8D7065AF375A225A70951C8AB74
                               4EC4D595E85225F08E2BC03FE1C42567

Set 1, vector#  9:
                         key = 00400000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 01F191C3A1F2CC6EBED78095A05E062E
                               1228154AF6BAE80A0E1A61DF2AE15FBC
                               C37286440F66780761413F23B0C2C9E4
                               678C628C5E7FB48C6EC1D82D47117D9F
            stream[192..255] = 86D6F824D58012A14A19858CFE137D76
                               8E77597B96A4285D6B65D88A7F1A8778
                               4BF1A3E44FC9D3525DDC784F5D99BA22
                               2712420181CABAB00C4B91AAEDFF521C
            stream[256..319] = 287A9DB3C4EEDCC96055251B73ED361B
                               A727C2F326EF6944F9449FB7A3DDC396
                               A88D9D0D853FADE365F82789D57F9B40
                               10F963BC498F176A93FD51723FCD4D55
            stream[448..511] = E0D62E2E3B37FDD906C934FAA35D5E8A
                               89A517DD0F24CF33DE8495C5FF24F4B1
                               476B3E826A1C90D74507C3991CEF4067
                               E316A04B97AEFFA5E9D1F33CB0609B9E
                  xor-digest = 44936C5AE8EA99630920CEC7C0FE9E8E
                               A6C5166366D543D3A6FCCE3EAE9B0DF6
                               28C61B62CABD61B44F5610440C6798E9
                               3B820711202105D120398ECB96C0C102

Set 1, vector# 18:
                         key = 00002000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = C29BA0DA9EBEBFACDEBBDD1D16E5F598
                               7E1CB12E9083D437EAAAA4BA0CDC909E
                               53D052AC387D86ACDA8D956BA9E6F654
                               3065F6912A7DF710B4B57F27809BAFE3
            stream[192..255] = 77DE29C19136852CC5DF78B5903CAC7B
                               8C91345350CF97529D90F18055ECB75A
                               C86A922B2BD3BD1DE3E2FB6DF9153166
                               09BDBAB298B37EA0C5ECD917788E2216
            stream[256..319] = 1985A31AA8484383B885418C78210D0E
                               84CBC7070A2ED22DCAAC6A739EAD5881
                               8E5F7755BE3BF0723A27DC69612F18DC
                               8BF9709077D22B78A365CE6131744651
            stream[448..511] = 9618FCA736A8ECA00BD1194FC9855085
                               526ECD47A8DE1F8DB298AD49FCE935EA
                               63B548597092ABAD6338F41AF87586A7
                               0505F2537902B81F55E53599DABA84CC
                  xor-digest = C442D7538E8129F048E38EA1A6FFA5F8
                               29F5B54D26A01DB1C0FA1B2E07418FB1
                               872C5D96CDC250746C26BD803903E28D
                               7DEC66DED9AB7DE6797C502B3D1B246D

Set 1, vector# 27:
                         key = 00000010000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = FF852567EB72687DC56C122D61B2FB2A
                               4FB9E8E8DA62313B618D10F8E0DA521B
                               176E879CD78E641043F0FA4A22211566
                               429B7C68EC645FF5E44B2505D61A2D71
            stream[192..255] = E5B040B199C3DFC8DB1F41C74C798AE2
                               62105477AEB1CE761D6FFF1CAB15AA1A
                               7B7CE26B9CCE6DC33FD4522BF8F73E70
                               B843D67FC06FA2258F9709DB14FBD54C
            stream[256..319] = 55706075E5FED81E2205994609868EFC
                               383B3E4CC295C4214356BA41FC72BFE5
                               4E6936FE6684EAF93C5973DDCD8E8F23
                               767B82D783953F89AF4E808C90BEEABD
            stream[448..511] = 7ECE71883742EE852C94F01AD85EA1A6
                               76CC7CBC6EDFCF1BAE751455A923FAAC
                               806BB72E6A982EC7A38F112445E25EB6
                               BC5B49C5E6C22DC8748DEE0942F6E8B2
                  xor-digest = 0058982850C947A63750238409A95FFF
                               CA5A559990EF1A60F038ADAAF965DD6B
                               3931693C24AF075CC27663683B7B15D1
                               0F7A4B6BD1AD61F35D67A7E632ADBF2D

Set 1, vector# 36:
                         key = 00000000080000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = AF6E2EE1D5021675A92F02C764AFD94A
                               F3097F53532FC965EB861D6D12A3A012
                               ABA683A5281238CE76E3AF3944736752
                               AD86A5FD16E7DAFAF241ECFB0ADBBDFE
            stream[192..255] = 19444E6D7C3D8BEC0957C3E785E1EEFD
                               56B857F21CF8D325A4285F8DEF5078FF
                               7B7EFB5E3B20F6E0906265B6F7580A04
                               9CEC5DF1872DCCB54081054C0FC15514
            stream[256..319] = 7EB544ADBF57D042E3A6753B13C65843
                               0399764CF90D007E48DAFE3DA1FE3F90
                               8EF4BFA6AF96DCD54197DA0D3A10FA35
                               6A374DA08B9A84044E70EC70ED050D46
            stream[448..511] = 57224DA912C62801DB393D5E3F4EDFF7
                               D61BA895F88C7391FE5C943B88CC4642
                               0D11C3F1884B628F03C04A3C10F03FFB
                               CFC652D066BFD8DBF52DA2A72B9B9AC5
                  xor-digest = 9796755E92C0BBCABFF65DCC865EF240
                               BF5C2105C98EFFD368155FB2DFD06D3F
                               2A4D7CCEE6A73DE51DF37F953BAF6BD7
                               EA28AE0B9DA94B7D2E05FC44389B4101

Set 1, vector# 45:
                         key = 00000000000400000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = D203CC523351942C94E215F6D5CC1425
                               C5FFB2EA9A916C0D4F7B343333A58D94
                               1DE20B5F543E3EE63C29D981469ACE48
                               86ED9DEF839D4FBD20CDF9D001F1B89B
            stream[192..255] = 9E37D2BE6473F4FA87ED294765816BB0
                               8CCA625418155F6704CB48082A860581
                               A9CF69D9145D0DCB2621E1515013DD3E
                               18819BEC5C186628ED545BFF7E4AC1C2
            stream[256..319] = B8648B92B5A7B3B991722F0053909A3F
                               052E8F7DABE7FE0E34498C1C550DE9D5
                               3CE0818DDBA82F0616B3F79AD72B0BF9
                               B5FA2F2B8032B1860FAB0804934FBD00
            stream[448..511] = 0CD554D10A975BEA79AEAC663F5FF984
                               15883EB558925C5ECFA53D77FAB4B884
                               FE4D705B1E1B34A938C1C2D8528E1FAB
                               4C9A7512F12707B78F2B6BFEE8D76E57
                  xor-digest = D570E4908F0A350A7D18A3039A94F070
                               3F5FB6A5630594BEB15E15EC175F8E46
                               E3F1A3ABD993DDE832224D3192DB8A1D
                               1E71A5009E2A6AB27C6A5989848F59C1

Set 1, vector# 54:
                         key = 00000000000002000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = C45E28A2C9A80AC07A760580DCD96340
                               26651B25BA2332FDAFC9AA16998317B9
                               751A446302CDE95525C709E79CB55951
                               4E4A54FD73ADAAF0AB3A3F1ADDABBADA
            stream[192..255] = 17937670127CBF691AFDAD6D36994F0A
                               40B3F369C21691B887CFE20B0F63D125
                               8896C88CAB669ED6FABE464A700DA937
                               C43AABB45E60F14E6EBA69FBC9F2FCF3
            stream[256..319] = 2690AB8F4616302C49D79CFE3AE29AA7
                               9C4D1036E0CBB1D24C4682BCA0E1C1A5
                               80904001185286AC3C63BFBF909F4A36
                               525D2A732D7D166A52E087444DE24469
            stream[448..511] = 9E5E91D8BE1E46B0BAD46ED9ACCD440A
                               01882556B51C2B7CCC987A6C554201FC
                               6CE8DA0B1CD42C011A085EB8FBA0F8F2
                               623B6B9627EAEB91C05CFA3090A28040
                  xor-digest = B33D0D25BFA4A067F09A452AC147B2F4
                               C46BEC615D17D020CB9F39393433CF8B
                               244607E5B7ACC4701A89EAB887D9035F
                               BC18163F59304929BAA5F25CE05FF43E

Set 1, vector# 63:
                         key = 00000000000000010000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 5F7B6B86B0C197B960D8250B5106CFEB
                               F6F4DE0D94D3958945FA979534AFE19C
                               D5305C55A1404C59302F05ACC819D3A3
                               B0BDB9D154A45C0DEE52F25012DAA445
            stream[192..255] = 20F99149AA74F631D22BEA8D85EC84A6
                               57C2E8703B45ED36458F0ED47408C3C7
                               E6624A184E7CED17C93CBC9960914A61
                               E71083308CB7A55D7723C2B9E6A2F087
            stream[256..319] = EBB0F7194EA7AE5D28B916D361B19394
                               A163A6EB124D37A372A798135E4F2FDF
                               2EF422997F5AA1F9DFA3B1826431AA62
                               99E0AEB44D844E297604D27974EAAD6B
            stream[448..511] = 65CA9CAE36B65F58085D561A91CFDBE1
                               EA0400CDEB4AA1B987FAC06702590D8B
                               39B6228E6F4B81BB91852971DE2D3436
                               C8C24FA193BC10BFC5534BF5915A245B
                  xor-digest = 884DDB758C2D7D1FA5B9DAC42756F87D
                               9C1CF9E1EAA1B0B5BB032D2E8763EAEE
                               261129C8340F163ABD66EEEEF507EAE5
                               BD230703C2A7C16FFCC23D063238F99C

Set 1, vector# 72:
                         key = 00000000000000000080000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = B96FCF5A182789AD14E53FB2E981E496
                               B47C6B44BE7EF95692F19AE24E193219
                               6E180778AC04A0EB2497497680587FEB
                               F412BB3A67E9538CA5B2A373E16E60F3
            stream[192..255] = 953544577886B26F2F8D7BD237D7AE8E
                               5D425523F6180C9591206E10E166C7E3
                               06537355EFD9C32FF1C8808537BA12D5
                               B0E303DBCEC7DB3DA6E3A16DACB1E7FB
            stream[256..319] = 9B416AA89BDC5589A1C9046D2D308B8A
                               CA852008C6503B373250C2639C693D9E
                               164FC0E94FCFBB35D67D45DE1A3D838F
                               302915E78470EB47654B87540AADF90A
            stream[448..511] = 3911737593809A1A9FD14F57950AEFCA
                               66E1E45475D39335DC01FFA72E431A85
                               01E146994FAA64BA37AF255F1951B33F
                               CB28AAC76BB08AA0917B53B9ED64CDAD
                  xor-digest = DFED948E3423CF3689FFCBFAD85BD98A
                               9785A39C7CAFCE10EC7C89686F9B45E0
                               F335D8FB649B07CA88D5D264CB47F133
                               32538D762A8A38C5E1EEE97D25DF6442

Set 1, vector# 81:
                         key = 00000000000000000000400000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 2B08D82E92AC352247211D5F0791DAC9
                               D585ABF67DADFBD7B5AC60EB2EEF4C72
                               F6F71CA110DEE4CB2F19FABE4F442B2F
                               5F9FB1C94FBD553C21CD5B0CEF139880
            stream[192..255] = AAD0055BF85562F06118CB260CB0BD5F
                               374CD798021593F03A67134EA8A73B22
                               F00F09BAB770D1287FFF17CCF5F1CF32
                               86833B57F4397B16A9F8351922042810
            stream[256..319] = 724D557F9D7DA4AFCB5DC6D1040DD8BF
                               A14A0CC61F7206606BC99385D15BFED8
                               9C4D69EFE5711A9E256C908AFF2734D6
                               501C9D1AEB7CCD1029413BF7FA40848C
            stream[448..511] = 8960F4D83E21984B3A6D5D1B667944ED
                               12814CD390B107A502A4BBA620E3CE9F
                               6DAF2D4629C828C59E86F09F1F435B4D
                               40A1595C3D5B6E0744FFA546B22EF865
                  xor-digest = E63E2A4BFE4910ACEAEF896FC499955A
                               04FCFE14F46F9A5273E9B61216A8AED3
                               77A50ECE7429AB7FD8FC3A97C3A1B70E
                               27B13A5B5486335B07132260FACD3E81

Set 1, vector# 90:
                         key = 00000000000000000000002000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = C9969A75572ABFAA28FBE769A287A676
                               3B534AF50B697C31B7F4CD8F50DDF2F2
                               17B3C5532E95F73AF11B0693D5A33A34
                               DAFBB64635A195EC9477FDFD69AE7540
            stream[192..255] = 6B358B53A60B9542F582FDE14B2711F2
                               6CD1B7421B4D872B95E347CDD7D763C8
                               73C2A8730A802AECA326FD63C8C4205C
                               FC1A6E2F4DF7A6ACF1E22A2BCA5379A9
            stream[256..319] = AF64A04DB6B9CA63429E0D81CE975FD0
                               2A5E3BB8C1A0C3D35636AE22F3733201
                               2DF59549BAC23E992A1E4DD481F91956
                               40C4D6EE0E083702DB18328D42D93BF7
            stream[448..511] = 3F3FD5559C9C0CE3B5B484BD15E75CAB
                               B252CC44961C1ACA86B1722FCF205408
                               EF9841F947224170ECAC6503F7A8FEAE
                               7281ED1D9A18C4C00D12C8E40F21876F
                  xor-digest = E759CFE55228EF54198A82FF8E11D269
                               05EC81CAD0A4736124A5391D34ADB50A
                               3956FA5321AFCD7AECA3E9D3E0FAF913
                               502358257426307F76EB55A8184DBA88

Set 1, vector# 99:
                         key = 00000000000000000000000010000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 698BFC90B147715FB9F0CA1DDC94EE10
                               3082316701CDD1DF2EAE752BA485F585
                               9E131D0D9233B16890BD5946CBCF116D
                               B50E8E2DCAE104162C7B76CB3D11445C
            stream[192..255] = 07D49AB7BA8451A2A68DF473C6D1E91D
                               407038568FADA2DB948ABFBBE408401F
                               DF5960241325F2981DC17EAF1C333CDC
                               91E27EC064734234656AED7A944AD78A
            stream[256..319] = C152FCF951DAECBD48EC1D0122A4EA00
                               9FB8FD03E35E283109DAA4E033783990
                               DADE92932BC6410CE1B6ADE414AAF782
                               8DA024FB2C3F4135DF6C42A347BD3E25
            stream[448..511] = BD0CD02750FE445A0C03D2EA30D73684
                               07DF4B13CBE8E3CE2DE2780F9A90983B
                               9EB919DEF1EC22EBEE10F584B6FE8F99
                               1374666D378C7C20CB5AD1771FA7C799
                  xor-digest = 7D24B8DDED30B6B17567C7AE46FE517B
                               C5700ED002A54CE30DB17A2ABC9F37F7
                               1F6395D4E2C7467AB73483B89AEDC212
                               B9FEEA18E4C08DBDFD8A39DA31FB7EC5

Set 1, vector#108:
                         key = 00000000000000000000000000080000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 07AE6801D7A94836ED52CCD69D9E97F6
                               34B136A234B978BAE4302F475B0A6B0E
                               A7905CEE090F648962BB969CB4D65228
                               03E1ACD1DCBEFC2E7482C0D426E4BD95
            stream[192..255] = 145DF9D539C59467F55E67D959FC8C8B
                               2CB0397F64D6F122C3F2F1A19E0D67B6
                               9696EADDC6DDA6E80D5A0C0AC1F555A9
                               21C054E0E75EBB246C8E20A854A38E93
            stream[256..319] = 2BF710E9709B5178E5E50B421BAAF59E
                               B1F267F41C60E9E91695D658BAD32497
                               B56868B8738BAA6A15BDE89D69900ED2
                               742F26285504C3D4748F77EECC0D4A67
            stream[448..511] = E93A249CE755F099C81FA40B5DA6256E
                               E185FA1EFC475EB404BB68C13A921FA5
                               78785537DD65964B9BF77F68DBAE4926
                               9F5061B19AF08B82C372AC69EB64D762
                  xor-digest = FA4EC052E9C9712474F1BCDB5C09A6D8
                               9A9F1843381F098CF3F9ADFDE0115133
                               FE977491C6365F5C5BB78BFF42B20EC2
                               60193927289B597BE42EA7A0FFF6C3A2

Set 1, vector#117:
                         key = 00000000000000000000000000000400
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = A374C1F86586B0D5A121E1F734EE70CC
                               7072284B322BF61F88336EBE84B53219
                               F4D1FEE2C5EECC4A421BA8AEA9D108E7
                               21A7A82DD979F2559BB0E45CC88C8780
            stream[192..255] = B0CA15C769D66B26CA4A6D4772AE3521
                               AEA4696890998954F33ACA8638FA50E2
                               9981C2F84596D9371644D18E3EB267E8
                               FCCC98D95A2FB38639D32468A3013B5F
            stream[256..319] = 1CC3AE9293EE9CA19C12D9ABD7000F99
                               047B86A868E82A839DD95418EECB23CB
                               4B4A08E3EF69CC639DBADF3F5F33FAD5
                               0762C2603DFC48882EE8D2346FDB426B
            stream[448..511] = 0D6EC570BB04230AC35B49A1271336CA
                               721E0395F63D306554158154CA12FB62
                               E8D45CF5E21A311554DE9DF5D90CA99E
                               9B7FAFEFAD3597B50A17FEEDD9966884
                  xor-digest = 4C47AD8677A22F3587C4C25DD7A4A8CF
                               E144E34FA7567088BD9DC4001911A531
                               83860E9CDCAB006CF50AC00B95C95725
                               BAE29E53648CD27B4183A213E5855E8A

Set 1, vector#126:
                         key = 00000000000000000000000000000002
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 19F23D5CB3C7303D56AFF18413835EF3
                               DF7405C30BE5F19C72FE8746BA04610D
                               D5D261FB3A0E8C11D2478F4A4D6CF820
                               9730187BB1386C03229F4EB02C5B4422
            stream[192..255] = 7B814D9DB8DC9C8397C23550DE194BE2
                               74694399A8B2BEF6B8095704C2A29E00
                               DEED66C8191F67BA9C048CA41DA4DB05
                               FDEAECBBD0727AD9664563991A22EA46
            stream[256..319] = 7B4DC904BA9FC0CBB054FB57DAE11C58
                               C9505A98E319B43FBB9C30DA2CA7E6B8
                               7A42F1E40774A6657EB3EB2C33B5D365
                               BB92A8CA0CCD5B71C17F7022DD840E14
            stream[448..511] = 5B2DB8E73DB53C289E8479F524953BAF
                               D881E8A366899440175CB2B93F8EBF25
                               3911652B3C7EA35B41B409B4BBD0BD93
                               95AE5A2AE2368B7A43A0F9844239E3C2
                  xor-digest = F80AD4DE8D937A57E230BFD00A4AB8C0
                               65DA8978DD9C51E6F998E393C0B888BA
                               C772F76BE8AD5D501425465AC5C05D82
                               63925189E928BCA52AC7F4A738D46102

Set 1, vector#135:
                         key = 00000000000000000000000000000000
                               01000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = B18CFBA23F81884FBFEA037648B1715C
                               EFAEF1D8CD5C98957353E82B838FE332
                               672B3D7C2905979698F6F6D98EAAE8F9
                               8DA16EF393CB150228FE6438440C5759
            stream[192..255] = BF285CEEEE6D66ED9A401AF86B4F1B0E
                               69B5ABF625D0C35220F9E6198FF5C225
                               A728EEBF67EDC8690ADFB6A2E43ED7BD
                               2956A4915A8FF4BC584C803C87B03956
            stream[256..319] = 0FBE7818D981B60177DD1C7ED21FC23F
                               F088EEB3A36A3DB18E37BAA312642BE6
                               481F6FBD4C6A3DCF6990D3F5E0F02813
                               F66F42B4384F3821E9F2A5CC7AC37029
            stream[448..511] = A72F53B68BF3E6972515790869B97667
                               E353E1CC089AFA194B8ACFCC4C033567
                               4B2E9E0290501D24D87B80AF12C636B9
                               3902F09252F77812802151798FDB831D
                  xor-digest = 8811ABBCDCD51C0E367CF0E2A78935BE
                               1FC6E462D026D995CC1B754E2DE7CB83
                               BC06112D2AC813006F2A32F8789AA939
                               4DDF3A43DF247BFE94F456054AA057A9

Set 1, vector#144:
                         key = 00000000000000000000000000000000
                               00008000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 0EEF3E17B6B9388FB55C2C0AEF9716CB
                               106786EEB0E606E124C41AB552EF3389
                               7902AA2AE93D9E4628E785B356C53AC9
                               70BDEE2A7DDBAB427371903EF3EC9FA5
            stream[192..255] = BA437BE85A1152B673AB7F39345534C2
                               6B53227FC8E99B6EEBCBBDC00B436DBD
                               E6AEF836EC78AC581F251D0C61F56404
                               D275B1DF39294B26CF24F4AC0792D176
            stream[256..319] = 381C3C583CFB20763CDBE072668FD1A2
                               557A35901CDC8595393181AF1610300E
                               D751154C050D8CE0354EFD30D05251A9
                               7F215A48F8924B4A68FD475C793A0543
            stream[448..511] = 15E30D96D2A42C99DB1030B5280A6313
                               2AA665B57DEB3AC6AAC8DDC1450C899B
                               D0DAE783A224134232687459917CC525
                               6D76929A153950DBFF7D12CA21EE77C9
                  xor-digest = 360A5D8ADE0ED311ADAFEDE072335406
                               51A262FA795DF4B5FD3FB0041702B961
                               49DD5DE99DE88D28EC9E87264AD9C5C1
                               6CC9C1A21E01678E6800B3140F6E34E8

Set 1, vector#153:
                         key = 00000000000000000000000000000000
                               00000040000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = AE5572D5E61A992162AEEE513815339C
                               93A994DB12576D087EA4A9A98EA5946C
                               F58794B43515A4B55C5E9B28A882DADE
                               7D3BFE82B32EC3B604D2C1E1B37B1B99
            stream[192..255] = 247616FFD99152BBFA71D2225AB667DD
                               1999ED6E2AC64F60F43B3DD1EA5E574A
                               47C52B82E3FBA3443996EB1E842D11EF
                               78572638CA556157674B0A38ADF26F8C
            stream[256..319] = 1BE7BBE4FA4078886183F1DC9E296911
                               96106D005F5D653AAE744B2506401723
                               30F38DA7C5CA81F38A879D79FAED5B23
                               37045434875074B65D7E126DAF8B728F
            stream[448..511] = 89048CF63BC3AC13B4637487735B9976
                               2707C4161EBD6788289F2BAE38D3B68D
                               14C9A49E26573E3604D8D9907D151C75
                               6728F3D9A2A6BC118E62390BC0DBACA9
                  xor-digest = 9FBC8803149CCF3117F715A0343789DE
                               B0D5B7B16E34689DF65B83B5111AF91C
                               590865CE8A73D8DD5A6D040832CA4604
                               488D0592F87F90D74545439F9A6DE8E5

Set 1, vector#162:
                         key = 00000000000000000000000000000000
                               00000000200000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = BA66E5BA75AD8C4030AE54B554E07A97
                               29685FDF033CCC35A153334E9FC93A90
                               3C79F281907BADF6F37123819ACA25E1
                               F03BA0AC69D9B2D5E447F59F31A7A402
            stream[192..255] = 6B0FC33710282B08A33917D23186B1CE
                               0964104B5B8FC229CFD79BAEFF04FF97
                               07AD12904B3673B15B72428BB3FDC0FD
                               DECFF9AF8606456774B1B3B53AE74C5F
            stream[256..319] = FFD0D5ECE17F9C1890199A4F201333F3
                               D55A0AE07B1DBC50A704FE66493B71AC
                               F802534FCD7BAF86B140CF87C582BC02
                               59EFE52CB2D1A64524F948A86F756E21
            stream[448..511] = 81EF72B6DD7F8043A078486BF0DFA634
                               7CF53FF6432432B45CC740533243D6E8
                               E936A5E6C1CB688388D6D97BFE48C430
                               0325A4B5DE69825E6CB5409FE9518708
                  xor-digest = 6F8913AC7CC348D3F4A795850332BA5B
                               9E6C87113F126686D5E10F728D8585AD
                               3872C4FD4CDB446C23FD56E288E1FEEF
                               3D65E9C600B7BAD4D476CCF40830B410

Set 1, vector#171:
                         key = 00000000000000000000000000000000
                               00000000001000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 59DBEE08FB86EBCBEBFFBF087F9DD881
                               2AFFFD75414B5162B5E7AE540BFA8777
                               5BEC4982E1F4B6985DC8B2B25F061947
                               61BD6BC5EFD66B2A1EB12833733E5490
            stream[192..255] = C54CDD55BBBC09038A772D1FEE876EF1
                               88110319FD6D7B306E9F5ACBF3C47824
                               9E4CD2C8C11900DBAA39F8F7D57724E3
                               70606016AFC49DEF5248964A416E0DC8
            stream[256..319] = EE1C6E2F9DA5404012821C3DBE703D47
                               1FF717042C20DDB4743246448F431DE1
                               53BADF69A059D161189D20B8F22F1F7C
                               C491B5B2F5CDFE7A779A0F9DB0C60586
            stream[448..511] = 85E92E3EA90E7EB79A9D3894D0B21153
                               DA80FCC6DA7631A1C38EB38C78A1BEF2
                               321265349CB5FCFA22E5FD02648BB37E
                               74D3152011F7640A0FD42DCC9457B2AC
                  xor-digest = AF4B9782FA0332C8A6344DECDAE4FC2A
                               B63769CA35799A664FCC0A91A6F3C0E0
                               689281D6D2BB4A856286BADB4986DD56
                               4B7BDEB2867F5A2E20FE0B4311C77924

Set 1, vector#180:
                         key = 00000000000000000000000000000000
                               00000000000008000000000000000000
                          IV = 0000000000000000
               stream[0..63] = FD1D039AE6D953654A63334A92CEC647
                               A671CAB6374DB63B89DA1A12B99C231D
                               C7B9418D44210CB0C88F114EAA54AE4A
                               096FEFCCBF51062E8EFD169715677F28
            stream[192..255] = 119152E46B97338C5E50A28DB78757E6
                               B21C9C03AA9D96B5FDAC9D352AADF2F9
                               FA0FCA07649582E7288297E9CC765846
                               2D929ACED1F14E3AEE634CD2086D1762
            stream[256..319] = F9C91CA01A70253BC6D88A8DFA00537C
                               E635634769E8867B279C1A052A921F14
                               8810FC8854BDF58F99E36FEDBC6E6E6F
                               78BC8F82DCD18D408B3B4F8BFEF12F12
            stream[448..511] = C22A3D49E727785EA32E83E79E349D62
                               C2647AC6D531BA2D466CCD7CF29D04D1
                               015D41A79C9BE4B0AE1844DBDBCD7FE6
                               765EB95A0D5E121F48840937AB399C6E
                  xor-digest = 422309F77B0CBD9E17E58372079547B2
                               A42EC06690AE889CB3095B8489734124
                               2D7951F73579E2F66B26503723CABA8F
                               5EAF869887D090BCF20344757A8105D0

Set 1, vector#189:
                         key = 00000000000000000000000000000000
                               00000000000000040000000000000000
                          IV = 0000000000000000
               stream[0..63] = 72491EC81A704E3694C83FCCC47CF5E8
                               7B66F7B7979F78D8150A606ACDCB4492
                               F64A9D7D9DAD5042F8738DB462F4728C
                               2475F5FDEE985CD3601FA31F576712C3
            stream[192..255] = 17566EFAC19AFD1ADDEC66F42695006C
                               EDFBA525E8F41DB02BE50D2AC4CB497E
                               A10C6DA38ACF39BB608F40AD854F69C4
                               4A0FC6696F6FA8361CF26D5411B1C7C9
            stream[256..319] = E3CE396F970BC54C9E46B6129B48616D
                               F7FBD0293B1EFEB772D99CA90BCE12A4
                               AF729DA0B94223A3D2F0B9605DC04BF9
                               AE82E065C1B963039802BE6354D3EB2C
            stream[448..511] = C0B2081FF9B7F2DDD59EE6808F6181F0
                               4CD19D4B0D3F032D5FC0EA2B81D49276
                               BD6E540648576CEAE720411523889D3C
                               F14BF05DA43D8D6155B7D98B021F269E
                  xor-digest = 85395D5EA1C9428817FCEA56DA0978E4
                               C4B244D9556FE87F19806664A8D9BC42
                               C0A69A717F1AAD4E09200120A517B73A
                               2F3FE01BE24B201508D801E416AD6AA6

Set 1, vector#198:
                         key = 00000000000000000000000000000000
                               00000000000000000200000000000000
                          IV = 0000000000000000
               stream[0..63] = E3D058FC000427B4F0802300E5D7FE9F
                               8E3F68E9E8339E9F4C5DE62252E14857
                               71371DE4D2E1C97DC4172AA378924AB4
                               2CADF887136B88D6FEB6514538EBA847
            stream[192..255] = 80CE800DC11805A7522E3B423699D68B
                               51BCCE201ECA4F8E465C5A58A558A71F
                               019A22593CBC148A76647A527E635A23
                               4096EB22F081F39B5A9DC7649277726B
            stream[256..319] = 30A91E7D2CDB7D1B080750B433A14F7B
                               6EE602EB53D67AC65B7E4219B533AA6C
                               CBC1FCAC070270D595CF9E90FD3C2D02
                               A707F7C1F97059DB3644F50D236933B0
            stream[448..511] = 79FA6D08B8DF687EFE868E67643CB5A9
                               FC5FECEEC258E67D831D20AD3C8CBECB
                               51F1712A0BAE64202FBF66A1FAE767C1
                               68A9B0C4BE89FCF2F6D2DBC5CA96A4BB
                  xor-digest = B76429935B5D578116D0A303D8A435C8
                               93BE1D132E2025922F192D5985E198BC
                               5F07F6F16A8FAD5CCFB4487826366AA9
                               B60FCED5994A8740B0DDFE3F9AD2A408

Set 1, vector#207:
                         key = 00000000000000000000000000000000
                               00000000000000000001000000000000
                          IV = 0000000000000000
               stream[0..63] = FF0D93064CDBD91A8D6BD0B9267A4F93
                               DF7D3C76BAA5D0D14798812203C55A34
                               3BD50E6931394DAB88F514F44E2A1FB5
                               8EF3A915F3B60DAB35E36174AD92B3B1
            stream[192..255] = 074A711F8BB92EA6953D21F9FD7AAEA9
                               1C12D18A2B18E8D325DB04029B5E8EBA
                               43C408D3D4EBE049440CFB716BC3ECA9
                               1929E009ED7EA0EA7273E32C13F44346
            stream[256..319] = 6BD5DE42827A81941C72012219EED591
                               BE1AFE19DF91C8B7284DF2AF4050D7EB
                               674DBE78680EF4F8963D59ACB05B43D6
                               A52B7CEBEBDED9D3268D0500699A036F
            stream[448..511] = 9748C1BA603FE3DD4435A25F2ABF18B4
                               9F25ECEBC3514785406425E03ACD369A
                               EC91463FDD5F3611F06870D513B10DB7
                               730F3328C22312DE7329DF8CB43DA5C2
                  xor-digest = BF19031B4F8884F9A290E543C517998E
                               C8442A227DE1D46F5F441D5D586A215D
                               B4A24A2965F5FD3C70ABF296C55C4333
                               E3C9B2FF671080BBA28C24FDBD2C8370

Set 1, vector#216:
                         key = 00000000000000000000000000000000
                               00000000000000000000008000000000
                          IV = 0000000000000000
               stream[0..63] = DCC597DC08E1AD1451E69D857AF803BB
                               DBF7CD6D510D5C59C9D6C66EB153CC79
                               F9A6228ADEE570983E959788628F174E
                               5833B5CFA350C0C2D8A18F7FE46BB4E1
            stream[192..255] = 8CCB839CB382DB591B5C80F6DD7EAE7E
                               AECB3C8BF29C9C6074058A5EA04E2E58
                               675B4537B8FD061BA7E4195AD2A3EC29
                               FD260FD19F0AAB3DCB7BD483ED8FB860
            stream[256..319] = 73E92E3449C863E55E9A41B0DB35805F
                               344FB07E4C3CEFF25B261819140C849B
                               E90639644C542880946582842CE5B1D9
                               FA2DF07B5589C8C68BED84E15DED4AF2
            stream[448..511] = 693C7F397D23C831431264E9BF4EE963
                               B8A43C6ED939B324FCB8AF1032BAC678
                               C71F1DE8BA3A8090948872FA9C747AB7
                               67F7D162FD8B6F484B81AA54151612A6
                  xor-digest = E6732941C20C6DFF6276F6A21A461777
                               638799041F1A360D7C8C88B1A8E9CA8D
                               989360840F8E16C72890220E51A7913B
                               6B5B74A70D75B7D9F26BF7FB5C8C7C78

Set 1, vector#225:
                         key = 00000000000000000000000000000000
                               00000000000000000000000040000000
                          IV = 0000000000000000
               stream[0..63] = C94A72C1B17F8B9F26420BF06B3A5445
                               20C658D5F77ED7D62CC65AF824BD5678
                               98EE4928AF0E2BEDEA64D5A7C22749C3
                               C16369D274EFD2A6DF2CFCCB130A1144
            stream[192..255] = 2130A7225D4C78BBBB8C5122C18851A9
                               32A78E360E62E56058027C624DA49EEC
                               34DCE5ED9F66D78B44334CE0E3317AFF
                               5BC78261FA4C96A642E846CDCEA4C242
            stream[256..319] = 575EAB318220A54E5B2B0A8EC7F54429
                               0719FE422C646E1114D807201416F37E
                               B5CECDB278AFC7CDE84E6DB5CA164840
                               2BF9654D1C4E96A3E7BF5C19C84CDA71
            stream[448..511] = EAFC6C17BF190180FFD817644D7933C2
                               F86989ADF705A72B04CDF8227A164596
                               7BADE4A0E706039BD84702395B9A44DC
                               7368E198B01335577A28028FE2F6056D
                  xor-digest = B26E45B88EB4359878EC471C5A5814D5
                               10F435CE44D1A77A50468BE21F48F7B3
                               7D5B2DDA0389405CEEA8998A9C3480CE
                               9F30A02408B065F28543BFCBBB159AC3

Set 1, vector#234:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000200000
                          IV = 0000000000000000
               stream[0..63] = 832A824C044E27605AD9A3201EF106C1
                               A19B6FC6EA5B328DC1D1FC59086C498D
                               47E7568CFA9616D7D5E63D9C087CC426
                               B4276752E0FF14D7F1E258F9A28A54BA
            stream[192..255] = CFC021E1EDACD733768D3412C0DA7177
                               7AF74D147D075BD5497BAD89B84D0A66
                               F7F4D0E46B77510AE3FB57C0DB9F9922
                               111337BDFF89A9169DB16B38F305BEC8
            stream[256..319] = CE311109342E1A41ADA17363B0AB030D
                               1BE9C62F15C2A5D8FEE2BC9819F2E064
                               6880D350E547824BDDFD5BE89C43F23D
                               FFA366BE34629F6EE929E2701EFA6829
            stream[448..511] = DCE864E5E336A7B51A7FFE9E4C8C1FBE
                               F5F4755A0877EE91D61D1F20F29485FA
                               A879323F2566590917417C4AC0076CB9
                               81EE78C58741506F725BC58743957CAC
                  xor-digest = 335D243CF73622A761D728CA87A083E8
                               F7EA67FEFE422833C9B963A9433DF43B
                               02606FC5D67FFAB71724AD78339F51DF
                               B7CFD78C0F5472EF7727AA5C967969EB

Set 1, vector#243:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000001000
                          IV = 0000000000000000
               stream[0..63] = 28DD9E566F018FDA0251E1E648057E85
                               211831E215AE21525E04C932736245C2
                               288AD4A197E4ECA04003B85C3B80D02A
                               9B82C28E7662A34467946A34257D8D0B
            stream[192..255] = DDC4A6A1AAF92AB32D2958DE67BBA593
                               338D7EE4E3A412C2374A5D63E6CD7F56
                               51F518251CEEFE1E63636DB2F432F407
                               88D4C0163738446515A62637695D782E
            stream[256..319] = 107AAEEDD6C459411921177468E3D013
                               50C40AEB41EE50AE196754BBCE5559B9
                               7276957DC73141981DC087209378F87F
                               89C8423ACE0EAE8C5EFEEDEBCBB20618
            stream[448..511] = A3FE61185B31AA80EA384B36CEC7F41F
                               19F2E55614BE22852E796963326B9F49
                               72E8A316D4A6653CCE3FE06014C0F5BB
                               6E4E64B439109608FEC6A44C15384C13
                  xor-digest = 3597FEEBE687F754D35F2BC480810C34
                               1A1D557F867C07B83CB8A8890CD090F0
                               0E6C6CA3CA9B804AD70B40747DCFF18C
                               7F830FA6630EFBAEAB4B022C22B892A6

Set 1, vector#252:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000008
                          IV = 0000000000000000
               stream[0..63] = E48C2F264BF9E8374B78FB652BAFF1E3
                               3ECB4B1C635D76A64ECFC4BDE00EE5C8
                               77E1094D6480CA382815CCCD5CC36770
                               46E801C29A860EB032420DCAEEBC36F4
            stream[192..255] = D2EEE83D63F96B0B7E6D8E0C72B6581D
                               50AF4081017CD62A73789C8C2DC5483F
                               CB4067C71FDBFD6EA8882FFBAC63BC9C
                               5E4F438A2ECBC71627646539A5BFE1DD
            stream[256..319] = BDDA0B90B24A4FF5D535E12D075DCE84
                               6D6741F809D105DC03552A3F13AC88B2
                               F98411A1C19CB32FA3F595CDD8F87608
                               3C057E42BDD903A055F13182CA080F4D
            stream[448..511] = 44E931EF73A9AFA565EB9A8E6AB1AA3B
                               9F14FC198B41909CB31B532F9EB776FA
                               B51FFD895E7F266D1D275463282BD7F6
                               62FBBBB5629890A4C68B6F6CF8200623
                  xor-digest = 151F615109FC211C0A7AA4DD2CEBB984
                               CFE9ED0486E8C552233AED574E9983F9
                               A9DD738D23F2D307DC313C634A42A805
                               18B616A250C0725694750A29413DA8F1

Test vectors -- set 2
=====================

Set 2, vector#  0:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000000
               stream[0..63] = 9A97F65B9B4C721B960A672145FCA8D4
                               E32E67F9111EA979CE9C4826806AEEE6
                               3DE9C0DA2BD7F91EBCB2639BF989C625
                               1B29BF38D39A9BDCE7C55F4B2AC12A39
            stream[192..255] = 2F3C3E10649160B44321B7F830D7D222
                               699FAE0E834C76C3997985B5404808AB
                               7E6E99AA1FEC2730749213E7F37A291A
                               A6B5AFD2E524C2D608F34D4959930436
            stream[256..319] = 8598D1FA94516B474B69DA83E3C1312C
                               49A05B8283B880B31872CD1EA7D8F1B2
                               D60A86CBA8184F949EA7AE8502A582DB
                               392E85C4D70D3D17B2E57D817A98ED6E
            stream[448..511] = F86C7489712FB77896706FC892D9A1C8
                               4BB53D081F6EB4AE1C68B1190CBB0B41
                               484E9E2B6FEA0A31BF124415921E5CF3
                               7C26493A5BC08F7620A8C80503C4C76F
                  xor-digest = 7C3A1499A63B507B0BC75824ABEEAA26
                               109101C5B915F0F554DD9950045D02FA
                               FF815CA8B2C7CFF3625765697B80B026
                               7EA87E25412564BD71DD05843A60465E

Set 2, vector#  9:
                         key = 09090909090909090909090909090909
                               09090909090909090909090909090909
                          IV = 0000000000000000
               stream[0..63] = 7041E747CEB22ED7812985465F503331
                               24F971DA1C5D6EFE5CA201B886F31046
                               E757E5C3EC914F60ED1F6BCE2819B681
                               0953F12B8BA1199BF82D746A8B8A88F1
            stream[192..255] = 4EE90AFB713AE7E01295C74381180A38
                               16D7020D5A396C0D97AAA783EAABB6EC
                               44D5111157F2212D1B1B8FCA7893E8B5
                               20CD482418C272AB119B569A2B9598EB
            stream[256..319] = 355624D12E79ADAB81153B58CD22EAF1
                               B2A32395DEDC4A1C66F4D274070B9800
                               EA95766F0245A8295F8AADB36DDBBDFA
                               936417C8DBC6235D19494036964D3E70
            stream[448..511] = 5CF38C1232023E6A6EF66C315BCB2A43
                               28642FAABB7CA1E889E039E7C444B34B
                               B3443F596AC730F3DF3DFCDB343C307C
                               80F76E43E8898C5E8F43DC3BB280ADD0
                  xor-digest = 8FE7F0A88BD73434364D9D9FFC95F837
                               2C8F2B8808A7996239C9FA7E81F61D46
                               AD3C4CD426F149B186A298C554CCE61E
                               661678F992DB556AFBDC541C814D3C21

Set 2, vector# 18:
                         key = 12121212121212121212121212121212
                               12121212121212121212121212121212
                          IV = 0000000000000000
               stream[0..63] = 7BCD4C5528F4BEAE0FC9F164CEBEC73E
                               D89CE32DA46EB68CA3CEDAA7C7A580FB
                               1C50D291F31C38DB2811864F6654098E
                               141A2213828593A98B7D0020BF0D6D93
            stream[192..255] = 87DCAB67C8D5A90D17AF198D3A22D432
                               BC82C06872F0E61B3A3D1A1FC14527D1
                               E8C3C9CA50E5BF529621C2860ED304F2
                               7E6E427A9BC64D0FC6E2E16BD40C434C
            stream[256..319] = 121F38D31A0ED8A6D72F4C6A4678A7B0
                               D3054A6268D02C9C6766069427722606
                               36CD6D79F81C64412A93F10DB68D1B86
                               962DFC41434B1C65AF4770F7D185514A
            stream[448..511] = BEDDFB9B60B204E0332726D7D7E90640
                               FF29318A164A9551D9FA477D7E437273
                               A0E08EC35046CAE10BDAEB959F44E9C2
                               A09FFFBAA7A89B7B9F1AF34948FFFE9D
                  xor-digest = 3F8C09148423C1FBE286530726434747
                               F6362345A359A66A6066EAD149C4B1C3
                               B33E35608825D5618D924A7D5CDE0CB8
                               F2A0626D34F894C293FCAA83D162A460

Set 2, vector# 27:
                         key = 1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B
                               1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B1B
                          IV = 0000000000000000
               stream[0..63] = 944B67EAB62DF3756085CEE577D0C1DA
                               4DD7CD17B85F9B9C51004107C8AA6935
                               7E413AEA37BB512BD8246F2D03E2748D
                               3BB24B60C1FBE4D1A55237FFE3D4D604
            stream[192..255] = A9574AD5FC6A0D4A57FBE98AB5122A54
                               E2C355524AAC38580C659AE4E906F14C
                               3FB5A096586FA808F5F266182D26C784
                               72B116652EE1874CB5CF007DF2E2BB5A
            stream[256..319] = EE5A306A60C83E209ACC5F3D60E17D90
                               FDDC0D790BBB7B1EEB635924A4C7AEBF
                               3ADE18F1F2F03C1E74093847B8F9225A
                               9588E92A826444BDD143B38CC3934FBD
            stream[448..511] = 33DDC526B91BD452296DC8ABAEE7C65A
                               E7D8CA37FE66166B67570726639841C8
                               559405236A37A104FAA3F5A1A1932D57
                               FFE36EC16D439B1C291DD11638C50730
                  xor-digest = 8FF9D4A8277BA858B70F05FBAF80FC6E
                               31AC1CC81E8D847721CB632FA757C4B4
                               BAF548A4764EBA7206009A5B99A00089
                               7717410EAA1D4ADC9F6D619EC2D6C511

Set 2, vector# 36:
                         key = 24242424242424242424242424242424
                               24242424242424242424242424242424
                          IV = 0000000000000000
               stream[0..63] = 0FDF243C21DA8B291097C9F385DFF2AD
                               4FDCA5EB4FA7E4C23CC61FA1A582EB23
                               5AE23454DF6F19B259E498F746F9EF35
                               491F77DC53BD596AACCB9FB7B5EE8ABC
            stream[192..255] = A92CE971EA8E2ED7614325F0C47CE1D7
                               200B94EEB7FB4E31CDE640696ED6449F
                               B29A9F19EABE323B776EE9460C2448E2
                               DF83206A401074E3254C5AD6C194BD99
            stream[256..319] = 6F988009D4C82F523611DE08FEA23680
                               02FA5A615E8EA831A76C7CABCC92E1BC
                               C02249FD76DDEA5C00FEBC391613857C
                               97CD684B23C6D9B40F1C5254404F7CA4
            stream[448..511] = 61503589A014A6F800A5D93803517581
                               988262122B30755A337F81EF3B326125
                               51ABCE838C0A57795EED2F26173DE6B7
                               E4BB6E37EE7F98383658A7BC47976321
                  xor-digest = 3AA2F283D77DD949C9EBF1B4EA95D9CE
                               AEC299832AE766A1BD617F56473D3031
                               2B81DF89D0EB79669F1A044042213F93
                               DC7F0A2D4B6F089153C6FF9095C4E69C

Set 2, vector# 45:
                         key = 2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D
                               2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D2D
                          IV = 0000000000000000
               stream[0..63] = 3D9EA1F4A3036C92CF9E0D6BB20824C0
                               F57818B3C84DF65AE4A1DE2A058F8BEE
                               242F9BEA42A78383F98AC998BE4B1EA5
                               401BEA5250611CFE6505AA5F43C9A262
            stream[192..255] = 8C2F23B3E0255982DB921D035B507433
                               2EB98C31143E19F5FAA40547D0819157
                               BBA1B6B5C3177AE45074CF5E711195F9
                               281A71E62617F3A1E582D4F89FDAEC4F
            stream[256..319] = 5D1ED872FD20FDE0C98FD76503F538B7
                               538F5061D3A3B12385B4BAE7C8CECA20
                               E47EBD5C96F88D78230B5D3909CA9B0A
                               4BDDA1FD1F561ABEC60524C51559EF45
            stream[448..511] = EA2F040B9DD538FB258C9289F5CB76B2
                               335C7D05F5B9B2CD591B55AC8FAB882D
                               07EC54EDD33D4B24D6AD69841C219C5D
                               26DDC827C67D0A6AC12D0A4E0DBE9A78
                  xor-digest = 04A255960FBBF45E8E9E0828035FA11D
                               A684C2A7099AB00DB1D3E117B78026F2
                               8F69523A4B9A9F570C49FC02E1F2FBE9
                               6F2CA223DC8105A5A6FD0E2CFCDC9AF0

Set 2, vector# 54:
                         key = 36363636363636363636363636363636
                               36363636363636363636363636363636
                          IV = 0000000000000000
               stream[0..63] = E0E9C87C82202453CDE753D368DA1842
                               9279F0B97446FB12A0436C6BE1AA7514
                               3E98B740F6F9CEC72A1EA38D4EF2BC65
                               E1AF3AE13C5ADF6DA16A2131739C0084
            stream[192..255] = A43046BAE6A4A2C288CA187C72A21E88
                               047CE98C64147F2F853617A54A3057C7
                               0F48823ECA4B82609924CC9453D57F1D
                               3ACF7D302592BCF9B1439F28B3EE5F34
            stream[256..319] = 08DFF1999015561E0817C20CED5E979C
                               6BED0512A69CCB4C6F6FA480CCE4348A
                               076F549355D22DDC52728F833447DAED
                               83D7012F3F59A8BE495078B72B299753
            stream[448..511] = C66109B099BAD13AF2F36F5AED7AA0F0
                               0320D8B109EABC7428362B7CC43C284D
                               04EC23DFA4F2A5ED2A7BE2A64CF42F9B
                               F973C6F2AFDB1AB7B7E5F9499B9DE964
                  xor-digest = 9F9D95E6B8F6E9682B03C0F78E06DD42
                               35E62C6DDBC601EAA3D36D6E6F8B95C4
                               50198564F812801FD2893F1B12A59158
                               B9506624CE698A648E1928A42BC72ACF

Set 2, vector# 63:
                         key = 3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F
                               3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F
                          IV = 0000000000000000
               stream[0..63] = 18B631E89190A2C763AD5F1DBC57B565
                               EAD588F7DC85C3DD75E7D7E74C1D4429
                               E2FB3C6CB687A620EB7050CCD49B54D0
                               F147302BFB7ADC6D1EB235A60338D190
            stream[192..255] = FE2017B0E26C72416B6789071D0EABE4
                               8DA7531CAD058597AB3742792C791678
                               44C84243B910FCA131C4EB3D39BD6341
                               842F96F4059261438A81423586EEE459
            stream[256..319] = 5FA44FAD6149C7E80BA6A98A8C861993
                               F7D39F1CAEAD07CEB96CBB9BD9153C97
                               8B8957C82F88EC2EDD1BCC207627CDB7
                               029AFC907BBEAFAA14444F66CB9A20EA
            stream[448..511] = CF4DD50E4D99B8A26A9ED0F8CEE5FC10
                               E8410C7071CCFD6939C09AE576C3A5ED
                               D2F03412E40C8BAD8DC72FAFD2ED76A1
                               AF3BDD674EC5428BD400E2D4AE9026EF
                  xor-digest = 1C945357BA98CA7AAFD28A5A39DE3DD5
                               B5F640CC7F0A567172706069637AF5C3
                               975923080CA3FAF9367294D495A02052
                               C0C473E4B779AFF3BDD9EE665D55EA9F

Set 2, vector# 72:
                         key = 48484848484848484848484848484848
                               48484848484848484848484848484848
                          IV = 0000000000000000
               stream[0..63] = 82492EEE44E22AD4DFCA2032BA401F73
                               7D4BC35CE8546EB6314EDC25E69DAC16
                               C8A9EBED6EAB895B7D72BFACEAA14E36
                               3F9A9773E43B077A1991EAC1EEA83EC5
            stream[192..255] = CB11B43F7E98D75576BB1B1AB33A4E6E
                               CD9CBCEEB36718B22C14F430A8BE7BCA
                               BCBCDE60D775DF441FCD808E79D05FAF
                               E3AA199D45DC174272EA3DD0057D9BD4
            stream[256..319] = 7D237FF28E20F0FDCAE42A7D0D7AEFEC
                               8AF23CF2906E305341FDF8FF75C0B9CB
                               C8F19696CE8D31D15E27EAB0AFFCE92A
                               AFD1BC29E9B80895B3A7CF57ED434D96
            stream[448..511] = 5ED806ACF2490F17AB82438484FCBF61
                               6A17015069B88DFC2C4CE76A2F564E4C
                               5786A7514CE542709E90101094DEBBF4
                               8954F9BF8F4773E06DEE7FB9231AA457
                  xor-digest = C77654229F128DE04A121608381F5F05
                               7F7EC90BA31DCA134CB4AB45EF911F8A
                               0AA71DCAB706277802FA880A73EE5153
                               7451838D05B4ADDD796FA005E7F136DD

Set 2, vector# 81:
                         key = 51515151515151515151515151515151
                               51515151515151515151515151515151
                          IV = 0000000000000000
               stream[0..63] = C7FC0F8C8D2064FE05BEC4A641560FCB
                               C41A60718B1DF62AA297E754756CDB68
                               48C5BF60721B49A854A7A4D4BF2D36EE
                               943A3B3922A638293B32F15A7E9A1357
            stream[192..255] = 987A15FE80E62B043B2C7C0953A27D04
                               83B2A7ECC03AD33C2F99FAB7FD2A7EE7
                               0181F7913429F89027E392FC3B73F4A7
                               5E475BA1D7DD4DA0F32D776BBABF270C
            stream[256..319] = CEBF798ED076B963AC8EA9465F7EBB90
                               6E09F80247C1FE09C86D1BEF3DE4F4AF
                               94B51FECC1C58E1E8CD225C2F68CCEAF
                               C36C029DDCE9380AE9FBC867E145F658
            stream[448..511] = FD7E885A72C796E642EA628C6ECDC508
                               9F465F57E55D51170C039B253B14EB9D
                               195A3712CDEA2624A5382880192DE3FA
                               0DA2A86EF3A61220DB949596FE1C318F
                  xor-digest = DBE3B90391D0169C64BC96AA925975E5
                               89CAE70CBE36AE6835496B1977D1ECF7
                               9F9D3E161698CFAA3F7AF57DE535488D
                               A06A8B686577A52BC358FE25F95EA2E6

Set 2, vector# 90:
                         key = 5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A
                               5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A5A
                          IV = 0000000000000000
               stream[0..63] = 6C3645C8621D8E7286911278BAB37C5E
                               EBAA2AD321AB8ECA62D13372156F8B87
                               FB87FBE02B1EFE39AB0EBE41553E5348
                               073053048A0D4DBDA1880230CD23A4F1
            stream[192..255] = BB161E8441B29DE15C9A02F447766354
                               E7E590B42AE566935F0A6D7E864AF5EB
                               B288C0C63812B0917970547225899573
                               7C804E58F7BEA1596B7343B0CBDC6AA3
            stream[256..319] = 6EC6A41251D6FE041CD87EB3996369F1
                               390E649F012712F9DA4D1F4DFF96CF74
                               91CAA6836C09BA8C55ABB656B4F51F7B
                               4AF829B5DC89F460287EFAD064C44F28
            stream[448..511] = 3D54A399D5B92252CCF9E6A0C054D4A5
                               EDBFA58A3B53981BBA50EE9BB379D71A
                               C9775A0D793AFC79A64C708D0F9A7D7B
                               E061D5A5D50DBF32480AABEBC128D198
                  xor-digest = C8340B28A1E4A5AADAC14966D92094DD
                               56EAB9C48C53327CFA62608FBF20456A
                               23DEC3B658FBEC9EC1FB6B56651245DB
                               58D6FF770F1404659BC697685BBDA62B

Set 2, vector# 99:
                         key = 63636363636363636363636363636363
                               63636363636363636363636363636363
                          IV = 0000000000000000
               stream[0..63] = D417644E8A37FF8840772A55960C4B06
                               4DA371869EA07FD02D7F8EFEF0BDB7CE
                               308173B8BAFDCA6064CEBE09609377B6
                               542CE73D44A0134C95C452D9B83A4B35
            stream[192..255] = 2974AF76C0EB09874EFAF061BFD45636
                               E6AD9C2BA71A1B4FAE493C04205B5CCA
                               A1D361DED0F1BF8C2FF2DE70F4B68E1E
                               B1B6E63B19EE1842DA4ABC52C88714D8
            stream[256..319] = 934392340254B83FA7A9888D1CA9959B
                               A221FF1C487B214FE6703C4BCE02EF62
                               4DE46A76670712B381E2EE017B67DBAA
                               3726CE1CFB39038FD0059EFCB2346385
            stream[448..511] = F234ED6FEFF11821E19D73E31BFAF745
                               126D80E0743623A179303C5A7827582A
                               ACFEE4845E8D3FD98AB990C710020B42
                               542DAB392D6A1BFE058E200FEFA00006
                  xor-digest = 42453E59A686950A5E37CE26842A5939
                               52CFAC24B39A3D7B2D7A4BA65BB95460
                               D5A493A51F5F1D97B30B6752A826BFD2
                               CD6EC49B87ED1815F2E47DBBE99BC904

Set 2, vector#108:
                         key = 6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C
                               6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C
                          IV = 0000000000000000
               stream[0..63] = 1456A98D271F43A5FF29B3D0BCC35B78
                               50C4D9DA5BBA43B752A1A541A4FC88DC
                               0FC4C89F35ACF1B540F5C3207A0BF359
                               490D482232936E5C0B818C3DE6EF2012
            stream[192..255] = E8DFC363183330BBCC8498913A28545C
                               6905F858D314939FA148C4C6600CD23A
                               941F88F2FF08D7567202F335F5A90A0E
                               A92B9D73A2C710CFE22BE0D180BA1A42
            stream[256..319] = 77ACAD59AC794EC38C13805E9638F145
                               DEE96C36C9C07A1811DCC1531A462144
                               AC1F4B2245A570C42B25EB646D4655D6
                               EA646776B0445C8B5670AB2B11203823
            stream[448..511] = 9A1BBE72AEC868E45B28B9FE3570381D
                               A759D1484B710A2AFB385DB7EAC5A2C6
                               5E2EFF9204C5DF6A684ED55C2D09FBD1
                               7E2FB6B4FF4BAD3ABD201DCEE340305A
                  xor-digest = A5832EF363D2FD5B01270B6776A5C8EE
                               C08491D8064752E4B5AC53993EED1A5C
                               24673C6B2A47FC31C91F2EEB995836D8
                               20E8061303E9DB8C81384798C4265AE9

Set 2, vector#117:
                         key = 75757575757575757575757575757575
                               75757575757575757575757575757575
                          IV = 0000000000000000
               stream[0..63] = 8F04C8F40319569CB4B04458528135E8
                               35AF2C69561F0F0F5B6009B540B85ED1
                               BC7612C9EC7A200B08AEDF07DB08ABC3
                               9FA48E63AC81974175AE3A4AC9429985
            stream[192..255] = DD98FBC3465BBD56ED0BF2F2367498B0
                               E2854E514A27C7410AAF8E0B44117EAF
                               A5EDA0C7FA2106C03DB8AF62E5ED136B
                               4BCA0B82CF2EA19FDADE4101C57117E2
            stream[256..319] = 7CA321B64434A90CE08E00A99D9456CB
                               7A0779D4F0FC12346C01A5A1310528DD
                               2E0EA2F58A8795BD138687645A7054DC
                               2FA74835B1B45F4B68E3CEAAA315C250
            stream[448..511] = 076AB5564DB74D830CF96E6B90897E5F
                               2E597619B47FF74B190C16735E902BDF
                               111FA384ED3F8055343F4561C731F783
                               7072FAB81825304DC3D4CC02404E539D
                  xor-digest = D725AAE2FE26DE0129790BC7BE3BEFC5
                               83A8C7D9F4508C8582A40855D4A79C00
                               098B8CAB2878A30F12FEE5F1B192C21F
                               DF3D41F5EBC784784B3DEC5C89D0716B

Set 2, vector#126:
                         key = 7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
                               7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E7E
                          IV = 0000000000000000
               stream[0..63] = DFD428440260E1B64579A6940EE53907
                               8CF48977E4B61DD0C708B52B42A607AB
                               C0A0774F49FD8599E4A4CA3B7C54FEDC
                               353D2467DEECDB9FFC8350C79414CFBB
            stream[192..255] = F4C7C343C6DFB6F7EA25DBF6DFBD31D2
                               595C45C4CD1C057308FFA60C1AF1BBCA
                               888C6C8097E97319566A7EBD80DA4F0E
                               DDBD22015CC363E5AC01BE42770660C8
            stream[256..319] = F1792B445D52BD4FC99557ABBECBCE74
                               257A62EEA110EF9CB3CB0388922A7FBB
                               5FCBCE5BCE44818F930284E4E360973D
                               49607E1B0E1D97C618EBA4D909A50375
            stream[448..511] = 7A2EB3ABE2F83C4B40A15F4AAA89D5C9
                               72B911AAFFF5069FA3E7396162CFDBBB
                               6A16E222C15878D9C8A00AD8201F1889
                               9F060851A3147AC2F3385FD8144BCD32
                  xor-digest = 65FAF34064FE19847014B10AD550DF15
                               B05A8A3D6B7EB64C94FD0EB61774A8E1
                               03DFB43B3C4E0BB074848DDC6A928449
                               2AE5E03E36FAAB8D46E8D647753B825A

Set 2, vector#135:
                         key = 87878787878787878787878787878787
                               87878787878787878787878787878787
                          IV = 0000000000000000
               stream[0..63] = 47756F1D1EEDDF06790A5E39083186D3
                               16E3258B9C5B7D25E478E817308E2B90
                               A5DC4A8C03A38AE1757B6EFAE73B058A
                               7CEA675CEE9A01E9BBC7B15DC5424E64
            stream[192..255] = FE6FB2E0BDF120B585D082602D2648D6
                               D95D14C3E8DF44F7D9BF650709578C0A
                               A5D775BAA12A3C1153CF44AE2A3BAC49
                               534210F8BB8AAE7F54DF049AE368678F
            stream[256..319] = DA0D9214302984F36B92EDCA76765B8D
                               5E748EE13176CFA41345AB0EFBD7CB54
                               737DC606DE60E4355233E63B1EDAF48A
                               B84DF854E47D1D746B3AA5CCC0A5DA62
            stream[448..511] = 8373EFD791B51A07B840A7FACA4307CE
                               9F5FB71A0C7891CEF7E7754A414B61D6
                               593A5EEB782FBF28998F4174C63733BF
                               A7EE172290A0A854AD6C36757AEE0911
                  xor-digest = 11BDAED16F1C5D38F8EAE0B9FC6E63BF
                               F0E7A087247A25A20135BB7A5500F937
                               F34ADA22153862AE37C963764901DFB0
                               18E9D8245DB4F275A38C10BA393EFAFB

Set 2, vector#144:
                         key = 90909090909090909090909090909090
                               90909090909090909090909090909090
                          IV = 0000000000000000
               stream[0..63] = 6AB7A8C769386FB6067059D0EE3DBC97
                               1EFAEF4AC10C74A2F17527EA5A8C6E0C
                               DF1FA10F27A29911BB57BF3E7A6DBDCE
                               4AF3E7BB730F47AC79DC917DA646A8B7
            stream[192..255] = 1DD701A2698617855C38017B0ADE1E17
                               D22D9717E21AD8635CE6A40CECC7EE43
                               83D5483F414B9F2285D200500CCA85C3
                               D45F4F25550E3701B675D7E1B8266C6B
            stream[256..319] = 5D331C1544CFD44E3588C2EA0D889F44
                               D5742E7AFE9581CAF23CB668B0530C84
                               A89D63F948969DBC0D0574911EC0307E
                               CE9CF38C5FCDE75462D1C472455A78ED
            stream[448..511] = A55713DFAA272076529BC5A33558A7D5
                               206C1C070648DBAA348C78556631AD99
                               F8F16DDDA2E5779B155DD9377A8E575C
                               257FE7E08ABE9B3A378027EA06539810
                  xor-digest = 8672FFC244BBC43DD6210AF1BDE1A607
                               C41F3243FC149BA8988B7FEF41C4A7E9
                               61F5E992F51CDD23B183C4DB710E89AB
                               BAFFC13FBDD613EA098F9D7375742F8C

Set 2, vector#153:
                         key = 99999999999999999999999999999999
                               99999999999999999999999999999999
                          IV = 0000000000000000
               stream[0..63] = E548ECEAF4B4AF1F8572F7113C7D8FF9
                               61837C15ECC6BEAAB80F38CB15022B50
                               BCB1FA414A798C954DAFB572CF22A9A4
                               D82F7561186C31BA0199EAE1678CC4CF
            stream[192..255] = 9E5D061279348E0D5DA552A82DDD3795
                               37F928DCA393AE75AED13F63BD60DEE4
                               32C96D1B2365B59FEE3C0E18515966D6
                               642F2E156C30C704A77DCB5629AC6167
            stream[256..319] = 9CDCAD9CB247AB21BA9E93C936936994
                               C6C320841C745D6DFC85110367B36C88
                               67CFAB60F6A67A1656C645BFDBF196AC
                               974A4165BF81FBE715CB6C3954E217FD
            stream[448..511] = FE5134E8B0BC016D3ED3594B6EEF2F06
                               FAFE2F4C89CB4E2627B232BACFDCA8A4
                               80B1C55DF4C0AF1E630A617CEDE0A48F
                               900A9CF815362C098A76D29360414735
                  xor-digest = 6C7EF2493D05F6A8BBBB583EF1E361FC
                               0F808F749BD13D2E6F23BDF76A9003FA
                               D08E8C1D840D2236C6C3686211353DB1
                               4B5B421A75DD362E413D33D02A5D5658

Set 2, vector#162:
                         key = A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2
                               A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2A2
                          IV = 0000000000000000
               stream[0..63] = D0854334E4619E3EFBB2A53D59F89866
                               F67220CE00A3116313FB9CB645339766
                               0CA976A8B3477F76FF8FA485D61E3758
                               3DA5F35A8FAD678B7C2B9EC97321DFD0
            stream[192..255] = 92D4924C3E682EECBF9AD3A5453BE7BD
                               56D9FD73F16BA0CA09FBD0C136BCD595
                               2FE55744B1871E4C8726611F291B282C
                               2219C817C88086A5A7BDC513DCCA473D
            stream[256..319] = CAC309E4AA3ED635D68E5AFD9F4CB0BA
                               DB229E8EB560B16645CA2A71B35B7C3D
                               757C156983F7D053B0430F9634402B8E
                               4FDE6926135473BA8560C3AE1FD5BF48
            stream[448..511] = 980DB26FDBF49D5D890B65EB01AAEBD5
                               CC118812BDE441A71871206D67683889
                               828622C6336DEA09DB6ADE0772A3D091
                               F77B1F3115E1341EF11F41F7CD0505D6
                  xor-digest = 3306A1B9675C78ADACEA0291207055CF
                               68522DB3DA07A5EC9C91361B015B1896
                               33E4451B8F7B811EF5CD0A056AC7A07D
                               BC1AB3A9DA16DEC28A006FD9928B53C4

Set 2, vector#171:
                         key = ABABABABABABABABABABABABABABABAB
                               ABABABABABABABABABABABABABABABAB
                          IV = 0000000000000000
               stream[0..63] = 6CD6B451B1C793485006B3B51470E6AB
                               20163502C30240C4A3C6406482A2770D
                               550AD77D0091632C719BA33769823D2D
                               8147396466F1A2A857060A42ECCE0A0E
            stream[192..255] = 81298474E6D86A66AE4CBCEE495D8740
                               502CBE5CC91174865A615B193B55BA4F
                               CD2337667292D3F3C428B9FEF090207E
                               2DEF037917A2244FFD3AE8161CEBA42A
            stream[256..319] = 367B062DFFD72A6EF6CEB3AE7FE59684
                               690F40A9F276E8021994ED475BE1F08F
                               A5C99E3A1AE1E68A92D02C5C14BE0E67
                               A1B989E7033274993D1685D4B2DAE6D0
            stream[448..511] = 43C53B82CFBB199FFF9C5719ED1EF470
                               AAAD578C5778A9DD3C2D77C7BAF41CC3
                               0F5F7B4C91FED81E9A661093EE20FC3B
                               BA55FF8447C899C6E12A0A0F5ECE3BA3
                  xor-digest = 7772EA572BE1609E5D20201E7F147A6F
                               DC25DCCF12D25DEBCAAFB9E9BD1E11A6
                               FD26D5B416743F495268D00B4B6CB798
                               B0AC43498541EFA188907F9E78AF0424

Set 2, vector#180:
                         key = B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4
                               B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4B4
                          IV = 0000000000000000
               stream[0..63] = EE879D01C8E20CE8CACDDB464348F69C
                               6551F70383A82933C3A765B8AC138581
                               8D67C69841FF2B4B8BC209ECFC0FE765
                               C44C42C9CD6EFF90E0A6DAB153F52D04
            stream[192..255] = 8D7D377A3072E9571F9AE00D25E875A4
                               D9BAB98A3EA348BF823F12F44DABAE28
                               317BAA3A71EB3D7C4C2EC3EF87E828CB
                               862FBFC99C7ECBC629D22DB8EB82156D
            stream[256..319] = 97B547A3E920FB054416A5787EAB5C76
                               38FA6CCDEC816613FC855EAAFB4887C1
                               3A38094D89570BF17E55E5E1EC275ECD
                               122142C9126DE5E9411F06805071983F
            stream[448..511] = CCA815558FFE08873C9AF373FAA546B2
                               FB3EA3059EFD02CB778D01962E87EFA8
                               5F24BC5BEFD4ED02C986C0229D70ABA0
                               D4E97328780FBD0ECB367A8C085414E9
                  xor-digest = 00AADA5BD15D2585CDB0EF205F20E4B3
                               3452AF75D4CE8C36925514CAFDB519EB
                               E387FAFF8DDC7856AD1CE68A7BBAEE29
                               5347BE2A647434AC4CFCE3D2C925905F

Set 2, vector#189:
                         key = BDBDBDBDBDBDBDBDBDBDBDBDBDBDBDBD
                               BDBDBDBDBDBDBDBDBDBDBDBDBDBDBDBD
                          IV = 0000000000000000
               stream[0..63] = DED8C79CC623162C2074FC7B4876F754
                               1B959209AC6573E6D25D1F1E649CC241
                               31A2F1B1B9E9E0FA639F8AF373CCAB88
                               3C659001BD120449997871E6A1D5AD8E
            stream[192..255] = 1E946CF03C4C89D19DDB9C48EACFE7FA
                               A48235899DF49232CE2A586130BAD63D
                               52540151FBC02E3BFEF082A63A900C42
                               0D6D7A11E289C34387A6155ABB71816A
            stream[256..319] = 3CCAA2AEA81296ED9171B608FD8DEAEA
                               3EA5B8A87B17B10751A01713EDE6A156
                               652783C26C0247E347860C06AD633AAE
                               2C0AFB239291A6E7729F8838A4D97533
            stream[448..511] = 065DCB330DDC528BD42DC6A0F85179A3
                               531CF900DC5F7D3B5455DC49D451161F
                               9AFD79A619DD951C854019412532D33C
                               9DE6F9AE44394208653CF12D316F4A70
                  xor-digest = 74D888BC6614CDD372E03DE0E92A0512
                               DAD7CE19C19B1E05F34BE79F1222BEFE
                               E92190397CDA031A0FDE5098FF31CEC1
                               CDC0FD7B422A6424119C46C506BF39CA

Set 2, vector#198:
                         key = C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6
                               C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6C6
                          IV = 0000000000000000
               stream[0..63] = 36AFBAFFF746195D8784CB72A16D12AA
                               604CDBF567955F15FB55DD42FAE8DDC4
                               E6CEA63B6F8E2815F3094005E403FEA3
                               0EEDD68B5F2573EFD03A4E2BC41AEC32
            stream[192..255] = 4F7E1CE5E727D83989222ACF56776F0A
                               FD1B00E9A5734408E1513313E0CA347C
                               C37D8DE7AF4F6C5C7EF311BDA97BD8F4
                               52F89B4D44411D63105BECADC661D558
            stream[256..319] = 2677C65207F10008A28E0D3D2C7D43A6
                               71A96CB9A98ED1ECDEBA8F5AFAF4DDF3
                               F7B078346EB1DAEB1047D2E656EFB331
                               F3A71302E6FB547568D6A8A2871EB5B2
            stream[448..511] = C39BC4103ED0D8FE8C7D5FC072C94080
                               DF9DAB70F627D8BD68719A721836554F
                               3A2CFD08616170F4E3C3B0420BB41FBE
                               9A84C43D405B9EE32285BB5051CD5E83
                  xor-digest = C6AFC707ACCB8F10DFDA45A836C85603
                               D2F5E30BFFC0A9FDDE48666153F395EE
                               0BF93F900D59C7FA70632F644521A5FC
                               FE28882311D315D53BC10755698FC81C

Set 2, vector#207:
                         key = CFCFCFCFCFCFCFCFCFCFCFCFCFCFCFCF
                               CFCFCFCFCFCFCFCFCFCFCFCFCFCFCFCF
                          IV = 0000000000000000
               stream[0..63] = AA68F6EB41DB62A2C5E4E9AAF21D7D43
                               1C29A66303854A68EF737872CBF7C505
                               918B87CE4DB6B3D84BC039906AC0561D
                               F79F0A57CFA762B8B9C2991F1DC98032
            stream[192..255] = 7BC0564BAF3C88CF14FCD2020433CEDF
                               65EE68DF4AFAB7E040DFC396A856617F
                               677217529B839EB9DF47AFD6758CAACD
                               75E734FCC653ED5AC25C8A7B1AEBAA49
            stream[256..319] = AD21BBE24EA84C0859B2EF3E09070493
                               6A6D2A97DF912207D3F50D63FCD56676
                               61A47AD0DF1FA8DDE08EAD7201AF15FA
                               85BCBA0962D7921397E35E60149BB4EB
            stream[448..511] = 8914307989CD704120A6DAC52789B845
                               7260A2939CA0E02A4C41C46ECE890305
                               9F58A2B0F3D93B45160D08A13737D51E
                               984B97CD4A28DC2D92155FCADA3F8033
                  xor-digest = 2EE47E155D995B266EFC7E0A995172EB
                               AD6A1201A20D9A9F5397FFB815AE6246
                               760EF488A9C45FB9B820E32A42E21634
                               E995CECF6E9E05FD14FFDCA92313AC0E

Set 2, vector#216:
                         key = D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8
                               D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8D8
                          IV = 0000000000000000
               stream[0..63] = 596EA70BBA1A4DE2F8ED2AF37A0CE6D1
                               2443354659CD0C41203EB345E160CF05
                               6F8D71314AA7221D86F868304F34D5B3
                               ED4D51072FE7B12568B859077B6F920D
            stream[192..255] = 26716254A9C7067808EDC0D31D54D289
                               88A3F655C10931E217B3C9A8A4B557D2
                               8AD6C701612A8D848FED1589CCFBBE7B
                               566496F4662B1D98FCFC70C1716E5347
            stream[256..319] = B33C15E9488DE8A97AFE67FBFAF47FFE
                               5C3934B05B5E2EA061A41A2BF0D81FB6
                               054C824B492775E3E8300DAD609BCEA5
                               837392668C0B54FECE2F2945F18160D3
            stream[448..511] = A1F72ECB02649F01D4396574EA80BBCB
                               8934FCF989CF1D7CF7410B0A93E08C10
                               0A229C952DA999789662E1666CA71C65
                               4DBEB2C5BBC20BB67DF67CD39B51B4CB
                  xor-digest = 7ABDAB4EA81129864F2CEB9157C01178
                               A413889D86A1D54F964F3D70C5A4326E
                               3FDBEA3C5B77F4EFBBE94CC2DB808B96
                               A81BCEF94D3FC039CB13A754D4E4A1E6

Set 2, vector#225:
                         key = E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1
                               E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1E1
                          IV = 0000000000000000
               stream[0..63] = 6D221A5561813E4B6BF1A3821F0BC95B
                               3D51004ED29EAECD26016E5B7F628BA0
                               6B2BA4D650D685C3BA9FB51E305EEB36
                               A11CA08C431E0740D59D521FBDDBF716
            stream[192..255] = 9C9EEBCA7428A88562FAD4EC9800EB7D
                               E4EBE571855B40D3F1D9770236EF0131
                               70A6BF8CF9C1880A1BC3C58193777098
                               89384D19F4F9D6E8098E8E326B9AC4B7
            stream[256..319] = 86ECBB7CA8E1526F538805A692C354B8
                               E335BAC919CB4355C15B40D721328BE9
                               81105395FD27BB6F0515A427469DF557
                               DC92EB010C49C332BFEB1A98154BF0AA
            stream[448..511] = 0503DAA102F9CDFBFF854D6015BF484A
                               201F69E6E789A757B8DAB005D5859027
                               849ECA4E951AE28126FB6C63BB65EF61
                               94C9661F9E40CAAB817CBE89595096EC
                  xor-digest = A3008548B817A82F3D4D2813B5777952
                               A5D46CD710AC4F8417273ABDF65BF0D3
                               D519C5D0CA1CFFE8F265338084F54DC3
                               65C61F376DF6F1DC4B2BC6BA63E2FF11

Set 2, vector#234:
                         key = EAEAEAEAEAEAEAEAEAEAEAEAEAEAEAEA
                               EAEAEAEAEAEAEAEAEAEAEAEAEAEAEAEA
                          IV = 0000000000000000
               stream[0..63] = 304665A82B0838D4EA0A7737855CEAB0
                               44583FBF2F8E68D7B3B191600ADAEB33
                               538942A71998F68DA9A0D4BAC36A5052
                               CBEAEFFCABC6B506E5F805F8105D5E79
            stream[192..255] = 96B62FE40229E2CEBEAE44431F01A0A4
                               3FA080D685215BEA4705B6B78187751B
                               E1DFA0DCC1C8D6A2040C0716F524CF40
                               42889F743A3EDC01EBDFD3A6FF3E92DD
            stream[256..319] = D1667A839D7725E602FD36A69117D039
                               AE92EC7032432323A61AFB1602F17E4F
                               B66F0BB5A5F4C54329F7217497B3546F
                               FF9938966B05789E0CA65CBF34DB1B2D
            stream[448..511] = 3557FC69A9D44C66FB022ED8D4D349C1
                               D82A41DA40E3687B197DFC070000B69C
                               2FD9B1F9F99C63BF3ED82F2CCBD2A6ED
                               20A14ABA05F6855078DF5C73A4D50493
                  xor-digest = AA453B1E7AC7D53F54827BDBAD419A21
                               AA49AC5A55E96622D028D3D600F37D89
                               2C084D404A006404B18620F84BDF872E
                               F7E90203875719F0B90FA8A900FDC22D

Set 2, vector#243:
                         key = F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3
                               F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3F3
                          IV = 0000000000000000
               stream[0..63] = BF9634C2D81B6400C2ADACFCC0C353CE
                               3AC45A2EB636AE9D2D6B8DB6107511C9
                               399FB22CA2DF6406307EADEED423E72B
                               72411E11530B1814AB196A74DFD4FA61
            stream[192..255] = 50F32FC8C94BEFCE5E51F3E774134ACA
                               D60BF3DE49BFE1F17DDD88395C4880AC
                               926528971A3D74796303A4064F67733B
                               A2AB545344B97F555525C0A5611151DE
            stream[256..319] = A6E426963373DCDCE54C1827F683859D
                               F11857D7BEB1EEA10FF137CF6B395635
                               53C79E92295B1FA385C59BC201612C70
                               39341B55D49139B88A16544AEDBDA967
            stream[448..511] = EB50C1AFCDFBF83EDA42011C141B67CD
                               041598209605800EAFF2EE6A99A6C958
                               9621B778FA4DB6D2FC4980030B86F3C8
                               670B46BED56A511B9A18E60B1FED27D5
                  xor-digest = BEE123C7EF4E2D25DB862CC720D9FEA7
                               4B329C3B1588342B6104DCA139FB1A3B
                               E0E1A1779D973C3F4473D76309B8FA2F
                               831F295B150445F44E4F46336014FA7D

Set 2, vector#252:
                         key = FCFCFCFCFCFCFCFCFCFCFCFCFCFCFCFC
                               FCFCFCFCFCFCFCFCFCFCFCFCFCFCFCFC
                          IV = 0000000000000000
               stream[0..63] = 356DD71DBC2B216B7A439E07BCC1348F
                               769F7EF482486C92E8FD8EB050224838
                               AB1F4DFCD2FB196AFD4C4FFBF51B9124
                               6BF45AE8131B8D5CAFA29FC3025A3597
            stream[192..255] = C09481306DB9FF12F1798A21A3031921
                               B237E1B54A73F724CC0378379DB2FD86
                               8DF08983A3D26C32379E3B132A6F1766
                               646A963AA56C8F5D45B35F79B24D27C0
            stream[256..319] = 6C198E30BBAD2E329A7A3ED5C383340F
                               90EADD9F44AB7F339E6BE9217366188C
                               4C8D721BD6DC5D5D192A8E854013EBE2
                               66633893015AFBED28EA42F928B27F60
            stream[448..511] = FF9B8ED2074ABD83B51AA93A65E5E303
                               774CD6874D344236B1EFD39A3605984E
                               DFEBCFB5B41AC09AAD500F71AF6D77A0
                               7CE81A5E0E1E29C857609143B5BE0BA6
                  xor-digest = 1858D5470ABE500EC2CC40158C700E5A
                               78CF094440F2081ED694C47AD054D7C0
                               0E77B67920631ED3E5C86B400FFD70D6
                               244CEC58F08DD3C4AE05778B514163FC

Test vectors -- set 3
=====================

Set 3, vector#  0:
                         key = 000102030405060708090A0B0C0D0E0F
                               101112131415161718191A1B1C1D1E1F
                          IV = 0000000000000000
               stream[0..63] = B580F7671C76E5F7441AF87C146D6B51
                               3910DC8B4146EF1B3211CF12AF4A4B49
                               E5C874B3EF4F85E7D7ED539FFEBA73EB
                               73E0CCA74FBD306D8AA716C7783E89AF
            stream[192..255] = 9B5B5406977968E7F472DE2924EFFD0E
                               8EA74C954D23FCC21E4ED87BBA9E0F79
                               D1477D1810368F02259F7F53966F91CE
                               B50ECD3DA10363E7F08EEAB83A0EF71A
            stream[256..319] = 68E43AA40C5D5718E636D8E3B0AB3830
                               D61698A12EB15BD9C923FF40A23E80BE
                               026B7E1349265AD9C20A6C8A60256F4A
                               CD1D7AD0DCBE1DFF3058ACD9E1B4C537
            stream[448..511] = 343ED5D011373AF376308D0B0DAB7806
                               A4B4D3BF9B898181D546EFCF83D7464C
                               FC56AE76F03F3711174DC67AC9363E69
                               84F5A447BD25642A00754F1133BFD953
                  xor-digest = 8C03E9237FEE95D5041C753C204D2B35
                               764E4A53035A76F9EFBADD7E63E60B69
                               BF23F7C5FD39B2249B0C628FB654D521
                               4EB588371E5D2F34BF51396AF3ACB666

Set 3, vector#  9:
                         key = 090A0B0C0D0E0F101112131415161718
                               191A1B1C1D1E1F202122232425262728
                          IV = 0000000000000000
               stream[0..63] = 0DD83B7F93629BA8E489E30FE4B6EE54
                               9BAFB44CB794AAEF2EF07116649FD4C4
                               4DAC52560EFB34FF1A2E56FC0DD86F2D
                               56C2C5C97089FC4C35C6788F36E6F142
            stream[192..255] = 19A8C09135CBB83C6140BBEB60099BDB
                               469178F58B6DC87AD2B33CAE53A83B46
                               A3BCE1289A68528D5A32A8867587FCC7
                               F4DFE8EEA78BB2A9C40B9F6D8797BFE3
            stream[256..319] = 2E4E97BAAE813AD2C14848ABAB7C51A7
                               4BF3153C63101F4E6E4EEA56B470F0A6
                               78FAC3AA6CC300A51A7A345356D3FE1E
                               3A56242086CA61A1E8E43F6703CDF6DE
            stream[448..511] = 306FBEFC44132B66D527F5E75D171868
                               EE8CBC6DAEFD6FC5B3730541CEA82CF6
                               7D41B8783D75117D266B924502D5AA5F
                               28FF44A13AA2179DD8F0F4AD4B29024F
                  xor-digest = 25B9F46F897C9060052B08E0E70C8FAC
                               C9FAC41A4057E304209D39EE0807987A
                               C77A8A723BE07A22E9AB6BB8DC358A54
                               14E6C31C1C6B9D9E107AF74594134307

Set 3, vector# 18:
                         key = 12131415161718191A1B1C1D1E1F2021
                               22232425262728292A2B2C2D2E2F3031
                          IV = 0000000000000000
               stream[0..63] = 4B094A8031FEA02C5CBDC1E2A64B13A9
                               A0976897FCBD92A15738330CD1F85448
                               EBD8B7E61A76855C64BE1BE78034ADEB
                               FFDEDFCF064AB92744760DFBF59F0A9D
            stream[192..255] = F807DF0420C6D87DAD3A1811A96B5E4D
                               2B2F284CD9130F51D307521BD2CABE72
                               1F1BAC0EF6219B7ACF8923C026C7F9AD
                               8762CC9A9F8847750511D3697E165689
            stream[256..319] = AFB3798B54C003AA6C05C7893C5DB290
                               AC7FAFE8C25D3E66AC699BBA3A880330
                               70D17C0314DAEAF51DBDA0C9DF36B713
                               A913BD397B41DA7FF410A593568AB2BE
            stream[448..511] = 67AFD443E67F5FF76A247EFCF3D54649
                               0649CDE396FE3AA34549C3ABC8F7447D
                               DB7A666C0402AFA25ADC47E95B8924B4
                               B1C955C11A746FD4C0DA15432C1B83B7
                  xor-digest = 842B6DBFACE3D7D24AC981C56F398BD9
                               C19DB3B086F4ECF5473CAB197AD6C170
                               BF57A238BD47FED65726CF2D58AD701F
                               B66E27C2026772AC8C706B77186BA837

Set 3, vector# 27:
                         key = 1B1C1D1E1F202122232425262728292A
                               2B2C2D2E2F303132333435363738393A
                          IV = 0000000000000000
               stream[0..63] = AE39508EAC9AECE7BF97BB20B9DEE41F
                               87D947F828913598DB72CC232948565E
                               837E0BF37D5D387B2D7102B43BB5D823
                               B04ADF3CECB6D93B9BA752BEC5D45059
            stream[192..255] = CF7F36734A7AD1EF4D9A4AA518A91C14
                               64184688F31E5E775E879E01E82FB42E
                               AEE8F382AA0701D54AF5DB788858CCDF
                               801DED1E18BA4195019AA3111BA111AC
            stream[256..319] = AB84E643D214E8DE9274720A1557A1E0
                               471F00394934A83A324D4270949BD448
                               A7BB6B5D5FA40E9831AE5B4EA7D8D34E
                               071EB56EFD84F127C8E34DA9BF633B46
            stream[448..511] = E757CA957797D6416E17F852AFFBF191
                               AF98EB8CF73DCBBA0BCE8EFA29B958E3
                               9C0085F0076E0B4E31289A4F2DF35855
                               ADD6BBEC725FC2860D4F49AB4EEA6C87
                  xor-digest = DCF2DD4BD229E2325045FBE0DA487A00
                               256DA072F2EC9FADF50897E8C0379308
                               28C6C1971EFBEA4155758DAEC6404CB0
                               E312243E7757060D600EB8094FB66995

Set 3, vector# 36:
                         key = 2425262728292A2B2C2D2E2F30313233
                               3435363738393A3B3C3D3E3F40414243
                          IV = 0000000000000000
               stream[0..63] = 5DDE22EEE0ED12CF83F433441A3799B3
                               A4415A2018A60BDE0A0F8E08993820C8
                               20998D420F346D8B808CBED40FC7CBD0
                               CC43949B0A16F0EF2577CECAD03DCAD6
            stream[192..255] = 5C86A6AB19AD083676D609D2C094FFC2
                               921CD8D4580815522BA72AA20FEC59D5
                               64F1EDF2E2AE4810C69701BCD515A939
                               D9C156254F28DE5C90C6CA2B0A385D53
            stream[256..319] = 956A71BB6344DDF03A8B828A03FEA914
                               8585BB8D21E52134F1FA9541A57519F4
                               4C2D56C8746E9FB40EB1FCF3551A5F95
                               38B90606924F3D082987B77C127D1DB7
            stream[448..511] = 2160DB576116DD75880E4DE9A7505308
                               05EBD00F48B6BFB62679F93EDBD42766
                               A51AD3052C64174B5B027F6D5DD02059
                               2F5BBC369D48708295259F4B9519B19B
                  xor-digest = 5923F3E59743F7BD9E6C9E81DB5E0007
                               02C2A1BF996C3F00C43D2BF32FBD0F62
                               38B3EC2AB846972C48171EE53B5A9722
                               2CCC6DF5B470C5C1ECC1F6FF89F74043

Set 3, vector# 45:
                         key = 2D2E2F303132333435363738393A3B3C
                               3D3E3F404142434445464748494A4B4C
                          IV = 0000000000000000
               stream[0..63] = BDF4E0BB6B36D01A31EE2E76F2379D33
                               286ABFA82F6872677955777DEE0B1662
                               A65D85EBC56A7995A6F6CF995154C444
                               C27CEF3EABC85B8985C7FA94C8ECB065
            stream[192..255] = 8835BF6D66FD567BCDA956673D9DA182
                               701921B79AAAB6039D65ABE1C7178923
                               BC39C8A56FDEC8FEAAC4C29707914F68
                               CA6CBEDE4DBE9FEAAF84DA2DFEC56E96
            stream[256..319] = A2751597632CF806C8246F7F9D9C4A72
                               DE85C8C0C36A769F32A062DFCD45635B
                               0C7131BFB38CE253886D4918CC4B7DBA
                               780CAE5FA0F22479F445C0AD1285F35D
            stream[448..511] = 1130339E16298874524D18F68266246C
                               A0B2060607B60689D025BD30BC6DE7FF
                               5DDB90249319C9EA13195200ACADB595
                               14D56FC358D7A0D3BAEA374E34EA2E9D
                  xor-digest = EBF45CE390507D94E9969EF42C62C8B3
                               C6649FF841003830CD716EF712BAD8F2
                               A47575AF99B8F93F12C14FAD7CC03D6F
                               0D4C5C5E5C6D997053C9C36DAA99BBCC

Set 3, vector# 54:
                         key = 363738393A3B3C3D3E3F404142434445
                               464748494A4B4C4D4E4F505152535455
                          IV = 0000000000000000
               stream[0..63] = 51B180F1C9C31388F8B3DE8734F3918F
                               F6DEC759689E6A54D0EAF8734DECAB2C
                               A2ACA4DFAA260AB781769B83CF94C2A0
                               166F2643585CAB42220D200F92074363
            stream[192..255] = 147CE4098C9884493CF00DD28B6439A5
                               B794F871CCC4FFE349CABF3963C6BACE
                               D799AAB7F778B59473EDE8CB475056A1
                               E7F5D0BE68DE84C535A8FB67724E0C6D
            stream[256..319] = 7F0BCA1B790CD5C8F8CFD047AFE1C5BF
                               DDA8C8E0BBAF0567D4AE6B63C9E32770
                               51D1200ED8740D60FBBADC20CAC825A0
                               819CB66398FF7CFA38F3CE5CF23BAC37
            stream[448..511] = 74C2B38820E2614D4AC42477185346D7
                               5EC3BB41DC9810610C5B745A1B423A3C
                               BF14A7E45C08C5E7C1CAE65B8839F030
                               A8E52500776B45EA65885322FC1B3A57
                  xor-digest = 109865F93CCF7C2EF35211EE91967DFE
                               6A45DD309EF3FEB5375F05F1429F7C88
                               0D712F67A547462D01CDC15D72AA32CD
                               A5F4D630AD5186D951E34A8E1F9F6D20

Set 3, vector# 63:
                         key = 3F404142434445464748494A4B4C4D4E
                               4F505152535455565758595A5B5C5D5E
                          IV = 0000000000000000
               stream[0..63] = AADBA970B29F5BB8522C3817E849E5D0
                               417863554D16D6FC42405CA5A826A82A
                               7F0ADD295D02DF3EB565E10CA1902E7E
                               E84CC977614F325AA0BCA298F64871C4
            stream[192..255] = 23453B14E9067B2733C88A3137650D83
                               BF2EDEA3BD78D336765151C9DC15A534
                               5394C7B0E1B0DD3BEF7C7BBBB84AB0B5
                               7992446F8DD102F90B0D72728686EC17
            stream[256..319] = 0291E9B6188CB3E43F98B576C9C114B4
                               E1165A39B33E32E7260D6767058C45B0
                               93717E09868B400557E750557417E7C7
                               F0DA6A8AB0179630023EEE17B0362575
            stream[448..511] = D98E6AF3B8A4BE5EE6CD4F067FDDE869
                               FA2569648498460C0B2E4A3A4652FB71
                               77D02D632BFEF2C3511F1D374AAADDE1
                               4542AC660114716E5CAF854AA5C2CF1A
                  xor-digest = 989C4606DDB85C89396FD3D07C6D0341
                               6D90B980843BDB761E51AE7887E10E6A
                               F845E1D71C310A84473701B2D27AC832
                               6721A660A63B5EA4E265D1F2B2027093

Set 3, vector# 72:
                         key = 48494A4B4C4D4E4F5051525354555657
                               58595A5B5C5D5E5F6061626364656667
                          IV = 0000000000000000
               stream[0..63] = 53AD3698A011F779AD71030F3EFBEBA0
                               A7EE3C55789681B1591EF33A7BE521ED
                               68FC36E58F53FFD6E1369B00E390E973
                               F656ACB097E0D603BE59A0B8F7975B98
            stream[192..255] = A04698274C6AC6EC03F66ED3F94C08B7
                               9FFDBF2A1610E6F5814905E73AD6D0D2
                               8164EEB8450D8ED0BB4B644761B43512
                               52DD5DDF00C31E3DABA0BC17691CCFDC
            stream[256..319] = B826C7F071E796D34E3BFFB3C96E76A1
                               209388392806947C7F19B86D379FA3AE
                               DFCD19EBF49803DACC6E577E5B97B0F6
                               D2036B6624D8196C96FCF02C865D30C1
            stream[448..511] = B505D41E2C207FA1C0A0E93413DDCFFC
                               9BECA8030AFFAC2466E56482DA0EF428
                               E63880B5021D3051F18679505A2B9D4F
                               9B2C5A2D271D276DE3F51DBEBA934436
                  xor-digest = 7849651A820B1CDFE36D5D6632716534
                               E0635EDEFD538122D80870B60FB055DB
                               637C7CA2B78B116F83AFF46E40F8F71D
                               4CD6D2E1B750D5E011D1DF2E80F7210A

Set 3, vector# 81:
                         key = 5152535455565758595A5B5C5D5E5F60
                               6162636465666768696A6B6C6D6E6F70
                          IV = 0000000000000000
               stream[0..63] = B2995CDC9255E4E6177398EECE05F338
                               BE14825E8025598C1B4B0B80013E5D4B
                               C195802ACF47326F309C58809E044CA0
                               2027CCE97D80F7AEBA6D0376C96BFD7A
            stream[192..255] = 0B89114F6F4111D2C7C33B0CC3DE682F
                               932E9B060BD3D1E17801ADBF7F034819
                               2D1F77F99104BE2FE62AA14CAF17D0C2
                               35243B76D298C9CB51F7E5E02914027D
            stream[256..319] = A93BEF16E18FB3D34FD342AEAC4EC93F
                               474910948F5E25F20C3C6AF50FBFFD14
                               8B8272DF4AAE7400843AE11502D06196
                               59F3F2484D5D5659BC340039CAC03B20
            stream[448..511] = 031AB90E5D0C95ED116B7D03EFDD3543
                               ACDA91FE89071680C1B025F305538F7E
                               7154BDF131351E68F0F0ADDD40FB5183
                               0DD7761114BB4BA9692BD72500E7B2A3
                  xor-digest = F72CEF83AF80636D318B6C1623368883
                               65438DF6E3B8611FBF3D602746BD8077
                               961880B5FD7ED4C6BD13C360B50BA7AF
                               E838ABA36FD1B21FED0AA6B095D04F60

Set 3, vector# 90:
                         key = 5A5B5C5D5E5F60616263646566676869
                               6A6B6C6D6E6F70717273747576777879
                          IV = 0000000000000000
               stream[0..63] = 447D16E09F139ADBFDBC742D248EC354
                               67F165D42937FBA97B816016613DE365
                               B0C23E4145CA71A3680B382CFF6D615C
                               E7B2B02AEE1B6CAE692E4D09B2B47CE4
            stream[192..255] = 49DEBE1A89CE85C6BC52DCE9E80422D0
                               523FA99D29132F3B292B695EC641C0E3
                               C3C339414349F83BAAF6E534E426DA98
                               2BB80981B58401128A158AEB75FD48E7
            stream[256..319] = E661F70FC1DCB4437D4DE0C4F6540EFC
                               14D319CF67906DDBF41BA8FA8FD1B17E
                               A8452CCB67F4078A8CEB2953218F97C7
                               73850D1CB882656A6486C0D12F9324EE
            stream[448..511] = 7916FA50772F5BCD5DBF87F6733466B7
                               E0DC28687A5AFDEE5BDFCA4A197E7B6D
                               82072AC49F2C7944519999FCE9438AF9
                               80EC5576BEF6454C43AEC151A488A405
                  xor-digest = 62E4E63373B0DD84227A80FDE16A2B25
                               27AF035FAFE70CCF3B67F0CB2C22DF85
                               AF7FFAF9B34E05C6AE9E42C9C6DC457B
                               C223D886718E3B0022BD15FF398FC2CE

Set 3, vector# 99:
                         key = 636465666768696A6B6C6D6E6F707172
                               737475767778797A7B7C7D7E7F808182
                          IV = 0000000000000000
               stream[0..63] = D356187B3A555932420B005EEA1703CB
                               6C568987D54316540561425C078A0BC1
                               6011BD3A1E88C62039608DDB65C35453
                               8E6E6BE417066D824B4CC3F4842D1B7D
            stream[192..255] = FC9DB2F6F1A10BB4690291F108119B07
                               C7D908E2A3C35BDEDF1F0B79041C04B9
                               1D63CE0D20459F3A99BF37AB195D907D
                               3EBF1C75C5B7272D29ED83C0ECAE915F
            stream[256..319] = 2193BE6883F2B56B74312E46F422441C
                               C1A54EF08360C87F70AF598751E24F28
                               5E7A0C2F886147DFEC52B34466F3A598
                               8DDAF657AF45A452495F852233F3E312
            stream[448..511] = 42822BF1D4BFD3122C2C842CE59BD9AD
                               4616D916AADBBADB1A7F710EED2F7211
                               653055D94569FA2BE4C2BA8B758E2956
                               2C7A3354074705A28891B5E66EB8A7D7
                  xor-digest = 8DBE66E2AAD0332B5D3B001C2EDFB91E
                               9335910ACC1E28F2150289D7A414DCF8
                               ACE6EC6CE70A985602E8E10B61F0736B
                               0076F2E181758E99DE3628079B9C41B6

Set 3, vector#108:
                         key = 6C6D6E6F707172737475767778797A7B
                               7C7D7E7F808182838485868788898A8B
                          IV = 0000000000000000
               stream[0..63] = 4C2EB1D4A9A84064F43082EAC25C741F
                               A49F2579FCB069A2B072B4D7EB704B38
                               E00DB35E0D9C2077E58B9403D73904B9
                               BDAF16A1C79A0A25B0B9BC06E49D2659
            stream[192..255] = DBB77843D3F626E1F577ED0AB0D90348
                               66237611BC25FEA9713D5D001D2FE59F
                               51A5C201D1EE6F7844BF231C34BB489A
                               CB3EA4434226248FDA91597AC400C8D2
            stream[256..319] = 3AC1C77E12C7B3CD306743B805738AAA
                               8269B47132D1902ECEAD7EC403E2CE6F
                               D3EA6DFF1FE350995BAC330874EB0777
                               EA659488C3991432A1FF9CDE7ABB9D34
            stream[448..511] = FFC9E408A4521EFDA22B2D4C30F22781
                               D17CB1C709C4ECB2FD03ABEF56B4DD98
                               6379C068662A5CBC01053A0A7B3D1A0E
                               9B9AB81EEB8F57EDED3BE1EE75ED340B
                  xor-digest = 0030ABB5B1F3B9615878BB05AE84977C
                               6F5DFC18FDD5C2C7CDA6AC2E00997C43
                               4DF73B9AE21A6C58D0D856801A072B23
                               DCBA58AC011983C8AEA55A48F8A38FCC

Set 3, vector#117:
                         key = 75767778797A7B7C7D7E7F8081828384
                               85868788898A8B8C8D8E8F9091929394
                          IV = 0000000000000000
               stream[0..63] = B36D9BB49A62689A751CF5C971A15F70
                               439E56DC516F15F958369E3DA2500EC4
                               D51CE469B050037570D03B0948D9FF82
                               F2AD1B1D65FA5D782CAE515E03BA6A60
            stream[192..255] = 0A4DE80091F11609F0AE9BE3AA9BE969
                               9AA1C0BDEE5C1DE5C00C36C642D7FF87
                               2195871708F2A2325DE93F81462E7305
                               4CECEFA7C1906CDAE88F874135D5B95D
            stream[256..319] = F69916317394BF360EB6E726751B7050
                               96C5BF1317554006E4E832123D7E43CE
                               74A06499BF685BB0AAC8E19C41C75B1C
                               840FD9375F656AD2B1377B5A0B26289A
            stream[448..511] = 5A49B471376394B09890CA0A5A72410A
                               B34ED9B829B127FB5677026E1BFC75B4
                               AFE9DBF53B5C1B4D8BEB5CEDB678D697
                               FE56DACBA9D6DEA9C57CD8243153755A
                  xor-digest = 9DEC1D2B309A5B7872F2AF1F30A5ACB0
                               FC656DFE3C88B5C098DB3D5CE76F3ADC
                               C9D4BEB8C29B81C3EF3BB24D34A93A52
                               DD659F62FD9BCBEAEC440BEB08B342D7

Set 3, vector#126:
                         key = 7E7F808182838485868788898A8B8C8D
                               8E8F909192939495969798999A9B9C9D
                          IV = 0000000000000000
               stream[0..63] = 4E7DB2320A4A7717959C27182A53072B
                               9D18874644B42B319963B5512340AA4D
                               C7088FE4803EE59CC25E77AC29D13E72
                               20654487F4A3BF2D39C073C7D231DB17
            stream[192..255] = 58A4B8F161BE5C1AC1573FB95C216AAE
                               ADBF17205072225CD2236439A574B40A
                               2AD76749E37AAEC60B52D79F5DA5459F
                               094244FDE783122FACE929D94E914A87
            stream[256..319] = BE41A549607DA00691D0C3734D1F9CF7
                               1A0D21056E50BC89F29135989432FDB5
                               C2340BFF6D181946BACD49D4B28A5104
                               97990B241CE021280159DFAAC44DA45C
            stream[448..511] = E7CEFE15DADB07044C730CE7650E4124
                               687B7781C85C472EF6D3DD6C7150B050
                               001904552B59778F2BAEA8C0CA29900F
                               0470F14CCED15E2D83FB1A06A0C57C7E
                  xor-digest = 81B127B36F506C56D5D62AA866E93BD5
                               2F97F96D353D50F613B9CF2270743A29
                               4B4082749139ADC010A6C12B05A77533
                               EA699A1FDDEFFE1B28880AC98F7FAD71

Set 3, vector#135:
                         key = 8788898A8B8C8D8E8F90919293949596
                               9798999A9B9C9D9E9FA0A1A2A3A4A5A6
                          IV = 0000000000000000
               stream[0..63] = EE17A6C5E4275B77E5CE6B0549B556A6
                               C3B98B508CC370E5FA9C4EA928F7B516
                               D8C481B89E3B6BE41F964EE23F226A97
                               E13F0B1D7F3C3FBBFF2E49A9A9B2A87F
            stream[192..255] = 1246C91147270CA53D2CEACA1D11D00B
                               F83BB8F1C893E6F10118807D71021972
                               586592F9935827B03EA663B7CF032AA7
                               ED9F1F9EE15409B18E08D12F4880E162
            stream[256..319] = 6B6AC56A7E4C7636D6589886D8D27462
                               41BACAF2A1C102C5D0DE1603E4C7A92B
                               42F609BCB73BC5BFC0927EF075C72656
                               7018B47870365138EE821345C958F917
            stream[448..511] = DA438732BA03CBB9AFFF4B796A0B4482
                               EA5880D7C3B02E2BE135B81D63DF351E
                               EECEFA571731184CD5CB7EEA0A1D1626
                               83BA706373017EE078B8068B14953FBF
                  xor-digest = C06DCD6409120BCC16F4412019C0D958
                               3BFDE4F32A6AE9B469A4112211B77654
                               355FB3ECEE657E1B8C20E570A83A9CC6
                               E58656A63AD343E0C241DE558EB4EFE3

Set 3, vector#144:
                         key = 909192939495969798999A9B9C9D9E9F
                               A0A1A2A3A4A5A6A7A8A9AAABACADAEAF
                          IV = 0000000000000000
               stream[0..63] = 14530F67317B09CB008EA4FD08813F80
                               4AC63D6B1D595D21E244E11AA4F153E1
                               256DF77976F713B4F7DD1DF64E7016BB
                               F9460A1A7CC7F3E9D28D8D19A69EB0B4
            stream[192..255] = 6C025A7A0A9F32AE768D35C56231AFFF
                               5E9A283260E54F442D1F3263A837545C
                               234F7701D1A5B568DDA76A5D596F532C
                               4F950425A2F79CD74203CCBB27293020
            stream[256..319] = CA585389DDA8D79B73CA2C64B476C776
                               0DC029271B359EB10D09B90FEF816E96
                               432CCEDFB51322F7AEA6DEB896E048FA
                               2AAD234F89C45FC25967DF99955B1234
            stream[448..511] = 7DECE5C4BA2E08A2A61A37D9DD56BC89
                               2E141874A572AE4342067CBD4E080933
                               1851640E5D6EF48F73A4A638C74471C1
                               85E731136BAC231B0803A66A4CDB6A4C
                  xor-digest = 99D13A0741CCC1C40D655993BE02D21C
                               6BDB707DCF4FE3EE7866FC62F9C23EBF
                               C1C57844796FF8B71CDC8F569E75D960
                               0EFA123DCDDD96E33C1090238E750B34

Set 3, vector#153:
                         key = 999A9B9C9D9E9FA0A1A2A3A4A5A6A7A8
                               A9AAABACADAEAFB0B1B2B3B4B5B6B7B8
                          IV = 0000000000000000
               stream[0..63] = 9B05907B8F2EE3E831D9A0BE6203DBED
                               012C381B7E3225B52282B9D0BA5A5A6A
                               A367F7C553177557B87FFAA73C59E123
                               B8B2F069B6C0F6DF25CC0A340CD2550D
            stream[192..255] = 4274D6C7996E9E605D378A52CB5AECCC
                               E6EF862FC0F40091C79FDC93DE2B7CF8
                               4B484FC874687BE243965F92080444D2
                               206123C6815E9A497610283D79EB8FA9
            stream[256..319] = B9EBAF94F5CD2CCDAA2F8804E586DE09
                               98A5E2E79D9C2E9F6267A16B314C3748
                               07E7DD80A3115D2F64F1A7B6AF174AD6
                               8EA04962D48C7F0BCA72D9CDA9945FB1
            stream[448..511] = A08547DA215E1372CED1AC1192431AF3
                               52B670CE9FF5F1F3A598CB17961D7780
                               F1D08A6C69BF2EF73BB54DAC8308D320
                               66CB8132DE497FDD9BB54739A54A57AC
                  xor-digest = 71B9D8900F2C9E50E4E8E31D22C1E008
                               9113A28C9E4039B00E181FC3AC2CBAC0
                               70EA10B1E44ADF4C46D767A1945D5414
                               42E4C1322099B3EAC9AF05736E2BFDDC

Set 3, vector#162:
                         key = A2A3A4A5A6A7A8A9AAABACADAEAFB0B1
                               B2B3B4B5B6B7B8B9BABBBCBDBEBFC0C1
                          IV = 0000000000000000
               stream[0..63] = 7D0FF0DCB7CAAC90E548E24BEEA22D10
                               1C927E0A9BD559BC32BA70B346659F41
                               8FD9E36202D3AF35CB836F1BD15087DE
                               0D01FFF0BD42BC24B01A65CAD6F38E2C
            stream[192..255] = 12E246BA025A6174789C631646D092A8
                               865094571FF71BC28A38BEACEB08A822
                               72441DE97C1F273A9AE185B1F05B2953
                               EC37C940EE4C3AB5C901FF563563CCC9
            stream[256..319] = 2B48A7B5979BD5D27E841D2A6ED203D7
                               9126471DB9201444D07FCEA31A66D22F
                               DC65636F451B8D51365639CE2F5090B8
                               D08E14FE955580CB3692F4A35410D9BA
            stream[448..511] = A94E650CCC1ADEE62D2BAC9AA8969BA1
                               911429B6B9287E2E8A553752EDDF6F82
                               132FA5620E1F4F671EDF9C2EF1B76DB1
                               CE63A8A61EDF905A8D5D195D8EE7A116
                  xor-digest = 6492816A5383705890130321A2A5AFB7
                               B76B54481A48AF1F307EAA0AF41FB5FD
                               45CA6F00FE72C7D5C09E48406575651B
                               5674BC9488CF5EE93986F830947BF1A6

Set 3, vector#171:
                         key = ABACADAEAFB0B1B2B3B4B5B6B7B8B9BA
                               BBBCBDBEBFC0C1C2C3C4C5C6C7C8C9CA
                          IV = 0000000000000000
               stream[0..63] = F943B21C04A85C22ED1FC5BFBACAAF93
                               2CB889EF7CD4472089B16B6DDA5C72E9
                               A8F11B66CFC7677D72FB8908018B2A32
                               F6B37A2AC811665D8266841199C066AE
            stream[192..255] = E877CA4C8570A4A0CF06FECCCF0430BB
                               C63077B80518C4BFEC10BA18ABB08C0B
                               3FD72D94EED86F1A9A38385AD4395A96
                               7ABB10B245D71680E50C2918CB5AE210
            stream[256..319] = 89B67848C1661AFE6D54D7B7A92EB3FF
                               AB5D4E1438B6BEB9E51DE6733F08A71F
                               F16B676851ADD55712C5EE91B3F89381
                               0352A3C0DC7093FCC6D11810C475F472
            stream[448..511] = 14ABC36FB047EB4137390D3AA3486407
                               7400CDF9AC001025BA6F45BEDD460ECD
                               2FD4C16064F5579C50ACC64361EE9470
                               468B39F5CABCF366E0AE7DEA4EB1FEB1
                  xor-digest = 85D10891442BBD49CB301840BC9BFFDC
                               AAC81AAAAD8E6CF18E35C17B8E14255F
                               10650F031203035A67B68E4DA9414BF3
                               3C229E3F7C253F55771460CA6E804B09

Set 3, vector#180:
                         key = B4B5B6B7B8B9BABBBCBDBEBFC0C1C2C3
                               C4C5C6C7C8C9CACBCCCDCECFD0D1D2D3
                          IV = 0000000000000000
               stream[0..63] = 5F76E49A712A9B36D646FDB1355FA862
                               DE02BDC06E9AA4DF8DC0749102ADB071
                               D575101D0CA6E36034EE3A039CF5239B
                               817466A88DE350081D91090D79842DF5
            stream[192..255] = 48AEECB9BA29A1B52B2A5F58597980CF
                               2B5A31CD6DB97B98A4DB560500705ED7
                               0BF7D9946DF6B2D26C77E2BC3152F23C
                               2302F08ADE124F97E9E45F2894832434
            stream[256..319] = BD9BFA707093FD92BE49E0B0FD0A9E89
                               0AFD92AC6A50375173CE0C966C9D9A87
                               E2B538445E697EA193BD33D60DC9F107
                               1784CDA56C8AAD2BC67E17C9F5BDBAF8
            stream[448..511] = 1477E6B19CA394B91496C5C1E1EFE3D4
                               68D157B035C87A4667F6559F56C84ABF
                               3CE27D85D85784C40081EA064835904D
                               AE34A9277900B6F2F0B67F44B6B41776
                  xor-digest = E7FDF2693C8481BADDA0503996EAA6F8
                               201C2422907DC27CF747F8325B5FAB10
                               0567204E731A896F0128BFD87993C5C0
                               80B05AA3C75C9675BB7F9CBF935F502A

Set 3, vector#189:
                         key = BDBEBFC0C1C2C3C4C5C6C7C8C9CACBCC
                               CDCECFD0D1D2D3D4D5D6D7D8D9DADBDC
                          IV = 0000000000000000
               stream[0..63] = 1D8D3CB0B17972779FBD8339BDBC5D0C
                               4178C943381AFA6FA974FF792C78B4BB
                               5E0D8A2D2F9988C01F0FF7CE8AD310B6
                               6FA3B8D8CB507E507C4516BC9E7603B6
            stream[192..255] = F32D0691B1832478889516518C441ADB
                               8F0FE2165B15043756BB37928EBCA33F
                               9C166A5907F7F85CCF45CE6BFB68E725
                               748FA39528149A0E96B0B6C656854F88
            stream[256..319] = 66A7226EA4CF4DB203592F0C678BA8D2
                               99F26E212F2874681E29426A579469B2
                               CA747B8620E7E48A7E77D50E5C45FF62
                               A733D6052B2FB4AAB4AC782539193A76
            stream[448..511] = 25CCCD9E6FF25D8D6525E621BC376F6A
                               F73C749E80213260F1418B0C191B1F24
                               C1922DAD397EFA6062BBE9E3612D35D5
                               30F49C5D9D4F11E4CB2B3A4E66731FA8
                  xor-digest = 92BE9D3D0940E7447B043A3C0150AE8A
                               E28BC1F1D3EF2318E447210936356401
                               729A21A8BBA3FE17EAC0334B9E42E2BE
                               FE94CF0FEDBA97460B1BC07813A48053

Set 3, vector#198:
                         key = C6C7C8C9CACBCCCDCECFD0D1D2D3D4D5
                               D6D7D8D9DADBDCDDDEDFE0E1E2E3E4E5
                          IV = 0000000000000000
               stream[0..63] = 9D2EB0E9A93A0EF9F8ABCE0916C06EEB
                               E9C8EBB52A8112CD352A8E2E4EE84DFD
                               44B7C8251D0D1A36EA69CEB8C595D527
                               DA0EF26A2C5A5F443DC3040C6BF2DA49
            stream[192..255] = A86842C08DA057352B70FB63EBD1516F
                               D56E7BB389BBBB22F8EDE940DC7036CF
                               E10104AB81A51F23CFE35CCCC07BF50D
                               40A2438F3B3AEAB62953406A9E7D7BF3
            stream[256..319] = 9EE5EE22FFEDB13C11A81B0E5EC82DB6
                               303F22A62F0FD0574CE7007AF1EA2FCC
                               23D9C4196EBE897AB0D00371429F518E
                               C150063EAE314EE72EFADB1AA7714AC6
            stream[448..511] = 125ACD159548C79FCC93BFEC7B832C5D
                               387AFD85A0537BB6A49A8C3F4673306B
                               D76E17AC601629E00AB5AFF62B269491
                               AD996A624C6B1888BF13785AD63DEC7C
                  xor-digest = 9772AADF9F5BE8C14EC8304D594AF93E
                               1285357C9DFE9C10A1D45E5EB7D5BDCC
                               FCBF529FDCA3E620EB097575BFE68B08
                               04E63DD07C5FE3C8D8E28E2277E0358E

Set 3, vector#207:
                         key = CFD0D1D2D3D4D5D6D7D8D9DADBDCDDDE
                               DFE0E1E2E3E4E5E6E7E8E9EAEBECEDEE
                          IV = 0000000000000000
               stream[0..63] = 1D99BD420A9EBE17CF6144EEBE46A4B5
                               D8CE913F571DCEDEE6C6E3CFA27572F5
                               9983D4B2CADC292A956983AF7250CA81
                               A23A9EDA42417CC150597891045FF321
            stream[192..255] = D53AB2E60871F42D10E6747FE358E562
                               14D7CE3E7BA38E51354C801B72E5D515
                               DD805F8FDBA9F1BC81C5926DBE8CDBD2
                               3B006714CC8D550671036F6FD2991825
            stream[256..319] = FD97553220FB51132C33EBDA78606A24
                               5C5E3578A69754BF4FC11D6242605160
                               B4085DFDFC3D11505F72DC15CC16C683
                               37798E0DABD37C67B2E8912E498EA940
            stream[448..511] = A2D9199683D73F01DDD77BD46CD5BCEF
                               37CD9D4ECBA40B6C51446DCC68BCAD18
                               9FBEFEFC3D82131ECF98263299DC0CA9
                               1DD349E4DD348A88B2E3D7AA2D20CC13
                  xor-digest = 0F8AA6C52B1A2A36DA0EBE5C16BA2360
                               2E48161F836C228A0B8A413F6E0699A0
                               4DA22789A18C53A5125CFE51B9D7B5A1
                               D9957CDED4D1F48744944B65D2AE2290

Set 3, vector#216:
                         key = D8D9DADBDCDDDEDFE0E1E2E3E4E5E6E7
                               E8E9EAEBECEDEEEFF0F1F2F3F4F5F6F7
                          IV = 0000000000000000
               stream[0..63] = B9751AF24FCF14907948F7AD36E2649A
                               9A07B637F84D34E961EE82B7C33A9CC3
                               7B96DA6A956AFF4A629546C422802767
                               AD9F24BB2E79F09FCD43775FAC965123
            stream[192..255] = 6C4CB6AD15DDCE11F1BF68FFF1376E0F
                               4CE35ABCE777F4AB1D6906D09184689D
                               B697D1CFFAF46C5B85AD9F21CFF0D756
                               3DF67CF86D4199FA055F4BE18AFA34C2
            stream[256..319] = 35F4A1BBB9DA8476A82367A5607C72A0
                               C273A8D1F94DC4D62FDB2FA303858678
                               FABCD6C6EBA64849640BFB6FE4ADB340
                               28FAE26F802EA0ECE37D2AC2F2560CE8
            stream[448..511] = 3D208E3CFAF58AF11BCC527F948A3B75
                               E1751A28A76CBFE94204783820AD7FEE
                               7C98B318EDA2DC87111D18978CEE0C0C
                               E39F1469E7CB3EEEDBD6BF30DA68DF34
                  xor-digest = 7843987CBFAF3BC7DABD22E793F0F1B6
                               599E7774A6FAAA79B81A956F7C20964A
                               884A766CD76FDCDB67AAFAEACF24D221
                               5C5CE400F056F81A9EB0951A468502D9

Set 3, vector#225:
                         key = E1E2E3E4E5E6E7E8E9EAEBECEDEEEFF0
                               F1F2F3F4F5F6F7F8F9FAFBFCFDFEFF00
                          IV = 0000000000000000
               stream[0..63] = EA444200CDE137A48DD3728CFC0FE82A
                               1CD6F0F412C0343639052B6471F8321C
                               3C9A38986A5F882A26ABCFB342D3FF50
                               4E2EBF01D8CDA2408AE1A9023F4D64CA
            stream[192..255] = 5C20B3CECA032C29E7B8118BB8B946F9
                               90A9DD8895D9D7FE620727087DB8C6E9
                               6973741552A24E8C3B9EC81FA2B06E5F
                               F4283201639C83CC0C6AF8AA20FBDDD9
            stream[256..319] = 4DB2FF5167737BB90AD337FE16C10BD9
                               E4D2B8D6FBD172F5448D099D24FEAEA9
                               B30224AB670781C667292D04C76EFEC2
                               476B2D33ADA7A7132677E4B8270C68CD
            stream[448..511] = 5AB9F03158EA17B1D845CDC688C3BB0F
                               F1AC5CEAA2F16DB3178223D1471D0191
                               0E9D5BB3C6D0C9CC652C0ACF527B4F44
                               94B0DE521164493800E132B272A42A22
                  xor-digest = E7CADB2D003E6B1FB7ED9E085806817E
                               A548D2F1AFEF99ADED650D1B3DDF3533
                               C1E86435B9040DEEC83CEA60A501F35F
                               8538B9A4B3836B7D23A909100E244801

Set 3, vector#234:
                         key = EAEBECEDEEEFF0F1F2F3F4F5F6F7F8F9
                               FAFBFCFDFEFF00010203040506070809
                          IV = 0000000000000000
               stream[0..63] = 99A8CCEC6C5B2A0B6E336CB20652241C
                               32B24D34ACC0457EF679178EDE7CF805
                               805A9305C7C49909683BD1A803327817
                               627CA46FE8B929B6DF0012BD864183BE
            stream[192..255] = 2D226C11F47B3C0CCD0959B61F59D5CC
                               30FCEF6DBB8CBB3DCC1CC25204FCD449
                               8C37426A63BEA3282B1A8A0D60E13EB2
                               FE59241A9F6AF426689866EDC769E1E6
            stream[256..319] = 482FE1C128A15C1123B5655ED546DF01
                               4CE0C455DBF5D3A13D9CD4F0E2D1DAB9
                               F12FB68C544261D7F88EAC1C6CBF993F
                               BBB8E0AA8510BFF8E73835A1E86EADBB
            stream[448..511] = 0597188A1C19255769BE1C210399AD17
                               2EB46C52F92FD541DF2EAD71B1FF8EA7
                               ADD380EC71A5FD7ADB5181EADD1825EC
                               02779A4509BE5832708CA2836C1693A5
                  xor-digest = 0F8D6440841701C8C9BA58C0A86262AE
                               0220D0655B0B8C6DE7D2987BCC211A59
                               F2A23C932D0C17DF87C6B5F80AACB5AC
                               5A7894CB6B7552D0C0E235F1FCEAC442

Set 3, vector#243:
                         key = F3F4F5F6F7F8F9FAFBFCFDFEFF000102
                               030405060708090A0B0C0D0E0F101112
                          IV = 0000000000000000
               stream[0..63] = B4C0AFA503BE7FC29A62058166D56F8F
                               5D27DC246F75B9AD8760C8C39DFD8749
                               2D3B76D5D9637F009EADA14458A52DFB
                               09815337E72672681DDDC24633750D83
            stream[192..255] = DBBA0683DF48C335A9802EEF02522563
                               54C9F763C3FDE19131A6BB7B85040624
                               B1D6CD4BF66D16F7482236C8602A6D58
                               505EEDCCA0B77AED574AB583115124B9
            stream[256..319] = F0C5F98BAE05E019764EF6B65E0694A9
                               04CB9EC9C10C297B1AB1A6052365BB78
                               E55D3C6CB9F06184BA7D425A92E7E987
                               757FC5D9AFD7082418DD64125CA6F2B6
            stream[448..511] = 5A5FB5C8F0AFEA471F0318A4A2792F7A
                               A5C67B6D6E0F0DDB79961C34E3A564BA
                               2EECE78D9AFF45E510FEAB1030B102D3
                               9DFCECB77F5798F7D2793C0AB09C7A04
                  xor-digest = E940A6B3F4FF6EEDB11FF692E60C1246
                               392EB04AF868088EE85D813B0600CA91
                               E8C384620F059B6537F29431A534ADFF
                               92DB33C3615465AE4B19E6196F14C0DE

Set 3, vector#252:
                         key = FCFDFEFF000102030405060708090A0B
                               0C0D0E0F101112131415161718191A1B
                          IV = 0000000000000000
               stream[0..63] = 2064790538ACDF1DE3852C465070D962
                               FE2993BDD20C96DED5B2E5FA33283374
                               2A6B03966D47F8874D39C501ECFE0045
                               725C463530967ED1499097906B9775C3
            stream[192..255] = 9F880124435347E31FDF6EF96981FAB3
                               1A912D0B70210CBED6DDC9813521CCE2
                               B5C2B80193A59DCD933026D262E8EC74
                               F5880028FBB06166E0A304453A3A54BB
            stream[256..319] = 8A3F922FCDE48CE6C2E324EAA639DECC
                               E7257A25C420A2435BBA98740DF6C92A
                               8FA18F1D4E67C5F75F314219BB769685
                               A0C028D115321D10D58B46E5D58ABB4E
            stream[448..511] = 905C86F2F2C1E0454963E21D7498E8F4
                               67ECF23F8B02671F57584322E9952223
                               58D4FD541714BF12EFB189ACEA624AFF
                               2D55B252974D39D8598E8A066536ACB2
                  xor-digest = 4750CDBC728F8AB112C58235A5891BA1
                               84B79BAF5172AC7C530F57023F1E9CFD
                               26071B4826FF3C6971DC2A7DD8FB35B0
                               054D59D5538746F0C4C2D1A8DE6DC771

Test vectors -- set 4
=====================

Set 4, vector#  0:
                         key = 0053A6F94C9FF24598EB3E91E4378ADD
                               3083D6297CCF2275C81B6EC11467BA0D
                          IV = 0000000000000000
               stream[0..63] = F9D2DC274BB55AEFC2A0D9F8A982830F
                               6916122BC0A6870F991C6ED8D00D2F85
                               94E3151DE4C5A19A9A06FBC191C87BF0
                               39ADF971314BAF6D02337080F2DAE5CE
        stream[65472..65535] = 05BDA8EE240BA6DC53A42C14C17F620F
                               6FA799A6BC88775E04EEF427B4B9DE5A
                               5349327FCADA077F385BA321DB4B3939
                               C0F49EA99801790B0FD32986AFC41B85
        stream[65536..65599] = FED5279620FBCBDD3C3980B11FCE4787
                               E6F9F97772BEAAD0EF215FDCD0B3A16F
                               BB56D72AFD5FD52E6A584BF840914168
                               D04A594FFDDA959A63EB4CF42694F03F
      stream[131008..131071] = F161DCE8FA4CF80F8143DDB21FA1BFA3
                               1CA4DC0A412233EDE80EF72DAA1B8039
                               4BCE3875CA1E1E195D58BC3197F803A8
                               9C433A59A0718C1A009BCB4DA2AC1778
                  xor-digest = 2052F9A2853E989133D10938222AC76D
                               B8B4CBA135ACB59970DDF9C074C6271A
                               5C4E2A7A00D2D697EDFC9B1FF9B365C8
                               7347B23020663A30711A71E3A02AB00C

Set 4, vector#  1:
                         key = 0558ABFE51A4F74A9DF04396E93C8FE2
                               3588DB2E81D4277ACD2073C6196CBF12
                          IV = 0000000000000000
               stream[0..63] = 2F634849A4EDC206CE3E3F89949DF4E6
                               EA9A0E3EE87F0AB108C4D3B789ACE673
                               07AC8C54F07F30BAD9640B7F6EDEEC9D
                               B15E51599EB15E1CA94739FEA5F1E3D7
        stream[65472..65535] = EB2B0FD63C7EEEAA5A4D712EEEFC0A7E
                               214BEB04D3FDA19C32250949868216D3
                               A659B312E13EC66C5832E970F9C91FF9
                               4F7463439A9827ECCA52248D3CC604CD
        stream[65536..65599] = 425E0DF93A3DE6B22E0871EB4E435691
                               D77B5C471228DE302A79001F89F7E77D
                               837C5CA0177B2206568EDC2EB0F169D5
                               6B414B9DCCDC928659B4BE1E0DEDFF73
      stream[131008..131071] = 6AA3D6938B6B54B4CB8D2885274A991B
                               4A0D5CCF35D981953EC64452FACC8640
                               B5ACFA39A372E38BE4E10EE68E7F1B50
                               5A5660CDFBAE8DCBFCC9A3847BBB6BA4
                  xor-digest = 61F90A34A70BEE706D298B31B281BFC7
                               2CF9E82394F6AD7277AAFE176CDB6D62
                               8E723AC403D892A85AC907D48DD2C3CB
                               CB6C622297670AD2590BE9A774B07D65

Set 4, vector#  2:
                         key = 0A5DB00356A9FC4FA2F5489BEE4194E7
                               3A8DE03386D92C7FD22578CB1E71C417
                          IV = 0000000000000000
               stream[0..63] = 0A8BBD088ABADC4D57D3389E32175878
                               125BD89DE7E9D05DBF29B753F5F0C2CB
                               F0EEF9333526E9308A114E06EB9564EB
                               35C28EA93C17BEF0466748079A355B9C
        stream[65472..65535] = F47FDFF047F0303F6CCE2510FA2475F0
                               7784D5F0FBD63D1746BD8CE4BB02802C
                               3052A375D7DE75D439174E7B19CEBA3B
                               9546DB027F14FFDB9EF542D5768CE5A7
        stream[65536..65599] = 40FEC0EE1697D63CB04299A17C446DE0
                               6B3407D10C6DD2143DFA24EB7362D09A
                               6857C6AA83A191D65B05EBBBC8133D12
                               2BDE75900C86FCD8785EECE48659C3B0
      stream[131008..131071] = 7820087794D46993E984536E7B74C615
                               67AB34C6C0A90090DB080E6EB79532FB
                               414CD1145A781A2C55519A3E3AD19FA6
                               D78790313EBE19A86F61068E4C8E508D
                  xor-digest = 67125CED828BA1AC0E22B29E75886255
                               ED129F94F30B83E81E9DACFC4D2BD1DD
                               782BCC1929B62D754D1CC0AB120A24A4
                               8FB8190CEF0519F73B404C97A83E3925

Set 4, vector#  3:
                         key = 0F62B5085BAE0154A7FA4DA0F34699EC
                               3F92E5388BDE3184D72A7DD02376C91C
                          IV = 0000000000000000
               stream[0..63] = 4A671A2AE75DB7555BEA5995DC53AF8D
                               C1E8776AF917A3AB2CA9827BCED53DA7
                               00B779820F17294751A2C37EF5CCCFE9
                               7BF7481E85AFC9ECAE431B7CF05F6153
        stream[65472..65535] = 15C415BE73C12230AC9505B92B2B1273
                               7F6FB2FAAF9C51F22ECCB8CBED36A27A
                               1E0738E1252D26E8E5E5651FE8AA02CC
                               9887D141A7CBAE80F01BE09B314005BB
        stream[65536..65599] = 1C48158413F5EC5E64D2FA4786D91D27
                               27DF6BECD614F6AE745CF2B6F35CD824
                               3E5F1C440BEDE01E6C8A1145F2AB77FA
                               24D634DE88F955D4F830D4A548A926D0
      stream[131008..131071] = A9BE2FB00C8BD01054153F77EC0C633C
                               E8DF7F78E994907B9F387FF090CB3B95
                               4271FEADF50C9084106F4285FF4F534D
                               AEC130AAE287D47033179BBAEEB36CE6
                  xor-digest = FE8E842CB0F33D020632A0A682AF9EF0
                               AD3715E3DBAF4CD3591D46B1CE47FCEB
                               6D3A04AE59AF466E22EE507FB8BD58F7
                               4C643E138029521638A3B066305F60DF

Test vectors -- set 5
=====================

Set 5, vector#  0:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 8000000000000000
               stream[0..63] = 2ABA3DC45B4947007B14C851CD694456
                               B303AD59A465662803006705673D6C3E
                               29F1D3510DFC0405463C03414E0E07E3
                               59F1F1816C68B2434A19D3EEE0464873
            stream[192..255] = EFF0C107DCA563B5C0048EB488B40341
                               ED34052790475CD204A947EB480F3D75
                               3EF5347CEBB0A21F25B6CC8DE6B48906
                               E604F554A6B01B23791F95C4A93A4717
            stream[256..319] = E3393E1599863B52DE8C52CF26C752FB
                               473B74A34D6D9FE31E9CA8DD6292522F
                               13EB456C5BE9E5432C06E1BA3965D454
                               48936BC98376BF903969F049347EA05D
            stream[448..511] = FC4B2EF3B6B3815C99A437F16BDB06C5
                               B948692786081D91C48CC7B072ABB901
                               C0491CC6900F2FEA217BFFC70C43EDD6
                               65E3E020B59AAA43868E9949FBB9AE22
                  xor-digest = FE40F57D1586D7664C2FCA5AB10BD7C7
                               9DE3234836E76949F9DC01CBFABC6D6C
                               42AB27DDC748B4DF7991092972AB4985
                               CEC19B3E7C2C85D6E25A338DEC288282

Set 5, vector#  9:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0040000000000000
               stream[0..63] = F28343BCF4C946FC95DCAAED9DA10B27
                               7E573FC8EBC8CEE246FDDC533D29C2EA
                               05451ED9A821C4161EE0AFA32EC0FCA0
                               DAD124B702DA9248B3D2AA64489C9D26
            stream[192..255] = C65F799168D6B229D0281309526B746C
                               490D3EDC0F6408A04339275FCE04BDF4
                               656AB5868495C32D238FDB97869A9332
                               E09CB7BE8031D38B8F565FB5469C8459
            stream[256..319] = 03E48FD41282FCD62C7217ED64153E55
                               B558F82A613245C3D8A885542346AA39
                               27DE9734C0581338C3DE5DB443EC4227
                               E3F82677D259D2D42601D187C79BF87A
            stream[448..511] = 551F95AD9751E4F4BACE7FD48B6A3C67
                               E86C4B1E5B747BA60377B07FE8365E09
                               F8973085F8A6086FC56BD88168D8C561
                               8B01B159EF29F658C85FD117925D46E0
                  xor-digest = 17E005D5257D3ECE55DBE678290C9E1F
                               E0D1E18CA8D54F6271E83D1A94580D8A
                               597403F680674F564D130D71111ADD2D
                               A17E21268D5A8407CB2721730776DC94

Set 5, vector# 18:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000200000000000
               stream[0..63] = 621F3014E0ADC8022868C3D9070BC49E
                               48BC6B504AFF11CB17957F0EBFB7612F
                               7FCB67C60A2FBD7A4BD7C312E8F50AF3
                               CA7520821D73DB47189DAD557C436DDC
            stream[192..255] = 42C8DFE869C90018825E2037BB5E2EBB
                               C4A4A42660AFEA8A2E385AFBBC63EF30
                               98D052FF4A52ED12107EE71C1AEC271E
                               6870538FCEAA1191B4224A6FFDCE5327
            stream[256..319] = 4214DA4FAF0DF7FC2955D81403C9D49E
                               E87116B1975C5823E28D9A08C5B1189D
                               C52BCBEF065B637F1870980CB778B75A
                               DDA41613F5F4728AD8D8D189FBF0E76D
            stream[448..511] = 4CA854257ECE95E67383FC8665C3A823
                               8B87255F815CA4DEC2D57DB72924C60C
                               B20A7EE40C559406AAAB25BE5F47184D
                               D187ED7EA191133F3000CB88DCBAC433
                  xor-digest = 3191EA19C819BC3382B2C085B709E569
                               54D91532C88F49E77BACCCBCE6312A46
                               A6079A13DF08EFEE0F1A6C95BE79E919
                               87677D1CAAE2E0CE253B3EE47F825EAF

Set 5, vector# 27:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000001000000000
               stream[0..63] = D2DB1A5CF1C1ACDBE81A7A4340EF5343
                               5E7F4B1A50523F8D283DCF851D696E60
                               F2DE7456181B8410D462BA6050F061F2
                               1C787FC12434AF58BF2C59CA9077F3B0
            stream[192..255] = 6CE020B3E83765A11F9AE157AD2D07D1
                               EA4E9FBBF386C83FEF54319746E5F997
                               D35BE9F73B99772DA97054FF07301314
                               3FF9E5B47C61966D8525F17265F48D08
            stream[256..319] = FFEAB16EEA5C43BFD08D2591F9A40293
                               24CDDC83A840B2C136B7CE99AF3A66CB
                               3084E4E2CA6F44AC5CEAF7A1157BE267
                               3DF688B43BD51B9A8444CE194E3CA7F2
            stream[448..511] = 0D3873FD47A7B3400115C40574469D21
                               5BCE0679ED5CF9E374E473B4427DE498
                               5804DD75151D72EE367A3F066E641B7F
                               5CF28A67215B74DD80EB3FC02E12A308
                  xor-digest = 838960663A70C2CACD205BC38D8BCC12
                               8438D6A03006727EF1440B1D3C7FD557
                               CC4A02AC9CF7D51DCFE3862ACCDEEEBE
                               B15393EE6D8E4483710932C4B44990B5

Set 5, vector# 36:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000008000000
               stream[0..63] = 22E129373F7589D9EAFFF18DEA63432E
                               38D0245BAE221D3635BEE176760552B8
                               9B6BC49CFEB7D9A5B358963C488ED8FA
                               D01F1C72307CADEEF9C20273FB5D6775
            stream[192..255] = 6E6FFCB8B324EE4FF55E64449B2A356B
                               D53D8AB7747DFFC0B3D044E0BE1A736B
                               4AB2109624600FE8CA7E6949A4DF82AC
                               A5C96D039F78B67767A1B66FAB0EF24B
            stream[256..319] = C3DF823DBA0F84D70E425D0C2C88DCE3
                               CAEC3ACCA435B5A2832BE2E0F0AA46AD
                               3F288AFE49BE5C345DC65445D26993F5
                               1E3F46E0C1B02B5AEDF73D68336AA04F
            stream[448..511] = 443B0FDC4F8365AB93A07682EBCA7B92
                               42259A26DAB3574B2E562CCABDB25633
                               96F331146347C26D5DB49C87054642F8
                               60FC1A0B87468ED0B5CB9C30D72EA8F7
                  xor-digest = 8D2110EA7CABD4A3DAEC4768131D8AED
                               9E7E1EB1E1F553F7EE38FB339B648444
                               0F43AB7C7F17BB593BF2CFB53688CC74
                               F00A32117039DA4745F78E66059000E6

Set 5, vector# 45:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000040000
               stream[0..63] = DC302570A4D1C44F31D9FA55C7712B11
                               AE770BFAA3F8631DFF924BCF00A09C90
                               6571B024CE5264215E516D73416BF3E3
                               CE373CAE669DB1A057EFD7EB184243B6
            stream[192..255] = A52427068F8048FC5E3E6E94A1A616CD
                               11F5A9ED4F8899F780F67836EEC4FADB
                               B19C183C6946541F182F224104DF9444
                               66D96A6CE7F2EFE723807A8738950AD9
            stream[256..319] = D1410A14DFA3DA5C9BDF18A34476F7C0
                               D7A8373331741ED62682C555EA8B62A8
                               1EDB10DB9479BAF2CD532CFB18357A92
                               FF90897315F69CEE526DE31329CFA06B
            stream[448..511] = 9CA44AF188E42090F9969FB5F771C987
                               557912B83261760EE80A809F7E398A66
                               D56049FFDFFBD3E16633537B84AFB38E
                               564B717A0C26EBFEE907B8EF7FDA31F0
                  xor-digest = BBF40E80EBBCBCC93067CAC497FD4403
                               D1E797AC131A593D06BCA34ED650CB3D
                               70403B26550EBB76E00CE04CB15F28CA
                               99F1700ABBA462F041D474E008488F0D

Set 5, vector# 54:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000200
               stream[0..63] = 98951956F4BD5E2E9DC624CCD2D79E60
                               6D24A4DB51D413FDAF9A9741A6F079B4
                               21400FDA0B4D8785578BB318BDAD4ABC
                               A8C2D1BA3BA4E18C2F5572499F345BC1
            stream[192..255] = C3A267F0EB87ED714E09CABC2780FEF6
                               E5F665BBBBB44C8448D8EB42D88275CD
                               62AD759AAC9F4080F73993DE50FF94E8
                               34E2CF7B74A91E68B38EACE9C12922C2
            stream[256..319] = 78BD0BB32A69E62362EE7E31F1DD9E96
                               CA6E196844EFD9459F270D612119DFA4
                               5DD1522967629143CECD585CFE62B7FD
                               9D1503A62A238C35A66595C49DD71575
            stream[448..511] = C17F946C14A492392A1C554993F406B2
                               EA806E4186D97FCB420C21FB4245A3DB
                               4EBA2BCB59D2C33CE2CD5044A79A96F9
                               5182112D9724E16AD9E965047DA71F05
                  xor-digest = 0094C2C02618E924D6CD7C96CBD6A44A
                               33D3983DEAD3084C1A694490B367CE6D
                               7732300C3BA3848C2EA143031A980BE4
                               9C1354A528D1E1E1B1A55AFF18BD0F92

Set 5, vector# 63:
                         key = 00000000000000000000000000000000
                               00000000000000000000000000000000
                          IV = 0000000000000001
               stream[0..63] = B47F96AA96786135297A3C4EC56A613D
                               0B80095324FF43239D684C57FFE42E1C
                               44F3CC011613DB6CDC880999A1E65AED
                               1287FCB11C839C37120765AFA73E5075
            stream[192..255] = 97128BD699DDC1B4B135D94811B5D2D6
                               B2ADCBDC1ED8D3CF86ECF65A1750DE66
                               CA5F1C2ED350DC2F497396E029DBD4A0
                               6FDDA6238BE7D120DD41E9F19E6DEEA2
            stream[256..319] = FF8065AD901A2DFC5C01642A840F7593
                               AE032946058E54EA67300FBF7B928C20
                               3244EF546762BA640032B6A2514122DE
                               0CA969283F70CE21F981A5D668274F0D
            stream[448..511] = 1309268BE548EFEC38D79DF4334CA949
                               AB15A2A1003E2B97969FE0CD74A16A06
                               5FE8691F03CBD0ECFCF6312F2EE0697F
                               44BD3BF3E60320B289CBF21B428C8922
                  xor-digest = 1115D387A0C41A67543BE13DD539AB84
                               4D9A2CC98C20BB6E7B092268C060884F
                               53774E3B044C6058B137CCCADF9F7026
                               96B3D40DFF3835341E4BF01BDD1C8FB1

Test vectors -- set 6
=====================

Set 6, vector#  0:
                         key = 0053A6F94C9FF24598EB3E91E4378ADD
                               3083D6297CCF2275C81B6EC11467BA0D
                          IV = 0D74DB42A91077DE
               stream[0..63] = F5FAD53F79F9DF58C4AEA0D0ED9A9601
                               F278112CA7180D565B420A48019670EA
                               F24CE493A86263F677B46ACE1924773D
                               2BB25571E1AA8593758FC382B1280B71
        stream[65472..65535] = B70C50139C63332EF6E77AC54338A407
                               9B82BEC9F9A403DFEA821B83F7860791
                               650EF1B2489D0590B1DE772EEDA4E3BC
                               D60FA7CE9CD623D9D2FD5758B8653E70
        stream[65536..65599] = 81582C65D7562B80AEC2F1A673A9D01C
                               9F892A23D4919F6AB47B9154E08E699B
                               4117D7C666477B60F8391481682F5D95
                               D96623DBC489D88DAA6956B9F0646B6E
      stream[131008..131071] = A13FFA1208F8BF50900886FAAB40FD10
                               E8CAA306E63DF39536A1564FB760B242
                               A9D6A4628CDC878762834E27A541DA2A
                               5E3B3445989C76F611E0FEC6D91ACACC
                  xor-digest = C349B6A51A3EC9B712EAED3F90D8BCEE
                               69B7628645F251A996F55260C62EF31F
                               D6C6B0AEA94E136C9D984AD2DF3578F7
                               8E457527B03A0450580DD874F63B1AB9

Set 6, vector#  1:
                         key = 0558ABFE51A4F74A9DF04396E93C8FE2
                               3588DB2E81D4277ACD2073C6196CBF12
                          IV = 167DE44BB21980E7
               stream[0..63] = 3944F6DC9F85B128083879FDF190F7DE
                               E4053A07BC09896D51D0690BD4DA4AC1
                               062F1E47D3D0716F80A9B4D85E6D6085
                               EE06947601C85F1A27A2F76E45A6AA87
        stream[65472..65535] = 36E03B4B54B0B2E04D069E690082C8C5
                               92DF56E633F5D8C7682A02A65ECD1371
                               8CA4352AACCB0DA20ED6BBBA62E177F2
                               10E3560E63BB822C4158CAA806A88C82
        stream[65536..65599] = 1B779E7A917C8C26039FFB23CF0EF8E0
                               8A1A13B43ACDD9402CF5DF38501098DF
                               C945A6CC69A6A17367BC03431A86B3ED
                               04B0245B56379BF997E25800AD837D7D
      stream[131008..131071] = 7EC6DAE81A105E67172A0B8C4BBE7D06
                               A7A8759F914FBEB1AF62C8A552EF4A4F
                               56967EA29C7471F46F3B07F7A3746E95
                               3D315821B85B6E8CB40122B96635313C
                  xor-digest = C3EAAF32836BACE32D04E1124231EF47
                               E101367D6305413A0EEB07C60698A287
                               6E4D031870A739D6FFDDD208597AFF0A
                               47AC17EDB0167DD67EBA84F1883D4DFD

Set 6, vector#  2:
                         key = 0A5DB00356A9FC4FA2F5489BEE4194E7
                               3A8DE03386D92C7FD22578CB1E71C417
                          IV = 1F86ED54BB2289F0
               stream[0..63] = 3FE85D5BB1960A82480B5E6F4E965A44
                               60D7A54501664F7D60B54B06100A37FF
                               DCF6BDE5CE3F4886BA77DD5B44E95644
                               E40A8AC65801155DB90F02522B644023
        stream[65472..65535] = C8D6E54C29CA204018A830E266CEEE0D
                               037DC47E921947302ACE40D1B996A6D8
                               0B598677F3352F1DAA6D9888F891AD95
                               A1C32FFEB71BB861E8B07058515171C9
        stream[65536..65599] = B79FD776542B4620EFCB88449599F234
                               03E74A6E91CACC50A05A8F8F3C0DEA8B
                               00E1A5E6081F5526AE975B3BC0450F1A
                               0C8B66F808F1904B971361137C93156F
      stream[131008..131071] = 7998204FED70CE8E0D027B206635C08C
                               8BC443622608970E40E3AEDF3CE790AE
                               EDF89F922671B45378E2CD03F6F62356
                               529C4158B7FF41EE854B1235373988C8
                  xor-digest = 3CD23C3DC90201ACC0CF49B440B6C417
                               F0DC8D8410A716D5314C059E14B1A8D9
                               A9FB8EA3D9C8DAE12B21402F674AA95C
                               67B1FC514E994C9D3F3A6E41DFF5BBA6

Set 6, vector#  3:
                         key = 0F62B5085BAE0154A7FA4DA0F34699EC
                               3F92E5388BDE3184D72A7DD02376C91C
                          IV = 288FF65DC42B92F9
               stream[0..63] = 5E5E71F90199340304ABB22A37B6625B
                               F883FB89CE3B21F54A10B81066EF87DA
                               30B77699AA7379DA595C77DD59542DA2
                               08E5954F89E40EB7AA80A84A6176663F
        stream[65472..65535] = 2DA2174BD150A1DFEC1796E921E9D6E2
                               4ECF0209BCBEA4F98370FCE629056F64
                               917283436E2D3F45556225307D5CC5A5
                               65325D8993B37F1654195C240BF75B16
        stream[65536..65599] = ABF39A210EEE89598B7133377056C2FE
                               F42DA731327563FB67C7BEDB27F38C7C
                               5A3FC2183A4C6B277F901152472C6B2A
                               BCF5E34CBE315E81FD3D180B5D66CB6C
      stream[131008..131071] = 1BA89DBD3F98839728F56791D5B7CE23
                               5036DE843CCCAB0390B8B5862F1E4596
                               AE8A16FB23DA997F371F4E0AACC26DB8
                               EB314ED470B1AF6B9F8D69DD79A9D750
                  xor-digest = E00EBCCD70D69152725F9987982178A2
                               E2E139C7BCBE04CA8A0E99E318D9AB76
                               F988C8549F75ADD790BA4F81C176DA65
                               3C1A043F11A958E169B6D2319F4EEC1A



End of test vectors
