// Copyright 2013 The Closure Library Authors. All Rights Reserved.
// Use of this source code is governed by the Apache License, Version 2.0.

////////////////////////// NOTE ABOUT EDITING THIS FILE ///////////////////////
//                                                                           //
// Any edits to this file must be applied to vec2d_test.js by running:       //
//   swap_type.sh vec2f_test.js > vec2d_test.js                              //
//                                                                           //
////////////////////////// NOTE ABOUT EDITING THIS FILE ///////////////////////

goog.provide('goog.vec.vec2fTest');
goog.setTestOnly('goog.vec.vec2fTest');

goog.require('goog.vec.Float32Array');
goog.require('goog.vec.vec2f');
goog.require('goog.testing.jsunit');

function testCreate() {
  var v = goog.vec.vec2f.create();
  assertElementsEquals([0, 0], v);
}

function testCreateFromArray() {
  var v = goog.vec.vec2f.createFromArray([1, 2]);
  assertElementsEquals([1, 2], v);
}

function testCreateFromValues() {
  var v = goog.vec.vec2f.createFromValues(1, 2);
  assertElementsEquals([1, 2], v);
}

function testClone() {
  var v0 = goog.vec.vec2f.createFromValues(1, 2);
  var v1 = goog.vec.vec2f.clone(v0);
  assertElementsEquals([1, 2], v1);
}

function testSet() {
  var v = goog.vec.vec2f.create();
  goog.vec.vec2f.setFromValues(v, 1, 2);
  assertElementsEquals([1, 2], v);

  goog.vec.vec2f.setFromArray(v, [4, 5]);
  assertElementsEquals([4, 5], v);

  var w = goog.vec.vec2f.create();
  goog.vec.vec2f.setFromValues(w, 1, 2);
  assertElementsEquals([1, 2], w);

  goog.vec.vec2f.setFromArray(w, [4, 5]);
  assertElementsEquals([4, 5], w);
}

function testAdd() {
  var v0 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [1, 2]);
  var v1 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [4, 5]);
  var v2 = goog.vec.vec2f.setFromVec2f(goog.vec.vec2f.create(), v0);

  goog.vec.vec2f.add(v2, v1, v2);
  assertElementsEquals([1, 2], v0);
  assertElementsEquals([4, 5], v1);
  assertElementsEquals([5, 7], v2);

  goog.vec.vec2f.add(goog.vec.vec2f.add(v0, v1, v2), v0, v2);
  assertElementsEquals([6, 9], v2);
}

function testSubtract() {
  var v0 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [1, 2]);
  var v1 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [4, 5]);
  var v2 = goog.vec.vec2f.setFromVec2f(goog.vec.vec2f.create(), v0);

  goog.vec.vec2f.subtract(v2, v1, v2);
  assertElementsEquals([1, 2], v0);
  assertElementsEquals([4, 5], v1);
  assertElementsEquals([-3, -3], v2);

  goog.vec.vec2f.setFromValues(v2, 0, 0);
  goog.vec.vec2f.subtract(v1, v0, v2);
  assertElementsEquals([3, 3], v2);

  v2 = goog.vec.vec2f.setFromVec2f(goog.vec.vec2f.create(), v0);
  goog.vec.vec2f.subtract(v2, v1, v2);
  assertElementsEquals([-3, -3], v2);

  goog.vec.vec2f.subtract(goog.vec.vec2f.subtract(v1, v0, v2), v0, v2);
  assertElementsEquals([2, 1], v2);
}

function testMultiply() {
  var v0 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [1, 2]);
  var v1 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [4, 5]);
  var v2 = goog.vec.vec2f.setFromVec2f(goog.vec.vec2f.create(), v0);

  goog.vec.vec2f.componentMultiply(v2, v1, v2);
  assertElementsEquals([1, 2], v0);
  assertElementsEquals([4, 5], v1);
  assertElementsEquals([4, 10], v2);

  goog.vec.vec2f.componentMultiply(goog.vec.vec2f.componentMultiply(v0, v1, v2), v0, v2);
  assertElementsEquals([4, 20], v2);
}

function testDivide() {
  var v0 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [1, 2]);
  var v1 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [4, 5]);
  var v2 = goog.vec.vec2f.setFromVec2f(goog.vec.vec2f.create(), v0);

  goog.vec.vec2f.componentDivide(v2, v1, v2);
  assertElementsRoughlyEqual([1, 2], v0, 10e-5);
  assertElementsRoughlyEqual([4, 5], v1, 10e-5);
  assertElementsRoughlyEqual([.25, .4], v2, 10e-5);

  goog.vec.vec2f.setFromValues(v2, 0, 0);
  goog.vec.vec2f.componentDivide(v1, v0, v2);
  assertElementsRoughlyEqual([4, 2.5], v2, 10e-5);

  v2 = goog.vec.vec2f.setFromVec2f(goog.vec.vec2f.create(), v0);
  goog.vec.vec2f.componentDivide(v2, v1, v2);
  assertElementsRoughlyEqual([.25, .4], v2, 10e-5);

  goog.vec.vec2f.componentDivide(goog.vec.vec2f.componentDivide(v1, v0, v2), v0, v2);
  assertElementsRoughlyEqual([4, 1.25], v2, 10e-5);
}

function testNegate() {
  var v0 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [1, 2]);
  var v1 = goog.vec.vec2f.create();

  goog.vec.vec2f.negate(v0, v1);
  assertElementsEquals([-1, -2], v1);
  assertElementsEquals([1, 2], v0);

  goog.vec.vec2f.negate(v0, v0);
  assertElementsEquals([-1, -2], v0);
}

function testAbs() {
  var v0 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [-1, -2]);
  var v1 = goog.vec.vec2f.create();

  goog.vec.vec2f.abs(v0, v1);
  assertElementsEquals([1, 2], v1);
  assertElementsEquals([-1, -2], v0);

  goog.vec.vec2f.abs(v0, v0);
  assertElementsEquals([1, 2], v0);
}

function testScale() {
  var v0 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [1, 2]);
  var v1 = goog.vec.vec2f.create();

  goog.vec.vec2f.scale(v0, 4, v1);
  assertElementsEquals([4, 8], v1);
  assertElementsEquals([1, 2], v0);

  goog.vec.vec2f.setFromArray(v1, v0);
  goog.vec.vec2f.scale(v1, 5, v1);
  assertElementsEquals([5, 10], v1);
}

function testMagnitudeSquared() {
  var v0 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [1, 2]);
  assertEquals(5, goog.vec.vec2f.magnitudeSquared(v0));
}

function testMagnitude() {
  var v0 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [1, 2]);
  assertEquals(Math.sqrt(5), goog.vec.vec2f.magnitude(v0));
}

function testNormalize() {
  var v0 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [2, 3]);
  var v1 = goog.vec.vec2f.create();
  var v2 = goog.vec.vec2f.create();
  goog.vec.vec2f.scale(
      v0, 1 / goog.vec.vec2f.magnitude(v0), v2);

  goog.vec.vec2f.normalize(v0, v1);
  assertElementsEquals(v2, v1);
  assertElementsEquals([2, 3], v0);

  goog.vec.vec2f.setFromArray(v1, v0);
  goog.vec.vec2f.normalize(v1, v1);
  assertElementsEquals(v2, v1);
}

function testDot() {
  var v0 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [1, 2]);
  var v1 = goog.vec.vec2f.setFromArray(goog.vec.vec2f.create(), [4, 5]);
  assertEquals(14, goog.vec.vec2f.dot(v0, v1));
  assertEquals(14, goog.vec.vec2f.dot(v1, v0));
}

function testDistanceSquared() {
  var v0 = goog.vec.vec2f.setFromValues(goog.vec.vec2f.create(), 1, 2);
  var v1 = goog.vec.vec2f.setFromValues(goog.vec.vec2f.create(), 1, 2);
  assertEquals(0, goog.vec.vec2f.distanceSquared(v0, v1));
  goog.vec.vec2f.setFromValues(v0, 1, 2);
  goog.vec.vec2f.setFromValues(v1, -1, -2);
  assertEquals(20, goog.vec.vec2f.distanceSquared(v0, v1));
}

function testDistance() {
  var v0 = goog.vec.vec2f.setFromValues(goog.vec.vec2f.create(), 1, 2);
  var v1 = goog.vec.vec2f.setFromValues(goog.vec.vec2f.create(), 1, 2);
  assertEquals(0, goog.vec.vec2f.distance(v0, v1));
  goog.vec.vec2f.setFromValues(v0, 2, 3);
  goog.vec.vec2f.setFromValues(v1, -2, 0);
  assertEquals(5, goog.vec.vec2f.distance(v0, v1));
}

function testDirection() {
  var v0 = goog.vec.vec2f.setFromValues(goog.vec.vec2f.create(), 1, 2);
  var v1 = goog.vec.vec2f.setFromValues(goog.vec.vec2f.create(), 1, 2);
  var dirVec = goog.vec.vec2f.setFromValues(goog.vec.vec2f.create(), 4, 5);
  goog.vec.vec2f.direction(v0, v1, dirVec);
  assertElementsEquals([0, 0], dirVec);
  goog.vec.vec2f.setFromValues(v0, 0, 0);
  goog.vec.vec2f.setFromValues(v1, 1, 0);
  goog.vec.vec2f.direction(v0, v1, dirVec);
  assertElementsEquals([1, 0], dirVec);
  goog.vec.vec2f.setFromValues(v0, 1, 1);
  goog.vec.vec2f.setFromValues(v1, 0, 0);
  goog.vec.vec2f.direction(v0, v1, dirVec);
  assertElementsRoughlyEqual(
      [-0.707106781, -0.707106781],
      dirVec, goog.vec.EPSILON);
}

function testLerp() {
  var v0 = goog.vec.vec2f.setFromValues(goog.vec.vec2f.create(), 1, 2);
  var v1 = goog.vec.vec2f.setFromValues(goog.vec.vec2f.create(), 10, 20);
  var v2 = goog.vec.vec2f.setFromVec2f(goog.vec.vec2f.create(), v0);

  goog.vec.vec2f.lerp(v2, v1, 0, v2);
  assertElementsEquals([1, 2], v2);
  goog.vec.vec2f.lerp(v2, v1, 1, v2);
  assertElementsEquals([10, 20], v2);
  goog.vec.vec2f.lerp(v0, v1, .5, v2);
  assertElementsEquals([5.5, 11], v2);
}

function testMax() {
  var v0 = goog.vec.vec2f.setFromValues(goog.vec.vec2f.create(), 10, 20);
  var v1 = goog.vec.vec2f.setFromValues(goog.vec.vec2f.create(), 5, 25);
  var v2 = goog.vec.vec2f.create();

  goog.vec.vec2f.max(v0, v1, v2);
  assertElementsEquals([10, 25], v2);
  goog.vec.vec2f.max(v1, v0, v1);
  assertElementsEquals([10, 25], v1);
  goog.vec.vec2f.max(v2, 20, v2);
  assertElementsEquals([20, 25], v2);
}

function testMin() {
  var v0 = goog.vec.vec2f.setFromValues(goog.vec.vec2f.create(), 10, 20);
  var v1 = goog.vec.vec2f.setFromValues(goog.vec.vec2f.create(), 5, 25);
  var v2 = goog.vec.vec2f.create();

  goog.vec.vec2f.min(v0, v1, v2);
  assertElementsEquals([5, 20], v2);
  goog.vec.vec2f.min(v1, v0, v1);
  assertElementsEquals([5, 20], v1);
  goog.vec.vec2f.min(v2, 10, v2);
  assertElementsEquals([5, 10], v2);
}

function testEquals() {
  var v0 = goog.vec.vec2f.setFromValues(goog.vec.vec2f.create(), 1, 2);
  var v1 = goog.vec.vec2f.setFromVec2f(goog.vec.vec2f.create(), v0);
  assertElementsEquals(v0, v1);

  v1[0] = 4;
  assertFalse(goog.vec.vec2f.equals(v0, v1));

  v1 = goog.vec.vec2f.setFromVec2f(goog.vec.vec2f.create(), v0);
  v1[1] = 4;
  assertFalse(goog.vec.vec2f.equals(v0, v1));
}
