/*
 GNU Maverik - a system for managing display and interaction in 
               Virtual Environment applications.
 Copyright (C) 1999 Advanced Interfaces Group

 This program is free software; you can redistribute it and/or
 modify it under the terms of the GNU General Public License
 as published by the Free Software Foundation; either version 2
 of the License, or (at your option) any later version.

 This program is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program; if not, write to the Free Software
 Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA


 The authors can be contacted via:
 www   - http://aig.cs.man.ac.uk
 email - maverik@aig.cs.man.ac.uk
 mail  - Advanced Interfaces Group, Room 2.90, Computer Science Building, 
         University of Manchester, Manchester, M13 9PL, UK
*/


#include "mavlib_kernel.h"
#include <strings.h>
#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#ifdef MAV_LINUX
int gethostname(char *name, size_t len);
#endif
#include <dlfcn.h>

const MAV_vector MAV_NULL_VECTOR={0,0,0};
const MAV_matrix MAV_ID_MATRIX={{{1,0,0,0},{0,1,0,0},{0,0,1,0},{0,0,0,1}}};
const MAV_quaternion MAV_ID_QUATERNION={1,0,0,0};

MAV_window *mav_win_all;
MAV_palette *mav_palette_default;
MAV_class *mav_class_all;
MAV_callback *mav_callback_delete;
MAV_SMSCallback *mav_SMSCallback_delete;
MAV_SMSCallback *mav_SMSCallback_objectRmv;
MAV_list *mav_module_list;
MAV_list *mav_sms_list;
MAV_viewParams mav_vp_default;
int mavlib_usedWin[MAV_MAX_WIN];
float mav_fps;
float mav_fps_avg;
int mav_mallocCount=0;
char mav_hostName[200];
int mav_opt_output= MAV_VERBOSE;
int mav_opt_objectTables= MAV_TRUE;
int mav_opt_WMPlacement= MAV_FALSE;
int mav_opt_singleBuf= MAV_FALSE;
int mav_opt_quadBuf= MAV_FALSE;
int mav_opt_accumBuf= MAV_FALSE;
int mav_opt_multiSample= MAV_DONTCARE;



/* Routines to identify and initalise the kernel */

char *mav_kernelID(void)
{
  return "GNU Maverik v4.3";
}

void mavlib_execInit(void *dlh, char *buf)
{
  MAV_moduleInitFn fn;

  fn= (MAV_moduleInitFn) dlsym(dlh, buf);
  if (fn) 
  {
    fn();
  }
  else
  {
    if (mav_opt_output) fprintf(stderr, "Error: Can not find function %s, exiting\n", buf);
    exit(1);
  }
}

void mav_initialise(void)
{
  FILE *f;
  char buf[100], homepath[200];
  void *dlh;
  int i, defaultInit=0;
  char *defaultMod[]={"mav_gfxModuleInit","mav_callbacksModuleInit","mav_SMSModuleInit","mav_windowsModuleInit","mav_navigationModuleInit","mav_objectsModuleInit",0};

  /* identify the kernel version */
  if (mav_opt_output==MAV_VERBOSE) fprintf(stderr, "%s - Copyright (C) 1999 Advanced Interfaces Group\n", mav_kernelID());

  /* create kernel defined callbacks to delete an object and SMS and remove an object from an SMS */
  mav_callback_delete= mav_callbackNew();
  mav_SMSCallback_delete= mav_SMSCallbackNew();
  mav_SMSCallback_objectRmv= mav_SMSCallbackNew();

  /* create the global window handle */
  mav_win_all= (MAV_window *) mav_malloc(sizeof(MAV_window));
  mav_win_all->id= 0;

  /* create the global class handle */
  mav_class_all= mav_classNew();

  /* create the globla palette handle */
  mav_palette_default= mav_paletteNew();
  mav_windowPaletteSet(mav_win_all, mav_palette_default);

  /* create a list of all windows and SMS's */
  mav_win_list= mav_listNew();
  mav_sms_list= mav_listNew();

  /* create a list frame functions */
  mavlib_frame0_list= mav_listNew();
  mavlib_frame1_list= mav_listNew();
  mavlib_frame2_list= mav_listNew();
  mavlib_frame3_list= mav_listNew();
  mavlib_frame4_list= mav_listNew();
  mavlib_duringFrame_addList= mav_listNew();
  mavlib_duringFrame_rmvList= mav_listNew();

  /* create a list of device functions */
  mavlib_devicePoll_list= mav_listNew();
  mavlib_deviceCalc_list= mav_listNew();
  mavlib_deviceEvent_list= mav_listNew();

  /* create a list of modules */
  mav_module_list= mav_listNew();

  /* default view params */
  mav_vp_default.eye.x=0;
  mav_vp_default.eye.y=0;
  mav_vp_default.eye.z=10;
  mav_vp_default.view.x=0;
  mav_vp_default.view.y=0;
  mav_vp_default.view.z=-1;
  mav_vp_default.up.x=0;
  mav_vp_default.up.y=1;
  mav_vp_default.up.z=0;
  mav_vp_default.fixed_up= mav_vp_default.up;
  mav_vp_default.mod= NULL;

  /* window used tables */
  for (i=1; i<MAV_MAX_WIN; i++) mavlib_usedWin[i]=0;

  /* initialise object lookup tables */
  if (mav_opt_objectTables) 
  {
    mavlib_setUpObjectTables();
  }
  else
  {
    if (mav_opt_output==MAV_VERBOSE) fprintf(stderr, "Warning: Not using object look-up tables.\n");
  }

  /* get hostname */
  gethostname(mav_hostName, 200);

  /* multisample hint  */
  if (mav_opt_multiSample==MAV_DONTCARE) {
    if (!strcmp(mav_hostName, "bigmachine"))
    {
      mav_opt_multiSample= MAV_TRUE;
    }
    else
    {
      mav_opt_multiSample= MAV_FALSE;
    }
  }

  /* find the correct .MavModules file or use default */
  f= fopen(".MavModules", "r");
  if (!f) {
    if (getenv("MAV_HOME"))
    {
      sprintf(homepath, "%s/.MavModules", getenv("MAV_HOME"));
      f= fopen(homepath, "r");
      if (!f) defaultInit=1;
    }
    else
    {
      defaultInit=1;
    }
  }

  /* initalise each module defined */
#ifdef MAV_SUNOS4
  dlh= dlopen(0, 1);
#else
  dlh= dlopen(0, RTLD_NOW);
#endif  
  
  if (defaultInit) 
  {
    i=0;
    while (defaultMod[i]) {
      mavlib_execInit(dlh, defaultMod[i]);
      i++;
    }
  }
  else
  {
    do {
      if (fscanf(f, "%s", buf)>0) mavlib_execInit(dlh, buf);
    } while (!feof(f));
    
    fclose(f);
  }
}



/* Wrappers to system malloc and free */

void *mav_malloc(int msize)
{
  void *result=(void *) malloc(msize);
  
  if (!result) {
    fprintf(stderr, "Error: malloc call failed. Requested size %i\n", msize);
    exit(1);
  } 

  mav_mallocCount++;

  return result;
}

void mav_free(void *mem_ptr) 
{
  mav_mallocCount--;

  free(mem_ptr);
}



/* Routine to perform micro second resolution sleep */

#ifdef MAV_IRIX5
#include <limits.h>
#include <unistd.h>
int sginap(long ticks);
#endif

#ifdef MAV_IRIX6
#include <limits.h>
#include <unistd.h>
long sginap(long ticks);
#endif

#ifdef MAV_LINUX
#include <unistd.h>
void usleep(unsigned long usec);
#endif

void mav_sleep(float len)
{
#ifdef MAV_IRIX5
  sginap((long) (len*CLK_TCK));
#else
#ifdef MAV_IRIX6
  sginap((long) (len*CLK_TCK));
#else
  usleep(len*1.0E+6);
#endif
#endif
}



/* Routine to add new modules and print a list of them */

void mav_moduleNew(MAV_moduleIDFn fn)
{
  mav_listItemAdd(mav_module_list, (void *) fn);
}

void mav_moduleDump(void)
{
  MAV_moduleIDFn fn;

  fprintf(stderr, "%s\n", mav_kernelID());
  mav_listPointerReset(mav_module_list);

  while (mav_listItemNext(mav_module_list, (void **) &fn)) fprintf(stderr, "Module: %s\n", (*fn)());
}



/* Wrappers to kernel defined callbacks */

void mav_callbackDeleteSet(MAV_window *w, MAV_class *c, MAV_callbackDeleteFn fn)
{
  mav_callbackSet(mav_callback_delete, w, c, (MAV_callbackFn) fn);
}

int mav_callbackDeleteExec(MAV_window *w, MAV_object *obj)
{
  return (mav_callbackExec(mav_callback_delete, w, obj, NULL, NULL));
}

void mav_SMSCallbackDeleteSet(MAV_SMSClass *c, MAV_SMSCallbackDeleteFn fn)
{
  mav_SMSCallbackSet(mav_SMSCallback_delete, c, (MAV_SMSCallbackFn) fn);
}

int mav_SMSCallbackDeleteExec(MAV_SMS *s, int o)
{
  return (mav_SMSCallbackExec(mav_SMSCallback_delete, s, (void *) &o, NULL, NULL, NULL));
}

void mav_SMSCallbackObjectRmvSet(MAV_SMSClass *c, MAV_SMSCallbackObjectRmvFn fn)
{
  mav_SMSCallbackSet(mav_SMSCallback_objectRmv, c, (MAV_SMSCallbackFn) fn);
}

int mav_SMSCallbackObjectRmvExec(MAV_SMS *s, MAV_object *obj)
{
  return (mav_SMSCallbackExec(mav_SMSCallback_objectRmv, s, (void *) obj, NULL, NULL, NULL));
}

int mav_SMSObjectRmv(MAV_SMS *s, MAV_object *o)
{
  return mav_SMSCallbackObjectRmvExec(s,o);
}
