/* SPDX-License-Identifier: GPL-2.0-only */
/*
 * Rockchip RK3308 internal audio codec driver -- register definitions
 *
 * Copyright (c) 2018, Fuzhou Rockchip Electronics Co., Ltd All rights reserved.
 * Copyright (c) 2022, Vivax-Metrotech Ltd
 */

#ifndef __RK3308_CODEC_H__
#define __RK3308_CODEC_H__

#define RK3308_GLB_CON				0x00

/* ADC DIGITAL REGISTERS */

/*
 * The ADC group are 0 ~ 3, that control:
 *
 * CH0: left_0(ADC1) and right_0(ADC2)
 * CH1: left_1(ADC3) and right_1(ADC4)
 * CH2: left_2(ADC5) and right_2(ADC6)
 * CH3: left_3(ADC7) and right_3(ADC8)
 */
#define RK3308_ADC_DIG_OFFSET(ch)		(((ch) & 0x3) * 0xc0 + 0x0)

#define RK3308_ADC_DIG_CON01(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x04)
#define RK3308_ADC_DIG_CON02(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x08)
#define RK3308_ADC_DIG_CON03(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x0c)
#define RK3308_ADC_DIG_CON04(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x10)
#define RK3308_ADC_DIG_CON05(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x14) // ver.C only
#define RK3308_ADC_DIG_CON06(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x18) // ver.C only
#define RK3308_ADC_DIG_CON07(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x1c)

#define RK3308_ALC_L_DIG_CON00(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x40)
#define RK3308_ALC_L_DIG_CON01(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x44)
#define RK3308_ALC_L_DIG_CON02(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x48)
#define RK3308_ALC_L_DIG_CON03(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x4c)
#define RK3308_ALC_L_DIG_CON04(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x50)
#define RK3308_ALC_L_DIG_CON05(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x54)
#define RK3308_ALC_L_DIG_CON06(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x58)
#define RK3308_ALC_L_DIG_CON07(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x5c)
#define RK3308_ALC_L_DIG_CON08(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x60)
#define RK3308_ALC_L_DIG_CON09(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x64)
#define RK3308_ALC_L_DIG_CON12(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x70)

#define RK3308_ALC_R_DIG_CON00(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x80)
#define RK3308_ALC_R_DIG_CON01(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x84)
#define RK3308_ALC_R_DIG_CON02(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x88)
#define RK3308_ALC_R_DIG_CON03(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x8c)
#define RK3308_ALC_R_DIG_CON04(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x90)
#define RK3308_ALC_R_DIG_CON05(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x94)
#define RK3308_ALC_R_DIG_CON06(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x98)
#define RK3308_ALC_R_DIG_CON07(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0x9c)
#define RK3308_ALC_R_DIG_CON08(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0xa0)
#define RK3308_ALC_R_DIG_CON09(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0xa4)
#define RK3308_ALC_R_DIG_CON12(ch)		(RK3308_ADC_DIG_OFFSET((ch)) + 0xb0)

/* DAC DIGITAL REGISTERS */
#define RK3308_DAC_DIG_OFFSET			0x300
#define RK3308_DAC_DIG_CON01			(RK3308_DAC_DIG_OFFSET + 0x04)
#define RK3308_DAC_DIG_CON02			(RK3308_DAC_DIG_OFFSET + 0x08)
#define RK3308_DAC_DIG_CON03			(RK3308_DAC_DIG_OFFSET + 0x0c)
#define RK3308_DAC_DIG_CON04			(RK3308_DAC_DIG_OFFSET + 0x10)
#define RK3308_DAC_DIG_CON05			(RK3308_DAC_DIG_OFFSET + 0x14)
#define RK3308_DAC_DIG_CON10			(RK3308_DAC_DIG_OFFSET + 0x28)
#define RK3308_DAC_DIG_CON11			(RK3308_DAC_DIG_OFFSET + 0x2c)
#define RK3308_DAC_DIG_CON13			(RK3308_DAC_DIG_OFFSET + 0x34)
#define RK3308_DAC_DIG_CON14			(RK3308_DAC_DIG_OFFSET + 0x38)

/* ADC ANALOG REGISTERS */
/*
 * The ADC group are 0 ~ 3, that control:
 *
 * CH0: left_0(ADC1) and right_0(ADC2)
 * CH1: left_1(ADC3) and right_1(ADC4)
 * CH2: left_2(ADC5) and right_2(ADC6)
 * CH3: left_3(ADC7) and right_3(ADC8)
 */
#define RK3308_ADC_ANA_OFFSET(ch)		(((ch) & 0x3) * 0x40 + 0x340)
#define RK3308_ADC_ANA_CON00(ch)		(RK3308_ADC_ANA_OFFSET((ch)) + 0x00)
#define RK3308_ADC_ANA_CON01(ch)		(RK3308_ADC_ANA_OFFSET((ch)) + 0x04)
#define RK3308_ADC_ANA_CON02(ch)		(RK3308_ADC_ANA_OFFSET((ch)) + 0x08)
#define RK3308_ADC_ANA_CON03(ch)		(RK3308_ADC_ANA_OFFSET((ch)) + 0x0c)
#define RK3308_ADC_ANA_CON04(ch)		(RK3308_ADC_ANA_OFFSET((ch)) + 0x10)
#define RK3308_ADC_ANA_CON05(ch)		(RK3308_ADC_ANA_OFFSET((ch)) + 0x14)
#define RK3308_ADC_ANA_CON06(ch)		(RK3308_ADC_ANA_OFFSET((ch)) + 0x18)
#define RK3308_ADC_ANA_CON07(ch)		(RK3308_ADC_ANA_OFFSET((ch)) + 0x1c)
#define RK3308_ADC_ANA_CON08(ch)		(RK3308_ADC_ANA_OFFSET((ch)) + 0x20)
#define RK3308_ADC_ANA_CON10(ch)		(RK3308_ADC_ANA_OFFSET((ch)) + 0x28)
#define RK3308_ADC_ANA_CON11(ch)		(RK3308_ADC_ANA_OFFSET((ch)) + 0x2c)

/* DAC ANALOG REGISTERS */
#define RK3308_DAC_ANA_OFFSET			0x440
#define RK3308_DAC_ANA_CON00			(RK3308_DAC_ANA_OFFSET + 0x00)
#define RK3308_DAC_ANA_CON01			(RK3308_DAC_ANA_OFFSET + 0x04)
#define RK3308_DAC_ANA_CON02			(RK3308_DAC_ANA_OFFSET + 0x08)
#define RK3308_DAC_ANA_CON03			(RK3308_DAC_ANA_OFFSET + 0x0c)
#define RK3308_DAC_ANA_CON04			(RK3308_DAC_ANA_OFFSET + 0x10)
#define RK3308_DAC_ANA_CON05			(RK3308_DAC_ANA_OFFSET + 0x14)
#define RK3308_DAC_ANA_CON06			(RK3308_DAC_ANA_OFFSET + 0x18)
#define RK3308_DAC_ANA_CON07			(RK3308_DAC_ANA_OFFSET + 0x1c)
#define RK3308_DAC_ANA_CON08			(RK3308_DAC_ANA_OFFSET + 0x20)
#define RK3308_DAC_ANA_CON12			(RK3308_DAC_ANA_OFFSET + 0x30)
#define RK3308_DAC_ANA_CON13			(RK3308_DAC_ANA_OFFSET + 0x34)
#define RK3308_DAC_ANA_CON14			(RK3308_DAC_ANA_OFFSET + 0x38)
#define RK3308_DAC_ANA_CON15			(RK3308_DAC_ANA_OFFSET + 0x3c)

/*
 * These are the bits for registers
 */

/* RK3308_GLB_CON - REG: 0x0000 */
#define RK3308_ADC_BIST_WORK			BIT(7)
#define RK3308_DAC_BIST_WORK			BIT(6)
#define RK3308_ADC_MCLK_GATING			BIT(5)
#define RK3308_DAC_MCLK_GATING			BIT(4)
#define RK3308_ADC_DIG_WORK			BIT(2)
#define RK3308_DAC_DIG_WORK			BIT(1)
#define RK3308_SYS_WORK				BIT(0)

/* RK3308_ADC_DIG_CON01 - REG: 0x0004 */
#define RK3308_ADC_I2S_LRC_POL_REVERSAL		BIT(7)
#define RK3308_ADC_I2S_VALID_LEN_SFT		5
#define RK3308_ADC_I2S_VALID_LEN_MSK		(0x3 << RK3308_ADC_I2S_VALID_LEN_SFT)
#define RK3308_ADC_I2S_VALID_LEN_32BITS		(0x3 << RK3308_ADC_I2S_VALID_LEN_SFT)
#define RK3308_ADC_I2S_VALID_LEN_24BITS		(0x2 << RK3308_ADC_I2S_VALID_LEN_SFT)
#define RK3308_ADC_I2S_VALID_LEN_20BITS		(0x1 << RK3308_ADC_I2S_VALID_LEN_SFT)
#define RK3308_ADC_I2S_VALID_LEN_16BITS		(0x0 << RK3308_ADC_I2S_VALID_LEN_SFT)
#define RK3308_ADC_I2S_MODE_SFT			3
#define RK3308_ADC_I2S_MODE_MSK			(0x3 << RK3308_ADC_I2S_MODE_SFT)
#define RK3308_ADC_I2S_MODE_PCM			(0x3 << RK3308_ADC_I2S_MODE_SFT)
#define RK3308_ADC_I2S_MODE_I2S			(0x2 << RK3308_ADC_I2S_MODE_SFT)
#define RK3308_ADC_I2S_MODE_LJ			(0x1 << RK3308_ADC_I2S_MODE_SFT)
#define RK3308_ADC_I2S_MODE_RJ			(0x0 << RK3308_ADC_I2S_MODE_SFT)
#define RK3308_ADC_I2S_LR_SWAP			BIT(1)
#define RK3308_ADC_I2S_MONO			BIT(0)

/* RK3308_ADC_DIG_CON02 - REG: 0x0008 */
#define RK3308_ADC_IO_MODE_MASTER		BIT(5)
#define RK3308_ADC_MODE_MASTER			BIT(4)
#define RK3308_ADC_I2S_FRAME_LEN_SFT		2
#define RK3308_ADC_I2S_FRAME_LEN_MSK		(0x3 << RK3308_ADC_I2S_FRAME_LEN_SFT)
#define RK3308_ADC_I2S_FRAME_32BITS		(0x3 << RK3308_ADC_I2S_FRAME_LEN_SFT)
#define RK3308_ADC_I2S_FRAME_24BITS		(0x2 << RK3308_ADC_I2S_FRAME_LEN_SFT)
#define RK3308_ADC_I2S_FRAME_20BITS		(0x1 << RK3308_ADC_I2S_FRAME_LEN_SFT)
#define RK3308_ADC_I2S_FRAME_16BITS		(0x0 << RK3308_ADC_I2S_FRAME_LEN_SFT)
#define RK3308_ADC_I2S_WORK			BIT(1)
#define RK3308_ADC_I2S_BIT_CLK_POL_REVERSAL	BIT(0)

/* RK3308_ADC_DIG_CON03 - REG: 0x000c */
#define RK3308_ADC_L_CH_BIST_SFT		2
#define RK3308_ADC_L_CH_BIST_MSK		(0x3 << RK3308_ADC_L_CH_BIST_SFT)
#define RK3308_ADC_L_CH_NORMAL_RIGHT		(0x3 << RK3308_ADC_L_CH_BIST_SFT) /* normal mode */
#define RK3308_ADC_L_CH_BIST_CUBE		(0x2 << RK3308_ADC_L_CH_BIST_SFT)
#define RK3308_ADC_L_CH_BIST_SINE		(0x1 << RK3308_ADC_L_CH_BIST_SFT)
#define RK3308_ADC_L_CH_NORMAL_LEFT		(0x0 << RK3308_ADC_L_CH_BIST_SFT) /* normal mode */
#define RK3308_ADC_R_CH_BIST_SFT		0
#define RK3308_ADC_R_CH_BIST_MSK		(0x3 << RK3308_ADC_R_CH_BIST_SFT)
#define RK3308_ADC_R_CH_NORMAL_LEFT		(0x3 << RK3308_ADC_R_CH_BIST_SFT) /* normal mode */
#define RK3308_ADC_R_CH_BIST_CUBE		(0x2 << RK3308_ADC_R_CH_BIST_SFT)
#define RK3308_ADC_R_CH_BIST_SINE		(0x1 << RK3308_ADC_R_CH_BIST_SFT)
#define RK3308_ADC_R_CH_NORMAL_RIGHT		(0x0 << RK3308_ADC_R_CH_BIST_SFT) /* normal mode */

/* RK3308_ADC_DIG_CON04 - REG: 0x0010 */
#define RK3308_ADC_HPF_PATH_DIS			BIT(2)
#define RK3308_ADC_HPF_CUTOFF_SFT		0
#define RK3308_ADC_HPF_CUTOFF_MSK		(0x3 << RK3308_ADC_HPF_CUTOFF_SFT)
#define RK3308_ADC_HPF_CUTOFF_612HZ		(0x2 << RK3308_ADC_HPF_CUTOFF_SFT)
#define RK3308_ADC_HPF_CUTOFF_245HZ		(0x1 << RK3308_ADC_HPF_CUTOFF_SFT)
#define RK3308_ADC_HPF_CUTOFF_20HZ		(0x0 << RK3308_ADC_HPF_CUTOFF_SFT)

/* RK3308_ADC_DIG_CON07 - REG: 0x001c */
#define RK3308_ADCL_DATA_SFT			4
#define RK3308_ADCR_DATA_SFT			2
#define RK3308_ADCL_DATA_SEL_ADCL		BIT(1)
#define RK3308_ADCR_DATA_SEL_ADCR		BIT(0)

/*
 * RK3308_ALC_L_DIG_CON00 - REG: 0x0040 + ch * 0xc0
 * RK3308_ALC_R_DIG_CON00 - REG: 0x0080 + ch * 0xc0
 */
#define RK3308_GAIN_ATTACK_JACK			BIT(6)
#define RK3308_CTRL_GEN_SFT			4
#define RK3308_CTRL_GEN_MSK			(0x3 << RK3308_ALC_CTRL_GEN_SFT)
#define RK3308_CTRL_GEN_JACK3			(0x3 << RK3308_ALC_CTRL_GEN_SFT)
#define RK3308_CTRL_GEN_JACK2			(0x2 << RK3308_ALC_CTRL_GEN_SFT)
#define RK3308_CTRL_GEN_JACK1			(0x1 << RK3308_ALC_CTRL_GEN_SFT)
#define RK3308_CTRL_GEN_NORMAL			(0x0 << RK3308_ALC_CTRL_GEN_SFT)
#define RK3308_AGC_HOLD_TIME_SFT		0
#define RK3308_AGC_HOLD_TIME_MSK		(0xf << RK3308_AGC_HOLD_TIME_SFT)
#define RK3308_AGC_HOLD_TIME_1S			(0xa << RK3308_AGC_HOLD_TIME_SFT)
#define RK3308_AGC_HOLD_TIME_512MS		(0x9 << RK3308_AGC_HOLD_TIME_SFT)
#define RK3308_AGC_HOLD_TIME_256MS		(0x8 << RK3308_AGC_HOLD_TIME_SFT)
#define RK3308_AGC_HOLD_TIME_128MS		(0x7 << RK3308_AGC_HOLD_TIME_SFT)
#define RK3308_AGC_HOLD_TIME_64MS		(0x6 << RK3308_AGC_HOLD_TIME_SFT)
#define RK3308_AGC_HOLD_TIME_32MS		(0x5 << RK3308_AGC_HOLD_TIME_SFT)
#define RK3308_AGC_HOLD_TIME_16MS		(0x4 << RK3308_AGC_HOLD_TIME_SFT)
#define RK3308_AGC_HOLD_TIME_8MS		(0x3 << RK3308_AGC_HOLD_TIME_SFT)
#define RK3308_AGC_HOLD_TIME_4MS		(0x2 << RK3308_AGC_HOLD_TIME_SFT)
#define RK3308_AGC_HOLD_TIME_2MS		(0x1 << RK3308_AGC_HOLD_TIME_SFT)
#define RK3308_AGC_HOLD_TIME_0MS		(0x0 << RK3308_AGC_HOLD_TIME_SFT)

/*
 * RK3308_ALC_L_DIG_CON01 - REG: 0x0044 + ch * 0xc0
 * RK3308_ALC_R_DIG_CON01 - REG: 0x0084 + ch * 0xc0
 */
#define RK3308_AGC_DECAY_TIME_SFT		4
#define RK3308_AGC_ATTACK_TIME_SFT		0

/*
 * RK3308_ALC_L_DIG_CON02 - REG: 0x0048 + ch * 0xc0
 * RK3308_ALC_R_DIG_CON02 - REG: 0x0088 + ch * 0xc0
 */
#define RK3308_AGC_MODE_LIMITER			BIT(7)
#define RK3308_AGC_ZERO_CRO_EN			BIT(6)
#define RK3308_AGC_AMP_RECOVER_GAIN		BIT(5)
#define RK3308_AGC_FAST_DEC_EN			BIT(4)
#define RK3308_AGC_NOISE_GATE_EN		BIT(3)
#define RK3308_AGC_NOISE_GATE_THRESH_SFT	0
#define RK3308_AGC_NOISE_GATE_THRESH_MSK	(0x7 << RK3308_AGC_NOISE_GATE_THRESH_SFT)

/*
 * RK3308_ALC_L_DIG_CON03 - REG: 0x004c + ch * 0xc0
 * RK3308_ALC_R_DIG_CON03 - REG: 0x008c + ch * 0xc0
 */
#define RK3308_AGC_PGA_ZERO_CRO_EN		BIT(5)
#define RK3308_AGC_PGA_GAIN_MAX			0x1f
#define RK3308_AGC_PGA_GAIN_MIN			0
#define RK3308_AGC_PGA_GAIN_SFT			0

/*
 * RK3308_ALC_L_DIG_CON04 - REG: 0x0050 + ch * 0xc0
 * RK3308_ALC_R_DIG_CON04 - REG: 0x0090 + ch * 0xc0
 */
#define RK3308_AGC_SLOW_CLK_EN			BIT(3)
#define RK3308_AGC_APPROX_RATE_SFT		0
#define RK3308_AGC_APPROX_RATE_MSK		(0x7 << RK3308_AGC_APPROX_RATE_SFT)

/*
 * RK3308_ALC_L_DIG_CON05 - REG: 0x0054 + ch * 0xc0
 * RK3308_ALC_R_DIG_CON05 - REG: 0x0094 + ch * 0xc0
 */
#define RK3308_AGC_LO_8BITS_AGC_MAX_MSK		0xff

/*
 * RK3308_ALC_L_DIG_CON06 - REG: 0x0058 + ch * 0xc0
 * RK3308_ALC_R_DIG_CON06 - REG: 0x0098 + ch * 0xc0
 */
#define RK3308_AGC_HI_8BITS_AGC_MAX_MSK		0xff

/*
 * RK3308_ALC_L_DIG_CON07 - REG: 0x005c + ch * 0xc0
 * RK3308_ALC_R_DIG_CON07 - REG: 0x009c + ch * 0xc0
 */
#define RK3308_AGC_LO_8BITS_AGC_MIN_MSK		0xff

/*
 * RK3308_ALC_L_DIG_CON08 - REG: 0x0060 + ch * 0xc0
 * RK3308_ALC_R_DIG_CON08 - REG: 0x00a0 + ch * 0xc0
 */
#define RK3308_AGC_HI_8BITS_AGC_MIN_MSK		0xff

/*
 * RK3308_ALC_L_DIG_CON09 - REG: 0x0064 + ch * 0xc0
 * RK3308_ALC_R_DIG_CON09 - REG: 0x00a4 + ch * 0xc0
 */
#define RK3308_AGC_FUNC_SEL			BIT(6)
#define RK3308_AGC_MAX_GAIN_PGA_MAX		0x7
#define RK3308_AGC_MAX_GAIN_PGA_MIN		0
#define RK3308_AGC_MAX_GAIN_PGA_SFT		3
#define RK3308_AGC_MAX_GAIN_PGA_MSK		(0x7 << RK3308_AGC_MAX_GAIN_PGA_SFT)
#define RK3308_AGC_MIN_GAIN_PGA_MAX		0x7
#define RK3308_AGC_MIN_GAIN_PGA_MIN		0
#define RK3308_AGC_MIN_GAIN_PGA_SFT		0
#define RK3308_AGC_MIN_GAIN_PGA_MSK		(0x7 << RK3308_AGC_MIN_GAIN_PGA_SFT)

/*
 * RK3308_ALC_L_DIG_CON12 - REG: 0x0068 + ch * 0xc0
 * RK3308_ALC_R_DIG_CON12 - REG: 0x00a8 + ch * 0xc0
 */
#define RK3308_AGC_GAIN_MSK			0x1f

/* RK3308_DAC_DIG_CON01 - REG: 0x0304 */
#define RK3308_DAC_I2S_LRC_POL_REVERSAL		BIT(7)
#define RK3308_DAC_I2S_VALID_LEN_SFT		5
#define RK3308_DAC_I2S_VALID_LEN_MSK		(0x3 << RK3308_DAC_I2S_VALID_LEN_SFT)
#define RK3308_DAC_I2S_VALID_LEN_32BITS		(0x3 << RK3308_DAC_I2S_VALID_LEN_SFT)
#define RK3308_DAC_I2S_VALID_LEN_24BITS		(0x2 << RK3308_DAC_I2S_VALID_LEN_SFT)
#define RK3308_DAC_I2S_VALID_LEN_20BITS		(0x1 << RK3308_DAC_I2S_VALID_LEN_SFT)
#define RK3308_DAC_I2S_VALID_LEN_16BITS		(0x0 << RK3308_DAC_I2S_VALID_LEN_SFT)
#define RK3308_DAC_I2S_MODE_SFT			3
#define RK3308_DAC_I2S_MODE_MSK			(0x3 << RK3308_DAC_I2S_MODE_SFT)
#define RK3308_DAC_I2S_MODE_PCM			(0x3 << RK3308_DAC_I2S_MODE_SFT)
#define RK3308_DAC_I2S_MODE_I2S			(0x2 << RK3308_DAC_I2S_MODE_SFT)
#define RK3308_DAC_I2S_MODE_LJ			(0x1 << RK3308_DAC_I2S_MODE_SFT)
#define RK3308_DAC_I2S_MODE_RJ			(0x0 << RK3308_DAC_I2S_MODE_SFT)
#define RK3308_DAC_I2S_LR_SWAP			BIT(2)

/* RK3308_DAC_DIG_CON02 - REG: 0x0308 */
#define RK3308BS_DAC_IO_MODE_MASTER		BIT(7)
#define RK3308BS_DAC_MODE_MASTER		BIT(6)
#define RK3308_DAC_IO_MODE_MASTER		BIT(5)
#define RK3308_DAC_MODE_MASTER			BIT(4)
#define RK3308_DAC_I2S_FRAME_LEN_SFT		2
#define RK3308_DAC_I2S_FRAME_LEN_MSK		(0x3 << RK3308_DAC_I2S_FRAME_LEN_SFT)
#define RK3308_DAC_I2S_FRAME_32BITS		(0x3 << RK3308_DAC_I2S_FRAME_LEN_SFT)
#define RK3308_DAC_I2S_FRAME_24BITS		(0x2 << RK3308_DAC_I2S_FRAME_LEN_SFT)
#define RK3308_DAC_I2S_FRAME_20BITS		(0x1 << RK3308_DAC_I2S_FRAME_LEN_SFT)
#define RK3308_DAC_I2S_FRAME_16BITS		(0x0 << RK3308_DAC_I2S_FRAME_LEN_SFT)
#define RK3308_DAC_I2S_WORK			BIT(1)
#define RK3308_DAC_I2S_BIT_CLK_POL_REVERSAL	BIT(0)

/* RK3308_DAC_DIG_CON03 - REG: 0x030C */
#define RK3308_DAC_L_CH_BIST_SFT		2
#define RK3308_DAC_L_CH_BIST_MSK		(0x3 << RK3308_DAC_L_CH_BIST_SFT)
#define RK3308_DAC_L_CH_BIST_LEFT		(0x3 << RK3308_DAC_L_CH_BIST_SFT) /* normal mode */
#define RK3308_DAC_L_CH_BIST_CUBE		(0x2 << RK3308_DAC_L_CH_BIST_SFT)
#define RK3308_DAC_L_CH_BIST_SINE		(0x1 << RK3308_DAC_L_CH_BIST_SFT)
#define RK3308_DAC_L_CH_BIST_RIGHT		(0x0 << RK3308_DAC_L_CH_BIST_SFT) /* normal mode */
#define RK3308_DAC_R_CH_BIST_SFT		0
#define RK3308_DAC_R_CH_BIST_MSK		(0x3 << RK3308_DAC_R_CH_BIST_SFT)
#define RK3308_DAC_R_CH_BIST_LEFT		(0x3 << RK3308_DAC_R_CH_BIST_SFT) /* normal mode */
#define RK3308_DAC_R_CH_BIST_CUBE		(0x2 << RK3308_DAC_R_CH_BIST_SFT)
#define RK3308_DAC_R_CH_BIST_SINE		(0x1 << RK3308_DAC_R_CH_BIST_SFT)
#define RK3308_DAC_R_CH_BIST_RIGHT		(0x0 << RK3308_DAC_R_CH_BIST_SFT) /* normal mode */

/* RK3308_DAC_DIG_CON04 - REG: 0x0310 */
/* Versions up to B: */
#define RK3308_DAC_MODULATOR_GAIN_SFT		4
#define RK3308_DAC_MODULATOR_GAIN_MSK		(0x7 << RK3308_DAC_MODULATOR_GAIN_SFT)
#define RK3308_DAC_CIC_IF_GAIN_SFT		0
#define RK3308_DAC_CIC_IF_GAIN_MSK		(0x7 << RK3308_DAC_CIC_IF_GAIN_SFT)
/* Version C: */
#define RK3308BS_DAC_DIG_GAIN_SFT		0
#define RK3308BS_DAC_DIG_GAIN_MSK		(0xff << RK3308BS_DAC_DIG_GAIN_SFT)
#define RK3308BS_DAC_DIG_GAIN_0DB		(0xed << RK3308BS_DAC_DIG_GAIN_SFT)

/* RK3308BS_ADC_DIG_CON05..06 (Version C only) */
#define RK3308_ADC_DIG_VOL_CON_x_SFT		0
#define RK3308_ADC_DIG_VOL_CON_x_MSK		(0xff << RK3308_ADC_DIG_VOL_CON_x_SFT)
#define RK3308_ADC_DIG_VOL_CON_x_0DB		(0xc2 << RK3308_ADC_DIG_VOL_CON_x_SFT)

/* RK3308_DAC_DIG_CON05 - REG: 0x0314 */
#define RK3308_DAC_L_REG_CTL_INDATA		BIT(2)
#define RK3308_DAC_R_REG_CTL_INDATA		BIT(1)

/* RK3308_DAC_DIG_CON10 - REG: 0x0328 */
#define RK3308_DAC_DATA_HI4(x)			((x) & 0xf)

/* RK3308_DAC_DIG_CON11 - REG: 0x032c */
#define RK3308_DAC_DATA_LO8(x)			((x) & 0xff)

/* RK3308_ADC_ANA_CON00 - REG: 0x0340 */
#define RK3308_ADC_CH1_CH2_MIC_ALL_MSK		(0xff << 0)
#define RK3308_ADC_CH1_CH2_MIC_ALL		0xff
#define RK3308_ADC_CH2_MIC_UNMUTE		BIT(7)
#define RK3308_ADC_CH2_MIC_WORK			BIT(6)
#define RK3308_ADC_CH2_MIC_EN			BIT(5)
#define RK3308_ADC_CH2_BUF_REF_EN		BIT(4)
#define RK3308_ADC_CH1_MIC_UNMUTE		BIT(3)
#define RK3308_ADC_CH1_MIC_WORK			BIT(2)
#define RK3308_ADC_CH1_MIC_EN			BIT(1)
#define RK3308_ADC_CH1_BUF_REF_EN		BIT(0)

/* RK3308_ADC_ANA_CON01 - REG: 0x0344
 *
 * The PGA of MIC-INs:
 * - HW version A:
 *   0x0 - MIC1~MIC8  0 dB (recommended when ADC used as loopback)
 *   0x3 - MIC1~MIC8 20 dB (recommended when ADC used as MIC input)
 * - HW version B:
 *   0x0 - MIC1~MIC8   0 dB
 *   0x1 - MIC1~MIC8 6.6 dB
 *   0x2 - MIC1~MIC8  13 dB
 *   0x3 - MIC1~MIC8  20 dB
 */
#define RK3308_ADC_CH2_MIC_GAIN_MAX		0x3
#define RK3308_ADC_CH2_MIC_GAIN_MIN		0
#define RK3308_ADC_CH2_MIC_GAIN_SFT		4
#define RK3308_ADC_CH2_MIC_GAIN_MSK		(0x3 << RK3308_ADC_CH2_MIC_GAIN_SFT)
#define RK3308_ADC_CH2_MIC_GAIN_20DB		(0x3 << RK3308_ADC_CH2_MIC_GAIN_SFT)
#define RK3308_ADC_CH2_MIC_GAIN_13DB		(0x2 << RK3308_ADC_CH2_MIC_GAIN_SFT)
#define RK3308_ADC_CH2_MIC_GAIN_6_6DB		(0x1 << RK3308_ADC_CH2_MIC_GAIN_SFT)
#define RK3308_ADC_CH2_MIC_GAIN_0DB		(0x0 << RK3308_ADC_CH2_MIC_GAIN_SFT)

#define RK3308_ADC_CH1_MIC_GAIN_MAX		0x3
#define RK3308_ADC_CH1_MIC_GAIN_MIN		0
#define RK3308_ADC_CH1_MIC_GAIN_SFT		0
#define RK3308_ADC_CH1_MIC_GAIN_MSK		(0x3 << RK3308_ADC_CH1_MIC_GAIN_SFT)
#define RK3308_ADC_CH1_MIC_GAIN_20DB		(0x3 << RK3308_ADC_CH1_MIC_GAIN_SFT)
#define RK3308_ADC_CH1_MIC_GAIN_13DB		(0x2 << RK3308_ADC_CH1_MIC_GAIN_SFT)
#define RK3308_ADC_CH1_MIC_GAIN_6_6DB		(0x1 << RK3308_ADC_CH1_MIC_GAIN_SFT)
#define RK3308_ADC_CH1_MIC_GAIN_0DB		(0x0 << RK3308_ADC_CH1_MIC_GAIN_SFT)

/* RK3308_ADC_ANA_CON02 - REG: 0x0348 */
#define RK3308_ADC_CH2_ZEROCROSS_DET_EN		BIT(6)
#define RK3308_ADC_CH2_ALC_WORK			BIT(5)
#define RK3308_ADC_CH2_ALC_EN			BIT(4)
#define RK3308_ADC_CH1_ZEROCROSS_DET_EN		BIT(2)
#define RK3308_ADC_CH1_ALC_WORK			BIT(1)
#define RK3308_ADC_CH1_ALC_EN			BIT(0)

/* RK3308_ADC_ANA_CON03 - REG: 0x034c */
#define RK3308_ADC_CH1_ALC_GAIN_MAX		0x1f
#define RK3308_ADC_CH1_ALC_GAIN_MIN		0
#define RK3308_ADC_CH1_ALC_GAIN_SFT		0
#define RK3308_ADC_CH1_ALC_GAIN_MSK		(0x1f << RK3308_ADC_CH1_ALC_GAIN_SFT)
#define RK3308_ADC_CH1_ALC_GAIN_0DB		(0x0c << RK3308_ADC_CH1_ALC_GAIN_SFT)

/* RK3308_ADC_ANA_CON04 - REG: 0x0350 */
#define RK3308_ADC_CH2_ALC_GAIN_MAX		0x1f
#define RK3308_ADC_CH2_ALC_GAIN_MIN		0
#define RK3308_ADC_CH2_ALC_GAIN_SFT		0
#define RK3308_ADC_CH2_ALC_GAIN_MSK		(0x1f << RK3308_ADC_CH2_ALC_GAIN_SFT)
#define RK3308_ADC_CH2_ALC_GAIN_0DB		(0x0c << RK3308_ADC_CH2_ALC_GAIN_SFT)

/* RK3308_ADC_ANA_CON05 - REG: 0x0354 */
#define RK3308_ADC_CH2_ADC_WORK			BIT(6)
#define RK3308_ADC_CH2_ADC_EN			BIT(5)
#define RK3308_ADC_CH2_CLK_EN			BIT(4)
#define RK3308_ADC_CH1_ADC_WORK			BIT(2)
#define RK3308_ADC_CH1_ADC_EN			BIT(1)
#define RK3308_ADC_CH1_CLK_EN			BIT(0)

/* RK3308_ADC_ANA_CON06 - REG: 0x0358 */
#define RK3308_ADC_CURRENT_EN			BIT(0)

/* RK3308_ADC_ANA_CON07 - REG: 0x035c */
/* Note: The register configuration is only valid for ADC2 */
#define RK3308_ADC_CH2_IN_SEL_SFT		6
#define RK3308_ADC_CH2_IN_SEL_MSK		(0x3 << RK3308_ADC_CH2_IN_SEL_SFT)
#define RK3308_ADC_CH2_IN_LINEIN_MIC		(0x3 << RK3308_ADC_CH2_IN_SEL_SFT)
#define RK3308_ADC_CH2_IN_LINEIN		(0x2 << RK3308_ADC_CH2_IN_SEL_SFT)
#define RK3308_ADC_CH2_IN_MIC			(0x1 << RK3308_ADC_CH2_IN_SEL_SFT)
#define RK3308_ADC_CH2_IN_NONE			(0x0 << RK3308_ADC_CH2_IN_SEL_SFT)
/* Note: The register configuration is only valid for ADC1 */
#define RK3308_ADC_CH1_IN_SEL_SFT		4
#define RK3308_ADC_CH1_IN_SEL_MSK		(0x3 << RK3308_ADC_CH1_IN_SEL_SFT)
#define RK3308_ADC_CH1_IN_LINEIN_MIC		(0x3 << RK3308_ADC_CH1_IN_SEL_SFT)
#define RK3308_ADC_CH1_IN_LINEIN		(0x2 << RK3308_ADC_CH1_IN_SEL_SFT)
#define RK3308_ADC_CH1_IN_MIC			(0x1 << RK3308_ADC_CH1_IN_SEL_SFT)
#define RK3308_ADC_CH1_IN_NONE			(0x0 << RK3308_ADC_CH1_IN_SEL_SFT)
#define RK3308_ADC_MIC_BIAS_BUF_EN		BIT(3)
#define RK3308_ADC_LEVEL_RANGE_MICBIAS_MAX	7
#define RK3308_ADC_LEVEL_RANGE_MICBIAS_SFT	0
#define RK3308_ADC_LEVEL_RANGE_MICBIAS_MSK	(0x7 << RK3308_ADC_LEVEL_RANGE_MICBIAS_SFT)

/* RK3308_ADC_ANA_CON08 - REG: 0x0360 */
#define RK3308_ADC_MICBIAS_CURRENT_EN		BIT(4)

/* RK3308_ADC_ANA_CON10 - REG: 0x0368 */
#define RK3308_ADC_REF_EN			BIT(7)
#define RK3308_ADC_CURRENT_CHARGE_SFT		0
#define RK3308_ADC_CURRENT_CHARGE_MSK		(0x7f << RK3308_ADC_CURRENT_CHARGE_SFT)

/* RK3308_ADC_ANA_CON11 - REG: 0x036c */
#define RK3308_ADC_ALCR_CON_GAIN_PGAR_EN	BIT(1)
#define RK3308_ADC_ALCL_CON_GAIN_PGAL_EN	BIT(0)

/* RK3308_DAC_ANA_CON00 - REG: 0x0440 */
#define RK3308_DAC_HEADPHONE_DET_EN		BIT(1)
#define RK3308_DAC_CURRENT_EN			BIT(0)

/* RK3308_DAC_ANA_CON01 - REG: 0x0444 */
#define RK3308_DAC_BUF_REF_R_EN			BIT(6)
#define RK3308_DAC_BUF_REF_L_EN			BIT(2)
#define RK3308_DAC_HPOUT_POP_SOUND_R_SFT	4
#define RK3308_DAC_HPOUT_POP_SOUND_L_SFT	0
// unshifted values for both L and R:
#define RK3308_DAC_HPOUT_POP_SOUND_x_MSK	0x3
#define RK3308_DAC_HPOUT_POP_SOUND_x_WORK	0x2
#define RK3308_DAC_HPOUT_POP_SOUND_x_INIT	0x1

/* RK3308_DAC_ANA_CON02 - REG: 0x0448 */
#define RK3308_DAC_R_DAC_WORK			BIT(7)
#define RK3308_DAC_R_DAC_EN			BIT(6)
#define RK3308_DAC_R_CLK_EN			BIT(5)
#define RK3308_DAC_R_REF_EN			BIT(4)
#define RK3308_DAC_L_DAC_WORK			BIT(3)
#define RK3308_DAC_L_DAC_EN			BIT(2)
#define RK3308_DAC_L_CLK_EN			BIT(1)
#define RK3308_DAC_L_REF_EN			BIT(0)

/* RK3308_DAC_ANA_CON03 - REG: 0x044c */
#define RK3308_DAC_R_HPOUT_WORK			BIT(6)
#define RK3308_DAC_R_HPOUT_EN			BIT(5)
#define RK3308_DAC_R_HPOUT_MUTE_SFT		4
#define RK3308_DAC_L_HPOUT_WORK			BIT(2)
#define RK3308_DAC_L_HPOUT_EN			BIT(1)
#define RK3308_DAC_L_HPOUT_MUTE_SFT		0

/* RK3308_DAC_ANA_CON04 - REG: 0x0450 */
#define RK3308_DAC_x_LINEOUT_GAIN_MAX		0x3
#define RK3308_DAC_R_LINEOUT_GAIN_SFT		6
#define RK3308_DAC_R_LINEOUT_GAIN_MSK		(0x3 << RK3308_DAC_R_LINEOUT_GAIN_SFT)
#define RK3308_DAC_R_LINEOUT_GAIN_0DB		(0x3 << RK3308_DAC_R_LINEOUT_GAIN_SFT)
#define RK3308_DAC_R_LINEOUT_GAIN_NDB_1_5	(0x2 << RK3308_DAC_R_LINEOUT_GAIN_SFT)
#define RK3308_DAC_R_LINEOUT_GAIN_NDB_3		(0x1 << RK3308_DAC_R_LINEOUT_GAIN_SFT)
#define RK3308_DAC_R_LINEOUT_GAIN_NDB_6		(0x0 << RK3308_DAC_R_LINEOUT_GAIN_SFT)
#define RK3308_DAC_R_LINEOUT_MUTE_SFT		5
#define RK3308_DAC_R_LINEOUT_EN			BIT(4)
#define RK3308_DAC_L_LINEOUT_GAIN_SFT		2
#define RK3308_DAC_L_LINEOUT_GAIN_MSK		(0x3 << RK3308_DAC_L_LINEOUT_GAIN_SFT)
#define RK3308_DAC_L_LINEOUT_GAIN_0DB		(0x3 << RK3308_DAC_L_LINEOUT_GAIN_SFT)
#define RK3308_DAC_L_LINEOUT_GAIN_NDB_1_5	(0x2 << RK3308_DAC_L_LINEOUT_GAIN_SFT)
#define RK3308_DAC_L_LINEOUT_GAIN_NDB_3		(0x1 << RK3308_DAC_L_LINEOUT_GAIN_SFT)
#define RK3308_DAC_L_LINEOUT_GAIN_NDB_6		(0x0 << RK3308_DAC_L_LINEOUT_GAIN_SFT)
#define RK3308_DAC_L_LINEOUT_MUTE_SFT		1
#define RK3308_DAC_L_LINEOUT_EN			BIT(0)

/* RK3308_DAC_ANA_CON05 - REG: 0x0454, step is 1.5db */
/* RK3308_DAC_ANA_CON06 - REG: 0x0458, step is 1.5db */
#define RK3308_DAC_x_HPOUT_GAIN_MAX		0x1e
#define RK3308_DAC_x_HPOUT_GAIN_SFT		0
#define RK3308_DAC_x_HPOUT_GAIN_MSK		(0x1f << RK3308_DAC_x_HPOUT_GAIN_SFT)
#define RK3308_DAC_x_HPOUT_GAIN_MIN		(0x00 << RK3308_DAC_x_HPOUT_GAIN_SFT)

/* RK3308_DAC_ANA_CON07 - REG: 0x045c */
#define RK3308_DAC_R_HPOUT_DRV_SFT		4
#define RK3308_DAC_R_HPOUT_DRV_MSK		(0xf << RK3308_DAC_R_HPOUT_DRV_SFT)
#define RK3308_DAC_L_HPOUT_DRV_SFT		0
#define RK3308_DAC_L_HPOUT_DRV_MSK		(0xf << RK3308_DAC_L_HPOUT_DRV_SFT)

/* RK3308_DAC_ANA_CON08 - REG: 0x0460 */
#define RK3308_DAC_R_LINEOUT_DRV_SFT		4
#define RK3308_DAC_R_LINEOUT_DRV_MSK		(0xf << RK3308_DAC_R_LINEOUT_DRV_SFT)
#define RK3308_DAC_L_LINEOUT_DRV_SFT		0
#define RK3308_DAC_L_LINEOUT_DRV_MSK		(0xf << RK3308_DAC_L_LINEOUT_DRV_SFT)

/* RK3308_DAC_ANA_CON12 - REG: 0x0470 */
#define RK3308_DAC_R_HPMIX_SEL_SFT		6
#define RK3308_DAC_R_HPMIX_SEL_MSK		(0x3 << RK3308_DAC_R_HPMIX_SEL_SFT)
#define RK3308_DAC_R_HPMIX_LINEIN_I2S		(0x3 << RK3308_DAC_R_HPMIX_SEL_SFT)
#define RK3308_DAC_R_HPMIX_LINEIN		(0x2 << RK3308_DAC_R_HPMIX_SEL_SFT)
#define RK3308_DAC_R_HPMIX_I2S			(0x1 << RK3308_DAC_R_HPMIX_SEL_SFT)
#define RK3308_DAC_R_HPMIX_NONE			(0x0 << RK3308_DAC_R_HPMIX_SEL_SFT)
#define RK3308_DAC_L_HPMIX_SEL_SFT		2
#define RK3308_DAC_L_HPMIX_SEL_MSK		(0x3 << RK3308_DAC_L_HPMIX_SEL_SFT)
#define RK3308_DAC_L_HPMIX_LINEIN_I2S		(0x3 << RK3308_DAC_L_HPMIX_SEL_SFT)
#define RK3308_DAC_L_HPMIX_LINEIN		(0x2 << RK3308_DAC_L_HPMIX_SEL_SFT)
#define RK3308_DAC_L_HPMIX_I2S			(0x1 << RK3308_DAC_L_HPMIX_SEL_SFT)
#define RK3308_DAC_L_HPMIX_NONE			(0x0 << RK3308_DAC_L_HPMIX_SEL_SFT)
#define RK3308_DAC_x_HPMIX_GAIN_MIN		0x1 /* 0x0 and 0x3 are reserved */
#define RK3308_DAC_x_HPMIX_GAIN_MAX		0x2
#define RK3308_DAC_R_HPMIX_GAIN_SFT		4
#define RK3308_DAC_R_HPMIX_GAIN_MSK		(0x3 << RK3308_DAC_R_HPMIX_GAIN_SFT)
#define RK3308_DAC_R_HPMIX_GAIN_0DB		(0x2 << RK3308_DAC_R_HPMIX_GAIN_SFT)
#define RK3308_DAC_R_HPMIX_GAIN_NDB_6		(0x1 << RK3308_DAC_R_HPMIX_GAIN_SFT)
#define RK3308_DAC_L_HPMIX_GAIN_SFT		0
#define RK3308_DAC_L_HPMIX_GAIN_MSK		(0x3 << RK3308_DAC_L_HPMIX_GAIN_SFT)
#define RK3308_DAC_L_HPMIX_GAIN_0DB		(0x2 << RK3308_DAC_L_HPMIX_GAIN_SFT)
#define RK3308_DAC_L_HPMIX_GAIN_NDB_6		(0x1 << RK3308_DAC_L_HPMIX_GAIN_SFT)

/* RK3308_DAC_ANA_CON13 - REG: 0x0474 */
#define RK3308_DAC_R_HPMIX_UNMUTE		BIT(6)
#define RK3308_DAC_R_HPMIX_WORK			BIT(5)
#define RK3308_DAC_R_HPMIX_EN			BIT(4)
#define RK3308_DAC_L_HPMIX_UNMUTE		BIT(2)
#define RK3308_DAC_L_HPMIX_WORK			BIT(1)
#define RK3308_DAC_L_HPMIX_EN			BIT(0)

/* RK3308_DAC_ANA_CON14 - REG: 0x0478 */
#define RK3308_DAC_VCM_LINEOUT_EN		(0x1 << 4)
#define RK3308_DAC_CURRENT_CHARGE_SFT		0
#define RK3308_DAC_CURRENT_CHARGE_MSK		(0xf << RK3308_DAC_CURRENT_CHARGE_SFT)

/* RK3308_DAC_ANA_CON15 - REG: 0x047C */
#define RK3308_DAC_LINEOUT_POP_SOUND_R_SFT	4
#define RK3308_DAC_LINEOUT_POP_SOUND_R_MSK	(0x3 << RK3308_DAC_LINEOUT_POP_SOUND_R_SFT)
#define RK3308_DAC_R_SEL_DC_FROM_INTERNAL	(0x2 << RK3308_DAC_LINEOUT_POP_SOUND_R_SFT)
#define RK3308_DAC_R_SEL_DC_FROM_VCM		(0x1 << RK3308_DAC_LINEOUT_POP_SOUND_R_SFT)
#define RK3308_DAC_R_SEL_LINEOUT_FROM_INTERNAL	(0x0 << RK3308_DAC_LINEOUT_POP_SOUND_R_SFT)
#define RK3308_DAC_LINEOUT_POP_SOUND_L_SFT	0
#define RK3308_DAC_LINEOUT_POP_SOUND_L_MSK	(0x3 << RK3308_DAC_LINEOUT_POP_SOUND_L_SFT)
#define RK3308_DAC_L_SEL_DC_FROM_INTERNAL	(0x2 << RK3308_DAC_LINEOUT_POP_SOUND_L_SFT)
#define RK3308_DAC_L_SEL_DC_FROM_VCM		(0x1 << RK3308_DAC_LINEOUT_POP_SOUND_L_SFT)
#define RK3308_DAC_L_SEL_LINEOUT_FROM_INTERNAL	(0x0 << RK3308_DAC_LINEOUT_POP_SOUND_L_SFT)

#endif /* __RK3308_CODEC_H__ */
