/*
 * symbol.c
 *
 * Copyright 1999 Michael Elizabeth Chastain, <mailto:mec@shout.net>.
 * Licensed under the Gnu Public License, Version 2.
 *
 * Symbol table.
 */

#include <stdio.h>
#include <string.h>

#include "mconfig.h"



/*
 * Some convenient piece constants.
 */

const piece_type piece_empty = {"", 0};
const piece_type piece_y = {"y", 1};
const piece_type piece_m = {"m", 1};
const piece_type piece_n = {"n", 1};
const piece_type piece_0 = {"0", 1};



/*
 * A little hash table.
 */

#define HASH_SIZE	1009

static symbol_type *symbol_table[HASH_SIZE];



/*
 * Compute a hash code.  This likes to hash on the last characters of the
 * string, which is fine, because the first characters are almost always
 * "CONFIG_" anyways.
 */

static unsigned int 
hash_key(piece_type name)
{
	int             i;
	unsigned int    n = 0;

	for (i = 0; i < name.len; i++)
		n = (n << 1) ^ (unsigned char) name.ptr[i];

	return n % HASH_SIZE;
}



/*
 * Create a symbol.  Returns an existing symbol if one exists.
 */

symbol_type    *
symbol_create(piece_type name)
{
	const unsigned int key = hash_key(name);
	symbol_type    *symbol;

	for (symbol = symbol_table[key]; symbol != NULL; symbol = symbol->hash_link) {
		if (name.len == symbol->name.len
		    && memcmp(name.ptr, symbol->name.ptr, name.len) == 0)
			return symbol;
	}

	symbol = grab_memory(sizeof(*symbol));
	symbol->next = NULL;
	symbol->def_line = NULL;
	symbol->type = type_unset;
	symbol->origin = origin_unset;
	symbol->name = name;
	symbol->value = piece_empty;
	symbol->hash_link = symbol_table[key];
	symbol_table[key] = symbol;
	return symbol;
}



/*
 * Lookup an existing symbol.
 */

symbol_type    *
symbol_lookup(piece_type name)
{
	const unsigned int key = hash_key(name);
	symbol_type    *symbol;

	for (symbol = symbol_table[key]; symbol != NULL; symbol = symbol->hash_link) {
		if (name.len == symbol->name.len
		    && memcmp(name.ptr, symbol->name.ptr, name.len) == 0)
			return symbol;
	}

	return NULL;
}



/*
 * Create a nonce symbol (an automatically-generated temporary).
 * FIXME: it would be nice to check that this really is a nonce!
 */

symbol_type    *
symbol_nonce(const char *s)
{
	static int      nonce_counter = 0;
	piece_type      piece;
	char           *buf;
	int             len;

	if (s == NULL)
		s = "";
	len = strlen(s);
	buf = grab_memory(len + 16);

	if (nonce_counter >= 1000000000)
		error_internal("symbol_nonce: too many nonces");
	sprintf(buf, "%s%d", s, nonce_counter++);
	piece.ptr = buf;
	piece.len = strlen(buf);

	if (symbol_lookup(piece) != NULL)
		error_internal("symbol_nonce: nonce already exists");

	return symbol_create(piece);
}



/*
 * Initialize the symbol table.  This gets called once at initialization,
 * and its job is to populate the table with symbols $ARCH and $arch.
 */

void 
symbol_table_init(const char *arch)
{
	if (arch != NULL) {
		symbol_type    *symbol;
		piece_type      name;
		piece_type      value;

		name.ptr = "ARCH";
		name.len = 4;
		value.ptr = arch;
		value.len = strlen(arch);

		symbol = symbol_create(name);
		symbol->type = type_string;
		symbol->value = value;
		symbol->origin = origin_permanent;

		name.ptr = "arch";
		symbol = symbol_create(name);
		symbol->type = type_string;
		symbol->value = value;
		symbol->origin = origin_permanent;
	}
}



/*
 * Reset the value of each symbol.
 */

void 
symbol_table_reset()
{
	int             i;
	symbol_type    *symbol;

	for (i = 0; i < HASH_SIZE; i++) {
		for (symbol = symbol_table[i]; symbol != NULL; symbol = symbol->hash_link) {
			if (symbol->origin != origin_permanent) {
				symbol->def_line = NULL;
				symbol->origin = origin_unset;
				symbol->value = piece_empty;
			}
		}
	}
}
