#!/bin/sh

if [ "$1" == "" -o "$2" == "" ]
then
    echo "Usage: $0 good_dev bad_dev"
    exit 1
fi

GOOD="$1"
BAD="$2"

GOOD_SECTORS=1952767
GOOD_SECTORSIZE=512

BAD_SECTORS=8015505
BAD_SECTORSIZE=512

BAD_ERROR_SECTOR=4292779

GOOD_REF="/mnt/scratch/ref-good.img"
BAD_REF="/mnt/scratch/ref-bad.img"
PAT_REF="/mnt/scratch/ref-pat.img"

DC3DD="src/dc3dd"

TEST_IMG="/mnt/scratch/test.img"
TEST_LOG="/mnt/scratch/test.log"

# no config below this line
#GOOD_SECTORS=`grep sectors /var/log/messages | grep $GOOD_DEV | tail -1 | awk '{print $9}'`
#GOOD_SECTORSIZE=`grep sectors /var/log/messages | grep $GOOD_DEV | tail -1 | awk '{print $10}' | cut -f1 -d-`
GOOD_BYTES=`expr $GOOD_SECTORS \* $GOOD_SECTORSIZE`

#BAD_SECTORS=`grep sectors /var/log/messages | grep $BAD_DEV | tail -1 | awk '{print $9}'`
#BAD_SECTORSIZE=`grep sectors /var/log/messages | grep $BAD_DEV | tail -1 | awk '{print $10}' | cut -f1 -d-`
BAD_BYTES=`expr $BAD_SECTORS \* $BAD_SECTORSIZE`

umount "$GOOD"
umount "$BAD"

if [ -L "$GOOD" ]
then
    GOOD=`readlink -f "$GOOD"`
fi

if [ -L "$BAD" ]
then
    BAD=`readlink -f "$BAD"`
fi

GOOD_DEV=`basename "$GOOD"`
BAD_DEV=`basename "$BAD"`

# clean up temp files from any previous runs
function cleanup()
{
    rm "$TEST_IMG" "$TEST_LOG" "$TEST_IMG".* "$TEST_LOG".*
    if [ -e "$TEST_IMG" -o -e "$TEST_LOG" ]
    then
        echo "Failed to remove test output files"
        exit 2
    fi
}

function good()
{
cleanup

# load reference image into good drive
dd if="$GOOD_REF" of="$GOOD" bs="$GOOD_SECTORSIZE"
if [ $? -ne 0 ]
then
    echo "Failed to load good reference image $GOOD_REF to good device $GOOD"
    exit 1
fi

# get image from good drive
"$DC3DD" if="$GOOD" of="$TEST_IMG" conv=noerror,sync iflag=direct log="$TEST_LOG" hash=md5,sha1 hashconv=after sizeprobe=on progress=on
if [ $? -ne 0 ]
then
    echo "Run Failed"
    exit 3
fi

# check the retrieved image

# a. file sizes
REF_BYTES=`stat -c %s "$GOOD_REF"`
IMG_BYTES=`stat -c %s "$TEST_IMG"`

if [ $REF_BYTES -ne $IMG_BYTES ]
then
    echo "Size mismatch"
    exit 3
fi

# b. checksums
REF_SUM=`md5sum "$GOOD_REF" | cut -f 1 -d \ `
IMG_SUM=`md5sum "$TEST_IMG" | cut -f 1 -d \ `
LOG_SUM=`grep "md5 TOTAL" "$TEST_LOG" | cut -f 3 -d \ `

if [ "$REF_SUM" != "$IMG_SUM" ]
then
    echo "Ref/Img Checksum mismatch"
    exit 3
fi

if [ "$REF_SUM" != "$LOG_SUM" ]
then
    echo "Ref/Log Checksum mismatch"
    exit 3
fi

# sector probe/count

LOG_SECTORMODE=`grep sector\ size\: "$TEST_LOG" | awk '{print $4}'`
LOG_SECTORSIZE=`grep sector\ size\: "$TEST_LOG" | awk '{print $3}'`
LOG_SECTORS_FULL=`grep sectors\ in "$TEST_LOG" | awk '{print $1}' | cut -f1 -d+`
LOG_SECTORS_PART=`grep sectors\ in "$TEST_LOG" | awk '{print $1}' | cut -f2 -d+`
LOG_BYTES=`grep bytes "$TEST_LOG" | awk '{print $1}'`

if [ "$LOG_SECTORMODE" != "(probed)" ]
then
    echo "Failed to probe sector size"
    exit 3
fi

if [ "$LOG_SECTORSIZE" != "$GOOD_SECTORSIZE" ]
then
    echo "Wrong sector size detected"
    exit 3
fi

if [ "$LOG_SECTORS_PART" != "0" ]
then
    echo "Partial sectors must be zero"
    exit 3
fi

if [ "$LOG_SECTORS_FULL" != "$GOOD_SECTORS" ]
then
    echo "Wrong sector count"
    exit 3
fi

if [ "$LOG_BYTES" != "$GOOD_BYTES" ]
then
    echo "Wrong byte count - log doesn't match size calculated from kernel sectors and sectorsize"
    exit 3
fi

if [ "$LOG_BYTES" != "$IMG_BYTES" ]
then
    echo "Log file bytes doesn't match output file size"
    exit 3
fi

if [ "$LOG_BYTES" != "$REF_BYTES" ]
then
    echo "Log file bytes doesn't match reference file size"
    exit 3
fi

MSGCOUNT=`egrep -c bytes.+copied "$TEST_LOG"`
if [ $MSGCOUNT -ne 1 ]
then
    echo "Log is missing 'bytes copied' message"
    exit 3
fi

# test verify mode

# a. drive to reference file
"$DC3DD" if="$GOOD" vf="$GOOD_REF" conv=noerror,sync iflag=direct log="$TEST_LOG" sizeprobe=on progress=on
if [ $? -ne 0 ]
then
    echo "Verify to Reference Image Failed"
    exit 3
fi

MSGCOUNT=`egrep -c bytes.+compared "$TEST_LOG"`
if [ $MSGCOUNT -ne 1 ]
then
    echo "Log is missing 'bytes compared' message"
    exit 3
fi

MSGCOUNT=`grep -c Verify\ PASSED "$TEST_LOG"`
if [ $MSGCOUNT -ne 1 ]
then
    echo "Log is missing 'Verify PASSED' message"
    exit 3
fi


# b. drive to produced image file
"$DC3DD" if="$GOOD" vf="$TEST_IMG" conv=noerror,sync iflag=direct log="$TEST_LOG" sizeprobe=on progress=on
if [ $? -ne 0 ]
then
    echo "Verify to Created Image Failed"
    exit 3
fi

MSGCOUNT=`egrep -c bytes.+compared "$TEST_LOG"`
if [ $MSGCOUNT -ne 2 ]
then
    echo "Log is missing 'bytes compared' message"
    exit 3
fi

MSGCOUNT=`grep -c Verify\ PASSED "$TEST_LOG"`
if [ $MSGCOUNT -ne 2 ]
then
    echo "Log is missing 'Verify PASSED' message"
    exit 3
fi


# test splitting
cleanup

"$DC3DD" if="$GOOD" of="$TEST_IMG" conv=noerror,sync iflag=direct log="$TEST_LOG" splitformat=000 split=128M hash=md5,sha1 hashconv=after sizeprobe=on progress=on
if [ $? -ne 0 ]
then
    echo "Image to split output failed"
    exit 3
fi

IMG_SUM=`cat "$TEST_IMG".* | md5sum | cut -f 1 -d \ `
LOG_SUM=`grep "md5 TOTAL" "$TEST_LOG" | cut -f 3 -d \ `

if [ "$REF_SUM" != "$LOG_SUM" ]
then
    echo "Split: Ref/Log Checksum mismatch"
    exit 3
fi

if [ "$IMG_SUM" != "$REF_SUM" ]
then
    echo "Split: Img/Ref Checksum mismatch"
    exit 3
fi

LOG_SECTORMODE=`grep sector\ size\: "$TEST_LOG" | awk '{print $4}'`
LOG_SECTORSIZE=`grep sector\ size\: "$TEST_LOG" | awk '{print $3}'`
LOG_SECTORS_FULL=`grep sectors\ in "$TEST_LOG" | awk '{print $1}' | cut -f1 -d+`
LOG_SECTORS_PART=`grep sectors\ in "$TEST_LOG" | awk '{print $1}' | cut -f2 -d+`
LOG_BYTES=`grep bytes "$TEST_LOG" | awk '{print $1}'`

if [ "$LOG_SECTORMODE" != "(probed)" ]
then
    echo "Split: Failed to probe sector size"
    exit 3
fi

if [ "$LOG_SECTORSIZE" != "$GOOD_SECTORSIZE" ]
then
    echo "Split: Wrong sector size detected"
    exit 3
fi

if [ "$LOG_SECTORS_PART" != "0" ]
then
    echo "Split: Partial sectors must be zero"
    exit 3
fi

if [ "$LOG_SECTORS_FULL" != "$GOOD_SECTORS" ]
then
    echo "Split: Wrong sector count"
    exit 3
fi

if [ "$LOG_BYTES" != "$GOOD_BYTES" ]
then
    echo "Split: Wrong byte count - log doesn't match size calculated from kernel sectors and sectorsize"
    exit 3
fi

IMG_BYTES=0
for a in "$TEST_IMG".*
do
    SPLIT_BYTES=`stat -c %s "$a"`
    IMG_BYTES=`expr $IMG_BYTES + $SPLIT_BYTES`
done

if [ "$IMG_BYTES" != "$REF_BYTES" ]
then
    echo "Split: image bytes doesn't match reference file size"
    exit 3
fi

if [ "$LOG_BYTES" != "$IMG_BYTES" ]
then
    echo "Split: Log file bytes doesn't match output file size"
    exit 3
fi

# test joining
"$DC3DD" if="$GOOD" vfjoin="$TEST_IMG.000" conv=noerror,sync iflag=direct sizeprobe=on progress=on
if [ $? -ne 0 ]
then
    echo "vfjoin failed"
    exit 3
fi

"$DC3DD" ifjoin="$TEST_IMG.000" vf="$GOOD_REF" conv=noerror,sync iflag=direct sizeprobe=on progress=on
if [ $? -ne 0 ]
then
    echo "ifjoin failed"
    exit 3
fi

# test interrupting

cleanup

set -m
"$DC3DD" if="$GOOD" of="$TEST_IMG" conv=noerror,sync iflag=direct log="$TEST_LOG" hash=md5,sha1 hashconv=after sizeprobe=on progress=on& pid=$!;
sleep 5; kill -s 2 %1
wait %1
set +m

#if [ $? -ne 0 ]
#then
#    echo "Run Failed"
#    exit 3
#fi

IMG_BYTES=`stat -c %s "$TEST_IMG"`
LOG_BYTES=`grep bytes "$TEST_LOG" | awk '{print $1}'`

if [ "$LOG_BYTES" != "$IMG_BYTES" ]
then
    echo "Interrupt: Log file bytes doesn't match output file size"
    exit 3
fi

IMG_SUM=`md5sum "$TEST_IMG" | cut -f 1 -d \ `
LOG_SUM=`grep "md5 TOTAL" "$TEST_LOG" | cut -f 3 -d \ `

if [ "$IMG_SUM" != "$LOG_SUM" ]
then
    echo "Interrupt: Img/Log Checksum mismatch"
    exit 3
fi

}


#############################################################
# test bad drive

function bad()
{

ERRS=21

cleanup

# get image from bad drive
"$DC3DD" if="$BAD" of="$TEST_IMG" conv=noerror,sync iflag=direct log="$TEST_LOG" hash=md5,sha1 hashconv=after sizeprobe=on progress=on
if [ $? -ne 0 ]
then
    echo "Run Failed"
    exit 4
fi

# check the retrieved image

# a. file sizes
REF_BYTES=`stat -c %s "$BAD_REF"`
IMG_BYTES=`stat -c %s "$TEST_IMG"`

if [ $REF_BYTES -ne $IMG_BYTES ]
then
    echo "Size mismatch"
    exit 4
fi

# b. checksums
REF_SUM=`md5sum "$BAD_REF" | cut -f 1 -d \ `
IMG_SUM=`md5sum "$TEST_IMG" | cut -f 1 -d \ `
LOG_SUM=`grep "md5 TOTAL" "$TEST_LOG" | cut -f 3 -d \ `

if [ "$REF_SUM" != "$IMG_SUM" ]
then
    echo "Ref/Img Checksum mismatch"
    exit 4
fi

if [ "$REF_SUM" != "$LOG_SUM" ]
then
    echo "Ref/Log Checksum mismatch"
    exit 4
fi

# sector probe/count
LOG_SECTORMODE=`grep sector\ size\: "$TEST_LOG" | awk '{print $4}'`
LOG_SECTORSIZE=`grep sector\ size\: "$TEST_LOG" | awk '{print $3}'`
LOG_SECTORS_FULL=`grep sectors\ in "$TEST_LOG" | awk '{print $1}' | cut -f1 -d+`
LOG_SECTORS_PART=`grep sectors\ in "$TEST_LOG" | awk '{print $1}' | cut -f2 -d+`
LOG_SECTORS_OUT_FULL=`grep sectors\ out "$TEST_LOG" | awk '{print $1}' | cut -f1 -d+`
LOG_SECTORS_OUT_PART=`grep sectors\ out "$TEST_LOG" | awk '{print $1}' | cut -f2 -d+`
LOG_BYTES=`grep bytes "$TEST_LOG" | awk '{print $1}'`

if [ "$LOG_SECTORMODE" != "(probed)" ]
then
    echo "Failed to probe sector size"
    exit 4
fi

if [ "$LOG_SECTORSIZE" != "$BAD_SECTORSIZE" ]
then
    echo "Wrong sector size detected"
    exit 4
fi

if [ "$LOG_SECTORS_PART" != "21" ]
then
    echo "Partial (error) sectors in must be 21"
    exit 4
fi

if [ `expr $LOG_SECTORS_FULL + $LOG_SECTORS_PART` != "$BAD_SECTORS" ]
then
    echo "Wrong sectors in total count"
    exit 4
fi

if [ "$LOG_SECTORS_OUT_PART" != "0" ]
then
    echo "Partial (error) sectors out must be 0"
    exit 4
fi

if [ "$LOG_SECTORS_OUT_FULL" != "$BAD_SECTORS" ]
then
    echo "Wrong sectors out total count"
    exit 4
fi

if [ "$LOG_BYTES" != "$BAD_BYTES" ]
then
    echo "Wrong byte count - log doesn't match size calculated from kernel sectors and sectorsize"
    exit 4
fi

if [ "$LOG_BYTES" != "$IMG_BYTES" ]
then
    echo "Log file bytes doesn't match output file size"
    exit 4
fi

if [ "$LOG_BYTES" != "$REF_BYTES" ]
then
    echo "Log file bytes doesn't match reference file size"
    exit 4
fi

MSGCOUNT=`egrep -c bytes.+copied "$TEST_LOG"`
if [ $MSGCOUNT -ne 1 ]
then
    echo "Log is missing 'bytes copied' message"
    exit 4
fi

for ((i=4292779; i<=4292799; i+=1))
do
    MSGCOUNT=`grep -c "at sector $i" "$TEST_LOG"`
    if [ $MSGCOUNT -ne 1 ]
    then
        echo "Log is missing error message for sector $i"
        exit 4
    fi
done

# test verify mode

# a. drive to reference file
"$DC3DD" if="$BAD" vf="$BAD_REF" conv=noerror,sync iflag=direct log="$TEST_LOG" sizeprobe=on progress=on
if [ $? -ne 0 ]
then
    echo "Verify to Reference Image Failed"
    exit 4
fi

MSGCOUNT=`egrep -c bytes.+compared "$TEST_LOG"`
if [ $MSGCOUNT -ne 1 ]
then
    echo "Log is missing 'bytes compared' message"
    exit 4
fi

MSGCOUNT=`grep -c Verify\ PASSED "$TEST_LOG"`
if [ $MSGCOUNT -ne 1 ]
then
    echo "Log is missing 'Verify PASSED' message"
    exit 4
fi


# b. drive to produced image file
"$DC3DD" if="$BAD" vf="$TEST_IMG" conv=noerror,sync iflag=direct log="$TEST_LOG" sizeprobe=on progress=on
if [ $? -ne 0 ]
then
    echo "Verify to Created Image Failed"
    exit 4
fi

MSGCOUNT=`egrep -c bytes.+compared "$TEST_LOG"`
if [ $MSGCOUNT -ne 2 ]
then
    echo "Log is missing 'bytes compared' message"
    exit 4
fi

MSGCOUNT=`grep -c Verify\ PASSED "$TEST_LOG"`
if [ $MSGCOUNT -ne 2 ]
then
    echo "Log is missing 'Verify PASSED' message"
    exit 4
fi


# test splitting
cleanup

"$DC3DD" if="$BAD" of="$TEST_IMG" conv=noerror,sync iflag=direct log="$TEST_LOG" splitformat=000 split=128M hash=md5,sha1 hashconv=after sizeprobe=on progress=on
if [ $? -ne 0 ]
then
    echo "Image to split output failed"
    exit 4
fi

IMG_SUM=`cat "$TEST_IMG".* | md5sum | cut -f 1 -d \ `
LOG_SUM=`grep "md5 TOTAL" "$TEST_LOG" | cut -f 3 -d \ `

if [ "$REF_SUM" != "$LOG_SUM" ]
then
    echo "Split: Ref/Log Checksum mismatch"
    exit 4
fi

if [ "$IMG_SUM" != "$REF_SUM" ]
then
    echo "Split: Img/Ref Checksum mismatch"
    exit 4
fi

LOG_SECTORMODE=`grep sector\ size\: "$TEST_LOG" | awk '{print $4}'`
LOG_SECTORSIZE=`grep sector\ size\: "$TEST_LOG" | awk '{print $3}'`
LOG_SECTORS_FULL=`grep sectors\ in "$TEST_LOG" | awk '{print $1}' | cut -f1 -d+`
LOG_SECTORS_PART=`grep sectors\ in "$TEST_LOG" | awk '{print $1}' | cut -f2 -d+`
LOG_SECTORS_OUT_FULL=`grep sectors\ out "$TEST_LOG" | awk '{print $1}' | cut -f1 -d+`
LOG_SECTORS_OUT_PART=`grep sectors\ out "$TEST_LOG" | awk '{print $1}' | cut -f2 -d+`
LOG_BYTES=`grep bytes "$TEST_LOG" | awk '{print $1}'`

if [ "$LOG_SECTORMODE" != "(probed)" ]
then
    echo "Split: Failed to probe sector size"
    exit 4
fi

if [ "$LOG_SECTORSIZE" != "$BAD_SECTORSIZE" ]
then
    echo "Split: Wrong sector size detected"
    exit 4
fi

if [ "$LOG_SECTORS_PART" != "21" ]
then
    echo "Split: Partial (error) sectors in must be 21"
    exit 4
fi

if [ `expr $LOG_SECTORS_FULL + $LOG_SECTORS_PART` != "$BAD_SECTORS" ]
then
    echo "Split: Wrong sectors in total count"
    exit 4
fi

if [ "$LOG_SECTORS_OUT_PART" != "0" ]
then
    echo "Split: Partial (error) sectors out must be 0"
    exit 4
fi

if [ "$LOG_BYTES" != "$BAD_BYTES" ]
then
    echo "Split: Wrong byte count - log doesn't match size calculated from kernel sectors and sectorsize"
    exit 4
fi

IMG_BYTES=0
for a in "$TEST_IMG".*
do
    SPLIT_BYTES=`stat -c %s "$a"`
    IMG_BYTES=`expr $IMG_BYTES + $SPLIT_BYTES`
done

if [ "$IMG_BYTES" != "$REF_BYTES" ]
then
    echo "Split: image bytes doesn't match reference file size"
    exit 4
fi

if [ "$LOG_BYTES" != "$IMG_BYTES" ]
then
    echo "Split: Log file bytes doesn't match output file size"
    exit 4
fi

}

function bad2()
{
    cleanup

	# TODO: should make sure drive actually produces read errors, otherwise we aren't testing anything useful

    # get image from bad drive - dd
	"dd" if="$BAD" of="$TEST_IMG.ddslow" conv=noerror,sync iflag=direct bs=512 skip=4292778 count=40
	if [ $? -ne 0 ]
	then
		echo "Run Failed"
		exit 7
	fi

	DDSLOW_SUM=`md5sum "$TEST_IMG.ddslow" | cut -f 1 -d \ `


    # get image from bad drive - dc3dd slow mode
	"$DC3DD" if="$BAD" of="$TEST_IMG.slow" conv=noerror,sync iflag=direct bs=512 skip=4292778 count=40 hash=md5,sha1 hashconv=after
	if [ $? -ne 0 ]
	then
		echo "Run Failed"
		exit 7
	fi

	SLOW_SUM=`md5sum "$TEST_IMG.slow" | cut -f 1 -d \ `

	# get image from bad drive - dc3dd fast (dynamic) mode
	"$DC3DD" if="$BAD" of="$TEST_IMG.fast" conv=noerror,sync iflag=direct skip=4292778 count=40 hash=md5,sha1 hashconv=after
	if [ $? -ne 0 ]
	then
		echo "Run Failed"
		exit 7
	fi

	FAST_SUM=`md5sum "$TEST_IMG.fast" | cut -f 1 -d \ `

	if [ "$DDSLOW_SUM" != "$SLOW_SUM" ]
	then
		echo "bad2: dc3dd slow mode checksum doesn't match dd checksum"
		exit 7
	fi

	if [ "$DDSLOW_SUM" != "$FAST_SUM" ]
	then
		echo "bad2: dc3dd fast mode checksum doesn't match dd checksum"
		exit 7
	fi

    rm "$TEST_IMG.ddslow"
    rm "$TEST_IMG.slow"
    rm "$TEST_IMG.fast"
}



function zero
{
    cleanup

    ZERO_COUNT=$1
    DEF_SECTORSIZE=512

    "$DC3DD" if="/dev/zero" of="/dev/null" log="$TEST_LOG" count="$ZERO_COUNT" sizeprobe=on progress=on
    if [ $? -ne 0 ]
    then
        echo "Imaging from /dev/zero"
        exit 5
    fi

    LOG_SECTORMODE=`grep sector\ size\: "$TEST_LOG" | awk '{print $4}'`
    LOG_SECTORSIZE=`grep sector\ size\: "$TEST_LOG" | awk '{print $3}'`
    LOG_SECTORS_FULL=`grep sectors\ in "$TEST_LOG" | awk '{print $1}' | cut -f1 -d+`
    LOG_SECTORS_PART=`grep sectors\ in "$TEST_LOG" | awk '{print $1}' | cut -f2 -d+`
    LOG_SECTORS_OUT_FULL=`grep sectors\ out "$TEST_LOG" | awk '{print $1}' | cut -f1 -d+`
    LOG_SECTORS_OUT_PART=`grep sectors\ out "$TEST_LOG" | awk '{print $1}' | cut -f2 -d+`
    LOG_BYTES=`grep bytes "$TEST_LOG" | awk '{print $1}'`

    if [ "$LOG_SECTORMODE" != "(assumed)" ]
    then
        echo "Split: should be assuming sector size for zero/pattern fill"
        exit 5
    fi

    if [ "$LOG_SECTORSIZE" != "$DEF_SECTORSIZE" ]
    then
        echo "Split: Wrong sector size assumed"
        exit 5
    fi

    if [ "$LOG_SECTORS_PART" != "0" ]
    then
        echo "Split: Partial (error) sectors in must be 0"
        exit 5
    fi

    if [ "$LOG_SECTORS_FULL" != "$ZERO_COUNT" ]
    then
        echo "Split: Wrong sectors in full count"
        exit 5
    fi

    if [ "$LOG_SECTORS_OUT_PART" != "0" ]
    then
        echo "Split: Partial (error) sectors out must be 0"
        exit 5
    fi

    if [ "$LOG_SECTORS_OUT_FULL" != "$ZERO_COUNT" ]
    then
        echo "Split: Full sectors out count wrong"
        exit 5
    fi

    if [ "$LOG_BYTES" != `expr $ZERO_COUNT \* $DEF_SECTORSIZE` ]
    then
        echo "Split: Wrong byte count - log doesn't match size calculated from kernel sectors and sectorsize"
        exit 5
    fi
}

function zeros
{
    zero       1 
    zero      10 
    zero     500 
    zero     512 
    zero    1000 
    zero    1024 
    zero 1000000
    zero 1048576
}

function pattern
{
    cleanup

    PAT="AABBCCDD"
    PAT_COUNT=12345

    # write pattern to test image file
    "$DC3DD" textpattern="$PAT" of="$TEST_IMG" log="$TEST_LOG" count="$PAT_COUNT" hash=md5,sha1 hashconv=after sizeprobe=on progress=on
    if [ $? -ne 0 ]
    then
        echo "Pattern: Run Failed"
        exit 3
    fi

    # check the retrieved image

    # a. file sizes
    REF_BYTES=`stat -c %s "$PAT_REF"`
    IMG_BYTES=`stat -c %s "$TEST_IMG"`

    if [ $REF_BYTES -ne $IMG_BYTES ]
    then
        echo "Pattern: Size mismatch"
        exit 3
    fi

    # b. checksums
    REF_SUM=`md5sum "$PAT_REF" | cut -f 1 -d \ `
    IMG_SUM=`md5sum "$TEST_IMG" | cut -f 1 -d \ `
    LOG_SUM=`grep "md5 TOTAL" "$TEST_LOG" | cut -f 3 -d \ `

    if [ "$REF_SUM" != "$IMG_SUM" ]
    then
        echo "Pattern: Ref/Img Checksum mismatch"
        exit 3
    fi

    if [ "$REF_SUM" != "$LOG_SUM" ]
    then
        echo "Pattern: Ref/Log Checksum mismatch"
        exit 3
    fi

    # sector probe/count
    PAT_SECTORS="$PAT_COUNT"
    PAT_SECTORSIZE=512
    PAT_BYTES=`expr $PAT_SECTORS \* $PAT_SECTORSIZE`

    LOG_SECTORMODE=`grep sector\ size\: "$TEST_LOG" | awk '{print $4}'`
    LOG_SECTORSIZE=`grep sector\ size\: "$TEST_LOG" | awk '{print $3}'`
    LOG_SECTORS_FULL=`grep sectors\ in "$TEST_LOG" | awk '{print $1}' | cut -f1 -d+`
    LOG_SECTORS_PART=`grep sectors\ in "$TEST_LOG" | awk '{print $1}' | cut -f2 -d+`
    LOG_BYTES=`grep bytes "$TEST_LOG" | awk '{print $1}'`

    if [ "$LOG_SECTORMODE" != "(assumed)" ]
    then
        echo "Pattern: should be assuming sector size"
        exit 3
    fi

    if [ "$LOG_SECTORSIZE" != "$PAT_SECTORSIZE" ]
    then
        echo "Pattern: Wrong sector size assumed"
        exit 3
    fi

    if [ "$LOG_SECTORS_PART" != "0" ]
    then
        echo "Pattern: Partial sectors must be zero"
        exit 3
    fi

    if [ "$LOG_SECTORS_FULL" != "$PAT_SECTORS" ]
    then
        echo "Pattern: Wrong sector count"
        exit 3
    fi

    if [ "$LOG_BYTES" != "$PAT_BYTES" ]
    then
        echo "Pattern: Wrong byte count - log doesn't match size calculated from known sectors and sectorsize"
        exit 3
    fi

    if [ "$LOG_BYTES" != "$IMG_BYTES" ]
    then
        echo "Pattern: Log file bytes doesn't match output file size"
        exit 3
    fi

    if [ "$LOG_BYTES" != "$REF_BYTES" ]
    then
        echo "Pattern: Log file bytes doesn't match reference file size"
        exit 3
    fi

    MSGCOUNT=`egrep -c bytes.+copied "$TEST_LOG"`
    if [ $MSGCOUNT -ne 1 ]
    then
        echo "Pattern: Log is missing 'bytes copied' message"
        exit 3
    fi
}

function messages
{
    cleanup

    "$DC3DD" if="/dev/zero" of="/dev/null" log="$TEST_LOG" count="1000" progress=on
    if [ $? -ne 0 ]
    then
        echo "Imaging to /dev/zero test exit message failed"
        exit 6
    fi   

    LOG_VERB=`egrep "dc3dd.+?at" "$TEST_LOG" | tail -1 | awk '{print $2}'`

    if [ "$LOG_VERB" != "completed" ]
    then
        echo "Exit message verb on imaging /dev/zero success should be 'completed'"
        exit 6
    fi

    cleanup

    "$DC3DD" if="/dev/zero" of="$TEST_IMG" log="$TEST_LOG" count="1000" progress=on
    if [ $? -ne 0 ]
    then
        echo "Imaging to /dev/zero test exit message failed"
        exit 6
    fi   

    LOG_VERB=`egrep "dc3dd.+?at" "$TEST_LOG" | tail -1 | awk '{print $2}'`

    if [ "$LOG_VERB" != "completed" ]
    then
        echo "Exit message verb on imaging /dev/zero success should be 'completed'"
        exit 6
    fi

    cleanup

    "$DC3DD" if="$GOOD" of="$TEST_IMG" log="$TEST_LOG" progress=on
    if [ $? -ne 0 ]
    then
        echo "Imaging good drive to test exit message failed"
        exit 6
    fi   

    LOG_VERB=`egrep "dc3dd.+?at" "$TEST_LOG" | tail -1 | awk '{print $2}'`

    if [ "$LOG_VERB" != "completed" ]
    then
        echo "Exit message verb on imaging good drive success should be 'completed'"
        exit 6
    fi

    cleanup

    "$DC3DD" if="$BAD" of="$TEST_IMG" log="$TEST_LOG" progress=on
    if [ $? -ne 1 ]
    then
        echo "Imaging bad drive to test exit message did not terminate with error condition"
        exit 6
    fi   

    LOG_VERB=`egrep "dc3dd.+?at" "$TEST_LOG" | tail -1 | awk '{print $2}'`

    if [ "$LOG_VERB" != "failed" ]
    then
        echo "Exit message verb on imaging bad drive failure should be 'failed'"
        exit 6
    fi

    cleanup

    "$DC3DD" if="$BAD" of="$TEST_IMG" conv=noerror,sync iflag=direct log="$TEST_LOG" progress=on
    if [ $? -ne 0 ]
    then
        echo "Imaging bad drive to test exit message failed"
        exit 6
    fi   

    LOG_VERB=`egrep "dc3dd.+?at" "$TEST_LOG" | tail -1 | awk '{print $2}'`

    if [ "$LOG_VERB" != "completed" ]
    then
        echo "Exit message verb on imaging bad drive success should be 'completed'"
        exit 6
    fi

    cleanup

    set -m
    "$DC3DD" if="$GOOD" of="$TEST_IMG" conv=noerror,sync iflag=direct log="$TEST_LOG" progress=on& pid=$!;
    sleep 5; kill -s 2 %1
    wait %1
    set +m

    LOG_VERB=`egrep "dc3dd.+?at" "$TEST_LOG" | tail -1 | awk '{print $2}'`

    if [ "$LOG_VERB" != "aborted" ]
    then
        echo "Exit message verb on imaging good drive interrupt should be 'aborted'"
        exit 6
    fi
}

function rj_cat
{
    HASHCONV="$1"
    BS="$2"

    rm "$TEST_LOG"
    cat "$TEST_IMG".* | "$DC3DD" of="$TEST_IMG" log="$TEST_LOG" progress=on progresscount=1000 hash=md5 "$HASHCONV" hashwindow=700M $BS
    JOIN_SUM=`grep "md5 TOTAL" "$TEST_LOG" | cut -f 3 -d \ `
    if [ "$ORIG_SUM" != "$JOIN_SUM" ]
    then
        echo "join regression: joining with cat & $HASHCONV $BS, final hash mismatch"
        exit 9
    fi

    egrep "(^md5|TOTAL)" "$TEST_LOG" > "${TEST_LOG}.join_hashes"
    diff -q "${TEST_LOG}.acquire_hashes" "${TEST_LOG}.join_hashes"
    if [ $? -ne 0 ]
    then
        echo "join regression: joining with cat & $HASHCONV $BS, piecewise hash mismatch"
        exit 9
    fi

    IMG_SUM=`md5sum "$TEST_IMG" | cut -f 1 -d \ `
    if [ "$JOIN_SUM" != "$IMG_SUM" ]
    then
        echo "join regression: joining with cat & $HASHCONV $BS, log total hash != output file hash"
        exit 9
    fi
}

function rj_ifjoin
{
    HASHCONV="$1"
    BS="$2"

    rm "$TEST_LOG"
    "$DC3DD" ifjoin="${TEST_IMG}.000" of="$TEST_IMG" log="$TEST_LOG" sizeprobe=on progress=on progresscount=1000 hash=md5 "$HASHCONV" hashwindow=700M $BS
    JOIN_SUM=`grep "md5 TOTAL" "$TEST_LOG" | cut -f 3 -d \ `
    if [ "$ORIG_SUM" != "$JOIN_SUM" ]
    then
        echo "join regression: joining with ifjoin & $HASHCONV $BS, final hash mismatch"
        exit 9
    fi

    egrep "(^md5|TOTAL)" "$TEST_LOG" > "${TEST_LOG}.join_hashes"
    diff -q "${TEST_LOG}.acquire_hashes" "${TEST_LOG}.join_hashes"
    if [ $? -ne 0 ]
    then
        echo "join regression: joining with ifjoin & $HASHCONV $BS, piecewise hash mismatch"
        exit 9
    fi

    IMG_SUM=`md5sum "$TEST_IMG" | cut -f 1 -d \ `
    if [ "$JOIN_SUM" != "$IMG_SUM" ]
    then
        echo "join regression: joining with ifjoin & $HASHCONV $BS, log total hash != output file hash"
        exit 9
    fi
}

function regressions_join
{
    cleanup

    # get reference image
    "$DC3DD" if="$BAD" of="$TEST_IMG" conv=noerror,sync iflag=direct log="$TEST_LOG" splitformat=000 split=700M hash=md5 hashconv=after hashwindow=700M sizeprobe=on progress=on progresscount=1000
    if [ $? -ne 0 ]
    then
        echo "join regression: split run failed"
        exit 9
    fi

    egrep "(^md5|TOTAL)" "$TEST_LOG" > "${TEST_LOG}.acquire_hashes"

    ORIG_SUM=`grep "md5 TOTAL" "$TEST_LOG" | cut -f 3 -d \ `

    rj_cat    "hashconv=before" "" 
    rj_cat    "hashconv=before" "bs=512" 
    rj_cat    "hashconv=after"  "" 
    rj_cat    "hashconv=after"  "bs=512" 

    rj_ifjoin "hashconv=before" "" 
    rj_ifjoin "hashconv=before" "bs=512" 
    rj_ifjoin "hashconv=after"  "" 
    rj_ifjoin "hashconv=after"  "bs=512" 
}

function regressions
{
    cleanup

    # seek crash
    "$DC3DD" if=/dev/zero of=/dev/null seek=10 count=10
    if [ $? -ne 0 ]
    then
        echo "seek operator causes failure"
	exit 7
    fi

    # ifjoin/vfjoin crash
    "$DC3DD" ifjoin=bar
    if [ $? -ne 1 ]
    then
	echo "Invalid ifjoin format not detected properly"
	exit 7
    fi

    "$DC3DD" vfjoin=bar
    if [ $? -ne 1 ]
    then
	echo "Invalid vfjoin format not detected properly"
	exit 7
    fi

    "$DC3DD" if=foo ifjoin=bar.000
    if [ $? -ne 1 ]
    then
	echo "Invalid combination of if/ifjoin not detected properly"
	exit 7
    fi

    "$DC3DD" vf=foo vfjoin=bar.000
    if [ $? -ne 1 ]
    then
	echo "Invalid combination of vf/vfjoin not detected properly"
	exit 7
    fi

    # ensure proper skip= offset is accounted for in error messages
    # when conv=sync,noerror not used

    rem=`expr $BAD_ERROR_SECTOR % 10000`
    skip=`expr $BAD_ERROR_SECTOR - $rem`

    "$DC3DD" if="$BAD" of=/dev/null skip=$skip count=5000 iflag=direct bs=512 log="$TEST_LOG.good" conv=sync,noerror 
    if [ $? -ne 0 ]
    then
        echo "checking skip: reference run failed"
        exit 7
    fi

    "$DC3DD" if="$BAD" of=/dev/null skip=$skip count=5000 iflag=direct bs=512 log="$TEST_LOG.test"
    if [ $? -ne 1 ]
    then
        echo "checking skip: test run failed"
        exit 7
    fi

    RIGHT_SECTOR=`grep reading "$TEST_LOG.good" | head -1 | perl -ne '/sectors?\s+(\d+)/; print $1 . "\n"'`
    CHECK_SECTOR=`grep reading "$TEST_LOG.test" | head -1 | perl -ne '/sectors?\s+(\d+)/; print $1 . "\n"'`

    if [ "$RIGHT_SECTOR" != "$CHECK_SECTOR" ]
    then
        echo "When conv=sync,noerror not used, error message fails to account for skip= offset"
        exit 7
    fi
}

function driveend
{
    NUM=500
    skip=`expr $BAD_SECTORS - $NUM`

    for b in "bs=512" "bs=32768" ""
    do
        for c in "conv=sync,noerror" ""
        do
            for i in "iflag=direct" ""
            do
                rm -f "$TEST_LOG"

                "$DC3DD" if="$BAD" of=/dev/null hash=md5 hashconv=after skip=$skip $c $i $b log="$TEST_LOG"
                if [ $? -ne 0 ]
                then
                    echo "acquiring end of drive failed, options used: $c $i $b"
                    exit 8
                fi

                LOG_SECTORS_IN=`grep sectors\ in "$TEST_LOG" | awk '{print $1}' | cut -f1 -d+`
                LOG_SECTORS_OUT=`grep sectors\ out "$TEST_LOG" | awk '{print $1}' | cut -f1 -d+`
                LOG_SUM=`grep "md5 TOTAL" "$TEST_LOG" | cut -f 3 -d \ `

                if [ "$b" == "" -a "$c" == "" -a "$i" == "iflag=direct" ]
                then
                    # don't care what happens
                    echo -n '' # TODO: better no-op
                elif [ "$b" == "bs=32768" -a "$c" == "" -a "$i" == "iflag=direct" ]
                then
                    # don't care what happens
                    echo -n '' # TODO: better no-op
                else
                    # better get $NUM sectors!
                    if [ "$LOG_SECTORS_IN" != "$NUM" ]
                    then
                        echo "acquiring end of drive, got wrong number of sectors in"
                        exit 8
                    fi

                    if [ "$LOG_SECTORS_OUT" != "$NUM" ]
                    then
                        echo "acquiring end of drive, got wrong number of sectors out"
                        exit 8
                    fi

                    if [ "$LOG_SUM" != "93a6f3f06d6c7f2f5de363e9136140cd" ]
                    then
                        echo "acquiring end of drive, got wrong hash"
                        exit 8
                    fi
                fi

            done
        done
    done
}

good
bad
bad2
zeros
pattern
messages
regressions
regressions_join
driveend
cleanup

