/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%                H   H   AAA   SSSSS  H   H  M   M   AAA   PPPP               %
%                H   H  A   A  SS     H   H  MM MM  A   A  P   P              %
%                HHHHH  AAAAA   SSS   HHHHH  M M M  AAAAA  PPPP               %
%                H   H  A   A     SS  H   H  M   M  A   A  P                  %
%                H   H  A   A  SSSSS  H   H  M   M  A   A  P                  %
%                                                                             %
%                                                                             %
%                  ImageMagick Hash-map and Linked-list Methods               %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                               December 2002                                 %
%                                                                             %
%                                                                             %
%  Copyright (C) 2003 ImageMagick Studio, a non-profit organization dedicated %
%  to making software imaging solutions freely available.                     %
%                                                                             %
%  Permission is hereby granted, free of charge, to any person obtaining a    %
%  copy of this software and associated documentation files ("ImageMagick"),  %
%  to deal in ImageMagick without restriction, including without limitation   %
%  the rights to use, copy, modify, merge, publish, distribute, sublicense,   %
%  and/or sell copies of ImageMagick, and to permit persons to whom the       %
%  ImageMagick is furnished to do so, subject to the following conditions:    %
%                                                                             %
%  The above copyright notice and this permission notice shall be included in %
%  all copies or substantial portions of ImageMagick.                         %
%                                                                             %
%  The software is provided "as is", without warranty of any kind, express or %
%  implied, including but not limited to the warranties of merchantability,   %
%  fitness for a particular purpose and noninfringement.  In no event shall   %
%  ImageMagick Studio be liable for any claim, damages or other liability,    %
%  whether in an action of contract, tort or otherwise, arising from, out of  %
%  or in connection with ImageMagick or the use or other dealings in          %
%  ImageMagick.                                                               %
%                                                                             %
%  Except as contained in this notice, the name of the ImageMagick Studio     %
%  shall not be used in advertising or otherwise to promote the sale, use or  %
%  other dealings in ImageMagick without prior written authorization from the %
%  ImageMagick Studio.                                                        %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  This module implements the standard handy hash and linked-list methods for
%  storing and retrieving large numbers of data elements.  It is loosely based
%  on the Java implementation of these algorithms.
%
*/

/*
  Include declarations.
*/
#include "magick/studio.h"
#include "magick/error.h"
#include "magick/hashmap.h"
#include "magick/semaphore.h"
#include "magick/signature.h"
#include "magick/utility.h"

/*
  Typedef declarations.
*/
typedef struct _ElementInfo
{
  void
    *data;

  struct _ElementInfo
    *next;
} ElementInfo;

typedef struct _EntryInfo
{
  size_t
    hash;

  void
    *key,
    *data;
} EntryInfo;

struct _HashMapInfo
{
  size_t
    (*hash)(const void *);

  void
    (*liberate_key)(void **),
    (*liberate_data)(void **);

  size_t
    maximum_entries,
    entries,
    next;

  unsigned int
    head_of_list;

  struct _LinkedListInfo
    **map;

  SemaphoreInfo
    *semaphore;

  unsigned long
    signature;
};

struct _LinkedListInfo
{
  size_t
    maximum_elements,
    elements;

  ElementInfo
    *head,
    *tail;

  SemaphoreInfo
    *semaphore;

  ElementInfo
    *next;

  unsigned long
    signature;
};

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   A c q u i r e H a s h M a p I n f o                                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  AcquireHashMap() returns a pointer to a HashMapInfo structure initialized
%  to default values.  The maximum number of elements that can be stored in the
%  hash is maximum elements squared.
%
%  The format of the AcquireHashMap method is:
%
%      HashMapInfo *AcquireHashMap(const size_t maximum_elements,
%        size_t (*hash)(const void *),void (*liberate_key)(void **),
%        void (*liberate_data)(void **))
%
%  A description of each parameter follows:
%
%    o maximum_elements: The maximum number elements in the hash-map;
%      typically SmallHashMapSize, MediumHashMapSize, or LargeHashSize.
%
%    o hash: The hash method, typically HashPointerType() or HashStringType().
%
%    o liberate_key: The key deallocation method, typically LiberateMemory(),
%      called whenever a key is removed from the hash-map.
%
%    o liberate_data: The data deallocation method;  typically LiberateMemory(),
%      called whenever a data object is removed from the hash-map.
%
%
*/
MagickExport HashMapInfo *AcquireHashMap(const size_t maximum_elements,
  size_t (*hash)(const void *),void (*liberate_key)(void **),
  void (*liberate_data)(void **))
{
  struct _HashMapInfo
    *hash_info;

  hash_info=(struct _HashMapInfo *) AcquireMemory(sizeof(*hash_info));
  if (hash_info == (struct _HashMapInfo *) NULL)
    return((HashMapInfo *) NULL);
  hash_info->hash=hash ? hash : HashPointerType;
  hash_info->liberate_key=liberate_key;
  hash_info->liberate_data=liberate_data;
  hash_info->entries=0;
  hash_info->maximum_entries=maximum_elements;
  hash_info->map=(struct _LinkedListInfo **)
    AcquireMemory(maximum_elements*sizeof(*hash_info->map));
  if (hash_info->map == (struct _LinkedListInfo **) NULL)
    {
      LiberateMemory((void **) &hash_info);
      return((HashMapInfo *) NULL);
    }
  memset(hash_info->map,0,maximum_elements*sizeof(*hash_info->map));
  hash_info->semaphore=(SemaphoreInfo *) NULL;
  hash_info->signature=MagickSignature;
  return(hash_info);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   A c q u i r e L i n k e d L i s t I n f o                                 %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  AcquireLinkedList() returns a pointer to a LinkedListInfo structure
%  initialized to default values.
%
%  The format of the Acquirestruct LinkedListInfomethod is:
%
%      LinkedListInfo *AcquireLinkedList(const size_t maximum_elements)
%
%  A description of each parameter follows:
%
%    o maximum_elements: The maximum number of elements in the list.
%
%
*/
MagickExport LinkedListInfo *AcquireLinkedList(const size_t maximum_elements)
{
  struct _LinkedListInfo
    *list_info;

  list_info=(LinkedListInfo *) AcquireMemory(sizeof(*list_info));
  if (list_info == (LinkedListInfo *) NULL)
    return((void *) NULL);
  list_info->maximum_elements=maximum_elements == 0 ? (~0) : maximum_elements;
  list_info->elements=0;
  list_info->head=(ElementInfo *) NULL;
  list_info->tail=(ElementInfo *) NULL;
  list_info->next=(ElementInfo *) NULL;
  list_info->semaphore=(SemaphoreInfo *) NULL;
  list_info->signature=MagickSignature;
  return(list_info);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   A p p e n d E l e m e n t T o L i n k e d L i s t                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  AppendElementToLinkedList() appends an element to the end of the linked-list.
%
%  The format of the AppendElementToLinkedList method is:
%
%      unsigned int AppendElementToLinkedList(LinkedListInfo *list_info,
%        void *data)
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
%    o data: The data.
%
%
*/
MagickExport unsigned int AppendElementToLinkedList(LinkedListInfo *list_info,
  void *data)
{
  register ElementInfo
    *next;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->elements == list_info->maximum_elements)
    return(False);
  next=(ElementInfo *) AcquireMemory(sizeof(*next));
  if (next == (ElementInfo *) NULL)
    return(False);
  next->data=data;
  next->next=(ElementInfo *) NULL;
  AcquireSemaphoreInfo(&list_info->semaphore);
  if (list_info->next == (ElementInfo *) NULL)
    list_info->next=next;
  if (list_info->elements == 0)
    list_info->head=next;
  else
    list_info->tail->next=next;
  list_info->tail=next;
  list_info->elements++;
  LiberateSemaphoreInfo(&list_info->semaphore);
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   C l e a r L i n k e d L i s t                                             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ClearLinkedList() clears all the elements from the linked-list.
%
%  The format of the ClearLinkedList method is:
%
%      unsigned int ClearLinkedList(LinkedListInfo *list_info,
%        void (*liberate_data)(void **))
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
%    o liberate_data: The data deallocation method; typically LiberateMemory().
%
%
*/
MagickExport void ClearLinkedList(LinkedListInfo *list_info,
  void (*liberate_data)(void **))
{
  ElementInfo
    *element;

  register ElementInfo
    *next;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  AcquireSemaphoreInfo(&list_info->semaphore);
  next=list_info->head;
  while (next != (ElementInfo *) NULL)
  {
    if (liberate_data != (void (*)(void **)) NULL)
      liberate_data((void **) &next->data);
    element=next;
    next=next->next;
    LiberateMemory((void **) &element);
  }
  list_info->head=(ElementInfo *) NULL;
  list_info->tail=(ElementInfo *) NULL;
  list_info->next=(ElementInfo *) NULL;
  list_info->elements=0;
  LiberateSemaphoreInfo(&list_info->semaphore);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   D e s t r o y H a s h M a p                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DestroyHashMap() frees the hash-map and all associated resources.
%
%  The format of the DestroyHashMap method is:
%
%      DestroyHashMap(struct _HashMapInfo *hash_info)
%
%  A description of each parameter follows:
%
%    o hash_info: The hash info.
%
%
*/
MagickExport void DestroyHashMap(HashMapInfo *hash_info)
{
  register EntryInfo
    *entry;

  register long
    i;

  struct _LinkedListInfo
    *list_info;

  assert(hash_info != (HashMapInfo *) NULL);
  assert(hash_info->signature == MagickSignature);
  AcquireSemaphoreInfo(&hash_info->semaphore);
  for (i=0; i < hash_info->maximum_entries; i++)
  {
    list_info=hash_info->map[i];
    if (list_info != (LinkedListInfo *) NULL)
      {
        ResetLinkedListIterator(list_info);
        entry=GetNextElementInLinkedList(list_info);
        while (entry != (EntryInfo *) NULL)
        {
          if (hash_info->liberate_key != (void (*)(void **)) NULL)
            hash_info->liberate_key((void **) &entry->key);
          if (hash_info->liberate_data != (void (*)(void **)) NULL)
            hash_info->liberate_data((void **) &entry->data);
          entry=GetNextElementInLinkedList(list_info);
        }
      }
    if (list_info != (LinkedListInfo *) NULL)
      DestroyLinkedList(list_info,LiberateMemory);
  }
  DestroySemaphoreInfo(&hash_info->semaphore);
  LiberateMemory((void **) &hash_info->map);
  LiberateMemory((void **) &hash_info);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   D e s t r o y L i n k e d L i s t                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  DestroyLinkedList() frees the linked-list and all associated resources.
%
%  The format of the DestroyLinkedList method is:
%
%      DestroyLinkedList(LinkedListInfo *list_info,
%        void (*liberate_data)(void **))
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
%    o liberate_data: The data deallocation method;  typically
%      LiberateMemory().
%
%
*/
MagickExport void DestroyLinkedList(LinkedListInfo *list_info,
  void (*liberate_data)(void **))
{
  ElementInfo
    *entry;

  register ElementInfo
    *next;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  AcquireSemaphoreInfo(&list_info->semaphore);
  next=list_info->head;
  while (next != (ElementInfo *) NULL)
  {
    if (liberate_data != (void (*)(void **)) NULL)
      liberate_data((void **) &next->data);
    entry=next;
    next=next->next;
    LiberateMemory((void **) &entry);
  }
  DestroySemaphoreInfo(&list_info->semaphore);
  LiberateMemory((void **) &list_info);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t E n t r y F r o m H a s h M a p                                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetEntryFromHashMap() gets an entry from the hash-map by its key.
%
%  The format of the GetEntryFromHashMap method is:
%
%      unsigned int GetEntryFromHashMap(const HashMapInfo *hash_info,void *key)
%
%  A description of each parameter follows:
%
%    o hash_info: The hash info.
%
%    o key: The key.
%
%
*/
MagickExport void *GetEntryFromHashMap(const HashMapInfo *hash_info,
  const void *key)
{
  struct _LinkedListInfo
    *list_info;

  register EntryInfo
    *entry;

  size_t
    hash;

  assert(hash_info != (HashMapInfo *) NULL);
  assert(hash_info->signature == MagickSignature);
  if (key == NULL)
    return((void *) NULL);
  hash=hash_info->hash(key);
  list_info=hash_info->map[hash % hash_info->maximum_entries];
  if (list_info != (LinkedListInfo *) NULL)
    {
      ResetLinkedListIterator(list_info);
      entry=GetNextElementInLinkedList(list_info);
      while (entry != (EntryInfo *) NULL)
      {
        if (entry->hash == hash)
          return(entry->data);
        entry=GetNextElementInLinkedList(list_info);
      }
    }
  return((void *) NULL);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t E l e m e n t F r o m L i n k e d L i s t                           %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetElementInLinkedList() gets an element from the linked-list by the
%  specified location.
%
%  The format of the GetElementInLinkedList method is:
%
%      unsigned int GetElementInLinkedList(const LinkedListInfo *list_info,
%        size_t index)
%
%  A description of each parameter follows:
%
%    o list_info: The linked_list info.
%
%    o index: The list index.
%
%
*/
MagickExport void *GetElementFromLinkedList(const LinkedListInfo *list_info,
  size_t index)
{
  register ElementInfo
    *next;

  register long
    i;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (index >= list_info->elements)
    return((void *) NULL);
  if (index == 0)
    return(list_info->head->data);
  if (index == (list_info->elements-1))
    return(list_info->tail->data);
  next=list_info->head;
  for (i=0; i < index; i++)
    next=next->next;
  return(next->data);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t L a s t E n t r y I n L i n k e d L i s t                           %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetLastElementInLinkedList() gets the last element in the linked-list.
%
%  The format of the GetLastElementInLinkedList method is:
%
%      void *GetLastElementInLinkedList(const LinkedListInfo *list_info)
%
%  A description of each parameter follows:
%
%    o list_info: The linked_list info.
%
%
*/
MagickExport void *GetLastElementInLinkedList(const LinkedListInfo *list_info)
{
  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->elements == 0)
    return((void *) NULL);
  return(list_info->tail->data);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t N e x t E n t r y I n H a s h M a p                                 %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetNextEntryInHashMap() gets the next entry in the hash-map.
%
%  The format of the GetNextEntryInHashMap method is:
%
%      unsigned int GetNextEntryInHashMap(HashMapInfo *hash_info,void *key)
%
%  A description of each parameter follows:
%
%    o hash_info: The hash info.
%
%    o key: The key.
%
%
*/
MagickExport void *GetNextEntryInHashMap(HashMapInfo *hash_info)
{
  register EntryInfo
    *entry;

  struct _LinkedListInfo
    *list_info;

  assert(hash_info != (HashMapInfo *) NULL);
  assert(hash_info->signature == MagickSignature);
  while (hash_info->next < hash_info->maximum_entries)
  {
    list_info=hash_info->map[hash_info->next];
    if (list_info != (LinkedListInfo *) NULL)
      {
        if (hash_info->head_of_list == False)
          {
            ResetLinkedListIterator(list_info);
            hash_info->head_of_list=True;
          }
      entry=GetNextElementInLinkedList(list_info);
      if (entry != (EntryInfo *) NULL)
        return(entry->key);
      hash_info->head_of_list=False;
    }
    hash_info->next++;
  }
  return((void *) NULL);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t N e x t E l e m e n t I n L i n k e d L i s t                       %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetNextElementInLinkedList() gets the next element in the linked-list.
%
%  The format of the GetNextElementInLinkedList method is:
%
%      unsigned int GetNextElementInLinkedList(LinkedListInfo *list_info)
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
%
*/
MagickExport void *GetNextElementInLinkedList(LinkedListInfo *list_info)
{
  void
    *data;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->next == (ElementInfo *) NULL)
    return((void *) NULL);
  data=list_info->next->data;
  list_info->next=list_info->next->next;
  return(data);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t N u m b e r O f E n t r i e s I n H a s h M a p                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetNumberOfEntriesInHashMap() returns the number of entries in the hash-map.
%
%  The format of the GetNumberOfEntriesInHashMap method is:
%
%      size_t GetNumberOfEntriesInHashMap(const HashMapInfo *hash_info)
%
%  A description of each parameter follows:
%
%    o hash_info: The hash info.
%
%
*/
MagickExport size_t GetNumberOfEntriesInHashMap(const HashMapInfo *hash_info)
{
  assert(hash_info != (HashMapInfo *) NULL);
  assert(hash_info->signature == MagickSignature);
  return(hash_info->entries);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   G e t N u m b e r O f E l e m e n t s I n L i n k e d L i s t             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  GetNumberOfElementsInLinkedList() returns the number of entries in the
%  linked-list.
%
%  The format of the GetNumberOfElementsInLinkedList method is:
%
%      size_t GetNumberOfElementsInLinkedList(const LinkedListInfo *list_info)
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
%
*/
MagickExport size_t GetNumberOfElementsInLinkedList(
  const LinkedListInfo *list_info)
{
  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  return(list_info->elements);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   I n s e r t E l e m e n t I n L i n k e d L i s t                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  InsertElementInLinkedList() inserts an element in the linked-list at the
%  specified location.
%
%  The format of the InsertElementInLinkedList method is:
%
%      unsigned int InsertElementInLinkedList(ListInfo *list_info,size_t index,
%        void *data)
%
%  A description of each parameter follows:
%
%    o list_info: The hash info.
%
%    o index: The index.
%
%    o data: The data.
%
%
*/
MagickExport unsigned int InsertElementInLinkedList(LinkedListInfo *list_info,
  size_t index,void *data)
{
  register ElementInfo
    *next;

  register long
    i;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (data == NULL)
    return(False);
  if ((index > list_info->elements) ||
      (list_info->elements == list_info->maximum_elements))
    return(False);
  next=(ElementInfo *) AcquireMemory(sizeof(*next));
  if (next == (ElementInfo *) NULL)
    return(False);
  next->data=data;
  next->next=(ElementInfo *) NULL;
  AcquireSemaphoreInfo(&list_info->semaphore);
  if (list_info->elements == 0)
    {
      if (list_info->next == (ElementInfo *) NULL)
        list_info->next=next;
      list_info->head=next;
      list_info->tail=next;
    }
  else
    {
      if (index == 0)
        {
          if (list_info->next == list_info->head)
            list_info->next=next;
           next->next=list_info->head;
           list_info->head=next;
        }
      else
        if (index == list_info->elements)
          {
            if (list_info->next == (ElementInfo *) NULL)
              list_info->next=next;
             list_info->tail->next=next;
             list_info->tail=next;
          }
        else
          {
            ElementInfo
              *element;

            element=list_info->head;
            next->next=element->next;
            for (i=1; i < index; i++)
            {
              element=element->next;
              next->next=element->next;
            }
            next=next->next;
            element->next=next;
            if (list_info->next == next->next)
              list_info->next=next;
          }
    }
  list_info->elements++;
  LiberateSemaphoreInfo(&list_info->semaphore);
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   I n s e r t E l e m e n t I n S o r t e d L i n k e d L i s t             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  InsertElementInSortedLinkedList() inserts an element in the sorted
%  linked-list.
%
%  The format of the InsertElementInSortedLinkedList method is:
%
%      unsigned int InsertElementInLinkedSortedList(ListInfo *list_info,
%        int (*compare)(const void *, const void *),void **replace,void *data)
%
%  A description of each parameter follows:
%
%    o list_info: The hash info.
%
%    o index: The index.
%
%    o compare: The compare method.
%
%    o replace: return previous element here.
%
%    o data: The data.
%
%
*/
MagickExport unsigned int InsertElementInSortedLinkedList(
  LinkedListInfo *list_info,int (*compare)(const void *, const void *),
  void **replace,void *data)
{
  ElementInfo
    *element;

  register ElementInfo
    *next;

  register long
    i;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if ((compare == NULL) || (data == NULL))
    return(False);
  if (list_info->elements == list_info->maximum_elements)
    return(False);
  next=(ElementInfo *) AcquireMemory(sizeof(*next));
  if (next == (ElementInfo *) NULL)
    return(False);
  next->data=data;
  element=(ElementInfo *) NULL;
  AcquireSemaphoreInfo(&list_info->semaphore);
  next->next=list_info->head;
  while (next->next != (ElementInfo *) NULL)
  {
    i=compare(data,next->next->data);
    if ((i < 0) || ((replace != (void **) NULL) && (i == 0)))
      {
        if (i == 0)
          {
            *replace=next->next->data;
            next->next=next->next->next;
            if (element != (ElementInfo *) NULL)
              LiberateMemory((void **) &element->next);
            list_info->elements--;
          }
        if (element != (ElementInfo *) NULL)
          element->next=next;
        else
          list_info->head=next;
        break;
      }
    element=next->next;
    next->next=next->next->next;
  }
  if (next->next == (ElementInfo *) NULL)
    {
      if (element != (ElementInfo *) NULL)
        element->next=next;
      else
        list_info->head=next;
      list_info->tail=next;
    }
  list_info->elements++;
  LiberateSemaphoreInfo(&list_info->semaphore);
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   I s H a s h M a p E m p t y                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  IsHashMapEmpty() returns True if the hash-map is empty.
%
%  The format of the IsHashMapEmpty method is:
%
%      unsigned int IsHashMapEmpty(HashMapInfo *hash_info)
%
%  A description of each parameter follows:
%
%    o hash_info: The hash info.
%
%
*/
MagickExport unsigned int IsHashMapEmpty(HashMapInfo *hash_info)
{
  assert(hash_info != (HashMapInfo *) NULL);
  assert(hash_info->signature == MagickSignature);
  return(hash_info->entries == 0);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   I s L i n k e d L i s t E m p t y                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  IsLinkedListEmpty() returns True if the linked-list is empty.
%
%  The format of the IsLinkedListEmpty method is:
%
%      unsigned int IsLinkedListEmpty(LinkedListInfo *list_info)
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
%
*/
MagickExport unsigned int IsLinkedListEmpty(const LinkedListInfo *list_info)
{
  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  return(list_info->elements == 0);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   H a s h P o i n t e r T y p e                                             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Specify the HashPointerType() method in AcquireHashMap() to find an entry
%  in a hash-map based on the address of a pointer.
%
%  The format of the HashPointerType method is:
%
%      size_t HashPointerType(const void *pointer)
%
%  A description of each parameter follows:
%
%    o pointer: compute the hash entry location from this pointer address.
%
%
*/
MagickExport size_t HashPointerType(const void *pointer)
{
  return((size_t) pointer);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   H a s h S t r i n g T y p e                                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  Specify the HashStringType() method in AcquireHashMap() to find an entry
%  in a hash-map based on the contents of a string.
%
%  The format of the HashStringType method is:
%
%      size_t HashStringType(const void *pointer)
%
%  A description of each parameter follows:
%
%    o pointer: compute the hash entry location from this string.
%
%
*/
MagickExport size_t HashStringType(const void *string)
{
  register long
    i;

  SignatureInfo
    signature_info;

  size_t
    hash;

  GetSignatureInfo(&signature_info);
  UpdateSignature(&signature_info,string,strlen(string));
  FinalizeSignature(&signature_info);
  hash=0;
  for (i=0; i < 8; i++)
    hash^=signature_info.digest[i];
  return(hash);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   L i n k e d L i s t T o A r r a y                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  LinkedListToArray() converts the linked-list to an array.
%
%  The format of the LinkedListToArray method is:
%
%      unsigned int LinkedListToArray(LinkedListInfo *list_info,void **array)
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
%    o array: The array.
%
%
*/
MagickExport unsigned int LinkedListToArray(LinkedListInfo *list_info,
  void **array)
{
  register ElementInfo
    *next;

  register long
    i;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (array == (void **) NULL)
    return(False);
  next=list_info->head;
  for (i=0; next != (ElementInfo *) NULL; i++)
  {
    array[i]=next->data;
    next=next->next;
  }
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   P u t E n t r y I n H a s h M a p                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  PutEntryInHashMap() puts an entry in the hash-map.  If the key hash
%  value already exists in the map it is first removed.
%
%  The format of the PutEntryInHashMap method is:
%
%      unsigned int PutEntryInHashMap(HashMapInfo *hash_info,void *key,
%        void *data)
%
%  A description of each parameter follows:
%
%    o hash_info: The hash info.
%
%    o key: The key.
%
%    o data: The data.
%
%
*/
MagickExport unsigned int PutEntryInHashMap(HashMapInfo *hash_info,void *key,
  void *data)
{
  EntryInfo
    *entry,
    *next;

  register long
    i;

  struct _LinkedListInfo
    *list_info;

  assert(hash_info != (HashMapInfo *) NULL);
  assert(hash_info->signature == MagickSignature);
  if ((key == NULL) || (data == NULL))
    return(False);
  next=(EntryInfo *) AcquireMemory(sizeof(*next));
  if (next == (EntryInfo *) NULL)
    return(False);
  AcquireSemaphoreInfo(&hash_info->semaphore);
  next->hash=hash_info->hash(key);
  next->key=key;
  next->data=data;
  list_info=hash_info->map[next->hash % hash_info->maximum_entries];
  if (list_info == (LinkedListInfo *) NULL)
    {
      list_info=AcquireLinkedList(hash_info->maximum_entries);
      if (list_info == (LinkedListInfo *) NULL)
        {
          LiberateMemory((void **) &next);
          LiberateSemaphoreInfo(&hash_info->semaphore);
          return(False);
        }
      hash_info->map[next->hash % hash_info->maximum_entries]=list_info;
    }
  else
    {
      ResetLinkedListIterator(list_info);
      entry=GetNextElementInLinkedList(list_info);
      for (i=0; entry != (EntryInfo *) NULL; i++)
      {
        if (entry->hash == next->hash)
          {
            RemoveElementFromLinkedList(list_info,i);
            if (hash_info->liberate_key != (void (*)(void **)) NULL)
              hash_info->liberate_key((void **) &entry->key);
            if (hash_info->liberate_data != (void (*)(void **)) NULL)
              hash_info->liberate_data((void **) &entry->data);
            LiberateMemory((void **) &entry);
            break;
          }
        entry=GetNextElementInLinkedList(list_info);
      }
    }
  if (!InsertElementInLinkedList(list_info,0,next))
    {
      LiberateMemory((void **) &next);
      LiberateSemaphoreInfo(&hash_info->semaphore);
      return(False);
    }
  hash_info->entries++;
  LiberateSemaphoreInfo(&hash_info->semaphore);
  return(True);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e m o v e E n t r y B y V a l u e F r o m L i n k e d L i s t           %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  RemoveElementByValueFromLinkedList() removes an element from the linked-list
%  by value.
%
%  The format of the RemoveElementByValueFromLinkedList method is:
%
%      void *RemoveElementByValueFromLinkedList(LinkedListInfo *list_info,
%        void *data)
%
%  A description of each parameter follows:
%
%    o list_info: The list info.
%
%    o data: The data.
%
%
*/
MagickExport void *RemoveElementByValueFromLinkedList(LinkedListInfo *list_info,
  void *data)
{
  ElementInfo
    *next;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if ((list_info->elements == 0) || (data == NULL))
    return((void *) NULL);
  AcquireSemaphoreInfo(&list_info->semaphore);
  if (data == list_info->head->data)
    {
      if (list_info->next == list_info->head)
        list_info->next=list_info->head->next;
      next=list_info->head;
      list_info->head=list_info->head->next;
      LiberateMemory((void **) &next);
    }
  else
    {
      ElementInfo
        *element;

      next=list_info->head;
      while ((next->next != (ElementInfo *) NULL) &&
             (next->next->data != data))
        next=next->next;
      if (next->next == (ElementInfo *) NULL)
        {
          LiberateSemaphoreInfo(&list_info->semaphore);
          return((void *) NULL);
        }
      element=next->next;
      next->next=element->next;
      if (element == list_info->tail)
        list_info->tail=next;
      if (list_info->next == element)
        list_info->next=element->next;
      LiberateMemory((void **) &element);
    }
  list_info->elements--;
  LiberateSemaphoreInfo(&list_info->semaphore);
  return(data);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e m o v e E l e m e n t F r o m L i n k e d L i s t                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  RemoveElementFromLinkedList() removes an element from the linked-list at the
%  specified location.
%
%  The format of the RemoveElementFromLinkedList method is:
%
%      void *RemoveElementFromLinkedList(LinkedListInfo *list_info,size_t index)
%
%  A description of each parameter follows:
%
%    o list_info: The linkd-list info.
%
%    o index: The index.
%
%
*/
MagickExport void *RemoveElementFromLinkedList(LinkedListInfo *list_info,
  size_t index)
{
  ElementInfo
    *next;

  register long
    i;

  void
    *entry;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (index >= list_info->elements)
    return((void *) NULL);
  AcquireSemaphoreInfo(&list_info->semaphore);
  if (index == 0)
    {
      if (list_info->next == list_info->head)
        list_info->next=list_info->head->next;
      entry=list_info->head->data;
      next=list_info->head;
      list_info->head=list_info->head->next;
      LiberateMemory((void **) &next);
    }
  else
    {
      ElementInfo
        *element;

      next=list_info->head;
      for (i=1; i < index; i++)
        next=next->next;
      element=next->next;
      next->next=element->next;
      if (element == list_info->tail)
        list_info->tail=next;
      if (list_info->next == element)
        list_info->next=element->next;
      entry=element->data;
      LiberateMemory((void **) &element);
    }
  list_info->elements--;
  LiberateSemaphoreInfo(&list_info->semaphore);
  return(entry);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e m o v e E n t r y F r o m H a s h M a p                               %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  RemoveEntryFromHashMap() removes an entry from the hash-map by its key.
%
%  The format of the RemoveEntryFromHashMap method is:
%
%      unsigned int RemoveEntryFromHashMap(HashMapInfo *hash_info,void *key)
%
%  A description of each parameter follows:
%
%    o hash_info: The hash info.
%
%    o key: The key.
%
%
*/
MagickExport void *RemoveEntryFromHashMap(HashMapInfo *hash_info,const void *key)
{
  EntryInfo
    *entry;

  register long
    i;

  size_t
    hash;

  struct _LinkedListInfo
    *list_info;

  void
    *data;

  assert(hash_info != (HashMapInfo *) NULL);
  assert(hash_info->signature == MagickSignature);
  if (key == NULL)
    return((void *) NULL);
  AcquireSemaphoreInfo(&hash_info->semaphore);
  hash=hash_info->hash(key);
  list_info=hash_info->map[hash % hash_info->maximum_entries];
  if (list_info != (LinkedListInfo *) NULL)
    {
      ResetLinkedListIterator(list_info);
      entry=GetNextElementInLinkedList(list_info);
      for (i=0; entry != (EntryInfo *) NULL; i++)
      {
        if (entry->hash == hash)
          {
            entry=RemoveElementFromLinkedList(list_info,i);
            if (entry == (EntryInfo *) NULL)
              {
                LiberateSemaphoreInfo(&hash_info->semaphore);
                return((void *) NULL);
              }
            if (hash_info->liberate_key != (void (*)(void **)) NULL)
              hash_info->liberate_key((void **) &entry->key);
            data=entry->data;
            LiberateMemory((void **) &entry);
            hash_info->entries--;
            LiberateSemaphoreInfo(&hash_info->semaphore);
            return(data);
          }
        entry=GetNextElementInLinkedList(list_info);
      }
    }
  LiberateSemaphoreInfo(&hash_info->semaphore);
  return((void *) NULL);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e m o v e L a s t E l e m e n t F r o m L i n k e d L i s t             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  RemoveLastElementFromLinkedList() removes the last element from the
%  linked-list.
%
%  The format of the RemoveLastElementFromLinkedList method is:
%
%      void *RemoveLastElementFromLinkedList(LinkedListInfo *list_info)
%
%  A description of each parameter follows:
%
%    o list_info: The linkd-list info.
%
%
*/
MagickExport void *RemoveLastElementFromLinkedList(LinkedListInfo *list_info)
{
  void
    *entry;

  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  if (list_info->elements == 0)
    return((void *) NULL);
  AcquireSemaphoreInfo(&list_info->semaphore);
  if (list_info->next == list_info->tail)
    list_info->next=(ElementInfo *) NULL;
  if (list_info->elements == 1)
    {
      entry=list_info->head->data;
      list_info->head=(ElementInfo *) NULL;
      list_info->tail=(ElementInfo *) NULL;
    }
  else
    {
      ElementInfo
        *next;

      entry=list_info->tail->data;
      next=list_info->head;
      while (next->next != list_info->tail)
        next=next->next;
      LiberateMemory((void **) &list_info->tail);
      list_info->tail=next;
      next->next=(ElementInfo *) NULL;
    }
  list_info->elements--;
  LiberateSemaphoreInfo(&list_info->semaphore);
  return(entry);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e s e t H a s h M a p I t e r a t o r                                   %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ResetHashMapIterator() resets the hash-map iterator.  Use it in conjunction
%  with GetNextEntryInHashMap() to iterate over all the keys in the hash-map.
%
%  The format of the ResetHashMapIterator method is:
%
%      ResetHashMapIterator(HashMapInfo *hash_info)
%
%  A description of each parameter follows:
%
%    o hash_info: The hash info.
%
%
*/
MagickExport void ResetHashMapIterator(HashMapInfo *hash_info)
{
  assert(hash_info != (HashMapInfo *) NULL);
  assert(hash_info->signature == MagickSignature);
  hash_info->next=0;
  hash_info->head_of_list=False;
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   R e s e t L i n k e d L i s t I t e r a t o r                             %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ResetLinkedListIterator() resets the lined-list iterator.  Use it in
%  conjunction with GetNextElementInLinkedList() to iterate over all the values
%  in the linked-list.
%
%  The format of the ResetLinkedListIterator method is:
%
%      ResetLinkedListIterator(LinkedListInfo *list_info)
%
%  A description of each parameter follows:
%
%    o list_info: The linked-list info.
%
%
*/
MagickExport void ResetLinkedListIterator(LinkedListInfo *list_info)
{
  assert(list_info != (LinkedListInfo *) NULL);
  assert(list_info->signature == MagickSignature);
  list_info->next=list_info->head;
}
