/* GnomeICU
 * Copyright (C) 1998-2002 Jeremy Wise
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


/*
 * Functions for SNAC family 0x13, server side contacts list management
 * first created by Patrick Sung (2002)
 */


#include "common.h"
#include "auth.h"
#include "cl_migrate.h"
#include "gnomecfg.h"
#include "gnomeicu.h"
#include "groups.h"
#include "response.h"
#include "showlist.h"
#include "util.h"
#include "v7recv.h"
#include "v7send.h"
#include "v7snac13.h"

#define S13_REQID(id) ( ((++s13_reqcnt) << 16) | (id) )
#define SNAC13_SEND(d,l,type,reqid) \
  snac_send (mainconnection, (d), (l), FAMILY_13_SAVED_LIST, \
             (type), NULL, (reqid))

static guint s13_reqid = 0;
static guint s13_reqcnt = 0;

/* structure holding infos in an add uin trancaction
 *
 * uins are added without asking for auth for the first time
 * if fails gnomeicu will try to add again but with auth req
 * This struct holds the data that is needed in case the first try fails.
 */
struct _add_uin_info {
  UIN_T uin;
  gchar nick[20]; /* FIXME */
  guint groupid;
  guint uinid;
  gboolean grant;
};

struct _invalid_record {
  gint len;
  gchar *tag;
  guint gid;
  guint uid;
  gint tlvlen;
  gchar *tlvval;
};
  
static struct _add_uin_info add_uin_info;
static struct _invalid_record invalid_record;

static void remove_tlv11_record (void);
static void snac13_construct_nickname (TLVstack *t, UIN_T uin, guint16 gid,
				       guint16 uid, const gchar *nick,
				       gboolean needauth);
static void snac13_construct_id_ingroup (TLVstack *t, WORD gid);
static void snac13_construct_visible (TLVstack *t, UIN_T uin, guint16 uid);
static void snac13_construct_invisible (TLVstack *t, UIN_T uin, guint16 uid);
static void snac13_construct_ignore (TLVstack *t, UIN_T uin, guint16 uid);
static void snac13_construct_status (TLVstack *t, gboolean status);
static void snac13_construct_authreq (TLVstack *t, UIN_T uin, const gchar *msg);

static void snac13_send_add_to_list (const TLVstack *tlvs);
static void snac13_send_update_list (const TLVstack *tlvs);
static void snac13_send_remove_from_list (const TLVstack *tlvs);
static void snac13_check_contacts_list_sanity (void);

/*
 * sends 0x13,0x02 and 0x13,0x04/0x13,0x05
 * request server side contacts list
 */
void
v7_request_contacts_list ()
{
  guchar stamp[6];

#ifdef TRACE_FUNCTION
  g_print("v7_requestcontactslist\n");
#endif

  /* request rights (0x13, 0x02) */
  SNAC13_SEND (NULL, 0, F13_CLIENT_REQ_RIGHTS, F13_CLIENT_REQ_RIGHTS);

  /* check pref setting, if server list pref not set, check from server*/
  if (!srvlist_exist) {
    s13_reqid = S13_REQID (F13_CLIENT_QUERY_SAVED_LIST);
    SNAC13_SEND (NULL, 0, F13_CLIENT_QUERY_SAVED_LIST, s13_reqid);
  } else {
    /* send request */
    DW_2_CharsBE (stamp, list_time_stamp);
    Word_2_CharsBE (&stamp[4], record_cnt);

    s13_reqid = S13_REQID (F13_CLIENT_REQ_SAVED_LIST);
    SNAC13_SEND (stamp, 6, F13_CLIENT_REQ_SAVED_LIST, s13_reqid);
  }
} /* end v7_request_contacts_list() */


/* read snac 0x13, 0x06 content
 * replace local contact list with the received one */
void
v7_read_contacts_list (Snac *snac, V7Connection *conn)
{
  static gint total_record = 0; /* total record count in user CL */
  static gboolean first_time = TRUE;
  gchar *here = snac->data;
  gchar *rec_str = NULL;
  gint rec_cnt = 0; /* number of CL record in current snac */
  gint i;
  gint len_str, gid, uid;
  TLV *tlv;
  gchar *nick;
  guint32 timestamp;
  GSList *contact;

  gboolean force_migration = FALSE;


#ifdef TRACE_FUNCTION
  g_print("v7_read_contact_list\n");
#endif

  Current_Status = STATUS_ONLINE;

  /* with this flag, we have to skip 8 bytes first */
  if (snac->flags[0] && 0x80)
    here += 8;

  here += 2;

  rec_cnt = *here;
  here++;

  total_record += rec_cnt;
  /*g_print ("total_record: %d.\n",total_record);*/

  /* reset all because we are reading them anyway */
  contact = Contacts;
  while (contact && first_time) {
    kontakt->vis_list = FALSE;
    kontakt->invis_list = FALSE;
    kontakt->ignore_list = FALSE;
    contact = contact->next;
  }

  first_time = FALSE;

  for (i=0; i < rec_cnt; i++) {
    /* read the BWS string on each CL record */
    len_str = CharsBE_2_Word (here);
    here += 2;
    if (len_str != 0) {
      rec_str = g_strndup (here, len_str);
      here += len_str;
    }

    /* read group-id and uin-id */
    gid = CharsBE_2_Word (here);
    here += 2;
    uid = CharsBE_2_Word (here);
    here += 2;


    /* read the TLV, find out the type */
    tlv = new_tlv (here);

    switch (tlv->type) {
      TLV *tlv2;

    case 0:  /* contains nickname, for user */
      len_str = CharsBE_2_Word (tlv->value + 2);
      nick = g_strndup (tlv->value+TLV_HEADER, len_str);
      /* check if auth flag is set */
      if (tlv->len > len_str+4) {
        if ( *(tlv->value + TLV_HEADER + len_str + 1) == 0x66 )
          /* NOTDONE */;  /* this uin is waiting for auth */
      }
      /* add uin to contact list */
      /* TODO: add uin to corresponding group */
      if ( (contact = Find_User (atoi (rec_str))) == NULL ) {
        contact = Add_User (atoi (rec_str), nick, TRUE);
      } else {
        len_str = (len_str > 19) ? 19 : len_str;
        strncpy (kontakt->nick, nick, len_str);
      }
      kontakt->uid = uid;
      kontakt->gid = gid;
      kontakt->last_status = STATUS_OFFLINE;
      g_free (nick);
      break;
    case 1:  /* contains id's (group or uin) */
      /* group id is not zero, contains uin id */
      if (gid) {
	groups_add (rec_str, gid);
      }

      /* no uin-id in this group, skip */
      if (tlv->len == 0)
	break;

      tlv2 = new_tlv (tlv->value);
      if (tlv2->type != 0xC8) {
	g_warning ("TLV(C8) expected: got TLV(0x%x).",tlv2->type);
      }
      delete_tlv (tlv2);
      break;
    case 2:  /* zero length; vis to uin */
      if ( (contact = Find_User (atoi (rec_str))) == NULL ) {
        contact = Add_User (atoi (rec_str), NULL, TRUE);
      }
      kontakt->vislist_uid = uid;
      kontakt->vis_list = TRUE;
      break;
    case 3:  /* zero length; inv to uin */
      if ( (contact = Find_User (atoi (rec_str))) == NULL ) {
        contact = Add_User (atoi (rec_str), NULL, TRUE);
      }
      kontakt->invlist_uid = uid;
      kontakt->invis_list = TRUE;
      break;
    case 4:  /* last online status (invis or not invis) */
      tlv2 = new_tlv (tlv->value);
      if (tlv2->type != 0xCA) {
        g_warning ("TLV(CA) expected in TLV(4): gto TLV(0x%x).",tlv2->type);
      } else {
        status_uid = uid;
        if (*(tlv2->value) == 0x03) {
          if (preset_status != STATUS_INVISIBLE)
            Current_Status = STATUS_INVISIBLE;
        } else
          Current_Status = preset_status;
      }
      delete_tlv (tlv2);
      break;
    case 9:  /* unknown meaning, ignore */
      break;
    case 0xE:  /* zero length; ignore to uin */
      if ( (contact = Find_User (atoi (rec_str))) == NULL ) {
        contact = Add_User (atoi (rec_str), NULL, TRUE);
      }
      kontakt->ignorelist_uid = uid;
      kontakt->ignore_list = TRUE;
      break;
    case 0x11: /* found when user used lite.icq.com before server list exists */
      if ((snac->reqid & 0xFF) == F13_CLIENT_QUERY_SAVED_LIST) {
        invalid_record.len = len_str;
        invalid_record.tag = g_strndup(rec_str, len_str);
        invalid_record.gid = gid;
        invalid_record.uid = uid;
        invalid_record.tlvlen = tlv->len;
        invalid_record.tlvval = g_strndup(tlv->value, tlv->len);
        force_migration = TRUE;
      }
      break;
    case 0x13:  /* CL first import time, not really useful to us, ignore */
      break;
    default:
      g_warning ("Unknow TLV from server contact list: type 0x%x.\n\n"
                 "Please send the following packet dump to the mailing-list "
                 "for debugging, thanks.\n\n",tlv->type);
      packet_print_v7_snac (snac);
      break;
    }

    here += TLV_HEADER + tlv->len;

    g_free (rec_str);
    rec_str = NULL;
    delete_tlv (tlv);
  } /* end for(rec_cnt) */

  /* done reading, time stamp remains, read it */
  timestamp = CharsBE_2_DW (here);

  /* no record found, and previous request is 'query contacts list'
     therefore we migrate */
  if (rec_cnt == 0 && (snac->reqid & 0xFF) == F13_CLIENT_QUERY_SAVED_LIST) {
    SNAC13_SEND (NULL, 0, F13_CLIENT_RDY_TO_USE_LIST,F13_CLIENT_RDY_TO_USE_LIST);
    v7_set_user_info (conn);
    v7_add_icbm (conn);
    v7_setstatus2 (Current_Status);
    v7_client_ready (conn);

    /* normal migration */
    g_idle_add (migration_dialog, (gpointer)FALSE);
    return;
  }

  /* done reading all contacts, need force migrate? */
  if (timestamp != 0 && force_migration) {
    SNAC13_SEND (NULL, 0, F13_CLIENT_RDY_TO_USE_LIST,F13_CLIENT_RDY_TO_USE_LIST);
    v7_set_user_info (conn);
    v7_add_icbm (conn);
    v7_setstatus2 (Current_Status);
    v7_client_ready (conn);

    remove_tlv11_record ();
    /* start the migration but use the existing "General" group in the list */
    g_idle_add (migration_dialog, (gpointer)TRUE);
    return;
  }

  /* cannot detect a case to start the migration, so do the default actions */

  if (timestamp != 0) {
    /* finished all reading */
    /* signal ready to use the contacts list */
    SNAC13_SEND (NULL, 0, F13_CLIENT_RDY_TO_USE_LIST,F13_CLIENT_RDY_TO_USE_LIST);
    v7_set_user_info (conn);
    v7_add_icbm (conn);
    v7_setstatus2 (Current_Status);
    v7_client_ready (conn);

    srvlist_exist = TRUE;
    list_time_stamp = timestamp;
    record_cnt = total_record;
    save_preferences ();
    Save_RC ();
  }

  /* update GUI */
  Show_Quick_Status_lower (UPDATE_RECONSTRUCT, NULL);

  /* check sanity of contacts list after reading it */
  snac13_check_contacts_list_sanity ();
}



/*
 * sned snac 0x13, 0x11
 * begin contacts list change
 */
void
v7_begin_CL_edit (gboolean migrate)
{
  gchar data[4] = "\x00\x01\x00\x00";

#ifdef TRACE_FUNCTION
  g_print("v7_begin_CL_edit\n");
#endif

  if (migrate)
    SNAC13_SEND (data, 4, F13_CLIENT_START_MOD_LIST, F13_CLIENT_START_MOD_LIST);
  else
    SNAC13_SEND (NULL, 0, F13_CLIENT_START_MOD_LIST, F13_CLIENT_START_MOD_LIST);
}

/*
 * send snac 0x13, 0x12
 * end contacts list change
 */
void
v7_end_CL_edit ()
{
#ifdef TRACE_FUNCTION
  g_print("v7_end_CL_edit\n");
#endif

  SNAC13_SEND (NULL, 0, F13_CLIENT_END_MOD_LIST, F13_CLIENT_END_MOD_LIST);
}


/* send the contact list (13,08)
 *  force - force sending contact, instead of wait until enough records
 * return:
 *  TRUE - if snac is sent
 *  FALSE - snac is not sent
 */
gboolean
v7_send_contacts_list (TLVstack *t, gboolean force)
{
  static gint rec_cnt = 0;

#ifdef TRACE_FUNCTION
  g_print("v7_send_contacts_list\n");
#endif

  rec_cnt++;

  if (force || rec_cnt >= 0x50) {
    rec_cnt = 0;

    if (t->len > 0)
      /* send the tlv stack */
      snac13_send_add_to_list (t);
    return TRUE;
  }

  return FALSE;
}


/* construct the contacts list */
void
v7_migrate_contacts_list ()
{
  GSList *contact;
  guint id, main_gid = 0;
  gint ignore_cnt = 0;
  gint vis_cnt = 0;
  gint inv_cnt = 0;
  TLVstack *tlvs = NULL;
  gboolean grp_exists;

#ifdef TRACE_FUNCTION
  g_print("v7_migrate_contacts_list\n");
#endif

  main_gid = groups_find_gid_by_name ("General");

  if (!main_gid) {
    main_gid = groups_gen_gid ();
    groups_add ("General", main_gid);
    grp_exists = FALSE;
  } else
    grp_exists = TRUE;

  tlvs = new_tlvstack (NULL, 0);

  contact = Contacts;
  /* add contacts first */
  while (contact) {
    /* skip ignore list */
    if (kontakt->ignore_list) {
      contact = contact->next;
      ignore_cnt++;
      continue;
    }

    /* count visible and invisible user */
    if (kontakt->vis_list) {
      vis_cnt++;
    } else if (kontakt->invis_list) {
      inv_cnt++;
    }

    id = contact_gen_uid ();

    snac13_construct_nickname (tlvs, kontakt->uin, main_gid, id, kontakt->nick,
			       FALSE);

    kontakt->gid = main_gid;
    kontakt->uid = id;

    contact = contact->next;

    if (v7_send_contacts_list (tlvs, FALSE)) {
      free_tlvstack (tlvs);
      tlvs = new_tlvstack (NULL, 0);
    }

  }

  /* send special users */
  contact = Contacts;
  while (contact && (ignore_cnt != 0 || vis_cnt != 0 || inv_cnt != 0)) {
    /* only ignore, vis, invis user */
    if (kontakt->ignore_list) {
      snac13_construct_ignore (tlvs, kontakt->uin, id = contact_gen_uid ());
      kontakt->ignorelist_uid = id;
      ignore_cnt--;
    } else if (kontakt->vis_list) {
      snac13_construct_visible (tlvs, kontakt->uin, id = contact_gen_uid ());
      kontakt->vislist_uid = id;
      vis_cnt--;
    } else if (kontakt->invis_list) {
      snac13_construct_invisible (tlvs, kontakt->uin, id =contact_gen_uid ());
      kontakt->invlist_uid = id;
      inv_cnt--;
    }

    contact = contact->next;

    if (v7_send_contacts_list (tlvs, FALSE)) {
      free_tlvstack (tlvs);
      tlvs = new_tlvstack (NULL, 0);
    }
  }

  v7_send_contacts_list (tlvs, TRUE);  /* force a send */
  free_tlvstack (tlvs);
  tlvs = new_tlvstack (NULL, 0);


  /* send the default group - "General" */
  snac13_construct_id_ingroup (tlvs, main_gid);
  v7_send_contacts_list (tlvs, TRUE);
  free_tlvstack (tlvs);
  tlvs = new_tlvstack (NULL, 0);

  if (!grp_exists) {
    /* send master group */
    snac13_construct_id_ingroup (tlvs, 0);
    v7_send_contacts_list (tlvs, TRUE);
    free_tlvstack (tlvs);
    tlvs = new_tlvstack (NULL, 0);

    /* send current status */
    snac13_construct_status (tlvs, TRUE);
    v7_send_contacts_list (tlvs, TRUE);
    free_tlvstack (tlvs);
    tlvs = new_tlvstack (NULL, 0);
  }

  add_uin_info.groupid = 0;
  add_uin_info.uinid = 0;

  free_tlvstack (tlvs);

} /* end v7_migrate_contacts_list () */


/* send snac 0x13,0x08
 * add a uin to the server contacts list
 *  - authtlv: TRUE add request authorization TLV(0x66)
 *             FALSE don't add the TLV(0x66)
 */
void
v7_addto_contacts_list (UIN_T uin, const gchar *nick,
			guint gid, gboolean needauth)
{
  TLVstack *tlvs;

#ifdef TRACE_FUNCTION
  g_print("v7_addto_contacts_list\n");
#endif

  v7_begin_CL_edit (FALSE);

  add_uin_info.uinid = contact_gen_uid ();

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_nickname (tlvs, uin, gid, add_uin_info.uinid, nick, needauth);
  snac13_send_add_to_list (tlvs);

  free_tlvstack (tlvs);
}

/*
 * send snac 0x13,0x14
 */
void
v7_send_grant_auth (UIN_T uin)
{
  TLVstack *tlvs;

  tlvs = new_tlvstack (NULL, 0);
  snac13_construct_authreq (tlvs, uin, NULL);

  SNAC13_SEND (tlvs->beg, tlvs->len, F13_CLIENT_AUTH_REQUEST_1,
               F13_CLIENT_AUTH_REQUEST_1);

  free_tlvstack (tlvs);
}

/*
 * send snac 0x13,0x18
 */
void
v7_send_auth_message (UIN_T uin, const gchar *authmsg)
{
  TLVstack *tlvs;

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_authreq (tlvs, uin, authmsg);
  
  SNAC13_SEND (tlvs->beg, tlvs->len, F13_CLIENT_AUTH_REQUEST_2,
               F13_CLIENT_AUTH_REQUEST_2);

  free_tlvstack (tlvs);
}

/*
 * read snac 0x13,0x0E
 * server ack for serveral snac 0x13 request
 */
void
v7_snac13_check_srv_ack (Snac *snac)
{
  guint reqid;
  gchar *here = snac->data;
  gchar state = 0;

  Contact_Member *contactdata;

#ifdef TRACE_FUNCTION
  g_print("v7_snac13_check_srv_ack\n");
#endif

  reqid = snac->reqid;

  /* with this flag, we have to skip 8 bytes first */
  if (snac->flags[0] && 0x80)
    here += 8;

  /* read the last byte */
  state = *(here + 1);

  if ((state & 0xFF) != 0x0E) { /* no error */
    if (reqid == s13_reqid) {
      s13_reqid = 0;
      switch (reqid & 0xff) {
      case F13_CLIENT_ADD_TO_LIST: /* 0x08 */ /* we are done */
	/* add to contacts list, update gui */
	contactdata = g_new0 (Contact_Member, 1);
	contactdata->uin = add_uin_info.uin;
	strncpy (contactdata->nick, add_uin_info.nick, 20);
	contactdata->status = STATUS_NA;
	contactdata->last_status = STATUS_OFFLINE;
	contactdata->show_again = TRUE;
	contactdata->info = g_new0 (USER_INFO_STRUCT, 1);
	contactdata->inlist = TRUE;
	contactdata->tcp_seq = -1;
	contactdata->confirmed = TRUE;
        contactdata->uid = add_uin_info.uinid;
        contactdata->gid = add_uin_info.groupid;
	Contacts = g_slist_append (Contacts, contactdata);

	/* update group list before finishing edit */
	if (add_uin_info.groupid == 0 || add_uin_info.uinid == 0)
	  break;
	v7_update_group_info (add_uin_info.groupid);
	v7_end_CL_edit ();

	Show_Quick_Status_lower (UPDATE_OFFLINE|UPDATE_ONLINE, NULL);

        add_uin_info.groupid = 0;
        add_uin_info.uinid = 0;
	break;
      case F13_CLIENT_UPDATE_LIST: /* 0x09 */
	break;
      case F13_CLIENT_REMOVE_FROM_LIST: /* 0x0A */
        if (add_uin_info.groupid == 0 || add_uin_info.uinid == 0)
          break;
	v7_update_group_info (add_uin_info.groupid);
	v7_end_CL_edit ();
        add_uin_info.groupid = 0;
        add_uin_info.uinid = 0;
	break;
      default:
	break;
      }
    }
  } else { /* error */
    if (reqid == s13_reqid) {
      s13_reqid = 0;
      switch (reqid & 0xff) {
      case F13_CLIENT_ADD_TO_LIST: /* 0x08 */ /* from add to contact */
	if (add_uin_info.groupid == 0 || add_uin_info.uinid == 0)
	  break;
	v7_end_CL_edit ();
	auth_request_msg_box (add_uin_info.nick, add_uin_info.uin);
        add_uin_info.groupid = 0;
        add_uin_info.uinid = 0;
	break;
      case F13_CLIENT_UPDATE_LIST: /* 0x09 */
	break;
      case F13_CLIENT_REMOVE_FROM_LIST: /* 0x0A */
	g_warning ("User remove failed.\n");
	break;
      default:
	break;
      }
    }
  }
} /* end void v7_snac13_check_srv_ack (Snac *snac) */


/*
 * send user status
 *
 * update stauts, TRUE = visible, FALSE = invisible
 */
void
v7_send_status_server (gboolean visible)
{
  TLVstack *tlvs;

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_status (tlvs, visible);
  snac13_send_update_list (tlvs);

  free_tlvstack (tlvs);
}

/*
 * send snac 0x13,0x09
 */
void
v7_update_nickname (guint gid, guint uid, UIN_T uin,
                    const gchar *nick, gint nick_len)
{
  TLVstack *tlvs;
  gchar *mynick;

  mynick = g_strndup (nick, nick_len);

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_nickname (tlvs, uin, gid, uid, nick, FALSE);
  snac13_send_update_list (tlvs);

  free_tlvstack (tlvs);
  g_free (mynick);
}

/*
 * send snac 0x13,0x09
 */
void
v7_update_group_info (guint gid)
{
  TLVstack *tlvs;

#ifdef TRACE_FUNCTION
  g_print("v7_update_group_info\n");
#endif

  tlvs = new_tlvstack (NULL, 0);

  /* FIXME */
  /* need to find a way to get the group name, when we have group support */
  snac13_construct_id_ingroup (tlvs, gid);
  snac13_send_update_list (tlvs);

  free_tlvstack (tlvs);
}

/* helper functions to add uin */

/* try to add user without using authorization */
void
v7_try_add_uin (UIN_T uin, const gchar *nick, guint gid, gboolean grant)
{
  if (grant) {
    v7_send_grant_auth (uin);
  }

  add_uin_info.uin = uin;

  strncpy (add_uin_info.nick, nick, 19); /* FIXME */
  add_uin_info.nick[19] = 0;

  add_uin_info.groupid = gid;
  add_uin_info.grant = grant;

  v7_addto_contacts_list (uin, nick, gid, FALSE); /* add without TLV(0x66) set */
}

void
v7_ask_uin_for_auth (gchar *authmsg)
{
  UIN_T uin;

  uin = add_uin_info.uin;

  v7_send_auth_message (uin, authmsg);
  if (add_uin_info.grant)
    v7_send_grant_auth (uin);

  v7_addto_contacts_list (uin, add_uin_info.nick, add_uin_info.groupid, TRUE);
}

/*
 * send snac 0x13,0x0A
 */
void
v7_remove_contact (const UIN_T uin)
{
  TLVstack *tlvs;
  GSList *contact;

#ifdef TRACE_FUNCTION
  g_print("v7_remove_contact\n");
#endif

  contact = Find_User(uin);
  
  v7_begin_CL_edit (FALSE);

  add_uin_info.groupid = kontakt->gid;
  add_uin_info.uinid = kontakt->uid;

  tlvs = new_tlvstack (NULL, 0);

  /* change to req_session_cookie for a better name, and make sure no leaks */
  strncpy (add_uin_info.nick, kontakt->nick, 19); /* FIXME */


  snac13_construct_nickname (tlvs, kontakt->uin, kontakt->gid, kontakt->uid,
                             kontakt->nick, FALSE);
  snac13_send_remove_from_list (tlvs);

  free_tlvstack (tlvs);

  Save_RC ();
}

#if 0
/*
 * send snac 0x13,0x09
 */
void
v7_remove_user_from_group (const UIN_T uin)
{
  TLVstack *tlvs;
  GSList *contact;
  WORD gid;

#ifdef TRACE_FUNCTION
  g_print("v7_remove_user_from_group\n");
#endif

  gid = groups_find_gid_by_name ("General");

  kontakt->gid = 0;

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_id_ingroup (tlvs, kontakt->gid);
  snac13_send_update_list (tlvs);

  free_tlvstack (tlvs);
}
#endif

/*
 * read snac 0x13,0x19
 */
void
v7_recv_auth_request (Snac *snac)
{
  gchar *here = snac->data;
  UIN_T uin = 0;
  gint str_len;
  gchar *str;

#ifdef TRACE_FUNCTION
  g_print("v7_recv_auth_request\n");
#endif

  /* with this flag, we have to skip 8 bytes first */
  if (snac->flags[0] && 0x80)
    here += 8;

  /* read requester uin */
  uin = strtol (here+1, NULL, 10);
  here = here + here[0] + 1;

  str_len = CharsBE_2_Word (here);
  here += 2;

  /* read the request msg */
  str = g_strndup (here, str_len);

  /* got all the info, now call the client auth handler */
  auth_receive_request (uin, str);

  g_free (str);
}

/*
 * send snac 0x13,0x1A
 */
void
v7_grant_auth_request (UIN_T uin)
{
  gint len;
  gchar *strtmp;
  TLVstack *t = NULL;

  t = new_tlvstack (NULL, 0);

  strtmp = g_strdup_printf ("?%d", uin);
  len = strlen (strtmp) - 1;
  strtmp[0] = (gchar)len;
  add_nontlv (t, strtmp, len+1);
  g_free (strtmp);

  add_nontlv (t, "\x1\0\0\0\0", 5);

  SNAC13_SEND (t->beg, t->len, F13_CLIENT_SEND_GRANT_AUTH,
               F13_CLIENT_SEND_GRANT_AUTH);

  free_tlvstack (t);
}

/*
 * visible list, add or remove a uin
 */
void
v7_visible (UIN_T uin, WORD luid, gboolean add)
{
  TLVstack *tlvs;

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_visible (tlvs, uin, luid);

  if (add) {
    add_uin_info.groupid = 0;
    add_uin_info.uinid = 0;
    snac13_send_add_to_list (tlvs);
  } else {
    add_uin_info.groupid = 0;
    add_uin_info.uinid = 0;
    snac13_send_remove_from_list (tlvs);
  }

  free_tlvstack (tlvs);
}

/*
 * invisible list, add or remove a uin
 */
void
v7_invisible (UIN_T uin, WORD luid, gboolean add)
{
  TLVstack *tlvs;

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_invisible (tlvs, uin, luid);

  if (add) {
    add_uin_info.groupid = 0;
    add_uin_info.uinid = 0;
    snac13_send_add_to_list (tlvs);
  } else {
    add_uin_info.groupid = 0;
    add_uin_info.uinid = 0;
    snac13_send_remove_from_list (tlvs);
  }

  free_tlvstack (tlvs);
}

/*
 * ignore list, add or remove a uin
 */
void
v7_ignore (UIN_T uin, WORD luid, gboolean add)
{
  TLVstack *tlvs;

  tlvs = new_tlvstack (NULL, 0);

  snac13_construct_ignore (tlvs, uin, luid);

  if (add) {
    add_uin_info.groupid = 0;
    add_uin_info.uinid = 0;
    snac13_send_add_to_list (tlvs);
  } else {
    add_uin_info.groupid = 0;
    add_uin_info.uinid = 0;
    snac13_send_remove_from_list (tlvs);
  }

  free_tlvstack (tlvs);
}

/* private functions */

/* remove the TLV(0x11) record */
void
remove_tlv11_record ()
{
  TLVstack *tlvs = NULL;

  tlvs = new_tlvstack (NULL, 0);

  add_nontlv_bws (tlvs, invalid_record.tag);

  add_nontlv_w_be (tlvs, invalid_record.gid);
  add_nontlv_w_be (tlvs, invalid_record.uid);

  add_tlv (tlvs, 0x11, invalid_record.tlvval, invalid_record.tlvlen);
  snac13_send_remove_from_list (tlvs);

  g_free (invalid_record.tag);
  g_free (invalid_record.tlvval);

  free_tlvstack (tlvs);
}

/* helpers for composing the snac packet */

/* construct TLV(0) (type 0) packet */
void
snac13_construct_nickname (TLVstack *t, UIN_T uin, guint16 gid, guint16 uid,
                           const gchar *nick, gboolean needauth)
{
  gchar *strtmp;
  TLVstack *tmptlv = NULL;

#ifdef TRACE_FUNCTION
  g_print("snac13_construct_nickname\n");
#endif

  /* uin */
  strtmp = g_strdup_printf ("%d", uin);
  add_nontlv_bws (t, strtmp);
  g_free (strtmp);
  
  /* group id, uin id */
  add_nontlv_w_be (t, gid);
  add_nontlv_w_be (t, uid);
  
  /* nickname */
  tmptlv = new_tlvstack (NULL, 0);

  add_tlv (tmptlv, 0x131, nick, strlen (nick));
  if (needauth) {
    add_tlv (tmptlv, 0x66, NULL, 0);
  }

  add_tlv (t, 0, tmptlv->beg, tmptlv->len);

  g_free (tmptlv);
}

void
snac13_construct_id_ingroup (TLVstack *t, WORD gid)
{
  TLVstack *idstlv, *ids;
  gchar *grpname;
  GSList *contact;
  GSList *ginfo;

#ifdef TRACE_FUNCTION
  g_print("snac13_construct_id_ingroup\n");
#endif

  grpname = groups_name_by_gid(gid);
  
  /* group name */
  add_nontlv_bws (t, grpname);

  /* group id, zero uin id */
  add_nontlv_w_be (t, gid);
  add_nontlv_w_be (t, 0);

  /* add uinid's */

  ids = new_tlvstack (NULL, 0);

  if (gid == 0) { /* master group */
    for (ginfo = Groups; ginfo != NULL; ginfo = ginfo->next)
      add_nontlv_w_be (ids, ((GroupInfo *)(ginfo->data))->gid);
  } else {
    /* Look up the contact list for members of the group and add them */
    for (contact = Contacts; contact != NULL; contact = contact->next) 
      if (kontakt->gid == gid)
        add_nontlv_w_be (ids, kontakt->uid);
  }

  idstlv = new_tlvstack (NULL, 0);

  add_tlv (idstlv, 0xC8, ids->beg, ids->len);

  free_tlvstack(ids);

  add_tlv (t, 0x1, idstlv->beg, idstlv->len);

  free_tlvstack (idstlv);
}

void
snac13_construct_visible (TLVstack *t, UIN_T uin, guint16 uid)
{
  gchar *strtmp;

#ifdef TRACE_FUNCTION
  g_print("snac13_construct_visible\n");
#endif

  /* uin */
  strtmp = g_strdup_printf ("%d", uin);
  add_nontlv_bws (t, strtmp);
  g_free (strtmp);

  /* zero gid, uin id */
  add_nontlv_w_be (t, 0);
  add_nontlv_w_be (t, uid);

  add_tlv (t, 0x02, NULL, 0);
}

void
snac13_construct_invisible (TLVstack *t, UIN_T uin, guint16 uid)
{
  gchar *strtmp;

#ifdef TRACE_FUNCTION
  g_print("snac13_construct_invisible\n");
#endif

  /* uin */
  strtmp = g_strdup_printf ("%d", uin);
  add_nontlv_bws (t, strtmp);
  g_free (strtmp);

  /* zero gid, uin id */
  add_nontlv_w_be (t, 0);
  add_nontlv_w_be (t, uid);

  add_tlv (t, 0x03, NULL, 0);
}

void
snac13_construct_ignore (TLVstack *t, UIN_T uin, guint16 uid)
{
  gchar *strtmp;

#ifdef TRACE_FUNCTION
  g_print("snac13_construct_ignore\n");
#endif

  /* uin */
  strtmp = g_strdup_printf ("%d", uin);
  add_nontlv_bws (t, strtmp);
  g_free (strtmp);

  /* zero gid, uin id */
  add_nontlv_w_be (t, 0);
  add_nontlv_w_be (t, uid);

  add_tlv (t, 0x0E, NULL, 0);
}

/*
 * status =
 *   TRUE: visible
 *   FALSE: invisible
 */
void
snac13_construct_status (TLVstack *t, gboolean status)
{
  TLVstack *sts;
  gchar buf;
  
#ifdef TRACE_FUNCTION
  g_print("snac13_construct_status\n");
#endif

  status_uid = status_uid ? status_uid : contact_gen_uid ();

  add_nontlv_bws (t, NULL);

  add_nontlv_w_be (t, 0);
  add_nontlv_w_be (t, status_uid);

  sts = new_tlvstack (NULL, 0);

  buf = status ? '\x4' : '\x3';

  add_tlv (sts, 0xCA, &buf, 1);
  add_tlv (t, 0x4, sts->beg, sts->len);

  free_tlvstack (sts);
}


void
snac13_construct_authreq (TLVstack *t, UIN_T uin, const gchar *msg)
{
  gint len;
  gchar *strtmp;

#ifdef TRACE_FUNCTION
  g_print("snac13_construct_authreq\n");
#endif

  /* uin */
  strtmp = g_strdup_printf ("?%d", uin);
  len = strlen (strtmp) - 1; /* length of the string uin */
  strtmp[0] = (guint8)len;
  add_nontlv (t, strtmp, len+1);
  g_free (strtmp);

  add_nontlv_bws (t, msg);

  add_nontlv (t, "\0\0", 2);
}

void
snac13_send_add_to_list (const TLVstack *t)
{
  s13_reqid = S13_REQID (F13_CLIENT_ADD_TO_LIST);
  SNAC13_SEND (t->beg, t->len, F13_CLIENT_ADD_TO_LIST, s13_reqid);
}

void
snac13_send_update_list (const TLVstack *t)
{
  s13_reqid = S13_REQID (F13_CLIENT_UPDATE_LIST);
  SNAC13_SEND (t->beg, t->len, F13_CLIENT_UPDATE_LIST, s13_reqid);
}

void
snac13_send_remove_from_list (const TLVstack *t)
{
  s13_reqid = S13_REQID (F13_CLIENT_REMOVE_FROM_LIST);
  SNAC13_SEND (t->beg, t->len, F13_CLIENT_REMOVE_FROM_LIST, s13_reqid);
}

void
snac13_check_contacts_list_sanity ()
{
  /* match group id from user records */
  /* not done */

  /* check status_uid record, if 0, that means we need to create one */
  if (status_uid == 0) {
    TLVstack *tlvs;

    tlvs = new_tlvstack (NULL, 0);
    snac13_construct_status (tlvs, TRUE);
    snac13_send_add_to_list (tlvs);
    free_tlvstack (tlvs);
  }

  sane_cl = TRUE;
  save_preferences ();
}
