/*
 *  Copyright (C) 2001 Jorn Baayen 
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "toolbar.h"
#include "toolbar_editor.h"
#include "glade.h"
#include "eel-gconf-extensions.h"

#include <string.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>
#include <libgnomeui/gnome-stock.h>

extern PreferencesDialog *pd;

/**
 * toolbar_editor_init: initialize toolbar editor 
 */
void
toolbar_editor_init (void)
{
	GtkCList *clist, *alist;
	ToolbarItemType type;
	GSList *icon_list;
	GSList *tmp;
	int i;

	/* get lists from dialog */
	clist = GTK_CLIST (pd->current_clist);
	alist = GTK_CLIST (pd->available_clist);
	g_return_if_fail (clist != NULL);
	g_return_if_fail (alist != NULL);

	/* clear them out */
	gtk_clist_clear (alist);
	gtk_clist_clear (clist);

	/* set them up */
	gtk_clist_set_sort_column (alist, 0);
	gtk_clist_set_sort_type (alist, GTK_SORT_ASCENDING);
	gtk_clist_set_use_drag_icons (clist, TRUE);
	gtk_clist_set_use_drag_icons (alist, TRUE);
	gtk_clist_set_reorderable (clist, TRUE);
	gtk_clist_freeze (alist);
	gtk_clist_freeze (clist);

        /* set button relief (glade is broken) */
	gtk_button_set_relief (GTK_BUTTON(pd->add_button),    GTK_RELIEF_NONE);
	gtk_button_set_relief (GTK_BUTTON(pd->remove_button), GTK_RELIEF_NONE);
	gtk_button_set_relief (GTK_BUTTON(pd->up_button),     GTK_RELIEF_NONE);
	gtk_button_set_relief (GTK_BUTTON(pd->down_button),   GTK_RELIEF_NONE);

	/* fill in available list with all possible items */
	for (i = 0; i < TOOLBAR_ITEM_COUNT; i++)
	{
		toolbar_editor_insert_item (alist, i);
	}

	/* fill in current list and remove current items from available list */
	icon_list = eel_gconf_get_integer_list(CONF_TOOLBAR_ICON_LIST);
	for (tmp = icon_list; tmp; tmp = g_slist_next (tmp))
	{
		/* get item type */
		type = GPOINTER_TO_INT (tmp->data);
		
		/* check for illegal values */
		if (type < 0 || type >= TOOLBAR_ITEM_COUNT)
		{
			g_warning (_("Illegal toolbar item type\n"));
			continue;
		}
		
		/* remove it from available, add it to current */
		if (type != TOOLBAR_ITEM_SEPARATOR)
		{
			toolbar_editor_remove_item (alist, type);
		}
		toolbar_editor_insert_item (clist, type);
	}

	g_slist_free (icon_list);

	/* show list updates */
	gtk_clist_sort (alist);
	gtk_clist_thaw (alist);
	gtk_clist_thaw (clist);
}

/**
 * toolbar_editor_save: save toolbar editor settings
 */
void
toolbar_editor_save (void)
{
	GtkCList *clist = GTK_CLIST (pd->current_clist);
	gint i;
	GSList *icon_list = NULL;
	GSList *tmp_new = NULL;
	GSList *old_values = 
		eel_gconf_get_integer_list(CONF_TOOLBAR_ICON_LIST);
	GSList *tmp_old = old_values;
	
	gboolean changed = FALSE;

	/* save used items */
	for (i = clist->rows-1; i >= 0; i--)
	{
		icon_list = g_slist_prepend(icon_list, 
					    gtk_clist_get_row_data (clist, i));
	}
	
	tmp_new = icon_list;
	/* has the toolbar elements been changed ? */
	while ((tmp_new != NULL) && (tmp_old != NULL) && (!changed)) {
		if (tmp_new->data != tmp_old->data) {
			changed = TRUE;
		}
		tmp_new = g_slist_next(tmp_new);
		tmp_old = g_slist_next(tmp_old);
	}
	
	if ((tmp_new != NULL) || (tmp_old != NULL)) 
		changed = TRUE;
		
	/* store toolbar state */
	if (changed == TRUE) {
		eel_gconf_set_integer_list (CONF_TOOLBAR_ICON_LIST, icon_list);
	}
	g_slist_free(icon_list);
	g_slist_free(old_values);
}

/**
 * toolbar_editor_insert_item: insert a given toolbar item into a clist
 */
gint
toolbar_editor_insert_item (GtkCList *clist, ToolbarItemType type)
{
	gchar *text[1] = { NULL };
	GtkWidget *pixmap;
	gint row;
	
	/* get a row number */
	row = gtk_clist_append (GTK_CLIST (clist), text);

	/* get stock item if available */
	if (toolbar_items[type].stock_icon)
	{
		/* build shrunken pixmap */
		pixmap = gnome_stock_pixmap_widget_at_size 
			(NULL, toolbar_items[type].stock_icon,
			 16, 16);

		/* use this to make a nice entry */
		gtk_clist_set_pixtext (clist, row, 0, 
				       _(toolbar_items[type].label), 2, 
				       GNOME_STOCK (pixmap)->pixmap.pixmap, 
				       GNOME_STOCK (pixmap)->pixmap.mask);
		
		/* no longer need the widget itself */
		gtk_object_unref (GTK_OBJECT (pixmap));
	}
	else
	{
		/* otherwise just insert dull old text */
		gtk_clist_set_text (clist, row, 0, 
				    _(toolbar_items[type].label));
		gtk_clist_set_shift (clist, row, 0, 0, 18);
	}

	/* label it so we can find it again later */
	gtk_clist_set_row_data (clist, row, GINT_TO_POINTER (type));

	/* return row */
	return row;
}

/**
 * toolbar_editor_remove_item: remove a given toolbar item from a clist
 */
void
toolbar_editor_remove_item (GtkCList *clist, ToolbarItemType type)
{
	gint row;

	/* find it by the row data we set */
	row = gtk_clist_find_row_from_data (GTK_CLIST (clist),
					    GINT_TO_POINTER (type));

	/* remove it */
	gtk_clist_remove (GTK_CLIST (clist), row);
}
