/*
 *  Copyright (C) 2002 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "misc_callbacks.h"
#include "misc_general.h"
#include "misc_gui.h"
#include "misc_string.h"
#include "embed.h"
#include "mozilla.h"
#include "bookmarks.h"
#include "prefs.h"
#include "window.h"
#include "mime.h"
#include "session.h"
#include "toolbar.h"
#include "prefs.h"
#include "themes.h"
#include "downloader.h"
#include "history.h"
#include "xlink.h"

#include <gtkmozembed.h>
#include <gtk/gtkfilesel.h>
#include <gtk/gtkvseparator.h>
#include <gtk/gtkhseparator.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-app.h>
#include <libgnomeui/gnome-app-helper.h>
#include <libgnomeui/gnome-stock.h>
#include <libgnomeui/gnome-popup-menu.h>

/* saved open URL dialog string */
gchar *open_url_str = NULL;

/* dummy entry, used for separator resizing */
static GtkWidget *dummy = NULL;

/**
 * closes a popup menu with a popup menus child item is selected
 */
gint popup_selection_done_cb (GtkMenuShell *menushell, GtkWidget *popup)
{
	gtk_menu_shell_deactivate (GTK_MENU_SHELL (popup));
	return FALSE;
}

void			
open_url_handle_link (GaleonWindow *window, LinkState state)
{
	GtkWidget *entry;
	gchar *string;

	return_if_not_window (window);
	entry = window->open_url_entry;
	g_return_if_fail (entry != NULL);

	string = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
	if (string)
	{
		gchar *string2;

		g_free (open_url_str);
		open_url_str = string;

		string2 = bookmarks_parse_nick (string, NULL);
		if (string2) 
		{
			string = string2;
		} 
		embed_activate_link (window->active_embed, NULL, 
				     string, state);
		g_free(string2);
	}

	gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (entry)));
}

/** 
 * open_url_ok_button_release_event_cb: open url dialog ok button pressed
 */
gboolean
open_url_ok_button_release_event_cb (GtkButton *button, GdkEventButton *event,
				     GaleonWindow *window)
{
	if (GTK_WIDGET (button)->window != event->window)
		return FALSE;
	open_url_handle_link (window,
		     misc_general_mouse_state_to_link_state (event->button,
			     				     event->state));
	return TRUE;
}

/** 
 * open_url_clear_button_clicked_cb: open url dialog clear button clicked
 */
void
open_url_clear_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	GtkWidget *entry = window->open_url_entry;

	g_return_if_fail (entry != NULL);

	gtk_editable_delete_text (GTK_EDITABLE (entry), 0, -1);
	gtk_widget_grab_focus (GTK_WIDGET (entry));
}

/** 
 * open_url_cancel_button_clicked_cb : open url dialog cancel button clicked
 */
void
open_url_cancel_button_clicked_cb (GtkButton *button, GaleonWindow *window)
{
	gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/**
 * history_menu_menuitem_activate_cb: go to a selected item in the 
 * back/forward history menu 
 */
void
history_menu_menuitem_activate_cb (GtkMenuItem *menuitem, gpointer user_data)
{
	GaleonEmbed *embed;
	gint index;

	/* fetch data */
	index = GPOINTER_TO_INT (user_data);
	embed = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	return_if_not_embed (embed);

	mozilla_session_history_go (embed, index);
}
/**
 * history_menu_menuitem_button_release_cb: go to a selected item in the 
 * up directory level menu 
 */
gboolean
history_menu_menuitem_button_release_cb (GtkMenuItem *menuitem,
					 GdkEventButton *event, 
					 gpointer user_data)
{
	GaleonEmbed *embed;
	gint index;
	char *url;
	/* Get the submenu widget */
	GtkWidget *menu = GTK_WIDGET (menuitem)->parent; 

	if ((event->button == 1 && !(event->state & GDK_SHIFT_MASK)) ||
			event->button == 3)
		return FALSE;

	/* fetch data */
	index = GPOINTER_TO_INT (user_data);
	embed = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	return_val_if_not_embed (embed, FALSE);

	gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
	url = mozilla_session_history_get_url (embed, index);
	embed_activate_link_mouse (embed, NULL, url, event);
	g_free (url);
	return TRUE;
}

/**
 * up_menu_menuitem_activate_cb: go to a selected item in the 
 * up directory level menu 
 */
void
up_menu_menuitem_activate_cb (GtkMenuItem *menuitem, gpointer user_data)
{
	GaleonEmbed *embed;
	gint levels;

	/* fetch data */
	levels = GPOINTER_TO_INT (user_data);
	embed = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	return_if_not_embed (embed);

	embed_go_up (embed, levels, 0);
}

/**
 * up_menu_menuitem_activate_cb: go to a selected item in the 
 * up directory level menu 
 */
gboolean
up_menu_menuitem_button_release_cb (GtkMenuItem *menuitem,
				GdkEventButton *event, gpointer user_data)
{
	GaleonEmbed *embed;
	gint levels;
	/* Get the submenu widget */
	GtkWidget *menu = GTK_WIDGET (menuitem)->parent;

	if ((event->button == 1 && !(event->state & GDK_SHIFT_MASK)) ||
			event->button == 3)
		return FALSE;
	/* fetch data */
	levels = GPOINTER_TO_INT (user_data);
	embed = gtk_object_get_user_data (GTK_OBJECT (menuitem));
	return_val_if_not_embed (embed, FALSE);

	gtk_menu_shell_deactivate (GTK_MENU_SHELL(menu));
	embed_go_up (embed, levels,
		misc_general_mouse_state_to_link_state (event->button,
							event->state));
	return TRUE;
}

/**
 * context_bm_toolbar_appearance_menu_cb: called by the toolbar 
 * appearance context menu. Set toolbar style.
 */
void
context_bm_toolbar_appearance_menu_cb (GtkRadioMenuItem *menuitem, 
				       gpointer bm)
{
	BookmarkItem *bi = (BookmarkItem *) bm;

	g_return_if_fail (BOOKMARK_ITEM_IS_FOLDER (bi));

	/* check being activated */
	if (!(GTK_CHECK_MENU_ITEM (menuitem)->active))
	{
		return;
	}

	/* button group is in reverse order! */
	bi->toolbar_style = misc_gui_radio_menuitem_index (menuitem);
	bookmarks_tb_dirty = TRUE;
	bookmarks_updated ();
}

/**
 * context_main_toolbar_appearance_menu_cb: called by the toolbar 
 * appearance context menu. Set toolbar style.
 */
void
context_main_toolbar_appearance_menu_cb (GtkRadioMenuItem *menuitem, 
					 gpointer data)
{
	/* check being activated */
	if (!(GTK_CHECK_MENU_ITEM (menuitem)->active))
	{
		return;
	}

	/* get new value */
	eel_gconf_set_integer (CONF_TOOLBAR_STYLE, 
			      misc_gui_radio_menuitem_index (menuitem));
}

/**
 * context_main_toolbar_edit_toolbar_cb:
 */
void
context_main_toolbar_edit_toolbar_cb (GtkMenuItem *menuitem, 
				      GaleonWindow *window)
{
	return_if_not_window (window);
	prefs_show_page (window, PREFS_PAGE_ID_USER_INTERFACE_TOOLBARS);
}

void
separator_toolbar_orientation_changed_cb (GtkToolbar *tb,
					  GtkOrientation o,
					  GtkWidget *sep)
{
	gboolean vsep = GTK_IS_VSEPARATOR (sep);
	GtkRequisition r, dummy_req;

	gtk_widget_size_request (GTK_WIDGET (tb), &r);

	if (dummy == NULL)
	{
		dummy = gtk_entry_new ();
	}

	gtk_widget_size_request (dummy, &dummy_req);
	
	if (o == GTK_ORIENTATION_HORIZONTAL)
	{
		if (vsep)
		{
			gint height = dummy_req.height;
			if (r.height > height) height = r.height;
			gtk_widget_set_usize (sep, 5, height / 2);
			gtk_widget_show (sep);
		}
		else
			gtk_widget_hide (sep);
	}
	else
	{
		if (!vsep)
		{
			gtk_widget_set_usize (sep, r.width / 2, 5);
			gtk_widget_show (sep);
		}
		else
			gtk_widget_hide (sep);
	}
}

/**
 * gtk_theme_changed_cb: called when a Gtk+ theme change is detected, update the
 * tab label styles and aa pixmaps
 * FIXME this seems to be called too many times
 */
GdkFilterReturn
gtk_theme_changed_cb (GdkXEvent *xevent, GdkEvent *event, gpointer data)
{
	GList *l;
	
	/* reload colour styles, reload pixmaps */
	misc_gui_colors_init ();
	themes_reset_cache ();
	bookmarks_load_icons ();
	bookmarks_tb_dirty = TRUE;
	bookmarks_updated ();
	for (l = all_windows; l != NULL; l = g_list_next (l))
	{
		GaleonWindow *window = (GaleonWindow *) l->data;

		window_save_layout (window);
		toolbar_recreate (window);
		window_restore_layout (window);
		window_update_tabs_appearance (window);
		window_statusbar_set_security_icon (window,
			window->active_embed->secure,
			window->active_embed->security_tooltip);
		bookmarks_menu_recreate (window);
	}

	/* process other GDK events */
	return GDK_FILTER_CONTINUE;
}

gboolean
generic_link_button_press_event_cb (GtkWidget *widget, GdkEventButton *event,
				    GaleonWindow *window)
{
	GtkWidget *menu;
	gint action;
	GaleonEmbed *embed;
	gchar *url = gtk_object_get_data (GTK_OBJECT (widget), "url");
	gchar *tmp = gtk_object_get_data (GTK_OBJECT (widget), "title");
	gchar *title = NULL;

	/* reload context menu */
	static GnomeUIInfo menu_uiinfo[] =
	{
		GNOMEUIINFO_ITEM_STOCK (N_("Open in a new window"),
					NULL, NULL, GNOME_STOCK_MENU_NEW),
		GNOMEUIINFO_ITEM_STOCK (N_("Open in a new tab"),
					NULL, NULL, GNOME_STOCK_MENU_NEW),
		GNOMEUIINFO_ITEM_STOCK (N_("Download link"),
					NULL, NULL, GNOME_STOCK_MENU_SAVE),
		GNOMEUIINFO_ITEM_STOCK (N_("Copy link location"),
					NULL, NULL, GNOME_STOCK_MENU_COPY),
		GNOMEUIINFO_ITEM_NONE (N_("Add bookmark"), NULL, NULL),
		GNOMEUIINFO_END
	};

	/* check args */
	return_val_if_not_window (window, FALSE);
	g_assert (url != NULL);

	embed = window->active_embed;
	return_val_if_not_embed (embed, TRUE);

	/* setup title string */
	if (tmp)
	{
		title = misc_string_strip_uline_accel (tmp);
	}

	/* handle button accordingly */	
	switch (event->button)
	{
	case 1:
		/* nothing -- handled by click callback */
		break;

	case 2:
		/* nothing on middle click */
		break;

	case 3:
		/* show context menu */
		menu = misc_gui_new_popup_menu_lock_accels (menu_uiinfo);
		action = gnome_popup_menu_do_popup_modal 
			(GTK_WIDGET (menu), NULL, NULL, event, NULL);

		/* do appropiate action */
		switch (action)
		{
		case 0: 
			embed_activate_link (window->active_embed, NULL,
					     url, LINKSTATE_NEWWIN);
			break;

		case 1:
			embed_activate_link (window->active_embed, NULL,
					     url, LINKSTATE_NEWTAB);
			break;
		case 2:
			downloader_save_link (embed, url);
			break;
		case 3: 
			embed_copy_text_to_clipboard (url,
						      window->active_embed);
			break;
		case 4:
			bookmarks_add_bookmark (title, url, NULL,
						GTK_WINDOW (window->wmain), 0);
			break;
		}

		/* destroy the popup menu */
		gtk_widget_unref (menu);
		
		if (GTK_IS_MENU (GTK_WIDGET (widget)->parent))
		{
			GtkMenu *menu = GTK_MENU (GTK_WIDGET (widget)->parent);
			gtk_menu_shell_deactivate (GTK_MENU_SHELL (menu));
		}
		
		if (title) g_free (title);
		return TRUE;
	}

	if (title) g_free (title);
	return FALSE;
}

/**
 * generic_editable_button_press_event_cb: context menu for an editable
 */
gboolean
generic_editable_button_press_event_cb (GtkWidget *entry,
				        GdkEventButton *event,
				        gpointer user_data)
{
	if (event->button == 3)
	{
		/* 3rd button clicked, display contextmenu */
		GtkWidget *menu;
		gint action;
		GtkEditable *editable = GTK_EDITABLE (entry);

		/* context menu */
		static GnomeUIInfo menu_uiinfo[] =
		{
			GNOMEUIINFO_ITEM_STOCK (N_("Cut"),
					NULL, NULL, GNOME_STOCK_MENU_CUT),
			GNOMEUIINFO_ITEM_STOCK (N_("Copy"),
					NULL, NULL, GNOME_STOCK_MENU_COPY),
			GNOMEUIINFO_ITEM_STOCK (N_("Paste"),
					NULL, NULL, GNOME_STOCK_MENU_PASTE),
			GNOMEUIINFO_SEPARATOR,
			GNOMEUIINFO_ITEM_STOCK (N_("Select all"),
					NULL, NULL, GNOME_STOCK_MENU_BLANK),
			GNOMEUIINFO_END
		};
	
		/* setup context menu */
		menu = misc_gui_new_popup_menu_lock_accels (menu_uiinfo);

		/* setup cut/copy/paste sensivity */
		if (!(editable->has_selection && editable->editable))
		{
			gtk_widget_set_sensitive (menu_uiinfo[0].widget, FALSE);
		}
		if (!editable->has_selection)
		{
			gtk_widget_set_sensitive (menu_uiinfo[1].widget, FALSE);
		}
		if (!(editable->editable && galeon_x_clipboard_present ()))
		{
			gtk_widget_set_sensitive (menu_uiinfo[2].widget, FALSE);
		}
		
		/* run */
		action = gnome_popup_menu_do_popup_modal 
			(GTK_WIDGET (menu), NULL, NULL, event, NULL);

		/* do appropiate action */
		switch (action)
		{
		case 0: 
			/* cut selection */
			gtk_editable_cut_clipboard (editable);
			break;
		case 1:
			/* copy selection */
			gtk_editable_copy_clipboard (editable);
			break;
		case 2:
			/* paste selection */
			gtk_editable_paste_clipboard (editable);
			break;
		case 4:
			/* select everything */
			gtk_editable_select_region (editable, 0, -1);
			break;
		}

		/* destroy the popup menu */
		gtk_widget_unref (menu);
		
		/* gtk+ goofiness */
		gtk_signal_emit_stop_by_name (GTK_OBJECT (entry),
				              "button-press-event");

		return TRUE;
	}

	return FALSE;
}
