#include <sys/time.h>
#include <linux/time.h>
#include <errno.h>

struct hrt_time_t {
  unsigned long secs;
  unsigned long sub_sec; /* Less than one second. */
  unsigned long resolution; /* Resolution of timer */
};

struct hrtcmd{
  int cmd;
  int clk;
  struct hrt_time_t interval;
  struct hrt_time_t tod;
  int flags;
  int error;
  int reserved[3];
};

int gettimeofday(struct timeval *tpnt)
{
  struct hrt_time_t hrtime;
  int status;

  if( tpnt == NULL )
    {
      return 0;
    }

  /*
   * Indicate that we want microsecond resolution on the usec field.
   */
  hrtime.resolution = 1000000;

  status = _hrtsys(0, 1, 1, 0, &hrtime);

  if(status != 0) return -1;

  tpnt->tv_sec = hrtime.secs;
  tpnt->tv_usec = hrtime.sub_sec;
  return 0;
}


static int set_hclk(int timer_nbr, struct hrtcmd * hcmd)
{
  switch(timer_nbr)
    {
    case ITIMER_REAL:
      hcmd->clk = 1;
      break;
    case ITIMER_VIRTUAL:
      hcmd->clk = 2;
      break;
    case ITIMER_PROF:
      hcmd->clk = 4;
      break;
    default:
      errno = EINVAL;
      return -1;
    }
  return 0;
}

/*
 * These functions interface to the hrtalarm function which is defined
 * in the ibcs2 emulator.
 */
int getitimer(int timer_nbr, struct itimerval *value)
{
  struct hrtcmd hcmd;
  int status;

  memset(&hcmd, 0, sizeof(hcmd));
  status = set_hclk(timer_nbr, &hcmd);
  if(status != 0) return -1;
	 
  memset(value, 0, sizeof(*value));

  hcmd.cmd = 0xd;
  /*
   * Get the requested information.
   */
  status = _hrtsys(1, &hcmd, 1, 0, 0);

  if(status != 1)
    {
      errno = EINVAL;
      return -1;
    }

  value->it_interval.tv_sec = hcmd.interval.secs;
  value->it_interval.tv_usec = hcmd.interval.sub_sec;
  return 0;
}

int setitimer(int timer_nbr, const struct itimerval *value,
          struct itimerval *ovalue)
{
  struct hrtcmd hcmd;
  int status;

  if(timer_nbr < ITIMER_REAL || timer_nbr > ITIMER_PROF
     || value == NULL)
    {
      goto oops;
    }

  /*
   * Look for some combinations which we know are illegal.
   */
  if( value->it_value.tv_sec < 0 
     || value->it_value.tv_usec < 0
     || value->it_interval.tv_sec < 0 
     || value->it_interval.tv_usec < 0)
    {
      goto oops;
    }

  if( value->it_value.tv_sec > 100000000 
     || value->it_value.tv_usec > 1000000
     || value->it_interval.tv_sec > 100000000
     || value->it_interval.tv_usec > 1000000)
    {
      goto oops;
    }

  if( ovalue != NULL )
    {
      status = getitimer(timer_nbr, ovalue);
      if( status != 0 )
	{
	  goto oops;
	}
    }

  memset(&hcmd, 0, sizeof(hcmd));
  status = set_hclk(timer_nbr, &hcmd);
  if(status != 0) return -1;

  /*
   * Now check to make sure that the timeval contains sensible
   * information.
   */

  if( value->it_value.tv_sec == 0 && value->it_value.tv_usec == 0)
    {
      /*
       * Turning off itimer.  Nothing much to do here.
       */
      hcmd.cmd = 0x10;
    }
  else if( value->it_interval.tv_sec == 0 && value->it_interval.tv_usec == 0 )
    {
      /*
       * One shot timer.
       */
      hcmd.cmd = 0xc;
      hcmd.interval.resolution = 1000000;
      hcmd.interval.secs = value->it_value.tv_sec;
      hcmd.interval.sub_sec = value->it_value.tv_usec;
    }
  else
    {
      /*
       * Enable repeating timer.
       */
      hcmd.cmd = 0xf;

      hcmd.interval.resolution = 1000000;
      hcmd.interval.secs = value->it_interval.tv_sec;
      hcmd.interval.sub_sec = value->it_interval.tv_usec;

      hcmd.tod.resolution = 1000000;
      hcmd.tod.secs = value->it_value.tv_sec;
      hcmd.tod.sub_sec = value->it_value.tv_usec;
    }

  return _hrtsys(1, &hcmd, 1, 0, 0);

 oops:
  errno = EINVAL;
  return -1;

}
