/*----------------------------------------------------------------------------

   libtunepimp -- The MusicBrainz tagging library.
                  Let a thousand taggers bloom!

   Copyright (C) Robert Kaye 2003

   This file is part of libtunepimp.

   libtunepimp is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   libtunepimp is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with libtunepimp; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   $Id: tunepimp.cpp,v 1.87 2005/07/10 04:33:21 robert Exp $

----------------------------------------------------------------------------*/
#ifdef WIN32
#if _MSC_VER == 1200
#pragma warning(disable:4786)
#endif
#endif

#include <stdio.h>
#include <map>
using namespace std;

#ifdef WIN32
#include <winsock.h>
#endif

#include "../config.h"
#include "tunepimp.h"
#include "watchdog.h"
#include "dirsearch.h"

#define DB         printf("%s:%d\n", __FILE__, __LINE__);
#ifndef WIN32
#define PLUGIN_DIR PREFIX"/lib/tunepimp/plugins"
#endif

TunePimp::TunePimp(const string &appName, const string &appVersion, int startThreads, const char *pluginDir)
{
    context.setTunePimp(this);

    callback = NULL;
    proxyPort = 0;

    plugins = new Plugins();

#ifndef WIN32
    if (pluginDir)
        plugins->load(pluginDir, 0);
    else
    {
        char *ptr = getenv("HOME");
        if (ptr)
        {
            string path(ptr);

            path += string("/.tunepimp/plugins");
            plugins->load(path.c_str(), 0);
        }
        plugins->load("plugins", 0);
        plugins->load(PLUGIN_DIR, 0);
    }
#else
    string path;

    if (pluginDir)
        plugins->load(pluginDir, 0);
    else
        plugins->load("plugins", 0);
#endif

    cache = new FileCache(this);
    submit = new SubmitInfo(this, appName, appVersion);

    if (startThreads & TP_THREAD_ANALYZER)
    {
        watchdog = new WatchdogThread(this);
        analyzer = new Analyzer(this, plugins, cache, submit, watchdog);
    }
    else
    {
        watchdog = NULL;
        analyzer = NULL;
    }

    if (startThreads & TP_THREAD_LOOKUPTRM)
        lookup = new LookupThread(this, cache, submit);
    else
        lookup = NULL;

    if (startThreads & TP_THREAD_LOOKUPFILE)
        fileLookup = new FileLookupThread(this, cache);
    else
        fileLookup = NULL;

    if (startThreads & TP_THREAD_READ)
        read = new ReadThread(this, cache, plugins);
    else
        read = NULL;

    if (startThreads & TP_THREAD_WRITE)
        write = new WriteThread(this, cache, plugins);
    else
        write = NULL;

    plugins->getSupportedExtensions(extList);

    if (analyzer)
       analyzer->start();
    if (read)
       read->start();
    if (lookup)
       lookup->start();
    if (fileLookup)
       fileLookup->start();
    if (write)
       write->start();

    if (watchdog)
       watchdog->start();
}

TunePimp::~TunePimp(void)
{
    Analyzer         *aTemp;
    LookupThread     *lTemp;
    FileLookupThread *fTemp;
    WriteThread      *rTemp;
    WatchdogThread   *wTemp;
    ReadThread       *mTemp;

    // Turn the watchdog thread off to make sure it doesn't spawn a new analyzer
    if (watchdog)
        watchdog->stop();

    if (analyzer)
    {
        aTemp = analyzer;
        analyzer = NULL;
        delete aTemp;
    }

    if (lookup)
    {
        lTemp = lookup;
        lookup = NULL;
        delete lTemp;
    }

    if (fileLookup)
    {
        fTemp = fileLookup;
        fileLookup = NULL;
        delete fTemp;
    }

    if (read)
    {
        mTemp = read;
        read = NULL;
        delete mTemp;
    }
    if (watchdog)
    {
        rTemp = write;
        write = NULL;
        delete rTemp;
    }

    if (watchdog)
    {
        wTemp = watchdog;
        watchdog = NULL;
        delete wTemp;
    }

    delete submit;
    delete cache;

    plugins->unload();
    delete plugins;
}

// Get the version of the library
void TunePimp::getVersion(int &major, int &minor, int &rev)
{
    sscanf(VERSION, "%d.%d.%d", &major, &minor, &rev);
}

string &TunePimp::getFileNameEncoding(void)
{
    return context.getFileNameEncoding();
}

void TunePimp::setFileNameEncoding(const string &encoding)
{
    context.setFileNameEncoding(encoding);
}

void TunePimp::setServer(const string &server, short port)
{
    this->server = server;
    this->port = port;
}

void TunePimp::setProxy(const string &proxyServer, short proxyPort)
{
    this->proxyServer = proxyServer;
    this->proxyPort = proxyPort;
}

void TunePimp::getServer(string &server, short &port)
{
    server = this->server;
    port = this->port;
}

void TunePimp::getProxy(string &proxyServer, short &proxyPort)
{
    proxyServer = this->proxyServer;
    proxyPort = this->proxyPort;
}

void TunePimp::getSupportedExtensions(vector<string> &extList)
{
    extList = this->extList;
}

void TunePimp::setWriteID3v1(bool writeID3v1)
{
    context.setWriteID3v1(writeID3v1);
}

bool TunePimp::getWriteID3v1(void)
{
    return context.getWriteID3v1();
}

void TunePimp::setWriteID3v2_3(bool writeID3v2_3)
{
    context.setWriteID3v2_3(writeID3v2_3);
}

bool TunePimp::getWriteID3v2_3(void)
{
    return context.getWriteID3v2_3();
}

void TunePimp::setID3Encoding(TPID3Encoding enc)
{
    context.setID3Encoding(enc);
}

TPID3Encoding TunePimp::getID3Encoding(void)
{
    return context.getID3Encoding();
}

void TunePimp::setClearTags(bool clearTags)
{
    context.setClearTags(clearTags);
}

bool TunePimp::getClearTags(void)
{
    return context.getClearTags();
}

void TunePimp::setDestDir(const string &destDir)
{
    context.setDestDir(destDir);
}

string &TunePimp::getDestDir(void)
{
    return context.getDestDir();
}

void TunePimp::setFileMask(const string &fileMask)
{
    context.setFileMask(fileMask);
}

string &TunePimp::getFileMask(void)
{
    return context.getFileMask();
}

void TunePimp::setAllowedFileCharacters(const string &allowedChars)
{
    context.setAllowedFileCharacters(allowedChars);
}

string &TunePimp::getAllowedFileCharacters(void)
{
    return context.getAllowedFileCharacters();
}

void TunePimp::setVariousFileMask(const string &variousFileMask)
{
    context.setVariousFileMask(variousFileMask);
}

string &TunePimp::getVariousFileMask(void)
{
    return context.getVariousFileMask();
}

void TunePimp::setNonAlbumFileMask(const string &nonAlbumFileMask)
{
    context.setNonAlbumFileMask(nonAlbumFileMask);
}

string &TunePimp::getNonAlbumFileMask(void)
{
    return context.getNonAlbumFileMask();
}

void TunePimp::setTopSrcDir(const string &srcDir)
{
    context.setTopSrcDir(srcDir);
}

string &TunePimp::getTopSrcDir(void)
{
    return context.getTopSrcDir();
}

void TunePimp::setMoveFiles(bool moveFiles)
{
    context.setMoveFiles(moveFiles);
}

bool TunePimp::getMoveFiles(void)
{
    return context.getMoveFiles();
}

void TunePimp::setRenameFiles(bool renameFiles)
{
    context.setRenameFiles(renameFiles);
}

bool TunePimp::getRenameFiles(void)
{
    return context.getRenameFiles();
}

void TunePimp::setTRMCollisionThreshold(int trmThreshold)
{
    if (trmThreshold > 100)
        trmThreshold = 100;

    context.setTRMCollisionThreshold(trmThreshold);
}

int TunePimp::getTRMCollisionThreshold(void)
{
    return context.getTRMCollisionThreshold();
}

void TunePimp::setAutoSaveThreshold(int autoSaveThreshold)
{
    context.setAutoSaveThreshold(autoSaveThreshold);
}

int TunePimp::getAutoSaveThreshold(void)
{
    return context.getAutoSaveThreshold();
}

void TunePimp::setMinTRMThreshold(int minThreshold)
{
    if (minThreshold > 100)
        minThreshold = 100;

    context.setMinTRMThreshold(minThreshold);
}

int TunePimp::getMinTRMThreshold(void)
{
    return context.getMinTRMThreshold();
}

void TunePimp::setMaxFileNameLen(int maxFileNameLen)
{
    context.setMaxFileNameLen(maxFileNameLen);
}

int TunePimp::getMaxFileNameLen(void)
{
    return context.getMaxFileNameLen();
}

void TunePimp::setAutoRemoveSavedFiles(bool autoRemoveSavedFiles)
{
    context.setAutoRemoveSavedFiles(autoRemoveSavedFiles);
}

bool TunePimp::getAutoRemoveSavedFiles(void)
{
    return context.getAutoRemoveSavedFiles();
}

void TunePimp::setAnalyzerPriority(TPThreadPriorityEnum pri)
{
    if (analyzer)
        analyzer->setPriority(pri);
    context.setAnalyzerPriority(pri);
}

TPThreadPriorityEnum TunePimp::getAnalyzerPriority(void)
{
    if (analyzer == NULL)
	return eNormal;

    return analyzer->getPriority();
}

void TunePimp::setAutoFileLookup(bool enable)
{
    if (lookup)
        lookup->setAutoFileLookup(enable);
}

bool TunePimp::getAutoFileLookup(void)
{
    if (lookup)
        return lookup->getAutoFileLookup();
    return false; 
}

void TunePimp::setCallback(TPCallback *callback)
{
    this->callback = callback;
}

TPCallback *TunePimp::getCallback(void)
{
    return callback;
}

void TunePimp::getError(string &error)
{
    error = err;
}

// Set the debug option -- if enabled debug output will be printed to stdout
void TunePimp::setDebug(bool debug)
{
    context.setDebug(debug);
}

bool TunePimp::getDebug(void)
{
    return context.getDebug();
}

void TunePimp::setUserInfo(const string &userName, const string &password)
{
    submit->setUserInfo(userName, password);
}

void TunePimp::getUserInfo(string &userName, string &password)
{
    submit->getUserInfo(userName, password);
}

// Stoopid windows helper functions to init the winsock layer.
#ifdef WIN32
void TunePimp::WSAInit(void)
{
    WSADATA sGawdIHateMicrosoft;
    WSAStartup(0x0002,  &sGawdIHateMicrosoft);
}

void TunePimp::WSAStop(void)
{
    WSACleanup();
}
#endif

int TunePimp::addFile(const string &fileName, bool readMetadataNow)
{
    int          fileId;

    fileId = cache->add(fileName);
    if (fileId < 0)
	return fileId;

    if (readMetadataNow)
    {
        ReadThread   *read;
        Metadata      mdata;
        Track        *track;

        read = new ReadThread(this, cache, plugins);
        track = cache->getTrack(fileId);
        if (track)
        {
            track->lock();
            read->readMetadata(track, true);
            track->unlock();
            cache->release(track);
        }
        delete read;
    }
    else
    {
        if (callback)
            callback->notify(this, tpFileAdded, fileId, eMetadataRead);

        if (read)
           read->wake();
    }

    return fileId;
}

int TunePimp::addDir(const string &dirPath)
{
    DirSearch search(this, extList);
    int       count, fileId;

    count = search.recurseDir(dirPath.c_str());
    if (count > 0)
    {
        vector<string>           fileList;
        vector<string>::iterator i;
        search.getFiles(fileList);
        for(i = fileList.begin(); i != fileList.end(); i++)
        {
            fileId = cache->add(*i);
            if (callback)
                callback->notify(this, tpFileAdded, fileId, eMetadataRead);
        }
        if (read)
            read->wake();
    }

    return count;
}

int TunePimp::getNumUnsubmitted(void)
{
    return submit->getNumItems();
}

int TunePimp::getNumUnsavedItems(void)
{
    return cache->getNumUnsavedItems();
}

void TunePimp::getTrackCounts(map<TPFileStatus, int> &counts)
{
    return cache->getCounts(counts);
}

void TunePimp::remove(int fileId)
{
    Metadata  data;
    Track    *track;

    // When removing a track, nuke the track id from the submit list
    track = cache->getTrack(fileId);
    if (track)
    {
        track->lock();
        track->getServerMetadata(data);
        track->unlock();
        cache->release(track);

        if (!data.trackId.empty())
            submit->remove(data.trackId);
    }

    cache->remove(fileId);
    if (callback)
        callback->notify(this, tpFileRemoved, fileId, eDeleted);
}

void TunePimp::getFileIds(vector<int> &ids)
{
    cache->getFileIds(ids);
}

int TunePimp::getNumFiles(void)
{
    return cache->getNumItems();
}

Track *TunePimp::getTrack(int fileId)
{
    return cache->getTrack(fileId);
}

void TunePimp::releaseTrack(Track *track)
{
    cache->release(track);
}

// Since this function blocks, the track is assumed to be locked
// and should not be unlocked.
void TunePimp::wake(Track *track)
{
    if (callback)
    {
        int fileId;

        fileId = cache->getFileIdFromTrack(track);
        if (fileId >= 0)
            callback->notify(this, tpFileChanged, fileId, track->getStatus());
    }

    if (analyzer)
        analyzer->wake();

    if (read)
        read->wake();

    if (lookup)
        lookup->wake();

    if (fileLookup)
        fileLookup->wake();

    if (write)
        write->wake();
}

void TunePimp::writeTagsComplete(bool error)
{
    if (callback)
        callback->notify(this, tpWriteTagsComplete, (int)error, eError);
}

void TunePimp::trackRemoved(int fileId)
{
    if (callback)
        callback->notify(this, tpFileRemoved, fileId, eDeleted);
}

void TunePimp::setStatus(const string &status)
{
    if (callback)
        callback->status(this, status);
}

TPError TunePimp::submitTRMs(void)
{
    LookupStatus ret;

    ret = submit->submit();
    if (ret != eSubmitOk)
    {
        submit->getError(err);
        return tpSubmitError;
    }
    submit->clear();

    return tpOk;
}

void TunePimp::addTRMSubmission(const string &trackId, const string &trmId)
{
    if (submit)
        submit->add(trackId, trmId);
}

int TunePimp::getRecognizedFileList (int threshold, vector<int> &fileIds)
{
    return cache->getRecognizedFileList(threshold, fileIds);
}

bool TunePimp::writeTags(vector<int> *fileIds)
{
    vector<Track *>            tracks;
    vector<Track *>::iterator  i;
    vector<int>::iterator      j;
    Track                     *track;

    if (fileIds)
    {
        for(j = fileIds->begin(); j != fileIds->end(); j++)
        {
            track = cache->getTrack(*j);
            if (!track)
            {
                err = "Invalid track in write tags list.";
                return false;
            }
            if (track->getStatus() != eRecognized)
            {
                err = "All tracks must be recognized before writing tags.";
                return false;
            }
            tracks.push_back(track);
        }
    }
    else
        cache->getTracksFromStatus(eRecognized, tracks);

    // First, change all the states 
    for(i = tracks.begin(); i != tracks.end(); i++)
    {
        (*i)->lock();
        (*i)->setStatus(eVerified);
        (*i)->unlock();
    }

    // And then wake all the tracks
    for(i = tracks.begin(); i != tracks.end(); i++)
    {
        wake(*i);
        cache->release(*i);
    }
    write->wake();

    return true;
}

TPError TunePimp::selectResult(Track *track, int resultIndex)
{
    vector<TPResult *>  results;
    TPResultType        type;
    TPResult           *result;
    Metadata            data;

    track->lock();

    track->getResults(type, results);
    if (resultIndex < 0 || resultIndex >= (int)results.size())
    {
        track->unlock();
        return tpInvalidIndex;
    }

    result = results[resultIndex];

    track->getServerMetadata(data);
    if (type == eArtistList)
    {
        TPArtistResult *res = (TPArtistResult *)result;

        data.artistId = res->getId();
        data.artist = res->getName();
        data.sortName = res->getSortName();

        track->setServerMetadata(data);
        track->setStatus(eFileLookup);
    }
    else
    if (type == eAlbumList)
    {
        TPAlbumResult *res = (TPAlbumResult *)result;
        TPArtistResult artist;

        artist = res->getArtist();

        data.artistId = artist.getId();
        data.artist = artist.getName();
        data.sortName = artist.getSortName();

        data.albumId = res->getId();
        data.album = res->getName();
        data.variousArtist = res->getVariousArtists();

	/* I'm not familiar enough with this code to know if this belongs here... */
        data.nonAlbum = res->getNonAlbum();

        data.albumType = res->getType();
        data.albumStatus = res->getStatus();
        data.releaseYear = res->getReleaseYear();
        data.releaseMonth = res->getReleaseMonth();
        data.releaseDay = res->getReleaseDay();
        data.releaseCountry = res->getReleaseCountry();

        track->setServerMetadata(data);

        // If we have track information, try and resolve that as well.
        // If not, leave things here -- there is nothing else to do.
        track->getLocalMetadata(data);
        if (!data.track.empty())
            track->setStatus(eFileLookup);
        else
            track->setStatus(eUnrecognized);
    }
    else
    if (type == eTrackList)
    {
        string trm;
        TPArtistResult artist;
        TPAlbumResult album;

        TPAlbumTrackResult *res = (TPAlbumTrackResult *)result;

        artist = res->getArtist();
        album = res->getAlbum();

        data.artistId = artist.getId();
        data.artist = artist.getName();
        data.sortName = artist.getSortName();

        data.albumId = album.getId();
        data.album = album.getName();

        data.variousArtist = album.getVariousArtists();
        data.nonAlbum = album.getNonAlbum();

        data.albumType = album.getType();
        data.albumStatus = album.getStatus();
        data.releaseYear = album.getReleaseYear();
        data.releaseMonth = album.getReleaseMonth();
        data.releaseDay = album.getReleaseDay();
        data.releaseCountry = album.getReleaseCountry();

        data.trackId = res->getId();
        data.track = res->getName();
        data.duration = res->getDuration();
        data.trackNum = res->getTrackNum();

        track->getTRM(trm);
        submit->add(data.trackId, trm);

        track->setServerMetadata(data);
        track->setStatus(eRecognized);
    }

    wake(track);
    track->unlock();

    return tpOk;
}

void TunePimp::misidentified(int fileId)
{
    Track        *track;
    TPFileStatus  status;

    track = cache->getTrack(fileId);
    if (track)
    {
        string   trm;
        Metadata data;

        track->lock();
        track->getTRM(trm);
        track->getServerMetadata(data);
        if (!data.trackId.empty() && !trm.empty())
            submit->remove(data.trackId);

        if (trm.empty())
        {
            track->setTRM(ANALYZER_REDO_STRING);
            track->setStatus(ePending);
            status = ePending;
        }
        else
        {
            if (lookup && lookup->getAutoFileLookup())
            {
                track->setStatus(eFileLookup);
                status = eFileLookup;
            }
            else
            {
                track->setStatus(eUnrecognized);
                status = eUnrecognized;
            }
        }

        string format = data.fileFormat;
        data.clear();
        data.fileFormat = format;

        track->setServerMetadata(data);
        track->setError("");
        track->unlock();
        wake(track);
        cache->release(track);

        if (callback)
            callback->notify(this, tpFileChanged, fileId, status);
    }
}

void TunePimp::identifyAgain(int fileId)
{
    Track *track;

    track = cache->getTrack(fileId);
    if (track)
    {
        string   trm;
        Metadata data;

        track->lock();
        track->getTRM(trm);
        track->getServerMetadata(data);
        if (!data.trackId.empty() && !trm.empty())
            submit->remove(data.trackId);

        data.clear();
        track->setTRM(ANALYZER_REDO_STRING);
        track->setServerMetadata(data);
        track->setError("");
        track->setStatus(ePending);
        track->unlock();
        wake(track);
        cache->release(track);

        if (callback)
            callback->notify(this, tpFileChanged, fileId, ePending);
    }
}

void TunePimp::analyzerDied(int fileId)
{
    Track  *track;
    Analyzer *oldAnalyzer;

    if (analyzer == NULL)
        return;

    track = cache->getTrack(fileId);
    if (track)
    {
        track->lock();
        track->setStatus(eError);
        track->setError("Cannot decode file. (Decoder crashed)");
        track->unlock();
        wake(track);

        // Here we need to call release TWICE since the crashed
        // analyzer had a reference on the track. 
        cache->release(track);
        cache->release(track);

        if (callback)
            callback->notify(this, tpFileChanged, fileId, eError);
    }

    oldAnalyzer = analyzer;
    analyzer = new Analyzer(this, plugins, cache, submit, watchdog);
    analyzer->start();

    delete oldAnalyzer;
}

// I'm not sure this function can be completely uncommented. But for now we'll go with it.
void TunePimp::trackChangedStatus(Track *track)
{
#if 0
    int fileId;

    if (callback)
    {
        fileId = cache->getFileIdFromTrack(track);
        if (fileId >= 0)
            callback->notify(this, tpFileChanged, fileId, track->getStatus());
    }
#endif
}
